// Based on gimli-rs/addr2line (https://github.com/gimli-rs/addr2line):
// > Copyright (c) 2016-2018 The gimli Developers
// >
// > Permission is hereby granted, free of charge, to any
// > person obtaining a copy of this software and associated
// > documentation files (the "Software"), to deal in the
// > Software without restriction, including without
// > limitation the rights to use, copy, modify, merge,
// > publish, distribute, sublicense, and/or sell copies of
// > the Software, and to permit persons to whom the Software
// > is furnished to do so, subject to the following
// > conditions:
// >
// > The above copyright notice and this permission notice
// > shall be included in all copies or substantial portions
// > of the Software.
// >
// > THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF
// > ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// > TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// > PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT
// > SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
// > CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// > OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
// > IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// > DEALINGS IN THE SOFTWARE.


#[derive(Default)]
pub(crate) struct RangeAttributes<R: gimli::Reader> {
    pub(crate) low_pc: Option<u64>,
    pub(crate) high_pc: Option<u64>,
    pub(crate) size: Option<u64>,
    pub(crate) ranges_offset: Option<gimli::RangeListsOffset<R::Offset>>,
}

impl<R: gimli::Reader> RangeAttributes<R> {
    pub(crate) fn for_each_range<F: FnMut(gimli::Range)>(
        &self,
        unit: gimli::UnitRef<'_, R>,
        mut f: F,
    ) -> gimli::Result<bool> {
        let mut added_any = false;
        let mut add_range = |range: gimli::Range| {
            if range.begin < range.end {
                f(range);
                added_any = true
            }
        };
        if let Some(ranges_offset) = self.ranges_offset {
            let mut range_list = unit.ranges(ranges_offset)?;
            while let Some(range) = range_list.next()? {
                add_range(range);
            }
        } else if let (Some(begin), Some(end)) = (self.low_pc, self.high_pc) {
            add_range(gimli::Range { begin, end });
        } else if let (Some(begin), Some(size)) = (self.low_pc, self.size) {
            add_range(gimli::Range {
                begin,
                end: begin + size,
            });
        }
        Ok(added_any)
    }

    pub(crate) fn bounds(&self) -> Option<gimli::Range> {
        if let (Some(begin), Some(end)) = (self.low_pc, self.high_pc) {
            Some(gimli::Range { begin, end })
        } else if let (Some(begin), Some(size)) = (self.low_pc, self.size) {
            Some(gimli::Range {
                begin,
                end: begin + size,
            })
        } else {
            None
        }
    }
}
