/* $Id: raddump.c,v 1.2 2004/09/22 15:15:52 iscjonm Exp $
 *
 * Copyright (C) 2004-2005 The Trustees of the University of Pennsylvania
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <unistd.h>

#ifdef HAVE_NET_IF_H
#  include <net/if.h>
#endif /* HAVE_NET_IF_H */

#ifdef HAVE_NET_ETHERNET_H
#  include <net/ethernet.h>
#else /* ! HAVE_NET_ETHERNET_H */
#  ifdef HAVE_NETINET_IF_ETHER_H
#    include <netinet/if_ether.h>
#  endif /* HAVE_NETINET_IF_ETHER_H */
#endif /* HAVE_NET_ETHERNET_H */

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <netdb.h>

#ifdef HAVE_PCAP_H
/* for tcpdump formatting */
#include <pcap.h>
#endif /* HAVE_PCAP_H */

#include "vlan.h"
#include "pktrecord.h"
#include "radius.h"

int verbose = 0;
int lookup_hostnames = 1;
int use_short_hostnames = 0;

static void usage(char *progname);
static void parse_cmdline(int argc, char *argv[]);

int main(int argc, char **argv) {

  struct prec_handle *ph = NULL;
  struct pktrecord p;
  struct ether_header eh;
  struct vlan_hdr vh;
  int tagged_vlan_found = 0;
  struct ip iph;
  struct udphdr uh;
  unsigned short int udp_srcport;
  unsigned short int udp_dstport;
  struct radius_hdr rh;
  struct radius_attr *ra;
  unsigned int offs;
  struct timeval tstamp;
  struct tm localt;
  int pkts_in = 0;
  int rh_offs = 0;

  char srchost[256];
  char dsthost[256];
  char vlan_id_string[256];

  rad_init();

  parse_cmdline(argc, argv);

  if (!prec_init(&ph)) {
    fprintf(stderr,"Unrecognized packet trace format\n");
    exit(EXIT_FAILURE);
  }
  if (!prec_is_eth(ph)) {
    fprintf(stderr,"These aren't Ethernet packets!\n");
    exit(EXIT_FAILURE);
  }
  pkts_in = 0;

  memset(&p, 0, sizeof(p));
  
  while(prec_next_packet(ph,&p)) {
    pkts_in++;

    offs = 0;

    /* locate Ethernet header */
    if (p.included_len < offs + sizeof(eh)) continue;
    memcpy(&eh, p.pkt_data + offs, sizeof(eh));

    /* check for 802.1q VLAN tags */
    tagged_vlan_found = 0;
    vlan_id_string[0] = '\0';
    switch(ntohs(eh.ether_type)) {
      case ETHERTYPE_IP:
	offs += sizeof(eh);
	break;
      case ETHERTYPE_VLAN:
	if (p.included_len < offs + sizeof(vh)) continue;
	memcpy(&vh, p.pkt_data + offs, sizeof(vh));
	tagged_vlan_found = 1;
	if (ntohs(vh.ether_type) != ETHERTYPE_IP) continue;
	offs += sizeof(vh);
	break;
      default:
	continue;
    }

    /* locate IP header */
    if (p.included_len < offs + sizeof(iph)) continue;
    memcpy(&iph, p.pkt_data + offs, sizeof(iph));
    offs += iph.ip_hl * 4; /* in case we have IP options */

    /* get UDP header */
    if (iph.ip_p != IPPROTO_UDP ||
	p.included_len < offs + sizeof(uh)) continue;
    memcpy(&uh, p.pkt_data + offs, sizeof(uh));
    offs += sizeof(uh);

    /* find RADIUS header */
#ifdef HAVE_STRUCT_UDPHDR_UH_SPORT
    udp_srcport = ntohs(uh.uh_sport);
    udp_dstport = ntohs(uh.uh_dport);
#else
#ifdef HAVE_STRUCT_UDPHDR_SOURCE
    udp_srcport = ntohs(uh.source);
    udp_dstport = ntohs(uh.dest);
#else
#error "Don't know how to interpret the struct udphdr on your system"
#endif
#endif
    if (udp_dstport != 1645 && udp_srcport != 1645 &&
	udp_dstport != 1812 && udp_srcport != 1812 &&
	udp_dstport != 1646 && udp_srcport != 1646 &&
	udp_dstport != 1813 && udp_srcport != 1813) continue;
    if (p.included_len < offs + sizeof(rh)) continue;
    memcpy(&rh, p.pkt_data + offs, sizeof(rh));
    rh_offs = offs;
    offs += sizeof(rh);
    
    tstamp.tv_sec = p.ts_secs;
    tstamp.tv_usec = p.ts_usecs;
    localtime_r((time_t *)(&tstamp.tv_sec), &localt);
    
    /* by default, print IP addresses */
    sprintf(srchost,"%s",inet_ntoa(iph.ip_src));
    sprintf(dsthost,"%s",inet_ntoa(iph.ip_dst));
    if (lookup_hostnames) {
      struct hostent *he = NULL;
      if ((he = gethostbyaddr((char *)&(iph.ip_src), 
			      sizeof(struct in_addr),
			      AF_INET)) != NULL) {
	strncpy(srchost,he->h_name,sizeof(srchost));
	if (use_short_hostnames) {
	  char *cp = srchost;
	  while (*cp != '.' && *cp != '\0') cp++;
	  if (*cp == '.') *cp = '\0';
	}
      }
      if ((he = gethostbyaddr((char *)&(iph.ip_dst), 
			      sizeof(struct in_addr),
			      AF_INET)) != NULL) {
	strncpy(dsthost,he->h_name,sizeof(dsthost));
	if (use_short_hostnames) {
	  char *cp = dsthost;
	  while (*cp != '.' && *cp != '\0') cp++;
	  if (*cp == '.') *cp = '\0';
	}
      }
    }

    if (tagged_vlan_found) {
      sprintf(vlan_id_string,"vlan %d ",VLAN_ID(&vh));
    }

    printf("%02d:%02d:%02d.%06d radius %s%s:%d > %s:%d %s %u\n",
	   localt.tm_hour, localt.tm_min, localt.tm_sec,
	   tstamp.tv_usec,
	   vlan_id_string,
	   srchost, udp_srcport,
	   dsthost, udp_dstport,
	   rad_code_str(rh.code),
	   ntohs(rh.len) - (unsigned short)sizeof(rh));

    if (verbose >= 1) {
      while(offs < p.included_len && offs < rh_offs + ntohs(rh.len)) {
	if (p.included_len < offs + 2) break;
	ra = (struct radius_attr *)(p.pkt_data + offs);
	offs += ra->len;
	
	printf("  %s = ",rad_attr_name(ra));
	rad_print_attr_val(ra);
	printf("\n");
      }
    }
  }
  prec_close(ph);
  exit(EXIT_SUCCESS);
}

static void usage(char *progname) {
  (void)printf("usage: %s [-?vsn]\n",progname);
  (void)printf("  -?: this help\n");
  (void)printf("  -v: more verbose (multiple -v allowed)\n");
  (void)printf("  -n: do not do reverse DNS lookups\n");
  (void)printf("  -s: use short hostnames\n");
}

static void parse_cmdline(int argc, char *argv[]) {
  char *optstring = "?vsn";
  int c;
  int args_processed = 0;
  int args_expected = 0;
  
  while ((c = getopt(argc,argv,optstring)) != EOF) {
    switch (c) {
    case '?':
      usage(argv[0]);
      exit(EXIT_SUCCESS);
      break;
    case 'v':
      verbose++;
      break;
    case 'n':
      lookup_hostnames = 0;
      break;
    case 's':
      use_short_hostnames = 1;
      break;
    default:
      (void)fprintf(stderr,"%s: unknown option '-%c'\n",argv[0],c);
      usage(argv[0]);
      exit(EXIT_FAILURE);
    }
  }
  while(args_processed < args_expected) {
    if (optind >= argc) {
      fprintf(stderr,"%s: missing args\n",argv[0]);
      usage(argv[0]);
      exit(EXIT_FAILURE);
    }
    switch(args_processed) {
    default:
      break;
    }
    optind++;
    args_processed++;
  }
  if (argc > optind) {
    fprintf(stderr,"%s: too many arguments\n",argv[0]);
    usage(argv[0]);
    exit(EXIT_FAILURE);
  }
}
