;;; vw-mode.el --- Major mode for editing VW-grammars -*- lexical-binding: t; -*-

;; Copyright (C) 2025  Jose E. Marchesi

;; Author: Jose E. Marchesi
;; Maintainer: Jose E. Marchesi <jemarch@gnu.org>
;; URL: https://git.sr.ht/~jemarch/vw-mode
;; Keywords: languages
;; Version: 1.0
;; Package-Requires: ((emacs "24.3"))

;; This file is NOT part of GNU Emacs.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;;; Commentary:

;; Major mode for editing programming language descriptions in the
;; style used by the Algol 68 Revised Report.
;;
;; The syntax of the language is expressed formally using a
;; van-Wijngaarden grammar, also known as VW-grammars or two-level
;; grammars.  These grammars have the same expression power than a
;; Turing machine, and can therefore cover much more than the usual
;; BNF-style representations which are restricted to what can be
;; expressed by context-free grammars.
;;
;; Pragmatic annotations and cross-references complement the grammar
;; rules, which are organized in numbered sections.

;;; Code:

(require 'font-lock)
(require 'smie)
(require 'syntax)

(eval-when-compile
  (require 'rx))

(defgroup vw nil
  "Major mode for editing VW grammars."
  :prefix "vw-"
  :group 'languages)

(defcustom vw-indent-level 3
  "Indentation step for VW grammars."
  :type 'integer
  :safe #'integerp)

(defvar vw-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map (kbd "C-j") #'newline-and-indent)
    (define-key map (kbd "C-c p") #'vw-hide-pragmatics-mode)
    map)
  "Keymap for VW major mode.")

(defface vw-section-face '((t :inherit org-level-1))
  "Face for section numbers in VW mode.")
(defvar vw-section-face 'vw-section-face)

(defface vw-level-1-face '((t :inherit org-level-1))
  "Face for level 1 headers in VW mode.")
(defvar vw-level-1-face 'vw-level-1-face)

(defface vw-level-2-face '((t :inherit vw-level-1-face))
  "Face for level 2 headers in VW mode.")
(defvar vw-level-2-face 'vw-level-2-face)

(defface vw-level-3-face '((t :inherit vw-level-2-face))
  "Face for level 3 headers in VW mode.")
(defvar vw-level-3-face 'vw-level-3-face)

(defface vw-level-4-face '((t :inherit vw-level-3-face))
  "Face for level 4 headers in VW mode.")
(defvar vw-level-4-face 'vw-level-4-face)

(defface vw-xref-face '((t :inherit font-lock-comment-face))
  "Face for cross references in VW rules.")
(defvar vw-xref-face 'vw-xref-face)

(defface vw-metanotion-face '((t :inherit font-lock-keyword-face))
  "Face for metanotions in VW mode.")
(defvar vw-metanotion-face 'vw-metanotion-face)

(defface vw-symbol-face '((t :inherit font-lock-string-face))
  "Face for symbols in VW mode.")
(defvar vw-symbol-face 'vw-symbol-face)

(defface vw-protonotion-face '((t :inherit font-lock-variable-name-face))
  "Face for protonotions in VW mode.")
(defvar vw-protonotion-face 'vw-protonotion-face)

(defface vw-metanotion-number-face '((t :inherit font-lock-type-face))
  "Face for metanotion numbers in VW mode.")
(defvar vw-metanotion-number-face 'vw-metanotion-number-face)

(defface vw-asterisk-face '((t :inherit font-lock-warning-face))
  "Face for asterisk in lhs of rule in VW mode.")
(defvar vw-asterisk-face 'vw-asterisk-face)

(defconst vw-font-lock-keywords
  (list
   (cons "\\*" 'vw-asterisk-face)
   (cons "^[0-9][0-9\\.]*" 'vw-section-face)
   (cons "^[0-9]+ \\(.*\\)$" '(1 'vw-level-1-face))
   (cons "^[0-9]+\\.[0-9]+ \\(.*\\)$" '(1 'vw-level-2-face))
   (cons "^[0-9]+\\.[0-9]+\\.[0-9]+ \\(.*\\)$" '(1 'vw-level-3-face))
   (cons "^[0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9] \\(.*\\)$" '(1 'vw-level-4-face))
   (cons "{[0-9a-zA-Z.]+}" 'vw-xref-face)
   (cons "^[a-zA-Z])" 'font-lock-doc-face)
   (cons "[A-Z]+\\([0-9]+\\)" '(1 'vw-metanotion-number-face))
   (cons "[A-Z]+" 'vw-metanotion-face)
   (cons "\\(\\w[ \t]*\\)+[ \t]*\\<symbol\\>" 'vw-symbol-face)
   (cons "\\(\\w*\\)" 'vw-protonotion-face))
  "Highlighting expressions for VW mode.")

(defvar vw--smie-grammar
  (smie-prec2->grammar
   (smie-bnf->prec2 '(;; Hyper-rules.
                      (hyperrule ("-)" hypernotion ":" hyperalternatives "."))
                      (hyperalternatives (hyperalternatives ";" hyperalternatives)
                                         (hyperalternative))
                      (hyperalternative (hyperalternative "," hypealternative)
                                        (hypernotion))
                      (alternative (hypernotion "," hypernotion)
                                   (hypernotion))
                      (hypernotion)
                      ;; Metaproduction rules.
                      (metaproductionrule ("-)" metanotion ":-meta" hypernotions "."))
                      (hypernotions (hypernotions ";" hypernotions)
                                    (hypernotion))
                      (metanotion))
                    '((assoc ",")
                      (assoc ";")))))

(defun vw--smie-rules (kind token)
  (pcase (cons kind token)
    (`(:elem . basic) vw-indent-level)
    (`(:after . ":")
     (when (smie-rule-hanging-p)
       (smie-rule-parent 5)))
    (`(:after . ":-meta")
     (when (smie-rule-hanging-p)
       (smie-rule-parent 5)))
    (`(:after . ",")
     (when (smie-rule-hanging-p)
       2))
    (`(,_ . ",") (smie-rule-separator kind))
    (`(,_ . ".") (when (smie-rule-parent-p)
                   (smie-rule-parent)))))

(defun vw--smie-forward-token ()
  (forward-comment (point-max))
  (cond
   ((looking-at "^[0-9]\\(.[0-9]\\)*")
    (goto-char (match-end 0))
    "-section")
   ((looking-at "^[a-zA-Z])")
    (goto-char (+ (point) 2))
    "-)")
   ((looking-at ",")
    (goto-char (+ (point) 1))
    ",")
   ((looking-at "\\.")
    (goto-char (+ (point) 1))
    ".")
   ((looking-at "\\*")
    (goto-char (+ (point) 1))
    "*")
   ((looking-at ";")
    (goto-char (+ (point) 1))
    ";")
   ((looking-at "::")
    (goto-char (match-end 0))
    ":-meta")
   ((looking-at ":")
    (goto-char (+ (point) 1))
    ":")
   (t (buffer-substring-no-properties (point)
                                      (progn (skip-syntax-forward "w_")
                                             (point))))))

(defun vw--smie-backward-token ()
  (forward-comment (- (point)))
  (cond
   ((looking-back "^[0-9]\\(.[0-9]\\)*" (point-min))
    (goto-char (match-beginning 0))
    "-section")
   ((looking-back "^[a-zA-Z])" (point-min))
    (goto-char (- (point) 2))
    "-)")
   ((eq (char-before) ?,)
    (goto-char (- (point) 1))
    ",")
   ((eq (char-before) ?.)
    (goto-char (- (point) 1))
    ".")
   ((eq (char-before) ?*)
    (goto-char (- (point) 1))
    "*")
   ((eq (char-before) ?\;)
    (goto-char (- (point) 1))
    ";")
   ((looking-back "::" (- (point) 2))
    (goto-char (match-beginning 0))
    ":-meta")
   ((eq (char-before) ?:)
    (goto-char (- (point) 1))
    ":")
   (t (buffer-substring-no-properties (point)
                                      (progn (skip-syntax-backward "w_")
                                             (point))))))

(defun vw-within-comment ()
  (nth 4 (syntax-ppss)))

(defvar vw-mode-syntax-table
  (let ((st (make-syntax-table)))
    ;; symbol
    (modify-syntax-entry ?\( "_" st)
    (modify-syntax-entry ?\) "_" st)
    (modify-syntax-entry ?{ "<" st)
    (modify-syntax-entry ?} ">" st)
    ;; Note { and } are nestable.
    (modify-syntax-entry ?{ "< n" st)
    (modify-syntax-entry ?} "> n" st)
    st))

;;;###autoload
(define-derived-mode vw-mode prog-mode "VW"
  "Major mode for editing VW grammars."
  (setq-local font-lock-defaults '(vw-font-lock-keywords))
  (smie-setup vw--smie-grammar #'vw--smie-rules
              :forward-token #'vw--smie-forward-token
              :backward-token #'vw--smie-backward-token)
  (setq-local comment-start "{")
  (setq-local comment-end "}"))

(defun vw--set-comment-invisible-attr (attr)
  (let ((start (point-min))
        (end (point-max))
        (bufmodp (buffer-modified-p))
        (buffer-read-only nil)
        cbeg cend)
    (unwind-protect
        (save-excursion
          (goto-char start)
          (while (and (< start end) (setq cbeg (comment-search-forward end 'NOERROR)))
            (setq cend (if (string= "" comment-end)
                           (min (1+ (line-end-position)) (point-max))
                         (search-forward comment-end end 'NOERROR)))
            (when (and cbeg cend)
              (put-text-property cbeg cend 'invisible attr)))))
    (set-buffer-modified-p bufmodp)))

;;;###autoload
(define-minor-mode vw-hide-pragmatics-mode
  "Toggle hidding pragmatics in VW mode."
  :group vw
  (if vw-hide-pragmatics-mode
      (vw--set-comment-invisible-attr t)
    (vw--set-comment-invisible-attr nil)))

;;;###autoload
(add-to-list 'auto-mode-alist '("\\.vw\\'" . vw-mode))

(provide 'vw-mode)

;;; vw-mode.el ends here
