//! x^n where n is an integer.
//!
//! Method described at: <https://stackoverflow.com/a/101613>
//!
//! Note: not optimal for all powers, is NP complete
//! <https://en.wikipedia.org/wiki/Addition-chain_exponentiation>
//! will result in an addition multiplication for x^15 for example

use super::{F32, MANTISSA_BITS};

impl F32 {
    /// Approximates a number raised to an integer power.
    pub fn powi(self, n: i32) -> Self {
        let mut base = self;
        let mut abs_n = i32::abs(n);
        let mut result = 1.0;

        if n < 0 {
            base = 1.0 / self;
        }

        if n == 0 {
            return Self::ONE;
        }

        // 0.0 == 0.0 and -0.0 according to IEEE standards.
        if self == Self::ZERO && n > 0 {
            return self;
        }

        // For values less than 2.0, but greater than 0.5 (1.0/2.0), you can multiply longer without
        // going over exponent, i.e. 1.1 multiplied against itself will grow slowly.
        if !(0.5..2.0).contains(&self.abs().0) {
            // Approximation if we end up outside of the range of floating point values,
            // then we end early
            let approx_final_exponent = self.extract_exponent_value() * n;
            let max_representable_exponent = 127;
            let min_representable_exponent = -126 - (MANTISSA_BITS as i32);
            if approx_final_exponent > max_representable_exponent
                || (self == Self::ZERO && approx_final_exponent < 0)
            {
                if self.is_sign_positive() || n & 1 == 0 {
                    return Self::INFINITY;
                } else {
                    return Self::NEG_INFINITY;
                }
            } else if approx_final_exponent < min_representable_exponent {
                // We may want to copy the sign and do the same thing as above,
                // but that seems like an awful amount of work when 99.99999% of people only care
                // about bare zero
                return Self::ZERO;
            }
        }

        loop {
            if (abs_n & 1) == 1 {
                result *= base;
            }

            abs_n >>= 1;

            if abs_n == 0 {
                return Self(result);
            }

            base *= base;
        }
    }
}

#[cfg(test)]
mod tests {
    use super::F32;

    /// Error builds up from both exp and ln approximation, so we double the error allowed.
    pub(crate) const MAX_ERROR: f32 = 0.002;

    /// `powi(3,x)` test vectors - `(input_base, input_power, output)`
    pub(crate) const TEST_VECTORS: &[(f32, f32, f32)] = &[
        (-100000.0, -128.0, 0.0),
        (-10.0, -128.0, 0.0),
        (-3.0, -128.0, 0.0),
        (-2.0, -128.0, 2.938736e-39),
        (-1.0, -128.0, 1.0),
        (1.0, -128.0, 1.0),
        (2.0, -128.0, 2.938736e-39),
        (3.0, -128.0, 0.0),
        (10.0, -128.0, 0.0),
        (100000.0, -128.0, 0.0),
        (-100000.0, -127.0, -0.0),
        (-10.0, -127.0, -0.0),
        (-3.0, -127.0, -0.0),
        (-2.0, -127.0, -5.877472e-39),
        (-1.0, -127.0, -1.0),
        (1.0, -127.0, 1.0),
        (2.0, -127.0, 5.877472e-39),
        (3.0, -127.0, 0.0),
        (10.0, -127.0, 0.0),
        (100000.0, -127.0, 0.0),
        (-100000.0, -126.0, 0.0),
        (-10.0, -126.0, 0.0),
        (-3.0, -126.0, 0.0),
        (-2.0, -126.0, 1.1754944e-38),
        (-1.0, -126.0, 1.0),
        (1.0, -126.0, 1.0),
        (2.0, -126.0, 1.1754944e-38),
        (3.0, -126.0, 0.0),
        (10.0, -126.0, 0.0),
        (100000.0, -126.0, 0.0),
        (-100000.0, -125.0, -0.0),
        (-10.0, -125.0, -0.0),
        (-3.0, -125.0, -0.0),
        (-2.0, -125.0, -2.3509887e-38),
        (-1.0, -125.0, -1.0),
        (1.0, -125.0, 1.0),
        (2.0, -125.0, 2.3509887e-38),
        (3.0, -125.0, 0.0),
        (10.0, -125.0, 0.0),
        (100000.0, -125.0, 0.0),
        (-100000.0, -124.0, 0.0),
        (-10.0, -124.0, 0.0),
        (-3.0, -124.0, 0.0),
        (-2.0, -124.0, 4.7019774e-38),
        (-1.0, -124.0, 1.0),
        (1.0, -124.0, 1.0),
        (2.0, -124.0, 4.7019774e-38),
        (3.0, -124.0, 0.0),
        (10.0, -124.0, 0.0),
        (100000.0, -124.0, 0.0),
        (-100000.0, -123.0, -0.0),
        (-10.0, -123.0, -0.0),
        (-3.0, -123.0, -0.0),
        (-2.0, -123.0, -9.403955e-38),
        (-1.0, -123.0, -1.0),
        (1.0, -123.0, 1.0),
        (2.0, -123.0, 9.403955e-38),
        (3.0, -123.0, 0.0),
        (10.0, -123.0, 0.0),
        (100000.0, -123.0, 0.0),
        (-100000.0, -122.0, 0.0),
        (-10.0, -122.0, 0.0),
        (-3.0, -122.0, 0.0),
        (-2.0, -122.0, 1.880791e-37),
        (-1.0, -122.0, 1.0),
        (1.0, -122.0, 1.0),
        (2.0, -122.0, 1.880791e-37),
        (3.0, -122.0, 0.0),
        (10.0, -122.0, 0.0),
        (100000.0, -122.0, 0.0),
        (-100000.0, -121.0, -0.0),
        (-10.0, -121.0, -0.0),
        (-3.0, -121.0, -0.0),
        (-2.0, -121.0, -3.761582e-37),
        (-1.0, -121.0, -1.0),
        (1.0, -121.0, 1.0),
        (2.0, -121.0, 3.761582e-37),
        (3.0, -121.0, 0.0),
        (10.0, -121.0, 0.0),
        (100000.0, -121.0, 0.0),
        (-100000.0, -120.0, 0.0),
        (-10.0, -120.0, 0.0),
        (-3.0, -120.0, 0.0),
        (-2.0, -120.0, 7.523164e-37),
        (-1.0, -120.0, 1.0),
        (1.0, -120.0, 1.0),
        (2.0, -120.0, 7.523164e-37),
        (3.0, -120.0, 0.0),
        (10.0, -120.0, 0.0),
        (100000.0, -120.0, 0.0),
        (-100000.0, -119.0, -0.0),
        (-10.0, -119.0, -0.0),
        (-3.0, -119.0, -0.0),
        (-2.0, -119.0, -1.5046328e-36),
        (-1.0, -119.0, -1.0),
        (1.0, -119.0, 1.0),
        (2.0, -119.0, 1.5046328e-36),
        (3.0, -119.0, 0.0),
        (10.0, -119.0, 0.0),
        (100000.0, -119.0, 0.0),
        (-100000.0, -118.0, 0.0),
        (-10.0, -118.0, 0.0),
        (-3.0, -118.0, 0.0),
        (-2.0, -118.0, 3.0092655e-36),
        (-1.0, -118.0, 1.0),
        (1.0, -118.0, 1.0),
        (2.0, -118.0, 3.0092655e-36),
        (3.0, -118.0, 0.0),
        (10.0, -118.0, 0.0),
        (100000.0, -118.0, 0.0),
        (-100000.0, -117.0, -0.0),
        (-10.0, -117.0, -0.0),
        (-3.0, -117.0, -0.0),
        (-2.0, -117.0, -6.018531e-36),
        (-1.0, -117.0, -1.0),
        (1.0, -117.0, 1.0),
        (2.0, -117.0, 6.018531e-36),
        (3.0, -117.0, 0.0),
        (10.0, -117.0, 0.0),
        (100000.0, -117.0, 0.0),
        (-100000.0, -116.0, 0.0),
        (-10.0, -116.0, 0.0),
        (-3.0, -116.0, 0.0),
        (-2.0, -116.0, 1.2037062e-35),
        (-1.0, -116.0, 1.0),
        (1.0, -116.0, 1.0),
        (2.0, -116.0, 1.2037062e-35),
        (3.0, -116.0, 0.0),
        (10.0, -116.0, 0.0),
        (100000.0, -116.0, 0.0),
        (-100000.0, -115.0, -0.0),
        (-10.0, -115.0, -0.0),
        (-3.0, -115.0, -0.0),
        (-2.0, -115.0, -2.4074124e-35),
        (-1.0, -115.0, -1.0),
        (1.0, -115.0, 1.0),
        (2.0, -115.0, 2.4074124e-35),
        (3.0, -115.0, 0.0),
        (10.0, -115.0, 0.0),
        (100000.0, -115.0, 0.0),
        (-100000.0, -114.0, 0.0),
        (-10.0, -114.0, 0.0),
        (-3.0, -114.0, 0.0),
        (-2.0, -114.0, 4.814825e-35),
        (-1.0, -114.0, 1.0),
        (1.0, -114.0, 1.0),
        (2.0, -114.0, 4.814825e-35),
        (3.0, -114.0, 0.0),
        (10.0, -114.0, 0.0),
        (100000.0, -114.0, 0.0),
        (-100000.0, -113.0, -0.0),
        (-10.0, -113.0, -0.0),
        (-3.0, -113.0, -0.0),
        (-2.0, -113.0, -9.62965e-35),
        (-1.0, -113.0, -1.0),
        (1.0, -113.0, 1.0),
        (2.0, -113.0, 9.62965e-35),
        (3.0, -113.0, 0.0),
        (10.0, -113.0, 0.0),
        (100000.0, -113.0, 0.0),
        (-100000.0, -112.0, 0.0),
        (-10.0, -112.0, 0.0),
        (-3.0, -112.0, 0.0),
        (-2.0, -112.0, 1.92593e-34),
        (-1.0, -112.0, 1.0),
        (1.0, -112.0, 1.0),
        (2.0, -112.0, 1.92593e-34),
        (3.0, -112.0, 0.0),
        (10.0, -112.0, 0.0),
        (100000.0, -112.0, 0.0),
        (-100000.0, -111.0, -0.0),
        (-10.0, -111.0, -0.0),
        (-3.0, -111.0, -0.0),
        (-2.0, -111.0, -3.85186e-34),
        (-1.0, -111.0, -1.0),
        (1.0, -111.0, 1.0),
        (2.0, -111.0, 3.85186e-34),
        (3.0, -111.0, 0.0),
        (10.0, -111.0, 0.0),
        (100000.0, -111.0, 0.0),
        (-100000.0, -110.0, 0.0),
        (-10.0, -110.0, 0.0),
        (-3.0, -110.0, 0.0),
        (-2.0, -110.0, 7.70372e-34),
        (-1.0, -110.0, 1.0),
        (1.0, -110.0, 1.0),
        (2.0, -110.0, 7.70372e-34),
        (3.0, -110.0, 0.0),
        (10.0, -110.0, 0.0),
        (100000.0, -110.0, 0.0),
        (-100000.0, -109.0, -0.0),
        (-10.0, -109.0, -0.0),
        (-3.0, -109.0, -0.0),
        (-2.0, -109.0, -1.540744e-33),
        (-1.0, -109.0, -1.0),
        (1.0, -109.0, 1.0),
        (2.0, -109.0, 1.540744e-33),
        (3.0, -109.0, 0.0),
        (10.0, -109.0, 0.0),
        (100000.0, -109.0, 0.0),
        (-100000.0, -108.0, 0.0),
        (-10.0, -108.0, 0.0),
        (-3.0, -108.0, 0.0),
        (-2.0, -108.0, 3.081488e-33),
        (-1.0, -108.0, 1.0),
        (1.0, -108.0, 1.0),
        (2.0, -108.0, 3.081488e-33),
        (3.0, -108.0, 0.0),
        (10.0, -108.0, 0.0),
        (100000.0, -108.0, 0.0),
        (-100000.0, -107.0, -0.0),
        (-10.0, -107.0, -0.0),
        (-3.0, -107.0, -0.0),
        (-2.0, -107.0, -6.162976e-33),
        (-1.0, -107.0, -1.0),
        (1.0, -107.0, 1.0),
        (2.0, -107.0, 6.162976e-33),
        (3.0, -107.0, 0.0),
        (10.0, -107.0, 0.0),
        (100000.0, -107.0, 0.0),
        (-100000.0, -106.0, 0.0),
        (-10.0, -106.0, 0.0),
        (-3.0, -106.0, 0.0),
        (-2.0, -106.0, 1.2325952e-32),
        (-1.0, -106.0, 1.0),
        (1.0, -106.0, 1.0),
        (2.0, -106.0, 1.2325952e-32),
        (3.0, -106.0, 0.0),
        (10.0, -106.0, 0.0),
        (100000.0, -106.0, 0.0),
        (-100000.0, -105.0, -0.0),
        (-10.0, -105.0, -0.0),
        (-3.0, -105.0, -0.0),
        (-2.0, -105.0, -2.4651903e-32),
        (-1.0, -105.0, -1.0),
        (1.0, -105.0, 1.0),
        (2.0, -105.0, 2.4651903e-32),
        (3.0, -105.0, 0.0),
        (10.0, -105.0, 0.0),
        (100000.0, -105.0, 0.0),
        (-100000.0, -104.0, 0.0),
        (-10.0, -104.0, 0.0),
        (-3.0, -104.0, 0.0),
        (-2.0, -104.0, 4.9303807e-32),
        (-1.0, -104.0, 1.0),
        (1.0, -104.0, 1.0),
        (2.0, -104.0, 4.9303807e-32),
        (3.0, -104.0, 0.0),
        (10.0, -104.0, 0.0),
        (100000.0, -104.0, 0.0),
        (-100000.0, -103.0, -0.0),
        (-10.0, -103.0, -0.0),
        (-3.0, -103.0, -0.0),
        (-2.0, -103.0, -9.8607613e-32),
        (-1.0, -103.0, -1.0),
        (1.0, -103.0, 1.0),
        (2.0, -103.0, 9.8607613e-32),
        (3.0, -103.0, 0.0),
        (10.0, -103.0, 0.0),
        (100000.0, -103.0, 0.0),
        (-100000.0, -102.0, 0.0),
        (-10.0, -102.0, 0.0),
        (-3.0, -102.0, 0.0),
        (-2.0, -102.0, 1.9721523e-31),
        (-1.0, -102.0, 1.0),
        (1.0, -102.0, 1.0),
        (2.0, -102.0, 1.9721523e-31),
        (3.0, -102.0, 0.0),
        (10.0, -102.0, 0.0),
        (100000.0, -102.0, 0.0),
        (-100000.0, -101.0, -0.0),
        (-10.0, -101.0, -0.0),
        (-3.0, -101.0, -0.0),
        (-2.0, -101.0, -3.9443045e-31),
        (-1.0, -101.0, -1.0),
        (1.0, -101.0, 1.0),
        (2.0, -101.0, 3.9443045e-31),
        (3.0, -101.0, 0.0),
        (10.0, -101.0, 0.0),
        (100000.0, -101.0, 0.0),
        (-100000.0, -100.0, 0.0),
        (-10.0, -100.0, 0.0),
        (-3.0, -100.0, 0.0),
        (-2.0, -100.0, 7.888609e-31),
        (-1.0, -100.0, 1.0),
        (1.0, -100.0, 1.0),
        (2.0, -100.0, 7.888609e-31),
        (3.0, -100.0, 0.0),
        (10.0, -100.0, 0.0),
        (100000.0, -100.0, 0.0),
        (-100000.0, -99.0, -0.0),
        (-10.0, -99.0, -0.0),
        (-3.0, -99.0, -0.0),
        (-2.0, -99.0, -1.5777218e-30),
        (-1.0, -99.0, -1.0),
        (1.0, -99.0, 1.0),
        (2.0, -99.0, 1.5777218e-30),
        (3.0, -99.0, 0.0),
        (10.0, -99.0, 0.0),
        (100000.0, -99.0, 0.0),
        (-100000.0, -98.0, 0.0),
        (-10.0, -98.0, 0.0),
        (-3.0, -98.0, 0.0),
        (-2.0, -98.0, 3.1554436e-30),
        (-1.0, -98.0, 1.0),
        (1.0, -98.0, 1.0),
        (2.0, -98.0, 3.1554436e-30),
        (3.0, -98.0, 0.0),
        (10.0, -98.0, 0.0),
        (100000.0, -98.0, 0.0),
        (-100000.0, -97.0, -0.0),
        (-10.0, -97.0, -0.0),
        (-3.0, -97.0, -0.0),
        (-2.0, -97.0, -6.3108872e-30),
        (-1.0, -97.0, -1.0),
        (1.0, -97.0, 1.0),
        (2.0, -97.0, 6.3108872e-30),
        (3.0, -97.0, 0.0),
        (10.0, -97.0, 0.0),
        (100000.0, -97.0, 0.0),
        (-100000.0, -96.0, 0.0),
        (-10.0, -96.0, 0.0),
        (-3.0, -96.0, 0.0),
        (-2.0, -96.0, 1.2621775e-29),
        (-1.0, -96.0, 1.0),
        (1.0, -96.0, 1.0),
        (2.0, -96.0, 1.2621775e-29),
        (3.0, -96.0, 0.0),
        (10.0, -96.0, 0.0),
        (100000.0, -96.0, 0.0),
        (-100000.0, -95.0, -0.0),
        (-10.0, -95.0, -0.0),
        (-3.0, -95.0, -0.0),
        (-2.0, -95.0, -2.524355e-29),
        (-1.0, -95.0, -1.0),
        (1.0, -95.0, 1.0),
        (2.0, -95.0, 2.524355e-29),
        (3.0, -95.0, 0.0),
        (10.0, -95.0, 0.0),
        (100000.0, -95.0, 0.0),
        (-100000.0, -94.0, 0.0),
        (-10.0, -94.0, 0.0),
        (-3.0, -94.0, 1e-45),
        (-2.0, -94.0, 5.04871e-29),
        (-1.0, -94.0, 1.0),
        (1.0, -94.0, 1.0),
        (2.0, -94.0, 5.04871e-29),
        (3.0, -94.0, 1e-45),
        (10.0, -94.0, 0.0),
        (100000.0, -94.0, 0.0),
        (-100000.0, -93.0, -0.0),
        (-10.0, -93.0, -0.0),
        (-3.0, -93.0, -4e-45),
        (-2.0, -93.0, -1.009742e-28),
        (-1.0, -93.0, -1.0),
        (1.0, -93.0, 1.0),
        (2.0, -93.0, 1.009742e-28),
        (3.0, -93.0, 4e-45),
        (10.0, -93.0, 0.0),
        (100000.0, -93.0, 0.0),
        (-100000.0, -92.0, 0.0),
        (-10.0, -92.0, 0.0),
        (-3.0, -92.0, 1.3e-44),
        (-2.0, -92.0, 2.019484e-28),
        (-1.0, -92.0, 1.0),
        (1.0, -92.0, 1.0),
        (2.0, -92.0, 2.019484e-28),
        (3.0, -92.0, 1.3e-44),
        (10.0, -92.0, 0.0),
        (100000.0, -92.0, 0.0),
        (-100000.0, -91.0, -0.0),
        (-10.0, -91.0, -0.0),
        (-3.0, -91.0, -3.8e-44),
        (-2.0, -91.0, -4.038968e-28),
        (-1.0, -91.0, -1.0),
        (1.0, -91.0, 1.0),
        (2.0, -91.0, 4.038968e-28),
        (3.0, -91.0, 3.8e-44),
        (10.0, -91.0, 0.0),
        (100000.0, -91.0, 0.0),
        (-100000.0, -90.0, 0.0),
        (-10.0, -90.0, 0.0),
        (-3.0, -90.0, 1.15e-43),
        (-2.0, -90.0, 8.077936e-28),
        (-1.0, -90.0, 1.0),
        (1.0, -90.0, 1.0),
        (2.0, -90.0, 8.077936e-28),
        (3.0, -90.0, 1.15e-43),
        (10.0, -90.0, 0.0),
        (100000.0, -90.0, 0.0),
        (-100000.0, -89.0, -0.0),
        (-10.0, -89.0, -0.0),
        (-3.0, -89.0, -3.43e-43),
        (-2.0, -89.0, -1.6155871e-27),
        (-1.0, -89.0, -1.0),
        (1.0, -89.0, 1.0),
        (2.0, -89.0, 1.6155871e-27),
        (3.0, -89.0, 3.43e-43),
        (10.0, -89.0, 0.0),
        (100000.0, -89.0, 0.0),
        (-100000.0, -88.0, 0.0),
        (-10.0, -88.0, 0.0),
        (-3.0, -88.0, 1.031e-42),
        (-2.0, -88.0, 3.2311743e-27),
        (-1.0, -88.0, 1.0),
        (1.0, -88.0, 1.0),
        (2.0, -88.0, 3.2311743e-27),
        (3.0, -88.0, 1.031e-42),
        (10.0, -88.0, 0.0),
        (100000.0, -88.0, 0.0),
        (-100000.0, -87.0, -0.0),
        (-10.0, -87.0, -0.0),
        (-3.0, -87.0, -3.094e-42),
        (-2.0, -87.0, -6.4623485e-27),
        (-1.0, -87.0, -1.0),
        (1.0, -87.0, 1.0),
        (2.0, -87.0, 6.4623485e-27),
        (3.0, -87.0, 3.094e-42),
        (10.0, -87.0, 0.0),
        (100000.0, -87.0, 0.0),
        (-100000.0, -86.0, 0.0),
        (-10.0, -86.0, 0.0),
        (-3.0, -86.0, 9.281e-42),
        (-2.0, -86.0, 1.2924697e-26),
        (-1.0, -86.0, 1.0),
        (1.0, -86.0, 1.0),
        (2.0, -86.0, 1.2924697e-26),
        (3.0, -86.0, 9.281e-42),
        (10.0, -86.0, 0.0),
        (100000.0, -86.0, 0.0),
        (-100000.0, -85.0, -0.0),
        (-10.0, -85.0, -0.0),
        (-3.0, -85.0, -2.7841e-41),
        (-2.0, -85.0, -2.5849394e-26),
        (-1.0, -85.0, -1.0),
        (1.0, -85.0, 1.0),
        (2.0, -85.0, 2.5849394e-26),
        (3.0, -85.0, 2.7841e-41),
        (10.0, -85.0, 0.0),
        (100000.0, -85.0, 0.0),
        (-100000.0, -84.0, 0.0),
        (-10.0, -84.0, 0.0),
        (-3.0, -84.0, 8.3524e-41),
        (-2.0, -84.0, 5.169879e-26),
        (-1.0, -84.0, 1.0),
        (1.0, -84.0, 1.0),
        (2.0, -84.0, 5.169879e-26),
        (3.0, -84.0, 8.3524e-41),
        (10.0, -84.0, 0.0),
        (100000.0, -84.0, 0.0),
        (-100000.0, -83.0, -0.0),
        (-10.0, -83.0, -0.0),
        (-3.0, -83.0, -2.50575e-40),
        (-2.0, -83.0, -1.0339758e-25),
        (-1.0, -83.0, -1.0),
        (1.0, -83.0, 1.0),
        (2.0, -83.0, 1.0339758e-25),
        (3.0, -83.0, 2.50575e-40),
        (10.0, -83.0, 0.0),
        (100000.0, -83.0, 0.0),
        (-100000.0, -82.0, 0.0),
        (-10.0, -82.0, 0.0),
        (-3.0, -82.0, 7.51722e-40),
        (-2.0, -82.0, 2.0679515e-25),
        (-1.0, -82.0, 1.0),
        (1.0, -82.0, 1.0),
        (2.0, -82.0, 2.0679515e-25),
        (3.0, -82.0, 7.51722e-40),
        (10.0, -82.0, 0.0),
        (100000.0, -82.0, 0.0),
        (-100000.0, -81.0, -0.0),
        (-10.0, -81.0, -0.0),
        (-3.0, -81.0, -2.255166e-39),
        (-2.0, -81.0, -4.135903e-25),
        (-1.0, -81.0, -1.0),
        (1.0, -81.0, 1.0),
        (2.0, -81.0, 4.135903e-25),
        (3.0, -81.0, 2.255166e-39),
        (10.0, -81.0, 0.0),
        (100000.0, -81.0, 0.0),
        (-100000.0, -80.0, 0.0),
        (-10.0, -80.0, 0.0),
        (-3.0, -80.0, 6.765496e-39),
        (-2.0, -80.0, 8.271806e-25),
        (-1.0, -80.0, 1.0),
        (1.0, -80.0, 1.0),
        (2.0, -80.0, 8.271806e-25),
        (3.0, -80.0, 6.765496e-39),
        (10.0, -80.0, 0.0),
        (100000.0, -80.0, 0.0),
        (-100000.0, -79.0, -0.0),
        (-10.0, -79.0, -0.0),
        (-3.0, -79.0, -2.0296487e-38),
        (-2.0, -79.0, -1.6543612e-24),
        (-1.0, -79.0, -1.0),
        (1.0, -79.0, 1.0),
        (2.0, -79.0, 1.6543612e-24),
        (3.0, -79.0, 2.0296487e-38),
        (10.0, -79.0, 0.0),
        (100000.0, -79.0, 0.0),
        (-100000.0, -78.0, 0.0),
        (-10.0, -78.0, 0.0),
        (-3.0, -78.0, 6.088946e-38),
        (-2.0, -78.0, 3.3087225e-24),
        (-1.0, -78.0, 1.0),
        (1.0, -78.0, 1.0),
        (2.0, -78.0, 3.3087225e-24),
        (3.0, -78.0, 6.088946e-38),
        (10.0, -78.0, 0.0),
        (100000.0, -78.0, 0.0),
        (-100000.0, -77.0, -0.0),
        (-10.0, -77.0, -0.0),
        (-3.0, -77.0, -1.8266838e-37),
        (-2.0, -77.0, -6.617445e-24),
        (-1.0, -77.0, -1.0),
        (1.0, -77.0, 1.0),
        (2.0, -77.0, 6.617445e-24),
        (3.0, -77.0, 1.8266838e-37),
        (10.0, -77.0, 0.0),
        (100000.0, -77.0, 0.0),
        (-100000.0, -76.0, 0.0),
        (-10.0, -76.0, 0.0),
        (-3.0, -76.0, 5.4800514e-37),
        (-2.0, -76.0, 1.323489e-23),
        (-1.0, -76.0, 1.0),
        (1.0, -76.0, 1.0),
        (2.0, -76.0, 1.323489e-23),
        (3.0, -76.0, 5.4800514e-37),
        (10.0, -76.0, 0.0),
        (100000.0, -76.0, 0.0),
        (-100000.0, -75.0, -0.0),
        (-10.0, -75.0, -0.0),
        (-3.0, -75.0, -1.6440154e-36),
        (-2.0, -75.0, -2.646978e-23),
        (-1.0, -75.0, -1.0),
        (1.0, -75.0, 1.0),
        (2.0, -75.0, 2.646978e-23),
        (3.0, -75.0, 1.6440154e-36),
        (10.0, -75.0, 0.0),
        (100000.0, -75.0, 0.0),
        (-100000.0, -74.0, 0.0),
        (-10.0, -74.0, 0.0),
        (-3.0, -74.0, 4.9320465e-36),
        (-2.0, -74.0, 5.293956e-23),
        (-1.0, -74.0, 1.0),
        (1.0, -74.0, 1.0),
        (2.0, -74.0, 5.293956e-23),
        (3.0, -74.0, 4.9320465e-36),
        (10.0, -74.0, 0.0),
        (100000.0, -74.0, 0.0),
        (-100000.0, -73.0, -0.0),
        (-10.0, -73.0, -0.0),
        (-3.0, -73.0, -1.4796139e-35),
        (-2.0, -73.0, -1.0587912e-22),
        (-1.0, -73.0, -1.0),
        (1.0, -73.0, 1.0),
        (2.0, -73.0, 1.0587912e-22),
        (3.0, -73.0, 1.4796139e-35),
        (10.0, -73.0, 0.0),
        (100000.0, -73.0, 0.0),
        (-100000.0, -72.0, 0.0),
        (-10.0, -72.0, 0.0),
        (-3.0, -72.0, 4.4388417e-35),
        (-2.0, -72.0, 2.1175824e-22),
        (-1.0, -72.0, 1.0),
        (1.0, -72.0, 1.0),
        (2.0, -72.0, 2.1175824e-22),
        (3.0, -72.0, 4.4388417e-35),
        (10.0, -72.0, 0.0),
        (100000.0, -72.0, 0.0),
        (-100000.0, -71.0, -0.0),
        (-10.0, -71.0, -0.0),
        (-3.0, -71.0, -1.3316525e-34),
        (-2.0, -71.0, -4.2351647e-22),
        (-1.0, -71.0, -1.0),
        (1.0, -71.0, 1.0),
        (2.0, -71.0, 4.2351647e-22),
        (3.0, -71.0, 1.3316525e-34),
        (10.0, -71.0, 0.0),
        (100000.0, -71.0, 0.0),
        (-100000.0, -70.0, 0.0),
        (-10.0, -70.0, 0.0),
        (-3.0, -70.0, 3.9949575e-34),
        (-2.0, -70.0, 8.4703295e-22),
        (-1.0, -70.0, 1.0),
        (1.0, -70.0, 1.0),
        (2.0, -70.0, 8.4703295e-22),
        (3.0, -70.0, 3.9949575e-34),
        (10.0, -70.0, 0.0),
        (100000.0, -70.0, 0.0),
        (-100000.0, -69.0, -0.0),
        (-10.0, -69.0, -0.0),
        (-3.0, -69.0, -1.1984873e-33),
        (-2.0, -69.0, -1.6940659e-21),
        (-1.0, -69.0, -1.0),
        (1.0, -69.0, 1.0),
        (2.0, -69.0, 1.6940659e-21),
        (3.0, -69.0, 1.1984873e-33),
        (10.0, -69.0, 0.0),
        (100000.0, -69.0, 0.0),
        (-100000.0, -68.0, 0.0),
        (-10.0, -68.0, 0.0),
        (-3.0, -68.0, 3.5954618e-33),
        (-2.0, -68.0, 3.3881318e-21),
        (-1.0, -68.0, 1.0),
        (1.0, -68.0, 1.0),
        (2.0, -68.0, 3.3881318e-21),
        (3.0, -68.0, 3.5954618e-33),
        (10.0, -68.0, 0.0),
        (100000.0, -68.0, 0.0),
        (-100000.0, -67.0, -0.0),
        (-10.0, -67.0, -0.0),
        (-3.0, -67.0, -1.07863854e-32),
        (-2.0, -67.0, -6.7762636e-21),
        (-1.0, -67.0, -1.0),
        (1.0, -67.0, 1.0),
        (2.0, -67.0, 6.7762636e-21),
        (3.0, -67.0, 1.07863854e-32),
        (10.0, -67.0, 0.0),
        (100000.0, -67.0, 0.0),
        (-100000.0, -66.0, 0.0),
        (-10.0, -66.0, 0.0),
        (-3.0, -66.0, 3.2359155e-32),
        (-2.0, -66.0, 1.3552527e-20),
        (-1.0, -66.0, 1.0),
        (1.0, -66.0, 1.0),
        (2.0, -66.0, 1.3552527e-20),
        (3.0, -66.0, 3.2359155e-32),
        (10.0, -66.0, 0.0),
        (100000.0, -66.0, 0.0),
        (-100000.0, -65.0, -0.0),
        (-10.0, -65.0, -0.0),
        (-3.0, -65.0, -9.7077466e-32),
        (-2.0, -65.0, -2.7105054e-20),
        (-1.0, -65.0, -1.0),
        (1.0, -65.0, 1.0),
        (2.0, -65.0, 2.7105054e-20),
        (3.0, -65.0, 9.7077466e-32),
        (10.0, -65.0, 0.0),
        (100000.0, -65.0, 0.0),
        (-100000.0, -64.0, 0.0),
        (-10.0, -64.0, 0.0),
        (-3.0, -64.0, 2.9123242e-31),
        (-2.0, -64.0, 5.421011e-20),
        (-1.0, -64.0, 1.0),
        (1.0, -64.0, 1.0),
        (2.0, -64.0, 5.421011e-20),
        (3.0, -64.0, 2.9123242e-31),
        (10.0, -64.0, 0.0),
        (100000.0, -64.0, 0.0),
        (-100000.0, -63.0, -0.0),
        (-10.0, -63.0, -0.0),
        (-3.0, -63.0, -8.736972e-31),
        (-2.0, -63.0, -1.0842022e-19),
        (-1.0, -63.0, -1.0),
        (1.0, -63.0, 1.0),
        (2.0, -63.0, 1.0842022e-19),
        (3.0, -63.0, 8.736972e-31),
        (10.0, -63.0, 0.0),
        (100000.0, -63.0, 0.0),
        (-100000.0, -62.0, 0.0),
        (-10.0, -62.0, 0.0),
        (-3.0, -62.0, 2.6210917e-30),
        (-2.0, -62.0, 2.1684043e-19),
        (-1.0, -62.0, 1.0),
        (1.0, -62.0, 1.0),
        (2.0, -62.0, 2.1684043e-19),
        (3.0, -62.0, 2.6210917e-30),
        (10.0, -62.0, 0.0),
        (100000.0, -62.0, 0.0),
        (-100000.0, -61.0, -0.0),
        (-10.0, -61.0, -0.0),
        (-3.0, -61.0, -7.863275e-30),
        (-2.0, -61.0, -4.3368087e-19),
        (-1.0, -61.0, -1.0),
        (1.0, -61.0, 1.0),
        (2.0, -61.0, 4.3368087e-19),
        (3.0, -61.0, 7.863275e-30),
        (10.0, -61.0, 0.0),
        (100000.0, -61.0, 0.0),
        (-100000.0, -60.0, 0.0),
        (-10.0, -60.0, 0.0),
        (-3.0, -60.0, 2.3589825e-29),
        (-2.0, -60.0, 8.6736174e-19),
        (-1.0, -60.0, 1.0),
        (1.0, -60.0, 1.0),
        (2.0, -60.0, 8.6736174e-19),
        (3.0, -60.0, 2.3589825e-29),
        (10.0, -60.0, 0.0),
        (100000.0, -60.0, 0.0),
        (-100000.0, -59.0, -0.0),
        (-10.0, -59.0, -0.0),
        (-3.0, -59.0, -7.0769475e-29),
        (-2.0, -59.0, -1.7347235e-18),
        (-1.0, -59.0, -1.0),
        (1.0, -59.0, 1.0),
        (2.0, -59.0, 1.7347235e-18),
        (3.0, -59.0, 7.0769475e-29),
        (10.0, -59.0, 0.0),
        (100000.0, -59.0, 0.0),
        (-100000.0, -58.0, 0.0),
        (-10.0, -58.0, 0.0),
        (-3.0, -58.0, 2.1230843e-28),
        (-2.0, -58.0, 3.469447e-18),
        (-1.0, -58.0, 1.0),
        (1.0, -58.0, 1.0),
        (2.0, -58.0, 3.469447e-18),
        (3.0, -58.0, 2.1230843e-28),
        (10.0, -58.0, 0.0),
        (100000.0, -58.0, 0.0),
        (-100000.0, -57.0, -0.0),
        (-10.0, -57.0, -0.0),
        (-3.0, -57.0, -6.3692525e-28),
        (-2.0, -57.0, -6.938894e-18),
        (-1.0, -57.0, -1.0),
        (1.0, -57.0, 1.0),
        (2.0, -57.0, 6.938894e-18),
        (3.0, -57.0, 6.3692525e-28),
        (10.0, -57.0, 0.0),
        (100000.0, -57.0, 0.0),
        (-100000.0, -56.0, 0.0),
        (-10.0, -56.0, 0.0),
        (-3.0, -56.0, 1.9107758e-27),
        (-2.0, -56.0, 1.3877788e-17),
        (-1.0, -56.0, 1.0),
        (1.0, -56.0, 1.0),
        (2.0, -56.0, 1.3877788e-17),
        (3.0, -56.0, 1.9107758e-27),
        (10.0, -56.0, 0.0),
        (100000.0, -56.0, 0.0),
        (-100000.0, -55.0, -0.0),
        (-10.0, -55.0, -0.0),
        (-3.0, -55.0, -5.7323275e-27),
        (-2.0, -55.0, -2.7755576e-17),
        (-1.0, -55.0, -1.0),
        (1.0, -55.0, 1.0),
        (2.0, -55.0, 2.7755576e-17),
        (3.0, -55.0, 5.7323275e-27),
        (10.0, -55.0, 0.0),
        (100000.0, -55.0, 0.0),
        (-100000.0, -54.0, 0.0),
        (-10.0, -54.0, 0.0),
        (-3.0, -54.0, 1.7196983e-26),
        (-2.0, -54.0, 5.551115e-17),
        (-1.0, -54.0, 1.0),
        (1.0, -54.0, 1.0),
        (2.0, -54.0, 5.551115e-17),
        (3.0, -54.0, 1.7196983e-26),
        (10.0, -54.0, 0.0),
        (100000.0, -54.0, 0.0),
        (-100000.0, -53.0, -0.0),
        (-10.0, -53.0, -0.0),
        (-3.0, -53.0, -5.159095e-26),
        (-2.0, -53.0, -1.110223e-16),
        (-1.0, -53.0, -1.0),
        (1.0, -53.0, 1.0),
        (2.0, -53.0, 1.110223e-16),
        (3.0, -53.0, 5.159095e-26),
        (10.0, -53.0, 0.0),
        (100000.0, -53.0, 0.0),
        (-100000.0, -52.0, 0.0),
        (-10.0, -52.0, 0.0),
        (-3.0, -52.0, 1.5477285e-25),
        (-2.0, -52.0, 2.220446e-16),
        (-1.0, -52.0, 1.0),
        (1.0, -52.0, 1.0),
        (2.0, -52.0, 2.220446e-16),
        (3.0, -52.0, 1.5477285e-25),
        (10.0, -52.0, 0.0),
        (100000.0, -52.0, 0.0),
        (-100000.0, -51.0, -0.0),
        (-10.0, -51.0, -0.0),
        (-3.0, -51.0, -4.643185e-25),
        (-2.0, -51.0, -4.440892e-16),
        (-1.0, -51.0, -1.0),
        (1.0, -51.0, 1.0),
        (2.0, -51.0, 4.440892e-16),
        (3.0, -51.0, 4.643185e-25),
        (10.0, -51.0, 0.0),
        (100000.0, -51.0, 0.0),
        (-100000.0, -50.0, 0.0),
        (-10.0, -50.0, 0.0),
        (-3.0, -50.0, 1.3929556e-24),
        (-2.0, -50.0, 8.881784e-16),
        (-1.0, -50.0, 1.0),
        (1.0, -50.0, 1.0),
        (2.0, -50.0, 8.881784e-16),
        (3.0, -50.0, 1.3929556e-24),
        (10.0, -50.0, 0.0),
        (100000.0, -50.0, 0.0),
        (-100000.0, -49.0, -0.0),
        (-10.0, -49.0, -0.0),
        (-3.0, -49.0, -4.1788668e-24),
        (-2.0, -49.0, -1.7763568e-15),
        (-1.0, -49.0, -1.0),
        (1.0, -49.0, 1.0),
        (2.0, -49.0, 1.7763568e-15),
        (3.0, -49.0, 4.1788668e-24),
        (10.0, -49.0, 0.0),
        (100000.0, -49.0, 0.0),
        (-100000.0, -48.0, 0.0),
        (-10.0, -48.0, 0.0),
        (-3.0, -48.0, 1.25366e-23),
        (-2.0, -48.0, 3.5527137e-15),
        (-1.0, -48.0, 1.0),
        (1.0, -48.0, 1.0),
        (2.0, -48.0, 3.5527137e-15),
        (3.0, -48.0, 1.25366e-23),
        (10.0, -48.0, 0.0),
        (100000.0, -48.0, 0.0),
        (-100000.0, -47.0, -0.0),
        (-10.0, -47.0, -0.0),
        (-3.0, -47.0, -3.7609802e-23),
        (-2.0, -47.0, -7.1054274e-15),
        (-1.0, -47.0, -1.0),
        (1.0, -47.0, 1.0),
        (2.0, -47.0, 7.1054274e-15),
        (3.0, -47.0, 3.7609802e-23),
        (10.0, -47.0, 0.0),
        (100000.0, -47.0, 0.0),
        (-100000.0, -46.0, 0.0),
        (-10.0, -46.0, 0.0),
        (-3.0, -46.0, 1.128294e-22),
        (-2.0, -46.0, 1.4210855e-14),
        (-1.0, -46.0, 1.0),
        (1.0, -46.0, 1.0),
        (2.0, -46.0, 1.4210855e-14),
        (3.0, -46.0, 1.128294e-22),
        (10.0, -46.0, 0.0),
        (100000.0, -46.0, 0.0),
        (-100000.0, -45.0, -0.0),
        (-10.0, -45.0, -0.0),
        (-3.0, -45.0, -3.384882e-22),
        (-2.0, -45.0, -2.842171e-14),
        (-1.0, -45.0, -1.0),
        (1.0, -45.0, 1.0),
        (2.0, -45.0, 2.842171e-14),
        (3.0, -45.0, 3.384882e-22),
        (10.0, -45.0, 0.0),
        (100000.0, -45.0, 0.0),
        (-100000.0, -44.0, 0.0),
        (-10.0, -44.0, 1e-44),
        (-3.0, -44.0, 1.0154646e-21),
        (-2.0, -44.0, 5.684342e-14),
        (-1.0, -44.0, 1.0),
        (1.0, -44.0, 1.0),
        (2.0, -44.0, 5.684342e-14),
        (3.0, -44.0, 1.0154646e-21),
        (10.0, -44.0, 1e-44),
        (100000.0, -44.0, 0.0),
        (-100000.0, -43.0, -0.0),
        (-10.0, -43.0, -1e-43),
        (-3.0, -43.0, -3.046394e-21),
        (-2.0, -43.0, -1.1368684e-13),
        (-1.0, -43.0, -1.0),
        (1.0, -43.0, 1.0),
        (2.0, -43.0, 1.1368684e-13),
        (3.0, -43.0, 3.046394e-21),
        (10.0, -43.0, 1e-43),
        (100000.0, -43.0, 0.0),
        (-100000.0, -42.0, 0.0),
        (-10.0, -42.0, 1e-42),
        (-3.0, -42.0, 9.139182e-21),
        (-2.0, -42.0, 2.2737368e-13),
        (-1.0, -42.0, 1.0),
        (1.0, -42.0, 1.0),
        (2.0, -42.0, 2.2737368e-13),
        (3.0, -42.0, 9.139182e-21),
        (10.0, -42.0, 1e-42),
        (100000.0, -42.0, 0.0),
        (-100000.0, -41.0, -0.0),
        (-10.0, -41.0, -1e-41),
        (-3.0, -41.0, -2.7417544e-20),
        (-2.0, -41.0, -4.5474735e-13),
        (-1.0, -41.0, -1.0),
        (1.0, -41.0, 1.0),
        (2.0, -41.0, 4.5474735e-13),
        (3.0, -41.0, 2.7417544e-20),
        (10.0, -41.0, 1e-41),
        (100000.0, -41.0, 0.0),
        (-100000.0, -40.0, 0.0),
        (-10.0, -40.0, 1e-40),
        (-3.0, -40.0, 8.2252636e-20),
        (-2.0, -40.0, 9.094947e-13),
        (-1.0, -40.0, 1.0),
        (1.0, -40.0, 1.0),
        (2.0, -40.0, 9.094947e-13),
        (3.0, -40.0, 8.2252636e-20),
        (10.0, -40.0, 1e-40),
        (100000.0, -40.0, 0.0),
        (-100000.0, -39.0, -0.0),
        (-10.0, -39.0, -1e-39),
        (-3.0, -39.0, -2.467579e-19),
        (-2.0, -39.0, -1.8189894e-12),
        (-1.0, -39.0, -1.0),
        (1.0, -39.0, 1.0),
        (2.0, -39.0, 1.8189894e-12),
        (3.0, -39.0, 2.467579e-19),
        (10.0, -39.0, 1e-39),
        (100000.0, -39.0, 0.0),
        (-100000.0, -38.0, 0.0),
        (-10.0, -38.0, 1e-38),
        (-3.0, -38.0, 7.402737e-19),
        (-2.0, -38.0, 3.637979e-12),
        (-1.0, -38.0, 1.0),
        (1.0, -38.0, 1.0),
        (2.0, -38.0, 3.637979e-12),
        (3.0, -38.0, 7.402737e-19),
        (10.0, -38.0, 1e-38),
        (100000.0, -38.0, 0.0),
        (-100000.0, -37.0, -0.0),
        (-10.0, -37.0, -1e-37),
        (-3.0, -37.0, -2.2208211e-18),
        (-2.0, -37.0, -7.275958e-12),
        (-1.0, -37.0, -1.0),
        (1.0, -37.0, 1.0),
        (2.0, -37.0, 7.275958e-12),
        (3.0, -37.0, 2.2208211e-18),
        (10.0, -37.0, 1e-37),
        (100000.0, -37.0, 0.0),
        (-100000.0, -36.0, 0.0),
        (-10.0, -36.0, 1e-36),
        (-3.0, -36.0, 6.6624634e-18),
        (-2.0, -36.0, 1.4551915e-11),
        (-1.0, -36.0, 1.0),
        (1.0, -36.0, 1.0),
        (2.0, -36.0, 1.4551915e-11),
        (3.0, -36.0, 6.6624634e-18),
        (10.0, -36.0, 1e-36),
        (100000.0, -36.0, 0.0),
        (-100000.0, -35.0, -0.0),
        (-10.0, -35.0, -1e-35),
        (-3.0, -35.0, -1.998739e-17),
        (-2.0, -35.0, -2.910383e-11),
        (-1.0, -35.0, -1.0),
        (1.0, -35.0, 1.0),
        (2.0, -35.0, 2.910383e-11),
        (3.0, -35.0, 1.998739e-17),
        (10.0, -35.0, 1e-35),
        (100000.0, -35.0, 0.0),
        (-100000.0, -34.0, 0.0),
        (-10.0, -34.0, 1e-34),
        (-3.0, -34.0, 5.996217e-17),
        (-2.0, -34.0, 5.820766e-11),
        (-1.0, -34.0, 1.0),
        (1.0, -34.0, 1.0),
        (2.0, -34.0, 5.820766e-11),
        (3.0, -34.0, 5.996217e-17),
        (10.0, -34.0, 1e-34),
        (100000.0, -34.0, 0.0),
        (-100000.0, -33.0, -0.0),
        (-10.0, -33.0, -1e-33),
        (-3.0, -33.0, -1.798865e-16),
        (-2.0, -33.0, -1.1641532e-10),
        (-1.0, -33.0, -1.0),
        (1.0, -33.0, 1.0),
        (2.0, -33.0, 1.1641532e-10),
        (3.0, -33.0, 1.798865e-16),
        (10.0, -33.0, 1e-33),
        (100000.0, -33.0, 0.0),
        (-100000.0, -32.0, 0.0),
        (-10.0, -32.0, 1e-32),
        (-3.0, -32.0, 5.396595e-16),
        (-2.0, -32.0, 2.3283064e-10),
        (-1.0, -32.0, 1.0),
        (1.0, -32.0, 1.0),
        (2.0, -32.0, 2.3283064e-10),
        (3.0, -32.0, 5.396595e-16),
        (10.0, -32.0, 1e-32),
        (100000.0, -32.0, 0.0),
        (-100000.0, -31.0, -0.0),
        (-10.0, -31.0, -1e-31),
        (-3.0, -31.0, -1.6189785e-15),
        (-2.0, -31.0, -4.656613e-10),
        (-1.0, -31.0, -1.0),
        (1.0, -31.0, 1.0),
        (2.0, -31.0, 4.656613e-10),
        (3.0, -31.0, 1.6189785e-15),
        (10.0, -31.0, 1e-31),
        (100000.0, -31.0, 0.0),
        (-100000.0, -30.0, 0.0),
        (-10.0, -30.0, 1e-30),
        (-3.0, -30.0, 4.8569356e-15),
        (-2.0, -30.0, 9.313226e-10),
        (-1.0, -30.0, 1.0),
        (1.0, -30.0, 1.0),
        (2.0, -30.0, 9.313226e-10),
        (3.0, -30.0, 4.8569356e-15),
        (10.0, -30.0, 1e-30),
        (100000.0, -30.0, 0.0),
        (-100000.0, -29.0, -0.0),
        (-10.0, -29.0, -1e-29),
        (-3.0, -29.0, -1.4570806e-14),
        (-2.0, -29.0, -1.862_645_1e-9),
        (-1.0, -29.0, -1.0),
        (1.0, -29.0, 1.0),
        (2.0, -29.0, 1.862_645_1e-9),
        (3.0, -29.0, 1.4570806e-14),
        (10.0, -29.0, 1e-29),
        (100000.0, -29.0, 0.0),
        (-100000.0, -28.0, 0.0),
        (-10.0, -28.0, 1e-28),
        (-3.0, -28.0, 4.3712423e-14),
        (-2.0, -28.0, 3.725_290_3e-9),
        (-1.0, -28.0, 1.0),
        (1.0, -28.0, 1.0),
        (2.0, -28.0, 3.725_290_3e-9),
        (3.0, -28.0, 4.3712423e-14),
        (10.0, -28.0, 1e-28),
        (100000.0, -28.0, 0.0),
        (-100000.0, -27.0, -0.0),
        (-10.0, -27.0, -1e-27),
        (-3.0, -27.0, -1.3113726e-13),
        (-2.0, -27.0, -7.450_581e-9),
        (-1.0, -27.0, -1.0),
        (1.0, -27.0, 1.0),
        (2.0, -27.0, 7.450_581e-9),
        (3.0, -27.0, 1.3113726e-13),
        (10.0, -27.0, 1e-27),
        (100000.0, -27.0, 0.0),
        (-100000.0, -26.0, 0.0),
        (-10.0, -26.0, 1e-26),
        (-3.0, -26.0, 3.9341178e-13),
        (-2.0, -26.0, 1.490_116_1e-8),
        (-1.0, -26.0, 1.0),
        (1.0, -26.0, 1.0),
        (2.0, -26.0, 1.490_116_1e-8),
        (3.0, -26.0, 3.9341178e-13),
        (10.0, -26.0, 1e-26),
        (100000.0, -26.0, 0.0),
        (-100000.0, -25.0, -0.0),
        (-10.0, -25.0, -1e-25),
        (-3.0, -25.0, -1.1802354e-12),
        (-2.0, -25.0, -2.980_232_2e-8),
        (-1.0, -25.0, -1.0),
        (1.0, -25.0, 1.0),
        (2.0, -25.0, 2.980_232_2e-8),
        (3.0, -25.0, 1.1802354e-12),
        (10.0, -25.0, 1e-25),
        (100000.0, -25.0, 0.0),
        (-100000.0, -24.0, 0.0),
        (-10.0, -24.0, 1e-24),
        (-3.0, -24.0, 3.5407061e-12),
        (-2.0, -24.0, 5.960_464_5e-8),
        (-1.0, -24.0, 1.0),
        (1.0, -24.0, 1.0),
        (2.0, -24.0, 5.960_464_5e-8),
        (3.0, -24.0, 3.5407061e-12),
        (10.0, -24.0, 1e-24),
        (100000.0, -24.0, 0.0),
        (-100000.0, -23.0, -0.0),
        (-10.0, -23.0, -1e-23),
        (-3.0, -23.0, -1.0622119e-11),
        (-2.0, -23.0, -1.192_092_9e-7),
        (-1.0, -23.0, -1.0),
        (1.0, -23.0, 1.0),
        (2.0, -23.0, 1.192_092_9e-7),
        (3.0, -23.0, 1.0622119e-11),
        (10.0, -23.0, 1e-23),
        (100000.0, -23.0, 0.0),
        (-100000.0, -22.0, 0.0),
        (-10.0, -22.0, 1e-22),
        (-3.0, -22.0, 3.1866357e-11),
        (-2.0, -22.0, 2.384_185_8e-7),
        (-1.0, -22.0, 1.0),
        (1.0, -22.0, 1.0),
        (2.0, -22.0, 2.384_185_8e-7),
        (3.0, -22.0, 3.1866357e-11),
        (10.0, -22.0, 1e-22),
        (100000.0, -22.0, 0.0),
        (-100000.0, -21.0, -0.0),
        (-10.0, -21.0, -1e-21),
        (-3.0, -21.0, -9.559906e-11),
        (-2.0, -21.0, -4.768_371_6e-7),
        (-1.0, -21.0, -1.0),
        (1.0, -21.0, 1.0),
        (2.0, -21.0, 4.768_371_6e-7),
        (3.0, -21.0, 9.559906e-11),
        (10.0, -21.0, 1e-21),
        (100000.0, -21.0, 0.0),
        (-100000.0, -20.0, 0.0),
        (-10.0, -20.0, 1e-20),
        (-3.0, -20.0, 2.867972e-10),
        (-2.0, -20.0, 9.536_743e-7),
        (-1.0, -20.0, 1.0),
        (1.0, -20.0, 1.0),
        (2.0, -20.0, 9.536_743e-7),
        (3.0, -20.0, 2.867972e-10),
        (10.0, -20.0, 1e-20),
        (100000.0, -20.0, 0.0),
        (-100000.0, -19.0, -0.0),
        (-10.0, -19.0, -1e-19),
        (-3.0, -19.0, -8.603916e-10),
        (-2.0, -19.0, -1.907_348_6e-6),
        (-1.0, -19.0, -1.0),
        (1.0, -19.0, 1.0),
        (2.0, -19.0, 1.907_348_6e-6),
        (3.0, -19.0, 8.603916e-10),
        (10.0, -19.0, 1e-19),
        (100000.0, -19.0, 0.0),
        (-100000.0, -18.0, 0.0),
        (-10.0, -18.0, 1e-18),
        (-3.0, -18.0, 2.581_174_8e-9),
        (-2.0, -18.0, 3.814_697_3e-6),
        (-1.0, -18.0, 1.0),
        (1.0, -18.0, 1.0),
        (2.0, -18.0, 3.814_697_3e-6),
        (3.0, -18.0, 2.581_174_8e-9),
        (10.0, -18.0, 1e-18),
        (100000.0, -18.0, 0.0),
        (-100000.0, -17.0, -0.0),
        (-10.0, -17.0, -1e-17),
        (-3.0, -17.0, -7.743_524e-9),
        (-2.0, -17.0, -7.629_394_5e-6),
        (-1.0, -17.0, -1.0),
        (1.0, -17.0, 1.0),
        (2.0, -17.0, 7.629_394_5e-6),
        (3.0, -17.0, 7.743_524e-9),
        (10.0, -17.0, 1e-17),
        (100000.0, -17.0, 0.0),
        (-100000.0, -16.0, 0.0),
        (-10.0, -16.0, 1e-16),
        (-3.0, -16.0, 2.323_057_3e-8),
        (-2.0, -16.0, 1.525_878_9e-5),
        (-1.0, -16.0, 1.0),
        (1.0, -16.0, 1.0),
        (2.0, -16.0, 1.525_878_9e-5),
        (3.0, -16.0, 2.323_057_3e-8),
        (10.0, -16.0, 1e-16),
        (100000.0, -16.0, 0.0),
        (-100000.0, -15.0, -0.0),
        (-10.0, -15.0, -1e-15),
        (-3.0, -15.0, -6.969_172e-8),
        (-2.0, -15.0, -3.051_757_8e-5),
        (-1.0, -15.0, -1.0),
        (1.0, -15.0, 1.0),
        (2.0, -15.0, 3.051_757_8e-5),
        (3.0, -15.0, 6.969_172e-8),
        (10.0, -15.0, 1e-15),
        (100000.0, -15.0, 0.0),
        (-100000.0, -14.0, 0.0),
        (-10.0, -14.0, 1e-14),
        (-3.0, -14.0, 2.090_751_5e-7),
        (-2.0, -14.0, 6.103_515_6e-5),
        (-1.0, -14.0, 1.0),
        (1.0, -14.0, 1.0),
        (2.0, -14.0, 6.103_515_6e-5),
        (3.0, -14.0, 2.090_751_5e-7),
        (10.0, -14.0, 1e-14),
        (100000.0, -14.0, 0.0),
        (-100000.0, -13.0, -0.0),
        (-10.0, -13.0, -1e-13),
        (-3.0, -13.0, -6.272_254_6e-7),
        (-2.0, -13.0, -0.00012207031),
        (-1.0, -13.0, -1.0),
        (1.0, -13.0, 1.0),
        (2.0, -13.0, 0.00012207031),
        (3.0, -13.0, 6.272_254_6e-7),
        (10.0, -13.0, 1e-13),
        (100000.0, -13.0, 0.0),
        (-100000.0, -12.0, 0.0),
        (-10.0, -12.0, 1e-12),
        (-3.0, -12.0, 1.881_676_4e-6),
        (-2.0, -12.0, 0.000_244_140_63),
        (-1.0, -12.0, 1.0),
        (1.0, -12.0, 1.0),
        (2.0, -12.0, 0.000_244_140_63),
        (3.0, -12.0, 1.881_676_4e-6),
        (10.0, -12.0, 1e-12),
        (100000.0, -12.0, 0.0),
        (-100000.0, -11.0, -0.0),
        (-10.0, -11.0, -1e-11),
        (-3.0, -11.0, -5.645_029_4e-6),
        (-2.0, -11.0, -0.00048828125),
        (-1.0, -11.0, -1.0),
        (1.0, -11.0, 1.0),
        (2.0, -11.0, 0.00048828125),
        (3.0, -11.0, 5.645_029_4e-6),
        (10.0, -11.0, 1e-11),
        (100000.0, -11.0, 0.0),
        (-100000.0, -10.0, 0.0),
        (-10.0, -10.0, 1e-10),
        (-3.0, -10.0, 1.693_508_8e-5),
        (-2.0, -10.0, 0.0009765625),
        (-1.0, -10.0, 1.0),
        (1.0, -10.0, 1.0),
        (2.0, -10.0, 0.0009765625),
        (3.0, -10.0, 1.693_508_8e-5),
        (10.0, -10.0, 1e-10),
        (100000.0, -10.0, 0.0),
        (-100000.0, -9.0, -0.0),
        (-10.0, -9.0, -1e-09),
        (-3.0, -9.0, -5.080_526e-5),
        (-2.0, -9.0, -0.001953125),
        (-1.0, -9.0, -1.0),
        (1.0, -9.0, 1.0),
        (2.0, -9.0, 0.001953125),
        (3.0, -9.0, 5.080_526e-5),
        (10.0, -9.0, 1e-09),
        (100000.0, -9.0, 0.0),
        (-100000.0, -8.0, 1e-40),
        (-10.0, -8.0, 1e-08),
        (-3.0, -8.0, 0.00015241579),
        (-2.0, -8.0, 0.00390625),
        (-1.0, -8.0, 1.0),
        (1.0, -8.0, 1.0),
        (2.0, -8.0, 0.00390625),
        (3.0, -8.0, 0.00015241579),
        (10.0, -8.0, 1e-08),
        (100000.0, -8.0, 1e-40),
        (-100000.0, -7.0, -1e-35),
        (-10.0, -7.0, -1e-07),
        (-3.0, -7.0, -0.00045724737),
        (-2.0, -7.0, -0.0078125),
        (-1.0, -7.0, -1.0),
        (1.0, -7.0, 1.0),
        (2.0, -7.0, 0.0078125),
        (3.0, -7.0, 0.00045724737),
        (10.0, -7.0, 1e-07),
        (100000.0, -7.0, 1e-35),
        (-100000.0, -6.0, 1e-30),
        (-10.0, -6.0, 1e-06),
        (-3.0, -6.0, 0.0013717421),
        (-2.0, -6.0, 0.015625),
        (-1.0, -6.0, 1.0),
        (1.0, -6.0, 1.0),
        (2.0, -6.0, 0.015625),
        (3.0, -6.0, 0.0013717421),
        (10.0, -6.0, 1e-06),
        (100000.0, -6.0, 1e-30),
        (-100000.0, -5.0, -1e-25),
        (-10.0, -5.0, -1e-05),
        (-3.0, -5.0, -0.004115226),
        (-2.0, -5.0, -0.03125),
        (-1.0, -5.0, -1.0),
        (1.0, -5.0, 1.0),
        (2.0, -5.0, 0.03125),
        (3.0, -5.0, 0.004115226),
        (10.0, -5.0, 1e-05),
        (100000.0, -5.0, 1e-25),
        (-100000.0, -4.0, 1e-20),
        (-10.0, -4.0, 1e-04),
        (-3.0, -4.0, 0.012345679),
        (-2.0, -4.0, 0.0625),
        (-1.0, -4.0, 1.0),
        (1.0, -4.0, 1.0),
        (2.0, -4.0, 0.0625),
        (3.0, -4.0, 0.012345679),
        (10.0, -4.0, 1e-04),
        (100000.0, -4.0, 1e-20),
        (-100000.0, -3.0, -1e-15),
        (-10.0, -3.0, -0.001),
        (-3.0, -3.0, -0.037037037),
        (-2.0, -3.0, -0.125),
        (-1.0, -3.0, -1.0),
        (1.0, -3.0, 1.0),
        (2.0, -3.0, 0.125),
        (3.0, -3.0, 0.037037037),
        (10.0, -3.0, 0.001),
        (100000.0, -3.0, 1e-15),
        (-100000.0, -2.0, 1e-10),
        (-10.0, -2.0, 0.01),
        (-3.0, -2.0, 0.11111111),
        (-2.0, -2.0, 0.25),
        (-1.0, -2.0, 1.0),
        (1.0, -2.0, 1.0),
        (2.0, -2.0, 0.25),
        (3.0, -2.0, 0.11111111),
        (10.0, -2.0, 0.01),
        (100000.0, -2.0, 1e-10),
        (-100000.0, -1.0, -1e-05),
        (-10.0, -1.0, -0.1),
        (-3.0, -1.0, -0.33333334),
        (-2.0, -1.0, -0.5),
        (-1.0, -1.0, -1.0),
        (1.0, -1.0, 1.0),
        (2.0, -1.0, 0.5),
        (3.0, -1.0, 0.33333334),
        (10.0, -1.0, 0.1),
        (100000.0, -1.0, 1e-05),
        (-100000.0, 0.0, 1.0),
        (-10.0, 0.0, 1.0),
        (-3.0, 0.0, 1.0),
        (-2.0, 0.0, 1.0),
        (-1.0, 0.0, 1.0),
        (1.0, 0.0, 1.0),
        (2.0, 0.0, 1.0),
        (3.0, 0.0, 1.0),
        (10.0, 0.0, 1.0),
        (100000.0, 0.0, 1.0),
        (-100000.0, 1.0, -100000.0),
        (-10.0, 1.0, -10.0),
        (-3.0, 1.0, -3.0),
        (-2.0, 1.0, -2.0),
        (-1.0, 1.0, -1.0),
        (1.0, 1.0, 1.0),
        (2.0, 1.0, 2.0),
        (3.0, 1.0, 3.0),
        (10.0, 1.0, 10.0),
        (100000.0, 1.0, 100000.0),
        (-100000.0, 2.0, 10000000000.0),
        (-10.0, 2.0, 100.0),
        (-3.0, 2.0, 9.0),
        (-2.0, 2.0, 4.0),
        (-1.0, 2.0, 1.0),
        (1.0, 2.0, 1.0),
        (2.0, 2.0, 4.0),
        (3.0, 2.0, 9.0),
        (10.0, 2.0, 100.0),
        (100000.0, 2.0, 10000000000.0),
        (-100000.0, 3.0, -1000000000000000.0),
        (-10.0, 3.0, -1000.0),
        (-3.0, 3.0, -27.0),
        (-2.0, 3.0, -8.0),
        (-1.0, 3.0, -1.0),
        (1.0, 3.0, 1.0),
        (2.0, 3.0, 8.0),
        (3.0, 3.0, 27.0),
        (10.0, 3.0, 1000.0),
        (100000.0, 3.0, 1000000000000000.0),
        (-100000.0, 4.0, 1e+20),
        (-10.0, 4.0, 10000.0),
        (-3.0, 4.0, 81.0),
        (-2.0, 4.0, 16.0),
        (-1.0, 4.0, 1.0),
        (1.0, 4.0, 1.0),
        (2.0, 4.0, 16.0),
        (3.0, 4.0, 81.0),
        (10.0, 4.0, 10000.0),
        (100000.0, 4.0, 1e+20),
        (-100000.0, 5.0, -1e+25),
        (-10.0, 5.0, -100000.0),
        (-3.0, 5.0, -243.0),
        (-2.0, 5.0, -32.0),
        (-1.0, 5.0, -1.0),
        (1.0, 5.0, 1.0),
        (2.0, 5.0, 32.0),
        (3.0, 5.0, 243.0),
        (10.0, 5.0, 100000.0),
        (100000.0, 5.0, 1e+25),
        (-100000.0, 6.0, 1e+30),
        (-10.0, 6.0, 1000000.0),
        (-3.0, 6.0, 729.0),
        (-2.0, 6.0, 64.0),
        (-1.0, 6.0, 1.0),
        (1.0, 6.0, 1.0),
        (2.0, 6.0, 64.0),
        (3.0, 6.0, 729.0),
        (10.0, 6.0, 1000000.0),
        (100000.0, 6.0, 1e+30),
        (-100000.0, 7.0, -1e+35),
        (-10.0, 7.0, -10000000.0),
        (-3.0, 7.0, -2187.0),
        (-2.0, 7.0, -128.0),
        (-1.0, 7.0, -1.0),
        (1.0, 7.0, 1.0),
        (2.0, 7.0, 128.0),
        (3.0, 7.0, 2187.0),
        (10.0, 7.0, 10000000.0),
        (100000.0, 7.0, 1e+35),
        (-100000.0, 8.0, f32::INFINITY),
        (-10.0, 8.0, 100000000.0),
        (-3.0, 8.0, 6561.0),
        (-2.0, 8.0, 256.0),
        (-1.0, 8.0, 1.0),
        (1.0, 8.0, 1.0),
        (2.0, 8.0, 256.0),
        (3.0, 8.0, 6561.0),
        (10.0, 8.0, 100000000.0),
        (100000.0, 8.0, f32::INFINITY),
        (-100000.0, 9.0, -f32::INFINITY),
        (-10.0, 9.0, -1000000000.0),
        (-3.0, 9.0, -19683.0),
        (-2.0, 9.0, -512.0),
        (-1.0, 9.0, -1.0),
        (1.0, 9.0, 1.0),
        (2.0, 9.0, 512.0),
        (3.0, 9.0, 19683.0),
        (10.0, 9.0, 1000000000.0),
        (100000.0, 9.0, f32::INFINITY),
        (-100000.0, 10.0, f32::INFINITY),
        (-10.0, 10.0, 10000000000.0),
        (-3.0, 10.0, 59049.0),
        (-2.0, 10.0, 1024.0),
        (-1.0, 10.0, 1.0),
        (1.0, 10.0, 1.0),
        (2.0, 10.0, 1024.0),
        (3.0, 10.0, 59049.0),
        (10.0, 10.0, 10000000000.0),
        (100000.0, 10.0, f32::INFINITY),
        (-100000.0, 11.0, -f32::INFINITY),
        (-10.0, 11.0, -100000000000.0),
        (-3.0, 11.0, -177147.0),
        (-2.0, 11.0, -2048.0),
        (-1.0, 11.0, -1.0),
        (1.0, 11.0, 1.0),
        (2.0, 11.0, 2048.0),
        (3.0, 11.0, 177147.0),
        (10.0, 11.0, 100000000000.0),
        (100000.0, 11.0, f32::INFINITY),
        (-100000.0, 12.0, f32::INFINITY),
        (-10.0, 12.0, 1000000000000.0),
        (-3.0, 12.0, 531441.0),
        (-2.0, 12.0, 4096.0),
        (-1.0, 12.0, 1.0),
        (1.0, 12.0, 1.0),
        (2.0, 12.0, 4096.0),
        (3.0, 12.0, 531441.0),
        (10.0, 12.0, 1000000000000.0),
        (100000.0, 12.0, f32::INFINITY),
        (-100000.0, 13.0, -f32::INFINITY),
        (-10.0, 13.0, -10000000000000.0),
        (-3.0, 13.0, -1594323.0),
        (-2.0, 13.0, -8192.0),
        (-1.0, 13.0, -1.0),
        (1.0, 13.0, 1.0),
        (2.0, 13.0, 8192.0),
        (3.0, 13.0, 1594323.0),
        (10.0, 13.0, 10000000000000.0),
        (100000.0, 13.0, f32::INFINITY),
        (-100000.0, 14.0, f32::INFINITY),
        (-10.0, 14.0, 100000000000000.0),
        (-3.0, 14.0, 4782969.0),
        (-2.0, 14.0, 16384.0),
        (-1.0, 14.0, 1.0),
        (1.0, 14.0, 1.0),
        (2.0, 14.0, 16384.0),
        (3.0, 14.0, 4782969.0),
        (10.0, 14.0, 100000000000000.0),
        (100000.0, 14.0, f32::INFINITY),
        (-100000.0, 15.0, -f32::INFINITY),
        (-10.0, 15.0, -1000000000000000.0),
        (-3.0, 15.0, -14348907.0),
        (-2.0, 15.0, -32768.0),
        (-1.0, 15.0, -1.0),
        (1.0, 15.0, 1.0),
        (2.0, 15.0, 32768.0),
        (3.0, 15.0, 14348907.0),
        (10.0, 15.0, 1000000000000000.0),
        (100000.0, 15.0, f32::INFINITY),
        (-100000.0, 16.0, f32::INFINITY),
        (-10.0, 16.0, 1e+16),
        (-3.0, 16.0, 43046720.0),
        (-2.0, 16.0, 65536.0),
        (-1.0, 16.0, 1.0),
        (1.0, 16.0, 1.0),
        (2.0, 16.0, 65536.0),
        (3.0, 16.0, 43046720.0),
        (10.0, 16.0, 1e+16),
        (100000.0, 16.0, f32::INFINITY),
        (-100000.0, 17.0, -f32::INFINITY),
        (-10.0, 17.0, -1e+17),
        (-3.0, 17.0, -129140160.0),
        (-2.0, 17.0, -131072.0),
        (-1.0, 17.0, -1.0),
        (1.0, 17.0, 1.0),
        (2.0, 17.0, 131072.0),
        (3.0, 17.0, 129140160.0),
        (10.0, 17.0, 1e+17),
        (100000.0, 17.0, f32::INFINITY),
        (-100000.0, 18.0, f32::INFINITY),
        (-10.0, 18.0, 1e+18),
        (-3.0, 18.0, 387420480.0),
        (-2.0, 18.0, 262144.0),
        (-1.0, 18.0, 1.0),
        (1.0, 18.0, 1.0),
        (2.0, 18.0, 262144.0),
        (3.0, 18.0, 387420480.0),
        (10.0, 18.0, 1e+18),
        (100000.0, 18.0, f32::INFINITY),
        (-100000.0, 19.0, -f32::INFINITY),
        (-10.0, 19.0, -1e+19),
        (-3.0, 19.0, -1162261500.0),
        (-2.0, 19.0, -524288.0),
        (-1.0, 19.0, -1.0),
        (1.0, 19.0, 1.0),
        (2.0, 19.0, 524288.0),
        (3.0, 19.0, 1162261500.0),
        (10.0, 19.0, 1e+19),
        (100000.0, 19.0, f32::INFINITY),
        (-100000.0, 20.0, f32::INFINITY),
        (-10.0, 20.0, 1e+20),
        (-3.0, 20.0, 3486784500.0),
        (-2.0, 20.0, 1048576.0),
        (-1.0, 20.0, 1.0),
        (1.0, 20.0, 1.0),
        (2.0, 20.0, 1048576.0),
        (3.0, 20.0, 3486784500.0),
        (10.0, 20.0, 1e+20),
        (100000.0, 20.0, f32::INFINITY),
        (-100000.0, 21.0, -f32::INFINITY),
        (-10.0, 21.0, -1e+21),
        (-3.0, 21.0, -10460354000.0),
        (-2.0, 21.0, -2097152.0),
        (-1.0, 21.0, -1.0),
        (1.0, 21.0, 1.0),
        (2.0, 21.0, 2097152.0),
        (3.0, 21.0, 10460354000.0),
        (10.0, 21.0, 1e+21),
        (100000.0, 21.0, f32::INFINITY),
        (-100000.0, 22.0, f32::INFINITY),
        (-10.0, 22.0, 1e+22),
        (-3.0, 22.0, 31381060000.0),
        (-2.0, 22.0, 4194304.0),
        (-1.0, 22.0, 1.0),
        (1.0, 22.0, 1.0),
        (2.0, 22.0, 4194304.0),
        (3.0, 22.0, 31381060000.0),
        (10.0, 22.0, 1e+22),
        (100000.0, 22.0, f32::INFINITY),
        (-100000.0, 23.0, -f32::INFINITY),
        (-10.0, 23.0, -1e+23),
        (-3.0, 23.0, -94143180000.0),
        (-2.0, 23.0, -8388608.0),
        (-1.0, 23.0, -1.0),
        (1.0, 23.0, 1.0),
        (2.0, 23.0, 8388608.0),
        (3.0, 23.0, 94143180000.0),
        (10.0, 23.0, 1e+23),
        (100000.0, 23.0, f32::INFINITY),
        (-100000.0, 24.0, f32::INFINITY),
        (-10.0, 24.0, 1e+24),
        (-3.0, 24.0, 282429520000.0),
        (-2.0, 24.0, 16777216.0),
        (-1.0, 24.0, 1.0),
        (1.0, 24.0, 1.0),
        (2.0, 24.0, 16777216.0),
        (3.0, 24.0, 282429520000.0),
        (10.0, 24.0, 1e+24),
        (100000.0, 24.0, f32::INFINITY),
        (-100000.0, 25.0, -f32::INFINITY),
        (-10.0, 25.0, -1e+25),
        (-3.0, 25.0, -847288600000.0),
        (-2.0, 25.0, -33554432.0),
        (-1.0, 25.0, -1.0),
        (1.0, 25.0, 1.0),
        (2.0, 25.0, 33554432.0),
        (3.0, 25.0, 847288600000.0),
        (10.0, 25.0, 1e+25),
        (100000.0, 25.0, f32::INFINITY),
        (-100000.0, 26.0, f32::INFINITY),
        (-10.0, 26.0, 1e+26),
        (-3.0, 26.0, 2541866000000.0),
        (-2.0, 26.0, 67108864.0),
        (-1.0, 26.0, 1.0),
        (1.0, 26.0, 1.0),
        (2.0, 26.0, 67108864.0),
        (3.0, 26.0, 2541866000000.0),
        (10.0, 26.0, 1e+26),
        (100000.0, 26.0, f32::INFINITY),
        (-100000.0, 27.0, -f32::INFINITY),
        (-10.0, 27.0, -1e+27),
        (-3.0, 27.0, -7625597500000.0),
        (-2.0, 27.0, -134217730.0),
        (-1.0, 27.0, -1.0),
        (1.0, 27.0, 1.0),
        (2.0, 27.0, 134217730.0),
        (3.0, 27.0, 7625597500000.0),
        (10.0, 27.0, 1e+27),
        (100000.0, 27.0, f32::INFINITY),
        (-100000.0, 28.0, f32::INFINITY),
        (-10.0, 28.0, 1e+28),
        (-3.0, 28.0, 22876793000000.0),
        (-2.0, 28.0, 268435460.0),
        (-1.0, 28.0, 1.0),
        (1.0, 28.0, 1.0),
        (2.0, 28.0, 268435460.0),
        (3.0, 28.0, 22876793000000.0),
        (10.0, 28.0, 1e+28),
        (100000.0, 28.0, f32::INFINITY),
        (-100000.0, 29.0, -f32::INFINITY),
        (-10.0, 29.0, -1e+29),
        (-3.0, 29.0, -68630377000000.0),
        (-2.0, 29.0, -536870900.0),
        (-1.0, 29.0, -1.0),
        (1.0, 29.0, 1.0),
        (2.0, 29.0, 536870900.0),
        (3.0, 29.0, 68630377000000.0),
        (10.0, 29.0, 1e+29),
        (100000.0, 29.0, f32::INFINITY),
        (-100000.0, 30.0, f32::INFINITY),
        (-10.0, 30.0, 1e+30),
        (-3.0, 30.0, 205891140000000.0),
        (-2.0, 30.0, 1073741800.0),
        (-1.0, 30.0, 1.0),
        (1.0, 30.0, 1.0),
        (2.0, 30.0, 1073741800.0),
        (3.0, 30.0, 205891140000000.0),
        (10.0, 30.0, 1e+30),
        (100000.0, 30.0, f32::INFINITY),
        (-100000.0, 31.0, -f32::INFINITY),
        (-10.0, 31.0, -1e+31),
        (-3.0, 31.0, -617673400000000.0),
        (-2.0, 31.0, -2147483600.0),
        (-1.0, 31.0, -1.0),
        (1.0, 31.0, 1.0),
        (2.0, 31.0, 2147483600.0),
        (3.0, 31.0, 617673400000000.0),
        (10.0, 31.0, 1e+31),
        (100000.0, 31.0, f32::INFINITY),
        (-100000.0, 32.0, f32::INFINITY),
        (-10.0, 32.0, 1e+32),
        (-3.0, 32.0, 1853020200000000.0),
        (-2.0, 32.0, 4294967300.0),
        (-1.0, 32.0, 1.0),
        (1.0, 32.0, 1.0),
        (2.0, 32.0, 4294967300.0),
        (3.0, 32.0, 1853020200000000.0),
        (10.0, 32.0, 1e+32),
        (100000.0, 32.0, f32::INFINITY),
        (-100000.0, 33.0, -f32::INFINITY),
        (-10.0, 33.0, -1e+33),
        (-3.0, 33.0, -5559060500000000.0),
        (-2.0, 33.0, -8589935000.0),
        (-1.0, 33.0, -1.0),
        (1.0, 33.0, 1.0),
        (2.0, 33.0, 8589935000.0),
        (3.0, 33.0, 5559060500000000.0),
        (10.0, 33.0, 1e+33),
        (100000.0, 33.0, f32::INFINITY),
        (-100000.0, 34.0, f32::INFINITY),
        (-10.0, 34.0, 1e+34),
        (-3.0, 34.0, 1.667_718_2e16),
        (-2.0, 34.0, 17179870000.0),
        (-1.0, 34.0, 1.0),
        (1.0, 34.0, 1.0),
        (2.0, 34.0, 17179870000.0),
        (3.0, 34.0, 1.667_718_2e16),
        (10.0, 34.0, 1e+34),
        (100000.0, 34.0, f32::INFINITY),
        (-100000.0, 35.0, -f32::INFINITY),
        (-10.0, 35.0, -1e+35),
        (-3.0, 35.0, -5.003_154_6e16),
        (-2.0, 35.0, -34359740000.0),
        (-1.0, 35.0, -1.0),
        (1.0, 35.0, 1.0),
        (2.0, 35.0, 34359740000.0),
        (3.0, 35.0, 5.003_154_6e16),
        (10.0, 35.0, 1e+35),
        (100000.0, 35.0, f32::INFINITY),
        (-100000.0, 36.0, f32::INFINITY),
        (-10.0, 36.0, 1e+36),
        (-3.0, 36.0, 1.500_946_4e17),
        (-2.0, 36.0, 68719480000.0),
        (-1.0, 36.0, 1.0),
        (1.0, 36.0, 1.0),
        (2.0, 36.0, 68719480000.0),
        (3.0, 36.0, 1.500_946_4e17),
        (10.0, 36.0, 1e+36),
        (100000.0, 36.0, f32::INFINITY),
        (-100000.0, 37.0, -f32::INFINITY),
        (-10.0, 37.0, -1e+37),
        (-3.0, 37.0, -4.502_839e17),
        (-2.0, 37.0, -137438950000.0),
        (-1.0, 37.0, -1.0),
        (1.0, 37.0, 1.0),
        (2.0, 37.0, 137438950000.0),
        (3.0, 37.0, 4.502_839e17),
        (10.0, 37.0, 1e+37),
        (100000.0, 37.0, f32::INFINITY),
        (-100000.0, 38.0, f32::INFINITY),
        (-10.0, 38.0, 1e+38),
        (-3.0, 38.0, 1.350_851_7e18),
        (-2.0, 38.0, 274877900000.0),
        (-1.0, 38.0, 1.0),
        (1.0, 38.0, 1.0),
        (2.0, 38.0, 274877900000.0),
        (3.0, 38.0, 1.350_851_7e18),
        (10.0, 38.0, 1e+38),
        (100000.0, 38.0, f32::INFINITY),
        (-100000.0, 39.0, -f32::INFINITY),
        (-10.0, 39.0, -f32::INFINITY),
        (-3.0, 39.0, -4.052_555_2e18),
        (-2.0, 39.0, -549755800000.0),
        (-1.0, 39.0, -1.0),
        (1.0, 39.0, 1.0),
        (2.0, 39.0, 549755800000.0),
        (3.0, 39.0, 4.052_555_2e18),
        (10.0, 39.0, f32::INFINITY),
        (100000.0, 39.0, f32::INFINITY),
        (-100000.0, 40.0, f32::INFINITY),
        (-10.0, 40.0, f32::INFINITY),
        (-3.0, 40.0, 1.215_766_5e19),
        (-2.0, 40.0, 1099511600000.0),
        (-1.0, 40.0, 1.0),
        (1.0, 40.0, 1.0),
        (2.0, 40.0, 1099511600000.0),
        (3.0, 40.0, 1.215_766_5e19),
        (10.0, 40.0, f32::INFINITY),
        (100000.0, 40.0, f32::INFINITY),
        (-100000.0, 41.0, -f32::INFINITY),
        (-10.0, 41.0, -f32::INFINITY),
        (-3.0, 41.0, -3.647_299_5e19),
        (-2.0, 41.0, -2199023300000.0),
        (-1.0, 41.0, -1.0),
        (1.0, 41.0, 1.0),
        (2.0, 41.0, 2199023300000.0),
        (3.0, 41.0, 3.647_299_5e19),
        (10.0, 41.0, f32::INFINITY),
        (100000.0, 41.0, f32::INFINITY),
        (-100000.0, 42.0, f32::INFINITY),
        (-10.0, 42.0, f32::INFINITY),
        (-3.0, 42.0, 1.094_189_9e20),
        (-2.0, 42.0, 4398046500000.0),
        (-1.0, 42.0, 1.0),
        (1.0, 42.0, 1.0),
        (2.0, 42.0, 4398046500000.0),
        (3.0, 42.0, 1.094_189_9e20),
        (10.0, 42.0, f32::INFINITY),
        (100000.0, 42.0, f32::INFINITY),
        (-100000.0, 43.0, -f32::INFINITY),
        (-10.0, 43.0, -f32::INFINITY),
        (-3.0, 43.0, -3.282_569_6e20),
        (-2.0, 43.0, -8796093000000.0),
        (-1.0, 43.0, -1.0),
        (1.0, 43.0, 1.0),
        (2.0, 43.0, 8796093000000.0),
        (3.0, 43.0, 3.282_569_6e20),
        (10.0, 43.0, f32::INFINITY),
        (100000.0, 43.0, f32::INFINITY),
        (-100000.0, 44.0, f32::INFINITY),
        (-10.0, 44.0, f32::INFINITY),
        (-3.0, 44.0, 9.847_709e20),
        (-2.0, 44.0, 17592186000000.0),
        (-1.0, 44.0, 1.0),
        (1.0, 44.0, 1.0),
        (2.0, 44.0, 17592186000000.0),
        (3.0, 44.0, 9.847_709e20),
        (10.0, 44.0, f32::INFINITY),
        (100000.0, 44.0, f32::INFINITY),
        (-100000.0, 45.0, -f32::INFINITY),
        (-10.0, 45.0, -f32::INFINITY),
        (-3.0, 45.0, -2.954_312_7e21),
        (-2.0, 45.0, -35184372000000.0),
        (-1.0, 45.0, -1.0),
        (1.0, 45.0, 1.0),
        (2.0, 45.0, 35184372000000.0),
        (3.0, 45.0, 2.954_312_7e21),
        (10.0, 45.0, f32::INFINITY),
        (100000.0, 45.0, f32::INFINITY),
        (-100000.0, 46.0, f32::INFINITY),
        (-10.0, 46.0, f32::INFINITY),
        (-3.0, 46.0, 8.862_938e21),
        (-2.0, 46.0, 70368744000000.0),
        (-1.0, 46.0, 1.0),
        (1.0, 46.0, 1.0),
        (2.0, 46.0, 70368744000000.0),
        (3.0, 46.0, 8.862_938e21),
        (10.0, 46.0, f32::INFINITY),
        (100000.0, 46.0, f32::INFINITY),
        (-100000.0, 47.0, -f32::INFINITY),
        (-10.0, 47.0, -f32::INFINITY),
        (-3.0, 47.0, -2.658_881_5e22),
        (-2.0, 47.0, -140737490000000.0),
        (-1.0, 47.0, -1.0),
        (1.0, 47.0, 1.0),
        (2.0, 47.0, 140737490000000.0),
        (3.0, 47.0, 2.658_881_5e22),
        (10.0, 47.0, f32::INFINITY),
        (100000.0, 47.0, f32::INFINITY),
        (-100000.0, 48.0, f32::INFINITY),
        (-10.0, 48.0, f32::INFINITY),
        (-3.0, 48.0, 7.976_644e22),
        (-2.0, 48.0, 281474980000000.0),
        (-1.0, 48.0, 1.0),
        (1.0, 48.0, 1.0),
        (2.0, 48.0, 281474980000000.0),
        (3.0, 48.0, 7.976_644e22),
        (10.0, 48.0, f32::INFINITY),
        (100000.0, 48.0, f32::INFINITY),
        (-100000.0, 49.0, -f32::INFINITY),
        (-10.0, 49.0, -f32::INFINITY),
        (-3.0, 49.0, -2.392_993_2e23),
        (-2.0, 49.0, -562949950000000.0),
        (-1.0, 49.0, -1.0),
        (1.0, 49.0, 1.0),
        (2.0, 49.0, 562949950000000.0),
        (3.0, 49.0, 2.392_993_2e23),
        (10.0, 49.0, f32::INFINITY),
        (100000.0, 49.0, f32::INFINITY),
        (-100000.0, 50.0, f32::INFINITY),
        (-10.0, 50.0, f32::INFINITY),
        (-3.0, 50.0, 7.17898e+23),
        (-2.0, 50.0, 1125899900000000.0),
        (-1.0, 50.0, 1.0),
        (1.0, 50.0, 1.0),
        (2.0, 50.0, 1125899900000000.0),
        (3.0, 50.0, 7.17898e+23),
        (10.0, 50.0, f32::INFINITY),
        (100000.0, 50.0, f32::INFINITY),
        (-100000.0, 51.0, -f32::INFINITY),
        (-10.0, 51.0, -f32::INFINITY),
        (-3.0, 51.0, -2.153_694e24),
        (-2.0, 51.0, -2251799800000000.0),
        (-1.0, 51.0, -1.0),
        (1.0, 51.0, 1.0),
        (2.0, 51.0, 2251799800000000.0),
        (3.0, 51.0, 2.153_694e24),
        (10.0, 51.0, f32::INFINITY),
        (100000.0, 51.0, f32::INFINITY),
        (-100000.0, 52.0, f32::INFINITY),
        (-10.0, 52.0, f32::INFINITY),
        (-3.0, 52.0, 6.461_081_6e24),
        (-2.0, 52.0, 4503599600000000.0),
        (-1.0, 52.0, 1.0),
        (1.0, 52.0, 1.0),
        (2.0, 52.0, 4503599600000000.0),
        (3.0, 52.0, 6.461_081_6e24),
        (10.0, 52.0, f32::INFINITY),
        (100000.0, 52.0, f32::INFINITY),
        (-100000.0, 53.0, -f32::INFINITY),
        (-10.0, 53.0, -f32::INFINITY),
        (-3.0, 53.0, -1.938_324_6e25),
        (-2.0, 53.0, -9007199000000000.0),
        (-1.0, 53.0, -1.0),
        (1.0, 53.0, 1.0),
        (2.0, 53.0, 9007199000000000.0),
        (3.0, 53.0, 1.938_324_6e25),
        (10.0, 53.0, f32::INFINITY),
        (100000.0, 53.0, f32::INFINITY),
        (-100000.0, 54.0, f32::INFINITY),
        (-10.0, 54.0, f32::INFINITY),
        (-3.0, 54.0, 5.814_973_6e25),
        (-2.0, 54.0, 1.801_439_9e16),
        (-1.0, 54.0, 1.0),
        (1.0, 54.0, 1.0),
        (2.0, 54.0, 1.801_439_9e16),
        (3.0, 54.0, 5.814_973_6e25),
        (10.0, 54.0, f32::INFINITY),
        (100000.0, 54.0, f32::INFINITY),
        (-100000.0, 55.0, -f32::INFINITY),
        (-10.0, 55.0, -f32::INFINITY),
        (-3.0, 55.0, -1.744_492_2e26),
        (-2.0, 55.0, -3.602_879_7e16),
        (-1.0, 55.0, -1.0),
        (1.0, 55.0, 1.0),
        (2.0, 55.0, 3.602_879_7e16),
        (3.0, 55.0, 1.744_492_2e26),
        (10.0, 55.0, f32::INFINITY),
        (100000.0, 55.0, f32::INFINITY),
        (-100000.0, 56.0, f32::INFINITY),
        (-10.0, 56.0, f32::INFINITY),
        (-3.0, 56.0, 5.233_476_3e26),
        (-2.0, 56.0, 7.205_759_4e16),
        (-1.0, 56.0, 1.0),
        (1.0, 56.0, 1.0),
        (2.0, 56.0, 7.205_759_4e16),
        (3.0, 56.0, 5.233_476_3e26),
        (10.0, 56.0, f32::INFINITY),
        (100000.0, 56.0, f32::INFINITY),
        (-100000.0, 57.0, -f32::INFINITY),
        (-10.0, 57.0, -f32::INFINITY),
        (-3.0, 57.0, -1.570_043e27),
        (-2.0, 57.0, -1.441_151_9e17),
        (-1.0, 57.0, -1.0),
        (1.0, 57.0, 1.0),
        (2.0, 57.0, 1.441_151_9e17),
        (3.0, 57.0, 1.570_043e27),
        (10.0, 57.0, f32::INFINITY),
        (100000.0, 57.0, f32::INFINITY),
        (-100000.0, 58.0, f32::INFINITY),
        (-10.0, 58.0, f32::INFINITY),
        (-3.0, 58.0, 4.710_128_8e27),
        (-2.0, 58.0, 2.882_303_8e17),
        (-1.0, 58.0, 1.0),
        (1.0, 58.0, 1.0),
        (2.0, 58.0, 2.882_303_8e17),
        (3.0, 58.0, 4.710_128_8e27),
        (10.0, 58.0, f32::INFINITY),
        (100000.0, 58.0, f32::INFINITY),
        (-100000.0, 59.0, -f32::INFINITY),
        (-10.0, 59.0, -f32::INFINITY),
        (-3.0, 59.0, -1.413_038_7e28),
        (-2.0, 59.0, -5.764_607_5e17),
        (-1.0, 59.0, -1.0),
        (1.0, 59.0, 1.0),
        (2.0, 59.0, 5.764_607_5e17),
        (3.0, 59.0, 1.413_038_7e28),
        (10.0, 59.0, f32::INFINITY),
        (100000.0, 59.0, f32::INFINITY),
        (-100000.0, 60.0, f32::INFINITY),
        (-10.0, 60.0, f32::INFINITY),
        (-3.0, 60.0, 4.239_116e28),
        (-2.0, 60.0, 1.152_921_5e18),
        (-1.0, 60.0, 1.0),
        (1.0, 60.0, 1.0),
        (2.0, 60.0, 1.152_921_5e18),
        (3.0, 60.0, 4.239_116e28),
        (10.0, 60.0, f32::INFINITY),
        (100000.0, 60.0, f32::INFINITY),
        (-100000.0, 61.0, -f32::INFINITY),
        (-10.0, 61.0, -f32::INFINITY),
        (-3.0, 61.0, -1.271_734_7e29),
        (-2.0, 61.0, -2.305_843e18),
        (-1.0, 61.0, -1.0),
        (1.0, 61.0, 1.0),
        (2.0, 61.0, 2.305_843e18),
        (3.0, 61.0, 1.271_734_7e29),
        (10.0, 61.0, f32::INFINITY),
        (100000.0, 61.0, f32::INFINITY),
        (-100000.0, 62.0, f32::INFINITY),
        (-10.0, 62.0, f32::INFINITY),
        (-3.0, 62.0, 3.815_204_4e29),
        (-2.0, 62.0, 4.611_686e18),
        (-1.0, 62.0, 1.0),
        (1.0, 62.0, 1.0),
        (2.0, 62.0, 4.611_686e18),
        (3.0, 62.0, 3.815_204_4e29),
        (10.0, 62.0, f32::INFINITY),
        (100000.0, 62.0, f32::INFINITY),
        (-100000.0, 63.0, -f32::INFINITY),
        (-10.0, 63.0, -f32::INFINITY),
        (-3.0, 63.0, -1.144_561_25e30),
        (-2.0, 63.0, -9.223_372e18),
        (-1.0, 63.0, -1.0),
        (1.0, 63.0, 1.0),
        (2.0, 63.0, 9.223_372e18),
        (3.0, 63.0, 1.144_561_25e30),
        (10.0, 63.0, f32::INFINITY),
        (100000.0, 63.0, f32::INFINITY),
        (-100000.0, 64.0, f32::INFINITY),
        (-10.0, 64.0, f32::INFINITY),
        (-3.0, 64.0, 3.433_684e30),
        (-2.0, 64.0, 1.844_674_4e19),
        (-1.0, 64.0, 1.0),
        (1.0, 64.0, 1.0),
        (2.0, 64.0, 1.844_674_4e19),
        (3.0, 64.0, 3.433_684e30),
        (10.0, 64.0, f32::INFINITY),
        (100000.0, 64.0, f32::INFINITY),
        (-100000.0, 65.0, -f32::INFINITY),
        (-10.0, 65.0, -f32::INFINITY),
        (-3.0, 65.0, -1.030_105_1e31),
        (-2.0, 65.0, -3.689_349e19),
        (-1.0, 65.0, -1.0),
        (1.0, 65.0, 1.0),
        (2.0, 65.0, 3.689_349e19),
        (3.0, 65.0, 1.030_105_1e31),
        (10.0, 65.0, f32::INFINITY),
        (100000.0, 65.0, f32::INFINITY),
        (-100000.0, 66.0, f32::INFINITY),
        (-10.0, 66.0, f32::INFINITY),
        (-3.0, 66.0, 3.090_315_4e31),
        (-2.0, 66.0, 7.378_698e19),
        (-1.0, 66.0, 1.0),
        (1.0, 66.0, 1.0),
        (2.0, 66.0, 7.378_698e19),
        (3.0, 66.0, 3.090_315_4e31),
        (10.0, 66.0, f32::INFINITY),
        (100000.0, 66.0, f32::INFINITY),
        (-100000.0, 67.0, -f32::INFINITY),
        (-10.0, 67.0, -f32::INFINITY),
        (-3.0, 67.0, -9.270_946_5e31),
        (-2.0, 67.0, -1.475_739_5e20),
        (-1.0, 67.0, -1.0),
        (1.0, 67.0, 1.0),
        (2.0, 67.0, 1.475_739_5e20),
        (3.0, 67.0, 9.270_946_5e31),
        (10.0, 67.0, f32::INFINITY),
        (100000.0, 67.0, f32::INFINITY),
        (-100000.0, 68.0, f32::INFINITY),
        (-10.0, 68.0, f32::INFINITY),
        (-3.0, 68.0, 2.781_284e32),
        (-2.0, 68.0, 2.951_479e20),
        (-1.0, 68.0, 1.0),
        (1.0, 68.0, 1.0),
        (2.0, 68.0, 2.951_479e20),
        (3.0, 68.0, 2.781_284e32),
        (10.0, 68.0, f32::INFINITY),
        (100000.0, 68.0, f32::INFINITY),
        (-100000.0, 69.0, -f32::INFINITY),
        (-10.0, 69.0, -f32::INFINITY),
        (-3.0, 69.0, -8.343_852e32),
        (-2.0, 69.0, -5.902_958e20),
        (-1.0, 69.0, -1.0),
        (1.0, 69.0, 1.0),
        (2.0, 69.0, 5.902_958e20),
        (3.0, 69.0, 8.343_852e32),
        (10.0, 69.0, f32::INFINITY),
        (100000.0, 69.0, f32::INFINITY),
        (-100000.0, 70.0, f32::INFINITY),
        (-10.0, 70.0, f32::INFINITY),
        (-3.0, 70.0, 2.503_155_5e33),
        (-2.0, 70.0, 1.180_591_6e21),
        (-1.0, 70.0, 1.0),
        (1.0, 70.0, 1.0),
        (2.0, 70.0, 1.180_591_6e21),
        (3.0, 70.0, 2.503_155_5e33),
        (10.0, 70.0, f32::INFINITY),
        (100000.0, 70.0, f32::INFINITY),
        (-100000.0, 71.0, -f32::INFINITY),
        (-10.0, 71.0, -f32::INFINITY),
        (-3.0, 71.0, -7.509_466e33),
        (-2.0, 71.0, -2.361_183_2e21),
        (-1.0, 71.0, -1.0),
        (1.0, 71.0, 1.0),
        (2.0, 71.0, 2.361_183_2e21),
        (3.0, 71.0, 7.509_466e33),
        (10.0, 71.0, f32::INFINITY),
        (100000.0, 71.0, f32::INFINITY),
        (-100000.0, 72.0, f32::INFINITY),
        (-10.0, 72.0, f32::INFINITY),
        (-3.0, 72.0, 2.25284e+34),
        (-2.0, 72.0, 4.722_366_5e21),
        (-1.0, 72.0, 1.0),
        (1.0, 72.0, 1.0),
        (2.0, 72.0, 4.722_366_5e21),
        (3.0, 72.0, 2.25284e+34),
        (10.0, 72.0, f32::INFINITY),
        (100000.0, 72.0, f32::INFINITY),
        (-100000.0, 73.0, -f32::INFINITY),
        (-10.0, 73.0, -f32::INFINITY),
        (-3.0, 73.0, -6.75852e+34),
        (-2.0, 73.0, -9.444_733e21),
        (-1.0, 73.0, -1.0),
        (1.0, 73.0, 1.0),
        (2.0, 73.0, 9.444_733e21),
        (3.0, 73.0, 6.75852e+34),
        (10.0, 73.0, f32::INFINITY),
        (100000.0, 73.0, f32::INFINITY),
        (-100000.0, 74.0, f32::INFINITY),
        (-10.0, 74.0, f32::INFINITY),
        (-3.0, 74.0, 2.027_556e35),
        (-2.0, 74.0, 1.888_946_6e22),
        (-1.0, 74.0, 1.0),
        (1.0, 74.0, 1.0),
        (2.0, 74.0, 1.888_946_6e22),
        (3.0, 74.0, 2.027_556e35),
        (10.0, 74.0, f32::INFINITY),
        (100000.0, 74.0, f32::INFINITY),
        (-100000.0, 75.0, -f32::INFINITY),
        (-10.0, 75.0, -f32::INFINITY),
        (-3.0, 75.0, -6.082_667_7e35),
        (-2.0, 75.0, -3.777_893_2e22),
        (-1.0, 75.0, -1.0),
        (1.0, 75.0, 1.0),
        (2.0, 75.0, 3.777_893_2e22),
        (3.0, 75.0, 6.082_667_7e35),
        (10.0, 75.0, f32::INFINITY),
        (100000.0, 75.0, f32::INFINITY),
        (-100000.0, 76.0, f32::INFINITY),
        (-10.0, 76.0, f32::INFINITY),
        (-3.0, 76.0, 1.824_800_3e36),
        (-2.0, 76.0, 7.555_786_4e22),
        (-1.0, 76.0, 1.0),
        (1.0, 76.0, 1.0),
        (2.0, 76.0, 7.555_786_4e22),
        (3.0, 76.0, 1.824_800_3e36),
        (10.0, 76.0, f32::INFINITY),
        (100000.0, 76.0, f32::INFINITY),
        (-100000.0, 77.0, -f32::INFINITY),
        (-10.0, 77.0, -f32::INFINITY),
        (-3.0, 77.0, -5.474_401e36),
        (-2.0, 77.0, -1.511_157_3e23),
        (-1.0, 77.0, -1.0),
        (1.0, 77.0, 1.0),
        (2.0, 77.0, 1.511_157_3e23),
        (3.0, 77.0, 5.474_401e36),
        (10.0, 77.0, f32::INFINITY),
        (100000.0, 77.0, f32::INFINITY),
        (-100000.0, 78.0, f32::INFINITY),
        (-10.0, 78.0, f32::INFINITY),
        (-3.0, 78.0, 1.642_320_3e37),
        (-2.0, 78.0, 3.022_314_5e23),
        (-1.0, 78.0, 1.0),
        (1.0, 78.0, 1.0),
        (2.0, 78.0, 3.022_314_5e23),
        (3.0, 78.0, 1.642_320_3e37),
        (10.0, 78.0, f32::INFINITY),
        (100000.0, 78.0, f32::INFINITY),
        (-100000.0, 79.0, -f32::INFINITY),
        (-10.0, 79.0, -f32::INFINITY),
        (-3.0, 79.0, -4.926_961e37),
        (-2.0, 79.0, -6.044_629e23),
        (-1.0, 79.0, -1.0),
        (1.0, 79.0, 1.0),
        (2.0, 79.0, 6.044_629e23),
        (3.0, 79.0, 4.926_961e37),
        (10.0, 79.0, f32::INFINITY),
        (100000.0, 79.0, f32::INFINITY),
        (-100000.0, 80.0, f32::INFINITY),
        (-10.0, 80.0, f32::INFINITY),
        (-3.0, 80.0, 1.478_088_3e38),
        (-2.0, 80.0, 1.208_925_8e24),
        (-1.0, 80.0, 1.0),
        (1.0, 80.0, 1.0),
        (2.0, 80.0, 1.208_925_8e24),
        (3.0, 80.0, 1.478_088_3e38),
        (10.0, 80.0, f32::INFINITY),
        (100000.0, 80.0, f32::INFINITY),
        (-100000.0, 81.0, -f32::INFINITY),
        (-10.0, 81.0, -f32::INFINITY),
        (-3.0, 81.0, -f32::INFINITY),
        (-2.0, 81.0, -2.417_851_6e24),
        (-1.0, 81.0, -1.0),
        (1.0, 81.0, 1.0),
        (2.0, 81.0, 2.417_851_6e24),
        (3.0, 81.0, f32::INFINITY),
        (10.0, 81.0, f32::INFINITY),
        (100000.0, 81.0, f32::INFINITY),
        (-100000.0, 82.0, f32::INFINITY),
        (-10.0, 82.0, f32::INFINITY),
        (-3.0, 82.0, f32::INFINITY),
        (-2.0, 82.0, 4.835_703_3e24),
        (-1.0, 82.0, 1.0),
        (1.0, 82.0, 1.0),
        (2.0, 82.0, 4.835_703_3e24),
        (3.0, 82.0, f32::INFINITY),
        (10.0, 82.0, f32::INFINITY),
        (100000.0, 82.0, f32::INFINITY),
        (-100000.0, 83.0, -f32::INFINITY),
        (-10.0, 83.0, -f32::INFINITY),
        (-3.0, 83.0, -f32::INFINITY),
        (-2.0, 83.0, -9.671_407e24),
        (-1.0, 83.0, -1.0),
        (1.0, 83.0, 1.0),
        (2.0, 83.0, 9.671_407e24),
        (3.0, 83.0, f32::INFINITY),
        (10.0, 83.0, f32::INFINITY),
        (100000.0, 83.0, f32::INFINITY),
        (-100000.0, 84.0, f32::INFINITY),
        (-10.0, 84.0, f32::INFINITY),
        (-3.0, 84.0, f32::INFINITY),
        (-2.0, 84.0, 1.934_281_3e25),
        (-1.0, 84.0, 1.0),
        (1.0, 84.0, 1.0),
        (2.0, 84.0, 1.934_281_3e25),
        (3.0, 84.0, f32::INFINITY),
        (10.0, 84.0, f32::INFINITY),
        (100000.0, 84.0, f32::INFINITY),
        (-100000.0, 85.0, -f32::INFINITY),
        (-10.0, 85.0, -f32::INFINITY),
        (-3.0, 85.0, -f32::INFINITY),
        (-2.0, 85.0, -3.868_562_6e25),
        (-1.0, 85.0, -1.0),
        (1.0, 85.0, 1.0),
        (2.0, 85.0, 3.868_562_6e25),
        (3.0, 85.0, f32::INFINITY),
        (10.0, 85.0, f32::INFINITY),
        (100000.0, 85.0, f32::INFINITY),
        (-100000.0, 86.0, f32::INFINITY),
        (-10.0, 86.0, f32::INFINITY),
        (-3.0, 86.0, f32::INFINITY),
        (-2.0, 86.0, 7.737_125_2e25),
        (-1.0, 86.0, 1.0),
        (1.0, 86.0, 1.0),
        (2.0, 86.0, 7.737_125_2e25),
        (3.0, 86.0, f32::INFINITY),
        (10.0, 86.0, f32::INFINITY),
        (100000.0, 86.0, f32::INFINITY),
        (-100000.0, 87.0, -f32::INFINITY),
        (-10.0, 87.0, -f32::INFINITY),
        (-3.0, 87.0, -f32::INFINITY),
        (-2.0, 87.0, -1.547_425_1e26),
        (-1.0, 87.0, -1.0),
        (1.0, 87.0, 1.0),
        (2.0, 87.0, 1.547_425_1e26),
        (3.0, 87.0, f32::INFINITY),
        (10.0, 87.0, f32::INFINITY),
        (100000.0, 87.0, f32::INFINITY),
        (-100000.0, 88.0, f32::INFINITY),
        (-10.0, 88.0, f32::INFINITY),
        (-3.0, 88.0, f32::INFINITY),
        (-2.0, 88.0, 3.094_850_1e26),
        (-1.0, 88.0, 1.0),
        (1.0, 88.0, 1.0),
        (2.0, 88.0, 3.094_850_1e26),
        (3.0, 88.0, f32::INFINITY),
        (10.0, 88.0, f32::INFINITY),
        (100000.0, 88.0, f32::INFINITY),
        (-100000.0, 89.0, -f32::INFINITY),
        (-10.0, 89.0, -f32::INFINITY),
        (-3.0, 89.0, -f32::INFINITY),
        (-2.0, 89.0, -6.189_700_2e26),
        (-1.0, 89.0, -1.0),
        (1.0, 89.0, 1.0),
        (2.0, 89.0, 6.189_700_2e26),
        (3.0, 89.0, f32::INFINITY),
        (10.0, 89.0, f32::INFINITY),
        (100000.0, 89.0, f32::INFINITY),
        (-100000.0, 90.0, f32::INFINITY),
        (-10.0, 90.0, f32::INFINITY),
        (-3.0, 90.0, f32::INFINITY),
        (-2.0, 90.0, 1.237_940_1e27),
        (-1.0, 90.0, 1.0),
        (1.0, 90.0, 1.0),
        (2.0, 90.0, 1.237_940_1e27),
        (3.0, 90.0, f32::INFINITY),
        (10.0, 90.0, f32::INFINITY),
        (100000.0, 90.0, f32::INFINITY),
        (-100000.0, 91.0, -f32::INFINITY),
        (-10.0, 91.0, -f32::INFINITY),
        (-3.0, 91.0, -f32::INFINITY),
        (-2.0, 91.0, -2.475_880_1e27),
        (-1.0, 91.0, -1.0),
        (1.0, 91.0, 1.0),
        (2.0, 91.0, 2.475_880_1e27),
        (3.0, 91.0, f32::INFINITY),
        (10.0, 91.0, f32::INFINITY),
        (100000.0, 91.0, f32::INFINITY),
        (-100000.0, 92.0, f32::INFINITY),
        (-10.0, 92.0, f32::INFINITY),
        (-3.0, 92.0, f32::INFINITY),
        (-2.0, 92.0, 4.951_760_2e27),
        (-1.0, 92.0, 1.0),
        (1.0, 92.0, 1.0),
        (2.0, 92.0, 4.951_760_2e27),
        (3.0, 92.0, f32::INFINITY),
        (10.0, 92.0, f32::INFINITY),
        (100000.0, 92.0, f32::INFINITY),
        (-100000.0, 93.0, -f32::INFINITY),
        (-10.0, 93.0, -f32::INFINITY),
        (-3.0, 93.0, -f32::INFINITY),
        (-2.0, 93.0, -9.903_520_3e27),
        (-1.0, 93.0, -1.0),
        (1.0, 93.0, 1.0),
        (2.0, 93.0, 9.903_520_3e27),
        (3.0, 93.0, f32::INFINITY),
        (10.0, 93.0, f32::INFINITY),
        (100000.0, 93.0, f32::INFINITY),
        (-100000.0, 94.0, f32::INFINITY),
        (-10.0, 94.0, f32::INFINITY),
        (-3.0, 94.0, f32::INFINITY),
        (-2.0, 94.0, 1.980_704_1e28),
        (-1.0, 94.0, 1.0),
        (1.0, 94.0, 1.0),
        (2.0, 94.0, 1.980_704_1e28),
        (3.0, 94.0, f32::INFINITY),
        (10.0, 94.0, f32::INFINITY),
        (100000.0, 94.0, f32::INFINITY),
        (-100000.0, 95.0, -f32::INFINITY),
        (-10.0, 95.0, -f32::INFINITY),
        (-3.0, 95.0, -f32::INFINITY),
        (-2.0, 95.0, -3.961_408_1e28),
        (-1.0, 95.0, -1.0),
        (1.0, 95.0, 1.0),
        (2.0, 95.0, 3.961_408_1e28),
        (3.0, 95.0, f32::INFINITY),
        (10.0, 95.0, f32::INFINITY),
        (100000.0, 95.0, f32::INFINITY),
        (-100000.0, 96.0, f32::INFINITY),
        (-10.0, 96.0, f32::INFINITY),
        (-3.0, 96.0, f32::INFINITY),
        (-2.0, 96.0, 7.922_816_3e28),
        (-1.0, 96.0, 1.0),
        (1.0, 96.0, 1.0),
        (2.0, 96.0, 7.922_816_3e28),
        (3.0, 96.0, f32::INFINITY),
        (10.0, 96.0, f32::INFINITY),
        (100000.0, 96.0, f32::INFINITY),
        (-100000.0, 97.0, -f32::INFINITY),
        (-10.0, 97.0, -f32::INFINITY),
        (-3.0, 97.0, -f32::INFINITY),
        (-2.0, 97.0, -1.584_563_3e29),
        (-1.0, 97.0, -1.0),
        (1.0, 97.0, 1.0),
        (2.0, 97.0, 1.584_563_3e29),
        (3.0, 97.0, f32::INFINITY),
        (10.0, 97.0, f32::INFINITY),
        (100000.0, 97.0, f32::INFINITY),
        (-100000.0, 98.0, f32::INFINITY),
        (-10.0, 98.0, f32::INFINITY),
        (-3.0, 98.0, f32::INFINITY),
        (-2.0, 98.0, 3.169_126_5e29),
        (-1.0, 98.0, 1.0),
        (1.0, 98.0, 1.0),
        (2.0, 98.0, 3.169_126_5e29),
        (3.0, 98.0, f32::INFINITY),
        (10.0, 98.0, f32::INFINITY),
        (100000.0, 98.0, f32::INFINITY),
        (-100000.0, 99.0, -f32::INFINITY),
        (-10.0, 99.0, -f32::INFINITY),
        (-3.0, 99.0, -f32::INFINITY),
        (-2.0, 99.0, -6.338_253e29),
        (-1.0, 99.0, -1.0),
        (1.0, 99.0, 1.0),
        (2.0, 99.0, 6.338_253e29),
        (3.0, 99.0, f32::INFINITY),
        (10.0, 99.0, f32::INFINITY),
        (100000.0, 99.0, f32::INFINITY),
        (-100000.0, 100.0, f32::INFINITY),
        (-10.0, 100.0, f32::INFINITY),
        (-3.0, 100.0, f32::INFINITY),
        (-2.0, 100.0, 1.267_650_6e30),
        (-1.0, 100.0, 1.0),
        (1.0, 100.0, 1.0),
        (2.0, 100.0, 1.267_650_6e30),
        (3.0, 100.0, f32::INFINITY),
        (10.0, 100.0, f32::INFINITY),
        (100000.0, 100.0, f32::INFINITY),
        (-100000.0, 101.0, -f32::INFINITY),
        (-10.0, 101.0, -f32::INFINITY),
        (-3.0, 101.0, -f32::INFINITY),
        (-2.0, 101.0, -2.535_301_2e30),
        (-1.0, 101.0, -1.0),
        (1.0, 101.0, 1.0),
        (2.0, 101.0, 2.535_301_2e30),
        (3.0, 101.0, f32::INFINITY),
        (10.0, 101.0, f32::INFINITY),
        (100000.0, 101.0, f32::INFINITY),
        (-100000.0, 102.0, f32::INFINITY),
        (-10.0, 102.0, f32::INFINITY),
        (-3.0, 102.0, f32::INFINITY),
        (-2.0, 102.0, 5.070_602_4e30),
        (-1.0, 102.0, 1.0),
        (1.0, 102.0, 1.0),
        (2.0, 102.0, 5.070_602_4e30),
        (3.0, 102.0, f32::INFINITY),
        (10.0, 102.0, f32::INFINITY),
        (100000.0, 102.0, f32::INFINITY),
        (-100000.0, 103.0, -f32::INFINITY),
        (-10.0, 103.0, -f32::INFINITY),
        (-3.0, 103.0, -f32::INFINITY),
        (-2.0, 103.0, -1.014_120_5e31),
        (-1.0, 103.0, -1.0),
        (1.0, 103.0, 1.0),
        (2.0, 103.0, 1.014_120_5e31),
        (3.0, 103.0, f32::INFINITY),
        (10.0, 103.0, f32::INFINITY),
        (100000.0, 103.0, f32::INFINITY),
        (-100000.0, 104.0, f32::INFINITY),
        (-10.0, 104.0, f32::INFINITY),
        (-3.0, 104.0, f32::INFINITY),
        (-2.0, 104.0, 2.028_241e31),
        (-1.0, 104.0, 1.0),
        (1.0, 104.0, 1.0),
        (2.0, 104.0, 2.028_241e31),
        (3.0, 104.0, f32::INFINITY),
        (10.0, 104.0, f32::INFINITY),
        (100000.0, 104.0, f32::INFINITY),
        (-100000.0, 105.0, -f32::INFINITY),
        (-10.0, 105.0, -f32::INFINITY),
        (-3.0, 105.0, -f32::INFINITY),
        (-2.0, 105.0, -4.056_482e31),
        (-1.0, 105.0, -1.0),
        (1.0, 105.0, 1.0),
        (2.0, 105.0, 4.056_482e31),
        (3.0, 105.0, f32::INFINITY),
        (10.0, 105.0, f32::INFINITY),
        (100000.0, 105.0, f32::INFINITY),
        (-100000.0, 106.0, f32::INFINITY),
        (-10.0, 106.0, f32::INFINITY),
        (-3.0, 106.0, f32::INFINITY),
        (-2.0, 106.0, 8.112_964e31),
        (-1.0, 106.0, 1.0),
        (1.0, 106.0, 1.0),
        (2.0, 106.0, 8.112_964e31),
        (3.0, 106.0, f32::INFINITY),
        (10.0, 106.0, f32::INFINITY),
        (100000.0, 106.0, f32::INFINITY),
        (-100000.0, 107.0, -f32::INFINITY),
        (-10.0, 107.0, -f32::INFINITY),
        (-3.0, 107.0, -f32::INFINITY),
        (-2.0, 107.0, -1.622_592_8e32),
        (-1.0, 107.0, -1.0),
        (1.0, 107.0, 1.0),
        (2.0, 107.0, 1.622_592_8e32),
        (3.0, 107.0, f32::INFINITY),
        (10.0, 107.0, f32::INFINITY),
        (100000.0, 107.0, f32::INFINITY),
        (-100000.0, 108.0, f32::INFINITY),
        (-10.0, 108.0, f32::INFINITY),
        (-3.0, 108.0, f32::INFINITY),
        (-2.0, 108.0, 3.245_185_5e32),
        (-1.0, 108.0, 1.0),
        (1.0, 108.0, 1.0),
        (2.0, 108.0, 3.245_185_5e32),
        (3.0, 108.0, f32::INFINITY),
        (10.0, 108.0, f32::INFINITY),
        (100000.0, 108.0, f32::INFINITY),
        (-100000.0, 109.0, -f32::INFINITY),
        (-10.0, 109.0, -f32::INFINITY),
        (-3.0, 109.0, -f32::INFINITY),
        (-2.0, 109.0, -6.490_371e32),
        (-1.0, 109.0, -1.0),
        (1.0, 109.0, 1.0),
        (2.0, 109.0, 6.490_371e32),
        (3.0, 109.0, f32::INFINITY),
        (10.0, 109.0, f32::INFINITY),
        (100000.0, 109.0, f32::INFINITY),
        (-100000.0, 110.0, f32::INFINITY),
        (-10.0, 110.0, f32::INFINITY),
        (-3.0, 110.0, f32::INFINITY),
        (-2.0, 110.0, 1.298_074_2e33),
        (-1.0, 110.0, 1.0),
        (1.0, 110.0, 1.0),
        (2.0, 110.0, 1.298_074_2e33),
        (3.0, 110.0, f32::INFINITY),
        (10.0, 110.0, f32::INFINITY),
        (100000.0, 110.0, f32::INFINITY),
        (-100000.0, 111.0, -f32::INFINITY),
        (-10.0, 111.0, -f32::INFINITY),
        (-3.0, 111.0, -f32::INFINITY),
        (-2.0, 111.0, -2.596_148_4e33),
        (-1.0, 111.0, -1.0),
        (1.0, 111.0, 1.0),
        (2.0, 111.0, 2.596_148_4e33),
        (3.0, 111.0, f32::INFINITY),
        (10.0, 111.0, f32::INFINITY),
        (100000.0, 111.0, f32::INFINITY),
        (-100000.0, 112.0, f32::INFINITY),
        (-10.0, 112.0, f32::INFINITY),
        (-3.0, 112.0, f32::INFINITY),
        (-2.0, 112.0, 5.192_297e33),
        (-1.0, 112.0, 1.0),
        (1.0, 112.0, 1.0),
        (2.0, 112.0, 5.192_297e33),
        (3.0, 112.0, f32::INFINITY),
        (10.0, 112.0, f32::INFINITY),
        (100000.0, 112.0, f32::INFINITY),
        (-100000.0, 113.0, -f32::INFINITY),
        (-10.0, 113.0, -f32::INFINITY),
        (-3.0, 113.0, -f32::INFINITY),
        (-2.0, 113.0, -1.038_459_4e34),
        (-1.0, 113.0, -1.0),
        (1.0, 113.0, 1.0),
        (2.0, 113.0, 1.038_459_4e34),
        (3.0, 113.0, f32::INFINITY),
        (10.0, 113.0, f32::INFINITY),
        (100000.0, 113.0, f32::INFINITY),
        (-100000.0, 114.0, f32::INFINITY),
        (-10.0, 114.0, f32::INFINITY),
        (-3.0, 114.0, f32::INFINITY),
        (-2.0, 114.0, 2.076_918_7e34),
        (-1.0, 114.0, 1.0),
        (1.0, 114.0, 1.0),
        (2.0, 114.0, 2.076_918_7e34),
        (3.0, 114.0, f32::INFINITY),
        (10.0, 114.0, f32::INFINITY),
        (100000.0, 114.0, f32::INFINITY),
        (-100000.0, 115.0, -f32::INFINITY),
        (-10.0, 115.0, -f32::INFINITY),
        (-3.0, 115.0, -f32::INFINITY),
        (-2.0, 115.0, -4.153_837_5e34),
        (-1.0, 115.0, -1.0),
        (1.0, 115.0, 1.0),
        (2.0, 115.0, 4.153_837_5e34),
        (3.0, 115.0, f32::INFINITY),
        (10.0, 115.0, f32::INFINITY),
        (100000.0, 115.0, f32::INFINITY),
        (-100000.0, 116.0, f32::INFINITY),
        (-10.0, 116.0, f32::INFINITY),
        (-3.0, 116.0, f32::INFINITY),
        (-2.0, 116.0, 8.307_675e34),
        (-1.0, 116.0, 1.0),
        (1.0, 116.0, 1.0),
        (2.0, 116.0, 8.307_675e34),
        (3.0, 116.0, f32::INFINITY),
        (10.0, 116.0, f32::INFINITY),
        (100000.0, 116.0, f32::INFINITY),
        (-100000.0, 117.0, -f32::INFINITY),
        (-10.0, 117.0, -f32::INFINITY),
        (-3.0, 117.0, -f32::INFINITY),
        (-2.0, 117.0, -1.661_535e35),
        (-1.0, 117.0, -1.0),
        (1.0, 117.0, 1.0),
        (2.0, 117.0, 1.661_535e35),
        (3.0, 117.0, f32::INFINITY),
        (10.0, 117.0, f32::INFINITY),
        (100000.0, 117.0, f32::INFINITY),
        (-100000.0, 118.0, f32::INFINITY),
        (-10.0, 118.0, f32::INFINITY),
        (-3.0, 118.0, f32::INFINITY),
        (-2.0, 118.0, 3.32307e+35),
        (-1.0, 118.0, 1.0),
        (1.0, 118.0, 1.0),
        (2.0, 118.0, 3.32307e+35),
        (3.0, 118.0, f32::INFINITY),
        (10.0, 118.0, f32::INFINITY),
        (100000.0, 118.0, f32::INFINITY),
        (-100000.0, 119.0, -f32::INFINITY),
        (-10.0, 119.0, -f32::INFINITY),
        (-3.0, 119.0, -f32::INFINITY),
        (-2.0, 119.0, -6.64614e+35),
        (-1.0, 119.0, -1.0),
        (1.0, 119.0, 1.0),
        (2.0, 119.0, 6.64614e+35),
        (3.0, 119.0, f32::INFINITY),
        (10.0, 119.0, f32::INFINITY),
        (100000.0, 119.0, f32::INFINITY),
        (-100000.0, 120.0, f32::INFINITY),
        (-10.0, 120.0, f32::INFINITY),
        (-3.0, 120.0, f32::INFINITY),
        (-2.0, 120.0, 1.329_228e36),
        (-1.0, 120.0, 1.0),
        (1.0, 120.0, 1.0),
        (2.0, 120.0, 1.329_228e36),
        (3.0, 120.0, f32::INFINITY),
        (10.0, 120.0, f32::INFINITY),
        (100000.0, 120.0, f32::INFINITY),
        (-100000.0, 121.0, -f32::INFINITY),
        (-10.0, 121.0, -f32::INFINITY),
        (-3.0, 121.0, -f32::INFINITY),
        (-2.0, 121.0, -2.658_456e36),
        (-1.0, 121.0, -1.0),
        (1.0, 121.0, 1.0),
        (2.0, 121.0, 2.658_456e36),
        (3.0, 121.0, f32::INFINITY),
        (10.0, 121.0, f32::INFINITY),
        (100000.0, 121.0, f32::INFINITY),
        (-100000.0, 122.0, f32::INFINITY),
        (-10.0, 122.0, f32::INFINITY),
        (-3.0, 122.0, f32::INFINITY),
        (-2.0, 122.0, 5.316_912e36),
        (-1.0, 122.0, 1.0),
        (1.0, 122.0, 1.0),
        (2.0, 122.0, 5.316_912e36),
        (3.0, 122.0, f32::INFINITY),
        (10.0, 122.0, f32::INFINITY),
        (100000.0, 122.0, f32::INFINITY),
        (-100000.0, 123.0, -f32::INFINITY),
        (-10.0, 123.0, -f32::INFINITY),
        (-3.0, 123.0, -f32::INFINITY),
        (-2.0, 123.0, -1.063_382_4e37),
        (-1.0, 123.0, -1.0),
        (1.0, 123.0, 1.0),
        (2.0, 123.0, 1.063_382_4e37),
        (3.0, 123.0, f32::INFINITY),
        (10.0, 123.0, f32::INFINITY),
        (100000.0, 123.0, f32::INFINITY),
        (-100000.0, 124.0, f32::INFINITY),
        (-10.0, 124.0, f32::INFINITY),
        (-3.0, 124.0, f32::INFINITY),
        (-2.0, 124.0, 2.126_764_8e37),
        (-1.0, 124.0, 1.0),
        (1.0, 124.0, 1.0),
        (2.0, 124.0, 2.126_764_8e37),
        (3.0, 124.0, f32::INFINITY),
        (10.0, 124.0, f32::INFINITY),
        (100000.0, 124.0, f32::INFINITY),
        (-100000.0, 125.0, -f32::INFINITY),
        (-10.0, 125.0, -f32::INFINITY),
        (-3.0, 125.0, -f32::INFINITY),
        (-2.0, 125.0, -4.253_529_6e37),
        (-1.0, 125.0, -1.0),
        (1.0, 125.0, 1.0),
        (2.0, 125.0, 4.253_529_6e37),
        (3.0, 125.0, f32::INFINITY),
        (10.0, 125.0, f32::INFINITY),
        (100000.0, 125.0, f32::INFINITY),
        (-100000.0, 126.0, f32::INFINITY),
        (-10.0, 126.0, f32::INFINITY),
        (-3.0, 126.0, f32::INFINITY),
        (-2.0, 126.0, 8.507_059e37),
        (-1.0, 126.0, 1.0),
        (1.0, 126.0, 1.0),
        (2.0, 126.0, 8.507_059e37),
        (3.0, 126.0, f32::INFINITY),
        (10.0, 126.0, f32::INFINITY),
        (100000.0, 126.0, f32::INFINITY),
        (-100000.0, 127.0, -f32::INFINITY),
        (-10.0, 127.0, -f32::INFINITY),
        (-3.0, 127.0, -f32::INFINITY),
        (-2.0, 127.0, -1.701_411_8e38),
        (-1.0, 127.0, -1.0),
        (1.0, 127.0, 1.0),
        (2.0, 127.0, 1.701_411_8e38),
        (3.0, 127.0, f32::INFINITY),
        (10.0, 127.0, f32::INFINITY),
        (100000.0, 127.0, f32::INFINITY),
        (-100000.0, 128.0, f32::INFINITY),
        (-10.0, 128.0, f32::INFINITY),
        (-3.0, 128.0, f32::INFINITY),
        (-2.0, 128.0, f32::INFINITY),
        (-1.0, 128.0, 1.0),
        (1.0, 128.0, 1.0),
        (2.0, 128.0, f32::INFINITY),
        (3.0, 128.0, f32::INFINITY),
        (10.0, 128.0, f32::INFINITY),
        (100000.0, 128.0, f32::INFINITY),
        (0.0, 3.0, 0.0),
        (-0.0, 3.0, 0.0),
        (0.0, -3.0, f32::INFINITY),
        (-0.0, -3.0, -f32::INFINITY),
        (-0.0, -2.0, f32::INFINITY),
        (0.75, -150.0, 5.505_674e18),
        (-0.75, -150.0, 5.505_674e18),
        (1.1, -150.0, 6.181_527_4e-7),
        (1.1, 150.0, 1617723.1),
    ];

    fn calc_relative_error(experimental: F32, expected: f32) -> F32 {
        if experimental.is_nan() && expected.is_nan() || experimental.0 == expected {
            F32::ZERO
        } else if expected != 0.0 {
            (experimental - expected).abs() / expected
        } else {
            (experimental - expected).abs() / (expected + 1.0e-20)
        }
    }

    #[test]
    fn sanity_check() {
        for &(base_input, power_input, expected) in TEST_VECTORS {
            let exp_x = F32(base_input).powi(power_input as i32);
            let relative_error = calc_relative_error(exp_x, expected);

            assert!(
                relative_error <= MAX_ERROR,
                "relative_error {} too large for input {}.powi({}) : {} vs {}",
                relative_error,
                base_input,
                power_input,
                exp_x,
                expected
            );
        }
    }
}
