/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <stdlib.h>

#include <glib.h>

#include <libraw1394/raw1394.h>

#include <librbc/rbcdisk.h>

#include <libsbp2/sbp2login.h>
#include <libsbp2/sbp2worker.h>

#include "app.h"
#include "worker.h"


typedef struct _WorkerState WorkerState;

struct _WorkerState
{
  EndpointApp *app;

  RBCDisk     *disk;
  SBP2Worker  *worker;

  GIOChannel  *reader;
  GIOChannel  *writer;
};


static gboolean reconnect (EndpointApp   *app,
                           WireReconnect *message);


static WireMsgFuncs wire_funcs = {
  .reconnect = reconnect
};

void
worker_process (EndpointApp *app,
                Process     *process,
                WireLogin   *message)
{
  WorkerState      *state;
  RBCDisk          *disk;
  SBP2Worker       *worker;
  WireWorkerActive *ret_message;

  g_return_if_fail (app != NULL);
  g_return_if_fail (process != NULL);

  disk = g_hash_table_lookup (app->disks, GINT_TO_POINTER (message->orb.lun));

  if (!rbc_disk_activate (disk))
    exit (1);

  worker = sbp2_worker_new (disk, app->context,
                            message->port, message->login_ID);

  if (!worker)
    exit (2);

  if (!sbp2_worker_login (worker, message->node, &message->pointer,
	                  &message->orb))
    exit (3);

  app->wire_funcs = &wire_funcs;

  state = g_new (WorkerState, 1);

  state->app = app;

  state->disk = disk;
  state->worker = worker;

  state->reader = process->reader;
  state->writer = process->writer;

  app->user_data = state;

  ret_message = wire_message_worker_active (message->login_ID);
  wire_message_send (state->writer, ret_message, app);
}

static gboolean
reconnect (EndpointApp   *app,
	   WireReconnect *message)
{
  WorkerState      *state;
  WireWorkerActive *ret_message;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->user_data != NULL, FALSE);
  g_return_val_if_fail (message != NULL, FALSE);

  state = app->user_data;

  if (sbp2_worker_reconnect (state->worker, message->node))
    {
      ret_message = wire_message_worker_active (message->login_ID);
      wire_message_send (state->writer, ret_message, app);
    }
  else
    exit (4);

  return TRUE;
}
