// @HEADER
//
// ***********************************************************************
//
//             Xpetra: A linear algebra interface package
//                  Copyright 2012 Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact
//                    Jonathan Hu       (jhu@sandia.gov)
//                    Andrey Prokopenko (aprokop@sandia.gov)
//                    Ray Tuminaro      (rstumin@sandia.gov)
//
// ***********************************************************************
//
// @HEADER
#ifndef XPETRA_EPETRACRSGRAPH_HPP
#define XPETRA_EPETRACRSGRAPH_HPP

/* this file is automatically generated - do not edit (see script/epetra.py) */

#include "Xpetra_EpetraConfigDefs.hpp"
#include "Xpetra_Exceptions.hpp"

#include "Xpetra_CrsGraph.hpp"

#include "Xpetra_EpetraExport.hpp"
#include "Xpetra_EpetraImport.hpp"
#include "Xpetra_EpetraMap.hpp"
#include "Xpetra_EpetraUtils.hpp"

#include <Epetra_CrsGraph.h>

namespace Xpetra {

// TODO: move that elsewhere
template <class GlobalOrdinal, class Node>
RCP<const CrsGraph<int, GlobalOrdinal, Node>>
toXpetra(const Epetra_CrsGraph &graph);

template <class GlobalOrdinal, class Node>
const Epetra_CrsGraph &
toEpetra(const RCP<const CrsGraph<int, GlobalOrdinal, Node>> &graph);

template <class GlobalOrdinal, class Node>
class EpetraCrsGraphT
  : public CrsGraph<int, GlobalOrdinal, Node> {
  typedef int LocalOrdinal;

  //! The specialization of Map used by this class.
  typedef Map<LocalOrdinal, GlobalOrdinal, Node> map_type;

#ifdef HAVE_XPETRA_TPETRA
  typedef typename Xpetra::CrsGraph<LocalOrdinal, GlobalOrdinal, Node>::local_graph_type local_graph_type;
  typedef typename Xpetra::CrsGraph<LocalOrdinal, GlobalOrdinal, Node>::node_type node_type;
  typedef typename node_type::execution_space execution_space;
#endif

 public:
  //! @name Constructor/Destructor Methods
  //@{
  //! Constructor for empty DynamicProfile graph (no space is preallocated).
  EpetraCrsGraphT(const RCP<const map_type> &rowMap, const RCP<Teuchos::ParameterList> &plist = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::RuntimeError,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
  }

  //! Constructor specifying fixed number of entries for each row.
  EpetraCrsGraphT(const RCP<const map_type> &rowMap, size_t maxNumEntriesPerRow, const RCP<Teuchos::ParameterList> &plist = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::RuntimeError,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
  }

  //! Constructor specifying column Map and fixed number of entries for each row.
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap, const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &colMap, size_t maxNumEntriesPerRow, const Teuchos::RCP<Teuchos::ParameterList> &plist = null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::RuntimeError,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
  }

  ////! Constructor specifying column Map and number of entries in each row.
  // Definition not in cpp, so comment out
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap, const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &colMap, const ArrayRCP<const size_t> &NumEntriesPerRowToAlloc, const RCP<ParameterList> &params = null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::RuntimeError,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
  }

  // Constructor for fused import
  EpetraCrsGraphT(const RCP<const CrsGraph<LocalOrdinal, GlobalOrdinal, Node>> &sourceGraph,
                  const Import<LocalOrdinal, GlobalOrdinal, Node> &importer,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &domainMap = Teuchos::null,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rangeMap  = Teuchos::null,
                  const RCP<Teuchos::ParameterList> &params                          = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::RuntimeError,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
  }

#ifdef HAVE_XPETRA_TPETRA
  //! Constructor specifying column Map, number of entries in each row and column indices in each row.
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &colMap,
                  const typename local_graph_type::row_map_type &rowPointers,
                  const typename local_graph_type::entries_type::non_const_type &columnIndices,
                  const RCP<Teuchos::ParameterList> &plist = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
  }

  //! Constructor specifying column Map, number of entries in each row and column indices in each row.
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &colMap,
                  const local_graph_type &lclGraph,
                  const RCP<Teuchos::ParameterList> &plist = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
  }

  //! Constructor specifying column Map, number of entries in each row and column indices in each row.
  EpetraCrsGraphT(const local_graph_type &lclGraph,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &colMap,
                  const Teuchos::RCP<const map_type> &domainMap = Teuchos::null,
                  const Teuchos::RCP<const map_type> &rangeMap  = Teuchos::null,
                  const RCP<Teuchos::ParameterList> &plist      = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
  }
#endif

  //! Destructor.
  virtual ~EpetraCrsGraphT() {}

  //@}

  //! @name Insertion/Removal Methods
  //@{

  //! Insert global indices into the graph.
  void insertGlobalIndices(GlobalOrdinal globalRow, const ArrayView<const GlobalOrdinal> &indices) {}

  //! Insert local indices into the graph.
  void insertLocalIndices(const LocalOrdinal localRow, const ArrayView<const LocalOrdinal> &indices) {}

  //! Remove all graph indices from the specified local row.
  void removeLocalIndices(LocalOrdinal localRow) {}

  //! Allocates the 1D pointer arrays of the graph
  void allocateAllIndices(size_t numNonZeros, ArrayRCP<size_t> &rowptr, ArrayRCP<LocalOrdinal> &colind) {}

  //! Sets the 1D pointer arrays of the graph.
  void setAllIndices(const ArrayRCP<size_t> &rowptr, const ArrayRCP<LocalOrdinal> &colind) {}

  //! Gets the 1D pointer arrays of the graph.
  void getAllIndices(ArrayRCP<const size_t> &rowptr, ArrayRCP<const LocalOrdinal> &colind) const {}

  //@}

  //! @name Transformational Methods
  //@{

  //! Signal that data entry is complete, specifying domain and range maps.
  void fillComplete(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &domainMap, const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rangeMap, const RCP<ParameterList> &params = null) {}

  //! Signal that data entry is complete.
  void fillComplete(const RCP<ParameterList> &params = null) {}

  //! Expert version of fillComplete
  void
  expertStaticFillComplete(const Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &domainMap,
                           const Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rangeMap,
                           const Teuchos::RCP<const Import<LocalOrdinal, GlobalOrdinal, Node>> &importer =
                               Teuchos::null,
                           const Teuchos::RCP<const Export<LocalOrdinal, GlobalOrdinal, Node>> &exporter =
                               Teuchos::null,
                           const Teuchos::RCP<Teuchos::ParameterList> &params =
                               Teuchos::null) {}

  //@}

  //! @name Methods implementing RowGraph.
  //@{

  //! Returns the communicator.
  RCP<const Comm<int>> getComm() const {
    return Teuchos::null;
  }

  //! Returns the Map that describes the row distribution in this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getRowMap() const { return Teuchos::null; }

  //! Returns the Map that describes the column distribution in this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getColMap() const { return Teuchos::null; }

  //! Returns the Map associated with the domain of this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getDomainMap() const { return Teuchos::null; }

  //! Returns the Map associated with the domain of this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getRangeMap() const { return Teuchos::null; }

  //! Returns the importer associated with this graph.
  RCP<const Import<LocalOrdinal, GlobalOrdinal, Node>> getImporter() const { return Teuchos::null; }

  //! Returns the exporter associated with this graph.
  RCP<const Export<LocalOrdinal, GlobalOrdinal, Node>> getExporter() const { return Teuchos::null; }

  //! Returns the number of global rows in the graph.
  global_size_t getGlobalNumRows() const { return 0; }

  //! Returns the number of global columns in the graph.
  global_size_t getGlobalNumCols() const { return 0; }

  //! Returns the number of graph rows owned on the calling node.
  size_t getLocalNumRows() const { return 0; }

  //! Returns the number of columns connected to the locally owned rows of this graph.
  size_t getLocalNumCols() const { return 0; }

  //! Returns the index base for global indices for this graph.
  GlobalOrdinal getIndexBase() const { return 0; }

  //! Returns the global number of entries in the graph.
  global_size_t getGlobalNumEntries() const { return 0; }

  //! Returns the local number of entries in the graph.
  size_t getLocalNumEntries() const { return 0; }

  //! Returns the current number of entries on this node in the specified global row.
  size_t getNumEntriesInGlobalRow(GlobalOrdinal globalRow) const { return 0; }

  //! Returns the current number of entries on this node in the specified local row.
  size_t getNumEntriesInLocalRow(LocalOrdinal localRow) const { return 0; }

  //! Returns the current number of allocated entries for this node in the specified global row .
  size_t getNumAllocatedEntriesInGlobalRow(GlobalOrdinal globalRow) const { return 0; }

  //! Returns the current number of allocated entries on this node in the specified local row.
  size_t getNumAllocatedEntriesInLocalRow(LocalOrdinal localRow) const { return 0; }

  //! Maximum number of entries in all rows over all processes.
  size_t getGlobalMaxNumRowEntries() const { return 0; }

  //! Maximum number of entries in all rows owned by the calling process.
  size_t getLocalMaxNumRowEntries() const { return 0; }

  //! Whether the graph has a column Map.
  bool hasColMap() const { return false; }

  //! Whether column indices are stored using local indices on the calling process.
  bool isLocallyIndexed() const { return false; }

  //! Whether column indices are stored using global indices on the calling process.
  bool isGloballyIndexed() const { return false; }

  //! Whether fillComplete() has been called and the graph is in compute mode.
  bool isFillComplete() const { return false; }

  //! Returns true if storage has been optimized.
  bool isStorageOptimized() const { return false; }

  //! Return a const, nonpersisting view of global indices in the given row.
  void getGlobalRowView(GlobalOrdinal GlobalRow, ArrayView<const GlobalOrdinal> &Indices) const {}

  //! Return a const, nonpersisting view of local indices in the given row.
  void getLocalRowView(LocalOrdinal LocalRow, ArrayView<const LocalOrdinal> &indices) const {}

#ifdef HAVE_XPETRA_TPETRA
  typename local_graph_type::HostMirror getLocalGraphHost() const {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
    TEUCHOS_UNREACHABLE_RETURN((typename local_graph_type::HostMirror()));
  }
#else
#ifdef __GNUC__
#warning "Xpetra Kokkos interface for CrsGraph is enabled (HAVE_XPETRA_KOKKOS_REFACTOR) but Tpetra is disabled. The Kokkos interface needs Tpetra to be enabled, too."
#endif
#endif

#ifdef HAVE_XPETRA_TPETRA
  local_graph_type getLocalGraphDevice() const {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
    TEUCHOS_UNREACHABLE_RETURN((local_graph_type()));
  }

  void getLocalDiagOffsets(const Kokkos::View<size_t *, typename Node::device_type, Kokkos::MemoryUnmanaged> &offsets) const {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::RuntimeError,
                               "Epetra does not support getLocalDiagOffsets!");
  }
#else
#ifdef __GNUC__
#warning "Xpetra Kokkos interface for CrsGraph is enabled (HAVE_XPETRA_KOKKOS_REFACTOR) but Tpetra is disabled. The Kokkos interface needs Tpetra to be enabled, too."
#endif

#endif

  //! Force the computation of global constants if we don't have them
  void computeGlobalConstants() {}

  //@}

  //! @name Overridden from Teuchos::Describable
  //@{

  //! Return a simple one-line description of this object.
  std::string description() const { return std::string(""); }

  //! Print the object with some verbosity level to an FancyOStream object.
  void describe(Teuchos::FancyOStream &out, const Teuchos::EVerbosityLevel verbLevel = Teuchos::Describable::verbLevel_default) const {}

  //@}

  //! Implements DistObject interface
  //{@

  //! Access function for the Tpetra::Map this DistObject was constructed with.
  Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getMap() const { return Teuchos::null; }

  //! Import.
  void doImport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &source,
                const Import<LocalOrdinal, GlobalOrdinal, Node> &importer, CombineMode CM) {}

  //! Export.
  void doExport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &dest,
                const Import<LocalOrdinal, GlobalOrdinal, Node> &importer, CombineMode CM) {}

  //! Import (using an Exporter).
  void doImport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &source,
                const Export<LocalOrdinal, GlobalOrdinal, Node> &exporter, CombineMode CM) {}

  //! Export (using an Importer).
  void doExport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &dest,
                const Export<LocalOrdinal, GlobalOrdinal, Node> &exporter, CombineMode CM) {}
  //@}

  //! @name Xpetra specific
  //@{

  //! EpetraCrsGraphT constructor to wrap a Epetra_CrsGraph object
  EpetraCrsGraphT(const Teuchos::RCP<Epetra_CrsGraph> &graph) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::RuntimeError,
                               "Xpetra::EpetraCrsGraph only available for GO=int or GO=long long with EpetraNode (Serial or OpenMP depending on configuration)");
  }

  //! Get the underlying Epetra graph
  RCP<const Epetra_CrsGraph> getEpetra_CrsGraph() const { return Teuchos::null; }

  //@}

 private:
};  // EpetraCrsGraphT class

// specialization on GO=int and Node=EpetraNode
#ifndef XPETRA_EPETRA_NO_32BIT_GLOBAL_INDICES
template <>
class EpetraCrsGraphT<int, EpetraNode>
  : public virtual CrsGraph<int, int, EpetraNode> {
  typedef int LocalOrdinal;
  typedef int GlobalOrdinal;
  typedef EpetraNode Node;

  //! The specialization of Map used by this class.
  typedef Map<LocalOrdinal, GlobalOrdinal, Node> map_type;

#ifdef HAVE_XPETRA_TPETRA
  typedef typename Xpetra::CrsGraph<LocalOrdinal, GlobalOrdinal, Node>::local_graph_type local_graph_type;
  typedef typename Xpetra::CrsGraph<LocalOrdinal, GlobalOrdinal, Node>::node_type node_type;
  typedef typename node_type::execution_space execution_space;
#endif

 public:
  //! @name Constructor/Destructor Methods
  //@{

  //! Constructor for empty DynamicProfile graph (no space is preallocated).
  EpetraCrsGraphT(const Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap)
    : graph_(Teuchos::rcp(new Epetra_CrsGraph(Copy, toEpetra<GlobalOrdinal, Node>(rowMap), 0, false))) {}

  //! Constructor specifying fixed number of entries for each row.
  EpetraCrsGraphT(const RCP<const map_type> &rowMap, size_t maxNumEntriesPerRow, const RCP<Teuchos::ParameterList> & /* plist */ = Teuchos::null)
    : graph_(Teuchos::rcp(new Epetra_CrsGraph(Copy, toEpetra<GlobalOrdinal, Node>(rowMap), maxNumEntriesPerRow, true))) {}

  ////! Constructor specifying (possibly different) number of entries in each row.
  // Definition not in cpp, so comment out
  // EpetraCrsGraphT(const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > &rowMap, const ArrayRCP< const size_t > &NumEntriesPerRowToAlloc, const RCP< ParameterList > &params=null);

  //! Constructor specifying column Map and fixed number of entries for each row.
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap, const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &colMap, size_t maxNumEntriesPerRow, const Teuchos::RCP<Teuchos::ParameterList> & /* plist */ = null)
    : graph_(Teuchos::rcp(new Epetra_CrsGraph(Copy, toEpetra<GlobalOrdinal, Node>(rowMap), toEpetra<GlobalOrdinal, Node>(colMap), maxNumEntriesPerRow, true))) {}

  ////! Constructor specifying column Map and number of entries in each row.
  // Definition not in cpp, so comment out
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap, const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &colMap, const ArrayRCP<const size_t> &NumEntriesPerRowToAlloc, const RCP<ParameterList> & /* params */ = null) {
    Teuchos::Array<int> numEntriesPerRowToAlloc(NumEntriesPerRowToAlloc.begin(), NumEntriesPerRowToAlloc.end());  // convert array of "size_t" to array of "int"
    graph_ = Teuchos::rcp(new Epetra_CrsGraph(Copy, toEpetra<GlobalOrdinal, Node>(rowMap), toEpetra<GlobalOrdinal, Node>(colMap), numEntriesPerRowToAlloc.getRawPtr(), true));
  }

  // Constructor for fused import
  EpetraCrsGraphT(const RCP<const CrsGraph<LocalOrdinal, GlobalOrdinal, Node>> &sourceGraph,
                  const Import<LocalOrdinal, GlobalOrdinal, Node> &importer,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &domainMap = Teuchos::null,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rangeMap  = Teuchos::null,
                  const RCP<Teuchos::ParameterList> &params                          = Teuchos::null) {
    XPETRA_DYNAMIC_CAST(const EpetraCrsGraphT<GlobalOrdinal XPETRA_COMMA Node>, *sourceGraph, tSourceGraph, "Xpetra::EpetraCrsGraphT() only accepts Xpetra::EpetraCrsGraphT as input arguments.");
    XPETRA_DYNAMIC_CAST(const EpetraImportT<GlobalOrdinal XPETRA_COMMA Node>, importer, tImporter, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraImportT as input arguments.");
    RCP<const Epetra_CrsGraph> eSourceGraph = tSourceGraph.getEpetra_CrsGraph();

    // NOTE: Unlike Tpetra, Epetra does not have a FusedTransfer for Graphs.  So we do this the slow way
    graph_ = Teuchos::rcp(new Epetra_CrsGraph(Copy, eSourceGraph->RowMap(), 0, false));
    graph_->Import(*eSourceGraph, *tImporter.getEpetra_Import(), Insert);

    const Epetra_BlockMap &myDomainMap = domainMap != Teuchos::null ? toEpetra<GlobalOrdinal, Node>(domainMap) : eSourceGraph->ColMap();
    const Epetra_BlockMap &myRangeMap  = rangeMap != Teuchos::null ? toEpetra<GlobalOrdinal, Node>(rangeMap) : toEpetra<LocalOrdinal, Node>(importer.getTargetMap());

    graph_->FillComplete(myDomainMap, myRangeMap);
  }

#ifdef HAVE_XPETRA_TPETRA
  //! Constructor specifying column Map, number of entries in each row and column indices in each row.
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* rowMap */,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* colMap */,
                  const typename local_graph_type::row_map_type & /* rowPointers */,
                  const typename local_graph_type::entries_type::non_const_type & /* columnIndices */,
                  const RCP<Teuchos::ParameterList> & /* plist */ = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Epetra does not support CrsGraph constructors using Kokkos Views!");
  }

  //! Constructor specifying column Map, number of entries in each row and column indices in each row.
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* rowMap */,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* colMap */,
                  const local_graph_type & /* lclGraph */,
                  const RCP<Teuchos::ParameterList> & /* plist */ = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Epetra does not support CrsGraph constructors using a local graph!");
  }

  //! Constructor specifying column Map, number of entries in each row and column indices in each row.
  EpetraCrsGraphT(const local_graph_type & /* lclGraph */,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* rowMap */,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* colMap */,
                  const Teuchos::RCP<const map_type> & /* domainMap */ = Teuchos::null,
                  const Teuchos::RCP<const map_type> & /* rangeMap */  = Teuchos::null,
                  const RCP<Teuchos::ParameterList> & /* plist */      = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Epetra does not support CrsGraph constructors using a local graph!");
  }
#endif

  //! Destructor.
  virtual ~EpetraCrsGraphT() {}

  //@}

  //! @name Insertion/Removal Methods
  //@{

  //! Insert global indices into the graph.
  void insertGlobalIndices(GlobalOrdinal globalRow, const ArrayView<const GlobalOrdinal> &indices) {
    XPETRA_MONITOR("EpetraCrsGraphT::insertGlobalIndices");

    GlobalOrdinal *indices_rawPtr = const_cast<GlobalOrdinal *>(indices.getRawPtr());  // there is no const in the Epetra interface :(
    XPETRA_ERR_CHECK(graph_->InsertGlobalIndices(globalRow, indices.size(), indices_rawPtr));
  }

  //! Insert local indices into the graph.
  void insertLocalIndices(const LocalOrdinal localRow, const ArrayView<const LocalOrdinal> &indices) {
    XPETRA_MONITOR("EpetraCrsGraphT::insertLocalIndices");

    int *indices_rawPtr = const_cast<int *>(indices.getRawPtr());  // there is no const in the Epetra interface :(
    XPETRA_ERR_CHECK(graph_->InsertMyIndices(localRow, indices.size(), indices_rawPtr));
  }

  //! Remove all graph indices from the specified local row.
  void removeLocalIndices(LocalOrdinal localRow) {
    XPETRA_MONITOR("EpetraCrsGraphT::removeLocalIndices");
    graph_->RemoveMyIndices(localRow);
  }

  //! Allocates and returns ArrayRCPs of the Crs arrays --- This is an Xpetra-only routine.
  //** \warning This is an expert-only routine and should not be called from user code. */
  void allocateAllIndices(size_t numNonZeros, ArrayRCP<size_t> &rowptr, ArrayRCP<LocalOrdinal> &colind) {
    XPETRA_MONITOR("EpetraCrsGraphT::allocateAllIndies");

    // Row offsets
    // Unfortunately, we cannot do this in the same manner as column indices
    // and values (see below).  The problem is that Tpetra insists on using
    // size_t, and Epetra uses int internally.  So we only resize here, and
    // will need to copy in setAllValues
    rowptr.resize(getLocalNumRows() + 1);

    int lowerOffset = 0;
    bool ownMemory  = false;

    // Column indices
    // Extract, resize, set colind
    Epetra_IntSerialDenseVector &myColind = graph_->ExpertExtractIndices();
    myColind.Resize(numNonZeros);
    colind = Teuchos::arcp(myColind.Values(), lowerOffset, numNonZeros, ownMemory);
  }

  //! Sets the 1D pointer arrays of the graph.
  void setAllIndices(const ArrayRCP<size_t> &rowptr, const ArrayRCP<LocalOrdinal> &colind) {
    XPETRA_MONITOR("EpetraCrsGraphT::setAllIndices");

    // Check sizes
    TEUCHOS_TEST_FOR_EXCEPTION(Teuchos::as<size_t>(rowptr.size()) != getLocalNumRows() + 1, Xpetra::Exceptions::RuntimeError,
                               "An exception is thrown to let you know that the size of your rowptr array is incorrect.");
    if (colind.size() > 0) {
      TEUCHOS_TEST_FOR_EXCEPTION(colind.getRawPtr() != graph_->ExpertExtractIndices().Values(), Xpetra::Exceptions::RuntimeError,
                                 "An exception is thrown to let you know that you mismatched your pointers.");
    }

    // We have to make a copy here, it is unavoidable
    // See comments in allocateAllIndices
    const size_t N = getLocalNumRows();

    Epetra_IntSerialDenseVector &myRowptr = graph_->ExpertExtractIndexOffset();
    myRowptr.Resize(N + 1);
    for (size_t i = 0; i < N + 1; i++)
      myRowptr[i] = Teuchos::as<int>(rowptr[i]);
  }

  //! Gets the 1D pointer arrays of the graph.
  void getAllIndices(ArrayRCP<const size_t> &rowptr, ArrayRCP<const LocalOrdinal> &colind) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getAllIndices");

    int lowerOffset = 0;
    bool ownMemory  = false;

    const size_t n   = getLocalNumRows();
    const size_t nnz = getLocalNumEntries();

    // Row offsets
    // We have to make a copy here, it is unavoidable (see comments in allocateAllValues)
    Epetra_IntSerialDenseVector &myRowptr = graph_->ExpertExtractIndexOffset();
    rowptr.resize(n + 1);
    for (size_t i = 0; i < n + 1; i++)
      (*const_cast<size_t *>(&rowptr[i])) = Teuchos::as<size_t>(myRowptr[i]);

    // Column indices
    colind = Teuchos::arcp(graph_->ExpertExtractIndices().Values(), lowerOffset, nnz, ownMemory);
  }

  //@}

  //! @name Transformational Methods
  //@{

  //! Signal that data entry is complete, specifying domain and range maps.
  void fillComplete(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &domainMap, const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rangeMap, const RCP<ParameterList> &params = null) {
    XPETRA_MONITOR("EpetraCrsGraphT::fillComplete");

    graph_->FillComplete(toEpetra<GlobalOrdinal, Node>(domainMap), toEpetra<GlobalOrdinal, Node>(rangeMap));
    bool doOptimizeStorage = true;
    if (params != null && params->get("Optimize Storage", true) == false) doOptimizeStorage = false;
    if (doOptimizeStorage) graph_->OptimizeStorage();
  }

  //! Signal that data entry is complete.
  void fillComplete(const RCP<ParameterList> &params = null) {
    XPETRA_MONITOR("EpetraCrsGraphT::fillComplete");

    graph_->FillComplete();
    bool doOptimizeStorage = true;
    if (params != null && params->get("Optimize Storage", true) == false) doOptimizeStorage = false;
    if (doOptimizeStorage) graph_->OptimizeStorage();
  }

  //! Expert version of fillComplete
  void
  expertStaticFillComplete(const Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &domainMap,
                           const Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rangeMap,
                           const Teuchos::RCP<const Import<LocalOrdinal, GlobalOrdinal, Node>> &importer =
                               Teuchos::null,
                           const Teuchos::RCP<const Export<LocalOrdinal, GlobalOrdinal, Node>> &exporter =
                               Teuchos::null,
                           const Teuchos::RCP<Teuchos::ParameterList> &params =
                               Teuchos::null) {
    // Not optimized
    graph_->FillComplete(toEpetra<GlobalOrdinal, Node>(domainMap), toEpetra<GlobalOrdinal, Node>(rangeMap));
    graph_->OptimizeStorage();
  }

  //@}

  //! @name Methods implementing RowGraph.
  //@{

  //! Returns the communicator.
  RCP<const Comm<int>> getComm() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getComm");
    return toXpetra(graph_->Comm());
  }

  //! Returns the Map that describes the row distribution in this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getRowMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getRowMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->RowMap());
  }

  //! Returns the Map that describes the column distribution in this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getColMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getColMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->ColMap());
  }

  //! Returns the Map associated with the domain of this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getDomainMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getDomainMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->DomainMap());
  }

  //! Returns the Map associated with the domain of this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getRangeMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getRangeMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->RangeMap());
  }

  //! Returns the importer associated with this graph.
  RCP<const Import<LocalOrdinal, GlobalOrdinal, Node>> getImporter() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getImporter");
    return toXpetra<GlobalOrdinal, Node>(graph_->Importer());
  }

  //! Returns the exporter associated with this graph.
  RCP<const Export<LocalOrdinal, GlobalOrdinal, Node>> getExporter() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getExporter");
    return toXpetra<GlobalOrdinal, Node>(graph_->Exporter());
  }

  //! Returns the number of global rows in the graph.
  global_size_t getGlobalNumRows() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalNumRows");
    return graph_->NumGlobalRows64();
  }

  //! Returns the number of global columns in the graph.
  global_size_t getGlobalNumCols() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalNumCols");
    return graph_->NumGlobalCols64();
  }

  //! Returns the number of graph rows owned on the calling node.
  size_t getLocalNumRows() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalNumRows");
    return graph_->NumMyRows();
  }

  //! Returns the number of columns connected to the locally owned rows of this graph.
  size_t getLocalNumCols() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalNumCols");
    return graph_->NumMyCols();
  }

  //! Returns the index base for global indices for this graph.
  GlobalOrdinal getIndexBase() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getIndexBase");
    return (GlobalOrdinal)graph_->IndexBase64();
  }

  //! Returns the global number of entries in the graph.
  global_size_t getGlobalNumEntries() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalNumEntries");
    return graph_->NumGlobalEntries64();
  }

  //! Returns the local number of entries in the graph.
  size_t getLocalNumEntries() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalNumEntries");
    return graph_->NumMyEntries();
  }

  //! Returns the current number of entries on this node in the specified global row.
  size_t getNumEntriesInGlobalRow(GlobalOrdinal globalRow) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getNumEntriesInGlobalRow");
    return graph_->NumGlobalIndices(globalRow);
  }

  //! Returns the current number of entries on this node in the specified local row.
  size_t getNumEntriesInLocalRow(LocalOrdinal localRow) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getNumEntriesInLocalRow");
    return graph_->NumMyIndices(localRow);
  }

  //! Returns the current number of allocated entries for this node in the specified global row .
  size_t getNumAllocatedEntriesInGlobalRow(GlobalOrdinal globalRow) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getNumAllocatedEntriesInGlobalRow");
    return graph_->NumAllocatedGlobalIndices(globalRow);
  }

  //! Returns the current number of allocated entries on this node in the specified local row.
  size_t getNumAllocatedEntriesInLocalRow(LocalOrdinal localRow) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getNumAllocatedEntriesInLocalRow");
    return graph_->NumAllocatedMyIndices(localRow);
  }

  //! Maximum number of entries in all rows over all processes.
  size_t getGlobalMaxNumRowEntries() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalMaxNumRowEntries");
    return graph_->GlobalMaxNumIndices();
  }

  //! Maximum number of entries in all rows owned by the calling process.
  size_t getLocalMaxNumRowEntries() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalMaxNumRowEntries");
    return graph_->MaxNumIndices();
  }

  //! Whether the graph has a column Map.
  bool hasColMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::hasColMap");
    return graph_->HaveColMap();
  }

  //! Whether column indices are stored using local indices on the calling process.
  bool isLocallyIndexed() const {
    XPETRA_MONITOR("EpetraCrsGraphT::isLocallyIndexed");
    return graph_->IndicesAreLocal();
  }

  //! Whether column indices are stored using global indices on the calling process.
  bool isGloballyIndexed() const {
    XPETRA_MONITOR("EpetraCrsGraphT::isGloballyIndexed");
    return graph_->IndicesAreGlobal();
  }

  //! Whether fillComplete() has been called and the graph is in compute mode.
  bool isFillComplete() const {
    XPETRA_MONITOR("EpetraCrsGraphT::isFillComplete");
    return graph_->Filled();
  }

  //! Returns true if storage has been optimized.
  bool isStorageOptimized() const {
    XPETRA_MONITOR("EpetraCrsGraphT::isStorageOptimized");
    return graph_->StorageOptimized();
  }

  //! Return a const, nonpersisting view of global indices in the given row.
  void getGlobalRowView(GlobalOrdinal GlobalRow, ArrayView<const GlobalOrdinal> &Indices) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalRowView");

    int numEntries;
    GlobalOrdinal *eIndices;

    XPETRA_ERR_CHECK(graph_->ExtractGlobalRowView(GlobalRow, numEntries, eIndices));
    if (numEntries == 0) {
      eIndices = NULL;
    }  // Cf. TEUCHOS_TEST_FOR_EXCEPT( p == 0 && size_in != 0 ) in Teuchos ArrayView constructor.

    Indices = ArrayView<const GlobalOrdinal>(eIndices, numEntries);
  }

  //! Return a const, nonpersisting view of local indices in the given row.
  void getLocalRowView(LocalOrdinal LocalRow, ArrayView<const LocalOrdinal> &indices) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalRowView");

    int numEntries;
    int *eIndices;

    XPETRA_ERR_CHECK(graph_->ExtractMyRowView(LocalRow, numEntries, eIndices));
    if (numEntries == 0) {
      eIndices = NULL;
    }  // Cf. TEUCHOS_TEST_FOR_EXCEPT( p == 0 && size_in != 0 ) in Teuchos ArrayView constructor.

    indices = ArrayView<const int>(eIndices, numEntries);
  }

#ifdef HAVE_XPETRA_TPETRA
  typename local_graph_type::HostMirror getLocalGraphHost() const {
    RCP<Epetra_CrsGraph> graph = Teuchos::rcp_const_cast<Epetra_CrsGraph>(getEpetra_CrsGraph());

    using local_graph_type_host = typename local_graph_type::HostMirror;

    const int numRows = graph->NumMyRows();
    const int nnz     = graph->NumMyNonzeros();

    TEUCHOS_TEST_FOR_EXCEPTION(!graph->StorageOptimized(), std::runtime_error, "Xpetra::CrsGraph<>::getLocalGraph: Epetra_CrsGraph not StorageOptimized");
    const int *rowptr = graph->ExpertExtractIndexOffset().Values();
    int *colind       = graph->ExpertExtractIndices().Values();

    // Transform int* rowptr array to size_type* array
    typename local_graph_type_host::row_map_type::non_const_type kokkosRowPtr(Kokkos::ViewAllocateWithoutInitializing("local row map"), numRows + 1);
    for (size_t i = 0; i < kokkosRowPtr.size(); i++)
      kokkosRowPtr(i) = Teuchos::asSafe<typename local_graph_type_host::row_map_type::value_type>(rowptr[i]);

    // create Kokkos::Views
    typename local_graph_type_host::entries_type kokkosColind(colind, nnz);

    local_graph_type_host localGraph = local_graph_type_host(kokkosColind, kokkosRowPtr);

    return localGraph;
  }

  local_graph_type getLocalGraphDevice() const {
    RCP<Epetra_CrsGraph> graph = Teuchos::rcp_const_cast<Epetra_CrsGraph>(getEpetra_CrsGraph());

    const int numRows = graph->NumMyRows();
    const int nnz     = graph->NumMyNonzeros();

    TEUCHOS_TEST_FOR_EXCEPTION(!graph->StorageOptimized(), std::runtime_error, "Xpetra::CrsGraph<>::getLocalGraph: Epetra_CrsGraph not StorageOptimized");
    const int *rowptr = graph->ExpertExtractIndexOffset().Values();
    int *colind       = graph->ExpertExtractIndices().Values();

    // Transform int* rowptr array to size_type* array
    typename local_graph_type::row_map_type::non_const_type kokkosRowPtr(Kokkos::ViewAllocateWithoutInitializing("local row map"), numRows + 1);
    for (size_t i = 0; i < kokkosRowPtr.size(); i++)
      kokkosRowPtr(i) = Teuchos::asSafe<typename local_graph_type::row_map_type::value_type>(rowptr[i]);

    // create Kokkos::Views
    typename local_graph_type::entries_type kokkosColind(colind, nnz);

    local_graph_type localGraph = local_graph_type(kokkosColind, kokkosRowPtr);

    return localGraph;
  }

  void getLocalDiagOffsets(const Kokkos::View<size_t *, typename Node::device_type, Kokkos::MemoryUnmanaged> &offsets) const {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::RuntimeError,
                               "Epetra does not support getLocalDiagOffsets!");
  }
#else
#ifdef __GNUC__
#warning "Xpetra Kokkos interface for CrsGraph is enabled (HAVE_XPETRA_KOKKOS_REFACTOR) but Tpetra is disabled. The Kokkos interface needs Tpetra to be enabled, too."
#endif
#endif

  //! Force the computation of global constants if we don't have them
  void computeGlobalConstants() {}

  //@}

  //! @name Overridden from Teuchos::Describable
  //@{

  //! Return a simple one-line description of this object.
  std::string description() const {
    XPETRA_MONITOR("EpetraCrsGraphT::description");
    return "NotImplemented";
  }

  //! Print the object with some verbosity level to an FancyOStream object.
  void describe(Teuchos::FancyOStream &out, const Teuchos::EVerbosityLevel /* verbLevel */ = Teuchos::Describable::verbLevel_default) const {
    XPETRA_MONITOR("EpetraCrsGraphT::describe");

    out << "EpetraCrsGraphT::describe : Warning, verbosity level is ignored by this method." << std::endl;
    const Epetra_BlockMap rowmap = graph_->RowMap();
    if (rowmap.Comm().MyPID() == 0) out << "** EpetraCrsGraphT **\n\nrowmap" << std::endl;
    rowmap.Print(out);
    graph_->Print(out);
  }

  //@}

  //! Implements DistObject interface
  //{@

  //! Access function for the Tpetra::Map this DistObject was constructed with.
  Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->Map());
  }

  //! Import.
  void doImport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &source,
                const Import<LocalOrdinal, GlobalOrdinal, Node> &importer, CombineMode CM) {
    XPETRA_MONITOR("EpetraCrsGraphT::doImport");

    XPETRA_DYNAMIC_CAST(const EpetraCrsGraphT<GlobalOrdinal XPETRA_COMMA Node>, source, tSource, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraCrsGraphT as input arguments.");
    XPETRA_DYNAMIC_CAST(const EpetraImportT<GlobalOrdinal XPETRA_COMMA Node>, importer, tImporter, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraImportT as input arguments.");

    RCP<const Epetra_CrsGraph> v = tSource.getEpetra_CrsGraph();
    int err                      = graph_->Import(*v, *tImporter.getEpetra_Import(), toEpetra(CM));
    TEUCHOS_TEST_FOR_EXCEPTION(err != 0, std::runtime_error, "Catch error code returned by Epetra.");
  }

  //! Export.
  void doExport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &dest,
                const Import<LocalOrdinal, GlobalOrdinal, Node> &importer, CombineMode CM) {
    XPETRA_MONITOR("EpetraCrsGraphT::doExport");

    XPETRA_DYNAMIC_CAST(const EpetraCrsGraphT<GlobalOrdinal XPETRA_COMMA Node>, dest, tDest, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraCrsGraphT as input arguments.");
    XPETRA_DYNAMIC_CAST(const EpetraImportT<GlobalOrdinal XPETRA_COMMA Node>, importer, tImporter, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraImportT as input arguments.");

    RCP<const Epetra_CrsGraph> v = tDest.getEpetra_CrsGraph();
    int err                      = graph_->Export(*v, *tImporter.getEpetra_Import(), toEpetra(CM));
    TEUCHOS_TEST_FOR_EXCEPTION(err != 0, std::runtime_error, "Catch error code returned by Epetra.");
  }

  //! Import (using an Exporter).
  void doImport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &source,
                const Export<LocalOrdinal, GlobalOrdinal, Node> &exporter, CombineMode CM) {
    XPETRA_MONITOR("EpetraCrsGraphT::doImport");

    XPETRA_DYNAMIC_CAST(const EpetraCrsGraphT<GlobalOrdinal XPETRA_COMMA Node>, source, tSource, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraCrsGraphT as input arguments.");
    XPETRA_DYNAMIC_CAST(const EpetraExportT<GlobalOrdinal XPETRA_COMMA Node>, exporter, tExporter, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraImportT as input arguments.");

    RCP<const Epetra_CrsGraph> v = tSource.getEpetra_CrsGraph();
    int err                      = graph_->Import(*v, *tExporter.getEpetra_Export(), toEpetra(CM));
    TEUCHOS_TEST_FOR_EXCEPTION(err != 0, std::runtime_error, "Catch error code returned by Epetra.");
  }

  //! Export (using an Importer).
  void doExport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &dest,
                const Export<LocalOrdinal, GlobalOrdinal, Node> &exporter, CombineMode CM) {
    XPETRA_MONITOR("EpetraCrsGraphT::doExport");

    XPETRA_DYNAMIC_CAST(const EpetraCrsGraphT<GlobalOrdinal XPETRA_COMMA Node>, dest, tDest, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraCrsGraphT as input arguments.");
    XPETRA_DYNAMIC_CAST(const EpetraExportT<GlobalOrdinal XPETRA_COMMA Node>, exporter, tExporter, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraImportT as input arguments.");

    RCP<const Epetra_CrsGraph> v = tDest.getEpetra_CrsGraph();
    int err                      = graph_->Export(*v, *tExporter.getEpetra_Export(), toEpetra(CM));
    TEUCHOS_TEST_FOR_EXCEPTION(err != 0, std::runtime_error, "Catch error code returned by Epetra.");
  }
  //@}

  //! @name Xpetra specific
  //@{

  //! EpetraCrsGraphT constructor to wrap a Epetra_CrsGraph object
  EpetraCrsGraphT(const Teuchos::RCP<Epetra_CrsGraph> &graph)
    : graph_(graph) {
    TEUCHOS_TEST_FOR_EXCEPTION(!graph->RowMap().GlobalIndicesIsType<GlobalOrdinal>(), std::runtime_error, "Xpetra::EpetraCrsGraphT: GlobalOrdinal mismatch.");
  }

  //! Get the underlying Epetra graph
  RCP<const Epetra_CrsGraph> getEpetra_CrsGraph() const { return graph_; }

  //@}

 private:
  RCP<Epetra_CrsGraph> graph_;
};

#endif  // specialization on Node=EpetraNode and GO=int

// specialization on GO=long long and Node=EpetraNode
#ifndef XPETRA_EPETRA_NO_64BIT_GLOBAL_INDICES
template <>
class EpetraCrsGraphT<long long, EpetraNode>
  : public virtual CrsGraph<int, long long, EpetraNode> {
  typedef int LocalOrdinal;
  typedef long long GlobalOrdinal;
  typedef EpetraNode Node;

  //! The specialization of Map used by this class.
  typedef Map<LocalOrdinal, GlobalOrdinal, Node> map_type;

#ifdef HAVE_XPETRA_TPETRA
  typedef typename Xpetra::CrsGraph<LocalOrdinal, GlobalOrdinal, Node>::local_graph_type local_graph_type;
  typedef typename Xpetra::CrsGraph<LocalOrdinal, GlobalOrdinal, Node>::node_type node_type;
  typedef typename node_type::execution_space execution_space;
#endif

 public:
  //! @name Constructor/Destructor Methods
  //@{

  //! Constructor for empty DynamicProfile graph (no space is preallocated).
  EpetraCrsGraphT(const Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap)
    : graph_(Teuchos::rcp(new Epetra_CrsGraph(Copy, toEpetra<GlobalOrdinal, Node>(rowMap), 0, false))) {}

  //! Constructor specifying fixed number of entries for each row.
  EpetraCrsGraphT(const RCP<const map_type> &rowMap, size_t maxNumEntriesPerRow, const RCP<Teuchos::ParameterList> & /* plist */ = Teuchos::null)
    : graph_(Teuchos::rcp(new Epetra_CrsGraph(Copy, toEpetra<GlobalOrdinal, Node>(rowMap), maxNumEntriesPerRow, true))) {}

  ////! Constructor specifying (possibly different) number of entries in each row.
  // Definition not in cpp, so comment out
  // EpetraCrsGraphT(const RCP< const Map< LocalOrdinal, GlobalOrdinal, Node > > &rowMap, const ArrayRCP< const size_t > &NumEntriesPerRowToAlloc, const RCP< ParameterList > &params=null);

  //! Constructor specifying column Map and fixed number of entries for each row.
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap, const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &colMap, size_t maxNumEntriesPerRow, const Teuchos::RCP<Teuchos::ParameterList> & /* plist */ = null)
    : graph_(Teuchos::rcp(new Epetra_CrsGraph(Copy, toEpetra<GlobalOrdinal, Node>(rowMap), toEpetra<GlobalOrdinal, Node>(colMap), maxNumEntriesPerRow, true))) {}

  ////! Constructor specifying column Map and number of entries in each row.
  // Definition not in cpp, so comment out
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rowMap, const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &colMap, const ArrayRCP<const size_t> &NumEntriesPerRowToAlloc, const RCP<ParameterList> & /* params */ = null) {
    Teuchos::Array<int> numEntriesPerRowToAlloc(NumEntriesPerRowToAlloc.begin(), NumEntriesPerRowToAlloc.end());  // convert array of "size_t" to array of "int"
    graph_ = Teuchos::rcp(new Epetra_CrsGraph(Copy, toEpetra<GlobalOrdinal, Node>(rowMap), toEpetra<GlobalOrdinal, Node>(colMap), numEntriesPerRowToAlloc.getRawPtr(), true));
  }

#ifdef HAVE_XPETRA_TPETRA
  //! Constructor specifying column Map, number of entries in each row and column indices in each row.
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* rowMap */,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* colMap */,
                  const typename local_graph_type::row_map_type & /* rowPointers */,
                  const typename local_graph_type::entries_type::non_const_type & /* columnIndices */,
                  const RCP<Teuchos::ParameterList> & /* plist */ = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Epetra does not support CrsGraph constructors using Kokkos Views!");
  }

  //! Constructor specifying column Map, number of entries in each row and column indices in each row.
  EpetraCrsGraphT(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* rowMap */,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* colMap */,
                  const local_graph_type & /* lclGraph */,
                  const RCP<Teuchos::ParameterList> & /* plist */ = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Epetra does not support CrsGraph constructors using a local graph!");
  }

  //! Constructor specifying column Map, number of entries in each row and column indices in each row.
  EpetraCrsGraphT(const local_graph_type & /* lclGraph */,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* rowMap */,
                  const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> & /* colMap */,
                  const Teuchos::RCP<const map_type> & /* domainMap */ = Teuchos::null,
                  const Teuchos::RCP<const map_type> & /* rangeMap */  = Teuchos::null,
                  const RCP<Teuchos::ParameterList> & /* plist */      = Teuchos::null) {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::NotImplemented,
                               "Epetra does not support CrsGraph constructors using a local graph!");
  }
#endif

  //! Destructor.
  virtual ~EpetraCrsGraphT() {}

  //@}

  //! @name Insertion/Removal Methods
  //@{

  //! Insert global indices into the graph.
  void insertGlobalIndices(GlobalOrdinal globalRow, const ArrayView<const GlobalOrdinal> &indices) {
    XPETRA_MONITOR("EpetraCrsGraphT::insertGlobalIndices");

    GlobalOrdinal *indices_rawPtr = const_cast<GlobalOrdinal *>(indices.getRawPtr());  // there is no const in the Epetra interface :(
    XPETRA_ERR_CHECK(graph_->InsertGlobalIndices(globalRow, indices.size(), indices_rawPtr));
  }

  //! Insert local indices into the graph.
  void insertLocalIndices(const LocalOrdinal localRow, const ArrayView<const LocalOrdinal> &indices) {
    XPETRA_MONITOR("EpetraCrsGraphT::insertLocalIndices");

    int *indices_rawPtr = const_cast<int *>(indices.getRawPtr());  // there is no const in the Epetra interface :(
    XPETRA_ERR_CHECK(graph_->InsertMyIndices(localRow, indices.size(), indices_rawPtr));
  }

  //! Remove all graph indices from the specified local row.
  void removeLocalIndices(LocalOrdinal localRow) {
    XPETRA_MONITOR("EpetraCrsGraphT::removeLocalIndices");
    graph_->RemoveMyIndices(localRow);
  }

  //! Allocates and returns ArrayRCPs of the Crs arrays --- This is an Xpetra-only routine.
  //** \warning This is an expert-only routine and should not be called from user code. */
  void allocateAllIndices(size_t numNonZeros, ArrayRCP<size_t> &rowptr, ArrayRCP<LocalOrdinal> &colind) {
    XPETRA_MONITOR("EpetraCrsGraphT::allocateAllIndies");

    // Row offsets
    // Unfortunately, we cannot do this in the same manner as column indices
    // and values (see below).  The problem is that Tpetra insists on using
    // size_t, and Epetra uses int internally.  So we only resize here, and
    // will need to copy in setAllValues
    rowptr.resize(getLocalNumRows() + 1);

    int lowerOffset = 0;
    bool ownMemory  = false;

    // Column indices
    // Extract, resize, set colind
    Epetra_IntSerialDenseVector &myColind = graph_->ExpertExtractIndices();
    myColind.Resize(numNonZeros);
    colind = Teuchos::arcp(myColind.Values(), lowerOffset, numNonZeros, ownMemory);
  }

  //! Sets the 1D pointer arrays of the graph.
  void setAllIndices(const ArrayRCP<size_t> &rowptr, const ArrayRCP<LocalOrdinal> &colind) {
    XPETRA_MONITOR("EpetraCrsGraphT::setAllIndices");

    // Check sizes
    TEUCHOS_TEST_FOR_EXCEPTION(Teuchos::as<size_t>(rowptr.size()) != getLocalNumRows() + 1, Xpetra::Exceptions::RuntimeError,
                               "An exception is thrown to let you know that the size of your rowptr array is incorrect.");
    if (colind.size() > 0) {
      TEUCHOS_TEST_FOR_EXCEPTION(colind.getRawPtr() != graph_->ExpertExtractIndices().Values(), Xpetra::Exceptions::RuntimeError,
                                 "An exception is thrown to let you know that you mismatched your pointers.");
    }

    // We have to make a copy here, it is unavoidable
    // See comments in allocateAllIndices
    const size_t N = getLocalNumRows();

    Epetra_IntSerialDenseVector &myRowptr = graph_->ExpertExtractIndexOffset();
    myRowptr.Resize(N + 1);
    for (size_t i = 0; i < N + 1; i++)
      myRowptr[i] = Teuchos::as<int>(rowptr[i]);
  }

  //! Gets the 1D pointer arrays of the graph.
  void getAllIndices(ArrayRCP<const size_t> &rowptr, ArrayRCP<const LocalOrdinal> &colind) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getAllIndices");

    int lowerOffset = 0;
    bool ownMemory  = false;

    const size_t n   = getLocalNumRows();
    const size_t nnz = getLocalNumEntries();

    // Row offsets
    // We have to make a copy here, it is unavoidable (see comments in allocateAllValues)
    Epetra_IntSerialDenseVector &myRowptr = graph_->ExpertExtractIndexOffset();
    rowptr.resize(n + 1);
    for (size_t i = 0; i < n + 1; i++)
      (*const_cast<size_t *>(&rowptr[i])) = Teuchos::as<size_t>(myRowptr[i]);

    // Column indices
    colind = Teuchos::arcp(graph_->ExpertExtractIndices().Values(), lowerOffset, nnz, ownMemory);
  }

  //@}

  //! @name Transformational Methods
  //@{

  //! Signal that data entry is complete, specifying domain and range maps.
  void fillComplete(const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &domainMap, const RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rangeMap, const RCP<ParameterList> &params = null) {
    XPETRA_MONITOR("EpetraCrsGraphT::fillComplete");

    graph_->FillComplete(toEpetra<GlobalOrdinal, Node>(domainMap), toEpetra<GlobalOrdinal, Node>(rangeMap));
    bool doOptimizeStorage = true;
    if (params != null && params->get("Optimize Storage", true) == false) doOptimizeStorage = false;
    if (doOptimizeStorage) graph_->OptimizeStorage();
  }

  //! Signal that data entry is complete.
  void fillComplete(const RCP<ParameterList> &params = null) {
    XPETRA_MONITOR("EpetraCrsGraphT::fillComplete");

    graph_->FillComplete();
    bool doOptimizeStorage = true;
    if (params != null && params->get("Optimize Storage", true) == false) doOptimizeStorage = false;
    if (doOptimizeStorage) graph_->OptimizeStorage();
  }

  //! Expert version of fillComplete
  void
  expertStaticFillComplete(const Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &domainMap,
                           const Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> &rangeMap,
                           const Teuchos::RCP<const Import<LocalOrdinal, GlobalOrdinal, Node>> &importer =
                               Teuchos::null,
                           const Teuchos::RCP<const Export<LocalOrdinal, GlobalOrdinal, Node>> &exporter =
                               Teuchos::null,
                           const Teuchos::RCP<Teuchos::ParameterList> &params =
                               Teuchos::null) {
    // Not optimized
    graph_->FillComplete(toEpetra<GlobalOrdinal, Node>(domainMap), toEpetra<GlobalOrdinal, Node>(rangeMap));
    graph_->OptimizeStorage();
  }

  //@}

  //! @name Methods implementing RowGraph.
  //@{

  //! Returns the communicator.
  RCP<const Comm<int>> getComm() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getComm");
    return toXpetra(graph_->Comm());
  }

  //! Returns the Map that describes the row distribution in this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getRowMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getRowMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->RowMap());
  }

  //! Returns the Map that describes the column distribution in this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getColMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getColMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->ColMap());
  }

  //! Returns the Map associated with the domain of this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getDomainMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getDomainMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->DomainMap());
  }

  //! Returns the Map associated with the domain of this graph.
  RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getRangeMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getRangeMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->RangeMap());
  }

  //! Returns the importer associated with this graph.
  RCP<const Import<LocalOrdinal, GlobalOrdinal, Node>> getImporter() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getImporter");
    return toXpetra<GlobalOrdinal, Node>(graph_->Importer());
  }

  //! Returns the exporter associated with this graph.
  RCP<const Export<LocalOrdinal, GlobalOrdinal, Node>> getExporter() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getExporter");
    return toXpetra<GlobalOrdinal, Node>(graph_->Exporter());
  }

  //! Returns the number of global rows in the graph.
  global_size_t getGlobalNumRows() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalNumRows");
    return graph_->NumGlobalRows64();
  }

  //! Returns the number of global columns in the graph.
  global_size_t getGlobalNumCols() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalNumCols");
    return graph_->NumGlobalCols64();
  }

  //! Returns the number of graph rows owned on the calling node.
  size_t getLocalNumRows() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalNumRows");
    return graph_->NumMyRows();
  }

  //! Returns the number of columns connected to the locally owned rows of this graph.
  size_t getLocalNumCols() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalNumCols");
    return graph_->NumMyCols();
  }

  //! Returns the index base for global indices for this graph.
  GlobalOrdinal getIndexBase() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getIndexBase");
    return (GlobalOrdinal)graph_->IndexBase64();
  }

  //! Returns the global number of entries in the graph.
  global_size_t getGlobalNumEntries() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalNumEntries");
    return graph_->NumGlobalEntries64();
  }

  //! Returns the local number of entries in the graph.
  size_t getLocalNumEntries() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalNumEntries");
    return graph_->NumMyEntries();
  }

  //! Returns the current number of entries on this node in the specified global row.
  size_t getNumEntriesInGlobalRow(GlobalOrdinal globalRow) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getNumEntriesInGlobalRow");
    return graph_->NumGlobalIndices(globalRow);
  }

  //! Returns the current number of entries on this node in the specified local row.
  size_t getNumEntriesInLocalRow(LocalOrdinal localRow) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getNumEntriesInLocalRow");
    return graph_->NumMyIndices(localRow);
  }

  //! Returns the current number of allocated entries for this node in the specified global row .
  size_t getNumAllocatedEntriesInGlobalRow(GlobalOrdinal globalRow) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getNumAllocatedEntriesInGlobalRow");
    return graph_->NumAllocatedGlobalIndices(globalRow);
  }

  //! Returns the current number of allocated entries on this node in the specified local row.
  size_t getNumAllocatedEntriesInLocalRow(LocalOrdinal localRow) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getNumAllocatedEntriesInLocalRow");
    return graph_->NumAllocatedMyIndices(localRow);
  }

  //! Maximum number of entries in all rows over all processes.
  size_t getGlobalMaxNumRowEntries() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalMaxNumRowEntries");
    return graph_->GlobalMaxNumIndices();
  }

  //! Maximum number of entries in all rows owned by the calling process.
  size_t getLocalMaxNumRowEntries() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalMaxNumRowEntries");
    return graph_->MaxNumIndices();
  }

  //! Whether the graph has a column Map.
  bool hasColMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::hasColMap");
    return graph_->HaveColMap();
  }

  //! Whether column indices are stored using local indices on the calling process.
  bool isLocallyIndexed() const {
    XPETRA_MONITOR("EpetraCrsGraphT::isLocallyIndexed");
    return graph_->IndicesAreLocal();
  }

  //! Whether column indices are stored using global indices on the calling process.
  bool isGloballyIndexed() const {
    XPETRA_MONITOR("EpetraCrsGraphT::isGloballyIndexed");
    return graph_->IndicesAreGlobal();
  }

  //! Whether fillComplete() has been called and the graph is in compute mode.
  bool isFillComplete() const {
    XPETRA_MONITOR("EpetraCrsGraphT::isFillComplete");
    return graph_->Filled();
  }

  //! Returns true if storage has been optimized.
  bool isStorageOptimized() const {
    XPETRA_MONITOR("EpetraCrsGraphT::isStorageOptimized");
    return graph_->StorageOptimized();
  }

  //! Return a const, nonpersisting view of global indices in the given row.
  void getGlobalRowView(GlobalOrdinal GlobalRow, ArrayView<const GlobalOrdinal> &Indices) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getGlobalRowView");

    int numEntries;
    GlobalOrdinal *eIndices;

    XPETRA_ERR_CHECK(graph_->ExtractGlobalRowView(GlobalRow, numEntries, eIndices));
    if (numEntries == 0) {
      eIndices = NULL;
    }  // Cf. TEUCHOS_TEST_FOR_EXCEPT( p == 0 && size_in != 0 ) in Teuchos ArrayView constructor.

    Indices = ArrayView<const GlobalOrdinal>(eIndices, numEntries);
  }

  //! Return a const, nonpersisting view of local indices in the given row.
  void getLocalRowView(LocalOrdinal LocalRow, ArrayView<const LocalOrdinal> &indices) const {
    XPETRA_MONITOR("EpetraCrsGraphT::getLocalRowView");

    int numEntries;
    int *eIndices;

    XPETRA_ERR_CHECK(graph_->ExtractMyRowView(LocalRow, numEntries, eIndices));
    if (numEntries == 0) {
      eIndices = NULL;
    }  // Cf. TEUCHOS_TEST_FOR_EXCEPT( p == 0 && size_in != 0 ) in Teuchos ArrayView constructor.

    indices = ArrayView<const int>(eIndices, numEntries);
  }

#ifdef HAVE_XPETRA_TPETRA
  typename local_graph_type::HostMirror getLocalGraphHost() const {
    RCP<Epetra_CrsGraph> graph = Teuchos::rcp_const_cast<Epetra_CrsGraph>(getEpetra_CrsGraph());

    using local_graph_type_host = typename local_graph_type::HostMirror;

    const int numRows = graph->NumMyRows();
    const int nnz     = graph->NumMyNonzeros();

    TEUCHOS_TEST_FOR_EXCEPTION(!graph->StorageOptimized(), std::runtime_error, "Xpetra::CrsGraph<>::getLocalGraph: Epetra_CrsGraph not StorageOptimized");
    const int *rowptr = graph->ExpertExtractIndexOffset().Values();
    int *colind       = graph->ExpertExtractIndices().Values();

    // Transform int* rowptr array to size_type* array
    typename local_graph_type_host::row_map_type::non_const_type kokkosRowPtr(Kokkos::ViewAllocateWithoutInitializing("local row map"), numRows + 1);
    for (size_t i = 0; i < kokkosRowPtr.size(); i++)
      kokkosRowPtr(i) = Teuchos::asSafe<typename local_graph_type_host::row_map_type::value_type>(rowptr[i]);

    // create Kokkos::Views
    typename local_graph_type_host::entries_type kokkosColind(colind, nnz);

    local_graph_type_host localGraph = local_graph_type_host(kokkosColind, kokkosRowPtr);

    return localGraph;
  }

  local_graph_type getLocalGraphDevice() const {
    RCP<Epetra_CrsGraph> graph = Teuchos::rcp_const_cast<Epetra_CrsGraph>(getEpetra_CrsGraph());

    const int numRows = graph->NumMyRows();
    const int nnz     = graph->NumMyNonzeros();

    TEUCHOS_TEST_FOR_EXCEPTION(!graph->StorageOptimized(), std::runtime_error, "Xpetra::CrsGraph<>::getLocalGraph: Epetra_CrsGraph not StorageOptimized");
    const int *rowptr = graph->ExpertExtractIndexOffset().Values();
    int *colind       = graph->ExpertExtractIndices().Values();

    // Transform int* rowptr array to size_type* array
    typename local_graph_type::row_map_type::non_const_type kokkosRowPtr(Kokkos::ViewAllocateWithoutInitializing("local row map"), numRows + 1);
    for (size_t i = 0; i < kokkosRowPtr.size(); i++)
      kokkosRowPtr(i) = Teuchos::asSafe<typename local_graph_type::row_map_type::value_type>(rowptr[i]);

    // create Kokkos::Views
    typename local_graph_type::entries_type kokkosColind(colind, nnz);

    local_graph_type localGraph = local_graph_type(kokkosColind, kokkosRowPtr);

    return localGraph;
  }

  void getLocalDiagOffsets(const Kokkos::View<size_t *, typename Node::device_type, Kokkos::MemoryUnmanaged> &offsets) const {
    TEUCHOS_TEST_FOR_EXCEPTION(true, Xpetra::Exceptions::RuntimeError,
                               "Epetra does not support getLocalDiagOffsets!");
  }
#else
#ifdef __GNUC__
#warning "Xpetra Kokkos interface for CrsGraph is enabled (HAVE_XPETRA_KOKKOS_REFACTOR) but Tpetra is disabled. The Kokkos interface needs Tpetra to be enabled, too."
#endif
#endif

  //! Force the computation of global constants if we don't have them
  void computeGlobalConstants() {}

  //@}

  //! @name Overridden from Teuchos::Describable
  //@{

  //! Return a simple one-line description of this object.
  std::string description() const {
    XPETRA_MONITOR("EpetraCrsGraphT::description");
    return "NotImplemented";
  }

  //! Print the object with some verbosity level to an FancyOStream object.
  void describe(Teuchos::FancyOStream &out, const Teuchos::EVerbosityLevel /* verbLevel */ = Teuchos::Describable::verbLevel_default) const {
    XPETRA_MONITOR("EpetraCrsGraphT::describe");

    out << "EpetraCrsGraphT::describe : Warning, verbosity level is ignored by this method." << std::endl;
    const Epetra_BlockMap rowmap = graph_->RowMap();
    if (rowmap.Comm().MyPID() == 0) out << "** EpetraCrsGraphT **\n\nrowmap" << std::endl;
    rowmap.Print(out);
    graph_->Print(out);
  }

  //@}

  //! Implements DistObject interface
  //{@

  //! Access function for the Tpetra::Map this DistObject was constructed with.
  Teuchos::RCP<const Map<LocalOrdinal, GlobalOrdinal, Node>> getMap() const {
    XPETRA_MONITOR("EpetraCrsGraphT::getMap");
    return toXpetra<GlobalOrdinal, Node>(graph_->Map());
  }

  //! Import.
  void doImport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &source,
                const Import<LocalOrdinal, GlobalOrdinal, Node> &importer, CombineMode CM) {
    XPETRA_MONITOR("EpetraCrsGraphT::doImport");

    XPETRA_DYNAMIC_CAST(const EpetraCrsGraphT<GlobalOrdinal XPETRA_COMMA Node>, source, tSource, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraCrsGraphT as input arguments.");
    XPETRA_DYNAMIC_CAST(const EpetraImportT<GlobalOrdinal XPETRA_COMMA Node>, importer, tImporter, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraImportT as input arguments.");

    RCP<const Epetra_CrsGraph> v = tSource.getEpetra_CrsGraph();
    int err                      = graph_->Import(*v, *tImporter.getEpetra_Import(), toEpetra(CM));
    TEUCHOS_TEST_FOR_EXCEPTION(err != 0, std::runtime_error, "Catch error code returned by Epetra.");
  }

  //! Export.
  void doExport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &dest,
                const Import<LocalOrdinal, GlobalOrdinal, Node> &importer, CombineMode CM) {
    XPETRA_MONITOR("EpetraCrsGraphT::doExport");

    XPETRA_DYNAMIC_CAST(const EpetraCrsGraphT<GlobalOrdinal XPETRA_COMMA Node>, dest, tDest, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraCrsGraphT as input arguments.");
    XPETRA_DYNAMIC_CAST(const EpetraImportT<GlobalOrdinal XPETRA_COMMA Node>, importer, tImporter, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraImportT as input arguments.");

    RCP<const Epetra_CrsGraph> v = tDest.getEpetra_CrsGraph();
    int err                      = graph_->Export(*v, *tImporter.getEpetra_Import(), toEpetra(CM));
    TEUCHOS_TEST_FOR_EXCEPTION(err != 0, std::runtime_error, "Catch error code returned by Epetra.");
  }

  //! Import (using an Exporter).
  void doImport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &source,
                const Export<LocalOrdinal, GlobalOrdinal, Node> &exporter, CombineMode CM) {
    XPETRA_MONITOR("EpetraCrsGraphT::doImport");

    XPETRA_DYNAMIC_CAST(const EpetraCrsGraphT<GlobalOrdinal XPETRA_COMMA Node>, source, tSource, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraCrsGraphT as input arguments.");
    XPETRA_DYNAMIC_CAST(const EpetraExportT<GlobalOrdinal XPETRA_COMMA Node>, exporter, tExporter, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraImportT as input arguments.");

    RCP<const Epetra_CrsGraph> v = tSource.getEpetra_CrsGraph();
    int err                      = graph_->Import(*v, *tExporter.getEpetra_Export(), toEpetra(CM));
    TEUCHOS_TEST_FOR_EXCEPTION(err != 0, std::runtime_error, "Catch error code returned by Epetra.");
  }

  //! Export (using an Importer).
  void doExport(const DistObject<GlobalOrdinal, LocalOrdinal, GlobalOrdinal, Node> &dest,
                const Export<LocalOrdinal, GlobalOrdinal, Node> &exporter, CombineMode CM) {
    XPETRA_MONITOR("EpetraCrsGraphT::doExport");

    XPETRA_DYNAMIC_CAST(const EpetraCrsGraphT<GlobalOrdinal XPETRA_COMMA Node>, dest, tDest, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraCrsGraphT as input arguments.");
    XPETRA_DYNAMIC_CAST(const EpetraExportT<GlobalOrdinal XPETRA_COMMA Node>, exporter, tExporter, "Xpetra::EpetraCrsGraphT::doImport only accept Xpetra::EpetraImportT as input arguments.");

    RCP<const Epetra_CrsGraph> v = tDest.getEpetra_CrsGraph();
    int err                      = graph_->Export(*v, *tExporter.getEpetra_Export(), toEpetra(CM));
    TEUCHOS_TEST_FOR_EXCEPTION(err != 0, std::runtime_error, "Catch error code returned by Epetra.");
  }
  //@}

  //! @name Xpetra specific
  //@{

  //! EpetraCrsGraphT constructor to wrap a Epetra_CrsGraph object
  EpetraCrsGraphT(const Teuchos::RCP<Epetra_CrsGraph> &graph)
    : graph_(graph) {
    TEUCHOS_TEST_FOR_EXCEPTION(!graph->RowMap().GlobalIndicesIsType<GlobalOrdinal>(), std::runtime_error, "Xpetra::EpetraCrsGraphT: GlobalOrdinal mismatch.");
  }

  //! Get the underlying Epetra graph
  RCP<const Epetra_CrsGraph> getEpetra_CrsGraph() const { return graph_; }

  //@}

 private:
  RCP<Epetra_CrsGraph> graph_;
};
#endif  // specialization on Node=EpetraNode and GO=int

}  // namespace Xpetra

#endif  // XPETRA_EPETRACRSGRAPH_HPP
