// @HEADER
//
// ***********************************************************************
//
//        MueLu: A package for multigrid based preconditioning
//                  Copyright 2012 Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact
//                    Jonathan Hu       (jhu@sandia.gov)
//                    Andrey Prokopenko (aprokop@sandia.gov)
//                    Ray Tuminaro      (rstumin@sandia.gov)
//
// ***********************************************************************
//
// @HEADER
#ifndef MUELU_REPLICATEPFACTORY_DECL_HPP
#define MUELU_REPLICATEPFACTORY_DECL_HPP

#include <Teuchos_SerialDenseVector.hpp>

#include <Xpetra_Matrix_fwd.hpp>

#include "MueLu_ConfigDefs.hpp"
#include "MueLu_PFactory.hpp"
#include "MueLu_ReplicatePFactory_fwd.hpp"

#include "MueLu_Level_fwd.hpp"

namespace MueLuTests {
// Forward declaration of friend tester class used to UnitTest ReplicatePFactory
template <class Scalar, class LocalOrdinal, class GlobalOrdinal, class Node>
class ReplicatePFactoryTester;
}  // namespace MueLuTests

namespace MueLu {

/*!
  @class ReplicatePFactory
  @ingroup MueLuTransferClasses
  @brief Prolongator factory that replicates 'Psubblock' matrix to create new prolongator suitable for PDE systems


  Takes a previously generated prolongator for a scalar PDE and effectively makes a block diagonal prolongator
  by replicating it "replicate: npdes" times so that it can be used with a PDE system. A normal use case
  would be to run an existing MueLu multigrid algorithm on a scalar PDE to generate a hierarchy.  Then use
  something like

  MueLu::HierarchyUtils<SC,LO,GO,NO>::CopyBetweenHierarchies(*scalarHierarchy,*systemHierarchy, "P",  "Psubblock", "RCP<Matrix>");

  to copy it to the new hierarchy. This new hierarchy would then have <Parameter name="multigrid algorithm" type="string"  value="replicate"/>
  in its parameter list to then invoke RepliatePFactory. Currently, this is used in src/Operators/MueLu_Maxwell1_def.hpp with an example
  in test/maxwell.

  ## Input/output of ReplicatePFactory ##

  ### User parameters of SemiCoarsenPFactory ###
  | Parameter   | type    | default   | master.xml | validated | requested | description                                                                      |
  | ------------|---------|-----------|:----------:|:---------:|:---------:|----------------------------------------------------------------------------------|
  |replicate: npdes | int |  1        |     *      |     *     |           | Specifies the number of Psubblock copies to make                                 |
  | Psubblock   | Matrix  |           |            |           |           | Matrix that will be replicated                                                   |



  The * in the @c master.xml column denotes that the parameter is defined in the @c master.xml file.<br>
  The * in the @c validated column means that the parameter is declared in the list of valid input parameters (see ReplicateCoarsenPFactory::GetValidParameters).<br>
  The * in the @c requested column states that the data is requested as input with all dependencies (see ReplicateCoarsenPFactory::DeclareInput).

  ### Variables provided by ReplicatePFactory ###

  After ReplicatePFactory::Build the following data is available (if requested)

  | Parameter         | generated by             | description
  |-------------------|--------------------------|------------------------------------------------------------------------------------------------------------------|
  | P                 | ReplicatePFactory | Prolongator                                                                                                      |

*/
template <class Scalar        = DefaultScalar,
          class LocalOrdinal  = DefaultLocalOrdinal,
          class GlobalOrdinal = DefaultGlobalOrdinal,
          class Node          = DefaultNode>
class ReplicatePFactory : public PFactory {
#undef MUELU_REPLICATEPFACTORY_SHORT
#include "MueLu_UseShortNames.hpp"

 public:
  friend class MueLuTests::ReplicatePFactoryTester<Scalar, LocalOrdinal, GlobalOrdinal, Node>;

  //! @name Constructors/Destructors.
  //@{

  //! Constructor
  ReplicatePFactory() {}

  //! Destructor.
  virtual ~ReplicatePFactory() {}
  //@}

  RCP<const ParameterList> GetValidParameterList() const;

  //! Input
  //@{

  void DeclareInput(Level& fineLevel, Level& coarseLevel) const;

  //@}

  //! @name Build methods.
  //@{

  void Build(Level& fineLevel, Level& coarseLevel) const;
  void BuildP(Level& fineLevel, Level& coarseLevel) const;

  //@}

 private:
  int numPDEs_;

};  // class ReplicatePFactory

}  // namespace MueLu

#define MUELU_REPLICATEPFACTORY_SHORT
#endif  // MUELU_REPLICATEPFACTORY_DECL_HPP
