/******************************************/
/* Wminet - Mini IP monitor in dock app   */
/******************************************/

/******************************************/
/* This program is free software; you can redistribute it and/or
/* modify it under the terms of the GNU General Public License
/* as published by the Free Software Foundation; either version 2
/* of the License, or (at your option) any later version.
/* 
/* This program is distributed in the hope that it will be useful,
/* but WITHOUT ANY WARRANTY; without even the implied warranty of
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/* GNU General Public License for more details.
/* 
/* You should have received a copy of the GNU General Public License
/* along with this program; if not, write to the Free Software
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
/******************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include "wmgeneral.h"
#include "misc.h"
#include "xpm/wminet-default.xpm"
#include "xpm/wminet-lcd.xpm"

#define MASK_WIDTH 64
#define MASK_HEIGHT 64
#define CHAR_WIDTH 6
#define CHAR_HEIGHT 7
#define LINE_LENGTH 9
#define LINE_HEIGHT 11
#define CHAR_X 4
#define CHAR_Y 6

#define CONN_ALIVE 1

/******************************************/
/* Structures                             */
/******************************************/

struct {
	int type;
	char name[6];
	unsigned long laddr[2];
	unsigned int lport[2];
	unsigned long raddr[2];
	unsigned int rport[2];
	int conn[2];
} inet[] = {
	{ 2, "PROCS", { 0, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 } },
	{ 0, "SSH", { 0, 0 }, { 22, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 } },
	{ 0, "NNTP", { 0, 0 }, { 119, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 } },
	{ 0, "HTTP", { 0, 0 }, { 80, 8080 }, { 0, 0 }, { 0, 0 }, { 0, 0 } },
	{ 0, "SMTP", { 0, 0 }, { 25, 0 }, { 0, 0 }, { 0, 0 }, { 0, 0 } },
};

struct {
	char **pixmap;
	char *description;
} themes[] = {
	{ wminet_default_xpm, "Light emitting diode" },
	{ wminet_lcd_xpm, "Liquid crystal display" },
};

/******************************************/
/* Globals                                */
/******************************************/

char mask[MASK_WIDTH * MASK_HEIGHT];
int monitor_tcp = 0;
int monitor_udp = 0;
int monitor_proc = 0;
int refresh = 100000;
int update = 300000;
int theme = 0;
int debug = 0;
char hostname[32];
char command[32];

/******************************************/
/* Functions                              */
/******************************************/

void read_config(int, char **);
unsigned long convert_ip(const char *);
int string_to_number(const char *, unsigned int *);
void read_stats(void);
void draw_base(void);
void draw_nums(void);
void blit_string(char *, int, int);
void usage(int, char **);

/******************************************/
/* Main                                   */
/******************************************/

int main(int argc, char *argv[])
{
	XEvent Event;
	int count = update;
	int i;

	read_config( argc, argv );

	createXBMfromXPM(mask, themes[theme].pixmap, MASK_WIDTH, MASK_HEIGHT);
	openXwindow(argc, argv, themes[theme].pixmap, mask, MASK_WIDTH, MASK_HEIGHT);

	draw_base();
	draw_nums();

	while (1) {
		waitpid(0, NULL, WNOHANG);

		if ( count >= update ) {
			count = 0;

			read_stats();

			for ( i = 0; i < 5; i++ ) {
				if ( inet[i].conn[0] != inet[i].conn[1] ) {
					draw_nums();
					RedrawWindow();
					break;
				}
			}
		}

		while ( XPending(display) ) {
			XNextEvent(display, &Event);
			switch ( Event.type ) {
				case Expose:
					RedrawWindow();
					break;
				case DestroyNotify:
					XCloseDisplay(display);
					exit(0);
				case ButtonRelease:
					if ( Event.xbutton.button == 3 && command )
						execCommand(command);
					break;
			}
		}

		usleep(refresh);
		count = count + refresh;
	}
}



/******************************************/
/* Read config file                       */
/******************************************/

void read_config(int argc, char **argv)
{
	int i, j;
	char buf[256];
	char left[32], right[32];
	char *c;
	FILE *fp;

	gethostname(hostname,32);

	strcpy( buf, getenv("HOME") );
	strcat( buf,  "/.wminetrc" );

	if ( fp = fopen( buf, "r" ) ) {
		while ( fgets( buf, sizeof(buf), fp ) ) {
			if ( sscanf( buf, "%s %s", &left, &right ) == 2 ) {
				if ( ! strcasecmp( left, "line" ) ) {
					i = -1;
					if ( ! strcasecmp( right, "one" ) )
						i = 0;
					else if ( ! strcasecmp( right, "two" ) )
						i = 1;
					else if ( ! strcasecmp( right, "three" ) )
						i = 2;
					else if ( ! strcasecmp( right, "four" ) )
						i = 3;
					else if ( ! strcasecmp( right, "five" ) )
						i = 4;
					if ( i >= 0 ) {
						inet[i].type = -1;
						strcpy( inet[i].name, "UNDEF" );
						for ( j = 0; j < 2; j++ ) {
							inet[i].laddr[j] = 0;
							inet[i].lport[j] = 0;
							inet[i].raddr[j] = 0;
							inet[i].rport[j] = 0;
						}
					}
				} else if ( ! strcasecmp( left, "type" ) ) {
					if ( ! strcasecmp( right, "tcp" ) )
						inet[i].type = 0;
					else if ( ! strcasecmp( right, "udp" ) )
						inet[i].type = 1;
					else if ( ! strcasecmp( right, "proc" ) )
						inet[i].type = 2;
					else if ( ! strcasecmp( right, "host" ) )
						inet[i].type = 3;
				} else if ( ! strcasecmp( left, "name" ) ) {
					strncpy( inet[i].name, right, 6 );
				} else if ( ! strcasecmp( left, "localport" ) ) {
					if ( ! inet[i].lport[0] )
						inet[i].lport[0] = atoi(right);
					else if ( ! inet[i].lport[1] )
						inet[i].lport[1] = atoi(right);
				} else if ( ! strcasecmp( left, "localaddress" ) ) {
					if ( ! inet[i].laddr[0] )
						inet[i].laddr[0] = convert_ip(right);
					else if ( ! inet[i].laddr[1] )
						inet[i].laddr[1] = convert_ip(right);
				} else if ( ! strcasecmp( left, "remoteport" ) ) {
					if ( ! inet[i].rport[0] )
						inet[i].rport[0] = atoi(right);
					else if ( ! inet[i].rport[1] )
						inet[i].rport[1] = atoi(right);
				} else if ( ! strcasecmp( left, "remoteaddress" ) ) {
					if ( ! inet[i].raddr[0] )
						inet[i].raddr[0] = convert_ip(right);
					else if ( ! inet[i].raddr[1] )
						inet[i].raddr[1] = convert_ip(right);
				} else if ( ! strcasecmp( left, "command" ) ) {
					strcpy( command, strstr( buf, right ) );
					if ( c = strchr( command, '\n' ) )
						*c = 0;
				} else if ( ! strcasecmp( left, "refresh" ) )
					refresh = atoi(right) * 1000;
				else if ( ! strcasecmp( left, "update" ) )
					update = atoi(right) * 1000;
				else if ( ! strcasecmp( left, "host" ) ) {
					if ( strcasecmp( hostname, right ) ) {
						while ( fgets( buf, sizeof(buf), fp ) ) {
							if ( buf[0] != '#' || buf[0] != '\n' ) {
								sscanf( buf, "%s %s", &left, &right );
								if ( ! strcasecmp( left, "host" ) ) {
									if ( ! strcasecmp( hostname, right ) )
										break;
								}
							}
						}
					}
				}
			}
		}
		fclose(fp);
	}

	while ( ( i = getopt(argc, argv, "s:r:c:tdh") ) != -1 ) {
		switch ( i ) {
			case 's':
				if ( optarg )
					update = atoi(optarg) * 1000;
				break;
			case 'r':
				if ( optarg )
					refresh = atoi(optarg) * 1000;
				break;
			case 'c':
				if ( optarg )
					strcpy( command, optarg );
				break;
			case 't':
				theme = 1;
				break;
			case 'd':
				debug = 1;
				break;
			case 'h':
			default:
				usage( argc, argv );
				exit(1);
		}
	}

	if ( debug )
		printf( "C: %s\n", command );

	for ( i = 0; i < 5; i++ ) {
		if ( debug )
			printf( "%d: %d %s %d %d %d %d %d %d %d %d %d %d\n", i, inet[i].type, inet[i].name,
				inet[i].laddr[0], inet[i].laddr[1], inet[i].lport[0], inet[i].lport[1],
				inet[i].raddr[0], inet[i].raddr[1], inet[i].rport[0], inet[i].rport[1],
				inet[i].conn[0], inet[i].conn[1] );
		if ( inet[i].type == 0 )
			monitor_tcp = 1;
		else if ( inet[i].type == 1 )
			monitor_udp = 1;
		else if ( inet[i].type == 2 )
			monitor_proc = 1;
	}
}

/******************************************/
/* Convert dotted IP to integer           */
/******************************************/

unsigned long convert_ip(const char *dotted)
{
	unsigned long addr;
	unsigned char *addrp;
	char *p, *q;
	unsigned int onebyte;
	int i;
	char buf[20];

	strncpy(buf, dotted, sizeof(buf) - 1);
	addrp = (unsigned char *) &addr;

	p = buf;
	for (i = 0; i < 3; i++) {
		if ((q = strchr(p, '.')) == NULL)
			return 0;

		*q = '\0';
		if (string_to_number(p, &onebyte) == -1)
			return 0;

		addrp[i] = (unsigned char) onebyte;
		p = q + 1;
	}

	if (string_to_number(p, &onebyte) == -1)
		return 0;

	addrp[3] = (unsigned char) onebyte;

	return addr;
}

int string_to_number(const char *in, unsigned int *out)
{
	long post;
	char *pre;

	post = strtol(in, &pre, 0);

	if ( ( *pre == '\0' && pre != in ) && ( post >= 0 && post <= 255 ) ) {
		*out = post;
		return 0;
	}

	return -1;
}

/******************************************/
/* Read statistics                        */
/******************************************/

void read_stats(void)
{
	int i;
	unsigned int lport, rport, active;
	unsigned long laddr, raddr;
	char buf[256];
	FILE *fp;

	for ( i = 0; i < 5; i++ ) {
		inet[i].conn[1] = inet[i].conn[0];
		inet[i].conn[0] = 0;
	}

	if ( monitor_tcp ) {
		if ( fp = fopen( "/proc/net/tcp", "r" ) ) {
			fgets( buf, sizeof(buf), fp );
			while ( fgets( buf, sizeof(buf), fp ) ) {
				sscanf( buf, "%*d: %lx:%x %lx:%x %x", &laddr, &lport, &raddr, &rport, &active );

				if ( active != CONN_ALIVE )
					continue;

				for ( i = 0; i < 5; i++ ) {
					if ( inet[i].type == 0 ) {
						if ( inet[i].lport[0] && ( lport != inet[i].lport[0] && lport != inet[i].lport[1] ) )
							continue;
						if ( inet[i].rport[0] && ( rport != inet[i].rport[0] && rport != inet[i].rport[1] ) )
							continue;
						if ( inet[i].laddr[0] && ( laddr != inet[i].laddr[0] && laddr != inet[i].laddr[1] ) )
							continue;
						if ( inet[i].raddr[0] && ( raddr != inet[i].raddr[0] && raddr != inet[i].raddr[1] ) )
							continue;
						inet[i].conn[0]++ ;
					}
				}
			}
			fclose(fp);
		}
	}

	if ( monitor_udp ) {
		if ( fp = fopen( "/proc/net/udp", "r") ) {
			fgets( buf, sizeof(buf), fp );
			while ( fgets(buf, sizeof(buf), fp ) ) {
				sscanf( buf, "%*d: %lx:%x %lx:%x %x", &laddr, &lport, &raddr, &rport, &active );

				if ( active != CONN_ALIVE )
					continue;

				for ( i = 0; i < 5; i++ ) {
					if (inet[i].type == 1 ) {
						if ( inet[i].lport[0] && ( lport != inet[i].lport[0] && lport != inet[i].lport[1] ) )
							continue;
						if ( inet[i].rport[0] && ( rport != inet[i].rport[0] && rport != inet[i].rport[1] ) )
							continue;
						if ( inet[i].laddr[0] && ( laddr != inet[i].laddr[0] && laddr != inet[i].laddr[1] ) )
							continue;
						if ( inet[i].raddr[0] && ( raddr != inet[i].raddr[0] && raddr != inet[i].raddr[1] ) )
							continue;
						inet[i].conn[0]++ ;
					}
				}
			}
			fclose(fp);
		}
	}

	if ( monitor_proc ) {
		if ( fp = fopen( "/proc/loadavg", "r" ) ) {
			fgets( buf, sizeof(buf), fp );
			sscanf( buf, "%*f %*f %*f %*d/%d %*d", &active );
			for ( i = 0; i < 5; i++ ) {
				if ( inet[i].type == 2 )
					inet[i].conn[0] = active;
			}
			fclose(fp);
		}
	}
}

/******************************************/
/* Draw base pixmap                       */
/******************************************/

void draw_base(void)
{
	int i;
	int ycoord;

	for ( i = 0; i < 5; i++ ) {
		ycoord = CHAR_Y + LINE_HEIGHT * i;
		if ( inet[i].type != 3 ) {
			blit_string( "     :   ", CHAR_X, ycoord );
			inet[i].name[5] = 0;
			blit_string( inet[i].name, CHAR_X, ycoord );
		} else {
			blit_string( "         ", CHAR_X, ycoord );
			hostname[9] = 0;
			blit_string( hostname, CHAR_X, ycoord );	
		}
	}
}

/******************************************/
/* Draw number of connections             */
/******************************************/

void draw_nums(void)
{
	int i;
	int xcoord, ycoord;
	char buf[4];

	for ( i = 0; i < 5; i++ ) {
		if ( inet[i].type == 3 )
			continue;

		xcoord = CHAR_X + CHAR_WIDTH * 6;
		ycoord = CHAR_Y + LINE_HEIGHT * i;

		if ( inet[i].conn[0] < 100 )
			sprintf(buf, " %02i", inet[i].conn[0] );
		else if ( inet[i].conn[0] < 1000 )
			sprintf(buf, "%i", inet[i].conn[0] );
		else
			sprintf(buf, "%iK", inet[i].conn[0] / 1000 );

		blit_string( buf, xcoord, ycoord );
	}

}
	
/******************************************/
/* Blit string at co-ordinates            */
/******************************************/

void blit_string(char *name, int x, int y)
{
	int i;
	int c;
	int k;

	k = x;

	for ( i = 0; name[i]; i++ ) {
		c = toupper( name[i] );
		if ( c >= '0' && c <= '9' ) {
			c -= '0';
			copyXPMArea( c * 6, 65, CHAR_WIDTH, CHAR_HEIGHT, k, y ); 
		} else if ( c >= 'A' && c <= 'J' ) {   
			c -= 'A';
			copyXPMArea( c * 6, 73, CHAR_WIDTH, CHAR_HEIGHT, k, y );
		} else if ( c >= 'K' && c <= 'T' ) {
			c -= 'K';
			copyXPMArea( c * 6, 81, CHAR_WIDTH, CHAR_HEIGHT, k, y );
		} else if ( c >= 'U' && c <= 'Z' ) {
			c -= 'U';
			copyXPMArea( c * 6, 89, CHAR_WIDTH, CHAR_HEIGHT, k, y );
		} else if ( c == ':' )
			copyXPMArea( 36, 89, CHAR_WIDTH, CHAR_HEIGHT, k, y );
		else
			copyXPMArea( 54, 97, CHAR_WIDTH, CHAR_HEIGHT, k, y );
		k += 6;
	}
}

/******************************************/
/* Usage                                  */
/******************************************/

void usage(int argc, char **argv)
{
	fprintf(stderr, "\nWminet - Internet Monitor Dock V%s\n\n", VERSION);
	fprintf(stderr, "Usage: %s [ options... ]\n\n", argv[0]);
	fprintf(stderr, "\t-display <display name>\n");
	fprintf(stderr, "\t-geometry +XPOS+YPOS\tinitial window postion\n");
	fprintf(stderr, "\t-s <...>\t\tsample rate in milliseconds  (default:%d)\n", update / 1000 );
	fprintf(stderr, "\t-r <...>\t\trefresh rate in milliseconds (default:%d)\n", refresh / 1000 );
	fprintf(stderr, "\t-c <...>\t\tprogram to run on right click\n");
	fprintf(stderr, "\t-d\t\t\tdebug\n");
	fprintf(stderr, "\t-t\t\t\tuse alternate pixmap\n");
	fprintf(stderr, "\t-h\t\t\tprints this help\n");
}
