/*
* Copyright(c) 2018 Intel Corporation
* SPDX - License - Identifier: BSD - 2 - Clause - Patent
*/

#include <stdlib.h>

#include "EbUtility.h"
#include "EbPictureOperators.h"
#include "EbDefinitions.h"
#include "EbTransforms.h"
#include "EbEncDecProcess.h"
#include "EbLambdaRateTables.h"

static inline EB_U32 CheckNZero4x4(
	EB_S16  *coeff,
	EB_U32   coeffStride
	)
{


	const EB_U32 stride = coeffStride / 4;

	EB_U64 * coefPtr = (EB_U64 *)coeff;

	if (coefPtr[0] > 0)
		return 1;
	else if (coefPtr[stride] > 0)
		return 1;
	else if (coefPtr[2 * stride] > 0)
		return 1;
	else if (coefPtr[3 * stride] > 0)
		return 1;

	return 0;

}

void MatMultOut(
	EB_S16           *coeff,
	const EB_U32     coeffStride,
	EB_S16*          coeffOut,
	const EB_U32     coeffOutStride,
	const EB_U16     *maskingMatrix,
	const EB_U32     maskingMatrixStride,
	const EB_U32     computeSize,
	const EB_S32     offset,
	const EB_S32     shiftNum,
	EB_U32			 *nonzerocoeff){

	EB_U32 coeffLocation = 0, coeffOutLocation=0;
	EB_U32 rowIndex, colIndex;
	EB_S32 coeffTemp;

	*nonzerocoeff = 0;

	for (rowIndex = 0; rowIndex < computeSize; ++rowIndex) {
		for (colIndex = 0; colIndex < computeSize; ++colIndex) {
			coeffTemp = (ABS(coeff[coeffLocation]) * maskingMatrix[colIndex +rowIndex *maskingMatrixStride] + offset) >> shiftNum;
			coeffTemp = (coeff[coeffLocation] < 0) ? -coeffTemp : coeffTemp;

			coeffOut[coeffOutLocation] = (EB_S16)CLIP3(MIN_NEG_16BIT_NUM, MAX_POS_16BIT_NUM, coeffTemp);

			(*nonzerocoeff) += (coeffTemp != 0);
			++coeffLocation;
			++coeffOutLocation;
		}
		coeffLocation += coeffStride - computeSize;
		coeffOutLocation += coeffOutStride - computeSize;
	}
}

EB_ERRORTYPE PmEstimateQuantCoeffLuma_SSE2(
	CabacCost_t                  *CabacCost,
	CabacEncodeContext_t         *cabacEncodeCtxPtr,
	EB_U32                        size,                 // Input: TU size
	EB_MODETYPE                   type,                 // Input: CU type (INTRA, INTER)
	EB_U32                        intraLumaMode,
	EB_U32                        intraChromaMode,
	EB_S16                       *coeffBufferPtr,
	const EB_U32                  coeffStride,
	EB_U32                        componentType,
	EB_U32                        numNonZeroCoeffs,
	EB_U64                       *coeffBitsLong);

EB_ERRORTYPE PmEstimateQuantCoeffChroma_SSE2(
	CabacCost_t                  *CabacCost,
	CabacEncodeContext_t         *cabacEncodeCtxPtr,
	EB_U32                        size,                 // Input: TU size
	EB_MODETYPE                   type,                 // Input: CU type (INTRA, INTER)
	EB_U32                        intraLumaMode,
	EB_U32                        intraChromaMode,
	EB_S16                       *coeffBufferPtr,
	const EB_U32                  coeffStride,
	EB_U32                        componentType,
	EB_U32                        numNonZeroCoeffs,
	EB_U64                       *coeffBitsLong);


typedef EB_ERRORTYPE(*PM_RATE_EST_TYPE)(
	CabacCost_t                  *CabacCost,
	CabacEncodeContext_t         *cabacEncodeCtxPtr,
	EB_U32                        size,
	EB_MODETYPE                   type,                 // Input: CU type (INTRA, INTER)
	EB_U32                        intraLumaMode,
	EB_U32                        intraChromaMode,
	EB_S16                       *coeffBufferPtr,
	const EB_U32                  coeffStride,
	EB_U32                        componentType,
	EB_U32                        numNonZeroCoeffs,
	EB_U64                       *coeffBitsLong);

static PM_RATE_EST_TYPE FUNC_TABLE CoeffRateEst4x4_funcPtrArray[EB_ASM_TYPE_TOTAL][2/*Luma+Chroma*/] =
{
	{ EstimateQuantizedCoefficients_Lossy, EstimateQuantizedCoefficients_Lossy },
	{ PmEstimateQuantCoeffLuma_SSE2, PmEstimateQuantCoeffChroma_SSE2 }
};


/*****************************
 * function header
 *****************************/

EB_U64 EbHevcGetPMCost(
	EB_U64                   lambda,
	EB_U64                   tuDistortion,
	EB_U64                   yTuCoeffBits
	);

EB_ERRORTYPE TuEstimateCoeffBits_R(
	EB_U32                     tuOriginIndex,
	EB_U32                     tuChromaOriginIndex,
	EB_U32                     componentMask,
	EntropyCoder_t            *entropyCoderPtr,
	EbPictureBufferDesc_t     *coeffPtr,
	EB_U32                     yCountNonZeroCoeffs,
	EB_U32                     cbCountNonZeroCoeffs,
	EB_U32                     crCountNonZeroCoeffs,
	EB_U64                    *yTuCoeffBits,
	EB_U64                    *cbTuCoeffBits,
	EB_U64                    *crTuCoeffBits,
	EB_U32                     transformSize,
	EB_U32                     transformChromaSize,
	EB_MODETYPE                type,
	EB_U32					   intraLumaMode,
	EB_U32                     intraChromaMode,
	EB_U32                     partialFrequencyN2Flag,
    EB_BOOL                    coeffCabacUpdate,
    CoeffCtxtMdl_t            *updatedCoeffCtxModel,
	CabacCost_t               *CabacCost);

/*****************************
 * Defines
 *****************************/

#define BETA_P              1
#define BETA_N              3


/********************************************
 * Constants
 ********************************************/

#define ALPHA_0000   0
#define ALPHA_0050   50

#define ALPHA_0100   100
#define ALPHA_0200   200
#define ALPHA_0300   300
#define ALPHA_0500   500
#define ALPHA_1000  1000


const EB_U8 ChromaQpMap[] =
{
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16,
    17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 29, 30, 31, 32,
    33, 33, 34, 34, 35, 35, 36, 36, 37, 37, 38, 39, 40, 41, 42, 43, 44,
    45, 46, 47, 48, 49, 50, 51
};


EB_EXTERN EB_ALIGN(16) const EB_S16 EbHevcTransformAsmConst[] = {
    2, 0, 2, 0, 2, 0, 2, 0,
    4, 0, 4, 0, 4, 0, 4, 0,
    8, 0, 8, 0, 8, 0, 8, 0,
    9, 0, 9, 0, 9, 0, 9, 0,
    13, 0, 13, 0, 13, 0, 13, 0,
    16, 0, 16, 0, 16, 0, 16, 0,
    18, 0, 18, 0, 18, 0, 18, 0,
    22, 0, 22, 0, 22, 0, 22, 0,
    25, 0, 25, 0, 25, 0, 25, 0,
    31, 0, 31, 0, 31, 0, 31, 0,
    36, 0, 36, 0, 36, 0, 36, 0,
    38, 0, 38, 0, 38, 0, 38, 0,
    43, 0, 43, 0, 43, 0, 43, 0,
    46, 0, 46, 0, 46, 0, 46, 0,
    50, 0, 50, 0, 50, 0, 50, 0,
    54, 0, 54, 0, 54, 0, 54, 0,
    57, 0, 57, 0, 57, 0, 57, 0,
    61, 0, 61, 0, 61, 0, 61, 0,
    67, 0, 67, 0, 67, 0, 67, 0,
    70, 0, 70, 0, 70, 0, 70, 0,
    73, 0, 73, 0, 73, 0, 73, 0,
    75, 0, 75, 0, 75, 0, 75, 0,
    78, 0, 78, 0, 78, 0, 78, 0,
    80, 0, 80, 0, 80, 0, 80, 0,
    82, 0, 82, 0, 82, 0, 82, 0,
    83, 0, 83, 0, 83, 0, 83, 0,
    85, 0, 85, 0, 85, 0, 85, 0,
    87, 0, 87, 0, 87, 0, 87, 0,
    88, 0, 88, 0, 88, 0, 88, 0,
    89, 0, 89, 0, 89, 0, 89, 0,
    90, 0, 90, 0, 90, 0, 90, 0,
    256, 0, 256, 0, 256, 0, 256, 0,
    512, 0, 512, 0, 512, 0, 512, 0,
    1024, 0, 1024, 0, 1024, 0, 1024, 0,
    83, 36, 83, 36, 83, 36, 83, 36,
    36, -83, 36, -83, 36, -83, 36, -83,
    89, 75, 89, 75, 89, 75, 89, 75,
    50, 18, 50, 18, 50, 18, 50, 18,
    75, -18, 75, -18, 75, -18, 75, -18,
    -89, -50, -89, -50, -89, -50, -89, -50,
    50, -89, 50, -89, 50, -89, 50, -89,
    18, 75, 18, 75, 18, 75, 18, 75,
    18, -50, 18, -50, 18, -50, 18, -50,
    75, -89, 75, -89, 75, -89, 75, -89,
    90, 87, 90, 87, 90, 87, 90, 87, // 16x16
    80, 70, 80, 70, 80, 70, 80, 70,
    57, 43, 57, 43, 57, 43, 57, 43,
    25, 9, 25, 9, 25, 9, 25, 9,
    87, 57, 87, 57, 87, 57, 87, 57,
    9, -43, 9, -43, 9, -43, 9, -43,
    -80, -90, -80, -90, -80, -90, -80, -90,
    -70, -25, -70, -25, -70, -25, -70, -25,
    80, 9, 80, 9, 80, 9, 80, 9,
    -70, -87, -70, -87, -70, -87, -70, -87,
    -25, 57, -25, 57, -25, 57, -25, 57,
    90, 43, 90, 43, 90, 43, 90, 43,
    70, -43, 70, -43, 70, -43, 70, -43,
    -87, 9, -87, 9, -87, 9, -87, 9,
    90, 25, 90, 25, 90, 25, 90, 25,
    -80, -57, -80, -57, -80, -57, -80, -57,
    57, -80, 57, -80, 57, -80, 57, -80,
    -25, 90, -25, 90, -25, 90, -25, 90,
    -9, -87, -9, -87, -9, -87, -9, -87,
    43, 70, 43, 70, 43, 70, 43, 70,
    43, -90, 43, -90, 43, -90, 43, -90,
    57, 25, 57, 25, 57, 25, 57, 25,
    -87, 70, -87, 70, -87, 70, -87, 70,
    9, -80, 9, -80, 9, -80, 9, -80,
    25, -70, 25, -70, 25, -70, 25, -70,
    90, -80, 90, -80, 90, -80, 90, -80,
    43, 9, 43, 9, 43, 9, 43, 9,
    -57, 87, -57, 87, -57, 87, -57, 87,
    9, -25, 9, -25, 9, -25, 9, -25,
    43, -57, 43, -57, 43, -57, 43, -57,
    70, -80, 70, -80, 70, -80, 70, -80,
    87, -90, 87, -90, 87, -90, 87, -90,
};

EB_ALIGN(16) const EB_S16 TransformAsmConst_SSE4_1[] = {
    2, 0, 2, 0, 2, 0, 2, 0,
    4, 0, 4, 0, 4, 0, 4, 0,
    8, 0, 8, 0, 8, 0, 8, 0,
    9, 0, 9, 0, 9, 0, 9, 0,
    13, 0, 13, 0, 13, 0, 13, 0,
    16, 0, 16, 0, 16, 0, 16, 0,
    18, 0, 18, 0, 18, 0, 18, 0,
    22, 0, 22, 0, 22, 0, 22, 0,
    25, 0, 25, 0, 25, 0, 25, 0,
    31, 0, 31, 0, 31, 0, 31, 0,
    36, 0, 36, 0, 36, 0, 36, 0,
    38, 0, 38, 0, 38, 0, 38, 0,
    43, 0, 43, 0, 43, 0, 43, 0,
    46, 0, 46, 0, 46, 0, 46, 0,
    50, 0, 50, 0, 50, 0, 50, 0,
    54, 0, 54, 0, 54, 0, 54, 0,
    57, 0, 57, 0, 57, 0, 57, 0,
    61, 0, 61, 0, 61, 0, 61, 0,
    67, 0, 67, 0, 67, 0, 67, 0,
    70, 0, 70, 0, 70, 0, 70, 0,
    73, 0, 73, 0, 73, 0, 73, 0,
    75, 0, 75, 0, 75, 0, 75, 0,
    78, 0, 78, 0, 78, 0, 78, 0,
    80, 0, 80, 0, 80, 0, 80, 0,
    82, 0, 82, 0, 82, 0, 82, 0,
    83, 0, 83, 0, 83, 0, 83, 0,
    85, 0, 85, 0, 85, 0, 85, 0,
    87, 0, 87, 0, 87, 0, 87, 0,
    88, 0, 88, 0, 88, 0, 88, 0,
    89, 0, 89, 0, 89, 0, 89, 0,
    90, 0, 90, 0, 90, 0, 90, 0,
    256, 0, 256, 0, 256, 0, 256, 0,
    512, 0, 512, 0, 512, 0, 512, 0,
    1024, 0, 1024, 0, 1024, 0, 1024, 0,
    83, 36, 83, 36, 83, 36, 83, 36,
    36, -83, 36, -83, 36, -83, 36, -83,
    89, 75, 89, 75, 89, 75, 89, 75,
    50, 18, 50, 18, 50, 18, 50, 18,
    75, -18, 75, -18, 75, -18, 75, -18,
    -89, -50, -89, -50, -89, -50, -89, -50,
    50, -89, 50, -89, 50, -89, 50, -89,
    18, 75, 18, 75, 18, 75, 18, 75,
    18, -50, 18, -50, 18, -50, 18, -50,
    75, -89, 75, -89, 75, -89, 75, -89,
    90, 87, 90, 87, 90, 87, 90, 87, // 16x16
    80, 70, 80, 70, 80, 70, 80, 70,
    57, 43, 57, 43, 57, 43, 57, 43,
    25, 9, 25, 9, 25, 9, 25, 9,
    87, 57, 87, 57, 87, 57, 87, 57,
    9, -43, 9, -43, 9, -43, 9, -43,
    -80, -90, -80, -90, -80, -90, -80, -90,
    -70, -25, -70, -25, -70, -25, -70, -25,
    80, 9, 80, 9, 80, 9, 80, 9,
    -70, -87, -70, -87, -70, -87, -70, -87,
    -25, 57, -25, 57, -25, 57, -25, 57,
    90, 43, 90, 43, 90, 43, 90, 43,
    70, -43, 70, -43, 70, -43, 70, -43,
    -87, 9, -87, 9, -87, 9, -87, 9,
    90, 25, 90, 25, 90, 25, 90, 25,
    -80, -57, -80, -57, -80, -57, -80, -57,
    57, -80, 57, -80, 57, -80, 57, -80,
    -25, 90, -25, 90, -25, 90, -25, 90,
    -9, -87, -9, -87, -9, -87, -9, -87,
    43, 70, 43, 70, 43, 70, 43, 70,
    43, -90, 43, -90, 43, -90, 43, -90,
    57, 25, 57, 25, 57, 25, 57, 25,
    -87, 70, -87, 70, -87, 70, -87, 70,
    9, -80, 9, -80, 9, -80, 9, -80,
    25, -70, 25, -70, 25, -70, 25, -70,
    90, -80, 90, -80, 90, -80, 90, -80,
    43, 9, 43, 9, 43, 9, 43, 9,
    -57, 87, -57, 87, -57, 87, -57, 87,
    9, -25, 9, -25, 9, -25, 9, -25,
    43, -57, 43, -57, 43, -57, 43, -57,
    70, -80, 70, -80, 70, -80, 70, -80,
    87, -90, 87, -90, 87, -90, 87, -90,
    90, 90, 90, 90, 90, 90, 90, 90, // 32x32
    88, 85, 88, 85, 88, 85, 88, 85,
    82, 78, 82, 78, 82, 78, 82, 78,
    73, 67, 73, 67, 73, 67, 73, 67,
    61, 54, 61, 54, 61, 54, 61, 54,
    46, 38, 46, 38, 46, 38, 46, 38,
    31, 22, 31, 22, 31, 22, 31, 22,
    13, 4, 13, 4, 13, 4, 13, 4,
    90, 82, 90, 82, 90, 82, 90, 82,
    67, 46, 67, 46, 67, 46, 67, 46,
    22, -4, 22, -4, 22, -4, 22, -4,
    -31, -54, -31, -54, -31, -54, -31, -54,
    -73, -85, -73, -85, -73, -85, -73, -85,
    -90, -88, -90, -88, -90, -88, -90, -88,
    -78, -61, -78, -61, -78, -61, -78, -61,
    -38, -13, -38, -13, -38, -13, -38, -13,
    88, 67, 88, 67, 88, 67, 88, 67,
    31, -13, 31, -13, 31, -13, 31, -13,
    -54, -82, -54, -82, -54, -82, -54, -82,
    -90, -78, -90, -78, -90, -78, -90, -78,
    -46, -4, -46, -4, -46, -4, -46, -4,
    38, 73, 38, 73, 38, 73, 38, 73,
    90, 85, 90, 85, 90, 85, 90, 85,
    61, 22, 61, 22, 61, 22, 61, 22,
    85, 46, 85, 46, 85, 46, 85, 46,
    -13, -67, -13, -67, -13, -67, -13, -67,
    -90, -73, -90, -73, -90, -73, -90, -73,
    -22, 38, -22, 38, -22, 38, -22, 38,
    82, 88, 82, 88, 82, 88, 82, 88,
    54, -4, 54, -4, 54, -4, 54, -4,
    -61, -90, -61, -90, -61, -90, -61, -90,
    -78, -31, -78, -31, -78, -31, -78, -31,
    82, 22, 82, 22, 82, 22, 82, 22,
    -54, -90, -54, -90, -54, -90, -54, -90,
    -61, 13, -61, 13, -61, 13, -61, 13,
    78, 85, 78, 85, 78, 85, 78, 85,
    31, -46, 31, -46, 31, -46, 31, -46,
    -90, -67, -90, -67, -90, -67, -90, -67,
    4, 73, 4, 73, 4, 73, 4, 73,
    88, 38, 88, 38, 88, 38, 88, 38,
    78, -4, 78, -4, 78, -4, 78, -4,
    -82, -73, -82, -73, -82, -73, -82, -73,
    13, 85, 13, 85, 13, 85, 13, 85,
    67, -22, 67, -22, 67, -22, 67, -22,
    -88, -61, -88, -61, -88, -61, -88, -61,
    31, 90, 31, 90, 31, 90, 31, 90,
    54, -38, 54, -38, 54, -38, 54, -38,
    -90, -46, -90, -46, -90, -46, -90, -46,
    73, -31, 73, -31, 73, -31, 73, -31,
    -90, -22, -90, -22, -90, -22, -90, -22,
    78, 67, 78, 67, 78, 67, 78, 67,
    -38, -90, -38, -90, -38, -90, -38, -90,
    -13, 82, -13, 82, -13, 82, -13, 82,
    61, -46, 61, -46, 61, -46, 61, -46,
    -88, -4, -88, -4, -88, -4, -88, -4,
    85, 54, 85, 54, 85, 54, 85, 54,
    67, -54, 67, -54, 67, -54, 67, -54,
    -78, 38, -78, 38, -78, 38, -78, 38,
    85, -22, 85, -22, 85, -22, 85, -22,
    -90, 4, -90, 4, -90, 4, -90, 4,
    90, 13, 90, 13, 90, 13, 90, 13,
    -88, -31, -88, -31, -88, -31, -88, -31,
    82, 46, 82, 46, 82, 46, 82, 46,
    -73, -61, -73, -61, -73, -61, -73, -61,
    61, -73, 61, -73, 61, -73, 61, -73,
    -46, 82, -46, 82, -46, 82, -46, 82,
    31, -88, 31, -88, 31, -88, 31, -88,
    -13, 90, -13, 90, -13, 90, -13, 90,
    -4, -90, -4, -90, -4, -90, -4, -90,
    22, 85, 22, 85, 22, 85, 22, 85,
    -38, -78, -38, -78, -38, -78, -38, -78,
    54, 67, 54, 67, 54, 67, 54, 67,
    54, -85, 54, -85, 54, -85, 54, -85,
    -4, 88, -4, 88, -4, 88, -4, 88,
    -46, -61, -46, -61, -46, -61, -46, -61,
    82, 13, 82, 13, 82, 13, 82, 13,
    -90, 38, -90, 38, -90, 38, -90, 38,
    67, -78, 67, -78, 67, -78, 67, -78,
    -22, 90, -22, 90, -22, 90, -22, 90,
    -31, -73, -31, -73, -31, -73, -31, -73,
    46, -90, 46, -90, 46, -90, 46, -90,
    38, 54, 38, 54, 38, 54, 38, 54,
    -90, 31, -90, 31, -90, 31, -90, 31,
    61, -88, 61, -88, 61, -88, 61, -88,
    22, 67, 22, 67, 22, 67, 22, 67,
    -85, 13, -85, 13, -85, 13, -85, 13,
    73, -82, 73, -82, 73, -82, 73, -82,
    4, 78, 4, 78, 4, 78, 4, 78,
    38, -88, 38, -88, 38, -88, 38, -88,
    73, -4, 73, -4, 73, -4, 73, -4,
    -67, 90, -67, 90, -67, 90, -67, 90,
    -46, -31, -46, -31, -46, -31, -46, -31,
    85, -78, 85, -78, 85, -78, 85, -78,
    13, 61, 13, 61, 13, 61, 13, 61,
    -90, 54, -90, 54, -90, 54, -90, 54,
    22, -82, 22, -82, 22, -82, 22, -82,
    31, -78, 31, -78, 31, -78, 31, -78,
    90, -61, 90, -61, 90, -61, 90, -61,
    4, 54, 4, 54, 4, 54, 4, 54,
    -88, 82, -88, 82, -88, 82, -88, 82,
    -38, -22, -38, -22, -38, -22, -38, -22,
    73, -90, 73, -90, 73, -90, 73, -90,
    67, -13, 67, -13, 67, -13, 67, -13,
    -46, 85, -46, 85, -46, 85, -46, 85,
    22, -61, 22, -61, 22, -61, 22, -61,
    85, -90, 85, -90, 85, -90, 85, -90,
    73, -38, 73, -38, 73, -38, 73, -38,
    -4, 46, -4, 46, -4, 46, -4, 46,
    -78, 90, -78, 90, -78, 90, -78, 90,
    -82, 54, -82, 54, -82, 54, -82, 54,
    -13, -31, -13, -31, -13, -31, -13, -31,
    67, -88, 67, -88, 67, -88, 67, -88,
    13, -38, 13, -38, 13, -38, 13, -38,
    61, -78, 61, -78, 61, -78, 61, -78,
    88, -90, 88, -90, 88, -90, 88, -90,
    85, -73, 85, -73, 85, -73, 85, -73,
    54, -31, 54, -31, 54, -31, 54, -31,
    4, 22, 4, 22, 4, 22, 4, 22,
    -46, 67, -46, 67, -46, 67, -46, 67,
    -82, 90, -82, 90, -82, 90, -82, 90,
    4, -13, 4, -13, 4, -13, 4, -13,
    22, -31, 22, -31, 22, -31, 22, -31,
    38, -46, 38, -46, 38, -46, 38, -46,
    54, -61, 54, -61, 54, -61, 54, -61,
    67, -73, 67, -73, 67, -73, 67, -73,
    78, -82, 78, -82, 78, -82, 78, -82,
    85, -88, 85, -88, 85, -88, 85, -88,
    90, -90, 90, -90, 90, -90, 90, -90,
};

#define PMP_PRECISION     8
#define PMP_MAX          (1<<PMP_PRECISION)

#define M_100      100*PMP_MAX/100
#define M_90        90*PMP_MAX/100
#define M_80        80*PMP_MAX/100
#define M_70        70*PMP_MAX/100
#define M_60        60*PMP_MAX/100
#define M_50        50*PMP_MAX/100
#define M_40        40*PMP_MAX/100
#define M_30        30*PMP_MAX/100
#define M_25        25*PMP_MAX/100
#define M_20        20*PMP_MAX/100
#define M_10        10*PMP_MAX/100
#define M_0		     0*PMP_MAX/100

// Level0
// 4K
// 4x4
static const EB_U16 MaskingMatrix4x4_Level0_4K[] = {
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100
};
// 8x8
static const EB_U16 MaskingMatrix8x8_Level0_4K[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};
// 16x16
static const EB_U16 MaskingMatrix16x16_Level0_4K[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};
// 32x32
static const EB_U16 MaskingMatrix32x32_Level0_4K[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};

// 1080
// 4x4
static const EB_U16 MaskingMatrix4x4_Level0_1080p[] = {
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100
};
// 8x8
static const EB_U16 MaskingMatrix8x8_Level0_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};
// 16x16
static const EB_U16 MaskingMatrix16x16_Level0_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};
// 32x32
static const EB_U16 MaskingMatrix32x32_Level0_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};


// Level1
// 4K


// 4x4
static const EB_U16 MaskingMatrix4x4_Level1_4K[] = {
    M_100, M_100, M_50, M_50,
    M_100, M_100, M_50, M_50,
    M_50, M_50, M_50, M_50,
    M_50, M_50, M_50, M_50,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_Level1_4K[] = {
    M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50,
};

static const EB_U16 MaskingMatrix8x8_Level1_MOD1_4K[] = {
    M_100, M_100, M_100, M_100, M_80, M_80, M_80, M_80,
    M_100, M_100, M_100, M_100, M_80, M_80, M_80, M_80,
    M_100, M_100, M_100, M_100, M_70, M_70, M_70, M_70,
    M_100, M_100, M_100, M_100, M_70, M_70, M_70, M_70,
    M_80, M_80, M_70, M_70, M_70, M_70, M_70, M_70,
    M_80, M_80, M_70, M_70, M_70, M_70, M_70, M_70,
    M_80, M_80, M_70, M_70, M_70, M_70, M_70, M_70,
    M_80, M_80, M_70, M_70, M_70, M_70, M_70, M_70,
};

// 16x16
static const EB_U16 MaskingMatrix16x16_Level1_4K[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
};
// 32x32
static const EB_U16 MaskingMatrix32x32_Level1_4K[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
};

// 1080

// 4x4
static const EB_U16 MaskingMatrix4x4_Level1_1080p[] = {
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100
};
// 8x8
static const EB_U16 MaskingMatrix8x8_Level1_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};
// 16x16
static const EB_U16 MaskingMatrix16x16_Level1_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};
// 32x32
static const EB_U16 MaskingMatrix32x32_Level1_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};

// Level2
// 4K
// 4x4
static const EB_U16 MaskingMatrix4x4_Level2_4K[] = {
    M_100, M_100, M_0, M_0,
    M_100, M_100, M_0, M_0,
    M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_Level2_4K[] = {
    M_100, M_100, M_100, M_100, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
};
// 16x16
static const EB_U16 MaskingMatrix16x16_Level2_4K[] = {
    M_100, M_100, M_100, M_100, M_90, M_90, M_90, M_90, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_90, M_90, M_90, M_90, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
};
// 32x32
static const EB_U16 MaskingMatrix32x32_Level2_4K[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_90, M_90, M_90, M_90, M_90, M_90, M_90, M_90, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_90, M_90, M_90, M_90, M_90, M_90, M_90, M_90, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_90, M_90, M_90, M_90, M_90, M_90, M_90, M_90, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_90, M_90, M_90, M_90, M_90, M_90, M_90, M_90, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_80, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
};


// 1080

// 4x4
static const EB_U16 MaskingMatrix4x4_Level2_1080p[] = {
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100
};
// 8x8
static const EB_U16 MaskingMatrix8x8_Level2_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};
// 16x16
static const EB_U16 MaskingMatrix16x16_Level2_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};
// 32x32
static const EB_U16 MaskingMatrix32x32_Level2_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100
};


// Level3
// 4K
// 4x4
static const EB_U16 MaskingMatrix4x4_Level3_4K[] = {
    M_100, M_90, M_0, M_0,
    M_90, M_90, M_0, M_0,
    M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0,
};

// 1080

// 4x4
static const EB_U16 MaskingMatrix4x4_Level3_1080p[] = {
    M_100, M_100, M_50, M_50,
    M_100, M_100, M_50, M_50,
    M_50, M_50, M_50, M_50,
    M_50, M_50, M_50, M_50,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_Level3_1080p[] = {
    M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50,
};
// 16x16
static const EB_U16 MaskingMatrix16x16_Level3_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
};
// 32x32
static const EB_U16 MaskingMatrix32x32_Level3_1080p[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
};

// Set 1
// 4x4
static const EB_U16 MaskingMatrix4x4_Level4_4K_Set1[] = {
    M_100, M_60, M_0, M_0,
    M_60, M_60, M_0, M_0,
    M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0,
};

// Set 1
// 4x4
static const EB_U16 MaskingMatrix4x4_Level5_4K_Set1[] = {
    M_100, M_50, M_0, M_0,
    M_50, M_50, M_0, M_0,
    M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0,
};

// 1080

// Set 1
// 4x4
static const EB_U16 MaskingMatrix4x4_Level6_4K_Set1[] = {
    M_100, M_25, M_0, M_0,
    M_25, M_25, M_0, M_0,
    M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_Level6_4K_Set1[] = {
    M_100, M_25, M_25, M_25, M_0, M_0, M_0, M_0,
    M_25, M_25, M_25, M_25, M_0, M_0, M_0, M_0,
    M_25, M_25, M_25, M_25, M_0, M_0, M_0, M_0,
    M_25, M_25, M_25, M_25, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
};

// Set 2
// 16x16
static const EB_U16 MaskingMatrix16x16_Level6_4K_Set2[] = {
    M_100, M_80, M_80, M_80, M_40, M_40, M_40, M_40, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_80, M_80, M_80, M_80, M_40, M_40, M_40, M_40, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_80, M_80, M_80, M_80, M_25, M_25, M_25, M_25, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_80, M_80, M_80, M_80, M_25, M_25, M_25, M_25, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_40, M_40, M_25, M_25, M_25, M_25, M_25, M_25, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_40, M_40, M_25, M_25, M_25, M_25, M_25, M_25, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_40, M_40, M_25, M_25, M_25, M_25, M_25, M_25, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_40, M_40, M_25, M_25, M_25, M_25, M_25, M_25, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,

};
// 32x32
static const EB_U16 MaskingMatrix32x32_Level6_4K_Set2[] = {
    M_100, M_90, M_90, M_90, M_60, M_60, M_60, M_60, M_20, M_20, M_20, M_20, M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_60, M_60, M_60, M_60, M_20, M_20, M_20, M_20, M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_50, M_50, M_50, M_50, M_20, M_20, M_20, M_20, M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_90, M_90, M_90, M_90, M_50, M_50, M_50, M_50, M_20, M_20, M_20, M_20, M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_60, M_60, M_50, M_50, M_50, M_50, M_50, M_50, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,

};

// Set 1
// 4x4
static const EB_U16 MaskingMatrix4x4_Level7_4K_Set1[] = {
    M_100, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0,
};

// Set 2
// 16x16

// 32x32
static const EB_U16 MaskingMatrix32x32_Level7_4K_Set2[] = {
    M_100, M_100, M_100, M_100, M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_20, M_20, M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_100, M_100, M_100, M_100, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_20, M_20, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,
    M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0, M_0,

};



// 1080
static const EB_U16 *MaskingMatrix[2][8][4] =//
{
    /****************** 4K ************************/
    {
        { MaskingMatrix4x4_Level0_4K, MaskingMatrix8x8_Level0_4K, MaskingMatrix16x16_Level0_4K, MaskingMatrix32x32_Level0_4K }, // Level 0 OFF
        { MaskingMatrix4x4_Level1_4K, MaskingMatrix8x8_Level1_4K, MaskingMatrix16x16_Level1_4K, MaskingMatrix32x32_Level1_4K }, // Level 1 I_PICTURE
        /*************************  L23_SETTING *************************/
        { MaskingMatrix4x4_Level2_4K, MaskingMatrix8x8_Level2_4K, MaskingMatrix16x16_Level2_4K, MaskingMatrix32x32_Level2_4K }, // Level 2 Base Intra
        { MaskingMatrix4x4_Level3_4K, MaskingMatrix8x8_Level1_MOD1_4K, MaskingMatrix16x16_Level1_4K, MaskingMatrix32x32_Level1_4K }, // Level 3 Base Inter
        /*************************  L45_SETTING *************************/
        { MaskingMatrix4x4_Level4_4K_Set1, MaskingMatrix8x8_Level2_4K, MaskingMatrix16x16_Level2_4K, MaskingMatrix32x32_Level2_4K }, // Level 4 Ref Intra
        { MaskingMatrix4x4_Level5_4K_Set1, MaskingMatrix8x8_Level1_MOD1_4K, MaskingMatrix16x16_Level1_4K, MaskingMatrix32x32_Level1_4K }, // Level 5 Ref Inter
        /*************************  L67_SETTING *************************/
        { MaskingMatrix4x4_Level6_4K_Set1, MaskingMatrix8x8_Level6_4K_Set1, MaskingMatrix16x16_Level6_4K_Set2, MaskingMatrix32x32_Level6_4K_Set2 }, // Level 6 Non Ref Intra
        { MaskingMatrix4x4_Level7_4K_Set1, MaskingMatrix8x8_Level1_MOD1_4K, MaskingMatrix16x16_Level1_4K, MaskingMatrix32x32_Level7_4K_Set2 }  // Level 7 Non Ref Inter
    },
    /****************** 1080P ************************/
    {
        { MaskingMatrix4x4_Level0_1080p, MaskingMatrix8x8_Level0_1080p, MaskingMatrix16x16_Level0_1080p, MaskingMatrix32x32_Level0_1080p }, // Level 0 OFF
        { MaskingMatrix4x4_Level1_1080p, MaskingMatrix8x8_Level1_1080p, MaskingMatrix16x16_Level1_1080p, MaskingMatrix32x32_Level1_1080p }, // Level 1 I_PICTURE
        /*************************  L23_SETTING *************************/
        { MaskingMatrix4x4_Level2_1080p, MaskingMatrix8x8_Level2_1080p, MaskingMatrix16x16_Level2_1080p, MaskingMatrix32x32_Level2_1080p }, // Level 2 Base Intra
        { MaskingMatrix4x4_Level2_1080p, MaskingMatrix8x8_Level2_1080p, MaskingMatrix16x16_Level2_1080p, MaskingMatrix32x32_Level2_1080p }, // Level 3 Base Inter
        /*************************  L45_SETTING *************************/
        { MaskingMatrix4x4_Level2_1080p, MaskingMatrix8x8_Level2_1080p, MaskingMatrix16x16_Level2_1080p, MaskingMatrix32x32_Level2_1080p }, // Level 4 Ref Intra
        { MaskingMatrix4x4_Level3_1080p, MaskingMatrix8x8_Level3_1080p, MaskingMatrix16x16_Level3_1080p, MaskingMatrix32x32_Level3_1080p }, // Level 5 Ref Inter
        /*************************  L67_SETTING *************************/
        { MaskingMatrix4x4_Level3_1080p, MaskingMatrix8x8_Level3_1080p, MaskingMatrix16x16_Level3_1080p, MaskingMatrix32x32_Level3_1080p }, // Level 6 Non Ref Intra
        { MaskingMatrix4x4_Level3_1080p, MaskingMatrix8x8_Level3_1080p, MaskingMatrix16x16_Level3_1080p, MaskingMatrix32x32_Level3_1080p }, // Level 7 Non Ref Inter
    },
};

#define M_095  95 * PMP_MAX / 100

#define M_090  90 * PMP_MAX / 100
#define M_080  80 * PMP_MAX / 100
#define M_070  70 * PMP_MAX / 100
#define M_060  60 * PMP_MAX / 100
#define M_050  50 * PMP_MAX / 100
#define M_040  40 * PMP_MAX / 100
#define M_030  30 * PMP_MAX / 100
#define M_025  25 * PMP_MAX / 100
#define M_020  20 * PMP_MAX / 100
#define M_010  10 * PMP_MAX / 100
#define M_000   0 * PMP_MAX / 100

static const EB_U16 MaskingMatrix4x4_100[] = {
	M_100, M_100, M_100, M_100,
	M_100, M_100, M_100, M_100,
	M_100, M_100, M_100, M_100,
	M_100, M_100, M_100, M_100
};
static const EB_U16 MaskingMatrix4x4_70[] = {
	M_70, M_70, M_70, M_70,
	M_70, M_70, M_70, M_70,
	M_70, M_70, M_70, M_70,
	M_70, M_70, M_70, M_70
};
static const EB_U16 MaskingMatrix4x4_50[] = {
	M_50, M_50, M_50, M_50,
	M_50, M_50, M_50, M_50,
	M_50, M_50, M_50, M_50,
	M_50, M_50, M_50, M_50
};

// Group 0 (identity)
// 4x4
static const EB_U16 MaskingMatrix4x4_0[] = {
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100
};
// 8x8
static const EB_U16 MaskingMatrix8x8_0[] = {
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100
};
// 16x16
static const EB_U16 MaskingMatrix16x16_0[] = {
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100
};
// 32x32
static const EB_U16 MaskingMatrix32x32_0[] = {
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100
};


// Group 1
// 4x4
static const EB_U16 MaskingMatrix4x4_1[] = {
    M_100, M_100, M_050, M_050,
    M_100, M_100, M_050, M_050,
    M_050, M_050, M_050, M_050,
    M_050, M_050, M_050, M_050,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_1[] = {
    M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,
};

// 16x16
static const EB_U16 MaskingMatrix16x16_1[] = {
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
};
// 32x32
static const EB_U16 MaskingMatrix32x32_1[] = {
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
    M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_100, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50, M_50,
};

// Group 2
// 32x32
static const EB_U16 MaskingMatrix32x32_2[] = {
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
};

static const EB_U16 MaskingMatrix4x4_1_1[] = {
    M_095, M_090, M_050, M_050,
    M_090, M_090, M_050, M_050,
    M_050, M_050, M_050, M_050,
    M_050, M_050, M_050, M_050,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_1_1[] = {
    M_095, M_090, M_090, M_090,     M_060, M_060, M_060, M_060,
    M_090, M_090, M_090, M_090,     M_060, M_060, M_060, M_060,
    M_090, M_090, M_090, M_090,     M_050, M_050, M_050, M_050,
    M_090, M_090, M_090, M_090,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,
};

// 16x16
static const EB_U16 MaskingMatrix16x16_1_1[] = {
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,

    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
};

// Group 2
// 32x32
static const EB_U16 MaskingMatrix32x32_2_1[] = {
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,

    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,

    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,

    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
    M_060, M_060, M_060, M_060,     M_060, M_060, M_060, M_060,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,     M_050, M_050, M_050, M_050,
};

// Group3_1
// 4x4
static const EB_U16 MaskingMatrix4x4_3_1[] = {
	M_095, M_095, M_000, M_000,
	M_095, M_095, M_000, M_000,
	M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_3_1[] = {
	M_095, M_095, M_095, M_095,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};
// 16x16
static const EB_U16 MaskingMatrix16x16_3_1[] = {
	M_095, M_095, M_095, M_095,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_090, M_090, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_090, M_090, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_090, M_090, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_090, M_090, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};
// 32x32
static const EB_U16 MaskingMatrix32x32_3_1[] = {
	M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_095, M_095, M_095, M_095,     M_095, M_095, M_095, M_095,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};

// Group4_1
// 4x4
static const EB_U16 MaskingMatrix4x4_4_1[] = {
	M_095, M_080, M_000, M_000,
	M_080, M_080, M_000, M_000,
	M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_4_1[] = {
	M_095, M_080, M_080, M_080, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
};

// 16x16
static const EB_U16 MaskingMatrix16x16_4_1[] = {
	M_095, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
};
// 32x32
static const EB_U16 MaskingMatrix32x32_4_1[] = {
	M_095, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_080, M_080, M_080, M_080, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_070, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
	M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
};



// Group3
// 4x4
static const EB_U16 MaskingMatrix4x4_3[] = {
    M_100, M_100, M_000, M_000,
    M_100, M_100, M_000, M_000,
    M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_3[] = {
    M_100, M_100, M_100, M_100,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};
// 16x16
static const EB_U16 MaskingMatrix16x16_3[] = {
    M_100, M_100, M_100, M_100,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_090, M_090, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};
// 32x32
static const EB_U16 MaskingMatrix32x32_3[] = {
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_100, M_100, M_100, M_100,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};

// Group4
// 4x4
static const EB_U16 MaskingMatrix4x4_4[] = {
    M_100, M_090, M_000, M_000,
    M_090, M_090, M_000, M_000,
    M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_4[] = {
    M_100, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};
// 16x16
static const EB_U16 MaskingMatrix16x16_4[] = {
    M_100, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_080, M_080, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_080, M_080, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_080, M_080, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_080, M_080, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};
// 32x32
static const EB_U16 MaskingMatrix32x32_4[] = {
    M_100, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_080, M_080, M_080, M_080,     M_080, M_080, M_080, M_080,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_090, M_090, M_090, M_090,     M_090, M_090, M_090, M_090,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_080, M_080, M_080, M_080,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_080, M_080, M_080, M_080,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_080, M_080, M_080, M_080,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_080, M_080, M_080, M_080,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_080, M_080, M_080, M_080,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_080, M_080, M_080, M_080,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_080, M_080, M_080, M_080,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_080, M_080, M_080, M_080,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_070, M_070, M_070, M_070,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};

// Group 5
// 4x4
static const EB_U16 MaskingMatrix4x4_5[] = {
    M_100, M_60, M_0, M_0,
    M_60, M_60, M_0, M_0,
    M_000, M_0, M_0, M_0,
    M_000, M_0, M_0, M_0,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_5[] = {
    M_100, M_060, M_060, M_060,     M_000, M_000, M_000, M_000,
    M_060, M_060, M_060, M_060,     M_000, M_000, M_000, M_000,
    M_060, M_060, M_060, M_060,     M_000, M_000, M_000, M_000,
    M_060, M_060, M_060, M_060,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};

// 16x16
static const EB_U16 MaskingMatrix16x16_5[] = {
    M_100, M_100, M_100, M_100,     M_040, M_040, M_040, M_040,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_040, M_040, M_040, M_040,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_040, M_040, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_040, M_040, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_040, M_040, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_040, M_040, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};
// 32x32
static const EB_U16 MaskingMatrix32x32_5[] = {
    M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_040, M_040, M_040, M_040,     M_040, M_040, M_040, M_040,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_060, M_060, M_060, M_060,     M_040, M_040, M_040, M_040,     M_040, M_040, M_040, M_040,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_040, M_040, M_040, M_040,     M_040, M_040, M_040, M_040,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_050, M_050, M_050, M_050,     M_040, M_040, M_040, M_040,     M_040, M_040, M_040, M_040,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_060, M_060, M_050, M_050,     M_050, M_050, M_050, M_050,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_040, M_040, M_040, M_040,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_040, M_040, M_040, M_040,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_040, M_040, M_040, M_040,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_040, M_040, M_040, M_040,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_040, M_040, M_040, M_040,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_040, M_040, M_040, M_040,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_040, M_040, M_040, M_040,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_040, M_040, M_040, M_040,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

};

// Group6

// 4x4
static const EB_U16 MaskingMatrix4x4_6[] = {
    M_100, M_050, M_000, M_000,
    M_050, M_050, M_000, M_000,
    M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,
};
// 8x8
static const EB_U16 MaskingMatrix8x8_6[] = {
    M_100, M_050, M_050, M_050, M_000, M_000, M_000, M_000,
    M_050, M_050, M_050, M_050, M_000, M_000, M_000, M_000,
    M_050, M_050, M_050, M_050, M_000, M_000, M_000, M_000,
    M_050, M_050, M_050, M_050, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
};

// 16x16
static const EB_U16 MaskingMatrix16x16_6[] = {
    M_100, M_100, M_100, M_100,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_025, M_025, M_025, M_025,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_025, M_025, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_025, M_025, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_025, M_025, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_025, M_025, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,     M_000, M_000, M_000, M_000,
};

// 32x32
static const EB_U16 MaskingMatrix32x32_6[] = {

    M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_025, M_025, M_025, M_025,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_025, M_025, M_025, M_025,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_025, M_025, M_025, M_025,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_025, M_025, M_025, M_025,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

    M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,  M_100, M_100, M_100, M_100,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

    M_100, M_100, M_100, M_100,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_100, M_100, M_100, M_100,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

    M_025, M_025, M_025, M_025,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_025, M_025, M_025, M_025,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_025, M_025, M_025, M_025,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_025, M_025, M_025, M_025,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,

    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
    M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,  M_000, M_000, M_000, M_000,   M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000, M_000,
};

// Set0
static const EB_U16 *MaskingMatrixSet0[8][4] =
{
	/*************************  L01_SETTING *************************/
	{ MaskingMatrix4x4_0, MaskingMatrix8x8_0, MaskingMatrix16x16_0, MaskingMatrix32x32_0 },
	{ MaskingMatrix4x4_1, MaskingMatrix8x8_1, MaskingMatrix16x16_1, MaskingMatrix32x32_1 },
	/*************************  L23_SETTING *************************/
	{ MaskingMatrix4x4_1_1, MaskingMatrix8x8_1_1, MaskingMatrix16x16_1_1, MaskingMatrix32x32_2_1 },
	{ MaskingMatrix4x4_1_1, MaskingMatrix8x8_1_1, MaskingMatrix16x16_1_1, MaskingMatrix32x32_2_1 },
	/*************************  L45_SETTING *************************/
	{ MaskingMatrix4x4_1_1, MaskingMatrix8x8_1_1, MaskingMatrix16x16_1_1, MaskingMatrix32x32_2_1 },
	{ MaskingMatrix4x4_1_1, MaskingMatrix8x8_1_1, MaskingMatrix16x16_1_1, MaskingMatrix32x32_2_1 },
	/*************************  L67_SETTING *************************/
	{ MaskingMatrix4x4_1_1, MaskingMatrix8x8_1_1, MaskingMatrix16x16_1_1, MaskingMatrix32x32_2_1 },
	{ MaskingMatrix4x4_1_1, MaskingMatrix8x8_1_1, MaskingMatrix16x16_1_1, MaskingMatrix32x32_2_1 },
};
// Set1
static const EB_U16 *MaskingMatrixSet1[8][4] =
{
    /*************************  L01_SETTING *************************/
    { MaskingMatrix4x4_0, MaskingMatrix8x8_0, MaskingMatrix16x16_0, MaskingMatrix32x32_0},
    { MaskingMatrix4x4_1_1, MaskingMatrix8x8_1_1, MaskingMatrix16x16_1_1, MaskingMatrix32x32_2_1},
    /*************************  L23_SETTING *************************/
    { MaskingMatrix4x4_1_1, MaskingMatrix8x8_1_1, MaskingMatrix16x16_1_1, MaskingMatrix32x32_2_1 },
    { MaskingMatrix4x4_1_1, MaskingMatrix8x8_1_1, MaskingMatrix16x16_1_1, MaskingMatrix32x32_2_1 },
    /*************************  L45_SETTING *************************/
	{ MaskingMatrix4x4_3_1, MaskingMatrix8x8_3_1, MaskingMatrix16x16_3_1, MaskingMatrix32x32_3_1 },
	{ MaskingMatrix4x4_4_1, MaskingMatrix8x8_4_1, MaskingMatrix16x16_4_1, MaskingMatrix32x32_4_1 },
    /*************************  L67_SETTING *************************/
	{ MaskingMatrix4x4_3_1, MaskingMatrix8x8_3_1, MaskingMatrix16x16_3_1, MaskingMatrix32x32_3_1 },
	{ MaskingMatrix4x4_4_1, MaskingMatrix8x8_4_1, MaskingMatrix16x16_4_1, MaskingMatrix32x32_4_1 },
};
// Set1 1080P
static const EB_U16 *MaskingMatrix1080PSet1[8][4] =
{
	/*************************  L01_SETTING *************************/
	{ MaskingMatrix4x4_0, MaskingMatrix8x8_0, MaskingMatrix16x16_0, MaskingMatrix32x32_0},
	{ MaskingMatrix4x4_1, MaskingMatrix8x8_1, MaskingMatrix16x16_1, MaskingMatrix32x32_2},
	/*************************  L23_SETTING *************************/
	{ MaskingMatrix4x4_3, MaskingMatrix8x8_3, MaskingMatrix16x16_3, MaskingMatrix32x32_3},
	{ MaskingMatrix4x4_4, MaskingMatrix8x8_4, MaskingMatrix16x16_4, MaskingMatrix32x32_4},
	/*************************  L45_SETTING *************************/
	{ MaskingMatrix4x4_3, MaskingMatrix8x8_3, MaskingMatrix16x16_3, MaskingMatrix32x32_3},
	{ MaskingMatrix4x4_4, MaskingMatrix8x8_4, MaskingMatrix16x16_4, MaskingMatrix32x32_4},
	/*************************  L67_SETTING *************************/
	{ MaskingMatrix4x4_3, MaskingMatrix8x8_3, MaskingMatrix16x16_3, MaskingMatrix32x32_3},
	{ MaskingMatrix4x4_4, MaskingMatrix8x8_4, MaskingMatrix16x16_4, MaskingMatrix32x32_4},
};

// Set2
static const EB_U16 *MaskingMatrixSet2[8][4] =
{
    /*************************  L01_SETTING *************************/
    { MaskingMatrix4x4_0, MaskingMatrix8x8_0, MaskingMatrix16x16_0, MaskingMatrix32x32_0},
    { MaskingMatrix4x4_1, MaskingMatrix8x8_1, MaskingMatrix16x16_1, MaskingMatrix32x32_2},
    /*************************  L23_SETTING *************************/
    { MaskingMatrix4x4_1, MaskingMatrix8x8_1, MaskingMatrix16x16_1, MaskingMatrix32x32_2 },
    { MaskingMatrix4x4_1, MaskingMatrix8x8_1, MaskingMatrix16x16_1, MaskingMatrix32x32_2 },
    /*************************  L45_SETTING *************************/
    { MaskingMatrix4x4_5, MaskingMatrix8x8_5, MaskingMatrix16x16_5, MaskingMatrix32x32_5},
    { MaskingMatrix4x4_6, MaskingMatrix8x8_6, MaskingMatrix16x16_6, MaskingMatrix32x32_6 },
    /*************************  L67_SETTING *************************/
    { MaskingMatrix4x4_5, MaskingMatrix8x8_5, MaskingMatrix16x16_5, MaskingMatrix32x32_5},
    { MaskingMatrix4x4_6, MaskingMatrix8x8_6, MaskingMatrix16x16_6, MaskingMatrix32x32_6 },

};

static const EB_U16 *MaskingMatrix1080PSet0[8][4] =
{
	/*************************  L01_SETTING *************************/
	{ MaskingMatrix4x4_0, MaskingMatrix8x8_0, MaskingMatrix16x16_0, MaskingMatrix32x32_0 },
	{ MaskingMatrix4x4_0, MaskingMatrix8x8_0, MaskingMatrix16x16_0, MaskingMatrix32x32_0 },
	/*************************  L23_SETTING *************************/
	{ MaskingMatrix4x4_0, MaskingMatrix8x8_0, MaskingMatrix16x16_0, MaskingMatrix32x32_0 },
	{ MaskingMatrix4x4_0, MaskingMatrix8x8_0, MaskingMatrix16x16_0, MaskingMatrix32x32_0 },
	/*************************  L45_SETTING *************************/
	{ MaskingMatrix4x4_0, MaskingMatrix8x8_0, MaskingMatrix16x16_0, MaskingMatrix32x32_0 },
	{ MaskingMatrix4x4_1, MaskingMatrix8x8_1, MaskingMatrix16x16_1, MaskingMatrix32x32_2 },
	/*************************  L67_SETTING *************************/
	{ MaskingMatrix4x4_1, MaskingMatrix8x8_1, MaskingMatrix16x16_1, MaskingMatrix32x32_2 },
	{ MaskingMatrix4x4_1, MaskingMatrix8x8_1, MaskingMatrix16x16_1, MaskingMatrix32x32_2 },
};






void MatMult(
    EB_S16           *coeff,
    const EB_U32     coeffStride,
    const EB_U16     *maskingMatrix,
    const EB_U32     maskingMatrixStride,
    const EB_U32     computeSize,
    const EB_S32     offset,
    const EB_S32     shiftNum,
    EB_U32			 *nonzerocoeff){

    EB_U32 coeffLocation = 0;
    EB_U32 rowIndex, colIndex;
    EB_S32 coeffTemp;

    *nonzerocoeff = 0;

    for (rowIndex = 0; rowIndex < computeSize; ++rowIndex) {
        for (colIndex = 0; colIndex < computeSize; ++colIndex) {
			coeffTemp = (ABS(coeff[coeffLocation]) * maskingMatrix[colIndex + rowIndex*maskingMatrixStride] + offset) >> shiftNum;
            coeffTemp = (coeff[coeffLocation] < 0) ? -coeffTemp : coeffTemp;

            coeff[coeffLocation] = (EB_S16)CLIP3(MIN_NEG_16BIT_NUM, MAX_POS_16BIT_NUM, coeffTemp);
            (*nonzerocoeff) += (coeffTemp != 0);
            ++coeffLocation;
        }
        coeffLocation += coeffStride - computeSize;
    }
}

void MaskTransformCoeffs(
    EB_S16          *coeff,
    const EB_U32     coeffStride,
    EB_U32           areaSize,
    EB_U32           activeAreaSize,
    EB_U8            pmpMaskingLevelEncDec,
    PictureControlSet_t  *pictureControlSetPtr,
    EB_U32          *nonzerocoeff)
{
    EB_U8 depthIndex;

    SequenceControlSet_t *sequenceControlSetPtr = (SequenceControlSet_t*)pictureControlSetPtr->ParentPcsPtr->sequenceControlSetWrapperPtr->objectPtr;
    EB_U8     pmpMaskingClass = (sequenceControlSetPtr->inputResolution == INPUT_SIZE_4K_RANGE) ? 0 : 1;
    depthIndex = LOG2F(areaSize) - 2;

	depthIndex = depthIndex < 4 ? depthIndex : 0;

	MatMul_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)][activeAreaSize >> 3](
        coeff,
        coeffStride,
        &MaskingMatrix[pmpMaskingClass][pmpMaskingLevelEncDec][depthIndex][0],
        areaSize,
        activeAreaSize,
        (PMP_MAX >> 1),
        PMP_PRECISION,
        nonzerocoeff);

}

/*********************************************************************
 * PerformTwoStagePm
 *
 * First phase (non-core): three different PM candidates are determined to be evaluated in the second phase. The PM candidates are distinct in term of VQ (i.e. the susceptibility to cause VQ artifact(s)),
 * and in term of bitrate (i.e. the capability to save bits). The PM candidates are constructed among pre-defined sets of matrices that are derived offline.
 * Second stage (core), the provided PM candidates are evaluated based on the rate distortion optimization (RDO) metric and the best candidate is selected at a 4x4 block basis.
 * In this process, each TU is divided into 4x4 blocks, and each 4x4 is evaluated with up to three masking matrices provided from the first phase.
 * Weighting factors are used to bias the decisions based on HVS concepts like distance from DC, temporal layer and the block mode.
  *********************************************************************/
void PerformTwoStagePm(
    SequenceControlSet_t *sequenceControlSetPtr,
    PictureControlSet_t  *pictureControlSetPtr,
	EncDecContext_t      *contextPtr,
    EB_S16               *coeff,
    const EB_U32          coeffStride,
    EB_S16               *quantCoeff,
    EB_S16               *reconCoeff,
    const EB_U32          qFunc,
    const EB_U32          adptive_qp_offset,
    const EB_S32          shiftedQBits,
    const EB_S32          shiftedFFunc,
    const EB_S32          iq_offset,
    const EB_S32          shiftNum,
    EB_U32                areaSize,
    const EB_U32          activeAreaSize,
    EB_U32               *yCountNonZeroCoeffs,
    EB_U8			      pmpMaskingLevelEncDec,
    EB_MODETYPE		      type,
    EB_U32                componentType) {

    EbPMCand_t  *pmCandBuffer = contextPtr->pmCandBuffer;
    EB_U8  canDi;
    EB_U8  candCount = 0;
    EB_U64 bestCost = MAX_CU_COST;
    EB_U32 bestCand = 0;
    EB_U32 blkAreaSize = 4;
    EB_U8  sampleIndex;

    const EB_U16 *MaskingMap4x4Ptr[3];

    EB_U32 depthIndex = LOG2F(areaSize) - 2;
    depthIndex = depthIndex < 4 ? depthIndex : 0;

    // Initialize maskingMatrixPtr
    const EB_U16 *maskingMatrixPtr = &MaskingMatrixSet1[pmpMaskingLevelEncDec][depthIndex][0];


	EB_U32 betta = 100;

	if (contextPtr->cuPtr->orgDeltaQp> 0)
	{
		betta = BETA_P * 100 * ABS(contextPtr->cuPtr->orgDeltaQp);
	}
	else if (contextPtr->cuPtr->orgDeltaQp < 0)
	{
		betta = BETA_N * 100 * ABS(contextPtr->cuPtr->orgDeltaQp);
	}

    // Use a more agressive set for 50/60 fps clips
    if ((sequenceControlSetPtr->staticConfig.frameRate >> 16) > 30)
        maskingMatrixPtr = &MaskingMatrixSet2[pmpMaskingLevelEncDec][depthIndex][0];

    EB_U32 regionThrshld0, regionThrshld1, regionThrshld2;
    regionThrshld0 = areaSize >> 3;
    regionThrshld1 = areaSize >> 2;
    regionThrshld2 = areaSize >> 1;
    EB_U32 colBlkIter, rowBlkIter, blkOffset;
    if (yCountNonZeroCoeffs)
    *yCountNonZeroCoeffs = 0;

    // Do the Quantization to avoid extra compute for areas without non zero coefficient
	QiQ_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)][activeAreaSize >> 3](
        coeff,
        coeffStride,
        quantCoeff,
        reconCoeff,
        qFunc,
        adptive_qp_offset,
        shiftedQBits,
        shiftedFFunc,
        iq_offset,
        shiftNum,
        activeAreaSize,
        &(*yCountNonZeroCoeffs));

    if (yCountNonZeroCoeffs)
    {

        *yCountNonZeroCoeffs = 0;
        // Loop over 4x4 blocks within each TU
        for (rowBlkIter = 0; rowBlkIter < activeAreaSize / 4; ++rowBlkIter) {
            for (colBlkIter = 0; colBlkIter < activeAreaSize / 4; ++colBlkIter) {
                blkOffset = colBlkIter * 4 + (rowBlkIter * 4 * coeffStride);

				if ( CheckNZero4x4(&quantCoeff[blkOffset], coeffStride) )

                {
                    //Fill the PM candidates
                    candCount = 0;
                    bestCost = MAX_CU_COST;
                    bestCand = 0;
                    EB_U32 alpha = ALPHA_1000;
                    EB_U32 matrixBlkOffset = colBlkIter * 4 + (rowBlkIter * 4 * areaSize);

					if (contextPtr->pmMode == PM_MODE_0){ // 4K
						MaskingMap4x4Ptr[0] = &MaskingMatrixSet0[pmpMaskingLevelEncDec][depthIndex][matrixBlkOffset];
						MaskingMap4x4Ptr[1] = &MaskingMatrixSet1[pmpMaskingLevelEncDec][depthIndex][matrixBlkOffset];
						MaskingMap4x4Ptr[2] = &maskingMatrixPtr[matrixBlkOffset];

						pmCandBuffer[candCount++].maskingLevel = 0;
						pmCandBuffer[candCount++].maskingLevel = 1;
						pmCandBuffer[candCount++].maskingLevel = 2;

						switch (pmpMaskingLevelEncDec){
						case 1:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0100;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0100;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_0100;
							}
							break;
						case 2:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0300;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0500;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_1000;
							}
							break;
						case 3:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0200;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0300;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_1000;
							}
							break;

						case 4:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0500;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0500;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_1000;
							}
							break;
						case 5:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0500;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0500;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_1000;
							}
							break;
						case 6:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0500;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0500;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_1000;
							}
							break;
						case 7:

							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0500;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0500;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_1000;
							}
							break;
						default:
							break;
						}

					}
					else{ // PM_MODE_1 Sub 4K

						MaskingMap4x4Ptr[candCount] = &MaskingMatrix1080PSet1[pmpMaskingLevelEncDec][depthIndex][matrixBlkOffset];

						pmCandBuffer[candCount].maskingLevel = candCount;
						candCount++;
						MaskingMap4x4Ptr[candCount] = &MaskingMatrix1080PSet0[pmpMaskingLevelEncDec][depthIndex][matrixBlkOffset];
						pmCandBuffer[candCount].maskingLevel = candCount;
						candCount++;

						switch (pmpMaskingLevelEncDec){
						case 1:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0000;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0000;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_0000;
							}
							break;
						case 2:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0100;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0100;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_0100;
							}
							break;
						case 3:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0050;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0100;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_0100;
							}
							break;

						case 4:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0100;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0050;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_0100;
							}
							break;
						case 5:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0200;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0200;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_0200;
							}
							break;
						case 6:
							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0100;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0050;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_0100;
							}
							break;
						case 7:

							if (colBlkIter + rowBlkIter < regionThrshld0 || colBlkIter + rowBlkIter == 0){
								alpha = ALPHA_0200;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld1){
								alpha = ALPHA_0200;
							}
							else if (colBlkIter + rowBlkIter < regionThrshld2){
								alpha = ALPHA_0200;
							}
							break;
						default:
							break;
						}
					}

					//to simulate chroma rate estimation bug. To embedd with above alpha setting later.
					if (componentType != COMPONENT_LUMA)
						alpha = alpha * 2;

                    // Loop over candidates
                    for (canDi = 0; canDi < candCount; canDi++)
                    {

                        EbPMCand_t  *pmCand = &pmCandBuffer[canDi];

						//There is Mismatch between ASM vs C !
						MatMulOut_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)](
							&coeff[blkOffset],
							coeffStride,
							pmCand->trCoeff,
							PM_STRIDE,
							&MaskingMap4x4Ptr[pmCand->maskingLevel][0],
							areaSize,
							blkAreaSize,
							(PMP_MAX >> 1),
							PMP_PRECISION,
							&pmCand->nzCoeff);

						// Bypass masking DC value, if it is greater than DC_TRSHLD1
						if (colBlkIter + rowBlkIter == 0 && ABS(quantCoeff[0]) > PM_DC_TRSHLD1){
							pmCand->trCoeff[0] = coeff[0];
						}


						QiQ_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)][blkAreaSize >> 3](
                            pmCand->trCoeff,
                            PM_STRIDE,
                            pmCand->quCoeff,
                            pmCand->iqCoeff,
                            qFunc,
                            adptive_qp_offset,
                            shiftedQBits,
                            shiftedFFunc,
                            iq_offset,
                            shiftNum,
                            blkAreaSize,
                            &pmCand->nzCoeff);

                        //Cost calculation
						EB_U64 sse[2];
                        EB_U64 coeffBits = 0;

						FullDistortionIntrinsic_funcPtrArray[!!(ASM_TYPES & PREAVX2_MASK)][pmCand->nzCoeff != 0][1][blkAreaSize >> 3](
                            &coeff[blkOffset],
                            coeffStride,
                            pmCand->iqCoeff,
                            PM_STRIDE,
                            sse,
                            blkAreaSize,
                            blkAreaSize);


                        EB_U32 shift = 2 * (7 - Log2f(areaSize));
                        sse[DIST_CALC_RESIDUAL] = (sse[DIST_CALC_RESIDUAL] + (EB_U64)(1 << (shift - 1))) >> shift;

						if (pmCand->nzCoeff)
							CoeffRateEst4x4_funcPtrArray[!!(ASM_TYPES & PREAVX2_MASK)][componentType != COMPONENT_LUMA](
								pictureControlSetPtr->cabacCost,
								NULL,
								4,
								type,
								0,
								0,
								pmCand->quCoeff,
								PM_STRIDE,
								componentType,
								pmCand->nzCoeff,
								&coeffBits
								);

                        pmCand->cost = EbHevcGetPMCost(

							(EB_U64)contextPtr->fullLambda*alpha*betta / 100 / 100,

                            sse[DIST_CALC_RESIDUAL],
                            coeffBits
                            );

                        //determine best cost
                        if (pmCand->cost < bestCost){
                            bestCost = pmCand->cost;
                            bestCand = canDi;
                        }
                    }
                    //overwrite regular path with best
                    EbPMCand_t  *pmBestCand = &pmCandBuffer[bestCand];
                    for (sampleIndex = 0; sampleIndex < blkAreaSize; sampleIndex++){
                        EB_MEMCPY(&quantCoeff[blkOffset + sampleIndex*coeffStride], &pmBestCand->quCoeff[sampleIndex * PM_STRIDE], blkAreaSize*sizeof(EB_S16));
                        EB_MEMCPY(&reconCoeff[blkOffset + sampleIndex*coeffStride], &pmBestCand->iqCoeff[sampleIndex * PM_STRIDE], blkAreaSize*sizeof(EB_S16));
                    }
                    *yCountNonZeroCoeffs += pmBestCand->nzCoeff;
                }
            }
        }
    }
}

void DecoupledQuantizeInvQuantizeLoops(
	EB_S16                        *coeff,
	const EB_U32                  coeffStride,
	EB_S16                        *quantCoeff,
	EB_S16                        *reconCoeff,
	CabacEncodeContext_t         *cabacEncodeCtxPtr,
	EB_U64                        lambda,
	EB_MODETYPE                   type,                 // Input: CU type (INTRA, INTER)
	EB_U32                        intraLumaMode,
	EB_U32                        intraChromaMode,
	EB_U32                        componentType,
	EB_U8                         temporalLayerIndex,
	EB_BOOL                       isUsedAsReferenceFlag,
	EB_U8                         chromaLambda,
    EB_U16                        qp,
    EB_U32                        bitDepth,
	CabacCost_t                  *CabacCost,
	const EB_U32                  qFunc,
	const EB_U32                  q_offset,
	const EB_S32                  shiftedQBits,
	const EB_S32                  shiftedFFunc,
	const EB_S32                  iq_offset,
	const EB_S32                  shiftNum,
	const EB_U32                  areaSize,
	EB_U32                        *nonzerocoeff,
    EB_RDOQ_PMCORE_TYPE            useRdoType)
{

	EB_U32 coeffLocation = 0;
	EB_U32 rowIndex, colIndex;


	EB_U32 adptive_qp_offset = q_offset;

    (void)chromaLambda;
    *nonzerocoeff = 0;

    if (bitDepth == (EB_U32)EB_10BIT) {
        qp -= QP_BD_OFFSET;  // adjust for chroma weighting factor derivation in 10-bit coding
    }

#define RDCOST_ITERATION 3

	adptive_qp_offset = q_offset;
	coeffLocation = 0;
	*nonzerocoeff = 0;
	QiQ_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)][areaSize >> 3](
		coeff,
        coeffStride,
        quantCoeff,
        reconCoeff,
        qFunc,
        adptive_qp_offset,
        shiftedQBits,
        shiftedFFunc,
        iq_offset,
        shiftNum,
        areaSize,
        &(*nonzerocoeff));

    EB_U64 coeffBitsLong = 0;

    EB_S32 rCoeffTmp, coeffTemp2;
	EB_U64 rdoqBits_start = 0;
	EB_BOOL first_non_zero_coef_done = EB_FALSE;

    if (useRdoType == EB_RDOQ || useRdoType == EB_LIGHT) {

        coeffLocation = (areaSize - 1) + ((areaSize - 1) * coeffStride);

        for (rowIndex = 0; rowIndex < areaSize; ++rowIndex) {
            for (colIndex = 0; colIndex < areaSize; ++colIndex) {

                EB_S16 orgQuantCoeff = quantCoeff[coeffLocation];
                EB_S16 orgNonZeroCoeff = (EB_S16)*nonzerocoeff;
                EB_S16 bestQuantCoeff = quantCoeff[coeffLocation];
                EB_S16 bestNonZeroCoeff = (EB_S16)*nonzerocoeff;

                if (quantCoeff[coeffLocation]) {
                    EB_S32 absVal = ABS(coeff[coeffLocation]);

                    EB_U32 level = 0;
                    EB_U32 iteration = 0;
                    EB_S32 startIteration = 1;
                    EB_U32 startLevel = ((EB_S64)absVal * qFunc >> (shiftedQBits)) - (EB_S64)startIteration >= 0 ? ((EB_S64)absVal * qFunc >> (shiftedQBits)) - startIteration : ((EB_S64)absVal * qFunc >> (shiftedQBits));
                    EB_U32 enddLevel = (absVal * qFunc >> (shiftedQBits)) + startIteration;


                    EB_U64 rdoqCost[RDCOST_ITERATION] = { 0 };
                    EB_U64 rdoqError[RDCOST_ITERATION] = { 0 };
                    EB_U64 rdoqBits[RDCOST_ITERATION] = { 0 };
                    EB_U64 bestCost = MAX_CU_COST;
                    for (level = startLevel; level <= enddLevel; level += startIteration) {
                        *nonzerocoeff = orgNonZeroCoeff;
                        EB_U32 Templevel = level;

                        EB_U64 rdoqErrorTemp = ABS((((EB_S64)absVal * qFunc - (EB_S64)(Templevel << shiftedQBits)) >> (shiftedQBits - 8)));
                        coeffTemp2 = ((Templevel * shiftedFFunc) + iq_offset) >> shiftNum;
                        rdoqErrorTemp = ABS((EB_S64)absVal - (EB_S64)(coeffTemp2));
                        rdoqError[iteration] = rdoqErrorTemp * rdoqErrorTemp;

                        quantCoeff[coeffLocation] = (orgQuantCoeff > 0) ? (EB_S16)level : -(EB_S16)level;

                        *nonzerocoeff -= (quantCoeff[coeffLocation] == 0);

                        coeffBitsLong = 0;

						if (useRdoType == EB_LIGHT) {
							if (*nonzerocoeff) {
								if (!first_non_zero_coef_done) {
									first_non_zero_coef_done = EB_TRUE;

									EstimateQuantizedCoefficients[1][!!(ASM_TYPES & PREAVX2_MASK)](
										CabacCost,
										cabacEncodeCtxPtr,
										areaSize,
										type,
										intraLumaMode,
										intraChromaMode,
										quantCoeff,
										coeffStride,
										componentType,
										*nonzerocoeff,
										&coeffBitsLong);
									rdoqBits_start = coeffBitsLong;
								}
								else {
									coeffBitsLong = rdoqBits_start + ((rdoqBits_start * iteration * 4) / 1000);
								}
							}
							rdoqBits[iteration] = coeffBitsLong;
						}
						else {
							if (*nonzerocoeff) {
								EstimateQuantizedCoefficients[1][!!(ASM_TYPES & PREAVX2_MASK)](
									CabacCost,
									cabacEncodeCtxPtr,
									areaSize,
									type,
									intraLumaMode,
									intraChromaMode,
									quantCoeff,
									coeffStride,
									componentType,
									*nonzerocoeff,
									&coeffBitsLong);
							}
							rdoqBits[iteration] = coeffBitsLong;
						}

                        EB_U32 shift = 2 * (7 - Log2f(areaSize));

                        rdoqError[iteration] = (rdoqError[iteration] + (EB_U64)(1 << (shift - 1))) >> shift;


                        if (componentType == COMPONENT_CHROMA) {
                            if (temporalLayerIndex == 0) {
                                rdoqError[iteration] = (((rdoqError[iteration] * ChromaWeightFactorRaBase[qp]) + CHROMA_WEIGHT_OFFSET) >> CHROMA_WEIGHT_SHIFT);
                            }
                            else if (isUsedAsReferenceFlag) {
                                rdoqError[iteration] = (((rdoqError[iteration] * ChromaWeightFactorRaRefNonBase[qp]) + CHROMA_WEIGHT_OFFSET) >> CHROMA_WEIGHT_SHIFT);
                            }
                            else {
                                rdoqError[iteration] = (((rdoqError[iteration] * ChromaWeightFactorRaNonRef[qp]) + CHROMA_WEIGHT_OFFSET) >> CHROMA_WEIGHT_SHIFT);
                            }

                            rdoqCost[iteration] = rdoqError[iteration] + (((lambda * rdoqBits[iteration]) + MD_OFFSET) >> MD_SHIFT);
                        }
                        else {
                            rdoqCost[iteration] = (rdoqError[iteration] << COST_PRECISION) + (((lambda * rdoqBits[iteration]) + MD_OFFSET) >> MD_SHIFT);
                        }

                        if (rdoqCost[iteration] < bestCost) {
                            // Add local bit counter to global bit counter
                            bestCost = rdoqCost[iteration];

                            bestQuantCoeff = quantCoeff[coeffLocation];
                            bestNonZeroCoeff = (EB_S16)*nonzerocoeff;
                        }

                        iteration++;
                    }

                    quantCoeff[coeffLocation] = bestQuantCoeff;
                    *nonzerocoeff = bestNonZeroCoeff;
                }

                --coeffLocation;
            }

            coeffLocation -= (coeffStride - areaSize);
        }
    }
    else

	if (useRdoType == EB_PMCORE){
	if (*nonzerocoeff && (componentType == COMPONENT_LUMA)){

#define NUM_PM_CANDIDATE 3

		EB_U8  canDi;
		EB_U8  candCount = 0;
		EB_U64 bestCost = MAX_CU_COST;
		EB_U32 bestCand = 0;
		EB_U32 blkAreaSize = 4;
		EB_U8  sampleIndex;
		EB_U32 activeAreaSize = areaSize;

		EB_U32 depthIndex = LOG2F(areaSize) - 2;
		depthIndex = depthIndex < 4 ? depthIndex : 0;

		// Initialize maskingMatrixPtr
		const EB_U16 *MaskingMap4x4Ptr[NUM_PM_CANDIDATE];
		EbPMCand_t  pmCandBuffer[NUM_PM_CANDIDATE];


		EB_U32 colBlkIter, rowBlkIter, blkOffset;

		*nonzerocoeff = 0;
		// Loop over 4x4 blocks within each TU
		for (rowBlkIter = 0; rowBlkIter < activeAreaSize / 4; ++rowBlkIter) {
			for (colBlkIter = 0; colBlkIter < activeAreaSize / 4; ++colBlkIter) {
				blkOffset = colBlkIter * 4 + (rowBlkIter * 4 * coeffStride);

				if (CheckNZero4x4(&quantCoeff[blkOffset], coeffStride))

				{
					//Fill the PM candidates
					candCount = 0;
					bestCost = MAX_CU_COST;
					bestCand = 0;

					MaskingMap4x4Ptr[0] = &MaskingMatrix4x4_100[0];
					MaskingMap4x4Ptr[1] = &MaskingMatrix4x4_70[0];
					MaskingMap4x4Ptr[2] = &MaskingMatrix4x4_50[0];

					pmCandBuffer[candCount++].maskingLevel = 0;
					pmCandBuffer[candCount++].maskingLevel = 1;
					pmCandBuffer[candCount++].maskingLevel = 2;



					// Loop over candidates
					for (canDi = 0; canDi < candCount; canDi++)
					{

						EbPMCand_t  *pmCand = &pmCandBuffer[canDi];

						//There is Mismatch between ASM vs C !
						MatMulOut_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)](
							&coeff[blkOffset],
							coeffStride,
							pmCand->trCoeff,
							PM_STRIDE,
							&MaskingMap4x4Ptr[pmCand->maskingLevel][0],
							PM_STRIDE,
							blkAreaSize,
							(PMP_MAX >> 1),
							PMP_PRECISION,
							&pmCand->nzCoeff);

						// Bypass masking DC value, if it is greater than DC_TRSHLD1
						if (colBlkIter + rowBlkIter == 0 && ABS(quantCoeff[0]) > PM_DC_TRSHLD1){
							pmCand->trCoeff[0] = coeff[0];
						}


						QiQ_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)][blkAreaSize >> 3](
							pmCand->trCoeff,
							PM_STRIDE,
							pmCand->quCoeff,
							pmCand->iqCoeff,
							qFunc,
							adptive_qp_offset,
							shiftedQBits,
							shiftedFFunc,
							iq_offset,
							shiftNum,
							blkAreaSize,
							&pmCand->nzCoeff);

						//Cost calculation
						EB_U64 sse[2];
						EB_U64 coeffBits = 0;

						FullDistortionIntrinsic_funcPtrArray[!!(ASM_TYPES & PREAVX2_MASK)][pmCand->nzCoeff != 0][1][blkAreaSize >> 3](
							&coeff[blkOffset],
							coeffStride,
							pmCand->iqCoeff,
							PM_STRIDE,
							sse,
							blkAreaSize,
							blkAreaSize);


						EB_U32 shift = 2 * (7 - Log2f(areaSize));
						sse[DIST_CALC_RESIDUAL] = (sse[DIST_CALC_RESIDUAL] + (EB_U64)(1 << (shift - 1))) >> shift;

						if (pmCand->nzCoeff)
							CoeffRateEst4x4_funcPtrArray[!!(ASM_TYPES & PREAVX2_MASK)][componentType != COMPONENT_LUMA](
							CabacCost,
							NULL,
							4,
							type,
							0,
							0,
							pmCand->quCoeff,
							PM_STRIDE,
							componentType,
							pmCand->nzCoeff,
							&coeffBits
							);

						if (componentType == COMPONENT_CHROMA){

							if (temporalLayerIndex == 0) {
								sse[DIST_CALC_RESIDUAL] = (((sse[DIST_CALC_RESIDUAL] * ChromaWeightFactorRaBase[qp]) + CHROMA_WEIGHT_OFFSET) >> CHROMA_WEIGHT_SHIFT);
							}
							else if (isUsedAsReferenceFlag) {
								sse[DIST_CALC_RESIDUAL] = (((sse[DIST_CALC_RESIDUAL] * ChromaWeightFactorRaRefNonBase[qp]) + CHROMA_WEIGHT_OFFSET) >> CHROMA_WEIGHT_SHIFT);
							}
							else {
								sse[DIST_CALC_RESIDUAL] = (((sse[DIST_CALC_RESIDUAL] * ChromaWeightFactorRaNonRef[qp]) + CHROMA_WEIGHT_OFFSET) >> CHROMA_WEIGHT_SHIFT);
							}

							pmCand->cost = sse[DIST_CALC_RESIDUAL] + (((lambda * coeffBits) + MD_OFFSET) >> MD_SHIFT);
						} else {
							pmCand->cost = (sse[DIST_CALC_RESIDUAL] << COST_PRECISION) + (((lambda * coeffBits) + MD_OFFSET) >> MD_SHIFT);
						}


						//determine best cost
						if (pmCand->cost < bestCost){
							bestCost = pmCand->cost;
							bestCand = canDi;
						}
					}
					//overwrite regular path with best
					EbPMCand_t  *pmBestCand = &pmCandBuffer[bestCand];
					for (sampleIndex = 0; sampleIndex < blkAreaSize; sampleIndex++){
						EB_MEMCPY(&quantCoeff[blkOffset + sampleIndex*coeffStride], &pmBestCand->quCoeff[sampleIndex * PM_STRIDE], blkAreaSize*sizeof(EB_S16));
						EB_MEMCPY(&reconCoeff[blkOffset + sampleIndex*coeffStride], &pmBestCand->iqCoeff[sampleIndex * PM_STRIDE], blkAreaSize*sizeof(EB_S16));
					}
					*nonzerocoeff += pmBestCand->nzCoeff;
				}
			}
		}
	}
    }

	coeffLocation = 0;
	for (rowIndex = 0; rowIndex < areaSize; ++rowIndex) {
		for (colIndex = 0; colIndex < areaSize; ++colIndex) {

			//iQ
			//coeffTemp = ( (CLIP3(MIN_NEG_16BIT_NUM, MAX_POS_16BIT_NUM, quantCoeff[quantLocation]) * shiftedFFunc) + iq_offset) >> shiftNum;
			coeffTemp2 = ((quantCoeff[coeffLocation] * shiftedFFunc) + iq_offset) >> shiftNum;
			rCoeffTmp = (EB_S16)CLIP3(MIN_NEG_16BIT_NUM, MAX_POS_16BIT_NUM, coeffTemp2);
			reconCoeff[coeffLocation] = (EB_S16)rCoeffTmp;

			++coeffLocation;
		}
		coeffLocation += coeffStride - areaSize;
	}


}

/*********************************************************************
 * UnifiedQuantizeInvQuantize
 *
 *  Unified Quant +iQuant
 *********************************************************************/
void UnifiedQuantizeInvQuantize(
	EncDecContext_t              *contextPtr,
	PictureControlSet_t          *pictureControlSetPtr,
	EB_S16                       *coeff,
	const EB_U32                  coeffStride,
	EB_S16                       *quantCoeff,
	EB_S16                       *reconCoeff,
	EB_U32                        qp,
	EB_U32                        bitDepth,
	EB_U32                        areaSize,
	EB_PICTURE                      sliceType,
	EB_U32                       *yCountNonZeroCoeffs,
	EB_U8                         transCoeffShape,
	EB_U8		                  cleanSparseCeoffPfEncDec,
	EB_U8			              pmpMaskingLevelEncDec,
	EB_MODETYPE		              type,
	EB_U32                        enableCbflag,
	EB_U8                         enableContouringQCUpdateFlag,
	EB_U32                        componentType,
	EB_U32                        temporalLayerIndex,
	EB_U32                        dZoffset,
	CabacEncodeContext_t         *cabacEncodeCtxPtr,
	EB_U64                        lambda,
	EB_U32                        intraLumaMode,
	EB_U32                        intraChromaMode,
	CabacCost_t                  *CabacCost)
{


    SequenceControlSet_t *sequenceControlSetPtr = (SequenceControlSet_t*)pictureControlSetPtr->ParentPcsPtr->sequenceControlSetWrapperPtr->objectPtr;

	if (contextPtr->mdContext->rdoqPmCoreMethod){

		//for the Quant
		const EB_S32 qpRem = (EB_S32)QpModSix[qp]; //the output is between 0-5
		const EB_S32 qpPer = (EB_S32)QpDivSix[qp] + TRANS_BIT_INCREMENT; //the output is between 0 and 8+TRANS_BIT_INCREMENT   (CHKN TRANS_BIT_INCREMENT =   0)
		const EB_U32 qFunc = QFunc[qpRem]; // 15 bits

		const EB_U32 internalBitDepth = (EB_U32)bitDepth + TRANS_BIT_INCREMENT;  //CHKN always 8 for 8 bit

		const EB_U32 transformShiftNum = MAX_TR_DYNAMIC_RANGE - internalBitDepth - Log2f(areaSize);
		const EB_S32 shiftedQBits = QUANT_SHIFT + qpPer + transformShiftNum;
		const EB_U32 q_offset = ((sliceType == EB_I_PICTURE || sliceType == EB_IDR_PICTURE) ? QUANT_OFFSET_I : QUANT_OFFSET_P) << (shiftedQBits - 9);

		//for the iQuant
		const EB_S32 shiftedFFunc = (qpPer > 8) ? (EB_S32)FFunc[qpRem] << (qpPer - 2) : (EB_S32)FFunc[qpRem] << qpPer; // this is 6+8+TRANS_BIT_INCREMENT
		const EB_S32 shiftNum = (qpPer > 8) ? QUANT_IQUANT_SHIFT - QUANT_SHIFT - transformShiftNum - 2 : QUANT_IQUANT_SHIFT - QUANT_SHIFT - transformShiftNum;
		const EB_S32 iq_offset = 1 << (shiftNum - 1);

		DecoupledQuantizeInvQuantizeLoops(
			coeff,
			coeffStride,
			quantCoeff,
			reconCoeff,
			cabacEncodeCtxPtr,
			lambda,
			type,
			intraLumaMode,
			intraChromaMode,
			componentType,
			pictureControlSetPtr->temporalLayerIndex,
			pictureControlSetPtr->ParentPcsPtr->isUsedAsReferenceFlag,
            (EB_U8)contextPtr->fullChromaLambda,
			(EB_U16)qp,
            bitDepth,
			CabacCost,
			qFunc,
			q_offset,
			shiftedQBits,
			shiftedFFunc,
			iq_offset,
			shiftNum,
			areaSize,
			&(*yCountNonZeroCoeffs),
            contextPtr->mdContext->rdoqPmCoreMethod);

	}
	else if (transCoeffShape == ONLY_DC_SHAPE) {

        // Quant
        const EB_S32 qpRem = (EB_S32)QpModSix[qp]; //the output is between 0-5
        const EB_S32 qpPer = (EB_S32)QpDivSix[qp] + TRANS_BIT_INCREMENT; //the output is between 0 and 8+TRANS_BIT_INCREMENT   (CHKN TRANS_BIT_INCREMENT =   0)
        const EB_U32 qFunc = QFunc[qpRem]; // 15 bits

        const EB_U32 internalBitDepth = (EB_U32)bitDepth + TRANS_BIT_INCREMENT;  //CHKN always 8 for 8 bit

        const EB_U32 transformShiftNum = MAX_TR_DYNAMIC_RANGE - internalBitDepth - Log2f(areaSize);
        const EB_S32 shiftedQBits = QUANT_SHIFT + qpPer + transformShiftNum;
        const EB_U32 q_offset = ((sliceType == EB_I_PICTURE || sliceType == EB_IDR_PICTURE) ? QUANT_OFFSET_I : QUANT_OFFSET_P) << (shiftedQBits - 9);

        // iQuant
        const EB_S32 shiftedFFunc = (qpPer > 8) ? (EB_S32)FFunc[qpRem] << (qpPer - 2) : (EB_S32)FFunc[qpRem] << qpPer; // this is 6+8+TRANS_BIT_INCREMENT
        const EB_S32 shiftNum = (qpPer > 8) ? QUANT_IQUANT_SHIFT - QUANT_SHIFT - transformShiftNum - 2 : QUANT_IQUANT_SHIFT - QUANT_SHIFT - transformShiftNum;
        const EB_S32 iq_offset = 1 << (shiftNum - 1);


        EB_S32 coeffTemp; // based on HM4.0, this variable which is used in quantized coef calculations can be up to 15+16 -transformSizeLog2 bits
        EB_S32 sign;

        EB_S32 coeffTemp1, coeffTemp2;
        EB_S16 qCoeffTmp, rCoeffTmp;
        *yCountNonZeroCoeffs = 0;

        coeffTemp = (EB_S32)coeff[0];
        sign = (coeffTemp < 0 ? -1 : 1);
        coeffTemp1 = ABS(coeffTemp);
        coeffTemp1 *= qFunc;
        coeffTemp1 += q_offset;
        coeffTemp1 >>= shiftedQBits;
        qCoeffTmp = (EB_S16)CLIP3(MIN_NEG_16BIT_NUM, MAX_POS_16BIT_NUM, (sign*coeffTemp1));
        quantCoeff[0] = qCoeffTmp;
        (*yCountNonZeroCoeffs) += (qCoeffTmp != 0);

        //iQ
        coeffTemp2 = ((qCoeffTmp * shiftedFFunc) + iq_offset) >> shiftNum;
        rCoeffTmp = (EB_S16)CLIP3(MIN_NEG_16BIT_NUM, MAX_POS_16BIT_NUM, coeffTemp2);
        reconCoeff[0] = rCoeffTmp;

    }
    else {

        //for the Quant
        const EB_S32 qpRem = (EB_S32)QpModSix[qp]; //the output is between 0-5
        const EB_S32 qpPer = (EB_S32)QpDivSix[qp] + TRANS_BIT_INCREMENT; //the output is between 0 and 8+TRANS_BIT_INCREMENT   (CHKN TRANS_BIT_INCREMENT =   0)
        const EB_U32 qFunc = QFunc[qpRem]; // 15 bits

        const EB_U32 internalBitDepth = (EB_U32)bitDepth + TRANS_BIT_INCREMENT;  //CHKN always 8 for 8 bit

        const EB_U32 transformShiftNum = MAX_TR_DYNAMIC_RANGE - internalBitDepth - Log2f(areaSize);
        const EB_S32 shiftedQBits = QUANT_SHIFT + qpPer + transformShiftNum;
        const EB_U32 q_offset = ((sliceType == EB_I_PICTURE || sliceType == EB_IDR_PICTURE) ? QUANT_OFFSET_I : QUANT_OFFSET_P) << (shiftedQBits - 9);

        //for the iQuant
        const EB_S32 shiftedFFunc = (qpPer > 8) ? (EB_S32)FFunc[qpRem] << (qpPer - 2) : (EB_S32)FFunc[qpRem] << qpPer; // this is 6+8+TRANS_BIT_INCREMENT
        const EB_S32 shiftNum = (qpPer > 8) ? QUANT_IQUANT_SHIFT - QUANT_SHIFT - transformShiftNum - 2 : QUANT_IQUANT_SHIFT - QUANT_SHIFT - transformShiftNum;
        const EB_S32 iq_offset = 1 << (shiftNum - 1);


        EB_U32 adptive_qp_offset;

        adptive_qp_offset = q_offset;

        adptive_qp_offset = dZoffset ? (dZoffset * (1 << shiftedQBits) / 20) : adptive_qp_offset;

        EB_U32 activeAreaSize = areaSize >> transCoeffShape;

		if (contextPtr->pmMethod && componentType != COMPONENT_LUMA) {

            if (pmpMaskingLevelEncDec)
            {
                MaskTransformCoeffs(
                    coeff,
                    coeffStride,
                    areaSize,
                    activeAreaSize,
                    pmpMaskingLevelEncDec,
                    pictureControlSetPtr,
                    &(*yCountNonZeroCoeffs));
            }
            //QiQ SSSE3 is hardcoded
            //QiQ   Use this for SW
			QiQ_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)][activeAreaSize >> 3](
                coeff,
                coeffStride,
                quantCoeff,
                reconCoeff,
                qFunc,
                adptive_qp_offset,
                shiftedQBits,
                shiftedFFunc,
                iq_offset,
                shiftNum,
                activeAreaSize,
                &(*yCountNonZeroCoeffs));

        }
        else {
            if (pmpMaskingLevelEncDec) {

                PerformTwoStagePm(
                    sequenceControlSetPtr,
                    pictureControlSetPtr,
                    contextPtr,
                    coeff,
                    coeffStride,
                    quantCoeff,
                    reconCoeff,
                    qFunc,
                    adptive_qp_offset,
                    shiftedQBits,
                    shiftedFFunc,
                    iq_offset,
                    shiftNum,
                    areaSize,
                    activeAreaSize,
                    yCountNonZeroCoeffs,
                    pmpMaskingLevelEncDec,
                    type,
                    componentType);

            } else {

				QiQ_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)][activeAreaSize >> 3](
                    coeff,
                    coeffStride,
                    quantCoeff,
                    reconCoeff,
                    qFunc,
                    adptive_qp_offset,
                    shiftedQBits,
                    shiftedFFunc,
                    iq_offset,
                    shiftNum,
                    activeAreaSize,
                    &(*yCountNonZeroCoeffs));
            }
        }




        if (cleanSparseCeoffPfEncDec && areaSize >= 8 && yCountNonZeroCoeffs && (*yCountNonZeroCoeffs) && sliceType != EB_I_PICTURE){
            EB_U32 coeffLocation = 0;
            EB_U32 rowIndex = 0;
            EB_U32 colIndex = 0;

            //// Zero out Isolated Quant Coeffs in 4x4 blocks
            if (activeAreaSize >= 4)
            {

                EB_U32 colBlkIter, rowBlkIter, blkOffset;
                EB_U8 nonZeroNeigh;
                coeffLocation = 0;

                for (colBlkIter = 0; colBlkIter < activeAreaSize / 4; ++colBlkIter) {
                    for (rowBlkIter = 0; rowBlkIter < activeAreaSize / 4; ++rowBlkIter) {

                        blkOffset = rowBlkIter * 4 + (colBlkIter * 4 * coeffStride);
                        nonZeroNeigh = 0;
                        for (colIndex = 0; colIndex < 4; ++colIndex) {
                            for (rowIndex = 0; rowIndex < 4; ++rowIndex) {


                                coeffLocation = blkOffset + (rowIndex)+(colIndex)* coeffStride;
                                if (quantCoeff[coeffLocation]){
                                    nonZeroNeigh++;
                                }
                            }
                        }
                        if (nonZeroNeigh == 1){
                            for (colIndex = 0; colIndex < 4; ++colIndex) {
                                for (rowIndex = 0; rowIndex < 4; ++rowIndex) {

                                    coeffLocation = blkOffset + (rowIndex)+(colIndex)* coeffStride;
                                    if (quantCoeff[coeffLocation] && coeffLocation){
                                        quantCoeff[coeffLocation] = 0;
                                        reconCoeff[coeffLocation] = 0;
                                        (*yCountNonZeroCoeffs)--;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        if (yCountNonZeroCoeffs != (EB_U32*)EB_NULL) {
            UpdateQiQCoef(
                quantCoeff,
                reconCoeff,
                coeffStride,
                shiftedFFunc,
                iq_offset,
                shiftNum,
                activeAreaSize,
                &(*yCountNonZeroCoeffs),
                componentType,
                sliceType,
                temporalLayerIndex,
                enableCbflag,
                enableContouringQCUpdateFlag);
        }
	}
}

/*********************************************************************
 * Transform
 *   Note there is an implicit assumption that TU Size <= PU Size,
 *   which is different than the HEVC requirements.
 *********************************************************************/
EB_ERRORTYPE EstimateTransform(
    EB_S16              *residualBuffer,
    EB_U32               residualStride,
    EB_S16              *coeffBuffer,
    EB_U32               coeffStride,
    EB_U32               transformSize,
    EB_S16              *transformInnerArrayPtr,
    EB_U32               bitIncrement,
    EB_BOOL              dstTansformFlag,
    EB_TRANS_COEFF_SHAPE transCoeffShape)

{
    EB_ERRORTYPE return_error = EB_ErrorNone;

    EB_U32 transformSizeFlag = Log2f(TRANSFORM_MAX_SIZE) - Log2f(transformSize);

    if (transCoeffShape == DEFAULT_SHAPE) {
		(*transformFunctionTableEstimate[!!(ASM_TYPES & AVX2_MASK)][transformSizeFlag + dstTansformFlag])(
            residualBuffer,
            residualStride,
            coeffBuffer,
            coeffStride,
            transformInnerArrayPtr,
            bitIncrement
            );
    }

    else if (transCoeffShape == N2_SHAPE) {

		(*PfreqN2TransformTable0[!!(ASM_TYPES & AVX2_MASK)][transformSizeFlag + dstTansformFlag])(
            residualBuffer,
            residualStride,
            coeffBuffer,
            coeffStride,
            transformInnerArrayPtr,
            0);
    }

    else if (transCoeffShape == N4_SHAPE) {

		(*PfreqN4TransformTable0[!!(ASM_TYPES & AVX2_MASK)][transformSizeFlag + dstTansformFlag])(
            residualBuffer,
            residualStride,
            coeffBuffer,
            coeffStride,
            transformInnerArrayPtr,
            0);
    }
    else { // transCoeffShape == ONLY_DC_SHAPE

        EB_S32 sumResidual;

		sumResidual = SumResidual_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)](
            residualBuffer,
            transformSize,
            residualStride);

        EB_U32 shift1st = Log2f(transformSize) - 1 + bitIncrement;
        EB_U32 shift2nd = Log2f(transformSize) + 6;
        EB_S32 shift = shift1st + shift2nd;
        EB_S32 offset = 1 << (shift - 1);
        EB_S16 dcCoeff = (EB_S16)((64 * 64 * sumResidual + offset) >> shift);
        coeffBuffer[0] = dcCoeff;

    }


    return return_error;
}

/*********************************************************************
 * Transform
 *   Note there is an implicit assumption that TU Size <= PU Size,
 *   which is different than the HEVC requirements.
 *********************************************************************/
EB_ERRORTYPE EncodeTransform(
    EB_S16              *residualBuffer,
    EB_U32               residualStride,
    EB_S16              *coeffBuffer,
    EB_U32               coeffStride,
    EB_U32               transformSize,
    EB_S16              *transformInnerArrayPtr,
    EB_U32               bitIncrement,
    EB_BOOL              dstTransformFlag,
    EB_TRANS_COEFF_SHAPE transCoeffShape)
{
    EB_ERRORTYPE return_error = EB_ErrorNone;

    EB_U32 transformSizeFlag = Log2f(TRANSFORM_MAX_SIZE) - Log2f(transformSize);

    if (transCoeffShape == DEFAULT_SHAPE) {
        if (!((!!(ASM_TYPES & AVX2_MASK)))) { // C Only
            (*transformFunctionTableEncode0[/*ASM_TYPES*/((bitIncrement & BIT_INCREMENT_10BIT) ? EB_ASM_C : (!!(ASM_TYPES & PREAVX2_MASK)))][transformSizeFlag + dstTransformFlag])(
                residualBuffer,
                residualStride,
                coeffBuffer,
                coeffStride,
                transformInnerArrayPtr,
                bitIncrement
                );
        }
        else {
            (*transformFunctionTableEncode1[/*ASM_TYPES*/((bitIncrement & BIT_INCREMENT_10BIT) ? EB_ASM_C : (!!(ASM_TYPES & PREAVX2_MASK)))][transformSizeFlag + dstTransformFlag])(
                residualBuffer,
                residualStride,
                coeffBuffer,
                coeffStride,
                transformInnerArrayPtr,
                bitIncrement
                );
        }
    }

    else if (transCoeffShape == N2_SHAPE) {
            if (!((!!(ASM_TYPES & AVX2_MASK)))) { // C Only
                (*PfreqN2TransformTable0[/*ASM_TYPES*/((bitIncrement & BIT_INCREMENT_10BIT) ? EB_ASM_C : (!!(ASM_TYPES & AVX2_MASK)))][transformSizeFlag + dstTransformFlag])(
                    residualBuffer,
                    residualStride,
                    coeffBuffer,
                    coeffStride,
                    transformInnerArrayPtr,
                    bitIncrement
                    );
            }
            else {
                (*PfreqN2TransformTable1[/*ASM_TYPES*/((bitIncrement & BIT_INCREMENT_10BIT) ? EB_ASM_C : (!!(ASM_TYPES & AVX2_MASK)))][transformSizeFlag + dstTransformFlag])(
                    residualBuffer,
                    residualStride,
                    coeffBuffer,
                    coeffStride,
                    transformInnerArrayPtr,
                    bitIncrement
                    );
            }
    }

    else if (transCoeffShape == N4_SHAPE) {
        if (!((!!(ASM_TYPES & AVX2_MASK)))) { // C Only
            (*PfreqN4TransformTable0[/*ASM_TYPES*/((bitIncrement & BIT_INCREMENT_10BIT) ? EB_ASM_C : (!!(ASM_TYPES & AVX2_MASK)))][transformSizeFlag + dstTransformFlag])(
                residualBuffer,
                residualStride,
                coeffBuffer,
                coeffStride,
                transformInnerArrayPtr,
                bitIncrement);
        }
        else {
            (*PfreqN4TransformTable1[/*ASM_TYPES*/((bitIncrement & BIT_INCREMENT_10BIT) ? EB_ASM_C : (!!(ASM_TYPES & AVX2_MASK)))][transformSizeFlag + dstTransformFlag])(
                residualBuffer,
                residualStride,
                coeffBuffer,
                coeffStride,
                transformInnerArrayPtr,
                bitIncrement);
        }
    }

    else { // transCoeffShape == ONLY_DC_SHAPE

        EB_S32 sumResidual;

		sumResidual = SumResidual_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)](
            residualBuffer,
            transformSize,
            residualStride);

        EB_U32 shift1st    = Log2f(transformSize) - 1 + bitIncrement;
        EB_S32 offset1st   = 1 << (shift1st - 1);

        EB_U32 shift2nd    = Log2f(transformSize) + 6;
        EB_S32 offset2nd   = 1 << (shift2nd - 1);

        EB_S16 dcCoeff;
        EB_S32 dcCoeffTemp;
        dcCoeffTemp = (EB_S32)((64 * sumResidual + offset1st) >> shift1st);
        dcCoeff = (EB_S16)((64 * dcCoeffTemp + offset2nd) >> shift2nd);

        coeffBuffer[0] = dcCoeff;

    }

    return return_error;
}

/*********************************************************************
 * Estimate Inverse Transform
 *********************************************************************/
EB_ERRORTYPE EstimateInvTransform(
    EB_S16      *coeffBuffer,
    EB_U32       coeffStride,
    EB_S16      *reconBuffer,
    EB_U32       reconStride,
    EB_U32       transformSize,
    EB_S16      *transformInnerArrayPtr,
    EB_U32       bitIncrement,
    EB_BOOL      dstTransformFlag,
    EB_U32       partialFrequencyN2Flag)
{
    EB_ERRORTYPE return_error = EB_ErrorNone;

    EB_U32 transformSizeFlag = Log2f(TRANSFORM_MAX_SIZE) - Log2f(transformSize);

    // Forward Transform
    // The input of this function is the quantized_inversequantized transformed residual
    //   but in order to avoid extra copying, it is overwritten in place. The
    //   input(residualBuffer) is the LCU residual buffer
    if (partialFrequencyN2Flag == EB_FALSE) {
		(*invTransformFunctionTableEstimate[!!(ASM_TYPES & PREAVX2_MASK)][transformSizeFlag + dstTransformFlag])(
            coeffBuffer,
            coeffStride,
            reconBuffer,
            reconStride,
            transformInnerArrayPtr,
            bitIncrement);
    }
    else {

        // EstimateInvTransform32x32_SSE2(
        PfreqEstimateInvTransform32x32_SSE2(
            coeffBuffer,
            coeffStride,
            reconBuffer,
            reconStride,
            transformInnerArrayPtr,
            bitIncrement);

    }

    return return_error;
}

/*********************************************************************
 * Encode Inverse Transform
 *********************************************************************/
EB_ERRORTYPE EncodeInvTransform(
    EB_BOOL      isOnlyDc,
    EB_S16      *coeffBuffer,
    EB_U32       coeffStride,
    EB_S16      *reconBuffer,
    EB_U32       reconStride,
    EB_U32       transformSize,
    EB_S16      *transformInnerArrayPtr,
    EB_U32       bitIncrement,
    EB_BOOL      dstTransformFlag)
{
    EB_ERRORTYPE return_error = EB_ErrorNone;

    EB_U32 transformSizeFlag = Log2f(TRANSFORM_MAX_SIZE) - Log2f(transformSize);

    if (isOnlyDc) {
        EB_S16  dcCoef              = coeffBuffer[0];

        EB_U32  shift1st            = SHIFT_INV_1ST;
        EB_U32  shift2nd            = SHIFT_INV_2ND - bitIncrement;

        EB_S32  offset1st           = 1 << (shift1st - 1);
        EB_S32  offset2nd           = 1 << (shift2nd - 1);

        EB_S16  invTranformedDcCoef;

        invTranformedDcCoef = (EB_S16) CLIP3(MIN_NEG_16BIT_NUM, MAX_POS_16BIT_NUM,((64 * dcCoef + offset1st) >> shift1st));
        invTranformedDcCoef = (EB_S16) CLIP3(MIN_NEG_16BIT_NUM, MAX_POS_16BIT_NUM,((64 * invTranformedDcCoef + offset2nd) >> shift2nd));

		memset16bitBlock_funcPtrArray[!!(ASM_TYPES & AVX2_MASK)](
            reconBuffer,
            reconStride,
            transformSize,
            invTranformedDcCoef);

    }
    else
    {

        // Forward Transform
        // The input of this function is the quantized_inversequantized transformed residual
        //   but in order to avoid extra copying, it is overwritten in place. The
        //   input(residualBuffer) is the LCU residual buffer
		(*invTransformFunctionTableEncode[!!(ASM_TYPES & PREAVX2_MASK)][transformSizeFlag + dstTransformFlag])(
            coeffBuffer,
            coeffStride,
            reconBuffer,
            reconStride,
            transformInnerArrayPtr,
            bitIncrement);
    }
    return return_error;
}

/*********************************************************************
 * Calculate CBF
 *********************************************************************/
EB_ERRORTYPE CalculateCbf(
    EB_S16  *buffer,
    EB_U32   stride,
    EB_U32   cuSize,
    EB_U32   transformSize,
    EB_U32  *cbfBuffer)
{
    EB_ERRORTYPE return_error = EB_ErrorNone;

    // CBFs
    // *Note - this logic assumes that TU size == PU size
    // *Note - this logic should eventually be moved to the InvTransform logic
    // so that the TUs can be coded in proper order

    EB_U32  xIndex;
    EB_U32  sampleCount;
    EB_U32  sampleTotalCount;
    EB_U32  cbf;
    EB_S16 *bufferSubBlock;

    EB_U32 x = 0;
    EB_U32 y = 0;

    EB_U32 transformBlockTotalCount;
    EB_U32 transformBlockCount;
    //EB_U32 transformSizeFlag;

    transformBlockTotalCount = (Log2f(cuSize) - Log2f(transformSize) == 0) ? 1 :
        (Log2f(cuSize) - Log2f(transformSize) == 1) ? 4 : 16;

    transformBlockCount = 0;
    //transformSizeFlag = Log2f(TRANSFORM_MAX_SIZE) - Log2f(transformSize);

    // Initialize CBF
    *cbfBuffer = 0;

    do {
        sampleCount = 0;
        sampleTotalCount = SQR(transformSize);
        cbf = 0;
        bufferSubBlock = &buffer[(y*stride + x)*transformSize];

        do {
            for (xIndex = 0; (xIndex < transformSize) && (cbf == 0); ++xIndex) {
                cbf = (bufferSubBlock[xIndex] == 0) ? cbf : 1;
            }
            bufferSubBlock += stride;
            sampleCount += transformSize;
        } while ((sampleCount < sampleTotalCount) && (cbf == 0));

        (*cbfBuffer) |= (cbf << transformBlockCount);

        ZOrderIncrement(&x, &y);

    } while (++transformBlockCount < transformBlockTotalCount);

    return return_error;
}

/*********************************************************************
 * Map Chroma QP
 *********************************************************************/
EB_U8 MapChromaQp(
    EB_U8  qp)
{
    return ChromaQpMap[qp];
}

void PfZeroOutUselessQuadrants(
    EB_S16* transformCoeffBuffer,
    EB_U32  transformCoeffStride,
    EB_U32  quadrantSize) {

    PicZeroOutCoef_funcPtrArray[!!(ASM_TYPES & PREAVX2_MASK)][quadrantSize >> 3](
        transformCoeffBuffer,
        transformCoeffStride,
        quadrantSize,
        quadrantSize,
        quadrantSize);

    PicZeroOutCoef_funcPtrArray[!!(ASM_TYPES & PREAVX2_MASK)][quadrantSize >> 3](
        transformCoeffBuffer,
        transformCoeffStride,
        quadrantSize * transformCoeffStride,
        quadrantSize,
        quadrantSize);

    PicZeroOutCoef_funcPtrArray[!!(ASM_TYPES & PREAVX2_MASK)][quadrantSize >> 3](
        transformCoeffBuffer,
        transformCoeffStride,
        quadrantSize * transformCoeffStride + quadrantSize,
        quadrantSize,
        quadrantSize);

}
