/* vim: set tabstop=4: */
/*
 * This file is part of TraceProto.
 * Copyright 2004-2005 Eric Hope and others; see the AUTHORS file for details.
 *
 * TraceProto is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * TraceProto is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TraceProto; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*! \file tp_output.h
 *  \brief Functions for the various types of output
 *
 * The report_* functions implement the various possible output
 * formats that can be specified by the -o flag.  The first
 * parameter is one of the output_items enumerated in traceproto.h
 * These must match the *_packet_id array provided by each of the
 * report functions.  Thus it should be easy to add new report
 * formats.  The third parameter similarly should match one of
 * the packet_numbers from traceproto.h.
 */

#ifndef TP_OUTPUT_H
#define TP_OUTPUT_H 1

#include "config.h"

/*! Spit out the header and nothing else (accept whatever accounting)
 */
void report_none ( int output_item, __attribute__((__unused__)) struct in_addr * from, __attribute__((__unused__)) int packet_type );

/*! Spit out a single char, all responses for each hop on the same line
 */
void report_minimum ( int output_item, __attribute__((__unused__)) struct in_addr * from, int packet_type );

/*! Minimal output designed for scripting, not necessarily for
 * human readability.
 */
void report_scriptable ( int output_item, struct in_addr * from, int packet_type );

/*! Output format that is as close as possible to the original traceroute.
 *  This still needs a bit of work to make it match exactly.
 */
void report_classic ( int output_item, struct in_addr * from, int packet_type );

/*! Traceproto's standard output format. It is somewhat more readable than the
 * original traceroute format but still fairly space efficient.
 */
void report_std ( int output_item, struct in_addr * from, int packet_type );

/*! An exececutive summary output format: each response on a separate line with
 * an ASCII line showing the time.
 */
void report_graphic ( int output_item, struct in_addr * from, int packet_type );

void report_curses ( int output_item, struct in_addr * from, __attribute__((__unused__)) int packet_type );

/*! A terse summary of how to invoke Traceproto
 */
void usage( void );

#endif /* TP_OUTPUT_H */
