/*  polarization.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "polarization.h"
#include "functions.h"
#include "ginacutils.h"
#include "propagator.h"
#include "color.h"
#include "files.h"
#include "feynmanrules.h"

// \todo define Ident() in Color

using namespace std;
using namespace GiNaC;

namespace Reduze {
//
//
//
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(metric4_tensor, tensor,
		print_func<print_dflt>(&metric4_tensor::do_print))

metric4_tensor::metric4_tensor() {
	setflag(status_flags::evaluated | status_flags::expanded);
}

GINAC_BIND_UNARCHIVER( metric4_tensor)
;

int metric4_tensor::compare_same_type(const GiNaC::basic& other) const {
	/* by default, the objects are always identical */
	return 0;
}

void metric4_tensor::do_print(const GiNaC::print_context& c, unsigned level) const {
	c.s << "metricl4";
}

template<class T>
static bool is_equal_or_dummy_pair(const T& a, const T& b) {
	return a.is_equal(b) || is_dummy_pair(a, b);
}

template<class T1, class T2>
static bool has_equal_or_dummy(const T1& v, const T2& e) {
	for (typename T1::const_iterator it = v.begin(); it != v.end(); ++it) {
		if (is_equal_or_dummy_pair(*it, e))
			return true;
	}
	return false;
}

// returns the vector where element 'i' is permuted cyclically to the front
// sign is set to -1 for an odd number of cyclic permutations
static GiNaC::exvector permute_cyclic_to_front(const GiNaC::exvector& v,
		const GiNaC::ex& i, int& sign) {
	ASSERT(has_equal_or_dummy(v, i));
	sign = 1;
	exvector res;
	for (exvector::const_iterator it = v.begin(); it != v.end(); ++it) {
		if (is_equal_or_dummy_pair(*it, i)) {
			res.reserve(v.size());
			exvector::const_iterator it2 = it;
			do {
				res.push_back(*it2);
				++it2;
				if (it2 == v.end()) // v not empty
					it2 = v.begin();
			} while (it2 != it);
			return res;
		} else {
			sign = -sign;
		}
	}
	ABORT("Index " << i << " or " << ex_to<varidx> (i).toggle_variance()
			<< " is not contained in the vector " << v);
	return exvector();
}

// returns the four canonically ordered metric4 indices in 'indices'
// such that ordered indices in 'ordering' appear 'as early as possible'
// e.g.: ordering = {i1, i2, i3}, v = {i4, i2, i3, i1} => returns {i1, i3, i2, i4}
// ordering may be either the indices themselves or their dummy counterpart
static void metric4_tensor_canonical_indices(GiNaC::exvector& indices,
		const GiNaC::exvector& ordering, int& sign) {
	if (indices.size() != 4)
		ABORT("indices must contain four elements");

	for (exvector::const_iterator it = ordering.begin(); it != ordering.end(); ++it)
		ASSERT(has_equal_or_dummy(indices, *it));

	sign = 1;
	if (ordering.size() >= 1) {
		// bring ordered_indices[0] to first position (always possible)
		indices = permute_cyclic_to_front(indices, ordering[0], sign);
		ASSERT(is_equal_or_dummy_pair(indices[0], ordering[0]));

		if (ordering.size() >= 2) {
			// employ the remaining permutation 1 <-> 3  which leaves 0 unchanged
			if (is_equal_or_dummy_pair(ordering[1], indices[1])) {
				// second index is as wanted, don't touch anymore
			} else if (is_equal_or_dummy_pair(ordering[1], indices[3])) {
				// bring ordering[1] to second position
				swap(indices[1], indices[3]);
				sign = -sign;
			} else if (ordering.size() > 2 && is_equal_or_dummy_pair(
					ordering[2], indices[3])) {
				// bring ordering[2] to second position
				swap(indices[1], indices[3]);
				sign = -sign;
			}
		}
	}
}

// canonicalizes indices in two-fold contraction of two metric4 tensors:
// return one of 3 possible combinations
// v1 = [a,b,x,y]  &&  v2 = [a,b,u,v]
// v1 = [a,b,x,y]  &&  v2 = [a,u,b,v]
// v1 = [a,x,b,y]  &&  v2 = [a,u,b,v]
static void metric4_tensor_permute_2dummies_to_front(GiNaC::exvector& v1,
		GiNaC::exvector& v2, const GiNaC::exvector& dummies, int& sign) {
	ASSERT(v1.size() == 4);
	ASSERT(v2.size() == 4);
	ASSERT(dummies.size() == 2); // dummies = [a,b]

	for (exvector::const_iterator it = dummies.begin(); it != dummies.end(); ++it) {
		ASSERT(has_equal_or_dummy(v1, *it));
		ASSERT(has_equal_or_dummy(v2, *it));
	}

	int sign1(1), sign2(1);
	metric4_tensor_canonical_indices(v1, dummies, sign1);
	metric4_tensor_canonical_indices(v2, dummies, sign2);
	sign = sign1 * sign2;
	if (is_dummy_pair(v1[2], v2[1]))
		v1.swap(v2);

	ASSERT(is_dummy_pair(v1[0], v2[0]));
	ASSERT(is_equal_or_dummy_pair(v1[1], v2[1]) || is_equal_or_dummy_pair(
					v1[1], v2[2]) || is_equal_or_dummy_pair(v1[2], v2[2]));
}

// canonicalizes indices in three-fold contraction of two metric4 tensors:
// returns one of 3 possible combinations
// v1 = [a,b,c,x]  v2 = [a,b,c,y]
// v1 = [a,b,c,x]  v2 = [a,c,b,y]
// v1 = [a,b,c,x]  v2 = [a,b,y,c]  (== +[b,a,c,y])
static void metric4_tensor_permute_3dummies_to_front(GiNaC::exvector& v1,
		GiNaC::exvector& v2, const GiNaC::exvector& dummies, int& sign) {
	ASSERT(v1.size() == 4);
	ASSERT(v2.size() == 4);
	ASSERT(dummies.size() == 3);

	for (exvector::const_iterator it = dummies.begin(); it != dummies.end(); ++it) {
		ASSERT(has_equal_or_dummy(v1, *it));
		ASSERT(has_equal_or_dummy(v2, *it));
	}

	// scan to free index in vector v1
	exvector::const_iterator free;
	for (free = v1.begin(); free != v1.end(); ++free) {
		exvector::const_iterator d;
		for (d = dummies.begin(); d != dummies.end(); ++d)
			if (is_equal_or_dummy_pair(*free, *d))
				break;
		if (d == dummies.end())
			break;
	}
	ASSERT(free != v1.end());

	// set up dummies in order as they appear in v1 after the free index
	exvector dummies_v1_order;
	dummies_v1_order.reserve(3);
	exvector::const_iterator i = free;
	while ((++i != v1.end() || (i = v1.begin(), true)) && i != free) // v1 non-empty
		dummies_v1_order.push_back(*i);
	ASSERT(dummies_v1_order.size() == 3); // dummies_v1_order = [a,b,c]

	int sign1(1), sign2(1);
	metric4_tensor_canonical_indices(v1, dummies_v1_order, sign1); // v1 = [a,b,c,x]
	metric4_tensor_canonical_indices(v2, dummies_v1_order, sign2);
	sign = sign1 * sign2;

	ASSERT(v1[0].is_equal(dummies_v1_order[0]));
	ASSERT(v1[1].is_equal(dummies_v1_order[1]));
	ASSERT(v1[2].is_equal(dummies_v1_order[2]));
	ASSERT(is_equal_or_dummy_pair(v1[0], v2[0]));
	ASSERT((is_equal_or_dummy_pair(v1[1], v2[1]) && is_equal_or_dummy_pair(
							v1[2], v2[2])) || (is_equal_or_dummy_pair(v1[1], v2[2])
					&& is_equal_or_dummy_pair(v1[2], v2[1]))
			|| (is_equal_or_dummy_pair(v1[1], v2[1]) && is_equal_or_dummy_pair(
							v1[2], v2[3])));

}

// canonicalizes indices in three-fold contraction of two metric4 tensors:
// returns one of 3 possible combinations
// v1 = [a,b,c,d]  v2 = [a,b,c,d]
// v1 = [a,b,c,d]  v2 = [a,b,d,c]
// v1 = [a,b,c,d]  v2 = [a,c,b,d]
static void metric4_tensor_permute_4dummies_to_front(GiNaC::exvector& v1,
		GiNaC::exvector& v2, int& sign) {
	ASSERT(v1.size() == 4);
	ASSERT(v2.size() == 4);
	metric4_tensor_canonical_indices(v2, v1, sign);
	ASSERT(is_dummy_pair(v1[0], v2[0]));
}

/** Automatic symbolic evaluation of an indexed metricl4 tensor. */
GiNaC::ex metric4_tensor::eval_indexed(const GiNaC::basic& i) const {
	ASSERT(is_a<indexed> (i));
	ASSERT(i.nops() == 5);
	ASSERT(is_a<metric4_tensor> (i.op(0)));
	ASSERT(is_a<varidx> (i.op(1)));
	ASSERT(is_a<varidx> (i.op(2)));
	ASSERT(is_a<varidx> (i.op(3)));
	ASSERT(is_a<varidx> (i.op(4)));

	const ex Dd = ex_to<varidx> (i.op(1)).get_dim();
	//static ex Dd = Files::instance()->kinematics()->get_dimension();
	const ex dm1 = Ident(Dd - 1);

	exvector indices_dummy =
			static_cast<const indexed &> (i).get_dummy_indices();

	if (indices_dummy.empty())
		return i.hold();

	exvector indices = static_cast<const indexed &> (i).get_indices();
	int sign = 1;
	exvector v = permute_cyclic_to_front(indices, indices_dummy[0], sign);

	// m(a,a,b,c) = (d-1) g(b,c);
	if (is_dummy_pair(v[0], v[1]))
		return sign * dm1 * lorentz_g(v[2], v[3]);

	// m(a,b,a,c) = 0
	if (is_dummy_pair(v[0], v[2]))
		return ex(0);

	// m(a,b,c,a) =  - (d-1) g(b,c);
	if (is_dummy_pair(v[0], v[3]))
		return -sign * dm1 * lorentz_g(v[1], v[2]);

	// m(a,a,b,b) = (d-1) d;
	if (is_dummy_pair(v[0], v[1]) && is_dummy_pair(v[2], v[3]))
		return sign * dm1 * Dd;

	// m(a,b,b,a) =  - (d-1) d;
	if (is_dummy_pair(v[0], v[3]) && is_dummy_pair(v[1], v[2]))
		return -sign * dm1 * Dd;

	ABORT("Logic Error: " << indices_dummy << ", " << indices);
	return 0;
}

///** Contraction of metric4 tensor with something else. */
bool metric4_tensor::contract_with(GiNaC::exvector::iterator self,
		GiNaC::exvector::iterator other, GiNaC::exvector & v) const {

	ASSERT(is_a<indexed> (*self));
	ASSERT(is_a<indexed> (*other));
	ASSERT(self->nops() == 5);
	ASSERT(is_a<metric4_tensor> (self->op(0)));
	ASSERT(is_a<varidx> (self->op(1)));
	ASSERT(is_a<varidx> (self->op(2)));
	ASSERT(is_a<varidx> (self->op(3)));
	ASSERT(is_a<varidx> (self->op(4)));

	//static ex Dd = Files::instance()->kinematics()->get_dimension();
	const ex Dd = ex_to<varidx> (self->op(1)).get_dim();
	const ex dm1 = Ident(Dd - 1);

	if (is_a<metric4_tensor> (other->op(0))) {
		ASSERT(is_a<varidx> (other->op(1)));
		ASSERT(is_a<varidx> (other->op(2)));
		ASSERT(is_a<varidx> (other->op(3)));
		ASSERT(is_a<varidx> (other->op(4)));

		exvector dummy_indices = ex_to<indexed> (*self).get_dummy_indices(
				ex_to<indexed> (*other));
		exvector v1 = ex_to<indexed> (*self).get_indices();
		exvector v2 = ex_to<indexed> (*other).get_indices();

		if (dummy_indices.size() == 2) {
			int sign(1);
			metric4_tensor_permute_2dummies_to_front(v1, v2, dummy_indices,
					sign);
			// 3 possible combinations
			// v1 = [a,b,x,y]  &&  v2 = [a,b,u,v]   ==> (d-1) * g(y,x) * g(u,v) -  m(y,x,u,v)
			// v1 = [a,b,x,y]  &&  v2 = [a,u,b,v]   ==> m(u,x,v,y)
			// v1 = [a,x,b,y]  &&  v2 = [a,u,b,v]   ==> 2 * m(u,x,v,y)

			ASSERT(is_dummy_pair(v1[0], v2[0]));

			if (is_dummy_pair(v1[1], v2[1])) {
				*self = dm1 * lorentz_g(v1[3], v1[2]) * lorentz_g(v2[2], v2[3])
						- metric_l_diff_wrapper(v1[3], v1[2], v2[2], v2[3]);
				*other = sign;
				return true;

			} else if (is_dummy_pair(v1[1], v2[2])) {
				*self = metric_l_diff_wrapper(v2[1], v1[2], v2[3], v1[3]);
				*other = sign;
				return true;

			} else if (is_dummy_pair(v1[2], v2[2])) {
				*self = metric_l_diff_wrapper(v2[1], v1[1], v2[3], v1[3]);
				*other = 2 * sign;
				return true;
			} else {
				ABORT("Logic error");
			}
			return false;
		}

		if (dummy_indices.size() == 3) {
			int sign(1);
			metric4_tensor_permute_3dummies_to_front(v1, v2, dummy_indices,
					sign);

			// 3 possible combinations
			// v1 = [a,b,c,x]  v2 = [a,b,c,y]  ==>  2 (-1 + d) g[x, y]
			// v1 = [a,b,c,x]  v2 = [a,c,b,y]  ==>    (-1 + d) g[x, y]
			// v1 = [a,b,c,x]  v2 = [a,b,y,c]  ==>    (-1 + d) g[x, y]

			if (/*   */is_dummy_pair(v1[0], v2[0]) //
					&& is_dummy_pair(v1[1], v2[1]) //
					&& is_dummy_pair(v1[2], v2[2])) {
				*self = lorentz_g(v1[3], v2[3]);
				*other = 2 * dm1 * sign;
				return true;
			}
			if (/*   */is_dummy_pair(v1[0], v2[0]) //
					&& is_dummy_pair(v1[1], v2[2])) {
				ASSERT(is_dummy_pair(v1[2], v2[1]));
				*self = lorentz_g(v1[3], v2[3]);
				*other = dm1 * sign;
				return true;
			}
			if (/*   */is_dummy_pair(v1[0], v2[0]) //
					&& is_dummy_pair(v1[1], v2[1]) //
					&& is_dummy_pair(v1[2], v2[3])) {
				*self = lorentz_g(v1[3], v2[2]);
				*other = dm1 * sign;
				return true;
			}

			ABORT("Logic error: " << v1 << ", " << v2);
		}

		if (dummy_indices.size() == 4) {
			int sign(1);
			metric4_tensor_permute_4dummies_to_front(v1, v2, sign);

			// 3 possible combinations
			// v1 = [a,b,c,d]  v2 = [a,b,c,d]  ==> 2 * d * (d-1)
			// v1 = [a,b,c,d]  v2 = [a,b,d,c]  ==> d * (d-1)
			// v1 = [a,b,c,d]  v2 = [a,c,b,d]  ==> d * (d-1)

			ASSERT(is_dummy_pair(v1[0], v2[0]));

			if (/*   */is_dummy_pair(v1[1], v2[1]) //
					&& is_dummy_pair(v1[2], v2[2])) {
				ASSERT(is_dummy_pair(v1[3], v2[3]));
				*self = 2 * Dd * dm1;
				*other = sign;
				return true;

			} else if (is_dummy_pair(v1[1], v2[1]) //
					&& is_dummy_pair(v1[2], v2[3])) {
				ASSERT(is_dummy_pair(v1[3], v2[2]));
				*self = Dd * dm1;
				*other = sign;
				return true;

			} else if (is_dummy_pair(v1[1], v2[2]) //
					&& is_dummy_pair(v1[2], v2[1])) {
				ASSERT(is_dummy_pair(v1[3], v2[3]));
				*self = Dd * dm1;
				*other = sign;
				return true;

			} else {
				ABORT("Logic error " << dummy_indices << ", " << v1 << ", "
						<< v2);
			}
		}

		// replace them by the metric tensors \todo better solution ???
		*self = lorentz_g(self->op(1), self->op(2)) * lorentz_g(self->op(3),
				self->op(4)) - lorentz_g(self->op(1), self->op(4)) * lorentz_g(
				self->op(2), self->op(3));
		*other = lorentz_g(other->op(1), other->op(2)) * lorentz_g(
				other->op(3), other->op(4)) - lorentz_g(other->op(1),
				other->op(4)) * lorentz_g(other->op(2), other->op(3));
		return true;

	} else /*if (is_a<momentum_tensor> (other->op(0)))*/{
		*self = lorentz_g(self->op(1), self->op(2)) * lorentz_g(self->op(3),
				self->op(4)) - lorentz_g(self->op(1), self->op(4)) * lorentz_g(
				self->op(2), self->op(3));
		return true;
	}

	return false;
}
//
//
//
// four_vector_tensor
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(four_vector_tensor, tensor,
		print_func<print_dflt>(&four_vector_tensor::do_print))

GINAC_BIND_UNARCHIVER(four_vector_tensor)
;

four_vector_tensor::four_vector_tensor() {
}

four_vector_tensor::four_vector_tensor(const GiNaC::ex& p) :
	vector_(p) {
}

void four_vector_tensor::do_print(const GiNaC::print_context& c, unsigned level) const {
	c.s << "fv(" << vector_ << ")";
}

int four_vector_tensor::compare_same_type(const GiNaC::basic& other) const {
	const four_vector_tensor& o =
			static_cast<const four_vector_tensor&> (other);
	if (!vector_.is_equal(o.vector_))
		return vector_.compare(o.vector_);
	return 0;
}

size_t four_vector_tensor::nops() const {
	return 1;
}

GiNaC::ex four_vector_tensor::op(size_t i) const {
	switch (i) {
	case 0:
		return vector_;
	default:
		throw std::range_error("Invalid operand");
	}
}

GiNaC::ex& four_vector_tensor::let_op(size_t i) {
	switch (i) {
	case 0:
		return vector_;
	default:
		throw std::range_error("Invalid operand");
	}
}

GiNaC::ex four_vector_tensor::get_vector() const {
	return vector_;
}

GiNaC::ex four_vector_tensor::eval_indexed(const GiNaC::basic& i) const {
	ASSERT(is_a<indexed> (i));
	ASSERT(i.nops() > 1);
	ASSERT(is_a<four_vector_tensor> (i.op(0)));
	ASSERT(is_a<varidx> (i.op(1)));
	return i.hold();
}

bool four_vector_tensor::contract_with(GiNaC::exvector::iterator self,
		GiNaC::exvector::iterator other, GiNaC::exvector & v) const {
	ASSERT(is_a<indexed> (*self));
	ASSERT(self->nops() == 2);
	ASSERT(is_a<four_vector_tensor> (self->op(0)));
	ASSERT(is_a<varidx> (self->op(1)));
	ASSERT(is_a<indexed> (*other));

	if (is_a<four_vector_tensor> (other->op(0))) {
		ASSERT(other->nops() == 2);
		ASSERT(is_a<varidx> (other->op(1)));
		ex p1 = ex_to<four_vector_tensor> (self->op(0)).get_vector();
		ex p2 = ex_to<four_vector_tensor> (other->op(0)).get_vector();
		*self = ScalarProduct(p1, p2);
		*other = 1;
		return true;
	}

	return false;
}
//
//
//
// Polarization
GINAC_IMPLEMENT_REGISTERED_CLASS(Polarization, basic)

Polarization::Polarization() :
	momentum_(four_vector_tensor(0)), mass_(0), conjugated_(false),
			orthogonal_vector_(four_vector_tensor(0)) {
}

Polarization::Polarization(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		bool conjugated, const GiNaC::ex& orthogonal_vector) :
	momentum_(momentum), mass_(mass), conjugated_(conjugated),
			orthogonal_vector_(orthogonal_vector) {

	if (!is_a<four_vector_tensor> (momentum_)) {
		throw std::invalid_argument("momentum " + to_string(momentum_)
				+ " is not of type four_vector_tensor");
	}
	if (!is_a<four_vector_tensor> (orthogonal_vector_)) {
		throw std::invalid_argument("orthogonal_vector " + to_string(
				orthogonal_vector_) + " is not of type four_vector_tensor");
	}
}

Polarization::~Polarization() {
}

int Polarization::compare_same_type(const GiNaC::basic& other) const {
	const Polarization& o = static_cast<const Polarization&> (other); // safe

	if (conjugated_ && !o.conjugated_)
		return -1;
	if (!conjugated_ && o.conjugated_)
		return 1;

	if (!momentum_.is_equal(o.momentum_))
		return momentum_.compare(o.momentum_);
	if (!mass_.is_equal(o.mass_))
		return mass_.compare(o.mass_);
	if (!orthogonal_vector_.is_equal(o.orthogonal_vector_))
		return orthogonal_vector_.compare(o.orthogonal_vector_);

	return 0;
}

size_t Polarization::nops() const {
	return 3;
}

GiNaC::ex Polarization::op(size_t i) const {
	switch (i) {
	case 0:
		return momentum_;
	case 1:
		return mass_;
	case 2:
		return orthogonal_vector_;
	default:
		throw std::range_error("Invalid operand");
	}
}

GiNaC::ex& Polarization::let_op(size_t i) {
	switch (i) {
	case 0:
		return momentum_;
	case 1:
		return mass_;
	case 2:
		return orthogonal_vector_;
	default:
		throw std::range_error("Invalid operand");
	}
}

GiNaC::ex Polarization::momentum_raw() const {
	ASSERT(is_a<four_vector_tensor>(momentum_));
	return ex_to<four_vector_tensor> (momentum_).get_vector();
}
GiNaC::ex Polarization::orthogonal_vector_raw() const {
	ASSERT(is_a<four_vector_tensor>(orthogonal_vector_));
	return ex_to<four_vector_tensor> (orthogonal_vector_).get_vector();
}

GiNaC::ex Polarization::conjugate() const {
	return Polarization(momentum_, mass_, !conjugated_, orthogonal_vector_);
}

std::string Polarization::unique_pol_sum_name(const Polarization& other) const {
	return pol_sum_name() + "*" + other.pol_sum_name();
}

std::string Polarization::pol_sum_name() const {
	string name(this->class_name());
	if (!orthogonal_vector_raw().is_zero())
		name += "/ortho";
	if (is_conjugated())
		name += "/c";
	return name;
}
//
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(GluonPolarization, Polarization,
		print_func<print_context>(&GluonPolarization::do_print))
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(GhostPolarization, Polarization,
		print_func<print_context>(&GhostPolarization::do_print))
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(PhotonPolarization, Polarization,
		print_func<print_context>(&PhotonPolarization::do_print))
GINAC_IMPLEMENT_REGISTERED_CLASS_OPT(Spinor, Polarization,
		print_func<print_context>(&Spinor::do_print))
//


GluonPolarization::GluonPolarization() :
	inherited() {
}

GluonPolarization::GluonPolarization(const GiNaC::ex& momentum,
		const GiNaC::ex& mass, bool conjugated,
		const GiNaC::ex& orthogonal_vector) :
	inherited(momentum, mass, conjugated, orthogonal_vector) {

}

GiNaC::ex GluonPolarization::conjugate() const {
	return GluonPolarization(momentum_, mass_, !conjugated_, orthogonal_vector_);
}

int GluonPolarization::compare_same_type(const GiNaC::basic& other) const {
	return inherited::compare_same_type(other);
}

void GluonPolarization::do_print(const GiNaC::print_context& c, unsigned level) const {
	string polar("pol");
	polar += conjugated_ ? "c_gluon" : "_gluon";
	c.s << polar << '(' << momentum_ << ',' << mass_;
	c.s << "," << orthogonal_vector_;
	c.s << ')';
}

//
//
//


GhostPolarization::GhostPolarization() :
	inherited() {
}
ex GhostPolarization::conjugate() const {
	return GhostPolarization(momentum_, mass_, !conjugated_);
}
int GhostPolarization::compare_same_type(const GiNaC::basic& other) const {
	return inherited::compare_same_type(other);
}
void GhostPolarization::do_print(const GiNaC::print_context& c, unsigned level) const {
	string polar("pol");
	polar += conjugated_ ? "c_ghost" : "_ghost";
	c.s << polar << '(' << momentum_ << ',' << mass_ << ')';
}

//
//
//


PhotonPolarization::PhotonPolarization() :
	inherited() {
}
GiNaC::ex PhotonPolarization::conjugate() const {
	return PhotonPolarization(momentum_, mass_, !conjugated_);
}
int PhotonPolarization::compare_same_type(const GiNaC::basic& other) const {
	return inherited::compare_same_type(other);
}
void PhotonPolarization::do_print(const GiNaC::print_context& c, unsigned level) const {
	string polar("pol");
	polar += conjugated_ ? "c_photon" : "_photon";
	c.s << polar << '(' << momentum_ << ',' << mass_ << ')';
}

//
//
//


Spinor::Spinor() :
	inherited(), u_type_(true) {
}
Spinor::Spinor(const GiNaC::ex& momentum, const GiNaC::ex& mass, bool u_type,
		bool conjugated) :
	inherited(momentum, mass, conjugated), u_type_(u_type) {
}
Spinor::~Spinor() {
}

int Spinor::compare_same_type(const GiNaC::basic& other) const {
	const Spinor& o = static_cast<const Spinor&> (other); // safe
	if (u_type_ && !o.u_type_)
		return 1;
	if (!u_type_ && o.u_type_)
		return -1;

	return inherited::compare_same_type(other);
}

void Spinor::do_print(const GiNaC::print_context& c, unsigned level) const {
	string spinor;
	spinor += u_type_ ? "u" : "v";
	spinor += conjugated_ ? "bar" : "";
	c.s << spinor << '(' << momentum_ << ',' << mass_ << ')';
}

GiNaC::ex Spinor::conjugate() const {
	return Spinor(momentum_, mass_, u_type_, !conjugated_);
}

std::string Spinor::pol_sum_name() const {
	string name(class_name());
	if (conjugated_)
		name += u_type_ ? "/ubar" : "/vbar";
	else
		name += u_type_ ? "/u" : "/v";
	return name;
}

//
// global functions
//


//GiNaC::ex polarization_sum(const GiNaC::ex& a, const GiNaC::ex& b,
//		const GiNaC::ex& dim) {
//
//	const FeynmanRules* fr = Files::instance()->get_feynmanrules();
//
//	cerr << endl << endl;
//	cerr << "request polarization sum for  " << a << " " << b << endl;
//	ex polar = fr->polarization_sum(a, b);
//	cerr << "pol " << polar << endl << endl;
//
//	if (a.nops() == 0 || b.nops() == 0)
//		ABORT("Can't perform polarization sum in " << a << " * " << b);
//
//	/* spin sum of (eps_\mu eps^*_\nu) is controlled by orthogonal_vector n:
//	 *   => delta_ab (-g_\mu\nu)                                     if n==0,
//	 *   => delta_ab (-g_\mu\nu + (k_\mu n_\nu + n_\mu k_\nu)/(k n)
//	 *                          + n^2 k_\mu k_\nu / (k n)^2) )       if n!=0
//	 */
//	if (is_a<GluonPolarization> (a.op(0)) && is_a<GluonPolarization> (b.op(0))) {
//		ASSERT(a.nops() == 3);
//		ASSERT(b.nops() == 3);
//		GluonPolarization ga = ex_to<GluonPolarization> (a.op(0));
//		GluonPolarization gb = ex_to<GluonPolarization> (b.op(0));
//		ASSERT(is_a<varidx> (a.op(1))); // mu
//		ASSERT(is_a<varidx> (b.op(1))); // mu
//		ASSERT(is_a<idx2> (a.op(2))); // a
//		ASSERT(is_a<idx2> (b.op(2))); // a
//
//		if (ga.has_same_mass_mom_ortho(gb) //
//				&& ga.is_conjugated() != gb.is_conjugated()) {
//			ex n = ga.orthogonal_vector();
//			ASSERT(is_a<four_vector_tensor> (n));
//			ex nraw = ex_to<four_vector_tensor> (n).get_vector();
//			ex deltacolor = delta_tensor(a.op(2), b.op(2));
//			ex mu = a.op(1);
//			ex nu = b.op(1);
//			ex gmunu = lorentz_g(mu, nu);
//			if (nraw.is_zero()) {
//				LOGX("using Feynman gluon polarization sum for " << ga << "*"
//						<< gb);
//				ex res = deltacolor * (-gmunu);
//				cerr << "res " << res << endl;
//				VERIFY(expand(polar - res).is_zero());
//				return deltacolor * (-gmunu);
//			} else {
//				LOGX("using physical gluon polarization sum for " << ga << "*"
//						<< gb);
//				ex k = ga.momentum();
//				ASSERT(is_a<four_vector_tensor> (k));
//				ex kraw = ex_to<four_vector_tensor> (k).get_vector();
//				ex kmu = indexed(k, mu);
//				ex knu = indexed(k, nu);
//				ex nmu = indexed(n, mu);
//				ex nnu = indexed(n, nu);
//				ex kn = ScalarProduct(kraw, nraw);
//				ex nn = ScalarProduct(nraw, nraw);
//
//				ex res = deltacolor * (-gmunu //
//						+ (kmu * nnu + nmu * knu) / kn //
//						+ nn * kmu * knu / pow(kn, 2));
//				cerr << "res " << res << endl;
//				VERIFY(expand(polar - res).is_zero());
//				return deltacolor * (-gmunu //
//						+ (kmu * nnu + nmu * knu) / kn //
//						+ nn * kmu * knu / pow(kn, 2));
//			}
//		}
//	}
//
//	/// sum_lambda pol_ghost polc_ghost = delta_ab
//	if (is_a<GhostPolarization> (a.op(0)) && is_a<GhostPolarization> (b.op(0))) {
//		ASSERT(a.nops() == 2);
//		ASSERT(b.nops() == 2);
//		GhostPolarization ga = ex_to<GhostPolarization> (a.op(0));
//		GhostPolarization gb = ex_to<GhostPolarization> (b.op(0));
//		ASSERT(is_a<idx2> (a.op(1))); // a
//		ASSERT(is_a<idx2> (b.op(1))); // a
//
//		if (ga.has_same_mass_mom_ortho(gb) && ga.is_conjugated()
//				!= gb.is_conjugated()) {
//			// \todo: return I*.... to adjust signs from ghosts
//			ex res = delta_tensor(a.op(1), b.op(1));
//			cerr << "res " << res << endl;
//			VERIFY(expand(polar - res).is_zero());
//			return delta_tensor(a.op(1), b.op(1));
//		}
//	}
//
//	/// sum_lambda pol_photon polc_photon = - g_{mu,nu}
//	if (is_a<PhotonPolarization> (a.op(0))
//			&& is_a<PhotonPolarization> (b.op(0))) {
//		ASSERT(a.nops() == 2);
//		ASSERT(b.nops() == 2);
//		PhotonPolarization ga = ex_to<PhotonPolarization> (a.op(0));
//		PhotonPolarization gb = ex_to<PhotonPolarization> (b.op(0));
//		ASSERT(is_a<varidx> (a.op(1))); // a
//		ASSERT(is_a<varidx> (b.op(1))); // a
//
//		if (ga.has_same_mass_mom_ortho(gb) && ga.is_conjugated()
//				!= gb.is_conjugated()) {
//			ex res = -lorentz_g(a.op(1), b.op(1));
//			cerr << "res " << res << endl;
//			VERIFY(expand(polar - res).is_zero());
//			return -lorentz_g(a.op(1), b.op(1));
//		}
//	}
//
//	if (is_a<Spinor> (a.op(0)) && is_a<Spinor> (b.op(0))) {
//		VERIFY(!dim.is_zero());
//		ASSERT(a.nops() == 2);
//		ASSERT(b.nops() == 2);
//		Spinor spina = ex_to<Spinor> (a.op(0));
//		Spinor spinb = ex_to<Spinor> (b.op(0));
//
//		// sum_sigma v vbar = (p_slash - m) delta_ij
//		if (spina.is_v() && spinb.is_vbar() && spina.has_same_mass_mom_ortho(
//				spinb)) {
//			ex p = spina.momentum();
//			ex m = spina.mass();
//			ex i = a.op(1);
//			ex j = b.op(1);
//			ex res = (dirac_slash(p, dim) - m * dirac_ONE()) * delta_tensor(i,
//					j);
//			cerr << "res " << res << endl;
//			VERIFY(expand(polar - res).is_zero());
//			return (dirac_slash(p, dim) - m * dirac_ONE()) * delta_tensor(i, j);
//		}
//
//		// sum_sigma u ubar = (p_slash + m) delta_ij
//		if (spina.is_u() && spinb.is_ubar() && spina.has_same_mass_mom_ortho(
//				spinb)) {
//			ex p = spina.momentum();
//			ex m = spina.mass();
//			ex i = a.op(1);
//			ex j = b.op(1);
//			ex res = (dirac_slash(p, dim) + m * dirac_ONE()) * delta_tensor(i,
//					j);
//			cerr << "res " << res << endl;
//			VERIFY(expand(polar - res).is_zero());
//			return (dirac_slash(p, dim) + m * dirac_ONE()) * delta_tensor(i, j);
//		}
//	}
//
//	ABORT("Can't perform polarization sum in " << a << " * " << b);
//	return 0; // suppress compiler warning
//}

/// returns the iterator 'pos' to the object in 'v' such that pos.op(0) == self.op(0).conjuagate()
static GiNaC::exvector::iterator conjugated_partner_of(GiNaC::exvector& v,
		GiNaC::exvector::iterator& self) {
	ex partner = self->conjugate();
	ASSERT(partner.nops() > 0);
	for (exvector::iterator it = v.begin(); it != v.end(); ++it) {
		if (it != self && it->nops() > 0 && it->op(0).is_equal(partner.op(0)))
			return it;
	}
	ABORT("No complex conjugated value found for: " << *self << " in " << v);
	return v.end();
}

// returns the polarization sum of the indexed Polarization objects in 'va' with corresponding one in 'vb'
GiNaC::ex polarization_sum(const GiNaC::exvector& va,
		const GiNaC::exvector& vb, const FeynmanRules* fr) {
	VERIFY(va.size() == vb.size());
	exvector v(va);
	exvector v2(vb);
	ex polarization(1);
	for (exvector::iterator it = v.begin(); it != v.end(); ++it) {
		if (!is_a<indexed> (*it))
			continue;
		ASSERT(it->nops() > 0);
		ASSERT(is_a<Polarization> (it->op(0)));
		exvector::iterator partner = conjugated_partner_of(v2, it);
		ASSERT(partner->nops() > 0);
		ASSERT(is_a<Polarization> (partner->op(0)));
		polarization *= fr->get_polarization_sum(*it, *partner);
		*it = 1;
		*partner = 1;
	}
	ASSERT(ex(mul(v)).is_equal(1));
	ASSERT(ex(mul(v2)).is_equal(1));
	return polarization;
}

// wrapper functions


GiNaC::ex metric_l_diff_wrapper(const GiNaC::ex& e1, const GiNaC::ex& e2,
		const GiNaC::ex& e3, const GiNaC::ex& e4) {
	if (!is_a<varidx> (e1) || !is_a<varidx> (e2) || !is_a<varidx> (e3)
			|| !is_a<varidx> (e4))
		throw(std::invalid_argument(
				"indices of metric_l_diff must be of type varidx"));
	ex f = (new metric4_tensor)->setflag(status_flags::dynallocated);
	return indexed(f, sy_none(sy_anti(0, 2), sy_anti(1, 3)), e1, e2, e3, e4);
}

GiNaC::ex four_vector_tensor_wrapper(const GiNaC::ex& p, const GiNaC::ex& mu) {
	if (!is_a<varidx> (mu))
		throw(std::invalid_argument(
				"index of four_vector_tensor must be of type varidx"));
	ex f = (new four_vector_tensor(p))->setflag(status_flags::dynallocated);
	return indexed(f, mu);
}

// gluon
GiNaC::ex pol_gluon_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& mu, const GiNaC::ex& a, const GiNaC::ex& ortho) {
	if (!is_a<varidx> (mu) || !is_a<idx2> (a))
		throw(std::invalid_argument(
				"indices of pol_gluon must be of type varidx and idx2"));
	ex f = (new GluonPolarization(momentum, mass, false, ortho))->setflag(
			status_flags::dynallocated);
	return indexed(f, sy_none(), mu, a);
}

GiNaC::ex polc_gluon_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& mu, const GiNaC::ex& a, const GiNaC::ex& ortho) {
	if (!is_a<varidx> (mu) || !is_a<idx2> (a))
		throw(std::invalid_argument(
				"indices of polc_gluon must be of type varidx and idx2"));
	ex f = (new GluonPolarization(momentum, mass, true, ortho))->setflag(
			status_flags::dynallocated);
	return indexed(f, sy_none(), mu, a);
}

GiNaC::ex pol_ghost_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& a) {
	if (!is_a<idx2> (a))
		throw(std::invalid_argument("index of pol_ghost must be of type idx2"));
	ex f = (new GhostPolarization(momentum, mass, false))->setflag(
			status_flags::dynallocated);
	return indexed(f, a);
}

GiNaC::ex polc_ghost_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& a) {
	if (!is_a<idx2> (a))
		throw(std::invalid_argument("index of polc_ghost must be of type idx2"));
	ex f = (new GhostPolarization(momentum, mass, true))->setflag(
			status_flags::dynallocated);
	return indexed(f, a);
}

GiNaC::ex pol_photon_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& mu) {
	if (!is_a<varidx> (mu))
		throw(std::invalid_argument(
				"index of pol_photon must be of type varidx"));
	ex f = (new PhotonPolarization(momentum, mass, false))->setflag(
			status_flags::dynallocated);
	return indexed(f, mu);
}

GiNaC::ex polc_photon_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& mu) {
	if (!is_a<varidx> (mu))
		throw(std::invalid_argument(
				"index of polc_photon must be of type varidx"));
	ex f = (new PhotonPolarization(momentum, mass, true))->setflag(
			status_flags::dynallocated);
	return indexed(f, mu);
}

GiNaC::ex u_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& i) {
	if (!is_a<idx> (i))
		throw(std::invalid_argument("index of u must be of type idx"));
	ex f = (new Spinor(momentum, mass, true, false))->setflag(
			status_flags::dynallocated);
	return indexed(f, i);
}

GiNaC::ex ubar_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& i) {
	if (!is_a<idx> (i))
		throw(std::invalid_argument("index of ubar must be of type idx"));
	ex f = (new Spinor(momentum, mass, true, true))->setflag(
			status_flags::dynallocated);
	return indexed(f, i);
}

GiNaC::ex v_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& i) {
	if (!is_a<idx> (i))
		throw(std::invalid_argument("index of v must be of type idx"));
	ex f = (new Spinor(momentum, mass, false, false))->setflag(
			status_flags::dynallocated);
	return indexed(f, i);
}

GiNaC::ex vbar_wrapper(const GiNaC::ex& momentum, const GiNaC::ex& mass,
		const GiNaC::ex& i) {
	if (!is_a<idx> (i))
		throw(std::invalid_argument("index of vbar must be of type idx"));
	ex f = (new Spinor(momentum, mass, false, true))->setflag(
			status_flags::dynallocated);
	return indexed(f, i);
}

} // namespace Reduze


