/*  job_setupsectormappingsalt.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_SETUPSECTORMAPPINGSALT_H_
#define JOB_SETUPSECTORMAPPINGSALT_H_

#include "job.h"
#include "sector.h"

namespace Reduze {

/// Job to set up zero sectors and shifts between sectors via combinatorics
class SetupSectorMappingsAlt: public Reduze::Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("setup_sector_mappings_alt");
		s.set_short_description(
				"Finds shifts between sectors via combinatorics.");
		s.set_long_description(""//
					"Job to set up or update sector mappings files."
					" This job finds shifts between sectors by a straight-forward"
					" attempt to match propagators in all combinatorically possible"
					" ways. For each such combination linear algebra methods"
					" are used to derive a shift of loop and external momenta."
					" Shifts of loop momenta are restricted to those with a"
					" modulus 1 Jacobi determinant. External momenta may be"
					" permuted."
					" Zero sectors may be found by an explicit zero test:"
					" the algorithm reduces a small set of IBPs and if the scalar"
					" (corner) integral is found to reduce to zero, the sector is"
					" considered to be zero."
					" All results are stored in the corresponding sectormappings*.yaml"
					" file which is used to eliminate redundant and zero sectors in future"
					" calculations."
					" This is a robust and general but also slow approach."
					" Often it is better to start with the graph "
					" based job 'setup_sector_mappings' and potentially improve its"
					" results with this job."
					" If this job is used to update existing sector mappings files,"
					" sectors for which a shift was found before are skipped."
					" Existing reductions can be updated with the job 'normalize'.");
		s.add_option("source_sectors", true, "sector_selection", ""//
					"Sectors for which rules should be generated. This option "
					" does not affect possible shift targets, for those all"
					" respective lower sectors are considered. Usually it is best"
				    " to just use select_all:true here.");
		s.add_option("find_sector_relations", false, "boolean", ""//
					"Whether to find shifts of loop and external momenta,"
					" which map a sector to a lower sector."
					" See help for 'sector' to find out about sector ordering.");
		s.add_option("find_crossed_sector_relations", false, "boolean", ""//
					"Whether to find shifts of loop and external momenta,"
					" which map a crossed sector onto another crossing of itself.");
		s.add_option("find_sector_symmetries", false, "boolean", ""//
					"Whether to find shifts of loop and external momenta,"
					" which map a sector onto itself. A possible crossing"
					" of external momenta is required to leave the kinematic"
					" invariants unchanged.");
		s.add_option("find_zero_sectors", false, "boolean", ""//
					"Whether to test for zero sectors by performing a small"
					" IBP reduction.");
		s.add_option("verify_permutation_symmetries", false, "boolean", ""//
					"Whether the permutation symmetries of the integral families"
					" should be verified.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}
	SetupSectorMappingsAlt() :
			find_zero_sectors_(true), find_sector_relations_(true),
			find_crossed_sector_relations_(true),
			find_sector_symmetries_(true),
			verify_permutation_symmetries_(true) {
		add_auto_options();
	}
	virtual ~SetupSectorMappingsAlt() {
	}
	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&,//
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;

protected:
	virtual void add_auto_options() {
		add_auto_io("source_sectors", source_sectors_);
		add_auto_io("find_zero_sectors", find_zero_sectors_);
		add_auto_io("find_sector_relations", find_sector_relations_);
		add_auto_io("find_crossed_sector_relations", find_crossed_sector_relations_);
		add_auto_io("find_sector_symmetries", find_sector_symmetries_);
		add_auto_io("verify_permutation_symmetries", verify_permutation_symmetries_);
	}
	virtual void init();

private:
	UncrossedReadSectorSelection source_sectors_;
	bool find_zero_sectors_;
	bool find_sector_relations_;
	bool find_crossed_sector_relations_;
	bool find_sector_symmetries_;
	bool verify_permutation_symmetries_;
};

}

#endif /* JOB_SETUPSECTORMAPPINGSALT_H_ */
