/*  job_setupsectormappingsalt.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_setupsectormappingsalt.h"

#include "kinematics.h"
#include "sectormappings.h"
#include "functions.h"
#include "files.h"
#include "ginacutils.h"
#include "graph.h"
#include "job_setupsectormappings.h"
#include "equation.h" // check if symmetry shift has trivial effect

using namespace std;
using namespace GiNaC;

namespace Reduze {

// register job type
namespace {
JobProxy<SetupSectorMappingsAlt> dummy;
}

bool SetupSectorMappingsAlt::find_dependencies(const set<string>& outothers, //
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	bool setup_crossings = false;
	bool res = SetupSectorMappings::get_dependencies(outothers, in, out,
			auxjobs, setup_crossings, verify_permutation_symmetries_,
			is_conditional());
	return res;
}


void SetupSectorMappingsAlt::init() {
}

std::string SetupSectorMappingsAlt::get_description() const {
	return string("setup sector mappings (alt)");
}

class LessExmap {
public:
	bool operator()(const GiNaC::exmap& a, const GiNaC::exmap& b) const {
		if (a.size() != b.size())
			return a.size() < b.size();
		exmap::const_iterator i, j;
		size_t arhs = 0;
		size_t brhs = 0;
		for (i = a.begin(); i != a.end(); ++i)
			arhs += i->second.nops();
		for (j = a.begin(); j != a.end(); ++j)
			brhs += j->second.nops();
		if (arhs != brhs)
			return arhs < brhs;
		for (i = a.begin(), j = b.begin(); i != a.end() && j != b.end(); ++i, ++j)
			if (!i->first.is_equal(j->first))
				return i->first.compare(j->first) < 0;
			else if (!i->second.is_equal(j->second))
				return i->second.compare(j->second) < 0;
		return false;
	}
};

// returns shifts of momentum combinations needed to match pa to pb up to a sign
// each element in the return value represents one such possibility, possible
// consisting of several equations (needed for bilinear)
stack<lst> find_propagator_mappings(const Propagator& sp, const Propagator& dp,
		const lst& srcmoms) {
	stack<lst> res;
	const ex& s1 = sp.momentum1();
	const ex& d1 = dp.momentum1();
	if (sp.has_squared_momentum() && dp.has_squared_momentum()) {
		if (sp.squaredmass().is_equal(dp.squaredmass())) {
			// no overall sign
#ifdef NEW_GINAC
            res.push(lst({s1 == d1}));
            res.push(lst({s1 == -d1}));
#else
            res.push(lst(s1 == d1));
            res.push(lst(s1 == -d1));
#endif
		}
	} else if (!sp.has_squared_momentum() && !dp.has_squared_momentum()) {
		vector<lst> l;
		l.reserve(8);
		const ex& s2 = sp.momentum2();
		const ex& d2 = dp.momentum2();
		if (sp.squaredmass().is_equal(dp.squaredmass())) {
			// no overall sign:
#ifdef NEW_GINAC
            l.push_back(lst({s1 == d1, s2 == d2}));
            l.push_back(lst({s1 == -d1, s2 == -d2}));
            l.push_back(lst({s1 == d2, s2 == d1}));
            l.push_back(lst({s1 == -d2, s2 == -d1}));
#else
            l.push_back(lst(s1 == d1, s2 == d2));
            l.push_back(lst(s1 == -d1, s2 == -d2));
            l.push_back(lst(s1 == d2, s2 == d1));
            l.push_back(lst(s1 == -d2, s2 == -d1));
#endif
		}
#ifdef GENERATE_UNHANDLED_SHIFTS
		// right now such shifts can't be handled by INT::apply_shift()
		// at some point we will improve on that
		if (sp.squaredmass().is_equal(-dp.squaredmass())) {
			// overall minus sign
			l.push_back(lst(s1 == d1, s2 == -d2));
			l.push_back(lst(s1 == -d1, s2 == d2));
			l.push_back(lst(s1 == d2, s2 == -d1));
			l.push_back(lst(s1 == -d2, s2 == d1));
		}
#endif
		for (vector<lst>::iterator sys = l.begin(); sys != l.end(); ++sys)
			if ((*sys = ex_to<lst>(lsolve(*sys, srcmoms))).nops() > 0)
				res.push(*sys);
	}
    return res;
}

// propagators are given in form of pair<momentum, squaredmass>
// shift of loop and crossing of ext. momenta is simultaneous trafo
bool find_shifts(//
		const vector<Propagator>& srcprops,//
		const lst& srcmoms, // src loop momenta (alt symbols)
		int srccuts,        // bit mask to mark cut propagators in srcprops
		const vector<Propagator>& destprops,//
		const lst& destmoms, // dest loop momenta
		int destcuts,        // bit mask to mark cut propagators in destprops
		list<lst>& shifts, bool stop_on_first_match) {
	stack<int> matched; // assigned dest propagators
	stack<lst> sysmatched; // eqs for assigned dest props
	stack<stack<lst> > altshifts; // shifts to try for specific propagator pair
	list<int> unmatched; // ordered list of unassigned dest propagators

	size_t t = srcprops.size();
	ASSERT(srcprops.size() == t && destprops.size() == t);
	for (size_t i = 0; i < t; ++i)
		unmatched.push_back(i);
	altshifts.push(stack<lst>());

	list<int>::iterator totry = unmatched.begin(); // next dest propagator to try
	while (totry != unmatched.end() || !matched.empty()) {
		if (totry != unmatched.end()) {
			// try to match one specific propagator
			if (altshifts.top().empty()) {
				// on first occurance, get the list of all possible shifts
				bool scut = (srccuts & (1 << (int) matched.size())) != 0;
				bool dcut = (destcuts & (1 << (int) *totry)) != 0;
				if (scut == dcut) { // don't mix cut with non-cut propagators
					const Propagator& sp = srcprops[matched.size()];
					const Propagator& dp = destprops[*totry];
					altshifts.top() = find_propagator_mappings(sp, dp, srcmoms);
				}
			}
			lst sys;
			if (!altshifts.top().empty()) { // try a possbible shift
				sys = (sysmatched.empty() ? lst() : sysmatched.top());
				sys = add_lst(sys, altshifts.top().top());
				altshifts.top().pop();
				if (matched.size() + 1 >= srcmoms.nops() || //
						unmatched.size() == 1 /* last prop to match ? */) {
					sys = ex_to<lst> (lsolve(sys, srcmoms));
				}
			}
			if (sys.nops() > 0) { // match succeeded ? then go one level deeper
				matched.push(*totry);
				unmatched.erase(totry);
				totry = unmatched.begin();
				sysmatched.push(sys);
				altshifts.push(stack<lst>());
			} else { // match failed, try next
				if (altshifts.top().empty())
					++totry;
			}
			if (unmatched.empty()) { // full match of all propagators ?
				ex det = abs(jacobi_determinant(sys, destmoms).expand());
				if (det == 1) {
					shifts.push_back(sys);
					if (stop_on_first_match)
						return true;
				}
			}
		} else if (!matched.empty()) {
			// next try at one level up, discard last matched
			int last = matched.top();
			matched.pop();
			sysmatched.pop();
			altshifts.pop();
			totry = lower_bound(unmatched.begin(), unmatched.end(), last);
			totry = unmatched.insert(totry, last);
			if (altshifts.top().empty())
				++totry;
		} else {
			// all tries done
			return false;
		}
	}
	return false; // never reached, suppresses compiler warning
}

// will put simplest shift to front
bool find_shifts(const Sector& src,//
		set<Sector>::const_iterator destbegin,//
		set<Sector>::const_iterator destend,//
		const list<Crossing>& crossings, /*crossings equivalent to identity*/
		list<pair<Sector, exmap> >& shifts, bool stop_on_first_match) {
	size_t t = src.t();
	int scuts, dcuts;
	vector<Propagator> srcprops = src.get_propagators(scuts);
	ASSERT(!crossings.empty()); // at least the identity
	ASSERT(srcprops.size() == t);
	// temp names for src loop momenta in case they coincide with those from ic
	const Kinematics* kin = Files::instance()->kinematics();
	const lst& extmoms = kin->independent_external_momenta();
	lst srcloopmoms = src.integralfamily()->loop_momenta();
	lst tmploopmoms, tmpextmoms; // temp symbols for srcloopmoms, extmoms
	exmap src2tmp, tmp2src;
	provide_alternative_symbols(srcloopmoms, tmploopmoms, src2tmp, tmp2src);
	provide_alternative_symbols(extmoms, tmpextmoms, src2tmp, tmp2src);

	for (set<Sector>::const_iterator dest = destbegin; dest != destend; ++dest) {
		//LOGX("checking " << src << " -> " << *dest);
		lst destloopmoms = dest->integralfamily()->loop_momenta();
		vector<Propagator> destprops = dest->get_propagators(dcuts);
		ASSERT(destprops.size() == t);
		if (srcloopmoms.nops() != destloopmoms.nops())
			continue; // different number of loop momenta

		// quick return if masses can't be matched (not necessary but useful)
		vector<ex> smass2(t), dmass2(t);
		for (size_t i = 0; i < t; ++i) {
			smass2[i] = srcprops[i].squaredmass();
			dmass2[i] = destprops[i].squaredmass();
		}
		sort(smass2.begin(), smass2.end(), ex_is_less());
		sort(dmass2.begin(), dmass2.end(), ex_is_less());
		if (!equal(smass2.begin(), smass2.end(), dmass2.begin()))
			continue; // different masses
		if (Sector::t_of_id(scuts) != Sector::t_of_id(dcuts))
			continue; // different number of cut propagators

		list<Crossing>::const_iterator c;
		for (c = crossings.begin(); c != crossings.end(); ++c) {
			//ASSERT(c->is_equivalent(Crossing(extmoms, invars))); // equiv to id
			ASSERT(c->is_equivalent_to_identity());
			lst eqns;
			exmap subst;
			for (lst::const_iterator p = extmoms.begin(); p != extmoms.end(); ++p) {
				ex lhs = p->subs(src2tmp);
				ex rhs = p->subs(c->rules_momenta());
				eqns.append(lhs == rhs);
				subst[lhs] = rhs;
			}
			vector<Propagator> sps(t), dps(t);
			try {
				for (size_t i = 0; i < t; ++i) {
					ex smom1 = srcprops[i].momentum1();
					ex smom2 = srcprops[i].momentum2();
					ex dmom1 = destprops[i].momentum1();
					ex dmom2 = destprops[i].momentum2();
					smom1 = smom1.subs(src2tmp).subs(subst);
					smom2 = smom2.subs(src2tmp).subs(subst);
					sps[i] = Propagator(smom1, smom2, srcprops[i].squaredmass());
					dps[i] = Propagator(dmom1, dmom2, destprops[i].squaredmass());
				}
			} catch (exception& e) {
				//WARNING("can't match non-standard propagators in " << src << " or " << *dest);
				break;
			}
			list<lst> matches;
			find_shifts(sps, tmploopmoms, scuts, dps, destloopmoms, dcuts,
					matches, stop_on_first_match);
			for (list<lst>::const_iterator m = matches.begin(); m
					!= matches.end(); ++m) {
				// note: shift might transform external momenta
				// undo with: Crossing invcross(kin, c->permutation().inverse());
				exmap mmap = equations_to_substitutions(add_lst(*m, eqns),
						tmp2src, true);
				// insert to front if better than previous best
				if (!shifts.empty() && !(*dest > shifts.front().first)//
						&& (*dest < shifts.front().first || LessExmap()(mmap,
								shifts.front().second)))
					shifts.push_front(make_pair(*dest, mmap));
				else
					shifts.push_back(make_pair(*dest, mmap));
			}
			if (stop_on_first_match && !matches.empty())
				return true;
		}
	}
	return false;
}

/*
 // NOTE: take care of this before using: may find non-minimal target sectors
 // will put simplest shift to front
 bool find_shifts_via_subsectors(const Sector& src,
 list<pair<Sector, exmap> >& shifts) {
 const IntegralFamily* ic = src.integralfamily();
 string name = ic->name();
 const SectorMappings* m = Files::instance()->sectormappings(name);
 set<int> sub = ic->get_immediate_subsector_equivalents(src.get_id());
 bool found = false;
 for (set<int>::const_iterator s = sub.begin(); s != sub.end(); ++s) {
 try {
 Sector sub(ic, *s);
 Sector subdest = SectorMappings::get_shifted_sector(sub);
 if (subdest == sub) // no shift
 continue;
 const IntegralFamily* destic = subdest.integralfamily();
 INT integral(ic, *s);
 exmap shift = m->find_shift_for_sector_relation(integral);
 ex integrand = INT(src).get_integrand().subs(shift);
 INT i(destic);
 ex c;
 //LOGX("testing shift " << sub << " -> " << subdest);
 if (destic->match_integrand(integrand, i, c) //
 && i.get_sector() < src) {
 // note: might be a sector which is reducible by a shift
 if (!shifts.empty() && shifts.front().first < i.get_sector())
 shifts.push_back(make_pair(i.get_equivalent().get_sector(), shift));
 else
 shifts.push_front(make_pair(i.get_equivalent().get_sector(), shift));
 found = true;
 }
 } catch (exception&) {
 }
 }
 return found;
 }
 */

bool is_trivial_shift(const IntegralFamily* fam, const exmap& shift) {
    // trivial trafo of momenta ?
	lst moms = fam->get_all_momenta();
	if (moms.subs(shift).is_equal(moms))
		return true;
	// trivial effect on propagators ?
	const vector<Propagator>& p = fam->propagators();
	for (size_t i = 0 ; i < p.size(); ++i) {
		const ex& sp = p[i].scalarproduct();
		if (!(sp - sp.subs(shift)).eval().is_zero())
			return false;
	}
	return true;
}

void prepare_selections(const UncrossedReadSectorSelection& srcsecs,
		set<Sector>& secs, set<string>& src_fams, list<Crossing>& cross,
		list<Crossing>& idcross, list<Crossing>& idequivcross) {
	// validate and post process the input
	LOG("Preparing selections");
	secs = srcsecs.find_sectors();
	set<const Kinematics*> src_kins;
	for (set<Sector>::iterator s = secs.begin(); s != secs.end();) {
		if (s->integralfamily()->is_crossed()) {
			LOG("ignoring sector " << *s << " from crossed integral family");
			secs.erase(s++);
		} else {
			src_fams.insert(s->integralfamily()->name());
			src_kins.insert(s->integralfamily()->kinematics());
			++s;
		}
	}
	for (set<const Kinematics*>::const_iterator sk = src_kins.begin();
			sk != src_kins.end(); ++sk) {
		(*sk)->print_info();
		Files::instance()->crossings((*sk)->name());
	}
	for (set<string>::const_iterator sf = src_fams.begin();	sf != src_fams.end(); ++sf)
		Files::instance()->integralfamily(*sf)->print_info();
	LOG("  found " << secs.size() << " source sectors");

	const Kinematics* kin = Files::instance()->kinematics();
	Crossing identity(kin);
	cross = Files::instance()->crossings(kin->name())->ordered_crossings();
	cross.push_front(identity);
	for (list<Crossing>::const_iterator c = cross.begin(); c != cross.end(); ++c)
		if (c->is_equivalent_to_identity())
			idequivcross.push_back(*c);
	idcross.push_front(identity);
}

map<int, set<Sector> > setup_src_by_t(const set<Sector>& secs) {
	LOG("\nFinding source sectors");
	map<int, set<Sector> > src_by_t;
	set<Sector>::const_iterator s;
	for (set<Sector>::iterator s = secs.begin(); s != secs.end(); ++s)
			src_by_t[s->t()].insert(*s);
	return src_by_t;
}

map<int, set<Sector> > setup_dest_by_t(const set<Sector>& secs,
		const list<Crossing>& cross, const list<Crossing>& idcross,
		const list<Crossing>& idequivcross) {
	LOG("\nFinding target sectors");
	map<int, set<Sector> > dest_by_t;
	set<Sector> dest;
	Files* files = Files::instance();
	list<IntegralFamily*> fams = files->integralfamilies();
	for (list<IntegralFamily*>::iterator f = fams.begin(); f != fams.end();
			++f) {
		if ((*f)->is_crossed())
			continue;
		set<int> ids;
		files->sectormappings((*f)->name())->find_shift_targets(ids);
		for (set<int>::const_iterator s = ids.begin(); s != ids.end(); ++s)
			dest.insert(Sector(*f, *s));
	}
	LOG("  found " << dest.size() << " target sectors (without crossings)");
	for (set<Sector>::iterator s = dest.begin(); s != dest.end(); ++s)
		for (list<Crossing>::const_iterator c = cross.begin(); c != cross.end(); ++c)
			dest_by_t[s->t()].insert(c->transform(*s));
	LOG("  using " << cross.size() << " crossings");
	LOG("  using " << idequivcross.size() << " crossings equivalent to identity");
	return dest_by_t;
}

void find_zero_sectors(set<Sector>& secs) {
	LOG("\nFinding zero sectors:");
	LOG("  analyzing " << secs.size() << " sectors");
	map<pair<int, string>, map<Sector, list<Sector> > > sec_by_fam;
	for (set<Sector>::iterator s = secs.begin(); s != secs.end(); ++s) {
		const IntegralFamily* f = s->integralfamily();
		sec_by_fam[make_pair(f->id(), f->name())][*s] = list<Sector>();
	}
	// Include missing zero sectors in the sector mappings files which have
	// been omitted by SectorSelection::find_sectors().
	// This might also include zero sectors which are not sectors of the
	// user-defined selection, but who cares?
	map<pair<int, string>, map<Sector, list<Sector> > >::iterator sf;
	for (sf = sec_by_fam.begin(); sf != sec_by_fam.end(); ++sf) {
		const string& n = sf->first.second; // name
		LOG("\n  integral family '" << n << "':");
		const IntegralFamily* f = Files::instance()->integralfamily(n);
		SectorMappings* m = Files::instance()->sectormappings(n);
		LOG("   add trivial zero sectors with t < " << f->loop_momenta().nops());
		LOGN("   finding zero sectors with t >= " << f->loop_momenta().nops() << ":");
		set<int> trivial_zeros = f->get_trivial_zero_sector_equivalents();
		m->insert_zero_sectors(trivial_zeros);
		bool allow_longtest = f->kinematics()->external_momenta().nops() != 0;
		m->find_zero_sectors(sf->second, allow_longtest);
		Files::instance()->save(*m);
	}
	set<Sector>::const_iterator s;
	for (set<Sector>::iterator s = secs.begin(); s != secs.end();)
		if (SectorMappings::is_zero(*s))
			secs.erase(s++);
		else
			++s;
}

void save_mappings(const set<string>& src_fams) {
	set<string>::const_iterator name = src_fams.begin();
	for (; name != src_fams.end(); ++name) {
		LOG("Saving updated sector mappings for " << *name);
		const SectorMappings * m = Files::instance()->sectormappings(*name);
		Files::instance()->save(*m);
	}
}

void find_sector_relations(map<int, set<Sector> >& src_by_t,
		map<int, set<Sector> >& dest_by_t, const list<Crossing>& cross,
		const list<Crossing>& idcross, const set<string>& src_fams) {
	LOG("\nFinding sector relations:");
	map<int, set<Sector> >::iterator sbt;
	for (sbt = src_by_t.begin(); sbt != src_by_t.end(); ++sbt) {
		int t = sbt->first;
		set<Sector>& src_secs = sbt->second;
		set<Sector>& dest_secs = dest_by_t[t];
		LOG("\nt = " << t);
		set<Sector>::iterator src, dest;
		for (src = src_secs.begin(); src != src_secs.end();) {
			LOGN("  " << *src << " ");
			string name = src->integralfamily()->name();
			SectorMappings* m = Files::instance()->sectormappings(name);
			list<pair<Sector, exmap> > shifts;
			find_shifts(*src, // from sec
						dest_secs.begin(), dest_secs.lower_bound(*src), // to secs
						idcross, shifts, true);
			if (!shifts.empty()) { // found a shift to lower sector for src
				const Sector& dest = shifts.front().first;
				const exmap& shift = shifts.front().second;
				LOG("shifted to " << dest << "  using " << shift);
				m->insert_shift(src->id(), dest, shift);
				// remove from dest_by_t and src_by_t
				for (list<Crossing>::const_iterator c = cross.begin();
						c != cross.end(); ++c)
					dest_secs.erase(c->transform(*src));
				src_secs.erase(src++);
			} else { // no shift to lower sector for src
				LOG("no shift");
				++src;
			}
		} // source sector loop
		save_mappings(src_fams);
		LOG("");
	} // t loop

	LOGX("Found following shift target sectors:");
	for (sbt = src_by_t.begin(); sbt != src_by_t.end(); ++sbt) {
		LOGX("  t=" << sbt->first << ":" << sbt->second.size() << " sectors");
		set<Sector>::const_iterator s;
		for (s = sbt->second.begin(); s != sbt->second.end(); ++s)
			LOGX("    " << *s);
	}
}

void find_sector_symmetries(map<int, set<Sector> >& src_by_t,
		const list<Crossing>& idequivcross, const set<string>& src_fams) {
	LOG("\nFinding sector symmetries:");
	map<int, set<Sector> >::iterator sbt;
	for (sbt = src_by_t.begin(); sbt != src_by_t.end(); ++sbt) {
		LOG("\nt = " << sbt->first);
		set<Sector>::iterator src;
		for (src = sbt->second.begin(); src != sbt->second.end(); ++src) {
			set<Sector> d;
			d.insert(*src);
			list<pair<Sector, exmap> > shifts;
			find_shifts(*src, d.begin(), d.end(), idequivcross, shifts, false);
			list<pair<Sector, exmap> >::const_iterator symm;
			list<exmap> rawshifts;
			for (symm = shifts.begin(); symm != shifts.end(); ++symm) {
				ASSERT(symm->first == *src);
				const exmap& shift = symm->second;
				if (!is_trivial_shift(src->integralfamily(), shift))
					rawshifts.push_back(shift);
			}
			LOG(*src << ": " << rawshifts.size() << " shifts onto itself");
			if (!rawshifts.empty()) {
				string name = src->integralfamily()->name();
				SectorMappings* m = Files::instance()->sectormappings(name);
				m->insert_symmetry_shifts(src->id(), rawshifts);
			}
		} // source sector loop
		save_mappings(src_fams);
		LOG("");
	} // t loop
}

void find_crossed_sector_relations(map<int, set<Sector> >& src_by_t,
		const list<Crossing>& cross, const list<Crossing>& idcross,
		const set<string>& src_fams) {
	LOG("\nFinding crossed sector relations:");
	map<int, set<Sector> >::iterator sbt;
	for (sbt = src_by_t.begin(); sbt != src_by_t.end(); ++sbt) {
		LOG("\nt = " << sbt->first);
		set<Sector>::iterator src;
		for (src = sbt->second.begin(); src != sbt->second.end(); ++src) {
			set<Sector> xsrc, d;
			for (list<Crossing>::const_iterator c = cross.begin();
					c != cross.end(); ++c) {
				d.insert(c->transform(*src));
				Sector s = Crossing::to_minimal_crossing(c->transform(*src));
				// add only if no shift present already
				string name = s.integralfamily()->name();
				SectorMappings* m = Files::instance()->sectormappings(name);
				if (m->get_shifted_sector(s) == s)
					xsrc.insert(s);
			}
			set<Sector>::const_iterator xs = xsrc.begin();
			for (++xs; xs != xsrc.end(); ++xs) {
				LOGN("  " << *xs << " ");
				list<pair<Sector, exmap> > shifts;
				find_shifts(*xs, d.begin(), d.lower_bound(*xs),
							idcross, shifts, true);
				if (!shifts.empty()) { // found a shift to lower sector for xs
					const Sector& dest = shifts.front().first;
					const exmap& shift = shifts.front().second;
					LOG("shifted to " << dest << "  using " << shift);
					string name = xs->integralfamily()->name();
					SectorMappings* m = Files::instance()->sectormappings(name);
					m->insert_shift(xs->id(), dest, shift);
				} else { // no shift to lower sector for xs
					LOG("no shift");
				}
			}
		} // source sector loop
		save_mappings(src_fams);
		LOG("");
	} // t loop
}

void SetupSectorMappingsAlt::run_serial() {
	set<string> src_fams;
	set<Sector> secs;
	list<Crossing> cross, idcross, idequivcross;
	prepare_selections(source_sectors_, secs, src_fams, cross, idcross, idequivcross);

	if (find_zero_sectors_)
		find_zero_sectors(secs);

	map<int, set<Sector> > src_by_t = setup_src_by_t(secs);
	map<int, set<Sector> > dest_by_t = setup_dest_by_t(secs,cross,idcross,idequivcross);

	if (find_sector_relations_)
		find_sector_relations(src_by_t, dest_by_t, cross, idcross, src_fams);

	if (find_sector_symmetries_)
		find_sector_symmetries(src_by_t, idequivcross, src_fams);

	if (find_crossed_sector_relations_)
		find_crossed_sector_relations(src_by_t, cross, idcross, src_fams);

}

} // namespace Reduze

