/*  job_selectreductions.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_selectreductions.h"

#include "functions.h"
#include "ginacutils.h"
#include "equationlist.h"
#include "files.h"
#include "filedata.h"
#include "kinematics.h"
#include "sectormappings.h"
#include "identitygenerator.h"

using namespace std;

namespace Reduze {

// register job type at JobFactory
namespace {
JobProxy<SelectReductions> dummy;
}

// SelectReductions


set<INT> find_selected_integrals(const string& input_filename,
		bool generate_crossed) {
	// read a file with integrals
	LOG("Reading integrals from file \"" << input_filename << '\"');
	set<INT> intset;
	InFileINTs in(input_filename);
	in.get_all(intset);
	in.close();
	LOG("Solutions for " << intset.size() << " different integrals requested");
	const Kinematics* kin = Files::instance()->kinematics();
	if (generate_crossed) {
		LOG("Generating crossed integrals");
		list<Crossing> cl =
				Files::instance()->crossings(kin->name())->ordered_crossings();
		set<INT> insert;
		for (set<INT>::const_iterator i = intset.begin(); i != intset.end(); ++i) {
			INT uncrossed = Crossing::uncross(*i).first;
			insert.insert(uncrossed);
			for (list<Crossing>::const_iterator c = cl.begin(); c != cl.end(); ++c)
				insert.insert(c->transform(uncrossed));
		}
		intset.insert(insert.begin(), insert.end());

		LOG("Solutions for " << intset.size() << " integrals remain");
	}
	return intset;
}

// write out shift identities (only real shifts) and add rhs integrals to 'intset'
// set foundsome to true if identities were written (not set otherwise)
void write_shift_identities(set<INT>& intset, OutFileEquations& of,//
		const map<Crossing, Crossing>& crossmap, bool& foundsome)//
{
	for (set<INT>::iterator i = intset.begin(); i != intset.end();) {
		LOGNX("Search shift for " << *i);
		Identity eq = IdentityGenerator::get_shift_identity(*i, true, false);
		if (eq.empty()) { // no shift available
			++i;
			LOGX(" => none");
			continue;
		}
		LOGX(" => found shift, select red for rhs");
		foundsome = true;
		of << eq;
		eq.find_subleading_INT(intset); // select rhs INTs (no shifts for them)
		intset.erase(i++);
	}
}

// find equivalent integrals (permutation sym. zero) and sort by sector
// ints_by_sec: sector -> equiv. -> selected
void sort_selected_integrals(const set<INT>& intset, map<Sector, multimap<INT,
		INT> >& ints_by_sec, bool& needreduction) {
	for (set<INT>::const_iterator i = intset.begin(); i != intset.end(); ++i) {
		INT integral = Crossing::uncross(*i).first;
		if (integral != *i)
			LOGX("Construct solution for: " << *i << " from result of " << integral << " via crossing");
		needreduction = true;
		INT equiv = SectorMappings::get_equivalent_or_unique_zero(integral);
		if (integral == equiv)
			LOGX("Selecting: " << integral);
		else
			LOGX("Selecting: " << integral << " equivalent to " << equiv);
		ints_by_sec[equiv.get_sector()].insert(make_pair(equiv, *i));
	}
	int num_sectors = ints_by_sec.size();
	LOG("Need reductions for " << num_sectors << " different sectors");
}

void SelectReductions::run_serial() {
	INT::load_preferred(preferred_masters_file_);
	LOG("Finding selected integrals");
	set<INT> intset;
	intset = find_selected_integrals(input_filename_, generate_crossed_);
	LOG("Finding crossing equivalences");
	const Kinematics* kin = Files::instance()->kinematics();
	const map<Crossing, Crossing>& crossmap = Files::instance()->crossings(
			kin->name())->equivalent_crossings();

	LOG("Opening output file '" + output_filename_ + "'");
	/// \todo: reactivate write_header(of);
	OutFileEquations of(output_filename_);
	LOG("Writing shift identities");
	bool needreduction = false;
	map<Sector, multimap<INT, INT> > ints_by_sec; // sector -> equiv. -> selected
	write_shift_identities(intset, of, crossmap, needreduction);
	sort_selected_integrals(intset, ints_by_sec, needreduction);

	// scan reduction files and write out identities

	map<Sector, multimap<INT, INT> >::iterator s;
	set<INT> notfound;
	for (s = ints_by_sec.begin(); s != ints_by_sec.end(); ++s) {
		LOG("Searching integrals from sector " << s->first);
		string fn = Files::instance()->get_filename_sectorreduction(s->first);
		if (SectorMappings::is_zero(s->first)) {
		} else if (!is_readable_file(fn)) {
			LOGX("File " << fn << " is not readable");
		} else {
			LOGX("Reading " << fn << std::flush);
			InFileEquations in(fn);
			EquationHLight eq;
			Timer timer_read, timer_search;
			while (timer_read.unpause(), timer_search.pause(), in.get(eq)) {
				timer_read.pause();
				timer_search.unpause();
				INT mi = eq.max_INT();
				pair<multimap<INT, INT>::iterator, multimap<INT, INT>::iterator>
						start_end = s->second.equal_range(mi);
				if (start_end.first == start_end.second)
					continue;
				LOGX(mi << ": found reduction");
				// write all requested results which are equivalent to one integral
				multimap<INT, INT>::iterator it;
				for (it = start_end.first; it != start_end.second; ++it) {
					Identity outeq(eq); // convert to algebraic coefficients
					if (it->first == it->second) {
						// same integral
						outeq.remove_zero_integral();
						of << outeq;
					} else if (it->first.integralfamily()
							== it->second.integralfamily()) {
						// different (but equivalent) integrals, same integral family
						outeq.insert(it->second, 1, false); // selected by user
						outeq.insert(it->first, -1, false); // reduction for this
						outeq.remove_zero_integral();
						of << outeq;
					} else {
						// different integral families (crossing)
						pair<INT, Crossing> cp = Crossing::uncross(it->second);
						map<Crossing, Crossing>::const_iterator c =
								crossmap.find(cp.second);
						ASSERT(c != crossmap.end());
						outeq = c->second.transform(outeq);
						outeq.insert(c->second.transform(cp.first), -1, false);
						outeq.insert(it->second, 1, false);
						outeq.remove_zero_integral();
						of << outeq;
						// simplify crossed subsectors
						set<INT> sints;
						outeq.find_subleading_INT(sints);
						set<INT>::const_iterator si;
						for (si = sints.begin(); si != sints.end(); ++si) {
						    needreduction = true;
							of << IdentityGenerator::get_shift_identity(*si,
									true, false);
						}
					}
				}
				s->second.erase(start_end.first, start_end.second);
			}
			LOG("  time to read reductions:   " << timer_read.get_wall_time() << " s");
			LOG("  time to search reductions: " << timer_search.get_wall_time() << " s");
		}
		// write reductions based on permutations or simpler crossings only,
		// remember if not found at all
		multimap<INT, INT>::const_iterator i;
		for (i = s->second.begin(); i != s->second.end(); ++i) {
			const INT& selected = i->second;
			const INT& equiv = i->first;
			if (selected == equiv) {
				LOGX(selected << ": no reduction");
				notfound.insert(selected);
			} else if (selected.integralfamily() == equiv.integralfamily()) {
				LOGX(selected << ": found permutation equivalence");
				Identity outeq;
				outeq.insert(equiv, -1, false);
				outeq.remove_zero_integral();
				outeq.insert(selected, 1, false);
				of << outeq;
			} else { // is crossed, try to find simpler crossing
				// crossing
				Crossing c(Crossing::uncross(selected).second);
				// get simpler crossing
				map<Crossing, Crossing>::const_iterator crossing =
						crossmap.find(c);
				ASSERT(crossing != crossmap.end());
				INT simplest = crossing->second.transform(equiv);
				if (simplest != selected) {
					LOGX(selected << ": found simpler crossing");
					Identity outeq;
					outeq.insert(simplest, -1, false);
					outeq.remove_zero_integral();
					outeq.insert(selected, 1, false);
					of << outeq;
				} else {
					LOGX(selected << ": no reduction");
					notfound.insert(selected);
				}
			}
		}
	} // sector loop
	of.finalize();
	LOG("Output written to file \"" << output_filename_ << '\"');

	if (notfound.size() > 0) {
		LOG("Couldn't find solutions for " << notfound.size() << " (selected) integral(s)");
		if (!notfound_file_.empty()) {
			LOG("  writing these integrals to file '" + notfound_file_ << "'");
			OutFileINTs out(notfound_file_);
			for (set<INT>::const_iterator i = notfound.begin(); i
					!= notfound.end(); ++i)
				out << *i;
			out.finalize();
		}
	} else {
		LOG("Found solutions for all integrals");
	}
	if (needreduction) {
		LOG("\nSome solutions were constructed on the fly via a shift symmetry");
		LOG("or a crossing. It is recommended to run the job reduce_files on");
		LOG("the output file to ensure fully reduced solutions.");
	}
}

bool SelectReductions::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	find_dependencies_reductions(outothers, in);
	in.push_back(input_filename_);
	out.push_back(output_filename_);
	return true;
}

void SelectReductions::init() {
	if (input_filename_.empty() || output_filename_.empty())
		throw runtime_error("Input or output file name empty");
	if (input_filename_ == output_filename_)
		throw runtime_error("Input and output file names are identical");
}

std::string SelectReductions::get_description() const {
	return string("select reductions: ") + short_filename(input_filename_);
}

}
