/*  job_normalize.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_NORMALIZE_H_
#define JOB_NORMALIZE_H_

#include "job.h"
#include "yamlconfigurable.h"

namespace Reduze {

/// Job to simplify linear combinations and equations
class Normalize: public Reduze::Job {
public:
	/// \todo: parallelize it
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("normalize");
		s.set_short_description("Simplifies linear combinations and equations.");
		s.set_long_description(""//
					"Job to simplify linear combinations and equations."
					" Coefficients of integrals may be normalized, permutations"
					" and shift symmetries may be employed to replace integrals"
					" (e.g. useful if permutations or shifts have been modified"
					" since production of the data).");
		s.add_option("input_file", true, "string", ""//
					"Input file name. File must contain a list of"
					" linear combinations or equations in 'reduze' format.");
		s.add_option("output_file", true, "string", ""//
					"Output file name. File is written in 'reduze' format.");
		s.add_option("normalize_coefficients", false, "boolean",
				"Whether to normalize coefficients.");
		s.add_option("factor_coefficients", false, "boolean",
				"Whether to factor coefficients. If enabled, the setting of"
				" 'normalize_coefficients' is ignored and a (more expensive)"
				" full factorization of the coefficients is performed."
				" Please note that GiNaC's factor() will be employed even if"
				" Reduze is configured to use Fermat.");
		s.add_option("apply_permutations", false, "boolean",
				"Whether to apply permutation symmetries and remove zero integrals.");
		s.add_option("apply_shifts", false, "boolean",
				"Whether to apply shift symmetries.");
		s.add_option("preferred_masters_file", false, "string",
				"Input file with preferred master integrals.");
		s.add_option("dimension_shift", false, "integer",
				"Shift dimension by the value specified.");
		s.add_option("map_on_coeff", false, "" //
					 "sequence of 2-element sequences", "" //
					 "Substitutions to be performed, e.g. [ [\"s\", \"-5\"] ]."
					 " This substitution is applied after other transformations.");
		//s.add_option("solve", false, "boolean",
		//		"Whether to solve equations for leading integral.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	Normalize() :
		normalize_coefficients_(true), factor_coefficients_(false),
		apply_permutations_(true), apply_shifts_(true),
		dimension_shift_(0) {
		add_auto_options();
	}
	virtual ~Normalize() {
	}

	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&,//
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;
	virtual void print_manual_options(YAML::Emitter& os) const;
	virtual void read_manual_options(const YAML::Node&);

protected:
	virtual void add_auto_options() {
		add_auto_io("input_file", input_filename_);
		add_auto_io("output_file", output_filename_);
		add_auto_io("normalize_coefficients", normalize_coefficients_);
		add_auto_io("factor_coefficients", factor_coefficients_);
		add_auto_io("apply_permutations", apply_permutations_);
		add_auto_io("apply_shifts", apply_shifts_);
		add_auto_io("preferred_masters_file", preferred_masters_file_);
		add_auto_io("dimension_shift", dimension_shift_);
	}
	virtual void init();

private:
	std::string input_filename_, output_filename_, preferred_masters_file_;
	bool normalize_coefficients_, factor_coefficients_;
	bool apply_permutations_, apply_shifts_;
	int dimension_shift_;
	GiNaC::exmap map_on_coeff_;
};

}

#endif /* JOB_NORMALIZE_H_ */
