/*  job.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job.h"

#include "integralfamily.h"
#include "files.h"
#include "sectormappings.h"
#include "functions.h"
#include "graph.h"
#include "amplitude.h"
#include "streamutils.h"
#include "yamlutils.h"

#ifdef HAVE_MPI
#include "jobcenter.h"
#endif

using namespace std;

namespace Reduze {

// Job

void Job::run_serial() {
	ABORT("No serial implementation available for:" << std::endl
			<< to_short_string() << std::endl);
}

#ifdef HAVE_MPI
void Job::run_manager(MPI::Intracomm*, int) {
	ABORT("No parallel implementation available for:" << std::endl
			<< to_short_string() << std::endl);
}

void Job::run_worker(MPI::Intracomm*, int) {
	ABORT("No parallel implementation available for:" << std::endl
			<< to_short_string() << std::endl);
}

int Job::manager_rank() const {
	return manager_rank_;
}
#endif // HAVE_MPI

int Job::id() const {
	return job_id;
}

void Job::set_id(unsigned id) {
	job_id = id;
}

#ifdef HAVE_MPI
void Job::set_communication_params(MPI::Intracomm* comm, int jobcenter_rank, int manager_rank) {
	communicator_ = comm;
	jobcenter_rank_ = jobcenter_rank;
	manager_rank_ = manager_rank;
}
#endif //HAVE_MPI

bool Job::is_conditional() const {
	return conditional_;
}

void Job::set_conditional(bool flag) {
	conditional_ = flag;
}

std::string Job::to_short_string(unsigned max_id) const {
	stringstream txt;
	int nmax = number_of_digits(max_id);
	int n = number_of_digits(job_id);
	string spaces(max(0, nmax - n), ' ');
	txt << "job(" << job_id << "): " << spaces << get_description();
	return txt.str();
}

// class DistributedJob

void DistributedJob::set_num_workers_range(unsigned min, unsigned max) {
	ASSERT(min <= max);
	min_workers_ = min;
	max_workers_ = max;
}

int DistributedJob::min_workers() const {
	return min_workers_;
}

int DistributedJob::max_workers() const {
	return max_workers_;
}

// global functions

bool find_dependencies_all_sectormappings(const set<string>& outothers,
		list<string>& in, list<Job*>& aux_jobs) {
	bool found_in_outothers = false;
	list<string> all = Files::instance()->get_filenames_all_sectormappings();
	for (list<string>::iterator f = all.begin(); f != all.end(); ++f) {
		string fn = *f;
		bool p = outothers.find(get_canonical_filename(fn)) != outothers.end();
		found_in_outothers |= p;
		if (p || is_readable_file(fn))
			in.push_back(fn);
	}
	return !found_in_outothers;
}

void find_dependencies_reductions(const std::set<std::string>& outothers,
		std::list<std::string>& in) {
	/// \todo: should we skip non-sector reduction files in reduction_directory() ?
	string reddir = Files::instance()->get_reductions_directory();
	set<string>::const_iterator f;
	for (f = outothers.begin(); f != outothers.end(); ++f) {
		if (get_directory_of_filename(*f) == reddir) {
			in.push_back(*f);
		}
	}
}

/*
 std::string Job::to_string_dot_node() const {
 std::stringstream s;
 s << "Job[" << get_id() << "]: [";
 if (get_status() == Pending) {
 s << "p";
 } else if (get_status() == Running) {
 s << "r";
 } else if (get_status() == Done) {
 s << "c";
 } else if (get_status() == Aborted) {
 s << "a";
 }
 s << "] " << rs_range.to_string();
 return s.str();
 }
 */

} // namespace Reduze

