/*  functions.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef REDUZE_FUNCTIONS_H
#define REDUZE_FUNCTIONS_H

#include <sstream>
#include <list>
#include <set>
#include <string>
#include <iostream>
#include <vector>
#include <map>
#include <stdint.h> // defines int8_t, uint32_t
#include "streamutils.h"

namespace YAML {
class Emitter;
class Node;
}

namespace Reduze {

/// alias for (8 bit) signed integer used for propagator exponents
typedef int8_t int8;

// GENERAL FUNCTIONS

/// split string into list of strings with maximal length
/** string is split at empty space character, single words
 ** with length > maxlen is are not divided, returned list
 ** contains at least one element */
std::list<std::string> split_string(const std::string& full, size_t maxlen);

/// chops off leading and trailing whitespaces
std::string chop_whitespaces(const std::string& s);

/// replaces characters contained in 'tokens' by 'replacewith'
std::string replace_chars(const std::string& s, const std::string& tokens,
		char replacewith);

/// returns the values of a map
template<class VAL>
std::set<VAL> get_map_values(const std::map<int, VAL>& amap) {
	std::set<VAL> result;
	typename std::map<int, VAL>::const_iterator m;
	for (m = amap.begin(); m != amap.end(); ++m)
		result.insert(m->second);
	return result;
}

/// return a map where values are mapped to consecutive integers starting with 0
template<class VAL>
std::map<VAL, int> map_values_to_base_0(const std::set<VAL>& values) {
	std::map<VAL, int> result;
	int lauf = 0;
	typename std::set<VAL>::const_iterator s;
	for (s = values.begin(); s != values.end(); ++s)
		result[*s] = lauf++;
	return result;
}

/// return a map where the values are mapped to consecutive integers starting with 0
template<class VAL>
std::map<int, int> map_values_to_base_0(const std::map<int, VAL>& amap) {
	std::set<VAL> values = get_map_values(amap);
	std::map<VAL, int> convert = map_values_to_base_0(values);
	std::map<int, int> result;
	typename std::map<int, VAL>::const_iterator m;
	for (m = amap.begin(); m != amap.end(); ++m)
		result[m->first] = convert.at(m->second);
	return result;
}

/// computes the a^b
int power_int(int a, int b);
unsigned long power_ulong(int a, int b);

unsigned long factorial_ulong(int n);

/** returns 0 if k < 0 or k > n */
/// computes the binomial coefficient n choose k
unsigned long binom_int(int n, int k);

unsigned long stirling2_ulong(int n, int k);

// n > 0, k > 0, r > 0: number of partitions into k subsets with each subset at least size r
// case r = 0:
// returns 1 for n = k = 0 otherwise returns 0
unsigned long associated_stirling_2(int n, int k, int r);

/// returns the number of digits of the number 'n'
int number_of_digits(unsigned long n);

/// removes all characters except for [a-zA-Z0-9] from the string
std::string to_safe_variable_name(const std::string&);

/// a unique run identification string
class RunIdentification {
public:
	static RunIdentification* instance();
	void set_number(long long id);
	long long number() const;
	const std::string& string() const;
	const std::string& long_string() const;
private:
	RunIdentification();
	long long number_;
	std::string string_, long_string_;
};

template<class T>
std::string to_string(const T& t) {
	std::stringstream ss;
	ss << t;
	return ss.str();
}

/// converts a string to an uintx, aborts if not successful
//uintx to_uintx(const std::string & str);

/// return a string containing the time
std::string time_string();

/// converts a time_t in the format hour:min:sec
std::string convert_seconds(time_t difftime);

/// add the number 'i' to each entry in the vector
void add_to_each_element(std::vector<int>& v, int i);
/// add the number 'i' to each entry in the vectors of the vector
void add_to_each_element(std::vector<std::vector<int> >& v, int i);

/// returns a Reduze welcome message
std::string welcome();

template<class T>
typename T::value_type front_pop(T& fifo) {
	typename T::value_type item = fifo.front();
	fifo.pop();
	return item;
}

// logging facilities

/// logging stream
/** uses a null buffer per default (throws away everything like /dev/null),
 ** set_* methods defined in this class assume ownership over rdbuf() **/
class LogStream: public std::ostream {
public:
	LogStream(int prefix_details = 1);
	virtual ~LogStream();
	/// prints some prefix info, e.g. rank of process ifdef HAVE_MPI
	void print_prefix();
	/// prints the date to the stream
	//void print_date();

	/// log to a file
	void set_file(const std::string& filename);
	/// log to std::cout
	void set_stdout();
	/// log to std::cerr
	void set_stderr();
	/// log to nothing
	void set_null();
	/// returns the filename of the opened file, empty string if none opened
	//std::string get_filename() const;
private:
	NullBuffer* nullbuffer;
	//std::string filename;
	std::ofstream* file;
	//unsigned long message_counter;
	//int details;
};

/// logging output stream for essential information
extern LogStream rlog1;
/// logging output stream for detailed information (used by ReduzeMonitor)
/** only one (root) process should report to this **/
extern LogStream rlog2;
/// logging output stream for very detailed information
extern LogStream rlog3;

/// aborts the program (wrapper for appropriate MPI or non-MPI abort)
void r_abort(int i);

// logging and error macros
// (macros, though evil, allow convenient auto-endl and context info support,
//  __PRETTY_FUNCTION__ gives full C++ description of function name,
//  __func__ only a short name)

// if DEBUG is #defined, context information is printed on rlogxx stream

#define LOG(str) (LOGX(str), Reduze::rlog1.print_prefix(), Reduze::rlog1 << str << std::endl)

#define LOGN(str) (LOGNX(str), Reduze::rlog1.print_prefix(), Reduze::rlog1 << str, Reduze::rlog1.flush())

#define LOGNX(str) (LOGXX(str), Reduze::rlog2.print_prefix(), Reduze::rlog2 << str, Reduze::rlog2.flush())

#define LOGX(str) (LOGXX(str), Reduze::rlog2.print_prefix(), Reduze::rlog2 << str << std::endl)

#ifndef DEBUG
#define LOGXX(str) ((void)0)
#define LOGNXX(str) ((void)0)
#else
//#define LOGXX(str) (Reduze::rlog3.print_prefix(), Reduze::rlog3 << str << std::endl)
#define LOGXX(str) (Reduze::rlog3.print_prefix(), \
		/*Reduze::rlog3 << __FILE__ << ":" << __LINE__ << ":" << __PRETTY_FUNCTION__ << ":"  << std::endl,*/ \
		Reduze::rlog3 << str << std::endl)
#define LOGNXX(str) (Reduze::rlog3.print_prefix(), \
		/*Reduze::rlog3 << __FILE__ << ":" << __LINE__ << ":" << __PRETTY_FUNCTION__ << ":"  << std::endl,*/ \
		Reduze::rlog3 << str)
#endif

class missing_file_error: public std::runtime_error {
public:
	explicit missing_file_error(const std::string& s) :
		std::runtime_error(s) {
	}
};
class missing_key_error: public std::runtime_error {
public:
	explicit missing_key_error(const std::string& s) :
		std::runtime_error(s) {
	}
};

/// a multiline verbose 'error' stamp
extern const char* error_asciiart;

class KnownError {
};

/// prints brief error message (no context info) and aborts
/** use this to abort if it's a "known" error and the user is going to
 ** understand the message */
#define ERROR(str) ( \
   (std::cerr << "\n" << error_asciiart << str << "\n" << std::endl), \
   LOGX("\n" << error_asciiart << str << "\n" << std::endl), \
   throw KnownError() \
)
//Reduze::r_abort(3)

class UnknownError {
};

/// prints error message with context info and throws UnrecoverableError
/** use this to abort if context information is important for possible bug
 ** tracking */
#define ABORT(str) ( \
   (std::cerr << "\n" << error_asciiart << __FILE__ << ":" << __LINE__ << ":" \
             << __PRETTY_FUNCTION__ << ":\n" << str << std::endl), \
   LOGX(std::endl << "\n" << error_asciiart << __FILE__ << ":" << __LINE__ << ":" \
                  << __PRETTY_FUNCTION__ << ":\n" << str << std::endl), \
   throw UnknownError() \
)
//throw UnrecoverableError()

// write everything to cerr (leaves have no log file)
#define WARNING(str) ( \
  std::cerr << std::endl << "WARNING: " << str << '\n' << std::endl, \
  LOGX(std::endl << "WARNING: " << str << '\n' << std::endl) \
)

/// DEBUG only: macro to assert an expression is true, aborts on failure
#define ASSERT(expr) \
   if (!(expr)) { \
	   ABORT("Assertion " << #expr << " failed"); \
   }
//	std::cerr << #expr << std::endl;
//    throw std::runtime_error("assertion failed");
//   }

/// macro to test also in production code an expression is true, aborts on failure
#define VERIFY(expr) \
   if (!(expr)) { \
	   ABORT("Verification " << #expr << " failed"); \
   }

// timers

/// timer to measure elapsed calender and CPU time
class Timer {
public:
	Timer() :
		wall_start_(0.), wall_stop_(0.), cpu_start_(0.), cpu_stop_(0.) {
		restart();
	}
	double restart();
	void unpause();
	void pause();
	/// wall seconds from timer start until now or until stop if stopped
	double get_wall_time() const;
	/// CPU seconds from timer start until now or until stop if stopped
	double get_cpu_time() const;
	// wall as a nice string
	std::string get_wall_time_nice_string() const;
	// string from double with tenth second precision, e.g. (Time: 22.1 s)
	static std::string print_nice_string(double d);

private:
	double get_absolute_wall_time() const;
	double get_absolute_cpu_time() const;
	double wall_start_, wall_stop_;
	double cpu_start_, cpu_stop_;
};

/// timer for a count down
class CountdownTimer: public Timer {
public:
	CountdownTimer(double duration) :
		duration(duration) {
	}
	bool is_elapsed() const {
		return duration > 0 && this->get_wall_time() > duration;
	}
private:
	double duration;
};

/// progress bar with ASCII output
class ProgressBar {
public:
	ProgressBar();
	ProgressBar(long offset, const std::string& titel, long max = 1);

	/// print title and reference bar
	void start();
	/// print newline
	void end(const std::string& message = "") const;

	/// use max_ and counter_ which increases by one
	void print();
	/// use max_ but a count in [0, max_)
	void print(long count);
	/// use count in [0, max), useful for slowly varying max
	void print(long count, long max);
	/// reset the max
	void set_max(long max);
	/// reset the max
	void set_title(const std::string& title);

private:

	long width_;
	std::string dot_;
	long printed_dots_;
	std::string titel_;
	long counter_;
	long max_;
	long offset_;
	bool first_;
};

// a class for a triple of object of different types (analogous to std::pair)
/*
template<class T1, class T2, class T3>
struct triple {

	triple() :
			first(), second(), third() {
	}

	triple(T1 a, T2 b, T3 c) :
			first(a), second(b), third(c) {
	}

	template<class U1, class U2, class U3>
	triple(const triple<U1, U2, U3>& other) :
			first(other.first), second(other.second), third(other.third) {
	}

	T1 first;
	T2 second;
	T3 third;

};

template<class T1, class T2, class T3>
inline bool operator==(const triple<T1, T2, T3>& a,
		const triple<T1, T2, T3>& b) {
	return a.first == b.first && a.second == b.second && a.third == b.third;
}

template<class T1, class T2, class T3>
inline bool operator<(const triple<T1, T2, T3>& a,
		const triple<T1, T2, T3>& b) {
	return a.first < b.first || (!(b.first < a.first) && a.second < b.second)
			|| (!(b.first < a.first) && (!(b.second < a.second))
					&& a.third < b.third);
}

template<class T1, class T2, class T3>
inline bool operator!=(const triple<T1, T2, T3>& a,
		const triple<T1, T2, T3>& b) {
	return !(a == b);
}

template<class T1, class T2, class T3>
inline bool operator>(const triple<T1, T2, T3>& a,
		const triple<T1, T2, T3>& b) {
	return b < a;
}

template<class T1, class T2, class T3>
inline bool operator<=(const triple<T1, T2, T3>& a,
		const triple<T1, T2, T3>& b) {
	return !(b < a);
}

template<class T1, class T2, class T3>
inline bool operator>=(const triple<T1, T2, T3>& a,
		const triple<T1, T2, T3>& b) {
	return !(a < b);
}

template<class T1, class T2, class T3>
inline triple<T1, T2, T3> make_triple(T1 a, T2 b, T3 c) {
	return triple<T1, T2, T3>(a, b, c);
}
*/

} // namespace Reduze

#endif
