/*
*         OpenPBS (Portable Batch System) v2.3 Software License
*
* Copyright (c) 1999-2000 Veridian Information Solutions, Inc.
* All rights reserved.
*
* ---------------------------------------------------------------------------
* For a license to use or redistribute the OpenPBS software under conditions
* other than those described below, or to purchase support for this software,
* please contact Veridian Systems, PBS Products Department ("Licensor") at:
*
*    www.OpenPBS.org  +1 650 967-4675                  sales@OpenPBS.org
*                        877 902-4PBS (US toll-free)
* ---------------------------------------------------------------------------
*
* This license covers use of the OpenPBS v2.3 software (the "Software") at
* your site or location, and, for certain users, redistribution of the
* Software to other sites and locations.  Use and redistribution of
* OpenPBS v2.3 in source and binary forms, with or without modification,
* are permitted provided that all of the following conditions are met.
* After December 31, 2001, only conditions 3-6 must be met:
*
* 1. Commercial and/or non-commercial use of the Software is permitted
*    provided a current software registration is on file at www.OpenPBS.org.
*    If use of this software contributes to a publication, product, or
*    service, proper attribution must be given; see www.OpenPBS.org/credit.html
*
* 2. Redistribution in any form is only permitted for non-commercial,
*    non-profit purposes.  There can be no charge for the Software or any
*    software incorporating the Software.  Further, there can be no
*    expectation of revenue generated as a consequence of redistributing
*    the Software.
*
* 3. Any Redistribution of source code must retain the above copyright notice
*    and the acknowledgment contained in paragraph 6, this list of conditions
*    and the disclaimer contained in paragraph 7.
*
* 4. Any Redistribution in binary form must reproduce the above copyright
*    notice and the acknowledgment contained in paragraph 6, this list of
*    conditions and the disclaimer contained in paragraph 7 in the
*    documentation and/or other materials provided with the distribution.
*
* 5. Redistributions in any form must be accompanied by information on how to
*    obtain complete source code for the OpenPBS software and any
*    modifications and/or additions to the OpenPBS software.  The source code
*    must either be included in the distribution or be available for no more
*    than the cost of distribution plus a nominal fee, and all modifications
*    and additions to the Software must be freely redistributable by any party
*    (including Licensor) without restriction.
*
* 6. All advertising materials mentioning features or use of the Software must
*    display the following acknowledgment:
*
*     "This product includes software developed by NASA Ames Research Center,
*     Lawrence Livermore National Laboratory, and Veridian Information
*     Solutions, Inc.
*     Visit www.OpenPBS.org for OpenPBS software support,
*     products, and information."
*
* 7. DISCLAIMER OF WARRANTY
*
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND. ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT
* ARE EXPRESSLY DISCLAIMED.
*
* IN NO EVENT SHALL VERIDIAN CORPORATION, ITS AFFILIATED COMPANIES, OR THE
* U.S. GOVERNMENT OR ANY OF ITS AGENCIES BE LIABLE FOR ANY DIRECT OR INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <limits.h>
#include <assert.h>
#include <ctype.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pbs_ifl.h"
#include "list_link.h"
#include "attribute.h"
#include "pbs_error.h"
#include "pbs_helper.h"

/*
 * This file contains functions for manipulating attributes of type
 * time: [[hh:]mm:]ss[.sss]
 *
 * Each set has functions for:
 * Decoding the value string to the machine representation.
 * Encoding the internal pbs_attribute to external form
 * Setting the value by =, + or - operators.
 * Comparing a (decoded) value with the pbs_attribute value.
 *
 * Some or all of the functions for an pbs_attribute type may be shared with
 * other pbs_attribute types.
 *
 * The prototypes are declared in "attribute.h"
 *
 * --------------------------------------------------
 * The Set of pbs_Attribute Functions for attributes with
 * value type "long"
 * --------------------------------------------------
 */
#define PBS_MAX_TIME (LONG_MAX - 1)



/*
 * time_str_to_seconds()
 *
 * Determines a number of seconds from a time string in the format:
 * [DD:]HH:MM:SS[.MS]
 *
 * @param str - the time string
 * @return the number of seconds or -1 if the format is invalid
 */

long time_str_to_seconds(

  const char *str)

  {
  long  time_val = 0;

  char *workval = strdup(str);
  char  msec[4];

  char *workvalsv = workval;
  bool  use_days = false;
  int   ncolon = 0;
  int   days = 0;

  if (workvalsv == NULL)
    {
    /* FAILURE - cannot alloc memory */

    return(-1);
    }

  for (int i = 0;i < 3;++i)
    msec[i] = '0';

  msec[3] = '\0';

  for (char *pc = workval; *pc; ++pc)
    {
    if (*pc == ':')
      {
      if (++ncolon > 3)
        {
        free(workvalsv);
        return(-1);
        }

      /* are days specified? */
      if (ncolon > 2)
        use_days = true;
      }
    }

  for (char *pc = workval; *pc; ++pc)
    {
    if (*pc == ':')
      {

      *pc = '\0';

      if (use_days)
        {
        days = strtol(workval, NULL, 10);
        use_days = false;
        }
      else
        {
        time_val = (time_val * 60) + strtol(workval, NULL, 10);
        }

      workval = pc + 1;

      }
    else if (*pc == '.')
      {
      *pc++ = '\0';

      for (int i = 0; (i < 3) && *pc; ++i)
        msec[i] = *pc++;

      break;
      }
    else if (!isdigit((int)*pc))
      {
      free(workvalsv);
      return(-1); /* bad value */
      }
    }

  time_val = (time_val * 60) + strtol(workval, NULL, 10);
  
  if (days > 0)
   time_val = time_val + (days * 24 * 3600);

  if (time_val > PBS_MAX_TIME)
    {
    free(workvalsv);
    return(-1);
    }

  if (strtol(msec, NULL, 10) >= 500)
    time_val++;

  free(workvalsv);

  return(time_val);
  } // END time_string_to_seconds()



/*
 * decode_time - decode time into into pbs_attribute structure of type ATR_TYPE_LONG
 *
 * Returns: 0 if ok
 *  >0 error number if error
 *  *patr elements set
 */


int decode_time(

  pbs_attribute *patr,  /* I/O (modified) */
  const char * UNUSED(name),  /* I - pbs_attribute name (not used) */
  const char * UNUSED(rescn), /* I - resource name (not used) */
  const char    *val,   /* I - pbs_attribute value */
  int          UNUSED(perm))  /* only used for resources */

  {
  long rv = 0;
  int  rc = PBSE_NONE;

  if ((val == NULL) || (strlen(val) == 0))
    {
    patr->at_flags = (patr->at_flags & ~ATR_VFLAG_SET) | ATR_VFLAG_MODIFY;

    patr->at_val.at_long = 0;

    /* SUCCESS */

    return(rc);
    }

  rv = time_str_to_seconds(val);

  if (rv >= 0)
    {
    // SUCCESS
    patr->at_val.at_long = rv;
    patr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
    }
  else
    rc = PBSE_BADATVAL;

  return(rc);
  }  /* END decode_time() */



/* 
 * get_time_string()
 *
 * Takes a number in seconds and translates it into days:minutes:seconds
 *
 * @param time_string (O) - the string where the time should be written
 * @param string_size (I) - the size of the output string
 * @param timeval (I) - the number of seconds in the time
 * @return PBSE_NONE
 */

int get_time_string(

  char *time_string,
  int   string_size,
  long  timeval)

  {
  int       hr;
  int       min;
  int       sec;
  
  hr      = timeval / 3600;
  timeval = timeval % 3600;
  min     = timeval / 60;
  timeval = timeval % 60;
  sec = timeval;
  
  snprintf(time_string, string_size, "%02d:%02d:%02d", hr, min, sec);

  return(PBSE_NONE);
  } // END get_time_string()



/*
 * encode_time - encode pbs_attribute of type long into attr_extern
 * with value in form of [[hh:]mm:]ss
 *
 * Returns: >0 if ok
 *   =0 if no value, no attrlist link added
 *   <0 if error
 */
/*ARGSUSED*/

/* NOTE:  if phead not specified, report output via atname (minsize=1024) */

#define ENCODE_TIME_SIZE 21

int encode_time(

  pbs_attribute  *attr,   /* ptr to pbs_attribute (value in attr->at_val.at_long) */
  tlist_head     *phead,  /* head of attrlist list (optional) */
  const char    *atname, /* pbs_attribute name */
  const char    *rsname, /* resource name (optional) */
  int            UNUSED(mode),   /* encode mode (not used) */
  int            UNUSED(perm))  /* only used for resources */

  {
  size_t    ct;
  char      cvnbuf[ENCODE_TIME_SIZE];
  svrattrl *pal;

  if ((attr == NULL)||(phead == NULL))
    {
    /* FAILURE */

    return(-1);
    }

  if (!(attr->at_flags & ATR_VFLAG_SET))
    {
    return(0);
    }

  get_time_string(cvnbuf, sizeof(cvnbuf), attr->at_val.at_long);

  ct = strlen(cvnbuf);

  pal = attrlist_create(atname, rsname, ct+1);

  if (pal == NULL)
    {
    /* FAILURE */

    return(-1);
    }

  memcpy(pal->al_value, cvnbuf, ct);

  pal->al_flags = attr->at_flags;

  append_link(phead, &pal->al_link, pal);
  /* SUCCESS */

  return(1);
  } // END encode_time()



/*
 * set_time  - use the function set_l()
 *
 * comp_time - use the funttion comp_l()
 *
 * free_l - use free_null to (not) free space
 */
