// $Id$

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.




#include "SDL.h"
#include "SDL_mixer.h"
#include "SDL_opengl.h"
#include <iostream>
#include <cstdlib>
#include "enums.h"
#include "Game_manager.h"
#include <cstring>
#include <ctime>
#include <fstream>
#include <string>
#include "User_input.h"
#include "FS_gfx.h"
#include <tr1/memory>

#ifdef __unix__
    #include <sys/stat.h>
    #include <errno.h>
#endif




// ************************
// *** GLOBAL VARIABLES ***
// ************************
bool test_level = false;
char* test_file;
bool joystick_available = false;
bool sound_available = false;
char* fs_dir;
SDL_Surface* screen;

std::tr1::shared_ptr<FS::FS_gfx> gfx_ptr;
std::tr1::shared_ptr<FS::User_input> input_ptr;

bool escape_pressed = false;
FS::Mode current_mode = FS::MODE_GAME;




// ************************
// *** STATIC VARIABLES ***
// ************************
static bool fullscreen = false;




// ***************************
// *** FUNCTION PROTOTYPES ***
// ***************************
void set_fs_dir();
void print_version();
void print_help();
void process_command_line_args(int argc, char* argv[]);




// **************
// *** main() ***
// **************

int main(int argc, char* argv[])
{
    // This ensures that everything is cleaned up on exit.
    // N.B. Functions are called in reverse order.
    atexit(SDL_Quit);
    atexit(Mix_CloseAudio);
    
    process_command_line_args(argc, argv);
    
    // initialize SDL - need video, audio and joystick subsystems
    if ( SDL_Init(SDL_INIT_VIDEO|SDL_INIT_AUDIO|SDL_INIT_JOYSTICK) < 0 )
    {
        std::cerr << "Can't initialize SDL: " << SDL_GetError() << std::endl;
        exit(EXIT_FAILURE);
    } // if
    
    // set up audio (via SDL mixer)
    if ( Mix_OpenAudio(22050, AUDIO_S16SYS, 2, 1024) < 0 )
    {
        std::cerr << "Can't open audio: " << Mix_GetError() << std::endl;
        //exit(EXIT_FAILURE);
    }
    else
    {
        sound_available = true;
    } // if ... else
    
    // set up joystick
    SDL_Joystick* js = NULL;
    if ( SDL_NumJoysticks() > 0 )
    {
        if ( (js = SDL_JoystickOpen(0)) )
        {
            joystick_available = true;
            std::cerr << "Joystick available: " << SDL_JoystickName(0) <<
                    std::endl;
        } // if
    } // if
    
    // Get an SDL_Surface. 
    screen = NULL;
    /*
    Uint32 flags = (SDL_SWSURFACE | SDL_ANYFORMAT | ((fullscreen) ? SDL_FULLSCREEN : 0));
    if ( (screen = SDL_SetVideoMode(FS::SCREEN_WIDTH, FS::SCREEN_HEIGHT, FS::BPP, flags)) == NULL )
    {
        std::cerr << "Can't get an SDL surface: " << SDL_GetError() << std::endl;
        exit(EXIT_FAILURE);
    } // if
    */
    SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
    screen = SDL_SetVideoMode( FS::SCREEN_WIDTH, FS::SCREEN_HEIGHT, 0, SDL_OPENGL );
    if ( !screen )
    {
        std::cerr << "Can't get an SDL surface: " << SDL_GetError() << std::endl;
        exit(EXIT_FAILURE);
    } // if
    glEnable(GL_TEXTURE_2D);
    glViewport(0, 0, FS::SCREEN_WIDTH, FS::SCREEN_HEIGHT);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, FS::SCREEN_WIDTH, FS::SCREEN_HEIGHT, 0.0, -1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();


    // set window/icon title
    SDL_WM_SetCaption( PACKAGE_STRING, PACKAGE_NAME );
    
    // hide mouse cursor
    SDL_ShowCursor(SDL_DISABLE);
    
    // seed random-number generator
    srand(time(0));

    set_fs_dir();
    
    // When assigning to a std::tr1::shared_ptr, rhs must be a std::tr1::shared_ptr!
    gfx_ptr = std::tr1::shared_ptr<FS::FS_gfx>( new FS::FS_gfx() );

    // load graphics
    //my_gfx = new FS::GfxStore();
    
    // Set up FontRepository object. This is needed by many classes under 
    // control of GameManager.
    //font_rep = new FS::FontRepository();
    input_ptr = std::tr1::shared_ptr<FS::User_input>( new FS::User_input() );

    // Pass control to GameManager.              
    FS::Game_manager gm;
    gm.start_loop();

    //delete font_rep;
    //delete my_gfx;
    delete [] fs_dir;
    
    SDL_JoystickClose(js);
    Mix_CloseAudio();
    SDL_Quit();
    
    return 0;
    
} // main()




// ****************************
// *** FUNCTION DEFINITIONS ***
// ****************************

// **************************************************

void set_fs_dir()
{
    bool ok = false;

#ifdef __unix__
        
    char* home_dir;
    
    std::cerr << "Checking for $HOME... ";
    if ( (home_dir = getenv("HOME")) )
    {
        std::cerr << home_dir << std::endl;
        
        fs_dir = new char[ strlen(home_dir) + strlen("/.fishsupper") + 1 ];
        strcpy(fs_dir, home_dir);
        strcat(fs_dir, "/.fishsupper");
        
        std::cerr << "Creating directory " << fs_dir << "... ";
        if ( !mkdir(fs_dir, (S_IRWXU|S_IRWXG|S_IRWXO)) )
        {
            std::cerr << "OK\n";
            
            // Write config.txt and table.txt in new dotdir.
            std::ofstream fout;
            std::string my_file;

            my_file = fs_dir;
            my_file.append("/config.txt");
            std::cerr << "Writing file " << my_file << "... ";
            fout.open( my_file.c_str() );
            if ( fout.is_open() )
            {
                fout << "joystick_selected=0" << std::endl <<
                        "sound=1" << std::endl <<
                        "fullscreen=0" << std::endl <<
                        "furthest_level=1" << std::endl;
                fout.close();
                std::cerr << "OK\n";
                    
                my_file = fs_dir;
                my_file.append("/table.txt");
                std::cerr << "Writing file " << my_file << "... ";
                fout.open( my_file.c_str() );
                if ( fout.is_open() )
                {   
                    fout << "Bertie=4000" << std::endl <<
                            "Agatha=3000" << std::endl <<
                            "Oofy=2000" << std::endl <<
                            "Dahlia=1000" << std::endl <<
                            "Spode=500" << std::endl;
                    fout.close();
                    std::cerr << "OK\n";
                    ok = true;
                }
                else
                {
                    // couldn't write table.txt
                    std::cerr << "failed\n";
                    delete [] fs_dir;
                } // if ... else
            }
            else
            {
                // couldn't write config.txt
                std::cerr << "failed\n";
                delete [] fs_dir;
            } // if ... else
        }
        else
        {
            // couldn't create dotdir
            // Is this because it already exists or has an error occurred?
            switch (errno)
            {
                case EEXIST:
                    std::cerr << "exists\n";
                    ok = true;
                    break;
                default:
                    std::cerr << "failed\n";
                    delete [] fs_dir;
                    break;
            } // switch
        } // if ... else
    }
    else
    {
        // couldn't find $HOME
        std::cerr << "failed\n";
    } // if ... else 
    
#endif
    
    if (!ok)
    {
        fs_dir = new char[ strlen(".") ];
        strcpy(fs_dir, ".");
    } // if ... else
    
} // set_fs_dir()

// **************************************************

void print_version()
{
    std::cerr << PACKAGE_STRING "\n"
                 "Copyright 2006, 2007, 2009 Matthew Clarke\n"
                 "This is free software; see the source for copying conditions. "
                 "There is NO\n"
                 "warranty; not even for MERCHANTABILITY or FITNESS FOR A "
                 "PARTICULAR PURPOSE.\n\n";
    
} // print_version()

// **************************************************

void print_help()
{
    std::cerr << "Usage: fishsupper [options]\n"
                 "Options:\n"
                 "  --fullscreen        Start game in fullscreen mode\n"
                 "  --help              Display this message and quit\n"
                 "  --testlevel=FILE    Play level called 'FILE' repeatedly\n"
                 "  --version           Display version number and quit\n\n";
    
} // print_help()

// **************************************************

void process_command_line_args(int argc, char* argv[])
{
    for (int i = 1; i < argc; ++i)
    {
        if (!strcmp(argv[i], "--help"))
        {
            print_help();
            exit(EXIT_SUCCESS);
        }
        else if (!strcmp(argv[i], "--version"))
        {
            print_version();
            exit(EXIT_SUCCESS);
        }
        // Also check that arg is > 12 (i.e. that at least one character follows '=').
        // FIXME: error checking and print message to screen?
        else if ( (!strncmp(argv[i], "--testlevel=", 12)) && (strlen(argv[i]) > 12) )
        {
            test_level = true;
            test_file = &(argv[i][12]);
        }
        else if (!strcmp(argv[i], "--fullscreen"))
        {
            fullscreen = true;
        } // if ... else
    } // for
    
} // process_command_line_args()
      
// **************************************************    

    
