/* $Id: op_alert_syslog.c,v 1.10 2004/03/16 04:18:20 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 1998-2001 Martin Roesch <roesch@sourcefire.com>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

/*  I N C L U D E S  *****************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <sys/types.h>
#include <netinet/in.h>

#include "strlcpyu.h"
#include "ConfigFile.h"
#include "plugbase.h"
#include "op_plugbase.h"
#include "mstring.h"
#include "util.h"
#include "sid.h"
#include "classification.h"
#include "input-plugins/dp_alert.h"
#include "barnyard.h"

/*  D A T A   S T R U C T U R E S  **************************************/
typedef struct _OpAlertSyslog_Data 
{
    int facility;
    int priority;
    int options;
} OpAlertSyslog_Data;


/* Output Plugin API */
static int OpAlertSyslog_Setup(OutputPlugin *, char *args);
static int OpAlertSyslog_Exit(OutputPlugin *);
static int OpAlertSyslog_Start(OutputPlugin *, void *);
static int OpAlertSyslog_Stop(OutputPlugin *);
static int OpAlertSyslog(void *, void *);
static int OpAlertSyslog_LogConfig(OutputPlugin *);


/* Internal functions */
static OpAlertSyslog_Data *OpAlertSyslog_ParseArgs(char *);

/* init routine makes this processor available for dataprocessor directives */
void OpAlertSyslog_Init()
{
    OutputPlugin *outputPlugin;
    
    outputPlugin = RegisterOutputPlugin("alert_syslog", "alert");

    outputPlugin->setupFunc = OpAlertSyslog_Setup;
    outputPlugin->exitFunc = OpAlertSyslog_Exit;
    outputPlugin->startFunc = OpAlertSyslog_Start;
    outputPlugin->stopFunc = OpAlertSyslog_Stop;
    outputPlugin->outputFunc = OpAlertSyslog;
    outputPlugin->logConfigFunc = OpAlertSyslog_LogConfig;
}

static int OpAlertSyslog_LogConfig(OutputPlugin *outputPlugin)
{
    OpAlertSyslog_Data *data = NULL;

    if(!outputPlugin || !outputPlugin->data)
        return -1;

    data = (OpAlertSyslog_Data *)outputPlugin->data;

    LogMessage("OpAlertSyslog configured\n");
    LogMessage("  Facility: %i\n", data->facility);
    LogMessage("  Priority: %i\n", data->priority);
    LogMessage("  Options: %i\n", data->options);
    return 0;
}


/* Setup the output plugin, process any arguments, link the functions to
 * the output functional node
 */
int OpAlertSyslog_Setup(OutputPlugin *outputPlugin, char *args)
{
    /* setup the run time context for this output plugin */
    outputPlugin->data = OpAlertSyslog_ParseArgs(args);

    return 0;
}

/* Inverse of the setup function, free memory allocated in Setup 
 * can't free the outputPlugin since it is also the list node itself
 */
int OpAlertSyslog_Exit(OutputPlugin *outputPlugin)
{
    return 0;
}

/* 
 * this function gets called at start time, you should open any output files
 * or establish DB connections, etc, here
 */
int OpAlertSyslog_Start(OutputPlugin *outputPlugin, void *spool_header)
{
    OpAlertSyslog_Data *data = (OpAlertSyslog_Data *)outputPlugin->data;

    if(data == NULL)
        FatalError("ERROR: Unable to find context for AlertSyslog startup!\n");

    if(pv.verbose >= 2)
        OpAlertSyslog_LogConfig(outputPlugin);
    
    openlog("barnyard", data->options, data->facility);

    return 0;
}

int OpAlertSyslog_Stop(OutputPlugin *outputPlugin)
{
    closelog();
    return 0;
}

/* 
 * this is the primary output function for the plugin, this is what gets called
 * for every record read 
 */
int OpAlertSyslog(void *context, void *data)
{
    Sid *sid = NULL;
    char src[16];
    char dest[16];
    ClassType *class_type;
    UnifiedAlertRecord *record = (UnifiedAlertRecord *)data; 
    OpAlertSyslog_Data *op_data = (OpAlertSyslog_Data *)context;

    sid = GetSid(record->event.sig_generator, record->event.sig_id);
    
    class_type = GetClassType(record->event.classification);
    
    snprintf(src, 16, "%u.%u.%u.%u", (record->sip & 0xff000000) >> 24,
            (record->sip & 0x00ff0000) >> 16, (record->sip & 0x0000ff00) >> 8,
            record->sip & 0x000000ff);
    snprintf(dest, 16, "%u.%u.%u.%u", (record->dip & 0xff000000) >> 24,
            (record->dip & 0x00ff0000) >> 16, (record->dip & 0x0000ff00) >> 8,
            record->dip & 0x000000ff);
    
    switch(record->protocol)
    {
        case IPPROTO_TCP:
        case IPPROTO_UDP:
            syslog(op_data->priority, "[%d:%d:%d] %s [Classification: %s] "
                    "[Priority: %d] {%s} %s:%d -> %s:%d", 
                    record->event.sig_generator, record->event.sig_id, 
                    record->event.sig_rev, sid != NULL ? sid->msg : "ALERT",
                    class_type != NULL ? class_type->name : "Unknown", 
                    record->event.priority, protocol_names[record->protocol], 
                    src, record->sp, dest, record->dp);
            break;
        case IPPROTO_ICMP:
            syslog(op_data->priority, "[%d:%d:%d] %s [Classification: %s] "
                    "[Priority: %d] {%s} %s -> %s", 
                    record->event.sig_generator, record->event.sig_id, 
                    record->event.sig_rev, sid != NULL ? sid->msg : "ALERT",
                    class_type != NULL ? class_type->name : "Unknown", 
                    record->event.priority, protocol_names[record->protocol], 
                    src, dest);
            break;
        default:
            syslog(op_data->priority, "[%d:%d:%d] %s [Classification: %s] "
                    "[Priority: %d] {%s} %s -> %s", 
                    record->event.sig_generator, record->event.sig_id, 
                    record->event.sig_rev, sid != NULL ? sid->msg : "ALERT",
                    class_type != NULL ? class_type->name : "Unknown", 
                    record->event.priority, protocol_names[record->protocol], 
                    src, dest);
            break;
    }
    return 0;
}


/* initialize the output processor for this particular instantiation */
OpAlertSyslog_Data *OpAlertSyslog_ParseArgs(char *args)
{
    OpAlertSyslog_Data *data;

    data = (OpAlertSyslog_Data *)SafeAlloc(sizeof(OpAlertSyslog_Data));

    data->options = 0;
    data->facility = LOG_AUTH;
    data->priority = LOG_INFO;
    
    if(args != NULL)
    {
        char **toks;
        int num_toks;
        int i;
        /* parse out your args */
        toks = mSplit(args, " |", 31, &num_toks, '\\');
        for(i = 0; i < num_toks; ++i)
        {
            char *tmp = toks[i];
            /* syslog options */
#ifdef LOG_CONS 
            if(!strcasecmp("LOG_CONS", tmp))
                data->options |= LOG_CONS;
#endif
#ifdef LOG_NDELAY 
            else if(!strcasecmp("LOG_NDELAY", tmp))
                data->options |= LOG_NDELAY;
#endif
#ifdef LOG_PERROR 
            else if(!strcasecmp("LOG_PERROR", tmp))
                data->options |= LOG_PERROR;
#endif
#ifdef LOG_PID 
            else if(!strcasecmp("LOG_PID", tmp))
                data->options |= LOG_PID;
#endif

            /* syslog facilities that make sense to use */
#ifdef LOG_AUTHPRIV 
            else if(!strcasecmp("LOG_AUTHPRIV", tmp))
                data->facility = LOG_AUTHPRIV;
#endif
#ifdef LOG_AUTH 
            else if(!strcasecmp("LOG_AUTH", tmp))
                data->facility = LOG_AUTH;
#endif
#ifdef LOG_DAEMON 
            else if(!strcasecmp("LOG_DAEMON", tmp))
                data->facility = LOG_DAEMON;
#endif
#ifdef LOG_LOCAL0 
            else if(!strcasecmp("LOG_LOCAL0", tmp))
                data->facility = LOG_LOCAL0;
#endif
#ifdef LOG_LOCAL1 
            else if(!strcasecmp("LOG_LOCAL1", tmp))
                data->facility = LOG_LOCAL1;
#endif
#ifdef LOG_LOCAL2 
            else if(!strcasecmp("LOG_LOCAL2", tmp))
                data->facility = LOG_LOCAL2;
#endif
#ifdef LOG_LOCAL3 
            else if(!strcasecmp("LOG_LOCAL3", tmp))
                data->facility = LOG_LOCAL3;
#endif
#ifdef LOG_LOCAL4 
            else if(!strcasecmp("LOG_LOCAL4", tmp))
                data->facility = LOG_LOCAL4;
#endif
#ifdef LOG_LOCAL5 
            else if(!strcasecmp("LOG_LOCAL5", tmp))
                data->facility = LOG_LOCAL5;
#endif
#ifdef LOG_LOCAL6 
            else if(!strcasecmp("LOG_LOCAL6", tmp))
                data->facility = LOG_LOCAL6;
#endif
#ifdef LOG_LOCAL7 
            else if(!strcasecmp("LOG_LOCAL7", tmp))
                data->facility = LOG_LOCAL7;
#endif
#ifdef LOG_USER 
            else if(!strcasecmp("LOG_USER", tmp))
                data->facility = LOG_USER;
#endif
            /* syslog priorities */

#ifdef LOG_EMERG 
            else if(!strcasecmp("LOG_EMERG", tmp))
                data->priority = LOG_EMERG;
#endif
#ifdef LOG_ALERT 
            else if(!strcasecmp("LOG_ALERT", tmp))
                data->priority = LOG_ALERT;
#endif
#ifdef LOG_CRIT 
            else if(!strcasecmp("LOG_CRIT", tmp))
                data->priority = LOG_CRIT;
#endif
#ifdef LOG_ERR 
            else if(!strcasecmp("LOG_ERR", tmp))
                data->priority = LOG_ERR;
#endif
#ifdef LOG_WARNING 
            else if(!strcasecmp("LOG_WARNING", tmp))
                data->priority = LOG_WARNING;
#endif
#ifdef LOG_NOTICE 
            else if(!strcasecmp("LOG_NOTICE", tmp))
                data->priority = LOG_NOTICE;
#endif
#ifdef LOG_INFO 
            else if(!strcasecmp("LOG_INFO", tmp))
                data->priority = LOG_INFO;
#endif
#ifdef LOG_DEBUG 
            else if(!strcasecmp("LOG_DEBUG", tmp))
                data->priority = LOG_DEBUG;
#endif
            else
            {
                fprintf(stderr, "WARNING %s (%d) => Unrecoginzed argument for "
                        "AlertSyslog plugin: %s\n", file_name, file_line, tmp);
            }
        }
        /* free your mSplit tokens */
        FreeToks(toks, num_toks);
    }

    return data;
}




