;;;
;;; qdbm.scm - qdbm interface
;;;
;;;   Copyright (c) 2003 Kimura Fuyuki, All rights reserved.
;;;
;;;   Redistribution and use in source and binary forms, with or without
;;;   modification, are permitted provided that the following conditions
;;;   are met:
;;;
;;;   1. Redistributions of source code must retain the above copyright
;;;      notice, this list of conditions and the following disclaimer.
;;;
;;;   2. Redistributions in binary form must reproduce the above copyright
;;;      notice, this list of conditions and the following disclaimer in the
;;;      documentation and/or other materials provided with the distribution.
;;;
;;;   3. Neither the name of the authors nor the names of its contributors
;;;      may be used to endorse or promote products derived from this
;;;      software without specific prior written permission.
;;;
;;;   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;;;   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;;;   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;;;   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
;;;   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;;;   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
;;;   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;;   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;;   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;;   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;;   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;;;
;;;  $Id: qdbm.scm,v 1.3 2003/11/28 02:08:32 fuyuki Exp $
;;;

(define-module dbm.qdbm
  (extend dbm)
  (export <qdbm>
          ;; low-level depot functions
          qdbm-file-of
          qdbm-dpversion qdbm-dpecode qdbm-dperrmsg
          qdbm-dpopen qdbm-dpclose qdbm-dpclosed?
          qdbm-dpput qdbm-dpout qdbm-dpget
          qdbm-dpvsiz qdbm-dpiterinit qdbm-dpiternext
          qdbm-dpsetalign qdbm-dpsync qdbm-dpoptimize
          qdbm-dpname qdbm-dpfsiz qdbm-dpbnum qdbm-dpbusenum
          qdbm-dprnum qdbm-dpwritable
          qdbm-dpfatalerror qdbm-dpinode qdbm-dpfdesc qdbm-dpremove
          qdbm-dpinnerhash qdbm-dpouterhash qdbm-dpprimenum
          |DP_OREADER| |DP_OWRITER| |DP_OCREAT| |DP_OTRUNC| |DP_ONOLCK|
          |DP_DOVER| |DP_DKEEP| |DP_DCAT|))
(select-module dbm.qdbm)
(dynamic-load "qdbm")

;; The following is heavily based on ext/dbm/gdbm.scm.

;;
;; Initialize
;;

(define-class <qdbm-meta> (<dbm-meta>)
  ())

(define-class <qdbm> (<dbm>)
  ((qdbm-file :accessor qdbm-file-of :init-value #f)
   (bnum :init-keyword :bnum :init-value 0)
   )
  :metaclass <qdbm-meta>)

(define-method dbm-open ((self <qdbm>))
  (next-method)
  (unless (slot-bound? self 'path)
    (error "path must be set to open qdbm database"))
  (when (qdbm-file-of self)
    (errorf "qdbm ~S already opened" qdbm))
  (let* ((path   (slot-ref self 'path))
         (rwmode (slot-ref self 'rw-mode))
         (omode  (case rwmode
                   ((:read) |DP_OREADER|)
                   ((:write) (+ |DP_OWRITER| |DP_OCREAT|))
                   ((:create) (+ |DP_OWRITER| |DP_OCREAT| |DP_OTRUNC|))))
         (fp     (qdbm-dpopen path omode (slot-ref self 'bnum))))
    (slot-set! self 'qdbm-file fp)
    self))

;;
;; close operation
;;

(define-method dbm-close ((self <qdbm>))
  (let ((qdbm (qdbm-file-of self)))
    (and qdbm (qdbm-dpclose qdbm))))

(define-method dbm-closed? ((self <qdbm>))
  (let ((qdbm (qdbm-file-of self)))
    (or (not qdbm) (qdbm-dpclosed? qdbm))))

;;
;; accessors
;;

(define-method dbm-put! ((self <qdbm>) key value)
  (next-method)
  (unless (qdbm-dpput (qdbm-file-of self)
                      (%dbm-k2s self key)
                      (%dbm-v2s self value)
                      |DP_DOVER|)
    (error "dbm-put! failed" self)))

(define-method dbm-get ((self <qdbm>) key . args)
  (next-method)
  (cond ((qdbm-dpget (qdbm-file-of self) (%dbm-k2s self key))
         => (lambda (v) (%dbm-s2v self v)))
        ((pair? args) (car args))     ;fall-back value
        (else  (errorf "qdbm: no data for key ~s in database ~s"
                       key (qdbm-file-of self)))))

(define-method dbm-exists? ((self <qdbm>) key)
  (next-method)
  (and (qdbm-dpget (qdbm-file-of self) (%dbm-k2s self key)) #t))

(define-method dbm-delete! ((self <qdbm>) key)
  (next-method)
  (unless (qdbm-dpout (qdbm-file-of self) (%dbm-k2s self key))
    (errorf "dbm-delete!: deleteting key ~s from ~s failed" key self)))

;;
;; Iterations
;;

(define-method dbm-fold ((self <qdbm>) proc knil)
  (let ((qdbm (qdbm-file-of self)))
    (qdbm-dpiterinit qdbm)
    (let loop ((key (qdbm-dpiternext qdbm))
               (r   knil))
      (if key
          (let ((val (qdbm-dpget qdbm key)))
            (loop (qdbm-dpiternext qdbm)
                  (proc (%dbm-s2k self key) (%dbm-s2v self val) r)))
          r))))

;;
;; Metaoperations
;;

(autoload file.util copy-file move-file)

(define (%with-qdbm-locking path thunk)
  (let1 db (qdbm-dpopen path |DP_OREADER|) ;; put read-lock
    (with-error-handler
        (lambda (e) (qdbm-dpclose db) (raise e))
      (lambda () (thunk) (qdbm-dpclose db)))))

(define-method dbm-db-exists? ((class <qdbm-meta>) name)
  (file-exists? name))

(define-method dbm-db-remove ((class <qdbm-meta>) name)
  (qdbm-dpremove name))

(define-method dbm-db-copy ((class <qdbm-meta>) from to . keys)
  (%with-qdbm-locking
   (lambda () (apply copy-file from to :safe #t keys))))

(define-method dbm-db-rename ((class <qdbm-meta>) from to . keys)
  (%with-qdbm-locking
   (lambda () (apply move-file from to :safe #t keys))))

(provide "dbm/qdbm")
