// We like hashes around raw string literals.
#![allow(clippy::needless_raw_string_hashes)]

// Work around dead code warnings: rust-lang issue #46379
pub mod common;
use common::*;

#[test]
fn run_parse_should_fail_tests() {
    test_parse_fails(r#"x{5,4}"#);
    test_parse_fails(r#"[abcd"#);
    test_parse_fails(r#"[z-a]"#);
    test_parse_fails(r#"^*"#);
    test_parse_fails(r#"(abc"#);
    test_parse_fails(r#"(?# abc"#);
    test_parse_fails(r#"{4,5}abc"#);
    test_parse_fails(r#")"#);
    test_parse_fails(r#"a[b-a]"#);
    test_parse_fails(r#"a["#);
    test_parse_fails(r#"*a"#);
    test_parse_fails(r#"abc)"#);
    test_parse_fails(r#"(abc"#);
    test_parse_fails(r#"a**"#);
    test_parse_fails(r#")("#);
    test_parse_fails(r#"a[b-a]"#);
    test_parse_fails(r#"a["#);
    test_parse_fails(r#"*a"#);
    test_parse_fails(r#"abc)"#);
    test_parse_fails(r#"(abc"#);
    test_parse_fails(r#"a**"#);
    test_parse_fails(r#")("#);
    test_parse_fails(r#":(?:"#);
    test_parse_fails(r#"a(?{)b"#);
    test_parse_fails(r#"a(?{{})b"#);
    test_parse_fails(r#"a(?{}})b"#);
    test_parse_fails(r#"a(?{"{"})b"#);
    test_parse_fails(r#"a(?{"{"}})b"#);
    test_parse_fails(r#"[a[:xyz:"#);
    test_parse_fails(r#"a{37,17}"#);
    test_parse_fails(r#"[\200-\110]"#);
    test_parse_fails(r#"["#);
    test_parse_fails(r#"[a-"#);
    test_parse_fails(r#"^[a-\Q\E]"#);
    test_parse_fails(r#"(ab|c)(?-1)"#);
    test_parse_fails(r#"x(?-0)y"#);
    test_parse_fails(r#"x(?-1)y"#);
    test_parse_fails(r#"(?|(abc)|(xyz))"#);
    test_parse_fails(r#"(x)(?|(abc)|(xyz))(x)"#);
    test_parse_fails(r#"(x)(?|(abc)(pqr)|(xyz))(x)"#);
    test_parse_fails(r#"(?|(abc)|(xyz))\1"#);
    test_parse_fails(r#"(?-+a)"#);
    test_parse_fails(r#"^\ca\cA\c[\c{\c:"#);
    test_parse_fails(r#"a(?)b"#);
    test_parse_fails(r#"(?|(abc)|(xyz))"#);
    test_parse_fails(r#"(x)(?|(abc)|(xyz))(x)"#);
    test_parse_fails(r#"(x)(?|(abc)(pqr)|(xyz))(x)"#);
}

#[test]
fn run_pcre_match_tests() {
    test_with_configs(run_pcre_match_tests_config)
}

#[rustfmt::skip]
fn run_pcre_match_tests_config(tc: TestConfig) {
    let run1_match = |pattern: &str, flags_str: &str, input: &str| -> String {
        let cr = tc.compilef(pattern, flags_str);
        cr.match1f(input)
    };

    let test_eq = |left: String, right: &str| {
        assert_eq!(left.as_str(), right)
    };

    test_eq(run1_match("abc", "i", "abc"), "abc"); // "abc"
    test_eq(run1_match("abc", "i", "defabc"), "abc"); // "abc"
    test_eq(run1_match("abc", "i", "Aabc"), "abc"); // "abc"
    test_eq(run1_match("abc", "i", "Adefabc"), "abc"); // "abc"
    test_eq(run1_match("abc", "i", "ABC"), "ABC"); // "abc"
    test_eq(run1_match("^abc", "i", "abc"), "abc"); // "^abc"
    test_eq(run1_match("^abc$", "i", "abc"), "abc"); // "^abc$"
    test_eq(run1_match("cat|dog|elephant", "i", "this sentence eventually mentions a cat"), "cat"); // "cat|dog|elephant"
    test_eq(run1_match("cat|dog|elephant", "i", "this sentences rambles on and on for a while and then reaches elephant"), "elephant"); // "cat|dog|elephant"
    test_eq(run1_match("cat|dog|elephant", "i", "this sentence eventually mentions a cat"), "cat"); // "cat|dog|elephant"
    test_eq(run1_match("cat|dog|elephant", "i", "this sentences rambles on and on for a while and then reaches elephant"), "elephant"); // "cat|dog|elephant"
    test_eq(run1_match("cat|dog|elephant", "i", "this sentence eventually mentions a CAT cat"), "CAT"); // "cat|dog|elephant"
    test_eq(run1_match("cat|dog|elephant", "i", "this sentences rambles on and on for a while to elephant ElePhant"), "elephant"); // "cat|dog|elephant"
    test_eq(run1_match("(a)(b)(c)\\2", "i", "abcb"), "abcb,a,b,c"); // "(a)(b)(c)\\2"
    test_eq(run1_match("(a)(b)(c)\\2", "i", "O0abcb"), "abcb,a,b,c"); // "(a)(b)(c)\\2"
    test_eq(run1_match("(a)(b)(c)\\2", "i", "O3abcb"), "abcb,a,b,c"); // "(a)(b)(c)\\2"
    test_eq(run1_match("(a)(b)(c)\\2", "i", "O6abcb"), "abcb,a,b,c"); // "(a)(b)(c)\\2"
    test_eq(run1_match("(a)(b)(c)\\2", "i", "O9abcb"), "abcb,a,b,c"); // "(a)(b)(c)\\2"
    test_eq(run1_match("(a)(b)(c)\\2", "i", "O12abcb"), "abcb,a,b,c"); // "(a)(b)(c)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "abc"), "abc,a,,"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "O0abc"), "abc,a,,"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "O3abc"), "abc,a,,"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "O6abc"), "abc,a,,"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "aba"), "aba,,a,b"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "O0aba"), "aba,,a,b"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "O3aba"), "aba,,a,b"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "O6aba"), "aba,,a,b"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "O9aba"), "aba,,a,b"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("(a)bc|(a)(b)\\2", "i", "O12aba"), "aba,,a,b"); // "(a)bc|(a)(b)\\2"
    test_eq(run1_match("abc$", "i", "abc"), "abc"); // "abc$"
    test_eq(run1_match("the quick brown fox", "i", "the quick brown fox"), "the quick brown fox"); // "the quick brown fox"
    test_eq(run1_match("the quick brown fox", "i", "this is a line with the quick brown fox"), "the quick brown fox"); // "the quick brown fox"
    test_eq(run1_match("^abc|def", "i", "abcdef"), "abc"); // "^abc|def"
    test_eq(run1_match("^abc|def", "i", "abcdefB"), "abc"); // "^abc|def"
    test_eq(run1_match(".*((abc)$|(def))", "i", "defabc"), "defabc,abc,abc,"); // ".*((abc)$|(def))"
    test_eq(run1_match(".*((abc)$|(def))", "i", "Zdefabc"), "Zdefabc,abc,abc,"); // ".*((abc)$|(def))"
    test_eq(run1_match("abc", "i", "abc"), "abc"); // "abc"
    test_eq(run1_match("^abc|def", "i", "abcdef"), "abc"); // "^abc|def"
    test_eq(run1_match("^abc|def", "i", "abcdefB"), "abc"); // "^abc|def"
    test_eq(run1_match(".*((abc)$|(def))", "i", "defabc"), "defabc,abc,abc,"); // ".*((abc)$|(def))"
    test_eq(run1_match(".*((abc)$|(def))", "i", "Zdefabc"), "Zdefabc,abc,abc,"); // ".*((abc)$|(def))"
    test_eq(run1_match("the quick brown fox", "i", "the quick brown fox"), "the quick brown fox"); // "the quick brown fox"
    test_eq(run1_match("the quick brown fox", "i", "The Quick Brown Fox"), "The Quick Brown Fox"); // "the quick brown fox"
    test_eq(run1_match("the quick brown fox", "i", "the quick brown fox"), "the quick brown fox"); // "the quick brown fox"
    test_eq(run1_match("the quick brown fox", "i", "The Quick Brown Fox"), "The Quick Brown Fox"); // "the quick brown fox"
    test_eq(run1_match("abc$", "i", "abc"), "abc"); // "abc$"
    test_eq(run1_match("(abc\\1)", "i", "abc"), "abc,abc"); // "(abc\\1)"
    test_eq(run1_match("[^aeiou ]{3,}", "i", "co-processors, and for"), "-pr"); // "[^aeiou ]{3,}"
    test_eq(run1_match("<.*>", "i", "abc<def>ghi<klm>nop"), "<def>ghi<klm>"); // "<.*>"
    test_eq(run1_match("<.*?>", "i", "abc<def>ghi<klm>nop"), "<def>"); // "<.*?>"
    test_eq(run1_match("<.*?>", "i", "abc<def>ghi<klm>nop"), "<def>"); // "<.*?>"
    test_eq(run1_match("a$", "i", "a"), "a"); // "a$"
    test_eq(run1_match("a$", "i", "Za"), "a"); // "a$"
    test_eq(run1_match("a$", "im", "a"), "a"); // "a$"
    test_eq(run1_match("a$", "im", "a\n"), "a"); // "a$"
    test_eq(run1_match("a$", "im", "Za\n"), "a"); // "a$"
    test_eq(run1_match("a$", "im", "Za"), "a"); // "a$"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "foo\nbarbar"), "b"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "***Failers"), "a"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "rhubarb"), "b"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "barbell"), "b"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "abc\nbarton"), "a"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "foo\nbarbar"), "b"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "***Failers"), "a"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "rhubarb"), "b"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "barbell"), "b"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "abc\nbarton"), "a"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "abc"), "a"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "def\nabc"), "a"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "*** Failers"), "a"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(?!alphabet)[ab]", "i", "defabc"), "a"); // "(?!alphabet)[ab]"
    test_eq(run1_match("(a)bc(d)", "i", "abcd"), "abcd,a,d"); // "(a)bc(d)"
    test_eq(run1_match("(a)bc(d)", "i", "abcdC2"), "abcd,a,d"); // "(a)bc(d)"
    test_eq(run1_match("(a)bc(d)", "i", "abcdC5"), "abcd,a,d"); // "(a)bc(d)"
    test_eq(run1_match("(.{20})", "i", "abcdefghijklmnopqrstuvwxyz"), "abcdefghijklmnopqrst,abcdefghijklmnopqrst"); // "(.{20})"
    test_eq(run1_match("(.{20})", "i", "abcdefghijklmnopqrstuvwxyzC1"), "abcdefghijklmnopqrst,abcdefghijklmnopqrst"); // "(.{20})"
    test_eq(run1_match("(.{20})", "i", "abcdefghijklmnopqrstuvwxyzG1"), "abcdefghijklmnopqrst,abcdefghijklmnopqrst"); // "(.{20})"
    test_eq(run1_match("(.{15})", "i", "abcdefghijklmnopqrstuvwxyz"), "abcdefghijklmno,abcdefghijklmno"); // "(.{15})"
    test_eq(run1_match("(.{15})", "i", "abcdefghijklmnopqrstuvwxyzC1G1"), "abcdefghijklmno,abcdefghijklmno"); // "(.{15})"
    test_eq(run1_match("(.{16})", "i", "abcdefghijklmnopqrstuvwxyz"), "abcdefghijklmnop,abcdefghijklmnop"); // "(.{16})"
    test_eq(run1_match("(.{16})", "i", "abcdefghijklmnopqrstuvwxyzC1G1L"), "abcdefghijklmnop,abcdefghijklmnop"); // "(.{16})"
    test_eq(run1_match("^(a|(bc))de(f)", "i", "adefG1G2G3G4L"), "adef,a,,f"); // "^(a|(bc))de(f)"
    test_eq(run1_match("^(a|(bc))de(f)", "i", "bcdefG1G2G3G4L"), "bcdef,bc,bc,f"); // "^(a|(bc))de(f)"
    test_eq(run1_match("^(a|(bc))de(f)", "i", "adefghijkC0"), "adef,a,,f"); // "^(a|(bc))de(f)"
    // Skipping Unicode-unfriendly ^abc\00def
    test_eq(run1_match("\\Biss\\B", "i", "Mississippi"), "iss"); // "\\Biss\\B"
    test_eq(tc.compilef("iss", "i").run_global_match("Mississippi"), "iss,iss"); // "iss"
    test_eq(tc.compilef("\\Biss\\B", "i").run_global_match("Mississippi"), "iss,iss"); // "\\Biss\\B"
    // Skipping global "\\Biss\\B" with string "MississippiA"
    // Skipping global "\\Biss\\B" with string "Mississippi"
    test_eq(tc.compilef("^iss", "i").run_global_match("ississippi"), "iss"); // "^iss"
    test_eq(tc.compilef(".*iss", "i").run_global_match("abciss\nxyzisspqr"), "abciss,xyziss"); // ".*iss"
    test_eq(tc.compilef(".i.", "i").run_global_match("Mississippi"), "Mis,sis,sip"); // ".i."
    // Skipping Unicode-unfriendly .i.
    // Skipping Unicode-unfriendly .i.
    // Skipping Unicode-unfriendly .i.
    test_eq(tc.compilef("^.is", "i").run_global_match("Mississippi"), "Mis"); // "^.is"
    test_eq(tc.compilef("^ab\\n", "i").run_global_match("ab\nab\ncd"), "ab\n"); // "^ab\\n"
    test_eq(tc.compilef("^ab\\n", "im").run_global_match("ab\nab\ncd"), "ab\n,ab\n"); // "^ab\\n"
    test_eq(run1_match("a?b?", "i", "a"), "a"); // "a?b?"
    test_eq(run1_match("a?b?", "i", "b"), "b"); // "a?b?"
    test_eq(run1_match("a?b?", "i", "ab"), "ab"); // "a?b?"
    test_eq(run1_match("a?b?", "i", "\\"), ""); // "a?b?"
    test_eq(run1_match("a?b?", "i", "*** Failers"), ""); // "a?b?"
    test_eq(run1_match("a?b?", "i", "N"), ""); // "a?b?"
    test_eq(run1_match("|-", "i", "abcd"), ""); // "|-"
    test_eq(run1_match("|-", "i", "-abc"), ""); // "|-"
    test_eq(run1_match("|-", "i", "Nab-c"), ""); // "|-"
    test_eq(run1_match("|-", "i", "*** Failers"), ""); // "|-"
    test_eq(run1_match("|-", "i", "Nabc"), ""); // "|-"
    test_eq(run1_match("a*(b+)(z)(z)", "i", "aaaabbbbzzzz"), "aaaabbbbzz,bbbb,z,z"); // "a*(b+)(z)(z)"
    test_eq(run1_match("a*(b+)(z)(z)", "i", "aaaabbbbzzzzO0"), "aaaabbbbzz,bbbb,z,z"); // "a*(b+)(z)(z)"
    test_eq(run1_match("a*(b+)(z)(z)", "i", "aaaabbbbzzzzO1"), "aaaabbbbzz,bbbb,z,z"); // "a*(b+)(z)(z)"
    test_eq(run1_match("a*(b+)(z)(z)", "i", "aaaabbbbzzzzO2"), "aaaabbbbzz,bbbb,z,z"); // "a*(b+)(z)(z)"
    test_eq(run1_match("a*(b+)(z)(z)", "i", "aaaabbbbzzzzO3"), "aaaabbbbzz,bbbb,z,z"); // "a*(b+)(z)(z)"
    test_eq(run1_match("a*(b+)(z)(z)", "i", "aaaabbbbzzzzO4"), "aaaabbbbzz,bbbb,z,z"); // "a*(b+)(z)(z)"
    test_eq(run1_match("a*(b+)(z)(z)", "i", "aaaabbbbzzzzO5"), "aaaabbbbzz,bbbb,z,z"); // "a*(b+)(z)(z)"
    test_eq(run1_match("^.?abcd", "i", "(abcd)"), "(abcd"); // "^.?abcd"
    test_eq(run1_match("^.?abcd", "i", "(abcd)xyz"), "(abcd"); // "^.?abcd"
    test_eq(run1_match("^.?abcd", "i", "abcd"), "abcd"); // "^.?abcd"
    test_eq(run1_match("^.?abcd", "i", "abcd)"), "abcd"); // "^.?abcd"
    test_eq(run1_match("^.?abcd", "i", "(abcd"), "(abcd"); // "^.?abcd"
    test_eq(run1_match("^.?abcd", "i", "(abcd)"), "(abcd"); // "^.?abcd"
    test_eq(run1_match("^.?abcd", "i", "(abcd(xyz<p>qrs)123)"), "(abcd"); // "^.?abcd"
    test_eq(run1_match("(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\w+)\\s+(\\270)", "i", "O900 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 64 65 66 67 68 69 70 71 72 73 74 75 76 77 78 79 80 81 82 83 84 85 86 87 88 89 90 91 92 93 94 95 96 97 98 99 100 101 102 103 104 105 106 107 108 109 110 111 112 113 114 115 116 117 118 119 120 121 122 123 124 125 126 127 128 129 130 131 132 133 134 135 136 137 138 139 140 141 142 143 144 145 146 147 148 149 150 151 152 153 154 155 156 157 158 159 160 161 162 163 164 165 166 167 168 169 170 171 172 173 174 175 176 177 178 179 180 181 182 183 184 185 186 187 188 189 190 191 192 193 194 195 196 197 198 199 200 201 202 203 204 205 206 207 208 209 210 211 212 213 214 215 216 217 218 219 220 221 222 223 224 225 226 227 228 229 230 231 232 233 234 235 236 237 238 239 240 241 242 243 244 245 246 247 248 249 250 251 252 253 254 255 256 257 258 259 260 261 262 263 264 265 266 267 268 269 ABC ABC"), "1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 64 65 66 67 68 69 70 71 72 73 74 75 76 77 78 79 80 81 82 83 84 85 86 87 88 89 90 91 92 93 94 95 96 97 98 99 100 101 102 103 104 105 106 107 108 109 110 111 112 113 114 115 116 117 118 119 120 121 122 123 124 125 126 127 128 129 130 131 132 133 134 135 136 137 138 139 140 141 142 143 144 145 146 147 148 149 150 151 152 153 154 155 156 157 158 159 160 161 162 163 164 165 166 167 168 169 170 171 172 173 174 175 176 177 178 179 180 181 182 183 184 185 186 187 188 189 190 191 192 193 194 195 196 197 198 199 200 201 202 203 204 205 206 207 208 209 210 211 212 213 214 215 216 217 218 219 220 221 222 223 224 225 226 227 228 229 230 231 232 233 234 235 236 237 238 239 240 241 242 243 244 245 246 247 248 249 250 251 252 253 254 255 256 257 258 259 260 261 262 263 264 265 266 267 268 269 ABC ABC,1 ,2 ,3 ,4 ,5 ,6 ,7 ,8 ,9 ,10 ,11 ,12 ,13 ,14 ,15 ,16 ,17 ,18 ,19 ,20 ,21 ,22 ,23 ,24 ,25 ,26 ,27 ,28 ,29 ,30 ,31 ,32 ,33 ,34 ,35 ,36 ,37 ,38 ,39 ,40 ,41 ,42 ,43 ,44 ,45 ,46 ,47 ,48 ,49 ,50 ,51 ,52 ,53 ,54 ,55 ,56 ,57 ,58 ,59 ,60 ,61 ,62 ,63 ,64 ,65 ,66 ,67 ,68 ,69 ,70 ,71 ,72 ,73 ,74 ,75 ,76 ,77 ,78 ,79 ,80 ,81 ,82 ,83 ,84 ,85 ,86 ,87 ,88 ,89 ,90 ,91 ,92 ,93 ,94 ,95 ,96 ,97 ,98 ,99 ,100 ,101 ,102 ,103 ,104 ,105 ,106 ,107 ,108 ,109 ,110 ,111 ,112 ,113 ,114 ,115 ,116 ,117 ,118 ,119 ,120 ,121 ,122 ,123 ,124 ,125 ,126 ,127 ,128 ,129 ,130 ,131 ,132 ,133 ,134 ,135 ,136 ,137 ,138 ,139 ,140 ,141 ,142 ,143 ,144 ,145 ,146 ,147 ,148 ,149 ,150 ,151 ,152 ,153 ,154 ,155 ,156 ,157 ,158 ,159 ,160 ,161 ,162 ,163 ,164 ,165 ,166 ,167 ,168 ,169 ,170 ,171 ,172 ,173 ,174 ,175 ,176 ,177 ,178 ,179 ,180 ,181 ,182 ,183 ,184 ,185 ,186 ,187 ,188 ,189 ,190 ,191 ,192 ,193 ,194 ,195 ,196 ,197 ,198 ,199 ,200 ,201 ,202 ,203 ,204 ,205 ,206 ,207 ,208 ,209 ,210 ,211 ,212 ,213 ,214 ,215 ,216 ,217 ,218 ,219 ,220 ,221 ,222 ,223 ,224 ,225 ,226 ,227 ,228 ,229 ,230 ,231 ,232 ,233 ,234 ,235 ,236 ,237 ,238 ,239 ,240 ,241 ,242 ,243 ,244 ,245 ,246 ,247 ,248 ,249 ,250 ,251 ,252 ,253 ,254 ,255 ,256 ,257 ,258 ,259 ,260 ,261 ,262 ,263 ,264 ,265 ,266 ,267 ,268 ,269 ,ABC,ABC"); // "(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\d+(?:\\s|$))(\\w+)\\s+(\\270)"
    test_eq(run1_match("(main(O)?)+", "i", "mainmain"), "mainmain,main,"); // "(main(O)?)+"
    test_eq(run1_match("(main(O)?)+", "i", "mainOmain"), "mainOmain,main,"); // "(main(O)?)+"
    test_eq(run1_match("^(a(b)?)+$", "i", "aba"), "aba,a,"); // "^(a(b)?)+$"
    test_eq(run1_match("^(aa(bb)?)+$", "i", "aabbaa"), "aabbaa,aa,"); // "^(aa(bb)?)+$"
    test_eq(run1_match("^(aa|aa(bb))+$", "i", "aabbaa"), "aabbaa,aa,"); // "^(aa|aa(bb))+$"
    test_eq(run1_match("^(aa(bb)??)+$", "i", "aabbaa"), "aabbaa,aa,"); // "^(aa(bb)??)+$"
    test_eq(run1_match("^(?:aa(bb)?)+$", "i", "aabbaa"), "aabbaa,"); // "^(?:aa(bb)?)+$"
    test_eq(run1_match("^(aa(b(b))?)+$", "i", "aabbaa"), "aabbaa,aa,,"); // "^(aa(b(b))?)+$"
    test_eq(run1_match("^(?:aa(b(b))?)+$", "i", "aabbaa"), "aabbaa,,"); // "^(?:aa(b(b))?)+$"
    test_eq(run1_match("^(?:aa(b(?:b))?)+$", "i", "aabbaa"), "aabbaa,"); // "^(?:aa(b(?:b))?)+$"
    test_eq(run1_match("^(?:aa(bb(?:b))?)+$", "i", "aabbbaa"), "aabbbaa,"); // "^(?:aa(bb(?:b))?)+$"
    test_eq(run1_match("^(?:aa(b(?:bb))?)+$", "i", "aabbbaa"), "aabbbaa,"); // "^(?:aa(b(?:bb))?)+$"
    test_eq(run1_match("^(?:aa(?:b(b))?)+$", "i", "aabbaa"), "aabbaa,"); // "^(?:aa(?:b(b))?)+$"
    test_eq(run1_match("^(?:aa(?:b(bb))?)+$", "i", "aabbbaa"), "aabbbaa,"); // "^(?:aa(?:b(bb))?)+$"
    test_eq(run1_match("^(aa(b(bb))?)+$", "i", "aabbbaa"), "aabbbaa,aa,,"); // "^(aa(b(bb))?)+$"
    test_eq(run1_match("^(aa(bb(bb))?)+$", "i", "aabbbbaa"), "aabbbbaa,aa,,"); // "^(aa(bb(bb))?)+$"
    test_eq(run1_match("[\\S]", "", "ab"), "a"); // "[\\S]"
    test_eq(run1_match("[\\S]", "", "aB"), "a"); // "[\\S]"
    test_eq(run1_match("[\\S]", "", "*** Failers"), "*"); // "[\\S]"
    test_eq(run1_match("[\\S]", "", "AB"), "A"); // "[\\S]"
    test_eq(run1_match("[\\S]", "", "ab"), "a"); // "[\\S]"
    test_eq(run1_match("[\\S]", "", "aB"), "a"); // "[\\S]"
    test_eq(run1_match("[\\S]", "", "*** Failers"), "*"); // "[\\S]"
    test_eq(run1_match("[\\S]", "", "AB"), "A"); // "[\\S]"
    test_eq(run1_match("((.*))\\d+\\1", "i", "abc123bc"), "bc123bc,bc,bc"); // "((.*))\\d+\\1"
    test_eq(run1_match("c|abc", "i", "abcdef"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "1234abcdef"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abcxyz"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abcxyzf"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "123abcdef"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "1234abcdef"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abcdef"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "\u{83}x0abcdef"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "123abcdef"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "123abcdefC+"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "123abcdefC-"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "123abcdefC!1"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abcabcabc"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abcabcC!1!3"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abcabcabcC!1!3"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "123C+"), "C"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "123456C+"), "C"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "123456789C+"), "C"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "xyzabcC+"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "XxyzabcC+"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abcdefC+"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abcxyzC+"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abbbbbcccC*1"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abbbbbcccC*1"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "xbc"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abc"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "a(b)c"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "a(b(c))d"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "a(b(c)d"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "Satan, oscillate my metallic sonatas!"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "A man, a plan, a canal: Panama!"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "The quick brown fox"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "<abcd>"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "<abc <123> hij>"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "<abc <def> hij>"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "<abc<>def>"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "<abc<>"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "<abc"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "abcdefabc"), "abc"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "a=bc"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "a=bc"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "acde"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "Satan, oscillate my metallic sonatas!"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "A man, a plan, a canal: Panama!"), "c"); // "c|abc"
    test_eq(run1_match("c|abc", "i", "The quick brown fox"), "c"); // "c|abc"
    test_eq(run1_match("(a+)*zz", "i", "zzaaCZ"), "zz,"); // "(a+)*zz"
    test_eq(run1_match("(a+)*zz", "i", "zzaaCA"), "zz,"); // "(a+)*zz"
    test_eq(run1_match("((w\\/|-|with)*(free|immediate)*.*?shipping\\s*[!.-]*)", "i", " Baby Bjorn Active Carrier - With free SHIPPING!!"), " Baby Bjorn Active Carrier - With free SHIPPING!!, Baby Bjorn Active Carrier - With free SHIPPING!!,,"); // "((w\\/|-|with)*(free|immediate)*.*?shipping\\s*[!.-]*)"
    test_eq(run1_match("((w\\/|-|with)*(free|immediate)*.*?shipping\\s*[!.-]*)", "i", " Baby Bjorn Active Carrier - With free SHIPPING!!"), " Baby Bjorn Active Carrier - With free SHIPPING!!, Baby Bjorn Active Carrier - With free SHIPPING!!,,"); // "((w\\/|-|with)*(free|immediate)*.*?shipping\\s*[!.-]*)"
    test_eq(run1_match("([ab]{1,4}c|xy){4,5}?123", "i", "aacaacaacaacaac123"), "aacaacaacaacaac123,aac"); // "([ab]{1,4}c|xy){4,5}?123"
    test_eq(run1_match("abcde", "i", "abcdeP"), "abcde"); // "abcde"
    test_eq(run1_match("[abc]?123", "i", "123P"), "123"); // "[abc]?123"
    test_eq(run1_match("[abc]?123", "i", "c123P"), "c123"); // "[abc]?123"
    test_eq(run1_match("^(?:\\d){3,5}X", "i", "123X"), "123X"); // "^(?:\\d){3,5}X"
    test_eq(run1_match("^(?:\\d){3,5}X", "i", "1234X"), "1234X"); // "^(?:\\d){3,5}X"
    test_eq(run1_match("^(?:\\d){3,5}X", "i", "12345X"), "12345X"); // "^(?:\\d){3,5}X"
    test_eq(run1_match("line\\nbreak", "i", "this is a line\nbreak"), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "i", "line one\nthis is a line\nbreak in the second line"), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "i", "this is a line\nbreak"), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "i", "line one\nthis is a line\nbreak in the second line"), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "im", "this is a line\nbreak"), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "im", "line one\nthis is a line\nbreak in the second line"), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("ab.cd", "i", "ab-cd"), "ab-cd"); // "ab.cd"
    test_eq(run1_match("ab.cd", "i", "ab=cd"), "ab=cd"); // "ab.cd"
    test_eq(run1_match("ab.cd", "i", "ab-cd"), "ab-cd"); // "ab.cd"
    test_eq(run1_match("ab.cd", "i", "ab=cd"), "ab=cd"); // "ab.cd"
    test_eq(run1_match("a(b)c", "i", "abc"), "abc,b"); // "a(b)c"
    test_eq(run1_match("a(b)c", "i", "abc"), "abc,b"); // "a(b)c"
    test_eq(run1_match("\\s*,\\s*", "i", "\u{b},\u{b}"), "\u{b},\u{b}"); // "\\s*,\\s*"
    test_eq(run1_match("\\s*,\\s*", "i", "\u{c},\u{d}"), "\u{c},\u{d}"); // "\\s*,\\s*"
    test_eq(run1_match("^abc", "im", "xyz\nabc"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\nabc<lf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}\nabc<lf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}abc<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}\nabc<crlf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\nabc<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}\nabc<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\nabc<crlf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}abc<crlf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}abc<lf>"), "abc"); // "^abc"
    test_eq(run1_match("abc$", "im", "xyzabc"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\n"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\npqr"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\u{d}<cr>"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\u{d}pqr<cr>"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\u{d}\n<crlf>"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\u{d}\npqr<crlf>"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\u{d}"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\u{d}pqr"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\u{d}\n"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "im", "xyzabc\u{d}\npqr"), "abc"); // "abc$"
    test_eq(run1_match("^abc", "im", "xyz\u{d}abcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\nabcdef<lf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\nabcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\nabcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}abcdef<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}abcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}\nabcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}abcdef<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "im", "xyz\u{d}abcdef"), "abc"); // "^abc"
    test_eq(run1_match("abc", "i", "xyz\u{d}abc<bad>"), "abc"); // "abc"
    test_eq(run1_match("abc", "i", "abc"), "abc"); // "abc"
    test_eq(run1_match(".*", "i", "abc\ndef"), "abc"); // ".*"
    test_eq(run1_match(".*", "i", "abc\u{d}def"), "abc"); // ".*"
    test_eq(run1_match(".*", "i", "abc\u{d}\ndef"), "abc"); // ".*"
    test_eq(run1_match(".*", "i", "<cr>abc\ndef"), "<cr>abc"); // ".*"
    test_eq(run1_match(".*", "i", "<cr>abc\u{d}def"), "<cr>abc"); // ".*"
    test_eq(run1_match(".*", "i", "<cr>abc\u{d}\ndef"), "<cr>abc"); // ".*"
    test_eq(run1_match(".*", "i", "<crlf>abc\ndef"), "<crlf>abc"); // ".*"
    test_eq(run1_match(".*", "i", "<crlf>abc\u{d}def"), "<crlf>abc"); // ".*"
    test_eq(run1_match(".*", "i", "<crlf>abc\u{d}\ndef"), "<crlf>abc"); // ".*"
    test_eq(run1_match("()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()(.(.))", "i", "XYO400"), "XY,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,XY,Y"); // "()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()()(.(.))"
    test_eq(run1_match("^a+A\\d", "", "aaaA5"), "aaaA5"); // "^a+A\\d"
    test_eq(run1_match("^a*A\\d", "i", "aaaA5"), "aaaA5"); // "^a*A\\d"
    test_eq(run1_match("^a*A\\d", "i", "aaaa5"), "aaaa5"); // "^a*A\\d"
    test_eq(run1_match("a*[^a]", "", "xyCabcCxyz"), "x"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "xyCabcCxyz"), "x"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "bXaX"), "b"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "bXbX"), "b"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "** Failers"), "*"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "aXaX"), "aX"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "aXbX"), "aX"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "xx"), "x"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "xy"), "x"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "yy"), "y"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "yx"), "y"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "xx"), "x"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "xy"), "x"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "yy"), "y"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "yx"), "y"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "bxay"), "b"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "bxby"), "b"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "** Failers"), "*"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "axby"), "ax"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "XxXxxx"), "X"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "XxXyyx"), "X"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "XxXyxx"), "X"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "** Failers"), "*"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "x"), "x"); // "a*[^a]"
    test_eq(run1_match("a*[^a]", "", "abcabc"), "ab"); // "a*[^a]"
    test_eq(run1_match("^(?:(?:\\1|X)(a|b))+", "", "Xaaa"), "Xaaa,a"); // "^(?:(?:\\1|X)(a|b))+"
    test_eq(run1_match("^(?:(?:\\1|X)(a|b))+", "", "Xaba"), "Xaba,a"); // "^(?:(?:\\1|X)(a|b))+"
    test_eq(run1_match("(?=(\\w+))\\1:", "i", "abcd:"), "abcd:,abcd"); // "(?=(\\w+))\\1:"
    test_eq(run1_match("(?=(\\w+))\\1:", "i", "abcd:"), "abcd:,abcd"); // "(?=(\\w+))\\1:"
    test_eq(run1_match("(?=(\\w+))\\1:", "i", "a:aaxyz"), "a:,a"); // "(?=(\\w+))\\1:"
    test_eq(run1_match("(?=(\\w+))\\1:", "i", "ab:ababxyz"), "ab:,ab"); // "(?=(\\w+))\\1:"
    test_eq(run1_match("(?=(\\w+))\\1:", "i", "a:axyz"), "a:,a"); // "(?=(\\w+))\\1:"
    test_eq(run1_match("(?=(\\w+))\\1:", "i", "ab:abxyz"), "ab:,ab"); // "(?=(\\w+))\\1:"
    test_eq(run1_match("^a.b", "", "a\u{85}b<anycrlf> "), "a\u{85}b"); // "^a.b"
    test_eq(run1_match("^a.b", "", "a\u{85}b<any> "), "a\u{85}b"); // "^a.b"
    test_eq(run1_match("^abc.", "m", "abc1 \nabc2 \u{b}abc3xx \u{c}abc4 \u{d}abc5xx \u{d}\nabc6 \u{85}abc7 JUNK"), "abc1"); // "^abc."
    test_eq(run1_match("abc.$", "m", "abc1\n abc2\u{b} abc3\u{c} abc4\u{d} abc5\u{d}\n abc6\u{85} abc7 abc9"), "abc1"); // "abc.$"
    // Skipping Unicode-unfriendly ^a\R*b
    // Skipping Unicode-unfriendly ^a[\R]b
    test_eq(run1_match(".+foo", "", "afoo"), "afoo"); // ".+foo"
    test_eq(run1_match(".+foo", "", "afoo"), "afoo"); // ".+foo"
    test_eq(run1_match(".+foo", "", "afoo"), "afoo"); // ".+foo"
    test_eq(run1_match(".+foo", "", "afoo"), "afoo"); // ".+foo"
    test_eq(run1_match("^$", "m", "abc\u{d}\u{d}xyz"), ""); // "^$"
    // Skipping global "^$" with string "abc\n\u{d}xyz  "
    // Skipping global "^$" with string "abc\u{d}\nxyz"
    // Skipping global "^$" with string "abc\u{d}\n\u{d}\n"
    // Skipping global "^$" with string "abc\u{d}\n\u{d}\n"
    // Skipping global "^$" with string "abc\u{d}\n\u{d}\n"
    test_eq(run1_match("abc.$", "m", "abc1\n abc2\u{b} abc3\u{c} abc4\u{d} abc5\u{d}\n abc6\u{85} abc9"), "abc1"); // "abc.$"
    test_eq(run1_match("^X", "m", "XABC"), "X"); // "^X"
    test_eq(run1_match("^X", "m", "XABCB"), "X"); // "^X"
    test_eq(run1_match("^X", "m", "XabcXabc  "), "X"); // "^X"
    // Skipping Unicode-unfriendly (foo)(\Kbar|baz)
    test_eq(run1_match("\\nA", "", "\u{d}\nA "), "\nA"); // "\\nA"
    test_eq(run1_match("[\\r\\n]A", "", "\u{d}\nA "), "\nA"); // "[\\r\\n]A"
    test_eq(run1_match("(\\r|\\n)A", "", "\u{d}\nA "), "\nA,\n"); // "(\\r|\\n)A"
    // Skipping Unicode-unfriendly ^(a|b\g<1>c)
    // Skipping Unicode-unfriendly ^(a|b\g'1'c)
    // Skipping Unicode-unfriendly ^(a|b\g'-1'c)
    // Skipping Unicode-unfriendly (^(a|b\g<-1>c))
    test_eq(run1_match("(\\3)(\\1)(a)", "", "cat"), "a,,,a"); // "(\\3)(\\1)(a)"
    test_eq(run1_match("(\\3)(\\1)(a)", "", "cat"), "a,,,a"); // "(\\3)(\\1)(a)"
    // Skipping Unicode-unfriendly TA]
    // Skipping Unicode-unfriendly TA]
    test_eq(run1_match("a[^]b", "", "aXb"), "aXb"); // "a[^]b"
    test_eq(run1_match("a[^]b", "", "a\nb "), "a\nb"); // "a[^]b"
    test_eq(run1_match("a[^]+b", "", "aXb"), "aXb"); // "a[^]+b"
    test_eq(run1_match("a[^]+b", "", "a\nX\nXb "), "a\nX\nXb"); // "a[^]+b"
    test_eq(run1_match("a.b", "", "acb"), "acb"); // "a.b"
    test_eq(run1_match("a.b", "", "a\u{7f}b"), "a\u{7f}b"); // "a.b"
    test_eq(run1_match("a(.*?)(.)", "", "a\u{c0}\u{88}b"), "a\u{c0},,\u{c0}"); // "a(.*?)(.)"
    test_eq(run1_match("a(.*?)(.)", "", "ax{100}b"), "ax,,x"); // "a(.*?)(.)"
    test_eq(run1_match("a(.*)(.)", "", "a\u{c0}\u{88}b"), "a\u{c0}\u{88}b,\u{c0}\u{88},b"); // "a(.*)(.)"
    test_eq(run1_match("a(.*)(.)", "", "ax{100}b"), "ax{100}b,x{100},b"); // "a(.*)(.)"
    test_eq(run1_match("a(.)(.)", "", "a\u{c0}\u{92}bcd"), "a\u{c0}\u{92},\u{c0},\u{92}"); // "a(.)(.)"
    test_eq(run1_match("a(.)(.)", "", "ax{240}bcd"), "ax{,x,{"); // "a(.)(.)"
    test_eq(run1_match("a(.?)(.)", "", "a\u{c0}\u{92}bcd"), "a\u{c0}\u{92},\u{c0},\u{92}"); // "a(.?)(.)"
    test_eq(run1_match("a(.?)(.)", "", "ax{240}bcd"), "ax{,x,{"); // "a(.?)(.)"
    test_eq(run1_match("a(.??)(.)", "", "a\u{c0}\u{92}bcd"), "a\u{c0},,\u{c0}"); // "a(.??)(.)"
    test_eq(run1_match("a(.??)(.)", "", "ax{240}bcd"), "ax,,x"); // "a(.??)(.)"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}xyb "), "ax{1234}xyb,x{1234}xy"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}x{4321}yb "), "ax{1234}x{4321}yb,x{1234}x{4321}y"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}x{4321}x{3412}b "), "ax{1234}x{4321}x{3412}b,x{1234}x{4321}x{3412}"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "axxxxbcdefghijb "), "axxxxbcdefghijb,xxxxbcdefghij"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}x{4321}x{3412}x{3421}b "), "ax{1234}x{4321}x{3412}x{3421}b,x{1234}x{4321}x{3412}x{3421}"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}b "), "ax{1234}b,x{1234}"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}xyb "), "ax{1234}xyb,x{1234}xy"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}x{4321}yb "), "ax{1234}x{4321}yb,x{1234}x{4321}y"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}x{4321}x{3412}b "), "ax{1234}x{4321}x{3412}b,x{1234}x{4321}x{3412}"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "axxxxbcdefghijb "), "axxxxb,xxxx"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}x{4321}x{3412}x{3421}b "), "ax{1234}x{4321}x{3412}x{3421}b,x{1234}x{4321}x{3412}x{3421}"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}b "), "ax{1234}b,x{1234}"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,5})b", "", "axxxxbcdefghijb "), "axxxxb,xxxx"); // "a(.{3,5})b"
    test_eq(run1_match("a(.{3,5})b", "", "axbxxbcdefghijb "), "axbxxb,xbxx"); // "a(.{3,5})b"
    test_eq(run1_match("a(.{3,5})b", "", "axxxxxbcdefghijb "), "axxxxxb,xxxxx"); // "a(.{3,5})b"
    test_eq(run1_match("a(.{3,5}?)b", "", "axxxxbcdefghijb "), "axxxxb,xxxx"); // "a(.{3,5}?)b"
    test_eq(run1_match("a(.{3,5}?)b", "", "axbxxbcdefghijb "), "axbxxb,xbxx"); // "a(.{3,5}?)b"
    test_eq(run1_match("a(.{3,5}?)b", "", "axxxxxbcdefghijb "), "axxxxxb,xxxxx"); // "a(.{3,5}?)b"
    // Skipping Unicode-unfriendly X\C*
    // Skipping Unicode-unfriendly X\C*?
    test_eq(tc.compile("[^a]+").run_global_match("bcd"), "bcd"); // "[^a]+"
    // Skipping Unicode-unfriendly [^a]+
    test_eq(run1_match("^[^a]{2}", "", "x{100}bc"), "x{"); // "^[^a]{2}"
    test_eq(run1_match("^[^a]{2,}", "", "x{100}bcAa"), "x{100}bcA"); // "^[^a]{2,}"
    test_eq(run1_match("^[^a]{2,}?", "", "x{100}bca"), "x{"); // "^[^a]{2,}?"
    test_eq(tc.compilef("[^a]+", "i").run_global_match("bcd"), "bcd"); // "[^a]+"
    // Skipping Unicode-unfriendly [^a]+
    test_eq(run1_match("^[^a]{2}", "i", "x{100}bc"), "x{"); // "^[^a]{2}"
    test_eq(run1_match("^[^a]{2,}", "i", "x{100}bcAa"), "x{100}bc"); // "^[^a]{2,}"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}bca"), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}x{100} "), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}x{100} "), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}x{100}x{100}x{100} "), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}x{100}x{100}x{100} "), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "Xyyyax{100}x{100}bXzzz"), "Xy"); // "^[^a]{2,}?"
    test_eq(run1_match("\\D", "", "1X2"), "X"); // "\\D"
    test_eq(run1_match("\\D", "", "1x{100}2 "), "x"); // "\\D"
    test_eq(run1_match(">\\S", "", "> >X Y"), ">X"); // ">\\S"
    test_eq(run1_match(">\\S", "", "> >x{100} Y"), ">x"); // ">\\S"
    test_eq(run1_match("\\d", "", "x{100}3"), "1"); // "\\d"
    test_eq(run1_match("\\s", "", "x{100} X"), " "); // "\\s"
    test_eq(run1_match("\\D+", "", "12abcd34"), "abcd"); // "\\D+"
    test_eq(run1_match("\\D+", "", "*** Failers"), "*** Failers"); // "\\D+"
    test_eq(run1_match("\\D+", "", "1234  "), "  "); // "\\D+"
    test_eq(run1_match("\\D{2,3}", "", "12abcd34"), "abc"); // "\\D{2,3}"
    test_eq(run1_match("\\D{2,3}", "", "12ab34"), "ab"); // "\\D{2,3}"
    test_eq(run1_match("\\D{2,3}", "", "*** Failers  "), "***"); // "\\D{2,3}"
    test_eq(run1_match("\\D{2,3}", "", "12a34  "), "  "); // "\\D{2,3}"
    test_eq(run1_match("\\D{2,3}?", "", "12abcd34"), "ab"); // "\\D{2,3}?"
    test_eq(run1_match("\\D{2,3}?", "", "12ab34"), "ab"); // "\\D{2,3}?"
    test_eq(run1_match("\\D{2,3}?", "", "*** Failers  "), "**"); // "\\D{2,3}?"
    test_eq(run1_match("\\D{2,3}?", "", "12a34  "), "  "); // "\\D{2,3}?"
    test_eq(run1_match("\\d+", "", "12abcd34"), "12"); // "\\d+"
    test_eq(run1_match("\\d{2,3}", "", "12abcd34"), "12"); // "\\d{2,3}"
    test_eq(run1_match("\\d{2,3}", "", "1234abcd"), "123"); // "\\d{2,3}"
    test_eq(run1_match("\\d{2,3}?", "", "12abcd34"), "12"); // "\\d{2,3}?"
    test_eq(run1_match("\\d{2,3}?", "", "1234abcd"), "12"); // "\\d{2,3}?"
    test_eq(run1_match("\\S+", "", "12abcd34"), "12abcd34"); // "\\S+"
    test_eq(run1_match("\\S+", "", "*** Failers"), "***"); // "\\S+"
    test_eq(run1_match("\\S{2,3}", "", "12abcd34"), "12a"); // "\\S{2,3}"
    test_eq(run1_match("\\S{2,3}", "", "1234abcd"), "123"); // "\\S{2,3}"
    test_eq(run1_match("\\S{2,3}", "", "*** Failers"), "***"); // "\\S{2,3}"
    test_eq(run1_match("\\S{2,3}?", "", "12abcd34"), "12"); // "\\S{2,3}?"
    test_eq(run1_match("\\S{2,3}?", "", "1234abcd"), "12"); // "\\S{2,3}?"
    test_eq(run1_match("\\S{2,3}?", "", "*** Failers"), "**"); // "\\S{2,3}?"
    test_eq(run1_match(">\\s+<", "", "12>      <34"), ">      <"); // ">\\s+<"
    test_eq(run1_match(">\\s{2,3}<", "", "ab>  <cd"), ">  <"); // ">\\s{2,3}<"
    test_eq(run1_match(">\\s{2,3}<", "", "ab>   <ce"), ">   <"); // ">\\s{2,3}<"
    test_eq(run1_match(">\\s{2,3}?<", "", "ab>  <cd"), ">  <"); // ">\\s{2,3}?<"
    test_eq(run1_match(">\\s{2,3}?<", "", "ab>   <ce"), ">   <"); // ">\\s{2,3}?<"
    test_eq(run1_match("\\w+", "", "12      34"), "12"); // "\\w+"
    test_eq(run1_match("\\w+", "", "*** Failers"), "Failers"); // "\\w+"
    test_eq(run1_match("\\w{2,3}", "", "ab  cd"), "ab"); // "\\w{2,3}"
    test_eq(run1_match("\\w{2,3}", "", "abcd ce"), "abc"); // "\\w{2,3}"
    test_eq(run1_match("\\w{2,3}", "", "*** Failers"), "Fai"); // "\\w{2,3}"
    test_eq(run1_match("\\w{2,3}?", "", "ab  cd"), "ab"); // "\\w{2,3}?"
    test_eq(run1_match("\\w{2,3}?", "", "abcd ce"), "ab"); // "\\w{2,3}?"
    test_eq(run1_match("\\w{2,3}?", "", "*** Failers"), "Fa"); // "\\w{2,3}?"
    test_eq(run1_match("\\W+", "", "12====34"), "===="); // "\\W+"
    test_eq(run1_match("\\W+", "", "*** Failers"), "*** "); // "\\W+"
    test_eq(run1_match("\\W+", "", "abcd "), " "); // "\\W+"
    test_eq(run1_match("\\W{2,3}", "", "ab====cd"), "==="); // "\\W{2,3}"
    test_eq(run1_match("\\W{2,3}", "", "ab==cd"), "=="); // "\\W{2,3}"
    test_eq(run1_match("\\W{2,3}", "", "*** Failers"), "***"); // "\\W{2,3}"
    test_eq(run1_match("\\W{2,3}?", "", "ab====cd"), "=="); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "ab==cd"), "=="); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers "), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers "), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "X  "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "X  "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "X  "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "x{200}X   "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "x{200}X   "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "x{200}X   "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("[\\xFF]", "", ">\u{ff}<"), "\u{ff}"); // "[\\xFF]"
    test_eq(run1_match("[^\\xFF]", "", "XYZ"), "X"); // "[^\\xFF]"
    test_eq(run1_match("[^\\xff]", "", "XYZ"), "X"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "x{123} "), "x"); // "[^\\xff]"
    test_eq(run1_match("(|a)", "", "catac"), ","); // "(|a)"
    // Skipping global "(|a)" with string "ax{256}a "
    // Skipping global "(|a)" with string "x{85}"
    // Skipping global "(|a)" with string "\u{1234} "
    // Skipping global "(|a)" with string "\u{1234} "
    // Skipping global "(|a)" with string "abcdefg"
    // Skipping global "(|a)" with string "ab"
    // Skipping global "(|a)" with string "a "
    test_eq(run1_match("\\S\\S", "", "Ax{a3}BC"), "Ax"); // "\\S\\S"
    test_eq(run1_match("\\S{2}", "", "Ax{a3}BC"), "Ax"); // "\\S{2}"
    test_eq(run1_match("\\W\\W", "", "+x{a3}== "), "}="); // "\\W\\W"
    test_eq(run1_match("\\W{2}", "", "+x{a3}== "), "}="); // "\\W{2}"
    test_eq(run1_match("\\S", "", "x{442}x{435}x{441}x{442}"), "x"); // "\\S"
    test_eq(run1_match("[\\S]", "", "x{442}x{435}x{441}x{442}"), "x"); // "[\\S]"
    test_eq(run1_match("\\D", "", "x{442}x{435}x{441}x{442}"), "x"); // "\\D"
    test_eq(run1_match("[\\D]", "", "x{442}x{435}x{441}x{442}"), "x"); // "[\\D]"
    test_eq(run1_match("\\W", "", "x{2442}x{2435}x{2441}x{2442}"), "{"); // "\\W"
    test_eq(run1_match("[\\W]", "", "x{2442}x{2435}x{2441}x{2442}"), "{"); // "[\\W]"
    test_eq(run1_match("[\\S\\s]*", "", "abc\n\u{d}x{442}x{435}x{441}x{442}xyz "), "abc\n\u{d}x{442}x{435}x{441}x{442}xyz "); // "[\\S\\s]*"
    test_eq(run1_match("[\\S\\s]*", "", "x{442}x{435}x{441}x{442}"), "x{442}x{435}x{441}x{442}"); // "[\\S\\s]*"
    test_eq(run1_match(".[^\\S].", "", "abc defx{442}x{443}xyz\npqr"), "c d"); // ".[^\\S]."
    test_eq(run1_match(".[^\\S\\n].", "", "abc defx{442}x{443}xyz\npqr"), "c d"); // ".[^\\S\\n]."
    test_eq(run1_match("^[^d]*?$", "", "abc"), "abc"); // "^[^d]*?$"
    test_eq(run1_match("^[^d]*?$", "", "abc"), "abc"); // "^[^d]*?$"
    test_eq(run1_match("^[^d]*?$", "i", "abc"), "abc"); // "^[^d]*?$"
    test_eq(run1_match("^[^d]*?$", "i", "abc"), "abc"); // "^[^d]*?$"
    test_eq(run1_match(".{3,5}X", "", "x{212ab}x{212ab}x{212ab}x{861}X"), "{861}X"); // ".{3,5}X"
    test_eq(run1_match(".{3,5}?", "", "x{212ab}x{212ab}x{212ab}x{861}"), "x{2"); // ".{3,5}?"
    test_eq(run1_match(".{3,5}?", "", "x{c0}b"), "x{c"); // ".{3,5}?"
    test_eq(run1_match(".{3,5}?", "", "ax{c0}aaaa/ "), "ax{"); // ".{3,5}?"
    test_eq(run1_match(".{3,5}?", "", "ax{c0}aaaa/ "), "ax{"); // ".{3,5}?"
    test_eq(run1_match(".{3,5}?", "", "ax{c0}ax{c0}aaa/ "), "ax{"); // ".{3,5}?"
    test_eq(run1_match(".{3,5}?", "", "ax{c0}aaaa/ "), "ax{"); // ".{3,5}?"
    test_eq(run1_match(".{3,5}?", "", "ax{c0}ax{c0}aaa/ "), "ax{"); // ".{3,5}?"
    test_eq(run1_match(".{3,5}?", "", "ax{c0}aaaa/ "), "ax{"); // ".{3,5}?"
    test_eq(run1_match(".{3,5}?", "", "ax{c0}ax{c0}aaa/ "), "ax{"); // ".{3,5}?"
    test_eq(run1_match(".{3,5}?", "", "Should produce an error diagnostic"), "Sho"); // ".{3,5}?"
    test_eq(run1_match("^[ab]", "", "bar"), "b"); // "^[ab]"
    test_eq(run1_match("^[^ab]", "", "c"), "c"); // "^[^ab]"
    test_eq(run1_match("^[^ab]", "", "x{ff}"), "x"); // "^[^ab]"
    test_eq(run1_match("^[^ab]", "", "x{100}  "), "x"); // "^[^ab]"
    test_eq(run1_match("^[^ab]", "", "*** Failers "), "*"); // "^[^ab]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{f1}"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{bf}"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{1000}   "), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "*** Failers"), "*"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{c0} "), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{f0} "), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "1234"), "1"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "\"1234\" "), "\""); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}1234"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "\"x{100}1234\"  "), "\""); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}x{100}12ab "), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}x{100}\"12\" "), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "*** Failers "), "*"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}x{100}abcd"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "A"), "A"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "Zx{100}"), "Z"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}Z"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "*** Failers "), "*"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "Zx{100}"), "Z"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}Z"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "*** Failers "), "*"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{104}"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "*** Failers"), "*"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{105}"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{ff}    "), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "x{100}"), "x"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[^ab\\xC0-\\xF0]", "", "\u{100} "), "\u{100}"); // "[^ab\\xC0-\\xF0]"
    test_eq(run1_match("[\\xFF]", "", ">\u{ff}<"), "\u{ff}"); // "[\\xFF]"
    test_eq(run1_match("[^\\xff]", "", "\u{d6} # Matches without Study"), "\u{d6}"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "x{d6}"), "x"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "\u{d6} <-- Same with Study"), "\u{d6}"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "x{d6}"), "x"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "\u{d6} # Matches without Study"), "\u{d6}"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "x{d6} "), "x"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "\u{d6} <-- Same with Study"), "\u{d6}"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "x{d6} "), "x"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "\u{fffd}]"), "\u{fffd}"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "\u{fffd}"), "\u{fffd}"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "\u{fffd}\u{fffd}\u{fffd}"), "\u{fffd}"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "\u{fffd}\u{fffd}\u{fffd}?"), "\u{fffd}"); // "[^\\xff]"
    test_eq(run1_match("\\W", "", "A.B"), "."); // "\\W"
    test_eq(run1_match("\\W", "", "Ax{100}B "), "{"); // "\\W"
    test_eq(run1_match("\\w", "", "x{100}X   "), "x"); // "\\w"
    test_eq(run1_match("\\w", "", "ax{1234}b"), "a"); // "\\w"
    test_eq(run1_match("^abc.", "m", "abc1 \nabc2 \u{b}abc3xx \u{c}abc4 \u{d}abc5xx \u{d}\nabc6 x{0085}abc7 x{2028}abc8 x{2029}abc9 JUNK"), "abc1"); // "^abc."
    test_eq(run1_match("abc.$", "m", "abc1\n abc2\u{b} abc3\u{c} abc4\u{d} abc5\u{d}\n abc6x{0085} abc7x{2028} abc8x{2029} abc9"), "abc1"); // "abc.$"
    // Skipping Unicode-unfriendly ^a\R*b
    test_eq(run1_match(".*$", "", "x{1ec5} "), "x{1ec5} "); // ".*$"
    test_eq(run1_match(".*a.*=.b.*", "", "QQQx{2029}ABCaXYZ=!bPQR"), "QQQx{2029}ABCaXYZ=!bPQR"); // ".*a.*=.b.*"
    test_eq(run1_match("a[^]b", "", "a\nb "), "a\nb"); // "a[^]b"
    test_eq(run1_match("a[^]+b", "", "aXb"), "aXb"); // "a[^]+b"
    test_eq(run1_match("a[^]+b", "", "a\nX\nXx{1234}b "), "a\nX\nXx{1234}b"); // "a[^]+b"
    test_eq(run1_match("X", "", "Ax{1ec5}ABCXYZ"), "X"); // "X"
    // Skipping Unicode-unfriendly [\p{Nd}]
    // Skipping Unicode-unfriendly [\p{Nd}]
    // Skipping Unicode-unfriendly [\P{Nd}]+
    test_eq(run1_match("\\D+", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "\\D+"
    test_eq(run1_match("\\D+", "", " "), " "); // "\\D+"
    test_eq(run1_match("\\D+", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "\\D+"
    test_eq(run1_match("[\\D]+", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "[\\D]+"
    test_eq(run1_match("[\\D\\P{Nd}]+", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "[\\D\\P{Nd}]+"
    // Skipping Unicode-unfriendly ^[\X]
    // Skipping Unicode-unfriendly ^(\X*)(.)
    // Skipping Unicode-unfriendly ^(\X*)(.)
    // Skipping Unicode-unfriendly ^(\X*?)(.)
    // Skipping Unicode-unfriendly ^(\X*?)(.)
    test_eq(run1_match("^[\\p{Any}]X", "", "AXYZ"), "AX"); // "^[\\p{Any}]X"
    // Skipping Unicode-unfriendly ^[\P{Any}]X
    test_eq(run1_match("^[\\p{Any}]?X", "", "XYZ"), "X"); // "^[\\p{Any}]?X"
    test_eq(run1_match("^[\\p{Any}]?X", "", "AXYZ"), "AX"); // "^[\\p{Any}]?X"
    test_eq(run1_match("^[\\P{Any}]?X", "", "XYZ"), "X"); // "^[\\P{Any}]?X"
    // Skipping Unicode-unfriendly ^[\P{Any}]?X
    test_eq(run1_match("^[\\p{Any}]+X", "", "AXYZ"), "AX"); // "^[\\p{Any}]+X"
    // Skipping Unicode-unfriendly ^[\P{Any}]+X
    test_eq(run1_match("^[\\p{Any}]*X", "", "XYZ"), "X"); // "^[\\p{Any}]*X"
    test_eq(run1_match("^[\\p{Any}]*X", "", "AXYZ"), "AX"); // "^[\\p{Any}]*X"
    test_eq(run1_match("^[\\P{Any}]*X", "", "XYZ"), "X"); // "^[\\P{Any}]*X"
    // Skipping Unicode-unfriendly ^[\P{Any}]*X
    // Skipping Unicode-unfriendly ([\pL]=(abc))*X
    test_eq(run1_match("(A)\\1", "i", "AA"), "AA,A"); // "(A)\\1"
    test_eq(run1_match("(A)\\1", "i", "Aa"), "Aa,A"); // "(A)\\1"
    test_eq(run1_match("(A)\\1", "i", "aa"), "aa,a"); // "(A)\\1"
    test_eq(run1_match("(A)\\1", "i", "aA"), "aA,a"); // "(A)\\1"
    test_eq(run1_match("abc", "", "abc"), "abc"); // "abc"
    test_eq(run1_match("ab*c", "", "abc"), "abc"); // "ab*c"
    test_eq(run1_match("ab*c", "", "abbbbc"), "abbbbc"); // "ab*c"
    test_eq(run1_match("ab*c", "", "ac"), "ac"); // "ab*c"
    test_eq(run1_match("ab+c", "", "abc"), "abc"); // "ab+c"
    test_eq(run1_match("ab+c", "", "abbbbbbc"), "abbbbbbc"); // "ab+c"
    test_eq(run1_match("a*", "", "a"), "a"); // "a*"
    test_eq(run1_match("a*", "", "aaaaaaaaaaaaaaaaa"), "aaaaaaaaaaaaaaaaa"); // "a*"
    test_eq(run1_match("a*", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaa "), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "a*"
    test_eq(run1_match("a*", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaF "), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "a*"
    test_eq(run1_match("(a|abcd|african)", "", "a"), "a,a"); // "(a|abcd|african)"
    test_eq(run1_match("(a|abcd|african)", "", "abcd"), "a,a"); // "(a|abcd|african)"
    test_eq(run1_match("(a|abcd|african)", "", "african"), "a,a"); // "(a|abcd|african)"
    test_eq(run1_match("^abc", "", "abcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "abcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\nabc    "), "abc"); // "^abc"
    test_eq(run1_match("x\\dy\\Dz", "", "x9yzz"), "x9yzz"); // "x\\dy\\Dz"
    test_eq(run1_match("x\\dy\\Dz", "", "x0y+z"), "x0y+z"); // "x\\dy\\Dz"
    test_eq(run1_match("x\\sy\\Sz", "", "x yzz"), "x yzz"); // "x\\sy\\Sz"
    test_eq(run1_match("x\\sy\\Sz", "", "x y+z"), "x y+z"); // "x\\sy\\Sz"
    test_eq(run1_match("x\\wy\\Wz", "", "xxy+z"), "xxy+z"); // "x\\wy\\Wz"
    test_eq(run1_match("x.y", "", "x+y"), "x+y"); // "x.y"
    test_eq(run1_match("x.y", "", "x-y"), "x-y"); // "x.y"
    test_eq(run1_match("x.y", "", "x+y"), "x+y"); // "x.y"
    test_eq(run1_match("x.y", "", "x-y"), "x-y"); // "x.y"
    test_eq(run1_match("a\\d$", "", "ba0"), "a0"); // "a\\d$"
    test_eq(run1_match("a\\d$", "m", "ba0"), "a0"); // "a\\d$"
    test_eq(run1_match("a\\d$", "m", "ba0\n"), "a0"); // "a\\d$"
    test_eq(run1_match("a\\d$", "m", "ba0\ncd   "), "a0"); // "a\\d$"
    test_eq(run1_match("abc", "i", "abc"), "abc"); // "abc"
    test_eq(run1_match("abc", "i", "aBc"), "aBc"); // "abc"
    test_eq(run1_match("abc", "i", "ABC"), "ABC"); // "abc"
    test_eq(run1_match("[^a]", "", "abcd"), "b"); // "[^a]"
    test_eq(run1_match("ab?\\w", "", "abz"), "abz"); // "ab?\\w"
    test_eq(run1_match("ab?\\w", "", "abbz"), "abb"); // "ab?\\w"
    test_eq(run1_match("ab?\\w", "", "azz  "), "az"); // "ab?\\w"
    test_eq(run1_match("x{0,3}yz", "", "ayzq"), "yz"); // "x{0,3}yz"
    test_eq(run1_match("x{0,3}yz", "", "axyzq"), "xyz"); // "x{0,3}yz"
    test_eq(run1_match("x{0,3}yz", "", "axxyz"), "xxyz"); // "x{0,3}yz"
    test_eq(run1_match("x{0,3}yz", "", "axxxyzq"), "xxxyz"); // "x{0,3}yz"
    test_eq(run1_match("x{0,3}yz", "", "axxxxyzq"), "xxxyz"); // "x{0,3}yz"
    test_eq(run1_match("x{3}yz", "", "axxxyzq"), "xxxyz"); // "x{3}yz"
    test_eq(run1_match("x{3}yz", "", "axxxxyzq"), "xxxyz"); // "x{3}yz"
    test_eq(run1_match("x{2,3}yz", "", "axxyz"), "xxyz"); // "x{2,3}yz"
    test_eq(run1_match("x{2,3}yz", "", "axxxyzq"), "xxxyz"); // "x{2,3}yz"
    test_eq(run1_match("x{2,3}yz", "", "axxxxyzq"), "xxxyz"); // "x{2,3}yz"
    test_eq(run1_match("[^a]+", "", "bac"), "b"); // "[^a]+"
    test_eq(run1_match("[^a]+", "", "bcdefax"), "bcdef"); // "[^a]+"
    test_eq(run1_match("[^a]+", "", "*** Failers"), "*** F"); // "[^a]+"
    test_eq(run1_match("[^a]+", "", "aaaaa   "), "   "); // "[^a]+"
    test_eq(run1_match("[^a]*", "", "bac"), "b"); // "[^a]*"
    test_eq(run1_match("[^a]*", "", "bcdefax"), "bcdef"); // "[^a]*"
    test_eq(run1_match("[^a]*", "", "*** Failers"), "*** F"); // "[^a]*"
    test_eq(run1_match("[^a]*", "", "aaaaa   "), ""); // "[^a]*"
    test_eq(run1_match("[^a]{3,5}", "", "xyz"), "xyz"); // "[^a]{3,5}"
    test_eq(run1_match("[^a]{3,5}", "", "awxyza"), "wxyz"); // "[^a]{3,5}"
    test_eq(run1_match("[^a]{3,5}", "", "abcdefa"), "bcdef"); // "[^a]{3,5}"
    test_eq(run1_match("[^a]{3,5}", "", "abcdefghijk"), "bcdef"); // "[^a]{3,5}"
    test_eq(run1_match("[^a]{3,5}", "", "*** Failers"), "*** F"); // "[^a]{3,5}"
    test_eq(run1_match("[^a]{3,5}", "", "aaaaa         "), "     "); // "[^a]{3,5}"
    test_eq(run1_match("\\d*", "", "1234b567"), "1234"); // "\\d*"
    test_eq(run1_match("\\d*", "", "xyz"), ""); // "\\d*"
    test_eq(run1_match("\\D*", "", "a1234b567"), "a"); // "\\D*"
    test_eq(run1_match("\\D*", "", "xyz"), "xyz"); // "\\D*"
    test_eq(run1_match("\\D*", "", " "), " "); // "\\D*"
    test_eq(run1_match("\\d+", "", "ab1234c56"), "1234"); // "\\d+"
    test_eq(run1_match("\\D+", "", "ab123c56"), "ab"); // "\\D+"
    test_eq(run1_match("\\D+", "", "*** Failers"), "*** Failers"); // "\\D+"
    test_eq(run1_match("\\d?A", "", "045ABC"), "5A"); // "\\d?A"
    test_eq(run1_match("\\d?A", "", "ABC"), "A"); // "\\d?A"
    test_eq(run1_match("\\D?A", "", "ABC"), "A"); // "\\D?A"
    test_eq(run1_match("\\D?A", "", "BAC"), "BA"); // "\\D?A"
    test_eq(run1_match("\\D?A", "", "9ABC             "), "A"); // "\\D?A"
    test_eq(run1_match("a+", "", "aaaa"), "aaaa"); // "a+"
    test_eq(run1_match("^.*xyz", "", "xyz"), "xyz"); // "^.*xyz"
    test_eq(run1_match("^.*xyz", "", "ggggggggxyz"), "ggggggggxyz"); // "^.*xyz"
    test_eq(run1_match("^.+xyz", "", "abcdxyz"), "abcdxyz"); // "^.+xyz"
    test_eq(run1_match("^.+xyz", "", "axyz"), "axyz"); // "^.+xyz"
    test_eq(run1_match("^.?xyz", "", "xyz"), "xyz"); // "^.?xyz"
    test_eq(run1_match("^.?xyz", "", "cxyz       "), "cxyz"); // "^.?xyz"
    test_eq(run1_match("^\\d{2,3}X", "", "12X"), "12X"); // "^\\d{2,3}X"
    test_eq(run1_match("^\\d{2,3}X", "", "123X"), "123X"); // "^\\d{2,3}X"
    test_eq(run1_match("^[abcd]\\d", "", "a45"), "a4"); // "^[abcd]\\d"
    test_eq(run1_match("^[abcd]\\d", "", "b93"), "b9"); // "^[abcd]\\d"
    test_eq(run1_match("^[abcd]\\d", "", "c99z"), "c9"); // "^[abcd]\\d"
    test_eq(run1_match("^[abcd]\\d", "", "d04"), "d0"); // "^[abcd]\\d"
    test_eq(run1_match("^[abcd]*\\d", "", "a45"), "a4"); // "^[abcd]*\\d"
    test_eq(run1_match("^[abcd]*\\d", "", "b93"), "b9"); // "^[abcd]*\\d"
    test_eq(run1_match("^[abcd]*\\d", "", "c99z"), "c9"); // "^[abcd]*\\d"
    test_eq(run1_match("^[abcd]*\\d", "", "d04"), "d0"); // "^[abcd]*\\d"
    test_eq(run1_match("^[abcd]*\\d", "", "abcd1234"), "abcd1"); // "^[abcd]*\\d"
    test_eq(run1_match("^[abcd]*\\d", "", "1234  "), "1"); // "^[abcd]*\\d"
    test_eq(run1_match("^[abcd]+\\d", "", "a45"), "a4"); // "^[abcd]+\\d"
    test_eq(run1_match("^[abcd]+\\d", "", "b93"), "b9"); // "^[abcd]+\\d"
    test_eq(run1_match("^[abcd]+\\d", "", "c99z"), "c9"); // "^[abcd]+\\d"
    test_eq(run1_match("^[abcd]+\\d", "", "d04"), "d0"); // "^[abcd]+\\d"
    test_eq(run1_match("^[abcd]+\\d", "", "abcd1234"), "abcd1"); // "^[abcd]+\\d"
    test_eq(run1_match("^a+X", "", "aX"), "aX"); // "^a+X"
    test_eq(run1_match("^a+X", "", "aaX "), "aaX"); // "^a+X"
    test_eq(run1_match("^[abcd]?\\d", "", "a45"), "a4"); // "^[abcd]?\\d"
    test_eq(run1_match("^[abcd]?\\d", "", "b93"), "b9"); // "^[abcd]?\\d"
    test_eq(run1_match("^[abcd]?\\d", "", "c99z"), "c9"); // "^[abcd]?\\d"
    test_eq(run1_match("^[abcd]?\\d", "", "d04"), "d0"); // "^[abcd]?\\d"
    test_eq(run1_match("^[abcd]?\\d", "", "1234  "), "1"); // "^[abcd]?\\d"
    test_eq(run1_match("^[abcd]{2,3}\\d", "", "ab45"), "ab4"); // "^[abcd]{2,3}\\d"
    test_eq(run1_match("^[abcd]{2,3}\\d", "", "bcd93"), "bcd9"); // "^[abcd]{2,3}\\d"
    test_eq(run1_match("^(abc)*\\d", "", "abc45"), "abc4,abc"); // "^(abc)*\\d"
    test_eq(run1_match("^(abc)*\\d", "", "abcabcabc45"), "abcabcabc4,abc"); // "^(abc)*\\d"
    test_eq(run1_match("^(abc)*\\d", "", "42xyz "), "4,"); // "^(abc)*\\d"
    test_eq(run1_match("^(abc)+\\d", "", "abc45"), "abc4,abc"); // "^(abc)+\\d"
    test_eq(run1_match("^(abc)+\\d", "", "abcabcabc45"), "abcabcabc4,abc"); // "^(abc)+\\d"
    test_eq(run1_match("^(abc)?\\d", "", "abc45"), "abc4,abc"); // "^(abc)?\\d"
    test_eq(run1_match("^(abc)?\\d", "", "42xyz "), "4,"); // "^(abc)?\\d"
    test_eq(run1_match("^(abc){2,3}\\d", "", "abcabc45"), "abcabc4,abc"); // "^(abc){2,3}\\d"
    test_eq(run1_match("^(abc){2,3}\\d", "", "abcabcabc45"), "abcabcabc4,abc"); // "^(abc){2,3}\\d"
    test_eq(run1_match("^(a*\\w|ab)=(a*\\w|ab)", "", "ab=ab"), "ab=ab,ab,ab"); // "^(a*\\w|ab)=(a*\\w|ab)"
    test_eq(run1_match("^(a*\\w|ab)=(a*\\w|ab)", "", "ab=ab"), "ab=ab,ab,ab"); // "^(a*\\w|ab)=(a*\\w|ab)"
    test_eq(run1_match("^abc", "", "abcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "", "abcdefB  "), "abc"); // "^abc"
    test_eq(run1_match("^(a*|xyz)", "", "bcd"), ","); // "^(a*|xyz)"
    test_eq(run1_match("^(a*|xyz)", "", "aaabcd"), "aaa,aaa"); // "^(a*|xyz)"
    test_eq(run1_match("^(a*|xyz)", "", "xyz"), ","); // "^(a*|xyz)"
    test_eq(run1_match("^(a*|xyz)", "", "xyzN  "), ","); // "^(a*|xyz)"
    test_eq(run1_match("^(a*|xyz)", "", "*** Failers"), ","); // "^(a*|xyz)"
    test_eq(run1_match("^(a*|xyz)", "", "bcdN   "), ","); // "^(a*|xyz)"
    test_eq(run1_match("xyz$", "", "xyz"), "xyz"); // "xyz$"
    test_eq(run1_match("xyz$", "m", "xyz"), "xyz"); // "xyz$"
    test_eq(run1_match("xyz$", "m", "xyz\n "), "xyz"); // "xyz$"
    test_eq(run1_match("xyz$", "m", "abcxyz\npqr "), "xyz"); // "xyz$"
    test_eq(run1_match("xyz$", "m", "abcxyz\npqrZ "), "xyz"); // "xyz$"
    test_eq(run1_match("xyz$", "m", "xyz\nZ    "), "xyz"); // "xyz$"
    test_eq(run1_match("^abcdef", "", "abcdefP"), "abcdef"); // "^abcdef"
    test_eq(run1_match("^a{2,4}\\d+z", "", "aa0zP"), "aa0z"); // "^a{2,4}\\d+z"
    test_eq(run1_match("^a{2,4}\\d+z", "", "aaaa4444444444444zP "), "aaaa4444444444444z"); // "^a{2,4}\\d+z"
    test_eq(run1_match("the quick brown fox", "", "the quick brown fox"), "the quick brown fox"); // "the quick brown fox"
    test_eq(run1_match("the quick brown fox", "", "What do you know about the quick brown fox?"), "the quick brown fox"); // "the quick brown fox"
    test_eq(run1_match("The quick brown fox", "i", "the quick brown fox"), "the quick brown fox"); // "The quick brown fox"
    test_eq(run1_match("The quick brown fox", "i", "The quick brown FOX"), "The quick brown FOX"); // "The quick brown fox"
    test_eq(run1_match("The quick brown fox", "i", "What do you know about the quick brown fox?"), "the quick brown fox"); // "The quick brown fox"
    test_eq(run1_match("The quick brown fox", "i", "What do you know about THE QUICK BROWN FOX?"), "THE QUICK BROWN FOX"); // "The quick brown fox"
    // Skipping Unicode-unfriendly abcd\t\n\r\f\a\e\071\x3b\$\\\?caxyz
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "abxyzpqrrrabbxyyyypqAzz"), "abxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "abxyzpqrrrabbxyyyypqAzz"), "abxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aabxyzpqrrrabbxyyyypqAzz"), "aabxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabxyzpqrrrabbxyyyypqAzz"), "aaabxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaaabxyzpqrrrabbxyyyypqAzz"), "aaaabxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "abcxyzpqrrrabbxyyyypqAzz"), "abcxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aabcxyzpqrrrabbxyyyypqAzz"), "aabcxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzpqrrrabbxyyyypAzz"), "aaabcxyzpqrrrabbxyyyypAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzpqrrrabbxyyyypqAzz"), "aaabcxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzpqrrrabbxyyyypqqAzz"), "aaabcxyzpqrrrabbxyyyypqqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzpqrrrabbxyyyypqqqAzz"), "aaabcxyzpqrrrabbxyyyypqqqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzpqrrrabbxyyyypqqqqAzz"), "aaabcxyzpqrrrabbxyyyypqqqqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzpqrrrabbxyyyypqqqqqAzz"), "aaabcxyzpqrrrabbxyyyypqqqqqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzpqrrrabbxyyyypqqqqqqAzz"), "aaabcxyzpqrrrabbxyyyypqqqqqqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaaabcxyzpqrrrabbxyyyypqAzz"), "aaaabcxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "abxyzzpqrrrabbxyyyypqAzz"), "abxyzzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aabxyzzzpqrrrabbxyyyypqAzz"), "aabxyzzzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabxyzzzzpqrrrabbxyyyypqAzz"), "aaabxyzzzzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaaabxyzzzzpqrrrabbxyyyypqAzz"), "aaaabxyzzzzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "abcxyzzpqrrrabbxyyyypqAzz"), "abcxyzzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aabcxyzzzpqrrrabbxyyyypqAzz"), "aabcxyzzzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzzzzpqrrrabbxyyyypqAzz"), "aaabcxyzzzzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaaabcxyzzzzpqrrrabbxyyyypqAzz"), "aaaabcxyzzzzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaaabcxyzzzzpqrrrabbbxyyyypqAzz"), "aaaabcxyzzzzpqrrrabbbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaaabcxyzzzzpqrrrabbbxyyyyypqAzz"), "aaaabcxyzzzzpqrrrabbbxyyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzpqrrrabbxyyyypABzz"), "aaabcxyzpqrrrabbxyyyypABzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", "aaabcxyzpqrrrabbxyyyypABBzz"), "aaabcxyzpqrrrabbxyyyypABBzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", ">>>aaabxyzpqrrrabbxyyyypqAzz"), "aaabxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", ">aaaabxyzpqrrrabbxyyyypqAzz"), "aaaabxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", "", ">>>>abcxyzpqrrrabbxyyyypqAzz"), "abcxyzpqrrrabbxyyyypqAzz"); // "a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz"
    test_eq(run1_match("^(abc){1,2}zz", "", "abczz"), "abczz,abc"); // "^(abc){1,2}zz"
    test_eq(run1_match("^(abc){1,2}zz", "", "abcabczz"), "abcabczz,abc"); // "^(abc){1,2}zz"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bc"), "bc,b"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bbc"), "bbc,b"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bbbc"), "bbbc,bb"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bac"), "bac,a"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bbac"), "bbac,a"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "aac"), "aac,a"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "abbbbbbbbbbbc"), "abbbbbbbbbbbc,bbbbbbbbbbb"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bbbbbbbbbbbac"), "bbbbbbbbbbbac,a"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+|a){1,2}c", "", "bc"), "bc,b"); // "^(b+|a){1,2}c"
    test_eq(run1_match("^(b+|a){1,2}c", "", "bbc"), "bbc,bb"); // "^(b+|a){1,2}c"
    test_eq(run1_match("^(b+|a){1,2}c", "", "bbbc"), "bbbc,bbb"); // "^(b+|a){1,2}c"
    test_eq(run1_match("^(b+|a){1,2}c", "", "bac"), "bac,a"); // "^(b+|a){1,2}c"
    test_eq(run1_match("^(b+|a){1,2}c", "", "bbac"), "bbac,a"); // "^(b+|a){1,2}c"
    test_eq(run1_match("^(b+|a){1,2}c", "", "aac"), "aac,a"); // "^(b+|a){1,2}c"
    test_eq(run1_match("^(b+|a){1,2}c", "", "abbbbbbbbbbbc"), "abbbbbbbbbbbc,bbbbbbbbbbb"); // "^(b+|a){1,2}c"
    test_eq(run1_match("^(b+|a){1,2}c", "", "bbbbbbbbbbbac"), "bbbbbbbbbbbac,a"); // "^(b+|a){1,2}c"
    test_eq(run1_match("^(b+|a){1,2}c", "", "bbc"), "bbc,bb"); // "^(b+|a){1,2}c"
    test_eq(run1_match("^(b*|ba){1,2}?bc", "", "babc"), "babc,ba"); // "^(b*|ba){1,2}?bc"
    test_eq(run1_match("^(b*|ba){1,2}?bc", "", "bbabc"), "bbabc,ba"); // "^(b*|ba){1,2}?bc"
    test_eq(run1_match("^(b*|ba){1,2}?bc", "", "bababc"), "bababc,ba"); // "^(b*|ba){1,2}?bc"
    test_eq(run1_match("^(ba|b*){1,2}?bc", "", "babc"), "babc,ba"); // "^(ba|b*){1,2}?bc"
    test_eq(run1_match("^(ba|b*){1,2}?bc", "", "bbabc"), "bbabc,ba"); // "^(ba|b*){1,2}?bc"
    test_eq(run1_match("^(ba|b*){1,2}?bc", "", "bababc"), "bababc,ba"); // "^(ba|b*){1,2}?bc"
    test_eq(run1_match("^[ab\\]cde]", "", "athing"), "a"); // "^[ab\\]cde]"
    test_eq(run1_match("^[ab\\]cde]", "", "bthing"), "b"); // "^[ab\\]cde]"
    test_eq(run1_match("^[ab\\]cde]", "", "]thing"), "]"); // "^[ab\\]cde]"
    test_eq(run1_match("^[ab\\]cde]", "", "cthing"), "c"); // "^[ab\\]cde]"
    test_eq(run1_match("^[ab\\]cde]", "", "dthing"), "d"); // "^[ab\\]cde]"
    test_eq(run1_match("^[ab\\]cde]", "", "ething"), "e"); // "^[ab\\]cde]"
    test_eq(run1_match("^[^ab\\]cde]", "", "fthing"), "f"); // "^[^ab\\]cde]"
    test_eq(run1_match("^[^ab\\]cde]", "", "[thing"), "["); // "^[^ab\\]cde]"
    test_eq(run1_match("^[^ab\\]cde]", "", "\\thing"), "\\"); // "^[^ab\\]cde]"
    test_eq(run1_match("^[^ab\\]cde]", "", "*** Failers"), "*"); // "^[^ab\\]cde]"
    test_eq(run1_match("^[0-9]+$", "", "0"), "0"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "1"), "1"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "2"), "2"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "3"), "3"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "4"), "4"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "5"), "5"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "6"), "6"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "7"), "7"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "8"), "8"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "9"), "9"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "10"), "10"); // "^[0-9]+$"
    test_eq(run1_match("^[0-9]+$", "", "100"), "100"); // "^[0-9]+$"
    test_eq(run1_match("^.*nter", "", "enter"), "enter"); // "^.*nter"
    test_eq(run1_match("^.*nter", "", "inter"), "inter"); // "^.*nter"
    test_eq(run1_match("^.*nter", "", "uponter"), "uponter"); // "^.*nter"
    test_eq(run1_match("^xxx[0-9]+$", "", "xxx0"), "xxx0"); // "^xxx[0-9]+$"
    test_eq(run1_match("^xxx[0-9]+$", "", "xxx1234"), "xxx1234"); // "^xxx[0-9]+$"
    test_eq(run1_match("^.+[0-9][0-9][0-9]$", "", "x123"), "x123"); // "^.+[0-9][0-9][0-9]$"
    test_eq(run1_match("^.+[0-9][0-9][0-9]$", "", "xx123"), "xx123"); // "^.+[0-9][0-9][0-9]$"
    test_eq(run1_match("^.+[0-9][0-9][0-9]$", "", "123456"), "123456"); // "^.+[0-9][0-9][0-9]$"
    test_eq(run1_match("^.+[0-9][0-9][0-9]$", "", "x1234"), "x1234"); // "^.+[0-9][0-9][0-9]$"
    test_eq(run1_match("^.+?[0-9][0-9][0-9]$", "", "x123"), "x123"); // "^.+?[0-9][0-9][0-9]$"
    test_eq(run1_match("^.+?[0-9][0-9][0-9]$", "", "xx123"), "xx123"); // "^.+?[0-9][0-9][0-9]$"
    test_eq(run1_match("^.+?[0-9][0-9][0-9]$", "", "123456"), "123456"); // "^.+?[0-9][0-9][0-9]$"
    test_eq(run1_match("^.+?[0-9][0-9][0-9]$", "", "x1234"), "x1234"); // "^.+?[0-9][0-9][0-9]$"
    test_eq(run1_match("^([^!]+)!(.+)=apquxz\\.ixr\\.zzz\\.ac\\.uk$", "", "abc!pqr=apquxz.ixr.zzz.ac.uk"), "abc!pqr=apquxz.ixr.zzz.ac.uk,abc,pqr"); // "^([^!]+)!(.+)=apquxz\\.ixr\\.zzz\\.ac\\.uk$"
    test_eq(run1_match(":", "", "Well, we need a colon: somewhere"), ":"); // ":"
    test_eq(run1_match("([\\da-f:]+)$", "i", "0abc"), "0abc,0abc"); // "([\\da-f:]+)$"
    test_eq(run1_match("([\\da-f:]+)$", "i", "abc"), "abc,abc"); // "([\\da-f:]+)$"
    test_eq(run1_match("([\\da-f:]+)$", "i", "fed"), "fed,fed"); // "([\\da-f:]+)$"
    test_eq(run1_match("([\\da-f:]+)$", "i", "E"), "E,E"); // "([\\da-f:]+)$"
    test_eq(run1_match("([\\da-f:]+)$", "i", "::"), "::,::"); // "([\\da-f:]+)$"
    test_eq(run1_match("([\\da-f:]+)$", "i", "5f03:12C0::932e"), "5f03:12C0::932e,5f03:12C0::932e"); // "([\\da-f:]+)$"
    test_eq(run1_match("([\\da-f:]+)$", "i", "fed def"), "def,def"); // "([\\da-f:]+)$"
    test_eq(run1_match("([\\da-f:]+)$", "i", "Any old stuff"), "ff,ff"); // "([\\da-f:]+)$"
    test_eq(run1_match("^.*\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$", "", ".1.2.3"), ".1.2.3,1,2,3"); // "^.*\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$"
    test_eq(run1_match("^.*\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$", "", "A.12.123.0"), "A.12.123.0,12,123,0"); // "^.*\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$"
    test_eq(run1_match("^(\\d+)\\s+IN\\s+SOA\\s+(\\S+)\\s+(\\S+)\\s*\\(\\s*$", "", "1 IN SOA non-sp1 non-sp2("), "1 IN SOA non-sp1 non-sp2(,1,non-sp1,non-sp2"); // "^(\\d+)\\s+IN\\s+SOA\\s+(\\S+)\\s+(\\S+)\\s*\\(\\s*$"
    test_eq(run1_match("^(\\d+)\\s+IN\\s+SOA\\s+(\\S+)\\s+(\\S+)\\s*\\(\\s*$", "", "1    IN    SOA    non-sp1    non-sp2   ("), "1    IN    SOA    non-sp1    non-sp2   (,1,non-sp1,non-sp2"); // "^(\\d+)\\s+IN\\s+SOA\\s+(\\S+)\\s+(\\S+)\\s*\\(\\s*$"
    test_eq(run1_match("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$", "", "a."), "a.,"); // "^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$"
    test_eq(run1_match("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$", "", "Z."), "Z.,"); // "^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$"
    test_eq(run1_match("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$", "", "2."), "2.,"); // "^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$"
    test_eq(run1_match("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$", "", "ab-c.pq-r."), "ab-c.pq-r.,.pq-r"); // "^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$"
    test_eq(run1_match("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$", "", "sxk.zzz.ac.uk."), "sxk.zzz.ac.uk.,.uk"); // "^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$"
    test_eq(run1_match("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$", "", "x-.y-."), "x-.y-.,.y-"); // "^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-Z\\d\\-]*)*\\.$"
    test_eq(run1_match("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", "", "*.a"), "*.a,,,"); // "^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$"
    test_eq(run1_match("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", "", "*.b0-a"), "*.b0-a,0-a,,"); // "^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$"
    test_eq(run1_match("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", "", "*.c3-b.c"), "*.c3-b.c,3-b,.c,"); // "^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$"
    test_eq(run1_match("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", "", "*.c-a.b-c"), "*.c-a.b-c,-a,.b-c,-c"); // "^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$"
    test_eq(run1_match("^(?=ab(de))(abd)(e)", "", "abde"), "abde,de,abd,e"); // "^(?=ab(de))(abd)(e)"
    test_eq(run1_match("^(?!(ab)de|x)(abd)(f)", "", "abdf"), "abdf,,abd,f"); // "^(?!(ab)de|x)(abd)(f)"
    test_eq(run1_match("^(?=(ab(cd)))(ab)", "", "abcd"), "ab,abcd,cd,ab"); // "^(?=(ab(cd)))(ab)"
    test_eq(run1_match("^[\\da-f](\\.[\\da-f])*$", "i", "a.b.c.d"), "a.b.c.d,.d"); // "^[\\da-f](\\.[\\da-f])*$"
    test_eq(run1_match("^[\\da-f](\\.[\\da-f])*$", "i", "A.B.C.D"), "A.B.C.D,.D"); // "^[\\da-f](\\.[\\da-f])*$"
    test_eq(run1_match("^[\\da-f](\\.[\\da-f])*$", "i", "a.b.c.1.2.3.C"), "a.b.c.1.2.3.C,.C"); // "^[\\da-f](\\.[\\da-f])*$"
    // Skipping Unicode-unfriendly ^\".*\"\s*(;.*)?$
    // Skipping Unicode-unfriendly ^\".*\"\s*(;.*)?$
    // Skipping Unicode-unfriendly ^\".*\"\s*(;.*)?$
    test_eq(run1_match("^ab\\sc$", "", "ab c"), "ab c"); // "^ab\\sc$"
    test_eq(run1_match("^ab\\sc$", "", "ab c"), "ab c"); // "^ab\\sc$"
    // Skipping Unicode-unfriendly ^a\ b[c]d$
    test_eq(run1_match("^(a(b(c)))(d(e(f)))(h(i(j)))(k(l(m)))$", "", "abcdefhijklm"), "abcdefhijklm,abc,bc,c,def,ef,f,hij,ij,j,klm,lm,m"); // "^(a(b(c)))(d(e(f)))(h(i(j)))(k(l(m)))$"
    test_eq(run1_match("^(?:a(b(c)))(?:d(e(f)))(?:h(i(j)))(?:k(l(m)))$", "", "abcdefhijklm"), "abcdefhijklm,bc,c,ef,f,ij,j,lm,m"); // "^(?:a(b(c)))(?:d(e(f)))(?:h(i(j)))(?:k(l(m)))$"
    test_eq(run1_match("^a*\\w", "", "z"), "z"); // "^a*\\w"
    test_eq(run1_match("^a*\\w", "", "az"), "az"); // "^a*\\w"
    test_eq(run1_match("^a*\\w", "", "aaaz"), "aaaz"); // "^a*\\w"
    test_eq(run1_match("^a*\\w", "", "a"), "a"); // "^a*\\w"
    test_eq(run1_match("^a*\\w", "", "aa"), "aa"); // "^a*\\w"
    test_eq(run1_match("^a*\\w", "", "aaaa"), "aaaa"); // "^a*\\w"
    test_eq(run1_match("^a*\\w", "", "a+"), "a"); // "^a*\\w"
    test_eq(run1_match("^a*\\w", "", "aa+"), "aa"); // "^a*\\w"
    test_eq(run1_match("^a*?\\w", "", "z"), "z"); // "^a*?\\w"
    test_eq(run1_match("^a*?\\w", "", "az"), "a"); // "^a*?\\w"
    test_eq(run1_match("^a*?\\w", "", "aaaz"), "a"); // "^a*?\\w"
    test_eq(run1_match("^a*?\\w", "", "a"), "a"); // "^a*?\\w"
    test_eq(run1_match("^a*?\\w", "", "aa"), "a"); // "^a*?\\w"
    test_eq(run1_match("^a*?\\w", "", "aaaa"), "a"); // "^a*?\\w"
    test_eq(run1_match("^a*?\\w", "", "a+"), "a"); // "^a*?\\w"
    test_eq(run1_match("^a*?\\w", "", "aa+"), "a"); // "^a*?\\w"
    test_eq(run1_match("^a+\\w", "", "az"), "az"); // "^a+\\w"
    test_eq(run1_match("^a+\\w", "", "aaaz"), "aaaz"); // "^a+\\w"
    test_eq(run1_match("^a+\\w", "", "aa"), "aa"); // "^a+\\w"
    test_eq(run1_match("^a+\\w", "", "aaaa"), "aaaa"); // "^a+\\w"
    test_eq(run1_match("^a+\\w", "", "aa+"), "aa"); // "^a+\\w"
    test_eq(run1_match("^a+?\\w", "", "az"), "az"); // "^a+?\\w"
    test_eq(run1_match("^a+?\\w", "", "aaaz"), "aa"); // "^a+?\\w"
    test_eq(run1_match("^a+?\\w", "", "aa"), "aa"); // "^a+?\\w"
    test_eq(run1_match("^a+?\\w", "", "aaaa"), "aa"); // "^a+?\\w"
    test_eq(run1_match("^a+?\\w", "", "aa+"), "aa"); // "^a+?\\w"
    test_eq(run1_match("^\\d{8}\\w{2,}", "", "1234567890"), "1234567890"); // "^\\d{8}\\w{2,}"
    test_eq(run1_match("^\\d{8}\\w{2,}", "", "12345678ab"), "12345678ab"); // "^\\d{8}\\w{2,}"
    test_eq(run1_match("^\\d{8}\\w{2,}", "", "12345678__"), "12345678__"); // "^\\d{8}\\w{2,}"
    test_eq(run1_match("^[aeiou\\d]{4,5}$", "", "uoie"), "uoie"); // "^[aeiou\\d]{4,5}$"
    test_eq(run1_match("^[aeiou\\d]{4,5}$", "", "1234"), "1234"); // "^[aeiou\\d]{4,5}$"
    test_eq(run1_match("^[aeiou\\d]{4,5}$", "", "12345"), "12345"); // "^[aeiou\\d]{4,5}$"
    test_eq(run1_match("^[aeiou\\d]{4,5}$", "", "aaaaa"), "aaaaa"); // "^[aeiou\\d]{4,5}$"
    test_eq(run1_match("^[aeiou\\d]{4,5}?", "", "uoie"), "uoie"); // "^[aeiou\\d]{4,5}?"
    test_eq(run1_match("^[aeiou\\d]{4,5}?", "", "1234"), "1234"); // "^[aeiou\\d]{4,5}?"
    test_eq(run1_match("^[aeiou\\d]{4,5}?", "", "12345"), "1234"); // "^[aeiou\\d]{4,5}?"
    test_eq(run1_match("^[aeiou\\d]{4,5}?", "", "aaaaa"), "aaaa"); // "^[aeiou\\d]{4,5}?"
    test_eq(run1_match("^[aeiou\\d]{4,5}?", "", "123456"), "1234"); // "^[aeiou\\d]{4,5}?"
    test_eq(run1_match("^From +([^ ]+) +[a-zA-Z][a-zA-Z][a-zA-Z] +[a-zA-Z][a-zA-Z][a-zA-Z] +[0-9]?[0-9] +[0-9][0-9]:[0-9][0-9]", "", "From abcd  Mon Sep 01 12:33:02 1997"), "From abcd  Mon Sep 01 12:33,abcd"); // "^From +([^ ]+) +[a-zA-Z][a-zA-Z][a-zA-Z] +[a-zA-Z][a-zA-Z][a-zA-Z] +[0-9]?[0-9] +[0-9][0-9]:[0-9][0-9]"
    test_eq(run1_match("^From\\s+\\S+\\s+([a-zA-Z]{3}\\s+){2}\\d{1,2}\\s+\\d\\d:\\d\\d", "", "From abcd  Mon Sep 01 12:33:02 1997"), "From abcd  Mon Sep 01 12:33,Sep "); // "^From\\s+\\S+\\s+([a-zA-Z]{3}\\s+){2}\\d{1,2}\\s+\\d\\d:\\d\\d"
    test_eq(run1_match("^From\\s+\\S+\\s+([a-zA-Z]{3}\\s+){2}\\d{1,2}\\s+\\d\\d:\\d\\d", "", "From abcd  Mon Sep  1 12:33:02 1997"), "From abcd  Mon Sep  1 12:33,Sep  "); // "^From\\s+\\S+\\s+([a-zA-Z]{3}\\s+){2}\\d{1,2}\\s+\\d\\d:\\d\\d"
    test_eq(run1_match("\\w+(?=\\t)", "", "the quick brown\u{9} fox"), "brown"); // "\\w+(?=\\t)"
    test_eq(run1_match("foo(?!bar)(.*)", "", "foobar is foolish see?"), "foolish see?,lish see?"); // "foo(?!bar)(.*)"
    test_eq(run1_match("(?:(?!foo)...|^.{0,2})bar(.*)", "", "foobar crowbar etc"), "rowbar etc, etc"); // "(?:(?!foo)...|^.{0,2})bar(.*)"
    test_eq(run1_match("(?:(?!foo)...|^.{0,2})bar(.*)", "", "barrel"), "barrel,rel"); // "(?:(?!foo)...|^.{0,2})bar(.*)"
    test_eq(run1_match("(?:(?!foo)...|^.{0,2})bar(.*)", "", "2barrel"), "2barrel,rel"); // "(?:(?!foo)...|^.{0,2})bar(.*)"
    test_eq(run1_match("(?:(?!foo)...|^.{0,2})bar(.*)", "", "A barrel"), "A barrel,rel"); // "(?:(?!foo)...|^.{0,2})bar(.*)"
    test_eq(run1_match("^(\\D*)(?=\\d)(?!123)", "", "abc456"), "abc,abc"); // "^(\\D*)(?=\\d)(?!123)"
    test_eq(run1_match("^1234", "", "1234"), "1234"); // "^1234"
    test_eq(run1_match("^1234", "", "1234"), "1234"); // "^1234"
    test_eq(run1_match("abcd", "", "abcd"), "abcd"); // "abcd"
    test_eq(run1_match("^abcd", "", "abcd"), "abcd"); // "^abcd"
    test_eq(run1_match("(?!^)abc", "", "the abc"), "abc"); // "(?!^)abc"
    test_eq(run1_match("(?=^)abc", "", "abc"), "abc"); // "(?=^)abc"
    test_eq(run1_match("^[ab]{1,3}(ab*|b)", "", "aabbbbb"), "aabb,b"); // "^[ab]{1,3}(ab*|b)"
    test_eq(run1_match("^[ab]{1,3}?(ab*|b)", "", "aabbbbb"), "aabbbbb,abbbbb"); // "^[ab]{1,3}?(ab*|b)"
    test_eq(run1_match("^[ab]{1,3}?(ab*?|b)", "", "aabbbbb"), "aa,a"); // "^[ab]{1,3}?(ab*?|b)"
    test_eq(run1_match("^[ab]{1,3}(ab*?|b)", "", "aabbbbb"), "aabb,b"); // "^[ab]{1,3}(ab*?|b)"
    // Skipping Unicode-unfriendly (?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\)|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")*<(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*,(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*)*:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*>)(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*
    // Skipping Unicode-unfriendly (?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\)|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")*<(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*,(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*)*:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*>)(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*
    // Skipping Unicode-unfriendly (?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\)|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")*<(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*,(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*)*:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*>)(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*
    // Skipping Unicode-unfriendly (?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\)|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")*<(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*,(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*)*:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*>)(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*
    // Skipping Unicode-unfriendly (?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\)|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")*<(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*,(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*)*:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*>)(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*
    // Skipping Unicode-unfriendly (?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\)|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")*<(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*,(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*)*:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*>)(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*
    // Skipping Unicode-unfriendly (?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\)|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")*<(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*,(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*)*:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*")(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"(?:[^\\\x80-\xff\n\015"]|\\[^\x80-\xff])*"))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*@(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])(?:(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*\.(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\]))*(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*>)(?:[\040\t]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff]|\((?:[^\\\x80-\xff\n\015()]|\\[^\x80-\xff])*\))*\))*
    // Skipping Unicode-unfriendly [\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*(?:(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*)*<[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*(?:,[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*)*:[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*>)
    // Skipping Unicode-unfriendly [\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*(?:(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*)*<[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*(?:,[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*)*:[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*>)
    // Skipping Unicode-unfriendly [\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*(?:(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*)*<[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*(?:,[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*)*:[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*>)
    // Skipping Unicode-unfriendly [\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*(?:(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*)*<[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*(?:,[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*)*:[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*>)
    // Skipping Unicode-unfriendly [\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*(?:(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*)*<[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*(?:,[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*)*:[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*>)
    // Skipping Unicode-unfriendly [\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*(?:(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*)*<[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*(?:,[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*)*:[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*>)
    // Skipping Unicode-unfriendly [\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*|(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*(?:(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[^()<>@,;:".\\\[\]\x80-\xff\000-\010\012-\037]*)*<[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*(?:,[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*)*:[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)?(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|"[^\\\x80-\xff\n\015"]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015"]*)*")[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*@[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:\.[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*(?:[^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff]+(?![^(\040)<>@,;:".\\\[\]\000-\037\x80-\xff])|\[(?:[^\\\x80-\xff\n\015\[\]]|\\[^\x80-\xff])*\])[\040\t]*(?:\([^\\\x80-\xff\n\015()]*(?:(?:\\[^\x80-\xff]|\([^\\\x80-\xff\n\015()]*(?:\\[^\x80-\xff][^\\\x80-\xff\n\015()]*)*\))[^\\\x80-\xff\n\015()]*)*\)[\040\t]*)*)*>)
    test_eq(run1_match("abc\\x0def\\x00pqr\\x000xyz\\x0000AB", "", "abc\u{d}ef\u{0}pqr\u{0}0xyz\u{0}00AB"), "abc\u{d}ef\u{0}pqr\u{0}0xyz\u{0}00AB"); // "abc\\x0def\\x00pqr\\x000xyz\\x0000AB"
    test_eq(run1_match("abc\\x0def\\x00pqr\\x000xyz\\x0000AB", "", "abc456 abc\u{d}ef\u{0}pqr\u{0}0xyz\u{0}00ABCDE"), "abc\u{d}ef\u{0}pqr\u{0}0xyz\u{0}00AB"); // "abc\\x0def\\x00pqr\\x000xyz\\x0000AB"
    // Skipping Unicode-unfriendly ^[\000-\037]
    // Skipping Unicode-unfriendly ^[\000-\037]
    // Skipping Unicode-unfriendly ^[\000-\037]
    test_eq(run1_match("\\0*", "", "\u{0}\u{0}\u{0}\u{0}"), "\u{0}\u{0}\u{0}\u{0}"); // "\\0*"
    test_eq(run1_match("^\\s", "", " abc"), " "); // "^\\s"
    test_eq(run1_match("^\\s", "", "\u{c}abc"), "\u{c}"); // "^\\s"
    test_eq(run1_match("^\\s", "", "\nabc"), "\n"); // "^\\s"
    test_eq(run1_match("^\\s", "", "\u{d}abc"), "\u{d}"); // "^\\s"
    test_eq(run1_match("^\\s", "", "\u{9}abc"), "\u{9}"); // "^\\s"
    test_eq(run1_match("^abc", "", "abc"), "abc"); // "^abc"
    test_eq(run1_match("ab{1,3}bc", "", "abbbbc"), "abbbbc"); // "ab{1,3}bc"
    test_eq(run1_match("ab{1,3}bc", "", "abbbc"), "abbbc"); // "ab{1,3}bc"
    test_eq(run1_match("ab{1,3}bc", "", "abbc"), "abbc"); // "ab{1,3}bc"
    test_eq(run1_match("([^.]*)\\.([^:]*):[T ]+(.*)", "", "track1.title:TBlah blah blah"), "track1.title:TBlah blah blah,track1,title,Blah blah blah"); // "([^.]*)\\.([^:]*):[T ]+(.*)"
    test_eq(run1_match("([^.]*)\\.([^:]*):[T ]+(.*)", "i", "track1.title:TBlah blah blah"), "track1.title:TBlah blah blah,track1,title,Blah blah blah"); // "([^.]*)\\.([^:]*):[T ]+(.*)"
    test_eq(run1_match("([^.]*)\\.([^:]*):[t ]+(.*)", "i", "track1.title:TBlah blah blah"), "track1.title:TBlah blah blah,track1,title,Blah blah blah"); // "([^.]*)\\.([^:]*):[t ]+(.*)"
    test_eq(run1_match("^[W-c]+$", "", "WXY_^abc"), "WXY_^abc"); // "^[W-c]+$"
    test_eq(run1_match("^[W-c]+$", "i", "WXY_^abc"), "WXY_^abc"); // "^[W-c]+$"
    test_eq(run1_match("^[W-c]+$", "i", "wxy_^ABC"), "wxy_^ABC"); // "^[W-c]+$"
    test_eq(run1_match("^[\\x3f-\\x5F]+$", "i", "WXY_^abc"), "WXY_^abc"); // "^[\\x3f-\\x5F]+$"
    test_eq(run1_match("^[\\x3f-\\x5F]+$", "i", "wxy_^ABC"), "wxy_^ABC"); // "^[\\x3f-\\x5F]+$"
    test_eq(run1_match("^abc$", "m", "abc"), "abc"); // "^abc$"
    test_eq(run1_match("^abc$", "m", "qqq\nabc"), "abc"); // "^abc$"
    test_eq(run1_match("^abc$", "m", "abc\nzzz"), "abc"); // "^abc$"
    test_eq(run1_match("^abc$", "m", "qqq\nabc\nzzz"), "abc"); // "^abc$"
    test_eq(run1_match("^abc$", "", "abc"), "abc"); // "^abc$"
    test_eq(run1_match("(?:b)|(?::+)", "", "b::c"), "b"); // "(?:b)|(?::+)"
    test_eq(run1_match("(?:b)|(?::+)", "", "c::b"), "::"); // "(?:b)|(?::+)"
    test_eq(run1_match("[-az]+", "", "az-"), "az-"); // "[-az]+"
    test_eq(run1_match("[-az]+", "", "*** Failers"), "a"); // "[-az]+"
    test_eq(run1_match("[az-]+", "", "za-"), "za-"); // "[az-]+"
    test_eq(run1_match("[az-]+", "", "*** Failers"), "a"); // "[az-]+"
    test_eq(run1_match("[a\\-z]+", "", "a-z"), "a-z"); // "[a\\-z]+"
    test_eq(run1_match("[a\\-z]+", "", "*** Failers"), "a"); // "[a\\-z]+"
    test_eq(run1_match("[a-z]+", "", "abcdxyz"), "abcdxyz"); // "[a-z]+"
    test_eq(run1_match("[\\d-]+", "", "12-34"), "12-34"); // "[\\d-]+"
    // Skipping Unicode-unfriendly [\d-z]+
    test_eq(run1_match("\\x5c", "", "\\\\"), "\\"); // "\\x5c"
    test_eq(run1_match("\\x20Z", "", "the Zoo"), " Z"); // "\\x20Z"
    // Skipping Unicode-unfriendly ab{3cd
    // Skipping Unicode-unfriendly ab{3,cd
    // Skipping Unicode-unfriendly ab{3,4a}cd
    // Skipping Unicode-unfriendly {4,5a}bc
    test_eq(run1_match("abc$", "", "abc"), "abc"); // "abc$"
    // Skipping Unicode-unfriendly (abc)\123
    // Skipping Unicode-unfriendly (abc)\223
    // Skipping Unicode-unfriendly (abc)\323
    // Skipping Unicode-unfriendly (abc)\100
    // Skipping Unicode-unfriendly (abc)\100
    // Skipping Unicode-unfriendly (abc)\100
    // Skipping Unicode-unfriendly (abc)\100
    // Skipping Unicode-unfriendly (abc)\100
    // Skipping Unicode-unfriendly (abc)\100
    // Skipping Unicode-unfriendly (abc)\100
    // Skipping Unicode-unfriendly (abc)\100
    // Skipping Unicode-unfriendly (a)(b)(c)(d)(e)(f)(g)(h)(i)(j)(k)\12\123
    // Skipping Unicode-unfriendly ab\idef
    test_eq(run1_match("a{0}bc", "", "bc"), "bc"); // "a{0}bc"
    test_eq(run1_match("(a|(bc)){0,0}?xyz", "", "xyz"), "xyz,,"); // "(a|(bc)){0,0}?xyz"
    // Skipping Unicode-unfriendly abc[\10]de
    // Skipping Unicode-unfriendly abc[\1]de
    // Skipping Unicode-unfriendly (abc)[\1]de
    test_eq(run1_match("^([^a])([^\\b])([^c]*)([^d]{3,4})", "", "baNOTccccd"), "baNOTcccc,b,a,NOT,cccc"); // "^([^a])([^\\b])([^c]*)([^d]{3,4})"
    test_eq(run1_match("^([^a])([^\\b])([^c]*)([^d]{3,4})", "", "baNOTcccd"), "baNOTccc,b,a,NOT,ccc"); // "^([^a])([^\\b])([^c]*)([^d]{3,4})"
    test_eq(run1_match("^([^a])([^\\b])([^c]*)([^d]{3,4})", "", "baNOTccd"), "baNOTcc,b,a,NO,Tcc"); // "^([^a])([^\\b])([^c]*)([^d]{3,4})"
    test_eq(run1_match("^([^a])([^\\b])([^c]*)([^d]{3,4})", "", "bacccd"), "baccc,b,a,,ccc"); // "^([^a])([^\\b])([^c]*)([^d]{3,4})"
    test_eq(run1_match("^([^a])([^\\b])([^c]*)([^d]{3,4})", "", "*** Failers"), "*** Failers,*,*,* Fail,ers"); // "^([^a])([^\\b])([^c]*)([^d]{3,4})"
    test_eq(run1_match("[^a]", "", "Abc"), "A"); // "[^a]"
    test_eq(run1_match("[^a]", "i", "Abc "), "b"); // "[^a]"
    test_eq(run1_match("[^a]+", "", "AAAaAbc"), "AAA"); // "[^a]+"
    test_eq(run1_match("[^a]+", "i", "AAAaAbc "), "bc "); // "[^a]+"
    test_eq(run1_match("[^a]+", "", "bbb\nccc"), "bbb\nccc"); // "[^a]+"
    test_eq(run1_match("[^k]$", "", "abc"), "c"); // "[^k]$"
    test_eq(run1_match("[^k]$", "", "*** Failers"), "s"); // "[^k]$"
    test_eq(run1_match("[^k]$", "", "abk   "), " "); // "[^k]$"
    test_eq(run1_match("[^k]{2,3}$", "", "abc"), "abc"); // "[^k]{2,3}$"
    test_eq(run1_match("[^k]{2,3}$", "", "kbc"), "bc"); // "[^k]{2,3}$"
    test_eq(run1_match("[^k]{2,3}$", "", "kabc "), "bc "); // "[^k]{2,3}$"
    test_eq(run1_match("[^k]{2,3}$", "", "*** Failers"), "ers"); // "[^k]{2,3}$"
    // Skipping Unicode-unfriendly ^\d{8,}\@.+[^k]$
    // Skipping Unicode-unfriendly ^\d{8,}\@.+[^k]$
    test_eq(run1_match("[^a]", "", "aaaabcd"), "b"); // "[^a]"
    test_eq(run1_match("[^a]", "", "aaAabcd "), "A"); // "[^a]"
    test_eq(run1_match("[^a]", "i", "aaaabcd"), "b"); // "[^a]"
    test_eq(run1_match("[^a]", "i", "aaAabcd "), "b"); // "[^a]"
    test_eq(run1_match("[^az]", "", "aaaabcd"), "b"); // "[^az]"
    test_eq(run1_match("[^az]", "", "aaAabcd "), "A"); // "[^az]"
    test_eq(run1_match("[^az]", "i", "aaaabcd"), "b"); // "[^az]"
    test_eq(run1_match("[^az]", "i", "aaAabcd "), "b"); // "[^az]"
    test_eq(run1_match("P[^*]TAIRE[^*]{1,6}?LL", "", "xxxxxxxxxxxPSTAIREISLLxxxxxxxxx"), "PSTAIREISLL"); // "P[^*]TAIRE[^*]{1,6}?LL"
    test_eq(run1_match("P[^*]TAIRE[^*]{1,}?LL", "", "xxxxxxxxxxxPSTAIREISLLxxxxxxxxx"), "PSTAIREISLL"); // "P[^*]TAIRE[^*]{1,}?LL"
    test_eq(run1_match("(\\.\\d\\d[1-9]?)\\d+", "", "1.230003938"), ".230003938,.23"); // "(\\.\\d\\d[1-9]?)\\d+"
    test_eq(run1_match("(\\.\\d\\d[1-9]?)\\d+", "", "1.875000282   "), ".875000282,.875"); // "(\\.\\d\\d[1-9]?)\\d+"
    test_eq(run1_match("(\\.\\d\\d[1-9]?)\\d+", "", "1.235  "), ".235,.23"); // "(\\.\\d\\d[1-9]?)\\d+"
    test_eq(run1_match("(\\.\\d\\d((?=0)|\\d(?=\\d)))", "", "1.230003938      "), ".23,.23,"); // "(\\.\\d\\d((?=0)|\\d(?=\\d)))"
    test_eq(run1_match("(\\.\\d\\d((?=0)|\\d(?=\\d)))", "", "1.875000282"), ".875,.875,5"); // "(\\.\\d\\d((?=0)|\\d(?=\\d)))"
    test_eq(run1_match("\\b(foo)\\s+(\\w+)", "i", "Food is on the foo table"), "foo table,foo,table"); // "\\b(foo)\\s+(\\w+)"
    test_eq(run1_match("foo(.*)bar", "", "The food is under the bar in the barn."), "food is under the bar in the bar,d is under the bar in the "); // "foo(.*)bar"
    test_eq(run1_match("foo(.*?)bar", "", "The food is under the bar in the barn."), "food is under the bar,d is under the "); // "foo(.*?)bar"
    test_eq(run1_match("(.*)(\\d*)", "", "I have 2 numbers: 53147"), "I have 2 numbers: 53147,I have 2 numbers: 53147,"); // "(.*)(\\d*)"
    test_eq(run1_match("(.*)(\\d+)", "", "I have 2 numbers: 53147"), "I have 2 numbers: 53147,I have 2 numbers: 5314,7"); // "(.*)(\\d+)"
    test_eq(run1_match("(.*?)(\\d*)", "", "I have 2 numbers: 53147"), ",,"); // "(.*?)(\\d*)"
    test_eq(run1_match("(.*?)(\\d+)", "", "I have 2 numbers: 53147"), "I have 2,I have ,2"); // "(.*?)(\\d+)"
    test_eq(run1_match("(.*)(\\d+)$", "", "I have 2 numbers: 53147"), "I have 2 numbers: 53147,I have 2 numbers: 5314,7"); // "(.*)(\\d+)$"
    test_eq(run1_match("(.*?)(\\d+)$", "", "I have 2 numbers: 53147"), "I have 2 numbers: 53147,I have 2 numbers: ,53147"); // "(.*?)(\\d+)$"
    test_eq(run1_match("(.*)\\b(\\d+)$", "", "I have 2 numbers: 53147"), "I have 2 numbers: 53147,I have 2 numbers: ,53147"); // "(.*)\\b(\\d+)$"
    test_eq(run1_match("(.*\\D)(\\d+)$", "", "I have 2 numbers: 53147"), "I have 2 numbers: 53147,I have 2 numbers: ,53147"); // "(.*\\D)(\\d+)$"
    test_eq(run1_match("^\\D*(?!123)", "", "ABC123"), "AB"); // "^\\D*(?!123)"
    test_eq(run1_match("^\\D*(?!123)", "", " "), " "); // "^\\D*(?!123)"
    test_eq(run1_match("^(\\D*)(?=\\d)(?!123)", "", "ABC445"), "ABC,ABC"); // "^(\\D*)(?=\\d)(?!123)"
    // Skipping Unicode-unfriendly ^[W-]46]
    // Skipping Unicode-unfriendly ^[W-]46]
    test_eq(run1_match("^[W-\\]46]", "", "W46]789 "), "W"); // "^[W-\\]46]"
    test_eq(run1_match("^[W-\\]46]", "", "Wall"), "W"); // "^[W-\\]46]"
    test_eq(run1_match("^[W-\\]46]", "", "Zebra"), "Z"); // "^[W-\\]46]"
    test_eq(run1_match("^[W-\\]46]", "", "Xylophone  "), "X"); // "^[W-\\]46]"
    test_eq(run1_match("^[W-\\]46]", "", "42"), "4"); // "^[W-\\]46]"
    test_eq(run1_match("^[W-\\]46]", "", "[abcd] "), "["); // "^[W-\\]46]"
    test_eq(run1_match("^[W-\\]46]", "", "]abcd["), "]"); // "^[W-\\]46]"
    test_eq(run1_match("^[W-\\]46]", "", "\\backslash "), "\\"); // "^[W-\\]46]"
    test_eq(run1_match("\\d\\d\\/\\d\\d\\/\\d\\d\\d\\d", "", "01/01/2000"), "01/01/2000"); // "\\d\\d\\/\\d\\d\\/\\d\\d\\d\\d"
    test_eq(run1_match("^(a){0,0}", "", "bcd"), ","); // "^(a){0,0}"
    test_eq(run1_match("^(a){0,0}", "", "abc"), ","); // "^(a){0,0}"
    test_eq(run1_match("^(a){0,0}", "", "aab     "), ","); // "^(a){0,0}"
    test_eq(run1_match("^(a){0,1}", "", "bcd"), ","); // "^(a){0,1}"
    test_eq(run1_match("^(a){0,1}", "", "abc"), "a,a"); // "^(a){0,1}"
    test_eq(run1_match("^(a){0,1}", "", "aab  "), "a,a"); // "^(a){0,1}"
    test_eq(run1_match("^(a){0,2}", "", "bcd"), ","); // "^(a){0,2}"
    test_eq(run1_match("^(a){0,2}", "", "abc"), "a,a"); // "^(a){0,2}"
    test_eq(run1_match("^(a){0,2}", "", "aab  "), "aa,a"); // "^(a){0,2}"
    test_eq(run1_match("^(a){0,3}", "", "bcd"), ","); // "^(a){0,3}"
    test_eq(run1_match("^(a){0,3}", "", "abc"), "a,a"); // "^(a){0,3}"
    test_eq(run1_match("^(a){0,3}", "", "aab"), "aa,a"); // "^(a){0,3}"
    test_eq(run1_match("^(a){0,3}", "", "aaa   "), "aaa,a"); // "^(a){0,3}"
    test_eq(run1_match("^(a){0,}", "", "bcd"), ","); // "^(a){0,}"
    test_eq(run1_match("^(a){0,}", "", "abc"), "a,a"); // "^(a){0,}"
    test_eq(run1_match("^(a){0,}", "", "aab"), "aa,a"); // "^(a){0,}"
    test_eq(run1_match("^(a){0,}", "", "aaa"), "aaa,a"); // "^(a){0,}"
    test_eq(run1_match("^(a){0,}", "", "aaaaaaaa    "), "aaaaaaaa,a"); // "^(a){0,}"
    test_eq(run1_match("^(a){1,1}", "", "abc"), "a,a"); // "^(a){1,1}"
    test_eq(run1_match("^(a){1,1}", "", "aab  "), "a,a"); // "^(a){1,1}"
    test_eq(run1_match("^(a){1,2}", "", "abc"), "a,a"); // "^(a){1,2}"
    test_eq(run1_match("^(a){1,2}", "", "aab  "), "aa,a"); // "^(a){1,2}"
    test_eq(run1_match("^(a){1,3}", "", "abc"), "a,a"); // "^(a){1,3}"
    test_eq(run1_match("^(a){1,3}", "", "aab"), "aa,a"); // "^(a){1,3}"
    test_eq(run1_match("^(a){1,3}", "", "aaa   "), "aaa,a"); // "^(a){1,3}"
    test_eq(run1_match("^(a){1,}", "", "abc"), "a,a"); // "^(a){1,}"
    test_eq(run1_match("^(a){1,}", "", "aab"), "aa,a"); // "^(a){1,}"
    test_eq(run1_match("^(a){1,}", "", "aaa"), "aaa,a"); // "^(a){1,}"
    test_eq(run1_match("^(a){1,}", "", "aaaaaaaa    "), "aaaaaaaa,a"); // "^(a){1,}"
    test_eq(run1_match(".*\\.gif", "", "borfle\nbib.gif\nno"), "bib.gif"); // ".*\\.gif"
    test_eq(run1_match(".{0,}\\.gif", "", "borfle\nbib.gif\nno"), "bib.gif"); // ".{0,}\\.gif"
    test_eq(run1_match(".*\\.gif", "m", "borfle\nbib.gif\nno"), "bib.gif"); // ".*\\.gif"
    test_eq(run1_match(".*\\.gif", "", "borfle\nbib.gif\nno"), "bib.gif"); // ".*\\.gif"
    test_eq(run1_match(".*\\.gif", "m", "borfle\nbib.gif\nno"), "bib.gif"); // ".*\\.gif"
    test_eq(run1_match(".*$", "", "borfle\nbib.gif\nno"), "no"); // ".*$"
    test_eq(run1_match(".*$", "m", "borfle\nbib.gif\nno"), "borfle"); // ".*$"
    test_eq(run1_match(".*$", "", "borfle\nbib.gif\nno"), "no"); // ".*$"
    test_eq(run1_match(".*$", "m", "borfle\nbib.gif\nno"), "borfle"); // ".*$"
    test_eq(run1_match(".*$", "", "borfle\nbib.gif\nno\n"), ""); // ".*$"
    test_eq(run1_match(".*$", "m", "borfle\nbib.gif\nno\n"), "borfle"); // ".*$"
    test_eq(run1_match(".*$", "", "borfle\nbib.gif\nno\n"), ""); // ".*$"
    test_eq(run1_match(".*$", "m", "borfle\nbib.gif\nno\n"), "borfle"); // ".*$"
    test_eq(run1_match("(.*X|^B)", "", "abcde\n1234Xyz"), "1234X,1234X"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "", "BarFoo "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "abcde\n1234Xyz"), "1234X,1234X"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "BarFoo "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "abcde\nBar  "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "", "abcde\n1234Xyz"), "1234X,1234X"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "", "BarFoo "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "abcde\n1234Xyz"), "1234X,1234X"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "BarFoo "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "abcde\nBar  "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "abcde\n1234Xyz"), "1234X,1234X"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "BarFoo "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "abcde\nBar  "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "abcde\n1234Xyz"), "1234X,1234X"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "BarFoo "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("(.*X|^B)", "m", "abcde\nBar  "), "B,B"); // "(.*X|^B)"
    test_eq(run1_match("^.*B", "", "B\n"), "B"); // "^.*B"
    test_eq(run1_match("^[0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9]", "", "123456654321"), "123456654321"); // "^[0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9]"
    test_eq(run1_match("^\\d\\d\\d\\d\\d\\d\\d\\d\\d\\d\\d\\d", "", "123456654321 "), "123456654321"); // "^\\d\\d\\d\\d\\d\\d\\d\\d\\d\\d\\d\\d"
    test_eq(run1_match("^[\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d]", "", "123456654321"), "123456654321"); // "^[\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d]"
    test_eq(run1_match("^[abc]{12}", "", "abcabcabcabc"), "abcabcabcabc"); // "^[abc]{12}"
    test_eq(run1_match("^[a-c]{12}", "", "abcabcabcabc"), "abcabcabcabc"); // "^[a-c]{12}"
    test_eq(run1_match("^(a|b|c){12}", "", "abcabcabcabc "), "abcabcabcabc,c"); // "^(a|b|c){12}"
    test_eq(run1_match("^[abcdefghijklmnopqrstuvwxy0123456789]", "", "n"), "n"); // "^[abcdefghijklmnopqrstuvwxy0123456789]"
    test_eq(run1_match("abcde{0,0}", "", "abcd"), "abcd"); // "abcde{0,0}"
    test_eq(run1_match("ab[cd]{0,0}e", "", "abe"), "abe"); // "ab[cd]{0,0}e"
    test_eq(run1_match("ab(c){0,0}d", "", "abd"), "abd,"); // "ab(c){0,0}d"
    test_eq(run1_match("a(b*)", "", "a"), "a,"); // "a(b*)"
    test_eq(run1_match("a(b*)", "", "ab"), "ab,b"); // "a(b*)"
    test_eq(run1_match("a(b*)", "", "abbbb"), "abbbb,bbbb"); // "a(b*)"
    test_eq(run1_match("a(b*)", "", "*** Failers"), "a,"); // "a(b*)"
    test_eq(run1_match("ab\\d{0}e", "", "abe"), "abe"); // "ab\\d{0}e"
    test_eq(run1_match("\"([^\\\\\"]+|\\\\.)*\"", "", "the \"quick\" brown fox"), "\"quick\",quick"); // "\"([^\\\\\"]+|\\\\.)*\""
    test_eq(run1_match("\"([^\\\\\"]+|\\\\.)*\"", "", "\"the \\\"quick\\\" brown fox\" "), "\"the \\\"quick\\\" brown fox\", brown fox"); // "\"([^\\\\\"]+|\\\\.)*\""
    test_eq(run1_match(".*?", "", "abc"), ""); // ".*?"
    test_eq(run1_match("\\b", "", "abc "), ""); // "\\b"
    test_eq(tc.compile("\\b").run_global_match("abc "), ","); // "\\b"
    // Skipping global "\\b" with string "abc"
    test_eq(run1_match("a[^a]b", "", "acb"), "acb"); // "a[^a]b"
    test_eq(run1_match("a[^a]b", "", "a\nb"), "a\nb"); // "a[^a]b"
    test_eq(run1_match("a.b", "", "acb"), "acb"); // "a.b"
    test_eq(run1_match("a[^a]b", "", "acb"), "acb"); // "a[^a]b"
    test_eq(run1_match("a[^a]b", "", "a\nb  "), "a\nb"); // "a[^a]b"
    test_eq(run1_match("a.b", "", "acb"), "acb"); // "a.b"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bac"), "bac,a"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bbac"), "bbac,a"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bbbac"), "bbbac,a"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bbbbac"), "bbbbac,a"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+?|a){1,2}?c", "", "bbbbbac "), "bbbbbac,a"); // "^(b+?|a){1,2}?c"
    test_eq(run1_match("^(b+|a){1,2}?c", "", "bac"), "bac,a"); // "^(b+|a){1,2}?c"
    test_eq(run1_match("^(b+|a){1,2}?c", "", "bbac"), "bbac,a"); // "^(b+|a){1,2}?c"
    test_eq(run1_match("^(b+|a){1,2}?c", "", "bbbac"), "bbbac,a"); // "^(b+|a){1,2}?c"
    test_eq(run1_match("^(b+|a){1,2}?c", "", "bbbbac"), "bbbbac,a"); // "^(b+|a){1,2}?c"
    test_eq(run1_match("^(b+|a){1,2}?c", "", "bbbbbac "), "bbbbbac,a"); // "^(b+|a){1,2}?c"
    // Skipping Unicode-unfriendly (?!\A)x
    // Skipping Unicode-unfriendly (?!\A)x
    test_eq(run1_match("(A|B)*?CD", "", "CD "), "CD,"); // "(A|B)*?CD"
    test_eq(run1_match("(A|B)*CD", "", "CD "), "CD,"); // "(A|B)*CD"
    test_eq(run1_match("(\\d+)(\\w)", "", "12345a"), "12345a,12345,a"); // "(\\d+)(\\w)"
    test_eq(run1_match("(\\d+)(\\w)", "", "12345+ "), "12345,1234,5"); // "(\\d+)(\\w)"
    test_eq(run1_match("(\\d+)(\\w)", "", "12345a"), "12345a,12345,a"); // "(\\d+)(\\w)"
    test_eq(run1_match("(\\d+)(\\w)", "", "12345+ "), "12345,1234,5"); // "(\\d+)(\\w)"
    test_eq(run1_match("(a+|b+|c+)*c", "", "aaabbbbccccd"), "aaabbbbcccc,ccc"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "((abc(ade)ufh()()x"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "(abc)"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "(abc(def)xyz)"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "a bcd e"), "bc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "a b cd e"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abcd e   "), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "a bcde "), "bc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "a bcde f"), "bc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abcdef  "), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abc"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "aBc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "Abc"), "bc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "ABc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abc"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "aBc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "aBc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "aBBc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abcd"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abcD     "), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abc"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "aBbc"), "bc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "aBBc "), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "Abc"), "bc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abc"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "aBc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abxxc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "aBxxc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "Abxxc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "ABxxc"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abc:"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "abc:"), "abc,b"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "cat"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "fcat"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "focat   "), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "foocat  "), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "cat"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "fcat"), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "focat   "), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(a+|b+|c+)*c", "", "foocat  "), "c,"); // "(a+|b+|c+)*c"
    test_eq(run1_match("(abc|)+", "", "abc"), "abc,abc"); // "(abc|)+"
    test_eq(run1_match("(abc|)+", "", "abcabc"), "abcabc,abc"); // "(abc|)+"
    test_eq(run1_match("(abc|)+", "", "abcabcabc"), "abcabcabc,abc"); // "(abc|)+"
    test_eq(run1_match("(abc|)+", "", "xyz      "), ","); // "(abc|)+"
    test_eq(run1_match("([a]*)*", "", "a"), "a,a"); // "([a]*)*"
    test_eq(run1_match("([a]*)*", "", "aaaaa "), "aaaaa,aaaaa"); // "([a]*)*"
    test_eq(run1_match("([ab]*)*", "", "a"), "a,a"); // "([ab]*)*"
    test_eq(run1_match("([ab]*)*", "", "b"), "b,b"); // "([ab]*)*"
    test_eq(run1_match("([ab]*)*", "", "ababab"), "ababab,ababab"); // "([ab]*)*"
    test_eq(run1_match("([ab]*)*", "", "aaaabcde"), "aaaab,aaaab"); // "([ab]*)*"
    test_eq(run1_match("([ab]*)*", "", "bbbb    "), "bbbb,bbbb"); // "([ab]*)*"
    test_eq(run1_match("([^a]*)*", "", "b"), "b,b"); // "([^a]*)*"
    test_eq(run1_match("([^a]*)*", "", "bbbb"), "bbbb,bbbb"); // "([^a]*)*"
    test_eq(run1_match("([^a]*)*", "", "aaa   "), ","); // "([^a]*)*"
    test_eq(run1_match("([^ab]*)*", "", "cccc"), "cccc,cccc"); // "([^ab]*)*"
    test_eq(run1_match("([^ab]*)*", "", "abab  "), ","); // "([^ab]*)*"
    test_eq(run1_match("([a]*?)*", "", "a"), "a,a"); // "([a]*?)*"
    test_eq(run1_match("([a]*?)*", "", "aaaa "), "aaaa,a"); // "([a]*?)*"
    test_eq(run1_match("([ab]*?)*", "", "a"), "a,a"); // "([ab]*?)*"
    test_eq(run1_match("([ab]*?)*", "", "b"), "b,b"); // "([ab]*?)*"
    test_eq(run1_match("([ab]*?)*", "", "abab"), "abab,b"); // "([ab]*?)*"
    test_eq(run1_match("([ab]*?)*", "", "baba   "), "baba,a"); // "([ab]*?)*"
    test_eq(run1_match("([^a]*?)*", "", "b"), "b,b"); // "([^a]*?)*"
    test_eq(run1_match("([^a]*?)*", "", "bbbb"), "bbbb,b"); // "([^a]*?)*"
    test_eq(run1_match("([^a]*?)*", "", "aaa   "), ","); // "([^a]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "c"), "c,c"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "cccc"), "cccc,c"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "baba   "), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "a"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aaabcde "), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aaaaa"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aabbaa "), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aaaaa"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aabbaa "), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "12-sep-98"), "12-sep-98,8"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "12-09-98"), "12-09-98,8"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "*** Failers"), "*** F,F"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "sep-12-98"), "sep-12-98,8"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "    "), "    , "); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "saturday"), "s,s"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "sunday"), "sund,d"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "Saturday"), "S,S"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "Sunday"), "Sund,d"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "SATURDAY"), "SATURDAY,Y"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "SUNDAY"), "SUNDAY,Y"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "SunDay"), "SunD,D"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "abcx"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aBCx"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "bbx"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "BBx"), "BBx,x"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "*** Failers"), "*** F,F"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "abcX"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aBCX"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "bbX"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "BBX               "), "BBX               , "); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "ac"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aC"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "bD"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "elephant"), "eleph,h"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "Europe "), "Europe , "); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "frog"), "frog,g"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "France"), "Fr,r"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "*** Failers"), "*** F,F"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "Africa     "), "Afric,c"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "ab"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aBd"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "xy"), "xy,y"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "xY"), "xY,Y"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "zebra"), "ze,e"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "Zambesi"), "Z,Z"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "*** Failers"), "*** F,F"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "aCD  "), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "XY  "), "XY  , "); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "foo\nbar"), "foo\n,\n"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "*** Failers"), "*** F,F"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "bar"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "baz\nbar   "), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "barbaz"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "barbarbaz "), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "koobarbaz "), "koo,o"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "*** Failers"), "*** F,F"); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "baz"), ","); // "([^ab]*?)*"
    test_eq(run1_match("([^ab]*?)*", "", "foobarbaz "), "foo,o"); // "([^ab]*?)*"
    test_eq(run1_match("abc", "", "abc"), "abc"); // "abc"
    test_eq(run1_match("abc", "", "xabcy"), "abc"); // "abc"
    test_eq(run1_match("abc", "", "ababc"), "abc"); // "abc"
    test_eq(run1_match("ab*c", "", "abc"), "abc"); // "ab*c"
    test_eq(run1_match("ab*bc", "", "abc"), "abc"); // "ab*bc"
    test_eq(run1_match("ab*bc", "", "abbc"), "abbc"); // "ab*bc"
    test_eq(run1_match("ab*bc", "", "abbbbc"), "abbbbc"); // "ab*bc"
    test_eq(run1_match(".{1}", "", "abbbbc"), "a"); // ".{1}"
    test_eq(run1_match(".{3,4}", "", "abbbbc"), "abbb"); // ".{3,4}"
    test_eq(run1_match("ab{0,}bc", "", "abbbbc"), "abbbbc"); // "ab{0,}bc"
    test_eq(run1_match("ab+bc", "", "abbc"), "abbc"); // "ab+bc"
    test_eq(run1_match("ab+bc", "", "abbbbc"), "abbbbc"); // "ab+bc"
    test_eq(run1_match("ab{1,}bc", "", "abbbbc"), "abbbbc"); // "ab{1,}bc"
    test_eq(run1_match("ab{1,3}bc", "", "abbbbc"), "abbbbc"); // "ab{1,3}bc"
    test_eq(run1_match("ab{3,4}bc", "", "abbbbc"), "abbbbc"); // "ab{3,4}bc"
    test_eq(run1_match("ab?bc", "", "abbc"), "abbc"); // "ab?bc"
    test_eq(run1_match("ab?bc", "", "abc"), "abc"); // "ab?bc"
    test_eq(run1_match("ab{0,1}bc", "", "abc"), "abc"); // "ab{0,1}bc"
    test_eq(run1_match("ab?c", "", "abc"), "abc"); // "ab?c"
    test_eq(run1_match("ab{0,1}c", "", "abc"), "abc"); // "ab{0,1}c"
    test_eq(run1_match("^abc$", "", "abc"), "abc"); // "^abc$"
    test_eq(run1_match("^abc", "", "abcc"), "abc"); // "^abc"
    test_eq(run1_match("abc$", "", "aabc"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "", "aabc"), "abc"); // "abc$"
    test_eq(run1_match("^", "", "abc"), ""); // "^"
    test_eq(run1_match("$", "", "abc"), ""); // "$"
    test_eq(run1_match("a.c", "", "abc"), "abc"); // "a.c"
    test_eq(run1_match("a.c", "", "axc"), "axc"); // "a.c"
    test_eq(run1_match("a.*c", "", "axyzc"), "axyzc"); // "a.*c"
    test_eq(run1_match("a[bc]d", "", "abd"), "abd"); // "a[bc]d"
    test_eq(run1_match("a[b-d]e", "", "ace"), "ace"); // "a[b-d]e"
    test_eq(run1_match("a[b-d]", "", "aac"), "ac"); // "a[b-d]"
    test_eq(run1_match("a[-b]", "", "a-"), "a-"); // "a[-b]"
    test_eq(run1_match("a[b-]", "", "a-"), "a-"); // "a[b-]"
    // Skipping Unicode-unfriendly a]
    test_eq(run1_match("a[^bc]d", "", "aed"), "aed"); // "a[^bc]d"
    test_eq(run1_match("a[^-b]c", "", "adc"), "adc"); // "a[^-b]c"
    test_eq(run1_match("\\ba\\b", "", "a-"), "a"); // "\\ba\\b"
    test_eq(run1_match("\\ba\\b", "", "-a"), "a"); // "\\ba\\b"
    test_eq(run1_match("\\ba\\b", "", "-a-"), "a"); // "\\ba\\b"
    test_eq(run1_match("\\Ba\\B", "", "*** Failers"), "a"); // "\\Ba\\B"
    test_eq(run1_match("\\By\\b", "", "xy"), "y"); // "\\By\\b"
    test_eq(run1_match("\\by\\B", "", "yz"), "y"); // "\\by\\B"
    test_eq(run1_match("\\By\\B", "", "xyz"), "y"); // "\\By\\B"
    test_eq(run1_match("\\w", "", "a"), "a"); // "\\w"
    test_eq(run1_match("\\W", "", "-"), "-"); // "\\W"
    test_eq(run1_match("\\W", "", "*** Failers"), "*"); // "\\W"
    test_eq(run1_match("\\W", "", "-"), "-"); // "\\W"
    test_eq(run1_match("a\\sb", "", "a b"), "a b"); // "a\\sb"
    test_eq(run1_match("a\\Sb", "", "a-b"), "a-b"); // "a\\Sb"
    test_eq(run1_match("a\\Sb", "", "a-b"), "a-b"); // "a\\Sb"
    test_eq(run1_match("\\d", "", "1"), "1"); // "\\d"
    test_eq(run1_match("\\D", "", "-"), "-"); // "\\D"
    test_eq(run1_match("\\D", "", "*** Failers"), "*"); // "\\D"
    test_eq(run1_match("\\D", "", "-"), "-"); // "\\D"
    test_eq(run1_match("[\\w]", "", "a"), "a"); // "[\\w]"
    test_eq(run1_match("[\\W]", "", "-"), "-"); // "[\\W]"
    test_eq(run1_match("[\\W]", "", "*** Failers"), "*"); // "[\\W]"
    test_eq(run1_match("[\\W]", "", "-"), "-"); // "[\\W]"
    test_eq(run1_match("a[\\s]b", "", "a b"), "a b"); // "a[\\s]b"
    test_eq(run1_match("a[\\S]b", "", "a-b"), "a-b"); // "a[\\S]b"
    test_eq(run1_match("a[\\S]b", "", "a-b"), "a-b"); // "a[\\S]b"
    test_eq(run1_match("[\\d]", "", "1"), "1"); // "[\\d]"
    test_eq(run1_match("[\\D]", "", "-"), "-"); // "[\\D]"
    test_eq(run1_match("[\\D]", "", "*** Failers"), "*"); // "[\\D]"
    test_eq(run1_match("[\\D]", "", "-"), "-"); // "[\\D]"
    test_eq(run1_match("ab|cd", "", "abc"), "ab"); // "ab|cd"
    test_eq(run1_match("ab|cd", "", "abcd"), "ab"); // "ab|cd"
    test_eq(run1_match("()ef", "", "def"), "ef,"); // "()ef"
    test_eq(run1_match("a\\(b", "", "a(b"), "a(b"); // "a\\(b"
    test_eq(run1_match("((a))", "", "abc"), "a,a,a"); // "((a))"
    test_eq(run1_match("(a)b(c)", "", "abc"), "abc,a,c"); // "(a)b(c)"
    test_eq(run1_match("a+b+c", "", "aabbabc"), "abc"); // "a+b+c"
    test_eq(run1_match("a{1,}b{1,}c", "", "aabbabc"), "abc"); // "a{1,}b{1,}c"
    test_eq(run1_match("a.+?c", "", "abcabc"), "abc"); // "a.+?c"
    test_eq(run1_match("(a+|b)*", "", "ab"), "ab,b"); // "(a+|b)*"
    test_eq(run1_match("(a+|b){0,}", "", "ab"), "ab,b"); // "(a+|b){0,}"
    test_eq(run1_match("(a+|b)+", "", "ab"), "ab,b"); // "(a+|b)+"
    test_eq(run1_match("(a+|b){1,}", "", "ab"), "ab,b"); // "(a+|b){1,}"
    test_eq(run1_match("(a+|b)?", "", "ab"), "a,a"); // "(a+|b)?"
    test_eq(run1_match("(a+|b){0,1}", "", "ab"), "a,a"); // "(a+|b){0,1}"
    test_eq(run1_match("[^ab]*", "", "cde"), "cde"); // "[^ab]*"
    test_eq(run1_match("([abc])*d", "", "abbbcd"), "abbbcd,c"); // "([abc])*d"
    test_eq(run1_match("([abc])*bcd", "", "abcd"), "abcd,a"); // "([abc])*bcd"
    test_eq(run1_match("a|b|c|d|e", "", "e"), "e"); // "a|b|c|d|e"
    test_eq(run1_match("(a|b|c|d|e)f", "", "ef"), "ef,e"); // "(a|b|c|d|e)f"
    test_eq(run1_match("abcd*efg", "", "abcdefg"), "abcdefg"); // "abcd*efg"
    test_eq(run1_match("ab*", "", "xabyabbbz"), "ab"); // "ab*"
    test_eq(run1_match("ab*", "", "xayabbbz"), "a"); // "ab*"
    test_eq(run1_match("(ab|cd)e", "", "abcde"), "cde,cd"); // "(ab|cd)e"
    test_eq(run1_match("[abhgefdc]ij", "", "hij"), "hij"); // "[abhgefdc]ij"
    test_eq(run1_match("(abc|)ef", "", "abcdef"), "ef,"); // "(abc|)ef"
    test_eq(run1_match("(a|b)c*d", "", "abcd"), "bcd,b"); // "(a|b)c*d"
    test_eq(run1_match("(ab|ab*)bc", "", "abc"), "abc,a"); // "(ab|ab*)bc"
    test_eq(run1_match("a([bc]*)c*", "", "abc"), "abc,bc"); // "a([bc]*)c*"
    test_eq(run1_match("a([bc]*)(c*d)", "", "abcd"), "abcd,bc,d"); // "a([bc]*)(c*d)"
    test_eq(run1_match("a([bc]+)(c*d)", "", "abcd"), "abcd,bc,d"); // "a([bc]+)(c*d)"
    test_eq(run1_match("a([bc]*)(c+d)", "", "abcd"), "abcd,b,cd"); // "a([bc]*)(c+d)"
    test_eq(run1_match("a[bcd]*dcdcde", "", "adcdcde"), "adcdcde"); // "a[bcd]*dcdcde"
    test_eq(run1_match("(ab|a)b*c", "", "abc"), "abc,ab"); // "(ab|a)b*c"
    test_eq(run1_match("((a)(b)c)(d)", "", "abcd"), "abcd,abc,a,b,d"); // "((a)(b)c)(d)"
    test_eq(run1_match("[a-zA-Z_][a-zA-Z0-9_]*", "", "alpha"), "alpha"); // "[a-zA-Z_][a-zA-Z0-9_]*"
    test_eq(run1_match("^a(bc+|b[eh])g|.h$", "", "abh"), "bh,"); // "^a(bc+|b[eh])g|.h$"
    test_eq(run1_match("(bc+d$|ef*g.|h?i(j|k))", "", "effgz"), "effgz,effgz,"); // "(bc+d$|ef*g.|h?i(j|k))"
    test_eq(run1_match("(bc+d$|ef*g.|h?i(j|k))", "", "ij"), "ij,ij,j"); // "(bc+d$|ef*g.|h?i(j|k))"
    test_eq(run1_match("(bc+d$|ef*g.|h?i(j|k))", "", "reffgz"), "effgz,effgz,"); // "(bc+d$|ef*g.|h?i(j|k))"
    test_eq(run1_match("((((((((((a))))))))))", "", "a"), "a,a,a,a,a,a,a,a,a,a,a"); // "((((((((((a))))))))))"
    test_eq(run1_match("(((((((((a)))))))))", "", "a"), "a,a,a,a,a,a,a,a,a,a"); // "(((((((((a)))))))))"
    test_eq(run1_match("multiple words", "", "multiple words, yeah"), "multiple words"); // "multiple words"
    test_eq(run1_match("(.*)c(.*)", "", "abcde"), "abcde,ab,de"); // "(.*)c(.*)"
    test_eq(run1_match("\\((.*), (.*)\\)", "", "(a, b)"), "(a, b),a,b"); // "\\((.*), (.*)\\)"
    test_eq(run1_match("abcd", "", "abcd"), "abcd"); // "abcd"
    test_eq(run1_match("a(bc)d", "", "abcd"), "abcd,bc"); // "a(bc)d"
    test_eq(run1_match("a[-]?c", "", "ac"), "ac"); // "a[-]?c"
    test_eq(run1_match("abc", "i", "ABC"), "ABC"); // "abc"
    test_eq(run1_match("abc", "i", "XABCY"), "ABC"); // "abc"
    test_eq(run1_match("abc", "i", "ABABC"), "ABC"); // "abc"
    test_eq(run1_match("ab*c", "i", "ABC"), "ABC"); // "ab*c"
    test_eq(run1_match("ab*bc", "i", "ABC"), "ABC"); // "ab*bc"
    test_eq(run1_match("ab*bc", "i", "ABBC"), "ABBC"); // "ab*bc"
    test_eq(run1_match("ab*?bc", "i", "ABBBBC"), "ABBBBC"); // "ab*?bc"
    test_eq(run1_match("ab{0,}?bc", "i", "ABBBBC"), "ABBBBC"); // "ab{0,}?bc"
    test_eq(run1_match("ab+?bc", "i", "ABBC"), "ABBC"); // "ab+?bc"
    test_eq(run1_match("ab+bc", "i", "ABBBBC"), "ABBBBC"); // "ab+bc"
    test_eq(run1_match("ab{1,}?bc", "i", "ABBBBC"), "ABBBBC"); // "ab{1,}?bc"
    test_eq(run1_match("ab{1,3}?bc", "i", "ABBBBC"), "ABBBBC"); // "ab{1,3}?bc"
    test_eq(run1_match("ab{3,4}?bc", "i", "ABBBBC"), "ABBBBC"); // "ab{3,4}?bc"
    test_eq(run1_match("ab??bc", "i", "ABBC"), "ABBC"); // "ab??bc"
    test_eq(run1_match("ab??bc", "i", "ABC"), "ABC"); // "ab??bc"
    test_eq(run1_match("ab{0,1}?bc", "i", "ABC"), "ABC"); // "ab{0,1}?bc"
    test_eq(run1_match("ab??c", "i", "ABC"), "ABC"); // "ab??c"
    test_eq(run1_match("ab{0,1}?c", "i", "ABC"), "ABC"); // "ab{0,1}?c"
    test_eq(run1_match("^abc$", "i", "ABC"), "ABC"); // "^abc$"
    test_eq(run1_match("^abc", "i", "ABCC"), "ABC"); // "^abc"
    test_eq(run1_match("abc$", "i", "AABC"), "ABC"); // "abc$"
    test_eq(run1_match("^", "i", "ABC"), ""); // "^"
    test_eq(run1_match("$", "i", "ABC"), ""); // "$"
    test_eq(run1_match("a.c", "i", "ABC"), "ABC"); // "a.c"
    test_eq(run1_match("a.c", "i", "AXC"), "AXC"); // "a.c"
    test_eq(run1_match("a.*?c", "i", "AXYZC"), "AXYZC"); // "a.*?c"
    test_eq(run1_match("a.*c", "i", "AABC"), "AABC"); // "a.*c"
    test_eq(run1_match("a[bc]d", "i", "ABD"), "ABD"); // "a[bc]d"
    test_eq(run1_match("a[b-d]e", "i", "ACE"), "ACE"); // "a[b-d]e"
    test_eq(run1_match("a[b-d]", "i", "AAC"), "AC"); // "a[b-d]"
    test_eq(run1_match("a[-b]", "i", "A-"), "A-"); // "a[-b]"
    test_eq(run1_match("a[b-]", "i", "A-"), "A-"); // "a[b-]"
    // Skipping Unicode-unfriendly a]
    test_eq(run1_match("a[^bc]d", "i", "AED"), "AED"); // "a[^bc]d"
    test_eq(run1_match("a[^-b]c", "i", "ADC"), "ADC"); // "a[^-b]c"
    test_eq(run1_match("ab|cd", "i", "ABC"), "AB"); // "ab|cd"
    test_eq(run1_match("ab|cd", "i", "ABCD"), "AB"); // "ab|cd"
    test_eq(run1_match("()ef", "i", "DEF"), "EF,"); // "()ef"
    test_eq(run1_match("a\\(b", "i", "A(B"), "A(B"); // "a\\(b"
    test_eq(run1_match("((a))", "i", "ABC"), "A,A,A"); // "((a))"
    test_eq(run1_match("(a)b(c)", "i", "ABC"), "ABC,A,C"); // "(a)b(c)"
    test_eq(run1_match("a+b+c", "i", "AABBABC"), "ABC"); // "a+b+c"
    test_eq(run1_match("a{1,}b{1,}c", "i", "AABBABC"), "ABC"); // "a{1,}b{1,}c"
    test_eq(run1_match("a.+?c", "i", "ABCABC"), "ABC"); // "a.+?c"
    test_eq(run1_match("a.*?c", "i", "ABCABC"), "ABC"); // "a.*?c"
    test_eq(run1_match("a.{0,5}?c", "i", "ABCABC"), "ABC"); // "a.{0,5}?c"
    test_eq(run1_match("(a+|b)*", "i", "AB"), "AB,B"); // "(a+|b)*"
    test_eq(run1_match("(a+|b){0,}", "i", "AB"), "AB,B"); // "(a+|b){0,}"
    test_eq(run1_match("(a+|b)+", "i", "AB"), "AB,B"); // "(a+|b)+"
    test_eq(run1_match("(a+|b){1,}", "i", "AB"), "AB,B"); // "(a+|b){1,}"
    test_eq(run1_match("(a+|b)?", "i", "AB"), "A,A"); // "(a+|b)?"
    test_eq(run1_match("(a+|b){0,1}", "i", "AB"), "A,A"); // "(a+|b){0,1}"
    test_eq(run1_match("(a+|b){0,1}?", "i", "AB"), ","); // "(a+|b){0,1}?"
    test_eq(run1_match("[^ab]*", "i", "CDE"), "CDE"); // "[^ab]*"
    test_eq(run1_match("([abc])*d", "i", "ABBBCD"), "ABBBCD,C"); // "([abc])*d"
    test_eq(run1_match("([abc])*bcd", "i", "ABCD"), "ABCD,A"); // "([abc])*bcd"
    test_eq(run1_match("a|b|c|d|e", "i", "E"), "E"); // "a|b|c|d|e"
    test_eq(run1_match("(a|b|c|d|e)f", "i", "EF"), "EF,E"); // "(a|b|c|d|e)f"
    test_eq(run1_match("abcd*efg", "i", "ABCDEFG"), "ABCDEFG"); // "abcd*efg"
    test_eq(run1_match("ab*", "i", "XABYABBBZ"), "AB"); // "ab*"
    test_eq(run1_match("ab*", "i", "XAYABBBZ"), "A"); // "ab*"
    test_eq(run1_match("(ab|cd)e", "i", "ABCDE"), "CDE,CD"); // "(ab|cd)e"
    test_eq(run1_match("[abhgefdc]ij", "i", "HIJ"), "HIJ"); // "[abhgefdc]ij"
    test_eq(run1_match("(abc|)ef", "i", "ABCDEF"), "EF,"); // "(abc|)ef"
    test_eq(run1_match("(a|b)c*d", "i", "ABCD"), "BCD,B"); // "(a|b)c*d"
    test_eq(run1_match("(ab|ab*)bc", "i", "ABC"), "ABC,A"); // "(ab|ab*)bc"
    test_eq(run1_match("a([bc]*)c*", "i", "ABC"), "ABC,BC"); // "a([bc]*)c*"
    test_eq(run1_match("a([bc]*)(c*d)", "i", "ABCD"), "ABCD,BC,D"); // "a([bc]*)(c*d)"
    test_eq(run1_match("a([bc]+)(c*d)", "i", "ABCD"), "ABCD,BC,D"); // "a([bc]+)(c*d)"
    test_eq(run1_match("a([bc]*)(c+d)", "i", "ABCD"), "ABCD,B,CD"); // "a([bc]*)(c+d)"
    test_eq(run1_match("a[bcd]*dcdcde", "i", "ADCDCDE"), "ADCDCDE"); // "a[bcd]*dcdcde"
    test_eq(run1_match("(ab|a)b*c", "i", "ABC"), "ABC,AB"); // "(ab|a)b*c"
    test_eq(run1_match("((a)(b)c)(d)", "i", "ABCD"), "ABCD,ABC,A,B,D"); // "((a)(b)c)(d)"
    test_eq(run1_match("[a-zA-Z_][a-zA-Z0-9_]*", "i", "ALPHA"), "ALPHA"); // "[a-zA-Z_][a-zA-Z0-9_]*"
    test_eq(run1_match("^a(bc+|b[eh])g|.h$", "i", "ABH"), "BH,"); // "^a(bc+|b[eh])g|.h$"
    test_eq(run1_match("(bc+d$|ef*g.|h?i(j|k))", "i", "EFFGZ"), "EFFGZ,EFFGZ,"); // "(bc+d$|ef*g.|h?i(j|k))"
    test_eq(run1_match("(bc+d$|ef*g.|h?i(j|k))", "i", "IJ"), "IJ,IJ,J"); // "(bc+d$|ef*g.|h?i(j|k))"
    test_eq(run1_match("(bc+d$|ef*g.|h?i(j|k))", "i", "REFFGZ"), "EFFGZ,EFFGZ,"); // "(bc+d$|ef*g.|h?i(j|k))"
    test_eq(run1_match("((((((((((a))))))))))", "i", "A"), "A,A,A,A,A,A,A,A,A,A,A"); // "((((((((((a))))))))))"
    test_eq(run1_match("(((((((((a)))))))))", "i", "A"), "A,A,A,A,A,A,A,A,A,A"); // "(((((((((a)))))))))"
    test_eq(run1_match("(?:(?:(?:(?:(?:(?:(?:(?:(?:(a))))))))))", "i", "A"), "A,A"); // "(?:(?:(?:(?:(?:(?:(?:(?:(?:(a))))))))))"
    test_eq(run1_match("(?:(?:(?:(?:(?:(?:(?:(?:(?:(a|b|c))))))))))", "i", "C"), "C,C"); // "(?:(?:(?:(?:(?:(?:(?:(?:(?:(a|b|c))))))))))"
    test_eq(run1_match("multiple words", "i", "MULTIPLE WORDS, YEAH"), "MULTIPLE WORDS"); // "multiple words"
    test_eq(run1_match("(.*)c(.*)", "i", "ABCDE"), "ABCDE,AB,DE"); // "(.*)c(.*)"
    test_eq(run1_match("\\((.*), (.*)\\)", "i", "(A, B)"), "(A, B),A,B"); // "\\((.*), (.*)\\)"
    test_eq(run1_match("abcd", "i", "ABCD"), "ABCD"); // "abcd"
    test_eq(run1_match("a(bc)d", "i", "ABCD"), "ABCD,BC"); // "a(bc)d"
    test_eq(run1_match("a[-]?c", "i", "AC"), "AC"); // "a[-]?c"
    test_eq(run1_match("a(?!b).", "", "abad"), "ad"); // "a(?!b)."
    test_eq(run1_match("a(?=d).", "", "abad"), "ad"); // "a(?=d)."
    test_eq(run1_match("a(?=c|d).", "", "abad"), "ad"); // "a(?=c|d)."
    test_eq(run1_match("a(?:b|c|d)(.)", "", "ace"), "ace,e"); // "a(?:b|c|d)(.)"
    test_eq(run1_match("a(?:b|c|d)*(.)", "", "ace"), "ace,e"); // "a(?:b|c|d)*(.)"
    test_eq(run1_match("a(?:b|c|d)+?(.)", "", "ace"), "ace,e"); // "a(?:b|c|d)+?(.)"
    test_eq(run1_match("a(?:b|c|d)+?(.)", "", "acdbcdbe"), "acd,d"); // "a(?:b|c|d)+?(.)"
    test_eq(run1_match("a(?:b|c|d)+(.)", "", "acdbcdbe"), "acdbcdbe,e"); // "a(?:b|c|d)+(.)"
    test_eq(run1_match("a(?:b|c|d){2}(.)", "", "acdbcdbe"), "acdb,b"); // "a(?:b|c|d){2}(.)"
    test_eq(run1_match("a(?:b|c|d){4,5}(.)", "", "acdbcdbe"), "acdbcdb,b"); // "a(?:b|c|d){4,5}(.)"
    test_eq(run1_match("a(?:b|c|d){4,5}?(.)", "", "acdbcdbe"), "acdbcd,d"); // "a(?:b|c|d){4,5}?(.)"
    test_eq(run1_match("((foo)|(bar))*", "", "foobar"), "foobar,bar,,bar"); // "((foo)|(bar))*"
    test_eq(run1_match("a(?:b|c|d){6,7}(.)", "", "acdbcdbe"), "acdbcdbe,e"); // "a(?:b|c|d){6,7}(.)"
    test_eq(run1_match("a(?:b|c|d){6,7}?(.)", "", "acdbcdbe"), "acdbcdbe,e"); // "a(?:b|c|d){6,7}?(.)"
    test_eq(run1_match("a(?:b|c|d){5,6}(.)", "", "acdbcdbe"), "acdbcdbe,e"); // "a(?:b|c|d){5,6}(.)"
    test_eq(run1_match("a(?:b|c|d){5,6}?(.)", "", "acdbcdbe"), "acdbcdb,b"); // "a(?:b|c|d){5,6}?(.)"
    test_eq(run1_match("a(?:b|c|d){5,7}(.)", "", "acdbcdbe"), "acdbcdbe,e"); // "a(?:b|c|d){5,7}(.)"
    test_eq(run1_match("a(?:b|c|d){5,7}?(.)", "", "acdbcdbe"), "acdbcdb,b"); // "a(?:b|c|d){5,7}?(.)"
    test_eq(run1_match("a(?:b|(c|e){1,2}?|d)+?(.)", "", "ace"), "ace,c,e"); // "a(?:b|(c|e){1,2}?|d)+?(.)"
    test_eq(run1_match("^(.+)?B", "", "AB"), "AB,A"); // "^(.+)?B"
    test_eq(run1_match("^([^a-z])|(\\^)$", "", "."), ".,.,"); // "^([^a-z])|(\\^)$"
    test_eq(run1_match("^[<>]&", "", "<&OUT"), "<&"); // "^[<>]&"
    test_eq(run1_match("(?:(f)(o)(o)|(b)(a)(r))*", "", "foobar"), "foobar,,,,b,a,r"); // "(?:(f)(o)(o)|(b)(a)(r))*"
    test_eq(run1_match("(?:(f)(o)(o)|(b)(a)(r))*", "", "ab"), ",,,,,,"); // "(?:(f)(o)(o)|(b)(a)(r))*"
    test_eq(run1_match("(?:(f)(o)(o)|(b)(a)(r))*", "", "*** Failers"), ",,,,,,"); // "(?:(f)(o)(o)|(b)(a)(r))*"
    test_eq(run1_match("(?:(f)(o)(o)|(b)(a)(r))*", "", "cb"), ",,,,,,"); // "(?:(f)(o)(o)|(b)(a)(r))*"
    test_eq(run1_match("(?:(f)(o)(o)|(b)(a)(r))*", "", "b"), ",,,,,,"); // "(?:(f)(o)(o)|(b)(a)(r))*"
    test_eq(run1_match("(?:(f)(o)(o)|(b)(a)(r))*", "", "ab"), ",,,,,,"); // "(?:(f)(o)(o)|(b)(a)(r))*"
    test_eq(run1_match("(?:(f)(o)(o)|(b)(a)(r))*", "", "b"), ",,,,,,"); // "(?:(f)(o)(o)|(b)(a)(r))*"
    test_eq(run1_match("(?:(f)(o)(o)|(b)(a)(r))*", "", "b"), ",,,,,,"); // "(?:(f)(o)(o)|(b)(a)(r))*"
    test_eq(run1_match("(?:..)*a", "", "aba"), "aba"); // "(?:..)*a"
    test_eq(run1_match("(?:..)*?a", "", "aba"), "a"); // "(?:..)*?a"
    test_eq(run1_match("^(){3,5}", "", "abc"), ","); // "^(){3,5}"
    test_eq(run1_match("^(a+)*ax", "", "aax"), "aax,a"); // "^(a+)*ax"
    test_eq(run1_match("^((a|b)+)*ax", "", "aax"), "aax,a,a"); // "^((a|b)+)*ax"
    test_eq(run1_match("^((a|bc)+)*ax", "", "aax"), "aax,a,a"); // "^((a|bc)+)*ax"
    test_eq(run1_match("(a|x)*ab", "", "cab"), "ab,"); // "(a|x)*ab"
    test_eq(run1_match("(a)*ab", "", "cab"), "ab,"); // "(a)*ab"
    test_eq(run1_match("(a)*ab", "", "ab"), "ab,"); // "(a)*ab"
    test_eq(run1_match("(a)*ab", "", "ab"), "ab,"); // "(a)*ab"
    test_eq(run1_match("(a)*ab", "", "ab"), "ab,"); // "(a)*ab"
    test_eq(run1_match("(a)*ab", "", "ab"), "ab,"); // "(a)*ab"
    test_eq(run1_match("(a)*ab", "", "ab"), "ab,"); // "(a)*ab"
    test_eq(run1_match("(a)*ab", "", "ab"), "ab,"); // "(a)*ab"
    test_eq(run1_match("(a)*ab", "", "ab"), "ab,"); // "(a)*ab"
    test_eq(run1_match("(a)*ab", "", "ab"), "ab,"); // "(a)*ab"
    test_eq(run1_match("(?:c|d)(?:)(?:a(?:)(?:b)(?:b(?:))(?:b(?:)(?:b)))", "", "cabbbb"), "cabbbb"); // "(?:c|d)(?:)(?:a(?:)(?:b)(?:b(?:))(?:b(?:)(?:b)))"
    test_eq(run1_match("(?:c|d)(?:)(?:aaaaaaaa(?:)(?:bbbbbbbb)(?:bbbbbbbb(?:))(?:bbbbbbbb(?:)(?:bbbbbbbb)))", "", "caaaaaaaabbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb"), "caaaaaaaabbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb"); // "(?:c|d)(?:)(?:aaaaaaaa(?:)(?:bbbbbbbb)(?:bbbbbbbb(?:))(?:bbbbbbbb(?:)(?:bbbbbbbb)))"
    test_eq(run1_match("foo\\w*\\d{4}baz", "", "foobar1234baz"), "foobar1234baz"); // "foo\\w*\\d{4}baz"
    test_eq(run1_match("x(~~)*(?:(?:F)?)?", "", "x~~"), "x~~,~~"); // "x(~~)*(?:(?:F)?)?"
    test_eq(run1_match("^a{3}c", "", "aaac"), "aaac"); // "^a{3}c"
    test_eq(run1_match("^a{3}c", "", "aaac"), "aaac"); // "^a{3}c"
    test_eq(run1_match("(\\w+:)+", "", "one:"), "one:,one:"); // "(\\w+:)+"
    test_eq(run1_match("([\\w:]+::)?(\\w+)$", "", "abcd"), "abcd,,abcd"); // "([\\w:]+::)?(\\w+)$"
    test_eq(run1_match("([\\w:]+::)?(\\w+)$", "", "xy:z:::abcd"), "xy:z:::abcd,xy:z:::,abcd"); // "([\\w:]+::)?(\\w+)$"
    test_eq(run1_match("^[^bcd]*(c+)", "", "aexycd"), "aexyc,c"); // "^[^bcd]*(c+)"
    test_eq(run1_match("(a*)b+", "", "caab"), "aab,aa"); // "(a*)b+"
    test_eq(run1_match("([\\w:]+::)?(\\w+)$", "", "abcd"), "abcd,,abcd"); // "([\\w:]+::)?(\\w+)$"
    test_eq(run1_match("([\\w:]+::)?(\\w+)$", "", "xy:z:::abcd"), "xy:z:::abcd,xy:z:::,abcd"); // "([\\w:]+::)?(\\w+)$"
    test_eq(run1_match("([\\w:]+::)?(\\w+)$", "", "*** Failers"), "Failers,,Failers"); // "([\\w:]+::)?(\\w+)$"
    test_eq(run1_match("^[^bcd]*(c+)", "", "aexycd"), "aexyc,c"); // "^[^bcd]*(c+)"
    test_eq(run1_match("([[:]+)", "", "a:[b]:"), ":[,:["); // "([[:]+)"
    test_eq(run1_match("([[=]+)", "", "a=[b]="), "=[,=["); // "([[=]+)"
    test_eq(run1_match("([[.]+)", "", "a.[b]."), ".[,.["); // "([[.]+)"
    test_eq(run1_match("((Z)+|A)*", "", "ZABCDEFG"), "ZA,A,"); // "((Z)+|A)*"
    test_eq(run1_match("(Z()|A)*", "", "ZABCDEFG"), "ZA,A,"); // "(Z()|A)*"
    test_eq(run1_match("(Z(())|A)*", "", "ZABCDEFG"), "ZA,A,,"); // "(Z(())|A)*"
    test_eq(run1_match("(Z(())|A)*", "", "ZABCDEFG"), "ZA,A,,"); // "(Z(())|A)*"
    test_eq(run1_match("(Z(())|A)*", "", "ZABCDEFG"), "ZA,A,,"); // "(Z(())|A)*"
    test_eq(run1_match("a*", "", "abbab"), "a"); // "a*"
    // Skipping Unicode-unfriendly a*
    test_eq(run1_match("a*", "", "-things"), ""); // "a*"
    // Skipping global "a*" with string "0digit"
    // Skipping global "a*" with string "*** Failers"
    // Skipping global "a*" with string "bcdef    "
    // Skipping Unicode-unfriendly ^[\d-a]
    // Skipping Unicode-unfriendly ^[\d-a]
    // Skipping Unicode-unfriendly ^[\d-a]
    test_eq(run1_match("[\\s]+", "", "> \u{9}\n\u{c}\u{d}\u{b}<"), " \u{9}\n\u{c}\u{d}\u{b}"); // "[\\s]+"
    test_eq(run1_match("[\\s]+", "", " "), " "); // "[\\s]+"
    test_eq(run1_match("\\s+", "", "> \u{9}\n\u{c}\u{d}\u{b}<"), " \u{9}\n\u{c}\u{d}\u{b}"); // "\\s+"
    test_eq(run1_match("\\s+", "", " "), " "); // "\\s+"
    test_eq(run1_match("abc.", "", "abc1abc2xyzabc3 "), "abc1"); // "abc."
    // Skipping global "abc." with string "XabcY"
    // Skipping global "abc." with string "XabcY  "
    // Skipping global "abc." with string "abcE"
    // Skipping Unicode-unfriendly [\z\C]
    // Skipping Unicode-unfriendly [\z\C]
    // Skipping Unicode-unfriendly \M
    test_eq(run1_match("(a+)*b", "", "bbbbc"), "b,"); // "(a+)*b"
    test_eq(run1_match("(a+)*b", "", "abc"), "ab,a"); // "(a+)*b"
    test_eq(run1_match("(a+)*b", "", "bca"), "b,"); // "(a+)*b"
    test_eq(run1_match("(a+)*b", "", "abc"), "ab,a"); // "(a+)*b"
    test_eq(run1_match("(a+)*b", "", "bca"), "b,"); // "(a+)*b"
    test_eq(run1_match("(a+)*b", "", "abc"), "ab,a"); // "(a+)*b"
    test_eq(run1_match("(a+)*b", "", "abc"), "ab,a"); // "(a+)*b"
    test_eq(run1_match("line\\nbreak", "", "this is a line\nbreak"), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "", "line one\nthis is a line\nbreak in the second line "), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "", "this is a line\nbreak"), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "", "line one\nthis is a line\nbreak in the second line "), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "m", "this is a line\nbreak"), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("line\\nbreak", "m", "line one\nthis is a line\nbreak in the second line "), "line\nbreak"); // "line\\nbreak"
    test_eq(run1_match("^", "m", "a\nb\nc\n"), ""); // "^"
    // Skipping global "^" with string " "
    // Skipping global "^" with string "A\nC\nC\n "
    // Skipping global "^" with string "AB"
    // Skipping global "^" with string "aB  "
    // Skipping global "^" with string "AB"
    // Skipping global "^" with string "aB  "
    // Skipping global "^" with string "AB"
    // Skipping global "^" with string "aB  "
    // Skipping global "^" with string "AB"
    // Skipping global "^" with string "aB  "
    test_eq(run1_match("Content-Type\\x3A[^\\r\\n]{6,}", "", "Content-Type:xxxxxyyy "), "Content-Type:xxxxxyyy "); // "Content-Type\\x3A[^\\r\\n]{6,}"
    test_eq(run1_match("Content-Type\\x3A[^\\r\\n]{6,}z", "", "Content-Type:xxxxxyyyz"), "Content-Type:xxxxxyyyz"); // "Content-Type\\x3A[^\\r\\n]{6,}z"
    test_eq(run1_match("Content-Type\\x3A[^a]{6,}", "", "Content-Type:xxxyyy "), "Content-Type:xxxyyy "); // "Content-Type\\x3A[^a]{6,}"
    test_eq(run1_match("Content-Type\\x3A[^a]{6,}z", "", "Content-Type:xxxyyyz"), "Content-Type:xxxyyyz"); // "Content-Type\\x3A[^a]{6,}z"
    test_eq(run1_match("^abc", "m", "xyz\nabc"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\nabc<lf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}\nabc<lf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}abc<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}\nabc<crlf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\nabc<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}\nabc<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\nabc<crlf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}abc<crlf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}abc<lf>"), "abc"); // "^abc"
    test_eq(run1_match("abc$", "m", "xyzabc"), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\n "), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\npqr "), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\u{d}<cr> "), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\u{d}pqr<cr> "), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\u{d}\n<crlf> "), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\u{d}\npqr<crlf> "), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\u{d} "), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\u{d}pqr "), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\u{d}\n "), "abc"); // "abc$"
    test_eq(run1_match("abc$", "m", "xyzabc\u{d}\npqr "), "abc"); // "abc$"
    test_eq(run1_match("^abc", "m", "xyz\u{d}abcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\nabcdef<lf>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\nabcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\nabcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}abcdef<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}abcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}\nabcdef"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}abcdef<cr>"), "abc"); // "^abc"
    test_eq(run1_match("^abc", "m", "xyz\u{d}abcdef"), "abc"); // "^abc"
    test_eq(run1_match(".*", "", "abc\ndef"), "abc"); // ".*"
    test_eq(run1_match(".*", "", "abc\u{d}def"), "abc"); // ".*"
    test_eq(run1_match(".*", "", "abc\u{d}\ndef"), "abc"); // ".*"
    test_eq(run1_match(".*", "", "<cr>abc\ndef"), "<cr>abc"); // ".*"
    test_eq(run1_match(".*", "", "<cr>abc\u{d}def"), "<cr>abc"); // ".*"
    test_eq(run1_match(".*", "", "<cr>abc\u{d}\ndef"), "<cr>abc"); // ".*"
    test_eq(run1_match(".*", "", "<crlf>abc\ndef"), "<crlf>abc"); // ".*"
    test_eq(run1_match(".*", "", "<crlf>abc\u{d}def"), "<crlf>abc"); // ".*"
    test_eq(run1_match(".*", "", "<crlf>abc\u{d}\ndef"), "<crlf>abc"); // ".*"
    test_eq(run1_match("^\\w+=.*(\\\\\\n.*)*", "", "abc=xyz\\\npqr"), "abc=xyz\\,"); // "^\\w+=.*(\\\\\\n.*)*"
    test_eq(run1_match("^(a()*)*", "", "aaaa"), "aaaa,a,"); // "^(a()*)*"
    test_eq(run1_match("^(?:a(?:(?:))*)*", "", "aaaa"), "aaaa"); // "^(?:a(?:(?:))*)*"
    test_eq(run1_match("^(a()+)+", "", "aaaa"), "aaaa,a,"); // "^(a()+)+"
    test_eq(run1_match("^(?:a(?:(?:))+)+", "", "aaaa"), "aaaa"); // "^(?:a(?:(?:))+)+"
    test_eq(run1_match("^abc.", "m", "abc1 \nabc2 \u{b}abc3xx \u{c}abc4 \u{d}abc5xx \u{d}\nabc6 \u{85}abc7 JUNK"), "abc1"); // "^abc."
    test_eq(run1_match("abc.$", "m", "abc1\n abc2\u{b} abc3\u{c} abc4\u{d} abc5\u{d}\n abc6\u{85} abc9"), "abc1"); // "abc.$"
    // Skipping Unicode-unfriendly ^a\R*b
    // Skipping Unicode-unfriendly ^a[\R]b
    test_eq(run1_match(".+foo", "", "afoo"), "afoo"); // ".+foo"
    test_eq(run1_match(".+foo", "", "afoo"), "afoo"); // ".+foo"
    test_eq(run1_match(".+foo", "", "afoo"), "afoo"); // ".+foo"
    test_eq(run1_match(".+foo", "", "afoo"), "afoo"); // ".+foo"
    test_eq(run1_match("^$", "m", "abc\u{d}\u{d}xyz"), ""); // "^$"
    // Skipping global "^$" with string "abc\n\u{d}xyz  "
    // Skipping global "^$" with string "abc\u{d}\nxyz"
    test_eq(run1_match("^X", "m", "XABC"), "X"); // "^X"
    test_eq(run1_match("^X", "m", "XABCB"), "X"); // "^X"
    test_eq(run1_match("\\nA", "", "\u{d}\nA "), "\nA"); // "\\nA"
    test_eq(run1_match("[\\r\\n]A", "", "\u{d}\nA "), "\nA"); // "[\\r\\n]A"
    test_eq(run1_match("(\\r|\\n)A", "", "\u{d}\nA "), "\nA,\n"); // "(\\r|\\n)A"
    test_eq(run1_match("a(?!)|\\wbc", "", "abc "), "abc"); // "a(?!)|\\wbc"
    test_eq(run1_match("a[^]b", "", "aXb"), "aXb"); // "a[^]b"
    test_eq(run1_match("a[^]b", "", "a\nb "), "a\nb"); // "a[^]b"
    test_eq(run1_match("a[^]+b", "", "aXb"), "aXb"); // "a[^]+b"
    test_eq(run1_match("a[^]+b", "", "a\nX\nXb "), "a\nX\nXb"); // "a[^]+b"
    test_eq(run1_match("\\bX", "", "Xoanon"), "X"); // "\\bX"
    test_eq(run1_match("\\bX", "", "+Xoanon"), "X"); // "\\bX"
    test_eq(run1_match("\\bX", "", "x{300}Xoanon "), "X"); // "\\bX"
    test_eq(run1_match("\\BX", "", "YXoanon"), "X"); // "\\BX"
    test_eq(run1_match("X\\b", "", "X+oanon"), "X"); // "X\\b"
    test_eq(run1_match("X\\b", "", "FAX "), "X"); // "X\\b"
    test_eq(run1_match("X\\B", "", "Xoanon  "), "X"); // "X\\B"
    test_eq(run1_match("X\\B", "", "ZXx{300}oanon "), "X"); // "X\\B"
    test_eq(run1_match("[^a]", "", "abcd"), "b"); // "[^a]"
    test_eq(run1_match("[^a]", "", "ax{100}   "), "x"); // "[^a]"
    test_eq(run1_match("[^a]", "", "ab99"), "b"); // "[^a]"
    test_eq(run1_match("[^a]", "", "x{123}x{123}45"), "x"); // "[^a]"
    test_eq(run1_match("[^a]", "", "x{400}x{401}x{402}6  "), "x"); // "[^a]"
    test_eq(run1_match("[^a]", "", "*** Failers"), "*"); // "[^a]"
    test_eq(run1_match("[^a]", "", "d99"), "d"); // "[^a]"
    test_eq(run1_match("[^a]", "", "x{123}x{122}4   "), "x"); // "[^a]"
    test_eq(run1_match("[^a]", "", "x{400}x{403}6  "), "x"); // "[^a]"
    test_eq(run1_match("[^a]", "", "x{400}x{401}x{402}x{402}6  "), "x"); // "[^a]"
    test_eq(run1_match("a.b", "", "acb"), "acb"); // "a.b"
    test_eq(run1_match("a.b", "", "a\u{7f}b"), "a\u{7f}b"); // "a.b"
    test_eq(run1_match("a(.*?)(.)", "", "a\u{c0}\u{88}b"), "a\u{c0},,\u{c0}"); // "a(.*?)(.)"
    test_eq(run1_match("a(.*?)(.)", "", "ax{100}b"), "ax,,x"); // "a(.*?)(.)"
    test_eq(run1_match("a(.*)(.)", "", "a\u{c0}\u{88}b"), "a\u{c0}\u{88}b,\u{c0}\u{88},b"); // "a(.*)(.)"
    test_eq(run1_match("a(.*)(.)", "", "ax{100}b"), "ax{100}b,x{100},b"); // "a(.*)(.)"
    test_eq(run1_match("a(.)(.)", "", "a\u{c0}\u{92}bcd"), "a\u{c0}\u{92},\u{c0},\u{92}"); // "a(.)(.)"
    test_eq(run1_match("a(.)(.)", "", "ax{240}bcd"), "ax{,x,{"); // "a(.)(.)"
    test_eq(run1_match("a(.?)(.)", "", "a\u{c0}\u{92}bcd"), "a\u{c0}\u{92},\u{c0},\u{92}"); // "a(.?)(.)"
    test_eq(run1_match("a(.?)(.)", "", "ax{240}bcd"), "ax{,x,{"); // "a(.?)(.)"
    test_eq(run1_match("a(.??)(.)", "", "a\u{c0}\u{92}bcd"), "a\u{c0},,\u{c0}"); // "a(.??)(.)"
    test_eq(run1_match("a(.??)(.)", "", "ax{240}bcd"), "ax,,x"); // "a(.??)(.)"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}xyb "), "ax{1234}xyb,x{1234}xy"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}x{4321}yb "), "ax{1234}x{4321}yb,x{1234}x{4321}y"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}x{4321}x{3412}b "), "ax{1234}x{4321}x{3412}b,x{1234}x{4321}x{3412}"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "axxxxbcdefghijb "), "axxxxbcdefghijb,xxxxbcdefghij"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}x{4321}x{3412}x{3421}b "), "ax{1234}x{4321}x{3412}x{3421}b,x{1234}x{4321}x{3412}x{3421}"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,})b", "", "ax{1234}b "), "ax{1234}b,x{1234}"); // "a(.{3,})b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}xyb "), "ax{1234}xyb,x{1234}xy"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}x{4321}yb "), "ax{1234}x{4321}yb,x{1234}x{4321}y"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}x{4321}x{3412}b "), "ax{1234}x{4321}x{3412}b,x{1234}x{4321}x{3412}"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "axxxxbcdefghijb "), "axxxxb,xxxx"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}x{4321}x{3412}x{3421}b "), "ax{1234}x{4321}x{3412}x{3421}b,x{1234}x{4321}x{3412}x{3421}"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,}?)b", "", "ax{1234}b "), "ax{1234}b,x{1234}"); // "a(.{3,}?)b"
    test_eq(run1_match("a(.{3,5})b", "", "axxxxbcdefghijb "), "axxxxb,xxxx"); // "a(.{3,5})b"
    test_eq(run1_match("a(.{3,5})b", "", "axbxxbcdefghijb "), "axbxxb,xbxx"); // "a(.{3,5})b"
    test_eq(run1_match("a(.{3,5})b", "", "axxxxxbcdefghijb "), "axxxxxb,xxxxx"); // "a(.{3,5})b"
    test_eq(run1_match("a(.{3,5}?)b", "", "axxxxbcdefghijb "), "axxxxb,xxxx"); // "a(.{3,5}?)b"
    test_eq(run1_match("a(.{3,5}?)b", "", "axbxxbcdefghijb "), "axbxxb,xbxx"); // "a(.{3,5}?)b"
    test_eq(run1_match("a(.{3,5}?)b", "", "axxxxxbcdefghijb "), "axxxxxb,xxxxx"); // "a(.{3,5}?)b"
    test_eq(run1_match("[^a]+", "", "bcd"), "bcd"); // "[^a]+"
    // Skipping Unicode-unfriendly [^a]+
    test_eq(run1_match("^[^a]{2}", "", "x{100}bc"), "x{"); // "^[^a]{2}"
    test_eq(run1_match("^[^a]{2,}", "", "x{100}bcAa"), "x{100}bcA"); // "^[^a]{2,}"
    test_eq(run1_match("^[^a]{2,}?", "", "x{100}bca"), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("[^a]+", "i", "bcd"), "bcd"); // "[^a]+"
    // Skipping Unicode-unfriendly [^a]+
    test_eq(run1_match("^[^a]{2}", "i", "x{100}bc"), "x{"); // "^[^a]{2}"
    test_eq(run1_match("^[^a]{2,}", "i", "x{100}bcAa"), "x{100}bc"); // "^[^a]{2,}"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}bca"), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}x{100} "), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}x{100} "), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}x{100}x{100}x{100} "), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "x{100}x{100}x{100}x{100} "), "x{"); // "^[^a]{2,}?"
    test_eq(run1_match("^[^a]{2,}?", "i", "Xyyyax{100}x{100}bXzzz"), "Xy"); // "^[^a]{2,}?"
    test_eq(run1_match("\\D", "", "1X2"), "X"); // "\\D"
    test_eq(run1_match("\\D", "", "1x{100}2 "), "x"); // "\\D"
    test_eq(run1_match(">\\S", "", "> >X Y"), ">X"); // ">\\S"
    test_eq(run1_match(">\\S", "", "> >x{100} Y"), ">x"); // ">\\S"
    test_eq(run1_match("\\d", "", "x{100}3"), "1"); // "\\d"
    test_eq(run1_match("\\s", "", "x{100} X"), " "); // "\\s"
    test_eq(run1_match("\\D+", "", "12abcd34"), "abcd"); // "\\D+"
    test_eq(run1_match("\\D+", "", "*** Failers"), "*** Failers"); // "\\D+"
    test_eq(run1_match("\\D+", "", "1234  "), "  "); // "\\D+"
    test_eq(run1_match("\\D{2,3}", "", "12abcd34"), "abc"); // "\\D{2,3}"
    test_eq(run1_match("\\D{2,3}", "", "12ab34"), "ab"); // "\\D{2,3}"
    test_eq(run1_match("\\D{2,3}", "", "*** Failers  "), "***"); // "\\D{2,3}"
    test_eq(run1_match("\\D{2,3}", "", "12a34  "), "  "); // "\\D{2,3}"
    test_eq(run1_match("\\D{2,3}?", "", "12abcd34"), "ab"); // "\\D{2,3}?"
    test_eq(run1_match("\\D{2,3}?", "", "12ab34"), "ab"); // "\\D{2,3}?"
    test_eq(run1_match("\\D{2,3}?", "", "*** Failers  "), "**"); // "\\D{2,3}?"
    test_eq(run1_match("\\D{2,3}?", "", "12a34  "), "  "); // "\\D{2,3}?"
    test_eq(run1_match("\\d+", "", "12abcd34"), "12"); // "\\d+"
    test_eq(run1_match("\\d{2,3}", "", "12abcd34"), "12"); // "\\d{2,3}"
    test_eq(run1_match("\\d{2,3}", "", "1234abcd"), "123"); // "\\d{2,3}"
    test_eq(run1_match("\\d{2,3}?", "", "12abcd34"), "12"); // "\\d{2,3}?"
    test_eq(run1_match("\\d{2,3}?", "", "1234abcd"), "12"); // "\\d{2,3}?"
    test_eq(run1_match("\\S+", "", "12abcd34"), "12abcd34"); // "\\S+"
    test_eq(run1_match("\\S+", "", "*** Failers"), "***"); // "\\S+"
    test_eq(run1_match("\\S{2,3}", "", "12abcd34"), "12a"); // "\\S{2,3}"
    test_eq(run1_match("\\S{2,3}", "", "1234abcd"), "123"); // "\\S{2,3}"
    test_eq(run1_match("\\S{2,3}", "", "*** Failers"), "***"); // "\\S{2,3}"
    test_eq(run1_match("\\S{2,3}?", "", "12abcd34"), "12"); // "\\S{2,3}?"
    test_eq(run1_match("\\S{2,3}?", "", "1234abcd"), "12"); // "\\S{2,3}?"
    test_eq(run1_match("\\S{2,3}?", "", "*** Failers"), "**"); // "\\S{2,3}?"
    test_eq(run1_match(">\\s+<", "", "12>      <34"), ">      <"); // ">\\s+<"
    test_eq(run1_match(">\\s{2,3}<", "", "ab>  <cd"), ">  <"); // ">\\s{2,3}<"
    test_eq(run1_match(">\\s{2,3}<", "", "ab>   <ce"), ">   <"); // ">\\s{2,3}<"
    test_eq(run1_match(">\\s{2,3}?<", "", "ab>  <cd"), ">  <"); // ">\\s{2,3}?<"
    test_eq(run1_match(">\\s{2,3}?<", "", "ab>   <ce"), ">   <"); // ">\\s{2,3}?<"
    test_eq(run1_match("\\w+", "", "12      34"), "12"); // "\\w+"
    test_eq(run1_match("\\w+", "", "*** Failers"), "Failers"); // "\\w+"
    test_eq(run1_match("\\w{2,3}", "", "ab  cd"), "ab"); // "\\w{2,3}"
    test_eq(run1_match("\\w{2,3}", "", "abcd ce"), "abc"); // "\\w{2,3}"
    test_eq(run1_match("\\w{2,3}", "", "*** Failers"), "Fai"); // "\\w{2,3}"
    test_eq(run1_match("\\w{2,3}?", "", "ab  cd"), "ab"); // "\\w{2,3}?"
    test_eq(run1_match("\\w{2,3}?", "", "abcd ce"), "ab"); // "\\w{2,3}?"
    test_eq(run1_match("\\w{2,3}?", "", "*** Failers"), "Fa"); // "\\w{2,3}?"
    test_eq(run1_match("\\W+", "", "12====34"), "===="); // "\\W+"
    test_eq(run1_match("\\W+", "", "*** Failers"), "*** "); // "\\W+"
    test_eq(run1_match("\\W+", "", "abcd "), " "); // "\\W+"
    test_eq(run1_match("\\W{2,3}", "", "ab====cd"), "==="); // "\\W{2,3}"
    test_eq(run1_match("\\W{2,3}", "", "ab==cd"), "=="); // "\\W{2,3}"
    test_eq(run1_match("\\W{2,3}", "", "*** Failers"), "***"); // "\\W{2,3}"
    test_eq(run1_match("\\W{2,3}?", "", "ab====cd"), "=="); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "ab==cd"), "=="); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers "), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers "), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "X  "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "X  "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "X  "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "x{200}X   "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "x{200}X   "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "x{200}X   "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "*** Failers"), "**"); // "\\W{2,3}?"
    test_eq(run1_match("\\W{2,3}?", "", "y    "), "  "); // "\\W{2,3}?"
    test_eq(run1_match("[\\xFF]", "", ">\u{ff}<"), "\u{ff}"); // "[\\xFF]"
    test_eq(run1_match("[^\\xFF]", "", "XYZ"), "X"); // "[^\\xFF]"
    test_eq(run1_match("[^\\xff]", "", "XYZ"), "X"); // "[^\\xff]"
    test_eq(run1_match("[^\\xff]", "", "x{123} "), "x"); // "[^\\xff]"
    test_eq(run1_match("(|a)", "", "catac"), ","); // "(|a)"
    // Skipping global "(|a)" with string "ax{256}a "
    // Skipping global "(|a)" with string "x{85}"
    test_eq(run1_match("^abc.", "m", "abc1 \nabc2 \u{b}abc3xx \u{c}abc4 \u{d}abc5xx \u{d}\nabc6 x{0085}abc7 x{2028}abc8 x{2029}abc9 JUNK"), "abc1"); // "^abc."
    test_eq(run1_match("abc.$", "m", "abc1\n abc2\u{b} abc3\u{c} abc4\u{d} abc5\u{d}\n abc6x{0085} abc7x{2028} abc8x{2029} abc9"), "abc1"); // "abc.$"
    // Skipping Unicode-unfriendly ^a\R*b
    test_eq(run1_match("X", "", "Ax{1ec5}ABCXYZ"), "X"); // "X"
    // Skipping Unicode-unfriendly \X?abc
    // Skipping Unicode-unfriendly \X?abc
    // Skipping Unicode-unfriendly \X?abc
    // Skipping Unicode-unfriendly \X?abc
    // Skipping Unicode-unfriendly ^\X?abc
    // Skipping Unicode-unfriendly \X*abc
    // Skipping Unicode-unfriendly \X*abc
    // Skipping Unicode-unfriendly \X*abc
    // Skipping Unicode-unfriendly \X*abc
    // Skipping Unicode-unfriendly ^\X*abc
    // Skipping Unicode-unfriendly [\p{Nd}]
    // Skipping Unicode-unfriendly [\p{Nd}]
    // Skipping Unicode-unfriendly [\P{Nd}]+
    test_eq(run1_match("\\D+", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "\\D+"
    test_eq(run1_match("\\D+", "", " "), " "); // "\\D+"
    test_eq(run1_match("\\D+", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "\\D+"
    test_eq(run1_match("[\\D]+", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "[\\D]+"
    test_eq(run1_match("[\\D\\P{Nd}]+", "", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"), "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"); // "[\\D\\P{Nd}]+"
    // Skipping Unicode-unfriendly ^[\X]
    // Skipping Unicode-unfriendly ^(\X*)(.)
    // Skipping Unicode-unfriendly ^(\X*)(.)
    // Skipping Unicode-unfriendly ^(\X*?)(.)
    // Skipping Unicode-unfriendly ^(\X*?)(.)
    test_eq(run1_match("^[\\p{Any}]X", "", "AXYZ"), "AX"); // "^[\\p{Any}]X"
    // Skipping Unicode-unfriendly ^[\P{Any}]X
    test_eq(run1_match("^[\\p{Any}]?X", "", "XYZ"), "X"); // "^[\\p{Any}]?X"
    test_eq(run1_match("^[\\p{Any}]?X", "", "AXYZ"), "AX"); // "^[\\p{Any}]?X"
    test_eq(run1_match("^[\\P{Any}]?X", "", "XYZ"), "X"); // "^[\\P{Any}]?X"
    // Skipping Unicode-unfriendly ^[\P{Any}]?X
    test_eq(run1_match("^[\\p{Any}]+X", "", "AXYZ"), "AX"); // "^[\\p{Any}]+X"
    // Skipping Unicode-unfriendly ^[\P{Any}]+X
    test_eq(run1_match("^[\\p{Any}]*X", "", "XYZ"), "X"); // "^[\\p{Any}]*X"
    test_eq(run1_match("^[\\p{Any}]*X", "", "AXYZ"), "AX"); // "^[\\p{Any}]*X"
    test_eq(run1_match("^[\\P{Any}]*X", "", "XYZ"), "X"); // "^[\\P{Any}]*X"
    // Skipping Unicode-unfriendly ^[\P{Any}]*X
}
