/*
    SPDX-FileCopyrightText: 2007 Friedrich W. H. Kossebau <kossebau@kde.org>

    SPDX-License-Identifier: LGPL-2.1-only OR LGPL-3.0-only OR LicenseRef-KDE-Accepted-LGPL
*/

#include "bytetablemodel.hpp"

// Okteta core
#include <Okteta/Character>
#include <Okteta/CharCodec>
#include <Okteta/ValueCodec>
// KF
#include <KLocalizedString>
#include <KColorScheme>
// Qt
#include <QApplication>
#include <QFontDatabase>

namespace Kasten {

static constexpr QChar ByteTableDefaultSubstituteChar =  QLatin1Char('.');
static constexpr QChar ByteTableDefaultUndefinedChar =   QChar(QChar::ReplacementCharacter);
static constexpr int ByteSetSize = 256;

ByteTableModel::ByteTableModel(QObject* parent)
    : QAbstractTableModel(parent)
    , mCharCodec(Okteta::CharCodec::createCodecForLocale())
    , mSubstituteChar(ByteTableDefaultSubstituteChar)
    , mUndefinedChar(ByteTableDefaultUndefinedChar)
    , mFixedFont(QFontDatabase::systemFont(QFontDatabase::FixedFont))
{
    constexpr Okteta::ValueCoding CodingIds[NofOfValueCodings] = {
        Okteta::DecimalCoding,
        Okteta::HexadecimalCoding,
        Okteta::OctalCoding,
        Okteta::BinaryCoding
    };
    for (std::size_t i = 0; i < NofOfValueCodings; ++i) {
        mValueCodec[i] = Okteta::ValueCodec::createCodec(CodingIds[i]);
    }
}

ByteTableModel::~ByteTableModel() = default;

void ByteTableModel::setSubstituteChar(QChar substituteChar)
{
    if (substituteChar.isNull()) {
        substituteChar = ByteTableDefaultSubstituteChar;
    }

    if (mSubstituteChar == substituteChar) {
        return;
    }

    mSubstituteChar = substituteChar;

    Q_EMIT dataChanged(index(0, CharacterId), index(ByteSetSize - 1, CharacterId));
}

void ByteTableModel::setUndefinedChar(QChar undefinedChar)
{
    if (undefinedChar.isNull()) {
        undefinedChar = ByteTableDefaultUndefinedChar;
    }

    if (mUndefinedChar == undefinedChar) {
        return;
    }

    mUndefinedChar = undefinedChar;

    Q_EMIT dataChanged(index(0, CharacterId), index(ByteSetSize - 1, CharacterId));
}

void ByteTableModel::setCharCodec(const QString& codecName)
{
    if (codecName == mCharCodec->name()) {
        return;
    }

    mCharCodec = Okteta::CharCodec::createCodec(codecName);

    Q_EMIT dataChanged(index(0, CharacterId), index(ByteSetSize - 1, CharacterId));
}

int ByteTableModel::rowCount(const QModelIndex& parent) const
{
    return (!parent.isValid()) ? ByteSetSize : 0;
}

int ByteTableModel::columnCount(const QModelIndex& parent) const
{
    return (!parent.isValid()) ? NoOfIds : 0;
}

QVariant ByteTableModel::data(const QModelIndex& index, int role) const
{
    QVariant result;
    if (role == Qt::DisplayRole) {
        QString content;

        const unsigned char byte = index.row();
        const int column = index.column();
        if (column == CharacterId) {
            const Okteta::Character decodedChar = mCharCodec->decode(byte);
            content =
                decodedChar.isUndefined() ?
                    QString(mUndefinedChar) :
                !decodedChar.isPrint() ?
                    QString(mSubstituteChar) :
                    QString(static_cast<QChar>(decodedChar));
            // TODO: show proper descriptions for all control values, incl. space and delete
            // cmp. KCharSelect
        } else if (column < CharacterId) {
            mValueCodec[column]->encode(&content, 0, byte);
        }

        result = content;
    } else if (role == Qt::ForegroundRole) {
        const int column = index.column();
        if (column == CharacterId) {
            const unsigned char byte = index.row();
            const Okteta::Character decodedChar = mCharCodec->decode(byte);
            if (decodedChar.isUndefined() || !decodedChar.isPrint()) {
                const QPalette& palette = QApplication::palette();
                const KColorScheme colorScheme(palette.currentColorGroup(), KColorScheme::View);
                result = colorScheme.foreground(KColorScheme::InactiveText);
            }
        }
    } else if (role == Qt::ToolTipRole) {
        const int column = index.column();
        if (column == CharacterId) {
            const unsigned char byte = index.row();
            const Okteta::Character decodedChar = mCharCodec->decode(byte);
            if (decodedChar.isUndefined()) {
                result = i18nc("@info:tooltip", "Character not defined.");
            }
        }
    } else if (role == Qt::TextAlignmentRole) {
        result = static_cast<int>(Qt::AlignVCenter | Qt::AlignRight);
    } else if (role == Qt::FontRole) {
        result = mFixedFont;
    }

    return result;
}

QVariant ByteTableModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    QVariant result;

    if (role == Qt::DisplayRole) {
        const QString title =
            section == DecimalId ?     i18nc("@title:column short for Decimal",     "Dec") :
            section == HexadecimalId ? i18nc("@title:column short for Hexadecimal", "Hex") :
            section == OctalId ?       i18nc("@title:column short for Octal",       "Oct") :
            section == BinaryId ?      i18nc("@title:column short for Binary",      "Bin") :
            section == CharacterId ?   i18nc("@title:column short for Character",   "Char") :
            QString();
        result = title;
    } else if (role == Qt::ToolTipRole) {
        const QString title =
            section == DecimalId ?
                i18nc("@info:tooltip column contains the value in decimal format",     "Decimal") :
            section == HexadecimalId ?
                i18nc("@info:tooltip column contains the value in hexadecimal format", "Hexadecimal") :
            section == OctalId ?
                i18nc("@info:tooltip column contains the value in octal format",       "Octal") :
            section == BinaryId ?
                i18nc("@info:tooltip column contains the value in binary format",      "Binary") :
            section == CharacterId ?
                i18nc("@info:tooltip column contains the character with the value",    "Character") :
                QString();
        result = title;
    } else {
        result = QAbstractTableModel::headerData(section, orientation, role);
    }

    return result;
}

}

#include "moc_bytetablemodel.cpp"
