// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use crate::graph::{GraphWriteError, MutableGraph, NodeID};
use crate::testing::graph_in_mem::InMemoryGraph;
use crate::testing::ordering::NodeComparator;
use hex::FromHex;
use hex::FromHexError;
use std::convert::From;
use std::io::{BufRead, Error, Lines};
use std::result::Result;
use std::result::Result::{Err, Ok};
use std::string::String;
use std::vec::Vec;

#[derive(Debug)]
pub enum GraphImportError {
    GraphWriteError(GraphWriteError),
    InvalidEntryError(String),
    DecodeError(FromHexError),
    IOError(std::io::Error),
}

impl From<GraphWriteError> for GraphImportError {
    fn from(error: GraphWriteError) -> Self {
        GraphImportError::GraphWriteError(error)
    }
}

impl From<FromHexError> for GraphImportError {
    fn from(error: FromHexError) -> Self {
        GraphImportError::DecodeError(error)
    }
}

impl From<std::io::Error> for GraphImportError {
    fn from(error: Error) -> Self {
        GraphImportError::IOError(error)
    }
}

pub fn read_graph<B: BufRead, C: NodeComparator>(
    lines: Lines<B>,
) -> Result<InMemoryGraph<C>, GraphImportError> {
    let mut graph = InMemoryGraph::<C>::new();

    for res_line in lines {
        let line = res_line?;
        let entry: Vec<&str> = line.split_ascii_whitespace().collect();

        match entry[..] {
            [id, p1, p2] => graph.push(
                <NodeID>::from_hex(id)?,
                <NodeID>::from_hex(p1)?,
                <NodeID>::from_hex(p2)?,
            )?,
            [] => continue, // blank line
            _ => return Err(GraphImportError::InvalidEntryError(line)),
        };
    }

    Ok(graph)
}

#[cfg(test)]
mod tests {
    use crate::graph::{Graph, LabelledGraph, NodeID, Parents, NULL_REVISION};
    use crate::testing::graph_io::read_graph;
    use crate::testing::ordering::NodeIDComparator;
    use hex::FromHex;
    use std::io::BufRead;
    use std::io::Cursor;

    #[test]
    fn test_read_graph() {
        let input = r#"
            15a36165a83fd3b7e32b57879424f6dbbce1b84d 0000000000000000000000000000000000000000 0000000000000000000000000000000000000000
            c6a9e12475572c7b1a26c4f6135aa22c2b8d8626 15a36165a83fd3b7e32b57879424f6dbbce1b84d 0000000000000000000000000000000000000000
            d9e73a237d40e9c6e95050b2a190eaa1c9e8e178 c6a9e12475572c7b1a26c4f6135aa22c2b8d8626 0000000000000000000000000000000000000000
        "#;

        let lines = Cursor::new(input).lines();
        let graph = read_graph::<_, NodeIDComparator>(lines).unwrap();

        let rev3_id =
            <NodeID>::from_hex("d9e73a237d40e9c6e95050b2a190eaa1c9e8e178")
                .unwrap();

        let rev = graph.rev_of_node_id(rev3_id).unwrap();
        assert_eq!(graph.node_id(rev).unwrap(), rev3_id);
        assert_eq!(graph.parents(rev).unwrap(), Parents([1, NULL_REVISION]));
    }
}
