/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef UTL_H
#define UTL_H

#include <stdlib.h>
#include <ctype.h>
#include <iostream>

#include "constant.h"
#include "dvec20.h"
#include "cube.h"

#ifdef PARALLEL
#include <mpi.h>
#endif

using namespace std;

class Utl {
public :
	//Get the min val of val1 and val2
	template <class T>
	inline static T getMin(const T val1, const T val2) {
		return val1 < val2 ? val1 : val2;
	}

	//--------------------------------------------------------------------
	//Get the min val of val1 and val2, val3
	template <class T>
	inline static T getMin(const T val1, const T val2, const T val3) {
		return getMin (val1, getMin (val2, val3));
	}

	//--------------------------------------------------------------------
	//Get the max val of val1 and val2
	template <class T>
	inline static T getMax(const T val1, const T val2) {
		return val1 > val2 ? val1 : val2;
	}

	//--------------------------------------------------------------------
	//Get the max val of val1 and val2, val3
	template <class T>
	inline static T getMax(const T val1, const T val2, const T val3) {
		return getMax (val1, getMax (val2, val3) );
	}

	//--------------------------------------------------------------------
	//check whether or not the val2 is at the middle of val1 and val3
	template <class T>
	static int isMid (const T val1, const T val2, const T val3) {
		if ( (val2 - val1) * (val2 - val3) < 0)
			return 1;
		else
			return 0;
	}

	//--------------------------------------------------------------------
	//Get the middle val of val1 and val2, val3
	template <class T>
	inline static T getMid(const T val1, const T val2, const T val3) {
		if (isMid (val1, val2, val3) == 1 )
			return val1;
		else
			if (isMid (val2, val1, val3) == 1 )
				return val2;
			else
				return val3;
	}

	//--------------------------------------------------------------------
	template <class T>
	inline static void swap(T &val1, T &val2) {
		T temp_ = val1;
		val1 = val2;
		val2 = temp_;
	}

	//--------------------------------------------------------------------
	//after orderInc val1 < val2
	template <class T>
	inline static void orderInc(T &val1, T &val2) {
		if (val1 > val2)
			swap(val1, val2);
	}

	//--------------------------------------------------------------------
	//after orderDec val1 > val2
	template <class T>
	inline static void orderDec(T &val1, T &val2) {
		if (val1 < val2)
			swap(val1, val2);
	}

	//--------------------------------------------------------------------
	//after orderDec val1 > val2 > val3
	template <class T>
	inline static void orderDec(T &val1, T &val2, T &val3) {
		orderDec (val1, val2);
		orderDec (val1, val3);
		orderDec (val2, val3);
	}

	//--------------------------------------------------------------------
	inline static void announceError (const char *error) {
#ifdef PARALLEL
		cout << "Worker " << mpi_myrank << "-> ";
#endif
		if (error == BAD_CMP_LI) {
			cout << error << " fixed" << endl;
		} else {
			std::cout << "Error: " << error << endl;

#ifdef PARALLEL
			MPI_Finalize();
#endif
			exit( EXIT_FAILURE);
		}
	}

	//--------------------------------------------------------------------
	//order the val1, val2, val3, val3 such that val1 < val2 < val3 < val4
	template <class T>
	static void orderInc (T &val1,  T &val2, T &val3, T &val4) {
		orderInc (val1, val2);
		orderInc (val1, val3);
		orderInc (val1, val4);
		orderInc (val2, val3);
		orderInc (val2, val4);
		orderInc (val3, val4);
	}

	//--------------------------------------------------------------------
	//check whether or not ch is a control character
	static int isControlChar (char ch) {
		if (ch <= LAST_CONTROL_CHAR)
			return 1;
		else
			return 0;
	}

	//--------------------------------------------------------------------
	//check whether (pair00, pair01) = (pair10, pair11)
	static int isEqualPair (int pair00, int pair01, int pair10, int pair11) {
		if ( (getMin (pair00, pair01) == getMin (pair10, pair11) ) &&
		        (getMax (pair00, pair01) == getMax (pair10, pair11) ) )
			return 1;
		return 0;
	}

	static void getMem (Cube &cube, int nX, int nY, int nZ) {
		cube = new PMat[nX];
		int x_, y_;
		for (x_ = 0; x_ < nX; x_ ++) {
			cube[x_] = new PVec[nY];
			for (y_ = 0; y_ < nY; y_ ++)
				cube[x_][y_] = new LDOUBLE[nZ];
		}
	}

	static void getMem (PMat &mat, int nX, int nY) {
		mat = new PVec[nX];
		int x_;
		for (x_ = 0; x_ < nX; x_ ++) {
			mat[x_] = new LDOUBLE[nY];
		}
	}

	static void getMem (PVec &vec, int nX) {
		vec = new LDOUBLE[nX];
	}


	static void delMem (Cube &cube, int nX, int nY) {
		if (cube == 0)
			return;
		int x_, y_;
		for (x_ = 0; x_ < nX; x_ ++) {
			for (y_ = 0; y_ < nY; y_ ++)
				delete [] cube[x_][y_];
			delete [] cube[x_];
		}
		delete [] cube;
		cube = 0;
	}

	static void delMem (PMat &mat, int nX) {
		if (mat == 0)
			return;
		int x_;
		for (x_ = 0; x_ < nX; x_ ++) {
			delete [] mat[x_];
		}
		delete [] mat;
		mat = 0;
	}

	static void delMem (PVec &vec) {
		if (vec == 0)
			return;
		delete [] vec;
		vec = 0;
	}

	static int isBlankLine (const char *line) {
		int pos_ = 0;
		while (line[pos_] != '\0') {
			if (line[pos_] > LAST_CONTROL_CHAR)
				return 0;
			pos_ ++;
		}
		return 1;
	}

	//=======================================================
	//check whether or not ch is a useful character for newick tree form
	static int isTreeChar (char ch) {
		if (
		    (  ch == '(') || (ch == ')'  ) ||
		    ( (ch >= '0') && (ch <= '9') ) ||
		    ( (tolower (ch) >= 'a') && (toupper (ch) <= 'z') ) ||
		    ( (ch == ':') || (ch == ',') || (ch == '_') || (ch == '.')  || (ch=='-') || ch==';')
		)
			return 1;
		else
			return 0;
	}

}
; //end of class

#endif
