/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <ctype.h>

#include "outstream.h"
#include "usertree.h"

//=======================================================
//the constructor function
UserTree::UserTree () {
	doConstructor ();
	isUrTreeForm_ = -1;
	newickTreeStatus_ = NEWICK_TREE_FORM;
}



//=======================================================
//check if it is a tree newick tree form
int UserTree::isNewickTree (Vec<char> &newickTree) {
	//first check the open and close brakets
	if (newickTree[0] != '(')
		return MISSING_FIRST_OPEN_BRACKET;

	int nOpenBracket_ = 0;
	int nCloseBracket_ = 0;
	int treeLen_ = newickTree.getSize ();
	for (int count_ = 0; count_ < treeLen_; count_ ++) {
		if (newickTree[count_] == '(')
			nOpenBracket_ ++;
		if (newickTree[count_] == ')' )
			nCloseBracket_ ++;
		if (nCloseBracket_ > nOpenBracket_)
			return WRONG_BRACKET_ORDER;
	}

	if (newickTree[treeLen_ - 2] != ')' || newickTree[treeLen_ - 1] != ';')
		return MISSING_LAST_CLOSE_BRACKET;

	if (nOpenBracket_ > nCloseBracket_)
		return MISSING_CLOSE_BRACKET;
	//finish checking the bracket errors
	return NEWICK_TREE_FORM;
}

//=======================================================
//read the user tree from the file
void UserTree::readFile (std::istream &in) {

	inputNewickTree_.set (MAX_NEWICK_TREE_LEN, 0);
	char ch_;
	while (!in.eof () ) {
		in >> ch_;
		if (ch_ == ';') {
			inputNewickTree_ += ch_;
			break;
		}

		if (Utl::isTreeChar (ch_) == 1) {
			inputNewickTree_ += ch_;
			ch_ = '*';
		}
	}
}



//=======================================================
//read the user tree from the file
void UserTree::readFile (const char *userTreeFile) {
	ifstream in;
	in.open (userTreeFile);
	if (in == 0)
		Utl::announceError ("Cannot open the user tree file ...");

	readFile(in);

	in.close ();
}

//=======================================================
//read the user tree from the file
void UserTree::setNewickTree (Vec<char> &newickTree) {
	inputNewickTree_ = newickTree;
}


//=======================================================
//compute the number external nodes of this newick tree form
int UserTree::cmpNumExNd (Vec<char> &newickTree) {
	int nExNd_ = 0;
	for (int count_ = 0; count_ < newickTree.getSize (); count_ ++)
		if ( (newickTree[count_] == ':') &&
		        (newickTree[count_ - 1] != ')' ) )
			nExNd_ ++;
	return nExNd_;
}

//=======================================================
/* (  (subTree0), subTree1, (subTree2), ... )
find the  eight sub tree point form the given point 
return -1, if not found*/
int UserTree::findRightSubTreePoint (Vec<char> &newickTree, int startPoint) {
	int treeLen_ = newickTree.getSize ();
	if (startPoint > treeLen_ - 1)
		return -1;

	int nOpenBracket_ = 0;
	int nCloseBracket_ = 0;
	for (int count_ = startPoint; count_ < treeLen_; count_ ++) {
		if ( (newickTree[count_] == ',') &&
		        (nCloseBracket_ == nOpenBracket_) )
			return (count_ - 1);
		if (newickTree[count_] == '(')
			nOpenBracket_ ++;

		if (newickTree[count_] ==')' )
			nCloseBracket_ ++;
	}

	return (treeLen_ - 2);
}

//=======================================================
// (  (subTree0), (subTree1), (subTree2), ... () )
//compute the number of sub trees of this newick tree form
int UserTree::cmpNumSubTree (Vec<char> &newickTree) {
	int rightSubTreePoint_ = -1;
	int nSubTree_ = 0;
	do {
		rightSubTreePoint_ = findRightSubTreePoint (newickTree, rightSubTreePoint_ + 2);
		if (rightSubTreePoint_ != -1)
			nSubTree_ ++;
	} while (rightSubTreePoint_ != -1);

	return nSubTree_;
}


//=======================================================
//check if the newick tree form is a rooted or unrooted tree form
int UserTree::isUrTreeForm (Vec<char> &newickTree) {
	if (isUrTreeForm_ == -1) {
		if (cmpNumSubTree (newickTree) > 2)
			isUrTreeForm_ = 1;
		else
			isUrTreeForm_ = 0;
	}

	return isUrTreeForm_;
}

//=======================================================
/*get the subtree of this newick tree. the subtree is end at rightPoint,
return the leftPoint of this subtree in the newick tree */
int UserTree::getSubTree (Vec<char> &newickTree, int rightPoint, Vec<char> &subTree, double &brLen_) {
	char colon_ = ':';
	int colonPoint_ = newickTree.findRightItem (colon_, rightPoint);

	if (colonPoint_ == -1) {
		newickTreeStatus_ = MISSING_COLON;
		return -1;
	}

	char *sBrLen_ = 0;
	sBrLen_ = new char [rightPoint - colonPoint_ + 1];

	int count_;
	for (count_ = colonPoint_ + 1; count_ <= rightPoint; count_ ++)
		sBrLen_[count_ - (colonPoint_ + 1) ] = newickTree[count_];
	sBrLen_[count_ - (colonPoint_ + 1)] = '\0';

	char *stopPoint_;
	//  brLen_ = strtod (sBrLen_, &stopPoint_) * COF_BR;
	brLen_ = strtod (sBrLen_, &stopPoint_);
	delete [] sBrLen_;

	int nOpenBracket_ = 0;
	int nCloseBracket_ = 0;
	subTree.set (colonPoint_, 0);
	if (newickTree[colonPoint_ - 1] != ')') {//the case the subtree will be a leaf
		for (count_ = colonPoint_ - 1; count_ >= 0; count_ --) {
			if (newickTree[count_] ==  '(' || newickTree[count_] ==  ',')
				return count_ + 1;

			else
				subTree.insert (0, newickTree[count_]);
		}// end of for count_

		return -1;
	} else {//the subTree is not a leaf
		for (count_ = colonPoint_ - 1; count_ >= 0; count_ --) {
			subTree.insert (0, newickTree[count_]);

			if (newickTree[count_] == '(' )
				nOpenBracket_ ++;
			if (newickTree[count_] == ')' )

				nCloseBracket_ ++;

			if (nOpenBracket_ == nCloseBracket_)
				return count_;
		} //end of for count_

		return -1;
	}
}

//=======================================================
//return 1, if this newickTree is a leaf tree
int UserTree::isLeaf (Vec<char> &newickTree) {
	if (newickTree[0] != '(')
		return 1;
	else
		return 0;
}

//=======================================================
/*decode the newick tree at an internal node
return -1 in case of not a newick tree form
       inNdNo_, otherwise
*/
int UserTree::decodeInNd (Vec<char> &newickTree, int parNdNo, int parBrNo) {
	//  cout << newickTree << endl;
	int inNdNo_ = inNdArr_.getNewNd ();

	InNd inNd_;
	inNd_.setId (inNdNo_);
	inNdArr_ += inNd_;

	if (parBrNo != -1)
		inNd_.addNeiNd (parNdNo);
	if (parNdNo != -1)
		inNd_.addBr (parBrNo);

	int brNo_, chiNdNo_;
	double brLen_;
	Br<double> br_;

	int rightSubTreePoint_ = -1;
	Vec<char> subTree_;

	do {
		rightSubTreePoint_ = findRightSubTreePoint (newickTree, rightSubTreePoint_ + 2);
		if (rightSubTreePoint_ != -1) {
			brNo_ = brArr_.getNewBr ();
			br_.setId (brNo_);
			brArr_ += br_;


			getSubTree (newickTree, rightSubTreePoint_, subTree_, brLen_);

			if (isLeaf (subTree_) == 1)
				chiNdNo_ = decodeExNd (subTree_, inNdNo_, brNo_);
			else
				chiNdNo_ = decodeInNd (subTree_, inNdNo_, brNo_);
			inNd_.addNeiNd (chiNdNo_);
			inNd_.addBr (brNo_);
			if (chiNdNo_ < maxNExNd_)
				br_.set (brNo_, inNdNo_, chiNdNo_, brLen_, EX);
			else
				br_.set (brNo_, inNdNo_, chiNdNo_, brLen_, IN);

			brArr_.change (br_);
		} //end if rightSubTreePoint_
	} while (rightSubTreePoint_ != -1);

	inNdArr_.changeItem (inNd_);
	return inNdNo_;
}


//=======================================================
/*decode the newick tree at an external node
return -1 in case of not a newick tree form
       exNdNo_, otherwise
*/
int UserTree::decodeExNd (Vec<char> &newickTree, int parNdNo, int parBrNo) {
	int exNdNo_ = alignment.getSeqNo (newickTree);
	if (exNdNo_ == -1) {
		std::cout << newickTree << endl;
		Utl::announceError (NEWICK_TREE_ALIGNMENT_ERROR);
	}

	ExNd exNd_;
	exNd_.set (exNdNo_, parNdNo, parBrNo);
	exNdArr_+= exNd_;

	return exNdNo_;
}


//=======================================================
//convert the newick tree form into the unrooted tree structure
int UserTree::createUrTree () {
	//OutStream::write (inputNewickTree_, std::cout) << endl;

	int status_ = isNewickTree (inputNewickTree_);
	if ( status_ != NEWICK_TREE_FORM)
		return status_;

	if (maxNExNd_ == 0)
		doConstructor ();
	else
		clean ();

	maxNExNd_ = cmpNumExNd (inputNewickTree_);

	if (maxNExNd_ != alignment.getNSeq ()) {
		Utl::announceError (NEWICK_TREE_ALIGNMENT_ERROR);
	}

	if (isUrTreeForm (inputNewickTree_) == 1)
		decodeInNd (inputNewickTree_, -1, -1);

	else {
		int rightSubTreePoint1_ = findRightSubTreePoint (inputNewickTree_, 1);
		Vec<char> subTree1_;
		double brLen1_;
		getSubTree (inputNewickTree_, rightSubTreePoint1_, subTree1_, brLen1_);

		int rightSubTreePoint2_ = findRightSubTreePoint (inputNewickTree_, rightSubTreePoint1_ + 2);
		Vec<char> subTree2_;
		double brLen2_;
		getSubTree (inputNewickTree_, rightSubTreePoint2_, subTree2_, brLen2_);

		if (isLeaf (subTree1_) == 1) {
			int brNo_ = brArr_.getNewBr ();
			Br<double> br_;
			br_.setId (brNo_);
			brArr_ += br_;

			int inNdNo2_ = decodeInNd (subTree2_, -1, -1);
			int exNdNo1_ = decodeExNd (subTree1_, inNdNo2_, brNo_);


			inNdArr_[inNdNo2_].addNeiNd (exNdNo1_);
			inNdArr_[inNdNo2_].addBr (brNo_);

			br_.set (brNo_, inNdNo2_, exNdNo1_, (brLen1_ + brLen2_) );
			brArr_.change (br_);
			return 1;
		}

		int inNdNo1_ = decodeInNd (subTree1_, -1, -1);

		if (isLeaf (subTree2_) == 1) {
			int brNo_ = brArr_.getNewBr ();
			Br<double> br_;
			br_.setId (brNo_);
			brArr_ += br_;

			int exNdNo2_ = decodeExNd (subTree2_, inNdNo1_, brNo_);

			inNdArr_[inNdNo1_].addNeiNd (exNdNo2_);
			inNdArr_[inNdNo1_].addBr (brNo_);

			br_.set (brNo_, inNdNo1_, exNdNo2_, (brLen1_ + brLen2_) );
			brArr_.change (br_);

		} else {

			int inNdNo2_ = decodeInNd (subTree2_, -1, -1);
			int brNo_ = brArr_.getNewBr ();
			Br<double> br_;
			br_.setId (brNo_);
			brArr_ += br_;

			inNdArr_[inNdNo1_].addNeiNd (inNdNo2_);
			inNdArr_[inNdNo1_].addBr (brNo_);

			inNdArr_[inNdNo2_].addNeiNd (inNdNo1_);
			inNdArr_[inNdNo2_].addBr (brNo_);

			br_.set (brNo_, inNdNo1_, inNdNo2_, (brLen1_ + brLen2_) );
			brArr_.change (br_);
		}
	}
	//create the direction table for this tree
	return NEWICK_TREE_FORM;
}

void UserTree::checkUrTree() {
	bool out_error = false;
	for (int i = 0; i < maxNBr_; i++) 
		if (brArr_[i].getLen() < MIN_BR_LEN) {
			brArr_[i].setLen(MIN_BR_LEN);
			if (!out_error && isMasterProc()) {
				cout << "Negative branch length observered! fixed" << endl;
				out_error = true;
			}
		}
}

//=======================================================
//the destructor function
UserTree::~UserTree () {}

