/***************************************************************************
 *                                                                         *
 *   clusterarr.cpp (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh}@cs.uni-duesseldorf.de                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "constant.h"
#include "clusterarr.h"
#include "ali.h"


ClusterArr::ClusterArr () {
	nDifCluster_ = 0;
	totalNCluster_ = 0;
	nSeq_ = alignment.getNSeq ();
	items_.set (nSeq_, nSeq_);
}

void ClusterArr::increaseSize() {
	if (nDifCluster_ >= items_.getSize()) {
		items_.addLimit(INC);
		items_.setSize(items_.getLimit());
	}
}

void ClusterArr::add (Vec<int> &exNdNoLs,  double brLen) {
	increaseSize();
	items_[nDifCluster_].add (exNdNoLs,  brLen);
	nDifCluster_ ++;
	totalNCluster_ ++;
}

//=====================================================
void ClusterArr::operator += (Cluster &cluster) {
	increaseSize();
	items_[nDifCluster_] = cluster;
	nDifCluster_ ++;
	totalNCluster_ ++;
}


//=====================================================
Cluster &ClusterArr::operator [] (int cNo) {
	return items_[cNo];
}

//=====================================================
int ClusterArr::findCluster (Cluster &cluster) {
	for (int cNo_ = 0; cNo_ < nDifCluster_; cNo_ ++)
		if (items_[cNo_] == cluster)
			return cNo_;
	return -1;
}

//=====================================================
void ClusterArr::setupIndex () {
	indexArr_.set (nSeq_, nSeq_);
	for (int count_ = 2; count_ < nSeq_; count_ ++)
		//indexArr_[count_].set (MAX_NUM_CLUSTER / count_, 0);
		indexArr_[count_].set (nSeq_* 16 / count_, 0);

	items_.set (nSeq_*16, nSeq_*16);
	//items_.set (MAX_NUM_CLUSTER, MAX_NUM_CLUSTER);
}

//=====================================================
int ClusterArr::findClusterConTree (Cluster &cluster) {
	int nObj_ = cluster.nObj_;
	for (int count_ = 0; count_ < indexArr_[nObj_].getSize (); count_ ++) {
		int clusterNo_ = indexArr_[nObj_][count_];
		if (cluster == items_[clusterNo_])
			return clusterNo_;
	}

	return -1;
}

//add cluster into cluster array
void ClusterArr::addConTree (Cluster &cluster) {
	totalNCluster_ ++;
	int clusterNo_ = findClusterConTree (cluster);
	if (clusterNo_ == -1) {
		int newClusterNo_ = nDifCluster_;
		increaseSize();
		items_[newClusterNo_] = cluster;
		items_[newClusterNo_].nSup_ = 1;
		int nObj_ = cluster.nObj_;
		indexArr_[nObj_] += newClusterNo_;
		nDifCluster_ ++;
	} else
		items_[clusterNo_].nSup_ ++;
}

//=====================================================
int ClusterArr::findMajorCluster (int nDifTree) {
	majorClusterArr_.set (nSeq_, 0);
	for (int nObj_ = 2; nObj_ <= nSeq_ / 2; nObj_ ++) {
		for (int  count_ = 0; count_ < indexArr_[nObj_].getSize(); count_ ++) {
			int clusterNo_ = indexArr_[nObj_][count_];
			if (items_[clusterNo_].nSup_ > nDifTree / 2)
				majorClusterArr_ += clusterNo_;
		}

		if (nSeq_ - nObj_ > nObj_)
			for (int  count_ = 0; count_ < indexArr_[nSeq_ - nObj_].getSize(); count_ ++) {
				int clusterNo_ = indexArr_[nSeq_ - nObj_][count_];
				if (items_[clusterNo_].nSup_ > nDifTree / 2)
					majorClusterArr_ += clusterNo_;
			}
	} //end for nObj_
	return majorClusterArr_.getSize ();
}

//=====================================================
void ClusterArr::findCluTree (int clusterNo, Vec<int> &cluTreeNoArr) {
	int nObj_ = items_[clusterNo].nObj_;
	Vec<int> seqMarkArr_ (nSeq_, nSeq_);

	int seqNo_, count_, nCluSeq_;
	if (nObj_ < nSeq_ - nObj_) {
		nCluSeq_ = nObj_;
		for (seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
			seqMarkArr_[seqNo_] = 0;
		for (count_ = 0; count_ < nObj_; count_ ++) {
			seqNo_ = items_[clusterNo][count_];
			seqMarkArr_[seqNo_] = 1;
		}
	} else {
		nCluSeq_ = nSeq_ - nObj_;
		for (seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
			seqMarkArr_[seqNo_] = 1;
		for (count_ = 0; count_ < nObj_; count_ ++) {
			seqNo_ = items_[clusterNo][count_];
			seqMarkArr_[seqNo_] = 0;
		}
	}

	cluTreeNoArr.set (nSeq_, 0);
	for (int cluTreeNo_ = 0; cluTreeNo_ < totalNCluTree_; cluTreeNo_ ++)
		if (cluTreeArr_[cluTreeNo_].isAct_ == 1)
			if (cluTreeArr_[cluTreeNo_].isSubClu (seqMarkArr_, nCluSeq_) == 1)
				cluTreeNoArr += cluTreeNo_;
}

//=====================================================
void ClusterArr::createNewClusterTree (int nDifTree, int clusterNo) {
	Vec<int> cluTreeNoArr_;
	findCluTree (clusterNo, cluTreeNoArr_);

	Vec<char> newCluTree_ (MAX_NEWICK_TREE_LEN, 0);
	Vec<int> newCluSeqArr_ (nSeq_, 0);

	//char colon_ = ':';
	char openBracket_ = '(';
	char comma_ = ',';
	char closeBracket_ = ')';

	newCluTree_ += openBracket_;
	int nCluTree_ = cluTreeNoArr_.getSize ();
	for (int count_ = 0; count_ < nCluTree_; count_ ++) {
		int cluTreeNo_ = cluTreeNoArr_[count_];
		newCluSeqArr_ += cluTreeArr_[cluTreeNo_].seqArr_;
		newCluTree_ += cluTreeArr_[cluTreeNo_].tree_;

		if (count_ < nCluTree_ - 1)
			newCluTree_ += comma_;

		cluTreeArr_[cluTreeNo_].isAct_ = 0;
	}

	newCluTree_ += closeBracket_;
	//newCluTree_ += colon_;

	Vec<char> sNSup_;
	int nSup_ = items_[clusterNo].nSup_;
	nSup_ = nSup_ * 100 / nDifTree;
	sNSup_.convert (nSup_);
	newCluTree_ += sNSup_;

	int newCluTreeNo_ = totalNCluTree_;
	cluTreeArr_[newCluTreeNo_].set (newCluSeqArr_, newCluTree_);

	totalNCluTree_ ++;
}

//=====================================================
//return number of clusters which are at least 50% supported
int ClusterArr::createConTree (int nDifTree, Vec<char> &conTree) {
	totalNCluTree_ = nSeq_;
	cluTreeArr_.set (nSeq_ * 2, nSeq_ * 2);
	for (int seqNo_ = 0; seqNo_ < nSeq_; seqNo_ ++)
		cluTreeArr_[seqNo_].set (seqNo_);

	int nMajorCluster_ = findMajorCluster (nDifTree);
	for (int count_ = 0; count_ < nMajorCluster_; count_ ++) {
		int clusterNo_ = majorClusterArr_[count_];
		createNewClusterTree (nDifTree, clusterNo_);
	}


	//   char colon_ = ':';
	char openBracket_ = '(';
	char comma_ = ',';
	char closeBracket_ = ')';
	char semiColon_ = ';';

	conTree.set (MAX_NEWICK_TREE_LEN, 0);
	conTree += openBracket_;
	for (int cluTreeNo_ = 0; cluTreeNo_ < totalNCluTree_; cluTreeNo_ ++)
		if (cluTreeArr_[cluTreeNo_].isAct_ == 1) {
			conTree += cluTreeArr_[cluTreeNo_].tree_;
			conTree += comma_;
		}
	conTree[ conTree.getSize () - 1] = closeBracket_;
	conTree += semiColon_;
	return 1;
}

void ClusterArr::printSplits(int nDifTree, int outGrpSeqNo, const char *filename) {
	ofstream out(filename);
	if (!out) {
		cout << "Error: Cannot write to file " << filename << endl;
		return;
	}
	Vec<int> sorted_id(nDifCluster_, nDifCluster_);
	int i, j;
	for (i = 0; i < nDifCluster_; i++) sorted_id[i] = i;
	for (i = 0; i < nDifCluster_-1; i++) 
		for (j = i+1; j < nDifCluster_; j++) 
			if (items_[sorted_id[i]].nSup_ < items_[sorted_id[j]].nSup_) {
				int k = sorted_id[i];
				sorted_id[i] = sorted_id[j];
				sorted_id[j] = k;
			}
	for (i = 0; i < nDifCluster_; i++) {
		Vec<bool> seq_present(nSeq_, nSeq_);
		for (j = 0; j < nSeq_; j++) seq_present[j] = false;
		for (j = 0; j < items_[sorted_id[i]].seqNoArr_.getSize(); j++)
			seq_present[items_[sorted_id[i]].seqNoArr_[j]] = true;
		if (!seq_present[outGrpSeqNo]) {
			// flip the split 
			for (j = 0; j < nSeq_; j++) seq_present[j] = 1-seq_present[j];
		}
		for (j = 0; j < nSeq_; j++)
			if (seq_present[j]) out << "*"; else out << ".";
		out << "\t";
		out.precision(4);
		out << (double)items_[sorted_id[i]].nSup_ / nDifTree << endl;
	}
	out.close();
}


//=====================================================
ClusterArr::~ClusterArr () {}
