// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package config

import (
	"errors"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"

	"github.com/golang/mock/gomock"
)

func TestCombine(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConfigArgs{
		User:  &core.User{Login: "octocat"},
		Repo:  &core.Repository{Slug: "octocat/hello-world", Config: ".drone.yml"},
		Build: &core.Build{After: "6d144de7"},
	}

	resp := &core.Config{Data: string(mockFile)}

	service := mock.NewMockConfigService(controller)
	service.EXPECT().Find(noContext, args).Return(resp, nil)

	result, err := Combine(service).Find(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}

	if result.Data != string(resp.Data) {
		t.Errorf("unexpected file contents")
	}
}

func TestCombineErr(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	resp := errors.New("")
	service := mock.NewMockConfigService(controller)
	service.EXPECT().Find(noContext, nil).Return(nil, resp)

	_, err := Combine(service).Find(noContext, nil)
	if err != resp {
		t.Errorf("expected config service error")
	}
}

func TestCombineNoConfig(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConfigArgs{
		User:  &core.User{Login: "octocat"},
		Repo:  &core.Repository{Slug: "octocat/hello-world", Config: ".drone.yml"},
		Build: &core.Build{After: "6d144de7"},
	}

	resp := &core.Config{Data: string(mockFile)}

	service1 := mock.NewMockConfigService(controller)
	service1.EXPECT().Find(noContext, args).Return(nil, nil)

	service2 := mock.NewMockConfigService(controller)
	service2.EXPECT().Find(noContext, args).Return(resp, nil)

	result, err := Combine(service1, service2).Find(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}

	if result.Data != string(resp.Data) {
		t.Errorf("unexpected file contents")
	}
}

func TestCombineEmptyConfig(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	args := &core.ConfigArgs{
		User:  &core.User{Login: "octocat"},
		Repo:  &core.Repository{Slug: "octocat/hello-world", Config: ".drone.yml"},
		Build: &core.Build{After: "6d144de7"},
	}

	resp1 := &core.Config{}
	resp2 := &core.Config{Data: string(mockFile)}

	service1 := mock.NewMockConfigService(controller)
	service1.EXPECT().Find(noContext, args).Return(resp1, nil)

	service2 := mock.NewMockConfigService(controller)
	service2.EXPECT().Find(noContext, args).Return(resp2, nil)

	result, err := Combine(service1, service2).Find(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}

	if result.Data != string(resp2.Data) {
		t.Errorf("unexpected file contents")
	}
}

func TestCombineNoConfigErr(t *testing.T) {
	// args := &core.ConfigArgs{
	// 	User:  &core.User{Login: "octocat"},
	// 	Repo:  &core.Repository{Slug: "octocat/hello-world", Config: ".drone.yml"},
	// 	Build: &core.Build{After: "6d144de7"},
	// }

	service := Combine()
	_, err := service.Find(noContext, nil)
	if err != errNotFound {
		t.Errorf("Expect not found error")
	}
}
