require 'fileutils'

require 'common/filesystem'
require 'common/dovecot_plugin'
require 'mv/mv_plugin'


# Handle moving (renaming) Dovecot users on the filesystem.
#
class DovecotMv

  include DovecotPlugin
  include MvPlugin


  # Move the Dovecot user *src* to *dst*. This relocates the user's
  # directory within the Dovecot mailstore (on the filesystem).
  #
  # This fails if the source user does not exist, or if the
  # destination user already exists before the move.
  #
  # But is it an error if the target domain does not exist? That's a
  # bit subtle... The domain may exist in the database, but if it
  # has not received any mail yet, then its directory won't exist
  # on-disk.
  #
  # There are two possible "oops" scenarios resulting from the fact
  # that we may run either the Postfixadmin move first or the
  # Dovecot move first. If we move the user in the database, we
  # definitely want to move him on disk (that is, we should create
  # the directory here). But if we move him on disk first, then we
  # don't know if the database move will fail! We don't want to move
  # his mail files if he won't get moved in the database.
  #
  # Faced with two equally-bad (but easy-to-fix) options, we do the
  # simplest thing and fail if the destination domain directory
  # doesn't exist. If nothing else, this is at least consistent.
  #
  # @param src [User] the source user to be moved.
  #
  # @param dst [User] the destination user being moved to.
  #
  def mv_user(src, dst)
    raise NonexistentUserError.new(src.to_s()) if not user_exists(src)
    raise UserAlreadyExistsError.new(dst.to_s()) if user_exists(dst)

    # See the docstring...
    if not self.domain_exists(dst.domain()) then
      raise NonexistentDomainError.new(dst.domainpart())
    end

    # We may need to create the target domain directory, even if the
    # domain exists in the database.
    FileUtils.mkdir_p(self.get_domain_path(dst.domain()))

    # The parent of dst_path exists because we just created it.The
    # source path should exist too, because the "source user" does,
    # and, well, how did we determine that?
    src_path = self.get_user_path(src)
    dst_path = self.get_user_path(dst)
    FileUtils.mv(src_path, dst_path)
  end

end
