# Convenience methods for working with the filesystem. This class
# only provides static methods, to be used analogously to the File
# class (for example, <tt>File.directory?</tt>).
#
class Filesystem

  # Return whether or not the given path begins with a dot (ASCII
  # period).
  #
  # @param path [String] the path whose first character you want to check.
  #
  # @return [Boolean] whether or not *path* begins with an ASCII period.
  #
  def self.begins_with_dot?(path)
    return (path[0..0] == '.')
  end


  # Get a list of all real subdirectories of the given directory.
  #
  # @param dir [String] the directory whose subdirectories you want.
  #
  # @return [Array<String>] a list of subdirectories of *dir*, not
  #   including the pseudo-directories "." and ".." (the current/parent
  #   directories).
  #
  def self.get_subdirs(dir)
    subdirs = []
    return subdirs if not File.directory?(dir)

    Dir.open(dir) do |d|
      d.each do |entry|
        relative_path = File.join(dir, entry)
        if File.directory?(relative_path) and not self.begins_with_dot?(entry)
          subdirs << entry
        end
      end
    end

    return subdirs
  end

end
