/*
  Copyright (c) 2001-2009 by Salvador E. Tropea (SET) <set@ieee.org>
  Copyright (c) 2003-2004 by Grzegorz Adam Hankiewicz
  
  This code is covered by the GPL license. A copy of the license should
  be provided in the same package.

  Modified by Grzegorz Adam Hankiewicz <gradha@hankiewicz.datanet.co.uk>
  to be used as a Turbo Vision driver.

  conio.h emulation library on top of Allegro
*/

#include <tv/configtv.h>

#define Uses_stdio
#define Uses_string
#define Uses_TScreen
#define Uses_TEvent
#define Uses_TGKey
#include <tv.h>
#include <tv/gkey.h>
#include <tv/alcon/alcon.h>

#ifdef HAVE_ALLEGRO

#include <allegro/internal/aintern.h>

// Forward declarations.
static void AlCon_LoadCustomFont(const char *filename, uchar *fdata, unsigned fw, unsigned fh);
static void AlCon_AllocateMemoryBuffer(void);

// Globals internal to AlCon.
static FONT *Alcon_CurrentFont=NULL;

static FONT Alcon_PrimaryFont;
static FONT_GLYPH *Alcon_ASCIIDataP[256];
static FONT_MONO_DATA Alcon_ASCIIMonoFontP =
{
 0x0, 0x100,                 /* begin, end characters */
 Alcon_ASCIIDataP,           /* the data set */
 0                           /* next */
};

static FONT Alcon_SecondaryFont;
static FONT_GLYPH *Alcon_ASCIIDataS[256];
static FONT_MONO_DATA Alcon_ASCIIMonoFontS =
{
 0x0, 0x100,                 /* begin, end characters */
 Alcon_ASCIIDataS,           /* the data set */
 0                           /* next */
};

static FONT Alcon_CursorFont;
static FONT_GLYPH *Alcon_CursorData[2];
static FONT_MONO_DATA Alcon_CursorMonoFont =
{
 0x0, 0x2,                   /* begin, end characters */
 Alcon_CursorData,           /* the data set */
 0                           /* next */
};

static
AlCon_Color BIOSPalette[16]={
{ 0x00, 0x00, 0x00, 0 },
{ 0x00, 0x00, 0xA8, 0 },
{ 0x00, 0xA8, 0x00, 0 },
{ 0x00, 0xA8, 0xA8, 0 },
{ 0xA8, 0x00, 0x00, 0 },
{ 0xA8, 0x00, 0xA8, 0 },
{ 0xA8, 0x54, 0x00, 0 },
{ 0xA8, 0xA8, 0xA8, 0 },
{ 0x54, 0x54, 0x54, 0 },
{ 0x54, 0x54, 0xFC, 0 },
{ 0x54, 0xFC, 0x54, 0 },
{ 0x54, 0xFC, 0xFC, 0 },
{ 0xFC, 0x54, 0x54, 0 },
{ 0xFC, 0x54, 0xFC, 0 },
{ 0xFC, 0xFC, 0x54, 0 },
{ 0xFC, 0xFC, 0xFC, 0 }};
AlCon_Color AlCon_CurPalette[16];

static int cursorX, cursorY;
static int cursorPX, cursorPY;

/*
   The variables AlCon_ScreenWidth, AlCon_ScreenHeight,
   AlCon_FontWidth and AlCon_FontHeight are read only. They are
   modified ONLY when the font is changed. The first two tell the
   number of text characters that fit into the physical screen. The
   last two tell the number of screen pixels a text cell has in
   Allegro. The font size is initialised by default to a 8x16 font so
   the AlCon_Init function can set a reasonable default screen size.
*/
static int AlCon_ScreenWidth, AlCon_ScreenHeight;
static int AlCon_FontWidth=8, AlCon_FontHeight=16;

static int colors[16];
static int fg,bg;
static char cursorEnabled=1, cursorInScreen=0;
static char AlCon_UseSecFont=0;
static unsigned char *chars, *attrs;
static unsigned char curAttr;
static char AlCon_CursorShapeFrom, AlCon_CursorShapeTo;
//static int al_text_mode;
static int al_mouse_buttons = 0;
static int al_mouse_wheel;

/* Extra flags reserved for silly cursor effects */
int AlCon_CursorBits = 0;
char AlCon_ForcedKeyboard[4] = "";

void AlCon_EnableSecFont()
{
 AlCon_UseSecFont=1;
}

void AlCon_DisableSecFont()
{
 AlCon_UseSecFont=0;
}

/*****************************************************************************

  Functions to hide/unhide the mouse and text cursors.
  I use the scare_mouse() stuff and also implemented the same concept for the
text cursor.
    
*****************************************************************************/


void AlCon_UnDrawCursor(int *aFgCol);

#define maxCurMode 32
char CursorMode[maxCurMode];
int CursorModeIndex=0;

static
void PushCursorMode()
{
 if (CursorModeIndex<maxCurMode)
    CursorMode[CursorModeIndex++]=cursorEnabled;
}

static
void PullCursorMode()
{
 if (CursorModeIndex)
    cursorEnabled=CursorMode[--CursorModeIndex];
}

void AlCon_ScareCursor()
{
 PushCursorMode();
 if (!cursorEnabled)
    return;
 cursorEnabled=0;
 if (cursorInScreen)
   {
    int color;
    scare_mouse(); /* Could be necesary. i.e. gotoxy */
    AlCon_UnDrawCursor(&color);
    unscare_mouse();
    cursorInScreen=0;
   }
}

void AlCon_UnScareCursor()
{
 PullCursorMode();
}

void AlCon_DisableAsync()
{
 scare_mouse();
 AlCon_ScareCursor();
}

void AlCon_EnableAsync()
{
 AlCon_UnScareCursor();
 unscare_mouse();
}

/*****************************************************************************

  Cursor positioning routines.
  They take care of the text cursor.
    
*****************************************************************************/


void AlCon_CursorHome()
{
 AlCon_GotoXY(1, 1);
}

void AlCon_GotoXY(int x, int y)
{
 ASSERT(x > 0 && x <= AlCon_ScreenWidth);
 ASSERT(y > 0 && y <= AlCon_ScreenHeight);
 AlCon_ScareCursor();
 cursorX = x; cursorY = y;
 cursorPX = (x - 1) * AlCon_FontWidth;
 cursorPY = (y - 1) * AlCon_FontHeight;
 AlCon_UnScareCursor();
}

int AlCon_WhereX()
{
 return cursorX;
}

int AlCon_WhereY()
{
 return cursorY;
}

int AlCon_ScreenCols()
{
 return AlCon_ScreenWidth;
}

int AlCon_ScreenRows()
{
 return AlCon_ScreenHeight;
}


/*****************************************************************************

  Color selection routines
    
*****************************************************************************/

void AlCon_SetColors(int aFg, int aBg)
{
 fg=aFg; bg=aBg;
 curAttr=(bg<<4) | (fg & 0xF);
}

void AlCon_TextAttr(int attr)
{
 unsigned char aBg=attr>>4;
 unsigned char aFg=attr & 0xF;
 curAttr=attr;
 fg=aFg; bg=aBg;
}

void AlCon_TextBackground(int aBg)
{
 AlCon_SetColors(fg,aBg);
}

void AlCon_TextColor(int aFg)
{
 AlCon_SetColors(aFg,bg);
}

/*****************************************************************************

  Clear routines
    
*****************************************************************************/

void AlCon_Clear()
{
 AlCon_DisableAsync();
 clear_to_color(screen,bg);
 AlCon_EnableAsync();
 memset(chars,' ', AlCon_ScreenWidth * AlCon_ScreenHeight);
 memset(attrs,curAttr, AlCon_ScreenWidth * AlCon_ScreenHeight);
}

void AlCon_ClrScr()
{
 AlCon_Clear();
 AlCon_CursorHome();
}

/*****************************************************************************

 Text output routines, including newline and tab management
    
*****************************************************************************/

void AlCon_NewLine()
{
 if (cursorY == AlCon_ScreenHeight)
   { // Scroll
    int ybot,xmax;
    unsigned offset;
    AlCon_DisableAsync();
    ybot = (AlCon_ScreenHeight - 1) * AlCon_FontHeight;
    xmax = AlCon_ScreenWidth * AlCon_FontWidth;
    blit(screen, screen, 0, AlCon_FontHeight, 0, 0, xmax, ybot);
    rectfill(screen,0,ybot,xmax, AlCon_ScreenHeight * AlCon_FontHeight, colors[bg]);
    AlCon_EnableAsync();

    // Same for buffers
    memcpy(chars, chars + AlCon_ScreenWidth, AlCon_ScreenWidth * (AlCon_ScreenHeight - 1));
    memcpy(attrs, attrs + AlCon_ScreenWidth, AlCon_ScreenWidth * (AlCon_ScreenHeight - 1));
    offset = (AlCon_ScreenHeight - 1) * AlCon_ScreenWidth;
    memset(chars+offset,' ', AlCon_ScreenWidth);
    memset(attrs+offset,curAttr, AlCon_ScreenWidth);

    AlCon_GotoXY(1,cursorY);
   }
 else
    AlCon_GotoXY(1,cursorY+1);
}

inline
void AlCon_TextOut(const char *str, unsigned x, unsigned y, int fg, int bg)
{
 if (AlCon_UseSecFont && (fg & 8))
    textout_ex(screen,&Alcon_SecondaryFont,str,x,y,colors[fg],colors[bg]);
 else
    textout_ex(screen,Alcon_CurrentFont,str,x,y,colors[fg],colors[bg]);
}

void AlCon_PutStr(const char *s)
{
 ASSERT(s);
 ASSERT(Alcon_CurrentFont);
 
 int l=strlen(s);
 AlCon_DisableAsync();
 while (cursorX + l > AlCon_ScreenWidth)
   {
    char b[AlCon_ScreenWidth + 1];
    int count = AlCon_ScreenWidth - cursorX + 1;
    unsigned offset;
    strncpy(b,s,count);
    b[count]=0;
    AlCon_TextOut(b,cursorPX,cursorPY,fg,bg);

    offset = (cursorX - 1) + (cursorY - 1) * AlCon_ScreenWidth;
    memcpy(chars+offset,b,count);
    memset(attrs+offset,curAttr,count);

    AlCon_NewLine();
    s+=count;
    l-=count;
   }
 if (*s)
   {
    unsigned offset;
    AlCon_TextOut(s,cursorPX,cursorPY,fg,bg);

    offset = (cursorX - 1) + (cursorY - 1) * AlCon_ScreenWidth;
    memcpy(chars+offset,s,l);
    memset(attrs+offset,curAttr,l);

    cursorX+=l;
    cursorPX += l * AlCon_FontWidth;
   }
 AlCon_EnableAsync();
}

void AlCon_Puts(const char *s)
{
 AlCon_PutStr(s);
 AlCon_NewLine();
}

void AlCon_Tab()
{
 int count=8-(cursorX & 7);
 char b[9];
 memset(b,' ',count);
 b[count]=0;
 AlCon_PutStr(b);
}

void AlCon_CPrintf(AL_CONST char *format, ...)
{
 char buf[4096];
 int ret;
 char *s,*next,aux;

 va_list ap;
 va_start(ap, format);
 // TODO: vsnprintf
 ret=vsprintf(buf, format, ap);
 if (ret>4090)
   {
    set_gfx_mode(GFX_TEXT,0,0,0,0);
    allegro_message("Buffer overflow in AlCon_CPrintf\n");
    exit(100);
   }
 va_end(ap);

 s=buf;
 while ((next=strpbrk(s,"\n\t"))!=0 && *next)
   {
    aux=*next; *next=0;
    AlCon_PutStr(s);
    *next=aux;
    if (aux=='\n')
       AlCon_NewLine();
    else if (aux=='\t')
       AlCon_Tab();
    s=next+1;
   }
 if (*s)
    AlCon_PutStr(s);
}

#if 0
// Avoiding to redraw characters that didn't change: 2.5 times faster
void AlCon_PutBuf(unsigned offset, uint16 *values, int count)
{
 unsigned x,y;
 x=(offset%AlCon_ScreenWidth)*AlCon_FontWidth;
 y=(offset/AlCon_ScreenWidth)*AlCon_FontHeight;

 uchar *b=(uchar *)values,newChar,newAttr;
 uchar *sc=(uchar *)(chars+offset);
 AlCon_DisableAsync();
 unsigned char aux[2];
 aux[1]=0;
 while (count--)
   {
    newChar=b[charPos];
    newAttr=b[attrPos];
    if (newChar!=chars[offset] || newAttr!=attrs[offset])
      {
       aux[0]=chars[offset]=newChar;
       attrs[offset]=newAttr;
       int nBg,nFg;
       nBg=b[attrPos]>>4;
       nFg=b[attrPos] & 0xF;
       AlCon_TextOut((char *)aux,x,y,nFg,nBg);
      }
    x+=AlCon_FontWidth; b+=2; offset++;
   }
 AlCon_EnableAsync();
}
#else
// Avoiding to redraw characters that didn't change: 2.5 times faster
// Grouping chars of the same colour: 9 to 15% faster
void AlCon_PutBuf(unsigned offset, uint16 *values, int count)
{
 unsigned x,y,xf;
 x=(offset%AlCon_ScreenWidth)*AlCon_FontWidth;
 y=(offset/AlCon_ScreenWidth)*AlCon_FontHeight;

 uchar *b=(uchar *)values,newChar,newAttr,oldAttr;
 AlCon_DisableAsync();
 uchar aux[count+1];
 newChar=b[charPos];
 newAttr=b[attrPos];
 while (count)
   {
    if (newChar!=chars[offset] || newAttr!=attrs[offset])
      {
       uchar *ind=aux;
       oldAttr=newAttr;
       xf=x;
       do
         {
          *(ind++)=chars[offset]=newChar;
          attrs[offset]=newAttr;
          xf+=AlCon_FontWidth;
          count--;
          if (!count)
             break;
          offset++;
          b+=2;
          newChar=b[charPos];
          newAttr=b[attrPos];
         }
       while (newAttr==oldAttr && newChar!=chars[offset] && newAttr!=attrs[offset]);
       *ind=0;
       int nBg,nFg;
       nBg=oldAttr>>4;
       nFg=oldAttr & 0xF;
       AlCon_TextOut((char *)aux,x,y,nFg,nBg);
       x=xf;
      }
    else
      {
       x+=AlCon_FontWidth;
       count--;
       offset++;
       b+=2;
       newChar=b[charPos];
       newAttr=b[attrPos];
      }
   }
 AlCon_EnableAsync();
}
#endif

// TODO: Optimize!!!
void AlCon_Redraw()
{
 ASSERT(Alcon_CurrentFont);

 int nBg,nFg;
 unsigned offset=0;
 unsigned mx=AlCon_ScreenWidth*AlCon_FontWidth;
 unsigned my=AlCon_ScreenHeight*AlCon_FontHeight;
 unsigned char aux[2];
 aux[1]=0;

 AlCon_DisableAsync();
 for (unsigned y=0; y<my; y+=AlCon_FontHeight)
    {
     for (unsigned x=0; x<mx; offset++, x+=AlCon_FontWidth)
        {
         aux[0]=chars[offset];
         nBg=attrs[offset]>>4;
         nFg=attrs[offset] & 0xF;
         AlCon_TextOut((char *)aux,x,y,nFg,nBg);
        }
    }
 AlCon_EnableAsync();
}

void AlCon_GetScrChars(unsigned offset, uint16 *buffer, unsigned count)
{
 unsigned char *b=(unsigned char *)buffer;

 while (count--)
   {
    b[charPos]=chars[offset];
    b[attrPos]=attrs[offset];
    b+=2;
    offset++;
   }
}

void AlCon_PutChar(unsigned offset, uint16 value)
{
 ASSERT(Alcon_CurrentFont);
 int nBg,nFg;
 unsigned char *b=(unsigned char *)&value;
 unsigned char aux[2];
 unsigned x,y;
 aux[1]=0;

 x = (offset % AlCon_ScreenWidth) * AlCon_FontWidth;
 y = (offset / AlCon_ScreenWidth) * AlCon_FontHeight;
 AlCon_DisableAsync();

 aux[0]=chars[offset]=b[charPos];
 attrs[offset]=b[attrPos];
 nBg=b[attrPos]>>4;
 nFg=b[attrPos] & 0xF;
 AlCon_TextOut((char *)aux,x,y,nFg,nBg);

 AlCon_EnableAsync();
}

/*****************************************************************************

  Text cursor routines.
    
*****************************************************************************/

void AlCon_UnDrawCursor(int *aFgCol)
{
 ASSERT(Alcon_CurrentFont);
 ASSERT(aFgCol);
 unsigned offset = (cursorX - 1) + (cursorY - 1) * AlCon_ScreenWidth;
 int bg=attrs[offset]>>4;
 int fg=attrs[offset] & 0xF;
 char b[2];
 b[0]=chars[offset]; b[1]=0;
 AlCon_TextOut(b,cursorPX,cursorPY,fg,bg);
 *aFgCol=colors[fg];
}

void AlCon_IntCursor()
{
 if (cursorEnabled)
   {
    int aFg;
    scare_mouse();
    cursorInScreen=!cursorInScreen;
    AlCon_UnDrawCursor(&aFg);
    if (cursorInScreen)
      {
       if (!(AlCon_CursorBits & ALCON_CURSOR_HIDE_NORMAL))
         textout_ex(screen, &Alcon_CursorFont, "\x1", cursorPX,cursorPY, aFg, -1);

       if (AlCon_CursorBits & ALCON_CURSOR_ON_SQUARE)
         rect(screen, cursorPX, cursorPY, cursorPX + AlCon_FontWidth - 1,
              cursorPY + AlCon_FontHeight - 1, aFg);

       if (AlCon_CursorBits & ALCON_CURSOR_ON_CROSS)
         {
          line(screen, cursorPX, cursorPY, cursorPX + AlCon_FontWidth - 1,
               cursorPY + AlCon_FontHeight - 1, aFg);
          line(screen, cursorPX, cursorPY + AlCon_FontHeight - 1,
               cursorPX + AlCon_FontWidth - 1, cursorPY, aFg);
         }
      }
    unscare_mouse();
   }
}

/**[txh]********************************************************************

  Description: Sets the cursor shape. Pass the start and end lines
  (inclusive) of the cursor. The lines must be in the range 0 to less than
  font's height. Since the start and end are inclusive, you cannot set an
  "invisible" cursor shape. Use AlCon_HideCursor() for that.
  
***************************************************************************/

void AlCon_SetCursorShape(int from, int to)
{
   ASSERT(from <= to);
   ASSERT(from >= 0);
   ASSERT(to < AlCon_FontHeight);
   
   unsigned wb=(AlCon_FontWidth+7)/8;
   memset(Alcon_CursorData[1]->dat, 0, wb*AlCon_FontHeight);
   memset(Alcon_CursorData[1]->dat + wb*from, 0xFF, wb*(to - from + 1));
   AlCon_CursorShapeFrom = from;
   AlCon_CursorShapeTo = to;
}

void AlCon_GetCursorShape(int *from, int *to)
{
   ASSERT(from || to);
   if (from)
      *from = AlCon_CursorShapeFrom;
   if (to)
      *to = AlCon_CursorShapeTo;
}

void Alcon_CreateNewCursor()
{
   Alcon_CursorData[0] =0;
   unsigned wb=(AlCon_FontWidth+7)/8;
   if (Alcon_CursorData[1])
      free(Alcon_CursorData[1]);
   Alcon_CursorData[1] =(FONT_GLYPH *)malloc(sizeof(FONT_GLYPH)+wb*AlCon_FontHeight);
   Alcon_CursorData[1]->w = AlCon_FontWidth;
   Alcon_CursorData[1]->h = AlCon_FontHeight;
   AlCon_SetCursorShape(87 * AlCon_FontHeight / 100, AlCon_FontHeight - 1);
}

void AlCon_HideCursor()
{
   cursorEnabled=0;
}

void AlCon_ShowCursor()
{
   cursorEnabled=1;
}

int AlCon_IsVisCursor()
{
   return cursorEnabled;
}

/*****************************************************************************

  Initialization routines
    
*****************************************************************************/

/**[txh]********************************************************************

  Description: Initialization of the AlCon layer. Pass the number of text
  characters you want to have with a default 8x16 font.
  
  Return: Zero if everything went fine, non zero otherwise.
  
***************************************************************************/

int AlCon_Init(int w, int h, int fw, int fh, uchar *fdata, AlCon_Color *pal)
{
   if (fw!=-1 && fh!=-1) {
      AlCon_FontWidth=fw;
      AlCon_FontHeight=fh;
   }

   set_uformat(U_ASCII);
   //set_uformat(U_UTF8);
  
   allegro_init();

   /* Maybe force a specific keyboard layout? */
   if (*AlCon_ForcedKeyboard) {
      AlCon_ForcedKeyboard[3] = 0;
      set_config_string("system", "keyboard", AlCon_ForcedKeyboard);
   }
   
   install_keyboard();
   al_mouse_buttons = install_mouse();
   al_mouse_wheel = mouse_z;
   // Clamp allegro return of "no mouse" to TVision interpretation.
   if (al_mouse_buttons < 0)
      al_mouse_buttons = 0;
   // And add "fake wheel buttons" if the mouse has at least two.
   if (al_mouse_buttons > 1)
      al_mouse_buttons = 5;
   install_timer();
  
   /* If we are in a graphic mode and we know the depth use it */
   if (desktop_color_depth()) {
      set_color_depth(desktop_color_depth());
   } else {
      #ifdef BPP
      set_color_depth(BPP);
      #else
      set_color_depth(8);
      #endif
   }
   
   if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, AlCon_FontWidth * w, AlCon_FontHeight * h,
         AlCon_FontWidth * w, AlCon_FontHeight * h)) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Can't initialize graphics mode\n%s\n",allegro_error);
      return 2;
   }
   // Set the size of the font and screen.
   AlCon_ScreenWidth =w;
   AlCon_ScreenHeight=h;

   /* Load a binary font. Hack to get the font/screen properties set. */
   ASSERT(fdata);
   AlCon_LoadCustomFont(0,fdata,AlCon_FontWidth,AlCon_FontHeight);
   //AlCon_LoadCustomFont("rom-PC437.016");
  
   /* Create default cursor shape */
   Alcon_CreateNewCursor();
  
   /* Create the text mode palette */
   AlCon_SetDisPaletteColors(0,16,pal ? pal : BIOSPalette);
  
   /* Allocate "screen" buffers */
   AlCon_AllocateMemoryBuffer();
  
   install_int(AlCon_IntCursor, 500);
  
   AlCon_SetColors(7, 0);
   AlCon_ClrScr();
   show_mouse(screen);
   return 0;
}

/**[txh]********************************************************************

  Description: Called when you want to change the font size. You
  can call this first with the font size you would like, and see
  if AlCon is able to resize the screen to that size.
  
  Return: Zero on success. A positive number if the resolution was
  not set, but AlCon was able to restore the previous screen size
  using the old parameters. A negative number if the resolution was
  not set, and trying to set it AlCon lost the previous screen,
  leaving the whole graphic mode in a mess you can only abort
  execution.
  
***************************************************************************/

int AlCon_Resize(unsigned sW, unsigned sH, int new_font_width,
                 int new_font_height)
{
   ASSERT(screen && "Did you really initialise me properly?");
   ASSERT(new_font_width > 0);
   ASSERT(new_font_height > 0);
   ASSERT(AlCon_ScreenWidth > 0);
   ASSERT(AlCon_ScreenHeight > 0);

   // Save here the previous parameters...
   const int old_w = SCREEN_W;
   const int old_h = SCREEN_H;
   const int old_card = gfx_driver->id;

   // Hide the mouse during the change. If we don't do it the cursor gets damaged.
   show_mouse(NULL);
   int temp;
   AlCon_UnDrawCursor(&temp);
   
   // Be brave and try the new screen resolution.
   // TODO: Get rid of explicit driver. Maybe use configuration file?
   int ret = set_gfx_mode(GFX_AUTODETECT_WINDOWED,
      sW * new_font_width, sH * new_font_height,
      sW * new_font_width, sH * new_font_height);

   if (ret == 0)
     {
      AlCon_ScreenWidth=sW;
      AlCon_ScreenHeight=sH;
      // Reallocate internal memory for text video.
      AlCon_AllocateMemoryBuffer();
      // Fix up cursor position, it may have ended up outside screen.
      if (cursorX > AlCon_ScreenWidth || cursorY > AlCon_ScreenHeight)
         AlCon_GotoXY(1, 1);
      
      // Clean up the screen, just in case...
      clear_to_color(screen, 0);
      
      show_mouse(screen);
      return 0;
     }

   // Oops, try to fix our mistake.
   ret = set_gfx_mode(old_card, old_w, old_h, old_w, old_h);
   if (ret != 0)
      return -1;  // Bail out completely.
   show_mouse(screen);
   return 1;   // At least we kept our head safe...
}

/**[txh]********************************************************************

  Description: Removes allegro from the system, freeing any internal
  memory and resetting the graphic mode and other subsystems.
  
***************************************************************************/

void AlCon_Exit()
{
 ASSERT(chars);
 ASSERT(attrs);
 
 remove_int(AlCon_IntCursor);
 set_gfx_mode(GFX_TEXT,0,0,0,0);
 allegro_exit();
 free(chars);
 free(attrs);
 chars = attrs = 0;
}

int AlCon_SetDisPaletteColors(int from, int number, AlCon_Color *c)
{
 int i;

 for (i=0; i<number; i++)
    {
     colors[from+i]=makecol(c[i].R,c[i].G,c[i].B);
     AlCon_CurPalette[from+i]=c[i];
    }

 return i;
}

/*****************************************************************************

  Keyboard input routines
    
*****************************************************************************/

int AlCon_GetCh()
{
 return readkey();
}

int AlCon_KbHit()
{
 return keypressed();
}

const char *AlCon_KeyNames[]=
{
"Unknown",
"A","B","C","D","E","F","G","H","I","J","K",
"L","M","N","O","P","Q","R","S","T","U","V",
"W","X","Y","Z",
"OpenBrace","BackSlash","CloseBrace","Pause","Esc",
"0","1","2","3","4","5","6","7","8","9",
"BackSpace","Tab","Enter","Colon","Quote","Grave",
"Comma","Stop","Slash","Asterisk","Space","Minus",
"Plus","PrnScr","Equal","F1","F2","F3","F4","F5",
"F6","F7","F8","F9","F10","F11","F12","Home",
"Up","PgUp","Left","Right","End","Down","PgDn",
"Insert","Delete","Caret","Admid","DobleQuote",
"Numeral","Dolar","Percent","Amper","OpenPar",
"ClosePar","DoubleDot","LessThan","GreaterThan",
"Question","A_Roba","Or","UnderLine","OpenCurly",
"CloseCurly","Tilde","Macro","WinLeft","WinRight","WinSel",
"Mouse"
};

#define NumKeyNames (sizeof(AlCon_KeyNames)/sizeof(char *))

static
uchar keyCode[KEY_MAX]=
{
 kbUnkNown,kbA,kbB,kbC,kbD,kbE,kbF,kbG, // 0-7
 kbH,kbI,kbJ,kbK,kbL,kbM,kbN,kbO, // 8-15
 kbP,kbQ,kbR,kbS,kbT,kbU,kbV,kbW, // 16-23
 kbX,kbY,kbZ,kb0,kb1,kb2,kb3,kb4, // 24-31
 kb5,kb6,kb7,kb8,kb9,kb0,kb1,kb2, // 32-39
 kb3,kb4,kb5,kb6,kb7,kb8,kb9,kbF1, // 40-47
 kbF2,kbF3,kbF4,kbF5,kbF6,kbF7,kbF8,kbF9, // 48-55
 kbF10,kbF11,kbF12,kbEsc,kbGrave,kbMinus,kbEqual,kbBackSpace, // 56-63
 kbTab,kbOpenBrace,kbCloseBrace,kbEnter,kbColon,kbQuote,kbBackSlash,kbBackSlash, // 64-71
 kbComma,kbStop,kbSlash,kbSpace,kbInsert,kbDelete,kbHome,kbEnd, // 72-79
 kbPgUp,kbPgDn,kbLeft,kbRight,kbUp,kbDown,kbSlash,kbAsterisk,
 kbMinus,kbPlus,kbDelete,kbEnter,kbPrnScr,kbPause,kbUnkNown,kbUnkNown,
 kbUnkNown,kbUnkNown,kbUnkNown,kbA_Roba,kbCaret,kbColon,kbUnkNown,kbUnkNown,
 kbUnkNown,kbUnkNown,kbUnkNown,kbUnkNown,kbUnkNown,kbUnkNown,kbWinLeft,
 kbWinRight,kbWinSel,kbUnkNown,kbUnkNown//,kbUnkNown
};

static
uchar KeyCodeByASCII[96]=
{
 kbSpace,kbAdmid,kbDobleQuote,kbNumeral,kbDolar,kbPercent,kbAmper,kbQuote,
 kbOpenPar,kbClosePar,kbAsterisk,kbPlus,kbComma,kbMinus,kbStop,kbSlash,
 kb0,kb1,kb2,kb3,kb4,kb5,kb6,kb7,
 kb8,kb9,kbDoubleDot,kbColon,kbLessThan,kbEqual,kbGreaterThan,kbQuestion,
 kbA_Roba,kbA,kbB,kbC,kbD,kbE,kbF,kbG,
 kbH,kbI,kbJ,kbK,kbL,kbM,kbN,kbO,
 kbP,kbQ,kbR,kbS,kbT,kbU,kbV,kbW,
 kbX,kbY,kbZ,kbOpenBrace,kbBackSlash,kbCloseBrace,kbCaret,kbUnderLine,
 kbGrave,kbA,kbB,kbC,kbD,kbE,kbF,kbG,
 kbH,kbI,kbJ,kbK,kbL,kbM,kbN,kbO,
 kbP,kbQ,kbR,kbS,kbT,kbU,kbV,kbW,
 kbX,kbY,kbZ,kbOpenCurly,kbOr,kbCloseCurly,kbTilde,kbDelete
};

int AlCon_GetKey(unsigned *aSymbol, uchar *aScan, unsigned *aFlags)
{
 int key=readkey();
 int scan=key>>8;
 int ascii=key & 0xFF;
 int name,flags;

 if (ascii>=32 && ascii<128)
    name=KeyCodeByASCII[ascii-32];
 else
    name=keyCode[scan];

 flags=0;
 if (key_shifts & KB_SHIFT_FLAG)
    flags|=kbShiftCode;
 if (key_shifts & KB_CTRL_FLAG)
    flags|=kbCtrlCode;
 if (key_shifts & KB_ALT_FLAG)
    flags|=kbAltLCode;

 *aSymbol=ascii;
 *aFlags=flags;
 *aScan=scan;

 //printf("key: 0x%04X scan: 0x%02X ascii: 0x%02X Abstract: 0x%02X\n",key,scan,ascii,name|flags);
 return name|flags;
}

void AlCon_ClearKeyBuf()
{
 clear_keybuf();
}

/*****************************************************************************

  Mouse input routines
    
*****************************************************************************/

// Returns 0 if there is no mouse.
int AlCon_GetMouseButtons(void)
{
 return al_mouse_buttons;
}

void AlCon_GetMousePos(int *x, int *y, int *buttons)
{
   if (al_mouse_buttons == 1)
      return ;
      
   poll_mouse();
   *x = mouse_x / AlCon_FontWidth;
   *y = mouse_y / AlCon_FontHeight;
   *buttons = mouse_b;
  
   // Emulate 4th and 5th buttons through wheelmouse control.
   if (mouse_z > al_mouse_wheel)
      *buttons |= 1 << 3;
   if (mouse_z < al_mouse_wheel)
      *buttons |= 1 << 4;
   al_mouse_wheel = mouse_z;
}

uchar AlCon_ShapeFont10x20[]=
{
 0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x00,0x61,0x80,0x4C,0x80,0x4C,0x80,0x7C,0x80,0x79,0x80,0x73,0x80,0x73,0x80,0x73,0x80,0x7F,0x80,0x73,0x80,0x73,0x80,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x39,0x80,0x6D,0x80,0x67,0x00,0x00,0x00,0x39,0x80,0x6D,0x80,0x67,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x80,0x36,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x67,0x80,0x66,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x36,0x00,0x1F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x00,0x6D,0x80,0x6D,0x80,0x6F,0x80,0x6C,0x00,0x6C,0x00,0x6D,0x80,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x3F,0x00,0x7F,0x80,0x7F,0x80,0x3F,0x00,0x1E,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x7E,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x00,0x00,0x1F,0x80,0x06,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x7C,0x00,0x60,0x00,0x60,0x00,0x78,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x0F,0x80,0x0C,0x00,0x0C,0x00,0x0F,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x3C,0x00,0x66,0x00,0x60,0x00,0x60,0x00,0x66,0x00,0x3C,0x00,0x00,0x00,0x1F,0x00,0x19,0x80,0x19,0x80,0x1F,0x00,0x1E,0x00,0x1B,0x00,0x19,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x3E,0x00,0x00,0x00,0x0F,0x80,0x0C,0x00,0x0C,0x00,0x0F,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x55,0x40,0x00,0x00,0xAA,0x80,0x00,0x00,0x55,0x40,0x00,0x00,0xAA,0x80,0x00,0x00,0x55,0x40,0x00,0x00,0xAA,0x80,0x00,0x00,0x55,0x40,0x00,0x00,0xAA,0x80,0x00,0x00,0x55,0x40,0x00,0x00,0xAA,0x80,0x00,0x00,
 0x55,0x40,0xAA,0x80,0x55,0x40,0xAA,0x80,0x55,0x40,0xAA,0x80,0x55,0x40,0xAA,0x80,0x55,0x40,0xAA,0x80,0x55,0x40,0xAA,0x80,0x55,0x40,0xAA,0x80,0x55,0x40,0xAA,0x80,0x55,0x40,0xAA,0x80,0x55,0x40,0xAA,0x80,
 0x55,0x40,0xFF,0xC0,0xAA,0x80,0xFF,0xC0,0x55,0x40,0xFF,0xC0,0xAA,0x80,0xFF,0xC0,0x55,0x40,0xFF,0xC0,0xAA,0x80,0xFF,0xC0,0x55,0x40,0xFF,0xC0,0xAA,0x80,0xFF,0xC0,0x55,0x40,0xFF,0xC0,0xAA,0x80,0xFF,0xC0,
 0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,
 0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0xFF,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,0xF8,0x00,
 0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,0x07,0xC0,
 0x00,0x00,0x00,0x00,0x66,0x00,0x76,0x00,0x7E,0x00,0x7E,0x00,0x6E,0x00,0x66,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x3C,0x00,0x3C,0x00,0x18,0x00,0x18,0x00,0x00,0x00,0x1F,0x80,0x06,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0x00,0x38,0x00,0x0E,0x00,0x03,0x80,0x0E,0x00,0x38,0x00,0xE0,0x00,0x00,0x00,0x00,0x00,0xFF,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x80,0x0E,0x00,0x38,0x00,0xE0,0x00,0x38,0x00,0x0E,0x00,0x03,0x80,0x00,0x00,0x00,0x00,0xFF,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x00,0x03,0x00,0x06,0x00,0x7F,0x80,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x18,0x00,0x30,0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x80,0x07,0x80,0x1F,0x80,0x7F,0x80,0x7F,0x80,0x1F,0x80,0x07,0x80,0x01,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x78,0x00,0x7E,0x00,0x7F,0x80,0x7F,0x80,0x7E,0x00,0x78,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x3F,0x00,0x6D,0x80,0x4C,0x80,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x4C,0x80,0x6D,0x80,0x3F,0x00,0x1E,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x06,0x00,0x03,0x00,0xFF,0x80,0xFF,0x80,0x03,0x00,0x06,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x30,0x00,0x60,0x00,0xFF,0x80,0xFF,0x80,0x60,0x00,0x30,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x3F,0x00,0x6D,0x80,0x4C,0x80,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x4C,0x80,0x6D,0x80,0x3F,0x00,0x1E,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x12,0x00,0x33,0x00,0x61,0x80,0xFF,0xC0,0xFF,0xC0,0x61,0x80,0x33,0x00,0x12,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x19,0x80,0x31,0x80,0x7F,0x80,0x7F,0x80,0x30,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x33,0x00,0x33,0x00,0x33,0x00,0x33,0x00,0x33,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x00,0x33,0x00,0x33,0x00,0x12,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0D,0x80,0x0D,0x80,0x0D,0x80,0x3F,0xC0,0x1B,0x00,0x1B,0x00,0x1B,0x00,0x7F,0x80,0x36,0x00,0x36,0x00,0x36,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x3F,0x00,0x6D,0x80,0x6C,0x00,0x6C,0x00,0x6C,0x00,0x3F,0x00,0x0D,0x80,0x0D,0x80,0x0D,0x80,0x6D,0x80,0x3F,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x39,0x80,0x6D,0x80,0x6F,0x00,0x3B,0x00,0x06,0x00,0x06,0x00,0x0C,0x00,0x0C,0x00,0x1B,0x80,0x1E,0xC0,0x36,0xC0,0x33,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1C,0x00,0x36,0x00,0x36,0x00,0x36,0x00,0x3C,0x00,0x18,0x00,0x38,0x00,0x6C,0x00,0x66,0xC0,0x63,0x80,0x63,0x00,0x77,0x80,0x3C,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x00,0x06,0x00,0x0C,0x00,0x0C,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x0C,0x00,0x0C,0x00,0x06,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x18,0x00,0x0C,0x00,0x0C,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x0C,0x00,0x0C,0x00,0x18,0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x00,0x1E,0x00,0x7F,0x80,0x1E,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x00,0x0E,0x00,0x1C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x00,0x0E,0x00,0x0E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x80,0x01,0x80,0x03,0x00,0x03,0x00,0x06,0x00,0x06,0x00,0x0C,0x00,0x0C,0x00,0x18,0x00,0x18,0x00,0x30,0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x33,0x00,0x1E,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1C,0x00,0x3C,0x00,0x6C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x01,0x80,0x01,0x80,0x03,0x00,0x0E,0x00,0x18,0x00,0x30,0x00,0x60,0x00,0x60,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x01,0x80,0x03,0x00,0x0E,0x00,0x03,0x00,0x01,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x00,0x07,0x00,0x0F,0x00,0x1B,0x00,0x33,0x00,0x63,0x00,0x63,0x00,0x7F,0x80,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x6E,0x00,0x73,0x00,0x01,0x80,0x01,0x80,0x01,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x00,0x60,0x00,0x60,0x00,0x6E,0x00,0x73,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x01,0x80,0x01,0x80,0x03,0x00,0x03,0x00,0x06,0x00,0x06,0x00,0x0C,0x00,0x0C,0x00,0x18,0x00,0x18,0x00,0x30,0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x01,0x80,0x01,0x80,0x21,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x00,0x0E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x00,0x0E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x00,0x0E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x00,0x0E,0x00,0x1C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x30,0x00,0x60,0x00,0x30,0x00,0x18,0x00,0x0C,0x00,0x06,0x00,0x03,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x30,0x00,0x18,0x00,0x0C,0x00,0x06,0x00,0x03,0x00,0x01,0x80,0x03,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x30,0x00,0x20,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x03,0x00,0x06,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x67,0x80,0x6F,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x6F,0x00,0x66,0x00,0x60,0x00,0x31,0x80,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x7F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0x00,0x66,0x00,0x63,0x00,0x63,0x00,0x63,0x00,0x66,0x00,0x7E,0x00,0x63,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x63,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0x00,0x63,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x63,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7E,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7E,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x67,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1E,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x7F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0F,0xC0,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x63,0x00,0x63,0x00,0x36,0x00,0x1C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x63,0x00,0x63,0x00,0x66,0x00,0x66,0x00,0x7C,0x00,0x66,0x00,0x66,0x00,0x63,0x00,0x63,0x00,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x73,0x80,0x73,0x80,0x7F,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x71,0x80,0x71,0x80,0x79,0x80,0x79,0x80,0x6D,0x80,0x6D,0x80,0x67,0x80,0x67,0x80,0x63,0x80,0x63,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0x00,0x63,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x63,0x00,0x7E,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x6D,0x80,0x67,0x80,0x33,0x00,0x1F,0x00,0x01,0x80,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0x00,0x63,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x63,0x00,0x7E,0x00,0x66,0x00,0x63,0x00,0x63,0x00,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x60,0x00,0x60,0x00,0x30,0x00,0x1E,0x00,0x03,0x00,0x01,0x80,0x01,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x33,0x00,0x33,0x00,0x1E,0x00,0x1E,0x00,0x1E,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x73,0x80,0x73,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x33,0x00,0x33,0x00,0x1E,0x00,0x1E,0x00,0x0C,0x00,0x1E,0x00,0x1E,0x00,0x33,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x33,0x00,0x33,0x00,0x1E,0x00,0x1E,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x01,0x80,0x01,0x80,0x03,0x00,0x06,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x18,0x00,0x30,0x00,0x60,0x00,0x60,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x30,0x00,0x18,0x00,0x18,0x00,0x0C,0x00,0x0C,0x00,0x06,0x00,0x06,0x00,0x03,0x00,0x03,0x00,0x01,0x80,0x01,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x0C,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x00,0x31,0x80,0x01,0x80,0x3F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x3E,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x6E,0x00,0x73,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x73,0x00,0x6E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x00,0x31,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x31,0x80,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x1D,0x80,0x33,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x7F,0x80,0x60,0x00,0x60,0x00,0x31,0x80,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0F,0x00,0x19,0x80,0x19,0x80,0x18,0x00,0x18,0x00,0x7E,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3E,0x80,0x63,0x80,0x63,0x00,0x63,0x00,0x63,0x00,0x3E,0x00,0x60,0x00,0x3F,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x3F,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x6E,0x00,0x73,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x3C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x80,0x01,0x80,0x00,0x00,0x07,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x31,0x80,0x31,0x80,0x31,0x80,0x1F,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x63,0x00,0x66,0x00,0x6C,0x00,0x78,0x00,0x7C,0x00,0x66,0x00,0x63,0x00,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x3C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x5B,0x00,0x7F,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x6E,0x00,0x73,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x6E,0x00,0x73,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x73,0x00,0x6E,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1D,0x80,0x33,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x6F,0x00,0x39,0x80,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x00,0x61,0x80,0x60,0x00,0x3F,0x00,0x01,0x80,0x01,0x80,0x61,0x80,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x7E,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x19,0x80,0x0F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x33,0x00,0x33,0x00,0x1E,0x00,0x1E,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x6D,0x80,0x6D,0x80,0x6D,0x80,0x7F,0x80,0x33,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x33,0x00,0x1E,0x00,0x0C,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x01,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x80,0x01,0x80,0x03,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x30,0x00,0x3F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x07,0x80,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x78,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x07,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x07,0x80,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x39,0x80,0x6D,0x80,0x67,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x33,0x00,0x33,0x00,0x1E,0x00,0x1E,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x07,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x0F,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,
 0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0F,0xC0,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,
 0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x0F,0xC0,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0xF8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0xFF,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,
 0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0xF8,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xC0,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,
 0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0xFF,0xC0,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x07,0xC0,0x04,0x00,0x07,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x13,0xC0,0x10,0x00,0x1F,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,
 0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0xC0,0x10,0x00,0x13,0xC0,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,
 0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x13,0xC0,0x10,0x00,0x13,0xC0,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x00,0x08,0x00,0xF8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0xF2,0x00,0x02,0x00,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xC0,0x00,0x00,0xFF,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0xF3,0xC0,0x00,0x00,0xFF,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x00,0x02,0x00,0xF2,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,
 0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0xF2,0x00,0x02,0x00,0xF2,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xC0,0x00,0x00,0xF3,0xC0,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,
 0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0xF3,0xC0,0x00,0x00,0xF3,0xC0,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x61,0x00,0x60,0x00,0x60,0x00,0x61,0x00,0x33,0x00,0x1E,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0F,0x00,0x19,0x80,0x19,0x80,0x18,0x00,0x18,0x00,0x7E,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x7C,0x00,0x56,0xC0,0x73,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x3F,0x00,0x6D,0x80,0x6C,0x00,0x6C,0x00,0x6C,0x00,0x3F,0x00,0x0D,0x80,0x0D,0x80,0x0D,0x80,0x6D,0x80,0x3F,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x3F,0x00,0x0C,0x00,0x3F,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x63,0x00,0x30,0x00,0x3C,0x00,0x66,0x00,0x33,0x00,0x19,0x80,0x0F,0x00,0x03,0x00,0x31,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x5E,0x80,0x52,0x80,0x50,0x80,0x52,0x80,0x5E,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x00,0x21,0x80,0x01,0x80,0x3F,0x80,0x61,0x80,0x61,0x80,0x3E,0x80,0x00,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x80,0x0D,0x80,0x1B,0x00,0x36,0x00,0x6C,0x00,0xD8,0x00,0x6C,0x00,0x36,0x00,0x1B,0x00,0x0D,0x80,0x04,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x7F,0x80,0x01,0x80,0x01,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x5E,0x80,0x52,0x80,0x5E,0x80,0x54,0x80,0x56,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x33,0x00,0x1E,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x0C,0x00,0x0C,0x00,0x00,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1C,0x00,0x36,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x30,0x00,0x3E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1C,0x00,0x36,0x00,0x06,0x00,0x0C,0x00,0x06,0x00,0x36,0x00,0x1C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x63,0x00,0x63,0x00,0x63,0x00,0x63,0x00,0x63,0x00,0x77,0x00,0x7D,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x80,0x7F,0x80,0x7D,0x80,0x7D,0x80,0x7D,0x80,0x3D,0x80,0x0D,0x80,0x0D,0x80,0x0D,0x80,0x0D,0x80,0x0D,0x80,0x0D,0x80,0x0D,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x1E,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x06,0x00,0x36,0x00,0x1C,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x38,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1C,0x00,0x36,0x00,0x63,0x00,0x63,0x00,0x63,0x00,0x36,0x00,0x1C,0x00,0x00,0x00,0x7F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x6C,0x00,0x36,0x00,0x1B,0x00,0x0D,0x80,0x06,0xC0,0x0D,0x80,0x1B,0x00,0x36,0x00,0x6C,0x00,0x48,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x60,0x00,0x20,0x80,0x21,0x00,0x72,0x00,0x04,0x00,0x09,0x00,0x13,0x00,0x25,0x00,0x4F,0x00,0x01,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x60,0x00,0x20,0x80,0x21,0x00,0x72,0x00,0x04,0x00,0x0B,0x00,0x14,0x80,0x20,0x80,0x41,0x00,0x02,0x00,0x07,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x70,0x00,0x08,0x00,0x30,0x80,0x09,0x00,0x72,0x00,0x04,0x00,0x09,0x00,0x13,0x00,0x25,0x00,0x4F,0x80,0x01,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x18,0x00,0x30,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x30,0x00,0x18,0x00,0x0C,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x7F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x03,0x00,0x06,0x00,0x0C,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x7F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x7F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x19,0x00,0x3F,0x00,0x26,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x7F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x7F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x1E,0x00,0x33,0x00,0x33,0x00,0x1E,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x7F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0F,0x80,0x1E,0x00,0x36,0x00,0x36,0x00,0x66,0x00,0x66,0x00,0x7F,0x80,0x66,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x67,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x61,0x80,0x33,0x00,0x1E,0x00,0x0C,0x00,0x06,0x00,0x36,0x00,0x1C,0x00,
 0x00,0x00,0x30,0x00,0x18,0x00,0x0C,0x00,0x00,0x00,0x7F,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7E,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x7F,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7E,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x7F,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7E,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7E,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x18,0x00,0x0C,0x00,0x06,0x00,0x00,0x00,0x3F,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x3F,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x3F,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x3F,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0x00,0x63,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0xF9,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x63,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x19,0x00,0x3F,0x00,0x26,0x00,0x00,0x00,0x61,0x80,0x71,0x80,0x79,0x80,0x79,0x80,0x6D,0x80,0x6D,0x80,0x67,0x80,0x67,0x80,0x63,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x18,0x00,0x0C,0x00,0x06,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x19,0x00,0x3F,0x00,0x26,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x41,0x00,0x63,0x00,0x36,0x00,0x1C,0x00,0x1C,0x00,0x36,0x00,0x63,0x00,0x41,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x80,0x1F,0x00,0x33,0x00,0x63,0x80,0x63,0x80,0x65,0x80,0x65,0x80,0x65,0x80,0x69,0x80,0x69,0x80,0x69,0x80,0x71,0x80,0x33,0x00,0x3E,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x18,0x00,0x0C,0x00,0x06,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x33,0x00,0x33,0x00,0x1E,0x00,0x1E,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x3F,0x00,0x31,0x80,0x31,0x80,0x31,0x80,0x31,0x80,0x31,0x80,0x3F,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x00,0x1B,0x00,0x31,0x80,0x31,0x80,0x33,0x00,0x76,0x00,0x36,0x00,0x33,0x00,0x31,0x80,0x31,0x80,0x31,0x80,0x33,0x00,0x36,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x0C,0x00,0x06,0x00,0x00,0x00,0x3F,0x00,0x61,0x80,0x01,0x80,0x3F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x3E,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x3F,0x00,0x61,0x80,0x01,0x80,0x3F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x3E,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x3F,0x00,0x61,0x80,0x01,0x80,0x3F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x3E,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x19,0x00,0x3F,0x00,0x26,0x00,0x00,0x00,0x3F,0x00,0x61,0x80,0x01,0x80,0x3F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x3E,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x3F,0x00,0x61,0x80,0x01,0x80,0x3F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x3E,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x33,0x00,0x1E,0x00,0x00,0x00,0x3F,0x00,0x61,0x80,0x01,0x80,0x3F,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x3E,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3B,0x00,0x4D,0x80,0x0D,0x80,0x0F,0x00,0x3C,0x00,0x6C,0x00,0x6C,0x80,0x37,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x00,0x31,0x80,0x60,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x31,0x80,0x1F,0x00,0x0C,0x00,0x06,0x00,0x36,0x00,0x1C,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x18,0x00,0x0C,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x7F,0x80,0x60,0x00,0x60,0x00,0x31,0x80,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x00,0x06,0x00,0x0C,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x7F,0x80,0x60,0x00,0x60,0x00,0x31,0x80,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x7F,0x80,0x60,0x00,0x60,0x00,0x31,0x80,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x7F,0x80,0x60,0x00,0x60,0x00,0x31,0x80,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x18,0x00,0x0C,0x00,0x00,0x00,0x3C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x3C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x3C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x3C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x7F,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x80,0x01,0x80,0x01,0x80,0x07,0xC0,0x01,0x80,0x1D,0x80,0x33,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x19,0x00,0x3F,0x00,0x26,0x00,0x00,0x00,0x6E,0x00,0x73,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x18,0x00,0x0C,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x00,0x06,0x00,0x0C,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x19,0x00,0x3F,0x00,0x26,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x1E,0x00,0x33,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x7F,0x80,0x7F,0x80,0x00,0x00,0x00,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x1F,0x00,0x33,0x00,0x65,0x80,0x65,0x80,0x69,0x80,0x69,0x80,0x33,0x00,0x3E,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x0C,0x00,0x06,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x00,0x06,0x00,0x0C,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x00,0x1E,0x00,0x33,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x01,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x3E,0x00,0x33,0x00,0x31,0x80,0x31,0x80,0x31,0x80,0x33,0x00,0x3E,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x33,0x00,0x00,0x00,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x61,0x80,0x33,0x80,0x1D,0x80,0x01,0x80,0x61,0x80,0x33,0x00,0x1E,0x00,
};

// Exported
uchar AlCon_ShapeFont8x16[]=
{
 0x00,0x00,0x7E,0xC3,0x99,0x99,0xF3,0xE7,0xE7,0xFF,0xE7,0xE7,0x7E,0x00,0x00,0x00, // 0
 0x00,0x00,0x00,0x00,0x00,0x76,0xDC,0x00,0x76,0xDC,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x6E,0xF8,0xD8,0xD8,0xDC,0xD8,0xD8,0xD8,0xF8,0x6E,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x6E,0xDB,0xDB,0xDF,0xD8,0xDB,0x6E,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x10,0x38,0x7C,0xFE,0x7C,0x38,0x10,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x88,0x88,0xF8,0x88,0x88,0x00,0x3E,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00, // 
 0x00,0xF8,0x80,0xE0,0x80,0x80,0x00,0x3E,0x20,0x38,0x20,0x20,0x00,0x00,0x00,0x00, // 
 0x00,0x70,0x88,0x80,0x88,0x70,0x00,0x3C,0x22,0x3C,0x24,0x22,0x00,0x00,0x00,0x00, // 
 0x00,0x80,0x80,0x80,0x80,0xF8,0x00,0x3E,0x20,0x38,0x20,0x20,0x00,0x00,0x00,0x00, // 
 0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x44,0x11,0x44, // \t
 0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA,0x55,0xAA, // \n
 0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,0x77,0xDD,0x77, // 
 0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF, // \r
 0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0,0xF0, // 
 0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F,0x0F, // 
 0x00,0x88,0xC8,0xA8,0x98,0x88,0x00,0x20,0x20,0x20,0x20,0x3E,0x00,0x00,0x00,0x00, // 
 0x00,0x88,0x88,0x50,0x50,0x20,0x00,0x3E,0x08,0x08,0x08,0x08,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x0E,0x38,0xE0,0x38,0x0E,0x00,0xFE,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0xE0,0x38,0x0E,0x38,0xE0,0x00,0xFE,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x06,0x0C,0xFE,0x18,0x30,0xFE,0x60,0xC0,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x06,0x1E,0x7E,0xFE,0x7E,0x1E,0x06,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0xC0,0xF0,0xFC,0xFE,0xFC,0xF0,0xC0,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x18,0x3C,0x7E,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x7E,0x3C,0x18,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x18,0x0C,0xFE,0x0C,0x18,0x00,0x00,0x00,0x00,0x00,0x00, // 0x1A
 0x00,0x00,0x00,0x00,0x00,0x30,0x60,0xFE,0x60,0x30,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x18,0x3C,0x7E,0x18,0x18,0x18,0x18,0x7E,0x3C,0x18,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x28,0x6C,0xFE,0x6C,0x28,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x06,0x36,0x66,0xFE,0x60,0x30,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x6C,0x6C,0x6C,0x6C,0x6C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, //  
 0x00,0x00,0x18,0x3C,0x3C,0x3C,0x18,0x18,0x18,0x00,0x18,0x18,0x00,0x00,0x00,0x00, // !
 0x00,0x66,0x66,0x66,0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // "
 0x00,0x00,0x00,0x6C,0x6C,0xFE,0x6C,0x6C,0x6C,0xFE,0x6C,0x6C,0x00,0x00,0x00,0x00, // #
 0x00,0x10,0x10,0x7C,0xD6,0xD0,0xD0,0x7C,0x16,0x16,0xD6,0x7C,0x10,0x10,0x00,0x00, // $
 0x00,0x00,0x00,0x00,0xC2,0xC6,0x0C,0x18,0x30,0x60,0xC6,0x86,0x00,0x00,0x00,0x00, // %
 0x00,0x00,0x38,0x6C,0x6C,0x38,0x76,0xDC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // &
 0x00,0x18,0x18,0x18,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // '
 0x00,0x00,0x0C,0x18,0x30,0x30,0x30,0x30,0x30,0x30,0x18,0x0C,0x00,0x00,0x00,0x00, // (
 0x00,0x00,0x30,0x18,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x18,0x30,0x00,0x00,0x00,0x00, // )
 0x00,0x00,0x00,0x00,0x00,0x66,0x3C,0xFF,0x3C,0x66,0x00,0x00,0x00,0x00,0x00,0x00, // *
 0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x7E,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00, // +
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x30,0x00,0x00,0x00, // ,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // -
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x00, // .
 0x00,0x00,0x00,0x00,0x00,0x06,0x0C,0x18,0x30,0x60,0xC0,0x00,0x00,0x00,0x00,0x00, // /
 0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xD6,0xD6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 0
 0x00,0x00,0x18,0x38,0x78,0x18,0x18,0x18,0x18,0x18,0x18,0x7E,0x00,0x00,0x00,0x00, // 1
 0x00,0x00,0x7C,0xC6,0x06,0x0C,0x18,0x30,0x60,0xC0,0xC6,0xFE,0x00,0x00,0x00,0x00, // 2
 0x00,0x00,0x7C,0xC6,0x06,0x06,0x3C,0x06,0x06,0x06,0xC6,0x7C,0x00,0x00,0x00,0x00, // 3
 0x00,0x00,0x0C,0x1C,0x3C,0x6C,0xCC,0xFE,0x0C,0x0C,0x0C,0x1E,0x00,0x00,0x00,0x00, // 4
 0x00,0x00,0xFE,0xC0,0xC0,0xC0,0xFC,0x06,0x06,0x06,0xC6,0x7C,0x00,0x00,0x00,0x00, // 5
 0x00,0x00,0x38,0x60,0xC0,0xC0,0xFC,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 6
 0x00,0x00,0xFE,0xC6,0x06,0x06,0x0C,0x18,0x30,0x30,0x30,0x30,0x00,0x00,0x00,0x00, // 7
 0x00,0x00,0x7C,0xC6,0xC6,0xC6,0x7C,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 8
 0x00,0x00,0x7C,0xC6,0xC6,0xC6,0x7E,0x06,0x06,0x06,0x0C,0x78,0x00,0x00,0x00,0x00, // 9
 0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x00,0x00, // :
 0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x18,0x18,0x30,0x00,0x00,0x00,0x00, // ;
 0x00,0x00,0x00,0x06,0x0C,0x18,0x30,0x60,0x30,0x18,0x0C,0x06,0x00,0x00,0x00,0x00, // <
 0x00,0x00,0x00,0x00,0x00,0xFE,0x00,0x00,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // =
 0x00,0x00,0x00,0x60,0x30,0x18,0x0C,0x06,0x0C,0x18,0x30,0x60,0x00,0x00,0x00,0x00, // >
 0x00,0x00,0x7C,0xC6,0xC6,0x0C,0x18,0x18,0x18,0x00,0x18,0x18,0x00,0x00,0x00,0x00, // ?
 0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xDE,0xDE,0xDE,0xDC,0xC0,0x7C,0x00,0x00,0x00,0x00, // @
 0x00,0x00,0x10,0x38,0x6C,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // A
 0x00,0x00,0xFC,0x66,0x66,0x66,0x7C,0x66,0x66,0x66,0x66,0xFC,0x00,0x00,0x00,0x00, // B
 0x00,0x00,0x3C,0x66,0xC2,0xC0,0xC0,0xC0,0xC0,0xC2,0x66,0x3C,0x00,0x00,0x00,0x00, // C
 0x00,0x00,0xF8,0x6C,0x66,0x66,0x66,0x66,0x66,0x66,0x6C,0xF8,0x00,0x00,0x00,0x00, // D
 0x00,0x00,0xFE,0x66,0x62,0x68,0x78,0x68,0x60,0x62,0x66,0xFE,0x00,0x00,0x00,0x00, // E
 0x00,0x00,0xFE,0x66,0x62,0x68,0x78,0x68,0x60,0x60,0x60,0xF0,0x00,0x00,0x00,0x00, // F
 0x00,0x00,0x3C,0x66,0xC2,0xC0,0xC0,0xDE,0xC6,0xC6,0x66,0x3A,0x00,0x00,0x00,0x00, // G
 0x00,0x00,0xC6,0xC6,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // H
 0x00,0x00,0x3C,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // I
 0x00,0x00,0x1E,0x0C,0x0C,0x0C,0x0C,0x0C,0xCC,0xCC,0xCC,0x78,0x00,0x00,0x00,0x00, // J
 0x00,0x00,0xE6,0x66,0x66,0x6C,0x78,0x78,0x6C,0x66,0x66,0xE6,0x00,0x00,0x00,0x00, // K
 0x00,0x00,0xF0,0x60,0x60,0x60,0x60,0x60,0x60,0x62,0x66,0xFE,0x00,0x00,0x00,0x00, // L
 0x00,0x00,0xC6,0xEE,0xFE,0xFE,0xD6,0xC6,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // M
 0x00,0x00,0xC6,0xE6,0xF6,0xFE,0xDE,0xCE,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // N
 0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // O
 0x00,0x00,0xFC,0x66,0x66,0x66,0x7C,0x60,0x60,0x60,0x60,0xF0,0x00,0x00,0x00,0x00, // P
 0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xD6,0xDE,0x7C,0x0C,0x0E,0x00,0x00, // Q
 0x00,0x00,0xFC,0x66,0x66,0x66,0x7C,0x6C,0x66,0x66,0x66,0xE6,0x00,0x00,0x00,0x00, // R
 0x00,0x00,0x7C,0xC6,0xC6,0x64,0x38,0x0C,0x06,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // S
 0x00,0x00,0x7E,0x7E,0x5A,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // T
 0x00,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // U
 0x00,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x6C,0x38,0x10,0x00,0x00,0x00,0x00, // V
 0x00,0x00,0xC6,0xC6,0xC6,0xC6,0xD6,0xD6,0xD6,0xFE,0xEE,0x6C,0x00,0x00,0x00,0x00, // W
 0x00,0x00,0xC6,0xC6,0x6C,0x7C,0x38,0x38,0x7C,0x6C,0xC6,0xC6,0x00,0x00,0x00,0x00, // X
 0x00,0x00,0x66,0x66,0x66,0x66,0x3C,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // Y
 0x00,0x00,0xFE,0xC6,0x86,0x0C,0x18,0x30,0x60,0xC2,0xC6,0xFE,0x00,0x00,0x00,0x00, // Z
 0x00,0x00,0x3C,0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x3C,0x00,0x00,0x00,0x00, // [
 0x00,0x00,0x00,0x00,0x00,0xC0,0x60,0x30,0x18,0x0C,0x06,0x00,0x00,0x00,0x00,0x00, // \ .
 0x00,0x00,0x3C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3C,0x00,0x00,0x00,0x00, // ]
 0x10,0x38,0x6C,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // ^
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00, // _
 0x00,0x30,0x30,0x30,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // `
 0x00,0x00,0x00,0x00,0x00,0x78,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // a
 0x00,0x00,0xE0,0x60,0x60,0x78,0x6C,0x66,0x66,0x66,0x66,0x7C,0x00,0x00,0x00,0x00, // b
 0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xC0,0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00, // c
 0x00,0x00,0x1C,0x0C,0x0C,0x3C,0x6C,0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // d
 0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00, // e
 0x00,0x00,0x38,0x6C,0x64,0x60,0xF0,0x60,0x60,0x60,0x60,0xF0,0x00,0x00,0x00,0x00, // f
 0x00,0x00,0x00,0x00,0x00,0x76,0xCC,0xCC,0xCC,0xCC,0xCC,0x7C,0x0C,0xCC,0x78,0x00, // g
 0x00,0x00,0xE0,0x60,0x60,0x6C,0x76,0x66,0x66,0x66,0x66,0xE6,0x00,0x00,0x00,0x00, // h
 0x00,0x00,0x18,0x18,0x00,0x38,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // i
 0x00,0x00,0x06,0x06,0x00,0x0E,0x06,0x06,0x06,0x06,0x06,0x06,0x66,0x66,0x3C,0x00, // j
 0x00,0x00,0xE0,0x60,0x60,0x66,0x6C,0x78,0x78,0x6C,0x66,0xE6,0x00,0x00,0x00,0x00, // k
 0x00,0x00,0x38,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // l
 0x00,0x00,0x00,0x00,0x00,0xEC,0xFE,0xD6,0xD6,0xD6,0xD6,0xC6,0x00,0x00,0x00,0x00, // m
 0x00,0x00,0x00,0x00,0x00,0xDC,0x66,0x66,0x66,0x66,0x66,0x66,0x00,0x00,0x00,0x00, // n
 0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // o
 0x00,0x00,0x00,0x00,0x00,0xDC,0x66,0x66,0x66,0x66,0x66,0x7C,0x60,0x60,0xF0,0x00, // p
 0x00,0x00,0x00,0x00,0x00,0x76,0xCC,0xCC,0xCC,0xCC,0xCC,0x7C,0x0C,0x0C,0x1E,0x00, // q
 0x00,0x00,0x00,0x00,0x00,0xDC,0x76,0x66,0x60,0x60,0x60,0xF0,0x00,0x00,0x00,0x00, // r
 0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0x60,0x38,0x0C,0xC6,0x7C,0x00,0x00,0x00,0x00, // s
 0x00,0x00,0x10,0x30,0x30,0xFC,0x30,0x30,0x30,0x30,0x36,0x1C,0x00,0x00,0x00,0x00, // t
 0x00,0x00,0x00,0x00,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // u
 0x00,0x00,0x00,0x00,0x00,0x66,0x66,0x66,0x66,0x66,0x3C,0x18,0x00,0x00,0x00,0x00, // v
 0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,0xD6,0xD6,0xD6,0xFE,0x6C,0x00,0x00,0x00,0x00, // w
 0x00,0x00,0x00,0x00,0x00,0xC6,0x6C,0x38,0x38,0x38,0x6C,0xC6,0x00,0x00,0x00,0x00, // x
 0x00,0x00,0x00,0x00,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7E,0x06,0x0C,0xF8,0x00, // y
 0x00,0x00,0x00,0x00,0x00,0xFE,0xCC,0x18,0x30,0x60,0xC6,0xFE,0x00,0x00,0x00,0x00, // z
 0x00,0x00,0x0E,0x18,0x18,0x18,0x70,0x18,0x18,0x18,0x18,0x0E,0x00,0x00,0x00,0x00, // {
 0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00, // |
 0x00,0x00,0x70,0x18,0x18,0x18,0x0E,0x18,0x18,0x18,0x18,0x70,0x00,0x00,0x00,0x00, // }
 0x00,0x00,0x76,0xDC,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // ~
 0x00,0x66,0x00,0x66,0x66,0x66,0x66,0x3C,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18, // 
 0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18, // 
 0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x1F,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xF8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18, // 
 0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xF8,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18, // 
 0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xFF,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0x00,0x00,0x00,0x00, // 
 0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x60,0x7F,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6F,0x60,0x7F,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C, // 
 0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x7F,0x60,0x6F,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C, // 
 0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6F,0x60,0x6F,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0xFC,0x0C,0xFC,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0xEC,0x0C,0xFC,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0xEF,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0xFC,0x0C,0xEC,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C, // 
 0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0xEC,0x0C,0xEC,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0x00,0xEF,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C, // 
 0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0xEF,0x00,0xEF,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C,0x6C, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x82,0xFE,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x18,0x18,0x18,0x3C,0x3C,0x3C,0x18,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x10,0x7C,0xD6,0xD0,0xD0,0xD0,0xD6,0x7C,0x10,0x00,0x00,0x00, // 
 0x00,0x00,0x38,0x6C,0x60,0x60,0xF0,0x60,0x60,0x66,0xF6,0x6C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0xC6,0x7C,0x6C,0x6C,0x7C,0xC6,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x66,0x66,0x3C,0x18,0x7E,0x18,0x7E,0x18,0x18,0x18,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x18,0x18,0x18,0x18,0x00,0x18,0x18,0x18,0x18,0x18,0x00,0x00,0x00,0x00, // 
 0x00,0x7C,0xC6,0x60,0x38,0x6C,0xC6,0xC6,0x6C,0x38,0x0C,0xC6,0x7C,0x00,0x00,0x00, // 
 0x00,0x6C,0x6C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x3C,0x42,0x99,0xA5,0xA1,0xA5,0x99,0x42,0x3C,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x3C,0x6C,0x6C,0x3E,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x36,0x6C,0xD8,0x6C,0x36,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x06,0x06,0x06,0x06,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x3C,0x42,0xB9,0xA5,0xB9,0xA5,0xA5,0x42,0x3C,0x00,0x00,0x00,0x00,0x00, // 
 0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x38,0x6C,0x6C,0x38,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x7E,0x18,0x18,0x00,0x7E,0x00,0x00,0x00,0x00, // 
 0x38,0x6C,0x18,0x30,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x38,0x6C,0x18,0x6C,0x38,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x18,0x30,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0xF6,0xC0,0xC0,0xC0,0x00, // 
 0x00,0x00,0x7F,0xD6,0xD6,0x76,0x36,0x36,0x36,0x36,0x36,0x36,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x6C,0x38,0x00, // 
 0x30,0x70,0x30,0x30,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x38,0x6C,0x6C,0x38,0x00,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0xD8,0x6C,0x36,0x6C,0xD8,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x60,0xE0,0x60,0x66,0x0C,0x18,0x30,0x66,0xCE,0x1A,0x3F,0x06,0x06,0x00,0x00, // 
 0x00,0x60,0xE0,0x60,0x66,0x0C,0x18,0x30,0x6E,0xDB,0x06,0x0C,0x1F,0x00,0x00,0x00, // 
 0x70,0xD8,0x30,0xD8,0x76,0x0C,0x18,0x30,0x66,0xCE,0x1A,0x3F,0x06,0x06,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x30,0x30,0x00,0x30,0x30,0x30,0x60,0xC6,0xC6,0x7C,0x00,0x00, // 
 0x60,0x30,0x00,0x38,0x6C,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // 
 0x0C,0x18,0x00,0x38,0x6C,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // 
 0x10,0x38,0x6C,0x00,0x38,0x6C,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // 
 0x76,0xDC,0x00,0x38,0x6C,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // 
 0x00,0x6C,0x00,0x38,0x6C,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // 
 0x38,0x6C,0x38,0x00,0x38,0x6C,0xC6,0xC6,0xFE,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x3E,0x78,0xD8,0xD8,0xFC,0xD8,0xD8,0xD8,0xD8,0xDE,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x3C,0x66,0xC2,0xC0,0xC0,0xC0,0xC0,0xC2,0x66,0x3C,0x0C,0x66,0x3C,0x00, // 
 0x60,0x30,0x00,0xFE,0x66,0x60,0x60,0x7C,0x60,0x60,0x66,0xFE,0x00,0x00,0x00,0x00, // 
 0x0C,0x18,0x00,0xFE,0x66,0x60,0x60,0x7C,0x60,0x60,0x66,0xFE,0x00,0x00,0x00,0x00, // 
 0x10,0x38,0x6C,0x00,0xFE,0x66,0x60,0x7C,0x60,0x60,0x66,0xFE,0x00,0x00,0x00,0x00, // 
 0x00,0x6C,0x00,0xFE,0x66,0x60,0x60,0x7C,0x60,0x60,0x66,0xFE,0x00,0x00,0x00,0x00, // 
 0x60,0x30,0x00,0x3C,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x06,0x0C,0x00,0x3C,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x18,0x3C,0x66,0x00,0x3C,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x00,0x66,0x00,0x3C,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0xF8,0x6C,0x66,0x66,0xF6,0x66,0x66,0x66,0x6C,0xF8,0x00,0x00,0x00,0x00, // 
 0x76,0xDC,0x00,0xC6,0xE6,0xF6,0xFE,0xDE,0xCE,0xC6,0xC6,0xC6,0x00,0x00,0x00,0x00, // 
 0x60,0x30,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x0C,0x18,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x10,0x38,0x6C,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x76,0xDC,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x6C,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x66,0x3C,0x18,0x3C,0x66,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x7E,0xC6,0xCE,0xCE,0xDE,0xF6,0xE6,0xE6,0xC6,0xFC,0x00,0x00,0x00,0x00, // 
 0x60,0x30,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x0C,0x18,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x10,0x38,0x6C,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x6C,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x06,0x0C,0x00,0x66,0x66,0x66,0x66,0x3C,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0xF0,0x60,0x7C,0x66,0x66,0x66,0x66,0x7C,0x60,0xF0,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xCC,0xC6,0xC6,0xC6,0xD6,0xDC,0x80,0x00,0x00,0x00, // 
 0x00,0x60,0x30,0x18,0x00,0x78,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x18,0x30,0x60,0x00,0x78,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x10,0x38,0x6C,0x00,0x78,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x76,0xDC,0x00,0x78,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x6C,0x00,0x78,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x38,0x6C,0x38,0x00,0x78,0x0C,0x7C,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x7E,0xDB,0x1B,0x7F,0xD8,0xDB,0x7E,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x7C,0xC6,0xC0,0xC0,0xC0,0xC6,0x7C,0x18,0x6C,0x38,0x00, // 
 0x00,0x60,0x30,0x18,0x00,0x7C,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x0C,0x18,0x30,0x00,0x7C,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x10,0x38,0x6C,0x00,0x7C,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x6C,0x00,0x7C,0xC6,0xFE,0xC0,0xC0,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x60,0x30,0x18,0x00,0x38,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x00,0x0C,0x18,0x30,0x00,0x38,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x00,0x18,0x3C,0x66,0x00,0x38,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x6C,0x00,0x38,0x18,0x18,0x18,0x18,0x18,0x3C,0x00,0x00,0x00,0x00, // 
 0x00,0x78,0x30,0x78,0x0C,0x7E,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x76,0xDC,0x00,0xDC,0x66,0x66,0x66,0x66,0x66,0x66,0x00,0x00,0x00,0x00, // 
 0x00,0x60,0x30,0x18,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x0C,0x18,0x30,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x10,0x38,0x6C,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x76,0xDC,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x6C,0x00,0x7C,0xC6,0xC6,0xC6,0xC6,0xC6,0x7C,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x18,0x00,0x7E,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0x00,0x00,0x7E,0xCE,0xDE,0xFE,0xF6,0xE6,0xFC,0x00,0x00,0x00,0x00, // 
 0x00,0x60,0x30,0x18,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x18,0x30,0x60,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x30,0x78,0xCC,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x00,0x00,0xCC,0x00,0xCC,0xCC,0xCC,0xCC,0xCC,0xCC,0x76,0x00,0x00,0x00,0x00, // 
 0x00,0x0C,0x18,0x30,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7E,0x06,0x0C,0xF8,0x00, // 
 0x00,0x00,0xF0,0x60,0x60,0x7C,0x66,0x66,0x66,0x66,0x7C,0x60,0x60,0xF0,0x00,0x00, // 
 0x00,0x00,0x00,0x6C,0x00,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x7E,0x06,0x0C,0xF8,0x00  // 255
};

void AlCon_SetFont(int which, uchar *fnt, unsigned w, unsigned h)
{
 unsigned sizeChanged=w!=(unsigned)AlCon_FontWidth ||
                      h!=(unsigned)AlCon_FontHeight;
 AlCon_FontWidth =w;
 AlCon_FontHeight=h;
 unsigned wb=(w+7)/8;
 unsigned sizeGlyph=wb*h;

 FONT_GLYPH **p=which ? Alcon_ASCIIDataS : Alcon_ASCIIDataP;
 // Copy font from buffer to internal structure.
 for (int i=0; i<0x100; i++)
    {// Free previously allocated memory.
     if (p[i])
        free(p[i]);
        
     p[i]=(FONT_GLYPH *)malloc(sizeof(FONT_GLYPH)+sizeGlyph);
     p[i]->w=AlCon_FontWidth;
     p[i]->h=AlCon_FontHeight;
 
     memcpy(p[i]->dat,fnt,sizeGlyph);
     fnt+=sizeGlyph;
    }
 Alcon_PrimaryFont.height  =
 Alcon_SecondaryFont.height=
 Alcon_CursorFont.height   =AlCon_FontHeight;

 if (sizeChanged)
    Alcon_CreateNewCursor();
}

/**[txh]********************************************************************

  Description: Loads the specified binary format font into an internal
  structure like Allegro's font and then sets the static global font pointer
  to that. If the font could not be loaded, the previously loaded custom
  font will be used. If this was the first custom font call, Allegro's
  default font will be used if everything fails. If the filename is a NULL
  pointer, the default font will be used.
  
***************************************************************************/

static
void AlCon_LoadCustomFont(const char *filename, uchar *fdata, unsigned fw,
                          unsigned fh)
{
   ASSERT(screen && "You have to call set_gfx_mode before.");
   //static bool one_custom_font_loaded = false;
   unsigned wb=(fw+7)/8;

   FILE *file = NULL;
   if (filename)
      file = fopen(filename, "rb");

   unsigned maxRead=wb*fh*0x100;
   uchar *font_buffer, buffer[maxRead];
   int read_bytes=0;
   if (!file) {
      if (fdata) {
         font_buffer=fdata;
         read_bytes=maxRead;
      } else {
         font_buffer=AlCon_ShapeFont8x16;
         wb=1; fw=8; fh=16;
         read_bytes=4096;
      }
   } else {
      font_buffer=buffer;
      int read_bytes=fread(font_buffer, 1, maxRead, file);
      fclose(file);
      if (read_bytes < 256) {// Absurd size limit
         allegro_message("Absurd custom font size (%d bytes)", read_bytes);
         exit(3);
      }
      //one_custom_font_loaded = true;
   }

   // Initialize global variables.
   Alcon_CurrentFont=&Alcon_PrimaryFont;
   Alcon_PrimaryFont.vtable=font->vtable;
   Alcon_PrimaryFont.data=&Alcon_ASCIIMonoFontP;
   Alcon_SecondaryFont.vtable=font->vtable;
   Alcon_SecondaryFont.data=&Alcon_ASCIIMonoFontS;
   Alcon_CursorFont.vtable=font->vtable;
   Alcon_CursorFont.data=&Alcon_CursorMonoFont;

   AlCon_SetFont(0,font_buffer,fw,read_bytes/(0x100*wb));
}

/**[txh]********************************************************************

  Description: Fills in the width and height of a text cell.
  
***************************************************************************/

void AlCon_GetFontGeometry(unsigned int *w, unsigned int *h)
{
   ASSERT(w || h);
   if (w)
      *w = AlCon_FontWidth;
   if (h)
      *h = AlCon_FontHeight;
}

/**[txh]********************************************************************

  Description: Allocates for the chars and attrs global buffers
  enough memory as specified by the variables AlCon_ScreenWidth
  and AlCon_ScreenHeight. If these global variables already had
  something allocated to them (ie: you are resizing the screen),
  that memory is freed before the new allocation takes place.
  
***************************************************************************/

static void AlCon_AllocateMemoryBuffer(void)
{
   if (chars) {
      ASSERT(attrs);
      free(chars);
      free(attrs);
   }
   chars = (unsigned char *)malloc(AlCon_ScreenWidth * AlCon_ScreenHeight);
   attrs = (unsigned char *)malloc(AlCon_ScreenWidth * AlCon_ScreenHeight);
   ASSERT(chars && attrs);
}

#endif // HAVE_ALLEGRO
