/*
 *  gd.xml.tiny package: classes for parsing XML documents
 *  Copyright (C) 1999  Tom Gibara <tom@srac.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package gd.xml.tiny;

import java.util.*;

class ParseNode implements ParsedXML {

    /** Type constant used to indicate that a node is a root node. */
    final static int ROOT      =  0;
    /** Type constant used to indicate that a node was generated by a tag. */
    final static int TAG       =  1;
    /** Type constant used to indicate that a node was generated by an DOCTYPE definition. */
    final static int DOCTYPE =  2;
    /** Type constant used to indicate that a node was generated by character data. */
    final static int CHARDATA  =  4;
    /** Type constant used to indicate that a node was generated by a CDATA tag. */
    final static int CDSECTION =  8;
    /** Type constant used to indicate that a node was generated by a program instruction. */
    final static int PI        = 16;
    /** Type constant used to indicate that a node was generated by a comment tag. */
    final static int COMMENT   = 32;

    private final static Vector emptyvec = new Vector();
    private final static Hashtable emptyhash = new Hashtable();

    private int type;
    private String name = null;
    private Hashtable attributes = emptyhash;
    private Vector elements = emptyvec;
    private String content = null;


    ParseNode(int t) { type = t; }


    int getType() { return type; }


    public String getTypeName() {
	switch (type) {
	case ParseNode.ROOT      : return "root";
	case ParseNode.TAG       : return "tag";
	case ParseNode.DOCTYPE   : return "dtd";
	case ParseNode.CHARDATA  : return "text";
	case ParseNode.CDSECTION : return "cdata";
	case ParseNode.PI        : return "pi";
	case ParseNode.COMMENT   : return "comment";
	default: return "unknown";
	}
    }


    void setName(String n) { name = n; }
    void setContent(String c) { content = c; }


    public String getName() { return name; }
    public String getContent() { return content; }


    public boolean isEmpty() { return (elements.size()==0); }

    void addNode(ParseNode pnode) {
	if (elements==emptyvec) elements = new Vector();
	elements.addElement(pnode);
    }

    public Enumeration elements() {
	return elements.elements();
    }


    /** Used to obtain an enumeration of this node's children
     *  restricted to nodes of particuar types.  The children of a
     *  node correspond to data contained between start and end tag
     *  pairs.  The enumeration is always empty for nodes which were
     *  not generated from tag pairs.
     *  @return the child nodes */

     Enumeration elements(int typeMask) {
	Vector v = new Vector();
	ParseNode pn;
	Enumeration e = elements();
	while (e.hasMoreElements()) {
	    pn = (ParseNode)e.nextElement();
	    if ( (typeMask |  pn.getType()) != 0)
		v.addElement(pn);
	}
	return v.elements();
    }


    void setAttributes(Hashtable ht) {
	if (ht==null) attributes = emptyhash;
	else attributes = ht;
    }

    public Enumeration attributes() {
	return attributes.keys();
    }

    public String getAttribute(String a) {
	return (String)attributes.get(a);
    }

}


