// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief B+ cross section at 7 and 13 TeV
  class LHCB_2017_I1630633 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(LHCB_2017_I1630633);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      declare(UnstableParticles(), "UFS");
      for (double eVal : allowedEnergies()) {
        const int en = int(eVal+0.5);
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;

        int ih(en==13000);
        book(_h_B[en], {2.0,2.5,3.0,3.5,4.0,4.5});
        for (auto& b : _h_B[en]->bins()) {
          book(b, 1+ih, 1, b.index());
        }
        book(_h_pT[en], 3, 1, 1+ih);
        book(_h_y[en],  4, 1, 1+ih);
      }
      raiseBeamErrorIf(_sqs==0);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Final state of unstable particles to get particle spectra
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      for (const Particle& p : ufs.particles(Cuts::pid==521)) {
        double absrap = p.absrap();
        if (absrap<2. || absrap>4.5) continue;
        double pT = p.pT();
        if (pT>40.*GeV) continue;
        _h_B[_sqs]->fill(absrap, pT/GeV);
        _h_pT[_sqs]->fill(pT/GeV);
        _h_y[_sqs]->fill(absrap);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      // 1/2 due rapidity folding +/-
      const double factor = 0.5*crossSection()/nanobarn/sumOfWeights();
      scale(_h_B, factor);
      divByGroupWidth(_h_B);
      scale(_h_pT, factor);
      scale(_h_y,  factor/1000.);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<int,Histo1DGroupPtr> _h_B;
    map<int,Histo1DPtr> _h_pT, _h_y;
    int _sqs = 0;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(LHCB_2017_I1630633);

}
