/*  $Id: main.cpp,v 1.16 2024/04/08 03:43:59 sarrazip Exp $
    main.cpp - main() function

    batrachians - A fly-eating frog game.
    Copyright (C) 2004-2011 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#include "BatrachiansEngine.h"

#include <stdlib.h>
#include <errno.h>
#include <time.h>


#ifdef HAVE_GETOPT_LONG
#include <getopt.h>

static struct option knownOptions[] =
{
    { "help",          no_argument,       NULL, 'h' },
    { "version",       no_argument,       NULL, 'v' },
    { "ms-per-frame",  required_argument, NULL, 'm' },
    { "max-num-flies", required_argument, NULL, 'F' },
    { "no-sound",      no_argument,       NULL, 'n' },
    { "full-screen",   no_argument,       NULL, 'f' },
    { "no-auto-pause", no_argument,       NULL, 'p' },
    { "rendering",     required_argument, NULL, 'r' },

    { NULL, 0, NULL, 0 }  // marks the end
};


static int millisecondsPerFrame = 1000 / FPS;
static size_t maxNumFlies = 2;


static void
displayVersionNo()
{
    printf("%s %s\n", PACKAGE, VERSION);
}


static void
displayHelp()
{
    printf("\n");

    displayVersionNo();

    printf(
"\n"
"Copyright (C) 2004-2011 Pierre Sarrazin <http://sarrazip.com/>\n"
"This program is free software; you may redistribute it under the terms of\n"
"the GNU General Public License.  This program has absolutely no warranty.\n"
    );

    printf(
"\n"
"Known options:\n"
"--help             Display this help page and exit\n"
"--version          Display this program's version number and exit\n"
"--ms-per-frame=N   N milliseconds per animation frame [default=%d]\n"
"                   Min. 1, max. 1000.  50 ms means 20 frames per second\n"
"--max-num-flies=N  Maximum simultaneous number of flies (default=%u)\n"
"--no-sound         Disable sound effects [default is to enable them]\n"
"--full-screen      Attempt to use full screen mode [default is window mode]\n"
"--no-auto-pause    Do not pause automatically when window loses focus\n"
"--rendering=X      Specify the type of rendering.\n"
"                   Replace X with 'a' for accelerated, 's' for software.\n"
"\n",
        millisecondsPerFrame,
        (unsigned) maxNumFlies);
}
#endif


int main(int argc, char *argv[])
{
    /*  Default game parameters:
    */
    bool useSound = true;
    bool fullScreen = false;
    bool processActiveEvent = true;
    bool useAcceleratedRendering = true;

    #ifdef HAVE_GETOPT_LONG

    /*  Interpret the command-line options:
    */
    int c;
    do
    {
        c = getopt_long(argc, argv, "", knownOptions, NULL);

        switch (c)
        {
            case EOF:
                break;  // nothing to do

            case 'm':
                {
                    errno = 0;
                    long n = strtol(optarg, NULL, 10);
                    if (errno == ERANGE || n < 1 || n > 1000)
                    {
                        printf("Invalid number of ms per frame.\n");
                        displayHelp();
                        return EXIT_FAILURE;
                    }

                    millisecondsPerFrame = int(n);
                }
                break;

            case 'F':
                {
                    errno = 0;
                    long n = strtol(optarg, NULL, 10);
                    if (errno == ERANGE || n < 1 || n > 1000)
                    {
                        printf("Invalid maximum number of flies.\n");
                        displayHelp();
                        return EXIT_FAILURE;
                    }

                    maxNumFlies = size_t(n);
                }
                break;

            case 'n':
                useSound = false;
                break;

            case 'f':
                fullScreen = true;
                break;

            case 'p':
                processActiveEvent = false;
                break;

            case 'r':
                if (!strcmp(optarg, "a"))
                    useAcceleratedRendering = true;
                else if (!strcmp(optarg, "s"))
                    useAcceleratedRendering = false;
                else
                {
                    cout << "Invalid argument for --rendering.\n";
                    displayHelp();
                    return EXIT_FAILURE;
                }
                break;

            case 'v':
                displayVersionNo();
                return EXIT_SUCCESS;

            case 'h':
            default:
                displayHelp();
                return EXIT_SUCCESS;
        }
    } while (c != EOF && c != '?');

    #endif


    /*  Initialize the random number generator:
    */
    const char *s = getenv("SEED");
    unsigned int seed = (s != NULL ? atol(s) : time(NULL));
    #ifndef NDEBUG
    fprintf(stderr, "seed = %u\n", seed);
    fprintf(stderr, "ms-per-frame = %d\n", millisecondsPerFrame);
    #endif
    srand(seed);


    try
    {
        BatrachiansEngine theEngine("Batrachians", maxNumFlies,
                                    useSound, fullScreen,
                                    processActiveEvent, useAcceleratedRendering);
        theEngine.run(millisecondsPerFrame);
    }
    catch (const string &s)
    {
        fprintf(stderr, "%s: engine init error: %s\n", PACKAGE, s.c_str());
        return EXIT_FAILURE;
    }
    catch (int e)
    {
        fprintf(stderr, "%s: init error # %d\n", PACKAGE, e);
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
