//////////////////////////////////////////////////////////////////////////////
//
// 		  Copyright (C) 1996,1997  Matthew Doar  doar@pobox.com
// 
// Permission to use, copy, and distribute this software and its documentation 
// for any purpose with or without fee is hereby granted, provided that the 
// above copyright notice appears with all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// 
// Permission to modify the software is granted, but not the right to 
// distribute the modified code. Modifications are to be distributed as 
// patches to the released version. 
// 
// This software is provided "as is" without express or implied warranty. 
//
//////////////////////////////////////////////////////////////////////////////

// tiers_config.cc

#include <fstream.h>	// for ifstream
#include <string.h>	// for strcpy
#include <stdlib.h>	// for strtoul

#ifndef _TIERS_HH
#include "tiers.hh"
#endif


////////////////////////////////////////////////////////////////////////
// Model:ProcessConfig
//
// Process a configuration file to set up the configuration for tiers
////////////////////////////////////////////////////////////////////////
bool
Model::ProcessConfig()
{
  bool ret = true;

  ifstream config_file("./tiers_config");
  if (!config_file)
	{
	  cerr << "Tiers:: cannot open file './tiers_config'" << endl;
	}

  // Read the contents of the file in, skipping blank lines or lines which 
  // start with a #
  enum ParseState
  {
	CHAR,
	COMMENT,
	PARAM,
	DESC,
	VALUE
  };
  ParseState parse_state = CHAR;

  unsigned int param = 0;	// the parameter number under consideration
  char *value = new char[32];	// the value under consideration
  unsigned int valpos = 0;	// the position in the value array

  char ch;
  while (config_file.get(ch))
	{
	  switch (parse_state)
		{
		case CHAR:
		  {
			switch (ch)
			  {
			  case '\n':
				{
				  // Do nothing, read the next char
				  break;
				} 
			  case '#':
				{
				  parse_state = COMMENT;
				  break;
				}
			  case 'P':
				{
				  parse_state = PARAM;
				  param++;
				  break;
				}
			  default:
				{
				  // Do nothing, read the next char
				  break;
				}
			  }
			break;
		  }
		case COMMENT:
		  {
			// End a comment at a new line
			if (ch == '\n') 
			  parse_state = CHAR;
			break;
		  }
		case PARAM:
		  {
			// The description DESC follows the .
			if (ch == '.') 
			  parse_state = DESC;
			// Keep reading until a . is seen
			break;
		  }
		case DESC:
		  {
			// Values come after a space after a DESC
			if (ch == ' ') 
			  parse_state = VALUE;
			break;
		  }
		case VALUE:
		  {
			// until the end of the line or a ; is reached as a parameter 
			// separator. A comment symbol (#) ends the line
			switch (ch)
			  {
			  case '#': 
			  case '\n':
				{
				  // Terminate the string
				  value[valpos] = '\0';
				  valpos = 0;
				  if(!set_param(param, value))
					{
					  cerr << "Parameter " << param << " value failed to be set"
						   << endl;
					  ret = false;
					}
				  parse_state = CHAR;
				  break;
				}
			  case ';':
				{
				  // A parameter separator. Terminate the string
				  value[valpos] = '\0';
				  valpos = 0;
				  if(!set_param(param, value))
					{
					  cerr << "Parameter " << param << " value failed to be set"
						   << endl;
					  ret = false;
					}
				  // Stay in the same state
				  parse_state = VALUE;
				  param++;
				  break;
				}
			  default:
				{
				  // Add the chars of the value into the array
				  // Leave space for a termination char
				  if (valpos < 31)
					{
					  value[valpos] = ch;
					  valpos++;
					}
				  else
					{
					  cerr << "Parameter " << param << " value too long" << endl;
					  ret = false;
					  parse_state = COMMENT;
					}
				  break;
				}
			  }
			break;
		  }
		default:
		  {
			cerr << "Unknown state whilst parsing the configuration file" << endl;
			ret = false;
			break;
		  }
		}
	} 

  delete [] value;
  config_file.close();

  return ret;

}



////////////////////////////////////////////////////////////////////////
// Model::set_param
//
// Set the parameter number Param to the value held in Value
////////////////////////////////////////////////////////////////////////
bool Model::set_param(unsigned int Param, char *value)
{
  bool ret = true;

  switch (Param)
	{
	case 1:
	  {
		// Type of output produced by the model, e.g. Model::GENERIC, Model::GNUPLOT
		// OP_TYPE = Model::GENERIC;
		if (value[1] == 'n')
		  OP_TYPE = Model::GNUPLOT;
		else
		  OP_TYPE = Model::GENERIC;
		break;
	  }
	case 2:
	  {
		// If defined, TIERS_VERBOSE causes the output to contain the
		// parameters used to generate the model and helpful headings
		// If undefined, the output is a plain list of nodes and edges
		if (value[0] == 'f')
		  TIERS_VERBOSE = false;
		else
		  TIERS_VERBOSE = true;
		break;
	  }
	case 3:
	  {
		// If set, TIERS_LABEL causes the output to contain a label 
		// (the unique node number)for each of the nodes
		if (value[0] == 'f')
		  TIERS_LABEL_WAN = false;
		else
		  TIERS_LABEL_WAN = true;
		break;
	  }
	case 4:
	  {
		// If set, TIERS_LABEL causes the output to contain a label 
		// (the unique node number)for each of the nodes
		if (value[0] == 'f')
		  TIERS_LABEL_MAN = false;
		else
		  TIERS_LABEL_MAN = true;
		break;
	  }
	case 5:
	  {
		// If set, TIERS_LABEL causes the output to contain a label 
		// (the unique node number)for each of the nodes
		if (value[0] == 'f')
		  TIERS_LABEL_LAN = false;
		else
		  TIERS_LABEL_LAN = true;
		break;
	  }
	case 6:
	  {
		// If defined, the operation of the model is more verbose
		// If undefined, the model (should) run with no debugging output
		TIERS_DEBUG = true; // UNUSED
		break;
	  }
	case 7:
	  {
		// If defined as true, then an undirected list of edges is produced in the 
		// output, i.e. the edges (i,j) and (j,i) only appear once in the output
		// Note that all LAN edges are assumed bidirectional
		if (value[0] == 'f')
		  REMOVE_DUP_EDGES = false;
		else
		  REMOVE_DUP_EDGES = true;
		break;
	  }
	case 8:
	  {
		// If this is true, apply proximity test to nodes in the network type
		if (value[0] == 'f')
		  PROXIMITY_TEST_WAN = false;
		else
		  PROXIMITY_TEST_WAN = true;
		break;
	  }
	case 9:
	  {
		// If this is true, apply proximity test to nodes in the network type
		if (value[0] == 'f')
		  PROXIMITY_TEST_MAN = false;
		else
		  PROXIMITY_TEST_MAN = true;
		break;
	  }
	case 10:
	  {
		// If this is true, apply proximity test to nodes in the network type
		if (value[0] == 'f')
		  PROXIMITY_TEST_LAN = false;
		else
		  PROXIMITY_TEST_LAN = true;
		break;
	  }
	case 11:
	  {
		// The size of a unit in WAN, MAN and LAN grids
		// e.g. 1000km, 1km, 1m would be best, but the current values fit with 
		// the size of integers and are good for viewing graphs
		WAN_SCALE = strtoul(value, NULL, 0);
		break;
	  }
	case 12:
	  {
		// The size of a unit in WAN, MAN and LAN grids
		// e.g. 1000km, 1km, 1m would be best, but the current values fit with 
		// the size of integers and are good for viewing graphs
		MAN_SCALE = strtoul(value, NULL, 0);
		break;
	  }
	case 13:
	  {
		// The size of a unit in WAN, MAN and LAN grids
		// e.g. 1000km, 1km, 1m would be best, but the current values fit with 
		// the size of integers and are good for viewing graphs
		LAN_SCALE = strtoul(value, NULL, 0);
		break;
	  }
	case 14:
	  {
		// GRID is the size of the square grid on which points are placed
		GRID = strtoul(value, NULL, 0);
		break;
	  }
	case 15:
	  {
		// The comment character for output 
		// TODO not implemented yet
		COM = '#';
		break;
	  }
	case 16:
	  {
		// The title for results from gnuplot
		title = new char[64];
		strcpy(title,value);
		break;
	  }
	case 17:
	  {
		// The output directory for results from gnuplot
		outputdir = new char[64];
		strcpy(outputdir,value);
		break;
	  }
	case 18:
	  {
		// The unique identifier for a particular network, default is the seed used 
		// to generate it
		//if (value[0] != 'M')
		//  ModelId = strtoul(value, NULL, 0);
		strtoul(value, NULL, 0);
		break;
	  }
	default:
	  {
		if (Param > 18 + (3*9) || Param < 19)
		  {
			cerr << "Unknown parameter number or missing case for parameter" 
				 << endl;
			ret = false;
		  }
		else
		  {
			// Process the cost table entries
			// 19 is (LAN,LAN).bw, 20 is (LAN,LAN).d1 etc
			unsigned int subfield = (Param-1)  % 3;		// 19 is bw
			unsigned int index = (Param-19)  / 3;

			switch (subfield)
			  {
			  case 0:
				{
				  theCostTable[index].bw = strtoul(value, NULL, 0);
				  break;
				}
			  case 1:
				{
				  theCostTable[index].d1 = strtoul(value, NULL, 0);
				  break;
				}
			  case 2:
				{
				  theCostTable[index].d2 = strtoul(value, NULL, 0);
				  break;
				}
			  default :
				{
				  cerr << "tiers_config:: x %3 != 0, 1 or 2." << endl;
				  ret = false;
				}
			  }
		  }
		break;
	  }
	}

  // Clear value
  for (unsigned int i = 0; i < 32 ; i++) value[i] = 0;

  return ret;
}


// end of file


