//////////////////////////////////////////////////////////////////////////////
//
// 		  Copyright (C) 1996,1997  Matthew Doar  doar@pobox.com
// 
// Permission to use, copy, and distribute this software and its documentation 
// for any purpose with or without fee is hereby granted, provided that the 
// above copyright notice appears with all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// 
// Permission to modify the software is granted, but not the right to 
// distribute the modified code. Modifications are to be distributed as 
// patches to the released version. 
// 
// This software is provided "as is" without express or implied warranty. 
//
//////////////////////////////////////////////////////////////////////////////

// tiers.hh

#ifndef _TIERS_HH
#define _TIERS_HH


////////////////////////////////////////////////////////////////////////
// Model
//
// The major class for the network model generation code
////////////////////////////////////////////////////////////////////////
class Node;
class Edge;
class NodesAndEdges;
class CostTableEntry;
class Model
{
public:
  Model(void);
  ~Model(void);

  bool ProcessConfig();
  bool CreateCostTable(CostTableEntry *aCostTable = 0);
  bool CreateModel(const unsigned long int aNW,
				   const unsigned long int aNM,
				   const unsigned long int aNL,
				   const unsigned long int aSW,
				   const unsigned long int aSM,
				   const unsigned long int aSL,
				   const unsigned long int aRW = 1,
				   const unsigned long int aRM = 1,
				   const unsigned long int aRL = 1,
				   const unsigned long int aRMW = 1,
				   const unsigned long int aRLM = 1);

  enum OpType
  {
	GENERIC,
	GNUPLOT
  };
  
  bool OutputModel(OpType OP_TYPE);

  enum NodeType
  {
	WAN,
	MAN,
	LAN
  };

  // Output code depends upon the two internetwork types being last
  enum EdgeState
  {
	INACTIVE,		// unused
	ACTIVE,	    	// part of the MST
	REDUNDANT,  	// a redundant part of the MST
	INTERNETWORK,	// an edge between networks of different types
	RED_INTERNETWORK	// a redundant edge between networks of different types
  };

private:
  // Functions
  bool CreateWanNodes();
  bool CreateManNodes();
  bool CreateLanNodes();
 
  bool CreateWanEdges();
  bool CreateManEdges();
  bool CreateLanEdges();

  bool CreateWanRedundancy();
  bool CreateManRedundancy();
  bool CreateLanRedundancy();

  bool ConnectMans();
  bool ConnectLans();

  bool OutputParams();
  bool OutputNodes();
  bool OutputEdges();

  bool OutputGnuplotHeader();
  bool CreateGlobalNodeList(Node* &nodes);
  bool OutputEdgesGnuplot();
  bool OutputGnuplotTrailer();

  // Calculate the Minimum Spanning Tree of a network and mark it active 
  // in the edge list of the network
  bool Prims(NodesAndEdges *aNetwork, unsigned long int R);

  // Distribution functions
  const unsigned long int SWdist(const unsigned long int aSW);
  const unsigned long int NMdist(const unsigned long int aNM);
  const unsigned long int SMdist(const unsigned long int aSM);
  const unsigned long int NLdist(const unsigned long int aNL);
  const unsigned long int SLdist(const unsigned long int aSL);
  const unsigned long int RWdist(const unsigned long int aRW);
  const unsigned long int RMdist(const unsigned long int aRM);
  const unsigned long int RLdist(const unsigned long int aRL);
  const unsigned long int RMWdist(const unsigned long int aRMW);
  const unsigned long int RLMdist(const unsigned long int aRLM);

  // Helper functions
  const unsigned long int Euclid(Node &A, Node &B);
  bool place(unsigned long int aNode, Node *nodes, unsigned long int NumNodes);
  bool proximity_test(unsigned long int aNode, Node *nodes, 
					  unsigned long int NumNodes);
  bool AddInternetworkEdge(NodesAndEdges *subnetwork,
						   NodesAndEdges *supernetwork,
						   unsigned long int gateway,
						   unsigned long int wangateway,	
						   unsigned long int NumManNodes,	
						   unsigned long int network,
						   EdgeState aState = INTERNETWORK);
  bool CreateRedundancy(NodesAndEdges *NetworkList, int network_type);
  bool RepositionNetwork(NodesAndEdges *Network,
						 Node *Base,
						 NodeType type);
  bool set_param(unsigned int Param, char *Value);

  // Variables
  unsigned long int NW;
  unsigned long int NM;
  unsigned long int NL;
  unsigned long int SW;
  unsigned long int SM;
  unsigned long int SL;
  unsigned long int RW;
  unsigned long int RM;
  unsigned long int RL;
  unsigned long int RMW;
  unsigned long int RLM;

  NodesAndEdges	*WanList;
  NodesAndEdges	*ManList;
  NodesAndEdges	*LanList;

  CostTableEntry	*theCostTable;

  unsigned long int	CurrNodeNum;	// for tracking node numbers
};



////////////////////////////////////////////////////////////////////////
// NodesAndEdges
//
// The representation of a single network such as a WAN, MAN or LAN
// Contains a list of nodes in the network and a list of edges in the network
////////////////////////////////////////////////////////////////////////
class Node;
class Edge;
class NodesAndEdges
{
public:
  NodesAndEdges();
  ~NodesAndEdges();

  // Public variables
  Node	*nodes;		// The list of nodes for this network
  Edge	*edges;		// The connectivity matrix for this network
  Edge	*newedges;	// The connectivity matrix for this network, sorted by cost

  // Access functions
  unsigned long int NumNodes();
  void NumNodes(const unsigned long int N);

  unsigned long int MaxNumEdges();
  void MaxNumEdges(const unsigned long int E);
  unsigned long int NumEdges(const unsigned long int index);
  void NumEdges(const unsigned long int index,
				const unsigned long int E);
  void IncNumEdges(const unsigned long int index);  

  unsigned long int NumNetworks();
  void NumNetworks(const unsigned long int N);

  // Helper functions
  bool newEdgeList(unsigned long int size);
  bool test_dup_edges(const unsigned long int to,
					  const unsigned long int from);
  void dump_edges(const unsigned int ArrayType = 1); 
  void dump_lanedges(); 

private:
  unsigned long int m_NumNodes;		// number of nodes in this network
  unsigned long int m_MaxNumEdges;	// number of edges from this node
  unsigned long int *m_NumEdges;	// number of edges from this node

  unsigned long int m_NumNetworks;	// number of networks attached to this 
                                    // network
};



////////////////////////////////////////////////////////////////////////
// Node
//
// Represents a node in a network
// Every node has a number which is unique in all the networks
// The numbers are assigned as the nodes are created, with 
// 	WAN < MAN < LAN
////////////////////////////////////////////////////////////////////////
struct Node
{
  unsigned long int	number;	// the node number, unique in the network
  unsigned long int 	x;		// x coordinate
  unsigned long int 	y;		// y coordinate
  Model::NodeType		type;	// Node type - WAN, MAN or LAN
};



////////////////////////////////////////////////////////////////////////
// Edge
//
// Represents an edge in a network
////////////////////////////////////////////////////////////////////////
struct Edge
{
  unsigned long int start;	// unique node number
  unsigned long int end;	// unique node number
  unsigned long int delay;	// delay - a typical cost to be minimised
  unsigned long int bw;		// bandwidth - a typical constraint
  Model::EdgeState state;	// state of an edge: inactive, active, 
                            // redundant, internetwork, redundant internetwork
};



////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
struct CostTableEntry
{
  unsigned long int bw;	// bandwidth
  unsigned long int d1; 	// fixed delay
  unsigned long int d2; 	// delay per unit distance
};


// 
// Global Variable Declarations
//
// Defined in tiers_config.cc

// Type of output produced by the model, e.g. Model::GENERIC, Model::GNUPLOT
Model::OpType OP_TYPE;

// If set, TIERS_VERBOSE causes the output to contain the
// parameters used to generate the model and helpful headings
// If undefined, the output is a simple list of nodes and edges
bool TIERS_VERBOSE;

// If set, TIERS_LABEL_XXX causes the output to contain a label for
// each of the nodes of type XXX
bool TIERS_LABEL_WAN;
bool TIERS_LABEL_MAN;
bool TIERS_LABEL_LAN;


// If defined as true, the operation of the model is more verbose
// If undefined, the model should run with no debugging output
bool TIERS_DEBUG;

// If defined as true, then an undirected list of edges is produced in the 
// output, i.e. the edges (i,j) and (j,i) only appear once in the output
// Note that all LAN edges are assumed bidirectional
bool REMOVE_DUP_EDGES;

// Set these true to apply proxmity tests to nodes in each network type
bool PROXIMITY_TEST_WAN;
bool PROXIMITY_TEST_MAN;
bool PROXIMITY_TEST_LAN;

// The size of a unit in WAN, MAN and LAN grids
// e.g. 1000km, 1km, 1m would be best, but the current values fit with 
// typical int sizes
unsigned long int WAN_SCALE;
unsigned long int MAN_SCALE;
unsigned long int LAN_SCALE;	

// GRID is the size of the square grid on which points are placed
unsigned long int GRID;

// The comment character for output
char COM;


//
// GNUPLOT variables
//

// The output directory for results from gnuplot
char *title;

// The title for results from gnuplot
char *outputdir;

// Create a unique identifier for each model
long int ModelId;

// Forward declaration
unsigned long int max(unsigned long int A, unsigned long int B);

#endif // _TIERS_HH

// end of file


