
/*

    dockhelper.c - Dock applets helper functions
    Copyright (C) 2002-2004 Mario Pascucci <ilpettegolo (a) yahoo it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    or
    visit the URL http://www.fsf.org/

*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include "dockhelper.h"



static struct dh_areas a[DH_MAXAREAS];    /* areas in the applet */
static int nareas = 0;	      /* number of areas defined and used */
static struct dh_color colors[DH_MAXCOLORS];  /* colors used for charts */
static int ncolors = 0;

static Display *display;	    /* X display struct */
static int screen;		    /* screen */
static Window root, win, iconwin;   /* root, main and icon window */
static Pixel fg, bg;		    /* foreground and background colors */
static GC maingc;		    /* window GC */
static Pixmap mask;		    /* shape mask */
static Pixmap backs;		    /* drawing pixmap */
static int sizex = 64, sizey = 64, offs = 2;  /* size and border */
static char *bgcolor;		    /* background color */
static char bevel_h[20], bevel_l[20];	/* bevel color */



/* create pixmaps for icons */
void dh_pixmap(Pixmap *p, Pixmap *m, char *xpm[]) {

  if (XpmCreatePixmapFromData(display, iconwin, xpm, p, m, NULL) != XpmSuccess) {
    fprintf(stderr, "FATAL: Not enough colors for main pixmap!\n");
    exit(1);
  }
}



/* copy pixmap for icons */
void dh_copyarea(Pixmap src, int xs, int ys, int xd, int yd, int w, int h) {

  XCopyArea (display, src, backs, maingc, xs, ys, w, h, xd, yd);
}



/* return display */
Display *dh_display(void) {

  return display;
}



/* return background color */
char *dh_background(void) {

  return bgcolor;
}



/* draw line using predefined gc */
void dh_line(int x1, int y1, int x2, int y2) {

  XDrawLine (display, backs, maingc, x1, y1, x2, y2);
}



/* create utility pixmap for scrolling area */
void dh_newpixmap(Pixmap *p, int w, int h) {

  *p = XCreatePixmap(display, backs, w, h, DefaultDepth(display,screen));
  if (! *p) {
    fprintf(stderr, "FATAL: Cannot create scroll pixmap!\n");
    exit(1);
  }
}



/* scroll pixmap of 1 pixel to right */
void dh_scroll(Pixmap p, int xs, int ys, int w, int h) {

  XCopyArea(display,backs,p,maingc,xs,ys,w,h,0,0);
  XCopyArea(display,p,backs,maingc,0,0,w,h,xs+1,ys);
}


      

/* draw hollow rectangle */
void dh_rectangle(int x, int y, int w, int h) {

  XDrawRectangle (display, backs, maingc, x, y, w, h);
}



/* draw filled rectangle */
void dh_fillrect(int x, int y, int w, int h) {

  XFillRectangle (display, backs, maingc, x, y, w, h);
}



/* set current color */
void dh_color(char *color) {

  XSetForeground (display, maingc, dh_getcolor(color));
}


      
/* prepare main window and shape mask */
int dh_mainwin(char *name, char *bground) {

  int i;
  XSizeHints sizehints;	    /* size hints */
  XGCValues gcval,gcv;      /* setting values for GCs */
  XClassHint classhint;     /* X class hints */
  XWMHints wmhints;	    /* wm class hints */
  GC mask_gc;		    /* shape mask gc */
  XColor l,b;		    /* some colors */

  if (name == NULL) return -1;
  if (strlen(name) == 0) return -1;
  if (bground == NULL) return -1;
  if (strlen(bground) == 0) return -1;
  display = XOpenDisplay(NULL);	  /* open X default display */
  if (!display) {
    fprintf(stderr, "Unable to open display\n");
    return -1;
  }
  
  bgcolor = bground;
  screen = DefaultScreen(display);    /* gets screen handle */
  root = DefaultRootWindow(display);  /* gets root window */

  sizehints.flags = USSize;	/* preferred window size */
  sizehints.width = sizex;
  sizehints.height = sizey;

  /* gets main colors */
  bg = BlackPixel(display, screen);
  fg = WhitePixel(display, screen);

  /* creates main and icon win */
  win = XCreateSimpleWindow(display, root, 0, 0, sizehints.width, sizehints.height, offs, fg, bg);
  iconwin = XCreateSimpleWindow(display, win, 0, 0, sizehints.width, sizehints.height, offs, fg, bg);

  /* setup name and class properties */
  XSetWMNormalHints(display, win, &sizehints);
  classhint.res_name = name;
  classhint.res_class = name;
  XSetClassHint(display, win, &classhint);

  /* events passed to win */
  XSelectInput(display, win, ExposureMask | StructureNotifyMask);
  XSelectInput(display, iconwin, ExposureMask | StructureNotifyMask);

  /* store names */
  XStoreName(display, win, name);
  XSetIconName(display, win, name);

  /* setup graphic contest */
  gcval.foreground = fg;
  gcval.background = bg;
  gcval.graphics_exposures = False;
  maingc = XCreateGC(display, iconwin, GCForeground | GCBackground | GCGraphicsExposures, &gcval);

  /* setup shape mask */
  mask = XCreatePixmap(display, iconwin, sizex, sizey, 1);
  gcv.foreground = 0;
  gcv.background = 1;
  gcv.graphics_exposures = 0;
  mask_gc = XCreateGC(display, mask, GCForeground | GCBackground | GCGraphicsExposures, &gcv);

  /* fill mask shape */
  XFillRectangle(display, mask, mask_gc, 0, 0, sizex, sizey);    

  /* prepare area masks */
  XSetForeground(display, mask_gc, 1);
  XSetBackground(display, mask_gc, 0);

  /* draw masks */
  for (i=0;i<nareas;i++) {
    XFillRectangle(display, mask, mask_gc, a[i].x, a[i].y, a[i].w, a[i].h);
  }

  XFreeGC(display, mask_gc);

  XShapeCombineMask(display, win, ShapeBounding, 0, 0, mask, ShapeSet);
  XShapeCombineMask(display, iconwin, ShapeBounding, 0, 0, mask, ShapeSet);

  /* colors for bevel */
  /* get rgb value for background */
  XLookupColor(display, DefaultColormap(display, screen), 
      bgcolor, &b, &l);
  /* computes color for 'up' bevel */
  /*fprintf(stderr,"rgb:%04x/%04x/%04x\n", b.red, b.green, b.blue); */
  /* color autoranging: 
   * - divide bg color by 2 for shadow side of bevel
   * - multiply by 8 for light side of bevel
   * - if bg color is too light, reduce multiplier to 4 and clip higher value */
  i = 8;
  if (b.red >= 0x2000U)
  {
    i = 4;
    b.red = 0x3fff;
  }
  if (b.green >= 0x2000U)
  {
    i = 4;
    b.green = 0x3fff;
  }
  if (b.blue >= 0x2000U)
  {
    i = 4;
    b.blue = 0x3fff;
  }
  snprintf(bevel_h,19,"rgb:%04x/%04x/%04x", (unsigned short) (b.red * i),
      (unsigned short) (b.green * i), (unsigned short) (b.blue * i));
  snprintf(bevel_l,19,"rgb:%04x/%04x/%04x", (unsigned short) b.red >> 1,
      (unsigned short) b.green >> 1, (unsigned short) b.blue >> 1);
  /* fprintf(stderr,"low bevel:%s\nhigh bevel:%s\n", bevel_l, bevel_h); */
  /* initial window status */
  wmhints.initial_state = WithdrawnState;
  wmhints.icon_window = iconwin;
  wmhints.icon_x = sizehints.x;
  wmhints.icon_y = sizehints.y;
  wmhints.window_group = win;
  wmhints.flags = StateHint | IconWindowHint | IconPositionHint | WindowGroupHint;
  XSetWMHints(display, win, &wmhints);

  /* map windows for display */
  XMapWindow(display, iconwin);
  XMapWindow(display, win);
  backs = XCreatePixmap(display, iconwin, sizex, sizey, DefaultDepth(display, screen));
  return 0;
}



/* draw bevel for individual area */
void dh_bevel(void) {

  int i;
  
  for (i = 0; i < nareas; i++) {
    if (!a[i].bevel) continue;	  /* no bevel */
    XSetForeground (display, maingc, dh_getcolor(bgcolor));
    XFillRectangle (display, backs, maingc, a[i].x, a[i].y,
	a[i].x + a[i].w - 1, a[i].y + a[i].h - 1);
    XSetForeground (display, maingc, dh_getcolor(bevel_h));
    XDrawLine (display, backs, maingc, a[i].x + a[i].w - 1 , a[i].y, 
        a[i].x + a[i].w - 1, a[i].y + a[i].h - 1);
    XDrawLine (display, backs, maingc, a[i].x, a[i].y + a[i].h - 1, 
        a[i].x + a[i].w - 1, a[i].y + a[i].h - 1);
    XSetForeground (display, maingc, dh_getcolor(bevel_l));
    XDrawLine (display, backs, maingc, a[i].x, a[i].y, 
        a[i].x + a[i].w - 1, a[i].y);
    XDrawLine (display, backs, maingc, a[i].x, a[i].y, 
        a[i].x, a[i].y + a[i].h - 1);
  }
}




/* updates applet area on screen */
void dh_update(void) {
  
  XCopyArea (display, backs, iconwin, maingc, 0, 0, sizex, sizey, 0, 0);
/*  XFlush(display);       usare solo nei test */
}
	



/* setup applet size */
int dh_setsize(int width, int height, int border) {

  if ((width < 10) || (height < 8) || 
      (width > DH_MAXSIZE) || (height > DH_MAXSIZE)) {
    fprintf(stderr,"Size exceeds limits, or too small\n");
    return -1;
  }
  if ((border < 1) || (border > DH_MAXBORDER)) {
    fprintf(stderr,"Border size exceeds limits\n");
    return -1;
  }
  sizex = width;
  sizey = height;
  offs = border;
  return 0;
}




/* find an area from coord */
int dh_getarea(int x, int y) {

  int i;
  
  for (i=0; i<nareas; i++)
  if ((x >= a[i].x) && (x <= a[i].x + a[i].w) &&
      (y >= a[i].y) && (y <= a[i].y + a[i].h))
    return i;
  return -1;
}



/* returns usable sizes of an area */
void dh_getsize(int handler, int *x, int *y, int *w, int *h) {

  if (handler < 1) {
    fprintf(stderr, "Invalid area handler.");
    return;
  }
  handler--;
  if (a[handler].bevel) {
    *x = a[handler].x + 1;
    *y = a[handler].y + 1;
    *w = a[handler].w - 2;
    *h = a[handler].h - 2;
  }
  else {
    *x = a[handler].x;
    *y = a[handler].y;
    *w = a[handler].w;
    *h = a[handler].h;
  }
  return;
}



/* define an area, return an handler (int) if ok */
int dh_addarea(int x, int y, int w, int h, int bevel, void (*handler)()) {
  
  int area;
  /* if already defined */
  if (((area = dh_getarea(x,y)) != -1) || 
      ((area = dh_getarea(x+w-1,y+h-1)) != -1)) {
    fprintf(stderr,"Area collision\n");;
    return -1;
  }
  /* too many areas */
  if (nareas >= DH_MAXAREAS) { 
    fprintf(stderr,"Too many areas defined\n"); 
    return -1; 
  }
  a[nareas].enabled = TRUE;
  a[nareas].bevel = bevel;
  a[nareas].x = x;
  a[nareas].y = y;
  a[nareas].w = w;
  a[nareas].h = h;
  a[nareas++].handler = handler;
  return nareas;
}


/* get and eventually allocate X color in colormap */
/* select colors by name */
long dh_getcolor(char *name) {

  int i;
  XColor  screencolor, exactcolor;
  
  for (i=0; i<ncolors; i++) 
    if (!strcasecmp(colors[i].name, name))
      return colors[i].pixel;

  if (!XAllocNamedColor(display, DefaultColormap(display, screen),
      name, &screencolor, &exactcolor) || (ncolors >= DH_MAXCOLORS)) {
    fprintf(stderr,"Cannot allocate X color\n");
    /* in case of colormap space exhausted, select last valid color */
    return colors[ncolors-1].pixel;
  }
  colors[ncolors].name = malloc(strlen(name) + 1);
  if (colors[ncolors].name == NULL) {
    fprintf(stderr,"Out of memory!\n");
    exit(1);
  }
  strcpy(colors[ncolors].name, name);
  colors[ncolors++].pixel = exactcolor.pixel;    
  return exactcolor.pixel;
}

