/* GKRELLM Vaio Brightness Control
 *
 * (c) Mike Kershaw <dragorn@nerv-un.net>
 *
 * This code is freely distributable under the terms of the GPL
 *
 * Controls the software-brightness of Sony Vaio laptops via gkrellm
 *
 * VERY HEAVILY borrowed and based on the GKRellM Demo plugins and the
 * Volume control plugin written by Sjoerd Simons
 *
 */

#include "vaiobright.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <getopt.h>
#include <errno.h>
#include <limits.h>
#include <linux/types.h>

// Define our ioctl's (stolen from spicctrl)
#define SONYPI_IOCGBRT _IOR('v', 0, __u8)
#define SONYPI_IOCSBRT _IOW('v', 0, __u8)

static void create_vaiobright_plug_config(GtkWidget *tab);
static void load_vaiobright_plug_config(gchar *arg);
static void save_vaiobright_plug_config(FILE *f);
static void apply_vaiobright_plug_config(void);
static void create_vaiobright_plugin(GtkWidget *vbox, gint first_create);
static void update_vaiobright_slider(void);

static int vaiobright_open_device(void);
static void vaiobright_close_device(void);
static void vaiobright_set_brightness(int level);
static int vaiobright_get_brightness(void);
static void vaiobright_device_entry_changed(GtkEntry *entry, gpointer *data);
static void vaiobright_toggle_toggled(GtkWidget *button, gint data);
static void vaiobright_create_help_text(GtkWidget *text);
static void vaiobright_do_toggle(void);
static gint vaiobright_plug_expose_event(GtkWidget *widget, GdkEventExpose *event);
static void vaiobright_slider_motion(GtkWidget *widget, GdkEventMotion *ev, gpointer data);
static void vaiobright_panel_button_release(GtkWidget *widget, GdkEventButton *ev, BrightSlider *slide);
static void vaiobright_panel_button_press(GtkWidget *widget, GdkEventButton *ev, BrightSlider *slide);
static void vaiobright_create_slider(int first_create);

static gint vaiobright_style_id;
static BrightSlider *vaiobright_bslider;

static GkrellmMonitor vaiobright_mon = {
    "Vaio Brightness",
    0,
    create_vaiobright_plugin, // create func*
    update_vaiobright_slider, // update func* (We don't want to do this.)
    create_vaiobright_plug_config, // config create
    apply_vaiobright_plug_config, // config apply
    save_vaiobright_plug_config, // config save
    load_vaiobright_plug_config, // config load
    CONFIG_KEYWORD, // config keyword
    NULL, // undef
    NULL, // undef2
    NULL, // undef3
    LOCATION, // Where
    NULL, // Handle, filled in by gkrellm
    NULL, // path, also filled in by gkrellm
};

static int vaiobright_open_device(void) {
    if ((vaiobright_bslider->sony_fd = open(vaiobright_bslider->device_path, O_RDWR)) == -1) {
        fprintf(stderr, "vaiobright: failed to open %s: %s\n",
                vaiobright_bslider->device_path, strerror(errno));
        vaiobright_bslider->broken = 1;
        return -1;
    }

    vaiobright_bslider->broken = 0;
    return 1;
}

static void vaiobright_close_device(void) {
    if (vaiobright_bslider->broken == 0)
        close(vaiobright_bslider->sony_fd);
}

// Write a new brightness level out to the control application
static void vaiobright_set_brightness(int level) {
    __u8 value = (__u8) level;

    // If we're broken, try again to open it
    if (vaiobright_bslider->broken == 1) {
        if (vaiobright_open_device() == -1) {
            return;
        }
    }

    if (ioctl(vaiobright_bslider->sony_fd, SONYPI_IOCSBRT, &value) < 0) {
        fprintf(stderr, "vaiobright set: ioctl failed: %s\n", strerror(errno));
        vaiobright_bslider->broken = 1;
        return;
    }
}

static int vaiobright_get_brightness() {
    __u8 value = 0;

    // If we're broken, try again to open it
    if (vaiobright_bslider->broken == 1) {
        if (vaiobright_open_device() == -1)
            return;
    }

    if (ioctl(vaiobright_bslider->sony_fd, SONYPI_IOCGBRT, &value) < 0) {
        fprintf(stderr, "vaiobright get: ioctl failed: %s\n", strerror(errno));
        vaiobright_bslider->broken = 1;
        return -1;
    }

    return value;
}

// Get the slider data from the register
static void update_vaiobright_slider(void) {
    static unsigned int count = 0;
    int x = vaiobright_bslider->setting;

    // Only fetch the brightness once every 5k updates.  This keeps us sane.
    if (count++ > 5000) {
        if (!vaiobright_bslider->broken) {
            x = vaiobright_get_brightness();
            vaiobright_bslider->setting = vaiobright_bslider->reversed ? MAX_VAL - x : x;
        }

        count = 0;
    }

    vaiobright_bslider->krell->previous = 0;
    gkrellm_update_krell(vaiobright_bslider->panel, vaiobright_bslider->krell, x);
    gkrellm_draw_panel_layers(vaiobright_bslider->panel);

}

// When someone middle-clicks, we flip the brightness.  If they're lightish,
// go full dark, if they're darkish, go full light
static void vaiobright_do_toggle(void) {
    // If we're normal and on the bright half, go to full dark, otherwise go full
    // bright.  Close the interface when we do this.
    if (vaiobright_bslider->setting < (MAX_VAL / 2)) {
        vaiobright_bslider->setting = MAX_VAL;
    } else {
        vaiobright_bslider->setting = MIN_VAL;
    }

    vaiobright_set_brightness(vaiobright_bslider->reversed ? MAX_VAL - vaiobright_bslider->setting : vaiobright_bslider->setting);
    vaiobright_bslider->krell->previous = 0;
    gkrellm_update_krell(vaiobright_bslider->panel, vaiobright_bslider->krell, vaiobright_bslider->setting);
    gkrellm_draw_panel_layers(vaiobright_bslider->panel);
}

// Handle drawing event
static gint vaiobright_plug_expose_event(GtkWidget *widget, GdkEventExpose *event) {
    if (widget == vaiobright_bslider->panel->drawing_area) {
        gdk_draw_pixmap(widget->window,
                        widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                        vaiobright_bslider->panel->pixmap,
                        event->area.x, event->area.y,
                        event->area.x, event->area.y,
                        event->area.width, event->area.height);
    }
    return FALSE;
}

// Handle the slider moving
static void vaiobright_slider_motion(GtkWidget *widget, GdkEventMotion *ev, gpointer data) {

    gint x;
    GdkModifierType state;

    if (vaiobright_bslider->broken)
        return;

    if (!vaiobright_bslider->slider_in_motion)
        return;

    // If the slider has been released, set the brightness to the
    // slider value
    state = ev->state;

    if (!(state & GDK_BUTTON1_MASK)) {
        vaiobright_bslider->slider_in_motion = 0;
        vaiobright_set_brightness(vaiobright_bslider->reversed ? MAX_VAL - vaiobright_bslider->setting : vaiobright_bslider->setting);
        return;
    }

    x = ev->x * vaiobright_bslider->krell->full_scale / (gkrellm_chart_width() - 1);

    if (x < MIN_VAL)
        x = 0;
    else if (x > MAX_VAL)
        x = MAX_VAL;

    vaiobright_bslider->krell->previous = 0;

    gkrellm_update_krell(vaiobright_bslider->panel, vaiobright_bslider->krell, (gulong) x);
    gkrellm_draw_panel_layers(vaiobright_bslider->panel);
    vaiobright_bslider->setting = x;
}

static void vaiobright_panel_button_release(GtkWidget *widget, GdkEventButton *ev, BrightSlider *slide) {
    gint x;

    if (ev->state & GDK_BUTTON1_MASK) {
        vaiobright_bslider->slider_in_motion = 0;
        vaiobright_set_brightness(vaiobright_bslider->reversed ? MAX_VAL - vaiobright_bslider->setting : vaiobright_bslider->setting);
    } else if (ev->state & GDK_BUTTON2_MASK)
        vaiobright_do_toggle();
    else if (vaiobright_bslider->broken)
        return;
    else if (ev->state & GDK_BUTTON4_MASK || ev->state & GDK_BUTTON5_MASK) {
        x = slide->setting;
        x = ev->state & GDK_BUTTON4_MASK ? x + WHEEL_ADJUST : x - WHEEL_ADJUST;

        if (x < MIN_VAL)
            x = MIN_VAL;
        if (x > MAX_VAL)
            x = MAX_VAL;

        vaiobright_set_brightness(slide->reversed ? MAX_VAL - x : x);

        slide->setting = x;
        slide->krell->previous = 0;

        gkrellm_update_krell(slide->panel, slide->krell, x);
        gkrellm_draw_panel_layers(slide->panel);
    }
}

static void vaiobright_panel_button_press(GtkWidget *widget, GdkEventButton *ev, BrightSlider *slide) {
    gint x;

    if (ev->button != 1 || vaiobright_bslider->broken)
        return;

    vaiobright_bslider->slider_in_motion = 1;

    x = ev->x * slide->krell->full_scale / (gkrellm_chart_width() - 1);

    if (x < MIN_VAL)
        x = MIN_VAL;

    slide->krell->previous = 0;
    slide->setting = x;
    gkrellm_update_krell(slide->panel, slide->krell, (gulong) x);
    gkrellm_draw_panel_layers(slide->panel);
    // vaiobright_set_brightness(x, 1);
}

static void vaiobright_create_slider(int first_create) {
    GkrellmPiximage *bg_image, *krell_image;
    GkrellmStyle *panel_style = gkrellm_meter_style(vaiobright_style_id);
    GkrellmStyle *slider_style = gkrellm_krell_slider_style();
    gint bright;
    char *label, *text = "Brightness", *device = DEFAULT_DEVICE_PATH;

    if ( first_create ) vaiobright_bslider->panel = gkrellm_panel_new0();

    bg_image = gkrellm_bg_meter_piximage(vaiobright_style_id);

    label = strdup(text);
    vaiobright_bslider->panel->textstyle = gkrellm_meter_textstyle(vaiobright_style_id);
    gkrellm_panel_configure(vaiobright_bslider->panel, label, panel_style);
    gkrellm_panel_create(vaiobright_bslider->panelbox, &vaiobright_mon, vaiobright_bslider->panel);

    krell_image = gkrellm_krell_slider_piximage();
    vaiobright_bslider->krell = gkrellm_create_krell(vaiobright_bslider->panel, krell_image,
                                          slider_style);
    vaiobright_bslider->krell->y0 = (vaiobright_bslider->panel->h - vaiobright_bslider->krell->h_frame)/2;
    vaiobright_bslider->krell->full_scale = MAX_VAL;

    vaiobright_bslider->broken = 0;

    if (vaiobright_bslider->device_path == NULL)
        vaiobright_bslider->device_path = strdup(device);
    if (vaiobright_bslider->reversed == -1)
        vaiobright_bslider->reversed = 0;

    vaiobright_bslider->config_device_path = strdup(vaiobright_bslider->device_path);
    vaiobright_bslider->config_reversed = vaiobright_bslider->reversed;

    // Attach the signal handlers
    if (first_create) {
        gtk_signal_connect(GTK_OBJECT(vaiobright_bslider->panel->drawing_area), "expose_event",
                           (GtkSignalFunc) vaiobright_plug_expose_event, NULL);
        gtk_signal_connect(GTK_OBJECT(vaiobright_bslider->panel->drawing_area), "button_press_event",
                           (GtkSignalFunc) vaiobright_panel_button_press, vaiobright_bslider);
        gtk_signal_connect(GTK_OBJECT(vaiobright_bslider->panel->drawing_area), "button_release_event",
                           (GtkSignalFunc) vaiobright_panel_button_release, vaiobright_bslider);
        gtk_signal_connect(GTK_OBJECT(vaiobright_bslider->panel->drawing_area), "motion_notify_event",
                           (GtkSignalFunc) vaiobright_slider_motion, vaiobright_bslider);
    }

    /*
    bright = vaiobright_get_brightness();
    vaiobright_bslider->setting = vaiobright_bslider->reversed ? MAX_VAL - bright : bright;
    */

    // Open the device
    vaiobright_open_device();

    // Setting is loaded from config.  Set our brightness to it.
    vaiobright_set_brightness(vaiobright_bslider->reversed ? MAX_VAL - vaiobright_bslider->setting : vaiobright_bslider->setting);
    vaiobright_bslider->krell->previous = 0;

    gkrellm_update_krell(vaiobright_bslider->panel, vaiobright_bslider->krell, vaiobright_bslider->setting);
    gkrellm_draw_panel_layers(vaiobright_bslider->panel);
}

// Create the plugin itself
static void create_vaiobright_plugin(GtkWidget *vbox, gint first_create) {
    if (vaiobright_bslider->panelbox == NULL) {
        vaiobright_bslider->panelbox = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(vbox), vaiobright_bslider->panelbox);
        gtk_widget_show(vaiobright_bslider->panelbox);
    }
    vaiobright_create_slider(first_create);

    update_vaiobright_slider();
}

// GTK Callback - Did they change the text for the path to the brightness
// command?
static void vaiobright_device_entry_changed(GtkEntry *entry, gpointer *data) {
    vaiobright_bslider->config_device_path = strdup(gtk_entry_get_text(entry));
}

// GTK Callback - Did they tap the "reverse brightness slider"?
static void vaiobright_toggle_toggled(GtkWidget *button, gint data) {
    // If we're the "reverse slider" toggle...
    if (data == REVERSE_TOGGLE) {
        if (vaiobright_bslider->reversed)
            vaiobright_bslider->config_reversed = 0;
        else
            vaiobright_bslider->config_reversed = 1;
    }
}

/* Print out the help for the config */
static void vaiobright_create_help_text(GtkWidget *text) {
    gchar *info_text[] =
    {
        "<b>This plugin allows you to control the brightness of the Sony Vaio laptops\n",
        "<b>User interface:\n",
        "* Left mouse button on the slider will change the brightness\n",
        "* Middle button will toggle max darkness and max brightness\n",
        "* Using your mousewheel will change the brightness by a configurable amount\n",
        "<b>Usage stuff:\n",
        "Setting the brightness is done via ioctl's to the SonyPI device node.  If\n",
        "the brightness control isn't working, make sure that the sonypi module is\n",
        "loaded and that the sonypi device node is writeable by the user you are running\n",
        "gkrellm as.\n",
        "<b>Configuration:\n",
        "Device file:  The location of the sonypi device node (/dev/sonypi in almost\n",
        "all cases.\n",
        "Reverse Slider: It appears some laptops (like mine) reverse the brightness\n",
        "values so that 0 is the brightest and 255 is the darkest.  This flips the slider\n",
        "accordingly.\n",
    };
    gkrellm_gtk_text_view_append_strings(text, info_text, sizeof(info_text)/sizeof(gchar *));
}


// Build the config tab to set our (few) options.
static void create_vaiobright_plug_config(GtkWidget *tab) {
    GtkWidget *tabs;
    GtkWidget *frame;
    GtkWidget *label;
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *button;
    GtkWidget *text;
    GtkWidget *scrolled;
    GtkWidget *ext_device_path;
    gchar     *plugin_about_text = NULL;

    tabs = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs),GTK_POS_TOP);
    gtk_box_pack_start(GTK_BOX(tab),tabs,TRUE,TRUE,0);

    /* ----------- Options Menu -------------*/
    frame = gtk_frame_new(NULL);
    gtk_container_border_width(GTK_CONTAINER(frame),3);
    label = gtk_label_new("Options");
    gtk_notebook_append_page(GTK_NOTEBOOK(tabs),frame,label);


    // The external brightness control command
    vbox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(frame),vbox);

    hbox = gtk_hbox_new(FALSE,0);
    label = gtk_label_new("SonyPI device: ");
    gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);

    ext_device_path = gtk_entry_new_with_max_length(255);
    gtk_entry_set_text(GTK_ENTRY(ext_device_path), vaiobright_bslider->device_path);
    gtk_entry_set_editable(GTK_ENTRY(ext_device_path), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), ext_device_path, TRUE, TRUE, 2);
    gtk_signal_connect(GTK_OBJECT(ext_device_path), "changed",
                       (GtkSignalFunc) vaiobright_device_entry_changed, NULL);

    gtk_container_add(GTK_CONTAINER(vbox),hbox);

    button = gtk_check_button_new_with_label("Reverse Brightness Slider");
    gtk_container_add(GTK_CONTAINER(vbox),button);
    gtk_toggle_button_set_state((GtkToggleButton *)
                                button, vaiobright_bslider->reversed);

    gtk_signal_connect(GTK_OBJECT(button),"toggled",
                       (GtkSignalFunc) vaiobright_toggle_toggled, REVERSE_TOGGLE);

    /* Help tab */
    frame = gtk_frame_new(NULL);
    gtk_container_border_width(GTK_CONTAINER(frame),3);
    scrolled = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
                                   GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);

    gtk_container_add(GTK_CONTAINER(frame),scrolled);
    label = gtk_label_new("Info");
    gtk_notebook_append_page(GTK_NOTEBOOK(tabs),frame,label);

    text = gtk_text_view_new();
    gtk_text_view_set_editable(GTK_TEXT_VIEW(text),FALSE);
    vaiobright_create_help_text(text);
    gtk_container_add(GTK_CONTAINER(scrolled),text);


    /* About tab */
    plugin_about_text = g_strdup_printf(
                                        "Vaio Brightness %d.%d\n" \
                                        "GKrellM Vaio LCD Brightness Control\n" \
                                        "Copyright (c) 2002 Mike Kershaw\n" \
                                        "dragorn@nerv-un.net\n" \
                                        "Released under the GNU Public License\n" \
                                        "Based on Volumeplugin by Sjoerd Simons\n",
                                        VBRIGHT_MAJOR_VERSION, VBRIGHT_MINOR_VERSION);

    text = gtk_label_new(plugin_about_text);
    label = gtk_label_new("About");
    gtk_notebook_append_page(GTK_NOTEBOOK(tabs),text,label);
    g_free(plugin_about_text);

}

// Store the config
static void save_vaiobright_plug_config(FILE *f) {
    fprintf(f, "%s REVERSED %d\n", CONFIG_KEYWORD, vaiobright_bslider->reversed);
    fprintf(f, "%s DEVICE %s\n", CONFIG_KEYWORD, vaiobright_bslider->device_path);
    fprintf(f, "%s BRIGHT %d\n", CONFIG_KEYWORD, vaiobright_bslider->setting);
}

// Load the plugin conf
static void load_vaiobright_plug_config(gchar *arg) {
    int n = 0;
    gchar config[10], item[256];

    if(sscanf(arg, "%s %[^\n]", config, item) != 2 ) return ;

    if(!strcmp("DEVICE", config)) {
        vaiobright_bslider->device_path = strdup(item);
    }

    /* all other config entry's use a numeric item */
    sscanf(item,"%d", &n);

    if (!strcmp("REVERSED", config)) {
        vaiobright_bslider->reversed = n;
    } else if (!strcmp("BRIGHT", config)) {
        vaiobright_bslider->setting = n;
    }

}

// Someone hit the "apply" button
static void apply_vaiobright_plug_config(void) {
    vaiobright_bslider->reversed = vaiobright_bslider->config_reversed;
    vaiobright_bslider->device_path = strdup(vaiobright_bslider->config_device_path);
    update_vaiobright_slider();
}

GkrellmMonitor *gkrellm_init_plugin(void) {
    vaiobright_bslider = g_new0(BrightSlider, 1);

    vaiobright_bslider->device_path = NULL;
    vaiobright_bslider->reversed = -1;

    vaiobright_style_id = gkrellm_add_meter_style(&vaiobright_mon, "brightness");
    return &vaiobright_mon;
}
