C
C     THIS FILE CONTAINS THE FOLLOWING OPTIMIZATION PACKAGES:
C
C     1) UNCMIN  - THE UNCMIN PACKAGE IS DOCUMENTED IN THE
C                  ARTICLE
C
C                  SCHNABEL, KOONTZ, AND WEISS "A MODULAR SYSTEM
C                  OF ALGORITHMS FOR UNCONSTRAINED OPTIMIZATION".
C
c                  THIS PACKAGE SUPPORTS 18 OPTIMIZATION METHODS:
C
C                  1) STEP SELECTION  - LINE, DOGLEG, HOOK STEP
C                  2) GRADIENTS       - NUMERIC OR ANALYTIC
C                  3) HESSIAN         - ANALYTIC, BFGS UPDATE, OR
C                                       FINITE DIFFERENCES
C
C                  THESE CAN BE USED INDEPENDTLY FOR A TOTAL OF
C                  18 ALGORITHMS (MORE LIKE 15 SINCE ANALYTIC
C                  HESSIANS WITH NUMERIC DERIVATIVES IS A BIT
C                  UNREALISTIC).
C
C                  NOTE THAT THE UNCMIN SOFTWARE IS USED TO
C                  IMPLEMENT THE DATAPLOT "OPTIMIZE" COMMAND.
C                  THE CODE INCLUDED HERE IS FOR INTERNAL DATAPLOT
C                  USAGE.  TO AVOID NAME CONFLICTS, THE ROUTINES
C                  HAVE BEEN RENAMED.
C
C     2) NNES    - ROD BAIN'S PACKAGE FOR OPTIMIZATION WITH
C                  BOUNDS.  CAN USE EITHER ANALYTIC OR NUMERIC
C                  GRADIENTS/HESSIANS.
C
C     3) TNBC    = STEPHEN NASH PACKAGE FOR EITHER UNCONSTRAINED
C                  OPTIMIZATION OR SIMPLE BOUND CONSTRAINED
C                  OPTIMIZATION.
C
C
C
C     THIS IS THE UNCMIN ROUTINE FOR UNCONSTRAINED OPTIMIATION.
C     SINCE WE ALSO USE THESE ROUTINES FOR USER-CALLABLE OPTIMIZATION,
C     RENAME THE SUBROUTINES TO AVOID NAME CONFLICTS (FOR INTERNAL
C     DATAPLOT USE, WE PROVIDE A SPECIFIC FUNCTION WHILE FOR
C     USER-DEFINABLE FUNCTIONS, WE CALL COMPIM, WHICH REQUIRES
C     MODIFICATION TO THE CALLING SEQUENCES.
C
C     ROUTINES ARE ALSO MODIFIED TO USE DATAPLOT I/O.
C
C     ROUTINE RENAME:
C     ORIGiNAL         NEW
C     ====================
C     BAKSLV           BAKSLZ
C     CHLHSN           CHLHSZ
C     CHOLDC           CHOLDZ
C     DFAULT           DFAULZ
C     DOGDRV           DOGDRZ
C     DOGSTP           DOGSTZ
C     FORSLZ           FORSLZ
C     FSTOCZ           FSTOCZ
C     FSTOFD           FSTOFZ
C     GRDCHK           GRDCHZ
C     HESCHK           HESCHZ
C     HOOKDR           HOOKDZ
C     HOOKST           HOOKSZ
C     HSNINT           HSNINZ
C     LLTSLV           LLTSLZ
C     LNSRCH           LNSRCZ
C     MVMLTL           MVMLTZ
C     MVMLTS           MVMLTY
C     MVMLTU           MVMLTX
C     OPTCHK           OPTCHZ
C     OPTDRV           OPTDRZ
C     OPTIF0           OPTIF0
C     OPTIF9           OPTIFZ
C     OPTSTP           OPTSTZ
C     QRAUX1           QRAUXZ
C     QRAUXY           QRAUXY
C     QRUPDT           QRUPDZ
C     RESULT           RESULZ
C     SCLMUZ           SCLMUZ
C     SECFAC           SECFAZ
C     SECUNF           SECUNZ
C     SNDOFD           SNDOFZ
C     TREGUP           TREGUZ
C
      SUBROUTINE BAKSLZ(NR,N,A,X,B)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C PURPOSE
C -------
C SOLVE  AX=B  WHERE A IS UPPER TRIANGULAR MATRIX.
C NOTE THAT A IS INPUT AS A LOWER TRIANGULAR MATRIX AND
C THAT THIS ROUTINE TAKES ITS TRANSPOSE IMPLICITLY.
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(N,N)       --> LOWER TRIANGULAR MATRIX (PRESERVED)
C X(N)        <--  SOLUTION VECTOR
C B(N)         --> RIGHT-HAND SIDE VECTOR
C
C NOTE
C ----
C IF B IS NO LONGER REQUIRED BY CALLING ROUTINE,
C THEN VECTORS B AND X MAY SHARE THE SAME STORAGE.
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*),X(N),B(N)
C
C SOLVE (L-TRANSPOSE)X=B. (BACK SOLVE)
C
      I=N
      X(I)=B(I)/A(I,I)
      IF(N.EQ.1) RETURN
   30 IP1=I
      I=I-1
      SUM=0.
      DO 40 J=IP1,N
        SUM=SUM+A(J,I)*X(J)
   40 CONTINUE
      X(I)=(B(I)-SUM)/A(I,I)
      IF(I.GT.1) GO TO 30
      RETURN
      END
      SUBROUTINE CHLHSZ(NR,N,A,EPSM,SX,UDIAG)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C PURPOSE
C -------
C FIND THE L(L-TRANSPOSE) [WRITTEN LL+] DECOMPOSITION OF THE PERTURBED
C MODEL HESSIAN MATRIX A+MU*I(WHERE MU\0 AND I IS THE IDENTITY MATRIX)
C WHICH IS SAFELY POSITIVE DEFINITE.  IF A IS SAFELY POSITIVE DEFINITE
C UPON ENTRY, THEN MU=0.
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(N,N)      <--> ON ENTRY; "A" IS MODEL HESSIAN (ONLY LOWER
C                  TRIANGULAR PART AND DIAGONAL STORED)
C                  ON EXIT:  A CONTAINS L OF LL+ DECOMPOSITION OF
C                  PERTURBED MODEL HESSIAN IN LOWER TRIANGULAR
C                  PART AND DIAGONAL AND CONTAINS HESSIAN IN UPPER
C                  TRIANGULAR PART AND UDIAG
C EPSM         --> MACHINE EPSILON
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C UDIAG(N)    <--  ON EXIT: CONTAINS DIAGONAL OF HESSIAN
C
C INTERNAL VARIABLES
C ------------------
C TOL              TOLERANCE
C DIAGMN           MINIMUM ELEMENT ON DIAGONAL OF A
C DIAGMX           MAXIMUM ELEMENT ON DIAGONAL OF A
C OFFMAX           MAXIMUM OFF-DIAGONAL ELEMENT OF A
C OFFROW           SUM OF OFF-DIAGONAL ELEMENTS IN A ROW OF A
C EVMIN            MINIMUM EIGENVALUE OF A
C EVMAX            MAXIMUM EIGENVALUE OF A
C
C DESCRIPTION
C -----------
C 1. IF "A" HAS ANY NEGATIVE DIAGONAL ELEMENTS, THEN CHOOSE MU>0
C SUCH THAT THE DIAGONAL OF A:=A+MU*I IS ALL POSITIVE
C WITH THE RATIO OF ITS SMALLEST TO LARGEST ELEMENT ON THE
C ORDER OF SQRT(EPSM).
C
C 2. "A" UNDERGOES A PERTURBED CHOLESKY DECOMPOSITION WHICH
C RESULTS IN AN LL+ DECOMPOSITION OF A+D, WHERE D IS A
C NON-NEGATIVE DIAGONAL MATRIX WHICH IS IMPLICITLY ADDED TO
C "A" DURING THE DECOMPOSITION IF "A" IS NOT POSITIVE DEFINITE.
C "A" IS RETAINED AND NOT CHANGED DURING THIS PROCESS BY
C COPYING L INTO THE UPPER TRIANGULAR PART OF "A" AND THE
C DIAGONAL INTO UDIAG.  THEN THE CHOLESKY DECOMPOSITION ROUTINE
C IS CALLED.  ON RETURN, ADDMAX CONTAINS MAXIMUM ELEMENT OF D.
C
C 3. IF ADDMAX=0, "A" WAS POSITIVE DEFINITE GOING INTO STEP 2
C AND RETURN IS MADE TO CALLING PROGRAM.  OTHERWISE,
C THE MINIMUM NUMBER SDD WHICH MUST BE ADDED TO THE
C DIAGONAL OF A TO MAKE IT SAFELY STRICTLY DIAGONALLY DOMINANT
C IS CALCULATED.  SINCE A+ADDMAX*I AND A+SDD*I ARE SAFELY
C POSITIVE DEFINITE, CHOOSE MU=MIN(ADDMAX,SDD) AND DECOMPOSE
C A+MU*I TO OBTAIN L.
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*),SX(N),UDIAG(N)
C
C SCALE HESSIAN
C PRE- AND POST- MULTIPLY "A" BY INV(SX)
C
      DO 20 J=1,N
        DO 10 I=J,N
          A(I,J)=A(I,J)/(SX(I)*SX(J))
   10   CONTINUE
   20 CONTINUE
C
C STEP1
C -----
C NOTE:  IF A DIFFERENT TOLERANCE IS DESIRED THROUGHOUT THIS
C ALGORITHM, CHANGE TOLERANCE HERE:
      TOL=SQRT(EPSM)
C
      DIAGMX=A(1,1)
      DIAGMN=A(1,1)
      IF(N.EQ.1) GO TO 35
      DO 30 I=2,N
        IF(A(I,I).LT.DIAGMN) DIAGMN=A(I,I)
        IF(A(I,I).GT.DIAGMX) DIAGMX=A(I,I)
   30 CONTINUE
   35 POSMAX=MAX(DIAGMX,0.D0)
C
C DIAGMN .LE. 0
C
      IF(DIAGMN.GT.POSMAX*TOL) GO TO 100
C     IF(DIAGMN.LE.POSMAX*TOL)
C     THEN
        AMU=TOL*(POSMAX-DIAGMN)-DIAGMN
        IF(AMU.NE.0.) GO TO 60
C       IF(AMU.EQ.0.)
C       THEN
C
C FIND LARGEST OFF-DIAGONAL ELEMENT OF A
          OFFMAX=0.
          IF(N.EQ.1) GO TO 50
          DO 45 I=2,N
            IM1=I-1
            DO 40 J=1,IM1
              IF(ABS(A(I,J)).GT.OFFMAX) OFFMAX=ABS(A(I,J))
   40       CONTINUE
   45     CONTINUE
   50     AMU=OFFMAX
          IF(AMU.NE.0.) GO TO 55
C         IF(AMU.EQ.0.)
C         THEN
            AMU=1.0
            GO TO 60
C         ELSE
   55       AMU=AMU*(1.0+TOL)
C         ENDIF
C       ENDIF
C
C A=A + MU*I
C
   60   DO 65 I=1,N
          A(I,I)=A(I,I)+AMU
   65   CONTINUE
        DIAGMX=DIAGMX+AMU
C     ENDIF
C
C STEP2
C -----
C COPY LOWER TRIANGULAR PART OF "A" TO UPPER TRIANGULAR PART
C AND DIAGONAL OF "A" TO UDIAG
C
  100 CONTINUE
      DO 110 J=1,N
        UDIAG(J)=A(J,J)
        IF(J.EQ.N) GO TO 110
        JP1=J+1
        DO 105 I=JP1,N
          A(J,I)=A(I,J)
  105   CONTINUE
  110 CONTINUE
C
      CALL CHOLDZ(NR,N,A,DIAGMX,TOL,ADDMAX)
C
C
C STEP3
C -----
C IF ADDMAX=0, "A" WAS POSITIVE DEFINITE GOING INTO STEP 2,
C THE LL+ DECOMPOSITION HAS BEEN DONE, AND WE RETURN.
C OTHERWISE, ADDMAX>0.  PERTURB "A" SO THAT IT IS SAFELY
C DIAGONALLY DOMINANT AND FIND LL+ DECOMPOSITION
C
      IF(ADDMAX.LE.0.) GO TO 170
C     IF(ADDMAX.GT.0.)
C     THEN
C
C RESTORE ORIGINAL "A" (LOWER TRIANGULAR PART AND DIAGONAL)
C
        DO 120 J=1,N
          A(J,J)=UDIAG(J)
          IF(J.EQ.N) GO TO 120
          JP1=J+1
          DO 115 I=JP1,N
            A(I,J)=A(J,I)
  115     CONTINUE
  120   CONTINUE
C
C FIND SDD SUCH THAT A+SDD*I IS SAFELY POSITIVE DEFINITE
C NOTE:  EVMIN<0 SINCE A IS NOT POSITIVE DEFINITE;
C
        EVMIN=0.
        EVMAX=A(1,1)
        DO 150 I=1,N
          OFFROW=0.
          IF(I.EQ.1) GO TO 135
          IM1=I-1
          DO 130 J=1,IM1
            OFFROW=OFFROW+ABS(A(I,J))
  130     CONTINUE
  135     IF(I.EQ.N) GO TO 145
          IP1=I+1
          DO 140 J=IP1,N
            OFFROW=OFFROW+ABS(A(J,I))
  140     CONTINUE
  145     EVMIN=MIN(EVMIN,A(I,I)-OFFROW)
          EVMAX=MAX(EVMAX,A(I,I)+OFFROW)
  150   CONTINUE
        SDD=TOL*(EVMAX-EVMIN)-EVMIN
C
C PERTURB "A" AND DECOMPOSE AGAIN
C
        AMU=MIN(SDD,ADDMAX)
        DO 160 I=1,N
          A(I,I)=A(I,I)+AMU
          UDIAG(I)=A(I,I)
  160   CONTINUE
C
C "A" NOW GUARANTEED SAFELY POSITIVE DEFINITE
C
        CALL CHOLDZ(NR,N,A,0.0D0,TOL,ADDMAX)
C     ENDIF
C
C UNSCALE HESSIAN AND CHOLESKY DECOMPOSITION MATRIX
C
  170 DO 190 J=1,N
        DO 175 I=J,N
          A(I,J)=SX(I)*A(I,J)
  175   CONTINUE
        IF(J.EQ.1) GO TO 185
        JM1=J-1
        DO 180 I=1,JM1
          A(I,J)=SX(I)*SX(J)*A(I,J)
  180   CONTINUE
  185   UDIAG(J)=UDIAG(J)*SX(J)*SX(J)
  190 CONTINUE
      RETURN
      END
      SUBROUTINE CHOLDZ(NR,N,A,DIAGMX,TOL,ADDMAX)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C PURPOSE
C -------
C FIND THE PERTURBED L(L-TRANSPOSE) [WRITTEN LL+] DECOMPOSITION
C OF A+D, WHERE D IS A NON-NEGATIVE DIAGONAL MATRIX ADDED TO A IF
C NECESSARY TO ALLOW THE CHOLESKY DECOMPOSITION TO CONTINUE.
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(N,N)      <--> ON ENTRY: MATRIX FOR WHICH TO FIND PERTURBED
C                       CHOLESKY DECOMPOSITION
C                  ON EXIT:  CONTAINS L OF LL+ DECOMPOSITION
C                  IN LOWER TRIANGULAR PART AND DIAGONAL OF "A"
C DIAGMX       --> MAXIMUM DIAGONAL ELEMENT OF "A"
C TOL          --> TOLERANCE
C ADDMAX      <--  MAXIMUM AMOUNT IMPLICITLY ADDED TO DIAGONAL OF "A"
C                  IN FORMING THE CHOLESKY DECOMPOSITION OF A+D
C INTERNAL VARIABLES
C ------------------
C AMINL    SMALLEST ELEMENT ALLOWED ON DIAGONAL OF L
C AMNLSQ   =AMINL**2
C OFFMAX   MAXIMUM OFF-DIAGONAL ELEMENT IN COLUMN OF A
C
C
C DESCRIPTION
C -----------
C THE NORMAL CHOLESKY DECOMPOSITION IS PERFORMED.  HOWEVER, IF AT ANY
C POINT THE ALGORITHM WOULD ATTEMPT TO SET L(I,I)=SQRT(TEMP)
C WITH TEMP < TOL*DIAGMX, THEN L(I,I) IS SET TO SQRT(TOL*DIAGMX)
C INSTEAD.  THIS IS EQUIVALENT TO ADDING TOL*DIAGMX-TEMP TO A(I,I)
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*)
C
      ADDMAX=0.
      AMINL=SQRT(DIAGMX*TOL)
      AMNLSQ=AMINL*AMINL
C
C FORM COLUMN J OF L
C
      DO 100 J=1,N
C FIND DIAGONAL ELEMENTS OF L
        SUM=0.
        IF(J.EQ.1) GO TO 20
        JM1=J-1
        DO 10 K=1,JM1
          SUM=SUM + A(J,K)*A(J,K)
   10   CONTINUE
   20   TEMP=A(J,J)-SUM
        IF(TEMP.LT.AMNLSQ) GO TO 30
C       IF(TEMP.GE.AMINL**2)
C       THEN
          A(J,J)=SQRT(TEMP)
          GO TO 40
C       ELSE
C
C FIND MAXIMUM OFF-DIAGONAL ELEMENT IN COLUMN
   30     OFFMAX=0.
          IF(J.EQ.N) GO TO 37
          JP1=J+1
          DO 35 I=JP1,N
            IF(ABS(A(I,J)).GT.OFFMAX) OFFMAX=ABS(A(I,J))
   35     CONTINUE
   37     IF(OFFMAX.LE.AMNLSQ) OFFMAX=AMNLSQ
C
C ADD TO DIAGONAL ELEMENT  TO ALLOW CHOLESKY DECOMPOSITION TO CONTINUE
          A(J,J)=SQRT(OFFMAX)
          ADDMAX=MAX(ADDMAX,OFFMAX-TEMP)
C       ENDIF
C
C FIND I,J ELEMENT OF LOWER TRIANGULAR MATRIX
   40   IF(J.EQ.N) GO TO 100
        JP1=J+1
        DO 70 I=JP1,N
          SUM=0.0
          IF(J.EQ.1) GO TO 60
          JM1=J-1
          DO 50 K=1,JM1
            SUM=SUM+A(I,K)*A(J,K)
   50     CONTINUE
   60     A(I,J)=(A(I,J)-SUM)/A(J,J)
   70   CONTINUE
  100 CONTINUE
      RETURN
      END
      SUBROUTINE DFAULZ(N,X,TYPSIZ,FSCALE,METHOD,IEXP,MSG,NDIGIT,
     +     ITNLIM,IAGFLG,IAHFLG,IPRZZ,DLT,GRADTL,STEPMX,STEPTL)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      REAL R1MACH
      INCLUDE 'DPCOMC.INC'
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C PURPOSE
C -------
C SET DEFAULT VALUES FOR EACH INPUT VARIABLE TO
C MINIMIZATION ALGORITHM.
C
C PARAMETERS
C ----------
C N            --> DIMENSION OF PROBLEM
C X(N)         --> INITIAL GUESS TO SOLUTION (TO COMPUTE MAX STEP SIZE)
C TYPSIZ(N)   <--  TYPICAL SIZE FOR EACH COMPONENT OF X
C FSCALE      <--  ESTIMATE OF SCALE OF MINIMIZATION FUNCTION
C METHOD      <--  ALGORITHM TO USE TO SOLVE MINIMIZATION PROBLEM
C IEXP        <--  =0 IF MINIMIZATION FUNCTION NOT EXPENSIVE TO EVALUATE
C MSG         <--  MESSAGE TO INHIBIT CERTAIN AUTOMATIC CHECKS + OUTPUT
C NDIGIT      <--  NUMBER OF GOOD DIGITS IN MINIMIZATION FUNCTION
C ITNLIM      <--  MAXIMUM NUMBER OF ALLOWABLE ITERATIONS
C IAGFLG      <--  =0 IF ANALYTIC GRADIENT NOT SUPPLIED
C IAHFLG      <--  =0 IF ANALYTIC HESSIAN NOT SUPPLIED
C IPR         <--  DEVICE TO WHICH TO SEND OUTPUT
C DLT         <--  TRUST REGION RADIUS
C GRADTL      <--  TOLERANCE AT WHICH GRADIENT CONSIDERED CLOSE ENOUGH
C                  TO ZERO TO TERMINATE ALGORITHM
C STEPMX      <--  VALUE OF ZERO TO TRIP DEFAULT MAXIMUM IN OPTCHZ
C STEPTL      <--  TOLERANCE AT WHICH SUCCESSIVE ITERATES CONSIDERED
C                  CLOSE ENOUGH TO TERMINATE ALGORITHM
C
      DIMENSION TYPSIZ(N),X(N)
      X(N)=X(N)
C
C SET TYPICAL SIZE OF X AND MINIMIZATION FUNCTION
      DO 10 I=1,N
        TYPSIZ(I)=1.0
   10 CONTINUE
      FSCALE=1.0
C
C SET TOLERANCES
      DLT=-1.0
      EPSM=D1MACH(4)
      GRADTL=EPSM**(1.0/3.0)
      STEPMX=0.0
      STEPTL=SQRT(EPSM)
C
C SET FLAGS
      METHOD=1
      IEXP=1
      MSG=0
      NDIGIT=-1
      ITNLIM=150
      IAGFLG=0
      IAHFLG=0
      IPR=I1MACH(2)
C
      RETURN
      END
      SUBROUTINE DOGDRZ(NR,N,X,F,G,A,P,XPLS,FPLS,FCN,SX,STEPMX,
     +     STEPTL,DLT,IRETCD,MXTAKE,SC,WRK1,WRK2,WRK3,IPRZZ)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C FIND A NEXT NEWTON ITERATE (XPLS) BY THE DOUBLE DOGLEG METHOD
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> OLD ITERATE X[K-1]
C F            --> FUNCTION VALUE AT OLD ITERATE, F(X)
C G(N)         --> GRADIENT  AT OLD ITERATE, G(X), OR APPROXIMATE
C A(N,N)       --> CHOLESKY DECOMPOSITION OF HESSIAN
C                  IN LOWER TRIANGULAR PART AND DIAGONAL
C P(N)         --> NEWTON STEP
C XPLS(N)     <--  NEW ITERATE X[K]
C FPLS        <--  FUNCTION VALUE AT NEW ITERATE, F(XPLS)
C FCN          --> NAME OF SUBROUTINE TO EVALUATE FUNCTION
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C STEPMX       --> MAXIMUM ALLOWABLE STEP SIZE
C STEPTL       --> RELATIVE STEP SIZE AT WHICH SUCCESSIVE ITERATES
C                  CONSIDERED CLOSE ENOUGH TO TERMINATE ALGORITHM
C DLT         <--> TRUST REGION RADIUS
C                  [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C IRETCD      <--  RETURN CODE
C                    =0 SATISFACTORY XPLS FOUND
C                    =1 FAILED TO FIND SATISFACTORY XPLS SUFFICIENTLY
C                       DISTINCT FROM X
C MXTAKE      <--  BOOLEAN FLAG INDICATING STEP OF MAXIMUM LENGTH USED
C SC(N)        --> WORKSPACE [CURRENT STEP]
C WRK1(N)      --> WORKSPACE (AND PLACE HOLDING ARGUMENT TO TREGUZ)
C WRK2(N)      --> WORKSPACE
C WRK3(N)      --> WORKSPACE
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),XPLS(N),G(N),P(N)
      DIMENSION SX(N)
      DIMENSION SC(N),WRK1(N),WRK2(N),WRK3(N)
      DIMENSION A(NR,*)
      LOGICAL FSTDOG,NWTAKE,MXTAKE
      EXTERNAL FCN
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      IRETCD=4
      FSTDOG=.TRUE.
      TMP=0.
      DO 5 I=1,N
        TMP=TMP+SX(I)*SX(I)*P(I)*P(I)
    5 CONTINUE
      RNWTLN=SQRT(TMP)
CCCCC WRITE(IPR,954) RNWTLN
C
  100 CONTINUE
C
C FIND NEW STEP BY DOUBLE DOGLEG ALGORITHM
      CALL DOGSTZ(NR,N,G,A,P,SX,RNWTLN,DLT,NWTAKE,FSTDOG,
     +     WRK1,WRK2,CLN,ETA,SC,IPRZZ,STEPMX)
C
C CHECK NEW POINT AND UPDATE TRUST REGION
      CALL TREGUZ(NR,N,X,F,G,A,FCN,SC,SX,NWTAKE,STEPMX,STEPTL,DLT,
     +     IRETCD,WRK3,FPLSP,XPLS,FPLS,MXTAKE,IPRZZ,2,WRK1)
      IF(IRETCD.LE.1) RETURN
      GO TO 100
CC950 FORMAT(42H DOGDRZ    INITIAL TRUST REGION NOT GIVEN.,
CCCCC+       22H  COMPUTE CAUCHY STEP.)
CC951 FORMAT(18H DOGDRZ    ALPHA =,E20.13/
CCCCC+       18H DOGDRZ    BETA  =,E20.13/
CCCCC+       18H DOGDRZ    DLT   =,E20.13/
CCCCC+       18H DOGDRZ    NWTAKE=,L1    )
CC952 FORMAT(28H DOGDRZ    CURRENT STEP (SC))
CC954 FORMAT(18H0DOGDRZ    RNWTLN=,E20.13)
CC955 FORMAT(14H DOGDRZ       ,5(E20.13,3X))
      END
      SUBROUTINE DOGSTZ(NR,N,G,A,P,SX,RNWTLN,DLT,NWTAKE,FSTDOG,
     +     SSD,V,CLN,ETA,SC,IPRZZ,STEPMX)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C FIND NEW STEP BY DOUBLE DOGLEG ALGORITHM
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C G(N)         --> GRADIENT AT CURRENT ITERATE, G(X)
C A(N,N)       --> CHOLESKY DECOMPOSITION OF HESSIAN IN
C                  LOWER PART AND DIAGONAL
C P(N)         --> NEWTON STEP
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C RNWTLN       --> NEWTON STEP LENGTH
C DLT         <--> TRUST REGION RADIUS
C NWTAKE      <--> BOOLEAN, =.TRUE. IF NEWTON STEP TAKEN
C FSTDOG      <--> BOOLEAN, =.TRUE. IF ON FIRST LEG OF DOGLEG
C SSD(N)      <--> WORKSPACE [CAUCHY STEP TO THE MINIMUM OF THE
C                  QUADRATIC MODEL IN THE SCALED STEEPEST DESCENT
C                  DIRECTION] [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C V(N)        <--> WORKSPACE  [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C CLN         <--> CAUCHY LENGTH
C                  [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C ETA              [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C SC(N)       <--  CURRENT STEP
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C STEPMX       --> MAXIMUM ALLOWABLE STEP SIZE
C
C INTERNAL VARIABLES
C ------------------
C CLN              LENGTH OF CAUCHY STEP
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION G(N),P(N)
      DIMENSION SX(N)
      DIMENSION SC(N),SSD(N),V(N)
      DIMENSION A(NR,*)
      LOGICAL NWTAKE,FSTDOG
      IPR=IPR
C
C CAN WE TAKE NEWTON STEP
C
      IF(RNWTLN.GT.DLT) GO TO 100
C     IF(RNWTLN.LE.DLT)
C     THEN
        NWTAKE=.TRUE.
        DO 10 I=1,N
          SC(I)=P(I)
   10   CONTINUE
        DLT=RNWTLN
C$      WRITE(IPR,951)
        GO TO 700
C     ELSE
C
C NEWTON STEP TOO LONG
C CAUCHY STEP IS ON DOUBLE DOGLEG CURVE
C
  100   NWTAKE=.FALSE.
        IF(.NOT.FSTDOG) GO TO 200
C       IF(FSTDOG)
C       THEN
C
C         CALCULATE DOUBLE DOGLEG CURVE (SSD)
          FSTDOG=.FALSE.
          ALPHA=0.
          DO 110 I=1,N
            ALPHA=ALPHA + (G(I)*G(I))/(SX(I)*SX(I))
  110     CONTINUE
          BETA=0.
          DO 130 I=1,N
            TMP=0.
            DO 120 J=I,N
              TMP=TMP + (A(J,I)*G(J))/(SX(J)*SX(J))
  120       CONTINUE
            BETA=BETA+TMP*TMP
  130     CONTINUE
          DO 140 I=1,N
            SSD(I)=-(ALPHA/BETA)*G(I)/SX(I)
  140     CONTINUE
          CLN=ALPHA*SQRT(ALPHA)/BETA
          ETA=.2 + (.8*ALPHA*ALPHA)/(-BETA*DDOT(N,G,1,P,1))
          DO 150 I=1,N
            V(I)=ETA*SX(I)*P(I) - SSD(I)
  150     CONTINUE
          IF (DLT .EQ. (-1.0)) DLT = MIN(CLN, STEPMX)
C$        WRITE(IPR,954) ALPHA,BETA,CLN,ETA
C$        WRITE(IPR,955)
C$        WRITE(IPR,960) (SSD(I),I=1,N)
C$        WRITE(IPR,956)
C$        WRITE(IPR,960) (V(I),I=1,N)
C       ENDIF
  200   IF(ETA*RNWTLN.GT.DLT) GO TO 220
C       IF(ETA*RNWTLN .LE. DLT)
C       THEN
C
C         TAKE PARTIAL STEP IN NEWTON DIRECTION
C
          DO 210 I=1,N
            SC(I)=(DLT/RNWTLN)*P(I)
  210     CONTINUE
C$        WRITE(IPR,957)
          GO TO 700
C       ELSE
  220     IF(CLN.LT.DLT) GO TO 240
C         IF(CLN.GE.DLT)
C         THEN
C           TAKE STEP IN STEEPEST DESCENT DIRECTION
C
            DO 230 I=1,N
              SC(I)=(DLT/CLN)*SSD(I)/SX(I)
  230       CONTINUE
C$          WRITE(IPR,958)
            GO TO 700
C         ELSE
C
C           CALCULATE CONVEX COMBINATION OF SSD AND ETA*P
C           WHICH HAS SCALED LENGTH DLT
C
  240       DOT1=DDOT(N,V,1,SSD,1)
            DOT2=DDOT(N,V,1,V,1)
            ALAM=(-DOT1+SQRT((DOT1*DOT1)-DOT2*(CLN*CLN-DLT*DLT)))/DOT2
            DO 250 I=1,N
              SC(I)=(SSD(I) + ALAM*V(I))/SX(I)
  250       CONTINUE
C$          WRITE(IPR,959)
C         ENDIF
C       ENDIF
C     ENDIF
  700 CONTINUE
C$    WRITE(IPR,952) FSTDOG,NWTAKE,RNWTLN,DLT
C$    WRITE(IPR,953)
C$    WRITE(IPR,960) (SC(I),I=1,N)
      RETURN
C
CC951 FORMAT(27H0DOGSTZ    TAKE NEWTON STEP)
CC952 FORMAT(18H DOGSTZ    FSTDOG=,L1/
CC   +       18H DOGSTZ    NWTAKE=,L1/
CC   +       18H DOGSTZ    RNWTLN=,E20.13/
CC   +       18H DOGSTZ    DLT   =,E20.13)
CC953 FORMAT(28H DOGSTZ    CURRENT STEP (SC))
CC954 FORMAT(18H DOGSTZ    ALPHA =,E20.13/
CC   +       18H DOGSTZ    BETA  =,E20.13/
CC   +       18H DOGSTZ    CLN   =,E20.13/
CC   +       18H DOGSTZ    ETA   =,E20.13)
CC955 FORMAT(28H DOGSTZ    CAUCHY STEP (SSD))
CC956 FORMAT(12H DOGSTZ    V)
CC957 FORMAT(48H0DOGSTZ    TAKE PARTIAL STEP IN NEWTON DIRECTION)
CC958 FORMAT(50H0DOGSTZ    TAKE STEP IN STEEPEST DESCENT DIRECTION)
CC959 FORMAT(39H0DOGSTZ    TAKE CONVEX COMBINATION STEP)
CC960 FORMAT(14H DOGSTZ       ,5(E20.13,3X))
      END
      SUBROUTINE FORSLZ(NR,N,A,X,B)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      REAL CPUMIN, CPUMAX
C
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C PURPOSE
C -------
C SOLVE  AX=B  WHERE A IS LOWER TRIANGULAR MATRIX
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(N,N)       --> LOWER TRIANGULAR MATRIX (PRESERVED)
C X(N)        <--  SOLUTION VECTOR
C B(N)         --> RIGHT-HAND SIDE VECTOR
C
C NOTE
C ----
C IF B IS NO LONGER REQUIRED BY CALLING ROUTINE,
C THEN VECTORS B AND X MAY SHARE THE SAME STORAGE.
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*),X(N),B(N)
C
C SOLVE LX=B. (FOREWARD SOLVE)
C
      X(1)=B(1)/A(1,1)
      IF(N.EQ.1) RETURN
      DO 20 I=2,N
        SUM=0.0
        IM1=I-1
        DO 10 J=1,IM1
          SUM=SUM+A(I,J)*X(J)
   10   CONTINUE
        X(I)=(B(I)-SUM)/A(I,I)
   20 CONTINUE
      RETURN
      END
      SUBROUTINE FSTOCZ (N, X, FCN, SX, RNOISE, G)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      REAL CPUMIN, CPUMAX
C
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C PURPOSE
C -------
C FIND CENTRAL DIFFERENCE APPROXIMATION G TO THE FIRST DERIVATIVE
C (GRADIENT) OF THE FUNCTION DEFINED BY FCN AT THE POINT X.
C
C PARAMETERS
C ----------
C N            --> DIMENSION OF PROBLEM
C X            --> POINT AT WHICH GRADIENT IS TO BE APPROXIMATED.
C FCN          --> NAME OF SUBROUTINE TO EVALUATE FUNCTION.
C SX           --> DIAGONAL SCALING MATRIX FOR X.
C RNOISE       --> RELATIVE NOISE IN FCN [F(X)].
C G           <--  CENTRAL DIFFERENCE APPROXIMATION TO GRADIENT.
C
C
      DIMENSION X(N)
      DIMENSION SX(N)
      DIMENSION G(N)
      EXTERNAL FCN
C
C FIND I TH  STEPSIZE, EVALUATE TWO NEIGHBORS IN DIRECTION OF I TH
C UNIT VECTOR, AND EVALUATE I TH  COMPONENT OF GRADIENT.
C
      THIRD = 1.0/3.0
      DO 10 I = 1, N
         STEPI = RNOISE**THIRD * MAX(ABS(X(I)), 1.0/SX(I))
         XTEMPI = X(I)
         X(I) = XTEMPI + STEPI
         CALL FCN (N, X, FPLUS)
         X(I) = XTEMPI - STEPI
         CALL FCN (N, X, FMINUS)
         X(I) = XTEMPI
         G(I) = (FPLUS - FMINUS)/(2.0*STEPI)
   10 CONTINUE
      RETURN
      END
      SUBROUTINE FSTOFZ(NR,M,N,XPLS,FCN,FPLS,A,SX,RNOISE,FHAT,ICASE)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C PURPOSE
C -------
C FIND FIRST ORDER FORWARD FINITE DIFFERENCE APPROXIMATION "A" TO THE
C FIRST DERIVATIVE OF THE FUNCTION DEFINED BY THE SUBPROGRAM "FNAME"
C EVALUATED AT THE NEW ITERATE "XPLS".
C
C
C FOR OPTIMIZATION USE THIS ROUTINE TO ESTIMATE:
C 1) THE FIRST DERIVATIVE (GRADIENT) OF THE OPTIMIZATION FUNCTION "FCN
C    ANALYTIC USER ROUTINE HAS BEEN SUPPLIED;
C 2) THE SECOND DERIVATIVE (HESSIAN) OF THE OPTIMIZATION FUNCTION
C    IF NO ANALYTIC USER ROUTINE HAS BEEN SUPPLIED FOR THE HESSIAN BUT
C    ONE HAS BEEN SUPPLIED FOR THE GRADIENT ("FCN") AND IF THE
C    OPTIMIZATION FUNCTION IS INEXPENSIVE TO EVALUATE
C
C NOTE
C ----
C _M=1 (OPTIMIZATION) ALGORITHM ESTIMATES THE GRADIENT OF THE FUNCTION
C      (FCN).   FCN(X) # F: R(N)-->R(1)
C _M=N (SYSTEMS) ALGORITHM ESTIMATES THE JACOBIAN OF THE FUNCTION
C      FCN(X) # F: R(N)-->R(N).
C _M=N (OPTIMIZATION) ALGORITHM ESTIMATES THE HESSIAN OF THE OPTIMIZATIO
C      FUNCTION, WHERE THE HESSIAN IS THE FIRST DERIVATIVE OF "FCN"
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C M            --> NUMBER OF ROWS IN A
C N            --> NUMBER OF COLUMNS IN A; DIMENSION OF PROBLEM
C XPLS(N)      --> NEW ITERATE:  X[K]
C FCN          --> NAME OF SUBROUTINE TO EVALUATE FUNCTION
C FPLS(M)      --> _M=1 (OPTIMIZATION) FUNCTION VALUE AT NEW ITERATE:
C                       FCN(XPLS)
C                  _M=N (OPTIMIZATION) VALUE OF FIRST DERIVATIVE
C                       (GRADIENT) GIVEN BY USER FUNCTION FCN
C                  _M=N (SYSTEMS)  FUNCTION VALUE OF ASSOCIATED
C                       MINIMIZATION FUNCTION
C A(NR,N)     <--  FINITE DIFFERENCE APPROXIMATION (SEE NOTE).  ONLY
C                  LOWER TRIANGULAR MATRIX AND DIAGONAL ARE RETURNED
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C RNOISE       --> RELATIVE NOISE IN FCN [F(X)]
C FHAT(M)      --> WORKSPACE
C ICASE        --> =1 OPTIMIZATION (GRADIENT)
C                  =2 SYSTEMS
C                  =3 OPTIMIZATION (HESSIAN)
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
C INTERNAL VARIABLES
C ------------------
C STEPSZ - STEPSIZE IN THE J-TH VARIABLE DIRECTION
C
      DIMENSION XPLS(N),FPLS(M)
      DIMENSION FHAT(M)
      DIMENSION SX(N)
      DIMENSION A(NR,*)
C
C FIND J-TH COLUMN OF A
C EACH COLUMN IS DERIVATIVE OF F(FCN) WITH RESPECT TO XPLS(J)
C
      DO 30 J=1,N
        STEPSZ=SQRT(RNOISE)*MAX(ABS(XPLS(J)),1./SX(J))
        XTMPJ=XPLS(J)
        XPLS(J)=XTMPJ+STEPSZ
        CALL FCN(N,XPLS,FHAT)
        XPLS(J)=XTMPJ
        DO 20 I=1,M
          A(I,J)=(FHAT(I)-FPLS(I))/STEPSZ
   20   CONTINUE
   30 CONTINUE
      IF(ICASE.NE.3) RETURN
C
C IF COMPUTING HESSIAN, A MUST BE SYMMETRIC
C
      IF(N.EQ.1) RETURN
      NM1=N-1
      DO 50 J=1,NM1
        JP1=J+1
        DO 40 I=JP1,M
          A(I,J)=(A(I,J)+A(J,I))/2.0
   40   CONTINUE
   50 CONTINUE
      RETURN
      END
      SUBROUTINE GRDCHZ(N,X,FCN,F,G,TYPSIZ,SX,FSCALE,RNF,
     +     ANALTL,WRK1,MSG,IPRZZ)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C CHECK ANALYTIC GRADIENT AGAINST ESTIMATED GRADIENT
C
C PARAMETERS
C ----------
C N            --> DIMENSION OF PROBLEM
C X(N)         --> ESTIMATE TO A ROOT OF FCN
C FCN          --> NAME OF SUBROUTINE TO EVALUATE OPTIMIZATION FUNCTION
C                  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C                       FCN:  R(N) --> R(1)
C F            --> FUNCTION VALUE:  FCN(X)
C G(N)         --> GRADIENT:  G(X)
C TYPSIZ(N)    --> TYPICAL SIZE FOR EACH COMPONENT OF X
C SX(N)        --> DIAGONAL SCALING MATRIX:  SX(I)=1./TYPSIZ(I)
C FSCALE       --> ESTIMATE OF SCALE OF OBJECTIVE FUNCTION FCN
C RNF          --> RELATIVE NOISE IN OPTIMIZATION FUNCTION FCN
C ANALTL       --> TOLERANCE FOR COMPARISON OF ESTIMATED AND
C                  ANALYTICAL GRADIENTS
C WRK1(N)      --> WORKSPACE
C MSG         <--  MESSAGE OR ERROR CODE
C                    ON OUTPUT: =-21, PROBABLE CODING ERROR OF GRADIENT
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C
      DIMENSION X(N),G(N)
      DIMENSION SX(N),TYPSIZ(N)
      DIMENSION WRK1(N)
      EXTERNAL FCN
      DIMENSION FTEMP(1)
      DIMENSION WRK(1)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C COMPUTE FIRST ORDER FINITE DIFFERENCE GRADIENT AND COMPARE TO
C ANALYTIC GRADIENT.
C
      FTEMP(1)=F
      CALL FSTOFZ(1,1,N,X,FCN,FTEMP,WRK1,SX,RNF,WRK,1)
      F=FTEMP(1)
      KER=0
      DO 5 I=1,N
        GS=MAX(ABS(F),FSCALE)/MAX(ABS(X(I)),TYPSIZ(I))
        IF(ABS(G(I)-WRK1(I)).GT.MAX(ABS(G(I)),GS)*ANALTL) KER=1
    5 CONTINUE
      IF(KER.NE.0) THEN
        WRITE(ICOUT,999)
  999   FORMAT(1X)
        CALL DPWRST('XXX','BUG ')
        WRITE(ICOUT,901)
  901   FORMAT('GRDCHZ    PROBABLE ERROR IN CODING OF ANALYTIC ',
     1         'GRADIENT FUNCTION.')
        CALL DPWRST('XXX','BUG ')
        WRITE(ICOUT,903)
  903   FORMAT('GRDCHZ     COMP',12X,'ANALYTIC',12X,'ESTIMATE')
        CALL DPWRST('XXX','BUG ')
        DO920I=1,N
          WRITE(ICOUT,921) I,G(I),WRK1(I)
  921     FORMAT(' GRDCHZ    ',I5,3X,E20.13,3X,E20.13)
          CALL DPWRST('XXX','BUG ')
  920   CONTINUE
        MSG=-21
      ENDIF
      RETURN
      END
      SUBROUTINE HESCHZ(NR,N,X,FCN,D1FCN,D2FCN,F,G,A,TYPSIZ,SX,RNF,
     +     ANALTL,IAGFLG,UDIAG,WRK1,WRK2,MSG,IPRZZ)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C CHECK ANALYTIC HESSIAN AGAINST ESTIMATED HESSIAN
C  (THIS MAY BE DONE ONLY IF THE USER SUPPLIED ANALYTIC HESSIAN
C   D2FCN FILLS ONLY THE LOWER TRIANGULAR PART AND DIAGONAL OF A)
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> ESTIMATE TO A ROOT OF FCN
C FCN          --> NAME OF SUBROUTINE TO EVALUATE OPTIMIZATION FUNCTION
C                  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C                       FCN:  R(N) --> R(1)
C D1FCN        --> NAME OF SUBROUTINE TO EVALUATE GRADIENT OF FCN.
C                  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C D2FCN        --> NAME OF SUBROUTINE TO EVALUATE HESSIAN OF FCN.
C                  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C F            --> FUNCTION VALUE:  FCN(X)
C G(N)        <--  GRADIENT:  G(X)
C A(N,N)      <--  ON EXIT:  HESSIAN IN LOWER TRIANGULAR PART AND DIAG
C TYPSIZ(N)    --> TYPICAL SIZE FOR EACH COMPONENT OF X
C SX(N)        --> DIAGONAL SCALING MATRIX:  SX(I)=1./TYPSIZ(I)
C RNF          --> RELATIVE NOISE IN OPTIMIZATION FUNCTION FCN
C ANALTL       --> TOLERANCE FOR COMPARISON OF ESTIMATED AND
C                  ANALYTICAL GRADIENTS
C IAGFLG       --> =1 IF ANALYTIC GRADIENT SUPPLIED
C UDIAG(N)     --> WORKSPACE
C WRK1(N)      --> WORKSPACE
C WRK2(N)      --> WORKSPACE
C MSG         <--> MESSAGE OR ERROR CODE
C                    ON INPUT : IF =1XX DO NOT COMPARE ANAL + EST HESS
C                    ON OUTPUT: =-22, PROBABLE CODING ERROR OF HESSIAN
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),G(N),A(NR,*)
      DIMENSION TYPSIZ(N),SX(N)
      DIMENSION UDIAG(N),WRK1(N),WRK2(N)
      EXTERNAL FCN,D1FCN
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C COMPUTE FINITE DIFFERENCE APPROXIMATION A TO THE HESSIAN.
C
      IF(IAGFLG.EQ.1) CALL FSTOFZ(NR,N,N,X,D1FCN,G,A,SX,RNF,WRK1,3)
      IF(IAGFLG.NE.1) CALL SNDOFZ(NR,N,X,FCN,F,A,SX,RNF,WRK1,WRK2)
C
      KER=0
C
C COPY LOWER TRIANGULAR PART OF "A" TO UPPER TRIANGULAR PART
C AND DIAGONAL OF "A" TO UDIAG
C
      DO 30 J=1,N
        UDIAG(J)=A(J,J)
        IF(J.EQ.N) GO TO 30
        JP1=J+1
        DO 25 I=JP1,N
          A(J,I)=A(I,J)
   25   CONTINUE
   30 CONTINUE
C
C COMPUTE ANALYTIC HESSIAN AND COMPARE TO FINITE DIFFERENCE
C APPROXIMATION.
C
      CALL D2FCN(NR,N,X,A)
      DO 40 J=1,N
        HS=MAX(ABS(G(J)),1.0D0)/MAX(ABS(X(J)),TYPSIZ(J))
        IF(ABS(A(J,J)-UDIAG(J)).GT.MAX(ABS(UDIAG(J)),HS)*ANALTL)
     +       KER=1
        IF(J.EQ.N) GO TO 40
        JP1=J+1
        DO 35 I=JP1,N
          IF(ABS(A(I,J)-A(J,I)).GT.MAX(ABS(A(I,J)),HS)*ANALTL) KER=1
   35   CONTINUE
   40 CONTINUE
C
      IF(KER.EQ.0) GO TO 90
        WRITE(IPR,901)
  901   FORMAT('HESCHZ    PROBABLE ERROR IN CODING OF ANALYTIC ',
     1         'HESSIAN FUNCTION.')
        CALL DPWRST('XXX','BUG ')
        WRITE(ICOUT,903)
  903   FORMAT(' HESCHZ      ROW  COL',14X,'ANALYTIC',14X,
     1         '(ESTIMATE)')
        CALL DPWRST('XXX','BUG ')
        DO 50 I=1,N
          IF(I.EQ.1) GO TO 45
          IM1=I-1
          DO 43 J=1,IM1
            WRITE(ICOUT,902) I,J,A(I,J),A(J,I)
  902       FORMAT(' HESCHZ    ',2I5,2X,E20.13,2X,'(',E20.13,')')
            CALL DPWRST('XXX','BUG ')
   43     CONTINUE
   45     WRITE(ICOUT,902) I,I,A(I,I),UDIAG(I)
          CALL DPWRST('XXX','BUG ')
   50   CONTINUE
        MSG=-22
C     ENDIF
   90 CONTINUE
      RETURN
      END
      SUBROUTINE HOOKDZ(NR,N,X,F,G,A,UDIAG,P,XPLS,FPLS,FCN,SX,STEPMX,
     +     STEPTL,DLT,IRETCD,MXTAKE,AMU,DLTP,PHI,PHIP0,
     +     SC,XPLSP,WRK0,EPSM,ITNCNT,IPRZZ)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C FIND A NEXT NEWTON ITERATE (XPLS) BY THE MORE-HEBDON METHOD
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> OLD ITERATE X[K-1]
C F            --> FUNCTION VALUE AT OLD ITERATE, F(X)
C G(N)         --> GRADIENT AT OLD ITERATE, G(X), OR APPROXIMATE
C A(N,N)       --> CHOLESKY DECOMPOSITION OF HESSIAN IN LOWER
C                  TRIANGULAR PART AND DIAGONAL.
C                  HESSIAN IN UPPER TRIANGULAR PART AND UDIAG.
C UDIAG(N)     --> DIAGONAL OF HESSIAN IN A(.,.)
C P(N)         --> NEWTON STEP
C XPLS(N)     <--  NEW ITERATE X[K]
C FPLS        <--  FUNCTION VALUE AT NEW ITERATE, F(XPLS)
C FCN          --> NAME OF SUBROUTINE TO EVALUATE FUNCTION
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C STEPMX       --> MAXIMUM ALLOWABLE STEP SIZE
C STEPTL       --> RELATIVE STEP SIZE AT WHICH SUCCESSIVE ITERATES
C                  CONSIDERED CLOSE ENOUGH TO TERMINATE ALGORITHM
C DLT         <--> TRUST REGION RADIUS
C IRETCD      <--  RETURN CODE
C                    =0 SATISFACTORY XPLS FOUND
C                    =1 FAILED TO FIND SATISFACTORY XPLS SUFFICIENTLY
C                       DISTINCT FROM X
C MXTAKE      <--  BOOLEAN FLAG INDICATING STEP OF MAXIMUM LENGTH USED
C AMU         <--> [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C DLTP        <--> [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C PHI         <--> [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C PHIP0       <--> [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C SC(N)        --> WORKSPACE
C XPLSP(N)     --> WORKSPACE
C WRK0(N)      --> WORKSPACE
C EPSM         --> MACHINE EPSILON
C ITNCNT       --> ITERATION COUNT
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),G(N),P(N),XPLS(N),SX(N)
      DIMENSION A(NR,*),UDIAG(N)
      DIMENSION SC(N),XPLSP(N),WRK0(N)
      LOGICAL MXTAKE,NWTAKE
      LOGICAL FSTIME
      EXTERNAL FCN
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      IRETCD=4
      FSTIME=.TRUE.
      TMP=0.
      DO 5 I=1,N
        TMP=TMP+SX(I)*SX(I)*P(I)*P(I)
    5 CONTINUE
      RNWTLN=SQRT(TMP)
CCCCC WRITE(IPR,954) RNWTLN
C
      IF(ITNCNT.GT.1) GO TO 100
C     IF(ITNCNT.EQ.1)
C     THEN
        AMU=0.
C
C       IF FIRST ITERATION AND TRUST REGION NOT PROVIDED BY USER,
C       COMPUTE INITIAL TRUST REGION.
C
        IF(DLT.NE. (-1.)) GO TO 100
C       IF(DLT.EQ. (-1.))
C       THEN
          ALPHA=0.
          DO 10 I=1,N
            ALPHA=ALPHA+(G(I)*G(I))/(SX(I)*SX(I))
   10     CONTINUE
          BETA=0.0
          DO 30 I=1,N
            TMP=0.
            DO 20 J=I,N
              TMP=TMP + (A(J,I)*G(J))/(SX(J)*SX(J))
   20       CONTINUE
            BETA=BETA+TMP*TMP
   30     CONTINUE
          DLT=ALPHA*SQRT(ALPHA)/BETA
          DLT = MIN(DLT, STEPMX)
CCCCC     WRITE(IPR,950)
CCCCC     WRITE(IPR,951) ALPHA,BETA,DLT
C       ENDIF
C     ENDIF
C
  100 CONTINUE
C
C FIND NEW STEP BY MORE-HEBDON ALGORITHM
      CALL HOOKSZ(NR,N,G,A,UDIAG,P,SX,RNWTLN,DLT,AMU,
     +     DLTP,PHI,PHIP0,FSTIME,SC,NWTAKE,WRK0,EPSM,IPRZZ)
      DLTP=DLT
C
C CHECK NEW POINT AND UPDATE TRUST REGION
      CALL TREGUZ(NR,N,X,F,G,A,FCN,SC,SX,NWTAKE,STEPMX,STEPTL,
     +         DLT,IRETCD,XPLSP,FPLSP,XPLS,FPLS,MXTAKE,IPRZZ,3,UDIAG)
      IF(IRETCD.LE.1) RETURN
      GO TO 100
C
CC950 FORMAT(43H HOOKDZ    INITIAL TRUST REGION NOT GIVEN. ,
CCCCC+       21H COMPUTE CAUCHY STEP.)
CC951 FORMAT(18H HOOKDZ    ALPHA =,E20.13/
CCCCC+       18H HOOKDZ    BETA  =,E20.13/
CCCCC+       18H HOOKDZ    DLT   =,E20.13)
CC952 FORMAT(28H HOOKDZ    CURRENT STEP (SC))
CC954 FORMAT(18H0HOOKDZ    RNWTLN=,E20.13)
CC955 FORMAT(14H HOOKDZ       ,5(E20.13,3X))
      END
      SUBROUTINE HOOKSZ(NR,N,G,A,UDIAG,P,SX,RNWTLN,DLT,AMU,
     +     DLTP,PHI,PHIP0,FSTIME,SC,NWTAKE,WRK0,EPSM,IPRZZ)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C FIND NEW STEP BY MORE-HEBDON ALGORITHM
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C G(N)         --> GRADIENT AT CURRENT ITERATE, G(X)
C A(N,N)       --> CHOLESKY DECOMPOSITION OF HESSIAN IN
C                  LOWER TRIANGULAR PART AND DIAGONAL.
C                  HESSIAN OR APPROX IN UPPER TRIANGULAR PART
C UDIAG(N)     --> DIAGONAL OF HESSIAN IN A(.,.)
C P(N)         --> NEWTON STEP
C SX(N)        --> DIAGONAL SCALING MATRIX FOR N
C RNWTLN       --> NEWTON STEP LENGTH
C DLT         <--> TRUST REGION RADIUS
C AMU         <--> [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C DLTP         --> TRUST REGION RADIUS AT LAST EXIT FROM THIS ROUTINE
C PHI         <--> [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C PHIP0       <--> [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C FSTIME      <--> BOOLEAN. =.TRUE. IF FIRST ENTRY TO THIS ROUTINE
C                  DURING K-TH ITERATION
C SC(N)       <--  CURRENT STEP
C NWTAKE      <--  BOOLEAN, =.TRUE. IF NEWTON STEP TAKEN
C WRK0         --> WORKSPACE
C EPSM         --> MACHINE EPSILON
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION G(N),P(N),SX(N),SC(N),WRK0(N)
      DIMENSION A(NR,*),UDIAG(N)
      LOGICAL NWTAKE,DONE
      LOGICAL FSTIME
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C HI AND ALO ARE CONSTANTS USED IN THIS ROUTINE.
C CHANGE HERE IF OTHER VALUES ARE TO BE SUBSTITUTED.
      IPR=IPR
      HI=1.5
      ALO=.75
C -----
      IF(RNWTLN.GT.HI*DLT) GO TO 15
C     IF(RNWTLN.LE.HI*DLT)
C     THEN
C
C       TAKE NEWTON STEP
C
        NWTAKE=.TRUE.
        DO 10 I=1,N
          SC(I)=P(I)
   10   CONTINUE
        DLT=MIN(DLT,RNWTLN)
        AMU=0.
CCCCC   WRITE(IPR,951)
        RETURN
C     ELSE
C
C       NEWTON STEP NOT TAKEN
C
   15   CONTINUE
CCCCC   WRITE(IPR,952)
        NWTAKE=.FALSE.
        IF(AMU.LE.0.) GO TO 20
C       IF(AMU.GT.0.)
C       THEN
          AMU=AMU- (PHI+DLTP) *((DLTP-DLT)+PHI)/(DLT*PHIP)
C$        WRITE(IPR,956) AMU
C       ENDIF
   20   CONTINUE
        PHI=RNWTLN-DLT
        IF(.NOT.FSTIME) GO TO 28
C       IF(FSTIME)
C       THEN
          DO 25 I=1,N
            WRK0(I)=SX(I)*SX(I)*P(I)
   25     CONTINUE
C
C         SOLVE L*Y = (SX**2)*P
C
          CALL FORSLZ(NR,N,A,WRK0,WRK0)
          PHIP0=-DNRM2(N,WRK0,1)**2/RNWTLN
          FSTIME=.FALSE.
C       ENDIF
   28   PHIP=PHIP0
        AMULO=-PHI/PHIP
        AMUUP=0.0
        DO 30 I=1,N
          AMUUP=AMUUP+(G(I)*G(I))/(SX(I)*SX(I))
   30   CONTINUE
        AMUUP=SQRT(AMUUP)/DLT
        DONE=.FALSE.
CCCCC   WRITE(IPR,956) AMU
CCCCC   WRITE(IPR,959) PHI
CCCCC   WRITE(IPR,960) PHIP
CCCCC   WRITE(IPR,957) AMULO
CCCCC   WRITE(IPR,958) AMUUP
C
C       TEST VALUE OF AMU; GENERATE NEXT AMU IF NECESSARY
C
  100   CONTINUE
        IF(DONE) RETURN
CCCCC   WRITE(IPR,962)
        IF(AMU.GE.AMULO .AND. AMU.LE.AMUUP) GO TO 110
C       IF(AMU.LT.AMULO .OR.  AMU.GT.AMUUP)
C       THEN
          AMU=MAX(SQRT(AMULO*AMUUP),AMUUP*1.0E-3)
CCCCC     WRITE(IPR,956) AMU
C       ENDIF
  110   CONTINUE
C
C       COPY (H,UDIAG) TO L
C       WHERE H <-- H+AMU*(SX**2) [DO NOT ACTUALLY CHANGE (H,UDIAG)]
        DO 130 J=1,N
          A(J,J)=UDIAG(J) + AMU*SX(J)*SX(J)
          IF(J.EQ.N) GO TO 130
          JP1=J+1
          DO 120 I=JP1,N
            A(I,J)=A(J,I)
  120     CONTINUE
  130   CONTINUE
C
C       FACTOR H=L(L+)
C
        CALL CHOLDZ(NR,N,A,0.0D0,SQRT(EPSM),ADDMAX)
C
C       SOLVE H*P = L(L+)*SC = -G
C
        DO 140 I=1,N
          WRK0(I)=-G(I)
  140   CONTINUE
        CALL LLTSLZ(NR,N,A,SC,WRK0)
CCCCC   WRITE(IPR,955)
CCCCC   WRITE(IPR,963) (SC(I),I=1,N)
C
C       RESET H.  NOTE SINCE UDIAG HAS NOT BEEN DESTROYED WE NEED DO
C       NOTHING HERE.  H IS IN THE UPPER PART AND IN UDIAG, STILL INTACT
C
        STEPLN=0.
        DO 150 I=1,N
          STEPLN=STEPLN + SX(I)*SX(I)*SC(I)*SC(I)
  150   CONTINUE
        STEPLN=SQRT(STEPLN)
        PHI=STEPLN-DLT
        DO 160 I=1,N
          WRK0(I)=SX(I)*SX(I)*SC(I)
  160   CONTINUE
        CALL FORSLZ(NR,N,A,WRK0,WRK0)
        PHIP=-DNRM2(N,WRK0,1)**2/STEPLN
CCCCC   WRITE(IPR,961) DLT,STEPLN
CCCCC   WRITE(IPR,959) PHI
CCCCC   WRITE(IPR,960) PHIP
        IF((ALO*DLT.GT.STEPLN .OR. STEPLN.GT.HI*DLT) .AND.
     +       (AMUUP-AMULO.GT.0.)) GO TO 170
C       IF((ALO*DLT.LE.STEPLN .AND. STEPLN.LE.HI*DLT) .OR.
C            (AMUUP-AMULO.LE.0.))
C       THEN
C
C         SC IS ACCEPTABLE HOOKSZEP
C
CCCCC     WRITE(IPR,954)
          DONE=.TRUE.
          GO TO 100
C       ELSE
C
C         SC NOT ACCEPTABLE HOOKSZEP.  SELECT NEW AMU
C
  170     CONTINUE
CCCCC     WRITE(IPR,953)
          AMULO=MAX(AMULO,AMU-(PHI/PHIP))
          IF(PHI.LT.0.) AMUUP=MIN(AMUUP,AMU)
          AMU=AMU-(STEPLN*PHI)/(DLT*PHIP)
CCCCC     WRITE(IPR,956) AMU
CCCCC     WRITE(IPR,957) AMULO
CCCCC     WRITE(IPR,958) AMUUP
          GO TO 100
C       ENDIF
C     ENDIF
C
CC951 FORMAT(27H0HOOKSZ    TAKE NEWTON STEP)
CC952 FORMAT(32H0HOOKSZ    NEWTON STEP NOT TAKEN)
CC953 FORMAT(31H HOOKSZ    SC IS NOT ACCEPTABLE)
CC954 FORMAT(27H HOOKSZ    SC IS ACCEPTABLE)
CC955 FORMAT(28H HOOKSZ    CURRENT STEP (SC))
CC956 FORMAT(18H HOOKSZ    AMU   =,E20.13)
CC957 FORMAT(18H HOOKSZ    AMULO =,E20.13)
CC958 FORMAT(18H HOOKSZ    AMUUP =,E20.13)
CC959 FORMAT(18H HOOKSZ    PHI   =,E20.13)
CC960 FORMAT(18H HOOKSZ    PHIP  =,E20.13)
CC961 FORMAT(18H HOOKSZ    DLT   =,E20.13/
CCCCC+       18H HOOKSZ    STEPLN=,E20.13)
CC962 FORMAT(23H0HOOKSZ    FIND NEW AMU)
CC963 FORMAT(14H HOOKSZ       ,5(E20.13,3X))
      END
      SUBROUTINE HSNINZ(NR,N,A,SX,METHOD)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C PROVIDE INITIAL HESSIAN WHEN USING SECANT UPDATES
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(N,N)      <--  INITIAL HESSIAN (LOWER TRIANGULAR MATRIX)
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C METHOD       --> ALGORITHM TO USE TO SOLVE MINIMIZATION PROBLEM
C                    =1,2 FACTORED SECANT METHOD USED
C                    =3   UNFACTORED SECANT METHOD USED
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*),SX(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DO 100 J=1,N
        IF(METHOD.EQ.3) A(J,J)=SX(J)*SX(J)
        IF(METHOD.NE.3) A(J,J)=SX(J)
        IF(J.EQ.N) GO TO 100
        JP1=J+1
        DO 90 I=JP1,N
          A(I,J)=0.
   90   CONTINUE
  100 CONTINUE
      RETURN
      END
      SUBROUTINE LLTSLZ(NR,N,A,X,B)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C SOLVE AX=B WHERE A HAS THE FORM L(L-TRANSPOSE)
C BUT ONLY THE LOWER TRIANGULAR PART, L, IS STORED.
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(N,N)       --> MATRIX OF FORM L(L-TRANSPOSE).
C                  ON RETURN A IS UNCHANGED.
C X(N)        <--  SOLUTION VECTOR
C B(N)         --> RIGHT-HAND SIDE VECTOR
C
C NOTE
C ----
C IF B IS NOT REQUIRED BY CALLING PROGRAM, THEN
C B AND X MAY SHARE THE SAME STORAGE.
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*),X(N),B(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C FORWARD SOLVE, RESULT IN X
C
      CALL FORSLZ(NR,N,A,X,B)
C
C BACK SOLVE, RESULT IN X
C
      CALL BAKSLZ(NR,N,A,X,X)
      RETURN
      END
      SUBROUTINE LNSRCZ(N,X,F,G,P,XPLS,FPLS,FCN,MXTAKE,
     +   IRETCD,STEPMX,STEPTL,SX,IPRZZ)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C PURPOSE
C -------
C FIND A NEXT NEWTON ITERATE BY LINE SEARCH.
C
C PARAMETERS
C ----------
C N            --> DIMENSION OF PROBLEM
C X(N)         --> OLD ITERATE:   X[K-1]
C F            --> FUNCTION VALUE AT OLD ITERATE, F(X)
C G(N)         --> GRADIENT AT OLD ITERATE, G(X), OR APPROXIMATE
C P(N)         --> NON-ZERO NEWTON STEP
C XPLS(N)     <--  NEW ITERATE X[K]
C FPLS        <--  FUNCTION VALUE AT NEW ITERATE, F(XPLS)
C FCN          --> NAME OF SUBROUTINE TO EVALUATE FUNCTION
C IRETCD      <--  RETURN CODE
C MXTAKE      <--  BOOLEAN FLAG INDICATING STEP OF MAXIMUM LENGTH USED
C STEPMX       --> MAXIMUM ALLOWABLE STEP SIZE
C STEPTL       --> RELATIVE STEP SIZE AT WHICH SUCCESSIVE ITERATES
C                  CONSIDERED CLOSE ENOUGH TO TERMINATE ALGORITHM
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C
C INTERNAL VARIABLES
C ------------------
C SLN              NEWTON LENGTH
C RLN              RELATIVE LENGTH OF NEWTON STEP
C
      INTEGER N,IRETCD
      DIMENSION SX(N)
      DIMENSION X(N),G(N),P(N)
      DIMENSION XPLS(N)
      LOGICAL MXTAKE
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      IPR=IPR
      MXTAKE=.FALSE.
      IRETCD=2
CCCCC WRITE(IPR,954)
CCCCC WRITE(IPR,955) (P(I),I=1,N)
      TMP=0.0
      DO 5 I=1,N
        TMP=TMP+SX(I)*SX(I)*P(I)*P(I)
    5 CONTINUE
      SLN=SQRT(TMP)
      IF(SLN.LE.STEPMX) GO TO 10
C
C NEWTON STEP LONGER THAN MAXIMUM ALLOWED
        SCL=STEPMX/SLN
        CALL SCLMUZ(N,SCL,P,P)
        SLN=STEPMX
C$      WRITE(IPR,954)
C$      WRITE(IPR,955) (P(I),I=1,N)
   10 CONTINUE
      SLP=DDOT(N,G,1,P,1)
      RLN=0.
      DO 15 I=1,N
        RLN=MAX(RLN,ABS(P(I))/MAX(ABS(X(I)),1./SX(I)))
   15 CONTINUE
      RMNLMB=STEPTL/RLN
      ALMBDA=1.0
CCCCC WRITE(IPR,952) SLN,SLP,RMNLMB,STEPMX,STEPTL
C
C LOOP
C CHECK IF NEW ITERATE SATISFACTORY.  GENERATE NEW LAMBDA IF NECESSARY.
C
  100 CONTINUE
      IF(IRETCD.LT.2) RETURN
      DO 105 I=1,N
        XPLS(I)=X(I) + ALMBDA*P(I)
  105 CONTINUE
      CALL FCN(N,XPLS,FPLS)
CCCCC WRITE(IPR,950) ALMBDA
CCCCC WRITE(IPR,951)
CCCCC WRITE(IPR,955) (XPLS(I),I=1,N)
CCCCC WRITE(IPR,953) FPLS
      IF(FPLS.GT. F+SLP*1.E-4*ALMBDA) GO TO 130
C     IF(FPLS.LE. F+SLP*1.E-4*ALMBDA)
C     THEN
C
C SOLUTION FOUND
C
        IRETCD=0
        IF(ALMBDA.EQ.1.0 .AND. SLN.GT. .99*STEPMX) MXTAKE=.TRUE.
        GO TO 100
C
C SOLUTION NOT (YET) FOUND
C
C     ELSE
  130   IF(ALMBDA .GE. RMNLMB) GO TO 140
C       IF(ALMBDA .LT. RMNLMB)
C       THEN
C
C NO SATISFACTORY XPLS FOUND SUFFICIENTLY DISTINCT FROM X
C
          IRETCD=1
          GO TO 100
C       ELSE
C
C CALCULATE NEW LAMBDA
C
  140     IF(ALMBDA.NE.1.0) GO TO 150
C         IF(ALMBDA.EQ.1.0)
C         THEN
C
C FIRST BACKTRACK: QUADRATIC FIT
C
            TLMBDA=-SLP/(2.*(FPLS-F-SLP))
            GO TO 170
C         ELSE
C
C ALL SUBSEQUENT BACKTRACKS: CUBIC FIT
C
  150       T1=FPLS-F-ALMBDA*SLP
            T2=PFPLS-F-PLMBDA*SLP
            T3=1.0/(ALMBDA-PLMBDA)
            A=T3*(T1/(ALMBDA*ALMBDA) - T2/(PLMBDA*PLMBDA))
            B=T3*(T2*ALMBDA/(PLMBDA*PLMBDA)
     +           - T1*PLMBDA/(ALMBDA*ALMBDA) )
            DISC=B*B-3.0*A*SLP
            IF(DISC.LE. B*B) GO TO 160
C           IF(DISC.GT. B*B)
C           THEN
C
C ONLY ONE POSITIVE CRITICAL POINT, MUST BE MINIMUM
C
              TLMBDA=(-B+SIGN(1.0D0,A)*SQRT(DISC))/(3.0*A)
              GO TO 165
C           ELSE
C
C BOTH CRITICAL POINTS POSITIVE, FIRST IS MINIMUM
C
  160         TLMBDA=(-B-SIGN(1.0D0,A)*SQRT(DISC))/(3.0*A)
C           ENDIF
  165       IF(TLMBDA.GT. .5*ALMBDA) TLMBDA=.5*ALMBDA
C         ENDIF
  170     PLMBDA=ALMBDA
          PFPLS=FPLS
          IF(TLMBDA.GE. ALMBDA*.1) GO TO 180
C         IF(TLMBDA.LT.ALMBDA/10.)
C         THEN
            ALMBDA=ALMBDA*.1
            GO TO 190
C         ELSE
  180       ALMBDA=TLMBDA
C         ENDIF
C       ENDIF
C     ENDIF
  190 GO TO 100
CC950 FORMAT(18H LNSRCZ    ALMBDA=,E20.13)
CC951 FORMAT(29H LNSRCZ    NEW ITERATE (XPLS))
CC952 FORMAT(18H LNSRCZ    SLN   =,E20.13/
CC   +       18H LNSRCZ    SLP   =,E20.13/
CC   +       18H LNSRCZ    RMNLMB=,E20.13/
CC   +       18H LNSRCZ    STEPMX=,E20.13/
CC   +       18H LNSRCZ    STEPTL=,E20.13)
CC953 FORMAT(19H LNSRCZ    F(XPLS)=,E20.13)
CC954 FORMAT(26H0LNSRCZ    NEWTON STEP (P))
CC955 FORMAT(14H LNSRCZ       ,5(E20.13,3X))
      END
      SUBROUTINE MVMLTZ(NR,N,A,X,Y)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C COMPUTE Y=LX
C WHERE L IS A LOWER TRIANGULAR MATRIX STORED IN A
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(N,N)       --> LOWER TRIANGULAR (N*N) MATRIX
C X(N)         --> OPERAND VECTOR
C Y(N)        <--  RESULT VECTOR
C
C NOTE
C ----
C X AND Y CANNOT SHARE STORAGE
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*),X(N),Y(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DO 30 I=1,N
        SUM=0.
        DO 10 J=1,I
          SUM=SUM+A(I,J)*X(J)
   10   CONTINUE
        Y(I)=SUM
   30 CONTINUE
      RETURN
      END
      SUBROUTINE MVMLTY(NR,N,A,X,Y)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C COMPUTE Y=AX
C WHERE "A" IS A SYMMETRIC (N*N) MATRIX STORED IN ITS LOWER
C TRIANGULAR PART AND X,Y ARE N-VECTORS
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(N,N)       --> SYMMETRIC (N*N) MATRIX STORED IN
C                  LOWER TRIANGULAR PART AND DIAGONAL
C X(N)         --> OPERAND VECTOR
C Y(N)        <--  RESULT VECTOR
C
C NOTE
C ----
C X AND Y CANNOT SHARE STORAGE.
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*),X(N),Y(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DO 30 I=1,N
        SUM=0.
        DO 10 J=1,I
          SUM=SUM+A(I,J)*X(J)
   10   CONTINUE
        IF(I.EQ.N) GO TO 25
        IP1=I+1
        DO 20 J=IP1,N
          SUM=SUM+A(J,I)*X(J)
   20   CONTINUE
   25   Y(I)=SUM
   30 CONTINUE
      RETURN
      END
      SUBROUTINE MVMLTX(NR,N,A,X,Y)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C COMPUTE Y=(L+)X
C WHERE L IS A LOWER TRIANGULAR MATRIX STORED IN A
C (L-TRANSPOSE (L+) IS TAKEN IMPLICITLY)
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(NR,1)       --> LOWER TRIANGULAR (N*N) MATRIX
C X(N)         --> OPERAND VECTOR
C Y(N)        <--  RESULT VECTOR
C
C NOTE
C ----
C X AND Y CANNOT SHARE STORAGE
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*),X(N),Y(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DO 30 I=1,N
        SUM=0.
        DO 10 J=I,N
          SUM=SUM+A(J,I)*X(J)
   10   CONTINUE
        Y(I)=SUM
   30 CONTINUE
      RETURN
      END
      SUBROUTINE OPTCHZ(N,X,TYPSIZ,SX,FSCALE,GRADTL,ITNLIM,NDIGIT,EPSM,
     +     DLT,METHOD,IEXP,IAGFLG,IAHFLG,STEPMX,MSG,IPRZZ)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C CHECK INPUT FOR REASONABLENESS
C
C PARAMETERS
C ----------
C N            --> DIMENSION OF PROBLEM
C X(N)         --> ON ENTRY, ESTIMATE TO ROOT OF FCN
C TYPSIZ(N)   <--> TYPICAL SIZE OF EACH COMPONENT OF X
C SX(N)       <--  DIAGONAL SCALING MATRIX FOR X
C FSCALE      <--> ESTIMATE OF SCALE OF OBJECTIVE FUNCTION FCN
C GRADTL       --> TOLERANCE AT WHICH GRADIENT CONSIDERED CLOSE
C                  ENOUGH TO ZERO TO TERMINATE ALGORITHM
C ITNLIM      <--> MAXIMUM NUMBER OF ALLOWABLE ITERATIONS
C NDIGIT      <--> NUMBER OF GOOD DIGITS IN OPTIMIZATION FUNCTION FCN
C EPSM         --> MACHINE EPSILON
C DLT         <--> TRUST REGION RADIUS
C METHOD      <--> ALGORITHM INDICATOR
C IEXP        <--> EXPENSE FLAG
C IAGFLG      <--> =1 IF ANALYTIC GRADIENT SUPPLIED
C IAHFLG      <--> =1 IF ANALYTIC HESSIAN SUPPLIED
C STEPMX      <--> MAXIMUM STEP SIZE
C MSG         <--> MESSAGE AND ERROR CODE
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C
      DIMENSION X(N),TYPSIZ(N),SX(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C CHECK THAT PARAMETERS ONLY TAKE ON ACCEPTABLE VALUES.
C IF NOT, SET THEM TO DEFAULT VALUES.
      IF(METHOD.LT.1 .OR. METHOD.GT.3) METHOD=1
      IF(IAGFLG.NE.1) IAGFLG=0
      IF(IAHFLG.NE.1) IAHFLG=0
      IF(IEXP.NE.0) IEXP=1
      IF(MOD(MSG/2,2).EQ.1 .AND. IAGFLG.EQ.0) GO TO 830
      IF(MOD(MSG/4,2).EQ.1 .AND. IAHFLG.EQ.0) GO TO 835
C
C CHECK DIMENSION OF PROBLEM
C
      IF(N.LE.0) GO TO 805
      IF(N.EQ.1 .AND. MOD(MSG,2).EQ.0) GO TO 810
C
C COMPUTE SCALE MATRIX
C
      DO 10 I=1,N
        IF(TYPSIZ(I).EQ.0.) TYPSIZ(I)=1.0
        IF(TYPSIZ(I).LT.0.) TYPSIZ(I)=-TYPSIZ(I)
        SX(I)=1.0/TYPSIZ(I)
   10 CONTINUE
C
C CHECK MAXIMUM STEP SIZE
C
      IF (STEPMX .GT. 0.0) GO TO 20
      STPSIZ = 0.0
      DO 15 I = 1, N
         STPSIZ = STPSIZ + X(I)*X(I)*SX(I)*SX(I)
   15 CONTINUE
      STPSIZ = SQRT(STPSIZ)
      STEPMX = MAX(1.0E3*STPSIZ, 1.0D3)
   20 CONTINUE
C CHECK FUNCTION SCALE
      IF(FSCALE.EQ.0.) FSCALE=1.0
      IF(FSCALE.LT.0.) FSCALE=-FSCALE
C
C CHECK GRADIENT TOLERANCE
      IF(GRADTL.LT.0.) GO TO 815
C
C CHECK ITERATION LIMIT
      IF(ITNLIM.LE.0) GO TO 820
C
C CHECK NUMBER OF DIGITS OF ACCURACY IN FUNCTION FCN
      IF(NDIGIT.EQ.0) GO TO 825
      IF(NDIGIT.LT.0) NDIGIT=-LOG10(EPSM)
C
C CHECK TRUST REGION RADIUS
      IF(DLT.LE.0.) DLT=-1.0
      IF (DLT .GT. STEPMX) DLT = STEPMX
      RETURN
C
C ERROR EXITS
C
  805 WRITE(ICOUT,901) N
      CALL DPWRST('XXX','BUG ')
      MSG=-1
      GO TO 895
  810 WRITE(ICOUT,902)
      CALL DPWRST('XXX','BUG ')
      MSG=-2
      GO TO 895
  815 WRITE(ICOUT,903) GRADTL
      CALL DPWRST('XXX','BUG ')
      MSG=-3
      GO TO 895
  820 WRITE(ICOUT,904) ITNLIM
      CALL DPWRST('XXX','BUG ')
      MSG=-4
      GO TO 895
  825 WRITE(ICOUT,905) NDIGIT
      CALL DPWRST('XXX','BUG ')
      MSG=-5
      GO TO 895
  830 WRITE(ICOUT,906) MSG
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,916)IAGFLG
      CALL DPWRST('XXX','BUG ')
      MSG=-6
      GO TO 895
  835 WRITE(ICOUT,907) MSG
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,917)IAHFLG
      CALL DPWRST('XXX','BUG ')
      MSG=-7
  895 RETURN
  901 FORMAT('***** FROM OPTCHK    ILLEGAL DIMENSION, N =',I5)
  902 FORMAT(
     +'***** FROM OPTCHK    +++ WARNING +++  THIS PACKAGE IS ',
     +'INEFFICIENT FOR PROBLEMS OF SIZE N=1.')
  903 FORMAT(
     +'***** FROM OPTCHK    ILLEGAL TOLERANCE.  GRADTL = ',E20.13)
  904 FORMAT(
     +'***** FROM OPTCHK    ILLEGAL ITERATION LIMIT.  ITNLIM = ',I5)
  905 FORMAT(
     +'***** FROM OPTCHK    MINIMIZATION FUNCTION HAS NO GOOD DIGITS.'
     +,'  NDIGIT = ',I5)
  906 FORMAT(
     +'***** FROM OPTCHK    USER REQUESTS THAT ANALYTIC GRADIENT BE',
     +' ACCEPTED AS PROPERLY CODED (MSG =',I5)
  916 FORMAT(
     +'                     BUT ANALYTIC GRADIENT NOT SUPPLIED',
     +'(IAGFLG = ',I5,'.')
  907 FORMAT(
     +'***** FROM OPTCHK    USER REQUESTS THAT ANALYTIC HESSIAN BE',
     +' ACCEPTED AS PROPERLY CODED (MSG =',I5)
  917 FORMAT(
     +'                     BUT ANALYTIC HESSIAN NOT SUPPLIED',
     +'(IAHFLG = ',I5,'.')
      END
      SUBROUTINE OPTDRZ(NR,N,X,FCN,D1FCN,D2FCN,TYPSIZ,FSCALE,
     +     METHOD,IEXP,MSG,NDIGIT,ITNLIM,IAGFLG,IAHFLG,IPRZZ,
     +     DLT,GRADTL,STEPMX,STEPTL,
     +     XPLS,FPLS,GPLS,ITRMCD,
     +     A,UDIAG,G,P,SX,WRK0,WRK1,WRK2,WRK3)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C DRIVER FOR NON-LINEAR OPTIMIZATION PROBLEM
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> ON ENTRY: ESTIMATE TO A ROOT OF FCN
C FCN          --> NAME OF SUBROUTINE TO EVALUATE OPTIMIZATION FUNCTION
C                  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C                            FCN: R(N) --> R(1)
C D1FCN        --> (OPTIONAL) NAME OF SUBROUTINE TO EVALUATE GRADIENT
C                  OF FCN.  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C D2FCN        --> (OPTIONAL) NAME OF SUBROUTINE TO EVALUATE HESSIAN OF
C                  OF FCN.  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C TYPSIZ(N)    --> TYPICAL SIZE FOR EACH COMPONENT OF X
C FSCALE       --> ESTIMATE OF SCALE OF OBJECTIVE FUNCTION
C METHOD       --> ALGORITHM TO USE TO SOLVE MINIMIZATION PROBLEM
C                    =1 LINE SEARCH
C                    =2 DOUBLE DOGLEG
C                    =3 MORE-HEBDON
C IEXP         --> =1 IF OPTIMIZATION FUNCTION FCN IS EXPENSIVE TO
C                  EVALUATE, =0 OTHERWISE.  IF SET THEN HESSIAN WILL
C                  BE EVALUATED BY SECANT UPDATE INSTEAD OF
C                  ANALYTICALLY OR BY FINITE DIFFERENCES
C MSG         <--> ON INPUT:  (.GT.0) MESSAGE TO INHIBIT CERTAIN
C                    AUTOMATIC CHECKS
C                  ON OUTPUT: (.LT.0) ERROR CODE; =0 NO ERROR
C NDIGIT       --> NUMBER OF GOOD DIGITS IN OPTIMIZATION FUNCTION FCN
C ITNLIM       --> MAXIMUM NUMBER OF ALLOWABLE ITERATIONS
C IAGFLG       --> =1 IF ANALYTIC GRADIENT SUPPLIED
C IAHFLG       --> =1 IF ANALYTIC HESSIAN SUPPLIED
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C DLT          --> TRUST REGION RADIUS
C GRADTL       --> TOLERANCE AT WHICH GRADIENT CONSIDERED CLOSE
C                  ENOUGH TO ZERO TO TERMINATE ALGORITHM
C STEPMX       --> MAXIMUM ALLOWABLE STEP SIZE
C STEPTL       --> RELATIVE STEP SIZE AT WHICH SUCCESSIVE ITERATES
C                  CONSIDERED CLOSE ENOUGH TO TERMINATE ALGORITHM
C XPLS(N)     <--> ON EXIT:  XPLS IS LOCAL MINIMUM
C FPLS        <--> ON EXIT:  FUNCTION VALUE AT SOLUTION, XPLS
C GPLS(N)     <--> ON EXIT:  GRADIENT AT SOLUTION XPLS
C ITRMCD      <--  TERMINATION CODE
C A(N,N)       --> WORKSPACE FOR HESSIAN (OR ESTIMATE)
C                  AND ITS CHOLESKY DECOMPOSITION
C UDIAG(N)     --> WORKSPACE [FOR DIAGONAL OF HESSIAN]
C G(N)         --> WORKSPACE (FOR GRADIENT AT CURRENT ITERATE)
C P(N)         --> WORKSPACE FOR STEP
C SX(N)        --> WORKSPACE (FOR DIAGONAL SCALING MATRIX)
C WRK0(N)      --> WORKSPACE
C WRK1(N)      --> WORKSPACE
C WRK2(N)      --> WORKSPACE
C WRK3(N)      --> WORKSPACE
C
C
C INTERNAL VARIABLES
C ------------------
C ANALTL           TOLERANCE FOR COMPARISON OF ESTIMATED AND
C                  ANALYTICAL GRADIENTS AND HESSIANS
C EPSM             MACHINE EPSILON
C F                FUNCTION VALUE: FCN(X)
C ITNCNT           CURRENT ITERATION, K
C RNF              RELATIVE NOISE IN OPTIMIZATION FUNCTION FCN.
C                       NOISE=10.**(-NDIGIT)
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),XPLS(N),G(N),GPLS(N),P(N)
      DIMENSION TYPSIZ(N),SX(N)
      DIMENSION A(NR,*),UDIAG(N)
      DIMENSION WRK0(N),WRK1(N),WRK2(N),WRK3(N)
      LOGICAL MXTAKE,NOUPDT
      EXTERNAL FCN,D1FCN,D2FCN
      DIMENSION FTEMP(1)
      DIMENSION WRK(1)
C
      INCLUDE 'DPCOF2.INC'
      REAL R1MACH
      INCLUDE 'DPCOMC.INC'
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C INITIALIZATION
C --------------
      DO 10 I=1,N
        P(I)=0.
   10 CONTINUE
      ITNCNT=0
      IRETCD=-1
      EPSM=D1MACH(4)
      CALL OPTCHZ(N,X,TYPSIZ,SX,FSCALE,GRADTL,ITNLIM,NDIGIT,EPSM,
     +     DLT,METHOD,IEXP,IAGFLG,IAHFLG,STEPMX,MSG,IPRZZ)
      IF(MSG.LT.0) RETURN
      RNF=MAX(10.0D0**(-NDIGIT),EPSM)
      ANALTL=MAX(1.0D-2,SQRT(RNF))
C
      IF(MOD(MSG/8,2).EQ.1) GO TO 15
      WRITE(ICOUT,901)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,900) (TYPSIZ(I),I=1,N)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,902)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,900) (SX(I),I=1,N)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,903) FSCALE
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,904) NDIGIT
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,914) IAGFLG
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,916)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,924) IAHFLG
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,926)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,934) IEXP
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,936) 
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,944) METHOD
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,946)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,954) ITNLIM
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,964) EPSM
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,905) STEPMX
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,915) STEPTL
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,925) GRADTL
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,935) DLT
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,945) RNF
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,955) ANALTL
      CALL DPWRST('XXX','BUG ')
   15 CONTINUE
C
C EVALUATE FCN(X)
C
      CALL FCN(N,X,F)
C
C EVALUATE ANALYTIC OR FINITE DIFFERENCE GRADIENT AND CHECK ANALYTIC
C GRADIENT, IF REQUESTED.
C
      IF (IAGFLG .EQ. 1) GO TO 20
C     IF (IAGFLG .EQ. 0)
C     THEN
        FTEMP(1)=F
        CALL FSTOFZ (1, 1, N, X, FCN, FTEMP, G, SX, RNF, WRK, 1)
        F=FTEMP(1)
        GO TO 25
C
   20 CALL D1FCN (N, X, G)
      IF (MOD(MSG/2,2) .EQ. 1) GO TO 25
C     IF (MOD(MSG/2,2).EQ.0)
C     THEN
        CALL GRDCHZ (N, X, FCN, F, G, TYPSIZ, SX, FSCALE,
     1    RNF, ANALTL, WRK1, MSG, IPR)
        IF (MSG .LT. 0) RETURN
   25 CONTINUE
C
      CALL OPTSTZ(N,X,F,G,WRK1,ITNCNT,ICSCMX,
     +            ITRMCD,GRADTL,STEPTL,SX,FSCALE,ITNLIM,IRETCD,MXTAKE,
     +            IPR,MSG)
      IF(ITRMCD.NE.0) GO TO 700
C
      IF(IEXP.NE.1) GO TO 80
C
C IF OPTIMIZATION FUNCTION EXPENSIVE TO EVALUATE (IEXP=1), THEN
C HESSIAN WILL BE OBTAINED BY SECANT UPDATES.  GET INITIAL HESSIAN.
C
      CALL HSNINZ(NR,N,A,SX,METHOD)
      GO TO 90
   80 CONTINUE
C
C EVALUATE ANALYTIC OR FINITE DIFFERENCE HESSIAN AND CHECK ANALYTIC
C HESSIAN IF REQUESTED (ONLY IF USER-SUPPLIED ANALYTIC HESSIAN
C ROUTINE D2FCN FILLS ONLY LOWER TRIANGULAR PART AND DIAGONAL OF A).
C
      IF (IAHFLG .EQ. 1) GO TO 82
C     IF (IAHFLG .EQ. 0)
C     THEN
         IF (IAGFLG .EQ. 1) CALL FSTOFZ (NR, N, N, X, D1FCN, G, A, SX,
     1      RNF, WRK1, 3)
         IF (IAGFLG .NE. 1) CALL SNDOFZ (NR, N, X, FCN, F, A, SX, RNF,
     1      WRK1, WRK2)
         GO TO 88
C
C     ELSE
   82    IF (MOD(MSG/4,2).EQ.0) GO TO 85
C        IF (MOD(MSG/4, 2) .EQ. 1)
C        THEN
            CALL D2FCN (NR, N, X, A)
            GO TO 88
C
C        ELSE
   85       CALL HESCHZ (NR, N, X, FCN, D1FCN, D2FCN, F, G, A, TYPSIZ,
     1         SX, RNF, ANALTL, IAGFLG, UDIAG, WRK1, WRK2, MSG, IPR)
C
C           HESCHZ EVALUATES D2FCN AND CHECKS IT AGAINST THE FINITE
C           DIFFERENCE HESSIAN WHICH IT CALCULATES BY CALLING FSTOFZ
C           (IF IAGFLG .EQ. 1) OR SNDOFZ (OTHERWISE).
C
            IF (MSG .LT. 0) RETURN
   88 CONTINUE
C
   90 IF(MOD(MSG/8,2).EQ.0)
     +     CALL RESULZ(NR,N,X,F,G,A,P,ITNCNT,1,IPRZZ)
C
C
C ITERATION
C ---------
  100 ITNCNT=ITNCNT+1
C
C FIND PERTURBED LOCAL MODEL HESSIAN AND ITS LL+ DECOMPOSITION
C (SKIP THIS STEP IF LINE SEARCH OR DOGSTEP TECHNIQUES BEING USED WITH
C SECANT UPDATES.  CHOLESKY DECOMPOSITION L ALREADY OBTAINED FROM
C SECFAZ.)
C
      IF(IEXP.EQ.1 .AND. METHOD.NE.3) GO TO 105
  103   CALL CHLHSZ(NR,N,A,EPSM,SX,UDIAG)
  105 CONTINUE
C
C SOLVE FOR NEWTON STEP:  AP=-G
C
      DO 110 I=1,N
        WRK1(I)=-G(I)
  110 CONTINUE
      CALL LLTSLZ(NR,N,A,P,WRK1)
C
C DECIDE WHETHER TO ACCEPT NEWTON STEP  XPLS=X + P
C OR TO CHOOSE XPLS BY A GLOBAL STRATEGY.
C
      IF (IAGFLG .NE. 0 .OR. METHOD .EQ. 1) GO TO 111
      DLTSAV = DLT
      IF (METHOD .EQ. 2) GO TO 111
      AMUSAV = AMU
      DLPSAV = DLTP
      PHISAV = PHI
      PHPSAV = PHIP0
  111 IF(METHOD.EQ.1)
     +     CALL LNSRCZ(N,X,F,G,P,XPLS,FPLS,FCN,MXTAKE,IRETCD,
     +     STEPMX,STEPTL,SX,IPRZZ)
      IF(METHOD.EQ.2)
     +     CALL DOGDRZ(NR,N,X,F,G,A,P,XPLS,FPLS,FCN,SX,STEPMX,
     +     STEPTL,DLT,IRETCD,MXTAKE,WRK0,WRK1,WRK2,WRK3,IPRZZ)
      IF(METHOD.EQ.3)
     +     CALL HOOKDZ(NR,N,X,F,G,A,UDIAG,P,XPLS,FPLS,FCN,SX,STEPMX,
     +     STEPTL,DLT,IRETCD,MXTAKE,AMU,DLTP,PHI,PHIP0,WRK0,
     +     WRK1,WRK2,EPSM,ITNCNT,IPRZZ)
C
C IF COULD NOT FIND SATISFACTORY STEP AND FORWARD DIFFERENCE
C GRADIENT WAS USED, RETRY USING CENTRAL DIFFERENCE GRADIENT.
C
      IF (IRETCD .NE. 1 .OR. IAGFLG .NE. 0) GO TO 112
C     IF (IRETCD .EQ. 1 .AND. IAGFLG .EQ. 0)
C     THEN
C
C        SET IAGFLG FOR CENTRAL DIFFERENCES
C
         IAGFLG = -1
         WRITE(ICOUT,906) ITNCNT
         CALL DPWRST('XXX','BUG ')
C
         CALL FSTOCZ (N, X, FCN, SX, RNF, G)
         IF (METHOD .EQ. 1) GO TO 105
         DLT = DLTSAV
         IF (METHOD .EQ. 2) GO TO 105
         AMU = AMUSAV
         DLTP = DLPSAV
         PHI = PHISAV
         PHIP0 = PHPSAV
         GO TO 103
C     ENDIF
C
C CALCULATE STEP FOR OUTPUT
C
  112 CONTINUE
      DO 114 I = 1, N
         P(I) = XPLS(I) - X(I)
  114 CONTINUE
C
C CALCULATE GRADIENT AT XPLS
C
      IF (IAGFLG .EQ. (-1)) GO TO 116
      IF (IAGFLG .EQ. 0) GO TO 118
C
C ANALYTIC GRADIENT
      CALL D1FCN (N, XPLS, GPLS)
      GO TO 120
C
C CENTRAL DIFFERENCE GRADIENT
  116 CALL FSTOCZ (N, XPLS, FCN, SX, RNF, GPLS)
      GO TO 120
C
C FORWARD DIFFERENCE GRADIENT
  118 CONTINUE
      FTEMP(1)=FPLS
      CALL FSTOFZ (1, 1, N, XPLS, FCN, FTEMP, GPLS, SX, RNF, WRK, 1)
      FPLS=FTEMP(1)
  120 CONTINUE
C
C CHECK WHETHER STOPPING CRITERIA SATISFIED
C
      CALL OPTSTZ(N,XPLS,FPLS,GPLS,X,ITNCNT,ICSCMX,
     +            ITRMCD,GRADTL,STEPTL,SX,FSCALE,ITNLIM,IRETCD,MXTAKE,
     +            IPR,MSG)
      IF(ITRMCD.NE.0) GO TO 690
C
C EVALUATE HESSIAN AT XPLS
C
      IF(IEXP.EQ.0) GO TO 130
      IF(METHOD.EQ.3)
     +     CALL SECUNZ(NR,N,X,G,A,UDIAG,XPLS,GPLS,EPSM,ITNCNT,RNF,
     +     IAGFLG,NOUPDT,WRK1,WRK2,WRK3)
      IF(METHOD.NE.3)
     +     CALL SECFAZ(NR,N,X,G,A,XPLS,GPLS,EPSM,ITNCNT,RNF,IAGFLG,
     +     NOUPDT,WRK0,WRK1,WRK2,WRK3)
      GO TO 150
  130 IF(IAHFLG.EQ.1) GO TO 140
      IF(IAGFLG.EQ.1)
     +     CALL FSTOFZ(NR,N,N,XPLS,D1FCN,GPLS,A,SX,RNF,WRK1,3)
      IF(IAGFLG.NE.1) CALL SNDOFZ(NR,N,XPLS,FCN,FPLS,A,SX,RNF,WRK1,WRK2)
      GO TO 150
  140 CALL D2FCN(NR,N,XPLS,A)
  150 CONTINUE
      IF(MOD(MSG/16,2).EQ.1)
     +     CALL RESULZ(NR,N,XPLS,FPLS,GPLS,A,P,ITNCNT,1,IPRZZ)
C
C X <-- XPLS  AND  G <-- GPLS  AND  F <-- FPLS
C
      F=FPLS
      DO 160 I=1,N
        X(I)=XPLS(I)
        G(I)=GPLS(I)
  160 CONTINUE
      GO TO 100
C
C TERMINATION
C -----------
C RESET XPLS,FPLS,GPLS,  IF PREVIOUS ITERATE SOLUTION
C
  690 IF(ITRMCD.NE.3) GO TO 710
  700 CONTINUE
      FPLS=F
      DO 705 I=1,N
        XPLS(I)=X(I)
        GPLS(I)=G(I)
  705 CONTINUE
C
C PRINT RESULTS
C
  710 CONTINUE
      IF(MOD(MSG/8,2).EQ.0)
     +     CALL RESULZ(NR,N,XPLS,FPLS,GPLS,A,P,ITNCNT,0,IPRZZ)
      MSG=0
CDPLT
CCCCC WRITE HESSIAN TO FILE DPST2F.DAT.  BEFORE WRITING, MAKE
CCCCC UPPER DIAGONAL OF MATRIX EQUAL TO LOWER DIAGONAL.
C
      IOUNI2=IST2NU
      DO9005I=1,N
        DO9007J=1,N
          A(J,I)=A(I,J)
 9007   CONTINUE
 9005 CONTINUE
C
      IF(N.LE.10)THEN
      WRITE(IOUNI2,9011)ITNCNT
      DO9010I=1,N
        WRITE(IOUNI2,9013)(A(I,J),J=1,N)
 9010 CONTINUE
 9011 FORMAT(1X,'HESSIAN MATRIX AT ITERATION ',I5)
 9013 FORMAT(10(1X,E15.7))
      ELSE
      WRITE(IOUNI2,9011)ITNCNT
      DO9020I=1,N
        DO9025J=1,N
          WRITE(IOUNI2,9023)A(I,J)
 9025   CONTINUE
        WRITE(IOUNI2,9027)
 9020 CONTINUE
 9023 FORMAT(1X,E15.7)
 9027 FORMAT(1X)
      ENDIF
C
      RETURN
C
  900 FORMAT('***** FROM OPTDRV       ',5(E20.13,3X))
  901 FORMAT('***** FROM OPTDRV    TYPICAL X')
  902 FORMAT('***** FROM OPTDRV    DIAGONAL SCALING MATRIX FOR X')
  903 FORMAT('***** FROM OPTDRV    TYPICAL F = ',E20.13)
  904 FORMAT('***** FROM OPTDRV    NUMBER OF GOOD DIGITS IN OPTFCN = ',
     +I5)
  914 FORMAT('                     GRADIENT FLAG  = ',I5,
     +' (=1 IF ')
  916 FORMAT('                     ANALYTIC GRADIENT SUPPLIED)')
  924 FORMAT('                     HESSIAN FLAG   = ',I5,
     +' (=1 IF ')
  926 FORMAT('                     ANALYTIC HESSIAN SUPPLIED)')
  934 FORMAT('                     EXPENSE FLAG   = ',I5,' (=1 IF ',
     +'MINIMIZATION ')
  936 FORMAT('                     FUNCTION EXPENSIVE TO EVALUATE)')
  944 FORMAT('                     METHOD TO USE  = ',I5,' (=1,2,3 ',
     +'FOR LINE SEARCH,')
  946 FORMAT('                     DOUBLE DOGLEG, MORE-HEBDON ',
     +' RESPECTIVELY)')
  954 FORMAT('                     ITERATION LIMIT = ',I5)
  964 FORMAT('                     MACHINE EPSILON = ',E20.13)
  905 FORMAT('***** FROM OPTDRV    MAXIMUM STEP SIZE    = ',E20.13)
  915 FORMAT('                     STEP TOLERANCE       = ',E20.13)
  925 FORMAT('                     GRADIENT TOLERANCE   = ',E20.13)
  935 FORMAT('                     TRUST REGION RADIUS  = ',E20.13)
  945 FORMAT('                     REL NOISE IN OPTFCN  = ',E20.13)
  955 FORMAT('                     ANAL-FD TOLERANCE    = ',E20.13)
  906 FORMAT('***** FROM OPTDRV    SHIFT FROM FORWARD TO CENTRAL ',
     +'DIFFERENCES IN ITERATION ', I5)
      END
      SUBROUTINE OPTIF0(NR,N,X,FCN,D1FCN,D2FCN,XPLS,FPLS,GPLS,
     +                  ITRMCD,A,WRK)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C PROVIDE SIMPLEST INTERFACE TO MINIMIZATION PACKAGE.
C USER HAS NO CONTROL OVER OPTIONS.
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> INITIAL ESTIMATE OF MINIMUM
C FCN          --> NAME OF ROUTINE TO EVALUATE MINIMIZATION FUNCTION.
C                  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE.
C XPLS(N)     <--  LOCAL MINIMUM
C FPLS        <--  FUNCTION VALUE AT LOCAL MINIMUM XPLS
C GPLS(N)     <--  GRADIENT AT LOCAL MINIMUM XPLS
C ITRMCD      <--  TERMINATION CODE
C A(N,N)       --> WORKSPACE
C WRK(N,9)     --> WORKSPACE
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),XPLS(N),GPLS(N)
      DIMENSION A(NR,*),WRK(NR,*)
      EXTERNAL FCN,D1FCN,D2FCN
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C EQUIVALENCE WRK(N,1) = UDIAG(N)
C             WRK(N,2) = G(N)
C             WRK(N,3) = P(N)
C             WRK(N,4) = TYPSIZ(N)
C             WRK(N,5) = SX(N)
C             WRK(N,6) = WRK0(N)
C             WRK(N,7) = WRK1(N)
C             WRK(N,8) = WRK2(N)
C             WRK(N,9) = WRK3(N)
C
      CALL DFAULZ(N,X,WRK(1,4),FSCALE,METHOD,IEXP,MSG,NDIGIT,
     +     ITNLIM,IAGFLG,IAHFLG,IPRZZ,DLT,GRADTL,STEPMX,STEPTL)
      CALL OPTDRZ(NR,N,X,FCN,D1FCN,D2FCN,WRK(1,4),FSCALE,
     +     METHOD,IEXP,MSG,NDIGIT,ITNLIM,IAGFLG,IAHFLG,IPRZZ,
     +     DLT,GRADTL,STEPMX,STEPTL,
     +     XPLS,FPLS,GPLS,ITRMCD,
     +     A,WRK(1,1),WRK(1,2),WRK(1,3),WRK(1,5),WRK(1,6),
     +     WRK(1,7),WRK(1,8),WRK(1,9))
      RETURN
      END
      SUBROUTINE OPTIFZ(NR,N,X,FCN,D1FCN,D2FCN,TYPSIZ,FSCALE,
     +     METHOD,IEXP,MSG,NDIGIT,ITNLIM,IAGFLG,IAHFLG,IPRZZ,
     +     DLT,GRADTL,STEPMX,STEPTL,
     +     XPLS,FPLS,GPLS,ITRMCD,A,WRK)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C PROVIDE COMPLETE INTERFACE TO MINIMIZATION PACKAGE.
C USER HAS FULL CONTROL OVER OPTIONS.
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> ON ENTRY: ESTIMATE TO A ROOT OF FCN
C FCN          --> NAME OF SUBROUTINE TO EVALUATE OPTIMIZATION FUNCTION
C                  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C                            FCN: R(N) --> R(1)
C D1FCN        --> (OPTIONAL) NAME OF SUBROUTINE TO EVALUATE GRADIENT
C                  OF FCN.  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C D2FCN        --> (OPTIONAL) NAME OF SUBROUTINE TO EVALUATE HESSIAN OF
C                  OF FCN.  MUST BE DECLARED EXTERNAL IN CALLING ROUTINE
C TYPSIZ(N)    --> TYPICAL SIZE FOR EACH COMPONENT OF X
C FSCALE       --> ESTIMATE OF SCALE OF OBJECTIVE FUNCTION
C METHOD       --> ALGORITHM TO USE TO SOLVE MINIMIZATION PROBLEM
C                    =1 LINE SEARCH
C                    =2 DOUBLE DOGLEG
C                    =3 MORE-HEBDON
C IEXP         --> =1 IF OPTIMIZATION FUNCTION FCN IS EXPENSIVE TO
C                  EVALUATE, =0 OTHERWISE.  IF SET THEN HESSIAN WILL
C                  BE EVALUATED BY SECANT UPDATE INSTEAD OF
C                  ANALYTICALLY OR BY FINITE DIFFERENCES
C MSG         <--> ON INPUT:  (.GT.0) MESSAGE TO INHIBIT CERTAIN
C                    AUTOMATIC CHECKS
C                  ON OUTPUT: (.LT.0) ERROR CODE; =0 NO ERROR
C NDIGIT       --> NUMBER OF GOOD DIGITS IN OPTIMIZATION FUNCTION FCN
C ITNLIM       --> MAXIMUM NUMBER OF ALLOWABLE ITERATIONS
C IAGFLG       --> =1 IF ANALYTIC GRADIENT SUPPLIED
C IAHFLG       --> =1 IF ANALYTIC HESSIAN SUPPLIED
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C DLT          --> TRUST REGION RADIUS
C GRADTL       --> TOLERANCE AT WHICH GRADIENT CONSIDERED CLOSE
C                  ENOUGH TO ZERO TO TERMINATE ALGORITHM
C STEPMX       --> MAXIMUM ALLOWABLE STEP SIZE
C STEPTL       --> RELATIVE STEP SIZE AT WHICH SUCCESSIVE ITERATES
C                  CONSIDERED CLOSE ENOUGH TO TERMINATE ALGORITHM
C XPLS(N)     <--> ON EXIT:  XPLS IS LOCAL MINIMUM
C FPLS        <--> ON EXIT:  FUNCTION VALUE AT SOLUTION, XPLS
C GPLS(N)     <--> ON EXIT:  GRADIENT AT SOLUTION XPLS
C ITRMCD      <--  TERMINATION CODE
C A(N,N)       --> WORKSPACE FOR HESSIAN (OR ESTIMATE)
C                  AND ITS CHOLESKY DECOMPOSITION
C WRK(N,8)     --> WORKSPACE
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),XPLS(N),GPLS(N),TYPSIZ(N)
      DIMENSION A(NR,*),WRK(NR,*)
      EXTERNAL FCN,D1FCN,D2FCN
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C EQUIVALENCE WRK(N,1) = UDIAG(N)
C             WRK(N,2) = G(N)
C             WRK(N,3) = P(N)
C             WRK(N,4) = SX(N)
C             WRK(N,5) = WRK0(N)
C             WRK(N,6) = WRK1(N)
C             WRK(N,7) = WRK2(N)
C             WRK(N,8) = WRK3(N)
C
      CALL OPTDRZ(NR,N,X,FCN,D1FCN,D2FCN,TYPSIZ,FSCALE,
     +     METHOD,IEXP,MSG,NDIGIT,ITNLIM,IAGFLG,IAHFLG,IPRZZ,
     +     DLT,GRADTL,STEPMX,STEPTL,
     +     XPLS,FPLS,GPLS,ITRMCD,
     +     A,WRK(1,1),WRK(1,2),WRK(1,3),WRK(1,4),WRK(1,5),
     +     WRK(1,6),WRK(1,7),WRK(1,8))
      RETURN
      END
      SUBROUTINE OPTSTZ(N,XPLS,FPLS,GPLS,X,ITNCNT,ICSCMX,
     +      ITRMCD,GRADTL,STEPTL,SX,FSCALE,ITNLIM,IRETCD,MXTAKE,
     +      IPRZZ,MSG)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C UNCONSTRAINED MINIMIZATION STOPPING CRITERIA
C --------------------------------------------
C FIND WHETHER THE ALGORITHM SHOULD TERMINATE, DUE TO ANY
C OF THE FOLLOWING:
C 1) PROBLEM SOLVED WITHIN USER TOLERANCE
C 2) CONVERGENCE WITHIN USER TOLERANCE
C 3) ITERATION LIMIT REACHED
C 4) DIVERGENCE OR TOO RESTRICTIVE MAXIMUM STEP (STEPMX) SUSPECTED
C
C
C PARAMETERS
C ----------
C N            --> DIMENSION OF PROBLEM
C XPLS(N)      --> NEW ITERATE X[K]
C FPLS         --> FUNCTION VALUE AT NEW ITERATE F(XPLS)
C GPLS(N)      --> GRADIENT AT NEW ITERATE, G(XPLS), OR APPROXIMATE
C X(N)         --> OLD ITERATE X[K-1]
C ITNCNT       --> CURRENT ITERATION K
C ICSCMX      <--> NUMBER CONSECUTIVE STEPS .GE. STEPMX
C                  [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C ITRMCD      <--  TERMINATION CODE
C GRADTL       --> TOLERANCE AT WHICH RELATIVE GRADIENT CONSIDERED CLOSE
C                  ENOUGH TO ZERO TO TERMINATE ALGORITHM
C STEPTL       --> RELATIVE STEP SIZE AT WHICH SUCCESSIVE ITERATES
C                  CONSIDERED CLOSE ENOUGH TO TERMINATE ALGORITHM
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C FSCALE       --> ESTIMATE OF SCALE OF OBJECTIVE FUNCTION
C ITNLIM       --> MAXIMUM NUMBER OF ALLOWABLE ITERATIONS
C IRETCD       --> RETURN CODE
C MXTAKE       --> BOOLEAN FLAG INDICATING STEP OF MAXIMUM LENGTH USED
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C MSG          --> IF MSG INCLUDES A TERM 8, SUPPRESS OUTPUT
C
C
      INTEGER N,ITNCNT,ICSCMX,ITRMCD,ITNLIM
      DIMENSION SX(N)
      DIMENSION XPLS(N),GPLS(N),X(N)
      LOGICAL MXTAKE
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      ITRMCD=0
C
C LAST GLOBAL STEP FAILED TO LOCATE A POINT LOWER THAN X
      IF(IRETCD.NE.1) GO TO 50
C     IF(IRETCD.EQ.1)
C     THEN
        JTRMCD=3
        GO TO 600
C     ENDIF
   50 CONTINUE
C
C FIND DIRECTION IN WHICH RELATIVE GRADIENT MAXIMUM.
C CHECK WHETHER WITHIN TOLERANCE
C
      D=MAX(ABS(FPLS),FSCALE)
      RGX=0.0
      DO 100 I=1,N
        RELGRD=ABS(GPLS(I))*MAX(ABS(XPLS(I)),1./SX(I))/D
        RGX=MAX(RGX,RELGRD)
  100 CONTINUE
      JTRMCD=1
      IF(RGX.LE.GRADTL) GO TO 600
C
      IF(ITNCNT.EQ.0) RETURN
C
C FIND DIRECTION IN WHICH RELATIVE STEPSIZE MAXIMUM
C CHECK WHETHER WITHIN TOLERANCE.
C
      RSX=0.0
      DO 120 I=1,N
        RELSTP=ABS(XPLS(I)-X(I))/MAX(ABS(XPLS(I)),1./SX(I))
        RSX=MAX(RSX,RELSTP)
  120 CONTINUE
      JTRMCD=2
      IF(RSX.LE.STEPTL) GO TO 600
C
C CHECK ITERATION LIMIT
C
      JTRMCD=4
      IF(ITNCNT.GE.ITNLIM) GO TO 600
C
C CHECK NUMBER OF CONSECUTIVE STEPS \ STEPMX
C
      IF(MXTAKE) GO TO 140
C     IF(.NOT.MXTAKE)
C     THEN
        ICSCMX=0
        RETURN
C     ELSE
  140   CONTINUE
        IF (MOD(MSG/8,2) .EQ. 0) WRITE(IPR,900)
        ICSCMX=ICSCMX+1
        IF(ICSCMX.LT.5) RETURN
        JTRMCD=5
C     ENDIF
C
C
C PRINT TERMINATION CODE
C
  600 ITRMCD=JTRMCD
      IF (MOD(MSG/8,2) .EQ. 0) THEN
         IF (ITRMCD.EQ.1) THEN
            WRITE(ICOUT,901)
            CALL DPWRST('XXX','BUG ')
            WRITE(ICOUT,911)
            CALL DPWRST('XXX','BUG ')
         ELSEIF (ITRMCD.EQ.2) THEN
            WRITE(ICOUT,902)
            CALL DPWRST('XXX','BUG ')
            WRITE(ICOUT,912)
            CALL DPWRST('XXX','BUG ')
         ELSEIF (ITRMCD.EQ.3) THEN
            WRITE(ICOUT,903)
            CALL DPWRST('XXX','BUG ')
            WRITE(ICOUT,913)
            CALL DPWRST('XXX','BUG ')
            WRITE(ICOUT,923)
            CALL DPWRST('XXX','BUG ')
         ELSEIF (ITRMCD.EQ.4) THEN
            WRITE(ICOUT,904)
            CALL DPWRST('XXX','BUG ')
         ELSEIF (ITRMCD.EQ.5) THEN
            WRITE(ICOUT,905)
            CALL DPWRST('XXX','BUG ')
            WRITE(ICOUT,915)
            CALL DPWRST('XXX','BUG ')
            WRITE(ICOUT,915)
            CALL DPWRST('XXX','BUG ')
        ENDIF
      ENDIF
C
  700 RETURN
C
  900 FORMAT(
     +'***** FROM OPTSTP    STEP OF MAXIMUM LENGTH (STEPMX) TAKEN')
  901 FORMAT(
     +'***** FROM OPTSTP    RELATIVE GRADIENT CLOSE TO ZERO.')
  911 FORMAT(
     +'                     CURRENT ITERATE IS PROBABLY SOLUTION.')
  902 FORMAT(
     +'***** FROM OPTSTP    SUCCESSIVE ITERATES WITHIN TOLERANCE.')
  912 FORMAT(
     +'                     CURRENT ITERATE IS PROBABLY SOLUTION.')
  903 FORMAT(
     +'***** FROM OPTSTP    LAST GLOBAL STEP FAILED TO LOCATE A ',
     +'POINT LOWER THAN X.')
  913 FORMAT(
     +'                     EITHER X IS AN APPROXIMATE LOCAL MINIMUM',
     + ' OF THE FUNCTION,')
  923 FORMAT(
     +'                     THE FUNCTION IS TOO NON-LINEAR FOR THIS ',
     +'ALGORITHM OR STEPTL IS TOO LARGE.')
  904 FORMAT(
     +'***** FROM OPTSTP    ALGORITHM FAILED BECAUSE ITERATION LIMIT',
     +'  EXCEEDED.')
  905 FORMAT(
     +'***** FROM OPTSTP    MAXIMUM STEP SIZE EXCEEDED 5 CONSECUTIVE',
     +'  TIMES.')
  915 FORMAT(
     +'                     EITHER THE FUNCTION IS UNBOUNDED BELOW, ',
     +'BECOMES ASYMPTOTIC TO A FINITE VALUE FROM ABOVE IN SOME ')
  925 FORMAT(
     +'                     DIRECTION, OR STEPMX IS TOO SMALL.')
      END
      SUBROUTINE QRAUXZ(NR,N,R,I)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C INTERCHANGE ROWS I,I+1 OF THE UPPER HESSENBERG MATRIX R,
C COLUMNS I TO N
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF MATRIX
C R(N,N)      <--> UPPER HESSENBERG MATRIX
C I            --> INDEX OF ROW TO INTERCHANGE (I.LT.N)
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION R(NR,*)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DO 10 J=I,N
        TMP=R(I,J)
        R(I,J)=R(I+1,J)
        R(I+1,J)=TMP
   10 CONTINUE
      RETURN
      END
      SUBROUTINE QRAUXY(NR,N,R,I,A,B)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C PRE-MULTIPLY R BY THE JACOBI ROTATION J(I,I+1,A,B)
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF MATRIX
C R(N,N)      <--> UPPER HESSENBERG MATRIX
C I            --> INDEX OF ROW
C A            --> SCALAR
C B            --> SCALAR
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION R(NR,*)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DEN=SQRT(A*A + B*B)
      C=A/DEN
      S=B/DEN
      DO 10 J=I,N
        Y=R(I,J)
        Z=R(I+1,J)
        R(I,J)=C*Y - S*Z
        R(I+1,J)=S*Y + C*Z
   10 CONTINUE
      RETURN
      END
      SUBROUTINE QRUPDZ(NR,N,A,U,V)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C FIND AN ORTHOGONAL (N*N) MATRIX (Q*) AND AN UPPER TRIANGULAR (N*N)
C MATRIX (R*) SUCH THAT (Q*)(R*)=R+U(V+)
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C A(N,N)      <--> ON INPUT:  CONTAINS R
C                  ON OUTPUT: CONTAINS (R*)
C U(N)         --> VECTOR
C V(N)         --> VECTOR
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION A(NR,*)
      DIMENSION U(N),V(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C DETERMINE LAST NON-ZERO IN U(.)
C
      K=N
   10 IF(U(K).NE.0. .OR. K.EQ.1) GO TO 20
C     IF(U(K).EQ.0. .AND. K.GT.1)
C     THEN
        K=K-1
        GO TO 10
C     ENDIF
C
C (K-1) JACOBI ROTATIONS TRANSFORM
C     R + U(V+) --> (R*) + (U(1)*E1)(V+)
C WHICH IS UPPER HESSENBERG
C
   20 IF(K.LE.1) GO TO 40
        KM1=K-1
        DO 30 II=1,KM1
          I=KM1-II+1
          IF(U(I).NE.0.) GO TO 25
C         IF(U(I).EQ.0.)
C         THEN
            CALL QRAUXZ(NR,N,A,I)
            U(I)=U(I+1)
            GO TO 30
C         ELSE
   25       CALL QRAUXY(NR,N,A,I,U(I),-U(I+1))
            U(I)=SQRT(U(I)*U(I) + U(I+1)*U(I+1))
C         ENDIF
   30   CONTINUE
C     ENDIF
C
C R <-- R + (U(1)*E1)(V+)
C
   40 DO 50 J=1,N
        A(1,J)=A(1,J) +U(1)*V(J)
   50 CONTINUE
C
C (K-1) JACOBI ROTATIONS TRANSFORM UPPER HESSENBERG R
C TO UPPER TRIANGULAR (R*)
C
      IF(K.LE.1) GO TO 100
        KM1=K-1
        DO 80 I=1,KM1
          IF(A(I,I).NE.0.) GO TO 70
C         IF(A(I,I).EQ.0.)
C         THEN
            CALL QRAUXZ(NR,N,A,I)
            GO TO 80
C         ELSE
   70       T1=A(I,I)
            T2=-A(I+1,I)
            CALL QRAUXY(NR,N,A,I,T1,T2)
C         ENDIF
   80   CONTINUE
C     ENDIF
  100 RETURN
      END
      SUBROUTINE RESULZ(NR,N,X,F,G,A,P,ITNCNT,IFLG,IPRZZ)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C PRINT INFORMATION
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> ITERATE X[K]
C F            --> FUNCTION VALUE AT X[K]
C G(N)         --> GRADIENT AT X[K]
C A(N,N)       --> HESSIAN AT X[K]
C P(N)         --> STEP TAKEN
C ITNCNT       --> ITERATION NUMBER K
C IFLG         --> FLAG CONTROLLING INFO TO PRINT
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),G(N),P(N),A(NR,*)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C PRINT ITERATION NUMBER
      WRITE(ICOUT,903) ITNCNT
      CALL DPWRST('XXX','BUG ')
      IF(IFLG.EQ.0) GO TO 120
C
C PRINT STEP
      WRITE(ICOUT,907)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,905) (P(I),I=1,N)
      CALL DPWRST('XXX','BUG ')
C
C PRINT CURRENT ITERATE
  120 CONTINUE
      WRITE(ICOUT,904)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,905) (X(I),I=1,N)
      CALL DPWRST('XXX','BUG ')
C
C PRINT FUNCTION VALUE
      WRITE(ICOUT,906)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,905) F
      CALL DPWRST('XXX','BUG ')
C
C PRINT GRADIENT
      WRITE(ICOUT,908)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,905) (G(I),I=1,N)
      CALL DPWRST('XXX','BUG ')
C
C PRINT HESSIAN FROM ITERATION K
      IF(IFLG.EQ.0) GO TO 140
      WRITE(ICOUT,901)
      CALL DPWRST('XXX','BUG ')
      DO 130 I=1,N
        WRITE(ICOUT,900) I
        CALL DPWRST('XXX','BUG ')
        WRITE(ICOUT,902) (A(I,J),J=1,I)
        CALL DPWRST('XXX','BUG ')
  130 CONTINUE
C
  140 RETURN
  900 FORMAT('****** FROM RESULT     ROW',I5)
  901 FORMAT('****** FROM RESULT       HESSIAN AT X(K)')
  902 FORMAT('****** FROM RESULT       ',5(2X,E20.13))
  903 FORMAT('****** FROM RESULT    ITERATE K=',I5)
  904 FORMAT('****** FROM RESULT       X(K)')
  905 FORMAT('****** FROM RESULT       ',5(2X,E20.13))
  906 FORMAT('****** FROM RESULT       FUNCTION AT X(K)')
  907 FORMAT('****** FROM RESULT       STEP')
  908 FORMAT('****** FROM RESULT       GRADIENT AT X(K)')
      END
      SUBROUTINE SCLMUZ(N,S,V,Z)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C MULTIPLY VECTOR BY SCALAR
C RESULT VECTOR MAY BE OPERAND VECTOR
C
C PARAMETERS
C ----------
C N            --> DIMENSION OF VECTORS
C S            --> SCALAR
C V(N)         --> OPERAND VECTOR
C Z(N)        <--  RESULT VECTOR
      DIMENSION V(N),Z(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DO 100 I=1,N
        Z(I)=S*V(I)
  100 CONTINUE
      RETURN
      END
      SUBROUTINE SECFAZ(NR,N,X,G,A,XPLS,GPLS,EPSM,ITNCNT,RNF,
     +     IAGFLG,NOUPDT,S,Y,U,W)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C UPDATE HESSIAN BY THE BFGS FACTORED METHOD
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> OLD ITERATE, X[K-1]
C G(N)         --> GRADIENT OR APPROXIMATE AT OLD ITERATE
C A(N,N)      <--> ON ENTRY: CHOLESKY DECOMPOSITION OF HESSIAN IN
C                    LOWER PART AND DIAGONAL.
C                  ON EXIT:  UPDATED CHOLESKY DECOMPOSITION OF HESSIAN
C                    IN LOWER TRIANGULAR PART AND DIAGONAL
C XPLS(N)      --> NEW ITERATE, X[K]
C GPLS(N)      --> GRADIENT OR APPROXIMATE AT NEW ITERATE
C EPSM         --> MACHINE EPSILON
C ITNCNT       --> ITERATION COUNT
C RNF          --> RELATIVE NOISE IN OPTIMIZATION FUNCTION FCN
C IAGFLG       --> =1 IF ANALYTIC GRADIENT SUPPLIED, =0 ITHERWISE
C NOUPDT      <--> BOOLEAN: NO UPDATE YET
C                  [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C S(N)         --> WORKSPACE
C Y(N)         --> WORKSPACE
C U(N)         --> WORKSPACE
C W(N)         --> WORKSPACE
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),XPLS(N),G(N),GPLS(N)
      DIMENSION A(NR,*)
      DIMENSION S(N),Y(N),U(N),W(N)
      LOGICAL NOUPDT,SKPUPD
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      IF(ITNCNT.EQ.1) NOUPDT=.TRUE.
      DO 10 I=1,N
        S(I)=XPLS(I)-X(I)
        Y(I)=GPLS(I)-G(I)
   10 CONTINUE
      DEN1=DDOT(N,S,1,Y,1)
      SNORM2=DNRM2(N,S,1)
      YNRM2=DNRM2(N,Y,1)
      IF(DEN1.LT.SQRT(EPSM)*SNORM2*YNRM2) GO TO 110
C     IF(DEN1.GE.SQRT(EPSM)*SNORM2*YNRM2)
C     THEN
        CALL MVMLTX(NR,N,A,S,U)
        DEN2=DDOT(N,U,1,U,1)
C
C       L <-- SQRT(DEN1/DEN2)*L
C
        ALP=SQRT(DEN1/DEN2)
        IF(.NOT.NOUPDT) GO TO 50
C       IF(NOUPDT)
C       THEN
          DO 30 J=1,N
            U(J)=ALP*U(J)
            DO 20 I=J,N
              A(I,J)=ALP*A(I,J)
   20       CONTINUE
   30     CONTINUE
          NOUPDT=.FALSE.
          DEN2=DEN1
          ALP=1.0
C       ENDIF
   50   SKPUPD=.TRUE.
C
C       W = L(L+)S = HS
C
        CALL MVMLTZ(NR,N,A,U,W)
        I=1
        IF(IAGFLG.NE.0) GO TO 55
C       IF(IAGFLG.EQ.0)
C       THEN
          RELTOL=SQRT(RNF)
          GO TO 60
C       ELSE
   55     RELTOL=RNF
C       ENDIF
   60   IF(I.GT.N .OR. .NOT.SKPUPD) GO TO 70
C       IF(I.LE.N .AND. SKPUPD)
C       THEN
          IF(ABS(Y(I)-W(I)) .LT. RELTOL*MAX(ABS(G(I)),ABS(GPLS(I))))
     +         GO TO 65
C         IF(ABS(Y(I)-W(I)) .GE. RELTOL*AMAX1(ABS(G(I)),ABS(GPLS(I))))
C         THEN
            SKPUPD=.FALSE.
            GO TO 60
C         ELSE
   65       I=I+1
            GO TO 60
C         ENDIF
C       ENDIF
   70   IF(SKPUPD) GO TO 110
C       IF(.NOT.SKPUPD)
C       THEN
C
C         W=Y-ALP*L(L+)S
C
          DO 75 I=1,N
            W(I)=Y(I)-ALP*W(I)
   75     CONTINUE
C
C         ALP=1/SQRT(DEN1*DEN2)
C
          ALP=ALP/DEN1
C
C         U=(L+)/SQRT(DEN1*DEN2) = (L+)S/SQRT((Y+)S * (S+)L(L+)S)
C
          DO 80 I=1,N
            U(I)=ALP*U(I)
   80     CONTINUE
C
C         COPY L INTO UPPER TRIANGULAR PART.  ZERO L.
C
          IF(N.EQ.1) GO TO 93
          DO 90 I=2,N
            IM1=I-1
            DO 85 J=1,IM1
              A(J,I)=A(I,J)
              A(I,J)=0.
   85       CONTINUE
   90     CONTINUE
C
C         FIND Q, (L+) SUCH THAT  Q(L+) = (L+) + U(W+)
C
   93     CALL QRUPDZ(NR,N,A,U,W)
C
C         UPPER TRIANGULAR PART AND DIAGONAL OF A NOW CONTAIN UPDATED
C         CHOLESKY DECOMPOSITION OF HESSIAN.  COPY BACK TO LOWER
C         TRIANGULAR PART.
C
          IF(N.EQ.1) GO TO 110
          DO 100 I=2,N
            IM1=I-1
            DO 95 J=1,IM1
              A(I,J)=A(J,I)
   95       CONTINUE
  100     CONTINUE
C       ENDIF
C     ENDIF
  110 RETURN
      END
      SUBROUTINE SECUNZ(NR,N,X,G,A,UDIAG,XPLS,GPLS,EPSM,ITNCNT,
     +     RNF,IAGFLG,NOUPDT,S,Y,T)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C UPDATE HESSIAN BY THE BFGS UNFACTORED METHOD
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> OLD ITERATE, X[K-1]
C G(N)         --> GRADIENT OR APPROXIMATE AT OLD ITERATE
C A(N,N)      <--> ON ENTRY: APPROXIMATE HESSIAN AT OLD ITERATE
C                    IN UPPER TRIANGULAR PART (AND UDIAG)
C                  ON EXIT:  UPDATED APPROX HESSIAN AT NEW ITERATE
C                    IN LOWER TRIANGULAR PART AND DIAGONAL
C                  [LOWER TRIANGULAR PART OF SYMMETRIC MATRIX]
C UDIAG        --> ON ENTRY: DIAGONAL OF HESSIAN
C XPLS(N)      --> NEW ITERATE, X[K]
C GPLS(N)      --> GRADIENT OR APPROXIMATE AT NEW ITERATE
C EPSM         --> MACHINE EPSILON
C ITNCNT       --> ITERATION COUNT
C RNF          --> RELATIVE NOISE IN OPTIMIZATION FUNCTION FCN
C IAGFLG       --> =1 IF ANALYTIC GRADIENT SUPPLIED, =0 OTHERWISE
C NOUPDT      <--> BOOLEAN: NO UPDATE YET
C                  [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C S(N)         --> WORKSPACE
C Y(N)         --> WORKSPACE
C T(N)         --> WORKSPACE
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),G(N),XPLS(N),GPLS(N)
      DIMENSION A(NR,*)
      DIMENSION UDIAG(N)
      DIMENSION S(N),Y(N),T(N)
      LOGICAL NOUPDT,SKPUPD
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C COPY HESSIAN IN UPPER TRIANGULAR PART AND UDIAG TO
C LOWER TRIANGULAR PART AND DIAGONAL
C
      DO 5 J=1,N
        A(J,J)=UDIAG(J)
        IF(J.EQ.N) GO TO 5
        JP1=J+1
        DO 4 I=JP1,N
          A(I,J)=A(J,I)
    4   CONTINUE
    5 CONTINUE
C
      IF(ITNCNT.EQ.1) NOUPDT=.TRUE.
      DO 10 I=1,N
        S(I)=XPLS(I)-X(I)
        Y(I)=GPLS(I)-G(I)
   10 CONTINUE
      DEN1=DDOT(N,S,1,Y,1)
      SNORM2=DNRM2(N,S,1)
      YNRM2=DNRM2(N,Y,1)
      IF(DEN1.LT.SQRT(EPSM)*SNORM2*YNRM2) GO TO 100
C     IF(DEN1.GE.SQRT(EPSM)*SNORM2*YNRM2)
C     THEN
        CALL MVMLTY(NR,N,A,S,T)
        DEN2=DDOT(N,S,1,T,1)
        IF(.NOT. NOUPDT) GO TO 50
C       IF(NOUPDT)
C       THEN
C
C         H <-- [(S+)Y/(S+)HS]H
C
          GAM=DEN1/DEN2
          DEN2=GAM*DEN2
          DO 30 J=1,N
            T(J)=GAM*T(J)
            DO 20 I=J,N
              A(I,J)=GAM*A(I,J)
   20       CONTINUE
   30     CONTINUE
          NOUPDT=.FALSE.
C       ENDIF
   50   SKPUPD=.TRUE.
C
C       CHECK UPDATE CONDITION ON ROW I
C
        DO 60 I=1,N
          TOL=RNF*MAX(ABS(G(I)),ABS(GPLS(I)))
          IF(IAGFLG.EQ.0) TOL=TOL/SQRT(RNF)
          IF(ABS(Y(I)-T(I)).LT.TOL) GO TO 60
C         IF(ABS(Y(I)-T(I)).GE.TOL)
C         THEN
            SKPUPD=.FALSE.
            GO TO 70
C         ENDIF
   60   CONTINUE
   70   IF(SKPUPD) GO TO 100
C       IF(.NOT.SKPUPD)
C       THEN
C
C         BFGS UPDATE
C
          DO 90 J=1,N
            DO 80 I=J,N
              A(I,J)=A(I,J)+Y(I)*Y(J)/DEN1-T(I)*T(J)/DEN2
   80       CONTINUE
   90     CONTINUE
C       ENDIF
C     ENDIF
  100 RETURN
      END
      SUBROUTINE SNDOFZ(NR,N,XPLS,FCN,FPLS,A,SX,RNOISE,STEPSZ,ANBR)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C PURPOSE
C -------
C FIND SECOND ORDER FORWARD FINITE DIFFERENCE APPROXIMATION "A"
C TO THE SECOND DERIVATIVE (HESSIAN) OF THE FUNCTION DEFINED BY THE SUBP
C "FCN" EVALUATED AT THE NEW ITERATE "XPLS"
C
C FOR OPTIMIZATION USE THIS ROUTINE TO ESTIMATE
C 1) THE SECOND DERIVATIVE (HESSIAN) OF THE OPTIMIZATION FUNCTION
C    IF NO ANALYTICAL USER FUNCTION HAS BEEN SUPPLIED FOR EITHER
C    THE GRADIENT OR THE HESSIAN AND IF THE OPTIMIZATION FUNCTION
C    "FCN" IS INEXPENSIVE TO EVALUATE.
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C XPLS(N)      --> NEW ITERATE:   X[K]
C FCN          --> NAME OF SUBROUTINE TO EVALUATE FUNCTION
C FPLS         --> FUNCTION VALUE AT NEW ITERATE, F(XPLS)
C A(N,N)      <--  FINITE DIFFERENCE APPROXIMATION TO HESSIAN
C                  ONLY LOWER TRIANGULAR MATRIX AND DIAGONAL
C                  ARE RETURNED
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C RNOISE       --> RELATIVE NOISE IN FNAME [F(X)]
C STEPSZ(N)    --> WORKSPACE (STEPSIZE IN I-TH COMPONENT DIRECTION)
C ANBR(N)      --> WORKSPACE (NEIGHBOR IN I-TH DIRECTION)
C
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION XPLS(N)
      DIMENSION SX(N)
      DIMENSION STEPSZ(N),ANBR(N)
      DIMENSION A(NR,*)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C FIND I-TH STEPSIZE AND EVALUATE NEIGHBOR IN DIRECTION
C OF I-TH UNIT VECTOR.
C
      OV3 = 1.0/3.0
      DO 10 I=1,N
        STEPSZ(I)=RNOISE**OV3 * MAX(ABS(XPLS(I)),1./SX(I))
        XTMPI=XPLS(I)
        XPLS(I)=XTMPI+STEPSZ(I)
        CALL FCN(N,XPLS,ANBR(I))
        XPLS(I)=XTMPI
   10 CONTINUE
C
C CALCULATE COLUMN I OF A
C
      DO 30 I=1,N
        XTMPI=XPLS(I)
        XPLS(I)=XTMPI+2.0*STEPSZ(I)
        CALL FCN(N,XPLS,FHAT)
        A(I,I)=((FPLS-ANBR(I))+(FHAT-ANBR(I)))/(STEPSZ(I)*STEPSZ(I))
C
C CALCULATE SUB-DIAGONAL ELEMENTS OF COLUMN
        IF(I.EQ.N) GO TO 25
        XPLS(I)=XTMPI+STEPSZ(I)
        IP1=I+1
        DO 20 J=IP1,N
          XTMPJ=XPLS(J)
          XPLS(J)=XTMPJ+STEPSZ(J)
          CALL FCN(N,XPLS,FHAT)
          A(J,I)=((FPLS-ANBR(I))+(FHAT-ANBR(J)))/(STEPSZ(I)*STEPSZ(J))
          XPLS(J)=XTMPJ
   20   CONTINUE
   25   XPLS(I)=XTMPI
   30 CONTINUE
      RETURN
      END
      SUBROUTINE TREGUZ(NR,N,X,F,G,A,FCN,SC,SX,NWTAKE,STEPMX,STEPTL,
     +     DLT,IRETCD,XPLSP,FPLSP,XPLS,FPLS,MXTAKE,IPRZZ,METHOD,UDIAG)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C PURPOSE
C -------
C DECIDE WHETHER TO ACCEPT XPLS=X+SC AS THE NEXT ITERATE AND UPDATE THE
C TRUST REGION DLT.
C
C PARAMETERS
C ----------
C NR           --> ROW DIMENSION OF MATRIX
C N            --> DIMENSION OF PROBLEM
C X(N)         --> OLD ITERATE X[K-1]
C F            --> FUNCTION VALUE AT OLD ITERATE, F(X)
C G(N)         --> GRADIENT AT OLD ITERATE, G(X), OR APPROXIMATE
C A(N,N)       --> CHOLESKY DECOMPOSITION OF HESSIAN IN
C                  LOWER TRIANGULAR PART AND DIAGONAL.
C                  HESSIAN OR APPROX IN UPPER TRIANGULAR PART
C FCN          --> NAME OF SUBROUTINE TO EVALUATE FUNCTION
C SC(N)        --> CURRENT STEP
C SX(N)        --> DIAGONAL SCALING MATRIX FOR X
C NWTAKE       --> BOOLEAN, =.TRUE. IF NEWTON STEP TAKEN
C STEPMX       --> MAXIMUM ALLOWABLE STEP SIZE
C STEPTL       --> RELATIVE STEP SIZE AT WHICH SUCCESSIVE ITERATES
C                  CONSIDERED CLOSE ENOUGH TO TERMINATE ALGORITHM
C DLT         <--> TRUST REGION RADIUS
C IRETCD      <--> RETURN CODE
C                    =0 XPLS ACCEPTED AS NEXT ITERATE;
C                       DLT TRUST REGION FOR NEXT ITERATION.
C                    =1 XPLS UNSATISFACTORY BUT ACCEPTED AS NEXT ITERATE
C                       BECAUSE XPLS-X .LT. SMALLEST ALLOWABLE
C                       STEP LENGTH.
C                    =2 F(XPLS) TOO LARGE.  CONTINUE CURRENT ITERATION
C                       WITH NEW REDUCED DLT.
C                    =3 F(XPLS) SUFFICIENTLY SMALL, BUT QUADRATIC MODEL
C                       PREDICTS F(XPLS) SUFFICIENTLY WELL TO CONTINUE
C                       CURRENT ITERATION WITH NEW DOUBLED DLT.
C XPLSP(N)    <--> WORKSPACE [VALUE NEEDS TO BE RETAINED BETWEEN
C                  SUCCESIVE CALLS OF K-TH GLOBAL STEP]
C FPLSP       <--> [RETAIN VALUE BETWEEN SUCCESSIVE CALLS]
C XPLS(N)     <--  NEW ITERATE X[K]
C FPLS        <--  FUNCTION VALUE AT NEW ITERATE, F(XPLS)
C MXTAKE      <--  BOOLEAN FLAG INDICATING STEP OF MAXIMUM LENGTH USED
C IPR          --> DEVICE TO WHICH TO SEND OUTPUT
C METHOD       --> ALGORITHM TO USE TO SOLVE MINIMIZATION PROBLEM
C                    =1 LINE SEARCH
C                    =2 DOUBLE DOGLEG
C                    =3 MORE-HEBDON
C UDIAG(N)     --> DIAGONAL OF HESSIAN IN A(.,.)
C
C***REVISION HISTORY  (YYMMDD)
C   000330  Modified array declarations.  (JEC)
C
      DIMENSION X(N),XPLS(N),G(N)
      DIMENSION SX(N),SC(N),XPLSP(N)
      DIMENSION A(NR,*)
      LOGICAL NWTAKE,MXTAKE
      DIMENSION UDIAG(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      IPR=IPR
      MXTAKE=.FALSE.
      DO 100 I=1,N
        XPLS(I)=X(I)+SC(I)
  100 CONTINUE
      CALL FCN(N,XPLS,FPLS)
      DLTF=FPLS-F
      SLP=DDOT(N,G,1,SC,1)
C
C NEXT STATEMENT ADDED FOR CASE OF COMPILERS WHICH DO NOT OPTIMIZE
C EVALUATION OF NEXT "IF" STATEMENT (IN WHICH CASE FPLSP COULD BE
C UNDEFINED).
      IF(IRETCD.EQ.4) FPLSP=0.0
CCCCc WRITE(IPR,961) IRETCD,FPLS,FPLSP,DLTF,SLP
      IF(IRETCD.NE.3 .OR. (FPLS.LT.FPLSP .AND. DLTF.LE. 1.E-4*SLP))
     +                                                     GO TO 130
C     IF(IRETCD.EQ.3 .AND. (FPLS.GE.FPLSP .OR. DLTF.GT. 1.E-4*SLP))
C     THEN
C
C       RESET XPLS TO XPLSP AND TERMINATE GLOBAL STEP
C
        IRETCD=0
        DO 110 I=1,N
          XPLS(I)=XPLSP(I)
  110   CONTINUE
        FPLS=FPLSP
        DLT=.5*DLT
CCCCC   WRITE(IPR,951)
        GO TO 230
C     ELSE
C
C       FPLS TOO LARGE
C
  130   IF(DLTF.LE. 1.E-4*SLP) GO TO 170
C       IF(DLTF.GT. 1.E-4*SLP)
C       THEN
CCCCC     WRITE(IPR,952)
          RLN=0.
          DO 140 I=1,N
            RLN=MAX(RLN,ABS(SC(I))/MAX(ABS(XPLS(I)),1./SX(I)))
  140     CONTINUE
CCCCC     WRITE(IPR,962) RLN
          IF(RLN.GE.STEPTL) GO TO 150
C         IF(RLN.LT.STEPTL)
C         THEN
C
C           CANNOT FIND SATISFACTORY XPLS SUFFICIENTLY DISTINCT FROM X
C
            IRETCD=1
CCCCC       WRITE(IPR,954)
            GO TO 230
C         ELSE
C
C           REDUCE TRUST REGION AND CONTINUE GLOBAL STEP
C
  150       IRETCD=2
            DLTMP=-SLP*DLT/(2.*(DLTF-SLP))
CCCCC       WRITE(IPR,963) DLTMP
            IF(DLTMP.GE. .1*DLT) GO TO 155
C           IF(DLTMP.LT. .1*DLT)
C           THEN
              DLT=.1*DLT
              GO TO 160
C           ELSE
  155         DLT=DLTMP
C           ENDIF
  160       CONTINUE
CCCCC       WRITE(IPR,955)
            GO TO 230
C         ENDIF
C       ELSE
C
C         FPLS SUFFICIENTLY SMALL
C
  170     CONTINUE
CCCCC     WRITE(IPR,958)
          DLTFP=0.
          IF (METHOD .EQ. 3) GO TO 180
C
C         IF (METHOD .EQ. 2)
C         THEN
C
          DO 177 I = 1, N
             TEMP = 0.0
             DO 173 J = I, N
                TEMP = TEMP + (A(J, I)*SC(J))
  173        CONTINUE
             DLTFP = DLTFP + TEMP*TEMP
  177     CONTINUE
          GO TO 190
C
C         ELSE
C
  180     DO 187 I = 1, N
             DLTFP = DLTFP + UDIAG(I)*SC(I)*SC(I)
             IF (I .EQ. N) GO TO 187
             TEMP = 0
             IP1 = I + 1
             DO 183 J = IP1, N
                TEMP = TEMP + A(I, J)*SC(I)*SC(J)
  183        CONTINUE
             DLTFP = DLTFP + 2.0*TEMP
  187     CONTINUE
C
C         END IF
C
  190     DLTFP = SLP + DLTFP/2.0
CCCCC     WRITE(IPR,964) DLTFP,NWTAKE
          IF(IRETCD.EQ.2 .OR. (ABS(DLTFP-DLTF).GT. .1*ABS(DLTF))
     +         .OR. NWTAKE .OR. (DLT.GT. .99*STEPMX)) GO TO 210
C         IF(IRETCD.NE.2 .AND. (ABS(DLTFP-DLTF) .LE. .1*ABS(DLTF))
C    +         .AND. (.NOT.NWTAKE) .AND. (DLT.LE. .99*STEPMX))
C         THEN
C
C           DOUBLE TRUST REGION AND CONTINUE GLOBAL STEP
C
            IRETCD=3
            DO 200 I=1,N
              XPLSP(I)=XPLS(I)
  200       CONTINUE
            FPLSP=FPLS
            DLT=MIN(2.*DLT,STEPMX)
CCCCC       WRITE(IPR,959)
            GO TO 230
C         ELSE
C
C           ACCEPT XPLS AS NEXT ITERATE.  CHOOSE NEW TRUST REGION.
C
  210       CONTINUE
CCCCC       WRITE(IPR,960)
            IRETCD=0
            IF(DLT.GT. .99*STEPMX) MXTAKE=.TRUE.
            IF(DLTF.LT. .1*DLTFP) GO TO 220
C           IF(DLTF.GE. .1*DLTFP)
C           THEN
C
C             DECREASE TRUST REGION FOR NEXT ITERATION
C
              DLT=.5*DLT
              GO TO 230
C           ELSE
C
C             CHECK WHETHER TO INCREASE TRUST REGION FOR NEXT ITERATION
C
  220         IF(DLTF.LE. .75*DLTFP) DLT=MIN(2.*DLT,STEPMX)
C           ENDIF
C         ENDIF
C       ENDIF
C     ENDIF
  230 CONTINUE
CCCCC WRITE(IPR,953)
CCCCC WRITE(IPR,956) IRETCD,MXTAKE,DLT,FPLS
CCCCC WRITE(IPR,957)
CCCCC WRITE(IPR,965) (XPLS(I),I=1,N)
      RETURN
C
CC951 FORMAT(55H TREGUZ    RESET XPLS TO XPLSP. TERMINATION GLOBAL STEP)
CC952 FORMAT(26H TREGUZ    FPLS TOO LARGE.)
CC953 FORMAT(38H0TREGUZ    VALUES AFTER CALL TO TREGUZ)
CC954 FORMAT(54H TREGUZ    CANNOT FIND SATISFACTORY XPLS DISTINCT FROM,
CC   +       27H X.  TERMINATE GLOBAL STEP.)
CC955 FORMAT(53H TREGUZ    REDUCE TRUST REGION. CONTINUE GLOBAL STEP.)
CC956 FORMAT(21H TREGUZ       IRETCD=,I3/
CC   +       21H TREGUZ       MXTAKE=,L1/
CC   +       21H TREGUZ       DLT   =,E20.13/
CC   +       21H TREGUZ       FPLS  =,E20.13)
CC957 FORMAT(32H TREGUZ       NEW ITERATE (XPLS))
CC958 FORMAT(35H TREGUZ    FPLS SUFFICIENTLY SMALL.)
CC959 FORMAT(54H TREGUZ    DOUBLE TRUST REGION.  CONTINUE GLOBAL STEP.)
CC960 FORMAT(50H TREGUZ    ACCEPT XPLS AS NEW ITERATE.  CHOOSE NEW,
CC   +       38H TRUST REGION.  TERMINATE GLOBAL STEP.)
CC961 FORMAT(18H TREGUZ    IRETCD=,I5/
CC   +       18H TREGUZ    FPLS  =,E20.13/
CC   +       18H TREGUZ    FPLSP =,E20.13/
CC   +       18H TREGUZ    DLTF  =,E20.13/
CC   +       18H TREGUZ    SLP   =,E20.13)
CC962 FORMAT(18H TREGUZ    RLN   =,E20.13)
CC963 FORMAT(18H TREGUZ    DLTMP =,E20.13)
CC964 FORMAT(18H TREGUZ    DLTFP =,E20.13/
CC   +       18H TREGUZ    NWTAKE=,L1)
CC965 FORMAT(14H TREGUZ       ,5(E20.13,3X))
      END
C
C     THESE ARE THE ROD BAIN NNES ROUTINES FOR EITHER UNCONSTRAINED
C     OPTIMIATION OR OPTIMIZATION WITH BOUNDS.  THESE ROUTINES CAN
C     USE EITHER ANALYTIC OR NUMERIC GRADIENTS/HESIANS.
C
C     ROUTINES ARE ALSO MODIFIED TO USE DATAPLOT I/O.
C
C     ROUTINES:
C     ====================
C     ABMUL    - MATRIX MULTIPLICATION (AB=C)
C     ASCALF   - ESTABLISH SCALING FACTORS FOR RESIDUAL VECTOR
C     ASCALX   - ESTABLISH SCALING FACTORS FOR COMPONENT VECTOR
C     ATAMUL   - MATRIX MULTIPLICATION (A^A=B)
C     ATAOV    - PRODUCT OF THE TRANSPOSE OF MATRIX A AND MATRIX A
C     ATBMUL   - MATRIX MULTIPLICATION (AB=C) WITH INNER LOOP UNROLLED
C                TO DEPTHS OF 32, 16, 8, AND 4
C     ATVOV    - PRODUCT OF THE TRANSPOSE OF MATRIX A AND THE VECTOR B
C     AVMUL    - MATRIX-VECTOR MULTIPLICATION AB=C
C     BAKDIF   - BACKWARD FINITE DIFFERENCE
C     BNDDIF   - FINITE DIFFERENCE CALCULATION WHEN THE BOUNDS FOR
C                COMPONENT J ARE SO CLOSE THAT NEITHER A FORWARD NOR
C                A BACKWARD DIFFERENCE CAN BE COMPUTED
C     BROYFA   - BROYDEN QUASI-NEWTON IS APPLIED TO FACTORED FORM
C                OF JACOBIAN
C     BROYUN  -  UPDATE JACOBIAN USING BROYDEN'S METHOD
C     CHOLDE  -  CHOLESKY DECOMPOSITION
C     CHSOLV  -  USE FORWARD/BACKWARD SUBSTITUTION TO SOLVE SYSTEM OF
C                LINEAR EQUATIONS
C     CONDNO  -  CONDITION NUMBER OF A QR-DECOMPOSED MATRIX
C     DELCAU  -  ESTABLISH INITIAL TRUST REGION DELTA
C     DEUFLS  -  CONDUCT A LINE SEARCH IN THE NEWTON DIRECTION IF
C                NO CONSTRAINTS ARE VIOLATED
C     DOGLEX  -  FIND A TRUST REGION USING THE DOUBLE DOGLEG METHOD
C     FCNEVL  -  EVALUATE THE OBJECTIVE FUNCTION
C                FCNNEW = 1/2(SCALEF*FVEC^SCALEF*FCVEC)
C     FORDIF  -  FORWARD FINITE DIFFERENCE
C     GRADF   -  COMPUTE GRADIENT OF FUNCTION
C                F = 1/2(SCALEF*FVECC)^(SCALEF*FCVECC)
C     INITCH  -  PERFORM SEVERAL CHECKS OF INPUT OPTIONS
C     INNERP  -  FIND INNER PRODUCT OF TWO VECTORS
C     JACCD   -  EVALUATE JACOBIAN USING CENTRAL DIFFERENCES
C     JACFD   -  EVALUATE JACOBIAN USING ONE-SIDED FINITE DIFFERENCES
C     JACOBI  -  EVALUATE THE JACOBIAN
C     JACROT  -  JACOBI (OR GIVENS) ROTATION
C     LINE    -  LINE SEARCH?
C     LLFA    -  LEE AND LEE QUASI=NEWTON METHOD APPLIED TO FACTORED
C                FORM OF JACOBIAN
C     LLUN    -  UPDATE JACOBIAN USING LEE AND LEE METHOD
C     LSOLV   -  SOLVES "B=RHS"
C     LTSOLV  -  SOLVES "L^Y=B"
C     MATCOP  -  COPY A CONTIGUOUS RECTANULAR PORTION OF ONE MATRIX
C                INTO ANOTHER ELEMENT
C     MATPRT  -  PRINTS RECTANULAR BLOCKS OF A MATRIX
C     MAXSTP  -  ESTABLISH MAXIMUM STEP LENGTH
C     NERSL   -  PRINT ITERATION RESULT
C     NESTOP  -  CHECKS FOR CONVERGENCE
C     NNES    -  MAIN ROUTINE FOR NNES
C     NSTPFA  -  FINDS NEWTON STEP
C     NSTPUN  -  FINDS NEWTON STEP
C     ONENRM  -  FIND 1-NORM OF H MATRIX IF PERTUBATION IS DESIRED AND
C                PERTURB DIAGONAL
C     QFORM   -  FOR Q^ FROM HOUSEHOLDER MATRICES
C                RENAME TO QFORMZ TO AVOID NAME CONFLICT WITH ANOTHER
C                DATAPLOT ROUTINE
C     QMIN    -  SET NEW TRUST REGION SIZE, DELTA, BASED ON A QUADRATIC
C                MINIMIZATION WHERE DELTA IS THE INDEPENDENT VARIABLE
C     QRDCOM  -  COMPUTE THE QR DECOMPOSITION
C     QRSOLV  -  SOLVES (QR)X=B  (Q AND R FROM QR DECOMPISITION)
C     QRUPDA  -  UPDATE QR DECOMPOSITION USING A SERIES OF GIVENS ROTATIONS
C     RCDPRT  -  DESCRIBE MEANING OF RETURN CODES
C     RSOLV   -  SOLVE BY BACKWARDS SUBSTITUTION RX=B
C     RTRMUL  -  FIND R^R FOR QR-DECOMPOSED JACOBIAN
C     SETUP   -  ASSIGN DEFAULT VALUES TO PARAMETERS
C     TITLE   -  PRINT TITLE FOR OUTPUT
C     TRSTUP  -  CHECK FOR ACCEPTANCE OF A TRUST REGION STEP GENERATED
C                BY DOUBLE DOGLEG METHOD
C     TWONRM  -  EVALUATES EUCLIDEAN NORM OF A VECTOR
C     UPDATE  -  RESETS CURRENT ESTIMATES OF SOLUTION AND UPDATES
C                OBJECTIVE FUNCTION VALUE AND THE TERMINATION CODE
C     UTBMUL  -  MATRIX MULTIPLICATION WHERE ONE OF THE MATRICES IS
C                UPPER TRIANGULAR
C     UTUMUL  -  MATRIX MULTIPLICATION WHERE ONE OF THE MATRICES IS
C                UPPER TRIANGULAR (WITH LOOP UNROLLING)
C     UVMUL   -  MATRIX-VECTOR MULTIPLICATION FOR TRIANGULAR MATRIX
C
        SUBROUTINE ABMUL(NRADEC,NRAACT,NCBDEC,NCBACT,NCDEC ,NCACT ,
     $                   AMAT  ,BMAT  ,CMAT  ,AROW)
C
C       FEB. 8, 1991
C
C       MATRIX MULTIPLICATION AB=C
C
C       VERSION WITH INNER LOOP UNROLLED TO DEPTHS 32, 16, 8 AND 4
C       EACH ROW OF MATRIX A IS SAVED AS A COLUMN, AROW, BEFORE USE.
C
C       NRADEC IS 1ST DIM. OF AMAT; NRAACT IS ACTUAL LIMIT FOR 1ST INDEX
C       NCBDEC IS 2ND DIM. OF BMAT; NCBACT IS ACTUAL LIMIT FOR 2ND INDEX
C       NCDEC IS COMMON DIMENSION OF AMAT & BMAT; NCACT IS ACTUAL LIMIT
C
C       I.E. NRADEC IS THE NUMBER OF ROWS OF A DECLARED
C            NCBDEC IS THE NUMBER OF COLUMNS OF B DECLARED
C            NCDEC IS THE COMMON DECLARED DIMENSION
C
C       MODIFICATION OF MATRIX MULTIPLIER DONATED BY PROF. JAMES
C       MACKINNON, QUEEN'S UNIVERSITY, KINGSTON, ONTARIO, CANADA
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION AMAT(NRADEC,NCDEC)  ,BMAT(NCDEC,NCBDEC),
     $            CMAT(NRADEC,NCBDEC) ,AROW(NCDEC)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO /0.0D0/
C
C       FIND NUMBER OF GROUPS OF SIZE 32, 16 ...
C
        NCC32=NCACT/32
        NCC32R=NCACT-32*NCC32
        NCC16=NCC32R/16
        NCC16R=NCC32R-16*NCC16
        NCC8=NCC16R/8
        NCC8R=NCC16R-8*NCC8
        NCC4=NCC8R/4
        NCC4R=NCC8R-4*NCC4
C
C       REASSIGN ROWS TO VECTOR AROW.
C
        DO 100 I=1,NRAACT
           K=0
           IF(NCC32.GT.0) THEN
              DO 200 KK=1,NCC32
                 K=K+32
                 AROW(K-31)=AMAT(I,K-31)
                 AROW(K-30)=AMAT(I,K-30)
                 AROW(K-29)=AMAT(I,K-29)
                 AROW(K-28)=AMAT(I,K-28)
                 AROW(K-27)=AMAT(I,K-27)
                 AROW(K-26)=AMAT(I,K-26)
                 AROW(K-25)=AMAT(I,K-25)
                 AROW(K-24)=AMAT(I,K-24)
                 AROW(K-23)=AMAT(I,K-23)
                 AROW(K-22)=AMAT(I,K-22)
                 AROW(K-21)=AMAT(I,K-21)
                 AROW(K-20)=AMAT(I,K-20)
                 AROW(K-19)=AMAT(I,K-19)
                 AROW(K-18)=AMAT(I,K-18)
                 AROW(K-17)=AMAT(I,K-17)
                 AROW(K-16)=AMAT(I,K-16)
                 AROW(K-15)=AMAT(I,K-15)
                 AROW(K-14)=AMAT(I,K-14)
                 AROW(K-13)=AMAT(I,K-13)
                 AROW(K-12)=AMAT(I,K-12)
                 AROW(K-11)=AMAT(I,K-11)
                 AROW(K-10)=AMAT(I,K-10)
                 AROW(K-9)=AMAT(I,K-9)
                 AROW(K-8)=AMAT(I,K-8)
                 AROW(K-7)=AMAT(I,K-7)
                 AROW(K-6)=AMAT(I,K-6)
                 AROW(K-5)=AMAT(I,K-5)
                 AROW(K-4)=AMAT(I,K-4)
                 AROW(K-3)=AMAT(I,K-3)
                 AROW(K-2)=AMAT(I,K-2)
                 AROW(K-1)=AMAT(I,K-1)
                 AROW(K)=AMAT(I,K)
200           CONTINUE
           END IF
           IF(NCC16.GT.0) THEN
              DO 300 KK=1,NCC16
                 K=K+16
                 AROW(K-15)=AMAT(I,K-15)
                 AROW(K-14)=AMAT(I,K-14)
                 AROW(K-13)=AMAT(I,K-13)
                 AROW(K-12)=AMAT(I,K-12)
                 AROW(K-11)=AMAT(I,K-11)
                 AROW(K-10)=AMAT(I,K-10)
                 AROW(K-9)=AMAT(I,K-9)
                 AROW(K-8)=AMAT(I,K-8)
                 AROW(K-7)=AMAT(I,K-7)
                 AROW(K-6)=AMAT(I,K-6)
                 AROW(K-5)=AMAT(I,K-5)
                 AROW(K-4)=AMAT(I,K-4)
                 AROW(K-3)=AMAT(I,K-3)
                 AROW(K-2)=AMAT(I,K-2)
                 AROW(K-1)=AMAT(I,K-1)
                 AROW(K)=AMAT(I,K)
300           CONTINUE
           END IF
           IF(NCC8.GT.0) THEN
              DO 400 KK=1,NCC8
                 K=K+8
                 AROW(K-7)=AMAT(I,K-7)
                 AROW(K-6)=AMAT(I,K-6)
                 AROW(K-5)=AMAT(I,K-5)
                 AROW(K-4)=AMAT(I,K-4)
                 AROW(K-3)=AMAT(I,K-3)
                 AROW(K-2)=AMAT(I,K-2)
                 AROW(K-1)=AMAT(I,K-1)
                 AROW(K)=AMAT(I,K)
400           CONTINUE
           END IF
           IF(NCC4.GT.0) THEN
              DO 500 KK=1,NCC4
                 K=K+4
                 AROW(K-3)=AMAT(I,K-3)
                 AROW(K-2)=AMAT(I,K-2)
                 AROW(K-1)=AMAT(I,K-1)
                 AROW(K)=AMAT(I,K)
500           CONTINUE
           END IF
           IF(NCC4R.GT.0) THEN
              DO 600 KK=1,NCC4R
                 K=K+1
                 AROW(K)=AMAT(I,K)
600           CONTINUE
           END IF
C
C          FIND ENTRY FOR MATRIX C USING COLUMN VECTOR AROW.
C
           DO 700 J=1,NCBACT
              SUM=ZERO
              K=0
              IF(NCC32.GT.0) THEN
                 DO 800 KK=1,NCC32
                    K=K+32
                    SUM=SUM
     $              +AROW(K-31)*BMAT(K-31,J)+AROW(K-30)*BMAT(K-30,J)
     $              +AROW(K-29)*BMAT(K-29,J)+AROW(K-28)*BMAT(K-28,J)
     $              +AROW(K-27)*BMAT(K-27,J)+AROW(K-26)*BMAT(K-26,J)
     $              +AROW(K-25)*BMAT(K-25,J)+AROW(K-24)*BMAT(K-24,J)
                    SUM=SUM
     $              +AROW(K-23)*BMAT(K-23,J)+AROW(K-22)*BMAT(K-22,J)
     $              +AROW(K-21)*BMAT(K-21,J)+AROW(K-20)*BMAT(K-20,J)
     $              +AROW(K-19)*BMAT(K-19,J)+AROW(K-18)*BMAT(K-18,J)
     $              +AROW(K-17)*BMAT(K-17,J)+AROW(K-16)*BMAT(K-16,J)
                    SUM=SUM
     $              +AROW(K-15)*BMAT(K-15,J)+AROW(K-14)*BMAT(K-14,J)
     $              +AROW(K-13)*BMAT(K-13,J)+AROW(K-12)*BMAT(K-12,J)
     $              +AROW(K-11)*BMAT(K-11,J)+AROW(K-10)*BMAT(K-10,J)
     $              +AROW(K-9) *BMAT(K-9,J) +AROW(K-8) *BMAT(K-8,J)
                    SUM=SUM
     $              +AROW(K-7)*BMAT(K-7,J)+AROW(K-6)*BMAT(K-6,J)
     $              +AROW(K-5)*BMAT(K-5,J)+AROW(K-4)*BMAT(K-4,J)
     $              +AROW(K-3)*BMAT(K-3,J)+AROW(K-2)*BMAT(K-2,J)
     $              +AROW(K-1)*BMAT(K-1,J)+AROW(K)  *BMAT(K,J)
800              CONTINUE
              END IF
              IF(NCC16.GT.0) THEN
                 DO 900 KK=1,NCC16
                    K=K+16
                    SUM=SUM
     $              +AROW(K-15)*BMAT(K-15,J)+AROW(K-14)*BMAT(K-14,J)
     $              +AROW(K-13)*BMAT(K-13,J)+AROW(K-12)*BMAT(K-12,J)
     $              +AROW(K-11)*BMAT(K-11,J)+AROW(K-10)*BMAT(K-10,J)
     $              +AROW(K-9) *BMAT(K-9,J) +AROW(K-8) *BMAT(K-8,J)
                    SUM=SUM
     $              +AROW(K-7)*BMAT(K-7,J)+AROW(K-6)*BMAT(K-6,J)
     $              +AROW(K-5)*BMAT(K-5,J)+AROW(K-4)*BMAT(K-4,J)
     $              +AROW(K-3)*BMAT(K-3,J)+AROW(K-2)*BMAT(K-2,J)
     $              +AROW(K-1)*BMAT(K-1,J)+AROW(K)  *BMAT(K,J)
900              CONTINUE
              END IF
              IF(NCC8.GT.0) THEN
                 DO 1000 KK=1,NCC8
                    K=K+8
                    SUM=SUM
     $              +AROW(K-7)*BMAT(K-7,J)+AROW(K-6)*BMAT(K-6,J)
     $              +AROW(K-5)*BMAT(K-5,J)+AROW(K-4)*BMAT(K-4,J)
     $              +AROW(K-3)*BMAT(K-3,J)+AROW(K-2)*BMAT(K-2,J)
     $              +AROW(K-1)*BMAT(K-1,J)+AROW(K)  *BMAT(K,J)
1000             CONTINUE
              END IF
              IF(NCC4.GT.0) THEN
                 DO 1100 KK=1,NCC4
                    K=K+4
                    SUM=SUM
     $              +AROW(K-3)*BMAT(K-3,J)+AROW(K-2)*BMAT(K-2,J)
     $              +AROW(K-1)*BMAT(K-1,J)+AROW(K)  *BMAT(K,J)
1100             CONTINUE
              END IF
              IF(NCC4R.GT.0) THEN
                 DO 1200 KK=1,NCC4R
                    K=K+1
                    SUM=SUM+AROW(K)*BMAT(K,J)
1200             CONTINUE
              END IF
              CMAT(I,J)=SUM
700        CONTINUE
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE ABMUL.
C
        END
        SUBROUTINE ASCALF(N,EPSMCH,FVECC,JAC,SCALEF)
C
C       FEB. 13, 1991
C
C       THIS SUBROUTINE ESTABLISHES SCALING FACTORS FOR THE
C       RESIDUAL VECTOR IF FUNCTION ADAPTIVE SCALING IS CHOSEN
C       USING INTEGER VARIABLE ITSCLF.
C
C       NOTE: IN QUASI-NEWTON METHODS THE SCALING FACTORS ARE
C             UPDATED ONLY WHEN THE JACOBIAN IS EVALUATED EXPLI-
C             CITLY.
C
C       SCALING FACTORS ARE DETERMINED FROM THE INFINITY NORMS
C       OF THE ROWS OF THE JACOBIAN AND THE VALUES OF THE CURRENT
C       FUNCTION VECTOR, FVECC.
C
C       A MINIMUM TOLERANCE ON THE SCALING FACTOR IS THE SQUARE
C       ROOT OF THE MACHINE PRECISION, SQRTEP.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N)
        DIMENSION        FVECC(N)  ,SCALEF(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,ONE /0.0D0,1.0D0/
C
        SQRTEP=SQRT(EPSMCH)
C
C       I COUNTS THE ROWS.
C
        DO 100 I=1,N
           AMAX=ZERO
C
C          FIND MAXIMUM ENTRY IN ROW I.
C
           DO 200 J=1,N
              AMAX=MAX(AMAX,ABS(JAC(I,J)))
200        CONTINUE
C
           AMAX=MAX(AMAX,FVECC(I))
C
C          SET SCALING FACTOR TO A DEFAULT OF ONE IF ITH ROW IS ZEROS.
C
           IF(AMAX.EQ.ZERO) AMAX=ONE
           AMAX=MAX(AMAX,SQRTEP)
           SCALEF(I)=ONE/AMAX
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE ASCALF.
C
        END
        SUBROUTINE ASCALX(N,EPSMCH,JAC,SCALEX)
C
C       FEB. 13, 1991
C
C       THIS SUBROUTINE ESTABLISHES SCALING FACTORS FOR THE
C       COMPONENET VECTOR IF ADAPTIVE SCALING IS CHOSEN USING
C       INTEGER ITSCLX.
C
C       NOTE: IN QUASI-NEWTON METHODS THE SCALING FACTORS ARE
C             UPDATED ONLY WHEN THE JACOBIAN IS EVALUATED EXPLI-
C             CITLY.
C
C       SCALING FACTORS ARE DETERMINED FROM THE INFINITY NORMS
C       OF THE COLUMNS OF THE JACOBIAN.
C
C       A MINIMUM TOLERANCE ON THE SCALING FACTOR IS THE SQUARE
C       ROOT OF THE MACHINE PRECISION, SQRTEP.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N)
        DIMENSION        SCALEX(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,ONE /0.0D0,1.0D0/
C
        SQRTEP=SQRT(EPSMCH)
C
C       J COUNTS COLUMNS.
C
        DO 100 J=1,N
           AMAX=ZERO
C
C          FIND MAXIMUM ENTRY IN JTH COLUMN.
C
           DO 200 I=1,N
              AMAX=MAX(AMAX,ABS(JAC(I,J)))
200        CONTINUE
C
C          IF A COLUMN IS ALL ZEROS SET AMAX TO ONE.
C
           IF(AMAX.EQ.ZERO) AMAX=ONE
           SCALEX(J)=MAX(AMAX,SQRTEP)
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE ASCALX.
C
        END
        SUBROUTINE ATAMUL(NRADEC,NCADEC,NRAACT,NCAACT,NRBDEC,NCBDEC,
     $                    AMAT  ,BMAT)
C
C       FEB. 8, 1991
C
C       MATRIX MULTIPLICATION:   A^A=B
C
C       VERSION WITH INNER LOOP UNROLLED TO DEPTHS 32, 16, 8 AND 4.
C
C       NRADEC IS NUMBER OF ROWS IN A DECLARED
C       NCADEC IS NUMBER OF COLUMNS IN A DECLARED
C       NRAACT IS THE LIMIT FOR THE 1ST INDEX IN A
C       NCAACT IS THE LIMIT FOR THE 2ND INDEX IN A
C       NRBDEC IS NUMBER OF ROWS IN B DECLARED
C       NCBDEC IS NUMBER OF COLUMNS IN B DECLARED
C
C       MODIFIED VERSION OF THE MATRIX MULTIPLIER DONATED BY PROF. JAMES
C       MACKINNON, QUEEN'S UNIVERSITY, KINGSTON, ONTARIO, CANADA
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION AMAT(NRADEC,NCADEC), BMAT(NRBDEC,NCBDEC)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO /0.0D0/
C
C       FIND NUMBER OF GROUPS OF SIZE 32, 16 ...
C
        NCC32=NRAACT/32
        NCC32R=NRAACT-32*NCC32
        NCC16=NCC32R/16
        NCC16R=NCC32R-16*NCC16
        NCC8=NCC16R/8
        NCC8R=NCC16R-8*NCC8
        NCC4=NCC8R/4
        NCC4R=NCC8R-4*NCC4
C
C       FIND ENTRY IN MATRIX B.
C
        DO 100 I=1,NCAACT
           DO 200 J=I,NCAACT
              SUM=ZERO
              K=0
              IF(NCC32.GT.0) THEN
                 DO 300 KK=1,NCC32
                    K=K+32
                    SUM=SUM
     $              +AMAT(K-31,I)*AMAT(K-31,J)+AMAT(K-30,I)*AMAT(K-30,J)
     $              +AMAT(K-29,I)*AMAT(K-29,J)+AMAT(K-28,I)*AMAT(K-28,J)
     $              +AMAT(K-27,I)*AMAT(K-27,J)+AMAT(K-26,I)*AMAT(K-26,J)
     $              +AMAT(K-25,I)*AMAT(K-25,J)+AMAT(K-24,I)*AMAT(K-24,J)
                    SUM=SUM
     $              +AMAT(K-23,I)*AMAT(K-23,J)+AMAT(K-22,I)*AMAT(K-22,J)
     $              +AMAT(K-21,I)*AMAT(K-21,J)+AMAT(K-20,I)*AMAT(K-20,J)
     $              +AMAT(K-19,I)*AMAT(K-19,J)+AMAT(K-18,I)*AMAT(K-18,J)
     $              +AMAT(K-17,I)*AMAT(K-17,J)+AMAT(K-16,I)*AMAT(K-16,J)
                    SUM=SUM
     $              +AMAT(K-15,I)*AMAT(K-15,J)+AMAT(K-14,I)*AMAT(K-14,J)
     $              +AMAT(K-13,I)*AMAT(K-13,J)+AMAT(K-12,I)*AMAT(K-12,J)
     $              +AMAT(K-11,I)*AMAT(K-11,J)+AMAT(K-10,I)*AMAT(K-10,J)
     $              +AMAT(K-9,I)*AMAT(K-9,J)  +AMAT(K-8,I)*AMAT(K-8,J)
                    SUM=SUM
     $              +AMAT(K-7,I)*AMAT(K-7,J)+AMAT(K-6,I)*AMAT(K-6,J)
     $              +AMAT(K-5,I)*AMAT(K-5,J)+AMAT(K-4,I)*AMAT(K-4,J)
     $              +AMAT(K-3,I)*AMAT(K-3,J)+AMAT(K-2,I)*AMAT(K-2,J)
     $              +AMAT(K-1,I)*AMAT(K-1,J)+AMAT(K,I)  *AMAT(K,J)
300              CONTINUE
              END IF
              IF(NCC16.GT.0) THEN
                 DO 400 KK=1,NCC16
                    K=K+16
                    SUM=SUM
     $              +AMAT(K-15,I)*AMAT(K-15,J)+AMAT(K-14,I)*AMAT(K-14,J)
     $              +AMAT(K-13,I)*AMAT(K-13,J)+AMAT(K-12,I)*AMAT(K-12,J)
     $              +AMAT(K-11,I)*AMAT(K-11,J)+AMAT(K-10,I)*AMAT(K-10,J)
     $              +AMAT(K-9,I)*AMAT(K-9,J)  +AMAT(K-8,I) *AMAT(K-8,J)
                    SUM=SUM
     $              +AMAT(K-7,I)*AMAT(K-7,J)+AMAT(K-6,I)*AMAT(K-6,J)
     $              +AMAT(K-5,I)*AMAT(K-5,J)+AMAT(K-4,I)*AMAT(K-4,J)
     $              +AMAT(K-3,I)*AMAT(K-3,J)+AMAT(K-2,I)*AMAT(K-2,J)
     $              +AMAT(K-1,I)*AMAT(K-1,J)+AMAT(K,I)*AMAT(K,J)
400              CONTINUE
              END IF
              IF(NCC8.GT.0) THEN
                 DO 500 KK=1,NCC8
                    K=K+8
                    SUM=SUM
     $              +AMAT(K-7,I)*AMAT(K-7,J)+AMAT(K-6,I)*AMAT(K-6,J)
     $              +AMAT(K-5,I)*AMAT(K-5,J)+AMAT(K-4,I)*AMAT(K-4,J)
     $              +AMAT(K-3,I)*AMAT(K-3,J)+AMAT(K-2,I)*AMAT(K-2,J)
     $              +AMAT(K-1,I)*AMAT(K-1,J)+AMAT(K,I)*AMAT(K,J)
500              CONTINUE
              END IF
              IF(NCC4.GT.0) THEN
                 DO 600 KK=1,NCC4
                    K=K+4
                    SUM=SUM
     $              +AMAT(K-3,I)*AMAT(K-3,J)+AMAT(K-2,I)*AMAT(K-2,J)
     $              +AMAT(K-1,I)*AMAT(K-1,J)+AMAT(K,I)*AMAT(K,J)
600              CONTINUE
              END IF
              IF(NCC4R.GT.0) THEN
                 DO 700 KK=1,NCC4R
                    K=K+1
                    SUM=SUM+AMAT(K,I)*AMAT(K,J)
700              CONTINUE
              END IF
              BMAT(I,J)=SUM
              IF(I.NE.J) BMAT(J,I)=BMAT(I,J)
200        CONTINUE
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE ATAMUL.
C
        END
      SUBROUTINE ATAOV(OVERFL,MAXEXP,N,NUNIT,OUTPUT,A,B,SCALEF)
C
C       SEPT. 8, 1991
C
C       THIS SUBROUTINE FINDS THE PRODUCT OF THE TRANSPOSE OF THE
C       MATRIX A AND MATRIX A.  EACH ENTRY IS CHECKED BEFORE BEING
C       ACCEPTED.  IF IT WOULD CAUSE AN OVERFLOW 10**MAXEXP IS
C       INSERTED IN ITS PLACE.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        INTEGER    OUTPUT
      DIMENSION  A(N,N)   ,B(N,N)   ,SCALEF(N)
      LOGICAL    OVERFL   ,WRNSUP
      COMMON/NNES_2/WRNSUP
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,ONE,TWO,TEN /0.0D0,1.0D0,2.0D0,10.0D0/
C
      EPS=TEN**(-MAXEXP)
        OVERFL=.FALSE.
C
      DO 100 I=1,N
         DO 200 J=I+1,N
            SUM=ZERO
            DO 300 K=1,N
          IF(LOG10(ABS(A(K,I))+EPS)+LOG10(ABS(A(K,J))+EPS)
     $              +TWO*LOG10(SCALEF(K)).GT.MAXEXP) THEN
             OVERFL=.TRUE.
             B(I,J)=SIGN(TEN**MAXEXP,A(K,I))*SIGN(ONE,A(K,J))
             IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
                WRITE(ICOUT,1)
1                      FORMAT(T3,'*',T74,'*')
                WRITE(ICOUT,2) B(I,J)
2                      FORMAT(T3,'*',4X,'WARNING: COMPONENT IN',
     $                 ' MATRIX-MATRIX PRODUCT SET TO ',1PD12.3,
     $                 T74,'*')
             END IF
             GO TO 201
          END IF
          SUM=SUM+A(K,I)*A(K,J)*SCALEF(K)*SCALEF(K)
300           CONTINUE
            B(I,J)=SUM
            B(J,I)=SUM
201           CONTINUE
200        CONTINUE
         SUM=ZERO
         DO 400 K=1,N
            IF(TWO*(LOG10(ABS(A(K,I))+EPS)+LOG10(SCALEF(K))).
     $           GT.MAXEXP) THEN
          OVERFL=.TRUE.
          B(I,I)=TEN**MAXEXP
          IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
             WRITE(ICOUT,1)
             WRITE(ICOUT,2) B(I,I)
          END IF
          GO TO 401
            END IF
            SUM=SUM+A(K,I)*A(K,I)*SCALEF(K)*SCALEF(K)
400        CONTINUE
         B(I,I)=SUM
401        CONTINUE
100     CONTINUE
      RETURN
C
C       LAST CARD OF SUBROUTINE ATAOV.
C
      END
        SUBROUTINE ATBMUL(NCADEC,NCAACT,NCBDEC,NCBACT,NCDEC,NCACT,
     $                    AMAT  ,BMAT  ,CMAT)
C
C       FEB. 8, 1991
C
C       MATRIX MULTIPLICATION:   A^B=C
C
C       VERSION WITH INNER LOOP UNROLLED TO DEPTHS 32, 16, 8 AND 4.
C
C       NCADEC IS 2ND DIM. OF AMAT; NCAACT IS ACTUAL LIMIT FOR 2ND INDEX
C       NCBDEC IS 2ND DIM. OF BMAT; NCBACT IS ACTUAL LIMIT FOR 2ND INDEX
C       NCDEC IS COMMON DIMENSION OF AMAT & BMAT; NCACT IS ACTUAL LIMIT
C
C       I.E.   NCADEC IS NUMBER OF COLUMNS OF A DECLARED
C              NCBDEC IS NUMBER OF COLUMNS OF B DECLARED
C              NCDEC  IS THE NUMBER OF ROWS IN BOTH A AND B DECLARED
C
C       MODIFIED VERSION OF THE MATRIX MULTIPLIER DONATED BY PROF. JAMES
C       MACKINNON, QUEEN'S UNIVERSITY, KINGSTON, ONTARIO, CANADA
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION AMAT(NCDEC,NCADEC), BMAT(NCDEC,NCBDEC),
     $            CMAT(NCADEC,NCBDEC)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO /0.0D0/
C
C       FIND NUMBER OF GROUPS OF SIZE 32, 16 ...
C
        NCC32=NCACT/32
        NCC32R=NCACT-32*NCC32
        NCC16=NCC32R/16
        NCC16R=NCC32R-16*NCC16
        NCC8=NCC16R/8
        NCC8R=NCC16R-8*NCC8
        NCC4=NCC8R/4
        NCC4R=NCC8R-4*NCC4
C
C       FIND ENTRY IN MATRIX C.
C
        DO 100 I=1,NCAACT
           DO 200 J=1,NCBACT
              SUM=ZERO
              K=0
              IF(NCC32.GT.0) THEN
                 DO 300 KK=1,NCC32
                    K=K+32
                    SUM=SUM
     $              +AMAT(K-31,I)*BMAT(K-31,J)+AMAT(K-30,I)*BMAT(K-30,J)
     $              +AMAT(K-29,I)*BMAT(K-29,J)+AMAT(K-28,I)*BMAT(K-28,J)
     $              +AMAT(K-27,I)*BMAT(K-27,J)+AMAT(K-26,I)*BMAT(K-26,J)
     $              +AMAT(K-25,I)*BMAT(K-25,J)+AMAT(K-24,I)*BMAT(K-24,J)
                    SUM=SUM
     $              +AMAT(K-23,I)*BMAT(K-23,J)+AMAT(K-22,I)*BMAT(K-22,J)
     $              +AMAT(K-21,I)*BMAT(K-21,J)+AMAT(K-20,I)*BMAT(K-20,J)
     $              +AMAT(K-19,I)*BMAT(K-19,J)+AMAT(K-18,I)*BMAT(K-18,J)
     $              +AMAT(K-17,I)*BMAT(K-17,J)+AMAT(K-16,I)*BMAT(K-16,J)
                    SUM=SUM
     $              +AMAT(K-15,I)*BMAT(K-15,J)+AMAT(K-14,I)*BMAT(K-14,J)
     $              +AMAT(K-13,I)*BMAT(K-13,J)+AMAT(K-12,I)*BMAT(K-12,J)
     $              +AMAT(K-11,I)*BMAT(K-11,J)+AMAT(K-10,I)*BMAT(K-10,J)
     $              +AMAT(K-9,I)*BMAT(K-9,J)  +AMAT(K-8,I) *BMAT(K-8,J)
                    SUM=SUM
     $              +AMAT(K-7,I)*BMAT(K-7,J)+AMAT(K-6,I)*BMAT(K-6,J)
     $              +AMAT(K-5,I)*BMAT(K-5,J)+AMAT(K-4,I)*BMAT(K-4,J)
     $              +AMAT(K-3,I)*BMAT(K-3,J)+AMAT(K-2,I)*BMAT(K-2,J)
     $              +AMAT(K-1,I)*BMAT(K-1,J)+AMAT(K,I)  *BMAT(K,J)
300              CONTINUE
              END IF
              IF(NCC16.GT.0) THEN
                 DO 400 KK=1,NCC16
                    K=K+16
                    SUM=SUM
     $              +AMAT(K-15,I)*BMAT(K-15,J)+AMAT(K-14,I)*BMAT(K-14,J)
     $              +AMAT(K-13,I)*BMAT(K-13,J)+AMAT(K-12,I)*BMAT(K-12,J)
     $              +AMAT(K-11,I)*BMAT(K-11,J)+AMAT(K-10,I)*BMAT(K-10,J)
     $              +AMAT(K-9,I)*BMAT(K-9,J)  +AMAT(K-8,I) *BMAT(K-8,J)
                    SUM=SUM
     $              +AMAT(K-7,I)*BMAT(K-7,J)+AMAT(K-6,I)*BMAT(K-6,J)
     $              +AMAT(K-5,I)*BMAT(K-5,J)+AMAT(K-4,I)*BMAT(K-4,J)
     $              +AMAT(K-3,I)*BMAT(K-3,J)+AMAT(K-2,I)*BMAT(K-2,J)
     $              +AMAT(K-1,I)*BMAT(K-1,J)+AMAT(K,I)  *BMAT(K,J)
400              CONTINUE
              END IF
              IF(NCC8.GT.0) THEN
                 DO 500 KK=1,NCC8
                    K=K+8
                    SUM=SUM
     $              +AMAT(K-7,I)*BMAT(K-7,J)+AMAT(K-6,I)*BMAT(K-6,J)
     $              +AMAT(K-5,I)*BMAT(K-5,J)+AMAT(K-4,I)*BMAT(K-4,J)
     $              +AMAT(K-3,I)*BMAT(K-3,J)+AMAT(K-2,I)*BMAT(K-2,J)
     $              +AMAT(K-1,I)*BMAT(K-1,J)+AMAT(K,I)  *BMAT(K,J)
500              CONTINUE
              END IF
              IF(NCC4.GT.0) THEN
                 DO 600 KK=1,NCC4
                    K=K+4
                    SUM=SUM
     $              +AMAT(K-3,I)*BMAT(K-3,J)+AMAT(K-2,I)*BMAT(K-2,J)
     $              +AMAT(K-1,I)*BMAT(K-1,J)+AMAT(K,I)  *BMAT(K,J)
600              CONTINUE
              END IF
              IF(NCC4R.GT.0) THEN
                 DO 700 KK=1,NCC4R
                    K=K+1
                    SUM=SUM+AMAT(K,I)*BMAT(K,J)
700              CONTINUE
              END IF
              CMAT(I,J)=SUM
200        CONTINUE
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE ATBMUL.
C
        END

      SUBROUTINE ATVOV(OVERFL,MAXEXP,N,NUNIT,OUTPUT,AMAT,BVEC,CVEC)
C
C       FEB. 8 ,1991
C
C       THIS SUBROUTINE FINDS THE PRODUCT OF THE TRANSPOSE OF THE
C       MATRIX A AND THE VECTOR B WHERE EACH ENTRY IS CHECKED TO
C       PREVENT OVERFLOWS.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        INTEGER    OUTPUT
      DIMENSION  AMAT(N,N)  ,BVEC(N)    ,CVEC(N)
      LOGICAL    OVERFL     ,WRNSUP
      COMMON/NNES_2/WRNSUP
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,ONE,TEN /0.0D0,1.0D0,10.0D0/
C
      EPS=TEN**(-MAXEXP)
        OVERFL=.FALSE.
C
      DO 100 I=1,N
         SUM=ZERO
         DO 200 J=1,N
            IF(LOG10(ABS(AMAT(J,I))+EPS)+LOG10(ABS(BVEC(J))+EPS)
     $           .GT.MAXEXP) THEN
         OVERFL=.TRUE.
         CVEC(I)=SIGN(TEN**MAXEXP,AMAT(J,I))
     $           *SIGN(ONE,BVEC(J))
         IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
            WRITE(ICOUT,1)
1                   FORMAT(T3,'*',T74,'*')
            WRITE(ICOUT,2) CVEC(I)
2                   FORMAT(T3,'*',4X,'WARNING: COMPONENT IN',
     $              ' MATRIX-VECTOR PRODUCT SET TO ',1PD12.3,T74,'*')
         END IF
         GO TO 101
            END IF
            SUM=SUM+AMAT(J,I)*BVEC(J)
200        CONTINUE
         CVEC(I)=SUM
101        CONTINUE
100     CONTINUE
      RETURN
C
C       LAST CARD OF SUBROUTINE ATVOV.
C
      END
        SUBROUTINE AVMUL(NRADEC,NRAACT,NCDEC ,NCACT ,AMAT  ,BVEC  ,CVEC)
C
C       FEB. 8, 1991
C
C       MATRIX-VECTOR MULTIPLICATION AB=C
C
C       VERSION WITH INNER LOOP UNROLLED TO DEPTHS 32, 16, 8 AND 4
C       EACH ROW OF MATRIX A IS SAVED AS A COLUMN BEFORE USE.
C
C       NRADEC IS 1ST DIM. OF AMAT; NRAACT IS ACTUAL LIMIT FOR 1ST INDEX
C       NCDEC IS COMMON DIMENSION OF AMAT & BVEC; NCACT IS ACTUAL LIMIT
C
C       I.E. NRADEC IS THE NUMBER OF ROWS OF A DECLARED
C            NCDEC IS THE COMMON DECLARED DIMENSION (COLUMNS OF A AND
C            ROWS OF B)
C
C       MODIFICATION OF THE MATRIX MULTIPLIER DONATED BY PROF. JAMES
C       MACKINNON, QUEEN'S UNIVERSITY, KINGSTON, ONTARIO, CANADA
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION AMAT(NRADEC,NCDEC),  BVEC(NCDEC),  CVEC(NRADEC)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO /0.0D0/
C
C       FIND NUMBER OF GROUPS OF SIZE 32, 16 ...
C
        NCC32=NCACT/32
        NCC32R=NCACT-32*NCC32
        NCC16=NCC32R/16
        NCC16R=NCC32R-16*NCC16
        NCC8=NCC16R/8
        NCC8R=NCC16R-8*NCC8
        NCC4=NCC8R/4
        NCC4R=NCC8R-4*NCC4
        DO 100 I=1,NRAACT
C
C          FIND ENTRY FOR VECTOR C.
C
           SUM=ZERO
           K=0
           IF(NCC32.GT.0) THEN
              DO 200 KK=1,NCC32
                 K=K+32
                 SUM=SUM
     $           +AMAT(I,K-31)*BVEC(K-31)+AMAT(I,K-30)*BVEC(K-30)
     $           +AMAT(I,K-29)*BVEC(K-29)+AMAT(I,K-28)*BVEC(K-28)
     $           +AMAT(I,K-27)*BVEC(K-27)+AMAT(I,K-26)*BVEC(K-26)
     $           +AMAT(I,K-25)*BVEC(K-25)+AMAT(I,K-24)*BVEC(K-24)
                 SUM=SUM
     $           +AMAT(I,K-23)*BVEC(K-23)+AMAT(I,K-22)*BVEC(K-22)
     $           +AMAT(I,K-21)*BVEC(K-21)+AMAT(I,K-20)*BVEC(K-20)
     $           +AMAT(I,K-19)*BVEC(K-19)+AMAT(I,K-18)*BVEC(K-18)
     $           +AMAT(I,K-17)*BVEC(K-17)+AMAT(I,K-16)*BVEC(K-16)
                 SUM=SUM
     $           +AMAT(I,K-15)*BVEC(K-15)+AMAT(I,K-14)*BVEC(K-14)
     $           +AMAT(I,K-13)*BVEC(K-13)+AMAT(I,K-12)*BVEC(K-12)
     $           +AMAT(I,K-11)*BVEC(K-11)+AMAT(I,K-10)*BVEC(K-10)
     $           +AMAT(I,K-9)*BVEC(K-9)  +AMAT(I,K-8) *BVEC(K-8)
                 SUM=SUM
     $           +AMAT(I,K-7)*BVEC(K-7)+AMAT(I,K-6)*BVEC(K-6)
     $           +AMAT(I,K-5)*BVEC(K-5)+AMAT(I,K-4)*BVEC(K-4)
     $           +AMAT(I,K-3)*BVEC(K-3)+AMAT(I,K-2)*BVEC(K-2)
     $           +AMAT(I,K-1)*BVEC(K-1)+AMAT(I,K)  *BVEC(K)
200           CONTINUE
           END IF
           IF(NCC16.GT.0) THEN
              DO 300 KK=1,NCC16
                 K=K+16
                 SUM=SUM
     $           +AMAT(I,K-15)*BVEC(K-15)+AMAT(I,K-14)*BVEC(K-14)
     $           +AMAT(I,K-13)*BVEC(K-13)+AMAT(I,K-12)*BVEC(K-12)
     $           +AMAT(I,K-11)*BVEC(K-11)+AMAT(I,K-10)*BVEC(K-10)
     $           +AMAT(I,K-9)*BVEC(K-9)  +AMAT(I,K-8) *BVEC(K-8)
                 SUM=SUM
     $           +AMAT(I,K-7)*BVEC(K-7)+AMAT(I,K-6)*BVEC(K-6)
     $           +AMAT(I,K-5)*BVEC(K-5)+AMAT(I,K-4)*BVEC(K-4)
     $           +AMAT(I,K-3)*BVEC(K-3)+AMAT(I,K-2)*BVEC(K-2)
     $           +AMAT(I,K-1)*BVEC(K-1)+AMAT(I,K)  *BVEC(K)
300           CONTINUE
           END IF
           IF(NCC8.GT.0) THEN
              DO 400 KK=1,NCC8
                 K=K+8
                 SUM=SUM
     $           +AMAT(I,K-7)*BVEC(K-7)+AMAT(I,K-6)*BVEC(K-6)
     $           +AMAT(I,K-5)*BVEC(K-5)+AMAT(I,K-4)*BVEC(K-4)
     $           +AMAT(I,K-3)*BVEC(K-3)+AMAT(I,K-2)*BVEC(K-2)
     $           +AMAT(I,K-1)*BVEC(K-1)+AMAT(I,K)  *BVEC(K)
400           CONTINUE
           END IF
           IF(NCC4.GT.0) THEN
              DO 500 KK=1,NCC4
                 K=K+4
                 SUM=SUM
     $           +AMAT(I,K-3)*BVEC(K-3)+AMAT(I,K-2)*BVEC(K-2)
     $           +AMAT(I,K-1)*BVEC(K-1)+AMAT(I,K)  *BVEC(K)
500           CONTINUE
           END IF
           IF(NCC4R.GT.0) THEN
              DO 600 KK=1,NCC4R
                 K=K+1
                 SUM=SUM+AMAT(I,K)*BVEC(K)
600           CONTINUE
           END IF
           CVEC(I)=SUM
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE AVMUL.
C
        END
      SUBROUTINE BAKDIF(OVERFL,J,N,DELTAJ,TEMPJ,FVEC,
     $                  FVECJ1,JACFDM,XC,FVECEV)
C
C       FEB. 6, 1991
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION JACFDM(N,N)
      DIMENSION FVEC(N),FVECJ1(N),XC(N)
      LOGICAL   OVERFL
      EXTERNAL  FVECEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DELTAJ=TEMPJ-XC(J)
      CALL FVECEV(OVERFL,N,FVECJ1,XC)
      IF(.NOT.OVERFL) THEN
         DO 100 I=1,N
            JACFDM(I,J)=(FVEC(I)-FVECJ1(I))/DELTAJ
100        CONTINUE
      END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE BAKDIF.
C
      END
      SUBROUTINE BNDDIF(OVERFL,J     ,N     ,EPSMCH,BOUNDL,BOUNDU,
     $                    FVECC ,FVECJ1,JACFDM,WV3   ,XC    ,FVECEV)
C
C       FEB. 15, 1991
C
C       FINITE DIFFERENCE CALCULATION WHEN THE BOUNDS FOR COMPONENT J
C       ARE SO CLOSE THAT NEITHER A FORWARD NOR BACKWARD DIFFERENCE
C       CAN BE PERFORMED.
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION  JACFDM(N,N)
      DIMENSION BOUNDL(N) ,BOUNDU(N), FVECC(N)  ,FVECJ1(N)  ,
     $            WV3(N)    ,XC(N)
      LOGICAL   OVERFL
      EXTERNAL  FVECEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO /0.0D0/
C
      EPS3Q=EPSMCH**0.75
C
C       STORE CURRENT
      DO 100 I=1,N
         WV3(I)=FVECC(I)
100     CONTINUE
      XC(J)=BOUNDU(J)
      CALL FVECEV(OVERFL,N,FVECJ1,XC)
      IF(.NOT.OVERFL) THEN
         XC(J)=BOUNDL(J)
         CALL FVECEV(OVERFL,N,FVECC,XC)
         IF(.NOT.OVERFL) THEN
            DO 200 I=1,N
C
C                ENSURE THAT THE JACOBIAN CALCULATION ISN'T JUST NOISE.
C
          IF(FVECJ1(I)-FVECC(I).GT.EPS3Q) THEN
             JACFDM(I,J)=(FVECJ1(I)-FVECC(I))/
     $                          (BOUNDU(J)-BOUNDL(J))
          ELSE
             JACFDM(I,J)=ZERO
          END IF
200           CONTINUE
         END IF
      END IF
      DO 300 I=1,N
         FVECC(I)=WV3(I)
300     CONTINUE
      RETURN
C
C       LAST CARD OF SUBROUTINE BNDDIF.
C
      END
      SUBROUTINE BROYFA(OVERCH,OVERFL,SCLFCH,SCLXCH,MAXEXP,
     $                    N     ,NUNIT ,OUTPUT,EPSMCH,A     ,
     $                    DELF  ,FVEC  ,FVECC ,JAC   ,RDIAG ,
     $                    S     ,SCALEF,SCALEX,T     ,W     ,
     $                    XC    ,XPLUS )
C
C       FEB. 23, 1992
C
C       THE BROYDEN QUASI-NEWTON METHOD IS APPLIED TO THE FACTORED
C       FORM OF THE JACOBIAN.
C
C       NOTE: T AND W ARE TEMPORARY WORKING VECTORS ONLY.
C
C       THE UPDATE OCCURS ONLY IF A SIGNIFICANT CHANGE IN THE
C       JACOBIAN WOULD RESULT, I.E., NOT ALL THE VALUES IN VECTOR W
C       ARE LESS THAN THE THRESHOLD IN MAGNITUDE.  IF THE VECTOR
C       W IS ESSENTIALLY ZERO THEN THE LOGICAL VARIABLE SKIPUP
C       REMAINS SET AT TRUE.
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION JAC(N,N)
      INTEGER          OUTPUT
      DIMENSION        A(N,N)   ,DELF(N) ,FVEC(N)   ,FVECC(N) ,
     $                   RDIAG(N) ,S(N)    ,SCALEF(N) ,SCALEX(N),
     $                   T(N)     ,W(N)    ,XC(N)     ,XPLUS(N)
      LOGICAL          OVERCH   ,OVERFL  ,SCLFCH    ,SCLXCH   ,
     $                   SKIPUP
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,TEN /0.0D0,10.0D0/
C
        OVERFL=.FALSE.
        EPS=TEN**(-MAXEXP)
        SQRTEP=SQRT(EPSMCH)
C
        DO 100 I=1,N
           A(I,I)=RDIAG(I)
           S(I)=XPLUS(I)-XC(I)
100     CONTINUE
C
C       R IS NOW IN THE UPPER TRIANGLE OF A.
C
        SKIPUP=.TRUE.
C
C       THE BROYDEN UPDATE IS CONDENSED INTO THE FORM
C
C       A(NEW) = A(OLD) + T S^
C
C       THE PRODUCT A*S IS FORMED IN TWO STAGES AS R IS IN THE UPPER
C       TRIANGLE OF MATRIX A AND Q^ IS IN JAC.
C
C       FIRST MULTIPLY R*S (A IS CONSIDERED UPPER TRIANGULAR)
C
        CALL UVMUL(N,N,N,N,A,S,T)
C
C       NOTE: THIS T IS TEMPORARY - IT IS THE T FROM BELOW WHICH
C             IS SENT TO SUBROUTINE QRUPDA.
C
        DO 200 I=1,N
         CALL INNERP(OVERCH,OVERFL,MAXEXP,N     ,N     ,N     ,
     $                 NUNIT ,OUTPUT,SUM   ,JAC(N*(I-1)+1,1),T)
           W(I)=SCALEF(I)*(FVEC(I)-FVECC(I))-SUM
C
C          TEST TO ENSURE VECTOR W IS NONZERO.  ANY VALUE GREATER
C          THAN THE THRESHOLD WILL SET SKIPUP TO FALSE.
C
           IF(ABS(W(I)).GT.SQRTEP*SCALEF(I)*(ABS(FVEC(I))+
     $     ABS(FVECC(I)))) THEN
              SKIPUP=.FALSE.
           ELSE
              W(I)=ZERO
           END IF
200     CONTINUE
C
C       IF W(I)=0 FOR ALL I THEN THE UPDATE IS SKIPPED.
C
        IF(.NOT.SKIPUP) THEN
C
C          T=Q^W; Q^ IS IN JAC.
C
         CALL AVMUL(N,N,N,N,JAC,W,T)
         IF(SCLXCH) THEN
            DO 300 I=1,N
               W(I)=S(I)*SCALEX(I)
300         CONTINUE
         ELSE
            CALL MATCOP(N,N,1,1,N,1,S,W)
         END IF
           CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,DENOM,W)
C
C          IF OVERFLOW WOULD OCCUR MAKE NO CHANGE TO JACOBIAN.
C
           IF(OVERFL.OR.LOG10(DENOM+EPS).GT.MAXEXP/2) THEN
              IF(OUTPUT.GT.3) THEN
                 WRITE(ICOUT,1)
1                FORMAT(2X,'*',69X,'*')
                 CALL DPWRST('XXX','WRIT')
                 WRITE(ICOUT,2)
2                FORMAT(2X,'*',4X,'WARNING: JACOBIAN NOT UPDATED',
     $           ' TO AVOID OVERFLOW IN DENOMINATOR OF',1X,'*')
                 CALL DPWRST('XXX','WRIT')
                 WRITE(ICOUT,3)
3                FORMAT(2X,'*',4X,'BROYDEN UPDATE',53X,'*')
                 CALL DPWRST('XXX','WRIT')
              END IF
              RETURN
           ELSE
            DENOM=DENOM*DENOM
           END IF
C
C          IF DENOM IS ZERO AVOID DIVIDE BY ZERO AND CONTINUE WITH
C          SAME JACOBIAN.
C
           IF(DENOM.EQ.ZERO) RETURN
C
C          THE SCALED VERSION OF S REPLACES THE ORIGINAL BEFORE
C          BEING SENT TO QRUPDA.
C
           DO 400 I=1,N
             S(I)=S(I)*SCALEX(I)*SCALEX(I)/DENOM
400        CONTINUE
C
C          UPDATE THE QR DECOMPOSITION USING A SERIES OF GIVENS
C          ROTATIONS.
C
           CALL QRUPDA(OVERFL,MAXEXP,N,EPSMCH,A,JAC,T,S)
C
C          RESET RDIAG AS DIAGONAL OF CURRENT R WHICH IS IN
C          THE UPPER TRIANGE OF A.
C
           DO 500 I=1,N
              RDIAG(I)=A(I,I)
500        CONTINUE
        END IF
C
C       UPDATE THE GRADIENT VECTOR, DELF.  THE NEW Q^ IS IN JAC.
C
C       DELF = (QR)^F = R^Q^F = R^JAC F
C
      IF(SCLFCH) THEN
         DO 600 I=1,N
            W(I)=FVEC(I)*SCALEF(I)
600        CONTINUE
      ELSE
         CALL MATCOP(N,N,1,1,N,1,FVEC,W)
      END IF
        CALL AVMUL(N,N,N,N,JAC,W,T)
        CALL UTBMUL(N,N,1,1,N,N,A,T,DELF)
        RETURN
C
C       LAST CARD OF SUBROUTINE BROYFA.
C
        END
        SUBROUTINE BROYUN(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,
     $                    EPSMCH,FVEC  ,FVECC ,JAC   ,SCALEX,
     $                    WV1   ,XC    ,XPLUS)
C
C       FEB. 23, 1992
C
C       UPDATE THE JACOBIAN USING BROYDEN'S METHOD.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N)
        INTEGER          OUTPUT
        DIMENSION        FVEC(N)  ,FVECC(N) ,SCALEX(N),WV1(N)  ,
     $                   XC(N)    ,XPLUS(N)
        LOGICAL          OVERFL
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,TEN /0.0D0,10.0D0/
C
        EPS=TEN**(-MAXEXP)
        SQRTEP=SQRT(EPSMCH)
C
        DO 100 I=1,N
           WV1(I)=(XPLUS(I)-XC(I))*SCALEX(I)
100     CONTINUE
        CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,DENOM,WV1)
C
C       IF OVERFLOW WOULD OCCUR MAKE NO CHANGE TO JACOBIAN.
C
        IF(OVERFL.OR.LOG10(DENOM+EPS).GT.MAXEXP/2) THEN
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
1             FORMAT(2X,'*',70X,'*')
              CALL DPWRST('XXX','WRIT')
              WRITE(ICOUT,2)
2             FORMAT(2X,'*',4X,'WARNING: JACOBIAN NOT UPDATED',
     $        ' TO AVOID OVERFLOW IN DENOMINATOR OF',1X,'*')
              CALL DPWRST('XXX','WRIT')
              WRITE(ICOUT,3)
3             FORMAT(2X,'*',4X,'BROYDEN UPDATE',52X,'*')
              CALL DPWRST('XXX','WRIT')
           END IF
           RETURN
        ELSE
         DENOM=DENOM*DENOM
        END IF
C
C       IF DENOM IS ZERO, AVOID OVERFLOW, CONTINUE WITH SAME JACOBIAN.
C
        IF(DENOM.EQ.ZERO) RETURN
C
C       UPDATE JACOBIAN BY ROWS.
C
        DO 200 I=1,N
           SUM=ZERO
           DO 300 J=1,N
              SUM=SUM+JAC(I,J)*(XPLUS(J)-XC(J))
300        CONTINUE
           TEMPI=FVEC(I)-FVECC(I)-SUM
C
C          CHECK TO ENSURE THAT SOME MEANINGFUL CHANGE IS BEING MADE
C          TO THE APPROXIMATE JACOBIAN; IF NOT, SKIP UPDATING ROW I.
C
           IF(ABS(TEMPI).GE.SQRTEP*(ABS(FVEC(I))+ABS(FVECC(I))))
     $        THEN
              TEMPI=TEMPI/DENOM
              DO 400 J=1,N
                 JAC(I,J)=JAC(I,J)+TEMPI*
     $                    (XPLUS(J)-XC(J))*SCALEX(J)*SCALEX(J)
400           CONTINUE
           END IF
200     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE BROYUN.
C
        END
        SUBROUTINE CHOLDE(N,MAXADD,MAXFFL,SQRTEP,H,L)
C
C       FEB. 23, 1992
C
C       THIS SUBROUTINE FINDS THE CHOLESKY DECOMPOSITION OF THE
C       MATRIX, H, AND RETURNS IT IN THE LOWER TRIANGLE OF
C       MATRIX, L.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION   L(N,N) ,MAXADD ,MAXFFL ,MINL  ,MINL2 ,
     $                     MINLJJ
        DIMENSION          H(N,N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO /0.0D0/
C
        MINL=SQRT(SQRTEP)*MAXFFL
C
C       MAXFFL EQUALS 0 WHEN THE MATRIX IS KNOWN TO BE POSITIVE
C       DEFINITE.
C
        IF(MAXFFL.EQ.ZERO) THEN
C
C          FIND SQUARE ROOT OF LARGEST MAGNITUDE DIAGONAL ELEMENT
C          AND SET MINL2.
C
           DO 100 I=1,N
              MAXFFL=MAX(MAXFFL,ABS(H(I,I)))
100        CONTINUE
           MAXFFL=SQRT(MAXFFL)
           MINL2=SQRTEP*MAXFFL
        END IF
C
C       MAXADD CONTAINS THE MAXIMUM AMOUNT WHICH IS IMPLICITLY ADDED
C       TO ANY DIAGONAL ELEMENT OF MATRIX H.
C
        MAXADD=ZERO
        DO 200 J=1,N
           SUM=ZERO
           DO 300 I=1,J-1
            SUM=SUM+L(J,I)*L(J,I)
300        CONTINUE
           L(J,J)=H(J,J)-SUM
           MINLJJ=ZERO
           DO 400 I=J+1,N
              SUM=ZERO
              DO 500 K=1,J-1
                 SUM=SUM+L(I,K)*L(J,K)
500           CONTINUE
              L(I,J)=H(J,I)-SUM
              MINLJJ=MAX(MINLJJ,ABS(L(I,J)))
400        CONTINUE
           MINLJJ=MAX(MINLJJ/MAXFFL,MINL)
           IF(L(J,J).GT.MINLJJ*MINLJJ) THEN
C
C             NORMAL CHOLESKY DECOMPOSITION.
C
              L(J,J)=SQRT(L(J,J))
           ELSE
C
C             IMPLICITLY PERTURB DIAGONAL OF H.
C
              IF(MINLJJ.LT.MINL2) THEN
                 MINLJJ=MINL2
              END IF
              MAXADD=MAX(MAXADD,MINLJJ*MINLJJ-L(J,J))
              L(J,J)=MINLJJ
           END IF
           DO 600 I=J+1,N
              L(I,J)=L(I,J)/L(J,J)
600        CONTINUE
200     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE CHOLDE.
C
        END
        SUBROUTINE CHSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                    OUTPUT,L     ,RHS   ,S     ,WV2   )
C
C       FEB. 14, 1991
C
C       THIS SUBROUTINE USES FORWARD/BACKWARD SUBSTITUTION TO SOLVE THE
C       SYSTEM OF LINEAR EQUATIONS:
C
C            (LL^)S=RHS
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION  L(N,N)
        INTEGER           OUTPUT
        DIMENSION         RHS(N) ,S(N)   ,WV2(N)
        LOGICAL           OVERCH ,OVERFL
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        CALL LSOLV(OVERCH,OVERFL,MAXEXP,N,NUNIT,OUTPUT,L,WV2,RHS)
        CALL LTSOLV(OVERCH,OVERFL,MAXEXP,N,NUNIT,OUTPUT,L,S,WV2)
C
        RETURN
C
C       LAST CARD OF SUBROUTINE CHSOLV.
C
        END
        SUBROUTINE CONDNO(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                    OUTPUT,CONNUM,A     ,P     ,PM    ,
     $                    Q     ,RDIAG )
C
C       FEB. 14, 1991
C
C       THIS SUBROUTINE ESTIMATES THE CONDITION NUMBER OF A
C       QR-DECOMPOSED MATRIX USING THE METHOD OF CLINE, MOLER,
C       STEWART AND WILKINSON (SIAM J. N.A. 16 P368 (1979) ).
C
C       IF A POTENTIAL OVERFLOW IS DETECTED AT ANY POINT THEN A
C       CONDITION NUMBER EQUIVALENT TO THAT OF A SINGULAR MATRIX
C       IS ASSIGNED BY THE CALLING SUBROUTINE.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION     A(N,N) ,P(N)  ,PM(N)  ,RDIAG(N)  ,Q(N)
        LOGICAL       OVERCH ,OVERFL
        INTEGER       OUTPUT
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,ONE,TEN /0.0D0,1.0D0,10.0D0/
C
        OVERFL=.FALSE.
        EPS=TEN**(-MAXEXP)
C
        CONNUM=ABS(RDIAG(1))
        DO 100 J=2,N
           TEMP=ZERO
           DO 200 I=1,J-1
              IF(OVERCH) THEN
                 IF(ABS(A(I,J)).GT.TEN**(MAXEXP-1)) THEN
                    OVERFL=.TRUE.
                    RETURN
                 END IF
              END IF
              TEMP=TEMP+ABS(A(I,J))
200        CONTINUE
           TEMP=TEMP+ABS(RDIAG(J))
           CONNUM=MAX(CONNUM,TEMP)
100     CONTINUE
        Q(1)=ONE/RDIAG(1)
        DO 300 I=2,N
           IF(OVERCH) THEN
              IF(LOG10(ABS(Q(1))+EPS)+LOG10(ABS(A(1,I))+EPS)
     $           .GT.MAXEXP) THEN
                 OVERFL=.TRUE.
                 RETURN
              END IF
           END IF
           P(I)=A(1,I)*Q(1)
300     CONTINUE
        DO 400 J=2,N
           IF(OVERCH) THEN
              IF(LOG10(ABS(P(J))+EPS)-LOG10(ABS(RDIAG(J))+EPS)
     $           .GT.MAXEXP) THEN
                 OVERFL=.TRUE.
                 RETURN
              END IF
           END IF
           QP=(ONE-P(J))/RDIAG(J)
           QM=(-ONE-P(J))/RDIAG(J)
           TEMP=ABS(QP)
           TEMPM=ABS(QM)
           DO 500 I=J+1,N
              IF(OVERCH) THEN
                 IF(LOG10(ABS(A(J,I))+EPS)+LOG10(ABS(QM)+EPS)
     $              .GT.MAXEXP) THEN
                    OVERFL=.TRUE.
                    RETURN
                 END IF
              END IF
              PM(I)=P(I)+A(J,I)*QM
              IF(OVERCH) THEN
                 IF(LOG10(ABS(PM(I))+EPS)-LOG10(ABS(RDIAG(I))+EPS)
     $              .GT.MAXEXP) THEN
                    OVERFL=.TRUE.
                    RETURN
                 END IF
              END IF
              TEMPM=TEMPM+(ABS(PM(I))/ABS(RDIAG(I)))
              IF(OVERCH) THEN
                 IF(TEMPM.GT.TEN**(MAXEXP-1)) THEN
                    OVERFL=.TRUE.
                    RETURN
                 END IF
              END IF
              IF(OVERCH) THEN
                 IF(LOG10(ABS(A(J,I))+EPS)+LOG10(ABS(QP)+EPS)
     $              .GT.MAXEXP) THEN
                    OVERFL=.TRUE.
                    RETURN
                 END IF
              END IF
              P(I)=P(I)+A(J,I)*QP
              IF(OVERCH) THEN
                 IF(LOG10(ABS(P(I))+EPS)-LOG10(ABS(RDIAG(I))+EPS)
     $              .GT.MAXEXP) THEN
                    OVERFL=.TRUE.
                    RETURN
                 END IF
              END IF
              TEMP=TEMP+(ABS(P(I))/ABS(RDIAG(I)))
              IF(OVERCH) THEN
                 IF(TEMP.GT.TEN**(MAXEXP-1)) THEN
                    OVERFL=.TRUE.
                    RETURN
                 END IF
              END IF
500        CONTINUE
           IF(TEMP.GE.TEMPM) THEN
              Q(J)=QP
           ELSE
              Q(J)=QM
              DO 600 I=J+1,N
                 P(I)=PM(I)
600           CONTINUE
           END IF
400     CONTINUE
        QNORM=ZERO
        DO 700 J=1,N
           QNORM=QNORM+ABS(Q(J))
           IF(OVERCH) THEN
              IF(QNORM.GT.TEN**(MAXEXP-1)) THEN
                 OVERFL=.TRUE.
                 RETURN
              END IF
           END IF
700     CONTINUE
        IF(LOG10(CONNUM)-LOG10(QNORM).GT.MAXEXP) THEN
           OVERFL=.TRUE.
           RETURN
        END IF
        CONNUM=CONNUM/QNORM
        CALL RSOLV(OVERCH,OVERFL,MAXEXP,N,NUNIT,OUTPUT,A,RDIAG,Q)
        IF(OVERFL) RETURN
        QNORM=ZERO
        DO 800 J=1,N
           QNORM=QNORM+ABS(Q(J))
           IF(OVERCH) THEN
              IF(QNORM.GT.TEN**(MAXEXP-1)) THEN
                 OVERFL=.TRUE.
                 RETURN
              END IF
           END IF
800     CONTINUE
        CONNUM=CONNUM*QNORM
        RETURN
C
C       LAST CARD OF SUBROUTINE CONDNO.
C
        END
        SUBROUTINE DELCAU(CAUCHY,OVERCH,OVERFL,ITNUM ,MAXEXP,
     $                    N     ,NUNIT ,OUTPUT,BETA  ,CAULEN,
     $                    DELTA ,EPSMCH,MAXSTP,NEWLEN,SQRTZ ,
     $                    A     ,DELF  ,SCALEX,WV1   )
C
C       FEB. 23, 1992
C
C       THIS SUBROUTINE ESTABLISHES AN INITIAL TRUST REGION, DELTA,
C       IF ONE IS NOT SPECIFIED BY THE USER AND FINDS THE LENGTH OF
C       THE SCALED CAUCHY STEP, CAULEN, AT EACH STEP IF THE DOUBLE
C       DOGLEG OPTION IS BEING USED.
C
C       THE USER HAS TWO CHOICES FOR THE INITIAL TRUST REGION:
C          1)  BASED ON THE SCALED CAUCHY STEP
C          2)  BASED ON THE SCALED NEWTON STEP
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION  MAXSTP   ,NEWLEN
        INTEGER           OUTPUT
        DIMENSION         A(N,N)   ,DELF(N) ,SCALEX(N) ,WV1(N)
        LOGICAL           CAUCHY   ,OVERCH  ,OVERFL
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,THREE,TEN/0.0D0,3.0D0,10.0D0/
C
        OVERFL=.FALSE.
        EPS=TEN**(-MAXEXP)
C
C       IF DELTA IS NOT GIVEN EVALUATE IT USING EITHER THE CAUCHY
C       STEP OR THE NEWTON STEP AS SPECIFIED BY THE USER.
C
C       THE SCALED CAUCHY LENGTH, CAULEN, IS REQUIRED IN TWO CASES.
C       1) WHEN SELECTED AS THE CRITERION FOR THE INITIAL DELTA
C       2) IN THE DOUBLE DOGLEG STEP REGARDLESS OF (1)
C
       IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
1          FORMAT(T3,'*',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
2          FORMAT(T3,'*',4X,'DETERMINATION OF SCALED CAUCHY',
     $     ' STEP LENGTH, CAULEN',T74,'*')
           CALL DPWRST('XXX','BUG')
      END IF
C
C       FIND FACTOR WHICH GIVES CAUCHY POINT WHEN MULTPLYING
C       STEEPEST DESCENT DIRECTION, DELF.
C
C       CAULEN= ZETA**1.5/BETA
C             =  SQRTZ**3/BETA
C
        DO 100 I=1,N
           WV1(I)=DELF(I)/SCALEX(I)
100     CONTINUE
        CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,SQRTZ,WV1)
        IF(OVERFL) THEN
           CAULEN=TEN**MAXEXP
           IF(OUTPUT.GT.4) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,3) SQRTZ
3             FORMAT(T3,'*',7X,'ZETA SET TO ',1PD11.3,' TO'
     $        ' AVOID OVERFLOW',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,4) CAULEN
4             FORMAT(T3,'*',7X,'SCALED CAUCHY LENGTH, CAULEN ',
     $        'SET TO',1PD9.2,' TO AVOID OVERFLOW',T74,'*')
              CALL DPWRST('XXX','BUG')
              IF(ITNUM.EQ.1) THEN
                 WRITE(ICOUT,5)
5                FORMAT(T3,'*',7X,'THE PROBLEM SHOULD BE RESCALED',
     $           ' OR A NEW STARTING POINT CHOSEN',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,6)
6                FORMAT(T3,'*',7X,'EXECUTION CONTINUES WITH',
     $           ' SUBSTITUTIONS AS LISTED ABOVE',T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
           END IF
        ELSE
           IF(OUTPUT.GT.4) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,7) SQRTZ
7             FORMAT(T3,'*',7X,'SQUARE ROOT OF ZETA, SQRTZ: ',
     $        1PD12.3,T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
        END IF
C
C       NOTE: THE LOWER TRIANGLE OF MATRIX A NOW CONTAINS THE
C             TRANSPOSE OF R WHERE A=QR.
C
      BETA=ZERO
        DO 200 I=1,N
         TEMP=ZERO
           DO 300 J=I,N
              IF(OVERCH) THEN
          IF(LOG10(ABS(A(J,I))+EPS)+LOG10(ABS(DELF(J))+EPS).
     $              GT.MAXEXP) THEN
             CAULEN=SQRT(EPSMCH)
                    GO TO 301
                 END IF
              END IF
            TEMP=TEMP+A(J,I)*DELF(J)/(SCALEX(J)*SCALEX(J))
300        CONTINUE
           BETA=BETA+TEMP*TEMP
200     CONTINUE
        IF(OUTPUT.GT.4) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,8) BETA
8          FORMAT(T3,'*',7X,'BETA: ',1PD11.3,6X,'NOTE: ',
     $     'CAULEN=ZETA**1.5/BETA',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
C
C       AVOID OVERFLOWS IN FINDING CAULEN.
C
      IF(THREE*LOG10(SQRTZ+EPS)-LOG10(BETA+EPS).LT.MAXEXP
     $  .AND.(.NOT.OVERFL).AND.BETA.NE.ZERO) THEN
C
C          NORMAL DETERMINATION.
C
         CAULEN=SQRTZ*SQRTZ*SQRTZ/BETA
C
C          THIS STEP AVOIDS DIVIDE BY ZERO IN DOGLEG IN THE
C          (RARE) CASE WHERE DELF(I)=0 FOR ALL I BUT THE
C          POINT IS NOT YET A SOLUTION - MOST LIKELY A BAD
C          STARTING ESTIMATE.
C
         CAULEN=MAX(CAULEN,TEN**(-MAXEXP))
C
        ELSE
C
C          SUBSTITUTION TO AVOID OVERFLOW.
C
         CAULEN=TEN**MAXEXP
        END IF
301     CONTINUE
        IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,9) CAULEN
9          FORMAT(T3,'*',7X,'SCALED CAUCHY LENGTH, CAULEN: ',
     $     1PD12.3,T74,'*')
           CALL DPWRST('XXX','BUG')
        END IF
C
C       ESTABLISH INITIAL TRUST REGION IF NEEDED.
C
      IF(DELTA.LT.ZERO) THEN
C
C          USE DISTANCE TO CAUCHY POINT OR LENGTH OF NEWTON STEP.
C
           IF(CAUCHY) THEN
              DELTA=MIN(CAULEN,MAXSTP)
           ELSE
              DELTA=MIN(NEWLEN,MAXSTP)
           END IF
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,10) DELTA
10            FORMAT(T3,'*',7X,'INITIAL TRUST REGION SIZE, DELTA: ',
     $        1PD12.3,T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE DELCAU.
C
      END
        SUBROUTINE DEUFLS(ABORT ,DEUFLH,GEOMS ,OVERCH,OVERFL,
     $                    QNFAIL,QRSING,RESTRT,SCLFCH,SCLXCH,
     $                    ACPCOD,ACPTCR,CONTYP,ITNUM ,JUPDM ,
     $                    MAXEXP,MAXLIN,N     ,NFUNC ,NUNIT ,
     $                    OUTPUT,QNUPDM,STOPCR,ALPHA ,CONFAC,
     $                    DELFTS,EPSMCH,FCNMAX,FCNNEW,FCNOLD,
     $                    LAMBDA,NEWMAX,SBRNRM,SIGMA ,A     ,
     $                    ASTORE,BOUNDL,BOUNDU,DELF  ,FVEC  ,
     $                    HHPI  ,JAC   ,RDIAG ,RHS   ,S     ,
     $                    SBAR  ,SCALEF,SCALEX,SN    ,WV2   ,
     $                    XC    ,XPLUS ,FVECEV)
C
C       FEB. 23, 1992
C
C       THIS SUBROUTINE CONDUCTS A LINE SEARCH IN THE NEWTON
C       DIRECTION IF NO CONSTRAINTS ARE VIOLATED.  IF THE FIRST
C       TRIAL IS A FAILURE THERE ARE TWO TYPES OF LINE SEARCH
C       AVAILABLE.
C         1)  REDUCE THE RELAXATION FACTOR, LAMBDA, TO
C             SIGMA*LAMBDA WHERE SIGMA IS USER-SPECIFIED
C             (GEOMETRIC LINE SEARCH)
C         2)  AT THE FIRST STEP MINIMIZE A QUADRATIC THROUGH
C             THE OBJECTIVE FUNCTION AT THE CURRENT POINT AND
C             TRIAL ESTIMATE (WHICH MUST BE A FAILURE) WITH
C             INITIAL SLOPE DELFTS.  AT SUBSEQUENT STEPS MINI-
C             MIZE A CUBIC THROUGH THE OBJECTIVE FUNCTION AT
C             THE TWO MOST RECENT FAILURES AND THE CURRENT
C             POINT, AGAIN USING THE INITIAL SLOPE, DELFTS.
C
C       CONVIO  INDICATES A CONSTRAINT VIOLATION BY ONE OR MORE
C               COMPONENTS
C       FRSTST  INDICATES THE FIRST STEP IN THE LINE SEARCH.
C
C       RATIO   RATIO OF PROPOSED STEP LENGTH IN (I)TH DIRECTION
C               TO DISTANCE FROM (I)TH COMPONENT TO BOUNDARY
C               VIOLATED
C       RATIOM  MINIMUM OF RATIOS FOR ALL CONSTAINTS VIOLATED
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N) ,LAMBDA     ,LAMPRE   ,LAMTMP   ,
     $                   NEWMAX
        INTEGER          ACPCOD   ,ACPTCR     ,CONTYP   ,OUTPUT   ,
     $                   STOPCR   ,QNUPDM
        DIMENSION        A(N,N)   ,ASTORE(N,N),BOUNDL(N),BOUNDU(N),
     $                   FVEC(N)  ,HHPI(N)    ,RDIAG(N) ,RHS(N)   ,
     $                   S(N)     ,DELF(N)    ,SBAR(N)  ,SCALEF(N),
     $                   SCALEX(N),SN(N)      ,WV2(N)   ,XC(N)    ,
     $                   XPLUS(N)
        LOGICAL          ABORT    ,CONVIO     ,DEUFLH   ,FRSTST   ,
     $                   GEOMS    ,OVERCH     ,OVERFL   ,QNFAIL   ,
     $                   QRSING   ,RESTRT     ,SCLFCH   ,SCLXCH   ,
     $                   WRNSUP
        COMMON/NNES_2/WRNSUP
        EXTERNAL FVECEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,POINT1,POINT5,ONE,TWO,THREE,TEN /0.0D0,0.1D0,0.5D0,
     $  1.0D0,2.0D0,3.0D0,10.0D0/
C
      FRSTST=.TRUE.
        OVERFL=.FALSE.
        EPS=TEN**(-MAXEXP)
        DO 100 K=1,MAXLIN
           RATIOM=ONE
C
           CONVIO=.FALSE.
C
C          FIND TRIAL POINT AND CHECK IF CONSTRAINTS VIOLATED (IF
C          CONTYP IS NOT EQUAL TO ZERO).
C
           DO 200 I=1,N
C
C             NOTE: WV2 MARKS VIOLATIONS.  WV2(I) CHANGES TO
C                   1 FOR LOWER BOUND VIOLATIONS AND TO 2 FOR
C                   FOR UPPER BOUND VIOLATIONS.  CONSTRAINT VIOL-
C                   ATIONS CAN ONLY OCCUR AT THE FIRST STEP.
C
              WV2(I)=-ONE
              XPLUS(I)=XC(I)+LAMBDA*SN(I)
              IF(CONTYP.GT.0.AND.FRSTST) THEN
                 IF(XPLUS(I).LT.BOUNDL(I)) THEN
                    CONVIO=.TRUE.
                    WV2(I)=ONE
                 ELSEIF(XPLUS(I).GT.BOUNDU(I)) THEN
                    CONVIO=.TRUE.
                    WV2(I)=TWO
                 ELSE
                    WV2(I)=-ONE
                 END IF
              END IF
200        CONTINUE
C
C          IF CONSTRAINTS ARE VIOLATED FIRST REDUCE THE STEP
C          SIZES FOR THE VIOLATING COMPONENTS TO OBTAIN A
C          FEASIBLE POINT.  IF THE DIRECTION TO THIS MODIFIED
C          POINT IS NOT A DESCENT DIRECTION OR IF THE MODIFIED
C          STEP DOES NOT LEAD TO AN ACCEPTABLE POINT THEN RETURN
C          TO THE NEWTON DIRECTION AND START A LINE SEARCH AT A
C          FEASIBLE POINT WHERE THE COMPONENT WHICH HAS THE
C          SMALLEST VALUE OF RATIO (DEFINED BELOW) IS TAKEN TO
C          "CONFAC" OF THE DISTANCE TO THE BOUNDARY.  DEFAULT
C          VALUE OF CONFAC IS 0.95.
C
           IF(CONVIO) THEN
              IF(OUTPUT.GT.3) THEN
                 WRITE(ICOUT,1)
1                FORMAT(T3,'*',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,2) K
2                FORMAT(T3,'*',10X,'LINE SEARCH STEP:',I3,T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,3) LAMBDA
3                FORMAT(T3,'*',10X,'LAMBDA FOR ATTEMPTED STEP: ',
     $           1PD12.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,4)
4                FORMAT(T3,'*',10X,'CONSTRAINT VIOLATED',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,5)
5                FORMAT(T3,'*',10X,'TRIAL ESTIMATES (VIOLATIONS',
     $           ' MARKED)',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 DO 300 I=1,N
                    IF(WV2(I).GT.ZERO) THEN
                       WRITE(ICOUT,6) I,XPLUS(I)
6                      FORMAT(T3,'*',13X,'XPLUS(',I3,') = ',1PD12.3,
     $                 2X,'*',T74,'*')
                       CALL DPWRST('XXX','BUG')
                    ELSE
                       WRITE(ICOUT,7) I,XPLUS(I)
7                      FORMAT(T3,'*',13X,'XPLUS(',I3,') = ',1PD12.3,
     $                 T74,'*')
                       CALL DPWRST('XXX','BUG')
                    END IF
300              CONTINUE
              END IF
              DO 400 I=1,N
                 IF(WV2(I).GT.ZERO) THEN
C
C                   FIND RATIO FOR THIS VIOLATING COMPONENT.
C
                    IF(WV2(I).EQ.ONE) THEN
                       RATIO=-(XC(I)-BOUNDL(I))/
     $                        (XPLUS(I)-XC(I))
                    ELSEIF(WV2(I).EQ.TWO) THEN
                       RATIO=(BOUNDU(I)-XC(I))/
     $                       (XPLUS(I)-XC(I))
                    END IF
C
C                   NOTE: THIS LINE IS FOR OUTPUT PURPOSES ONLY.
C
                    WV2(I)=RATIO
C
                    RATIOM=MIN(RATIOM,RATIO)
                    IF(RATIO.GT.POINT5) THEN
                       S(I)=CONFAC*RATIO*LAMBDA*SN(I)
                    ELSE
C
C                      WITHIN BUFFER ZONE - ONLY TAKE 1/2
C                      OF THE STEP YOU WOULD TAKE OTHERWISE.
C
                       S(I)=POINT5*CONFAC*RATIO*LAMBDA*SN(I)
                    END IF
C
C                   ESTABLISH MODIFIED TRIAL POINT.
C
                    XPLUS(I)=XC(I)+S(I)
                 ELSE
C
C                   FOR NONVIOLATORS XPLUS REMAINS UNCHANGED BUT
C                   THE COMPONENT OF S IS LOADED TO CHECK THE
C                   DIRECTIONAL DERIVATIVE.
C
                    S(I)=LAMBDA*SN(I)
                 END IF
400           CONTINUE
              IF(OUTPUT.GT.3) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,8)
8                FORMAT(T3,'*',7X,'NEW S AND XPLUS VECTORS',
     $           ' (WITH RATIOS FOR VIOLATIONS)',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,9)
9                FORMAT(T3,'*',7X,'NOTE: RATIOS ARE RATIO OF',
     $           ' LENGTH TO BOUNDARY FROM CURRENT',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,10)
10               FORMAT(T3,'*',13X,'X VECTOR TO MAGNITUDE OF',
     $           ' CORRESPONDING PROPOSED STEP',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 DO 500 I=1,N
                    IF(WV2(I).LT.ZERO) THEN
                       WRITE(ICOUT,11) I,S(I),I,XPLUS(I)
11                     FORMAT(T3,'*',7X,'S(',I3,') = ',1PD12.3,4X,
     $                 'XPLUS(',I3,') = ',1PD12.3,T74,'*')
                       CALL DPWRST('XXX','BUG')
                    ELSE
                       WRITE(ICOUT,12) I,S(I),I,XPLUS(I),WV2(I)
12                     FORMAT(T3,'*',7X,'S(',I3,') = ',1PD12.3,4X,
     $                 'XPLUS(',I3,') = ',1PD12.3,1X,1PD11.3,T74,'*')
                       CALL DPWRST('XXX','BUG')
                    END IF
500              CONTINUE
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,13) RATIOM
13               FORMAT(T3,'*',7X,'MINIMUM OF RATIOS, RATIOM: ',
     $           1PD12.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
C
C             CHECK DIRECTIONAL DERIVATIVE FOR MODIFIED POINT, DLFTSM.
C
            CALL INNERP(OVERCH,OVERFL,MAXEXP,N     ,N     ,N     ,
     $                    NUNIT ,OUTPUT,DLFTSM,DELF  ,S     )
              OVERFL=.FALSE.
              IF(OUTPUT.GT.3) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,14) DLFTSM
14               FORMAT(T3,'*',7X,'INNER PRODUCT OF DELF AND S FOR',
     $           ' MODIFIED S: ',1PD12.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
C
C             IF INNER PRODUCT IS POSITIVE RETURN TO NEWTON DIRECTION
C             AND CONDUCT A LINE SEARCH WITHIN THE FEASIBLE REGION.
C
              IF(DLFTSM.GT.ZERO) THEN
                 IF(OUTPUT.GT.3) THEN
                    WRITE(ICOUT,15)
15                  FORMAT(T3,'*',7X,'DELFTS > 0','  START LINE',
     $              ' SEARCH AT LAMBDA=CONFAC*LAMBDA*RATIOM',T74,'*')
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,16)
16                  FORMAT(T3,'*',7X,'NOTE: NO TRIAL POINT WAS',
     $              ' EVALUATED AT THIS STEP OF LINE SEARCH',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
C
C                THE STARTING POINT IS SET AT JUST INSIDE THE MOST
C                VIOLATED BOUNDARY.
C
                 LAMBDA=CONFAC*RATIOM*LAMBDA
C
C                LAMBDA IS ALREADY SET - SKIP NORMAL PROCEDURE.
C
                 GO TO 101
              END IF
C
           END IF
C
C          NO CONSTRAINTS VIOLATED - EVALUATE RESIDUAL VECTOR
C          AT NEW POINT.
C
           CALL FVECEV(OVERFL,N,FVEC,XPLUS)
           NFUNC=NFUNC+1
C
C          CHECK FOR OVERFLOW IN FUNCTION VECTOR EVALUATION.
C          IF SO, REDUCE STEP LENGTH AND CONTINUE LINE SEARCH.
C
           IF(OVERFL) THEN
C
              IF(OUTPUT.GT.3) THEN
                 WRITE(ICOUT,17)
17               FORMAT(T3,'*',7X,'OVERFLOW IN FUNCTION VECTOR',
     $           ' - STEP LENGTH REDUCED',T74,'*')
                 CALL DPWRST('XXX','BUG')
C
C                FORCE STEP TO BE WITHIN CONSTRAINTS - DON'T CALL
C                THIS THE FIRST STEP, I.E. FRSTST STAYS AT TRUE.
C
              END IF
              IF(CONVIO) THEN
                 LAMBDA=RATIOM*CONFAC*LAMBDA
              ELSE
                 LAMBDA=SIGMA*LAMBDA
              END IF
C
C             LAMBDA IS ALREADY SET - SKIP NORMAL PROCEDURE.
C
              GO TO 101
           END IF
C
C          EVALUATE (POSSIBLY SCALED) OBJECTIVE FUNCTION AT NEW POINT.
C
           CALL FCNEVL(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,
     $                 EPSMCH,FCNNEW,FVEC  ,SCALEF,WV2   )
           IF(OVERFL) THEN
C
              IF(OUTPUT.GT.3) THEN
                 WRITE(ICOUT,18)
18               FORMAT(T3,'*',7X,'OVERFLOW IN OBJECTIVE FUNCTION',
     $           ' - STEP LENGTH REDUCED',T74,'*')
                 CALL DPWRST('XXX','BUG')
C
C                FORCE STEP TO BE WITHIN CONSTRAINTS - DON'T CALL
C                THIS THE FIRST STEP, I.E. FRSTST STAYS AT TRUE.
C
              END IF
              IF(CONVIO) THEN
                 LAMBDA=RATIOM*CONFAC*LAMBDA
              ELSE
                 LAMBDA=SIGMA*LAMBDA
              END IF
              GO TO 101
           END IF
C
C          IF DEUFLHARD'S METHOD IS BEING USED FOR EITHER
C          RELAXATION FACTOR INITIALIZATION OR THE SECOND
C          ACCEPTANCE CRITERION THEN EVALUATE SBAR.  EVALU-
C          ATION METHOD DEPENDS UPON WHETHER THE JACOBIAN
C          WAS PERTURBED IN THE SOLUTION OF THE LINEAR SYSTEM.
C          LOGICAL VARIABLE QRSING IS TRUE IF PERTURBATION
C          TOOK PLACE.
C
           IF(DEUFLH.OR.ACPTCR.EQ.12) THEN
            IF(QRSING) THEN
C
C                FORM -J^F AS RIGHT HAND SIDE - METHOD DEPENDS ON
C                WHETHER QNUPDM EQUALS 0 OR 1 IF A QUASI-NEWTON
C                UPDATE IS BEING USED.  IF JUPDM IS 0 THEN THE NEWTON
C                STEP HAS BEEN FOUND IN SUBROUTINE NSTPUN.
C
          IF(JUPDM.EQ.0.OR.QNUPDM.EQ.0) THEN
C
C                   UNSCALED JACOBIAN IN MATRIX JAC.
C
             DO 600 I=1,N
                IF(SCLFCH) THEN
              WV2(I)=-FVEC(I)*SCALEF(I)*SCALEF(I)
                ELSE
              WV2(I)=-FVEC(I)
                END IF
600                 CONTINUE
                    CALL ATBMUL(N,N,1,1,N,N,JAC,WV2,RHS)
                 ELSE
C
C                   R IN UPPER TRIANGLE OF A PLUS RDIAG AND Q^ IN JAC
C                   - FROM QR DECOMPOSITION OF SCALED JACOBIAN.
C
                    DO 700 I=1,N
                       WV2(I)=ZERO
                       DO 800 J=1,N
                          WV2(I)=WV2(I)-JAC(I,J)*FVEC(J)*SCALEF(J)
800                    CONTINUE
700                 CONTINUE
                    RHS(1)=RDIAG(1)*WV2(1)
                    DO 900 J=2,N
                       RHS(J)=ZERO
                       DO 1000 I=1,J-1
                          RHS(J)=RHS(J)+A(I,J)*WV2(I)
1000                   CONTINUE
                       RHS(J)=RHS(J)+RDIAG(J)*WV2(J)
900                 CONTINUE
                 END IF
                 CALL CHSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                       OUTPUT,A     ,RHS   ,SBAR  ,WV2   )
              ELSE
C
C                RIGHT HAND SIDE IS -FVEC.
C
                 IF(QNUPDM.EQ.0.OR.JUPDM.EQ.0) THEN
C
C                   QR DECOMPOSITION OF SCALED JACOBIAN STORED IN
C                   ASTORE.
C
                    DO 1100 I=1,N
                       SBAR(I)=-FVEC(I)*SCALEF(I)
1100                CONTINUE
                    CALL QRSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                          OUTPUT,ASTORE,HHPI  ,RDIAG ,SBAR  )
                 ELSE
C
C                   SET UP RIGHT HAND SIDE - MULTIPLY -FVEC BY Q^
C                   (STORED IN JAC).
C
                    DO 1200 I=1,N
                       WV2(I)=-FVEC(I)*SCALEF(I)
1200                CONTINUE
                    CALL AVMUL(N,N,N,N,JAC,WV2,SBAR)
                    CALL RSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                         OUTPUT,A     ,RDIAG ,SBAR  )
                 END IF
              END IF
C
C             NORM OF SCALED SBAR IS NEEDED FOR SECOND ACCEPTANCE TEST.
C
              IF(ACPTCR.EQ.12) THEN
          DO 1300 I=1,N
             WV2(I)=SCALEX(I)*SBAR(I)
1300             CONTINUE
          CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,SBRNRM,WV2)
              END IF
           END IF
C
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,2) K
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,3) LAMBDA
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              IF(.NOT.CONVIO) THEN
                 WRITE(ICOUT,19)
19               FORMAT(T3,'*',10X,'NEW COMPONENT/FCN VECTORS',
     $           '  (XPLUS(I)=XC(I)+LAMBDA*SN(I))',T74,'*')
                 CALL DPWRST('XXX','BUG')
              ELSE
                 WRITE(ICOUT,20)
20               FORMAT(T3,'*',10X,'NEW FUNCTION VECTORS',
     $           ' AT MODIFIED POINT',T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
            DO 1400 I=1,N
                 WRITE(ICOUT,21) I,XPLUS(I),I,FVEC(I)
21               FORMAT(T3,'*',10X,'XPLUS(',I3,') = ',1PD12.3,
     $           5X,'FVEC(',I3,') = ',1PD12.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
1400          CONTINUE
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            IF(.NOT.SCLFCH) THEN
               WRITE(ICOUT,22) FCNNEW
22             FORMAT(T3,'*',10X,'OBJECTIVE FUNCTION VALUE',
     $         ' AT XPLUS: .........'1PD12.3,T74,'*')
               CALL DPWRST('XXX','BUG')
            ELSE
               WRITE(10,23) FCNNEW
23             FORMAT(T3,'*',10X,'SCALED OBJECTIVE FUNCTION VALUE',
     $         ' AT XPLUS: ..'1PD12.3,T74,'*')
               CALL DPWRST('XXX','BUG')
            END IF
            WRITE(ICOUT,24) FCNMAX+ALPHA*LAMBDA*DELFTS
24          FORMAT(T3,'*',10X,'FCNMAX+ALPHA*LAMBDA*DELFTS:',
     $      ' ................',1PD12.3,T74,'*')
            CALL DPWRST('XXX','BUG')
            IF(DEUFLH.OR.ACPTCR.EQ.12) THEN
          IF(ITNUM .GT.0) THEN
             IF(.NOT.SCLXCH) THEN
                WRITE(ICOUT,1)
                CALL DPWRST('XXX','BUG')
                WRITE(ICOUT,25)
25                     FORMAT(T3,'*',10X,'DEUFLHARD SBAR VECTOR',
     $                 T74,'*')
                CALL DPWRST('XXX','BUG')
                WRITE(ICOUT,1)
                CALL DPWRST('XXX','BUG')
                DO 1500 I=1,N
                   WRITE(ICOUT,26) I,SBAR(I)
26                 FORMAT(T3,'*',10X,'SBAR(',I3,') = ',
     $                    1PD12.3,T74,'*')
                   CALL DPWRST('XXX','BUG')
1500            CONTINUE
             ELSE
                WRITE(ICOUT,1)
                CALL DPWRST('XXX','BUG')
                WRITE(ICOUT,27)
27              FORMAT(T3,'*',10X,'DEUFLHARD SBAR VECTOR',
     $                 14X,'IN SCALED X UNITS',T74,'*')
                WRITE(ICOUT,1)
                CALL DPWRST('XXX','BUG')
                DO 1600 I=1,N
                   WRITE(ICOUT,28) I,SBAR(I),I,SCALEX(I)*SBAR(I)
28                 FORMAT(T3,'*',10X,'SBAR(',I3,') = ',1PD12.3,
     $                    8X,'SBAR(',I3,') = ',1PD12.3,T74,'*')
                   CALL DPWRST('XXX','BUG')
1600            CONTINUE
             END IF
          END IF
              END IF
              IF(ACPTCR.EQ.12) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 IF(.NOT.SCLXCH) THEN
                    WRITE(ICOUT,29) SBRNRM
29                  FORMAT(T3,'*',10X,'VALUE OF SBRNRM',
     $              ' AT XPLUS: ..................'1PD12.3,T74,'*')
                    CALL DPWRST('XXX','BUG')
                 ELSE
                    WRITE(ICOUT,30) SBRNRM
30                  FORMAT(T3,'*',10X,'VALUE OF SCALED SBRNRM',
     $              ' AT XPLUS: ...........'1PD12.3,T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 WRITE(ICOUT,31) NEWMAX
31               FORMAT(T3,'*',10X,'NEWMAX:',' ..............',
     $           '......................',1PD12.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
           END IF
C
C          CHECK FOR ACCEPTABLE STEP.
C
           IF(FCNNEW.LT.FCNMAX+ALPHA*LAMBDA*DELFTS) THEN
              ACPCOD=1
C
C             NOTE: STEP WILL BE ACCEPTED REGARDLESS OF NEXT TEST.
C                   THIS SECTION IS FOR BOOKKEEPING ONLY.
C
              IF(ACPTCR.EQ.12) THEN
                 IF(SBRNRM.LT.NEWMAX) THEN
                    ACPCOD=12
                 END IF
              END IF
C
              RETURN
           END IF
           IF(ACPTCR.EQ.12.AND.SBRNRM.LT.NEWMAX) THEN
              ACPCOD=2
              RETURN
           END IF
C
C          FAILURE OF STEP ACCEPTANCE TEST.
C
           IF(CONVIO) THEN
              LAMBDA=CONFAC*RATIOM*LAMBDA
C
C             LAMBDA IS ALREADY SET - SKIP NORMAL PROCEDURE.
C
              GO TO 101
           END IF
           IF(LAMBDA.EQ.ZERO) THEN
              IF(OUTPUT.GT.0) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,32)
32               FORMAT(T3,'*',7X,'LAMBDA IS 0.0:  NO PROGRESS',
     $           ' POSSIBLE - CHECK BOUNDS OR START',T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
              ABORT=.TRUE.
              RETURN
           END IF
           IF(GEOMS) THEN
C
C             GEOMETRIC LINE SEARCH
C
              LAMBDA=SIGMA*LAMBDA
C
           ELSE
C
              IF(FRSTST) THEN
                 FRSTST=.FALSE.
C
C                FIND MINIMUM OF QUADRATIC AT FIRST STEP.
C
                 LAMTMP=-(LAMBDA*LAMBDA)*DELFTS/
     $                  (TWO*(FCNNEW-FCNOLD-LAMBDA*DELFTS))
                 IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,33) LAMTMP
33                  FORMAT(T3,'*',13X,'TEMPORARY LAMBDA FROM',
     $              ' QUADRATIC MODEL:  ',1PD11.3,T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
C
              ELSE
C
C                FIND MINIMUM OF CUBIC AT SUBSEQUENT STEPS.
C
                 FACTOR=ONE/(LAMBDA-LAMPRE)
                 IF(LAMBDA*LAMBDA.EQ.ZERO) THEN
                    LAMBDA=SIGMA*LAMBDA
C
C                   NOTE: IF THIS LAMBDA**2 WAS ZERO ANY SUBSEQUENT
C                         LAMBDA**2 WILL ALSO BE ZERO.
C
                    GO TO 101
                 END IF
                 ACUBIC=FACTOR*((ONE/LAMBDA*LAMBDA)*(FCNNEW-FCNOLD-
     $           LAMBDA*DELFTS)-((ONE/LAMPRE*LAMBDA)*(FPLPRE-
     $           FCNOLD-LAMPRE*DELFTS)))
                 BCUBIC=FACTOR*((-LAMPRE/LAMBDA*LAMBDA)*(FCNNEW-FCNOLD-
     $           LAMBDA*DELFTS)+((LAMBDA/LAMPRE*LAMBDA)*(FPLPRE-
     $           FCNOLD-LAMPRE*DELFTS)))
                 IF(TWO*LOG10(ABS(BCUBIC)+EPS).GT.DBLE(MAXEXP))
     $              THEN
                    LAMTMP=SIGMA*LAMBDA
                    IF(OUTPUT.GT.4) THEN
                       WRITE(ICOUT,1)
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,34)
34                     FORMAT(T3,'*',13X,'POTENTIAL OVERFLOW IN'
     $                 ' CALCULATING TRIAL LAMBDA FROM',T74,'*')
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,35)
35                     FORMAT(T3,'*',13X,'CUBIC MODEL - LAMBDA',
     $                 ' SET TO SIGMA*LAMBDA',T74,'*')
                       CALL DPWRST('XXX','BUG')
                    END IF
                 ELSE
                    DISC=BCUBIC*BCUBIC-THREE*ACUBIC*DELFTS
                    IF(ABS(ACUBIC).LE.EPSMCH) THEN
                       LAMTMP=-DELFTS/(TWO*BCUBIC)
                    ELSE
                       IF(DISC.LT.ZERO) THEN
                          LAMTMP=SIGMA*LAMBDA
                       ELSE
                          LAMTMP=(-BCUBIC+SQRT(DISC))/(THREE*ACUBIC)
                       END IF
                    END IF
                    IF(OUTPUT.GT.4) THEN
                       WRITE(ICOUT,1)
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,36) LAMTMP
36                     FORMAT(T3,'*',13X,'TEMPORARY LAMBDA FROM',
     $                 ' CUBIC MODEL: .....',1PD11.3,T74,'*')
                       CALL DPWRST('XXX','BUG')
                    END IF
                 END IF
                 IF(LAMTMP.GT.SIGMA*LAMBDA) THEN
                    LAMTMP=SIGMA*LAMBDA
                    IF(OUTPUT.GT.4) THEN
                       WRITE(ICOUT,1)
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,37)
37                     FORMAT(T3,'*',13X,'LAMTMP TOO LARGE - REDUCED',
     $                 ' TO SIGMA*LAMBDA',T74,'*')
                       CALL DPWRST('XXX','BUG')
                    END IF
                 END IF
              END IF
              LAMPRE=LAMBDA
              FPLPRE=FCNNEW
              IF(LAMTMP.LT.POINT1*LAMBDA) THEN
                 IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,38)
38                  FORMAT(T3,'*',13X,'LAMTMP TOO SMALL - INCREASED',
     $               ' TO 0.1*LAMBDA',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 LAMBDA=POINT1*LAMBDA
              ELSE
                 IF(OUTPUT.GT.4.AND.LAMTMP.NE.SIGMA*LAMBDA) THEN
                    WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,39)
39                  FORMAT(T3,'*',13X,'LAMTMP WITHIN LIMITS - ',
     $              'LAMBDA SET TO LAMTMP',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 LAMBDA=LAMTMP
              END IF
           END IF
101        CONTINUE
100     CONTINUE
C
C       FAILURE IN LINE SEARCH
C
        ACPCOD=0
C
C       IF A QUASI-NEWTON STEP HAS FAILED IN THE LINE SEARCH THEN
C       SET QNFAIL TO TRUE ANS RETURN TO SUBROUTINE NNES.  THIS WILL
C       CAUSE THE JACOBIAN TO BE RE-EVALUATED EXPLICITLY AND A LINE
C       SEARCH IN THE NEW DIRECTION CONDUCTED.
C
        IF(.NOT.RESTRT) THEN
           QNFAIL=.TRUE.
           RETURN
        END IF
C
C       FALL THROUGH MAIN LOOP - WARNING GIVEN.
C
        IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,40) MAXLIN
40         FORMAT(T3,'*',7X,'WARNING: ',I3,' CYCLES COMPLETED',
     $            ' IN LINE SEARCH WITHOUT SUCCESS',T74,'*')
           CALL DPWRST('XXX','BUG')
        END IF
        IF(STOPCR.EQ.2.OR.STOPCR.EQ.3) THEN
           STOPCR=12
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,41)
41         FORMAT(T3,'*',7X,'STOPPING CRITERION RESET FROM ',
     $           '2 TO 12 TO AVOID HANGING',T74,'*')
           CALL DPWRST('XXX','BUG')
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE DEUFLS.
C
        END
C
CCCCC   RENAME THIS ROUTINE TO AVOID NAME CONFLICTS.
C
CCCCC   SUBROUTINE DOGLEG(FRSTDG,NEWTKN,OVERCH,OVERFL,MAXEXP,
        SUBROUTINE DOGLEX(FRSTDG,NEWTKN,OVERCH,OVERFL,MAXEXP,
     $                    N     ,NOTRST,NUNIT ,OUTPUT,BETA  ,
     $                    CAULEN,DELTA ,ETAFAC,NEWLEN,SQRTZ ,
     $                    DELF  ,S     ,SCALEX,SN    ,SSDHAT,
     $                    VHAT  )
C
C       FEB. 23, 1992
C
C       THIS SUBROUTINE FINDS A TRUST REGION STEP USING THE
C       (DOUBLE) DOGLEG METHOD.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION LAMBDA   ,NEWLEN
        INTEGER          OUTPUT
        DIMENSION        DELF(N)  ,S(N)    ,SCALEX(N),SN(N)  ,
     $                   SSDHAT(N),VHAT(N)
        LOGICAL          FRSTDG   ,NEWTKN  ,OVERCH   ,OVERFL ,
     $                   WRNSUP
      COMMON/NNES_2/WRNSUP
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,ONE /0.0D0,1.0D0/
C
        OVERFL=.FALSE.
C
        IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
1          FORMAT(T3,'*',70X,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2) NOTRST,DELTA
2          FORMAT(T3,'*',4X,'TRUST REGION STEP:',I6,2X,
     $     'TRUST REGION LENGTH, DELTA:',1PD11.3,2X,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,3) NEWLEN
3          FORMAT(T3,'*',7X,'LENGTH OF NEWTON STEP, NEWLEN: ',
     $     1PD11.3,21X,'*')
           CALL DPWRST('XXX','BUG')
        END IF
C
C       CHECK FOR NEWTON STEP WITHIN TRUST REGION - IF SO USE
C       NEWTON STEP.
C
        IF(NEWLEN.LE.DELTA) THEN
           DO 100 I=1,N
              S(I)=SN(I)
100        CONTINUE
           NEWTKN=.TRUE.
           TEMP=DELTA
           DELTA=NEWLEN
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,4)
4             FORMAT(T3,'*',7X,'NEWTON STEP WITHIN ACCEPTABLE RANGE',
     $        ' ( <= THAN DELTA)',11X,'*')
              CALL DPWRST('XXX','BUG')
              IF(TEMP.EQ.DELTA) THEN
                 WRITE(ICOUT,5) DELTA
5                FORMAT(T3,'*',7X,'DELTA STAYS AT LENGTH OF NEWTON',
     $           ' STEP: ',1PD11.3,14X,'*')
                 CALL DPWRST('XXX','BUG')
              ELSE
                 WRITE(ICOUT,6)
6                FORMAT(T3,'*',7X,'DELTA SET TO LENGTH OF NEWTON',
     $           ' STEP',29X,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,7)
7             FORMAT(T3,'*',7X,'FULL NEWTON STEP ATTEMPTED',37X,'*')
              CALL DPWRST('XXX','BUG')
           END IF
           RETURN
        ELSE
C
C          NEWTON STEP NOT WITHIN TRUST REGION - APPLY (DOUBLE)
C          DOGLEG PROCEDURE.  (IF ETAFAC EQUALS 1.0 THEN THE SINGLE
C          DOGLEG PROCEDURE IS BEING APPLIED).
C
           NEWTKN=.FALSE.
           IF(FRSTDG) THEN
C
C             SPECIAL SECTION FOR FIRST DOGLEG STEP - CALCULATES
C             CAUCHY POINT (MINIMIZER OF MODEL FUNCTION IN STEEPEST
C             DESCENT DIRECTION OF THE OBJECTIVE FUNCTION).
C
              FRSTDG=.FALSE.
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
          IF(ETAFAC.EQ.ONE) THEN
                    WRITE(ICOUT,8)
8                   FORMAT(T3,'*',7X,'FIRST SINGLE DOGLEG STEP',
     $              39X,'*')
                    CALL DPWRST('XXX','BUG')
                 ELSE
                    WRITE(ICOUT,9)
9                   FORMAT(T3,'*',7X,'FIRST DOUBLE DOGLEG STEP',
     $              39X,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,10)
10               FORMAT(T3,'*',10X,'SCALED CAUCHY STEP',42X,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
              END IF
C
C             NOTE: BETA AND SQRTZ WERE CALCULATED IN SUBROUTINE DELCAU.
C
              ZETA=SQRTZ*SQRTZ
C
C             FIND STEP TO CAUCHY POINT.
C
              FACTOR=-(ZETA/BETA)
              DO 200 I=1,N
                 SSDHAT(I)=FACTOR*(DELF(I)/SCALEX(I))
                 IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,11) I,SSDHAT(I)
11                  FORMAT(T3,'*',13X,'SSDHAT(',I3,') = ',1PD12.3,
     $              31X,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
200           CONTINUE
              CALL INNERP(OVERCH,OVERFL,MAXEXP,N,N,N     ,NUNIT ,
     $                    OUTPUT,DELFTS,DELF  ,SN    )
              OVERFL=.FALSE.
C
C             PROTECT AGAINST (RARE) CASE WHEN CALCULATED DIRECTIONAL
C             DERIVATIVE EQUALS ZERO.
C
            IF(DELFTS.NE.ZERO) THEN
C
C                STANDARD EXECUTION.
C
                 GAMMA=(ZETA/ABS(DELFTS))*(ZETA/BETA)
          ETA=ETAFAC+(ONE-ETAFAC)*GAMMA
              ELSE
                 IF(OUTPUT.GT.1.AND.(.NOT.WRNSUP)) THEN
                    WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,12)
12                  FORMAT(T3,'*',4X,'WARNING: DELFTS=0; ETA SET',
     $              ' TO 1.0 TO AVOID DIVISION BY ZERO',8X,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
          ETA=ONE
              END IF
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,13) ETA
13               FORMAT(T3,'*',10X,'ETA = ',1PD11.3,43X,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,14)
14               FORMAT(T3,'*',10X,'VHAT VECTOR     VHAT(I)=ETA*',
     $           'SN(I)*SCALEX(I)-SSDHAT(I)',7X,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
              END IF
              DO 300 I=1,N
                 VHAT(I)=ETA*SCALEX(I)*SN(I)-SSDHAT(I)
                 IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,15) I,VHAT(I)
15                  FORMAT(T3,'*',13X,'VHAT(',I3,') = ',1PD12.3,
     $              33X,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
300           CONTINUE
           END IF
C
C          ETA*NEWLEN <= DELTA MEANS TAKE STEP IN NEWTON DIRECTION
C          TO TRUST REGION BOUNDARY.
C
           IF(ETA*NEWLEN.LE.DELTA) THEN
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,16)
16               FORMAT(T3,'*',10X,'ETA*NEWLEN <= DELTA     S(I)',
     $           '= (DELTA/NEWLEN)*SN(I)',10X,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
              DO 400 I=1,N
                 S(I)=(DELTA/NEWLEN)*SN(I)
400           CONTINUE
           ELSE
C
C             DISTANCE TO CAUCHY POINT >= DELTA MEANS TAKE STEP IN
C             STEEPEST DESCENT DIRECTION TO TRUST REGION BOUNDARY.
C
              IF(CAULEN.GE.DELTA) THEN
                 IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,17)
17                  FORMAT(T3,'*',10X,'CAULEN >= DELTA   S(I)',
     $              '=(DELTA/CAULEN)*(SSDHAT(I)/SCALEX(I))',1X,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 DO 500 I=1,N
                    S(I)=(DELTA/CAULEN)*(SSDHAT(I)/SCALEX(I))
500              CONTINUE
              ELSE
C
C                TAKE (DOUBLE) DOGLEG STEP.
C
                 CALL INNERP(OVERCH,OVERFL,MAXEXP,N,N,N     ,NUNIT ,
     $                       OUTPUT,TEMP  ,SSDHAT,VHAT  )
                 CALL INNERP(OVERCH,OVERFL,MAXEXP,N,N,N     ,NUNIT ,
     $                       OUTPUT,TEMPV ,VHAT  ,VHAT  )
                 OVERFL=.FALSE.
                 LAMBDA=(-TEMP+SQRT(TEMP*TEMP-TEMPV*(CAULEN*CAULEN
     $                   -DELTA*DELTA)))/TEMPV
                 IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,18)
18                  FORMAT(T3,'*',10X,'S(I)=(SSDHAT(I)+LAMBDA*VHAT(I))',
     $              '/SCALEX(I)',19X,'*')
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,19) LAMBDA
19                  FORMAT(T3,'*',10X,'WHERE LAMBDA = ',1PD12.3,33X,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 DO 600 I=1,N
                    S(I)=(SSDHAT(I)+LAMBDA*VHAT(I))/SCALEX(I)
600              CONTINUE
              END IF
           END IF
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,20)
20            FORMAT(T3,'*',10X,'REVISED STEP FROM SUBROUTINE',
     $        ' DOGLEG',25X,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              DO 700 I=1,N
                 WRITE(ICOUT,21) I,S(I)
21               FORMAT(T3,'*',13X,'S(',I3,') = ',1PD12.3,36X,'*')
                 CALL DPWRST('XXX','BUG')
700           CONTINUE
           END IF
        END IF
        RETURN
        END
       SUBROUTINE FCNEVL(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,
     $                   EPSMCH,FCNNEW,FVEC  ,SCALEF,WV1   )
C
C       FEB. 23, 1992
C
C       THE OBJECTIVE FUNCTION, FCNNEW, DEFINED BY:
C
C          FCNNEW:=1/2(SCALEF*FVEC^SCALEF*FVEC)
C
C       IS EVALUATED BY THIS SUBROUTINE.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        INTEGER     OUTPUT
        DIMENSION   FVEC(N) ,SCALEF(N) ,WV1(N)
        LOGICAL     OVERFL  ,WRNSUP
        COMMON/NNES_2/WRNSUP
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA TWO,TEN /2.0D0,10.0D0/
C
        OVERFL=.FALSE.
        EPS=TEN**(-MAXEXP)
C
        DO 100 I=1,N
           WV1(I)=FVEC(I)*SCALEF(I)
100     CONTINUE
        CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,FCNNEW,WV1)
C
C       IF AN OVERFLOW WOULD OCCUR SUBSTITUTE A LARGE VALUE
C       FOR FCNNEW.
C
      IF(OVERFL.OR.TWO*LOG10(FCNNEW+EPS).GT.MAXEXP)  THEN
           OVERFL=.TRUE.
           FCNNEW=TEN**MAXEXP
           IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
              WRITE(ICOUT,1)
1             FORMAT(T3,'*',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,2) FCNNEW
2             FORMAT(T3,'*',4X,'WARNING: TO AVOID OVERFLOW',
     $        ' OBJECTIVE FUNCTION SET TO: ',1PD11.3,T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
           RETURN
        END IF
        FCNNEW=FCNNEW*FCNNEW/TWO
C
        RETURN
        END
      SUBROUTINE FORDIF(OVERFL,J     ,N     ,DELTAJ,FVEC  ,
     $                  FVECJ1,JACFDM,XC    ,FVECEV)
C
C       FEB. 6, 1991
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION JACFDM(N,N)
      DIMENSION FVEC(N),FVECJ1(N),XC(N)
      LOGICAL   OVERFL
      EXTERNAL  FVECEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      CALL FVECEV(OVERFL,N,FVECJ1,XC)
      IF(.NOT.OVERFL) THEN
         DO 100 I=1,N
            JACFDM(I,J)=(FVECJ1(I)-FVEC(I))/DELTAJ
100      CONTINUE
      END IF

      RETURN
C
C       LAST CARD OF SUBROUTINE FORDIF.
C
      END
      SUBROUTINE GRADF(OVERCH,OVERFL,RESTRT,SCLFCH,SCLXCH,JUPDM ,
     $                   MAXEXP,N     ,NUNIT ,OUTPUT,QNUPDM,DELF  ,
     $                   FVECC ,JAC   ,SCALEF,SCALEX,WV1   )
C
C       FEB. 23, 1992
C
C       THIS SUBROUTINE COMPUTES THE GRADIENT OF THE FUNCTION
C
C              F=1/2{SCALEF*FVECC)^(SCALEF*FVECC}
C
C       WHICH IS USED AS THE OBJECTIVE FUNCTION FOR MINIMIZATION.
C
C       NOTE: WHEN THE FACTORED FORM OF THE JACOBIAN IS UPDATED IN
C             QUASI-NEWTON METHODS THE GRADIENT IS UPDATED AS WELL
C             IN THE SAME SUBROUTINE - IT IS PRINTED HERE THOUGH.
C             IN THESE CASES QNUPDM > 0.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION  JAC(N,N)
        INTEGER           OUTPUT  ,QNUPDM
      DIMENSION         DELF(N) ,FVECC(N),SCALEF(N),SCALEX(N),WV1(N)
      LOGICAL           OVERCH  ,OVERFL  ,RESTRT   ,SCLFCH   ,SCLXCH
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        IF(RESTRT.OR.JUPDM.EQ.0.OR.QNUPDM.EQ.0) THEN
C
C          GRADIENT NOT ALREADY UPDATED:    FIND DELF = J^F.
C
         DO 100 I=1,N
            IF(SCLFCH) THEN
               WV1(I)=FVECC(I)*SCALEF(I)*SCALEF(I)
            ELSE
               WV1(I)=FVECC(I)
            END IF
100      CONTINUE
C        CHECK EACH ENTRY INDIVIDUALLY
         IF(OVERCH) THEN
            CALL ATVOV(OVERFL,MAXEXP,N     ,NUNIT ,
     $                   OUTPUT,JAC   ,WV1   ,DELF  )
         ELSE
            CALL ATBMUL(N,N,1,1,N,N,JAC,WV1,DELF)
         END IF
      END IF
C
C       PRINT GRADIENT VECTOR, DELF.
C
      IF(OUTPUT.GT.3) THEN
         IF(.NOT.SCLXCH) THEN
            WRITE(ICOUT,1)
1           FORMAT(T3,'*',T74,'*')
            CALL DPWRST('XXX','BUG')
            IF(.NOT.SCLFCH) THEN
               WRITE(ICOUT,2)
2              FORMAT(T3,'*',4X,'GRADIENT OF OBJECTIVE FUNCTION',
     $                T74,'*')
               CALL DPWRST('XXX','BUG')
            ELSE
               WRITE(ICOUT,3)
3              FORMAT(T3,'*',4X,'GRADIENT OF SCALED OBJECTIVE',
     $                ' FUNCTION',T74,'*')
               CALL DPWRST('XXX','BUG')
            END IF
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            DO 300 I=1,N
               WRITE(ICOUT,4) I,DELF(I)
4              FORMAT(T3,'*',6X,'DELF(',I3,') = ',1PD12.3,T74,'*')
               CALL DPWRST('XXX','BUG')
300         CONTINUE
         ELSE
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,5)
5           FORMAT(T3,'*',4X,'GRADIENT OF OBJECTIVE FUNCTION',9X,
     $             'IN SCALED X UNITS',T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            DO 400 I=1,N
               WRITE(ICOUT,6) I,DELF(I),I,SCALEF(I)*SCALEF(I)*
     $                        DELF(I)/SCALEX(I)
6              FORMAT(T3,'*',6X,'DELF(',I3,') = ',1PD12.3,
     $                9X,'DELF(',I3,') = ',1PD12.3,T74,'*')
               CALL DPWRST('XXX','BUG')
400         CONTINUE
         END IF
      END IF
C
      RETURN
C
C       LAST CARD OF SUBROUTINE GRADF.
C
      END
      SUBROUTINE INITCH(INSTOP,LINESR,NEWTON,OVERFL,SCLFCH,
     $                    SCLXCH,ACPTCR,CONTYP,JACTYP,JUPDM ,
     $                    MAXEXP,N     ,NUNIT ,OUTPUT,QNUPDM,
     $                    STOPCR,TRUPDM,EPSMCH,FCNOLD,FTOL  ,
     $                    BOUNDL,BOUNDU,FVECC ,SCALEF,SCALEX,
     $                    WV1   ,XC    ,FVECEV)
C
C       AUG. 27, 1991
C
C       THIS SUBROUTINE FIRST CHECKS TO SEE IF N IS WITHIN THE
C       ACCEPTABLE RANGE.
C
C       THE SECOND CHECK IS TO SEE IF THE INITIAL ESTIMATE IS
C       ALREADY A SOLUTION BY THE FUNCTION VALUE CRITERION, FTOL.
C
C       THE THIRD CHECK IS MADE TO SEE IF THE NEWTON OPTION IS BEING
C       USED WITH THE LINE SEARCH.  IF NOT A WARNING IS GIVEN AND
C       THE LINE SEARCH OPTION IS INVOKED.
C
C       THE FOURTH CHECK IS TO ENSURE APPLICABILITY OF SELECTED
C       VALUES FOR INTEGER CONSTANTS.
C
C       THE FIFTH CHECK IS TO WARN THE USER IF INITIAL ESTIMATES
C       ARE NOT WITHIN THE RANGES SET BY THE BOUNDL AND BOUNDU
C       VECTORS. CONTYP IS CHANGED FROM 0 TO 1 IF ANY BOUND HAS
C       BEEN SET BY THE USER
C
C       THE SIXTH CHECK ENSURES BOUNDL(I) < BOUNDU(I) FOR ALL I.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        INTEGER           ACPTCR   ,CONTYP   ,OUTPUT   ,QNUPDM   ,
     $                    STOPCR   ,TRUPDM
        DIMENSION         BOUNDL(N),BOUNDU(N),FVECC(N) ,SCALEF(N),
     $                    SCALEX(N),WV1(N)   ,XC(N)
        LOGICAL           FRSTER   ,INSTOP   ,LINESR   ,NEWTON   ,
     $                    OVERFL   ,SCLFCH   ,SCLXCH
        EXTERNAL FVECEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,ONE,TEN /0.0D0,1.0D0,10.0D0/
C
        INSTOP=.FALSE.
        TEMP1=-TEN**MAXEXP
        TEMP2=TEN**MAXEXP
C
C       CHECK FOR N IN RANGE.
C
        IF(N.LE.0) THEN
           INSTOP=.TRUE.
           WRITE(ICOUT,1)
1          FORMAT(T3,72('*'))
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
2          FORMAT(T3,'*',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,3)
3          FORMAT(T3,'*',2X,'N IS OUT OF RANGE - RESET TO POSITIVE',
     $     ' INTEGER',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
C
C       CHECK FOR SCALING FACTORS POSITIVE.
C
      FRSTER=.TRUE.
      SCLFCH=.FALSE.
      SCLXCH=.FALSE.
        DO 100 I=1,N
           IF(SCALEF(I).LE.ZERO) THEN
              IF(FRSTER) THEN
          INSTOP=.TRUE.
                 FRSTER=.FALSE.
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
              END IF
              WRITE(ICOUT,2)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,4) I,SCALEF(I)
4             FORMAT(T3,'*',7X,'SCALEF(',I3,') = ',1PD12.3,
     $               4X,'SHOULD BE POSITIVE',T74,'*')
              CALL DPWRST('XXX','BUG')
         END IF
         IF(SCALEF(I).NE.ONE) SCLFCH=.TRUE.
           IF(SCALEX(I).LE.ZERO) THEN
              IF(FRSTER) THEN
          INSTOP=.TRUE.
                 FRSTER=.FALSE.
                 WRITE(ICOUT,1)
              END IF
              WRITE(ICOUT,2)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,5) I,SCALEX(I)
5             FORMAT(T3,'*',7X,'SCALEX(',I3,') = ',1PD12.3,
     $               4X,'SHOULD BE POSITIVE',T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
         IF(SCALEX(I).NE.ONE) SCLXCH=.TRUE.
100     CONTINUE
      IF(.NOT.FRSTER) THEN
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
C
C       EVALUATE INITIAL RESIDUAL VECTOR AND OBJECTIVE FUNCTION AND
C       CHECK TO SEE IF THE INITIAL GUESS IS ALREADY A SOLUTION.
C
        CALL FVECEV(OVERFL,N,FVECC,XC)
C
C       NOTE: NUMBER OF LINE SEARCH FUNCTION EVALUATIONS, NFUNC,
C             INITIALIZED AT 1 WHICH REPRESENTS THIS EVALUATION.
C
        IF(OVERFL) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,6)
6          FORMAT(T3,'*',7X,'OVERFLOW IN INITIAL FUNCTION',
     $     ' VECTOR EVALUATION',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           INSTOP=.TRUE.
           RETURN
        END IF
        CALL FCNEVL(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,EPSMCH,
     $              FCNOLD,FVECC ,SCALEF,WV1   )
        IF(OVERFL) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,7)
7          FORMAT(T3,'*',7X,'OVERFLOW IN INITIAL OBJECTIVE ',
     $     ' FUNCTION EVALUATION',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           INSTOP=.TRUE.
           RETURN
        END IF
C
C       CHECK FOR SOLUTION USING SECOND STOPPING CRITERION.
C
      DO 200 I=1,N
         IF(ABS(FVECC(I)).GT.FTOL) GO TO 201
200     CONTINUE
      INSTOP=.TRUE.
      WRITE(ICOUT,1)
        CALL DPWRST('XXX','BUG')
      WRITE(ICOUT,2)
        CALL DPWRST('XXX','BUG')
      WRITE(ICOUT,8)
8       FORMAT(T3,'*',2X,'WARNING: THIS IS ALREADY A SOLUTION',
     $         ' BY THE CRITERIA OF THE SOLVER',T74,'*')
        CALL DPWRST('XXX','BUG')
      WRITE(ICOUT,2)
        CALL DPWRST('XXX','BUG')
C
C       IF THE PROBLEM IS BADLY SCALED THE OBJECTIVE FUNCTION
C       MAY MEET THE TOLERANCE ALTHOUGH THE INITIAL ESTIMATE
C       IS NOT THE SOLUTION.
C
      WRITE(ICOUT,9)
9       FORMAT(T3,'*',2X,'THIS MAY POSSIBLY BE ALLEVIATED BY ',
     $  'RESCALING THE PROBLEM IF THE',T74,'*')
        CALL DPWRST('XXX','BUG')
      WRITE(ICOUT,10)
10      FORMAT(T3,'*',2X,'INITIAL ESTIMATE IS KNOWN NOT TO BE',
     $  ' A SOLUTION',T74,'*')
        CALL DPWRST('XXX','BUG')
      WRITE(ICOUT,2)
        CALL DPWRST('XXX','BUG')
      WRITE(ICOUT,1)
201     CONTINUE ! FVEC(I) > FTOL FOR SOME I FROM 200 LOOP
        CALL DPWRST('XXX','BUG')
C
C       CHECK FOR NEWTON'S METHOD REQUESTED BUT LINE SEARCH NOT
C       BEING USED.
C
        IF(NEWTON.AND.(.NOT.LINESR)) THEN
           LINESR=.TRUE.
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,11)
11         FORMAT(T3,'*',2X,'WARNING: INCOMPATIBLE OPTIONS',
     $     ': NEWTON=.TRUE. AND LINESR=.FALSE.',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,12)
12         FORMAT(T3,'*',2X,'LINESR SET TO .TRUE.; EXECUTION'
     $     ' OF NEWTON METHOD CONTINUING',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
C
C       CHECK INTEGER CONSTANTS.
C
        IF(ACPTCR.NE.1.AND.ACPTCR.NE.12) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,13) ACPTCR
13         FORMAT(T3,'*',2X,'ACPTCR NOT AN ACCEPTABLE VALUE: ',
     $     I5,T74,'*')
           CALL DPWRST('XXX','BUG')
           INSTOP=.TRUE.
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
        IF(JACTYP.LT.0.OR.JACTYP.GT.3) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,14) JACTYP
14         FORMAT(T3,'*',2X,'JACTYP:',I5,' - NOT IN PROPER RANGE',
     $     T74,'*')
           CALL DPWRST('XXX','BUG')
           INSTOP=.TRUE.
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
        IF(STOPCR.NE.1.AND.STOPCR.NE.12.AND.STOPCR.NE.2.
     $     AND.STOPCR.NE.3) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,15) STOPCR
15         FORMAT(T3,'*',2X,'STOPCR NOT AN ACCEPTABLE VALUE: ',
     $     I5,T74,'*')
           CALL DPWRST('XXX','BUG')
           INSTOP=.TRUE.
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
        IF(QNUPDM.LT.0.OR.QNUPDM.GT.1) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,16) QNUPDM
16         FORMAT(T3,'*',2X,'QNUPDM:',I5,' - NOT IN PROPER RANGE',
     $     T74,'*')
           CALL DPWRST('XXX','BUG')
           INSTOP=.TRUE.
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
        IF(TRUPDM.LT.0.OR.TRUPDM.GT.1) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,17) TRUPDM
17         FORMAT(T3,'*',2X,'TRUPDM:',I5,' - NOT IN PROPER RANGE',
     $     T74,'*')
           CALL DPWRST('XXX','BUG')
           INSTOP=.TRUE.
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
        IF(JUPDM.LT.0.OR.JUPDM.GT.2) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,18) JUPDM
18         FORMAT(T3,'*',2X,'JUPDM:',I5,' - NOT IN PROPER RANGE',
     $     T74,'*')
           CALL DPWRST('XXX','BUG')
           INSTOP=.TRUE.
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
C
C       CHECK FOR INITIAL ESTIMATES NOT WITHIN SPECIFIED BOUNDS AND
C       SET CONTYP TO 1 => AT LEAST ONE BOUND IS IN EFFECT.
C
        CONTYP=0
      DO 300 I=1,N
           IF((BOUNDL(I).NE.TEMP1.OR.BOUNDU(I).NE.TEMP2)) THEN
              CONTYP=1
            GO TO 301
           END IF
300     CONTINUE
301     CONTINUE
        FRSTER=.TRUE.
        IF(CONTYP.NE.0) THEN
         DO 400 I=1,N
C
C             CHECK FOR INITIAL ESTIMATES OUT OF RANGE AND LOWER
C             BOUND GREATER THAN OR EQUAL TO THE UPPER BOUND.
C
              IF(XC(I).LT.BOUNDL(I).OR.XC(I).GT.BOUNDU(I)) THEN
          IF(FRSTER) THEN
             INSTOP=.TRUE.
             FRSTER=.FALSE.
             WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
             WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
             WRITE(ICOUT,19)
19                  FORMAT(T3,'*',7X,'COMPONENTS MUST BE WITHIN',
     $              ' BOUNDS',T74,'*')
                    CALL DPWRST('XXX','BUG')
             WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
             WRITE(ICOUT,20)
20                  FORMAT(T3,'*',8X,'NO.',9X,'XC',16X,'BOUNDL',10X,
     $              'BOUNDU',T74,'*')
                    CALL DPWRST('XXX','BUG')
             WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                 END IF
          WRITE(ICOUT,21) I,XC(I),BOUNDL(I),BOUNDU(I)
21               FORMAT(T3,'*',7X,I3,3X,1PD12.3,9X,1PD12.3,4X,
     $           1PD12.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
400        CONTINUE
         IF(.NOT.FRSTER) THEN
              WRITE(ICOUT,2)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
           END IF
C
           FRSTER=.TRUE.
         DO 500 I=1,N
              IF(BOUNDL(I).GE.BOUNDU(I)) THEN
                 IF(FRSTER) THEN
                    FRSTER=.FALSE.
             WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
             WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
             WRITE(ICOUT,22)
22                  FORMAT(T3,'*',7X,'LOWER BOUND MUST BE LESS THAN',
     $              ' UPPER BOUND - VIOLATIONS LISTED',T74,'*')
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                 END IF
          WRITE(ICOUT,23) I,BOUNDL(I),I,BOUNDU(I)
23               FORMAT(T3,'*',7X,'BOUNDL(',I3,') = ',1PD12.3,
     $           4X,'BOUNDU(',I3,') = ',1PD12.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
500        CONTINUE
           IF(.NOT.FRSTER) THEN
              WRITE(ICOUT,2)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
           END IF
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE INITCH.
C
        END
        SUBROUTINE INNERP(OVERCH,OVERFL,MAXEXP,LDIMA ,LDIMB ,N     ,
     $                    NUNIT ,OUTPUT,DTPRO ,A     ,B     )
C
C       FEB. 14, 1991
C
C       THIS SUBROUTINE FINDS THE INNER PRODUCT OF TWO VECTORS,
C       A AND B.  IF OVERCH IS FALSE, UNROLLED LOOPS ARE USED.
C
C       LDIMA IS THE DIMENSION OF A
C       LDIMB IS THE DIMENSION OF B
C       N IS THE DEPTH INTO A AND B THE INNER PRODUCT IS DESIRED.
C       (USUALLY LDIMA=LDIMB=N)
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        INTEGER    OUTPUT
        DIMENSION  A(LDIMA) ,B(LDIMB)
        LOGICAL    OVERCH   ,OVERFL   ,WRNSUP
        COMMON/NNES_2/WRNSUP
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,ONE,TEN /0.0D0,1.0D0,10.0D0/
C
        EPS=TEN**(-MAXEXP)
        OVERFL=.FALSE.
C
        DTPRO=ZERO
        IF(OVERCH) THEN
           DO 100 I=1,N
              IF(LOG10(ABS(A(I))+EPS)+LOG10(ABS(B(I))+EPS)
     $           .GT.MAXEXP) THEN
                 OVERFL=.TRUE.
                 DTPRO=SIGN(TEN**MAXEXP,A(I))*SIGN(ONE,B(I))
                 IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
                    WRITE(ICOUT,1)
1                   FORMAT(T3,'*',T74,'*')
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,2) DTPRO
2                   FORMAT(T3,'*',4X,'WARNING: TO AVOID OVERFLOW,',
     $              ' INNER PRODUCT SET TO ',1PD12.3,T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 RETURN
              END IF
              DTPRO=DTPRO+A(I)*B(I)
100        CONTINUE
        ELSE
C
C          SET NUMBER OF GROUPS OF EACH SIZE.
C
           NG32=N/32
           NG32R=N-32*NG32
           NG16=NG32R/16
           NG16R=NG32R-16*NG16
           NG8=NG16R/8
           NG8R=NG16R-8*NG8
           NG4=NG8R/4
           NG4R=NG8R-4*NG4
C
C          FIND INNER PRODUCT.
C
           K=0
           IF(NG32.GT.0) THEN
              DO 200 KK=1,NG32
                 K=K+32
                 DTPRO=DTPRO
     $           +A(K-31)*B(K-31)+A(K-30)*B(K-30)
     $           +A(K-29)*B(K-29)+A(K-28)*B(K-28)
     $           +A(K-27)*B(K-27)+A(K-26)*B(K-26)
     $           +A(K-25)*B(K-25)+A(K-24)*B(K-24)
                 DTPRO=DTPRO
     $           +A(K-23)*B(K-23)+A(K-22)*B(K-22)
     $           +A(K-21)*B(K-21)+A(K-20)*B(K-20)
     $           +A(K-19)*B(K-19)+A(K-18)*B(K-18)
     $           +A(K-17)*B(K-17)+A(K-16)*B(K-16)
                 DTPRO=DTPRO
     $           +A(K-15)*B(K-15)+A(K-14)*B(K-14)
     $           +A(K-13)*B(K-13)+A(K-12)*B(K-12)
     $           +A(K-11)*B(K-11)+A(K-10)*B(K-10)
     $           +A(K-9) *B(K-9)  +A(K-8) *B(K-8)
                 DTPRO=DTPRO
     $           +A(K-7)*B(K-7)+A(K-6)*B(K-6)
     $           +A(K-5)*B(K-5)+A(K-4)*B(K-4)
     $           +A(K-3)*B(K-3)+A(K-2)*B(K-2)
     $           +A(K-1)*B(K-1)+A(K)  *B(K)
200           CONTINUE
           END IF
           IF(NG16.GT.0) THEN
              DO 300 KK=1,NG16
                 K=K+16
                 DTPRO=DTPRO
     $           +A(K-15)*B(K-15)+A(K-14)*B(K-14)
     $           +A(K-13)*B(K-13)+A(K-12)*B(K-12)
     $           +A(K-11)*B(K-11)+A(K-10)*B(K-10)
     $           +A(K-9) *B(K-9)  +A(K-8)*B(K-8)
                 DTPRO=DTPRO
     $           +A(K-7)*B(K-7)+A(K-6)*B(K-6)
     $           +A(K-5)*B(K-5)+A(K-4)*B(K-4)
     $           +A(K-3)*B(K-3)+A(K-2)*B(K-2)
     $           +A(K-1)*B(K-1)+A(K)  *B(K)
300           CONTINUE
           END IF
           IF(NG8.GT.0) THEN
              DO 400 KK=1,NG8
                 K=K+8
                 DTPRO=DTPRO
     $           +A(K-7)*B(K-7)+A(K-6)*B(K-6)
     $           +A(K-5)*B(K-5)+A(K-4)*B(K-4)
     $           +A(K-3)*B(K-3)+A(K-2)*B(K-2)
     $           +A(K-1)*B(K-1)+A(K)  *B(K)
400           CONTINUE
           END IF
           IF(NG4.GT.0) THEN
              DO 500 KK=1,NG4
                 K=K+4
                 DTPRO=DTPRO
     $           +A(K-3)*B(K-3)+A(K-2)*B(K-2)
     $           +A(K-1)*B(K-1)+A(K)  *B(K)
500           CONTINUE
           END IF
           IF(NG4R.GT.0) THEN
              DO 600 KK=1,NG4R
                 K=K+1
                 DTPRO=DTPRO+A(K)*B(K)
600           CONTINUE
           END IF
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE INNERP.
C
        END
        SUBROUTINE JACCD(N     ,NUNIT ,OUTPUT,EPSMCH,FVECJ1,
     $                   FVECJ2,JACFDM,SCALEX,XC    ,FVECEV)
C
C       FEB. 11, 1991
C
C       THIS SUBROUTINE EVALUATES THE JACOBIAN USING CENTRAL
C       DIFFERENCES.
C
C       FVECJ1 AND FVECJ2 ARE TEMPORARY VECTORS TO HOLD THE
C       RESIDUAL VECTORS FOR THE CENTRAL DIFFERENCE CALCULATION.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JACFDM(N,N)
        INTEGER          OUTPUT
        DIMENSION        FVECJ1(N)   ,FVECJ2(N) ,SCALEX(N) ,XC(N)
        LOGICAL          OVERFL      ,WRNSUP
        COMMON/NNES_2/WRNSUP
        EXTERNAL FVECEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ONE,TWO /1.0D0,2.0D0/
C
        OVERFL=.FALSE.
        CURTEP=EPSMCH**0.33
C
        DO 100 J=1,N
           DELTAJ=(CURTEP)*SIGN((MAX(ABS(XC(J)),
     $            ONE/SCALEX(J))),XC(J))
           TEMPJ=XC(J)
           XC(J)=XC(J)+DELTAJ
C
C          NOTE: THIS STEP IS FOR FLOATING POINT ACCURACY ONLY.
C
           DELTAJ=XC(J)-TEMPJ
C
           CALL FVECEV(OVERFL,N,FVECJ1,XC)
           XC(J)=TEMPJ-DELTAJ
           CALL FVECEV(OVERFL,N,FVECJ2,XC)
           IF(OVERFL.AND.OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
              WRITE(ICOUT,1)
1             FORMAT(T3,'*',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,2)
2             FORMAT(T3,'*',4X,'WARNING: OVERFLOW IN FUNCTION',
     $        ' VECTOR IN "JACCD"',T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
           DO 200 I=1,N
              JACFDM(I,J)=(FVECJ1(I)-FVECJ2(I))/(TWO*DELTAJ)
200        CONTINUE
           XC(J)=TEMPJ
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE JACCD.
C
        END
      SUBROUTINE JACFD(JACTYP,N     ,NUNIT ,OUTPUT,EPSMCH,
     $                   BOUNDL,BOUNDU,FVECC ,FVECJ1,JACFDM,
     $                   SCALEX,WV3   ,XC    ,FVECEV)
C
C       FEB. 15, 1991
C
C       THIS SUBROUTINE EVALUATES THE JACOBIAN USING
C       ONE-SIDED FINITE DIFFERENCES.
C
C       JACTYP "1" SIGNIFIES FORWARD DIFFERENCES
C       JACTYP "2" SIGNIFIES BACKWARD DIFFERENCES
C
C       FVECJ1 IS A TEMPORARY VECTOR WHICH STORES THE RESIDUAL
C       VECTOR FOR THE FINITE DIFFERENCE CALCULATION.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION  JACFDM(N,N)
        INTEGER           OUTPUT
      DIMENSION         BOUNDL(N)  ,BOUNDU(N) ,FVECC(N)  ,FVECJ1(N),
     $                    SCALEX(N)  ,WV3(N)    ,XC(N)
        LOGICAL           OVERFL     ,WRNSUP
        COMMON/NNES_2/WRNSUP
        EXTERNAL FVECEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,ONE /0.0D0,1.0D0/
C
        SQRTEP=SQRT(EPSMCH)
C
C       FINITE-DIFFERENCE CALCULATION BY COLUMNS.
C
        DO 100 J=1,N
C
C          DELTAJ IS THE STEP SIZE - IT IS ALWAYS POSITIVE.
C
           DELTAJ=SQRTEP*MAX(ABS(XC(J)),ONE/SCALEX(J))
           TEMPJ=XC(J)  ! TEMPORARY STORAGE OF XC(J)
           IF(JACTYP.EQ.1) THEN
              IF(XC(J)+DELTAJ.LE.BOUNDU(J)) THEN
C
C                STEP WITHIN BOUNDS - COMPLETE FORWARD DIFFERENCE.
C
                 XC(J)=XC(J)+DELTAJ
          DELTAJ=XC(J)-TEMPJ
          CALL FORDIF(OVERFL,J     ,N     ,DELTAJ,FVECC ,
     $                       FVECJ1,JACFDM,XC    ,FVECEV)
              ELSE
C
C                STEP WOULD VIOLATE BOUNDU - TRY BACKWARD DIFFERENCE.
C
                 IF(XC(J)-DELTAJ.GE.BOUNDL(J)) THEN
                    XC(J)=XC(J)-DELTAJ
                    CALL BAKDIF(OVERFL,J     ,N     ,DELTAJ,TEMPJ ,
     $                          FVECC ,FVECJ1,JACFDM,XC    ,FVECEV)
                 ELSE
C
C                   STEP WOULD ALSO VIOLATE BOUNDL - IF THE DIFFERENCE
C                   IN THE BOUNDS, (BOUNDU-BOUNDL), IS GREATER THAN
C                   DELTAJ CALCULATE THE FUNCTION VECTOR AT EACH BOUND
C                   AND USE THIS DIFFERENCE - THIS REQUIRES ONE EXTRA
C                   FUNCTION EVALUATION. THE CURRENT FVECC IS STORED
C                   IN WV3, THEN REPLACED.
C
                    IF(BOUNDU(J)-BOUNDL(J).GE.DELTAJ) THEN
                CALL BNDDIF(OVERFL,J     ,N     ,EPSMCH,BOUNDL,
     $                             BOUNDU,FVECC ,FVECJ1,JACFDM,WV3   ,
     $                             XC    ,FVECEV)
                       IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP).AND.
     $                    (.NOT.OVERFL)) THEN
                          WRITE(ICOUT,1)
1                         FORMAT(T3,'*',T74,'*')
                          CALL DPWRST('XXX','BUG')
                          WRITE(ICOUT,2)
2                         FORMAT(T3,'*',4X,'WARNING: BOUNDS TOO CLOSE',
     $                    ' FOR 1-SIDED FINITE-DIFFERENCES',T74,'*')
                          CALL DPWRST('XXX','BUG')
                          WRITE(ICOUT,3) J
3                         FORMAT(T3,'*',13X,'LOWER AND UPPER BOUNDS',
     $                    ' USED FOR JACOBIAN COLUMN: ',I3,T74,'*')
                          CALL DPWRST('XXX','BUG')
                          WRITE(ICOUT,4)
4                         FORMAT(T3,'*',13X,'THIS REQUIRED ONE EXTRA',
     $                    ' FUNCTION EVALUATION',T74,'*')
                          CALL DPWRST('XXX','BUG')
                       END IF
             ELSE
C
C                      BOUNDS ARE EXTREMELY CLOSE (BUT NOT EQUAL OR
C                      THE PROGRAM WOULD HAVE STOPPED IN INITCH).
C
                       IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP).AND.
     $                    (.NOT.OVERFL)) THEN
                          WRITE(ICOUT,1)
                          CALL DPWRST('XXX','BUG')
                          WRITE(ICOUT,2)
                          CALL DPWRST('XXX','BUG')
                          WRITE(ICOUT,5) J
5                         FORMAT(T3,'*',13X,'BOUNDS ARE EXTREMELY',
     $                    ' CLOSE FOR COMPONENT: ',I3,T74,'*')
                          CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,6)
6                         FORMAT(T3,'*',13X,'FINITE DIFFERENCE',
     $                    ' JACOBIAN IS UNRELIABLE',T74,'*')
                          CALL DPWRST('XXX','BUG')
                       END IF
                CALL BNDDIF(OVERFL,J     ,N     ,EPSMCH,BOUNDL,
     $                             BOUNDU,FVECC ,FVECJ1,JACFDM,WV3   ,
     $                             XC    ,FVECEV)
             END IF
                 END IF
              END IF
           ELSE
              IF(XC(J)-DELTAJ.GE.BOUNDL(J)) THEN
                 XC(J)=XC(J)-DELTAJ
          CALL BAKDIF(OVERFL,J     ,N     ,DELTAJ,TEMPJ ,FVECC ,
     $                       FVECJ1,JACFDM,XC    ,FVECEV)
              ELSE
                 IF(XC(J)+DELTAJ.LE.BOUNDU(J)) THEN
                    XC(J)=XC(J)+DELTAJ
             CALL FORDIF(OVERFL,J     ,N     ,DELTAJ,
     $                          FVECC ,FVECJ1,JACFDM,XC    ,FVECEV)
                 ELSE
                    IF(BOUNDU(J)-BOUNDL(J).GE.DELTAJ) THEN
                CALL BNDDIF(OVERFL,J     ,N     ,EPSMCH,BOUNDL,
     $                             BOUNDU,FVECC ,FVECJ1,JACFDM,WV3   ,
     $                             XC    ,FVECEV)
                       IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP).AND.
     $                    (.NOT.OVERFL)) THEN
                          WRITE(ICOUT,1)
                          CALL DPWRST('XXX','BUG')
                          WRITE(ICOUT,2)
                          CALL DPWRST('XXX','BUG')
                          WRITE(ICOUT,3) J
                          CALL DPWRST('XXX','BUG')
                          WRITE(ICOUT,4)
                          CALL DPWRST('XXX','BUG')
                       END IF
                    ELSE
                CALL BNDDIF(OVERFL,J     ,N     ,EPSMCH,BOUNDL,
     $                             BOUNDU,FVECC ,FVECJ1,JACFDM,WV3   ,
     $                             XC    ,FVECEV)
                DO 300 I=1,N
                          JACFDM(I,J)=ZERO
300                    CONTINUE
                       IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP).AND.
     $                    (.NOT.OVERFL)) THEN
                          WRITE(ICOUT,1)
                          CALL DPWRST('XXX','BUG')
                          WRITE(ICOUT,2)
                          CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,5) J
                          CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,6)
                          CALL DPWRST('XXX','BUG')
                       END IF
                    END IF
                END IF
              END IF
           END IF
           IF(OVERFL.AND.OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,7)
7             FORMAT(T3,'*',4X,'WARNING: OVERFLOW IN FUNCTION',
     $        ' VECTOR IN SUBROUTINE JACFD',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,5) J
              CALL DPWRST('XXX','BUG')
              DO 400 I=1,N
                 JACFDM(I,J)=ZERO
400           CONTINUE
           END IF
           XC(J)=TEMPJ
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE JACFD.
C
        END
        SUBROUTINE JACOBI(CHECKJ,JACERR,OVERFL,JACTYP,N     ,
     $                    NUNIT ,OUTPUT,EPSMCH,FDTOLJ,BOUNDL,
     $                    BOUNDU,FVECC ,FVECJ1,FVECJ2,JAC   ,
     $                    JACFDM,SCALEX,WV3   ,XC    ,FVECEV,
     $                    JACEV )
C
C       APR. 13, 1991
C
C       THIS SUBROUTINE EVALUATES THE JACOBIAN.  IF CHECKJ IS TRUE
C       THEN THE ANALYTICAL JACOBIAN IS CHECKED NUMERICALLY.
C
C       JACEV IS A USER-SUPPLIED ANALYTICAL JACOBIAN USED ONLY IF
C       JACTYP=0.  THE JACOBIAN NAME MAY BE CHANGED BY USING THE
C       EXTERNAL STATEMENT IN THE MAIN DRIVER.
C
C       JACFD ESTIMATES THE JACOBIAN USING FINITE DIFFERENCES:
C       FORWARD IF JACTYP=1 OR BACKWARD IF JACTYP=2.
C
C       JACCD ESTIMATES THE JACOBIAN USING CENTRAL DIFFERENCES.
C
C       IF THE ANALYTICAL JACOBIAN IS CHECKED THE FINITE DIFFERENCE
C       JACOBIAN IS STORED IN "JACFDM" AND THEN COMPARED.
C
C       FRSTER  INDICATES FIRST ERROR - USED ONLY TO SET BORDERS
C               FOR OUTPUT
C       JACERR  FLAG TO INDICATE TO THE CALLING PROGRAM AN ERROR
C               IN THE ANALYTICAL JACOBIAN
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N) ,JACFDM(N,N)
        INTEGER          OUTPUT
      DIMENSION        BOUNDL(N) ,BOUNDU(N) ,FVECC(N) ,FVECJ1(N) ,
     $                   FVECJ2(N) ,SCALEX(N) ,WV3(N)   ,XC(N)
        LOGICAL          CHECKJ    ,FRSTER    ,JACERR   ,OVERFL
        EXTERNAL FVECEV,JACEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ONE /1.0D0/
C
        FRSTER=.TRUE.
        JACERR=.FALSE.
        OVERFL=.FALSE.
C
        IF(JACTYP.EQ.0) THEN
           CALL JACEV(OVERFL,N,JAC,XC)
        ELSEIF(JACTYP.EQ.1.OR.JACTYP.EQ.2) THEN
           CALL JACFD(JACTYP,N     ,NUNIT ,OUTPUT,EPSMCH,BOUNDL,BOUNDU,
     $                FVECC ,FVECJ1,JAC   ,SCALEX,WV3   ,XC    ,FVECEV)
        ELSE
           CALL JACCD(N     ,NUNIT ,OUTPUT,EPSMCH,FVECJ1,
     $                FVECJ2,JAC   ,SCALEX,XC    ,FVECEV)
        END IF
C
        IF(JACTYP.EQ.0.AND.CHECKJ) THEN
C
C          NOTE: JACTYP=0 SENT TO JACFD PRODUCES A FORWARD
C                DIFFERENCE ESTIMATE OF THE JACOBIAN.
C
           CALL JACFD(JACTYP,N     ,NUNIT ,OUTPUT,EPSMCH,BOUNDL,
     $                BOUNDU,FVECC ,FVECJ1,JACFDM,SCALEX,WV3   ,
     $                XC    ,FVECEV)
           DO 100 J=1,N
              DO 200 I=1,N
                 IF(ABS((JACFDM(I,J)-JAC(I,J))/MAX(ABS(JAC(I,J)),
     $              ABS(JACFDM(I,J)),ONE)).GT.FDTOLJ) THEN
                    JACERR=.TRUE.
                    IF(OUTPUT.GE.0) THEN
                       IF(FRSTER) THEN
                          FRSTER=.FALSE.
                          WRITE(ICOUT,1)
1                         FORMAT(T3,72('*'))
                          CALL DPWRST('XXX','BUG')
                       END IF
                       WRITE(ICOUT,2)
2                      FORMAT(T3,'*',T74,'*')
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,3) I,J
3                      FORMAT(T3,'*',4X,'CHECK JACOBIAN TERM (',I3,
     $                 ',',I3,')',T74,'*')
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,2)
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,4) JAC(I,J)
4                      FORMAT(T3,'*',4X,'ANALYTICAL DERIVATIVE IS ',
     $                 1PD12.3,T74,'*')
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,5) JACFDM(I,J)
5                      FORMAT(T3,'*',4X,' NUMERICAL DERIVATIVE IS ',
     $                 1PD12.3,T74,'*')
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,2)
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,1)
                       CALL DPWRST('XXX','BUG')
                    END IF
                 END IF
200           CONTINUE
100        CONTINUE
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE JACOBI.
C
        END
        SUBROUTINE JACROT(OVERFL,I,MAXEXP,N,AROT,BROT,EPSMCH,A,JAC)
C
C       FEB. 11, 1991
C
C       JACOBI (OR GIVENS) ROTATION.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N)
        DIMENSION        A(N,N)  ,HOLD(2)
        LOGICAL          OVERFL
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,ONE /0.0D0,1.0D0/
        IF(AROT.EQ.ZERO) THEN
           C=ZERO
           S=-SIGN(ONE,BROT)
        ELSE
           HOLD(1)=AROT
           HOLD(2)=BROT
           LDHOLD=2
           CALL TWONRM(OVERFL,MAXEXP,LDHOLD,EPSMCH,DENOM,HOLD)
           C=AROT/DENOM
           S=-BROT/DENOM
        END IF
        DO 100 J=I,N
           Y=A(I,J)
           W=A(I+1,J)
           A(I,J)=C*Y-S*W
           A(I+1,J)=S*Y+C*W
100     CONTINUE
        DO 200 J=1,N
           Y=JAC(I,J)
           W=JAC(I+1,J)
           JAC(I,J)=C*Y-S*W
           JAC(I+1,J)=S*Y+C*W
200     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE JACROT.
C
        END
      SUBROUTINE LINE(ABORT ,ABSNEW,DEUFLH,GEOMS ,NEWTON,
     $                  OVERCH,OVERFL,QNFAIL,QRSING,RESTRT,
     $                  SCLFCH,SCLXCH,ACPCOD,ACPTCR,CONTYP,
     $                  ISEJAC,ITNUM ,JUPDM ,MAXEXP,MAXLIN,
     $                  MGLL  ,MNEW  ,N     ,NARMIJ,NFUNC ,
     $                  NUNIT ,OUTPUT,QNUPDM,STOPCR,TRMCOD,
     $                  ALPHA ,CONFAC,EPSMCH,FCNMAX,FCNNEW,
     $                  FCNOLD,LAM0  ,MAXSTP,NEWLEN,SBRNRM,
     $                  SIGMA ,A     ,BOUNDL,BOUNDU,DELF  ,
     $                  FTRACK,FVEC  ,H     ,HHPI  ,JAC   ,
     $                  RDIAG ,RHS   ,S     ,SBAR  ,SCALEF,
     $                  SCALEX,SN    ,STRACK,WV2   ,XC    ,
     $                  XPLUS ,FVECEV)
C
C       SEPT. 9, 1991
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N) ,LAM0     ,LAMBDA    ,MAXSTP   ,
     $                   MU       ,NEWLEN   ,NEWMAX    ,NORM     ,
     $                   NRMPRE
        INTEGER          ACPTCR   ,CONTYP   ,OUTPUT    ,QNUPDM   ,
     $                   STOPCR   ,TRMCOD
        DIMENSION        A(N,N)   ,BOUNDL(N),BOUNDU(N) ,DELF(N)  ,
     $                   FTRACK(0:MGLL-1)   ,FVEC(N)   ,H(N,N)   ,
     $                   HHPI(N)  ,RDIAG(N) ,RHS(N)    ,S(N)     ,
     $                   SBAR(N)  ,SCALEF(N),SCALEX(N) ,SN(N)    ,
     $                   STRACK(0:MGLL-1)   ,WV2(N)    ,XC(N)    ,
     $                   XPLUS(N)
      LOGICAL          ABORT    ,ABSNEW   ,CONVIO    ,DEUFLH   ,
     $                   GEOMS    ,NEWTON   ,OVERCH    ,OVERFL   ,
     $                   QNFAIL   ,QRSING   ,RESTRT    ,SCLFCH   ,
     $                   SCLXCH   ,WRNSUP
        COMMON/NNES_2/WRNSUP
        EXTERNAL FVECEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,TWO,TEN /0.0D0,2.0D0,10.0D0/
        SAVE
C
      CONVIO=.FALSE.
        OVERFL=.FALSE.
      IF(NEWTON.OR.ABSNEW) THEN
         IF(NEWTON) THEN
C
C             FIND NEXT ITERATE FOR PURE NEWTON'S METHOD.
C
            DO 100 I=1,N
               XPLUS(I)=XC(I)+SN(I)
100         CONTINUE
         ELSE
C
C             FIND NEXT ITERATE FOR "ABSOLUTE" NEWTON'S METHOD.
C             IF COMPONENT I WOULD BE OUTSIDE ITS BOUND THEN TAKE
C             ABSOLUTE VALUE OF THE VIOLATION AND GO THIS DISTANCE
C             INTO THE FEASIBLE REGION.  ENSURE THAT THIS REFLECTION
C             OFF ONE BOUND DOES NOT VIOLATE THE OTHER.
C
            DO 200 I=1,N
          WV2(I)=ZERO
          IF(SN(I).GE.ZERO) THEN
             IF(XC(I)+SN(I).GT.BOUNDU(I)) THEN
                CONVIO=.TRUE.
                WV2(I)=TWO
                XPLUS(I)=MAX(TWO*BOUNDU(I)-XC(I)-SN(I),
     $                              BOUNDL(I))
             ELSE
                XPLUS(I)=XC(I)+SN(I)
             END IF
          ELSE
             IF(XC(I)+SN(I).LT.BOUNDL(I)) THEN
                CONVIO=.TRUE.
                WV2(I)=-TWO
                XPLUS(I)=MIN(TWO*BOUNDL(I)-XC(I)-SN(I),
     $                              BOUNDU(I))
             ELSE
                XPLUS(I)=XC(I)+SN(I)
             END IF
          END IF
200           CONTINUE
         END IF
         IF(CONVIO.AND.OUTPUT.GT.4) THEN
            WRITE(ICOUT,1)
1           FORMAT(T3,'*',T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,2)
2           FORMAT(T3,'*',7X,'CONSTRAINT VIOLATORS IN ABSOLUTE',
     $             ' NEWTON''','S METHOD',T74,'*',/T3,'*',T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,3)
3           FORMAT(T3,'*',7X,'COMPONENT',2X,'PROPOSED POINT',2X,
     $             'VIOLATED BOUND',2X,'FEASIBLE VALUE',T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,33)
33          FORMAT(T3,'*',T74,'*')
            CALL DPWRST('XXX','BUG')
            DO 300 I=1,N
               IF(WV2(I).GT.ZERO) THEN
                 WRITE(ICOUT,4) I,XC(I)+SN(I),BOUNDU(I),XPLUS(I)
                 CALL DPWRST('XXX','BUG')
               ELSEIF(WV2(I).LT.ZERO) THEN
                 WRITE(ICOUT,4) I,XC(I)+SN(I),BOUNDL(I),XPLUS(I)
                 CALL DPWRST('XXX','BUG')
               END IF
4              FORMAT(T3,'*',7X,I6,5X,1PD12.3,4X,1PD12.3,4X,1PD12.3,
     $                T74,'*')
300           CONTINUE
         END IF
         CALL FVECEV(OVERFL,N,FVEC,XPLUS)
         NFUNC=NFUNC+1
         IF(OVERFL) THEN
            OVERFL=.FALSE.
            FCNNEW=10.0**MAXEXP
            IF(OUTPUT.GT.2) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,5)
5             FORMAT(T3,'*',7X,'POTENTIAL OVERFLOW DETECTED',
     $               ' IN FUNCTION EVALUATION',T74,'*')
              CALL DPWRST('XXX','BUG')
            END IF
            GO TO 101
         END IF
         CALL FCNEVL(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,
     $               EPSMCH,FCNNEW,FVEC  ,SCALEF,WV2   )
101   CONTINUE
C
C          RETURN FROM PURE NEWTON'S METHOD - OTHERWISE CONDUCT
C          LINE SEARCH.
C
           RETURN
        END IF
        IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,6)
6          FORMAT(T3,'*',4X,'SUMMARY OF LINE SEARCH',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
C
C       SHORTEN NEWTON STEP IF LENGTH IS GREATER THAN MAXSTP.
C
        IF(NEWLEN.GT.MAXSTP) THEN
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,7)
7             FORMAT(T3,'*',7X,'LENGTH OF NEWTON STEP SHORTENED TO',
     $        ' MAXSTP',T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
         DO 400 I=1,N
              SN(I)=SN(I)*MAXSTP/NEWLEN
400        CONTINUE
        END IF
C
C       CHECK DIRECTIONAL DERIVATIVE (MAGNITUDE AND SIGN).
C
      CALL INNERP(OVERCH,OVERFL,MAXEXP,N     ,N     ,N     ,
     $              NUNIT ,OUTPUT,DELFTS,DELF  ,SN    )
        IF(OVERFL) THEN
           OVERFL=.FALSE.
           IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,8) DELFTS
8             FORMAT(T3,'*',4X,'WARNING: DIRECTIONAL DERIVATIVE',
     $        ', DELFTS, SET TO ',1PD12.3,T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
        END IF
C
C       REVERSE SEARCH DIRECTION IF DIRECTIONAL DERIVATIVE IS
C       POSITIVE.
C
        IF(DELFTS.GT.ZERO) THEN
         DO 500 I=1,N
              SN(I)=-SN(I)
500      CONTINUE
         DELFTS=-DELFTS
         IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,9)
9             FORMAT(T3,'*',4X,'WARNING: DIRECTIONAL DERIVATIVE IS'
     $        ' POSITIVE: DIRECTION REVERSED',T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
        END IF
C
C       OUTPUT INFORMATION.
C
        IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,10) DELFTS
           CALL DPWRST('XXX','BUG')
10         FORMAT(T3,'*',7X,'INNER PRODUCT OF DELF AND SN, DELFTS: '
     $     ,'.......',1PD12.3,T74,'*')
           CALL DPWRST('XXX','BUG')
           IF(.NOT.SCLXCH) THEN
              WRITE(ICOUT,11) NEWLEN
11            FORMAT(T3,'*',7X,'LENGTH OF NEWTON STEP, NEWLEN: '
     $        ,'..............',1PD12.3,T74,'*')
              CALL DPWRST('XXX','BUG')
         ELSE
              WRITE(ICOUT,12) NEWLEN
12            FORMAT(T3,'*',7X,'LENGTH OF SCALED NEWTON STEP, NEWLEN: '
     $        ,'.......',1PD12.3,T74,'*')
              CALL DPWRST('XXX','BUG')
         END IF
         WRITE(ICOUT,13) MAXSTP
13       FORMAT(T3,'*',7X,'MAXIMUM STEP LENGTH ALLOWED, MAXSTP: ',
     $          '........',1PD12.3,T74,'*')
         CALL DPWRST('XXX','BUG')
        END IF
C
C       ESTABLISH INITIAL RELAXATION FACTOR.
C
        IF(DEUFLH) THEN
C
C          AT FIRST STEP IN DAMPED NEWTON OR AFTER EXPLICIT
C          JACOBIAN EVALUATION IN QUASI-NEWTON OR IF THE STEP
C          SIZE IS WITHIN STOPPING TOLERANCE BUT STOPCR=3 THE
C          LINE SEARCH IS STARTED AT LAMBDA=1.
C
           IF(ISEJAC.EQ.1.OR.(TRMCOD.EQ.1.AND.STOPCR.EQ.3)) THEN
              LAMBDA=LAM0
           ELSE
              DO 600 I=1,N
                 WV2(I)=(SBAR(I)-SN(I))*SCALEX(I)
600           CONTINUE
              CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,NORM,WV2)
C
C             PREVENT DIVIDE BY ZERO IF NORM IS ZERO (UNDERFLOWS).
C
              IF(NORM.LT.EPSMCH) THEN
C
C                START LINE SEARCH AT LAMBDA=LAM0, USE DUMMY MU.
C
                 MU=TEN
              ELSE
                 MU=NRMPRE*LAMBDA/NORM
              END IF
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,14) MU
14               FORMAT(T3,'*',7X,'DEUFLHARD TEST RATIO, MU: ',
     $           ' ',1PD11.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
C
C             SET INITIAL LAMBDA DEPENDING ON MU. THIS IS A
C             MODIFICATION OF DEUFLHARD'S METHOD WHERE THE CUTOFF
C             VALUE WOULD BE 0.7 FOR LAM0=1.0.
C
              IF(MU.GT.LAM0/TEN) THEN
                 LAMBDA=LAM0
              ELSE
                 LAMBDA=LAM0/TEN
              END IF
           END IF
        ELSE
           LAMBDA=LAM0
        END IF
C
C       STORE LENGTH OF NEWTON STEP.  IF NEWTON STEP LENGTH WAS
C       GREATER THAN MAXSTP IT WAS SHORTENED TO MAXSTP.
C
        NRMPRE=MIN(MAXSTP,NEWLEN)
C
C       ESTABLISH FCNMAX AND NEWMAX FOR NONMONOTONIC LINE SEARCH.
C
        NEWMAX=NEWLEN
        FCNMAX=FCNOLD
        IF(ISEJAC.GT.NARMIJ) THEN
           IF(ISEJAC.LT.NARMIJ+MGLL) THEN
            DO 700 J=1,MNEW
                 FCNMAX=MAX(FCNMAX,FTRACK(J-1))
                 NEWMAX=MAX(NEWMAX,STRACK(J-1))
700           CONTINUE
           ELSE
            DO 800 J=0,MNEW
                 FCNMAX=MAX(FCNMAX,FTRACK(J))
                 NEWMAX=MAX(NEWMAX,STRACK(J))
800           CONTINUE
           END IF
        END IF
C
        IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           IF(.NOT.SCLXCH) THEN
              WRITE(ICOUT,15)
15            FORMAT(T3,'*',7X,'LINE SEARCH',T74,'*')
              CALL DPWRST('XXX','BUG')
           ELSE
              WRITE(ICOUT,16)
16            FORMAT(T3,'*',7X,'LINE SEARCH (X''','S GIVEN IN',
     $        ' UNSCALED UNITS)',T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
        END IF
C
C       CONDUCT LINE SEARCH.
C
      ICPCOD=INT(ACPCOD)
      CALL DEUFLS(ABORT ,DEUFLH,GEOMS ,OVERCH,OVERFL,
     $              QNFAIL,QRSING,RESTRT,SCLFCH,SCLXCH,
CCCCC$              ACPCOD,ACPTCR,CONTYP,ITNUM ,JUPDM ,
     $              ICPCOD,ACPTCR,CONTYP,ITNUM ,JUPDM ,
     $              MAXEXP,MAXLIN,N     ,NFUNC ,NUNIT ,
     $              OUTPUT,QNUPDM,STOPCR,ALPHA ,CONFAC,
     $              DELFTS,EPSMCH,FCNMAX,FCNNEW,FCNOLD,
     $              LAMBDA,NEWMAX,SBRNRM,SIGMA ,A     ,
     $              H     ,BOUNDL,BOUNDU,DELF  ,FVEC  ,
     $              HHPI  ,JAC   ,RDIAG ,RHS   ,S     ,
     $              SBAR  ,SCALEF,SCALEX,SN    ,WV2   ,
     $              XC    ,XPLUS ,FVECEV)
        ACPCOD=REAL(ICPCOD)
        RETURN
C
C       LAST CARD OF SUBROUTINE LINE.
C
      END
      SUBROUTINE LLFA(OVERCH,OVERFL,SCLFCH,SCLXCH,ISEJAC,
     $                  MAXEXP,N     ,NUNIT ,OUTPUT,EPSMCH,
     $                  OMEGA ,A     ,DELF  ,FVEC  ,FVECC ,
     $                  JAC   ,PLEE  ,RDIAG ,S     ,SCALEF,
     $                  SCALEX,T     ,W     ,WV3   ,XC    ,
     $                  XPLUS )
C
C       FEB. 23, 1991
C
C       THE LEE AND LEE QUASI-NEWTON METHOD IS APPLIED TO
C       THE FACTORED FORM OF THE JACOBIAN.
C
C       NOTE: T AND W ARE TEMPORARY WORKING VECTORS ONLY.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N)
        INTEGER          OUTPUT
        DIMENSION        A(N,N)   ,DELF(N) ,FVEC(N)  ,FVECC(N) ,
     $                   PLEE(N,N),RDIAG(N),S(N)     ,SCALEF(N),
     $                   SCALEX(N),T(N)    ,W(N)     ,WV3(N)   ,
     $                   XC(N)    ,XPLUS(N)
      LOGICAL          OVERCH   ,OVERFL  ,SCLFCH   ,SCLXCH   ,
     $                   SKIPUP
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO /0.0D0/
C
      OVERFL=.FALSE.
      SQRTEP=SQRT(EPSMCH)
      SKIPUP=.TRUE.
C
        DO 100 I=1,N
         A(I,I)=RDIAG(I)
           S(I)=XPLUS(I)-XC(I)
100     CONTINUE
C
C       R IS IN THE UPPER TRIANGLE OF A.
C
C       T=RS
C
        CALL UVMUL(N,N,N,N,A,S,T)
C
C       FORM PART OF NUMERATOR AND CHECK TO SEE IF A SIGNIFICANT
C       CHANGE WOULD BE MADE TO THE JACOBIAN.
C
        DO 200 I=1,N
         CALL INNERP(OVERCH,OVERFL,MAXEXP,N,N,N,NUNIT,OUTPUT,SUM,
     $                 JAC(N*(I-1)+1,1),T)
           W(I)=SCALEF(I)*(FVEC(I)-FVECC(I))-SUM
C
C          TEST TO ENSURE VECTOR W IS NONZERO.  IF W(I)=0 FOR
C          ALL I THEN THE UPDATE IS SKIPPED - SKIPUP IS TRUE.
C
           IF(ABS(W(I)).GT.SQRTEP*SCALEF(I)*(ABS(FVEC(I))+
     $     ABS(FVECC(I)))) THEN
            SKIPUP=.FALSE.  ! UPDATE TO BE PERFORMED
           ELSE
            W(I)=ZERO
           END IF
200     CONTINUE
        IF(.NOT.SKIPUP) THEN
C
C          T=Q^W   Q^ IS STORED IN JAC.
C
         CALL AVMUL(N,N,N,N,JAC,W,T)
C
C          FIND DENOMINATOR; FORM W=S^P (P IS SYMMETRIC SO PS IS FOUND).
C
         CALL AVMUL(N,N,N,N,PLEE,S,W)
         IF(SCLXCH) THEN
C
C             SCALE W TO FIND DENOMINATOR.
C
            DO 300 I=1,N
          WV3(I)=W(I)*SCALEX(I)*SCALEX(I)
300           CONTINUE
         ELSE
            CALL MATCOP(N,N,1,1,N,1,W,WV3)
         END IF
         CALL INNERP(OVERCH,OVERFL,MAXEXP,N     ,N     ,N     ,
     $                 NUNIT ,OUTPUT,DENOM ,WV3   ,S     )
C
C          IF OVERFLOW WOULD OCCUR MAKE NO CHANGE TO JACOBIAN.
C
           IF(OVERFL) THEN
              IF(OUTPUT.GT.3) THEN
                 WRITE(ICOUT,1)
1                FORMAT(T3,'*',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,2)
2                FORMAT(T3,'*',4X,'WARNING: JACOBIAN NOT UPDATED',
     $           ' TO AVOID OVERFLOW IN DENOMINATOR OF',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,222)
222              FORMAT(T3,'*',4X,'LEE AND LEE UPDATE',T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
              RETURN
           END IF
C
C          IF DENOM IS ZERO THE SOLVER IS PROBABLY NEAR SOLUTION -
C          AVOID OVERFLOW AND CONTINUE WITH SAME JACOBIAN.
C
         IF(DENOM.EQ.ZERO) RETURN
C
C          THE SCALED VERSION OF S IS TAKEN TO THE UPDATE.
C
           DO 400 I=1,N
              W(I)=W(I)*SCALEX(I)*SCALEX(I)/DENOM
400        CONTINUE
C
C          UPDATE THE QR DECOMPOSITION USING A SERIES OF GIVENS
C          (JACOBI) ROTATIONS.
C
           CALL QRUPDA(OVERFL,MAXEXP,N,EPSMCH,A,JAC,T,W)
C
C          RESET RDIAG AS DIAGONAL OF CURRENT R WHICH IS IN
C          THE UPPER TRIANGLE OF A.
C
           DO 500 I=1,N
              RDIAG(I)=A(I,I)
500        CONTINUE
C
C          UPDATE P MATRIX
C
           DENOM=OMEGA**(ISEJAC+2)+DENOM
           PLEE(1,1)=PLEE(1,1)-WV3(1)*WV3(1)/DENOM
           DO 600 J=2,N
              DO 700 I=1,J-1
                 PLEE(I,J)=PLEE(I,J)-WV3(I)*WV3(J)/DENOM
                 PLEE(J,I)=PLEE(I,J)
700           CONTINUE
              PLEE(J,J)=PLEE(J,J)-WV3(J)*WV3(J)/DENOM
600        CONTINUE
        END IF
C
C       UPDATE THE GRADIENT VECTOR, DELF.
C
C       DELF = (QR)^F = R^Q^F = R^JAC F
C
      IF(SCLFCH) THEN
         DO 800 I=1,N
            W(I)=FVEC(I)*SCALEF(I)
800      CONTINUE
      ELSE
         CALL MATCOP(N,N,1,1,N,1,FVEC,W)
      END IF
      CALL AVMUL(N,N,N,N,JAC,W,T)
      CALL UTBMUL(N,N,1,1,N,N,A,T,DELF)
      RETURN
C
C       LAST CARD OF SUBROUTINE LLFA.
C
      END
        SUBROUTINE LLUN(OVERCH,OVERFL,ISEJAC,MAXEXP,N     ,
     $                  NUNIT ,OUTPUT,EPSMCH,OMEGA ,FVEC  ,
     $                  FVECC ,JAC   ,PLEE  ,S     ,SCALEX,
     $                  WV1   ,XC    ,XPLUS)
C
C       FEB. 13, 1991
C
C       UPDATE THE JACOBIAN USING THE LEE AND LEE METHOD.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N)
        INTEGER          OUTPUT
        DIMENSION        FVEC(N)  ,FVECC(N) ,PLEE(N,N) ,S(N)   ,
     $                   SCALEX(N),WV1(N)   ,XC(N)     ,XPLUS(N)
        LOGICAL          OVERCH   ,OVERFL
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO /0.0D0/
C
        SQRTEP=SQRT(EPSMCH)
C
        DO 100 I=1,N
           S(I)=(XPLUS(I)-XC(I))*SCALEX(I)
100     CONTINUE
        DO 200 I=1,N
           WV1(I)=ZERO
           DO 300 J=1,N
              WV1(I)=WV1(I)+S(J)*PLEE(J,I)
300        CONTINUE
200     CONTINUE
        CALL INNERP(OVERCH,OVERFL,MAXEXP,N     ,N     ,N     ,
     $              NUNIT ,OUTPUT, DENOM,WV1   ,S     )
C
C       IF OVERFLOW WOULD OCCUR MAKE NO CHANGE TO JACOBIAN.
C
        IF(OVERFL) THEN
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
1             FORMAT(T3,'*',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,2)
2             FORMAT(T3,'*',4X,'WARNING: JACOBIAN NOT UPDATED',
     $        ' TO AVOID OVERFLOW IN DENOMINATOR OF',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,3)
3             FORMAT(T3,'*',4X,'LEE AND LEE UPDATE',T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
           RETURN
        END IF
C
C       IF DENOM IS ZERO THE SOLVER MUST BE VERY NEAR SOLUTION -
C       AVOID OVERFLOW AND CONTINUE WITH SAME JACOBIAN.
C
        IF(DENOM.EQ.ZERO) RETURN
        DO 400 I=1,N
           SUM=ZERO
           DO 500 J=1,N
              SUM=SUM+JAC(I,J)*(XPLUS(J)-XC(J))
500        CONTINUE
           TEMPI=FVEC(I)-FVECC(I)-SUM
           IF(ABS(TEMPI).GE.SQRTEP*(ABS(FVEC(I))+ABS(FVECC(I))))
     $        THEN
              TEMPI=TEMPI/DENOM
              DO 600 J=1,N
                 JAC(I,J)=JAC(I,J)+TEMPI*WV1(J)*SCALEX(J)
600           CONTINUE
           END IF
400     CONTINUE
C
C       UPDATE P MATRIX.
C
        DENOM=OMEGA**(ISEJAC+2)+DENOM
        PLEE(1,1)=PLEE(1,1)-WV1(1)*WV1(1)/DENOM
        DO 700 J=2,N
           DO 800 I=1,J-1
              PLEE(I,J)=PLEE(I,J)-WV1(I)*WV1(J)/
     $                  (DENOM*SCALEX(I)*SCALEX(J))
              PLEE(J,I)=PLEE(I,J)
800        CONTINUE
           PLEE(J,J)=PLEE(J,J)-WV1(J)*WV1(J)/DENOM
700     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE LLUN.
C
        END
        SUBROUTINE LSOLV(OVERCH,OVERFL,MAXEXP,N,NUNIT,OUTPUT,L,B,RHS)
C
C       FEB. 14, 1991
C
C       THIS SUBROUTINE SOLVES:
C
C              LB=RHS
C
C              WHERE    L     IS TAKEN FROM THE CHOLESKY DECOMPOSITION
C                       RHS   IS A GIVEN RIGHT HAND SIDE WHICH IS NOT
C                             OVERWRITTEN
C                       B     IS THE SOLUTION VECTOR
C
C       FRSTER IS USED FOR OUTPUT PURPOSES ONLY.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION L(N,N)  ,MAXLOG
        INTEGER          OUTPUT
        DIMENSION        B(N)    ,RHS(N)
        LOGICAL          FRSTER  ,OVERCH  ,OVERFL
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,ONE,TEN /0.0D0,1.0D0,10.0D0/
C
        FRSTER=.TRUE.
        OVERFL=.FALSE.
        EPS=TEN**(-MAXEXP)
C
        IF(OVERCH) THEN
           IF(LOG10(ABS(RHS(1))+EPS)-LOG10(ABS(L(1,1))+EPS)
     $        .GT.MAXEXP) THEN
              OVERFL=.TRUE.
              B(1)=SIGN(TEN**MAXEXP,RHS(1))*
     $                   SIGN(ONE,L(1,1))
              IF(OUTPUT.GT.3) THEN
                 WRITE(ICOUT,1)
1                FORMAT(T3,'*',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,2) 1,B(1)
2                FORMAT(T3,'*',4X,'WARNING: COMPONENT ',I3,
     $           ' SET TO ',1PD11.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
              GO TO 101
           END IF
        END IF
        B(1)=RHS(1)/L(1,1)
101     CONTINUE
        DO 200 I=2,N
           IF(OVERCH) THEN
C
C             CHECK TO FIND IF ANY TERMS IN THE EVALUATION WOULD
C             OVERFLOW.
C
              MAXLOG=LOG10(ABS(RHS(I))+EPS)-LOG10(ABS(L(I,I))+EPS)
              JSTAR=0
              DO 300 J=1,I-1
                 TMPLOG=LOG10(ABS(L(I,J))+EPS)+LOG10(ABS(B(J))+EPS)-
     $                  LOG10(ABS(L(I,I))+EPS)
                 IF(TMPLOG.GT.MAXLOG) THEN
                    JSTAR=J
                    MAXLOG=TMPLOG
                 END IF
300           CONTINUE
C
C             IF AN OVERFLOW WOULD OCCUR ASSIGN A VALUE FOR THE
C             TERM WITH CORRECT SIGN.
C
              IF(MAXLOG.GT.MAXEXP) THEN
                 OVERFL=.TRUE.
                 IF(JSTAR.EQ.0) THEN
                    B(I)=SIGN(TEN**MAXEXP,RHS(I))*
     $                   SIGN(ONE,L(I,I))
                 ELSE
                    B(I)=-SIGN(TEN**MAXEXP,L(I,JSTAR))*
     $                    SIGN(ONE,B(JSTAR))*
     $                    SIGN(ONE,L(I,I))
                 END IF
                 IF(FRSTER) THEN
                    FRSTER=.FALSE.
                    WRITE(ICOUT,1)
                 END IF
                 IF(OUTPUT.GT.3) WRITE(ICOUT,2) I,B(I)
                 GO TO 201
              END IF
           END IF
C
C          SUM FOR EACH TERM, ORDERING OPERATIONS TO MINIMIZE
C          POSSIBILITY OF OVERFLOW.
C
           SUM=ZERO
           DO 400 J=1,I-1
              SUM=SUM+(MIN(ABS(L(I,J)),ABS(B(J)))/L(I,I))*
     $            (MAX(ABS(L(I,J)),ABS(B(J))))*
     $            SIGN(ONE,L(I,J))*SIGN(ONE,B(J))
400        CONTINUE
           B(I)=RHS(I)/L(I,I)-SUM
201     CONTINUE
200     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE LSOLV.
C
        END
        SUBROUTINE LTSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                    OUTPUT,L     ,Y     ,B     )
C
C       FEB. 14, 1991
C
C       THIS SUBROUTINE SOLVES:
C
C              L^Y=B
C
C              WHERE    L  IS TAKEN FROM THE CHOLESKY DECOMPOSITION
C                       B  IS A GIVEN RIGHT HAND SIDE WHICH IS NOT
C                          OVERWRITTEN
C                       Y  IS THE SOLUTION VECTOR
C
C       FRSTER IS USED FOR OUTPUT PURPOSES ONLY.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION L(N,N)  ,MAXLOG
        INTEGER          OUTPUT
        DIMENSION        B(N)    ,Y(N)
        LOGICAL          FRSTER  ,OVERCH  ,OVERFL
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA ZERO,ONE,TEN /0.0D0,1.0D0,10.0D0/
C
        FRSTER=.TRUE.
        OVERFL=.FALSE.
        EPS=TEN**(-MAXEXP)
C
        IF(OVERCH) THEN
           IF(LOG10(ABS(B(N))+EPS)-LOG10(ABS(L(N,N))+EPS)
     $        .GT.MAXEXP) THEN
              OVERFL=.TRUE.
              Y(N)=SIGN(TEN**MAXEXP,B(N))*
     $                   SIGN(ONE,L(N,N))
              IF(OUTPUT.GT.3) THEN
                 FRSTER=.FALSE.
                 WRITE(ICOUT,1)
1                FORMAT(T3,'*',70X,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,2) N,Y(N)
2                FORMAT(T3,'*',4X,'WARNING: COMPONENT ',I3,
     $           ' SET TO ',1PD11.3,25X,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
              GO TO 101
           END IF
        END IF
        Y(N)=B(N)/L(N,N)
101     CONTINUE
        DO 200 I=N-1,1,-1
           IF(OVERCH) THEN
C
C             CHECK TO FIND IF ANY TERMS IN THE EVALUATION WOULD
C             OVERFLOW.
C
              MAXLOG=LOG10(ABS(B(I))+EPS)-LOG10(ABS(L(I,I))+EPS)
              JSTAR=0
              DO 300 J=I+1,N
                 TMPLOG=LOG10(ABS(L(J,I))+EPS)+LOG10(ABS(Y(J))+EPS)-
     $                  LOG10(ABS(L(I,I))+EPS)
                 IF(TMPLOG.GT.MAXLOG) THEN
                    JSTAR=J
                    MAXLOG=TMPLOG
                 END IF
300           CONTINUE
C
C             IF AN OVERFLOW WOULD OCCUR ASSIGN A VALUE FOR THE
C             TERM WITH CORRECT SIGN.
C
              IF(MAXLOG.GT.MAXEXP) THEN
                 OVERFL=.TRUE.
                 IF(JSTAR.EQ.0) THEN
                    Y(I)=SIGN(TEN**MAXEXP,B(I))*
     $                   SIGN(ONE,L(I,I))
                 ELSE
                    Y(I)=-SIGN(TEN**MAXEXP,L(JSTAR,I))*
     $                          SIGN(ONE,Y(JSTAR))*
     $                    SIGN(ONE,L(I,I))
                 END IF
                 IF(FRSTER) THEN
                    FRSTER=.FALSE.
                    WRITE(ICOUT,1)
                 END IF
                 IF(OUTPUT.GT.3) WRITE(ICOUT,2) I,Y(I)
                 GO TO 201
              END IF
           END IF
C
C          SUM FOR EACH TERM ORDERING OPERATIONS TO MINIMIZE
C          POSSIBILITY OF OVERFLOW.
C
           SUM=ZERO
           DO 400 J=I+1,N
              SUM=SUM+(MIN(ABS(L(J,I)),ABS(Y(J)))/L(I,I))*
     $            (MAX(ABS(L(J,I)),ABS(Y(J))))*
     $            SIGN(ONE,L(J,I))*SIGN(ONE,Y(J))
400        CONTINUE
           Y(I)=B(I)/L(I,I)-SUM
201     CONTINUE
200     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE LTSOLV.
C
        END
      SUBROUTINE MATCOP(NRADEC,NRAACT,NCADEC,NCAACT,NRBDEC,
     $                    NCBDEC,AMAT  ,BMAT  )
C
C       SEPT. 15, 1991
C
C       COPY A CONTINGUOUS RECTANGULAR PORTION OF ONE MATRIX
C       INTO ANOTHER (ELEMENT (1,1) MUST BE INCLUDED).
C
C       NRADEC IS 1ST DIMENSION OF AMAT, NRAACT IS LIMIT OF 1ST INDEX
C       NCADEC IS 2ND DIMENSION OF AMAT, NCAACT IS LIMIT OF 2ND INDEX
C       NRBDEC IS 1ST DIMENSION OF BMAT
C       NCBDEC IS 2ND DIMENSION OF BMAT
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DIMENSION  AMAT(NRADEC,NCADEC) ,BMAT(NRBDEC,NCBDEC)
C
C       FIND NUMBER OF GROUPS OF SIZE 32, 16 ...
C
      NCC32=NRAACT/32
      NCC32R=NRAACT-32*NCC32
      NCC16=NCC32R/16
      NCC16R=NCC32R-16*NCC16
      NCC8=NCC16R/8
      NCC8R=NCC16R-8*NCC8
      NCC4=NCC8R/4
      NCC4R=NCC8R-4*NCC4
      DO 100 J=1,NCAACT
C
C          COPY ENTRIES INTO MATRIX B BY COLUMN.
C
           K=0
           IF(NCC32.GT.0) THEN
              DO 200 KK=1,NCC32
                 K=K+32
          BMAT(K-31,J)=AMAT(K-31,J)
          BMAT(K-30,J)=AMAT(K-30,J)
          BMAT(K-29,J)=AMAT(K-29,J)
          BMAT(K-28,J)=AMAT(K-28,J)
          BMAT(K-27,J)=AMAT(K-27,J)
          BMAT(K-26,J)=AMAT(K-26,J)
          BMAT(K-25,J)=AMAT(K-25,J)
          BMAT(K-24,J)=AMAT(K-24,J)
          BMAT(K-23,J)=AMAT(K-23,J)
          BMAT(K-22,J)=AMAT(K-22,J)
          BMAT(K-21,J)=AMAT(K-21,J)
          BMAT(K-20,J)=AMAT(K-20,J)
          BMAT(K-19,J)=AMAT(K-19,J)
          BMAT(K-18,J)=AMAT(K-18,J)
          BMAT(K-17,J)=AMAT(K-17,J)
          BMAT(K-16,J)=AMAT(K-16,J)
          BMAT(K-15,J)=AMAT(K-15,J)
          BMAT(K-14,J)=AMAT(K-14,J)
          BMAT(K-13,J)=AMAT(K-13,J)
          BMAT(K-12,J)=AMAT(K-12,J)
          BMAT(K-11,J)=AMAT(K-11,J)
          BMAT(K-10,J)=AMAT(K-10,J)
          BMAT(K- 9,J)=AMAT(K- 9,J)
          BMAT(K- 8,J)=AMAT(K- 8,J)
          BMAT(K- 7,J)=AMAT(K- 7,J)
          BMAT(K- 6,J)=AMAT(K- 6,J)
          BMAT(K- 5,J)=AMAT(K- 5,J)
          BMAT(K- 4,J)=AMAT(K- 4,J)
          BMAT(K- 3,J)=AMAT(K- 3,J)
          BMAT(K- 2,J)=AMAT(K- 2,J)
          BMAT(K- 1,J)=AMAT(K- 1,J)
          BMAT(K   ,J)=AMAT(K   ,J)
200           CONTINUE
           END IF
           IF(NCC16.GT.0) THEN
              DO 300 KK=1,NCC16
                 K=K+16
          BMAT(K-15,J)=AMAT(K-15,J)
          BMAT(K-14,J)=AMAT(K-14,J)
          BMAT(K-13,J)=AMAT(K-13,J)
          BMAT(K-12,J)=AMAT(K-12,J)
          BMAT(K-11,J)=AMAT(K-11,J)
          BMAT(K-10,J)=AMAT(K-10,J)
          BMAT(K- 9,J)=AMAT(K- 9,J)
          BMAT(K- 8,J)=AMAT(K- 8,J)
          BMAT(K- 7,J)=AMAT(K- 7,J)
          BMAT(K- 6,J)=AMAT(K- 6,J)
          BMAT(K- 5,J)=AMAT(K- 5,J)
          BMAT(K- 4,J)=AMAT(K- 4,J)
          BMAT(K- 3,J)=AMAT(K- 3,J)
          BMAT(K- 2,J)=AMAT(K- 2,J)
          BMAT(K- 1,J)=AMAT(K- 1,J)
          BMAT(K   ,J)=AMAT(K   ,J)
300           CONTINUE
           END IF
           IF(NCC8.GT.0) THEN
              DO 400 KK=1,NCC8
                 K=K+8
          BMAT(K- 7,J)=AMAT(K- 7,J)
          BMAT(K- 6,J)=AMAT(K- 6,J)
          BMAT(K- 5,J)=AMAT(K- 5,J)
          BMAT(K- 4,J)=AMAT(K- 4,J)
          BMAT(K- 3,J)=AMAT(K- 3,J)
          BMAT(K- 2,J)=AMAT(K- 2,J)
          BMAT(K- 1,J)=AMAT(K- 1,J)
          BMAT(K   ,J)=AMAT(K   ,J)
400           CONTINUE
           END IF
           IF(NCC4.GT.0) THEN
              DO 500 KK=1,NCC4
                 K=K+4
          BMAT(K- 3,J)=AMAT(K- 3,J)
          BMAT(K- 2,J)=AMAT(K- 2,J)
          BMAT(K- 1,J)=AMAT(K- 1,J)
          BMAT(K   ,J)=AMAT(K   ,J)
500           CONTINUE
           END IF
           IF(NCC4R.GT.0) THEN
              DO 600 KK=1,NCC4R
                 K=K+1
          BMAT(K,J)=AMAT(K,J)
600           CONTINUE
           END IF
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE MATCOP.
C
        END
      SUBROUTINE MATPRT(NROWA,NCOLA,NROWPR,NCOLPR,NUNIT,A)
C
C       FEB. 6, 1991
C
C       THIS SUBROUTINE PRINTS RECTANGULAR BLOCKS STARTING WITH
C       ELEMENT A(1,1) OF SIZE NROWPR BY NCOLPR FOR MATRIX A
C       (WHICH HAS DECLARED SIZE NROWA BY NCOLA). THE MATRIX IS
C       PRINTED AS A BLOCK FOR SIZES UP TO 5X5 OR BY COLUMNS IF
C       IT IS LARGER.
C
C       NROWA IS THE NUMBER OF DECLARED ROWS IN THE MATRIX
C       NCOLA IS THE NUMBER OF DECLARED COLUMNS IN THE MATRIX
C
C       NROWPR IS THE NUMBER OF ROWS TO BE PRINTED
C       NCOLPR IS THE NUMBER OF COLUMNS TO BE PRINTED
C
C       IF MATRIX PRINTING IS TO BE SUPPRESSED THEN LOGICAL
C       VARIABLE MATSUP MUST BE SET TO TRUE BEFORE THE CALL
C       TO NNES.
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION A(NROWA,NCOLA)
      LOGICAL MATSUP
      COMMON/NNES_1/MATSUP
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      IF(MATSUP) THEN
         WRITE(ICOUT,1)
1        FORMAT(T3,'*',T74,'*')
         CALL DPWRST('XXX','BUG')
         WRITE(ICOUT,2)
2        FORMAT(T3,'*',7X,'MATRIX PRINTING SUPPRESSED',T74,'*')
         CALL DPWRST('XXX','BUG')
         RETURN
      END IF
C
C       FOR NCOLPR <= 5 WRITE MATRIX AS A WHOLE.
C
      WRITE(ICOUT,1)
      CALL DPWRST('XXX','BUG')
      IF(NCOLPR.LE.5) THEN
         WRITE(ICOUT,3) (K,K=1,NCOLPR)
3        FORMAT(T74,'*',T3,'*',2X,5(I12:))
         CALL DPWRST('XXX','BUG')
         WRITE(ICOUT,1)
         CALL DPWRST('XXX','BUG')
         DO 100 I=1,NROWPR
            WRITE(ICOUT,4) I,(A(I,K),K=1,NCOLPR)
4           FORMAT(T74,'*',T3,'*',3X,I3,5(1PD12.3:))
            CALL DPWRST('XXX','BUG')
100      CONTINUE
        ELSE
C
C          LIMIT IS THE NUMBER OF GROUPS OF 5 COLUMNS.
C
         LIMIT=NCOLPR/5
C
C          WRITE COMPLETE BLOCKS FIRST (LEFTOVERS LATER).
C
         DO 200 J=1,LIMIT
            WRITE(ICOUT,5) (K,K=1+(J-1)*5,5+(J-1)*5)
5           FORMAT(T3,'*',2X,5I12,T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            DO 300 I=1,NROWPR
               WRITE(ICOUT,6) I,(A(I,K),K=1+(J-1)*5,5+(J-1)*5)
6              FORMAT(T3,'*',3X,I3,5(1PD12.3),T74,'*')
               CALL DPWRST('XXX','BUG')
300         CONTINUE
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
200      CONTINUE
C
C        WRITE REMAINING ELEMENTS.
C
         WRITE(ICOUT,7) (K,K=5*LIMIT+1,NCOLPR)
7        FORMAT(T74,'*',T3,'*',2X,4(I12:))
         CALL DPWRST('XXX','BUG')
         WRITE(ICOUT,1)
         CALL DPWRST('XXX','BUG')
         DO 400 I=1,NROWPR
            WRITE(ICOUT,8) I,(A(I,K),K=5*LIMIT+1,NCOLPR)
8           FORMAT(T74,'*',T3,'*',3X,I3,4(1PD12.3:))
            CALL DPWRST('XXX','BUG')
400      CONTINUE
      END IF
      RETURN
C
C       LAST CARD OF SUBROUTINE MATPRT.
C
      END

        SUBROUTINE MAXST(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,EPSMCH,
     $                   MAXSTP,MSTPF ,SCALEX,WV1   ,XC    )
C
C       FEB. 11, 1991
C
C       THIS SUBROUTINE ESTABLISHES A MAXIMUM STEP LENGTH BASED ON
C       THE 2-NORMS OF THE INITIAL ESTIMATES AND THE SCALING FACTORS
C       MULTIPLIED BY A FACTOR MSTPF.
C
C            MAXSTP=MSTPF*MAX{ NORM1 , NORM2 }
C
C                 WHERE   MSTPF  USER-CHOSEN FACTOR (DEFAULT: 1000)
C                         NORM1  2-NORM OF SCALED STARTING ESTIMATES
C                         NORM2  2-NORM OF COMPONENT SCALING FACTORS
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION   MAXSTP   ,MSTPF   ,NORM1    ,NORM2
        INTEGER            OUTPUT
        DIMENSION          SCALEX(N),WV1(N)  ,XC(N)
        LOGICAL            OVERFL   ,WRNSUP
        COMMON/NNES_2/WRNSUP
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
        DATA TEN /10.0D0/
C
        OVERFL=.FALSE.
C
        DO 100 I=1,N
           WV1(I)=SCALEX(I)*XC(I)
100     CONTINUE
        CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,NORM1,WV1)
        IF(OVERFL) THEN
           MAXSTP=TEN**MAXEXP
           IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
              WRITE(ICOUT,1)
1             FORMAT(T3,'*',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,2) NORM1
2             FORMAT(T3,'*',4X,'WARNING: NORM OF SCALED INITIAL ',
     $        'ESTIMATE SET TO ',1PD12.3,T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,3) MAXSTP
3             FORMAT(T3,'*',7X,'MAXIMUM STEP SIZE, MAXSTP, SET TO ',
     $        1PD12.3,T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
           RETURN
        END IF
        CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,NORM2,SCALEX)
        IF(OVERFL) THEN
           MAXSTP=TEN**MAXEXP
           IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,4) NORM2
4             FORMAT(T3,'*',4X,'WARNING: NORM OF SCALING FACTORS ',
     $        'SET TO ',1PD12.3,T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,3) MAXSTP
              CALL DPWRST('XXX','BUG')
           END IF
           RETURN
        END IF
        MAXSTP=MSTPF*MAX(NORM1,NORM2)
        RETURN
C
C       LAST CARD OF SUBROUTINE MAXST.
C
        END
      SUBROUTINE NERSL(NEWTON,RESTRT,SCLFCH,SCLXCH,ACPCOD,JUPDM ,
     $                   N     ,NUNIT ,OUTPUT,FCNNEW,FVEC  ,XPLUS )
C
C       SEPT. 2, 1991
C
C       THE RESULTS OF EACH ITERATION ARE PRINTED.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        INTEGER    ACPCOD  ,OUTPUT
        DIMENSION  FVEC(N) ,XPLUS(N)
      LOGICAL    NEWTON  ,RESTRT  ,SCLFCH  ,SCLXCH
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      WRITE(ICOUT,1)
1     FORMAT(T3,'*',T74,'*')
      CALL DPWRST('XXX','BUG')
      WRITE(ICOUT,1)
      CALL DPWRST('XXX','BUG')
      IF(.NOT.SCLXCH) THEN
         WRITE(ICOUT,2)
2        FORMAT(T3,'*',4X,'SUMMARY OF ITERATION RESULTS',T74,'*')
         CALL DPWRST('XXX','BUG')
      ELSE
         WRITE(ICOUT,3)
3        FORMAT(T3,'*',4X,'SUMMARY OF ITERATION RESULTS (X''','S',
     $          ' GIVEN IN UNSCALED UNITS)',T74,'*')
         CALL DPWRST('XXX','BUG')
      END IF
      WRITE(ICOUT,1)
      CALL DPWRST('XXX','BUG')
      WRITE(ICOUT,4)
4     FORMAT(T3,'*',8X,'UPDATED ESTIMATES',16X,
     $       'UPDATED FUNCTION VALUES',T74,'*')
      CALL DPWRST('XXX','BUG')
      WRITE(ICOUT,1)
      CALL DPWRST('XXX','BUG')
      DO 100 I=1,N
         WRITE(ICOUT,5) I,XPLUS(I),I,FVEC(I)
5        FORMAT(T3,'*',6X,'X(',I3,') = ',1PD12.3,15X,
     $          'F(',I3,') = ',1PD12.3,T74,'*')
         CALL DPWRST('XXX','BUG')
100   CONTINUE
      WRITE(ICOUT,1)
      CALL DPWRST('XXX','BUG')
      IF(.NOT.SCLFCH) THEN
         WRITE(ICOUT,6) FCNNEW
6        FORMAT(T3,'*',6X,'OBJECTIVE FUNCTION VALUE: ',1PD12.3,
     $          T74,'*')
         CALL DPWRST('XXX','BUG')
      ELSE
         WRITE(ICOUT,7) FCNNEW
7        FORMAT(T3,'*',6X,'SCALED OBJECTIVE FUNCTION VALUE: ',
     $          1PD12.3,T74,'*')
         CALL DPWRST('XXX','BUG')
      END IF
      WRITE(ICOUT,1)
      CALL DPWRST('XXX','BUG')
      IF(OUTPUT.GT.3.AND.(.NOT.NEWTON)) THEN
         WRITE(ICOUT,8) ACPCOD
8        FORMAT(T3,'*',6X,'STEP ACCEPTANCE CODE, ACPCOD:',I9,
     $          T74,'*')
         CALL DPWRST('XXX','BUG')
      END IF
      IF(RESTRT.AND.OUTPUT.GE.3.AND.JUPDM.NE.0) THEN
        IF(OUTPUT.GT.3) WRITE(ICOUT,1)
        CALL DPWRST('XXX','BUG')
        WRITE(ICOUT,9)
9       FORMAT(T3,'*',6X,'NOTE: JACOBIAN EVALUATED EXPLICITLY',
     $         ' AT THIS STEP',T74,'*')
        CALL DPWRST('XXX','BUG')
        WRITE(ICOUT,1)
        CALL DPWRST('XXX','BUG')
      END IF
      RETURN
C
C     LAST CARD OF SUBROUTINE NERSL.
C
      END
      SUBROUTINE NESTOP(ABSNEW,LINESR,NEWTON,SCLFCH,SCLXCH,
     $                    ACPTCR,ITNUM ,N     ,NAC1  ,NAC2  ,
     $                    NAC12 ,NFUNC ,NJETOT,NUNIT ,OUTPUT,
     $                    STOPCR,TRMCOD,FCNNEW,FTOL  ,NSTTOL,
     $                    STPMAX,STPTOL,FVEC  ,SCALEF,SCALEX,
     $                    XC    ,XPLUS)
C
C       FEB. 23, 1992
C
C       THIS SUBROUTINE CHECKS TO SEE IF THE CONVERGENCE CRITERIA
C       HAVE BEEN MET.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION MAX1     ,MAX2      ,NSTTOL
        INTEGER          ACPTCR   ,OUTPUT    ,STOPCR    ,TRMCOD
        DIMENSION        FVEC(N)  ,SCALEF(N) ,SCALEX(N) ,XC(N)  ,
     $                   XPLUS(N)
      LOGICAL          ABSNEW   ,LINESR    ,NEWTON    ,SCLFCH ,
     $                   SCLXCH
        COMMON/NNES_4/NFETOT
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,ONE,TWO /0.0D0,1.0D0,2.0D0/
C
        IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
1          FORMAT(T3,'*',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
2          FORMAT(T3,'*',4X,'CONVERGENCE TESTING',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
C
C       IF THE NEWTON STEP WAS WITHIN TOLERANCE THEN TRMCOD IS 1.
C
        IF(TRMCOD.EQ.1) THEN
           IF(OUTPUT.GT.3) THEN
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            IF(.NOT.SCLXCH) THEN
               WRITE(ICOUT,3) STPMAX
3              FORMAT(T3,'*',6X,'MAXIMUM NEWTON STEP LENGTH',
     $                ' STPMAX:',1PD12.3,T74,'*')
               CALL DPWRST('XXX','BUG')
            ELSE
               WRITE(ICOUT,4) STPMAX
4              FORMAT(T3,'*',6X,'MAXIMUM SCALED NEWTON STEP',
     $                ' LENGTH STPMAX:',1PD12.3,T74,'*')
               CALL DPWRST('XXX','BUG')
            END IF
            WRITE(ICOUT,5) NSTTOL
5           FORMAT(T3,'*',6X,'FIRST CONVERGENCE CRITERION MET; ',
     $             ' NSTTOL IS:',1PD12.3,T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
C
C             SKIP CHECKING OTHER STEP SIZE CRITERION AS TRMCOD IS
C             ALREADY 1.
C
              GO TO 101
           END IF
        END IF
C
C       IF THE NEWTON STEP WAS NOT WITHIN TOLERANCE THEN, IF
C       STOPCR IS NOT EQUAL TO 2, THE SECOND STEP SIZE STOPPING
C       CRITERION MUST BE CHECKED.
C
        IF(STOPCR.NE.2.AND.TRMCOD.NE.1) THEN
           MAX1=ZERO
           DO 100 I=1,N
              RATIO1=(ABS(XPLUS(I)-XC(I)))/MAX(ABS(XPLUS(I)),
     $               ONE/SCALEX(I))
              MAX1=MAX(MAX1,RATIO1)
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,6) I,RATIO1
6                FORMAT(T3,'*',6X,'RELATIVE STEP SIZE (',I3,') = ',
     $           1PD12.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
100        CONTINUE
           IF(OUTPUT.GT.4) WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
            IF(OUTPUT.GT.3) THEN
               IF(.NOT.SCLXCH) THEN
                 WRITE(ICOUT,7) MAX1,STPTOL
7                FORMAT(T3,'*',6X,'MAXIMUM STEP SIZE:',
     $           1PD12.3,3X,'STPTOL:',1PD11.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
            ELSE
                 WRITE(ICOUT,8) MAX1,STPTOL
8                FORMAT(T3,'*',6X,'MAXIMUM RELATIVE STEP SIZE:',
     $           1PD12.3,3X,'STPTOL:',1PD11.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
            END IF
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
           END IF
           IF(MAX1.LT.STPTOL) THEN
              TRMCOD=1
           END IF
        END IF
C
C       NOTE: CONTINUATION AT 101 MEANS THAT TRMCOD WAS 1 ON ENTRY
C             SO THE STEP SIZE CRITERION ABOVE DID NOT NEED TO BE
C             CHECKED.
C
101     CONTINUE
C
C       THE SECOND STOPPING CRITERION IS CHECKED IF NEEDED.
C
        IF(STOPCR.EQ.2.OR.STOPCR.EQ.12.OR.
     $      (STOPCR.EQ.3.AND.TRMCOD.EQ.1)) THEN
           MAX2=ZERO
           DO 200 I=1,N
              MAX2=MAX(MAX2,SCALEF(I)*ABS(FVEC(I)))
            IF(OUTPUT.GT.4) THEN
               IF(.NOT.SCLFCH) THEN
                  WRITE(ICOUT,9) I,ABS(FVEC(I))
9                 FORMAT(T3,'*',6X,'ABSOLUTE FUNCTION VECTOR (',I3,
     $                   ') = ',1PD12.3,T74,'*')
                  CALL DPWRST('XXX','BUG')
               ELSE
                  WRITE(ICOUT,10) I,SCALEF(I)*ABS(FVEC(I))
10                FORMAT(T3,'*',6X,'ABSOLUTE SCALED FUNCTION',
     $                   ' VECTOR (',I3,') = ',1PD12.3,T74,'*')
                  CALL DPWRST('XXX','BUG')
               END IF
            END IF
200        CONTINUE
           IF(OUTPUT.GT.4) WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           IF(OUTPUT.GT.3) THEN
              IF(.NOT.SCLFCH) THEN
                 WRITE(ICOUT,11) MAX2,FTOL
11               FORMAT(T3,'*',6X,'MAXIMUM ABSOLUTE FUNCTION:',
     $           1PD12.3,5X,'FTOL:',1PD11.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
            ELSE
                 WRITE(ICOUT,12) MAX2,FTOL
12               FORMAT(T3,'*',6X,'MAX ABSOLUTE SCALED',
     $           ' FUNCTION:',1PD12.3,5X,'FTOL:',1PD11.3,T74,'*')
                 CALL DPWRST('XXX','BUG')
            END IF
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
           END IF
           IF(MAX2.LT.FTOL) THEN
              IF(STOPCR.EQ.3.AND.TRMCOD.EQ.1) THEN
C
C                BOTH NEEDED STOPPING CRITERIA HAVE BEEN MET.
C
                 TRMCOD=3
              ELSEIF(STOPCR.EQ.12.AND.TRMCOD.EQ.1) THEN
C
C                BOTH STOPPING CRITERIA HAVE BEEN MET ALTHOUGH
C                EITHER ONE WOULD BE SATISFACTORY.
C
                 TRMCOD=12
              ELSEIF(STOPCR.EQ.2.OR.STOPCR.EQ.12) THEN
                 TRMCOD=2
              END IF
           ELSEIF(STOPCR.EQ.3) THEN
C
C             ONLY THE FIRST STOPPING CRITERION WAS MET - TRMCOD
C             MUST BE RESET FROM 1 BACK TO 0.
C
              TRMCOD=0
           END IF
        END IF
C
C       PRINT FINAL RESULTS IF CONVERGENCE REACHED.
C
        IF(TRMCOD.GT.0) THEN
           IF(OUTPUT.GT.0) THEN
            IF(OUTPUT.EQ.1) WRITE(ICOUT,13)
13          FORMAT(T3,72('*'))
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,14) TRMCOD
14          FORMAT(T3,'*',6X,'CONVERGENCE REACHED;',
     $             ' TERMINATION CODE: ...............',I6,T74,'*')
            CALL DPWRST('XXX','BUG')
C
C             ITERATION RESULTS NOT PRINTED IN NERSL.
C
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            IF(SCLFCH.OR.SCLXCH) THEN
               WRITE(ICOUT,15)
15             FORMAT(T3,'*',29X,'UNSCALED RESULTS',T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
           END IF
           WRITE(ICOUT,16)
16         FORMAT(T3,'*',10X,'FINAL ESTIMATES',19X,'FINAL',
     $            ' FUNCTION VALUES',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           DO 300 I=1,N
              WRITE(ICOUT,17) I,XPLUS(I),I,FVEC(I)
17            FORMAT(T3,'*',6X,'X(',I3,') = ',1PD14.5,14X,
     $               'F(',I3,') = ',1PD14.5,T74,'*')
              CALL DPWRST('XXX','BUG')
300        CONTINUE
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           IF(SCLFCH) THEN
C
C                NEED UNSCALED OBJECTIVE FUNCTION.
C
          SUM=ZERO
          DO 400 I=1,N
             SUM=SUM+FVEC(I)*FVEC(I)
400       CONTINUE
          WRITE(ICOUT,18) SUM/TWO
          CALL DPWRST('XXX','BUG')
            ELSE
          WRITE(ICOUT,18) FCNNEW
          CALL DPWRST('XXX','BUG')
            END IF
18            FORMAT(T3,'*',6X,'FINAL OBJECTIVE FUNCTION VALUE:',
     $        1PD12.3,T74,'*')
            IF(SCLFCH.OR.SCLXCH) THEN
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,19)
19             FORMAT(T3,'*',30X,'SCALED RESULTS',T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,1998)
1998           FORMAT(T3,'*',T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,1999)
1999           FORMAT(T3,'*',10X,'FINAL ESTIMATES',19X,'FINAL',
     $                ' FUNCTION VALUES',T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               DO 500 I=1,N
                  WRITE(ICOUT,17) I,SCALEX(I)*XPLUS(I),
     $                            I,SCALEF(I)*FVEC(I)
                  CALL DPWRST('XXX','BUG')
500            CONTINUE
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,18) FCNNEW
               CALL DPWRST('XXX','BUG')
            END IF
         END IF
        ELSE
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,20)
20            FORMAT(T3,'*',6X,'CONVERGENCE NOT REACHED',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
           END IF
           RETURN
        END IF
C
C       TERMINATION HAS BEEN REACHED.
C
        IF(OUTPUT.GT.0) THEN
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,21) ITNUM
21          FORMAT(T3,'*',6X,'TOTAL NUMBER OF ITERATIONS',
     $             ': ..........................',I6,T74,'*')
            CALL DPWRST('XXX','BUG')
            IF(.NOT.NEWTON.AND.(.NOT.ABSNEW)) THEN
              IF(LINESR) THEN
                 WRITE(ICOUT,22) NFUNC
22               FORMAT(T3,'*',6X,'TOTAL NUMBER OF LINE SEARCH ',
     $           'FUNCTION EVALUATIONS: ....',I6,T74,'*')
                 CALL DPWRST('XXX','BUG')
              ELSE
                 WRITE(ICOUT,23) NFUNC
23               FORMAT(T3,'*',6X,'TOTAL NUMBER OF TRUST REGION',
     $           ' FUNCTION EVALUATIONS: ...',I6,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
           END IF
           WRITE(ICOUT,24) NJETOT
24         FORMAT(T3,'*',6X,'TOTAL NUMBER OF EXPLICIT JACOBIAN',
     $            ' EVALUATIONS: .......',I6,T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,25) NFETOT
25         FORMAT(T3,'*',6X,'TOTAL NUMBER OF FUNCTION',
     $            ' EVALUATIONS: ................',I6,T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           IF(.NOT.NEWTON.AND.(.NOT.ABSNEW).AND.ACPTCR.NE.1.
     $        AND.OUTPUT.GT.2) THEN
              WRITE(ICOUT,26) NAC1
26            FORMAT(T3,'*',6X,'NUMBER OF STEPS ACCEPTED',
     $               ' BY FUNCTION VALUE ONLY: .....',I6,T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,27) NAC2
27            FORMAT(T3,'*',6X,'NUMBER OF STEPS ACCEPTED',
     $               ' BY STEP SIZE VALUE ONLY: ....',I6,T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,28) NAC12
28            FORMAT(T3,'*',6X,'NUMBER OF STEPS ACCEPTED',
     $               ' BY EITHER CRITERION: ........',I6,T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
           END IF
         WRITE(ICOUT,13)
         CALL DPWRST('XXX','BUG')
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE NESTOP.
C
        END
      SUBROUTINE NNES(ABSNEW ,CAUCHY ,DEUFLH ,GEOMS  ,LINESR ,
     $                  NEWTON ,OVERCH ,ACPTCR ,ITSCLF ,ITSCLX ,
     $                  JACTYP ,JUPDM  ,MAXEXP ,MAXIT  ,MAXNS  ,
     $                  MAXQNS ,MGLL   ,MINQNS ,N      ,NARMIJ ,
     $                  NIEJEV ,NJACCH ,NJETOT ,NUNIT  ,OUTPUT ,
     $                  QNUPDM ,STOPCR ,SUPPRS ,TRMCOD ,TRUPDM ,
     $                  ALPHA  ,CONFAC ,DELTA  ,DELFAC ,EPSMCH ,
     $                  ETAFAC ,FCNNEW ,FDTOLJ ,FTOL   ,LAM0   ,
     $                  MSTPF  ,NSTTOL ,OMEGA  ,RATIOF ,SIGMA  ,
     $                  STPTOL ,A      ,BOUNDL ,BOUNDU ,DELF   ,
     $                  FSAVE  ,FTRACK ,FVEC   ,FVECC  ,H      ,
     $                  HHPI   ,JAC    ,PLEE   ,RDIAG  ,S      ,
     $                  SBAR   ,SCALEF ,SCALEX ,SN     ,SSDHAT ,
     $                  STRACK ,VHAT   ,WV1    ,WV2    ,WV3    ,
     $                  WV4    ,XC     ,XPLUS  ,XSAVE  ,HELP   ,
     $                  FVECEV ,JACEV  )
C
C       FEB. 28, 1992
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION JAC(N,N)  ,LAM0       ,MAXSTP     ,MSTPF    ,
     $                   NEWLEN    ,NEWMAX     ,NSTTOL
      INTEGER          ACPCOD    ,ACPSTR     ,ACPTCR     ,CONTYP   ,
     $                   COUNTR    ,OUTPUT     ,OUTTMP     ,QNUPDM   ,
     $                   RETCOD    ,STOPCR     ,SUPPRS     ,TRMCOD   ,
     $                   TRUPDM
        DIMENSION        A(N,N)    ,BOUNDL(N)  ,BOUNDU(N)  ,DELF(N)  ,
     $                   FSAVE(N)  ,FTRACK(0:MGLL-1 )      ,FVEC(N)  ,
     $                   FVECC(N)  ,H(N,N)     ,HHPI(N)    ,PLEE(N,N),
     $                   RDIAG(N)  ,S(N)       ,SBAR(N)    ,SCALEF(N),
     $                   SCALEX(N) ,SN(N)      ,SSDHAT(N)  ,
     $                   STRACK(0:MGLL-1)      ,VHAT(N)    ,WV1(N)   ,
     $                   WV2(N)    ,WV3(N)     ,WV4(N)     ,XC(N)    ,
     $                   XPLUS(N)  ,XSAVE(N)
      LOGICAL          ABORT     ,ABSNEW     ,CAUCHY     ,CHECKJ   ,
     $                   DEUFLH    ,FRSTDG     ,GEOMS      ,INSTOP   ,
     $                   JACERR    ,LINESR     ,MATSUP     ,NEWTON   ,
     $                   NEWTKN    ,OVERCH     ,OVERFL     ,QNFAIL   ,
     $                   QRSING    ,RESTRT     ,SAVEST     ,SCLFCH   ,
     $                   SCLXCH    ,WRNSUP
      CHARACTER*6      HELP
      COMMON/NNES_1/MATSUP
        COMMON/NNES_2/WRNSUP
        COMMON/NNES_6/ITNUM,NFUNC
        EXTERNAL FVECEV,JACEV
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,PT01,ONE /0.0D0,0.01D0,1.0D0/
C
C       PRINT HELP IF REQUESTED.
C
CCCCC IF(HELP(1:4).NE.'NONE') THEN
CCCCC    CALL OLHELP(NUNIT,HELP)
CCCCC    RETURN
CCCCC END IF
        QNFAIL=.FALSE.
        RESTRT=.TRUE.
        SAVEST=.FALSE.
        COUNTR=0
        ISEJAC=0
        MNEW=0
        NAC1=0
        NAC2=0
        NAC12=0
        NFUNC=1
        NJETOT=0
        OUTTMP=OUTPUT
        TRMCOD=0
C
C       ESTABLISH INITIAL FUNCTION VALUE AND CHECK FOR STARTING
C       ESTIMATE WHICH IS A SOLUTION.  ALSO, CHECK FOR INCOMPAT-
C       IBILITIES IN INPUT PARAMETERS.
C
      CALL INITCH(INSTOP,LINESR,NEWTON,OVERFL,SCLFCH,
     $              SCLXCH,ACPTCR,CONTYP,JACTYP,JUPDM ,
     $              MAXEXP,N     ,NUNIT ,OUTPUT,QNUPDM,
     $              STOPCR,TRUPDM,EPSMCH,FCNOLD,FTOL  ,
     $              BOUNDL,BOUNDU,FVECC ,SCALEF,SCALEX,
     $              WV1   ,XC    ,FVECEV)
C
C       IF A FATAL ERROR IS DETECTED IN INITCH RETURN TO MAIN
C       PROGRAM.  NOTE: SOME INCOMPATIBILITIES ARE CORRECTED
C       WITHIN INITCH AND EXECUTION CONTINUES.  WARNINGS ARE
C       GENERATED WITHIN INITCH.
C
        IF(INSTOP) RETURN
C
C       ESTABLISH MAXIMUM STEP LENGTH ALLOWED (USUALLY THIS IS MUCH
C       LARGER THAN ACTUAL STEP SIZES - IT IS ONLY TO PREVENT
C       EXCESSIVELY LARGE STEPS).  THE FACTOR MSTPF CONTROLS THE
C       MAGNITUDE OF MAXSTP AND IS SET BY THE USER (DEFAULT=1000).
C
      CALL MAXST(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,EPSMCH,
     $             MAXSTP,MSTPF ,SCALEX,WV1   ,XC    )
C
C       WRITE TITLE AND RECORD PARAMETERS.
C
        CALL TITLE(CAUCHY,DEUFLH,GEOMS ,LINESR,NEWTON,
     $             OVERCH,ACPTCR,CONTYP,ITSCLF,ITSCLX,
     $             JACTYP,JUPDM ,MAXIT ,MAXNS ,MAXQNS,
     $             MGLL  ,MINQNS,N     ,NARMIJ,NIEJEV,
     $             NJACCH,NUNIT ,OUTPUT,QNUPDM,STOPCR,
     $             TRUPDM,ALPHA ,CONFAC,DELFAC,DELTA ,
     $             EPSMCH,ETAFAC,FCNOLD,FTOL  ,LAM0  ,
     $             MAXSTP,MSTPF ,NSTTOL,OMEGA ,RATIOF,
     $             SIGMA ,STPTOL,BOUNDL,BOUNDU,FVECC ,
     $             SCALEF,SCALEX,XC    )
C
C       INITIALIZE FTRACK AND STRACK VECTORS (FTRACK STORES
C       (TRACKS) THE FUNCTION VALUES FOR THE NONMONOTONIC
C       COMPARISON AND STRACK, SIMILARLY, STORES THE LENGTH
C       OF THE NEWTON STEPS - WHICH ARE USED IN CONJUNCTION
C       WITH DEUFLHARD'S SECOND ACCEPTANCE CRITERION).
C
        DO 100 J=0,MGLL-1
         FTRACK(J)=ZERO
         STRACK(J)=ZERO
100     CONTINUE
C
C       MAIN ITERATIVE LOOP - MAXIT IS SPECIFIED BY USER.
C
C       ITNUM COUNTS OVERALL ITERATIONS.
C       ISEJAC COUNTS ITERATIONS SINCE LAST EXPLICIT JACOBIAN
C       EVALUATION IF A QUASI-NEWTON METHOD IS BEING USED.
C
        DO 200 ITNUM=1,MAXIT
C
C          SUPPRESS OUTPUT IF DESIRED - USED IF DETAILED OUTPUT
C          IS DESIRED FOR LATER ITERATIONS ONLY (DEFAULT VALUE
C          FOR SUPPRS IS 0 - I.E. NO SUPPRESSION).
C
           IF(ITNUM.LT.SUPPRS) THEN
              OUTPUT=3
           ELSE
              OUTPUT=OUTTMP
         END IF
C
           IF(OUTPUT.GT.2) THEN
              WRITE(ICOUT,1)
1             FORMAT(T3,72('*'))
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,2)
2             FORMAT(T3,'*',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,3) ITNUM
3             FORMAT(T3,'*',2X,'ITERATION NUMBER: ',I5,T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
C
C          UPDATE ITERATION COUNTER, ISEJAC, FOR QUASI-NEWTON METHODS
C          ONLY (I.E. JUPDM > 0).
C
           IF(JUPDM.GT.0.AND.RESTRT.AND.(.NOT.NEWTON)) THEN
              ISEJAC=1
           ELSE
              ISEJAC=ISEJAC+1
           END IF
           IF(OUTPUT.GT.4.AND.JUPDM.GT.0.AND.(.NOT.NEWTON)) THEN
              WRITE(ICOUT,2)
              CALL DPWRST('XXX','BUG')
              IF(RESTRT) THEN
                 IF(ITNUM.GT.NIEJEV) THEN
                    WRITE(ICOUT,4)
4                   FORMAT(T3,'*',4X,'RESTRT IS TRUE, ISEJAC SET',
     $                     ' TO 1',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
              ELSE
                 WRITE(ICOUT,5) ISEJAC
5                FORMAT(T3,'*',4X,'# OF ITERATIONS SINCE EXPLICIT',
     $                  ' JACOBIAN, ISEJAC, INCREASED TO',I4,T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
           END IF
C
C          WHEN AN EXPLICIT JACOBIAN IS BEING USED IN QUASI-NEWTON
C          METHODS THEN MAXNS STEPS ARE ALLOWED IN THE LINE SEARCH.
C          FOR STEPS BASED ON A QUASI-NEWTON APPROXIMATION MAXQNS
C          ARE ALLOWED.  SIMILARLY MAXNS AND MAXQNS STEPS ARE ALLOWED
C          IN TRUST REGION METHODS RESPECTIVELY. THIS IS AN ATTEMPT TO
C          AVOID AN EXCESSIVE NUMBER OF FUNCTION EVALUATIONS IN A
C          DIRECTION WHICH WOULD NOT LEAD TO A SIGNIFICANT REDUCTION.
C
           IF(.NOT.NEWTON) THEN
              IF(LINESR) THEN
                 IF(JUPDM.GT.0) THEN
                    IF(ISEJAC.EQ.1) THEN
C
C                      JACOBIAN UPDATED EXPLICITLY.
C
                       MAXLIN=MAXNS
                    ELSE
C
C                      QUASI-NEWTON UPDATE.
C
                       MAXLIN=MAXQNS
                    END IF
                 ELSE
                    MAXLIN=MAXNS
                 END IF
              ELSE
                 IF(JUPDM.GT.0) THEN
                    IF(ISEJAC.EQ.1) THEN
C
C                      JACOBIAN UPDATED EXPLICITLY.
C
                       MAXTRS=MAXNS
                    ELSE
C
C                      QUASI-NEWTON UPDATE.
C
                       MAXTRS=MAXQNS
                    END IF
                 ELSE
                    MAXTRS=MAXNS
                 END IF
              END IF
              IF(JUPDM.GT.0.AND.OUTPUT.GT.4) THEN
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 IF(LINESR) THEN
                    WRITE(ICOUT,6) MAXLIN
6                   FORMAT(T3,'*',4X,'MAXLIN SET TO:',I5,T74,'*')
                    CALL DPWRST('XXX','BUG')
                 ELSE
                    WRITE(ICOUT,7) MAXTRS
7                   FORMAT(T3,'*',4X,'MAXTRS SET TO:',I5,T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
              END IF
           END IF
C
C          ESTABLISH WHETHER JACOBIAN IS TO BE CHECKED NUMERICALLY -
C          NJACCH ESTABLISHES THE NUMBER OF ITERATIONS FOR WHICH
C          JACOBIAN CHECKING IS DESIRED.  IF CHECKJ IS TRUE A FORWARD
C          DIFFERENCE NUMERICAL APPROXIMATION OF THE JACOBIAN IS
C          COMPARED TO THE ANALYTICAL VERSION.  STORE THE NUMBER
C          OF FUNCTION EVALUATIONS SO THAT THESE "EXTRA" ARE NOT
C          INCLUDED IN OVERALL STATISTICS.
C
           IF(JACTYP.EQ.0) THEN
              IF(ITNUM.GT.NJACCH) THEN
                 CHECKJ=.FALSE.
              ELSE
                 CHECKJ=.TRUE.
                 NFESTR=NFETOT
              END IF
           END IF
C
C          EVALUATE JACOBIAN AT FIRST STEP OR IF NO QUASI-NEWTON
C          UPDATE IS BEING USED (RESTRT IS FALSE ONLY IN QUASI-
C          NEWTON METHODS WHEN THE QUASI-NEWTON UPDATE IS BEING
C          USED).
C
           IF(RESTRT.OR.JUPDM.EQ.0) THEN
C
C             IF MORE THAN ONE DAMPED NEWTON STEP HAS BEEN REQUESTED
C             AT THE START, IDENTIFY THIS AS THE REASON FOR EXPLICIT
C             JACOBIAN EVALUATION.
C
            IF(JUPDM.GT.0.AND.(ITNUM.LE.NIEJEV.AND.ITNUM.GT.1)
     $           .AND.OUTPUT.GT.4) THEN
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,8)
8                FORMAT(T3,'*',4X,'AS ITNUM <= NIEJEV',
     $                  ' JACOBIAN EVALUATED EXPLICITLY',T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
C
C             OTHERWISE JUPDM IS 0 OR RESTRT IS TRUE AND ITNUM IS
C             GREATER THEN NIEJEV.
C
              IF(JUPDM.GT.0.AND.ITNUM.GT.1.AND.OUTPUT.GT.4.
     $           AND.ITNUM.GT.NIEJEV) THEN
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,9)
9                FORMAT(T3,'*',4X,'RESTRT IS TRUE - JACOBIAN',
     $                  ' EVALUATED EXPLICITLY',T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
C
C             NOTE: MATRIX H IS USED HERE TO HOLD THE FINITE
C                   DIFFERENCE ESTIMATION USED IN CHECKING THE
C                   ANALYTICAL JACOBIAN IF CHECKJ IS TRUE.
C                   VECTORS WV1 AND, FOR CENTRAL DIFFERENCES,
C                   WV2 TEMPORARILY HOLD THE FINITE DIFFERENCE
C                   FUNCTION EVALUATIONS.
C
              CALL JACOBI(CHECKJ,JACERR,OVERFL,JACTYP,N     ,
     $                    NUNIT ,OUTPUT,EPSMCH,FDTOLJ,BOUNDL,
     $                    BOUNDU,FVECC ,WV1   ,WV2   ,JAC   ,
     $                    H     ,SCALEX,WV3   ,XC    ,FVECEV,
     $                    JACEV )
            NJETOT=NJETOT+1
C
C             RETURN IF ANALYTICAL AND NUMERICAL JACOBIANS DON'T
C             AGREE (APPLICABLE ONLY IF CHECKJ IS TRUE AND A DIS-
C             CREPANCY IS FOUND WITHIN SUBROUTINE JACOBI).  A
C             WARNING IS GIVEN FROM WITHIN JACOBI.
C
              IF(JACERR) RETURN
C
C             RESET TOTAL NUMBER OF FUNCTION EVALUATIONS TO NEGLECT
C             THOSE USED IN CHECKING ANALYTICAL JACOBIAN.
C
              IF(CHECKJ) NFETOT=NFESTR
C
              IF(JUPDM.GT.0) THEN
C
C                FCNMIN IS THE MINIMUM OF THE OBJECTIVE FUNCTION FOUND
C                SINCE THE LAST EXPLICIT JACOBIAN EVALUATION.  IT IS
C                USED TO ESTABLISH WHICH STEP THE PROGRAM RETURNS TO
C                WHEN A QUASI-NEWTON STEP FAILS.
C
                 FCNMIN=FCNOLD
C
C                POWTAU IS THE TAU FROM POWELL'S TRUST REGION UPDATING
C                SCHEME (USED WHEN JUPDM=1).  iT IS RESET TO 1.0 AT
C                EVERY EXPLICIT JACOBIAN EVALUATION IN QUASI-NEWTON
C                METHODS.
C
          POWTAU=ONE
C
C                AT EVERY EXPLICIT JACOBIAN EVALUATION EXCEPT
C                POSSIBLY THE FIRST, IN QUASI-NEWTON METHODS, A
C                NEW TRUST REGION IS CALCULATED INTERNALLY USING
C                EITHER THE NEWTON STEP OR THE CAUCHY STEP AS
C                SPECIFIED BY THE USER IN LOGICAL VARIABLE "CAUCHY"
C                IN SUBROUTINE DELCAU.  THIS IS FORCED BY SETTING
C                DELTA TO A NEGATIVE NUMBER.
C
          IF(ITNUM.GT.1) DELTA=-ONE
C
C                RESET COUNTER FOR NUMBER OF FAILURES OF RATIO
C                TEST (IS FCNNEW/FCNOLD LESS THAN RATIOF?) SINCE
C                LAST EXPLICIT JACOBIAN UPDATE.
C
                 NFAIL=0
                 IF(OUTPUT.GT.4.AND.ITNUM.GT.NIEJEV) THEN
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,10)
10                  FORMAT(T3,'*',4X,'NUMBER OF FAILURES OF RATIO',
     $                     ' TEST, NFAIL, SET BACK TO 0',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
C
C                ESTABLISH A NEW MAXIMUM STEP LENGTH ALLOWED.
C
                 IF(ITNUM.GT.1) CALL MAXST(OVERFL,MAXEXP,N     ,NUNIT,
     $                                     OUTPUT,EPSMCH,MAXSTP,MSTPF ,
     $                                     SCALEX,WV1   ,XC    )
C
C                SET "P" MATRIX TO IDENTITY FOR LEE AND LEE UPDATE
C                (JUPDM=2).
C
          IF(JUPDM.EQ.2.AND.ITNUM.GE.NIEJEV) THEN
                    DO 300 J=1,N
                       DO 400 I=1,N
              PLEE(I,J)=ZERO
400                    CONTINUE
                PLEE(J,J)=ONE
300                 CONTINUE
                 END IF
              END IF
           END IF
         IF((RESTRT.OR.QNUPDM.EQ.0).AND.OUTPUT.GT.4.AND.
     $     (.NOT.MATSUP)) THEN
C
C             WRITE JACOBIAN MATRIX.
C
              WRITE(ICOUT,2)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,11)
11            FORMAT(T3,'*',4X,'JACOBIAN MATRIX',T74,'*')
              CALL DPWRST('XXX','BUG')
              CALL MATPRT(N,N,N,N,NUNIT,JAC)
         END IF
C
C          ESTABLISH SCALING MATRICES IF DESIRED (ITSCLF=0 => NO
C          ADAPTIVE SCALING, WHILE ITSCLF > 0 MEANS ADAPTIVE SCALING
C          STARTS AFTER THE (ITSCLF)TH ITERATION).  SIMILARLY FOR
C          COMPONENT SCALING ... .  NOTE: SCALING FACTORS ARE UPDATED
C          ONLY WHEN THE JACOBIAN IS UPDATED EXPLICITLY IN QUASI-NEWTON
C          METHODS.
C
C          FUNCTION SCALING.
C
           IF(RESTRT.AND.ITSCLF.GT.0.AND.ITSCLF.LE.ITNUM) THEN
C
            CALL ASCALF(N,EPSMCH,FVECC,JAC,SCALEF)
C
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,12)
12               FORMAT(T3,'*',4X,'FUNCTION SCALING MATRIX',
     $                  T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 DO 500 I=1,N
                    WRITE(ICOUT,13) I,SCALEF(I)
13                  FORMAT(T3,'*',7X,'SCALEF(',I3,') = ',1PD12.3,
     $                     T74,'*')
                    CALL DPWRST('XXX','BUG')
500              CONTINUE
              END IF
C
C             RECALCULATE OBJECTIVE FUNCTION WITH NEW SCALING
C             FACTORS.  THIS AVOIDS PREMATURE FAILURES IF THE
C             CHANGE IS SCALING FACTORS WOULD MAKE THE PREVIOUS
C             OBJECTIVE FUNCTION VALUE SMALLER.
C
              CALL FCNEVL(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,
     $                    EPSMCH,FCNOLD,FVECC ,SCALEF,WV1   )
           END IF
C
C          COMPONENT SCALING.
C
           IF(RESTRT.AND.ITSCLX.GT.0.AND.ITSCLX.LE.ITNUM) THEN
C
            CALL ASCALX(N,EPSMCH,JAC,SCALEX)
C
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,14)
14               FORMAT(T3,'*',7X,'COMPONENT SCALING MATRIX',
     $                  T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 DO 600 I=1,N
                    WRITE(ICOUT,15) I,SCALEX(I)
15                  FORMAT(T3,'*',10X,'SCALEX(',I3,') = ',1PD12.3,
     $                     T74,'*')
                    CALL DPWRST('XXX','BUG')
600              CONTINUE
              END IF
           END IF
C
C          FIND GRADIENT OF 1/2 FVECC^FVECC (NOT USED IF DECOMPOSED
C          MATRIX IS UPDATED IN WHICH CASE THE GRADIENT IS FOUND
C          WITHIN THAT SUBROUTINE - CALL IS MADE FOR OUTPUT ONLY).
C
C
         CALL GRADF(OVERCH,OVERFL,RESTRT,SCLFCH,SCLXCH,JUPDM ,
     $                MAXEXP,N     ,NUNIT ,OUTPUT,QNUPDM,DELF  ,
     $                FVECC ,JAC   ,SCALEF,SCALEX,WV1   )
C
C          FIND NEWTON STEP USING QR DECOMPOSITION.
C
C          IF JUPDM = 0 OR QNUPDM = 0 THEN THE UNFACTORED FORM
C          FOR CALCULATING THE NEWTON STEP IS USED.
C
           IF(JUPDM.EQ.0.OR.QNUPDM.EQ.0) THEN
C
C             NEWTON STEP - UNFACTORED FORM BEING UPDATED.
C
              CALL NSTPUN(ABORT ,LINESR,OVERCH,OVERFL,QRSING,
     $                    SCLFCH,SCLXCH,ITNUM ,MAXEXP,N     ,
     $                    NUNIT ,OUTPUT,EPSMCH,A     ,DELF  ,
     $                    FVECC ,H     ,HHPI  ,JAC   ,RDIAG ,
     $                    SCALEF,SCALEX,SN    ,WV1   ,WV2   ,
     $                    WV3   )
           ELSE
C
C             NEWTON STEP - FACTORED FORM BEING UPDATED.
C
            CALL NSTPFA(ABORT ,LINESR,OVERCH,OVERFL,QRSING,
     $                    RESTRT,SCLFCH,SCLXCH,ITNUM ,MAXEXP,
     $                    N     ,NEWSTM,NUNIT ,OUTPUT,EPSMCH,
     $                    A     ,DELF  ,FVECC ,H     ,HHPI  ,
     $                    JAC   ,RDIAG ,SCALEF,SCALEX,SN    ,
     $                    WV1   ,WV2   ,WV3)
C
           END IF
C
C          RUN IS ABORTED IF THE JACOBIAN BECOMES ESSENTIALLY
C          ALL ZEROS.  NOTE: 203 FOLLOWS END OF MAIN 200 LOOP.
C
           IF(ABORT) GO TO 203
C
C          CHECK FOR CONVERGENCE ON LENGTH OF NEWTON STEP IF
C          STOPCR = 1, 12 OR 3.
C
           IF(STOPCR.NE.2) THEN
            STPMAX=ZERO
              DO 700 I=1,N
                 STPMAX=MAX(STPMAX,ABS(SN(I))/MAX(XC(I),SCALEX(I)))
                 WV1(I)=SCALEX(I)*XC(I)
700           CONTINUE
            CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,XNORM,WV1)
            IF(STPMAX.LE.NSTTOL*(ONE+XNORM)) THEN
                 TRMCOD=1
C
C                IF STOPCR=3 THEN OBJECTIVE FUNCTION VALUE MUST
C                BE DETERMINED AS WELL - OTHERWISE A SOLUTION
C                HAS BEEN FOUND.
C
                 IF(STOPCR.NE.3) THEN
                    GO TO 202
C
C                   NOTE: STATEMENT 202 PRECEDES CONVERGENCE
C                         CHECKING SUBROUTINE.
C
                 END IF
              END IF
           END IF
C
C          FIND LENGTH OF (SCALED) NEWTON STEP, NEWLEN.
C
           DO 800 I=1,N
              WV1(I)=SCALEX(I)*SN(I)
800        CONTINUE
         CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,NEWLEN,WV1)
C
C          FOR ITERATIONS AFTER THE ARMIJO STEPS HAVE BEEN COMPLETED
C          AT THE BEGINNING (IN OTHER WORDS THE MONOTONIC STEPS)
C          STORE THE FUNCTION AND, POSSIBLY, THE NEWTON STEP LENGTHS
C          IN THE FTRACK AND STRACK VECTORS, RESPECTIVELY.
C
           IF(ISEJAC.GE.NARMIJ) THEN
              IF(ISEJAC.EQ.1) THEN
                 STRACK(0)=NEWLEN
C
C                NEWMAX IS USED TO KEEP A BOUND ON THE ENTRIES IN
C                THE STRACK VECTOR.
C
                 NEWMAX=NEWLEN
              ELSE
                 STRACK(COUNTR)=MIN(NEWMAX,NEWLEN)
              END IF
C
C             THE OBJECTIVE FUNCTION VALUE IS STORED EVEN IF IT IS
C             GREATER THAN ANY PRECEEDING FUNCTION VALUE.
C
              FTRACK(COUNTR)=FCNOLD
C
C             WRITE FTRACK AND STRACK VECTORS IF DESIRED.  SINCE ONLY
C             THE LAST MGLL VALUES ARE NEEDED THE COUNTER CIRCULATES
C             THROUGH THE VECTOR CAUSING ONLY THE MGLL MOST RECENT
C             VALUES TO BE KEPT.  NOTE: THESE VECTORS ARE NOT APPLI-
C             CABLE IF NEWTON'S METHOD IS BEING USED.
C
              IF(.NOT.NEWTON.AND.OUTPUT.GT.4) THEN
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
C
C                IF ONLY THE FUNCTION VALUE ACCEPTANCE TEST IS BEING
C                USED, THUS ACPTCR=1, THEN ONLY THE FTRACK VECTOR
C                IS APPLICABLE.
C
                 IF(ACPTCR.EQ.1) THEN
                    WRITE(ICOUT,16) COUNTR
16                  FORMAT(T3,'*',4X,'CURRENT FTRACK VECTOR;',
     $                     2X,'LATEST CHANGE:  ELEMENT',I4,T74,'*')
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                    DO 900 J=0,MGLL-1
                       WRITE(ICOUT,17) J,FTRACK(J)
17                     FORMAT(T3,'*',7X,'FTRACK(',I3,') = ',1PD11.3,
     $                        T74,'*')
                       CALL DPWRST('XXX','BUG')
900                CONTINUE
C
                 ELSE
C
C                   BOTH THE FUNCTION VALUE AND THE STEP SIZE
C                   ACCEPTANCE TESTS ARE BEING USED, ACPTCR=12.
C
                    WRITE(ICOUT,18) COUNTR
18                  FORMAT(T3,'*',4X,'CURRENT FTRACK AND STRACK',
     $                     ' VECTORS;',2X,'LATEST CHANGE:  ELEMENT',I4,
     $                     T74,'*')
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                    DO 1000 J=0,MGLL-1
                       WRITE(ICOUT,19) J,FTRACK(J),J,STRACK(J)
19                     FORMAT(T3,'*',7X,'FTRACK(',I3,') = ',
     $                        1PD11.3,'  STRACK(',I3,') = ',1PD11.3,
     $                        T74,'*')
                       CALL DPWRST('XXX','BUG')
1000                CONTINUE
                 END IF
              END IF
C
C             UPDATE COUNTING INTEGER, COUNTR. RECYCLE IF COUNTR
C             HAS REACHED MGLL-1.
C
              IF(COUNTR.EQ.MGLL-1) THEN
                 COUNTR=0
              ELSE
                 COUNTR=COUNTR+1
              END IF
           END IF
C
C          RESET STEP ACCEPTANCE CODE AND DELSTR.
C
           ACPCOD=0
         IF(.NOT.LINESR) DELSTR=ZERO
C
C          RESET QNFAIL TO FALSE TO AVOID PREMATURE STOPPING.
C
           QNFAIL=.FALSE.
C
           IF(LINESR) THEN
C
C             THE MAIN LINE SEARCH IS CALLED IN SUBROUTINE LINE.
C
            CALL LINE(ABORT ,ABSNEW,DEUFLH,GEOMS ,NEWTON,
     $                  OVERCH,OVERFL,QNFAIL,QRSING,RESTRT,
     $                  SCLFCH,SCLXCH,ACPCOD,ACPTCR,CONTYP,
     $                  ISEJAC,ITNUM ,JUPDM ,MAXEXP,MAXLIN,
     $                  MGLL  ,MNEW  ,N     ,NARMIJ,NFUNC ,
     $                  NUNIT ,OUTPUT,QNUPDM,STOPCR,TRMCOD,
     $                  ALPHA ,CONFAC,EPSMCH,FCNMAX,FCNNEW,
     $                  FCNOLD,LAM0  ,MAXSTP,NEWLEN,SBRNRM,
     $                  SIGMA ,A     ,BOUNDL,BOUNDU,DELF  ,
     $                  FTRACK,FVEC  ,H     ,HHPI  ,JAC   ,
     $                  RDIAG ,WV1   ,S     ,SBAR  ,SCALEF,
     $                  SCALEX,SN    ,STRACK,WV2   ,XC    ,
     $                  XPLUS ,FVECEV)
              IF(ABORT) THEN
                 IF(OUTPUT.GT.0) THEN
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,1)
                    CALL DPWRST('XXX','BUG')
                 END IF
                 RETURN
              END IF
C
C             NOTE: 201 PRECEDES PRINTING OF ITERATION RESULTS.
C
              IF(NEWTON) GO TO 201
C
           ELSE
C
C             TRUST REGION METHOD
C
C             ESTABLISH INITIAL TRUST REGION SIZE, DELTA, AND/OR
C             FIND LENGTH OF SCALED DESCENT STEP, CAULEN.
C
              CALL DELCAU(CAUCHY,OVERCH,OVERFL,ISEJAC,MAXEXP,
     $                    N     ,NUNIT ,OUTPUT,BETA  ,CAULEN,
     $                    DELTA ,EPSMCH,MAXSTP,NEWLEN,SQRTZ ,
     $                    A     ,DELF  ,SCALEX,WV1   )
C
              FRSTDG=.TRUE.
C
              IF(OUTPUT.GT.3) THEN
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,20)
20               FORMAT(T3,'*',4X,'SUMMARY OF TRUST REGION',
     $                  ' METHOD USING (DOUBLE) DOGLEG STEP',T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
C
C             MAIN INTERNAL LOOP FOR TRUST REGION METHOD.
C
C             THE TRUST REGION SIZE IS STORED FOR COMPARISON
C             LATER TO SET THE PARAMETER POWTAU USED IN POWELL'S
C             TRUST REGION UPDATING SCHEME (QUASI-NEWTON, TRUPDM=1).
C
              DELTA0=DELTA
C
              DO 1100 NOTRST=1,MAXTRS
C
                 CALL DOGLEX(FRSTDG,NEWTKN,OVERCH,OVERFL,MAXEXP,
     $                       N     ,NOTRST,NUNIT ,OUTPUT,BETA  ,
     $                       CAULEN,DELTA ,ETAFAC,NEWLEN,SQRTZ ,
     $                       DELF  ,S     ,SCALEX,SN    ,SSDHAT,
     $                       VHAT  )
C
C                NOTE: WV1 AND WV4 HOLD THE COMPONENT AND RESIDUAL
C                      VECTOR RESPECTIVELY FOR A TRIAL POINT WHICH
C                      HAS BEEN FOUND TO BE ACCEPTABLE WHILE THE
C                      TRUST REGION IS EXPANDED AND A NEW TRIAL
C                      POINT TESTED.  WV2 AND WV3 ARE WORK VECTORS.
C                      H IS CALLED ASTORE INSIDE TRSTUP.
C
          CALL TRSTUP(GEOMS ,NEWTKN,OVERCH,OVERFL,QRSING,
     $                       SCLFCH,SCLXCH,ACPCOD,ACPSTR,ACPTCR,
     $                       CONTYP,ISEJAC,JUPDM ,MAXEXP,MGLL  ,
     $                       MNEW  ,N     ,NARMIJ,NFUNC ,NOTRST,
     $                       NUNIT ,OUTPUT,QNUPDM,RETCOD,TRUPDM,
     $                       ALPHA ,CONFAC,DELFAC,DELSTR,DELTA ,
     $                       EPSMCH,FCNMAX,FCNNEW,FCNOLD,FCNPRE,
     $                       MAXSTP,NEWLEN,NEWMAX,POWTAU,RELLEN,
     $                       STPTOL,A     ,H     ,BOUNDL,BOUNDU,
     $                       DELF  ,WV1   ,FTRACK,FVEC  ,FVECC ,
     $                       HHPI  ,JAC   ,RDIAG ,WV2   ,S     ,
     $                       SBAR  ,SCALEF,SCALEX,STRACK,WV3   ,
     $                       XC    ,WV4   ,XPLUS ,FVECEV)
C
                 IF(OUTPUT.GT.4.OR.(RETCOD.EQ.7.AND.OUTPUT.GT.2))
     $           CALL RCDPRT(NUNIT,RETCOD,DELTA,RELLEN,STPTOL)
C
C                IF NO PROGRESS WAS BEING MADE (RETCOD=7) IN A
C                QUASI-NEWTON STEP RETRY WITH AN EXPLICIT JACOBIAN
C                EVALUATION.
C
                 IF(RETCOD.EQ.7.AND.(.NOT.RESTRT)) QNFAIL=.TRUE.
C
C                RETURN CODE LESS THAN 8 EXITS FROM TRUST REGION
C                LOOP.
C
                 IF(RETCOD.LT.8) GO TO 1101
C
1100          CONTINUE
C
C             IF NO SUCCESSFUL STEP FOUND IN A QUASI-NEWTON STEP
C             RETRY WITN AN EXPLICIT JACOBIAN EVALUATION.
C
              IF(.NOT.RESTRT) QNFAIL=.TRUE.
C
           END IF
C
1101       CONTINUE
C
           IF(.NOT.LINESR) THEN
            IF(DELTA.LT.DELTA0) POWTAU=ONE
              DELTA=MAX(DELTA,1.0D-10)
           END IF
C
C          IF RETCOD=7 AND STOPCR=2 RESET STOPPING CRITERION TO
C          AVOID HANGING IN TRUST REGION METHOD. (RETCOD=7 MEANS
C          THE RELATIVE STEP LENGTH WAS LESS THAN STPTOL).
C
           IF(.NOT.LINESR.AND.RETCOD.EQ.7.AND.STOPCR.EQ.2.AND.
     $        (.NOT.QNFAIL)) THEN
              STOPCR=12
           END IF
C
C          RETAIN NUMBER OF STEPS ACCEPTED BY EACH CRITERION
C          FOR PERFORMANCE EVALUATION.
C
           IF(.NOT.NEWTON) THEN
              IF(ACPCOD.EQ.1) THEN
                 NAC1=NAC1+1
              ELSEIF(ACPCOD.EQ.2) THEN
                 NAC2=NAC2+1
              ELSEIF(ACPCOD.EQ.12) THEN
                 NAC12=NAC12+1
              END IF
           END IF
C
201        CONTINUE
C
C          PRINT RESULTS OF ITERATION.
C
           IF(OUTPUT.GT.2) THEN
C
            CALL NERSL(NEWTON,RESTRT,SCLFCH,SCLXCH,ACPCOD,JUPDM ,
     $                   N     ,NUNIT ,OUTPUT,FCNNEW,FVEC  ,XPLUS )
C
           END IF
C
C          CHECK FOR CONVERGENCE.  STATEMENT 202 IS USED IF THE
C          STEP SIZE OF THE NEWTON STEP IS FOUND TO BE WITHIN
C          THE SPECIFIED TOLERANCE AND STOPCR IS 1 OR 12.
C
202        CONTINUE
C
           IF(.NOT.QNFAIL) THEN
C
C             IF QNFAIL IS TRUE THE QUASI-NEWTON SEARCH FAILED TO
C             FIND A SATISFACTORY STEP - SINCE THE JACOBIAN IS TO
C             BE RE-EVALUATED AVOID PREMATURE STOPPAGES IN NESTOP.
C
            CALL NESTOP(ABSNEW,LINESR,NEWTON,SCLFCH,SCLXCH,
     $                    ACPTCR,ITNUM ,N     ,NAC1  ,NAC2  ,
     $                    NAC12 ,NFUNC ,NJETOT,NUNIT ,OUTPUT,
     $                    STOPCR,TRMCOD,FCNNEW,FTOL  ,NSTTOL,
     $                    STPMAX,STPTOL,FVEC  ,SCALEF,SCALEX,
     $                    XC    ,XPLUS )
           END IF
C
C          IF THE TERMINATION CODE, TRMCOD, IS GREATER THAN 0 THEN
C          CONVERGANCE HAS BEEN REACHED.
C
           IF(TRMCOD.GT.0) RETURN
C
C          QUASI-NEWTON UPDATING - JUPDM > 0.
C
           IF(JUPDM.GT.0) THEN
C
C             QNFAIL MEANS A FAILURE IN THE QUASI-NEWTON SEARCH.
C             RE-EVALUATE JACOBIAN AND TRY A DAMPED NEWTON STEP.
C             MAXLIN IS CHANGED FROM MAXQNS TO MAXNS OR MAXTRS IS
C             CHANGED, SIMILARLY, AT THE START OF THE LOOP.
C
              IF(QNFAIL) THEN
                 IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,2)
                    WRITE(ICOUT,21)
21                  FORMAT(T3,'*',7X,'FAILURE IN QUASI-NEWTON',
     $              ' SEARCH: QNFAIL IS TRUE',T74,'*')
                 END IF
C
              ELSEIF(.NOT.NEWTON) THEN
C
C                QNFAIL IS FALSE - THE STEP HAS BEEN ACCEPTED.
C
          IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,22) FCNNEW,FCNOLD,FCNNEW/FCNOLD
22                  FORMAT(T3,'*',7X,'FCNNEW= ',1PD11.3,2X,
     $              'FCNOLD= ',1PD11.3,2X,'RATIO= ',1PD11.3,T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 IF(FCNNEW/FCNOLD.GT.RATIOF) THEN
C
C                   STEP ACCEPTED BUT NOT A SIGNIFICANT IMPROVEMENT.
C
                    NFAIL=NFAIL+1
                    IF(OUTPUT.GT.4) THEN
                       WRITE(ICOUT,2)
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,23) NFAIL
23                     FORMAT(T3,'*',7X,'RATIO > RATIOF SO',
     $                 ' NFAIL INCREASED TO: ',I5,T74,'*')
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,2)
                       CALL DPWRST('XXX','BUG')
                    END IF
                 ELSE
C
C                   STEP ACCEPTED WITH A SIGNIFICANT IMPROVEMENT.
C
             IF(FCNNEW/FCNOLD.GT.PT01) THEN
                       NOSUBT=0
                    ELSE
                       NOSUBT=1
                    END IF
C
C                   ITEMPT IS USED LOCALLY FOR OUTPUT CONTROL.
C
                    ITEMP=NFAIL
                    NFAIL=MAX(NFAIL-NOSUBT,0)
                    IF(OUTPUT.GT.4) THEN
                       WRITE(ICOUT,2)
                       CALL DPWRST('XXX','BUG')
                       IF(ITEMP.EQ.NFAIL) THEN
                          WRITE(ICOUT,24) NFAIL
24                        FORMAT(T3,'*',7X,'NFAIL STAYS AT: ',I5,
     $                           T74,'*')
                          CALL DPWRST('XXX','BUG')
                       ELSE
                          WRITE(ICOUT,25) NFAIL
25                        FORMAT(T3,'*',7X,'NFAIL CHANGED TO: ',
     $                           I5,T74,'*')
                          CALL DPWRST('XXX','BUG')
                       END IF
                    END IF
                 END IF
C
C                SAVE THE RESULTS FOR RESTART IF A FAILURE IN THE
C                QUASI-NEWTON METHOD OCCURS - ESSENTIALLY THIS
C                FINDS THE BEST POINT SO FAR.
C
                 IF(ISEJAC.EQ.1.OR.NFAIL.EQ.1.OR.(NFAIL.LE.MINQNS
     $              .AND.FCNNEW/FCNMIN.LT.ONE)) THEN
                    SAVEST=.TRUE.
                    FCNMIN=FCNNEW
                 END IF
                 IF(SAVEST) THEN
                    SAVEST=.FALSE.
                    FSTORE=FCNNEW
                    IF(OUTPUT.GT.4) THEN
                       WRITE(ICOUT,2)
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,26)
26                     FORMAT(T3,'*',7X,'STEP IS SAVED',T74,'*')
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,2)
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,27)
27                     FORMAT(T3,'*',10X,'SAVED COMPONENT AND',
     $                 ' FUNCTION VALUES',T74,'*')
                       CALL DPWRST('XXX','BUG')
                       WRITE(ICOUT,2)
                       CALL DPWRST('XXX','BUG')
                    END IF
                    DO 1200 I=1,N
                       XSAVE(I)=XPLUS(I)
                       FSAVE(I)=FVEC(I)
                       IF(OUTPUT.GT.4) THEN
                          WRITE(ICOUT,28) I,XSAVE(I),I,FSAVE(I)
28                        FORMAT(T3,'*',7X,'XSAVE(',I3,') = ',
     $                    1PD12.3,6X,'FSAVE(',I3,') = ',1PD12.3,
     $                    T74,'*')
                          CALL DPWRST('XXX','BUG')
                       END IF
1200                CONTINUE
                    IF(OUTPUT.GT.4) WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                    ITSTR=ITNUM
                 END IF
              END IF
C
C             NOTE: IF QNFAIL IS TRUE THEN NFAIL CANNOT HAVE
C                   INCREASED IMPLYING THAT NFAIL CANNOT NOW
C                   BE GREATER THAN MINQNS.
C
              IF(QNFAIL.OR.NFAIL.GT.MINQNS) THEN
C
C                RESTART FROM BEST POINT FOUND SO FAR.
C
                 RESTRT=.TRUE.
                 IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,29)
29                  FORMAT(T3,'*',7X,'RESTRT IS TRUE',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 DO 1300 J=0,MGLL-1
             FTRACK(J)=ZERO
1300             CONTINUE
                 IF(ACPTCR.EQ.12) THEN
                    DO 1400 J=0,MGLL-1
                STRACK(J)=ZERO
1400                CONTINUE
                 END IF
                 COUNTR=0
                 ISEJAC=0
                 MNEW=0
                 TRMCOD=0
                 FCNOLD=FSTORE
                 IF(OUTPUT.GT.4) THEN
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,30) ITSTR
30                  FORMAT(T3,'*',7X,'RETURN TO ITERATION:',
     $                     I5,3X,'WHERE',T74,'*')
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,2)
                    CALL DPWRST('XXX','BUG')
                 END IF
                 DO 1500 I=1,N
                    XC(I)=XSAVE(I)
                    FVECC(I)=FSAVE(I)
                    IF(OUTPUT.GT.4) THEN
                       WRITE(ICOUT,31) I,XC(I),I,FVECC(I)
31                     FORMAT(T3,'*',7X,'XC(',I3,') = ',
     $                        1PD12.3,3X,'FVECC(',I3,') = ',1PD12.3,
     $                        T74,'*')
                       CALL DPWRST('XXX','BUG')
                    END IF
1500             CONTINUE
                 IF(OUTPUT.GT.4) WRITE(ICOUT,2)
                 CALL DPWRST('XXX','BUG')
              ELSE
          IF(ITNUM.GE.NIEJEV) RESTRT=.FALSE.
              END IF
           END IF
C
C          UPDATE JACOBIAN IF DESIRED:
C                QNUPDM = 0   => ACTUAL JACOBIAN BEING UPDATED
C                QNUPDM = 1   => FACTORED JACOBIAN BEING UPDATED
C
           IF(.NOT.RESTRT) THEN
C
              IF(QNUPDM.EQ.0) THEN
C
                 IF(JUPDM.EQ.1) THEN
C
C                   USE BROYDEN UPDATE.
C
                    CALL BROYUN(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,
     $                          EPSMCH,FVEC  ,FVECC ,JAC   ,SCALEX,
     $                          WV1   ,XC    ,XPLUS)
C
                 ELSEIF(JUPDM.EQ.2) THEN
C
C                   USE LEE AND LEE UPDATE.
C
                    CALL LLUN(OVERCH,OVERFL,ISEJAC,MAXEXP,N     ,
     $                        NUNIT ,OUTPUT,EPSMCH,OMEGA ,FVEC  ,
     $                        FVECC ,JAC   ,PLEE  ,S     ,SCALEX,
     $                        WV1   ,XC    ,XPLUS)
C
                 END IF
C
              ELSE
C
C                THE FACTORED FORM OF THE JACOBIAN IS UPDATED.
C
                 IF(JUPDM.EQ.1) THEN
C
             CALL BROYFA(OVERCH,OVERFL,SCLFCH,SCLXCH,MAXEXP,
     $                          N     ,NUNIT ,OUTPUT,EPSMCH,A     ,
     $                          DELF  ,FVEC  ,FVECC ,JAC   ,RDIAG ,
     $                          S     ,SCALEF,SCALEX,WV1   ,WV2   ,
     $                          XC    ,XPLUS )
C
                 ELSEIF(JUPDM.EQ.2) THEN
C
             CALL LLFA(OVERCH,OVERFL,SCLFCH,SCLXCH,ISEJAC,
     $                        MAXEXP,N     ,NUNIT ,OUTPUT,EPSMCH,
     $                        OMEGA ,A     ,DELF  ,FVEC  ,FVECC ,
     $                        JAC   ,PLEE  ,RDIAG ,S     ,SCALEF,
     $                        SCALEX,WV1   ,WV2   ,WV3   ,XC    ,
     $                        XPLUS )
C
                 END IF
C
              END IF
C
           END IF
C
C          UPDATE CURRENT VALUES - RESET TRMCOD TO ZERO.
C
C          UPDATE M "VECTOR" (ACTUALLY ONLY THE LATEST VALUE IS
C          NEEDED).
C
           MOLD=MNEW
           IF(ISEJAC.LT.NARMIJ) THEN
              MNEW=0
           ELSE
              MNEW=MIN(MOLD+1,MGLL-1)
           END IF
           IF(JUPDM.EQ.0.OR.(JUPDM.GT.0.AND.(.NOT.RESTRT))) THEN
C
              CALL UPDATE(MNEW  ,MOLD  ,N     ,TRMCOD,FCNNEW,
     $                    FCNOLD,FVEC  ,FVECC ,XC    ,XPLUS )
           END IF
C
200     CONTINUE
203     CONTINUE
C
        IF(OUTPUT.GT.0) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,32) ITNUM-1
32         FORMAT(T3,'*',2X,'NO SOLUTION FOUND AFTER',I6,
     $            ' ITERATION(S)',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,33)
33         FORMAT(T3,'*',9X,'FINAL ESTIMATES',15X,
     $            'FINAL FUNCTION VALUES',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           DO 1600 I=1,N
              WRITE(ICOUT,34) I,XPLUS(I),I,FVEC(I)
34            FORMAT(T3,'*',6X,'X(',I3,') = ',1PD12.3,12X,
     $               'F(',I3,') = ',1PD12.3,T74,'*')
              CALL DPWRST('XXX','BUG')
1600       CONTINUE
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,36) FCNNEW
36         FORMAT(T3,'*',2X,'FINAL OBJECTIVE FUNCTION VALUE = ',
     $            1PD12.3,T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE NNES.
C
      END
        SUBROUTINE NSTPFA(ABORT ,LINESR,OVERCH,OVERFL,QRSING,
     $                    RESTRT,SCLFCH,SCLXCH,ITNUM ,MAXEXP,
     $                    N     ,NEWSTM,NUNIT ,OUTPUT,EPSMCH,
     $                    A     ,DELF  ,FVECC ,H     ,HHPI  ,
     $                    JAC   ,RDIAG ,SCALEF,SCALEX,SN    ,
     $                    WV1   ,WV2   ,WV3)
C
C       FEB. 23, 1992
C
C       THIS SUBROUTINE FINDS THE NEWTON STEP.
C
C       IF THE JACOBIAN IS DETECTED AS SINGULAR OR IF THE ESTIMATED
C       CONDITION NUMBER IS TOO HIGH (GREATER THAN EPSMCH**(-2/3))
C       THEN H:=J^J IS FORMED AND THE DIAGONAL IS PERTURBED BY ADDING
C       SQRT(N*EPSMCH)*H1NORM*SCALEX(I)**2 TO THE CORRESPONDING
C       ELEMENT.  A CHOLESKY DECOMPOSITION IS PERFORMED ON THIS
C       MODIFIED MATRIX PRODUCING A PSEUDO-NEWTON STEP.
C
C       IF THE CONDITION NUMBER IS SMALL THEN THE NEWTON STEP, SN,
C       IS FOUND DIRECTLY BY BACK SUBSTITUTION.
C
C       ABORT    IF THE 1-NORM OF MATRIX H BECOMES TOO SMALL
C                ALTHOUGH BUT NOT AT A SOLUTION
C       BYPASS   ALLOWS BYPASSING OF THE SPECIAL TREATMENT FOR
C                BADLY CONDITIONED JACOBIANS
C       QRSING   INDICATES SINGULAR JACOBIAN DETECTED
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION MAXADD   ,MAXFFL   ,JAC(N,N)
        INTEGER          OUTPUT
        DIMENSION        A(N,N)   ,DELF(N)  ,FVECC(N) ,H(N,N)   ,
     $                   HHPI(N)  ,RDIAG(N) ,SCALEF(N),SCALEX(N),
     $                   SN(N)    ,WV1(N)   ,WV2(N)   ,WV3(N)
      LOGICAL          ABORT    ,BYPASS   ,LINESR   ,MATSUP   ,
     $                   OVERCH   ,OVERFL   ,PERTRB   ,QRSING   ,
     $                   RESTRT   ,SCLFCH   ,SCLXCH   ,WRNSUP
      COMMON/NNES_1/MATSUP
        COMMON/NNES_2/WRNSUP
        COMMON/NNES_3/BYPASS
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,ONE /0.0D0,1.0D0/
C
        ABORT=.FALSE.
        OVERFL=.FALSE.
        SQRTEP=SQRT(EPSMCH)
C
        IF(RESTRT) THEN
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
1             FORMAT(T3,'*',T74,'*')
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,2)
2             FORMAT(T3,'*',4X,'SOLUTION OF LINEAR SYSTEM FOR',
     $               ' NEWTON STEP, SN',T74,'*')
              CALL DPWRST('XXX','BUG')
           END IF
C
C          STORE (POSSIBLY SCALED) JACOBIAN IN MATRIX A.
C
         IF(.NOT.SCLFCH) THEN
            CALL MATCOP(N,N,N,N,N,N,JAC,A)
         ELSE
            DO 100 I=1,N
          IF(SCALEF(I).NE.ONE) THEN
             SCALFI=SCALEF(I)
             DO 200 J=1,N
                A(I,J)=JAC(I,J)*SCALFI
200                 CONTINUE
          ELSE
             DO 300 J=1,N
                A(I,J)=JAC(I,J)
300                 CONTINUE
          END IF
100           CONTINUE
         END IF
C
C          SCALED JACOBIAN IS PRINTED ONLY IF AT LEAST ONE SCALING
C          FACTOR IS NOT 1.0.
C
           IF(OUTPUT.GT.4.AND.SCLFCH) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,3)
3             FORMAT(T3,'*',7X,'SCALED JACOBIAN MATRIX',T74,'*')
              CALL DPWRST('XXX','BUG')
              CALL MATPRT(N,N,N,N,NUNIT,A)
           END IF
C
C          QR DECOMPOSITION OF (POSSIBLY SCALED) JACOBIAN.
C
           CALL QRDCOM(QRSING,N,EPSMCH,A,HHPI,RDIAG)
C
         IF(OUTPUT.GT.4.AND.N.GT.1.AND.(.NOT.MATSUP)) THEN
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            IF(.NOT.SCLFCH) THEN
               WRITE(ICOUT,4)
4              FORMAT(T3,'*',7X,'QR DECOMPOSITION OF JACOBIAN',
     $                ' MATRIX',T74,'*')
               CALL DPWRST('XXX','BUG')
            ELSE
               WRITE(ICOUT,5)
5              FORMAT(T3,'*',7X,'QR DECOMPOSITION OF SCALED',
     $                ' JACOBIAN MATRIX',T74,'*')
               CALL DPWRST('XXX','BUG')
            END IF
            CALL MATPRT(N,N,N,N,NUNIT,A)
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,6)
6           FORMAT(T3,'*',12X,'DIAGONAL OF R',10X,'PI FACTORS',
     $             ' FROM QR DECOMPOSITION',T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            DO 400 I=1,N-1
               WRITE(ICOUT,7) I,RDIAG(I),I,HHPI(I)
7              FORMAT(T3,'*',7X,'RDIAG(',I3,') = ',1PD11.3,
     $                8X,'HHPI(',I3,') = ',1PD12.3,T74,'*')
               CALL DPWRST('XXX','BUG')
400         CONTINUE
            WRITE(ICOUT,8) N,RDIAG(N)
8           FORMAT(T3,'*',7X,'RDIAG(',I3,') = ',1PD11.3,
     $             T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            IF(ITNUM.EQ.1) THEN
               WRITE(ICOUT,9)
9              FORMAT(T3,'*',7X,'NOTE: R IS IN STRICT UPPER',
     $                ' TRIANGLE OF MATRIX A PLUS RDIAG',T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,10)
10             FORMAT(T3,'*',13X,'THE COLUMNS OF THE LOWER',
     $                ' TRIANGLE OF MATRIX A PLUS',T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,11)
11             FORMAT(T3,'*',13X,'THE ELEMENTS OF VECTOR HHPI',
     $                ' FORM THE HOUSEHOLDER',T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,12)
12             FORMAT(T3,'*',13X,'MATRICES WHICH, WHEN',
     $                ' MULTIPLIED TOGETHER, FORM Q',T74,'*')
               CALL DPWRST('XXX','BUG')
              END IF
           END IF
C
C          FORM THE ACTUAL Q^ MATRIX FROM THE HOUSEHOLDER
C          TRANSFORMATIONS STORED IN THE LOWER TRIANGLE OF A
C          AND THE FACTORS IN HHPI: STORE IT IN JAC.
C
           CALL QFORMZ(N,A,HHPI,JAC)
C
C          COMPLETE THE UPPER TRIANGULAR R MATRIX BY REPLACING THE
C          DIAGONAL OF A.  THE QR DECOMPOSITION IS NOW AVAILABLE.
C
         DO 500 I=1,N
              A(I,I)=RDIAG(I)
500        CONTINUE
C
        ELSE
C
C          USING UPDATED FACTORED FORM OF JACOBIAN - CHECK FOR
C          SINGULARITY.
C
           QRSING=.FALSE.
         DO 600 I=1,N
              IF(A(I,I).EQ.ZERO) QRSING=.TRUE.
600        CONTINUE
C
        END IF
C
C       ESTIMATE CONDITION NUMBER IF JACOBIAN IS NOT SINGULAR.
C
      IF(.NOT.BYPASS.AND.(.NOT.QRSING).AND.N.GT.1) THEN
         IF(SCLXCH) THEN
C
C             SET UP FOR CONDITION NUMBER ESTIMATION - SCALE R WRT X'S.
C
            DO 700 J=1,N
          IF(SCALEX(J).NE.ONE) THEN
             SCALXJ=SCALEX(J)
             RDIAG(J)=RDIAG(J)/SCALXJ
             DO 800 I=1,J-1
                A(I,J)=A(I,J)/SCALXJ
800                 CONTINUE
          END IF
700           CONTINUE
         END IF
C
         CALL CONDNO(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,
     $                 CONNUM,A     ,WV1   ,WV2   ,WV3   ,RDIAG )
C
C          UNSCALE R IF IT WAS SCALED BEFORE THE CALL TO CONDNO.
C
         IF(SCLXCH) THEN
            DO 900 J=1,N
          IF(SCALEX(J).NE.ONE) THEN
             SCALXJ=SCALEX(J)
             RDIAG(J)=RDIAG(J)*SCALXJ
             DO 1000 I=1,J-1
                A(I,J)=A(I,J)*SCALXJ
1000                CONTINUE
          END IF
900           CONTINUE
         END IF
C
C          IF OVERFLOW DETECTED IN CONDITION NUMBER ESTIMATOR ASSIGN
C          QRSING AS TRUE SO THAT THE JACOBIAN WILL BE PERTURBED.
C
           IF(OVERFL) QRSING=.TRUE.
C
C          NOTE: OVERFL SWITCHED TO FALSE BEFORE FORMATION OF H.
C
        ELSE
           IF(N.EQ.1) THEN
              CONNUM=ONE
           ELSE
              CONNUM=ZERO
           END IF
      END IF
C
C       MATRIX H=JAC^JAC IS FORMED IN TWO CASES:
C          1)  THE (SCALED) JACOBIAN IS SINGULAR
C          2)  THE CONDITION NUMBER IS TOO HIGH AND THE
C              OPTION TO BYPASS THE PERTURBATION OF THE
C              JACOBIAN IS NOT BEING USED
C          3)  REQUESTED BY THE USER THROUGH NEWSTM.
C
        IF(QRSING.OR.((.NOT.BYPASS).AND.CONNUM.GT.
     $     ONE/SQRTEP**1.333).OR.NEWSTM.EQ.77) THEN
C
C          FORM H:=(DF*JAC)^(DF*JAC) WHERE DF=DIAG(SCALEF).  USE
C          PREVIOUSLY COMPUTED QR DECOMPOSITION OF (SCALED) JACOBIAN
C          WHERE R IS STORED IN THE UPPER TRIANGLE OF A AND RDIAG.
C
         OVERFL=.FALSE.
         IF(OVERCH) THEN
            CALL ATAOV(OVERFL,MAXEXP,N,NUNIT,OUTPUT,JAC,H,SCALEF)
         ELSE
            CALL RTRMUL(N,A,H,RDIAG,WV1)
         END IF
         IF(OUTPUT.GT.3) THEN
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            IF(QRSING.AND.(.NOT.OVERFL)) THEN
               WRITE(ICOUT,13)
13             FORMAT(T3,'*',7X,'SINGULAR JACOBIAN DETECTED:',
     $                ' JACOBIAN PERTURBED',T74,'*')
               CALL DPWRST('XXX','BUG')
            ELSE
               IF(OVERFL) THEN
                  WRITE(ICOUT,14)
14                FORMAT(T3,'*',7X,'POTENTIAL OVERFLOW DETECTED',
     $                   ' IN CONDITION NUMBER ESTIMATOR',T74,'*')
                  CALL DPWRST('XXX','BUG')
                  WRITE(ICOUT,15)
15                FORMAT(T3,'*',7X,'MATRIX "ASSIGNED" AS ',
     $                   'SINGULAR AND JACOBIAN PERTURBED',T74,'*')
                  CALL DPWRST('XXX','BUG')
               ELSE
                  WRITE(ICOUT,16) CONNUM
16                FORMAT(T3,'*',7X,'CONDITION NUMBER TOO HIGH: ',
     $                   1PD12.3,', JACOBIAN PERTURBED',T74,'*')
                  CALL DPWRST('XXX','BUG')
               END IF
              END IF
         END IF
         OVERFL=.FALSE.
         IF(NEWSTM.NE.77) THEN
C
C             FIND 1-NORM OF H MATRIX AND PERTURB DIAGONAL.
C
            CALL ONENRM(ABORT ,PERTRB,N     ,NUNIT ,OUTPUT,EPSMCH,
     $                    H1NORM,H     ,SCALEX)
         END IF
C
C          CHOLESKY DECOMPOSITION OF H MATRIX - MAXFFL=0 INDICATES
C          THAT H IS KNOWNTO BE POSITIVE DEFINITE.
C
           MAXFFL=ZERO
           CALL CHOLDE(N,MAXADD,MAXFFL,SQRTEP,H,A)
           IF(OUTPUT.GT.4) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,17)
17            FORMAT(T3,'*',5X,'CHOLESKY DECOMPOSITION OF H MATRIX',
     $               T74,'*')
              CALL DPWRST('XXX','BUG')
              CALL MATPRT(N,N,N,N,NUNIT,A)
           END IF
C
C          FIND NEWTON STEP FROM CHOLESKY DECOMPOSITION.  IF THE
C          DIAGONAL HAS BEEN PERTURBED THEN THIS IS NOT THE ACTUAL
C          NEWTON STEP BUT ONLY AN APPROXIMATION THEREOF.
C
         DO 1100 I=1,N
              WV1(I)=-DELF(I)
1100       CONTINUE
           CALL CHSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                 OUTPUT,A     ,WV1   ,SN    ,WV2)
         OVERFL=.FALSE.
         IF(OUTPUT.GT.3) THEN
            IF(.NOT.SCLXCH) THEN
               IF(.NOT.PERTRB) THEN
                  WRITE(ICOUT,18)
18                FORMAT(T3,'*',5X,'NEWTON STEP FROM CHOLESKY',
     $                   ' DECOMPOSITION',T74,'*')
                  CALL DPWRST('XXX','BUG')
               ELSE
                  WRITE(ICOUT,19)
19                FORMAT(T3,'*',5X,'APPROXIMATE NEWTON STEP FROM',
     $                   ' PERTURBED JACOBIAN',T74,'*')
                  CALL DPWRST('XXX','BUG')
               END IF
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               DO 1200 I=1,N
                  WRITE(ICOUT,20) I,SN(I)
20                FORMAT(T3,'*',7X,'SN(',I3,') = ',1PD12.3,T74,'*')
                  CALL DPWRST('XXX','BUG')
1200           CONTINUE
            ELSE
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               IF(.NOT.PERTRB) THEN
                  WRITE(ICOUT,21)
21                FORMAT(T3,'*',5X,'NEWTON STEP FROM CHOLESKY',
     $            ' DECOMPOSITION',3X,' IN SCALED UNITS',T74,'*')
                  CALL DPWRST('XXX','BUG')
               ELSE
                  WRITE(ICOUT,22)
22                FORMAT(T3,'*',5X,'APPROXIMATE NEWTON STEP FROM',
     $            ' PERTURBED JACOBIAN',3X,'IN SCALED UNITS',T74,'*')
                  CALL DPWRST('XXX','BUG')
               END IF
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               DO 1300 I=1,N
                  WRITE(ICOUT,23) I,SN(I),I,SCALEX(I)*SN(I)
23                FORMAT(T3,'*',7X,'SN(',I3,') = ',1PD12.3,
     $                   15X,'SN(',I3,') = ',1PD12.3,T74,'*')
                  CALL DPWRST('XXX','BUG')
1300           CONTINUE
            END IF
         END IF
C
C          SET QRSING TO TRUE SO THAT THE CORRECT MATRIX FACTORIZATION
C          IS USED IN THE BACK-CALCULATION OF SBAR FOR DEUFLHARD
C          RELAXATION FACTOR INITIALIZATION.
C
         QRSING=.TRUE.
      ELSE
         IF(OUTPUT.GT.3.AND.N.GT.1) THEN
            IF(.NOT.BYPASS.AND.CONNUM.LE.ONE/SQRTEP**1.33) THEN
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,24) CONNUM
24             FORMAT(T3,'*',7X,'CONDITION NUMBER ACCEPTABLE, ',
     $                1PD9.2,', JACOBIAN NOT PERTURBED',T74,'*')
               CALL DPWRST('XXX','BUG')
            END IF
            IF(BYPASS.AND.CONNUM.GT.ONE/SQRTEP**1.33) THEN
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,25) CONNUM
25             FORMAT(T3,'*',7X,'CONDITION NUMBER HIGH, ',
     $                1PD9.2,', JACOBIAN NOT PERTURBED AS',T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,26)
26             FORMAT(T3,'*',7X,'BYPASS IS TRUE',T74,'*')
               CALL DPWRST('XXX','BUG')
            END IF
         END IF
         DO 1400 I=1,N
            SUM=ZERO
            DO 1500 J=1,N
               SUM=SUM-JAC(I,J)*SCALEF(J)*FVECC(J)
1500        CONTINUE
            SN(I)=SUM
1400     CONTINUE
         CALL RSOLV(OVERCH,OVERFL,MAXEXP,N,NUNIT,OUTPUT,A,RDIAG,SN)
         OVERFL=.FALSE.
         IF(OUTPUT.GT.3) THEN
            IF(.NOT.SCLXCH) THEN
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,27)
27             FORMAT(T3,'*',7X,'NEWTON STEP FROM QR DECOMPOSITION '
     $                ,T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               DO 1600 I=1,N
                  WRITE(ICOUT,20) I,SN(I)
                  CALL DPWRST('XXX','BUG')
1600           CONTINUE
            ELSE
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,28)
28             FORMAT(T3,'*',7X,'NEWTON STEP FROM QR DECOMPOSITION '
     $                ,7X,'IN SCALED UNITS',T74,'*')
               CALL DPWRST('XXX','BUG')
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               DO 1700 I=1,N
                  WRITE(ICOUT,23) I,SN(I),I,SCALEX(I)*SN(I)
                  CALL DPWRST('XXX','BUG')
1700           CONTINUE
            END IF
           END IF
C
C          TRANSFORM MATRICES FOR SUBSEQUENT CALCULATIONS IN TRUST
C          REGION METHOD.
C
           IF(.NOT.LINESR) THEN
              DO 1800 I=2,N
                 DO 1900 J=1,I-1
                    A(I,J)=A(J,I)
1900             CONTINUE
1800          CONTINUE
           END IF
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE NSTPFA.
C
      END

      SUBROUTINE NSTPUN(ABORT ,LINESR,OVERCH,OVERFL,QRSING,
     $                    SCLFCH,SCLXCH,ITNUM ,MAXEXP,N     ,
     $                    NUNIT ,OUTPUT,EPSMCH,A     ,DELF  ,
     $                    FVECC ,H     ,HHPI  ,JAC   ,RDIAG ,
     $                    SCALEF,SCALEX,SN    ,WV1   ,WV2   ,
     $                    WV3   )
C
C       FEB. 23, 1992
C
C       THIS SUBROUTINE FINDS THE NEWTON STEP.
C
C       IF THE JACOBIAN IS DETECTED AS SINGULAR OR IF THE ESTIMATED
C       CONDITION NUMBER IS TOO HIGH (GREATER THAN EPSMCH**(-2/3))
C       THEN H:=J^J IS FORMED AND THE DIAGONAL IS PERTURBED BY ADDING
C       SQRT(N*EPSMCH)*H1NORM*SCALEX(I)**2 TO THE CORRESPONDING
C       ELEMENT.  A CHOLESKY DECOMPOSITION IS PERFORMED ON THIS
C       MODIFIED MATRIX PRODUCING A PSEUDO-NEWTON STEP.
C       NOTE: THIS PROCEDURE MAY BE BE BYPASSED FOR ILL-CONDITIONED
C       JACOBIANS BY SETTING THE LOGICAL VARIABLE BYPASS TO TRUE
C       IN THE DRIVER.
C
C       ABORT    IF THE 1-NORM OF MATRIX H BECOMES TOO SMALL
C                ALTHOUGH BUT NOT AT A SOLUTION
C       BYPASS   ALLOWS BYPASSING OF THE SPECIAL TREATMENT FOR
C                BADLY CONDITIONED JACOBIANS
C       QRSING   INDICATES SINGULAR JACOBIAN DETECTED
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N) ,MAXADD   ,MAXFFL
        INTEGER          OUTPUT
        DIMENSION        A(N,N)   ,DELF(N)  ,FVECC(N)  ,H(N,N)   ,
     $                   HHPI(N)  ,RDIAG(N) ,SCALEF(N) ,SCALEX(N),
     $                   SN(N)    ,WV1(N)   ,WV2(N)    ,WV3(N)
      LOGICAL          ABORT    ,BYPASS   ,LINESR    ,MATSUP   ,
     $                   OVERCH   ,OVERFL   ,PERTRB    ,QRSING   ,
     $                   SCLFCH   ,SCLXCH   ,WRNSUP
      COMMON/NNES_1/MATSUP
      COMMON/NNES_2/WRNSUP
        COMMON/NNES_3/BYPASS
      COMMON/ILL/NEWSTM
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO,ONE /0.0D0,1.0D0/
C
      ABORT=.FALSE.
      OVERFL=.FALSE.
      PERTRB=.FALSE.
      SQRTEP=SQRT(EPSMCH)
C
      IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
1          FORMAT(T3,'*',T74,'*')
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,2)
2          FORMAT(T3,'*',4X,'SOLUTION OF LINEAR SYSTEM FOR',
     $            ' NEWTON STEP, SN',T74,'*')
           CALL DPWRST('XXX','BUG')
      END IF
C
C       STORE (POSSIBLY SCALED) JACOBIAN IN MATRIX A.
C
      IF(.NOT.SCLFCH) THEN
         CALL MATCOP(N,N,N,N,N,N,JAC,A)
      ELSE
         DO 100 I=1,N
            IF(SCALEF(I).NE.ONE) THEN
          SCALFI=SCALEF(I)
          DO 200 J=1,N
             A(I,J)=JAC(I,J)*SCALFI
200              CONTINUE
            ELSE
          DO 300 J=1,N
             A(I,J)=JAC(I,J)
300              CONTINUE
            END IF
100        CONTINUE
      END IF
C
C       SCALED JACOBIAN IS PRINTED ONLY IF AT LEAST ONE SCALING
C       FACTOR IS NOT 1.0.
C
        IF(OUTPUT.GT.4.AND.SCLFCH) THEN
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,1)
           CALL DPWRST('XXX','BUG')
           WRITE(ICOUT,3)
           CALL DPWRST('XXX','BUG')
3          FORMAT(T3,'*',7X,'SCALED JACOBIAN MATRIX',T74,'*')
           CALL DPWRST('XXX','BUG')
           CALL MATPRT(N,N,N,N,NUNIT,A)
        END IF
C
C       QR DECOMPOSITION OF (POSSIBLY SCALED) JACOBIAN.
C
      CALL QRDCOM(QRSING,N,EPSMCH,A,HHPI,RDIAG)
C
C       SAVE MATRIX A FOR BACK SUBSTITUTION TO CHECK DEUFLHARDS'S
C       SECOND STEP ACCEPTANCE CRITERION IN LINE SEARCH OR TRUST
C       REGION METHOD.
C
      CALL MATCOP(N,N,N,N,N,N,A,H)
C
      IF(OUTPUT.GT.4.AND.N.GT.1.AND.(.NOT.MATSUP)) THEN
         WRITE(ICOUT,1)
         CALL DPWRST('XXX','BUG')
         IF(.NOT.SCLFCH) THEN
            WRITE(ICOUT,4)
4           FORMAT(T3,'*',7X,'QR DECOMPOSITION OF JACOBIAN',
     $             ' MATRIX',T74,'*')
            CALL DPWRST('XXX','BUG')
         ELSE
            WRITE(ICOUT,5)
5           FORMAT(T3,'*',7X,'QR DECOMPOSITION OF SCALED JACOBIAN',
     $             ' MATRIX',T74,'*')
            CALL DPWRST('XXX','BUG')
         END IF
         CALL MATPRT(N,N,N,N,NUNIT,A)
         WRITE(ICOUT,1)
         CALL DPWRST('XXX','BUG')
         WRITE(ICOUT,6)
6        FORMAT(T3,'*',12X,'DIAGONAL OF R',10X,'PI FACTORS',
     $          ' FROM QR DECOMPOSITION',T74,'*')
         CALL DPWRST('XXX','BUG')
         WRITE(ICOUT,1)
         CALL DPWRST('XXX','BUG')
         DO 400 I=1,N-1
            WRITE(ICOUT,7) I,RDIAG(I),I,HHPI(I)
7           FORMAT(T3,'*',7X,'RDIAG(',I3,') = ',1PD11.3,
     $             8X,'HHPI(',I3,') = ',1PD12.3,T74,'*')
            CALL DPWRST('XXX','BUG')
400      CONTINUE
         WRITE(ICOUT,8) N,RDIAG(N)
8        FORMAT(T3,'*',7X,'RDIAG(',I3,') = ',1PD11.3,T74,'*')
         CALL DPWRST('XXX','BUG')
         WRITE(ICOUT,1)
         CALL DPWRST('XXX','BUG')
         IF(ITNUM.EQ.1) THEN
            WRITE(ICOUT,9)
9           FORMAT(T3,'*',7X,'NOTE: R IS IN STRICT UPPER TRIANGLE',
     $             ' OF MATRIX A PLUS RDIAG',T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1)
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,10)
10          FORMAT(T3,'*',13X,'THE COLUMNS OF THE LOWER TRIANGLE',
     $        ' OF MATRIX A PLUS',T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,1010)
1010        FORMAT(T3,'*',13X,'THE ELEMENTS',
     $             ' OF VECTOR HHPI FORM THE HOUSEHOLDER',T74,'*')
            CALL DPWRST('XXX','BUG')
            WRITE(ICOUT,11)
11          FORMAT(T3,'*',13X,'MATRICES WHICH, WHEN MULTIPLIED',
     $             ' TOGETHER, FORM Q',T74,'*')
            CALL DPWRST('XXX','BUG')
         END IF
        END IF
C
C       ESTIMATE CONDITION NUMBER IF (SCALED) JACOBIAN IS NOT SINGULAR.
C
      IF(.NOT.BYPASS.AND.(.NOT.QRSING).AND.N.GT.1) THEN
         IF(SCLXCH) THEN
C
C             SET UP FOR CONDITION NUMBER ESTIMATOR - SCALE R WRT X'S.
C
            DO 500 J=1,N
          IF(SCALEX(J).NE.ONE) THEN
             SCALXJ=SCALEX(J)
             RDIAG(J)=RDIAG(J)/SCALXJ
             DO 600 I=1,J-1
                A(I,J)=A(I,J)/SCALXJ
600                 CONTINUE
          END IF
500           CONTINUE
         END IF
C
         CALL CONDNO(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,
     $                 CONNUM,A     ,WV1   ,WV2   ,WV3   ,RDIAG )
C
C          IF OVERFLOW DETECTED IN CONDITION NUMBER ESTIMATOR ASSIGN
C          QRSING AS TRUE.
C
           IF(OVERFL) QRSING=.TRUE.
C
C          NOTE: OVERFL SWITCHED TO FALSE BEFORE FORMATION OF H LATER.
C
        ELSE
C
C          ASSIGN DUMMY TO CONNUM FOR SINGULAR JACOBIAN UNLESS N=1.
C
           IF(N.EQ.1) THEN
              CONNUM=ONE
           ELSE
              CONNUM=ZERO
           END IF
        END IF
C
C       MATRIX H=JAC^JAC IS FORMED IN THREE CASES:
C          1)  THE (SCALED) JACOBIAN IS SINGULAR
C          2)  THE CONDITION NUMBER IS TOO HIGH AND THE
C              OPTION TO BYPASS THE PERTURBATION OF THE
C              JACOBIAN IS NOT BEING USED.
C          3)  REQUESTED BY USER THROUGH NEWSTM.
C
        IF(QRSING.OR.((.NOT.BYPASS).AND.CONNUM.GT.
     $     ONE/SQRTEP**1.333).OR.NEWSTM.EQ.77) THEN
C
C          FORM H:=(DF*JAC)^(DF*JAC) WHERE DF=DIAG(SCALEF).  USE
C          PREVIOUSLY COMPUTED QR DECOMPOSITION OF (SCALED) JACOBIAN
C          WHERE R IS STORED IN THE UPPER TRIANGLE OF A AND RDIAG.
C
         IF(OVERCH) THEN
            OVERFL=.FALSE.
            CALL ATAOV(OVERFL,MAXEXP,N,NUNIT,OUTPUT,JAC,H,SCALEF)
         ELSE
            CALL RTRMUL(N,A,H,RDIAG,WV1)
         END IF
         IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              IF(QRSING.AND.(.NOT.OVERFL)) THEN
                 WRITE(ICOUT,12)
12               FORMAT(T3,'*',7X,'SINGULAR JACOBIAN DETECTED:',
     $                  ' JACOBIAN PERTURBED',T74,'*')
                 CALL DPWRST('XXX','BUG')
              ELSE
C
C                NOTE: IF OVERFL IS TRUE THEN QRSING MUST BE TRUE.
C
                 IF(OVERFL) THEN
                    WRITE(ICOUT,13)
13                  FORMAT(T3,'*',7X,'POTENTIAL OVERFLOW DETECTED',
     $                     ' IN CONDITION NUMBER ESTIMATOR',T74,'*')
                    CALL DPWRST('XXX','BUG')
                    WRITE(ICOUT,14)
14                  FORMAT(T3,'*',7X,'MATRIX "ASSIGNED" AS ',
     $                     'SINGULAR AND JACOBIAN PERTURBED',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 ELSE
                    WRITE(ICOUT,15) CONNUM
15                  FORMAT(T3,'*',7X,'CONDITION NUMBER TOO HIGH: ',
     $                     1PD12.3,', JACOBIAN PERTURBED',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
              END IF
           END IF
         OVERFL=.FALSE.
         IF(NEWSTM.NE.77) THEN
C
C             FIND 1-NORM OF H MATRIX AND PERTURB DIAGONAL.
C
            CALL ONENRM(ABORT ,PERTRB,N     ,NUNIT ,OUTPUT,EPSMCH,
     $                    H1NORM,H     ,SCALEX)
            IF(ABORT) RETURN
         END IF
C
C          CHOLESKY DECOMPOSITION OF H MATRIX - MAXFFL=0 IMPLIES
C          THAT H IS KNOWN TO BE POSITIVE DEFINITE.
C
           MAXFFL=ZERO
           CALL CHOLDE(N,MAXADD,MAXFFL,SQRTEP,H,A)
           IF(OUTPUT.GT.4) THEN
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,1)
              CALL DPWRST('XXX','BUG')
              WRITE(ICOUT,16)
16            FORMAT(T3,'*',5X,'CHOLESKY DECOMPOSITION OF H MATRIX'
     $               ,T74,'*')
              CALL DPWRST('XXX','BUG')
              CALL MATPRT(N,N,N,N,NUNIT,A)
           END IF
C
C          FIND NEWTON STEP FROM CHOLESKY DECOMPOSITION.  IF THE
C          DIAGONAL HAS BEEN PERTURBED THEN THIS IS NOT THE ACTUAL
C          NEWTON STEP BUT ONLY AN APPORXIMATION THEREOF.
C
         DO 700 I=1,N
              WV1(I)=-DELF(I)
700        CONTINUE
           CALL CHSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                 OUTPUT,A     ,WV1   ,SN    ,WV2)
           OVERFL=.FALSE.
           IF(OUTPUT.GT.3) THEN
              IF(.NOT.SCLXCH) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 IF(.NOT.PERTRB) THEN
                    WRITE(ICOUT,17)
17                  FORMAT(T3,'*',5X,'NEWTON STEP FROM CHOLESKY',
     $                     ' DECOMPOSITION',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 ELSE
                    WRITE(ICOUT,18)
18                  FORMAT(T3,'*',5X,'APPROXIMATE NEWTON STEP FROM',
     $                     ' PERTURBED JACOBIAN',T74,'*')
                    CALL DPWRST('XXX','BUG')
                 END IF
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 DO 800 I=1,N
                    WRITE(ICOUT,19) I,SN(I)
19                  FORMAT(T3,'*',7X,'SN(',I3,') = ',1PD12.3,T74,'*')
                    CALL DPWRST('XXX','BUG')
800              CONTINUE
            ELSE
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               IF(.NOT.PERTRB) THEN
                   WRITE(ICOUT,20)
20                 FORMAT(T3,'*',5X,'NEWTON STEP FROM CHOLESKY',
     $             ' DECOMPOSITION',3X,' IN SCALED UNITS',T74,'*')
                   CALL DPWRST('XXX','BUG')
               ELSE
                   WRITE(ICOUT,21)
21                 FORMAT(T3,'*',5X,'APPROXIMATE NEWTON STEP FROM',
     $             ' PERTURBED JACOBIAN',3X,'IN SCALED UNITS',T74,
     $             '*')
                   CALL DPWRST('XXX','BUG')
               END IF
               WRITE(ICOUT,1)
               CALL DPWRST('XXX','BUG')
               DO 900 I=1,N
                  WRITE(ICOUT,22) I,SN(I),I,SCALEX(I)*SN(I)
22                FORMAT(T3,'*',7X,'SN(',I3,') = ',1PD12.3,
     $                   15X,'SN(',I3,') = ',1PD12.3,T74,'*')
                  CALL DPWRST('XXX','BUG')
900            CONTINUE
            END IF
         END IF
C
C          SET QRSING TO TRUE SO THAT THE CORRECT MATRIX
C          FACTORIZATION IS USED IN THE BACK-CALCULATION OF
C          SBAR FOR DEUFLHARD RELAXATION FACTOR INITIALIZATION
C          IN LINE SEARCH (ONLY MATTERS WHEN JACOBIAN IS ILL-
C          CONDITIONED BUT NOT SINGULAR).
C
           QRSING=.TRUE.
        ELSE
         IF(OUTPUT.GT.3.AND.N.GT.1) THEN
            IF(.NOT.BYPASS.AND.CONNUM.LE.ONE/SQRTEP**1.33) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,23) CONNUM
23               FORMAT(T3,'*',7X,'CONDITION NUMBER ACCEPTABLE, ',
     $           1PD9.2,', JACOBIAN NOT PERTURBED',T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
              IF(BYPASS.AND.CONNUM.GT.ONE/SQRTEP**1.33) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,24) CONNUM
24               FORMAT(T3,'*',7X,'CONDITION NUMBER HIGH, ',
     $           1PD9.2,', JACOBIAN NOT PERTURBED AS',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,25)
25               FORMAT(T3,'*',7X,'BYPASS IS TRUE',T74,'*')
                 CALL DPWRST('XXX','BUG')
              END IF
           END IF
C
C          NOTE: HERE SN STORES THE R.H.S. - IT IS OVERWRITTEN.
C
         DO 1000 I=1,N
            SN(I)=-FVECC(I)*SCALEF(I)
1000       CONTINUE
         IF(.NOT.BYPASS.AND.SCLXCH) THEN
C
C             R WAS SCALED BEFORE THE CONDITION NUMBER ESTIMATOR -
C             THIS CONVERTS IT BACK TO THE UNSCALED FORM.
C
            DO 1100 J=1,N
          IF(SCALEX(J).NE.ONE) THEN
             SCALXJ=SCALEX(J)
             RDIAG(J)=RDIAG(J)*SCALXJ
             DO 1200 I=1,J-1
                A(I,J)=A(I,J)*SCALXJ
1200                CONTINUE
          END IF
1100          CONTINUE
         END IF
C
C          ACCEPTABLE CONDITION NUMBER - USE BACK SUBSTITUTION TO
C          FIND NEWTON STEP FROM QR DECOMPOSITION.
C
           CALL QRSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                 OUTPUT,A     ,HHPI  ,RDIAG ,SN    )
           OVERFL=.FALSE.
C
         IF(OUTPUT.GT.3) THEN
            IF(.NOT.SCLXCH) THEN
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,26)
26               FORMAT(T3,'*',7X,'NEWTON STEP FROM QR DECOMPOSITION '
     $                  ,T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 DO 1300 I=1,N
                    WRITE(ICOUT,19) I,SN(I)
                    CALL DPWRST('XXX','BUG')
1300             CONTINUE
            ELSE
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,27)
27               FORMAT(T3,'*',7X,'NEWTON STEP FROM QR DECOMPOSITION '
     $                  ,7X,'IN SCALED UNITS',T74,'*')
                 CALL DPWRST('XXX','BUG')
                 WRITE(ICOUT,1)
                 CALL DPWRST('XXX','BUG')
                 DO 1400 I=1,N
                    WRITE(ICOUT,22) I,SN(I),I,SCALEX(I)*SN(I)
                    CALL DPWRST('XXX','BUG')
1400             CONTINUE
            END IF
         END IF
C
C          TRANSFORM MATRICES FOR SUBSEQUENT CALCULATIONS IN TRUST
C          REGION METHODS (A IS STORED ABOVE IN H).
C
           IF(.NOT.LINESR) THEN
            DO 1500 I=1,N
                 A(I,I)=RDIAG(I)
                 DO 1600 J=1,I-1
                    A(I,J)=A(J,I)
1600             CONTINUE
1500        CONTINUE
           END IF
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE NSTPUN.
C
        END
      SUBROUTINE ONENRM(ABORT ,PERTRB,N     ,NUNIT ,OUTPUT,EPSMCH,
     $                    H1NORM,H     ,SCALEX)
C
C       FEB. 23, 1992
C
C       FIND 1-NORM OF H MATRIX IF PERTURBATION IS DESIRED AND
C       PERTURB DIAGONAL.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      INTEGER   OUTPUT
      DIMENSION H(N,N) ,SCALEX(N)
      LOGICAL   ABORT  ,PERTRB
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DATA ZERO /0.0D0/
C
      SQRTEP=SQRT(EPSMCH)
      IF(OUTPUT.GT.4) THEN
         WRITE(ICOUT,1)
1          FORMAT(T3,'*',T74,'*')
         WRITE(ICOUT,2)
2          FORMAT(T3,'*',7X,'DIAGONAL OF MATRIX H ',
     $     '(=JAC^JAC) BEFORE BEING PERTURBED',T74,'*')
         WRITE(ICOUT,1)
         DO 100 I=1,N
            WRITE(ICOUT,3) I,I,H(I,I)
3             FORMAT(T3,'*',10X,'H(',I3,',',I3,') = ',
     $        1PD12.3,T74,'*')
100        CONTINUE
      END IF
      H1NORM=ZERO
      DO 200 J=1,N
         H1NORM=H1NORM+ABS(H(1,J))/SCALEX(J)
200     CONTINUE
      H1NORM=H1NORM/SCALEX(1)
      DO 300 I=2,N
         TEMP=ZERO
         DO 400 J=1,I
            TEMP=TEMP+ABS(H(J,I))/SCALEX(J)
400        CONTINUE
         DO 500 J=I+1,N
            TEMP=TEMP+ABS(H(I,J))/SCALEX(J)
500        CONTINUE
         H1NORM=MAX(H1NORM,TEMP/SCALEX(I))
300     CONTINUE
      IF(OUTPUT.GT.4) THEN
         WRITE(ICOUT,1)
         WRITE(ICOUT,4) H1NORM
4          FORMAT(T3,'*',7X,'1-NORM OF MATRIX H: ',
     $     1PD11.3,T74,'*')
      END IF
      IF(H1NORM.LT.EPSMCH) THEN
         IF(OUTPUT.GT.0) THEN
            WRITE(ICOUT,5)
5             FORMAT(T3,72('*'))
            WRITE(ICOUT,1)
            WRITE(ICOUT,6)
6             FORMAT(T3,'*',4X,'PROGRAM FAILS AS 1-NORM OF',
     $        ' JACOBIAN IS TOO SMALL',T74,'*')
            WRITE(ICOUT,1)
            WRITE(ICOUT,5)
         END IF
         ABORT=.TRUE.
         RETURN
      ELSE
C
C          PERTURB DIAGONAL OF MATRIX H - USE THIS TO FIND "SN".
C
         PERTRB=.TRUE.
         DO 600 I=1,N
            H(I,I)=H(I,I)
     $               +SQRT(DBLE(N))*SQRTEP*H1NORM*SCALEX(I)*SCALEX(I)
600        CONTINUE
         IF(OUTPUT.GT.4) THEN
            WRITE(ICOUT,1)
            WRITE(ICOUT,1)
            WRITE(ICOUT,7)
7             FORMAT(T3,'*',4X,'PERTURBED H MATRIX',T74,'*')
            CALL MATPRT(N,N,N,N,NUNIT,H)
         END IF
      END IF
      RETURN
C
C       LAST CARD OF SUBROUTINE ONENRM.
C
      END
CCCCC   SUBROUTINE QFORM(N,A,HHPI,JAC)
        SUBROUTINE QFORMZ(N,A,HHPI,JAC)
C
C       FEB. 14, 1991
C
C       FORM Q^  FROM THE HOUSEHOLDER MATRICES STORED IN
C       MATRICES A AND HHPI AND STORE IT IN JAC.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N)
        DIMENSION        A(N,N)   ,HHPI(N)
        DATA ZERO,ONE /0.0D0,1.0D0/
C
        DO 100 J=1,N
           DO 200 I=1,N
              JAC(I,J)=ZERO
200        CONTINUE
           JAC(J,J)=ONE
100     CONTINUE
        DO 300 K=1,N-1
           IF(HHPI(K).NE.ZERO) THEN
              DO 400 J=1,N
                 TAU=ZERO
                 DO 500 I=K,N
                    TAU=TAU+A(I,K)*JAC(I,J)
500              CONTINUE
                 TAU=TAU/HHPI(K)
                 DO 600 I=K,N
                    JAC(I,J)=JAC(I,J)-TAU*A(I,K)
600              CONTINUE
400           CONTINUE
           END IF
300     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE QFORM.
C
        END
        SUBROUTINE QMIN(NUNIT,OUTPUT,DELFTS,DELTA,DELTAF,STPLEN)
C
C       FEB. 9, 1991
C
C       SET THE NEW TRUST REGION SIZE, DELTA, BASED ON A QUADRATIC
C       MINIMIZATION WHERE DELTA IS THE INDEPENDENT VARIABLE.
C
C       DELTAF IS THE DIFFERENCE IN THE SUM-OF-SQUARES OBJECTIVE
C       FUNCTION VALUE AND DELFTS IS THE DIRECTIONAL DERIVATIVE IN
C       THE DIRECTION OF THE CURRENT STEP, S, WHICH HAS STEP LENGTH
C       STPLEN.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        INTEGER  OUTPUT
        DATA ZERO,POINT1,TWO /0.0D0,0.1D0,2.0D0/
C
        IF(DELTAF-DELFTS.NE.ZERO) THEN
C
C          CALCULATE DELTA WHERE MINIMUM WOULD OCCUR - DELTMP.
C          THIS IS PROVISIONAL AS IT MUST BE WITHIN CERTAIN
C          LIMITS TO BE ACCEPTED.
C
           DELTMP=-DELFTS*STPLEN/(TWO*(DELTAF-DELFTS))
           IF(OUTPUT.GT.4) THEN
              WRITE(ICOUT,1)
1             FORMAT(T3,'*',T74,'*')
              WRITE(ICOUT,2) DELTMP
2             FORMAT(T3,'*',7X,'TEMPORARY DELTA FROM QUADRATIC',
     $        ' MINIMIZATION: ',1PD12.3,T74,'*')
              WRITE(ICOUT,3) DELTA
3             FORMAT(T3,'*',30X,'VERSUS CURRENT DELTA: ',1PD12.3,
     $        T74,'*')
           END IF
C
C          REDUCE DELTA DEPENDING ON THE MAGNITUDE OF DELTMP.
C          IT MUST BE WITHIN [.1DELTA,.5DELTA] TO BE ACCEPTED -
C          OTHERWISE THE NEAREST ENDPOINT OF THE INTERVAL IS USED.
C
           IF(DELTMP.LT.POINT1*DELTA) THEN
              DELTA=POINT1*DELTA
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,1)
                 WRITE(ICOUT,4)
4                FORMAT(T3,'*',7X,'NEW DELTA SET TO 0.1',
     $           ' CURRENT DELTA',T74,'*')
              END IF
           ELSEIF(DELTMP.GT.DELTA/TWO) THEN
              DELTA=DELTA/TWO
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,1)
                 WRITE(ICOUT,5)
5                FORMAT(T3,'*',7X,'NEW DELTA SET TO 0.5',
     $           ' CURRENT DELTA',T74,'*')
              END IF
           ELSE
              DELTA=DELTMP
              IF(OUTPUT.GT.4) THEN
                 WRITE(ICOUT,1)
                 WRITE(ICOUT,6)
6                FORMAT(T3,'*',7X,'NEW DELTA SET TO DELTMP',T74,'*')
              END IF
           END IF
        ELSE
           IF(OUTPUT.GT.4) THEN
              WRITE(ICOUT,1)
              WRITE(ICOUT,7)
7             FORMAT(T3,'*',7X,'TO AVOID OVERFLOW NEW DELTA',
     $        ' SET TO 0.5 CURRENT DELTA',T74,'*')
           END IF
           DELTA=DELTA/TWO
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE QMIN.
C
        END
        SUBROUTINE QRDCOM(QRSING,N,EPSMCH,A,HHPI,RDIAG)
C
C       FEB. 23, 1992
C
C       THIS SUBROUTINE COMPUTES THE QR DECOMPOSITION OF THE
C       MATRIX A.  THE DECOMPOSITION IS COMPLETED EVEN IF
C       A SINGULARITY IS DETECTED (WHEREUPON QRSING IS SET TO
C       TRUE).
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION  A(N,N)   ,HHPI(N) ,RDIAG(N)
        LOGICAL    QRSING
        DATA ZERO /0.0D0/
C
        QRSING=.FALSE.
C
        DO 100 K=1,N-1
           ETA=ZERO
           DO 200 I=K,N
              ETA=MAX(ETA,ABS(A(I,K)))
200        CONTINUE
           IF(ETA.LT.EPSMCH) THEN
              QRSING=.TRUE.
              HHPI(K)=ZERO
              RDIAG(K)=ZERO
           ELSE
              DO 300 I=K,N
                 A(I,K)=A(I,K)/ETA
300           CONTINUE
              SIGMA=ZERO
              DO 400 I=K,N
           SIGMA=SIGMA+A(I,K)*A(I,K)
400           CONTINUE
              SIGMA=SIGN(SQRT(SIGMA),A(K,K))
              A(K,K)=A(K,K)+SIGMA
              HHPI(K)=SIGMA*A(K,K)
              RDIAG(K)=-ETA*SIGMA
              DO 500 J=K+1,N
                 TAU=ZERO
                 DO 600 I=K,N
                    TAU=TAU+A(I,K)*A(I,J)
600              CONTINUE
                 TAU=TAU/HHPI(K)
                 DO 700 I=K,N
                    A(I,J)=A(I,J)-TAU*A(I,K)
700              CONTINUE
500           CONTINUE
           END IF
100     CONTINUE
        RDIAG(N)=A(N,N)
        IF(ABS(RDIAG(N)).LT.EPSMCH) QRSING=.TRUE.
        RETURN
C
C       LAST CARD OF SUBROUTINE QRDCOM.
C
        END
        SUBROUTINE QRSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                    OUTPUT,A     ,HHPI  ,RDIAG ,B     )
C
C       FEB. 2, 1991
C
C       THIS SUBROUTINE SOLVES
C
C            (QR)X=B
C
C            WHERE  Q AND R ARE OBTAINED FROM THE QR DECOMPOSITION
C                   B IS A GIVEN RIGHT HAND SIDE WHICH IS
C                     OVERWRITTEN
C
C                   R IS CONTAINED IN THE STRICT UPPER TRIANGLE OF
C                     MATRIX A AND THE VECTOR RDIAG
C                   Q IS "CONTAINED" IN THE LOWER TRIANGLE OF MATRIX A
C
C       FRSTOV  INDICATES FIRST OVERFLOW - USED ONLY TO SET BORDER
C               FOR OUTPUT
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        INTEGER    OUTPUT
        DIMENSION  A(N,N)  ,HHPI(N)  ,B(N)   ,RDIAG(N)
        LOGICAL    FRSTOV  ,OVERCH   ,OVERFL ,WRNSUP
        COMMON/NNES_2/WRNSUP
      DATA ZERO,ONE,TEN /0.0D0,1.0D0,10.0D0/
C
      EPS=TEN**(-MAXEXP)
        FRSTOV=.TRUE.
        OVERFL=.FALSE.
C
C       MULTIPLY RIGHT HAND SIDE BY Q^ THEN SOLVE USING R
C       STORED IN MATRIX A.
C
        DO 100 J=1,N-1
         TAU=ZERO
           DO 200 I=J,N
              IF(OVERCH) THEN
                 IF(LOG10(ABS(A(I,J))+EPS)+LOG10(ABS(B(I))+EPS)
     $              -LOG10(HHPI(J)+EPS).GT.MAXEXP) THEN
                    OVERFL=.TRUE.
             TAU=SIGN(TEN**MAXEXP,A(I,J))*
     $                  SIGN(ONE,B(J))
                    GO TO 201
                 END IF
              END IF
              TAU=TAU+A(I,J)*B(I)/HHPI(J)
201        CONTINUE
200        CONTINUE
           DO 300 I=J,N
              IF(OVERCH) THEN
                 IF(LOG10(ABS(TAU)+EPS)+LOG10(ABS(A(I,J))+EPS)
     $              .GT.MAXEXP) THEN
                    OVERFL=.TRUE.
             B(I)=-SIGN(TEN**MAXEXP,TAU)
     $                   *SIGN(ONE,A(I,J))
                    IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
                       IF(FRSTOV) THEN
                          WRITE(ICOUT,1)
1                         FORMAT(T3,'*',T74,'*')
                       END IF
                       WRITE(ICOUT,2) I,B(I)
2                      FORMAT(T3,'*',4X,'WARNING: COMPONENT ',I3,
     $                 ' SET TO ',1PD11.3,' IN QRSOLV BEFORE',
     $                 ' RSOLV',T74,'*')
                    END IF
                    GO TO 301
                 END IF
              END IF
              B(I)=B(I)-TAU*A(I,J)
301           CONTINUE
300        CONTINUE
100     CONTINUE
        CALL RSOLV(OVERCH,OVERFL,MAXEXP,N,NUNIT,OUTPUT,A,RDIAG,B)
        RETURN
C
C       LAST CARD OF SUBROUTINE QRSOLV.
C
      END
        SUBROUTINE QRUPDA(OVERFL,MAXEXP,N,EPSMCH,A,JAC,U,V)
C
C       FEB. 12, 1991
C
C       UPDATE QR DECOMPOSITION USING A SERIES OF GIVENS ROTATIONS.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION JAC(N,N)
        DIMENSION        A(N,N)   ,HOLD(2)  ,U(N)   ,V(N)
        LOGICAL          OVERFL
        DATA ZERO /0.0D0/
C
C       REPLACE SUBDIAGONAL WITH ZEROS SO THAT WHEN R IS MULTIPLIED
C       BY GIVENS (JACOBI) ROTATIONS THE SUBDIAGONAL ELEMENTS DO
C       NOT AFFECT THE OUTCOME.
C
        DO 100 I=2,N
           A(I,I-1)=ZERO
100     CONTINUE
C
C       FIND LARGEST K FOR WHICH U(K) DOES NOT EQUAL ZERO.
C
        K=N
        DO 200 L=1,N
           IF(U(K).EQ.ZERO) THEN
              IF(K.GT.1) THEN
                 K=K-1
              ELSE
                 GO TO 201
              END IF
           ELSE
              GO TO 201
           END IF
200     CONTINUE
201     CONTINUE
C
C       MULTIPLY UV^ BY A SERIES OF ROTATIONS SO THAT ALL BUT THE
C       TOP ROW IS MADE ZERO (THEORETICALLY THIS IS WHAT HAPPENS
C       ALTHOUGH THIS MATRIX ISN'T ACTUALLY FORMED).
C
        DO 300 I=K-1,1,-1
           CALL JACROT(OVERFL,I,MAXEXP,N,U(I),U(I+1),EPSMCH,A,JAC)
           IF(U(I).EQ.ZERO) THEN
C
C             THIS STEP JUST AVOIDS ADDING ZERO.
C
              U(I)=ABS(U(I+1))
           ELSE
              HOLD(1)=U(I)
              HOLD(2)=U(I+1)
              LDHOLD=2
              CALL TWONRM(OVERFL,MAXEXP,LDHOLD,EPSMCH,EUCNRM,HOLD)
              U(I)=EUCNRM
           END IF
300     CONTINUE
C
C       ADD THE TOP ROW TO THE TOP ROW OF A - THIS FORMS THE
C       UPPER HESSENBERG MATRIX.
C
        DO 400 J=1,N
           A(1,J)=A(1,J)+U(1)*V(J)
400     CONTINUE
C
C       FORM THE UPPER TRIANGULAR R MATRIX BY A SERIES OF ROTATIONS
C       TO ZERO OUT THE SUBDIAGONALS.
C
        DO 500 I=1,K-1
           CALL JACROT(OVERFL,I     ,MAXEXP,N     ,A(I,I)  ,A(I+1,I),
     $                 EPSMCH,A     ,JAC)
500     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE QRUPDA.
C
        END
        SUBROUTINE RCDPRT(NUNIT,RETCOD,DELTA,RELLEN,STPTOL)
C
C       FEB. 14, 1991
C
C       DESCRIBE MEANING OF RETURN CODES, RETCOD, FROM TRUST REGION
C       UPDATING.
C
        DOUBLE PRECISION  DELTA ,RELLEN ,STPTOL
        INTEGER           RETCOD
C
        WRITE(ICOUT,1)
1       FORMAT(T3,'*',T74,'*')
        WRITE(ICOUT,2) RETCOD
2       FORMAT(T3,'*',7X,'RETCOD, FROM TRUST REGION UPDATING:',I5,
     $  T74,'*')
        WRITE(ICOUT,1)
        IF(RETCOD.EQ.1) THEN
           WRITE(ICOUT,3)
3          FORMAT(T3,'*',7X,'PROMISING STEP FOUND; DELTA',
     $     ' HAS BEEN INCREASED TO NEWLEN BUT',T74,'*')
           WRITE(ICOUT,4)
4          FORMAT(T3,'*',7X,'BECAUSE OF OVERFLOWS IN THE FUNCTION',
     $     ' VECTOR(S) IN SUBSEQUENT',T74,'*')
           WRITE(ICOUT,5)
5          FORMAT(T3,'*',7X,'STEP(S) THE PROJECTED DELTA IS LESS',
     $     ' THAN THAT AT THE ALREADY SUCCESSFUL',T74,'*')
           WRITE(ICOUT,6)
6          FORMAT(T3,'*',7X,'STEP - RETURN TO SUCCESSFUL STEP',
     $     ' AND ACCEPT AS NEW POINT',T74,'*')
        ELSEIF(RETCOD.EQ.2) THEN
           WRITE(ICOUT,3)
           WRITE(ICOUT,7)
7          FORMAT(T3,'*',7X,'BECAUSE OF OVERFLOWS IN THE OBJECTIVE',
     $     ' FUNCTION IN SUBSEQUENT',T74,'*')
           WRITE(ICOUT,5)
           WRITE(ICOUT,6)
        ELSEIF(RETCOD.EQ.3) THEN
           WRITE(ICOUT,3)
           WRITE(ICOUT,8)
8          FORMAT(T3,'*',7X,'BECAUSE OF SUBSEQUENT FAILURES IN',
     $     ' THE STEP ACCEPTANCE TEST(S)',T74,'*')
           WRITE(ICOUT,9)
9          FORMAT(T3,'*',7X,'THE PROJECTED DELTA IS LESS',
     $     ' THAN THAT AT THE ALREADY',T74,'*')
           WRITE(ICOUT,10)
10         FORMAT(T3,'*',7X,'SUCCESSFUL STEP - RETURN TO',
     $     ' SUCCESSFUL STEP AND ACCEPT',T74,'*')
        ELSEIF(RETCOD.EQ.4) THEN
           WRITE(ICOUT,11)
11         FORMAT(T3,'*',7X,'STEP ACCEPTED BY STEP SIZE CRITERION',
     $     ' ONLY - DELTA REDUCED',T74,'*')
        ELSEIF(RETCOD.EQ.5) THEN
           WRITE(ICOUT,12)
12         FORMAT(T3,'*',7X,'STEP ACCEPTED - NEW FUNCTION VALUE'
     $     ' GREATER THAN PREVIOUS =>',T74,'*')
           WRITE(ICOUT,13)
13         FORMAT(T3,'*',7X,'REDUCE TRUST REGION',T74,'*')
        ELSEIF(RETCOD.EQ.6) THEN
           WRITE(ICOUT,14)
14         FORMAT(T3,'*',7X,'STEP ACCEPTED - DELTA CHANGED'
     $     ' AS DETAILED ABOVE',T74,'*')
        ELSEIF(RETCOD.EQ.7) THEN
           WRITE(ICOUT,15)
15         FORMAT(T3,'*',7X,'NO PROGRESS MADE: RELATIVE STEP',
     $     ' SIZE IS TOO SMALL',T74,'*')
           WRITE(ICOUT,16) RELLEN,STPTOL
16         FORMAT(T3,'*',7X,'REL. STEP SIZE, RELLEN = ',
     $     1PD12.3,', STPTOL = ',1PD12.3,T74,'*')
        ELSEIF(RETCOD.EQ.8) THEN
           WRITE(ICOUT,17)
17         FORMAT(T3,'*',7X,'POINT MODIFIED BY CONSTRAINTS',
     $     ' NOT A DESCENT DIRECTION',T74,'*')
           WRITE(ICOUT,18)
18         FORMAT(T3,'*',7X,'DELTA REDUCED TO CONFAC*RATIOM*DELTA',
     $     T74,'*')
        ELSEIF(RETCOD.EQ.9) THEN
           WRITE(ICOUT,19)
19         FORMAT(T3,'*',7X,'OVERFLOW DETECTED IN FUNCTION VECTOR',
     $     ' - DELTA REDUCED',T74,'*')
        ELSEIF(RETCOD.EQ.10) THEN
           WRITE(ICOUT,20)
20         FORMAT(T3,'*',7X,'OVERFLOW IN OBJECTIVE FUNCTION',
     $     ' - DELTA REDUCED',T74,'*')
        ELSEIF(RETCOD.EQ.11) THEN
           WRITE(ICOUT,21)
21         FORMAT(T3,'*',7X,'STEP NOT ACCEPTED - REDUCE TRUST',
     $     ' REGION SIZE BY MINIMIZATION',T74,'*')
           WRITE(ICOUT,22)
22         FORMAT(T3,'*',7X,'OF QUADRATIC MODEL IN STEP',
     $     ' DIRECTION',T74,'*')
        ELSE
           WRITE(ICOUT,23)
23         FORMAT(T3,'*',7X,'PROMISING STEP - INCREASE DELTA TO',
     $     ' NEWLEN AND TRY A NEW STEP',T74,'*')
        END IF
        WRITE(ICOUT,1)
        WRITE(ICOUT,24) DELTA
24      FORMAT(T3,'*',7X,'DELTA ON RETURN FROM TRUST REGION',
     $  ' UPDATING: ',1PD11.3,T74,'*')
        RETURN
C
C       LAST CARD OF SUBROUTINE RCDPRT.
C
        END
        SUBROUTINE RSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                   OUTPUT,A     ,RDIAG ,B     )
C
C       FEB. 14, 1991
C
C       THIS SUBROUTINE SOLVES, BY BACKWARDS SUBSTITUTION,
C
C              RX=B
C
C              WHERE    R IS TAKEN FROM THE QR DECOMPOSITION AND
C                         IS STORED IN THE STRICT UPPER TRIANGLE
C                         OF MATRIX A AND THE VECTOR, RDIAG
C                       B IS A GIVEN RIGHT HAND SIDE WHICH IS
C                         OVERWRITTEN
C
C       FRSTOV  INDICATES FIRST OVERFLOW - USED ONLY TO SET
C               BORDERS FOR OUTPUT
C
       IMPLICIT DOUBLE PRECISION (A-H,O-Z)
       DOUBLE PRECISION    MAXLOG
       INTEGER             OUTPUT
       DIMENSION           A(N,N) ,B(N)   ,RDIAG(N)
       LOGICAL             FRSTOV ,OVERCH ,OVERFL  ,WRNSUP
       COMMON/NNES_2/WRNSUP
       DATA ZERO,ONE,TEN /0.0D0,1.0D0,10.0D0/
C
       FRSTOV=.TRUE.
       OVERFL=.FALSE.
       EPS=TEN**(-MAXEXP)
C
       IF(OVERCH) THEN
          IF(LOG10(ABS(B(N))+EPS)-LOG10(ABS(RDIAG(N))+EPS)
     $       .GT.MAXEXP) THEN
             OVERFL=.TRUE.
             B(N)=SIGN(TEN**MAXEXP,B(N))*SIGN(ONE,RDIAG(N))
             IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) THEN
                FRSTOV=.FALSE.
                WRITE(ICOUT,1)
1               FORMAT(T3,'*',T74,'*')
                WRITE(ICOUT,2) N,B(N)
2               FORMAT(T3,'*',4X,'WARNING: COMPONENT ',I3,
     $          ' SET TO ',1PD12.3,T74,'*')
             END IF
             GO TO 101
          END IF
       END IF
       B(N)=B(N)/RDIAG(N)
101    CONTINUE
       DO 200 I=N-1,1,-1
          IF(OVERCH) THEN
C
C            CHECK TO FIND IF ANY TERMS IN THE EVALUATION WOULD
C            OVERFLOW.
C
             MAXLOG=LOG10(ABS(B(I))+EPS)-LOG10(ABS(RDIAG(I))+EPS)
             JSTAR=0
             DO 300 J=I+1,N
                TMPLOG=LOG10(ABS(A(I,J))+EPS)+LOG10(ABS(B(J))+EPS)-
     $                  LOG10(ABS(RDIAG(I))+EPS)
                IF(TMPLOG.GT.MAXLOG) THEN
                   JSTAR=J
                   MAXLOG=TMPLOG
                END IF
300          CONTINUE
C
C            IF AN OVERFLOW WOULD OCCUR ASSIGN A VALUE FOR THE
C            TERM WITH CORRECT SIGN.
C
             IF(MAXLOG.GT.MAXEXP) THEN
                OVERFL=.TRUE.
                IF(JSTAR.EQ.0) THEN
                   B(I)=SIGN(TEN**MAXEXP,B(I))*
     $                  SIGN(ONE,RDIAG(I))
                ELSE
                   B(I)=-SIGN(TEN**MAXEXP,A(I,JSTAR))*
     $                   SIGN(ONE,B(JSTAR))*SIGN(ONE,RDIAG(I))
                END IF
                IF(FRSTOV) THEN
                   FRSTOV=.FALSE.
                   WRITE(ICOUT,1)
                END IF
                IF(OUTPUT.GT.2.AND.(.NOT.WRNSUP)) WRITE(ICOUT,2) I,B(I)
                GO TO 301
             END IF
          END IF
C
C         SUM FOR EACH TERM ORDERING OPERATIONS TO MINIMIZE
C         POSSIBILITY OF OVERFLOW.
C
          SUM=ZERO
          DO 400 J=I+1,N
             SUM=SUM+(MIN(ABS(A(I,J)),ABS(B(J)))/RDIAG(I))
     $           *(MAX(ABS(A(I,J)),ABS(B(J))))
     $           *SIGN(ONE,A(I,J))*SIGN(ONE,B(J))
400        CONTINUE
           B(I)=B(I)/RDIAG(I)-SUM
301     CONTINUE
200     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE RSOLV.
C
        END
      SUBROUTINE RTRMUL(N,A,H,RDIAG,WV1)
C
C       SEPT. 4, 1991
C
C       FIND R^R FOR QR-DECOMPOSED JACOBIAN.
C
C       R IS STORED IN STRICT UPPER TRIANGLE OF A AND RDIAG.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      DIMENSION A(N,N) ,H(N,N) ,RDIAG(N) ,WV1(N)
C
C       TEMPORARILY REPLACE DIAGONAL OF R IN A (A IS RESTORED LATER).
C
      DO 100 I=1,N
         WV1(I)=A(I,I)
         A(I,I)=RDIAG(I)
100     CONTINUE
      CALL UTUMUL(N,N,N,N,N,N,A,H)
      DO 200 I=1,N
         A(I,I)=WV1(I)
200     CONTINUE
      RETURN
C
C       LAST CARD OF SUBROUTINE RTRMUL.
C
      END


      SUBROUTINE SETUP(ABSNEW ,CAUCHY ,DEUFLH ,GEOMS  ,LINESR ,
     $                   NEWTON ,OVERCH ,ACPTCR ,ITSCLF ,ITSCLX ,
     $                   JACTYP ,JUPDM  ,MAXEXP ,MAXIT  ,MAXNS  ,
     $                   MAXQNS ,MINQNS ,N      ,NARMIJ ,NIEJEV ,
     $                   NJACCH ,OUTPUT ,QNUPDM ,STOPCR ,SUPPRS ,
     $                   TRUPDM ,ALPHA  ,CONFAC ,DELTA  ,DELFAC ,
     $                   EPSMCH ,ETAFAC ,FDTOLJ ,FTOL   ,LAM0   ,
     $                   MSTPF  ,NSTTOL ,OMEGA  ,RATIOF ,SIGMA  ,
     $                   STPTOL ,BOUNDL ,BOUNDU ,SCALEF ,SCALEX ,
     $                   HELP)
C
C       DEC. 7, 1991
C
C       SUBROUTINE SETUP ASSIGNS DEFAULT VALUES TO ALL
C       REQUISITE PARAMETERS.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DOUBLE PRECISION  LAM0     ,MSTPF    ,NSTTOL
        INTEGER           ACPTCR   ,OUTPUT   ,QNUPDM   ,STOPCR   ,
     $                    SUPPRS   ,TRUPDM
        DIMENSION         BOUNDL(N),BOUNDU(N),SCALEF(N),SCALEX(N)
      LOGICAL           ABSNEW   ,BYPASS   ,CAUCHY   ,DEUFLH   ,
     $                    GEOMS    ,LINESR   ,MATSUP   ,NEWTON   ,
     $                    OVERCH   ,WRNSUP
      CHARACTER*6 HELP
        COMMON/NNES_1/MATSUP
        COMMON/NNES_2/WRNSUP
        COMMON/NNES_3/BYPASS
        COMMON/NNES_4/NFETOT
        COMMON/NNES_5/SMALLB,BIGB,SMALLS,BIGS,BIGR
CCCCC integer i1mach
CCCCC double precision d1mach
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      REAL R1MACH
      INCLUDE 'DPCOMC.INC'
C
C       LOGICAL VALUES.
C
      ABSNEW=.FALSE.
        BYPASS=.FALSE.
        CAUCHY=.FALSE.
        DEUFLH=.TRUE.
        GEOMS=.TRUE.
        LINESR=.TRUE.
        MATSUP=.FALSE.
        NEWTON=.FALSE.
        OVERCH=.FALSE.
        WRNSUP=.FALSE.
C
C       INTEGER VALUES.
C
        ACPTCR=12
        ITSCLF=0
        ITSCLX=0
        JACTYP=1
        JUPDM=0
        MAXIT=250
        MAXNS=50
        MAXQNS=10
        MINQNS=7
        NARMIJ=1
        NFETOT=0
      NIEJEV=1
        NJACCH=1
        OUTPUT=2
        QNUPDM=1
        STOPCR=12
        SUPPRS=0
        TRUPDM=0
C
C       REAL VALUES.
C
        ALPHA   = 1.0D-04
        CONFAC  = 0.95D0
        DELTA   =-1.0D0
        DELFAC  = 2.0D0
        ETAFAC  = 0.2D0
        LAM0    = 1.0D0
        MSTPF   = 1.0D3
        OMEGA   = 0.1D0
        RATIOF  = 0.70D0
      SIGMA   = 0.5D0
C
C       CHARACTER VARIABLE.
C
      HELP(1:4)='NONE'
C
C       NOTE: NOTATIONAL CHANGES IN CALLING PROGRAM FROM MACHAR
C             1)  EPSMCH DENOTES MACHINE EPSILON
C             2)  MINEBB DENOTES MINIMUM EXPONENT BASE BETA
C             3)  MAXEBB DENOTES MAXIMUM EXPONENT BASE BETA
C
*       CALL MACHAR(IBETA ,IT    ,IRND  ,NGRD  ,MACHEP,
*    $              NEGEP ,IEXP  ,MINEBB,MAXEBB,EPSMCH,
*    $              EPSNEG,XMIN  ,XMAX  )
      IT = i1mach(14)
      IBETA = i1mach(10)
      MINEBB = i1mach(15)
      MAXEBB = i1mach(16)
      EPSMCH = d1mach(4)
      XMAX = d1mach(2)
        MAXEXP=INT(DBLE(MAXEBB)*LOG(DBLE(IBETA))/LOG(10.0D0))
C
C       VALUES FOR TWO-NORM CALCULATIONS.
C
        SMALLB=DBLE(IBETA)**((MINEBB+1)/2)
        BIGB=  DBLE(IBETA)**((MAXEBB-IT+1)/2)
        SMALLS=DBLE(IBETA)**((MINEBB-1)/2)
        BIGS=  DBLE(IBETA)**((MAXEBB+IT-1)/2)
        BIGR=  XMAX
C
C       SET STOPPING CRITERIA PARAMETERS.
C
        FDTOLJ  = 1.0D-06
        FTOL    = EPSMCH**0.333
        NSTTOL  = FTOL*FTOL
        STPTOL  = NSTTOL
C
C       VECTOR VALUES.
C
      TEMP=-10.0D0**MAXEXP
        DO 100 I=1,N
           BOUNDL(I)=TEMP
         BOUNDU(I)=-TEMP
           SCALEF(I)=1.0D0
           SCALEX(I)=1.0D0
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE SETUP.
C
      END

      SUBROUTINE TITLE(CAUCHY,DEUFLH,GEOMS ,LINESR,NEWTON,
     $                   OVERCH,ACPTCR,CONTYP,ITSCLF,ITSCLX,
     $                   JACTYP,JUPDM ,MAXIT ,MAXNS ,MAXQNS,
     $                   MGLL  ,MINQNS,N     ,NARMIJ,NINITN,
     $                   NJACCH,NUNIT ,OUTPUT,QNUPDM,STOPCR,
     $                   TRUPDM,ALPHA ,CONFAC,DELFAC,DELTA ,
     $                   EPSMCH,ETAFAC,FCNOLD,FTOL  ,LAM0  ,
     $                   MAXSTP,MSTPF ,NSTTOL,OMEGA ,RATIOF,
     $                   SIGMA ,STPTOL,BOUNDL,BOUNDU,FVECC ,
     $                   SCALEF,SCALEX,XC    )
C
C       APR. 13, 1991
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION LAM0     ,MAXSTP   ,MSTPF   ,NSTTOL
        INTEGER          ACPTCR   ,CONTYP   ,QNUPDM  ,OUTPUT   ,
     $                   STOPCR   ,TRUPDM
        DIMENSION        BOUNDL(N),BOUNDU(N),FVECC(N),SCALEF(N),
     $                   SCALEX(N),XC(N)
        LOGICAL          CAUCHY   ,DEUFLH   ,GEOMS   ,LINESR   ,
     $                   NEWTON   ,OVERCH
      DATA ZERO,ONE /0.0D0,1.0D0/
C
        IF(OUTPUT.LT.2) RETURN
        WRITE(ICOUT,1)
1       FORMAT(////,T3,72('*'))
        WRITE(ICOUT,2)
2       FORMAT(T3,72('*'))
        WRITE(ICOUT,2)
        WRITE(ICOUT,3)
3       FORMAT(T3,'*',T74,'*')
        WRITE(ICOUT,4)
4       FORMAT(T3,'*',33X,'NNES',T74,'*')
        WRITE(ICOUT,3)
        WRITE(ICOUT,5)
5       FORMAT(T3,'*',9X,'NONMONOTONIC NONLINEAR EQUATION SOLVER',
     $  '  VERSION 1.05',9X,'*')
        WRITE(ICOUT,3)
        WRITE(ICOUT,6)
6       FORMAT(T3,'*',21X,'COPYRIGHT 1991, BY R.S. BAIN',T74,'*')
        WRITE(ICOUT,3)
        WRITE(ICOUT,2)
        WRITE(ICOUT,2)
        WRITE(ICOUT,2)
        WRITE(ICOUT,7)
7       FORMAT(///)
        IF(OUTPUT.LT.3) GO TO 101
        WRITE(ICOUT,8)
8       FORMAT('1',T3,72('*'))
        WRITE(ICOUT,2)
        WRITE(ICOUT,3)
        IF(NEWTON) THEN
           IF(JUPDM.EQ.0) THEN
              WRITE(ICOUT,9)
9             FORMAT(T3,'*',2X,'METHOD: NEWTON (NO LINE SEARCH)',
     $        T74,'*')
           ELSEIF(JUPDM.EQ.1) THEN
              WRITE(ICOUT,10)
10            FORMAT(T3,'*',2X,'METHOD: QUASI-NEWTON (NO LINE SEARCH)',
     $        ' USING BROYDEN UPDATE',T74,'*')
           ELSEIF(JUPDM.EQ.2) THEN
              WRITE(ICOUT,11)
11            FORMAT(T3,'*',2X,'METHOD: QUASI-NEWTON (NO LINE SEARCH)',
     $        ' USING LEE AND LEE UPDATE',T74,'*')
           END IF
           WRITE(ICOUT,3)
           IF(OVERCH) THEN
              WRITE(ICOUT,12)
12            FORMAT(T3,'*',2X,'OVERLOW CHECKING IN USE',T74,'*')
           ELSE
              WRITE(ICOUT,13)
13            FORMAT(T3,'*',2X,'OVERFLOW CHECKING NOT IN USE',
     $        T74,'*')
           END IF
           IF(JACTYP.EQ.0) THEN
              IF(NJACCH.GT.0) THEN
                 WRITE(ICOUT,14) NJACCH
14               FORMAT(T3,'*',2X,'ANALYTICAL JACOBIAN USED',
     $           ', CHECKED NUMERICALLY, NJACCH: ',I5,T74,'*')
              ELSE
                 WRITE(ICOUT,15)
15               FORMAT(T3,'*',2X,'ANALYTICAL JACOBIAN USED; NOT',
     $           ' CHECKED',T74,'*')
              END IF
           ELSEIF(JACTYP.EQ.1) THEN
              WRITE(ICOUT,16)
16            FORMAT(T3,'*',2X,'JACOBIAN ESTIMATED USING FORWARD',
     $        ' DIFFERENCES',T74,'*')
           ELSEIF(JACTYP.EQ.2) THEN
              WRITE(ICOUT,17)
17            FORMAT(T3,'*',2X,'JACOBIAN ESTIMATED USING BACKWARD',
     $        ' DIFFERENCES',T74,'*')
           ELSE
              WRITE(ICOUT,18)
18            FORMAT(T3,'*',2X,'JACOBIAN ESTIMATED USING CENTRAL',
     $        ' DIFFERENCES',T74,'*')
           END IF
           WRITE(ICOUT,3)
           WRITE(ICOUT,2)
        ELSE
           IF(LINESR) THEN
              WRITE(ICOUT,3)
              IF(DEUFLH) THEN
                 WRITE(ICOUT,19)
19               FORMAT(T3,'*',2X,'DEUFLHARD RELAXATION FACTOR ',
     $           'INITIALIZATION IN EFFECT',T74,'*')
              ELSE
                 WRITE(ICOUT,20)
20               FORMAT(T3,'*',2X,'DEUFLHARD RELAXATION FACTOR ',
     $           'INITIALIZATION NOT IN EFFECT',T74,'*')
              END IF
           ELSE
            IF(ETAFAC.EQ.ONE) THEN
                 WRITE(ICOUT,21)
21               FORMAT(T3,'*',2X,'METHOD: TRUST REGION USING',
     $           ' SINGLE DOGLEG STEPS',T74,'*')
              ELSE
                 WRITE(ICOUT,22)
22               FORMAT(T3,'*',2X,'METHOD: TRUST REGION USING',
     $           ' DOUBLE DOGLEG STEPS',T74,'*')
              END IF
              WRITE(ICOUT,3)
              IF(CAUCHY) THEN
                 WRITE(ICOUT,23)
23               FORMAT(T3,'*',2X,'INITIAL STEP CONSTRAINED BY',
     $           ' SCALED CAUCHY STEP',T74,'*')
              ELSE
                 WRITE(ICOUT,24)
24               FORMAT(T3,'*',2X,'INITIAL STEP CONSTRAINED BY',
     $           ' SCALED NEWTON STEP',T74,'*')
              END IF
           END IF
           IF(GEOMS) THEN
              WRITE(ICOUT,25)
25            FORMAT(T3,'*',2X,'METHOD: GEOMETRIC SEARCH',
     $        T74,'*')
           ELSE
              WRITE(ICOUT,26)
26            FORMAT(T3,'*',2X,'METHOD: SEARCH BASED ON',
     $        ' SUCCESSIVE MINIMIZATIONS',T74,'*')
           END IF
           IF(OVERCH) THEN
              WRITE(ICOUT,12)
           ELSE
              WRITE(ICOUT,13)
           END IF
           IF(JUPDM.EQ.0) THEN
              WRITE(ICOUT,27)
27            FORMAT(T3,'*',2X,'NO QUASI-NEWTON UPDATE USED',T74,'*')
           END IF
           IF(JUPDM.EQ.1) THEN
              IF(QNUPDM.EQ.0) THEN
                 WRITE(ICOUT,28)
28               FORMAT(T3,'*',2X,'BROYDEN QUASI-NEWTON UPDATE',
     $           ' OF UNFACTORED JACOBIAN',T74,'*')
              ELSE
                 WRITE(ICOUT,29)
29               FORMAT(T3,'*',2X,'BROYDEN QUASI-NEWTON UPDATE',
     $           ' OF FACTORED JACOBIAN',T74,'*')
              END IF
           ELSEIF(JUPDM.EQ.2) THEN
              IF(QNUPDM.EQ.0) THEN
                 WRITE(ICOUT,30)
30               FORMAT(T3,'*',2X,'LEE AND LEE QUASI-NEWTON UPDATE',
     $           ' OF UNFACTORED JACOBIAN',T74,'*')
              ELSE
                 WRITE(ICOUT,31)
31               FORMAT(T3,'*',2X,'LEE AND LEE QUASI-NEWTON UPDATE',
     $           ' OF FACTORED JACOBIAN',T74,'*')
              END IF
           END IF
           IF(JACTYP.EQ.0) THEN
              IF(NJACCH.GT.0) THEN
                 WRITE(ICOUT,14)
              ELSE
                 WRITE(ICOUT,15)
              END IF
           ELSEIF(JACTYP.EQ.1) THEN
              WRITE(ICOUT,16)
           ELSEIF(JACTYP.EQ.2) THEN
              WRITE(ICOUT,17)
           ELSE
              WRITE(ICOUT,18)
           END IF
           IF(.NOT.LINESR) THEN
              WRITE(ICOUT,3)
              IF(TRUPDM.EQ.0.AND.JUPDM.GT.0) THEN
                 WRITE(ICOUT,32)
32               FORMAT(T3,'*',2X,'TRUST REGION UPDATED USING',
     $           ' POWELL STRATEGY',T74,'*')
              ELSE
                 WRITE(ICOUT,33)
33               FORMAT(T3,'*',2X,'TRUST REGION UPDATED USING',
     $           ' DENNIS AND SCHNABEL STRATEGY',T74,'*')
              END IF
           END IF
           WRITE(ICOUT,3)
           WRITE(ICOUT,2)
           WRITE(ICOUT,3)
           IF(ITSCLF.NE.0) THEN
              WRITE(ICOUT,34) ITSCLF
34            FORMAT(T3,'*',2X,'ADAPTIVE FUNCTION SCALING STARTED AT',
     $        ' ITERATION: ..........',I6,T74,'*')
              WRITE(ICOUT,3)
           END IF
           IF(ITSCLX.NE.0) THEN
              WRITE(ICOUT,35) ITSCLX
35            FORMAT(T3,'*',2X,'ADAPTIVE VARIABLE SCALING STARTED AT',
     $        ' ITERATION: ..........',I6,T74,'*')
              WRITE(ICOUT,3)
           END IF
           IF(LINESR) THEN
              IF(JUPDM.EQ.0) THEN
                 WRITE(ICOUT,36) MAXNS
36               FORMAT(T3,'*',2X,'MAXIMUM NUMBER OF STEPS IN LINE',
     $           ' SEARCH, MAXNS: ...........',I6,T74,'*')
              ELSE
                 WRITE(ICOUT,37) MAXNS
37               FORMAT(T3,'*',2X,'MAXIMUM NUMBER OF NEWTON LINE',
     $           ' SEARCH STEPS, MAXNS: .......',I6,T74,'*')
                 WRITE(ICOUT,38) MAXQNS
38               FORMAT(T3,'*',2X,'MAXIMUM NUMBER OF QUASI-NEWTON',
     $           ' LINE SEARCH STEPS, MAXQNS: ',I6,T74,'*')
              END IF
           ELSE
              IF(JUPDM.EQ.0) THEN
                 WRITE(ICOUT,39) MAXNS
39               FORMAT(T3,'*',2X,'MAXIMUM NUMBER OF TRUST REGION',
     $         ' UPDATES, MAXNS: ','...........',I6,T74,'*')
              ELSE
                 WRITE(ICOUT,40) MAXNS
40               FORMAT(T3,'*',2X,'MAXIMUM NO. OF NEWTON TRUST',
     $           ' REGION UPDATES, MAXNS: .......',I6,T74,'*')
                 WRITE(ICOUT,41) MAXQNS
41               FORMAT(T3,'*',2X,'MAXIMUM NO. OF QUASI-NEWTON',
     $           ' TRUST REGION UPDATES, MAXQNS: ',I6,T74,'*')
              END IF
           END IF
           IF(NARMIJ.LT.MAXIT) THEN
              WRITE(ICOUT,3)
              WRITE(ICOUT,42) MGLL
42            FORMAT(T3,'*',2X,'NUMBER OF OBJECTIVE FUNCTION',
     $        ' VALUES COMPARED, MGLL: ','......',I6,T74,'*')
           END IF
           IF(JUPDM.GT.0) THEN
              IF(NARMIJ.EQ.MAXIT) WRITE(ICOUT,3)
              WRITE(ICOUT,43) MINQNS
43            FORMAT(T3,'*',2X,'MINIMUM NUMBER OF STEPS BETWEEN',
     $        ' JACOBIAN UPDATES, MINQNS: ',I6,T74,'*')
              WRITE(ICOUT,44) NINITN
44            FORMAT(T3,'*',2X,'NUMBER OF NON-QUASI-NEWTON',
     $        ' STEPS AT START, NINITN: .......',I6,T74,'*')
           END IF
           WRITE(ICOUT,45) NARMIJ
45         FORMAT(T3,'*',2X,'NUMBER OF ARMIJO STEPS AT START,',
     $     ' NARMIJ:',' .................',I6,T74,'*')
        END IF
        WRITE(ICOUT,3)
        IF(STOPCR.EQ.3) THEN
           WRITE(ICOUT,46) STOPCR
46         FORMAT(T3,'*',2X,'FUNCTION AND STEP SIZE STOPPING'
     $     ' CRITERIA, STOPCR: ........',I6,T74,'*')
        ELSEIF(STOPCR.EQ.12) THEN
           WRITE(ICOUT,47) STOPCR
47         FORMAT(T3,'*',2X,'FUNCTION OR STEP SIZE STOPPING',
     $     ' CRITERIA, STOPCR: .........',I6,T74,'*')
        ELSEIF(STOPCR.EQ.1) THEN
           WRITE(ICOUT,48) STOPCR
48         FORMAT(T3,'*',2X,'STEP SIZE STOPPING CRITERION,',
     $     ' STOPCR: ','....................',I6,T74,'*')
        ELSE
           WRITE(ICOUT,49) STOPCR
49         FORMAT(T3,'*',2X,'FUNCTION STOPPING CRITERION,',
     $     ' STOPCR: ','.....................',I6,T74,'*')
        END IF
        IF(.NOT.NEWTON) THEN
           WRITE(ICOUT,3)
           IF(ACPTCR.EQ.12) THEN
              WRITE(ICOUT,50) ACPTCR
50            FORMAT(T3,'*',2X,'FUNCTION AND STEP SIZE ACCEPTANCE'
     $        ' CRITERIA, ACPTCR: ......',I6,T74,'*')
           ELSEIF(ACPTCR.EQ.2) THEN
              WRITE(ICOUT,51) ACPTCR
51            FORMAT(T3,'*',2X,'STEP SIZE ACCEPTANCE CRITERION, ',
     $        'ACPTCR: ..................',I6,T74,'*')
           ELSE
              WRITE(ICOUT,52) ACPTCR
52            FORMAT(T3,'*',2X,'FUNCTION ACCEPTANCE CRITERION, ',
     $        'ACPTCR: ...................',I6,T74,'*')
           END IF
           IF(CONTYP.NE.0) THEN
              WRITE(ICOUT,3)
              WRITE(ICOUT,53) CONTYP
53            FORMAT(T3,'*',2X,'CONSTRAINTS IN USE, CONTYP: ',
     $        '..............................',I6,T74,'*')
           END IF
        END IF
        WRITE(ICOUT,3)
        WRITE(ICOUT,2)
        WRITE(ICOUT,3)
      WRITE(ICOUT,54) EPSMCH
54      FORMAT(T3,'*',2X,'ESTIMATED MACHINE EPSILON, EPSMCH:',
     $  ' ...................',1PD10.3,T74,'*')
        WRITE(ICOUT,3)
        WRITE(ICOUT,55) MSTPF
55      FORMAT(T3,'*',2X,'FACTOR TO ESTABLISH MAXIMUM STEP SIZE',
     $  ', MSTPF: ........',1PD10.3,T74,'*')
        WRITE(ICOUT,56) MAXSTP
56      FORMAT(T3,'*',2X,'CALCULATED MAXIMUM STEP SIZE, MAXSTP:',
     $  ' ................',1PD10.3,T74,'*')
        IF(.NOT.LINESR) THEN
         IF(DELTA.LT.ZERO) THEN
              WRITE(ICOUT,3)
              WRITE(ICOUT,57)
57            FORMAT(T3,'*',2X,'INITIAL TRUST REGION NOT PROVIDED',
     $        T74,'*')
           ELSE
              WRITE(ICOUT,3)
              WRITE(ICOUT,58) DELTA
58            FORMAT(T3,'*',2X,'INITIAL TRUST REGION SIZE, DELTA:',
     $        ' ....................',1PD10.3,T74,'*')
           END IF
         IF(ETAFAC.LT.ONE) THEN
              WRITE(ICOUT,59) ETAFAC
59            FORMAT(T3,'*',2X,'FACTOR TO SET DIRECTION OF',
     $        ' TRUST REGION STEP, ETAFAC: ... ',F7.4,T74,'*')
           END IF
           WRITE(ICOUT,60) DELFAC
60         FORMAT(T3,'*',2X,'TRUST REGION UPDATING FACTOR, DELFAC: ',
     $     '................',1PD10.3,T74,'*')
        END IF
        IF(.NOT.NEWTON) THEN
           WRITE(ICOUT,3)
           WRITE(ICOUT,61) ALPHA
61         FORMAT(T3,'*',2X,'FACTOR IN OBJECTIVE FUNCTION',
     $     ' COMPARISON, ALPHA: ......',1PD10.3,T74,'*')
           IF(LINESR.AND.(.NOT.NEWTON)) THEN
              WRITE(ICOUT,3)
              WRITE(ICOUT,62) SIGMA
62            FORMAT(T3,'*',2X,'REDUCTION FACTOR FOR RELAXATION'
     $        ' FACTOR, SIGMA: .......'1PD10.3,T74,'*')
           END IF
           IF(JUPDM.NE.0) THEN
              WRITE(ICOUT,3)
              WRITE(ICOUT,63) RATIOF
63            FORMAT(T3,'*',2X,'REDUCTION REQUIRED IN OBJ. FUNCTION',
     $        ' FOR QN STEP, RATIOF:  ',F7.4,T74,'*')
           END IF
           IF(JUPDM.EQ.2) THEN
              WRITE(ICOUT,3)
              WRITE(ICOUT,64) OMEGA
64            FORMAT(T3,'*',2X,'FACTOR IN LEE AND LEE UPDATE, OMEGA:',
     $        ' .....................',F7.4,T74,'*')
           END IF
        END IF
        WRITE(ICOUT,3)
        IF(STOPCR.NE.2) THEN
           WRITE(ICOUT,65) STPTOL
65         FORMAT(T3,'*',2X,'STOPPING TOLERANCE FOR STEP SIZE,'
     $     ' STPTOL: ............',1PD10.3,T74,'*')
           WRITE(ICOUT,66) NSTTOL
66         FORMAT(T3,'*',2X,'STOPPING TOLERANCE FOR NEWTON STEP,'
     $     ' NSTTOL: ..........',1PD10.3,T74,'*')
        END IF
        IF(STOPCR.NE.1) THEN
        WRITE(ICOUT,67) FTOL
67      FORMAT(T3,'*',2X,'STOPPING TOLERANCE FOR OBJECTIVE'
     $  ' FUNCTION, FTOL: .....',1PD10.3,T74,'*')
        END IF
      IF(LINESR.AND.(.NOT.NEWTON).AND.LAM0.LT.ONE) THEN
           WRITE(ICOUT,3)
           WRITE(ICOUT,68) LAM0
68         FORMAT(T3,'*',2X,'INITIAL LAMBDA IN LINE SEARCH,',
     $     ' LAM0: .................',1PD10.3,T74,'*')
        END IF
        IF(CONTYP.GT.0) THEN
           WRITE(ICOUT,3)
           WRITE(ICOUT,69) CONFAC
69         FORMAT(T3,'*'2X,'FACTOR TO ENSURE STEP WITHIN',
     $     ' CONSTRAINTS, CONFAC: ........',F7.4,T74,'*')
        END IF
        WRITE(ICOUT,3)
        WRITE(ICOUT,2)
        WRITE(ICOUT,3)
        WRITE(ICOUT,70)
70      FORMAT(T3,'*',2X,'SCALING FACTORS',T74,'*')
        WRITE(ICOUT,3)
        WRITE(ICOUT,71)
71      FORMAT(T3,'*',6X,'COMPONENT VALUES',24X,'FUNCTION VALUES',
     $  T74,'*')
        WRITE(ICOUT,3)
        DO 100 I=1,N
           WRITE(ICOUT,72) I,SCALEX(I),I,SCALEF(I)
72         FORMAT(T3,'*',2X,'SCALEX(',I3,') = ',1PD10.3,15X,
     $     'SCALEF(',I3,') = ',1PD10.3,T74,'*')
100     CONTINUE
        IF(CONTYP.GT.0) THEN
           WRITE(ICOUT,3)
           WRITE(ICOUT,2)
           WRITE(ICOUT,3)
           WRITE(ICOUT,73)
73         FORMAT(T3,'*',2X,'LOWER AND UPPER BOUNDS',T74,'*')
           WRITE(ICOUT,3)
           WRITE(ICOUT,74)
74         FORMAT(T3,'*',8X,'LOWER BOUNDS',27X,'UPPER BOUNDS',
     $     T74,'*')
           WRITE(ICOUT,3)
           DO 200 I=1,N
              WRITE(ICOUT,75) I,BOUNDL(I),I,BOUNDU(I)
75            FORMAT(T3,'*',2X,'BOUNDL(',I3,') = ',1PD10.3,15X,
     $        'BOUNDU(',I3,') = ',1PD10.3,T74,'*')
200        CONTINUE
        END IF
        WRITE(ICOUT,3)
101     CONTINUE
        IF(OUTPUT.EQ.2) WRITE(ICOUT,2)
        WRITE(ICOUT,2)
        WRITE(ICOUT,3)
        WRITE(ICOUT,76)
76      FORMAT(T3,'*',4X,'INITIAL ESTIMATES',16X,
     $  'INITIAL FUNCTION VALUES',T74,'*')
        WRITE(ICOUT,3)
        DO 300 I=1,N
           WRITE(ICOUT,77) I,XC(I),I,FVECC(I)
77         FORMAT(T3,'*',2X,'X(',I3,') = ',1PD12.3,15X,
     $     'F(',I3,') = ',1PD12.3,T74,'*')
300     CONTINUE
        WRITE(ICOUT,3)
        WRITE(ICOUT,78) FCNOLD
78      FORMAT(T3,'*',2X,'INITIAL OBJECTIVE FUNCTION VALUE = ',
     $  1PD10.3,T74,'*')
        WRITE(ICOUT,3)
        WRITE(ICOUT,2)
        WRITE(ICOUT,2)
        IF(OUTPUT.LT.3) RETURN
        WRITE(ICOUT,79)
79      FORMAT('1',//,T3,24X,23('*'))
        WRITE(ICOUT,80)
80      FORMAT(T3,24X,23('*'))
        WRITE(ICOUT,81)
81      FORMAT(T3,24X,'*',21X,'*')
        WRITE(ICOUT,82)
82      FORMAT(T3,24X,'*  UPDATED ESTIMATES  *')
        WRITE(ICOUT,81)
        WRITE(ICOUT,80)
        WRITE(ICOUT,80)
        WRITE(ICOUT,83)
83      FORMAT(//)
        RETURN
C
C       LAST CARD OF SUBROUTINE TITLE.
C
      END
        SUBROUTINE TRSTUP(GEOMS ,NEWTKN,OVERCH,OVERFL,QRSING,
     $                    SCLFCH,SCLXCH,ACPCOD,ACPSTR,ACPTCR,
     $                    CONTYP,ISEJAC,JUPDM ,MAXEXP,MGLL  ,
     $                    MNEW  ,N     ,NARMIJ,NFUNC ,NOTRST,
     $                    NUNIT ,OUTPUT,QNUPDM,RETCOD,TRUPDM,
     $                    ALPHA ,CONFAC,DELFAC,DELSTR,DELTA ,
     $                    EPSMCH,FCNMAX,FCNNEW,FCNOLD,FCNPRE,
     $                    MAXSTP,NEWLEN,NEWMAX,POWTAU,RELLEN,
     $                    STPTOL,A     ,ASTORE,BOUNDL,BOUNDU,
     $                    DELF  ,FPLPRE,FTRACK,FVEC  ,FVECC ,
     $                    HHPI  ,JAC   ,RDIAG ,RHS   ,S     ,
     $                    SBAR  ,SCALEF,SCALEX,STRACK,WV3   ,
     $                    XC    ,XPLPRE,XPLUS ,FVECEV)
C
C       FEB. 28, 1992
C
C       THIS SUBROUTINE CHECKS FOR ACCEPTANCE OF A TRUST REGION
C       STEP GENERATED BY THE DOUBLE DOGLEG METHOD.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION   JAC(N,N)  ,MAXSTP     ,NEWLEN   ,NEWMAX
        INTEGER            ACPCOD    ,ACPSTR     ,ACPTCR   ,CONTYP   ,
     $                     OUTPUT    ,QNUPDM     ,RETCOD   ,TRUPDM
        DIMENSION          A(N,N)    ,ASTORE(N,N),BOUNDL(N),BOUNDU(N),
     $                     DELF(N)   ,FTRACK(0:MGLL-1)     ,FPLPRE(N),
     $                     FVEC(N)   ,FVECC(N)   ,HHPI(N)  ,RDIAG(N) ,
     $                     RHS(N)    ,S(N)       ,SBAR(N)  ,SCALEF(N),
     $                     SCALEX(N) ,STRACK(0:MGLL-1)     ,WV3(N)   ,
     $                     XC(N)     ,XPLPRE(N)  ,XPLUS(N)
        LOGICAL            CONVIO    ,GEOMS      ,NEWTKN   ,OVERCH   ,
     $                     OVERFL    ,QRSING     ,SCLFCH   ,SCLXCH   ,
     $                     WRNSUP
        COMMON/NNES_2/WRNSUP
        EXTERNAL FVECEV
      DATA ZERO,PT5,THREEQ,ONE,ONEPT1,TWO,TEN
     $  /0.0D0,0.5D0,0.75D0,1.0D0,1.1D0,2.0D0,10.0D0/
C
C       NOTE: ACCEPTANCE CODE, ACPCOD, IS 0 ON ENTRANCE TO TRSTUP
C
        CONVIO=.FALSE.
        OVERFL=.FALSE.
C
        IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
1          FORMAT(T3,'*',T74,'*')
         WRITE(ICOUT,1)
         IF(.NOT.SCLFCH.AND.(.NOT.SCLXCH)) THEN
            WRITE(ICOUT,2)
2             FORMAT(T3,'*',4X,'TRUST REGION UPDATING',T74,'*')
         ELSE
            WRITE(ICOUT,3)
3             FORMAT(T3,'*',4X,'TRUST REGION UPDATING (ALL X''','S',
     $        ' AND F''','S IN UNSCALED UNITS)',T74,'*')
         END IF
         WRITE(ICOUT,1)
        END IF
C
C       CHECK TO MAKE SURE "S" IS A DESCENT DIRECTION - FIND
C       DIRECTIONAL DERIVATIVE AT CURRENT XC USING S GENERATED
C       BY DOGLEG SUBROUTINE.
C
      CALL INNERP(OVERCH,OVERFL,MAXEXP,N     ,N     ,N     ,NUNIT ,
     $              OUTPUT,DELFTS,DELF  ,S     )
        IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
         WRITE(ICOUT,4) DELFTS
4          FORMAT(T3,'*',7X,'INNER PRODUCT OF DELF AND S, DELFTS: '
     $     ,'........',1PD13.4,T74,'*')
        END IF
      IF(DELFTS.GT.ZERO) THEN
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
            WRITE(ICOUT,5)
5             FORMAT(T3,'*',7X,'DIRECTIONAL DERIVATIVE POSITIVE',
     $        '; SEARCH DIRECTION REVERSED',T74,'*')
           END IF
           DO 100 I=1,N
              S(I)=-S(I)
100        CONTINUE
        END IF
C
C       FIND MAXIMUM OBJECTIVE FUNCTION VALUE AND MAXIMIUM STEP
C       LENGTH FOR NONMONOTONIC SEARCH.  THIS HAS TO BE DONE ONLY
C       ONCE DURING EACH ITERATION (WHERE NOTRST=1).
C
        IF(NOTRST.EQ.1) THEN
           NEWMAX=NEWLEN
           FCNMAX=FCNOLD
           IF(ISEJAC.GT.NARMIJ) THEN
              IF(ISEJAC.LT.NARMIJ+MGLL) THEN
                 DO 200 J=1,MNEW
                    FCNMAX=MAX(FCNMAX,FTRACK(J-1))
                    NEWMAX=MAX(NEWMAX,STRACK(J-1))
200              CONTINUE
              ELSE
                 DO 300 J=0,MNEW
                    FCNMAX=MAX(FCNMAX,FTRACK(J))
                    NEWMAX=MAX(NEWMAX,STRACK(J))
300              CONTINUE
              END IF
           END IF
        END IF
C
C       TEST TRIAL POINT - FIND XPLUS AND TEST FOR CONSTRAINT
C       VIOLATIONS IF CONTYP DOES NOT EQUAL 0.
C
        DO 400 I=1,N
         WV3(I)=-ONE
C
C          WV3 IS A MARKER FOR "VIOLATORS" - IT CHANGES TO 1 OR 2.
C
           XPLUS(I)=XC(I)+S(I)
           IF(CONTYP.GT.0) THEN
              IF(XPLUS(I).LT.BOUNDL(I)) THEN
                 CONVIO=.TRUE.
          WV3(I)=ONE
              ELSEIF(XPLUS(I).GT.BOUNDU(I)) THEN
                 CONVIO=.TRUE.
          WV3(I)=TWO
              END IF
           END IF
400     CONTINUE
C
C       IF CONSTRAINT IS VIOLATED ...
C
        IF(CONVIO) THEN
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
            WRITE(ICOUT,6)
6             FORMAT(T3,'*',7X,'CONSTRAINT VIOLATED',T74,'*',/T3,'*',
     $        T74,'*',/T3,'*',10X,'TRIAL ESTIMATES (VIOLATIONS MARKED)'
     $        ,T74,'*')
              WRITE(ICOUT,1)
              DO 500 I=1,N
          IF(WV3(I).GT.ZERO) THEN
                    WRITE(ICOUT,7) I,XPLUS(I)
7                   FORMAT(T3,'*',13X,'XPLUS(',I3,') = ',1PD12.3,
     $              2X,'*',T74,'*')
                 ELSE
                    WRITE(ICOUT,8) I,XPLUS(I)
8                   FORMAT(T3,'*',13X,'XPLUS(',I3,') = ',1PD12.3,
     $              T74,'*')
                 END IF
500           CONTINUE
           END IF
C
C          FIND STEP WITHIN CONSTRAINED REGION.
C
C          FIND THE RATIO OF THE DISTANCE FROM THE (I)TH
C          COMPONENT TO ITS CONSTRAINT TO THE LENGTH OF THE
C          PROPOSED STEP, XPLUS(I)-XC(I).  MULTIPLY THIS BY
C          CONFAC (DEFAULT 0.95) TO ENSURE THE NEW STEP STAYS
C          WITHIN THE ACCEPTABLE REGION UNLESS XC IS CLOSE TO
C          THE BOUNDARY (RATIO <= 1/2).  IN SUCH CASES A FACTOR
C          OF 0.5*CONFAC IS USED.
C
C          NOTE: ONLY THE VIOLATING COMPONENTS ARE REDUCED.
C
         RATIOM=ONE
C
C          RATIOM STORES THE MINIMUM VALUE OF RATIO.
C
           DO 600 I=1,N
            IF(WV3(I).EQ.ONE) THEN
                 RATIO=(BOUNDL(I)-XC(I))/S(I)
            ELSEIF(WV3(I).EQ.TWO) THEN
                 RATIO=(BOUNDU(I)-XC(I))/S(I)
              END IF
            IF(WV3(I).GT.ZERO) THEN
C
C                NOTE: RATIO IS STORED IN WV3 FOR OUTPUT ONLY.
C
                 WV3(I)=RATIO
C
                 RATIOM=MIN(RATIOM,RATIO)
          IF(RATIO.GT.PT5) THEN
                    XPLUS(I)=XC(I)+CONFAC*RATIO*S(I)
                 ELSE
C
C                   WITHIN BUFFER ZONE.
C
             XPLUS(I)=XC(I)+CONFAC*RATIO*S(I)/TWO
                 END IF
                 S(I)=XPLUS(I)-XC(I)
              END IF
600        CONTINUE
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
              WRITE(ICOUT,1)
              WRITE(ICOUT,9)
9             FORMAT(T3,'*',7X,'NEW S AND XPLUS VECTORS',
     $        ' (WITH RATIOS FOR VIOLATIONS)',T74,'*')
              WRITE(ICOUT,1)
              WRITE(ICOUT,10)
10            FORMAT(T3,'*',7X,'NOTE: RATIOS ARE RATIO OF',
     $        ' LENGTH TO BOUNDARY FROM CURRENT',T74,'*')
              WRITE(ICOUT,11)
11            FORMAT(T3,'*',7X,'X VECTOR TO MAGNITUDE OF',
     $        ' CORRESPONDING PROPOSED STEP',T74,'*')
              WRITE(ICOUT,1)
              DO 700 I=1,N
          IF(WV3(I).LT.ZERO) THEN
                    WRITE(ICOUT,12) I,S(I),I,XPLUS(I)
12                  FORMAT(T3,'*',7X,'S(',I3,') = ',1PD12.3,4X,
     $              'XPLUS(',I3,') = ',1PD12.3,T74,'*')
                 ELSE
                    WRITE(ICOUT,13) I,S(I),I,XPLUS(I),WV3(I)
13                  FORMAT(T3,'*',7X,'S(',I3,') = ',1PD12.3,4X,
     $              'XPLUS(',I3,') = ',1PD12.3,1X,1PD11.3,T74,'*')
                 END IF
700           CONTINUE
              WRITE(ICOUT,1)
              WRITE(ICOUT,14) RATIOM
14            FORMAT(T3,'*',7X,'MINIMUM OF RATIOS, RATIOM: ',
     $        1PD12.3,T74,'*')
           END IF
C
C          THE NEW POINT, XPLUS, IS NOT NECESSARILY IN A DESCENT
C          DIRECTION.  CHECK DIRECTIONAL DERIVATIVE FOR MODIFIED
C          STEP, DLFTSM.
C
         CALL INNERP(OVERCH,OVERFL,MAXEXP,N     ,N     ,N     ,
     $                 NUNIT ,OUTPUT,DLFTSM,DELF  ,S     )
           IF(OUTPUT.GT.3) THEN
              WRITE(ICOUT,1)
              WRITE(ICOUT,15) DLFTSM
15            FORMAT(T3,'*',7X,'INNER PRODUCT OF DELF AND MODIFIED S',
     $        ', DLFTSM: ',1PD12.3,T74,'*')
           END IF
C
C          IF DLFTSM IS POSITIVE REDUCE TRUST REGION.  IF NOT, TEST
C          NEW POINT.
C
         IF(DLFTSM.GT.ZERO) THEN
              DELTA=CONFAC*RATIOM*DELTA
              RETCOD=8
              RETURN
           END IF
        END IF
C
C       CONSTRAINTS NOT (OR NO LONGER) VIOLATED - TEST NEW POINT.
C
        CALL FVECEV(OVERFL,N,FVEC,XPLUS)
        NFUNC=NFUNC+1
C
C       IF OVERFLOW AT NEW POINT REDUCE TRUST REGION AND RETURN.
C
        IF(OVERFL) THEN
C
C          IF THE OVERFLOW COMES AS A RESULT OF INCREASING DELTA
C          WITHIN THE CURRENT ITERATION (IMPLYING DELSTR IS POSITIVE)
C          AND DIVIDING DELTA BY 10 WOULD PRODUCE A DELTA WHICH
C          IS SMALLER THAN THAT AT THE STORED POINT, THEN USE
C          STORED POINT AS THE UPDATED RESULT.
C
         IF(DELSTR.GT.DELTA/TEN) THEN
            CALL MATCOP(N,N,1,1,N,1,XPLPRE,XPLUS)
            CALL MATCOP(N,N,1,1,N,1,FPLPRE,FVEC)
              ACPCOD=ACPSTR
              DELTA=DELSTR
              FCNNEW=FCNPRE
              RETCOD=1
           ELSE
            DELTA=DELTA/TEN
              RETCOD=9
           END IF
           RETURN
        END IF
C
C       NO OVERFLOW IN RESIDUAL VECTOR.
C
        IF(OUTPUT.GT.3) THEN
           WRITE(ICOUT,1)
           WRITE(ICOUT,16)
16         FORMAT(T3,'*',12X,'TRIAL ESTIMATES',18X,
     $     'FUNCTION VALUES',T74,'*')
           WRITE(ICOUT,1)
           DO 800 I=1,N
              WRITE(ICOUT,17) I,XPLUS(I),I,FVEC(I)
17            FORMAT(T3,'*',7X,'XPLUS(',I3,') = ',1PD12.3,9X,
     $        'FVEC(',I3,') = ',1PD12.3,T74,'*')
800        CONTINUE
        END IF
C
C       IF NO OVERFLOW WITHIN RESIDUAL VECTOR FIND OBJECTIVE
C       FUNCTION.
C
        CALL FCNEVL(OVERFL,MAXEXP,N     ,NUNIT ,OUTPUT,
     $              EPSMCH,FCNNEW,FVEC  ,SCALEF,WV3   )
C
C       IF OVERFLOW IN OBJECTIVE FUNCTION EVALUATION REDUCE
C       TRUST REGION AND RETURN.
C
        IF(OVERFL) THEN
C
C          IF THE OVERFLOW COMES AS A RESULT OF INCREASING DELTA
C          WITHIN THE CURRENT ITERATION (SO THAT DELSTR IS POSITIVE)
C          AND DIVIDING DELTA BY 10 WOULD PRODUCE A DELTA WHICH
C          IS SMALLER THAN THAT AT THE STORED POINT THEN USE
C          STORED POINT AS THE UPDATED RESULT.
C
         IF(DELSTR.GT.DELTA/TEN) THEN
            CALL MATCOP(N,N,1,1,N,1,XPLPRE,XPLUS)
            CALL MATCOP(N,N,1,1,N,1,FPLPRE,FVEC)
              ACPCOD=ACPSTR
              DELTA=DELSTR
              FCNNEW=FCNPRE
              RETCOD=2
           ELSE
            DELTA=DELTA/TEN
              RETCOD=10
           END IF
           RETURN
        ELSE
C
C          NO OVERFLOW AT TRIAL POINT - COMPARE OBJECTIVE FUNCTION
C          TO FCNMAX.
C
           IF(OUTPUT.GT.3) THEN
            WRITE(ICOUT,1)
            IF(.NOT.SCLFCH) THEN
          WRITE(ICOUT,18) FCNNEW
18               FORMAT(T3,'*',7X,'OBJECTIVE FUNCTION AT XPLUS,',
     $           ' FCNNEW: .........',1PD12.4,T74,'*')
            ELSE
          WRITE(ICOUT,19) FCNNEW
19               FORMAT(T3,'*',7X,'SCALED OBJECTIVE FUNCTION AT XPLUS',
     $           ', FCNNEW: ..',1PD12.4,T74,'*')
            END IF
            WRITE(ICOUT,20) FCNMAX+ALPHA*DELFTS
20            FORMAT(T3,'*',7X,'COMPARE TO FCNMAX+ALPHA*DELFTS: ',
     $        14('.'),1PD12.4,T74,'*')
           END IF
        END IF
C
C       IF ACPTCR=12 CHECK SECOND DEUFLHARD STEP ACCEPTANCE TEST
C       BY FINDING 2-NORM OF SBAR.  THERE ARE FOUR POSSIBILITIES
C       DEPENDING ON WHETHER THE JACOBIAN IS OR IS NOT SINGULAR
C       AND WHETHER QNUPDM IS 0 OR 1.
C
        IF(ACPTCR.EQ.12) THEN
           IF(QRSING) THEN
C
C             FORM -J^F AS RIGHT HAND SIDE - METHOD DEPENDS ON
C             WHETHER QNUPDM EQUALS 0 OR 1 (UNFACTORED OR FACTORED).
C
              IF(QNUPDM.EQ.0) THEN
C
C                UNSCALED JACOBIAN IS IN MATRIX JAC.
C
                 DO 900 I=1,N
             WV3(I)=-FVEC(I)*SCALEF(I)*SCALEF(I)
900              CONTINUE
          CALL ATBMUL(N,N,1,1,N,N,JAC,WV3,RHS)
C
            ELSE
C
C                R IN UPPER TRIANGLE OF A PLUS RDIAG AND Q^ IN JAC
C                FROM QR DECOMPOSITION OF SCALED JACOBIAN.
C
                 DO 1000 I=1,N
             SUM=ZERO
                    DO 1100 J=1,N
                       SUM=SUM-JAC(I,J)*FVEC(J)*SCALEF(J)
1100                CONTINUE
             WV3(I)=SUM
1000             CONTINUE
                 RHS(1)=RDIAG(1)*WV3(1)
                 DO 1200 J=2,N
             SUM=ZERO
                    DO 1300 I=1,J-1
                       SUM=SUM+A(I,J)*WV3(I)
1300                CONTINUE
                    RHS(J)=SUM+RDIAG(J)*WV3(J)
1200             CONTINUE
              END IF
              CALL CHSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                    OUTPUT,A     ,RHS   ,SBAR  ,WV3   )
         ELSE
C
C             RIGHT HAND SIDE IS -FVEC.
C
              IF(QNUPDM.EQ.0.OR.JUPDM.EQ.0) THEN
C
C                QR DECOMPOSITION OF SCALED JACOBIAN STORED IN
C                ASTORE.
C
                 DO 1400 I=1,N
                    SBAR(I)=-FVEC(I)*SCALEF(I)
1400             CONTINUE
                 CALL QRSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                       OUTPUT,ASTORE,HHPI  ,RDIAG ,SBAR  )
              ELSE
C
C                SET UP RIGHT HAND SIDE - MULTIPLY -FVEC BY Q^
C                (STORED IN JAC).  RHS IS A WORK VECTOR ONLY HERE.
C
                 DO 1500 I=1,N
                    WV3(I)=-FVEC(I)*SCALEF(I)
1500             CONTINUE
          CALL AVMUL(N,N,N,N,JAC,WV3,SBAR)
          CALL RSOLV(OVERCH,OVERFL,MAXEXP,N     ,NUNIT ,
     $                      OUTPUT,A     ,RDIAG ,SBAR  )
              END IF
           END IF
C
C          NORM OF (SCALED) SBAR IS NEEDED FOR SECOND ACCEPTANCE TEST.
C
         DO 1600 I=1,N
            WV3(I)=SCALEX(I)*SBAR(I)
1600       CONTINUE
         CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,SBRNRM,WV3)
C
           IF(OUTPUT.GT.4) THEN
            WRITE(ICOUT,1)
            IF(.NOT.SCLXCH) THEN
          WRITE(ICOUT,21)
21               FORMAT(T3,'*',10X,'DEUFLHARD SBAR VECTOR',T74,'*')
          WRITE(ICOUT,1)
          DO 1700 I=1,N
             WRITE(ICOUT,22) I,SBAR(I)
22                  FORMAT(T3,'*',10X,'SBAR(',I3,') = ',
     $              1PD12.3,T74,'*')
1700             CONTINUE
            ELSE
          WRITE(ICOUT,23)
23               FORMAT(T3,'*',10X,'DEUFLHARD SBAR VECTOR',14X,
     $           'IN SCALED X UNITS',T74,'*')
          WRITE(ICOUT,1)
          DO 1800 I=1,N
             WRITE(ICOUT,24) I,SBAR(I),I,SCALEX(I)*SBAR(I)
24                  FORMAT(T3,'*',10X,'SBAR(',I3,') = ',1PD12.3,
     $              8X,'SBAR(',I3,') = ',1PD12.3,T74,'*')
1800             CONTINUE
            END IF
         END IF
           IF(OUTPUT.GT.3) THEN
            WRITE(ICOUT,1)
            IF(.NOT.SCLXCH) THEN
          WRITE(ICOUT,25) SBRNRM
25               FORMAT(T3,'*',10X,'VALUE OF SBRNRM',
     $           ' AT XPLUS: .................'1PD12.4,T74,'*')
            ELSE
          WRITE(ICOUT,26) SBRNRM
26               FORMAT(T3,'*',10X,'VALUE OF SCALED SBRNRM',
     $           ' AT XPLUS: ..........'1PD12.4,T74,'*')
            END IF
            WRITE(ICOUT,27) NEWMAX
27            FORMAT(T3,'*',10X,'NEWMAX: ',35('.'),1PD12.4,T74,'*')
           END IF
           IF(SBRNRM.LT.NEWMAX) ACPCOD=2
C
C          FUNCTION VALUE ACCEPTANCE IS ALSO CHECKED REGARDLESS
C          OF WHETHER SECOND STEP ACCEPTANCE CRITERION WAS MET.
C
        END IF
C
C       ESTABLISH DELTAF FOR USE IN COMPARISON TO PREDICTED
C       CHANGE IN OBJECTIVE FUNCTION, DELFPR, LATER.
C
        DELTAF=FCNNEW-FCNOLD
        IF(FCNNEW.GE.FCNMAX+ALPHA*DELFTS) THEN
C
C          FAILURE OF FIRST STEP ACCEPTANCE TEST. TEST LENGTH OF
C          STEP TO ENSURE PROGRESS IS STILL BEING MADE.
C
         RELLEN=ZERO
         DO 1900 I=1,N
              RELLEN=MAX(RELLEN,ABS(S(I))/
     $               MAX((ABS(XPLUS(I))),ONE/SCALEX(I)))
1900       CONTINUE
           IF(RELLEN.LT.STPTOL) THEN
C
C             NO PROGRESS BEING MADE - RETCOD = 7 STOPS PROGRAM.
C
            CALL MATCOP(N,N,1,1,N,1,XC,XPLUS)
              RETCOD=7
              RETURN
           ELSE
C
C             FAILURE OF STEP BY OBJECTIVE FUNCTION CRITERION.
C             ESTABLISH A NEW DELTA FROM EITHER SIMPLE DIVISION
C             BY DELFAC OR BY FINDING THE MINIMUM OF A QUADRATIC
C             MODEL.
C
              IF(GEOMS) THEN
                 DELTA=DELTA/DELFAC
              ELSE
C
C                FIRST FIND LENGTH OF TRUST REGION STEP.
C
                 DO 2000 I=1,N
                    WV3(I)=S(I)*SCALEX(I)
2000             CONTINUE
          CALL TWONRM(OVERFL,MAXEXP,N,EPSMCH,STPLEN,WV3)
C
                 CALL QMIN(NUNIT,OUTPUT,DELFTS,DELTA,DELTAF,STPLEN)
C
              END IF
              IF(DELTA.LT.DELSTR) THEN
C
C                IF DELTA HAS BEEN INCREASED AT THIS ITERATION
C                AND THE DELTA FROM QMIN IS LESS THAN THE DELTA
C                AT THE PREVIOUSLY ACCEPTED (STORED) POINT THEN
C                RETURN TO THAT POINT AND ACCEPT IT AS THE UPDATED
C                ITERATE.
C
          CALL MATCOP(N,N,1,1,N,1,XPLPRE,XPLUS)
          CALL MATCOP(N,N,1,1,N,1,FPLPRE,FVEC)
                 ACPCOD=ACPSTR
                 DELTA=DELSTR
                 FCNNEW=FCNPRE
                 RETCOD=3
                 RETURN
              END IF
C
C             IF THE SECOND ACCEPTANCE TEST HAS BEEN PASSED RETURN
C             WITH NEW TRUST REGION AND CONTINUE ON TO NEXT ITER-
C             ATION; OTHERWISE TRY A NEW STEP WITH REDUCED DELTA.
C
              IF(ACPCOD.EQ.2) THEN
                 RETCOD=4
              ELSE
C
C                FAILURE OF FIRST STEP ACCEPTANCE TEST.
C
                 RETCOD=11
              END IF
              RETURN
           END IF
        ELSE
C
C          OBJECTIVE FUNCTION MEETS FIRST ACCEPTANCE CRITERION.
C          IN NONMONOTONIC SEARCHES IT MAY BE GREATER THAN THE
C          PREVIOUS OBJECTIVE FUNCTION VALUE - CONSIDER THIS
C          CASE FIRST.
C
           IF(DELTAF.GE.ALPHA*DELFTS) THEN
C
C             AN ACCEPTABLE STEP HAS BEEN FOUND FOR THE
C             NONMONOTONIC SEARCH BUT THE OBJECTIVE FUNCTION
C             VALUE IS NOT A "DECREASE" FROM THE PREVIOUS
C             ITERATION (ACTUALLY IT MIGHT BE BETWEEN ZERO AND
C             ALPHA*DELFTS).  ACCEPT STEP BUT REDUCE DELTA.
C
              DELTA=DELTA/DELFAC
              RETCOD=5
              IF(ACPCOD.EQ.2) THEN
                 ACPCOD=12
              ELSE
                 ACPCOD=1
              END IF
              RETURN
           END IF
C
C          COMPARE DELTAF TO DELTAF PREDICTED, DELFPR, TO DETERMINE
C          NEXT TRUST REGION SIZE.  NOTE: DELTAF MUST BE LESS THAN
C          ALPHA*DELFTS (IN ESSENCE NEGATIVE) TO HAVE REACHED THIS
C          POINT IN TRSTUP.  R IS IN UPPER TRIANGLE OF MATRIX A SO
C          THE FOLLOWING CODE FINDS:
C
C          DELFPR = DELF^S + 1/2 S^J^JS = DELF^S + 1/2 S^R^RS
C
         DELFPR=DELFTS
         CALL UVMUL(N,N,N,N,A,S,WV3)
         DO 2100 I=1,N
            DELFPR=DELFPR+WV3(I)*WV3(I)/TWO
2100       CONTINUE
           IF(OUTPUT.GT.4) THEN
              WRITE(ICOUT,1)
            WRITE(ICOUT,28) DELFPR
28            FORMAT(T3,'*',7X,'PREDICTED CHANGE IN OBJECTIVE',
     $        ' FUNCTION, DELFPR:',1PD12.3,T74,'*')
            WRITE(ICOUT,29) DELTAF
29            FORMAT(T3,'*',7X,'   ACTUAL CHANGE IN OBJECTIVE',
     $        ' FUNCTION, DELTAF:',1PD12.3,T74,'*')
           END IF
         IF(RETCOD.LE.6.AND.(ABS(DELFPR-DELTAF).LE.
     $     ABS(DELTAF)/TEN.OR.DELTAF.LE.DELFTS).AND.(.NOT.NEWTKN)
     $     .AND.(.NOT.CONVIO).AND.DELSTR.EQ.ZERO) THEN
            IF(MIN(NEWLEN,MAXSTP)/DELTA.GT.ONEPT1) THEN
C
C                PROMISING STEP - INCREASE TRUST REGION.
C
C                STORE CURRENT POINT.
C
          CALL MATCOP(N,N,1,1,N,1,XPLUS,XPLPRE)
          CALL MATCOP(N,N,1,1,N,1,FVEC,FPLPRE)
                 DELSTR=DELTA
                 FCNPRE=FCNNEW
C
C                IF NONMONOTONIC STEPS ARE BEING USED EXPAND TRUST
C                REGION TO NEWLEN, OTHERWISE EXPAND BY DELFAC.
C
                 IF(ISEJAC.GT.NARMIJ) THEN
                    DELTA=MIN(NEWLEN,MAXSTP)
                 ELSE
                    DELTA=MIN(DELFAC*DELTA,MAXSTP)
                 END IF
                 RETCOD=12
                 IF(ACPCOD.EQ.2) THEN
                    ACPSTR=12
                 ELSE
                    ACPSTR=1
                 END IF
                 ACPCOD=0
              ELSE
                 RETCOD=0
                 IF(ACPCOD.EQ.2) THEN
                    ACPCOD=12
                 ELSE
                    ACPCOD=1
                 END IF
              END IF
              RETURN
           ELSE
C
C             CHANGE TRUST REGION SIZE DEPENDING ON DELTAF AND
C             DELFPR.
C
              RETCOD=6
              IF(ACPCOD.EQ.2) THEN
                 ACPCOD=12
              ELSE
                 ACPCOD=1
              END IF
            IF(DELTAF.GE.DELFPR/TEN) THEN
                 DELTA=DELTA/DELFAC
                 IF(OUTPUT.GT.3) THEN
                    WRITE(ICOUT,1)
             WRITE(ICOUT,30)
30                  FORMAT(T3,'*',7X,'CHANGE IN F, DELTAF, IS GREATER',
     $              ' THAN .1 DELFPR - REDUCE DELTA',T74,'*')
                 END IF
              ELSEIF(TRUPDM.EQ.0.AND.JUPDM.GT.0) THEN
C
C                POWELL'S UPDATING SCHEME - FIND JAC S FIRST.
C
                 IF(QNUPDM.EQ.0) THEN
C
C                   UNSCALED JACOBIAN IN JAC.
C
                    DO 2200 I=1,N
                       RHS(I)=S(I)*SCALEF(I)
2200                CONTINUE
             CALL AVMUL(N,N,N,N,JAC,RHS,WV3)
                 ELSE
C
C                   MULTIPLY BY R FIRST.
C
             CALL UVMUL(N,N,N,N,A,S,RHS)
C
C                   THEN Q (IN JAC^)
C
             CALL ATBMUL(N,N,1,1,N,N,JAC,RHS,WV3)
          END IF
          DMULT=DELFPR/TEN-DELTAF
          SP=ZERO
          SS=ZERO
                 DO 2300 K=1,N
                    WV3(K)=WV3(K)+FVECC(K)
                    SP=SP+ABS(FVEC(K)*(FVEC(K)-WV3(K)))
                    SS=SS+(FVEC(K)-WV3(K))*(FVEC(K)-WV3(K))
2300             CONTINUE
          IF(SP+SQRT(SP*SP+DMULT*SS).LT.EPSMCH) THEN
             POWLAM=TEN
                 ELSE
             POWLAM=ONE+DMULT/(SP+SQRT(SP*SP+DMULT*SS))
                 END IF
                 POWLAM=SQRT(POWLAM)
                 POWMU=MIN(DELFAC,POWLAM,POWTAU)
                 POWTAU=POWLAM/POWMU
                 IF(OUTPUT.GT.3) THEN
                    WRITE(ICOUT,1)
             WRITE(ICOUT,31)
31                  FORMAT(T3,'*',7X,'FACTORS IN POWELL UPDATING',
     $              ' SCHEME',T74,'*')
                    WRITE(ICOUT,1)
             WRITE(ICOUT,32) POWLAM,POWMU,POWTAU
32                  FORMAT(T3,'*',7X,'LAMBDA: ',1PD12.3,4X,'MU: ',
     $              1PD12.3,4X,'TAU: ',1PD12.3,T74,'*')
             WRITE(ICOUT,33)
33                  FORMAT(T3,'*',7X,'DELTA IS MINIMUM OF MU*DELTA',
     $              ' AND MAXSTP',T74,'*')
                 END IF
                 DELTA=MIN(POWMU*DELTA,MAXSTP)
              ELSE
          IF(DELTAF.LT.THREEQ*DELFPR) THEN
                    DELTA=MIN(DELFAC*DELTA,MAXSTP)
                    IF(OUTPUT.GT.3) THEN
                       WRITE(ICOUT,1)
                WRITE(ICOUT,34)
34                     FORMAT(T3,'*',7X,'CHANGE IN F, DELTAF, IS LESS',
     $                 ' THAN .75 X PREDICTED',T74,'*')
                WRITE(ICOUT,35) DELTA
35                     FORMAT(T3,'*',7X,'DELTA INCREASED TO: ',1PD12.3,
     $                 T74,'*')
                    END IF
                 ELSE
                    IF(OUTPUT.GT.3) THEN
                       WRITE(ICOUT,1)
                WRITE(ICOUT,36)
36                     FORMAT(T3,'*',7X,'DELTAF BETWEEN 0.1 AND 0.75',
     $                 ' DELFPR - LEAVE DELTA UNCHANGED',T74,'*')
                    END IF
                 END IF
              END IF
           END IF
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE TRSTUP.
C
      END

        SUBROUTINE TWONRM(OVERFL,MAXEXP,N,EPSMCH,EUCNRM,V)
C
C       FEB. 23 ,1992
C
C       THIS SUBROUTINE EVALUATES THE EUCLIDEAN NORM OF A VECTOR, V.
C       IT FOLLOWS THE ALGORITHM OF J.L. BLUE IN ACM TOMS V4 15 (1978)
C       BUT USES SLIGHTLY DIFFERENT CUTS. THE CONSTANTS IN COMMON BLOCK
C       NNES_5 ARE CALCULATED IN THE SUBROUTINE MACHAR OR ARE PROVIDED
C       BY THE USER IN THE DRIVER.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION        V(N)
        LOGICAL          OVERFL
        COMMON/NNES_5/SMALLB,BIGB,SMALLS,BIGS,BIGR
        DATA ZERO,ONE,TWO,TEN /0.0D0,1.0D0,2.0D0,10.0D0/
C
        OVERFL=.FALSE.
        SQRTEP=SQRT(EPSMCH)
C
        ASMALL=ZERO
        AMED=ZERO
        ABIG=ZERO
        DO 100 I=1,N
C
C          ACCUMULATE SUMS OF SQUARES IN ONE OF THREE ACCULULATORS,
C          ABIG, AMED AND ASMALL, DEPENDING ON THEIR SIZES.
C
           ABSVI=ABS(V(I))
C
C          THIS COMPARISON RESTRICTS THE MAXIMUM VALUE OF AMED TO BE
C          B/N => CANNOT SUM SO THAT AMED OVERFLOWS.
C
           IF(ABSVI.GT.BIGB/DBLE(N)) THEN
C
C             THIS DIVISOR OF 10N RESTRICTS ABIG FROM (PATHALOGICALLY)
C             OVERFLOWING FROM SUMMATION.
C
            ABIG=ABIG+((V(I)/(TEN*DBLE(N)*BIGS)))**2
C
           ELSEIF(ABSVI.LT.SMALLS) THEN
C
            ASMALL=ASMALL+((V(I)/SMALLS))**2
C
           ELSE
C
            AMED=AMED+V(I)*V(I)
C
           END IF
C
100     CONTINUE
        IF(ABIG.GT.ZERO) THEN
C
C          IF OVERFLOW WOULD OCCUR ASSIGN BIGR AS NORM AND SIGNAL TO
C          CALLING SUBROUTINE VIA OVERFL.
C
           IF(LOG10(ABIG)/TWO+LOG10(BIGS)+1+LOG10(DBLE(N))
     $        .GT.MAXEXP) THEN
              EUCNRM=BIGR
              OVERFL=.TRUE.
              RETURN
           END IF
C
C          IF AMED IS POSITIVE IT COULD CONTRIBUTE TO THE NORM -
C          DETERMINATION IS DELAYED UNTIL LATER TO SAVE REPEATING
C          CODE.
C
           IF(AMED.GT.ZERO) THEN
              YMIN=MIN(SQRT(AMED),TEN*DBLE(N)*BIGS*SQRT(ABIG))
              YMAX=MAX(SQRT(AMED),TEN*DBLE(N)*BIGS*SQRT(ABIG))
           ELSE
C
C             AMED DOESN'T CONTRIBUTE AND ASMALL WON'T MATTER IF
C             ABIG IS NONZERO - FIND NORM USING ABIG AND RETURN.
C
              EUCNRM=TEN*DBLE(N)*BIGS*SQRT(ABIG)
              RETURN
           END IF
        ELSEIF(ASMALL.GT.ZERO) THEN
           IF(AMED.GT.ZERO) THEN
              YMIN=MIN(SQRT(AMED),SMALLS*SQRT(ASMALL))
              YMAX=MAX(SQRT(AMED),SMALLS*SQRT(ASMALL))
           ELSE
C
C             ABIG AND AMED ARE ZERO SO USE ASMALL ONLY.
C
              EUCNRM=SMALLS*SQRT(ASMALL)
              RETURN
           END IF
        ELSE
           EUCNRM=SQRT(AMED)
           RETURN
        END IF
        IF(YMIN.LT.SQRTEP*YMAX) THEN
C
C          SMALLER PORTION DOES NOT CONTRIBUTE TO NORM.
C
           EUCNRM=YMAX
        ELSE
C
C          SMALLER PORTION CONTRIBUTES TO NORM.
C
           EUCNRM=YMAX*SQRT((ONE+YMIN/YMAX)*(ONE+YMIN/YMAX))
        END IF
        RETURN
C
C       LAST CARD OF SUBROUTINE TWONRM.
C
        END
        SUBROUTINE UPDATE(MNEW  ,MOLD  ,N     ,TRMCOD,FCNNEW,
     $                    FCNOLD,FVEC  ,FVECC ,XC    ,XPLUS )
C
C       FEB. 9, 1991
C
C       THIS SUBROUTINE RESETS CURRENT ESTIMATES OF SOLUTION
C       AND UPDATES THE OBJECTIVE FUNCTION VALUE, M (USED TO
C       SET HOW MANY PREVIOUS VALUES TO LOOK AT IN THE NON-
C       MONOTONIC COMPARISONS) AND THE TERMINATION CODE, TRMCOD.
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        INTEGER     TRMCOD
      DIMENSION   FVEC(N) ,FVECC(N) ,XC(N) ,XPLUS(N)
        FCNOLD=FCNNEW
        MOLD=MNEW
        TRMCOD=0
        DO 100 I=1,N
           FVECC(I)=FVEC(I)
           XC(I)=XPLUS(I)
100     CONTINUE
      RETURN
C
C       LAST CARD OF SUBROUTINE UPDATE.
C
        END
      SUBROUTINE UTBMUL(NCADEC,NCAACT,NCBDEC,NCBACT,NCDEC,NCACT,
     $                    AMAT  ,BMAT  ,CMAT)
C
C       FEB. 8, 1991
C
C       MATRIX MULTIPLICATION:   A^B=C    WHERE A IS UPPER TRIANGULAR
C
C       VERSION WITH INNER LOOP UNROLLED TO DEPTHS 32, 16, 8 AND 4.
C
C       NCADEC IS 2ND DIM. OF AMAT; NCAACT IS ACTUAL LIMIT FOR 2ND INDEX
C       NCBDEC IS 2ND DIM. OF BMAT; NCBACT IS ACTUAL LIMIT FOR 2ND INDEX
C       NCDEC IS COMMON DIMENSION OF AMAT & BMAT; NCACT IS ACTUAL LIMIT
C
C       I.E.   NCADEC IS NUMBER OF COLUMNS OF A DECLARED
C              NCBDEC IS NUMBER OF COLUMNS OF B DECLARED
C              NCDEC IS THE NUMBER OF ROWS IN BOTH A AND B DECLARED
C
C       MODIFICATION OF THE MATRIX MULTIPLIER DONATED BY PROF. JAMES
C       MACKINNON, QUEEN'S UNIVERSITY, KINGSTON, ONTARIO, CANADA
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION AMAT(NCDEC,NCADEC), BMAT(NCDEC,NCBDEC),
     $            CMAT(NCADEC,NCBDEC)
        DATA ZERO /0.0D0/
      DO 100 I=1,NCAACT
C
C          FIND NUMBER OF GROUPS OF SIZE 32, 16 ...
C
         NEND=MIN(I,NCACT)
C
C          THIS ADJUSTMENT IS REQUIRED WHEN NCACT IS LESS THAN NCAACT.
C
         NCC32=NEND/32
         NCC32R=NEND-32*NCC32
         NCC16=NCC32R/16
         NCC16R=NCC32R-16*NCC16
         NCC8=NCC16R/8
         NCC8R=NCC16R-8*NCC8
         NCC4=NCC8R/4
         NCC4R=NCC8R-4*NCC4
C
C          FIND ENTRY IN MATRIX C.
C
         DO 200 J=1,NCBACT
            SUM=ZERO
            K=0
            IF(NCC32.GT.0) THEN
          DO 300 KK=1,NCC32
             K=K+32
             SUM=SUM
     $              +AMAT(K-31,I)*BMAT(K-31,J)+AMAT(K-30,I)*BMAT(K-30,J)
     $              +AMAT(K-29,I)*BMAT(K-29,J)+AMAT(K-28,I)*BMAT(K-28,J)
     $              +AMAT(K-27,I)*BMAT(K-27,J)+AMAT(K-26,I)*BMAT(K-26,J)
     $              +AMAT(K-25,I)*BMAT(K-25,J)+AMAT(K-24,I)*BMAT(K-24,J)
             SUM=SUM
     $              +AMAT(K-23,I)*BMAT(K-23,J)+AMAT(K-22,I)*BMAT(K-22,J)
     $              +AMAT(K-21,I)*BMAT(K-21,J)+AMAT(K-20,I)*BMAT(K-20,J)
     $              +AMAT(K-19,I)*BMAT(K-19,J)+AMAT(K-18,I)*BMAT(K-18,J)
     $              +AMAT(K-17,I)*BMAT(K-17,J)+AMAT(K-16,I)*BMAT(K-16,J)
             SUM=SUM
     $              +AMAT(K-15,I)*BMAT(K-15,J)+AMAT(K-14,I)*BMAT(K-14,J)
     $              +AMAT(K-13,I)*BMAT(K-13,J)+AMAT(K-12,I)*BMAT(K-12,J)
     $              +AMAT(K-11,I)*BMAT(K-11,J)+AMAT(K-10,I)*BMAT(K-10,J)
     $              +AMAT(K-9,I) *BMAT(K-9,J) +AMAT(K-8,I) *BMAT(K-8,J)
             SUM=SUM
     $              +AMAT(K-7,I)*BMAT(K-7,J)+AMAT(K-6,I)*BMAT(K-6,J)
     $              +AMAT(K-5,I)*BMAT(K-5,J)+AMAT(K-4,I)*BMAT(K-4,J)
     $              +AMAT(K-3,I)*BMAT(K-3,J)+AMAT(K-2,I)*BMAT(K-2,J)
     $              +AMAT(K-1,I)*BMAT(K-1,J)+AMAT(K,I)  *BMAT(K,J)
300              CONTINUE
            END IF
            IF(NCC16.GT.0) THEN
          DO 400 KK=1,NCC16
             K=K+16
             SUM=SUM
     $              +AMAT(K-15,I)*BMAT(K-15,J)+AMAT(K-14,I)*BMAT(K-14,J)
     $              +AMAT(K-13,I)*BMAT(K-13,J)+AMAT(K-12,I)*BMAT(K-12,J)
     $              +AMAT(K-11,I)*BMAT(K-11,J)+AMAT(K-10,I)*BMAT(K-10,J)
     $              +AMAT(K-9,I)*BMAT(K-9,J)  +AMAT(K-8,I) *BMAT(K-8,J)
             SUM=SUM
     $              +AMAT(K-7,I)*BMAT(K-7,J)+AMAT(K-6,I)*BMAT(K-6,J)
     $              +AMAT(K-5,I)*BMAT(K-5,J)+AMAT(K-4,I)*BMAT(K-4,J)
     $              +AMAT(K-3,I)*BMAT(K-3,J)+AMAT(K-2,I)*BMAT(K-2,J)
     $              +AMAT(K-1,I)*BMAT(K-1,J)+AMAT(K,I)  *BMAT(K,J)
400              CONTINUE
            END IF
            IF(NCC8.GT.0) THEN
          DO 500 KK=1,NCC8
             K=K+8
             SUM=SUM
     $              +AMAT(K-7,I)*BMAT(K-7,J)+AMAT(K-6,I)*BMAT(K-6,J)
     $              +AMAT(K-5,I)*BMAT(K-5,J)+AMAT(K-4,I)*BMAT(K-4,J)
     $              +AMAT(K-3,I)*BMAT(K-3,J)+AMAT(K-2,I)*BMAT(K-2,J)
     $              +AMAT(K-1,I)*BMAT(K-1,J)+AMAT(K,I)  *BMAT(K,J)
500              CONTINUE
            END IF
            IF(NCC4.GT.0) THEN
          DO 600 KK=1,NCC4
             K=K+4
             SUM=SUM
     $              +AMAT(K-3,I)*BMAT(K-3,J)+AMAT(K-2,I)*BMAT(K-2,J)
     $              +AMAT(K-1,I)*BMAT(K-1,J)+AMAT(K,I)  *BMAT(K,J)
600              CONTINUE
            END IF
            IF(NCC4R.GT.0) THEN
              DO 700 KK=1,NCC4R
                 K=K+1
                 SUM=SUM+AMAT(K,I)*BMAT(K,J)
700           CONTINUE
            END IF
            CMAT(I,J)=SUM
200        CONTINUE
100     CONTINUE
      RETURN
C
C       LAST CARD OF SUBROUTINE UTBMUL.
C
      END
      SUBROUTINE UTUMUL(NRADEC,NCADEC,NRAACT,NCAACT,NRBDEC,NCBDEC,
     $                    AMAT  ,BMAT)
C
C       FEB. 8, 1991
C
C       MATRIX MULTIPLICATION:   A^A=B   WHERE A IS UPPER TRIANGULAR
C
C       VERSION WITH INNER LOOP UNROLLED TO DEPTHS 32, 16, 8 AND 4.
C
C       NRADEC IS NUMBER OF ROWS IN A DECLARED
C       NCADEC IS NUMBER OF COLUMNS IN A DECLARED
C       NRAACT IS THE LIMIT FOR THE 1ST INDEX IN A
C       NCAACT IS THE LIMIT FOR THE 2ND INDEX IN A
C       NRBDEC IS NUMBER OF ROWS IN B DECLARED
C       NCBDEC IS NUMBER OF COLUMNS IN B DECLARED
C
C       MODIFIED VERSION OF THE MATRIX MULTIPLIER DONATED BY PROF. JAMES
C       MACKINNON, QUEEN'S UNIVERSITY, KINGSTON, ONTARIO, CANADA
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION AMAT(NRADEC,NCADEC), BMAT(NRBDEC,NCBDEC)
        DATA ZERO /0.0D0/
C
C       FIND ENTRY IN MATRIX B.
C
        DO 100 I=1,NCAACT
C
C          FIND NUMBER OF GROUPS OF SIZE 32, 16 ...
C
         NEND=MIN(I,NRAACT)
C
         NCC32=NEND/32
         NCC32R=NEND-32*NCC32
         NCC16=NCC32R/16
         NCC16R=NCC32R-16*NCC16
         NCC8=NCC16R/8
         NCC8R=NCC16R-8*NCC8
         NCC4=NCC8R/4
         NCC4R=NCC8R-4*NCC4
         DO 200 J=I,NCAACT
              SUM=ZERO
              K=0
              IF(NCC32.GT.0) THEN
                 DO 300 KK=1,NCC32
                    K=K+32
                    SUM=SUM
     $              +AMAT(K-31,I)*AMAT(K-31,J)+AMAT(K-30,I)*AMAT(K-30,J)
     $              +AMAT(K-29,I)*AMAT(K-29,J)+AMAT(K-28,I)*AMAT(K-28,J)
     $              +AMAT(K-27,I)*AMAT(K-27,J)+AMAT(K-26,I)*AMAT(K-26,J)
     $              +AMAT(K-25,I)*AMAT(K-25,J)+AMAT(K-24,I)*AMAT(K-24,J)
                    SUM=SUM
     $              +AMAT(K-23,I)*AMAT(K-23,J)+AMAT(K-22,I)*AMAT(K-22,J)
     $              +AMAT(K-21,I)*AMAT(K-21,J)+AMAT(K-20,I)*AMAT(K-20,J)
     $              +AMAT(K-19,I)*AMAT(K-19,J)+AMAT(K-18,I)*AMAT(K-18,J)
     $              +AMAT(K-17,I)*AMAT(K-17,J)+AMAT(K-16,I)*AMAT(K-16,J)
                    SUM=SUM
     $              +AMAT(K-15,I)*AMAT(K-15,J)+AMAT(K-14,I)*AMAT(K-14,J)
     $              +AMAT(K-13,I)*AMAT(K-13,J)+AMAT(K-12,I)*AMAT(K-12,J)
     $              +AMAT(K-11,I)*AMAT(K-11,J)+AMAT(K-10,I)*AMAT(K-10,J)
     $              +AMAT(K-9,I)*AMAT(K-9,J)  +AMAT(K-8,I)*AMAT(K-8,J)
                    SUM=SUM
     $              +AMAT(K-7,I)*AMAT(K-7,J)+AMAT(K-6,I)*AMAT(K-6,J)
     $              +AMAT(K-5,I)*AMAT(K-5,J)+AMAT(K-4,I)*AMAT(K-4,J)
     $              +AMAT(K-3,I)*AMAT(K-3,J)+AMAT(K-2,I)*AMAT(K-2,J)
     $              +AMAT(K-1,I)*AMAT(K-1,J)+AMAT(K,I)  *AMAT(K,J)
300              CONTINUE
              END IF
              IF(NCC16.GT.0) THEN
                 DO 400 KK=1,NCC16
                    K=K+16
                    SUM=SUM
     $              +AMAT(K-15,I)*AMAT(K-15,J)+AMAT(K-14,I)*AMAT(K-14,J)
     $              +AMAT(K-13,I)*AMAT(K-13,J)+AMAT(K-12,I)*AMAT(K-12,J)
     $              +AMAT(K-11,I)*AMAT(K-11,J)+AMAT(K-10,I)*AMAT(K-10,J)
     $              +AMAT(K-9,I)*AMAT(K-9,J)  +AMAT(K-8,I) *AMAT(K-8,J)
                    SUM=SUM
     $              +AMAT(K-7,I)*AMAT(K-7,J)+AMAT(K-6,I)*AMAT(K-6,J)
     $              +AMAT(K-5,I)*AMAT(K-5,J)+AMAT(K-4,I)*AMAT(K-4,J)
     $              +AMAT(K-3,I)*AMAT(K-3,J)+AMAT(K-2,I)*AMAT(K-2,J)
     $              +AMAT(K-1,I)*AMAT(K-1,J)+AMAT(K,I)*AMAT(K,J)
400              CONTINUE
              END IF
              IF(NCC8.GT.0) THEN
                 DO 500 KK=1,NCC8
                    K=K+8
                    SUM=SUM
     $              +AMAT(K-7,I)*AMAT(K-7,J)+AMAT(K-6,I)*AMAT(K-6,J)
     $              +AMAT(K-5,I)*AMAT(K-5,J)+AMAT(K-4,I)*AMAT(K-4,J)
     $              +AMAT(K-3,I)*AMAT(K-3,J)+AMAT(K-2,I)*AMAT(K-2,J)
     $              +AMAT(K-1,I)*AMAT(K-1,J)+AMAT(K,I)*AMAT(K,J)
500              CONTINUE
              END IF
              IF(NCC4.GT.0) THEN
                 DO 600 KK=1,NCC4
                    K=K+4
                    SUM=SUM
     $              +AMAT(K-3,I)*AMAT(K-3,J)+AMAT(K-2,I)*AMAT(K-2,J)
     $              +AMAT(K-1,I)*AMAT(K-1,J)+AMAT(K,I)*AMAT(K,J)
600              CONTINUE
              END IF
              IF(NCC4R.GT.0) THEN
                 DO 700 KK=1,NCC4R
                    K=K+1
                    SUM=SUM+AMAT(K,I)*AMAT(K,J)
700              CONTINUE
              END IF
              BMAT(I,J)=SUM
              IF(I.NE.J) BMAT(J,I)=BMAT(I,J)
200        CONTINUE
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE UTUMUL.
C
        END


        SUBROUTINE UVMUL(NRADEC,NRAACT,NCDEC ,NCACT ,AMAT  ,BVEC  ,CVEC)
C
C       FEB. 8, 1991
C
C       MATRIX-VECTOR MULTIPLICATION:  AB=C  WHERE A IS UPPER TRIANGULAR
C
C       VERSION WITH INNER LOOP UNROLLED TO DEPTHS 32, 16, 8 AND 4
C       EACH ROW OF MATRIX A IS SAVED AS A COLUMN BEFORE USE.
C
C       NRADEC IS 1ST DIM. OF AMAT; NRAACT IS ACTUAL LIMIT FOR 1ST INDEX
C       NCDEC IS COMMON DIMENSION OF AMAT & BVEC; NCACT IS ACTUAL LIMIT
C
C       I.E. NRADEC IS THE NUMBER OF ROWS OF A DECLARED
C            NCDEC IS THE COMMON DECLARED DIMENSION (COLUMNS OF A AND
C            ROWS OF B)
C
C       MODIFICATION OF THE MATRIX MULTIPLIER DONATED BY PROF. JAMES
C       MACKINNON, QUEEN'S UNIVERSITY, KINGSTON, ONTARIO, CANADA
C
        IMPLICIT DOUBLE PRECISION (A-H,O-Z)
        DIMENSION AMAT(NRADEC,NCDEC),  BVEC(NCDEC),  CVEC(NRADEC)
        DATA ZERO /0.0D0/
        DO 100 I=1,NRAACT
C
C          FIND NUMBER OF GROUPS OF SIZE 32, 16 ...
C
           NCC32=(NCACT-(I-1))/32
           NCC32R=(NCACT-(I-1))-32*NCC32
           NCC16=NCC32R/16
           NCC16R=NCC32R-16*NCC16
           NCC8=NCC16R/8
           NCC8R=NCC16R-8*NCC8
           NCC4=NCC8R/4
           NCC4R=NCC8R-4*NCC4
C
C          FIND ENTRY FOR VECTOR C.
C
           SUM=ZERO
           K=I-1
           IF(NCC32.GT.0) THEN
              DO 200 KK=1,NCC32
                 K=K+32
                 SUM=SUM
     $           +AMAT(I,K-31)*BVEC(K-31)+AMAT(I,K-30)*BVEC(K-30)
     $           +AMAT(I,K-29)*BVEC(K-29)+AMAT(I,K-28)*BVEC(K-28)
     $           +AMAT(I,K-27)*BVEC(K-27)+AMAT(I,K-26)*BVEC(K-26)
     $           +AMAT(I,K-25)*BVEC(K-25)+AMAT(I,K-24)*BVEC(K-24)
                 SUM=SUM
     $           +AMAT(I,K-23)*BVEC(K-23)+AMAT(I,K-22)*BVEC(K-22)
     $           +AMAT(I,K-21)*BVEC(K-21)+AMAT(I,K-20)*BVEC(K-20)
     $           +AMAT(I,K-19)*BVEC(K-19)+AMAT(I,K-18)*BVEC(K-18)
     $           +AMAT(I,K-17)*BVEC(K-17)+AMAT(I,K-16)*BVEC(K-16)
                 SUM=SUM
     $           +AMAT(I,K-15)*BVEC(K-15)+AMAT(I,K-14)*BVEC(K-14)
     $           +AMAT(I,K-13)*BVEC(K-13)+AMAT(I,K-12)*BVEC(K-12)
     $           +AMAT(I,K-11)*BVEC(K-11)+AMAT(I,K-10)*BVEC(K-10)
     $           +AMAT(I,K-9)*BVEC(K-9)  +AMAT(I,K-8) *BVEC(K-8)
                 SUM=SUM
     $           +AMAT(I,K-7)*BVEC(K-7)+AMAT(I,K-6)*BVEC(K-6)
     $           +AMAT(I,K-5)*BVEC(K-5)+AMAT(I,K-4)*BVEC(K-4)
     $           +AMAT(I,K-3)*BVEC(K-3)+AMAT(I,K-2)*BVEC(K-2)
     $           +AMAT(I,K-1)*BVEC(K-1)+AMAT(I,K)  *BVEC(K)
200           CONTINUE
           END IF
           IF(NCC16.GT.0) THEN
              DO 300 KK=1,NCC16
                 K=K+16
                 SUM=SUM
     $           +AMAT(I,K-15)*BVEC(K-15)+AMAT(I,K-14)*BVEC(K-14)
     $           +AMAT(I,K-13)*BVEC(K-13)+AMAT(I,K-12)*BVEC(K-12)
     $           +AMAT(I,K-11)*BVEC(K-11)+AMAT(I,K-10)*BVEC(K-10)
     $           +AMAT(I,K-9)*BVEC(K-9)  +AMAT(I,K-8) *BVEC(K-8)
                 SUM=SUM
     $           +AMAT(I,K-7)*BVEC(K-7)+AMAT(I,K-6)*BVEC(K-6)
     $           +AMAT(I,K-5)*BVEC(K-5)+AMAT(I,K-4)*BVEC(K-4)
     $           +AMAT(I,K-3)*BVEC(K-3)+AMAT(I,K-2)*BVEC(K-2)
     $           +AMAT(I,K-1)*BVEC(K-1)+AMAT(I,K)  *BVEC(K)
300           CONTINUE
           END IF
           IF(NCC8.GT.0) THEN
              DO 400 KK=1,NCC8
                 K=K+8
                 SUM=SUM
     $           +AMAT(I,K-7)*BVEC(K-7)+AMAT(I,K-6)*BVEC(K-6)
     $           +AMAT(I,K-5)*BVEC(K-5)+AMAT(I,K-4)*BVEC(K-4)
     $           +AMAT(I,K-3)*BVEC(K-3)+AMAT(I,K-2)*BVEC(K-2)
     $           +AMAT(I,K-1)*BVEC(K-1)+AMAT(I,K)  *BVEC(K)
400           CONTINUE
           END IF
           IF(NCC4.GT.0) THEN
              DO 500 KK=1,NCC4
                 K=K+4
                 SUM=SUM
     $           +AMAT(I,K-3)*BVEC(K-3)+AMAT(I,K-2)*BVEC(K-2)
     $           +AMAT(I,K-1)*BVEC(K-1)+AMAT(I,K)  *BVEC(K)
500           CONTINUE
           END IF
           IF(NCC4R.GT.0) THEN
              DO 600 KK=1,NCC4R
                 K=K+1
                 SUM=SUM+AMAT(I,K)*BVEC(K)
600           CONTINUE
           END IF
           CVEC(I)=SUM
100     CONTINUE
        RETURN
C
C       LAST CARD OF SUBROUTINE UVMUL.
C
        END
C
C     THESE ARE THE STEVE NASH TN/TNBC ROUTINES FOR EITHER UNCONSTRAINED
C     OPTIMIATION OR OPTIMIZATION WITH BOUNDS.  THESE ROUTINES CAN
C     USE EITHER ANALYTIC OR NUMERIC GRADIENTS/HESIANS.
C
C     ROUTINES ARE ALSO MODIFIED TO USE DATAPLOT I/O.  THE BLAS ROUTINES
C     USED BY TN/TNBC ARE NOT LISTED HERE (THESE ARE OTHER DATAPLOT
C     SOURCE FILES, NOT HERE).
C
C     ROUTINES:
C     ====================
C     CHKUCP   - CHECKS PARAMETERS AND SETS CONSTANTS WHICH ARE
C                COMMON TO BOTH DERIVATIE AND NON-DERIVATIVE ALGORITHMS
C     CNVTST   - TEST FOR CONVERGENCE
C     CRASH    - INITIALIZE CONSTRAINT INFORMATION
C     GETPTC   - ALGORITHM FOR FINDING A STEPLENGTH
C     GTIMS    - COMPUTE PRODUCT OF MATRIX G TIMES VECTOR V
C                AND STORE RESULT IN VECTOR GV (FINITE DIFFERENCE
C                VERSION
C     INITP3   - INITIALIZATION
C     INITP4   - INITIALIZATION (RENAMED FROM INITPC TO AVOID
C                CONFLICT WITH DATAPLOT NAME 
C     LINDER   - LINE SEARCH ALGORITHMS OF GILL AND MURRAY
C     LMQN     - THIS ROUTINE IS A TRUNCATED-NEWTON METHOD.
C                THE TRUNCATED-NEWTON METHOD IS PRECONDITIONED BY A
C                LIMITED-MEMORY QUASI-NEWTON METHOD (THIS
C                PRECONDITIONING STRATEGY IS DEVELOPED IN THIS ROUTINE)
C                WITH A FURTHER DIAGONAL SCALING (SEE ROUTINE NDIA3).
C     LMQNBC   - THIS ROUTINE IS A BOUNDS-CONSTRAINED TRUNCATED-NEWTON
C                METHOD.  THE TRUNCATED-NEWTON METHOD IS PRECONDITIONED
C                BY A LIMITED-MEMORY QUASI-NEWTON METHOD (THIS
C                PRECONDITIONING STRATEGY IS DEVELOPED IN THIS ROUTINE)
C                WITH A FURTHER DIAGONAL SCALING (SEE ROUTINE NDIA3).
C     LSOUT    - ERROR PRINTOUTS FOR GETPTC
C     MCHPR1   - SMALLEST POSSIBLE REAL NUMBER SUCH THAT 1.0 + EPS
C                IS GREATER THAN 1.0
C     MODLNP   - PERFORM A PRECONDITIONED CONJUGATE-GRADIENT
C                ITERATION IN ORDER TO SOLVE THE NEWTON EQUATIONS FOR
C                A SEARCH DIRECTION FOR A TRUNCATED-NEWTON ALGORITHM.
C                WHEN THE VALUE OF THE QUADRATIC MODEL IS SUFFICIENTLY
C                REDUCED, THE ITERATION IS TERMINATED.
C     MODZ     - UPDATE THE CONSTRAINT MATRIX IF A NEW CONSTRAINT IS
C                ENCOUNTERED
C     MONIT    - PRINT RESULTS OF CURRENT ITERATION
C     MSLV     - THIS ROUTINE ACTS AS A PRECONDITIONING STEP FOR THE
C                LINEAR CONJUGATE-GRADIENT ROUTINE.  IT IS ALSO THE
C                METHOD OF COMPUTING THE SEARCH DIRECTION FROM THE
C                GRADIENT FOR THE NON-LINEAR CONJUGATE-GRADIENT CODE.
C                IT REPRESENTS A TWO-STEP SELF-SCALED BFGS FORMULA.
C     MSOLVE   - SETUP ARRAYS FOR MSOLVE
C     NDIA3    - UPDATE THE PRECONDITIONING MATRIX BASED ON A DIAGONAL
C                VERSION OF THE BFGS QUASI-NEWTON UPDATE
C     NEGVEC   - SERVICE ROUTINE FOR OPTIMIZATION
C     SETPAR   - SET UP PARAMETERS FOR OPTIMIZATION ROUTINE
C     SETUCR   - CHECK INPUT PARAMETERS, COMPUTE INITIAL FUNCTION
C                VALUE, SET CONSTANTS FOR THE SUBSEQUENT MINIMIZATION
C     SSBFGS   - SELF-SCALED BFGS
C     STEP1    - RETURN LENGTH OF INITIAL STEP TO BE TAKEN ALONG
C                VECTOR P IN THE NEXT LINEAR SEARCH
C     STPMAX   - COMPUTE MAXIMUM ALLOWABLE STEP LENGTH
C     TNBC     - MAIN ROUTINE FOR BOUND CONSTRAINED OPTIMIZATION
C     TN       - MAIN ROUTINE FOR UNCONSTRAINED OPTIMIZATION
C     ZTIME    - MULTIPLY VECTOR X BY THE CONSTRAINT MATRIX Z
C
C
      SUBROUTINE CHKUCP(LWTEST,MAXFUN,NWHY,N,ALPHA,EPSMCH,
     *     ETA,PEPS,RTEPS,RTOL,RTOLSQ,STEPMX,TEST,
     *     XTOL,XNORM,X,LW,SMALL,TINY,ACCRCY)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      INTEGER LW,LWTEST,MAXFUN,NWHY,N
      DOUBLE PRECISION ACCRCY,ALPHA,EPSMCH,ETA,PEPS,RTEPS,RTOL,
     *     RTOLSQ,STEPMX,TEST,XTOL,XNORM,SMALL,TINY
      DOUBLE PRECISION X(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C CHECKS PARAMETERS AND SETS CONSTANTS WHICH ARE COMMON TO BOTH
C DERIVATIVE AND NON-DERIVATIVE ALGORITHMS
C
      DOUBLE PRECISION DABS,DSQRT,MCHPR1
      EPSMCH = MCHPR1()
      SMALL = EPSMCH*EPSMCH
      TINY = SMALL
      NWHY = -1
      RTEPS = DSQRT(EPSMCH)
      RTOL = XTOL
      IF (DABS(RTOL) .LT. ACCRCY) RTOL = 1.D1*RTEPS
C
C CHECK FOR ERRORS IN THE INPUT PARAMETERS
C
      IF (LW .LT. LWTEST
     *      .OR. N .LT. 1 .OR. RTOL .LT. 0.D0 .OR. ETA .GE. 1.D0 .OR.
     *      ETA .LT. 0.D0 .OR. STEPMX .LT. RTOL .OR.
     *      MAXFUN .LT. 1) RETURN
      NWHY = 0
C
C SET CONSTANTS FOR LATER
C
      RTOLSQ = RTOL*RTOL
      PEPS = ACCRCY**0.6666D0
      XNORM = DNRM2(N,X,1)
      ALPHA = 0.D0
      TEST = 0.D0
      RETURN
      END
C
C
      SUBROUTINE CNVTST(CONV,ALPHA,PNORM,TOLEPS,XNORM,DIFNEW,RTLEPS,
     *     FTEST,GTG,PEPS,EPSMCH,GTPNEW,FNEW,FLAST,G,IPIVOT,N,ACCRCY)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      LOGICAL CONV,LTEST
      INTEGER IPIVOT(N)
      DOUBLE PRECISION G(N), ALPHA, PNORM, TOLEPS, XNORM, DIFNEW,
     *     RTLEPS, FTEST, GTG, PEPS, EPSMCH, GTPNEW, FNEW, FLAST, ONE,
     *     CMAX, T, ACCRCY
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C TEST FOR CONVERGENCE
C
      IMAX = 0
      CMAX = 0.D0
      LTEST = FLAST - FNEW .LE. -5.D-1*GTPNEW
      DO 10 I = 1,N
         IF (IPIVOT(I) .EQ. 0 .OR. IPIVOT(I) .EQ. 2) GO TO 10
         T = -IPIVOT(I)*G(I)
         IF (T .GE. 0.D0) GO TO 10
         CONV = .FALSE.
         IF (LTEST) GO TO 10
         IF (CMAX .LE. T) GO TO 10
         CMAX = T
         IMAX = I
10    CONTINUE
      IF (IMAX .EQ. 0) GO TO 15
      IPIVOT(IMAX) = 0
      FLAST = FNEW
      RETURN
15    CONTINUE
      CONV = .FALSE.
      ONE = 1.D0
      IF ((ALPHA*PNORM .GE. TOLEPS*(ONE + XNORM)
     *     .OR. DABS(DIFNEW) .GE. RTLEPS*FTEST
     *     .OR. GTG .GE. PEPS*FTEST*FTEST)
     *     .AND. GTG .GE. 1.D-4*ACCRCY*FTEST*FTEST) RETURN
      CONV = .TRUE.
C
C FOR DETAILS, SEE GILL, MURRAY, AND WRIGHT (1981, P. 308) AND
C FLETCHER (1981, P. 116).  THE MULTIPLIER TESTS (HERE, TESTING
C THE SIGN OF THE COMPONENTS OF THE GRADIENT) MAY STILL NEED TO
C MODIFIED TO INCORPORATE TOLERANCES FOR ZERO.
C
      RETURN
      END
C
C
      SUBROUTINE CRASH(N,X,IPIVOT,LOW,UP,IER)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION X(N),LOW(N),UP(N)
      INTEGER IPIVOT(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C THIS INITIALIZES THE CONSTRAINT INFORMATION, AND ENSURES THAT THE
C INITIAL POINT SATISFIES  LOW <= X <= UP.
C THE CONSTRAINTS ARE CHECKED FOR CONSISTENCY.
C
      IER = 0
      DO 30 I = 1,N
         IF (X(I) .LT. LOW(I)) X(I) = LOW(I)
         IF (X(I) .GT. UP(I)) X(I) = UP(I)
         IPIVOT(I) = 0
         IF (X(I) .EQ. LOW(I)) IPIVOT(I) = -1
         IF (X(I) .EQ. UP(I)) IPIVOT(I) = 1
         IF (UP(I) .EQ. LOW(I)) IPIVOT(I) = 2
         IF (LOW(I) .GT. UP(I)) IER = -I
30    CONTINUE
      RETURN
      END
C
C NOTE: FOR DATAPLOT, RENAME DXPY TO DXPYTN.  DATAPLOT ALREADY
C       USES DXPY ELSEWHERE AND THE DATAPLOT VERSION HAS A
C       DIFFERENT CALL LIST.  FOR TN/TNBC, USE THE VERSION
C       GIVEN HERE, BUT RENAME.
C
C******************************************************************
C SPECIAL BLAS FOR Y = X+Y
C******************************************************************
CCCCC SUBROUTINE DXPY(N,DX,INCX,DY,INCY)
      SUBROUTINE DXPYTN(N,DX,INCX,DY,INCY)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C     VECTOR PLUS A VECTOR.
C     USES UNROLLED LOOPS FOR INCREMENTS EQUAL TO ONE.
C     STEPHEN G. NASH 5/30/89.
C
      DOUBLE PRECISION DX(1),DY(1)
      INTEGER I,INCX,INCY,IX,IY,M,MP1,N
C
      IF(N.LE.0)RETURN
      IF(INCX.EQ.1.AND.INCY.EQ.1)GO TO 20
C
C        CODE FOR UNEQUAL INCREMENTS OR EQUAL INCREMENTS
C          NOT EQUAL TO 1
C
      IX = 1
      IY = 1
      IF(INCX.LT.0)IX = (-N+1)*INCX + 1
      IF(INCY.LT.0)IY = (-N+1)*INCY + 1
      DO 10 I = 1,N
        DY(IY) = DY(IY) + DX(IX)
        IX = IX + INCX
        IY = IY + INCY
   10 CONTINUE
      RETURN
C
C        CODE FOR BOTH INCREMENTS EQUAL TO 1
C
C
C        CLEAN-UP LOOP
C
   20 M = MOD(N,4)
      IF( M .EQ. 0 ) GO TO 40
      DO 30 I = 1,M
        DY(I) = DY(I) + DX(I)
   30 CONTINUE
      IF( N .LT. 4 ) RETURN
   40 MP1 = M + 1
      DO 50 I = MP1,N,4
        DY(I) = DY(I) + DX(I)
        DY(I + 1) = DY(I + 1) + DX(I + 1)
        DY(I + 2) = DY(I + 2) + DX(I + 2)
        DY(I + 3) = DY(I + 3) + DX(I + 3)
   50 CONTINUE
      RETURN
      END
C
C
      SUBROUTINE GETPTC(BIG,SMALL,RTSMLL,RELTOL,ABSTOL,TNYTOL,
     *     FPRESN,ETA,RMU,XBND,U,FU,GU,XMIN,FMIN,GMIN,
     *     XW,FW,GW,A,B,OLDF,B1,SCXBND,E,STEP,FACTOR,
     *     BRAKTD,GTEST1,GTEST2,TOL,IENTRY,ITEST)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      LOGICAL BRAKTD
      INTEGER IENTRY,ITEST
      DOUBLE PRECISION BIG,SMALL,RTSMLL,RELTOL,ABSTOL,TNYTOL,
     *     FPRESN,ETA,RMU,XBND,U,FU,GU,XMIN,FMIN,GMIN,
     *     XW,FW,GW,A,B,OLDF,B1,SCXBND,E,STEP,FACTOR,
     *     GTEST1,GTEST2,TOL,DENOM
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C ************************************************************
C GETPTC, AN ALGORITHM FOR FINDING A STEPLENGTH, CALLED REPEATEDLY BY
C ROUTINES WHICH REQUIRE A STEP LENGTH TO BE COMPUTED USING CUBIC
C INTERPOLATION. THE PARAMETERS CONTAIN INFORMATION ABOUT THE INTERVAL
C IN WHICH A LOWER POINT IS TO BE FOUND AND FROM THIS GETPTC COMPUTES A
C POINT AT WHICH THE FUNCTION CAN BE EVALUATED BY THE CALLING PROGRAM.
C THE VALUE OF THE INTEGER PARAMETERS IENTRY DETERMINES THE PATH TAKEN
C THROUGH THE CODE.
C ************************************************************
C
      LOGICAL CONVRG
      DOUBLE PRECISION ABGMIN,ABGW,ABSR,A1,CHORDM,CHORDU,
     *     D1,D2,P,Q,R,S,SCALE,SUMSQ,TWOTOL,XMIDPT
      DOUBLE PRECISION ZERO, POINT1,HALF,ONE,THREE,FIVE,ELEVEN
C
C THE FOLLOWING STANDARD FUNCTIONS AND SYSTEM FUNCTIONS ARE CALLED
C WITHIN GETPTC
C
      DOUBLE PRECISION DABS, DSQRT
C
      ZERO = 0.D0
      POINT1 = 1.D-1
      HALF = 5.D-1
      ONE = 1.D0
      THREE = 3.D0
      FIVE = 5.D0
      ELEVEN = 11.D0
C
C      BRANCH TO APPROPRIATE SECTION OF CODE DEPENDING ON THE
C      VALUE OF IENTRY.
C
CCCCC GOTO (10,20), IENTRY
      IF(IENTRY.EQ.1)GOTO10
      IF(IENTRY.EQ.2)GOTO20
C
C      IENTRY=1
C      CHECK INPUT PARAMETERS
C
10      ITEST = 2
      IF (U .LE. ZERO .OR. XBND .LE. TNYTOL .OR. GU .GT. ZERO)
     *     RETURN
      ITEST = 1
      IF (XBND .LT. ABSTOL) ABSTOL = XBND
      TOL = ABSTOL
      TWOTOL = TOL + TOL
C
C A AND B DEFINE THE INTERVAL OF UNCERTAINTY, X AND XW ARE POINTS
C WITH LOWEST AND SECOND LOWEST FUNCTION VALUES SO FAR OBTAINED.
C INITIALIZE A,SMIN,XW AT ORIGIN AND CORRESPONDING VALUES OF
C FUNCTION AND PROJECTION OF THE GRADIENT ALONG DIRECTION OF SEARCH
C AT VALUES FOR LATEST ESTIMATE AT MINIMUM.
C
      A = ZERO
      XW = ZERO
      XMIN = ZERO
      OLDF = FU
      FMIN = FU
      FW = FU
      GW = GU
      GMIN = GU
      STEP = U
      FACTOR = FIVE
C
C      THE MINIMUM HAS NOT YET BEEN BRACKETED.
C
      BRAKTD = .FALSE.
C
C SET UP XBND AS A BOUND ON THE STEP TO BE TAKEN. (XBND IS NOT COMPUTED
C EXPLICITLY BUT SCXBND IS ITS SCALED VALUE.)  SET THE UPPER BOUND
C ON THE INTERVAL OF UNCERTAINTY INITIALLY TO XBND + TOL(XBND).
C
      SCXBND = XBND
      B = SCXBND + RELTOL*DABS(SCXBND) + ABSTOL
      E = B + B
      B1 = B
C
C COMPUTE THE CONSTANTS REQUIRED FOR THE TWO CONVERGENCE CRITERIA.
C
      GTEST1 = -RMU*GU
      GTEST2 = -ETA*GU
C
C SET IENTRY TO INDICATE THAT THIS IS THE FIRST ITERATION
C
      IENTRY = 2
      GO TO 210
C
C IENTRY = 2
C
C UPDATE A,B,XW, AND XMIN
C
20      IF (FU .GT. FMIN) GO TO 60
C
C IF FUNCTION VALUE NOT INCREASED, NEW POINT BECOMES NEXT
C ORIGIN AND OTHER POINTS ARE SCALED ACCORDINGLY.
C
      CHORDU = OLDF - (XMIN + U)*GTEST1
      IF (FU .LE. CHORDU) GO TO 30
C
C THE NEW FUNCTION VALUE DOES NOT SATISFY THE SUFFICIENT DECREASE
C CRITERION. PREPARE TO MOVE THE UPPER BOUND TO THIS POINT AND
C FORCE THE INTERPOLATION SCHEME TO EITHER BISECT THE INTERVAL OF
C UNCERTAINTY OR TAKE THE LINEAR INTERPOLATION STEP WHICH ESTIMATES
C THE ROOT OF F(ALPHA)=CHORD(ALPHA).
C
      CHORDM = OLDF - XMIN*GTEST1
      GU = -GMIN
      DENOM = CHORDM-FMIN
      IF (DABS(DENOM) .GE. 1.D-15) GO TO 25
          DENOM = 1.D-15
          IF (CHORDM-FMIN .LT. 0.D0)  DENOM = -DENOM
25    CONTINUE
      IF (XMIN .NE. ZERO) GU = GMIN*(CHORDU-FU)/DENOM
      FU = HALF*U*(GMIN+GU) + FMIN
      IF (FU .LT. FMIN) FU = FMIN
      GO TO 60
30      FW = FMIN
      FMIN = FU
      GW = GMIN
      GMIN = GU
      XMIN = XMIN + U
      A = A-U
      B = B-U
      XW = -U
      SCXBND = SCXBND - U
      IF (GU .LE. ZERO) GO TO 40
      B = ZERO
      BRAKTD = .TRUE.
      GO TO 50
40    A = ZERO
50    TOL = DABS(XMIN)*RELTOL + ABSTOL
      GO TO 90
C
C IF FUNCTION VALUE INCREASED, ORIGIN REMAINS UNCHANGED
C BUT NEW POINT MAY NOW QUALIFY AS W.
C
60    IF (U .LT. ZERO) GO TO 70
      B = U
      BRAKTD = .TRUE.
      GO TO 80
70    A = U
80    XW = U
      FW = FU
      GW = GU
90    TWOTOL = TOL + TOL
      XMIDPT = HALF*(A + B)
C
C CHECK TERMINATION CRITERIA
C
      CONVRG = DABS(XMIDPT) .LE. TWOTOL - HALF*(B-A) .OR.
     *     DABS(GMIN) .LE. GTEST2 .AND. FMIN .LT. OLDF .AND.
     *     (DABS(XMIN - XBND) .GT. TOL .OR. .NOT. BRAKTD)
      IF (.NOT. CONVRG) GO TO 100
      ITEST = 0
      IF (XMIN .NE. ZERO) RETURN
C
C IF THE FUNCTION HAS NOT BEEN REDUCED, CHECK TO SEE THAT THE RELATIVE
C CHANGE IN F(X) IS CONSISTENT WITH THE ESTIMATE OF THE DELTA-
C UNIMODALITY CONSTANT, TOL.  IF THE CHANGE IN F(X) IS LARGER THAN
C EXPECTED, REDUCE THE VALUE OF TOL.
C
      ITEST = 3
      IF (DABS(OLDF-FW) .LE. FPRESN*(ONE + DABS(OLDF))) RETURN
      TOL = POINT1*TOL
      IF (TOL .LT. TNYTOL) RETURN
      RELTOL = POINT1*RELTOL
      ABSTOL = POINT1*ABSTOL
      TWOTOL = POINT1*TWOTOL
C
C CONTINUE WITH THE COMPUTATION OF A TRIAL STEP LENGTH
C
100   R = ZERO
      Q = ZERO
      S = ZERO
      IF (DABS(E) .LE. TOL) GO TO 150
C
C FIT CUBIC THROUGH XMIN AND XW
C
      R = THREE*(FMIN-FW)/XW + GMIN + GW
      ABSR = DABS(R)
      Q = ABSR
      IF (GW .EQ. ZERO .OR. GMIN .EQ. ZERO) GO TO 140
C
C COMPUTE THE SQUARE ROOT OF (R*R - GMIN*GW) IN A WAY
C WHICH AVOIDS UNDERFLOW AND OVERFLOW.
C
      ABGW = DABS(GW)
      ABGMIN = DABS(GMIN)
      S = DSQRT(ABGMIN)*DSQRT(ABGW)
      IF ((GW/ABGW)*GMIN .GT. ZERO) GO TO 130
C
C COMPUTE THE SQUARE ROOT OF R*R + S*S.
C
      SUMSQ = ONE
      P = ZERO
      IF (ABSR .GE. S) GO TO 110
C
C THERE IS A POSSIBILITY OF OVERFLOW.
C
      IF (S .GT. RTSMLL) P = S*RTSMLL
      IF (ABSR .GE. P) SUMSQ = ONE +(ABSR/S)**2
      SCALE = S
      GO TO 120
C
C THERE IS A POSSIBILITY OF UNDERFLOW.
C
110   IF (ABSR .GT. RTSMLL) P = ABSR*RTSMLL
      IF (S .GE. P) SUMSQ = ONE + (S/ABSR)**2
      SCALE = ABSR
120   SUMSQ = DSQRT(SUMSQ)
      Q = BIG
      IF (SCALE .LT. BIG/SUMSQ) Q = SCALE*SUMSQ
      GO TO 140
C
C COMPUTE THE SQUARE ROOT OF R*R - S*S
C
130   Q = DSQRT(DABS(R+S))*DSQRT(DABS(R-S))
      IF (R .GE. S .OR. R .LE. (-S)) GO TO 140
      R = ZERO
      Q = ZERO
      GO TO 150
C
C COMPUTE THE MINIMUM OF FITTED CUBIC
C
140   IF (XW .LT. ZERO) Q = -Q
      S = XW*(GMIN - R - Q)
      Q = GW - GMIN + Q + Q
      IF (Q .GT. ZERO) S = -S
      IF (Q .LE. ZERO) Q = -Q
      R = E
      IF (B1 .NE. STEP .OR. BRAKTD) E = STEP
C
C CONSTRUCT AN ARTIFICIAL BOUND ON THE ESTIMATED STEPLENGTH
C
150   A1 = A
      B1 = B
      STEP = XMIDPT
      IF (BRAKTD) GO TO 160
      STEP = -FACTOR*XW
      IF (STEP .GT. SCXBND) STEP = SCXBND
      IF (STEP .NE. SCXBND) FACTOR = FIVE*FACTOR
      GO TO 170
C
C IF THE MINIMUM IS BRACKETED BY 0 AND XW THE STEP MUST LIE
C WITHIN (A,B).
C
160   IF ((A .NE. ZERO .OR. XW .GE. ZERO) .AND. (B .NE. ZERO .OR.
     *     XW .LE. ZERO)) GO TO 180
C
C IF THE MINIMUM IS NOT BRACKETED BY 0 AND XW THE STEP MUST LIE
C WITHIN (A1,B1).
C
      D1 = XW
      D2 = A
      IF (A .EQ. ZERO) D2 = B
C THIS LINE MIGHT BE
C     IF (A .EQ. ZERO) D2 = E
      U = - D1/D2
      STEP = FIVE*D2*(POINT1 + ONE/U)/ELEVEN
      IF (U .LT. ONE) STEP = HALF*D2*DSQRT(U)
170   IF (STEP .LE. ZERO) A1 = STEP
      IF (STEP .GT. ZERO) B1 = STEP
C
C REJECT THE STEP OBTAINED BY INTERPOLATION IF IT LIES OUTSIDE THE
C REQUIRED INTERVAL OR IT IS GREATER THAN HALF THE STEP OBTAINED
C DURING THE LAST-BUT-ONE ITERATION.
C
180   IF (DABS(S) .LE. DABS(HALF*Q*R) .OR.
     *     S .LE. Q*A1 .OR. S .GE. Q*B1) GO TO 200
C
C A CUBIC INTERPOLATION STEP
C
      STEP = S/Q
C
C THE FUNCTION MUST NOT BE EVALUTATED TOO CLOSE TO A OR B.
C
      IF (STEP - A .GE. TWOTOL .AND. B - STEP .GE. TWOTOL) GO TO 210
      IF (XMIDPT .GT. ZERO) GO TO 190
      STEP = -TOL
      GO TO 210
190   STEP = TOL
      GO TO 210
200   E = B-A
C
C IF THE STEP IS TOO LARGE, REPLACE BY THE SCALED BOUND (SO AS TO
C COMPUTE THE NEW POINT ON THE BOUNDARY).
C
210   IF (STEP .LT. SCXBND) GO TO 220
      STEP = SCXBND
C
C MOVE SXBD TO THE LEFT SO THAT SBND + TOL(XBND) = XBND.
C
      SCXBND = SCXBND - (RELTOL*DABS(XBND)+ABSTOL)/(ONE + RELTOL)
220   U = STEP
      IF (DABS(STEP) .LT. TOL .AND. STEP .LT. ZERO) U = -TOL
      IF (DABS(STEP) .LT. TOL .AND. STEP .GE. ZERO) U = TOL
      ITEST = 1
      RETURN
      END
C
C
      SUBROUTINE GTIMS(V,GV,N,X,G,W,LW,SFUN,FIRST,DELTA,ACCRCY,XNORM,
     *                 XOBS,NOBS)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION V(N),GV(N),DINV,DELTA,G(N)
      DOUBLE PRECISION F,X(N),W(LW),ACCRCY,DSQRT,XNORM
      DOUBLE PRECISION XOBS(*)
      LOGICAL FIRST
      EXTERNAL SFUN
      COMMON/SUBSCR/ LGV,LZ1,LZK,LV,LSK,LYK,LDIAGB,LSR,LYR,
     *     LHYR,LHG,LHYK,LPK,LEMAT,LWTEST
C
C     NOTE: FOR DATAPLOT, ADD XOBS AND NOBS TO CALL LIST.  THESE
C           WILL BE PASSED TO THE SFUN ROUTINE.
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C THIS ROUTINE COMPUTES THE PRODUCT OF THE MATRIX G TIMES THE VECTOR
C V AND STORES THE RESULT IN THE VECTOR GV (FINITE-DIFFERENCE VERSION)
C
      IF (.NOT. FIRST) GO TO 20
      DELTA = DSQRT(ACCRCY)*(1.D0+XNORM)
      FIRST = .FALSE.
20    CONTINUE
      DINV = 1.D0/DELTA
      IHG = LHG
      DO 30 I = 1,N
         W(IHG) = X(I) + DELTA*V(I)
         IHG = IHG + 1
30    CONTINUE
      CALL SFUN(N,W(LHG),F,GV,XOBS,NOBS)
      DO 40 I = 1,N
         GV(I) = (GV(I) - G(I))*DINV
40    CONTINUE
      RETURN
      END
C
C
      SUBROUTINE INITP3(DIAGB,EMAT,N,LRESET,YKSK,YRSR,BSK,
     *     SK,YK,SR,YR,MODET,UPD1)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION DIAGB(N),EMAT(N),YKSK,YRSR,BSK(N),SK(N),
     *     YK(N),COND,SR(N),YR(N),DDOT,SDS,SRDS,YRSK,TD,D1,DN
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      LOGICAL LRESET,UPD1
      IF (UPD1) GO TO 90
      IF (LRESET) GO TO 60
      DO 10 I = 1,N
         BSK(I) = DIAGB(I)*SR(I)
10    CONTINUE
      SDS = DDOT(N,SR,1,BSK,1)
      SRDS = DDOT(N,SK,1,BSK,1)
      YRSK = DDOT(N,YR,1,SK,1)
      DO 20 I = 1,N
         TD = DIAGB(I)
         BSK(I) = TD*SK(I) - BSK(I)*SRDS/SDS+YR(I)*YRSK/YRSR
         EMAT(I) = TD-TD*TD*SR(I)*SR(I)/SDS+YR(I)*YR(I)/YRSR
20    CONTINUE
      SDS = DDOT(N,SK,1,BSK,1)
      DO 30 I = 1,N
         EMAT(I) = EMAT(I) - BSK(I)*BSK(I)/SDS+YK(I)*YK(I)/YKSK
30    CONTINUE
      GO TO 110
60    CONTINUE
      DO 70 I = 1,N
         BSK(I) = DIAGB(I)*SK(I)
70    CONTINUE
      SDS = DDOT(N,SK,1,BSK,1)
      DO 80 I = 1,N
         TD = DIAGB(I)
         EMAT(I) = TD - TD*TD*SK(I)*SK(I)/SDS + YK(I)*YK(I)/YKSK
80    CONTINUE
      GO TO 110
90    CONTINUE
      CALL DCOPY(N,DIAGB,1,EMAT,1)
110   CONTINUE
      IF (MODET .LT. 1) RETURN
      D1 = EMAT(1)
      DN = EMAT(1)
      DO 120 I = 1,N
         IF (EMAT(I) .LT. D1) D1 = EMAT(I)
         IF (EMAT(I) .GT. DN) DN = EMAT(I)
120   CONTINUE
      COND = DN/D1
      WRITE(ICOUT,99)
  99  FORMAT(1X)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,99)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,800) D1,DN,COND
800   FORMAT(9X,'DMIN =',1PD12.4,'  DMAX =',1PD12.4,' COND =',1PD12.4)
      WRITE(ICOUT,99)
      CALL DPWRST('XXX','BUG ')
      RETURN
      END
C
C ROUTINES TO INITIALIZE PRECONDITIONER
C
C RENAME TO AVOID NAME CONFLICT WITH DATAPLOT INITPC
C ROUTINE
C
CCCCC SUBROUTINE INITPC(DIAGB,EMAT,N,W,LW,MODET,
CCCCC*     UPD1,YKSK,GSK,YRSR,LRESET)
      SUBROUTINE INITP4(DIAGB,EMAT,N,W,LW,MODET,
     *     UPD1,YKSK,GSK,YRSR,LRESET)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION DIAGB(N),EMAT(N),W(LW)
      DOUBLE PRECISION YKSK,GSK,YRSR
      LOGICAL LRESET,UPD1
      COMMON/SUBSCR/ LGV,LZ1,LZK,LV,LSK,LYK,LDIAGB,LSR,LYR,
     *     LHYR,LHG,LHYK,LPK,LEMAT,LWTEST
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      CALL INITP3(DIAGB,EMAT,N,LRESET,YKSK,YRSR,W(LHYK),
     *     W(LSK),W(LYK),W(LSR),W(LYR),MODET,UPD1)
      RETURN
      END
C
C      LINE SEARCH ALGORITHMS OF GILL AND MURRAY
C
      SUBROUTINE LINDER(N,SFUN,SMALL,EPSMCH,RELTOL,ABSTOL,
     *     TNYTOL,ETA,SFTBND,XBND,P,GTP,X,F,ALPHA,G,NFTOTL,
     *     IFLAG,W,LW,XOBS,NOBS)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      INTEGER N,NFTOTL,IFLAG,LW
      DOUBLE PRECISION SMALL,EPSMCH,RELTOL,ABSTOL,TNYTOL,ETA,
     *     SFTBND,XBND,GTP,F,ALPHA
      DOUBLE PRECISION P(N),X(N),G(N),W(LW)
      DOUBLE PRECISION XOBS(*)
C
C     NOTE: FOR DATAPLOT, ADD XOBS AND NOBS TO CALL LIST.  THESE
C           WILL BE PASSED TO THE SFUN ROUTINE.
C
      INTEGER I,IENTRY,ITEST,L,LG,LX,NUMF,ITCNT
      DOUBLE PRECISION A,B,B1,BIG,E,FACTOR,FMIN,FPRESN,FU,
     *     FW,GMIN,GTEST1,GTEST2,GU,GW,OLDF,SCXBND,STEP,
     *     TOL,U,XMIN,XW,RMU,RTSMLL,UALPHA
      LOGICAL BRAKTD
C
C      THE FOLLOWING STANDARD FUNCTIONS AND SYSTEM FUNCTIONS ARE
C      CALLED WITHIN LINDER
C
      DOUBLE PRECISION DDOT,DSQRT
      EXTERNAL SFUN
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C      ALLOCATE THE ADDRESSES FOR LOCAL WORKSPACE
C
      LX = 1
      LG = LX + N
      LSPRNT = 0
      NPRNT  = 10000
      RTSMLL = DSQRT(SMALL)
      BIG = 1.D0/SMALL
      ITCNT = 0
C
C      SET THE ESTIMATED RELATIVE PRECISION IN F(X).
C
      FPRESN = 10.D0*EPSMCH
      NUMF = 0
      U = ALPHA
      FU = F
      FMIN = F
      GU = GTP
      RMU = 1.0D-4
C
C      FIRST ENTRY SETS UP THE INITIAL INTERVAL OF UNCERTAINTY.
C
      IENTRY = 1
10    CONTINUE
C
C TEST FOR TOO MANY ITERATIONS
C
      ITCNT = ITCNT + 1
      IFLAG = 1
      IF (ITCNT .GT. 20) GO TO 50
      IFLAG = 0
      CALL GETPTC(BIG,SMALL,RTSMLL,RELTOL,ABSTOL,TNYTOL,
     *     FPRESN,ETA,RMU,XBND,U,FU,GU,XMIN,FMIN,GMIN,
     *     XW,FW,GW,A,B,OLDF,B1,SCXBND,E,STEP,FACTOR,
     *     BRAKTD,GTEST1,GTEST2,TOL,IENTRY,ITEST)
CLSOUT
      IF (LSPRNT .GE. NPRNT) CALL LSOUT(IENTRY,ITEST,XMIN,FMIN,GMIN,
     *     XW,FW,GW,U,A,B,TOL,RELTOL,SCXBND,XBND)
C
C      IF ITEST=1, THE ALGORITHM REQUIRES THE FUNCTION VALUE TO BE
C      CALCULATED.
C
      IF (ITEST .NE. 1) GO TO 30
      UALPHA = XMIN + U
      L = LX
      DO 20 I = 1,N
         W(L) = X(I) + UALPHA*P(I)
         L = L + 1
20    CONTINUE
      CALL SFUN(N,W(LX),FU,W(LG),XOBS,NOBS)
      NUMF = NUMF + 1
      GU = DDOT(N,W(LG),1,P,1)
C
C      THE GRADIENT VECTOR CORRESPONDING TO THE BEST POINT IS
C      OVERWRITTEN IF FU IS LESS THAN FMIN AND FU IS SUFFICIENTLY
C      LOWER THAN F AT THE ORIGIN.
C
      IF (FU .LE. FMIN .AND. FU .LE. OLDF-UALPHA*GTEST1)
     *     CALL DCOPY(N,W(LG),1,G,1)
      GOTO 10
C
C      IF ITEST=2 OR 3 A LOWER POINT COULD NOT BE FOUND
C
30    CONTINUE
      NFTOTL = NUMF
      IFLAG = 1
      IF (ITEST .NE. 0) GO TO 50
C
C      IF ITEST=0 A SUCCESSFUL SEARCH HAS BEEN MADE
C
      IFLAG = 0
      F = FMIN
      ALPHA = XMIN
      DO 40 I = 1,N
         X(I) = X(I) + ALPHA*P(I)
40    CONTINUE
50    RETURN
      END
C
C
      SUBROUTINE LMQN (IFAIL, N, X, F, G, W, LW, SFUN,
     *            MSGLVL, MAXIT, MAXFUN, ETA, STEPMX, ACCRCY, XTOL,
     *            XOBS,NOBS)
      IMPLICIT          DOUBLE PRECISION (A-H,O-Z)
      INTEGER           MSGLVL, N, MAXFUN, IFAIL, LW
      DOUBLE PRECISION  X(N), G(N), W(LW), ETA, XTOL, STEPMX, F, ACCRCY
      DOUBLE PRECISION  XOBS(*)
C
C NOTE: FOR DATAPLOT, ADD XOBS AND NOBS TO THE CALL LIST.  THESE WILL
C       BE PASSED TO THE SFUN ROUTINE.
C
C THIS ROUTINE IS A TRUNCATED-NEWTON METHOD.
C THE TRUNCATED-NEWTON METHOD IS PRECONDITIONED BY A LIMITED-MEMORY
C QUASI-NEWTON METHOD (THIS PRECONDITIONING STRATEGY IS DEVELOPED
C IN THIS ROUTINE) WITH A FURTHER DIAGONAL SCALING (SEE ROUTINE NDIA3).
C FOR FURTHER DETAILS ON THE PARAMETERS, SEE ROUTINE TN.
C
      INTEGER I, ICYCLE, IOLDG, IPK, IYK, LOLDG, LPK, LSR,
     *     LWTEST, LYK, LYR, NFTOTL, NITER, NM1, NUMF, NWHY
      DOUBLE PRECISION ABSTOL, ALPHA, DIFNEW, DIFOLD, EPSMCH,
     *     EPSRED, FKEEP, FM, FNEW, FOLD, FSTOP, FTEST, GNORM, GSK,
     *     GTG, GTPNEW, OLDF, OLDGTP, ONE, PE, PEPS, PNORM, RELTOL,
     *     RTEPS, RTLEPS, RTOL, RTOLSQ, SMALL, SPE, TINY,
     *     TNYTOL, TOLEPS, XNORM, YKSK, YRSR, ZERO
      LOGICAL LRESET, UPD1
      INTEGER IPIVOT(1)
C
C THE FOLLOWING IMSL AND STANDARD FUNCTIONS ARE USED
C
      DOUBLE PRECISION DABS, DDOT, DSQRT, STEP1, DNRM2
      EXTERNAL SFUN
      COMMON /SUBSCR/ LGV,LZ1,LZK,LV,LSK,LYK,LDIAGB,LSR,LYR,
     *     LOLDG,LHG,LHYK,LPK,LEMAT,LWTEST
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C INITIALIZE PARAMETERS AND CONSTANTS
C
      IF (MSGLVL .GE. -2) THEN
         WRITE(ICOUT,99)
   99    FORMAT(1X)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,800)
800      FORMAT(' NIT   NF   CG', 9X, 'F', 21X, 'GTG')
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
      ENDIF
      CALL SETPAR(N)
      UPD1 = .TRUE.
      IRESET = 0
      NFEVAL = 0
      NMODIF = 0
      NLINCG = 0
      FSTOP = F
      ZERO = 0.D0
      ONE = 1.D0
      NM1 = N - 1
C
C WITHIN THIS ROUTINE THE ARRAY W(LOLDG) IS SHARED BY W(LHYR)
C
      LHYR = LOLDG
C
C CHECK PARAMETERS AND SET CONSTANTS
C
      CALL CHKUCP(LWTEST,MAXFUN,NWHY,N,ALPHA,EPSMCH,
     *     ETA,PEPS,RTEPS,RTOL,RTOLSQ,STEPMX,FTEST,
     *     XTOL,XNORM,X,LW,SMALL,TINY,ACCRCY)
      IF (NWHY .LT. 0) GO TO 120
      CALL SETUCR(SMALL,NFTOTL,NITER,N,F,FNEW,
     *     FM,GTG,OLDF,SFUN,G,X,XOBS,NOBS)
      FOLD = FNEW
      IF (MSGLVL .GE. 1) THEN
         WRITE(ICOUT,810) NITER,NFTOTL,NLINCG,FNEW,GTG
810      FORMAT(' ',I3,1X,I4,1X,I4,1X,1PD22.15,2X,1PD15.8)
         CALL DPWRST('XXX','BUG ')
      ENDIF
C
C CHECK FOR SMALL GRADIENT AT THE STARTING POINT.
C
      FTEST = ONE + DABS(FNEW)
      IF (GTG .LT. 1.D-4*EPSMCH*FTEST*FTEST) GO TO 90
C
C SET INITIAL VALUES TO OTHER PARAMETERS
C
      ICYCLE = NM1
      TOLEPS = RTOL + RTEPS
      RTLEPS = RTOLSQ + EPSMCH
      GNORM  = DSQRT(GTG)
      DIFNEW = ZERO
      EPSRED = 5.0D-2
      FKEEP  = FNEW
C
C SET THE DIAGONAL OF THE APPROXIMATE HESSIAN TO UNITY.
C
      IDIAGB = LDIAGB
      DO 10 I = 1,N
         W(IDIAGB) = ONE
         IDIAGB = IDIAGB + 1
10    CONTINUE
C
C ..................START OF MAIN ITERATIVE LOOP..........
C
C COMPUTE THE NEW SEARCH DIRECTION
C
      MODET = MSGLVL - 3
      CALL MODLNP(MODET,W(LPK),W(LGV),W(LZ1),W(LV),
     *     W(LDIAGB),W(LEMAT),X,G,W(LZK),
     *     N,W,LW,NITER,MAXIT,NFEVAL,NMODIF,
     *     NLINCG,UPD1,YKSK,GSK,YRSR,LRESET,SFUN,.FALSE.,IPIVOT,
     *     ACCRCY,GTPNEW,GNORM,XNORM,XOBS,NOBS)
20    CONTINUE
      CALL DCOPY(N,G,1,W(LOLDG),1)
      PNORM = DNRM2(N,W(LPK),1)
      OLDF = FNEW
      OLDGTP = GTPNEW
C
C PREPARE TO COMPUTE THE STEP LENGTH
C
      PE = PNORM + EPSMCH
C
C COMPUTE THE ABSOLUTE AND RELATIVE TOLERANCES FOR THE LINEAR SEARCH
C
      RELTOL = RTEPS*(XNORM + ONE)/PE
      ABSTOL = - EPSMCH*FTEST/(OLDGTP - EPSMCH)
C
C COMPUTE THE SMALLEST ALLOWABLE SPACING BETWEEN POINTS IN
C THE LINEAR SEARCH
C
      TNYTOL = EPSMCH*(XNORM + ONE)/PE
      SPE = STEPMX/PE
C
C SET THE INITIAL STEP LENGTH.
C
      ALPHA = STEP1(FNEW,FM,OLDGTP,SPE)
C
C PERFORM THE LINEAR SEARCH
C
      CALL LINDER(N,SFUN,SMALL,EPSMCH,RELTOL,ABSTOL,TNYTOL,
     *     ETA,ZERO,SPE,W(LPK),OLDGTP,X,FNEW,ALPHA,G,NUMF,
     *     NWHY,W,LW,XOBS,NOBS)
C
      FOLD = FNEW
      NITER = NITER + 1
      NFTOTL = NFTOTL + NUMF
      GTG = DDOT(N,G,1,G,1)
      IF (MSGLVL .GE. 1) THEN
         WRITE(ICOUT,810) NITER,NFTOTL,NLINCG,FNEW,GTG
         CALL DPWRST('XXX','BUG ')
      ENDIF
      IF (NWHY .LT. 0) GO TO 120
      IF (NWHY .EQ. 0 .OR. NWHY .EQ. 2) GO TO 30
C
C THE LINEAR SEARCH HAS FAILED TO FIND A LOWER POINT
C
      NWHY = 3
      GO TO 100
30    IF (NWHY .LE. 1) GO TO 40
      CALL SFUN(N,X,FNEW,G,XOBS,NOBS)
      NFTOTL = NFTOTL + 1
C
C TERMINATE IF MORE THAN MAXFUN EVALUTATIONS HAVE BEEN MADE
C
40    NWHY = 2
      IF (NFTOTL .GT. MAXFUN) GO TO 110
      NWHY = 0
C
C SET UP PARAMETERS USED IN CONVERGENCE AND RESETTING TESTS
C
      DIFOLD = DIFNEW
      DIFNEW = OLDF - FNEW
C
C IF THIS IS THE FIRST ITERATION OF A NEW CYCLE, COMPUTE THE
C PERCENTAGE REDUCTION FACTOR FOR THE RESETTING TEST.
C
      IF (ICYCLE .NE. 1) GO TO 50
      IF (DIFNEW .GT. 2.0D0 *DIFOLD) EPSRED = EPSRED + EPSRED
      IF (DIFNEW .LT. 5.0D-1*DIFOLD) EPSRED = 5.0D-1*EPSRED
50    CONTINUE
      GNORM = DSQRT(GTG)
      FTEST = ONE + DABS(FNEW)
      XNORM = DNRM2(N,X,1)
C
C TEST FOR CONVERGENCE
C
      IF ((ALPHA*PNORM .LT. TOLEPS*(ONE + XNORM)
     *     .AND. DABS(DIFNEW) .LT. RTLEPS*FTEST
     *     .AND. GTG .LT. PEPS*FTEST*FTEST)
     *     .OR. GTG .LT. 1.D-4*ACCRCY*FTEST*FTEST) GO TO 90
C
C COMPUTE THE CHANGE IN THE ITERATES AND THE CORRESPONDING CHANGE
C IN THE GRADIENTS
C
      ISK = LSK
      IPK = LPK
      IYK = LYK
      IOLDG = LOLDG
      DO 60 I = 1,N
         W(IYK) = G(I) - W(IOLDG)
         W(ISK) = ALPHA*W(IPK)
         IPK = IPK + 1
         ISK = ISK + 1
         IYK = IYK + 1
         IOLDG = IOLDG + 1
60    CONTINUE
C
C SET UP PARAMETERS USED IN UPDATING THE DIRECTION OF SEARCH.
C
      YKSK = DDOT(N,W(LYK),1,W(LSK),1)
      LRESET = .FALSE.
      IF (ICYCLE .EQ. NM1 .OR. DIFNEW .LT.
     *     EPSRED*(FKEEP-FNEW)) LRESET = .TRUE.
      IF (LRESET) GO TO 70
      YRSR = DDOT(N,W(LYR),1,W(LSR),1)
      IF (YRSR .LE. ZERO) LRESET = .TRUE.
70    CONTINUE
      UPD1 = .FALSE.
C
C      COMPUTE THE NEW SEARCH DIRECTION
C
      MODET = MSGLVL - 3
      CALL MODLNP(MODET,W(LPK),W(LGV),W(LZ1),W(LV),
     *     W(LDIAGB),W(LEMAT),X,G,W(LZK),
     *     N,W,LW,NITER,MAXIT,NFEVAL,NMODIF,
     *     NLINCG,UPD1,YKSK,GSK,YRSR,LRESET,SFUN,.FALSE.,IPIVOT,
     *     ACCRCY,GTPNEW,GNORM,XNORM,XOBS,NOBS)
      IF (LRESET) GO TO 80
C
C      STORE THE ACCUMULATED CHANGE IN THE POINT AND GRADIENT AS AN
C      "AVERAGE" DIRECTION FOR PRECONDITIONING.
C
      CALL DXPYTN(N,W(LSK),1,W(LSR),1)
      CALL DXPYTN(N,W(LYK),1,W(LYR),1)
      ICYCLE = ICYCLE + 1
      GOTO 20
C
C RESET
C
80    IRESET = IRESET + 1
C
C INITIALIZE THE SUM OF ALL THE CHANGES IN X.
C
      CALL DCOPY(N,W(LSK),1,W(LSR),1)
      CALL DCOPY(N,W(LYK),1,W(LYR),1)
      FKEEP = FNEW
      ICYCLE = 1
      GO TO 20
C
C ...............END OF MAIN ITERATION.......................
C
90    IFAIL = 0
      F = FNEW
      RETURN
100   OLDF = FNEW
C
C LOCAL SEARCH HERE COULD BE INSTALLED HERE
C
110    F = OLDF
C
C SET IFAIL
C
120   IFAIL = NWHY
      RETURN
      END
C
C
      SUBROUTINE LMQNBC (IFAIL, N, X, F, G, W, LW, SFUN, LOW, UP,
     *   IPIVOT, MSGLVL, MAXIT, MAXFUN, ETA, STEPMX, ACCRCY, XTOL,
     *   XOBS, NOBS)
      IMPLICIT         DOUBLE PRECISION (A-H,O-Z)
      INTEGER          MSGLVL,N,MAXFUN,IFAIL,LW
      INTEGER          IPIVOT(N)
      DOUBLE PRECISION ETA,XTOL,STEPMX,F,ACCRCY
      DOUBLE PRECISION X(N),G(N),W(LW),LOW(N),UP(N)
      DOUBLE PRECISION XOBS(*)
C
C NOTE: FOR DATAPLOT, ADD XOBS AND NOBS TO CALL LIST.  THESE WILL
C       BE PASSED TO THE SFUN CALL LIST
C
C THIS ROUTINE IS A BOUNDS-CONSTRAINED TRUNCATED-NEWTON METHOD.
C THE TRUNCATED-NEWTON METHOD IS PRECONDITIONED BY A LIMITED-MEMORY
C QUASI-NEWTON METHOD (THIS PRECONDITIONING STRATEGY IS DEVELOPED
C IN THIS ROUTINE) WITH A FURTHER DIAGONAL SCALING (SEE ROUTINE NDIA3).
C FOR FURTHER DETAILS ON THE PARAMETERS, SEE ROUTINE TNBC.
C
      INTEGER I, ICYCLE, IOLDG, IPK, IYK, LOLDG, LPK, LSR,
     *     LWTEST, LYK, LYR, NFTOTL, NITER, NM1, NUMF, NWHY
      DOUBLE PRECISION ABSTOL, ALPHA, DIFNEW, DIFOLD, EPSMCH, EPSRED,
     *     FKEEP, FLAST, FM, FNEW, FOLD, FSTOP, FTEST, GNORM, GSK,
     *     GTG, GTPNEW, OLDF, OLDGTP, ONE, PE, PEPS, PNORM, RELTOL,
     *     RTEPS, RTLEPS, RTOL, RTOLSQ, SMALL, SPE, TINY,
     *     TNYTOL, TOLEPS, XNORM, YKSK, YRSR, ZERO
      LOGICAL CONV, LRESET, UPD1, NEWCON
C
C THE FOLLOWING STANDARD FUNCTIONS AND SYSTEM FUNCTIONS ARE USED
C
      DOUBLE PRECISION DABS, DDOT, DNRM2, DSQRT, STEP1
      EXTERNAL SFUN
      COMMON/SUBSCR/ LGV, LZ1, LZK, LV, LSK, LYK, LDIAGB, LSR, LYR,
     *     LOLDG, LHG, LHYK, LPK, LEMAT, LWTEST
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C CHECK THAT INITIAL X IS FEASIBLE AND THAT THE BOUNDS ARE CONSISTENT
C
      CALL CRASH(N,X,IPIVOT,LOW,UP,IER)
      IF (IER .NE. 0) THEN
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,802)
802      FORMAT('***** ERROR IN TN/TNBC OPTIMIZATION--')
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,800)
800      FORMAT(' THERE IS NO FEASIBLE POINT; TERMINATING ALGORITHM')
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         RETURN
      ENDIF
      IF (MSGLVL .GE. 1) THEN
         WRITE(ICOUT,99)
   99    FORMAT(1X)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,810)
810      FORMAT('  NIT   NF   CG', 9X, 'F', 21X, 'GTG')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
      ENDIF
C
C INITIALIZE VARIABLES
C
      CALL SETPAR(N)
      UPD1 = .TRUE.
      IRESET = 0
      NFEVAL = 0
      NMODIF = 0
      NLINCG = 0
      FSTOP = F
      CONV = .FALSE.
      ZERO = 0.D0
      ONE = 1.D0
      NM1 = N - 1
C
C WITHIN THIS ROUTINE THE ARRAY W(LOLDG) IS SHARED BY W(LHYR)
C
      LHYR = LOLDG
C
C CHECK PARAMETERS AND SET CONSTANTS
C
      CALL CHKUCP(LWTEST,MAXFUN,NWHY,N,ALPHA,EPSMCH,
     *     ETA,PEPS,RTEPS,RTOL,RTOLSQ,STEPMX,FTEST,
     *     XTOL,XNORM,X,LW,SMALL,TINY,ACCRCY)
      IF (NWHY .LT. 0) GO TO 160
      CALL SETUCR(SMALL,NFTOTL,NITER,N,F,FNEW,
     *     FM,GTG,OLDF,SFUN,G,X,XOBS,NOBS)
      FOLD = FNEW
      FLAST = FNEW
C
C TEST THE LAGRANGE MULTIPLIERS TO SEE IF THEY ARE NON-NEGATIVE.
C BECAUSE THE CONSTRAINTS ARE ONLY LOWER BOUNDS, THE COMPONENTS
C OF THE GRADIENT CORRESPONDING TO THE ACTIVE CONSTRAINTS ARE THE
C LAGRANGE MULTIPLIERS.  AFTERWORDS, THE PROJECTED GRADIENT IS FORMED.
C
      DO 10 I = 1,N
         IF (IPIVOT(I) .EQ. 2) GO TO 10
         IF (-IPIVOT(I)*G(I) .GE. 0.D0) GO TO 10
         IPIVOT(I) = 0
10    CONTINUE
      CALL ZTIME(N,G,IPIVOT)
      GTG = DDOT(N,G,1,G,1)
      IF (MSGLVL .GE. 1)
     *    CALL MONIT(N,X,FNEW,G,NITER,NFTOTL,NFEVAL,IRESET,IPIVOT)
CCCCC*    CALL MONIT(N,X,FNEW,G,NITER,NFTOTL,NFEVAL,LRESET,IPIVOT)
C
C CHECK IF THE INITIAL POINT IS A LOCAL MINIMUM.
C
      FTEST = ONE + DABS(FNEW)
      IF (GTG .LT. 1.D-4*EPSMCH*FTEST*FTEST) GO TO 130
C
C SET INITIAL VALUES TO OTHER PARAMETERS
C
      ICYCLE = NM1
      TOLEPS = RTOL + RTEPS
      RTLEPS = RTOLSQ + EPSMCH
      GNORM  = DSQRT(GTG)
      DIFNEW = ZERO
      EPSRED = 5.0D-2
      FKEEP  = FNEW
C
C SET THE DIAGONAL OF THE APPROXIMATE HESSIAN TO UNITY.
C
      IDIAGB = LDIAGB
      DO 15 I = 1,N
         W(IDIAGB) = ONE
         IDIAGB = IDIAGB + 1
15    CONTINUE
C
C ..................START OF MAIN ITERATIVE LOOP..........
C
C COMPUTE THE NEW SEARCH DIRECTION
C
      MODET = MSGLVL - 3
      CALL MODLNP(MODET,W(LPK),W(LGV),W(LZ1),W(LV),
     *     W(LDIAGB),W(LEMAT),X,G,W(LZK),
     *     N,W,LW,NITER,MAXIT,NFEVAL,NMODIF,
     *     NLINCG,UPD1,YKSK,GSK,YRSR,LRESET,SFUN,.TRUE.,IPIVOT,
     *     ACCRCY,GTPNEW,GNORM,XNORM,XOBS,NOBS)
20    CONTINUE
      CALL DCOPY(N,G,1,W(LOLDG),1)
      PNORM = DNRM2(N,W(LPK),1)
      OLDF = FNEW
      OLDGTP = GTPNEW
C
C PREPARE TO COMPUTE THE STEP LENGTH
C
      PE = PNORM + EPSMCH
C
C COMPUTE THE ABSOLUTE AND RELATIVE TOLERANCES FOR THE LINEAR SEARCH
C
      RELTOL = RTEPS*(XNORM + ONE)/PE
      ABSTOL = - EPSMCH*FTEST/(OLDGTP - EPSMCH)
C
C COMPUTE THE SMALLEST ALLOWABLE SPACING BETWEEN POINTS IN
C THE LINEAR SEARCH
C
      TNYTOL = EPSMCH*(XNORM + ONE)/PE
      CALL STPMAX(STEPMX,PE,SPE,N,X,W(LPK),IPIVOT,LOW,UP)
C
C SET THE INITIAL STEP LENGTH.
C
      ALPHA = STEP1(FNEW,FM,OLDGTP,SPE)
C
C PERFORM THE LINEAR SEARCH
C
      CALL LINDER(N,SFUN,SMALL,EPSMCH,RELTOL,ABSTOL,TNYTOL,
     *     ETA,ZERO,SPE,W(LPK),OLDGTP,X,FNEW,ALPHA,G,NUMF,
     *     NWHY,W,LW,XOBS,NOBS)
      NEWCON = .FALSE.
      IF (DABS(ALPHA-SPE) .GT. 1.D1*EPSMCH) GO TO 30
      NEWCON = .TRUE.
      NWHY   = 0
      CALL MODZ(N,X,W(LPK),IPIVOT,EPSMCH,LOW,UP,FLAST,FNEW)
      FLAST = FNEW
C
30    IF (MSGLVL .GE. 3) THEN
         WRITE(ICOUT,820) ALPHA,PNORM
820      FORMAT('        LINESEARCH RESULTS:  ALPHA,PNORM',2(1PD12.4))
         CALL DPWRST('XXX','BUG ')
      ENDIF
      FOLD = FNEW
      NITER = NITER + 1
      NFTOTL = NFTOTL + NUMF
C
C IF REQUIRED, PRINT THE DETAILS OF THIS ITERATION
C
      IF (MSGLVL .GE. 1)
     *    CALL MONIT(N,X,FNEW,G,NITER,NFTOTL,NFEVAL,IRESET,IPIVOT)
CCCCC*    CALL MONIT(N,X,FNEW,G,NITER,NFTOTL,NFEVAL,LRESET,IPIVOT)
      IF (NWHY .LT. 0) GO TO 160
      IF (NWHY .EQ. 0 .OR. NWHY .EQ. 2) GO TO 40
C
C THE LINEAR SEARCH HAS FAILED TO FIND A LOWER POINT
C
      NWHY = 3
      GO TO 140
40    IF (NWHY .LE. 1) GO TO 50
      CALL SFUN(N,X,FNEW,G)
      NFTOTL = NFTOTL + 1
C
C TERMINATE IF MORE THAN MAXFUN EVALUATIONS HAVE BEEN MADE
C
50    NWHY = 2
      IF (NFTOTL .GT. MAXFUN) GO TO 150
      NWHY = 0
C
C SET UP PARAMETERS USED IN CONVERGENCE AND RESETTING TESTS
C
      DIFOLD = DIFNEW
      DIFNEW = OLDF - FNEW
C
C IF THIS IS THE FIRST ITERATION OF A NEW CYCLE, COMPUTE THE
C PERCENTAGE REDUCTION FACTOR FOR THE RESETTING TEST.
C
      IF (ICYCLE .NE. 1) GO TO 60
      IF (DIFNEW .GT. 2.D0*DIFOLD) EPSRED = EPSRED + EPSRED
      IF (DIFNEW .LT. 5.0D-1*DIFOLD) EPSRED = 5.0D-1*EPSRED
60    CALL DCOPY(N,G,1,W(LGV),1)
      CALL ZTIME(N,W(LGV),IPIVOT)
      GTG = DDOT(N,W(LGV),1,W(LGV),1)
      GNORM = DSQRT(GTG)
      FTEST = ONE + DABS(FNEW)
      XNORM = DNRM2(N,X,1)
C
C TEST FOR CONVERGENCE
C
      CALL CNVTST(CONV,ALPHA,PNORM,TOLEPS,XNORM,DIFNEW,RTLEPS,
     *     FTEST,GTG,PEPS,EPSMCH,GTPNEW,FNEW,FLAST,G,IPIVOT,N,ACCRCY)
      IF (CONV) GO TO 130
      CALL ZTIME(N,G,IPIVOT)
C
C COMPUTE THE CHANGE IN THE ITERATES AND THE CORRESPONDING CHANGE
C IN THE GRADIENTS
C
      IF (NEWCON) GO TO 90
      ISK = LSK
      IPK = LPK
      IYK = LYK
      IOLDG = LOLDG
      DO 70 I = 1,N
         W(IYK) = G(I) - W(IOLDG)
         W(ISK) = ALPHA*W(IPK)
         IPK = IPK + 1
         ISK = ISK + 1
         IYK = IYK + 1
         IOLDG = IOLDG + 1
70    CONTINUE
C
C SET UP PARAMETERS USED IN UPDATING THE PRECONDITIONING STRATEGY.
C
      YKSK = DDOT(N,W(LYK),1,W(LSK),1)
      LRESET = .FALSE.
      IF (ICYCLE .EQ. NM1 .OR. DIFNEW .LT.
     *     EPSRED*(FKEEP-FNEW)) LRESET = .TRUE.
      IF (LRESET) GO TO 80
      YRSR = DDOT(N,W(LYR),1,W(LSR),1)
      IF (YRSR .LE. ZERO) LRESET = .TRUE.
80    CONTINUE
      UPD1 = .FALSE.
C
C      COMPUTE THE NEW SEARCH DIRECTION
C
90    IF (UPD1 .AND. MSGLVL .GE. 3) THEN
         WRITE(ICOUT,830)
830      FORMAT(' UPD1 IS TRUE - TRIVIAL PRECONDITIONING')
         CALL DPWRST('XXX','BUG ')
      ENDIF
      IF (NEWCON .AND. MSGLVL .GE. 3) THEN
         WRITE(ICOUT,840)
840      FORMAT(' NEWCON IS TRUE - CONSTRAINT ADDED IN LINESEARCH')
         CALL DPWRST('XXX','BUG ')
      ENDIF
      MODET = MSGLVL - 3
      CALL MODLNP(MODET,W(LPK),W(LGV),W(LZ1),W(LV),
     *     W(LDIAGB),W(LEMAT),X,G,W(LZK),
     *     N,W,LW,NITER,MAXIT,NFEVAL,NMODIF,
     *     NLINCG,UPD1,YKSK,GSK,YRSR,LRESET,SFUN,.TRUE.,IPIVOT,
     *     ACCRCY,GTPNEW,GNORM,XNORM,XOBS,NOBS)
      IF (NEWCON) GO TO 20
      IF (LRESET) GO TO 110
C
C COMPUTE THE ACCUMULATED STEP AND ITS CORRESPONDING
C GRADIENT DIFFERENCE.
C
      CALL DXPYTN(N,W(LSK),1,W(LSR),1)
      CALL DXPYTN(N,W(LYK),1,W(LYR),1)
      ICYCLE = ICYCLE + 1
      GOTO 20
C
C RESET
C
110   IRESET = IRESET + 1
C
C INITIALIZE THE SUM OF ALL THE CHANGES IN X.
C
      CALL DCOPY(N,W(LSK),1,W(LSR),1)
      CALL DCOPY(N,W(LYK),1,W(LYR),1)
      FKEEP = FNEW
      ICYCLE = 1
      GO TO 20
C
C ...............END OF MAIN ITERATION.......................
C
130   IFAIL = 0
      F = FNEW
      RETURN
140   OLDF = FNEW
C
C LOCAL SEARCH COULD BE INSTALLED HERE
C
150   F = OLDF
      IF (MSGLVL .GE. 1) CALL MONIT(N,X,
     *     F,G,NITER,NFTOTL,NFEVAL,IRESET,IPIVOT)
C
C SET IFAIL
C
160   IFAIL = NWHY
      RETURN
      END
C
C
      SUBROUTINE LSOUT(ILOC,ITEST,XMIN,FMIN,GMIN,XW,FW,GW,U,A,
     *     B,TOL,EPS,SCXBD,XLAMDA)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION XMIN,FMIN,GMIN,XW,FW,GW,U,A,B,
     *     TOL,EPS,SCXBD,XLAMDA
C
C ERROR PRINTOUTS FOR GETPTC
C
      DOUBLE PRECISION YA,YB,YBND,YW,YU
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      YU = XMIN + U
      YA = A + XMIN
      YB = B + XMIN
      YW = XW + XMIN
      YBND = SCXBD + XMIN
C
      WRITE(ICOUT,99)
   99 FORMAT(1X)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,99)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,99)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,800)
800   FORMAT(' OUTPUT FROM LINEAR SEARCH')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,810)
810   FORMAT('  TOL AND EPS')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,811) TOL,EPS
811   FORMAT(2D25.14)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,820)
820   FORMAT('  CURRENT UPPER AND LOWER BOUNDS')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,821) YA,YB
821   FORMAT(2D25.14)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,830)
830   FORMAT('  STRICT UPPER BOUND')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,831) YBND
831   FORMAT(D25.14)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,840)
840   FORMAT('  XW, FW, GW')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,841) YW,FW,GW
841   FORMAT(3D25.14)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,850)
850   FORMAT('  XMIN, FMIN, GMIN')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,851) XMIN,FMIN,GMIN
851   FORMAT(3D25.14)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,860)
860   FORMAT('  NEW ESTIMATE')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,861) YU
861   FORMAT(2D25.14)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,870)
870   FORMAT('  ILOC AND ITEST')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,871) ILOC,ITEST
871   FORMAT(2I3)
      CALL DPWRST('XXX','BUG ')
      RETURN
      END
C
C
      DOUBLE PRECISION FUNCTION MCHPR1()
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION X
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C RETURNS THE VALUE OF EPSMCH, WHERE EPSMCH IS THE SMALLEST POSSIBLE
C REAL NUMBER SUCH THAT 1.0 + EPSMCH .GT. 1.0
C
C FOR VAX
C
CCCCC MCHPR1 = 1.D-17
C
C FOR SUN
C
      MCHPR1 = 1.0842021724855D-19
      RETURN
      END
C
C THE VECTORS SK AND YK, ALTHOUGH NOT IN THE CALL,
C ARE USED (VIA THEIR POSITION IN W) BY THE ROUTINE MSOLVE.
C
      SUBROUTINE MODLNP(MODET,ZSOL,GV,R,V,DIAGB,EMAT,
     *     X,G,ZK,N,W,LW,NITER,MAXIT,NFEVAL,NMODIF,NLINCG,
     *     UPD1,YKSK,GSK,YRSR,LRESET,SFUN,BOUNDS,IPIVOT,ACCRCY,
     *     GTP,GNORM,XNORM,XOBS,NOBS)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      INTEGER MODET,N,NITER,IPIVOT(1)
      DOUBLE PRECISION ZSOL(N),G(N),GV(N),R(N),V(N),DIAGB(N),W(LW)
      DOUBLE PRECISION EMAT(N),ZK(N),X(N),ACCRCY
      DOUBLE PRECISION ALPHA,BETA,DELTA,GSK,GTP,PR,
     *     QOLD,QNEW,QTEST,RHSNRM,RNORM,RZ,RZOLD,TOL,VGV,YKSK,YRSR
      DOUBLE PRECISION GNORM,XNORM
      DOUBLE PRECISION DDOT,DNRM2
      DOUBLE PRECISION XOBS(*)
      LOGICAL FIRST,UPD1,LRESET,BOUNDS
      EXTERNAL SFUN
C
C     NOTE: FOR DATAPLOT, ADD XOBS AND NOBS TO CALL LIST.  THESE
C           WILL BE PASSED TO SFUN.
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C THIS ROUTINE PERFORMS A PRECONDITIONED CONJUGATE-GRADIENT
C ITERATION IN ORDER TO SOLVE THE NEWTON EQUATIONS FOR A SEARCH
C DIRECTION FOR A TRUNCATED-NEWTON ALGORITHM.  WHEN THE VALUE OF THE
C QUADRATIC MODEL IS SUFFICIENTLY REDUCED,
C THE ITERATION IS TERMINATED.
C
C PARAMETERS
C
C MODET       - INTEGER WHICH CONTROLS AMOUNT OF OUTPUT
C ZSOL        - COMPUTED SEARCH DIRECTION
C G           - CURRENT GRADIENT
C GV,GZ1,V    - SCRATCH VECTORS
C R           - RESIDUAL
C DIAGB,EMAT  - DIAGONAL PRECONDITONING MATRIX
C NITER       - NONLINEAR ITERATION #
C FEVAL       - VALUE OF QUADRATIC FUNCTION
C
C *************************************************************
C INITIALIZATION
C *************************************************************
C
C GENERAL INITIALIZATION
C
      IF (MODET .GT. 0) THEN
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(*,800)
800      FORMAT(' ENTERING MODLNP')
         CALL DPWRST('XXX','BUG ')
      ENDIF
      IF (MAXIT .EQ. 0) RETURN
      FIRST = .TRUE.
      RHSNRM = GNORM
      TOL = 1.D-12
      QOLD = 0.D0
C
C INITIALIZATION FOR PRECONDITIONED CONJUGATE-GRADIENT ALGORITHM
C
CCCCC CALL INITPC(DIAGB,EMAT,N,W,LW,MODET,
      CALL INITP4(DIAGB,EMAT,N,W,LW,MODET,
     *            UPD1,YKSK,GSK,YRSR,LRESET)
      DO 10 I = 1,N
         R(I) = -G(I)
         V(I) = 0.D0
         ZSOL(I) = 0.D0
10    CONTINUE
C
C ************************************************************
C MAIN ITERATION
C ************************************************************
C
      DO 30 K = 1,MAXIT
         NLINCG = NLINCG + 1
         IF (MODET .GT. 1) THEN
            WRITE(ICOUT,99)
            CALL DPWRST('XXX','BUG ')
            WRITE(ICOUT,99)
            CALL DPWRST('XXX','BUG ')
            WRITE(*,810) K
810         FORMAT(' ### ITERATION ',I2,' ###')
            CALL DPWRST('XXX','BUG ')
         ENDIF
C
C CG ITERATION TO SOLVE SYSTEM OF EQUATIONS
C
         IF (BOUNDS) CALL ZTIME(N,R,IPIVOT)
         CALL MSOLVE(R,ZK,N,W,LW,UPD1,YKSK,GSK,
     *                 YRSR,LRESET,FIRST)
         IF (BOUNDS) CALL ZTIME(N,ZK,IPIVOT)
         RZ = DDOT(N,R,1,ZK,1)
         IF (RZ/RHSNRM .LT. TOL) GO TO 80
         IF (K .EQ. 1) BETA = 0.D0
         IF (K .GT. 1) BETA = RZ/RZOLD
         DO 20 I = 1,N
            V(I) = ZK(I) + BETA*V(I)
20       CONTINUE
         IF (BOUNDS) CALL ZTIME(N,V,IPIVOT)
         CALL GTIMS(V,GV,N,X,G,W,LW,SFUN,FIRST,DELTA,ACCRCY,XNORM,
     1              XOBS,NOBS)
         IF (BOUNDS) CALL ZTIME(N,GV,IPIVOT)
         NFEVAL = NFEVAL + 1
         VGV = DDOT(N,V,1,GV,1)
         IF (VGV/RHSNRM .LT. TOL) GO TO 50
         CALL NDIA3(N,EMAT,V,GV,R,VGV,MODET)
C
C COMPUTE LINEAR STEP LENGTH
C
         ALPHA = RZ / VGV
         IF (MODET .GE. 1) THEN
            WRITE(ICOUT,820) ALPHA
820         FORMAT(' ALPHA',1PD16.8)
            CALL DPWRST('XXX','BUG ')
         ENDIF
C
C COMPUTE CURRENT SOLUTION AND RELATED VECTORS
C
         CALL DAXPY(N,ALPHA,V,1,ZSOL,1)
         CALL DAXPY(N,-ALPHA,GV,1,R,1)
C
C TEST FOR CONVERGENCE
C
         GTP = DDOT(N,ZSOL,1,G,1)
         PR = DDOT(N,R,1,ZSOL,1)
         QNEW = 5.D-1 * (GTP + PR)
         QTEST = K * (1.D0 - QOLD/QNEW)
         IF (QTEST .LT. 0.D0) GO TO 70
         QOLD = QNEW
         IF (QTEST .LE. 5.D-1) GO TO 70
C
C PERFORM CAUTIONARY TEST
C
         IF (GTP .GT. 0) GO TO 40
         RZOLD = RZ
30    CONTINUE
C
C TERMINATE ALGORITHM
C
      K = K-1
      GO TO 70
C
C TRUNCATE ALGORITHM IN CASE OF AN EMERGENCY
C
40    IF (MODET .GE. -1) THEN
         WRITE(ICOUT,830) K
830      FORMAT(' G(T)Z POSITIVE AT ITERATION ',I2,
     *     ' - TRUNCATING METHOD')
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
   99    FORMAT(1X)
         CALL DPWRST('XXX','BUG ')
      ENDIF
      CALL DAXPY(N,-ALPHA,V,1,ZSOL,1)
      GTP = DDOT(N,ZSOL,1,G,1)
      GO TO 90
50    CONTINUE
      IF (MODET .GT. -2) THEN
         WRITE(ICOUT,840)
840      FORMAT(' ',10X,'HESSIAN NOT POSITIVE-DEFINITE')
         CALL DPWRST('XXX','BUG ')
      ENDIF
60    IF (K .GT. 1) GO TO 70
      CALL MSOLVE(G,ZSOL,N,W,LW,UPD1,YKSK,GSK,YRSR,LRESET,FIRST)
      CALL NEGVEC(N,ZSOL)
      IF (BOUNDS) CALL ZTIME(N,ZSOL,IPIVOT)
      GTP = DDOT(N,ZSOL,1,G,1)
70    CONTINUE
      IF (MODET .GE. -1) THEN
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(*,850) K,RNORM
850      FORMAT(8X,'MODLAN TRUNCATED AFTER ',I3,' ITERATIONS',
     *          '  RNORM = ',1PD14.6)
         CALL DPWRST('XXX','BUG ')
      ENDIF
      GO TO 90
80    CONTINUE
      IF (MODET .GE. -1) THEN
         WRITE(*,860)
860      FORMAT(' PRECONDITIONING NOT POSITIVE-DEFINITE')
         CALL DPWRST('XXX','BUG ')
      ENDIF
      IF (K .GT. 1) GO TO 70
      CALL DCOPY(N,G,1,ZSOL,1)
      CALL NEGVEC(N,ZSOL)
      IF (BOUNDS) CALL ZTIME(N,ZSOL,IPIVOT)
      GTP = DDOT(N,ZSOL,1,G,1)
      GO TO 70
C
C STORE (OR RESTORE) DIAGONAL PRECONDITIONING
C
90    CONTINUE
      CALL DCOPY(N,EMAT,1,DIAGB,1)
      RETURN
      END
C
C
      SUBROUTINE MODZ(N,X,P,IPIVOT,EPSMCH,LOW,UP,FLAST,FNEW)
      IMPLICIT         DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION X(N), P(N), EPSMCH, DABS, TOL, LOW(N), UP(N),
     *                 FLAST, FNEW
      INTEGER          IPIVOT(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C UPDATE THE CONSTRAINT MATRIX IF A NEW CONSTRAINT IS ENCOUNTERED
C
      DO 10 I = 1,N
         IF (IPIVOT(I) .NE. 0) GO TO 10
         IF (P(I) .EQ. 0.D0) GO TO 10
         IF (P(I) .GT. 0.D0) GO TO 5
         TOL = 1.D1 * EPSMCH * (DABS(LOW(I)) + 1.D0)
         IF (X(I)-LOW(I) .GT. TOL) GO TO 10
         FLAST = FNEW
         IPIVOT(I) = -1
         X(I) = LOW(I)
         GO TO 10
5        TOL = 1.D1 * EPSMCH * (DABS(UP(I)) + 1.D0)
         IF (UP(I)-X(I) .GT. TOL) GO TO 10
         FLAST = FNEW
         IPIVOT(I) = 1
         X(I) = UP(I)
10    CONTINUE
      RETURN
      END
C
C
      SUBROUTINE MONIT(N,X,F,G,NITER,NFTOTL,NFEVAL,IRESET,IPIVOT)
C
C PRINT RESULTS OF CURRENT ITERATION
C
      IMPLICIT         DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION X(N),F,G(N),GTG
      INTEGER          IPIVOT(N)
      INTEGER          IRESET
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      GTG = 0.D0
      DO 10 I = 1,N
         IF (IPIVOT(I) .NE. 0) GO TO 10
         GTG = GTG + G(I)*G(I)
10    CONTINUE
      WRITE(*,800) NITER,NFTOTL,NFEVAL,F,GTG
800   FORMAT(' ',I4,1X,I4,1X,I4,1X,1PD22.15,2X,1PD15.8)
      CALL DPWRST('XXX','BUG ')
      RETURN
      END
      SUBROUTINE MSLV(G,Y,N,SK,YK,DIAGB,SR,YR,HYR,HG,HYK,
     *     UPD1,YKSK,GSK,YRSR,LRESET,FIRST)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION G(N),Y(N)
C
C THIS ROUTINE ACTS AS A PRECONDITIONING STEP FOR THE
C LINEAR CONJUGATE-GRADIENT ROUTINE.  IT IS ALSO THE
C METHOD OF COMPUTING THE SEARCH DIRECTION FROM THE
C GRADIENT FOR THE NON-LINEAR CONJUGATE-GRADIENT CODE.
C IT REPRESENTS A TWO-STEP SELF-SCALED BFGS FORMULA.
C
      DOUBLE PRECISION DDOT,YKSK,GSK,YRSR,RDIAGB,YKHYK,GHYK,
     *     YKSR,YKHYR,YRHYR,GSR,GHYR
      DOUBLE PRECISION SK(N),YK(N),DIAGB(N),SR(N),YR(N),HYR(N),HG(N),
     *     HYK(N),ONE
      LOGICAL LRESET,UPD1,FIRST
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      IF (UPD1) GO TO 100
      ONE = 1.D0
      GSK = DDOT(N,G,1,SK,1)
      IF (LRESET) GO TO 60
C
C COMPUTE HG AND HY WHERE H IS THE INVERSE OF THE DIAGONALS
C
      DO 57 I = 1,N
         RDIAGB = 1.0D0/DIAGB(I)
         HG(I) = G(I)*RDIAGB
         IF (FIRST) HYK(I) = YK(I)*RDIAGB
         IF (FIRST) HYR(I) = YR(I)*RDIAGB
57    CONTINUE
      IF (FIRST) YKSR = DDOT(N,YK,1,SR,1)
      IF (FIRST) YKHYR = DDOT(N,YK,1,HYR,1)
      GSR = DDOT(N,G,1,SR,1)
      GHYR = DDOT(N,G,1,HYR,1)
      IF (FIRST) YRHYR = DDOT(N,YR,1,HYR,1)
      CALL SSBFGS(N,ONE,SR,YR,HG,HYR,YRSR,
     *     YRHYR,GSR,GHYR,HG)
      IF (FIRST) CALL SSBFGS(N,ONE,SR,YR,HYK,HYR,YRSR,
     *     YRHYR,YKSR,YKHYR,HYK)
      YKHYK = DDOT(N,HYK,1,YK,1)
      GHYK = DDOT(N,HYK,1,G,1)
      CALL SSBFGS(N,ONE,SK,YK,HG,HYK,YKSK,
     *     YKHYK,GSK,GHYK,Y)
      RETURN
60    CONTINUE
C
C COMPUTE GH AND HY WHERE H IS THE INVERSE OF THE DIAGONALS
C
      DO 65 I = 1,N
         RDIAGB = 1.D0/DIAGB(I)
         HG(I) = G(I)*RDIAGB
         IF (FIRST) HYK(I) = YK(I)*RDIAGB
65    CONTINUE
      IF (FIRST) YKHYK = DDOT(N,YK,1,HYK,1)
      GHYK = DDOT(N,G,1,HYK,1)
      CALL SSBFGS(N,ONE,SK,YK,HG,HYK,YKSK,
     *     YKHYK,GSK,GHYK,Y)
      RETURN
100   CONTINUE
      DO 110 I = 1,N
110      Y(I) = G(I) / DIAGB(I)
      RETURN
      END
C
C
      SUBROUTINE MSOLVE(G,Y,N,W,LW,UPD1,YKSK,GSK,
     *     YRSR,LRESET,FIRST)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION G(N),Y(N),W(LW),YKSK,GSK,YRSR
      LOGICAL UPD1,LRESET,FIRST
C
C THIS ROUTINE SETS UPT THE ARRAYS FOR MSLV
C
      COMMON/SUBSCR/ LGV,LZ1,LZK,LV,LSK,LYK,LDIAGB,LSR,LYR,
     *     LHYR,LHG,LHYK,LPK,LEMAT,LWTEST
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      CALL MSLV(G,Y,N,W(LSK),W(LYK),W(LDIAGB),W(LSR),W(LYR),W(LHYR),
     *     W(LHG),W(LHYK),UPD1,YKSK,GSK,YRSR,LRESET,FIRST)
      RETURN
      END
C
C
      SUBROUTINE NDIA3(N,E,V,GV,R,VGV,MODET)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION E(N),V(N),GV(N),R(N),VGV,VR,DDOT
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C UPDATE THE PRECONDITIOING MATRIX BASED ON A DIAGONAL VERSION
C OF THE BFGS QUASI-NEWTON UPDATE.
C
      VR = DDOT(N,V,1,R,1)
      DO 10 I = 1,N
         E(I) = E(I) - R(I)*R(I)/VR + GV(I)*GV(I)/VGV
         IF (E(I) .GT. 1.D-6) GO TO 10
         IF (MODET .GT. -2) THEN
            WRITE(*,800) E(I)
800         FORMAT(' *** EMAT NEGATIVE:  ',1PD16.8)
            CALL DPWRST('XXX','BUG ')
         ENDIF
         E(I) = 1.D0
10    CONTINUE
      RETURN
      END
C
C      SERVICE ROUTINES FOR OPTIMIZATION
C
      SUBROUTINE NEGVEC(N,V)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      INTEGER N
      DOUBLE PRECISION V(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C NEGATIVE OF THE VECTOR V
C
      INTEGER I
      DO 10 I = 1,N
         V(I) = -V(I)
10    CONTINUE
      RETURN
      END
C
C
      SUBROUTINE SETPAR(N)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      INTEGER LSUB(14)
      COMMON/SUBSCR/ LSUB,LWTEST
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C SET UP PARAMETERS FOR THE OPTIMIZATION ROUTINE
C
      DO 10 I = 1,14
          LSUB(I) = (I-1)*N + 1
10    CONTINUE
      LWTEST = LSUB(14) + N - 1
      RETURN
      END
C
C
      SUBROUTINE SETUCR(SMALL,NFTOTL,NITER,N,F,FNEW,
     *            FM,GTG,OLDF,SFUN,G,X,XOBS,NOBS)
      IMPLICIT         DOUBLE PRECISION (A-H,O-Z)
      INTEGER          NFTOTL,NITER,N
      DOUBLE PRECISION F,FNEW,FM,GTG,OLDF,SMALL
      DOUBLE PRECISION G(N),X(N)
      DOUBLE PRECISION XOBS(*)
      EXTERNAL         SFUN
C
C     NOTE: FOR DATAPLOT, ADD XOBS AND NOBS TO CALL LIST.  THESE
C           WILL BE PASSED TO SFUN.
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C CHECK INPUT PARAMETERS, COMPUTE THE INITIAL FUNCTION VALUE, SET
C CONSTANTS FOR THE SUBSEQUENT MINIMIZATION
C
      FM = F
C
C COMPUTE THE INITIAL FUNCTION VALUE
C
      CALL SFUN(N,X,FNEW,G,XOBS,NOBS)
      NFTOTL = 1
C
C SET CONSTANTS FOR LATER
C
      NITER = 0
      OLDF = FNEW
      GTG = DDOT(N,G,1,G,1)
      RETURN
      END
C
C
      SUBROUTINE SSBFGS(N,GAMMA,SJ,YJ,HJV,HJYJ,YJSJ,YJHYJ,
     *     VSJ,VHYJ,HJP1V)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      INTEGER N
      DOUBLE PRECISION GAMMA,YJSJ,YJHYJ,VSJ,VHYJ
      DOUBLE PRECISION SJ(N),YJ(N),HJV(N),HJYJ(N),HJP1V(N)
C
C SELF-SCALED BFGS
C
      INTEGER I
      DOUBLE PRECISION BETA,DELTA
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      DELTA = (1.D0 + GAMMA*YJHYJ/YJSJ)*VSJ/YJSJ
     *     - GAMMA*VHYJ/YJSJ
      BETA = -GAMMA*VSJ/YJSJ
      DO 10 I = 1,N
         HJP1V(I) = GAMMA*HJV(I) + DELTA*SJ(I) + BETA*HJYJ(I)
10    CONTINUE
      RETURN
      END
C
C
      DOUBLE PRECISION FUNCTION STEP1(FNEW,FM,GTP,SMAX)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION FNEW,FM,GTP,SMAX
C
C ********************************************************
C STEP1 RETURNS THE LENGTH OF THE INITIAL STEP TO BE TAKEN ALONG THE
C VECTOR P IN THE NEXT LINEAR SEARCH.
C ********************************************************
C
      DOUBLE PRECISION ALPHA,D,EPSMCH
      DOUBLE PRECISION DABS,MCHPR1
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
      EPSMCH = MCHPR1()
      D = DABS(FNEW-FM)
      ALPHA = 1.D0
      IF (2.D0*D .LE. (-GTP) .AND. D .GE. EPSMCH)
     *     ALPHA = -2.D0*D/GTP
      IF (ALPHA .GE. SMAX) ALPHA = SMAX
      STEP1 = ALPHA
      RETURN
      END
C
C
      SUBROUTINE STPMAX(STEPMX,PE,SPE,N,X,P,IPIVOT,LOW,UP)
      IMPLICIT         DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION LOW(N),UP(N),X(N),P(N),STEPMX,PE,SPE,T
      INTEGER          IPIVOT(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C COMPUTE THE MAXIMUM ALLOWABLE STEP LENGTH
C
      SPE = STEPMX / PE
C SPE IS THE STANDARD (UNCONSTRAINED) MAX STEP
      DO 10 I = 1,N
         IF (IPIVOT(I) .NE. 0) GO TO 10
         IF (P(I) .EQ. 0.D0) GO TO 10
         IF (P(I) .GT. 0.D0) GO TO 5
         T = LOW(I) - X(I)
         IF (T .GT. SPE*P(I)) SPE = T / P(I)
         GO TO 10
5        T = UP(I) - X(I)
         IF (T .LT. SPE*P(I)) SPE = T / P(I)
10    CONTINUE
      RETURN
      END
C
C
      SUBROUTINE TNBC (IERROR, N, X, F, G, W, LW, SFUN, LOW, UP, IPIVOT,
     1                 XOBS,NOBS)
      IMPLICIT          DOUBLE PRECISION (A-H,O-Z)
      INTEGER           IERROR, N, LW, IPIVOT(N)
      DOUBLE PRECISION  X(N), G(N), F, W(LW), LOW(N), UP(N)
      DOUBLE PRECISION  XOBS(*)
C
C NOTE: FOR DATAPLOT, ADD XOBS AND NOBS TO THE CALL LIST.  THESE
C       WILL BE PASSED TO THE SFUN ROUTINE.
C
C THIS ROUTINE SOLVES THE OPTIMIZATION PROBLEM
C
C   MINIMIZE     F(X)
C      X
C   SUBJECT TO   LOW <= X <= UP
C
C WHERE X IS A VECTOR OF N REAL VARIABLES.  THE METHOD USED IS
C A TRUNCATED-NEWTON ALGORITHM (SEE "NEWTON-TYPE MINIMIZATION VIA
C THE LANCZOS ALGORITHM" BY S.G. NASH (TECHNICAL REPORT 378, MATH.
C THE LANCZOS METHOD" BY S.G. NASH (SIAM J. NUMER. ANAL. 21 (1984),
C PP. 770-778).  THIS ALGORITHM FINDS A LOCAL MINIMUM OF F(X).  IT DOES
C NOT ASSUME THAT THE FUNCTION F IS CONVEX (AND SO CANNOT GUARANTEE A
C GLOBAL SOLUTION), BUT DOES ASSUME THAT THE FUNCTION IS BOUNDED BELOW.
C IT CAN SOLVE PROBLEMS HAVING ANY NUMBER OF VARIABLES, BUT IT IS
C ESPECIALLY USEFUL WHEN THE NUMBER OF VARIABLES (N) IS LARGE.
C
C SUBROUTINE PARAMETERS:
C
C IERROR  - (INTEGER) ERROR CODE
C           ( 0 => NORMAL RETURN
C           ( 2 => MORE THAN MAXFUN EVALUATIONS
C           ( 3 => LINE SEARCH FAILED TO FIND LOWER
C           (          POINT (MAY NOT BE SERIOUS)
C           (-1 => ERROR IN INPUT PARAMETERS
C N       - (INTEGER) NUMBER OF VARIABLES
C X       - (REAL*8) VECTOR OF LENGTH AT LEAST N; ON INPUT, AN INITIAL
C           ESTIMATE OF THE SOLUTION; ON OUTPUT, THE COMPUTED SOLUTION.
C G       - (REAL*8) VECTOR OF LENGTH AT LEAST N; ON OUTPUT, THE FINAL
C           VALUE OF THE GRADIENT
C F       - (REAL*8) ON INPUT, A ROUGH ESTIMATE OF THE VALUE OF THE
C           OBJECTIVE FUNCTION AT THE SOLUTION; ON OUTPUT, THE VALUE
C           OF THE OBJECTIVE FUNCTION AT THE SOLUTION
C W       - (REAL*8) WORK VECTOR OF LENGTH AT LEAST 14*N
C LW      - (INTEGER) THE DECLARED DIMENSION OF W
C SFUN    - A USER-SPECIFIED SUBROUTINE THAT COMPUTES THE FUNCTION
C           AND GRADIENT OF THE OBJECTIVE FUNCTION.  IT MUST HAVE
C           THE CALLING SEQUENCE
C             SUBROUTINE SFUN (N, X, F, G)
C             INTEGER           N
C             DOUBLE PRECISION  X(N), G(N), F
C
C           NOTE: FOR DATAPLOT, ADD XOBS AND NOBS TO CALL LIST
C
C LOW, UP - (REAL*8) VECTORS OF LENGTH AT LEAST N CONTAINING
C           THE LOWER AND UPPER BOUNDS ON THE VARIABLES.  IF
C           THERE ARE NO BOUNDS ON A PARTICULAR VARIABLE, SET
C           THE BOUNDS TO -1.D38 AND 1.D38, RESPECTIVELY.
C IPIVOT  - (INTEGER) WORK VECTOR OF LENGTH AT LEAST N, USED
C           TO RECORD WHICH VARIABLES ARE AT THEIR BOUNDS.
C
C THIS IS AN EASY-TO-USE DRIVER FOR THE MAIN OPTIMIZATION ROUTINE
C LMQNBC.  MORE EXPERIENCED USERS WHO WISH TO CUSTOMIZE PERFORMANCE
C OF THIS ALGORITHM SHOULD CALL LMQBC DIRECTLY.
C
C----------------------------------------------------------------------
C THIS ROUTINE SETS UP ALL THE PARAMETERS FOR THE TRUNCATED-NEWTON
C ALGORITHM.  THE PARAMETERS ARE:
C
C ETA    - SEVERITY OF THE LINESEARCH
C MAXFUN - MAXIMUM ALLOWABLE NUMBER OF FUNCTION EVALUATIONS
C XTOL   - DESIRED ACCURACY FOR THE SOLUTION X*
C STEPMX - MAXIMUM ALLOWABLE STEP IN THE LINESEARCH
C ACCRCY - ACCURACY OF COMPUTED FUNCTION VALUES
C MSGLVL - CONTROLS QUANTITY OF PRINTED OUTPUT
C          0 = NONE, 1 = ONE LINE PER MAJOR ITERATION.
C MAXIT  - MAXIMUM NUMBER OF INNER ITERATIONS PER STEP
C
      DOUBLE PRECISION  ETA, ACCRCY, XTOL, STEPMX, DSQRT, MCHPR1
      EXTERNAL          SFUN
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C SET PARAMETERS FOR THE OPTIMIZATION ROUTINE
C
      MAXIT = N/2
      IF (MAXIT .GT. 50) MAXIT = 50
      IF (MAXIT .LE. 0) MAXIT = 1
      MSGLVL = 1
      MAXFUN = 150*N
      ETA = .25D0
      STEPMX = 1.D1
      ACCRCY = 1.D2*MCHPR1()
      XTOL = DSQRT(ACCRCY)
C
C MINIMIZE FUNCTION
C
      CALL LMQNBC (IERROR, N, X, F, G, W, LW, SFUN, LOW, UP, IPIVOT,
     *            MSGLVL, MAXIT, MAXFUN, ETA, STEPMX, ACCRCY, XTOL,
     *            XOBS,NOBS)
C
C PRINT RESULTS
C
      IF (IERROR .NE. 0) THEN
         WRITE(ICOUT,99)
   99    FORMAT(1X)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,800) IERROR
800      FORMAT(' ERROR CODE =', I3)
         CALL DPWRST('XXX','BUG ')
      ENDIF
      WRITE(ICOUT,99)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,810) F
810   FORMAT(' OPTIMAL FUNCTION VALUE = ', 1PD22.15)
      CALL DPWRST('XXX','BUG ')
      IF (MSGLVL .LT. 1) RETURN
      WRITE(ICOUT,820)
820   FORMAT(10X, 'CURRENT SOLUTION IS (AT MOST 10 COMPONENTS)')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,821)
821   FORMAT(14X, 'I', 11X, 'X(I)')
      NMAX = 10
      IF (N .LT. NMAX) NMAX = N
      WRITE(ICOUT,830) (I,X(I),I=1,NMAX)
830   FORMAT(10X, I5, 2X, 1PD22.15)
      CALL DPWRST('XXX','BUG ')
      RETURN
      END
C
C   TRUNCATED-NEWTON METHOD:  SUBROUTINES
C   FOR OTHER MACHINES, MODIFY ROUTINE MCHPR1 (MACHINE EPSILON)
C   WRITTEN BY:  STEPHEN G. NASH
C                OPERATIONS RESEARCH AND APPLIED STATISTICS DEPT.
C                GEORGE MASON UNIVERSITY
C                FAIRFAX, VA 22030
C******************************************************************
      SUBROUTINE TN (IERROR, N, X, F, G, W, LW, SFUN, XOBS, NOBS)
      IMPLICIT          DOUBLE PRECISION (A-H,O-Z)
      INTEGER           IERROR, N, LW
      DOUBLE PRECISION  X(N), G(N), F, W(LW)
      DOUBLE PRECISION  XOBS(*)
C
C NOTE: FOR DATAPLOT, ADD XOBS AND NOBS TO THE CALL LIST.  THESE
C       WILL BE PASSED TO THE SFUN ROUTINE.
C
C THIS ROUTINE SOLVES THE OPTIMIZATION PROBLEM
C
C            MINIMIZE F(X)
C               X
C
C WHERE X IS A VECTOR OF N REAL VARIABLES.  THE METHOD USED IS
C A TRUNCATED-NEWTON ALGORITHM (SEE "NEWTON-TYPE MINIMIZATION VIA
C THE LANCZOS METHOD" BY S.G. NASH (SIAM J. NUMER. ANAL. 21 (1984),
C PP. 770-778).  THIS ALGORITHM FINDS A LOCAL MINIMUM OF F(X).  IT DOES
C NOT ASSUME THAT THE FUNCTION F IS CONVEX (AND SO CANNOT GUARANTEE A
C GLOBAL SOLUTION), BUT DOES ASSUME THAT THE FUNCTION IS BOUNDED BELOW.
C IT CAN SOLVE PROBLEMS HAVING ANY NUMBER OF VARIABLES, BUT IT IS
C ESPECIALLY USEFUL WHEN THE NUMBER OF VARIABLES (N) IS LARGE.
C
C SUBROUTINE PARAMETERS:
C
C IERROR - (INTEGER) ERROR CODE
C          ( 0 => NORMAL RETURN)
C          ( 2 => MORE THAN MAXFUN EVALUATIONS)
C          ( 3 => LINE SEARCH FAILED TO FIND
C          (          LOWER POINT (MAY NOT BE SERIOUS)
C          (-1 => ERROR IN INPUT PARAMETERS)
C N      - (INTEGER) NUMBER OF VARIABLES
C X      - (REAL*8) VECTOR OF LENGTH AT LEAST N; ON INPUT, AN INITIAL
C          ESTIMATE OF THE SOLUTION; ON OUTPUT, THE COMPUTED SOLUTION.
C G      - (REAL*8) VECTOR OF LENGTH AT LEAST N; ON OUTPUT, THE FINAL
C          VALUE OF THE GRADIENT
C F      - (REAL*8) ON INPUT, A ROUGH ESTIMATE OF THE VALUE OF THE
C          OBJECTIVE FUNCTION AT THE SOLUTION; ON OUTPUT, THE VALUE
C          OF THE OBJECTIVE FUNCTION AT THE SOLUTION
C W      - (REAL*8) WORK VECTOR OF LENGTH AT LEAST 14*N
C LW     - (INTEGER) THE DECLARED DIMENSION OF W
C SFUN   - A USER-SPECIFIED SUBROUTINE THAT COMPUTES THE FUNCTION
C          AND GRADIENT OF THE OBJECTIVE FUNCTION.  IT MUST HAVE
C          THE CALLING SEQUENCE
C             SUBROUTINE SFUN (N, X, F, G)
C             INTEGER           N
C             DOUBLE PRECISION  X(N), G(N), F
C
C          NOTE: ADD XOBS AND NOBS TO THE CALL LIST
C
C THIS IS AN EASY-TO-USE DRIVER FOR THE MAIN OPTIMIZATION ROUTINE
C LMQN.  MORE EXPERIENCED USERS WHO WISH TO CUSTOMIZE PERFORMANCE
C OF THIS ALGORITHM SHOULD CALL LMQN DIRECTLY.
C
C----------------------------------------------------------------------
C THIS ROUTINE SETS UP ALL THE PARAMETERS FOR THE TRUNCATED-NEWTON
C ALGORITHM.  THE PARAMETERS ARE:
C
C ETA    - SEVERITY OF THE LINESEARCH
C MAXFUN - MAXIMUM ALLOWABLE NUMBER OF FUNCTION EVALUATIONS
C XTOL   - DESIRED ACCURACY FOR THE SOLUTION X*
C STEPMX - MAXIMUM ALLOWABLE STEP IN THE LINESEARCH
C ACCRCY - ACCURACY OF COMPUTED FUNCTION VALUES
C MSGLVL - DETERMINES QUANTITY OF PRINTED OUTPUT
C          0 = NONE, 1 = ONE LINE PER MAJOR ITERATION.
C MAXIT  - MAXIMUM NUMBER OF INNER ITERATIONS PER STEP
C
      DOUBLE PRECISION ETA, ACCRCY, XTOL, STEPMX, DSQRT, MCHPR1
      EXTERNAL         SFUN
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C SET UP PARAMETERS FOR THE OPTIMIZATION ROUTINE
C
      MAXIT = N/2
      IF (MAXIT .GT. 50) MAXIT = 50
      IF (MAXIT .LE. 0) MAXIT = 1
      MSGLVL = 1
      MAXFUN = 150*N
      ETA = .25D0
      STEPMX = 1.D1
      ACCRCY = 1.D2*MCHPR1()
      XTOL = DSQRT(ACCRCY)
C
C MINIMIZE THE FUNCTION
C
      CALL LMQN (IERROR, N, X, F, G, W, LW, SFUN,
     *     MSGLVL, MAXIT, MAXFUN, ETA, STEPMX, ACCRCY, XTOL,
     *     XOBS,NOBS)
C
C PRINT THE RESULTS
C
      IF (IERROR .NE. 0) THEN
         WRITE(ICOUT,99)
   99    FORMAT(1X)
         CALL DPWRST('XXX','BUG ')
         WRITE(ICOUT,99)
         CALL DPWRST('XXX','BUG ')
         WRITE(*,800) IERROR
800      FORMAT(' ERROR CODE =', I3)
         CALL DPWRST('XXX','BUG ')
      ENDIF
      WRITE(ICOUT,99)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,99)
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,810) F
810   FORMAT(' OPTIMAL FUNCTION VALUE = ', 1PD22.15)
      CALL DPWRST('XXX','BUG ')
      IF (MSGLVL .LT. 1) RETURN
      WRITE(ICOUT,820)
820   FORMAT(10X, 'CURRENT SOLUTION IS (AT MOST 10 COMPONENTS)')
      CALL DPWRST('XXX','BUG ')
      WRITE(ICOUT,821)
821   FORMAT(14X, 'I', 11X, 'X(I)')
      CALL DPWRST('XXX','BUG ')
      NMAX = 10
      IF (N .LT. NMAX) NMAX = N
      WRITE(ICOUT,830) (I,X(I),I=1,NMAX)
830   FORMAT(10X, I5, 2X, 1PD22.15)
      CALL DPWRST('XXX','BUG ')
      RETURN
      END
C
C
      SUBROUTINE ZTIME(N,X,IPIVOT)
      IMPLICIT         DOUBLE PRECISION (A-H,O-Z)
      DOUBLE PRECISION X(N)
      INTEGER          IPIVOT(N)
C
      REAL CPUMIN
      REAL CPUMAX
      CHARACTER*4 IFEEDB
      CHARACTER*4 IPRINT
      CHARACTER*240 ICOUT
C
      COMMON /MACH/IRD,IPR,CPUMIN,CPUMAX,NUMBPC,NUMCPW,NUMBPW
      COMMON /PRINT/IFEEDB,IPRINT
      COMMON /TEXTOU/ICOUT,NCOUT,ILOUT
C
C THIS ROUTINE MULTIPLIES THE VECTOR X BY THE CONSTRAINT MATRIX Z
C
      DO 10 I = 1,N
         IF (IPIVOT(I) .NE. 0) X(I) = 0.D0
10    CONTINUE
      RETURN
      END
