#include "SensorBase.h"
#include "TMP36.h"
#include "HC_SR04.h"
#include "SRF02.h"
#include "RFMxx.h"
#include "Voltage.h"
#include "PowerSave.h"
#include "LevelSenderLib.h"
#include "Wire.h"

#define NAME         "LevelSender"
#define VERSION      "1.0.6"

// --- Configuration ---------------------------------------------------------
#define SENSOR_TYPE           4     // 2=SRF02   4=HC-SR04
#define TRANSMIT_INTERVAL     60    // Transmit interval in seconds
#define SENSOR_ID             1     // ID of this LevelSender ( 1 ... 15)
#define DATA_RATE             3     // Bit coded: 1 = 17241   2 = 9579   4 = 8842  
#define VOLTAGE_DIVIDER       11.0  // Measure or try until voltage is right
#define TMP36_OFFSET          0     // Apply offset to measured temperature of TMP36
#define USE_LED               0     // Set to 0 to save battery power

// --- Connecions -------------------------------------------------------------
#define POWER_5V_PIN          7    // Pin for the 5V Step-Up power supply
#define POWER_PULLUP_PIN      6    // Pin for the SRF02 pull up power supply
#define TMP36_PIN             A0   // Pin for the temperature 
#define VOLTAGE_PIN           A3   // Pin for the battery voltage 
#define HC_SR04_TRIGGER_PIN   5    // Trigger pin HC-SR04
#define HC_SR04_ECHO_PIN      4    // Echo pin HC-SR04
#define LED_PIN               15   // LED
#define SCL_PIN               19   // I2C
#define SDA_PIN               18   // I2C

// --- Globals ----------------------------------------------------------------
TMP36 tmp36(TMP36_PIN, TMP36::ReferenceVoltage11, TMP36::HalfDegrees);
HC_SR04 sr04(HC_SR04_TRIGGER_PIN, HC_SR04_ECHO_PIN);
SRF02 srf02(112, SRF_MODE_CENTIMETERS);
RFMxx rfm(11, 12, 13, 10, 2);
Voltage voltage = Voltage(VOLTAGE_PIN, Voltage::ReferenceVoltage11, VOLTAGE_DIVIDER);

void setup() {
  LevelSenderLib::SetDebugMode(false);

  pinMode(LED_PIN, OUTPUT);
  
  // 5V Step-Up
  pinMode(POWER_5V_PIN, OUTPUT);
  digitalWrite(POWER_5V_PIN, LOW);

  pinMode(POWER_PULLUP_PIN, OUTPUT);
  digitalWrite(POWER_PULLUP_PIN, LOW);
  
  analogReference(INTERNAL);
  
  // Wait until everything is up and running
  delay(200);

  Serial.begin(57600);
  Serial.println("\n\nStarted");

  // Initialize the RFM12
  rfm.InitialzeLaCrosse();

}

void loop() {
  // We just woke up so wait a moment to get stable
  delay(50);
  
  // Flash the LED
  if (USE_LED) {
    digitalWrite(LED_PIN, HIGH);
    delay(5);
    digitalWrite(LED_PIN, LOW);
  }

  // Enable 5V power for the sensors and the RFM12
  digitalWrite(POWER_5V_PIN, HIGH);
  digitalWrite(POWER_PULLUP_PIN, HIGH);
  unsigned long startMillis = millis();
  delay(100);

  LevelSenderLib::Frame frame;
  frame.Header = 11;
  frame.ID = SENSOR_ID;

  // Measure the voltage
  for (int i = 0; i < 5; i++) { voltage.ReadVoltage(); }
  frame.Voltage = voltage.ReadVoltage();
 
  // Measure the temperature
  for (int i = 0; i < 5; i++) { tmp36.ReadTemperature(); }
  frame.Temperature = tmp36.ReadTemperature() + TMP36_OFFSET;

  // Measure the distance
  frame.Level = -1.0;
  if (SENSOR_TYPE == 2) {
    // Warm up
    for (int i = 0; i < 2; i++) { srf02.ReadDistance(); }

    // Measure
    frame.Level = srf02.ReadDistance();
  }
  else if (SENSOR_TYPE == 4) {
    // Warm up
    for (int i = 0; i < 3; i++) { sr04.ReadDistance(); delay(25); }

    // Measure
    frame.Level = sr04.ReadDistance(frame.Temperature);
  }

  // Send the measured values to the serial port
  Serial.print("Values: ");

  Serial.print(frame.Level, 1);
  Serial.print(" cm   ");

  Serial.print(frame.Temperature);
  Serial.print(" C   ");

  Serial.print(frame.Voltage);
  Serial.print(" V");

  Serial.println();

  // Encode it
  byte bytes[LevelSenderLib::FRAME_LENGTH];
  LevelSenderLib::EncodeFrame(&frame, bytes);

  // Check if valid
  // Temperature: -40 ... +60 C
  // Level:         0 ... 400
  // Battery:       0 ... 13V
  bool isValid = true;

  if (frame.Temperature < -40.0 || frame.Temperature > 60.0) {
    isValid = false;
  }
  if (frame.Level < 0.0 || frame.Level > 400.0) {
    isValid = false;
  }
  if (frame.Voltage < 0.0 || frame.Voltage > 13.0) {
    isValid = false;
  }

  if (isValid) {
    // Send with all configured data rates
    if (bitRead(DATA_RATE, 0)) {
      rfm.SetDataRate(17241ul);
      rfm.SendArray(bytes, LevelSenderLib::FRAME_LENGTH);
      rfm.SendArray(bytes, LevelSenderLib::FRAME_LENGTH);
      Serial.println("Sent with 17241");
    }
    if (bitRead(DATA_RATE, 1)) {
      rfm.SetDataRate(9579ul);
      rfm.SendArray(bytes, LevelSenderLib::FRAME_LENGTH);
      rfm.SendArray(bytes, LevelSenderLib::FRAME_LENGTH);
      Serial.println("Sent with 9579");
    }
    if (bitRead(DATA_RATE, 2)) {
      rfm.SetDataRate(8842ul);
      rfm.SendArray(bytes, LevelSenderLib::FRAME_LENGTH);
      rfm.SendArray(bytes, LevelSenderLib::FRAME_LENGTH);
      Serial.println("Sent with 8842");
    }

    Serial.print(millis() - startMillis);
    Serial.println(" ms powered");
    Serial.println();

  }

  Serial.flush();
  
  // RFM and 5V off
  rfm.PowerDown();
  digitalWrite(POWER_PULLUP_PIN, LOW);
  digitalWrite(POWER_5V_PIN, LOW);
  digitalWrite(SCL_PIN, LOW);
  digitalWrite(SDA_PIN, LOW);

  
  // Good night and sleep well
  PowerSave.Sleep(TRANSMIT_INTERVAL * 1000ul);
}




