/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.painter.text;

import java.awt.*;
import java.awt.image.BufferedImage;

import javax.swing.JComponent;

/**
 * Base interface for Substance text painters.
 * 
 * @author Kirill Grouchnikov
 * @since version 4.2
 */
public interface SubstanceTextPainter {
	/**
	 * Callback for custom background painting.
	 * 
	 * @author Kirill Grouchnikov
	 */
	public static interface BackgroundPaintingCallback {
		/**
		 * Performs custom background painting on the specified surface.
		 * 
		 * @param g
		 *            Graphic context.
		 */
		public void paintBackground(Graphics g);
	}

	/**
	 * Background painting callback that draws an image at the specified
	 * location.
	 * 
	 * @author Kirill Grouchnikov
	 */
	public static class ImageBackgroundPaintingCallback implements
			BackgroundPaintingCallback {
		/**
		 * Image to paint.
		 */
		private BufferedImage backgroundImage;

		/**
		 * Image anchor.
		 */
		private Point backgroundImageAnchor;

		/**
		 * Alpha channel.
		 */
		private float alpha;

		/**
		 * Creates a new background painting callback.
		 * 
		 * @param backgroundImage
		 *            Image to paint.
		 * @param backgroundImageAnchor
		 *            Image anchor.
		 * @param alpha
		 *            Alpha channel.
		 */
		public ImageBackgroundPaintingCallback(BufferedImage backgroundImage,
				Point backgroundImageAnchor, float alpha) {
			super();
			this.backgroundImage = backgroundImage;
			this.backgroundImageAnchor = backgroundImageAnchor;
			this.alpha = alpha;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.painter.text.SubstanceTextPainter.BackgroundPaintingCallback#paintBackground(java.awt.Graphics)
		 */
		public void paintBackground(Graphics g) {
			Graphics2D g2d = (Graphics2D) g.create();
			g2d.setComposite(AlphaComposite.getInstance(
					AlphaComposite.SRC_OVER, this.alpha));
			if (this.backgroundImageAnchor == null)
				g2d.drawImage(this.backgroundImage, 0, 0, null);
			else
				g2d.drawImage(this.backgroundImage,
						this.backgroundImageAnchor.x,
						this.backgroundImageAnchor.y, null);
			g2d.dispose();
		}
	}

	/**
	 * Re-initializes the contents of this painter. Is usually called at the
	 * beginning of the component painting cycle. The painter implementation is
	 * expected to remove all registered callbacks.
	 * 
	 * @param comp
	 *            Component.
	 * @param clip
	 *            Clip to apply. The painter implementation should respect this
	 *            clip, especially on components such as lists or tables that
	 *            use cell renderers. In these components, the specific cell is
	 *            usually much smaller than the whole component, and the
	 *            painting is done on each visible cell in a loop.
	 * @param toEnforceRenderOnNoTexts
	 *            If <code>true</code>, the painter implementation <b>must</b>
	 *            fill the background and invoke all callbacks even when there
	 *            are no attached texts or all texts are empty.
	 * @see AbstractTextPainter
	 */
	public void init(JComponent comp, Rectangle clip,
			boolean toEnforceRenderOnNoTexts);

	/**
	 * Returns indication whether this painter requires background to be fully
	 * set before painting the text (with {@link #renderSurface(Graphics)}).
	 * 
	 * @return <code>true</code> if this painter requires background to be
	 *         fully set before painting the text, <code>false</code>
	 *         otherwise.
	 */
	public boolean needsBackgroundImage();

	/**
	 * Sets the fill color of the background.
	 * 
	 * @param comp
	 *            Component.
	 * @param backgroundFillColor
	 *            Fill color for the background.
	 * @param toOverlayWatermark
	 *            If <code>true</code>, the background fill should be
	 *            overlayed with the current watermark.
	 * @param watermarkOffsetX
	 *            The X offset for the watermark relative to the screen position
	 *            of the component. This is relevant for cell-based components
	 *            such as lists, tables and trees.
	 * @param watermarkOffsetY
	 *            The Y offset for the watermark relative to the screen position
	 *            of the component. This is relevant for cell-based components
	 *            such as lists, tables and trees.
	 */
	public void setBackgroundFill(JComponent comp, Color backgroundFillColor,
			boolean toOverlayWatermark, int watermarkOffsetX,
			int watermarkOffsetY);

	/**
	 * Attaches the specified background painting callback.
	 * 
	 * @param backgroundPaintingCallback
	 *            Background painting callback to attach.
	 */
	public void attachCallback(
			BackgroundPaintingCallback backgroundPaintingCallback);

	/**
	 * Attaches the specified text to paint.
	 * 
	 * @param comp
	 *            Component.
	 * @param textRect
	 *            Text rectangle.
	 * @param text
	 *            The text itself.
	 * @param mnemonicIndex
	 *            Optional mnemonic index. Can be -1 if no mnemonic.
	 * @param font
	 *            Font for this text.
	 * @param color
	 *            Color for this text.
	 * @param clip
	 *            Clip rectangle for this text.
	 */
	public void attachText(JComponent comp, Rectangle textRect, String text,
			int mnemonicIndex, java.awt.Font font, java.awt.Color color,
			java.awt.Rectangle clip);

	/**
	 * Attaches the specified vertical text to paint.
	 * 
	 * @param comp
	 *            Component.
	 * @param textRect
	 *            Text rectangle.
	 * @param text
	 *            The text itself.
	 * @param mnemonicIndex
	 *            Optional mnemonic index. Can be -1 if no mnemonic.
	 * @param font
	 *            Font for this text.
	 * @param color
	 *            Color for this text.
	 * @param clip
	 *            Clip rectangle for this text.
	 * @param isFromBottomToTop
	 *            If <code>true</code>, the text will be painted from bottom
	 *            to top (so that the top of the letters is on the left). If
	 *            <code>false</code>, the text will be painted from top to
	 *            bottom (so that the top of the letters is on the right).
	 */
	public void attachVerticalText(JComponent comp, Rectangle textRect,
			String text, int mnemonicIndex, java.awt.Font font,
			java.awt.Color color, java.awt.Rectangle clip,
			boolean isFromBottomToTop);

	/**
	 * Paints the surface. The painter implementation should respect the clip
	 * set in {@link #init(JComponent, Rectangle)}, background fill set in
	 * {@link #setBackgroundFill(JComponent, Color, boolean, int, int)}, invoke
	 * the background callbacks added with
	 * {@link #attachCallback(BackgroundPaintingCallback)} and paint all the
	 * texts added with
	 * {@link #attachText(JComponent, Rectangle, String, int, Font, Color)}.
	 * 
	 * @param g
	 *            Graphic context.
	 */
	public void renderSurface(Graphics g);

	/**
	 * Returns <code>true</code> if this text painter uses native font
	 * rendering.
	 * 
	 * @return <code>true</code> if this text painter uses native font
	 *         rendering.
	 */
	public boolean isNative();
	
	/**
	 * Disposes the resources allocated by this text painter.
	 */
	public void dispose();
	
	public java.awt.Dimension getTextBounds(Component comp, java.awt.Font font, String text);
}
