/*
 *   blobject.c
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                         
 *   blwtt.h
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
//  This class encapsulates everything needed to track call restrictions
//  for several devices.  It is built up from an array of Device objects.
//  All calls to the Blacklist API are redirected through this object.
//  An overview of operations can be found in mwbl.txt.
//
//  The public member functions of this class are protected by a
//  semiphore so that only one thread may operate at a time.  This
//  protects the data used by dialing devices from being reset by a
//  viewing utility while it is being used.  This also protects the
//  viewing utility from reading data while the devices are changing it.
//
//****************************************************************************

#include <blobject.h>  // Header file for this class
#include <idevice.h>   // Header file for the individual device class
#include <mwwttbl.h>   // Header file for WTT functions
#include <stdio.h>     // for debug stuff
#include <string.h>    // for debug stuff
#include <ctype.h>

//extern WTTsettings WTT;

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  Reset
//
//     This function resets the blacklist and reloads the WT settings.
//     It does not reset any existing device registration.
//
//
//     Params:   none
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG objReset (BLobject *pBL)
{
  ULONG  ulError;
  USHORT  usDevice;  // used as an array index, the compiler expects 16 bits
  
  objWaitSemiphore (pBL);
  
  MW_SYSLOG_1(TRACE_MWMBL,"blobject::objReset entry\n");
  for (usDevice = 0; usDevice < MAX_NUMBER_OF_DEVICES; usDevice++) {
    // Overhead device settings are set when the device registers so they can
    // not be reset when the device is individually reset by Device::Reset since
    // the registration is still valid.  Device::Reset does not affect the
    // registration.
    
    ulError = devReset(&pBL->DeviceArray[usDevice]);
    if (ulError) {
      MW_SYSLOG_3(TRACE_MWMBL,"blobject::objReset, ERROR Device#%x with ulError %lx\n", usDevice, ulError);
      objEndSemiphore(pBL); // cancel the semiphore for this session
      return ulError;
    }
  }
  
  //----------------------------------------------------------------------------
  // Now load the blacklist settings from the WT Table
  ulError = wttReset(&pBL->WTT);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objReset, ERROR call to wttReset with ulError %lu\n", ulError);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return ulError;
  }

  objEndSemiphore(pBL); // cancel the semiphore for this session
  return BL_SUCCESSFUL;
  
} 

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  ResetDevice
//
//     This function resets call restrictions for a device, it does not
//     reset the high level information about the device such as device
//     type and name.
//
//
//     Params:   ulDeviceHandle - The registered device handle.
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG objResetDevice (BLobject *pBL, ULONG ulDeviceHandle)
{
  ULONG ulError;

  objWaitSemiphore (pBL);

  //Check to make sure the WT Table settings loaded correctly.
  if (! wttIsWTTloaded(&pBL->WTT)) {
    MW_SYSLOG_1(TRACE_MWMBL,"blobject::objResetDevice ERROR - WTT not loaded\n");
    return BLERR_WTT_READ_FAILED;
  }
  
  if ((ulDeviceHandle > 0) && (ulDeviceHandle <= MAX_NUMBER_OF_DEVICES)) {
    // The Device::Reset function will mark the device as initialized without
    // affecting the registration.
    ulError = devReset(&pBL->DeviceArray[(int)ulDeviceHandle-1]);
    if (ulError) {
      MW_SYSLOG_3(TRACE_MWMBL,"blobject::objResetDevice, ERROR call to devReset for device #%lx with ulError %lx\n", ulDeviceHandle, ulError);
      objEndSemiphore(pBL); // cancel the semiphore for this session
      return ulError;
    }
  } else {
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return BLERR_INVALID_DEVICE_HANDLE; // out of range
  }
  
  objEndSemiphore(pBL); // cancel the semiphore for this session
  return BL_SUCCESSFUL;
} 


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  RegisterDevice
//
//     This function needs to be called first to register a device.  Each
//     device such as modem, fax, and voice needs to be registered.  The
//     handle returned is used to identify the memory structure used while
//     working with unsuccessful number status.  Also, some countries make
//     distictions in the call restrictions for certain devices.  In this
//     case there is typically a more severe restriction on fax machines
//     then on modems or voice calls.
//
//     A semiphore is implemented in the Device class to prevent the same
//     Device memory from being passed out to two devices.
//
//
//
//     Params:   ulDeviceType      - The device type defined above.
//
//               szDeviceName      - The name that the device is known by
//                                   in the system such as in the config.
//                                   This will be used by the blacklist viewer
//                                   in identifying status for a particular
//                                   device/line.
//
//               lpulDeviceHandle  - Pointer to where the ulDeviceHandle needs
//                                   to be returned.  This is the device
//                                   registration number.
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG objRegisterDevice  (BLobject *pBL,
			  ULONG ulDeviceType,
			  char FAR *lpszDeviceName,
			  ULONG FAR *lpulDeviceHandle)
     
{
  ULONG ulError;
  USHORT usDeviceHandle;  // used as an array index, the compiler expects 16 bits
  
  objWaitSemiphore (pBL);

  ulError=wttReset(&pBL->WTT);
  if (ulError) return ulError;
  
  //Check to make sure the WT Table settings loaded correctly.
  if ( ! wttIsWTTloaded(&pBL->WTT)) {
    MW_SYSLOG_1(TRACE_MWMBL,"blobject::objRegisterDevice, ERROR WTT not loaded\n");
    return BLERR_WTT_READ_FAILED;
  }
  
  // First we need to step through the Device array to find an available slot
  for (usDeviceHandle = 1; usDeviceHandle <= MAX_NUMBER_OF_DEVICES; usDeviceHandle++) {
    if (devIsDeviceReserved(&pBL->DeviceArray[usDeviceHandle-1]) == FALSE) {
      // OK, we found a potetially available handle.  Due to races this is not
      // guaranteed.  Next we will attempt to register.  If we get the
      // registration then we are done, otherwise we need to keep looking.
      ulError = devRegister(&pBL->DeviceArray[usDeviceHandle-1],ulDeviceType, lpszDeviceName);
      
      if (ulError == BLERR_DEVICE_REGISTRATION_FAILED) continue;
      else if (ulError) {
        MW_SYSLOG_2(TRACE_MWMBL,"blobject::objRegisterDevice, ERROR failed call to devRegister with ulError %lx\n", ulError);
        objEndSemiphore(pBL); // cancel the semiphore for this session
        return ulError;
      }
      
      // At this point we are registered, let's return the handle
      *lpulDeviceHandle = (ULONG) usDeviceHandle;

      MW_SYSLOG_3(TRACE_MWMBL,"blobject::objRegisterDevice, device number %x is %s\n", usDeviceHandle, lpszDeviceName);
      MW_SYSLOG_2(TRACE_MWMBL,"blobject::objRegisterDevice, device type is %lx\n", ulDeviceType);
      
      objEndSemiphore(pBL); // cancel the semiphore for this session
      return BL_SUCCESSFUL;
    }
  } 

  // If we dropped out of the for loop without registering then everything must be
  // in use.
  objEndSemiphore(pBL); // cancel the semiphore for this session
  return BLERR_NO_AVAIL_DEVICE_HANDLES;

} //end RegisterDevice


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  UnRegisterDevice
//
//     This function will unregister a device from the system.
//
//
//     Params:   ulDeviceHandle - The registered device handle.
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG objUnRegisterDevice (BLobject *pBL,ULONG ulDeviceHandle)
{
  objWaitSemiphore (pBL);
  
  //Check to make sure the WT Table settings loaded correctly.
  if ( ! wttIsWTTloaded(&pBL->WTT)) {
    MW_SYSLOG_1(TRACE_MWMBL,"blobject::objUnRegisterDevice, ERROR WTT not loaded\n");
    return BLERR_WTT_READ_FAILED;
  }
  
  // Is the specified handle within range?
  if ((ulDeviceHandle == 0) || (ulDeviceHandle > MAX_NUMBER_OF_DEVICES)) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objUnRegisterDevice, ERROR specifed ulDeviceHandle %lx is out of range\n", ulDeviceHandle);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return BLERR_INVALID_DEVICE_HANDLE;
  }
  
  objEndSemiphore(pBL); // cancel the semiphore for this session
  return devUnregister(&pBL->DeviceArray[(int)ulDeviceHandle-1]);
} 


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  RequestDialPermission
//
//     This function is used to request permission to dial.  Permission is
//     granted if BL_DIAL_GRANTED is returned.  In this case, a valid
//     ulDialHandle will be returned which needs to be used in reporting
//     DialResults.
//
//     In the case of a manual dial, permission will always be granted.
//     This is used to obtain the ulDialHandle which can then be used
//     to clean calling restrictions for the number if the manual dial
//     succeeded.
//
//
//     Params:   ulDeviceHandle - The registered device handle.
//               ulDialType     - The type of dial:  MANUAL_DIAL or AUTOMATIC_DIAL.
//               pszPhoneNumber - A pointer to the phone number that dialing is requested for.
//               lpulDialStatus - A pointer to return the status of the request:
//                                BL_DIAL_GRANTED       - OK to dial
//                                BL_NUMBER_BLACKLISTED - permission denied
//                                BL_NUMBER_DELAYED     - permission denied for
//                                                        lpulDelayTime more seconds
//
//               lpulDialHandle - A pointer to the returned dial handle.
//               lpulDelayTime  - A pointer to the returned delay time in seconds.
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG objRequestDialPermission (BLobject *pBL,
				ULONG ulDeviceHandle,
				ULONG ulDialType,
				char  FAR *lpszPhoneNumber,
				ULONG FAR *lpulDialStatus,
				ULONG FAR *lpulDialHandle,
				ULONG FAR *lpulDelayTime)
{
  ULONG ulError;

  objWaitSemiphore(pBL);

  //Check to make sure the WT Table settings loaded correctly.
  if (! wttIsWTTloaded(&pBL->WTT)) {
    MW_SYSLOG_1(TRACE_MWMBL,"blobject::objRequestDialPermission, ERROR WTT not loaded\n");
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return BLERR_WTT_READ_FAILED;
  }
  
  // Initialize these controls in case a user does not check return codes.
  *lpulDialStatus = BL_NUMBER_BLACKLISTED;
  *lpulDialHandle = NO_DIAL_HANDLE;
  *lpulDelayTime  = 0;

  // First we must do all the overhead checking to make sure everything is OK
  // Check the device handle
  if ((ulDeviceHandle == 0) || (ulDeviceHandle > MAX_NUMBER_OF_DEVICES)) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objRequestDialPermission, ERROR device handle %lx out of range\n", ulDeviceHandle);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return BLERR_INVALID_DEVICE_HANDLE;
  }
  
  // Check to make sure the device handle is reserved (registered)
  if (! devIsDeviceReserved(&pBL->DeviceArray[(int)ulDeviceHandle-1])) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objRequestDialPermission, ERROR device handle %lx is not registered\n", ulDeviceHandle);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return BLERR_DEVICE_HANDLE_NOT_IN_USE;
  }
  
  // Next we need to scrub the phone number to remove non-digit characters.
  ulError = objCleanDialString (pBL,lpszPhoneNumber);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objRequestDialPermission, ERROR  failed call to objCleanDialString with ulError %lx\n", ulError);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return ulError;
  }
  
  // Now we pass this request to the Device object to determine if
  // it is OK to dial.
  ulError=devRequestDialPermission(&pBL->DeviceArray[(int)ulDeviceHandle-1],ulDialType,
				lpszPhoneNumber, lpulDialStatus, lpulDialHandle, lpulDelayTime);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objRequestDialPermission - failed call to RequestDialPermission with ulError %lx\n", ulError);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return ulError;
  }

  objEndSemiphore(pBL); // cancel the semiphore for this session
  return BL_SUCCESSFUL;
} 


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  DialResults
//
//     This function is used to handle the results of a dial from the list
//     in mwblapi.h.  For manual dials, unsuccessful results do not change any
//     phone number status, however, a successful result will clear call
//     restrictions for auto-dialing.
//
//
//
//     Params:   ulDeviceHandle - The registered device handle.
//               lpulDialHandle - The dial handle from RequestDialPermission.
//               ulDialStatus   - The dial status defined in mwblapi.h as
//                                   SUCCESSFUL
//                                   NO_DIAL_TONE
//                                   LINE_BUSY
//                                   USER_ABORT
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG objDialResults (BLobject *pBL,ULONG ulDeviceHandle, ULONG ulDialHandle,
		      ULONG ulDialStatus)
{
  ULONG ulError;

  objWaitSemiphore(pBL);

  if (! wttIsWTTloaded(&pBL->WTT)) {
    MW_SYSLOG_1(TRACE_MWMBL,"blobject::objDialResults, ERROR WTT not loaded\n");
    return BLERR_WTT_READ_FAILED;
  }

  // First we have to do overhead checking to make sure everything seems OK
  // Check the device handle
  if ((ulDeviceHandle == 0) || (ulDeviceHandle > MAX_NUMBER_OF_DEVICES)) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objDialResults, ERROR device handle %lx out of range\n", ulDeviceHandle);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return BLERR_INVALID_DEVICE_HANDLE;
  }
  
  // Check to make sure the device handle is reserved (registered)
  if (! devIsDeviceReserved(&pBL->DeviceArray[(int)ulDeviceHandle-1])) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objDialResults, ERROR device handle %lx is not registered\n", ulDeviceHandle);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return BLERR_DEVICE_HANDLE_NOT_IN_USE;
  }
  
  // At this point we will let the device object handle the dial results.
  ulError=devDialResults(&pBL->DeviceArray[(int)ulDeviceHandle-1],ulDialHandle, ulDialStatus);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objDialResults, ERROR failed call to devDialResults with ulError %lx\n", ulError);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return ulError;
  }
  
  objEndSemiphore(pBL); // cancel the semiphore for this session
  return BL_SUCCESSFUL;
} 


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  GetInfo
//
//     This function is used to get information about the call restrictions
//     in effect for a device.
//
//
//     Params:   ulDeviceHandle - A possible device handle.  If no device
//                                currently uses this the handle,
//                                BLERR_HANDLE_NOT_IN_USE is returned.
//
//               lpBLsettings   - A pointer to where to return the settings.
//
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG objGetInfo(BLobject *pBL, ULONG ulDeviceHandle,
		 struct BL_Settings FAR *lpBLsettings)
{
  ULONG ulError;
  
  objWaitSemiphore(pBL);

  //Check to make sure the WT Table settings loaded correctly.
  if (! wttIsWTTloaded(&pBL->WTT)) {
    MW_SYSLOG_1(TRACE_MWMBL,"blobject::objGetInfo, ERROR WTT not loaded\n");
    return BLERR_WTT_READ_FAILED;
  }

  // First we have to do overhead checking to make sure everything seems OK
  // Check the device handle 
  if ((ulDeviceHandle == 0) || (ulDeviceHandle > MAX_NUMBER_OF_DEVICES)) { 
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objGetInfo, ERROR device handle %lx out of range\n", ulDeviceHandle);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return BLERR_INVALID_DEVICE_HANDLE;
  }
  
  // OK, it seems we have a good device handle, let's call the device object
  // to fill in the struct BL_Settings
  ulError=devGetInfo(&pBL->DeviceArray[(int)ulDeviceHandle-1],lpBLsettings);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objGetInfo, ERROR  failed call to devGetInfo with ulError %lx\n", ulError);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return ulError;
  }

  objEndSemiphore(pBL); // cancel the semiphore for this session
  return BL_SUCCESSFUL;
} //end GetInfo

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  GetNumberList
//
//     This function is used to get a list of numbers with call restrictions.
//     Memory must be provided by the calling routine.  This function will
//     fill in up to ulMaxListSize number of NumberListItem structures in
//     the specified array with call restricted information.  The first
//     structure in the array with a NULL phone number ends the list.
//
//
//
//     Params:   ulDeviceHandle - The registered device handle.
//
//               ulMaxListSize  - The size of the array pointed to by lpNumberList.
//                                The maximum number of phone numbers tracked.
//
//               lpNumberList   - A pointer to the array where call restriction
//                                information needs to be returned.
//
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG objGetNumberList (BLobject *pBL, ULONG ulDeviceHandle, ULONG ulMaxListSize,
			struct BL_NumberListItem FAR *lpNumberList)
{
  ULONG ulError;
  
  objWaitSemiphore(pBL);

  // Do the overhead checking
  //Check to make sure the WT Table settings loaded correctly.
  if (! wttIsWTTloaded(&pBL->WTT)) {
    MW_SYSLOG_1(TRACE_MWMBL,"blobject::objGetNumberList, ERROR WTT not loaded\n");
    return BLERR_WTT_READ_FAILED;
  }
  
  // Check the device handle
  if ((ulDeviceHandle == 0) || (ulDeviceHandle > MAX_NUMBER_OF_DEVICES)) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objGetNumberList, ERROR device handle %lx out of range\n", ulDeviceHandle);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return BLERR_INVALID_DEVICE_HANDLE;
  }
  
  // OK, now pass the request to the device class to handle
  ulError=devGetNumberList(&pBL->DeviceArray[((int)ulDeviceHandle)-1],ulMaxListSize, lpNumberList);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"blobject::objGetNumberList, ERROR failed call to devGetNumberList with ulError %lx\n", ulError);
    objEndSemiphore(pBL); // cancel the semiphore for this session
    return ulError;
  }

  objEndSemiphore(pBL); // cancel the semiphore for this session
  return BL_SUCCESSFUL;
}

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  WaitSemiphore
//
//     This function allows only one function to operate at a time.  If it
//     is called while another function is in operation, then a wait loop
//     occurs until the semiphore is free.
//
//     Params:   none
//
//     Returns:  none
//
//-----------------------------------------------------------------------------
void objWaitSemiphore (BLobject *pBL)
{

  while (pBL->usSemiphore1){};  // wait until the semiphore #1 is zero

  pBL->usSemiphore1++;

  while (pBL->usSemiphore1 > 1){};  // only the first task to test the semiphore at 1
                               // will get by

  while (pBL->usSemiphore2){};  // wait until the semiphore #2 is zero

  pBL->usSemiphore2++;

  while (pBL->usSemiphore2 > 1){};  // only the first task to test the semiphore at 1
                               // will get by

  // There are holes in not using a single cycle set and test instruction,
  // however, it is inconceivable that task switch will be occuring fast
  // enough to make the above steping stones fail.

}


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  EndSemiphore
//
//     This function releases the semiphore in use.
//
//     Params:   none
//
//     Returns:  none
//
//-----------------------------------------------------------------------------
void objEndSemiphore (BLobject *pBL)
{
  pBL->usSemiphore2--;
  pBL->usSemiphore1--;
} 


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  CleanDialString
//
//     This function removes non-digit characters from the phone number string.
//
//
//     Params:   pszPhoneNumber - The phone number string to clean.
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG objCleanDialString (BLobject *pBL,char FAR *lpszPhoneNumber)
{
  USHORT usIndexInput, usIndexOutput;  // used as an array index, the compiler expects 16 bits
  char cInputChar;
  int i;
  
  // First we need to get rid of mixed case
  MW_SYSLOG_2(TRACE_MWMBL,"blobject::objCleanDialString, lpszPhoneNumber %s\n",lpszPhoneNumber);
  for (i=0;i<strlen(lpszPhoneNumber);i++) {
    lpszPhoneNumber[i]=toupper(lpszPhoneNumber[i]);
  }
  MW_SYSLOG_2(TRACE_MWMBL,"blobject::objCleanDialString, after uppercasing lpszPhoneNumber %s\n",lpszPhoneNumber);

  // Next we need to scrub the phone number to remove non-digit characters.
  // This is done by stepping through the array and shifting left the good digits.
  usIndexInput  = 0;
  usIndexOutput = 0;
  
  do {
    cInputChar = lpszPhoneNumber[usIndexInput];
    
    if (usIndexInput > MAX_BL_DIAL_STRING) {
      MW_SYSLOG_1(TRACE_MWMBL,"blobject::objCleanDialString, ERROR dial string not terminated\n");
      return BLERR_DIAL_STRING_TOO_LARGE;
    }
    
    // if it is in range then copy it
    if ( ((cInputChar >= '0') && (cInputChar <= '9')) ||
         ((cInputChar >= 'A') && (cInputChar <= 'D')) ||
	 (cInputChar == '*') || (cInputChar == '#') ) {
      lpszPhoneNumber[usIndexOutput] = cInputChar;
      usIndexOutput++;
    } else if (cInputChar == '\0') {
      lpszPhoneNumber [usIndexOutput] = cInputChar;
    }
    usIndexInput++;
  }
  while (cInputChar != '\0');
  
  if (lpszPhoneNumber[0] == '\0')
    return BLERR_PHONE_NUMBER_IS_BLANK;
  
  return BL_SUCCESSFUL;

} 
