/*
 *   dspipc.c -- Routines for IPC functionality 
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include <string.h>
#include "dspmgr.h"

#include "dspipc.h"                    // Include function prototypes
#include "dspbios.h"         // Include function prototypes for dspcalle.c

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ConnectIPC                                              */
/*                                                                          */
/* FUNCTION: This routines will Connect an Interrupt Handling Entry Point   */
/*           to a Task.                                                     */
/*                                                                          */
/* INPUT:                                                                   */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ConnectIPC(PRTSK prtsk,PFN pfnIPCNotice,USHORT usType,PHIPC phIPC)

{
   PRIPCTE    pripcte;

   /*************************************************************************/
   /* Check the parameters to make sure everything OK.                      */
   /* Traverse the table and check for matching prtsk,                      */
   /*************************************************************************/

   MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspipc::ConnectIPC entry prtsk %x pfnIPCNotice %x usType %x\n",
	   (int)prtsk,(int)pfnIPCNotice,usType);


   if (prtsk == NULL)
      return (DSP_INV_HANDLE);

   if (prtsk->TSK_achValid != TSKVALIDATE)
      return (DSP_INV_HANDLE);

   if (pfnIPCNotice == NULL)
      return (DSP_INV_PARAMETER);

   if (phIPC == NULL)
      return (DSP_INV_PARAMETER);

   /*************************************************************************/
   /* If the IPC is a Ring 3 IPC, put the pripcte in the table              */
   /* instead of the Callback Address.  For Ring 3 IPCs, the                */
   /* pripcte is used as the Event ID for RUNs and BLOCKs.                  */
   /*************************************************************************/

   if (prtsk->TSK_pripcte == NULL)
      return (DSP_INV_IPC);

   pripcte = prtsk->TSK_pripcte;

   if (pripcte->IPCTE_pfnCallBack != NULL)
      return (DSP_IPC_COMMITTED);      /* say already claimed               */
   pripcte->IPCTE_usType = usType;
   *phIPC = (HIPC)pripcte;
   pripcte->IPCTE_usIntCount = 0;
   if (usType == IPCTE_RING3CALL)
      pripcte->IPCTE_pfnCallBack = (PFN)pripcte;
   else {
      pripcte->IPCTE_pfnCallBack = pfnIPCNotice;

   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspipc::ConnectIPC exit hIPC %x\n",(int)*phIPC);
   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: DisconnectIPC                                           */
/*                                                                          */
/* FUNCTION: This routines will DisConnect Interrupt Handling Entry Point   */
/*           from Task.                                                     */
/*                                                                          */
/* INPUT:                                                                   */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC DisconnectIPC(PRIPCTE pripcte)
{
   RC         ulRC = DSP_NOERROR;      // Error return code

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspipc::DisconnectIPC entry pripcte %x\n",(int)pripcte);
   if (pripcte == NULL)
      return (DSP_INV_HANDLE);
   if (pripcte->IPCTE_achValid != IPCTEVALIDATE)
      return (DSP_INV_HANDLE);
   if (pripcte->IPCTE_pfnCallBack != NULL) {/* was it connected?            */
      pripcte->IPCTE_pfnCallBack = NULL;

   }
   else
      return (DSP_INV_HANDLE);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspipc::DisconnectIPC exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddIPC                                                  */
/*                                                                          */
/* FUNCTION: This routine will add an IPC entry into the IPC table.         */
/*           It searches the table until an unused entry is found and adds  */
/*           the entry                                                      */
/*                                                                          */
/* INPUT  PRIPC pripc - ptr to ipc entry to be added to table               */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC AddIPC(PRTSK prtsk)
{
   RC         ulRC = DSP_NOERROR;      // Error return code
   PRIPCTE    pripcte;
   USHORT     i;
   BOOL       bfound;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspipc::AddIPC entry prtsk %x\n",(int)prtsk);
   /*************************************************************************/
   /* Setup pointer to IPC Table                                            */
   /* Search the table from 1-16 for a table entry.                         */
   /*************************************************************************/

   pripcte = prtsk->TSK_prdsp->DSP_prIPCTable;
   bfound = FALSE;

   /*bump through table looking for first unused entry                      */

   for (i = 1; ((bfound == FALSE) && (i <= NUMIPC)); i++) {

      if (pripcte->IPCTE_prtsk == NULL) {/* if NULL, this entry is unused   */
         prtsk->TSK_pripcte = pripcte; /* fix up task pointer               */
         pripcte->IPCTE_prtsk = prtsk; /* associate new entry with task     */
         pripcte->IPCTE_usIntCount = 0;/* initialize interrupt count        */
         bfound = TRUE;                /* indicate we found an entry        */
      }                                /* endif                             */
      pripcte++;                       /* bump to next table entry          */
   }                                   /* endfor                            */
   if (bfound == FALSE)
      ulRC = DSP_INS_IPC_CHANNELS;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspipc::AddIPC exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveIPC                                               */
/*                                                                          */
/* FUNCTION: This routine will remove an IPC entry into the IPC table.      */
/*           It searches the table until an task   entry is found and       */
/*           removes the entry.                                             */
/* INPUT:                                                                   */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveIPC(PRTSK prtsk)
{
   RC         ulRC = DSP_NOERROR;      // Error return code
   PRIPCTE    pripcte;

   /* clear prtsk and pfnCallBack table entries                             */

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspipc::RemoveIPC entry prtsk %x\n",(int)prtsk);
   pripcte = prtsk->TSK_pripcte;

   prtsk->TSK_pripcte = NULL;
   pripcte->IPCTE_prtsk = NULL;
   if (pripcte->IPCTE_pfnCallBack != NULL) {
      pripcte->IPCTE_pfnCallBack = NULL;

   }                                   /* endif                             */


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspipc::RemoveIPC exit ulRC %lx\n",ulRC);
   return (ulRC);
}

