/***************************************************************************
    Geneve 9640 system board

    Onboard SRAM configuration:
    There is an adjustable SRAM configuration on board, representing the
    various enhancements by users.

    The standard memory configuration as reported by chkdsk (32 KiB):
    557056 bytes of total memory

    With 64 KiB SRAM:
    589824 bytes of total memory

    With 384 KiB SRAM:
    917504 bytes of total memory

    The original 32 KiB SRAM memory needs to be expanded to 64 KiB for
    MDOS 2.50s and higher, or the system will lock up. Therefore the emulation
    default is 64 KiB.

    The ultimate expansion is a 512 KiB SRAM circuit wired to the gate array
    to provide 48 pages of fast static RAM. This also requires to build an
    adapter for a larger socket. From the 512 KiB, only 384 KiB will be
    accessed, since the higher pages are hidden behind the EPROM pages.

    === Address map ===
    p,q = page value bit (q = AMC, AMB, AMA)
    c = address offset within 8 KiB page

    p pqqq pppc cccc cccc cccc

    0 0... .... .... .... .... on-board dram 512 KiB

    0 1... .... .... .... .... on-board future expansion 512 KiB or Memex with Genmod

    1 00.. .... .... .... .... p-box AMA=0 (256 KiB)
    1 010. .... .... .... .... p-box AMA=1 AMB=0 (128 KiB)
    1 0110 .... .... .... .... p-box AMA=1 AMB=1 AMC=0 (64 KiB)

    1 0111 00.. .... .... .... p-box address block 0xxx, 2xxx
    1 0111 010. .... .... .... p-box address block 4xxx (DSR)
    1 0111 011. .... .... .... p-box address block 6xxx
    1 0111 100. .... .... .... p-box address block 8xxx (Speech at 0x9000)
    1 0111 101. .... .... .... p-box address block axxx
    1 0111 11.. .... .... .... p-box address block cxxx, exxx

    1 100. .... .... .... .... on-board sram (128K) -\
    1 101. .... .... .... .... on-board sram (128K) --+- maximum SRAM expansion
    1 1100 .... .... .... .... on-board sram (64K) --/
    1 1101 0... .... .... .... on-board sram (32K) - additional 32 KiB required for MDOS 2.50s and higher
    1 1101 1... .... .... .... on-board sram (32K) - standard setup

    1 111. ..0. .... .... .... on-board boot1
    1 111. ..1. .... .... .... on-board boot2

    The TI console (or more precise, the Flex Cable Interface) sets the AMA/B/C
    lines to 1. Most cards actually check for AMA/B/C=1. However, this decoding
    was forgotten in third party cards which cause the card address space
    to be mirrored. The usual DSR space at 4000-5fff which would be reachable
    via page 0xba is then mirrored on a number of other pages:

    10 xxx 010x = 82, 8a, 92, 9a, a2, aa, b2, ba

    Another block to take care of is 0xbc which covers 8000-9fff since this
    area contains the speech synthesizer port at 9000/9400.

    For the standard Geneve, only prefix 10 is routed to the P-Box. The Genmod
    modification wires these address lines to pins 8 and 9 in the P-Box as AMD and
    AME. This requires all cards to be equipped with an additional selection logic
    to detect AMD=0, AME=1. Otherwise these cards, although completely decoding the
    19-bit address, would reappear at 512 KiB distances.

    For the page numbers we get:
    Standard:
    00-3f are internal (DRAM)
    40-7f are internal expansion, never used
    80-bf are the P-Box address space (optional Memex and peripheral cards at that location)
    c0-ff are internal (SRAM, EPROM)

    Genmod:
    00-3f are the P-Box address space (expect Memex at that location)
    40-7f are the P-Box address space (expect Memex at that location)
    80-bf are the P-Box address space (expect Memex at that location)
    c0-ef are the P-Box address space (expect Memex at that location)
    f0-ff are internal (EPROM)

    Genmod's double switch box is also emulated. There are two switches:
    - Turbo mode: Activates or deactivates the wait state logic on the Geneve
      board. This switch may be changed at any time.
    - TI mode: Selects between the on-board memory, which is obviously required
      for the GPL interpreter, and the external Memex memory. This switch
      triggers a reset when changed.

    Waitstate handling
    ------------------
    Waitstates are caused by a cleared READY line of the TMS9995 processor
    during an external memory cycle. That means that waitstates have no effect
    for operations within the on-chip memory, and only when an access to the
    external memory or other devices occurs, a delay will be noticed.

    The waitstates are generated by the custom Gate Array chip on the board
    and the PAL 16R4, both lacking proper documentation. All of the following
    numbers have been determined by experiments with the real machine.

    Waitstates are generated for:
    - memory-mapped devices (mapper, clock, keyboard): 1 WS
    - accesses to the peripheral expansion box: 1 WS
    - accesses to on-board DRAM: 1 WS
    - accesses to video: 14 WS
    - accesses to sound: ~25 WS
    - accesses to SRAM: 0 WS

    Additional waitstates are created when one of the CRU bits is set. In that
    case, all delays are extended to 2 WS (including SRAM).

    Sound waitstates are somewhat unpredictable. It seems as if they depend
    on the clock of the sound chip; the theory is that the READY line is
    pulled down until the next clock pulse, which may take some value between
    18 CPU cycles and 30 CPU cycles.

    There is also a flag for video wait states. Waiting for video at first
    means an access to a memory-mapped device, giving 1 or 2 WS, and an
    additional 14 WS after the access. As said, these waitstates will run
    unnoticed when the code is located in the on-chip RAM. This makes emulation
    a bit non-straightforward.

    These 14 WS are simulated by a timer device which is started together with
    the video access, and when some external access occurs later, the
    remaining time will be used to produce wait states.

    Michael Zapf, October 2011
***************************************************************************/

#include "emu.h"
#include "genboard.h"
#include "video/v9938.h"
#include "videowrp.h"
#include "sound/sn76496.h"
#include "machine/mm58274c.h"
#include "peribox.h"

#define KEYQUEUESIZE 256
#define MAXKEYMSGLENGTH 10
#define KEYAUTOREPEATDELAY 30
#define KEYAUTOREPEATRATE 6

#define CYCLE 333			// 333 ns for CLKOUT

#define SRAM_SIZE 384*1024   // maximum SRAM expansion on-board
#define DRAM_SIZE 512*1024

#define VERBOSE 1
#define LOG logerror

typedef struct _genboard_state
{
	/* Mouse support */
	int		last_mx;
	int		last_my;

	/* Joystick support */
	int 	joySel;

	/* Keyboard support */
	UINT8	keyQueue[KEYQUEUESIZE];
	int 	keyQueueHead;
	int 	keyQueueLen;
	bool	keyInBuf;
	int 	keyReset;
	UINT32	keyStateSave[4];
	int 	keyNumLockState;
	int 	keyCtrlState;
	int 	keyAltState;
	int 	keyRealShiftState;
	int 	keyFakeShiftState;
	int 	keyFakeUnshiftState;
	int 	keyAutoRepeatKey;
	int 	keyAutoRepeatTimer;

	/* Mode flags */
	bool	palvideo;		// uncertain, could not be verified with real system
	bool	capslock;
	bool	keyboard_clock;
	bool	keep_keybuf;
	bool	extra_waitstates;

	bool	genmod;
	bool	turbo;
	bool	timode;

	/* GROM simulation */
	int		grom_address;
	bool	gromraddr_LSB;
	bool	gromwaddr_LSB;

	/* Memory */
	UINT8	*sram;
	UINT8	*dram;
	UINT8	*eprom;
	int		sram_mask, sram_val;

	/* Mapper */
	bool	geneve_mode;
	bool	direct_mode;
	bool	cartridge_size_8K;
	bool	cartridge_secondpage;
	bool	cartridge6_writable;
	bool	cartridge7_writable;
	int		map[8];

	int		line_count;

	/* Devices */
	device_t	*cpu;
	device_t	*video;
	device_t	*tms9901;
	device_t	*clock;
	device_t	*peribox;
	device_t	*sound;

	/* VDP Waitstate timer */
	bool		video_waitstates;
	emu_timer	*video_waittimer;
	bool		video_waiting;

} genboard_state;

static const UINT8 keyboard_mf1_code[0xe] =
{
	/* extended keys that are equivalent to non-extended keys */
	0x1c,	/* keypad enter */
	0x1d,	/* right control */
	0x38,	/* alt gr */
	// extra codes are 0x5b for Left Windows, 0x5c for Right Windows, 0x5d
	// for Menu, 0x5e for power, 0x5f for sleep, 0x63 for wake, but I doubt
	// any Geneve program would take advantage of these. */

	// extended key that is equivalent to a non-extended key
	// with shift off
	0x35,	/* pad slash */

	// extended keys that are equivalent to non-extended keys
	// with numlock off
	0x47,	/* home */
	0x48,	/* up */
	0x49,	/* page up */
	0x4b,	/* left */
	0x4d,	/* right */
	0x4f,	/* end */
	0x50,	/* down */
	0x51,	/* page down */
	0x52,	/* insert */
	0x53	/* delete */
};

static void poll_keyboard(device_t *device);
static void poll_mouse(device_t *device);
static void signal_when_key_available(genboard_state *board);
static UINT8 get_recent_key(device_t *device);

static const char *const keynames[] = { "KEY0", "KEY1", "KEY2", "KEY3", "KEY4", "KEY5", "KEY6", "KEY7" };

INLINE genboard_state *get_safe_token(device_t *device)
{
	assert(device != NULL);
	assert(device->type() == GENBOARD);

	return (genboard_state *)downcast<legacy_device_base *>(device)->token();
}

/*
    Callback for VDP waitstate. Not really needed, just for completeness.
*/
static TIMER_CALLBACK(video_wait_callback)
{
	device_t *device = (device_t *)ptr;
	genboard_state *board = get_safe_token(device);
	board->video_waiting = false;
}

/*
    scanline interrupt
*/
INTERRUPT_GEN( geneve_hblank_interrupt )
{
	device_t *dev = device->machine().device("geneve_board");
	genboard_state *board = get_safe_token(dev);
	v9938_interrupt(device->machine(), 0);
	board->line_count++;
	if (board->line_count == 262)
	{
		board->line_count = 0;
		poll_keyboard(dev);
		poll_mouse(dev);
	}
}

void set_gm_switches(device_t *board, int number, int value)
{
	if (number==1)
	{
		// Turbo switch. May be changed at any time.
		if (VERBOSE>0) LOG("genboard / genmod: Setting turbo flag to %d\n", value);
		((genboard_state*)board)->turbo = (value!=0);
	}
	else
	{
		// TIMode switch. Causes reset when changed.
		if (VERBOSE>0) LOG("genboard / genmod: Setting timode flag to %d\n", value);
		((genboard_state*)board)->timode = (value!=0);
		board->machine().schedule_hard_reset();
	}
}
/****************************************************************************
    GROM simulation. The Geneve board simulated GROM circuits within its gate
    array.
*****************************************************************************/

/*
    Simulates GROM. The real Geneve does not use GROMs but simulates them
    within the gate array. Unlike with real GROMs, no address wrapping occurs,
    and the complete 64K space is available.
*/
static READ8_DEVICE_HANDLER( read_grom )
{
	genboard_state *board = get_safe_token(device);
	UINT8 reply;
	if (offset & 0x0002)
	{
		// GROM address handling
		board->gromwaddr_LSB = false;

		if (board->gromraddr_LSB)
		{
			reply = board->grom_address & 0xff;
			board->gromraddr_LSB = false;
		}
		else
		{
			reply = (board->grom_address >> 8) & 0xff;
			board->gromraddr_LSB = true;
		}
	}
	else
	{
		// GROM data handling
		// GROMs are stored in pages 38..3f
		int page = 0x38;
		reply = board->dram[page*0x2000 + board->grom_address];
		board->grom_address = (board->grom_address + 1) & 0xffff;
		board->gromraddr_LSB = board->gromwaddr_LSB = false;
	}
	return reply;
}

/*
    Simulates GROM. The real Geneve does not use GROMs but simulates them
    within the gate array.
*/
static WRITE8_DEVICE_HANDLER( write_grom )
{
	genboard_state *board = get_safe_token(device);
	if (offset & 0x0002)
	{
		// set address
		board->gromraddr_LSB = false;
		if (board->gromwaddr_LSB)
		{
			board->grom_address = (board->grom_address & 0xff00) | data;
			board->grom_address = (board->grom_address + 1) & 0xffff;
			board->gromwaddr_LSB = false;
		}
		else
		{
			board->grom_address = (board->grom_address & 0x00ff) | ((UINT16)data<<8);
			board->gromwaddr_LSB = true;
		}
	}
	else
	{	// write GPL data
		// The Geneve GROM simulator allows for GROM writing (verified with a real system)
		int page = 0x38;
		board->dram[page*0x2000 + board->grom_address] = data;

		board->grom_address = (board->grom_address + 1) & 0xffff;
		board->gromraddr_LSB = board->gromwaddr_LSB = false;
	}
}

/*
Geneve mode:
f100 / fff5: v9938_r
f110 / fff8: mapper_r
f118 / ffff: key_r
f130 / fff0: clock_r

TI mode:
8000 / fff8: mapper_r
8008 / ffff: key_r
8010 / fff0: clock_r
8800 / fffd: v9938_r
9000 / ffc0: speech_r
9800 / ffc0: grom_r

*/

static void start_video_wait(device_t *device, int ws)
{
	genboard_state *board = get_safe_token(device);
	board->video_waittimer->adjust(attotime::from_nsec(CYCLE * ws));
	board->video_waiting = true;
}

static void do_wait(device_t *device, int min)
{
	long ws;
	genboard_state *board = get_safe_token(device);

	// Extra waitstates?
	if (board->extra_waitstates && min < 2) min = 2;

	if (board->video_waiting) {
		attotime left = board->video_waittimer->remaining();
		ws = (left.as_attoseconds()/ATTOSECONDS_PER_NANOSECOND) / CYCLE;
	}
	else ws = 0;

	// Now wait for the remaining time or for at least the min time
	device_adjust_icount(board->cpu, -4 * ((ws < min)? min : ws));
}

READ8_DEVICE_HANDLER( geneve_r )
{
	genboard_state *board = get_safe_token(device);
	UINT8 value = 0;
	int page;
	UINT32	physaddr;

	// Premature access. The CPU reads the start vector before RESET.
	if (board->peribox==NULL) return 0;

	if (board->geneve_mode)
	{
		// TODO: shortcut offset & 0xffc0 = 0xf100
		if ((offset & 0xfff5)==0xf100)
		{
			// video
			// ++++ ++++ ++++ -+-+
			// 1111 0001 0000 0000
			// 1111 0001 0000 0010
			// 1111 0001 0000 1000
			// 1111 0001 0000 1010

			// 1 WS is always added; any pending video waitstates are canceled
			board->video_waiting = false;
			do_wait(device, 1);

			// Initialize waitstate timer
			// Create 16 waitstates (2 more than expected, experimenting)
			if (board->video_waitstates)
				start_video_wait(device, 16);

			gen_v9938_rz(board->video, offset, &value);
			return value;
		}
		if ((offset & 0xfff8)==0xf110)
		{
			// mapper
			value = board->map[offset & 0x0007];

			// Add appropriate number of waitstates
			// 1 WS is added at least
			do_wait(device, 1);
			return value;
		}
		if ((offset & 0xfff8) == 0xf118)
		{
			// key
			value = get_recent_key(device);
			do_wait(device, 1);
			return value;
		}
		if ((offset & 0xfff0)==0xf130)
		{
			// clock
			// tests on the real machine showed that
			// upper nibble is 0xf (probably because of the location at 0xf130?)
			value = mm58274c_r(board->clock, offset & 0x000f) | 0xf0;
			do_wait(device, 1);
			return value;
		}
	}
	else // TI mode
	{
		if ((offset & 0xfff8)==0x8000)
		{
			// mapper
			value = board->map[offset & 0x0007];
			do_wait(device, 1);
			return value;
		}
		if ((offset & 0xfff8)== 0x8008)
		{
			// key
			value = get_recent_key(device);
			do_wait(device, 1);
			return value;
		}
		if ((offset & 0xfff0)==0x8010)
		{
			// clock
			// upper nibble is 1, only last byte gets a 2
			// probably because of the location at 8010...8020?
			// (TI mode used swapped byte order)
			// unless we use a workspace at >F000, in which case we get 8x values
			// Obscure, needs more investigation. We might as well ignore this,
			// as the high nibble is obviously undefined and takes some past
			// value floating around.
			value = mm58274c_r(board->clock, offset & 0x000f);
			value |= ((offset & 0x000f)==0x000f)? 0x20 : 0x10;

			do_wait(device, 1);
			return value;
		}
		if ((offset & 0xfc01)==0x8800)
		{
			// video
			// ++++ ++-- ---- ---+
			// 1000 1000 0000 00x0
			//  device_adjust_icount(board->cpu, -4);

			// 1 WS is always added; any pending video waitstates are canceled
			board->video_waiting = false;
			do_wait(device, 1);

			// Initialize waitstate timer
			// Create 14 waitstates (+2, see above)
			if (board->video_waitstates)
				start_video_wait(device, 16);

			gen_v9938_rz(board->video, offset, &value);
			return value;
		}
		if ((offset & 0xfc01)==0x9000)
		{
			// speech
			// ++++ ++-- ---- ---+
			// 1001 0000 0000 0000
			// We need to add the address prefix bits
			ti99_peb_data_rz(board->peribox, offset | ((board->genmod)? 0x170000 : 0x070000), &value);

			do_wait(device, 1);
			return value;
		}
		if ((offset & 0xfc01)==0x9800)
		{
			// grom simulation
			// ++++ ++-- ---- ---+
			// 1001 1000 0000 00x0
			value = read_grom(device, offset);

			do_wait(device, 1);
			return value;
		}
	}

	page = (offset & 0xe000) >> 13;

	if (board->direct_mode)
	{
		physaddr = 0x1e0000; // points to boot eprom
	}
	else
	{
		if (!board->geneve_mode && page==3)
		{
			// Cartridge paging in TI mode
			// See also cartridge type "paged" in gromport.h
			// value 0x36 = 0 0110 110x xxxx xxxx xxxx (page 1)
			// value 0x37 = 0 0110 111x xxxx xxxx xxxx (page 2)
			// Only use this if there are 2*8 KiB cartridge ROM
			if (!board->cartridge_size_8K && board->cartridge_secondpage) physaddr = 0x06e000;
			else physaddr = 0x06c000;
		}
		else
		{
			physaddr = (board->map[page] << 13);
		}
	}

	physaddr |= (offset & 0x1fff);

	if (!board->genmod)
	{
		// Standard Geneve
		if ((physaddr & 0x180000)==0x000000)
		{
			// DRAM. One wait state.
			// device_adjust_icount(board->cpu, -4);
			do_wait(device, 1);

			value = board->dram[physaddr & 0x07ffff];
//          printf("dram read physaddr = %06x logaddr = %04x value = %02x\n", physaddr, offset, value);
			return value;
		}

		if ((physaddr & 0x180000)==0x080000)
		{
			// On-board memory expansion for standard Geneve (never used)
			// device_adjust_icount(board->cpu, -4);
			do_wait(device, 1);
			return 0;
		}

		if ((physaddr & 0x1e0000)==0x1e0000)
		{
			// 1 111. ..xx xxxx xxxx xxxx on-board eprom (16K)
			// mirrored for f0, f2, f4, ...; f1, f3, f5, ...
			value = board->eprom[physaddr & 0x003fff];
//          printf("eprom read physaddr = %06x logaddr = %04x value = %02x\n", physaddr, offset, value);
			do_wait(device, 1);
			return value;
		}

		if ((physaddr & 0x180000)==0x180000)
		{
			if ((physaddr & board->sram_mask)==board->sram_val)
			{
				value = board->sram[physaddr & ~board->sram_mask];
			}
			// Return in any case
//          printf("sram read physaddr = %06x logaddr = %04x value = %02x\n", physaddr, offset, value);
			do_wait(device, 0);
			return value;
		}

		// Route everything else to the P-Box
		//   0x000000-0x07ffff for the stock Geneve (AMC,AMB,AMA,A0 ...,A15)
		//   0x000000-0x1fffff for the GenMod.(AME,AMD,AMC,AMB,AMA,A0 ...,A15)
		// Add a wait state
		do_wait(device, 1);

		physaddr = (physaddr & 0x0007ffff);  // 19 bit address (with AMA..AMC)
		ti99_peb_data_rz(board->peribox, physaddr, &value);
		return value;
	}
	else
	{
		// GenMod mode
		if ((board->timode) && ((physaddr & 0x180000)==0x000000))
		{
			// DRAM. One wait state.
			value = board->dram[physaddr & 0x07ffff];
			if (!board->turbo) do_wait(device, 1);
			return value;
		}

		if ((physaddr & 0x1e0000)==0x1e0000)
		{
			// 1 111. ..xx xxxx xxxx xxxx on-board eprom (16K)
			// mirrored for f0, f2, f4, ...; f1, f3, f5, ...
			value = board->eprom[physaddr & 0x003fff];
			do_wait(device, 0);
			return value;
		}
		// Route everything else to the P-Box
		physaddr = (physaddr & 0x001fffff);  // 21 bit address for Genmod

		if (!board->turbo) do_wait(device, 1);
		// Check: Are waitstates completely turned off for turbo mode, or
		// merely the waitstates for DRAM memory access and box access?

		ti99_peb_data_rz(board->peribox, physaddr, &value);
		return value;
	}
}


/*
Geneve mode:
f100 / fff1: v9938_w
f110 / fff8: mapper_w
f120 / fff0: sound_w
f130 / fff0: clock_w

TI mode:
8000 / fff8: mapper_w
8010 / fff0: clock_w
8400 / fc00: sound_w
8c00 / fff9: v9938_w
9400 / fc00: speech_w
9c00 / fffd: grom_w

*/

WRITE8_DEVICE_HANDLER( geneve_w )
{
	genboard_state *board = get_safe_token(device);
	UINT32	physaddr;
	int page;

	if (board->peribox==NULL) return;

	if (board->geneve_mode)
	{
		if ((offset & 0xfff1)==0xf100)
		{
			// video
			// ++++ ++++ ++++ ---+
			// 1111 0001 0000 .cc0
			// device_adjust_icount(board->cpu, -4);

			gen_v9938_w(board->video, offset, data);

			// 1 WS is always added; any pending video waitstates are canceled
			board->video_waiting = false;
			do_wait(device, 1);

			// Initialize waitstate timer
			// Create 14 waitstates (+3, experimenting)
			if (board->video_waitstates)
				start_video_wait(device, 17);

			return;
		}
		if ((offset & 0xfff8)==0xf110)
		{
			// mapper
			board->map[offset & 0x0007] = data;
			do_wait(device, 1);
			return;
		}
		if ((offset & 0xfff1)==0xf120)
		{
			// sound
			// ++++ ++++ ++++ ---+
			sn76496_w(board->sound, 0, data);

			// Add 24 waitstates. This is an approximation, as the
			// waitstate generation seems to depend on an external timer of
			// the sound chip
			device_adjust_icount(board->cpu, -4 * 24);
			return;
		}
		if ((offset & 0xfff0)==0xf130)
		{
			// clock
			// ++++ ++++ ++++ ----
			mm58274c_w(board->clock, offset & 0x00f, data);
			do_wait(device, 1);
			return;
		}
	}
	else
	{
		// TI mode
		if ((offset & 0xfff8)==0x8000)
		{
			// mapper
			board->map[offset & 0x0007] = data;
			do_wait(device, 1);
			return;
		}
		// No key write at 8008
		if ((offset & 0xfff0)==0x8010)
		{
			// clock
			mm58274c_w(board->clock, offset & 0x00f, data);
			do_wait(device, 1);
			return;
		}
		if ((offset & 0xfc01)==0x8400)
		{
			// sound
			// ++++ ++-- ---- ---+
			// 1000 0100 0000 0000

			sn76496_w(board->sound, 0, data);
			// Add 24 waitstates. This is an approximation, as the
			// waitstate generation seems to depend on an external timer of
			// the sound chip
			device_adjust_icount(board->cpu, -4 * 24);
			return;
		}
		if ((offset & 0xfc01)==0x8c00)
		{
			// video
			// ++++ ++-- ---- ---+
			// 1000 1100 0000 00c0
			// device_adjust_icount(board->cpu, -4);
			// Initialize waitstate timer
			gen_v9938_w(board->video, offset, data);

			// 1 WS is always added; any pending video waitstates are canceled
			board->video_waiting = false;
			do_wait(device, 1);

			// Initialize waitstate timer
			// Create 14 waitstates (+3)
			if (board->video_waitstates)
				start_video_wait(device, 17);

			return;
		}
		if ((offset & 0xfc01)==0x9400)
		{
			// speech
			// ++++ ++-- ---- ---+
			// 1001 0100 0000 0000
			// We need to add the address prefix bits
			ti99_peb_data_w(board->peribox, offset | ((board->genmod)? 0x170000 : 0x070000), data);
			do_wait(device, 1);
			return;
		}
		if ((offset & 0xfc01)==0x9c00)
		{
			// grom simulation
			// ++++ ++-- ---- ---+
			// 1001 1100 0000 00c0
			write_grom(device, offset, data);
			do_wait(device, 1);
			return;
		}
	}

	page = (offset & 0xe000) >> 13;
	if (board->direct_mode)
	{
		physaddr = 0x1e0000; // points to boot eprom
	}
	else
	{
		if (!board->geneve_mode && page==3)
		{
			if (!board->cartridge_size_8K)
			{
				// Writing to 0x6000 selects page 1,
				// writing to 0x6002 selects page 2
				board->cartridge_secondpage = ((offset & 0x0002)!=0);
				do_wait(device, 1);
				return;
			}
			else
			{
				// writing into cartridge rom space (no bankswitching)
				if ((((offset & 0x1000)==0x0000) && !board->cartridge6_writable)
					|| (((offset & 0x1000)==0x1000) && !board->cartridge7_writable))
				{
					if (VERBOSE>0) LOG("genboard: Writing to protected cartridge space %04x ignored\n", offset);
					return;
				}
				else
					// TODO: Check whether secondpage is really ignored
					physaddr = 0x06c000;
			}
		}
		else
			physaddr = (board->map[page] << 13);
	}

	physaddr |= offset & 0x1fff;

//  printf("write physaddr = %06x logaddr = %04x value = %02x\n", physaddr, offset, data);

	if (!board->genmod)
	{
		if ((physaddr & 0x180000)==0x000000)
		{
			// DRAM write. One wait state. (only for normal Geneve)
			board->dram[physaddr & 0x07ffff] = data;
			do_wait(device, 1);
			return;
		}

		if ((physaddr & 0x180000)==0x080000)
		{
			// On-board memory expansion for standard Geneve (never used)
			do_wait(device, 1);
			return;
		}

		if ((physaddr & 0x1e0000)==0x1e0000)
		{
			// 1 111. ..xx xxxx xxxx xxxx on-board eprom (16K)
			// mirrored for f0, f2, f4, ...; f1, f3, f5, ...
			// Ignore EPROM write
			do_wait(device, 1);
			return;
		}

		if ((physaddr & 0x180000)==0x180000)
		{
			if ((physaddr & board->sram_mask)==board->sram_val)
			{
				board->sram[physaddr & ~board->sram_mask] = data;
			}
			do_wait(device, 0);
			// Return in any case
			return;
		}
		// Route everything else to the P-Box
		// Add a wait state

		// only AMA, AMB, AMC are used; AMD and AME are not used
		physaddr = (physaddr & 0x0007ffff);  // 19 bit address
		ti99_peb_data_w(board->peribox, physaddr, data);
		do_wait(device, 1);
	}
	else
	{
		// GenMod mode
		if ((board->timode) && ((physaddr & 0x180000)==0x000000))
		{
			// DRAM. One wait state.
			board->dram[physaddr & 0x07ffff] = data;
			if (!board->turbo) do_wait(device, 1);
			return;
		}

		if ((physaddr & 0x1e0000)==0x1e0000)
		{
			// 1 111. ..xx xxxx xxxx xxxx on-board eprom (16K)
			// mirrored for f0, f2, f4, ...; f1, f3, f5, ...
			// Ignore EPROM write
			if (!board->turbo) do_wait(device, 1);
			return;
		}
		// Route everything else to the P-Box
		physaddr = (physaddr & 0x001fffff);  // 21 bit address for Genmod
		ti99_peb_data_w(board->peribox, physaddr, data);
		if (!board->turbo) do_wait(device, 1);
	}
}

/****************************************************************************
    CRU handling
*****************************************************************************/

#define CRU_CONTROL_BASE 0x1ee0
#define CRU_SSTEP_BASE 0x13c0

WRITE8_DEVICE_HANDLER ( geneve_cru_w )
{
	genboard_state *board = get_safe_token(device);

	int addroff = offset << 1;
	bool rising_edge = false;
	bool falling_edge = false;

	// Single step
	// 13c0 - 13fe: 0001 0011 11xx xxx0
	if ((addroff & 0xffc0) == CRU_SSTEP_BASE)
	{
		int bit = (addroff & 0x003e)>>1;
		if (VERBOSE>0) LOG("genboard: Single step not implemented; bit %d set to %d\n", bit, data);
		return;
	}

	if ((addroff & 0xffe0) == CRU_CONTROL_BASE)
	{
		int bit = (addroff & 0x001e)>>1;
		switch (bit)
		{
		case 5:
			board->palvideo = (data!=0);
			break;
		case 7:
			board->capslock = (data!=0);
			break;
		case 8:
			rising_edge = (!board->keyboard_clock && (data!=0));
			board->keyboard_clock = (data!=0);
			if (rising_edge)
				signal_when_key_available(board);
			break;
		case 9:
			rising_edge = (!board->keep_keybuf && (data!=0));
			falling_edge = (board->keep_keybuf && (data==0));
			board->keep_keybuf = (data!=0);

			if (rising_edge)
				signal_when_key_available(board);
			else
			{
				if (falling_edge)
				{
					if (board->keyQueueLen != 0)
					{
						board->keyQueueHead = (board->keyQueueHead + 1) % KEYQUEUESIZE;
						board->keyQueueLen--;
					}
					/* clear keyboard interrupt */
					tms9901_set_single_int(board->tms9901, 8, CLEAR_LINE);
					board->keyInBuf = false;
				}
			}
			break;
		case 10:
			board->geneve_mode = (data!=0);
			break;
		case 11:
			board->direct_mode = (data!=0);
			break;
		case 12:
			board->cartridge_size_8K = (data!=0);
			break;
		case 13:
			board->cartridge6_writable = (data!=0);
			break;
		case 14:
			board->cartridge7_writable = (data!=0);
			break;
		case 15:
			board->extra_waitstates = (data==0);  // let's use the inverse semantics
			break;
		default:
			if (VERBOSE>0) LOG("genboard: set CRU address %04x=%02x ignored\n", addroff, data);
			break;
		}
	}
	else
	{
		ti99_peb_cru_w(board->peribox, addroff, data);
	}
}


READ8_DEVICE_HANDLER ( geneve_cru_r )
{
	UINT8 value = 0;
	genboard_state *board = get_safe_token(device);
	int addroff = offset << 4;

	// Single step
	// 13c0 - 13fe: 0001 0011 11xx xxx0
	if ((addroff & 0xffc0) == CRU_SSTEP_BASE)
	{
		int bit = (addroff & 0x003e)>>1;
		if (VERBOSE>0) LOG("genboard: Single step not implemented; attempting to read bit %d\n", bit);
		return value;
	}

	// TMS9995-internal CRU locations (1ee0-1efe) are handled within the 99xxcore
	// implementation (read_single_cru), so we don't arrive here

	// Propagate the CRU access to external devices
	ti99_peb_cru_rz(board->peribox, addroff, &value);
	return value;
}

/****************************************************************************
    Keyboard support
*****************************************************************************/

UINT8 get_recent_key(device_t *device)
{
	genboard_state *board = get_safe_token(device);
	if (board->keyInBuf)
		return board->keyQueue[board->keyQueueHead];
	else
		return 0;
}

static void signal_when_key_available(genboard_state *board)
{
	// if keyboard reset is not asserted, and key clock is enabled, and key
	// buffer clear is disabled, and key queue is not empty. */
	if ((!board->keyReset) && (board->keyboard_clock) && (board->keep_keybuf) && (board->keyQueueLen != 0))
	{
		tms9901_set_single_int(board->tms9901, 8, ASSERT_LINE);
		board->keyInBuf = true;
	}
}

INLINE void post_in_keyQueue(genboard_state *board, int keycode)
{
	board->keyQueue[(board->keyQueueHead + board->keyQueueLen) % KEYQUEUESIZE] = keycode;
	board->keyQueueLen++;
}

static void poll_keyboard(device_t *device)
{
	genboard_state *board = get_safe_token(device);

	UINT32 keystate;
	UINT32 key_transitions;
	int i, j;
	int keycode;
	int pressed;

	if (board->keyReset)
		return;

	/* Poll keyboard */
	for (i = 0; (i < 4) && (board->keyQueueLen <= (KEYQUEUESIZE-MAXKEYMSGLENGTH)); i++)
	{
		keystate = input_port_read(device->machine(), keynames[2*i]) | (input_port_read(device->machine(), keynames[2*i + 1]) << 16);
		key_transitions = keystate ^ board->keyStateSave[i];
		if (key_transitions)
		{
			for (j = 0; (j < 32) && (board->keyQueueLen <= (KEYQUEUESIZE-MAXKEYMSGLENGTH)); j++)
			{
				if ((key_transitions >> j) & 1)
				{
					keycode = (i << 5) | j;
					pressed = ((keystate >> j) & 1);
					if (pressed)
						board->keyStateSave[i] |= (1 << j);
					else
						board->keyStateSave[i] &= ~ (1 << j);

					/* Update auto-repeat */
					if (pressed)
					{
						board->keyAutoRepeatKey = keycode;
						board->keyAutoRepeatTimer = KEYAUTOREPEATDELAY+1;
					}
					else /*if (keycode == board->keyAutoRepeatKey)*/
						board->keyAutoRepeatKey = 0;

					// Release Fake Shift/Unshift if another key is pressed
					// We do so if a key is released, though it is actually
					// required only if it is a modifier key
					/*if (pressed)*/
					//{
					if (board->keyFakeShiftState)
					{
						/* Fake shift release */
						post_in_keyQueue(board, 0xe0);
						post_in_keyQueue(board, 0xaa);
						board->keyFakeShiftState = 0;
					}
					if (board->keyFakeUnshiftState)
					{
						/* Fake shift press */
						post_in_keyQueue(board, 0xe0);
						post_in_keyQueue(board, 0x2a);
						board->keyFakeUnshiftState = 0;
					}
					//}

					/* update shift and numlock state */
					if ((keycode == 0x2a) || (keycode == 0x36))
						board->keyRealShiftState = board->keyRealShiftState + (pressed ? +1 : -1);
					if ((keycode == 0x1d) || (keycode == 0x61))
						board->keyCtrlState = board->keyCtrlState + (pressed ? +1 : -1);
					if ((keycode == 0x38) || (keycode == 0x62))
						board->keyAltState = board->keyAltState + (pressed ? +1 : -1);
					if ((keycode == 0x45) && pressed)
						board->keyNumLockState = ! board->keyNumLockState;

					if ((keycode >= 0x60) && (keycode < 0x6e))
					{	/* simpler extended keys */
						/* these keys are emulated */

						if ((keycode >= 0x63) && pressed)
						{
							/* Handle shift state */
							if (keycode == 0x63)
							{	/* non-shifted key */
								if (board->keyRealShiftState)
									/* Fake shift unpress */
									board->keyFakeUnshiftState = 1;
							}
							else /*if (keycode >= 0x64)*/
							{	/* non-numlock mode key */
								if (board->keyNumLockState & ! board->keyRealShiftState)
									/* Fake shift press if numlock is active */
									board->keyFakeShiftState = 1;
								else if ((! board->keyNumLockState) & board->keyRealShiftState)
									/* Fake shift unpress if shift is down */
									board->keyFakeUnshiftState = 1;
							}

							if (board->keyFakeShiftState)
							{
								post_in_keyQueue(board, 0xe0);
								post_in_keyQueue(board, 0x2a);
							}

							if (board->keyFakeUnshiftState)
							{
								post_in_keyQueue(board, 0xe0);
								post_in_keyQueue(board, 0xaa);
							}
						}

						keycode = keyboard_mf1_code[keycode-0x60];
						if (! pressed)
							keycode |= 0x80;
						post_in_keyQueue(board, 0xe0);
						post_in_keyQueue(board, keycode);
					}
					else if (keycode == 0x6e)
					{	/* emulate Print Screen / System Request (F13) key */
						/* this is a bit complex, as Alt+PrtScr -> SysRq */
						/* Additionally, Ctrl+PrtScr involves no fake shift press */
						if (board->keyAltState)
						{
							/* SysRq */
							keycode = 0x54;
							if (! pressed)
								keycode |= 0x80;
							post_in_keyQueue(board, keycode);
						}
						else
						{
							/* Handle shift state */
							if (pressed && (! board->keyRealShiftState) && (! board->keyCtrlState))
							{	/* Fake shift press */
								post_in_keyQueue(board, 0xe0);
								post_in_keyQueue(board, 0x2a);
								board->keyFakeShiftState = 1;
							}

							keycode = 0x37;
							if (! pressed)
								keycode |= 0x80;
							post_in_keyQueue(board, 0xe0);
							post_in_keyQueue(board, keycode);
						}
					}
					else if (keycode == 0x6f)
					{	// emulate pause (F15) key
						// this is a bit complex, as Pause -> Ctrl+NumLock and
						// Ctrl+Pause -> Ctrl+ScrLock.  Furthermore, there is no
						// repeat or release.
						if (pressed)
						{
							if (board->keyCtrlState)
							{
								post_in_keyQueue(board, 0xe0);
								post_in_keyQueue(board, 0x46);
								post_in_keyQueue(board, 0xe0);
								post_in_keyQueue(board, 0xc6);
							}
							else
							{
								post_in_keyQueue(board, 0xe1);
								post_in_keyQueue(board, 0x1d);
								post_in_keyQueue(board, 0x45);
								post_in_keyQueue(board, 0xe1);
								post_in_keyQueue(board, 0x9d);
								post_in_keyQueue(board, 0xc5);
							}
						}
					}
					else
					{
						if (! pressed)
							keycode |= 0x80;
						post_in_keyQueue(board, keycode);
					}
					signal_when_key_available(board);
				}
			}
		}
	}

	/* Handle auto-repeat */
	if ((board->keyQueueLen <= (KEYQUEUESIZE-MAXKEYMSGLENGTH)) && board->keyAutoRepeatKey && (--board->keyAutoRepeatTimer == 0))
	{
		if ((board->keyAutoRepeatKey >= 0x60) && (board->keyAutoRepeatKey < 0x6e))
		{
			post_in_keyQueue(board, 0xe0);
			post_in_keyQueue(board, keyboard_mf1_code[board->keyAutoRepeatKey-0x60]);
		}
		else if (board->keyAutoRepeatKey == 0x6e)
		{
			if (board->keyAltState)
				post_in_keyQueue(board, 0x54);
			else
			{
				post_in_keyQueue(board, 0xe0);
				post_in_keyQueue(board, 0x37);
			}
		}
		else if (board->keyAutoRepeatKey == 0x6f)
			;
		else
		{
			post_in_keyQueue(board, board->keyAutoRepeatKey);
		}
		signal_when_key_available(board);
		board->keyAutoRepeatTimer = KEYAUTOREPEATRATE;
	}
}

/****************************************************************************
    Mouse support
****************************************************************************/

static void poll_mouse(device_t *device)
{
	genboard_state *board = get_safe_token(device);

	int new_mx, new_my;
	int delta_x, delta_y, buttons;

	buttons = input_port_read(device->machine(), "MOUSE0");
	new_mx = input_port_read(device->machine(), "MOUSEX");
	new_my = input_port_read(device->machine(), "MOUSEY");

	/* compute x delta */
	delta_x = new_mx - board->last_mx;

	/* check for wrap */
	if (delta_x > 0x80)
		delta_x = 0x100-delta_x;
	if  (delta_x < -0x80)
		delta_x = -0x100-delta_x;

	board->last_mx = new_mx;

	/* compute y delta */
	delta_y = new_my - board->last_my;

	/* check for wrap */
	if (delta_y > 0x80)
		delta_y = 0x100-delta_y;
	if  (delta_y < -0x80)
		delta_y = -0x100-delta_y;

	board->last_my = new_my;

	video_update_mouse(board->video, delta_x, delta_y, buttons & 3);
}

/****************************************************************************
    TMS9901 and attached devices handling
****************************************************************************/

/*
    Called by the 9901 core whenever the state of INTREQ and IC0-3 changes
*/
static TMS9901_INT_CALLBACK( tms9901_interrupt_callback )
{
	/* INTREQ is connected to INT1 (IC0-3 are not connected) */
	cputag_set_input_line(device->machine(), "maincpu", 0, intreq? ASSERT_LINE : CLEAR_LINE);
}

/*
    Read pins INT3*-INT7* of Geneve 9901.

    signification:
     (bit 1: INT1 status)
     (bit 2: INT2 status)
     bit 3-7: joystick status
*/
static READ8_DEVICE_HANDLER( R9901_0 )
{
	device_t *dev = device->machine().device("geneve_board");
	genboard_state *board = get_safe_token(dev);
	int answer;
	answer = input_port_read(device->machine(), "JOY") >> (board->joySel * 8);
	return answer;
}

/*
    Read pins INT8*-INT15* of Geneve 9901.

    signification:
     (bit 0: keyboard interrupt)
     bit 1: unused
     bit 2: mouse right button
     (bit 3: clock interrupt)
     (bit 4: INTB from PE-bus)
     bit 5 & 7: used as output
     bit 6: unused
*/
static READ8_DEVICE_HANDLER( R9901_1 )
{
	int answer;
	answer = (input_port_read(device->machine(), "MOUSE0") & 4) ^ 4;
	return answer;
}

/*
    Read pins P0-P7 of Geneve 9901.
*/
static READ8_DEVICE_HANDLER( R9901_2 )
{
	return 0;
}

/*
    Read pins P8-P15 of Geneve 9901.
    bit 4: mouse right button
*/
static READ8_DEVICE_HANDLER( R9901_3 )
{
	int answer = 0;

	if (! (input_port_read(device->machine(), "MOUSE0") & 4))
		answer |= 0x10;

	return answer;
}

/*
    Write PE bus reset line
*/
static WRITE8_DEVICE_HANDLER( W9901_PE_bus_reset )
{
	if (VERBOSE>0) LOG("genboard: PE bus reset request, ignoring.\n");
}

/*
    Write VDP reset line
*/
static WRITE8_DEVICE_HANDLER( W9901_VDP_reset )
{
	if (VERBOSE>0) LOG("genboard: Video reset request, ignoring.\n");
}

/*
    Write joystick select line
*/
static WRITE8_DEVICE_HANDLER( W9901_joySel )
{
	device_t *dev = device->machine().device("geneve_board");
	genboard_state *board = get_safe_token(dev);
	board->joySel = data;
}

static WRITE8_DEVICE_HANDLER( W9901_keyboardReset )
{
	device_t *dev = device->machine().device("geneve_board");
	genboard_state *board = get_safe_token(dev);

	board->keyReset = !data;

	if (board->keyReset)
	{
		/* reset -> clear keyboard key queue, but not geneve key buffer */
		board->keyQueueLen = (board->keyInBuf)? 1 : 0;
		memset(board->keyStateSave, 0, sizeof(board->keyStateSave));
		board->keyNumLockState = 0;
		board->keyCtrlState = 0;
		board->keyAltState = 0;
		board->keyRealShiftState = 0;
		board->keyFakeShiftState = 0;
		board->keyFakeUnshiftState = 0;
		board->keyAutoRepeatKey = 0;
	}
	/*else
        poll_keyboard(space->machine());*/
}

/*
    Write external mem cycles (0=long, 1=short)
*/
static WRITE8_DEVICE_HANDLER( W9901_ext_mem_wait_states )
{
	if (VERBOSE>0) LOG("genboard: external bus wait states set to %d, not implemented yet.\n", data);
}

/*
    Write vdp wait cycles (1=add 14 cycles, 0=add none)
    see above for waitstate handling
*/
static WRITE8_DEVICE_HANDLER( W9901_VDP_wait_states )
{
	device_t *dev = device->machine().device("geneve_board");
	genboard_state *board = get_safe_token(dev);
	if (VERBOSE>1) LOG("genboard: vdp wait states set to %d\n", data);
	board->video_waitstates = (data!=0);
}

/* tms9901 setup */
const tms9901_interface tms9901_wiring_geneve =
{
	TMS9901_INT1 | TMS9901_INT2 | TMS9901_INT8 | TMS9901_INTB | TMS9901_INTC,	/* only input pins whose state is always known */

	{	/* read handlers */
		R9901_0,					// INTA, VDPint, Joystick
		R9901_1,					// Keyb, Mouse, Clock, INTB
		R9901_2,					// -
		R9901_3						// Mouse right button
	},

	{	/* write handlers */
		W9901_PE_bus_reset,
		W9901_VDP_reset,
		W9901_joySel,
		NULL,
		NULL,
		NULL,
		W9901_keyboardReset,
		W9901_ext_mem_wait_states,
		NULL,
		W9901_VDP_wait_states,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL
	},

	/* interrupt handler */
	tms9901_interrupt_callback,

	/* clock rate = 3MHz */
	3000000.
};

/*
    inta is connected to both tms9901 IRQ1 line and to tms9995 INT4/EC line.
*/
WRITE_LINE_DEVICE_HANDLER( board_inta )
{
	tms9901_set_single_int(device->machine().device("tms9901"), 1, state);
	cputag_set_input_line(device->machine(), "maincpu", 1, state);
}

/*
    intb is connected to tms9901 IRQ12 line.
*/
WRITE_LINE_DEVICE_HANDLER( board_intb )
{
	tms9901_set_single_int(device->machine().device("tms9901"), 12, state);

}

WRITE_LINE_DEVICE_HANDLER( board_ready )
{
	if (VERBOSE>0) LOG("genboard: READY line set ... not yet connected, level=%02x\n", state);
}

/*
    set the state of int2 (called by the v9938 core)
*/
void tms9901_gen_set_int2(running_machine &machine, int state)
{
	tms9901_set_single_int(machine.device("tms9901"), 2, state);
}

/***************************************************************************
    DEVICE LIFECYCLE FUNCTIONS
***************************************************************************/

static const mm58274c_interface geneve_mm58274c_interface =
{
	1,	/*  mode 24*/
	0   /*  first day of week */
};

static MACHINE_CONFIG_FRAGMENT( genboard )
	MCFG_MM58274C_ADD("mm58274c", geneve_mm58274c_interface)
MACHINE_CONFIG_END

static DEVICE_START( genboard )
{
	if (VERBOSE>0) LOG("genboard: Starting Geneve board\n");
	genboard_state *board = get_safe_token(device);

	board->video = device->siblingdevice("video");
	board->tms9901 = device->siblingdevice("tms9901");
	board->peribox = device->siblingdevice("peribox");
	board->cpu = device->siblingdevice("maincpu");
	board->sound = device->siblingdevice("soundgen");

	board->clock = device->subdevice("mm58274c");

	board->sram = (UINT8*)malloc(SRAM_SIZE);
	board->dram = (UINT8*)malloc(DRAM_SIZE);
	board->eprom = device->machine().region("maincpu")->base();
	assert(board->video && board->tms9901);
	assert(board->peribox && board->cpu);
	assert(board->sound && board->clock);
	assert(board->eprom);
	board->geneve_mode = false;
	board->direct_mode = true;

	// Used for generating the VDP waitstates. These are created in the
	// Gate Array on the board and only affect external memory accesses.
	// Therefore we cannot use the usual adjust_icount
	board->video_waittimer = device->machine().scheduler().timer_alloc(FUNC(video_wait_callback), (void *)device);

	memset(board->sram, 0x00, SRAM_SIZE);
	memset(board->dram, 0x00, DRAM_SIZE);
}

static DEVICE_STOP( genboard )
{
	if (VERBOSE>0) LOG("genboard: Stopping Geneve board\n");
	genboard_state *board = get_safe_token(device);
	free(board->sram);
	free(board->dram);
}

static DEVICE_RESET( genboard )
{
	genboard_state *board = get_safe_token(device);
	if (VERBOSE>0) LOG("genboard: Resetting Geneve board\n");
	board->joySel = 0;
	board->keyQueueHead = board->keyQueueLen = 0;
	memset(board->keyStateSave, 0, sizeof(board->keyStateSave));
	board->keyNumLockState = 0;
	board->keyCtrlState = 0;
	board->keyAltState = 0;
	board->keyRealShiftState = 0;
	board->keyFakeShiftState = 0;
	board->keyFakeUnshiftState = 0;
	board->keyAutoRepeatKey = 0;
	board->keyInBuf = false;
	board->keyReset = 1;
	board->last_mx = 0;
	board->last_my = 0;
	board->line_count = 0;

	board->palvideo = false;
	board->capslock = false;
	board->keyboard_clock = false;
	board->keep_keybuf = false;
	board->extra_waitstates = false;
	board->video_waitstates = true;
	board->video_waiting = false;

	board->geneve_mode =false;
	board->direct_mode = true;
	board->cartridge_size_8K = false;
	board->cartridge_secondpage = false;
	board->cartridge6_writable = false;
	board->cartridge7_writable = false;
	for (int i=0; i < 8; i++) board->map[i] = 0;

	board->genmod = false;
	if (input_port_read(device->machine(), "MODE")==0)
	{
		switch (input_port_read(device->machine(), "BOOTROM"))
		{
		case GENEVE_098:
			if (VERBOSE>0) LOG("genboard: Using 0.98 boot eprom\n");
			board->eprom = device->machine().region("maincpu")->base() + 0x4000;
			break;
		case GENEVE_100:
			if (VERBOSE>0) LOG("genboard: Using 1.00 boot eprom\n");
			board->eprom = device->machine().region("maincpu")->base();
			break;
		}
	}
	else
	{
		if (VERBOSE>0) LOG("genboard: Using GenMod modification\n");
		board->eprom = device->machine().region("maincpu")->base() + 0x8000;
		if (board->eprom[0] != 0xf0)
		{
			fatalerror("genboard: GenMod boot rom missing.\n");
		}
		board->genmod = true;
		board->turbo = ((input_port_read(device->machine(), "GENMODDIPS") & GM_TURBO)!=0);
		board->timode = ((input_port_read(device->machine(), "GENMODDIPS") & GM_TIM)!=0);
	}

	switch (input_port_read(device->machine(), "SRAM"))
	{
/*  1 100. .... .... .... .... on-board sram (128K) -+
    1 101. .... .... .... .... on-board sram (128K) -+-- maximum SRAM expansion
    1 1100 .... .... .... .... on-board sram (64K) --+
    1 1101 0... .... .... .... on-board sram (32K) - additional 32 KiB required for MDOS 2.50s and higher
    1 1101 1... .... .... .... on-board sram (32K) - standard setup
*/
	case 0: // 32 KiB
		board->sram_mask =	0x1f8000;
		board->sram_val =	0x1d8000;
		break;
	case 1: // 64 KiB
		board->sram_mask =	0x1f0000;
		board->sram_val =	0x1d0000;
		break;
	case 2: // 384 KiB (actually 512 KiB, but the EPROM masks the upper 128 KiB)
		board->sram_mask =	0x180000;
		board->sram_val =   0x180000;
		break;
	}
}

static const char DEVTEMPLATE_SOURCE[] = __FILE__;

#define DEVTEMPLATE_ID(p,s)             p##genboard##s
#define DEVTEMPLATE_FEATURES            DT_HAS_START | DT_HAS_STOP | DT_HAS_RESET | DT_HAS_MACHINE_CONFIG
#define DEVTEMPLATE_NAME                "Geneve system board"
#define DEVTEMPLATE_FAMILY              "Internal component"
#include "devtempl.h"

DEFINE_LEGACY_DEVICE( GENBOARD, genboard );
