package EnsEMBL::Web::Form::Field;

use strict;

use base qw(EnsEMBL::Web::DOM::Node::Element::Div);

################## STRUCTURE OF FIELD ##################
##  <div>                                             ##
##  <label>label</label><!--left column -->           ##
##  <div>Head notes</div><!-- right col (optional)--> ##
##  <div>Elements</div><!-- right col (multiple)-->   ##
##  <div>Foot notes</div><!-- right col (optional)--> ##
##  </div>                                            ##
########################################################

use constant {
  CSS_CLASS                 => 'form-field',
  CSS_CLASS_NOTES           => 'ff-notes',
  CSS_CLASS_LABEL           => 'ff-label',
  CSS_CLASS_ELEMENT_DIV     => 'ff-right',
  CSS_CLASS_INLINE_WRAPPER  => 'inline',
  
  _FLAG_FOOT_NOTES          => '_is_foot_note',
  _FLAG_HEAD_NOTES          => '_is_head_note',
  _FLAG_INLINE              => '_can_be_inline',
  _FLAG_ELEMENT             => '_is_field_element',
};

sub render {
  ## @overrides
  ## Sets the "for" attribute of <label> to the first element in the field before returning html
  my $self = shift;
  
  $self->set_attribute('class', $self->CSS_CLASS);

  my $label = $self->first_child && $self->first_child->node_name eq 'label' ? $self->first_child : undef;
  if ($label) {
    my $inputs = $self->inputs;
    if (scalar @$inputs
      && ($inputs->[0]->node_name =~ /^(select|textarea)$/
        || $inputs->[0]->node_name eq 'input' && $inputs->[0]->get_attribute('type') =~ /^(text|password|file)$/
        || $inputs->[0]->node_name eq 'input' && $inputs->[0]->get_attribute('type') =~ /^(checkbox|radio)$/ && scalar @$inputs == 1
      )
    ) {
      $inputs->[0]->id($self->unique_id) unless $inputs->[0]->id;
      $label->set_attribute('for', $inputs->[0]->id);
    }
  }
  return $self->SUPER::render;
}

sub label {
  ## Gets, modifies or adds new label to field
  ## @params String innerHTML for label
  ## @return DOM::Node::Element::Label object
  my $self = shift;
  my $label = $self->first_child && $self->first_child->node_name eq 'label'
    ? $self->first_child
    : $self->prepend_child($self->dom->create_element('label'));
  $label->set_attribute('class', $self->CSS_CLASS_LABEL);
  if (@_) {
    my $inner_HTML = shift;
    $inner_HTML .= ':' if $inner_HTML !~ /:$/;
    $label->inner_HTML($inner_HTML);
  }
  return $label;
}

sub head_notes {
  ## Gets, modifies or adds head notes to the field
  return shift->_notes('head', @_);
}

sub foot_notes {
  ## Gets, modifies or adds foot notes to the field
  return shift->_notes('foot', @_);
}

sub elements {
  ## Returns all the elements inside the field
  ## @return ArrayRef of Form::Element drived objects
  my $self = shift;
  return $self->get_child_nodes_by_flag($self->_FLAG_ELEMENT);
}

sub add_element {
  ## Adds a new element under existing label
  ## @params HashRef of standard parameters required for Form::Element->configure
  ## @params Inline flag - if on, tries to add the element inline with the previous element if possible
  my ($self, $params, $inline) = @_;
  
  my $children = $self->child_nodes;
  my $div = undef;

  my $element = $self->dom->create_element('form-element-'.$params->{'type'});
  
  #error handling
  if (!$element) {
    warn qq(DOM Could not create element "$params->{'type'}". Perhaps there's no corresponding class in Form::Element, or has not been mapped in Form::Element::map_element_class);
    return undef;
  }
  $element->configure($params);
  
  if ($inline && $element->node_name =~ /^(input|textarea|select)$/) { #if possible to fulfil the request for inline
    for (reverse @{$children}) {
      next if $_->get_flag($self->_FLAG_FOOT_NOTES);
      last unless $_->get_flag($self->_FLAG_INLINE);
      $div = $_;
      $div->set_attribute('class', $self->CSS_CLASS_INLINE_WRAPPER);
      last;
    }
  }
  unless ($div) {
    $div = $self->dom->create_element('div');
    $div->set_flag($self->_FLAG_INLINE);
    scalar @$children && $children->[-1]->get_flag($self->_FLAG_FOOT_NOTES) ? $self->insert_before($div, $children->[-1]) : $self->append_child($div);
  }

  if ($div->is_empty && $element->node_name eq 'div') { #to avoid nesting of divs
    $self->replace_child($element, $div);
    $div = $element;
  }
  else {
    $div->append_child($element);
  }
  $div->set_attribute('class', $self->CSS_CLASS_ELEMENT_DIV);
  $div->set_flag($self->_FLAG_ELEMENT);
  return $div;
}

sub inputs {
  ## Gets all input, select or textarea nodes present in the field
  ## @return ArrayRef of DOM::Node::Element::Select|TextArea and DOM::Node::Element::Input::*
  return shift->get_elements_by_tag_name([qw(input select textarea)]);
}

sub _notes {
  my $self = shift;
  my $location = shift eq 'head' ? 'head' : 'foot';
  my $identity_flag = $location eq 'head' ? $self->_FLAG_HEAD_NOTES : $self->_FLAG_FOOT_NOTES;
  my $notes = $self->get_child_nodes_by_flag($identity_flag);
  if (scalar @$notes) {
    $notes = shift @$notes;
  }
  else {
    $notes = $self->dom->create_element('div');
    $notes->set_flag($identity_flag);
    $notes->set_attribute('class', $self->CSS_CLASS_NOTES);
    if ($location eq 'head') {
      $self->first_child && $self->first_child->node_name eq 'label' ? $self->insert_after($notes, $self->first_child) : $self->prepend_child($notes);
    }
    else {
      $self->append_child($notes);
    }
  }
  $notes->inner_HTML(shift) if @_;
  return $notes;
}

1;