=head1 LICENSE

 Copyright (c) 1999-2011 The European Bioinformatics Institute and
 Genome Research Limited.  All rights reserved.

 This software is distributed under a modified Apache license.
 For license details, please see

   http://www.ensembl.org/info/about/code_licence.html

=head1 CONTACT

 Please email comments or questions to the public Ensembl
 developers list at <dev@ensembl.org>.

 Questions may also be sent to the Ensembl help desk at
 <helpdesk@ensembl.org>.

=cut

=head1 NAME

Bio::EnsEMBL::Variation::Utils::CondelConstants

=head1 DESCRIPTION

This module defines various constants used by the Bio::EnsEMBL::Variation::Utils::Condel
module to calculate the Condel (Consensus Deleteriousness) weighted average score for a 
missense mutation that has both a SIFT and PolyPhen-2 score. Condel is developed by the 
Biomedical Genomics Group of the Universitat Pompeu Fabra, at the Barcelona Biomedical 
Research Park (bg.upf.edu/Condel). The values in this module were provided by this group.

=cut

package Bio::EnsEMBL::Variation::Utils::CondelConstants;

use strict;
use warnings;

use base qw(Exporter);

our @EXPORT_OK = qw($CONDEL_CONFIG $CONDEL_SIFT_DATA $CONDEL_POLYPHEN_DATA);

our $CONDEL_CONFIG = {
          'max.HumVar.polyphen' => '1',
          'cutoff.HumVar.polyphen' => '0.28',
          'max.HumVar.sift' => '1',
          'cutoff.HumVar.sift' => '0.15'
        }
;

our $CONDEL_SIFT_DATA = {
          'tn' => {
                    '0.214' => '0.524',
                    '0.503' => '0.778',
                    '0.529' => '0.792',
                    '0.621' => '0.866',
                    '0.045' => '0.273',
                    '0.174' => '0.475',
                    '0.399' => '0.690',
                    '0.626' => '0.866',
                    '0.013' => '0.181',
                    '0.453' => '0.744',
                    '0.596' => '0.850',
                    '0.311' => '0.625',
                    '0.196' => '0.504',
                    '0.933' => '0.988',
                    '0.175' => '0.475',
                    '0.277' => '0.585',
                    '0.318' => '0.625',
                    '0.611' => '0.861',
                    '0.149' => '0.433',
                    '0.594' => '0.850',
                    '0.225' => '0.532',
                    '0.870' => '0.973',
                    '0.536' => '0.801',
                    '0.217' => '0.524',
                    '0.067' => '0.314',
                    '0.810' => '0.952',
                    '0.204' => '0.513',
                    '0.749' => '0.930',
                    '0.698' => '0.908',
                    '0.919' => '0.984',
                    '0.123' => '0.408',
                    '0.266' => '0.575',
                    '0.706' => '0.911',
                    '0.252' => '0.567',
                    '0.825' => '0.958',
                    '0.336' => '0.645',
                    '0.228' => '0.532',
                    '0.735' => '0.925',
                    '0.897' => '0.980',
                    '0.150' => '0.433',
                    '0.022' => '0.218',
                    '0.977' => '0.997',
                    '0.236' => '0.544',
                    '0.837' => '0.962',
                    '0.999' => '1.000',
                    '0.365' => '0.668',
                    '0.491' => '0.770',
                    '0.146' => '0.433',
                    '0.581' => '0.841',
                    '0.345' => '0.653',
                    '0.361' => '0.668',
                    '0.790' => '0.946',
                    '0.241' => '0.555',
                    '0.274' => '0.585',
                    '0.522' => '0.792',
                    '0.844' => '0.966',
                    '0.725' => '0.921',
                    '0.800' => '0.950',
                    '0.309' => '0.612',
                    '0.560' => '0.816',
                    '0.424' => '0.720',
                    '0.496' => '0.770',
                    '0.179' => '0.475',
                    '0.696' => '0.908',
                    '0.947' => '0.990',
                    '0.411' => '0.711',
                    '0.935' => '0.988',
                    '0.299' => '0.604',
                    '0.379' => '0.676',
                    '0.079' => '0.329',
                    '0.803' => '0.952',
                    '0.089' => '0.348',
                    '0.113' => '0.390',
                    '0.116' => '0.390',
                    '0.511' => '0.785',
                    '0.681' => '0.902',
                    '0.600' => '0.850',
                    '0.302' => '0.612',
                    '0.484' => '0.762',
                    '0.556' => '0.816',
                    '0.502' => '0.778',
                    '0.328' => '0.635',
                    '0.473' => '0.756',
                    '0.924' => '0.986',
                    '0.221' => '0.532',
                    '0.386' => '0.682',
                    '0.641' => '0.877',
                    '0.927' => '0.986',
                    '0.982' => '0.999',
                    '0.261' => '0.575',
                    '0.996' => '1.000',
                    '0.921' => '0.986',
                    '0.315' => '0.625',
                    '0.400' => '0.690',
                    '0.772' => '0.943',
                    '0.767' => '0.939',
                    '0.675' => '0.896',
                    '0.375' => '0.676',
                    '0.954' => '0.992',
                    '0.510' => '0.785',
                    '0.651' => '0.881',
                    '0.876' => '0.974',
                    '0.090' => '0.348',
                    '0.043' => '0.273',
                    '0.740' => '0.925',
                    '0.868' => '0.973',
                    '0.168' => '0.462',
                    '0.723' => '0.921',
                    '0.636' => '0.872',
                    '0.433' => '0.728',
                    '0.820' => '0.958',
                    '0.193' => '0.504',
                    '0.715' => '0.916',
                    '0.098' => '0.363',
                    '0.444' => '0.736',
                    '0.366' => '0.668',
                    '0.403' => '0.702',
                    '0.099' => '0.363',
                    '0.224' => '0.532',
                    '0.838' => '0.962',
                    '0.107' => '0.376',
                    '0.226' => '0.532',
                    '0.455' => '0.744',
                    '0.240' => '0.555',
                    '0.398' => '0.690',
                    '0.828' => '0.958',
                    '0.452' => '0.744',
                    '0.078' => '0.329',
                    '0.535' => '0.801',
                    '0.593' => '0.850',
                    '0.635' => '0.872',
                    '0.251' => '0.567',
                    '0.465' => '0.750',
                    '0.934' => '0.988',
                    '0.726' => '0.921',
                    '0.990' => '0.999',
                    '0.943' => '0.990',
                    '0.014' => '0.181',
                    '0.183' => '0.490',
                    '0.015' => '0.181',
                    '0.072' => '0.329',
                    '0.816' => '0.956',
                    '0.325' => '0.635',
                    '0.903' => '0.981',
                    '0.734' => '0.925',
                    '0.760' => '0.939',
                    '0.034' => '0.248',
                    '0.629' => '0.866',
                    '0.969' => '0.995',
                    '0.374' => '0.676',
                    '0.942' => '0.990',
                    '0.342' => '0.653',
                    '0.490' => '0.762',
                    '0.066' => '0.314',
                    '0.898' => '0.980',
                    '0.267' => '0.575',
                    '0.312' => '0.625',
                    '0.235' => '0.544',
                    '0.761' => '0.939',
                    '0.276' => '0.585',
                    '0.871' => '0.974',
                    '0.901' => '0.981',
                    '1.000' => '1.000',
                    '0.209' => '0.513',
                    '0.610' => '0.855',
                    '0.438' => '0.728',
                    '0.176' => '0.475',
                    '0.114' => '0.390',
                    '0.413' => '0.711',
                    '0.563' => '0.826',
                    '0.697' => '0.908',
                    '0.062' => '0.314',
                    '0.432' => '0.728',
                    '0.821' => '0.958',
                    '0.009' => '0.111',
                    '0.387' => '0.682',
                    '0.762' => '0.939',
                    '0.167' => '0.462',
                    '0.485' => '0.762',
                    '0.615' => '0.861',
                    '0.184' => '0.490',
                    '0.642' => '0.877',
                    '0.980' => '0.997',
                    '0.004' => '0.111',
                    '0.867' => '0.973',
                    '0.910' => '0.981',
                    '0.414' => '0.711',
                    '0.087' => '0.348',
                    '0.824' => '0.958',
                    '0.805' => '0.952',
                    '0.595' => '0.850',
                    '0.577' => '0.834',
                    '0.115' => '0.390',
                    '0.280' => '0.585',
                    '0.587' => '0.841',
                    '0.768' => '0.939',
                    '0.426' => '0.720',
                    '0.195' => '0.504',
                    '0.853' => '0.970',
                    '0.303' => '0.612',
                    '0.467' => '0.750',
                    '0.518' => '0.785',
                    '0.660' => '0.881',
                    '0.088' => '0.348',
                    '0.001' => '0.111',
                    '0.159' => '0.448',
                    '0.785' => '0.946',
                    '0.423' => '0.720',
                    '0.203' => '0.513',
                    '0.101' => '0.376',
                    '0.289' => '0.593',
                    '0.337' => '0.645',
                    '0.360' => '0.662',
                    '0.155' => '0.448',
                    '0.962' => '0.995',
                    '0.879' => '0.974',
                    '0.687' => '0.902',
                    '0.649' => '0.877',
                    '0.538' => '0.801',
                    '0.029' => '0.218',
                    '0.298' => '0.604',
                    '0.808' => '0.952',
                    '0.352' => '0.662',
                    '0.804' => '0.952',
                    '0.953' => '0.992',
                    '0.519' => '0.785',
                    '0.404' => '0.702',
                    '0.676' => '0.896',
                    '0.040' => '0.273',
                    '0.448' => '0.736',
                    '0.998' => '1.000',
                    '0.132' => '0.420',
                    '0.652' => '0.881',
                    '0.408' => '0.702',
                    '0.724' => '0.921',
                    '0.936' => '0.988',
                    '0.716' => '0.916',
                    '0.796' => '0.950',
                    '0.827' => '0.958',
                    '0.223' => '0.532',
                    '0.765' => '0.939',
                    '0.441' => '0.736',
                    '0.406' => '0.702',
                    '0.565' => '0.826',
                    '0.717' => '0.916',
                    '0.562' => '0.826',
                    '0.703' => '0.911',
                    '0.944' => '0.990',
                    '0.018' => '0.181',
                    '0.060' => '0.314',
                    '0.320' => '0.635',
                    '0.005' => '0.111',
                    '0.481' => '0.762',
                    '0.523' => '0.792',
                    '0.733' => '0.925',
                    '0.609' => '0.855',
                    '0.659' => '0.881',
                    '0.985' => '0.999',
                    '0.727' => '0.921',
                    '0.855' => '0.970',
                    '0.475' => '0.756',
                    '0.656' => '0.881',
                    '0.931' => '0.988',
                    '0.385' => '0.682',
                    '0.291' => '0.604',
                    '0.513' => '0.785',
                    '0.887' => '0.977',
                    '0.321' => '0.635',
                    '0.435' => '0.728',
                    '0.873' => '0.974',
                    '0.021' => '0.218',
                    '0.339' => '0.645',
                    '0.624' => '0.866',
                    '0.991' => '1.000',
                    '0.598' => '0.850',
                    '0.373' => '0.676',
                    '0.047' => '0.273',
                    '0.680' => '0.896',
                    '0.301' => '0.612',
                    '0.547' => '0.807',
                    '0.065' => '0.314',
                    '0.554' => '0.816',
                    '0.884' => '0.977',
                    '0.125' => '0.408',
                    '0.618' => '0.861',
                    '0.037' => '0.248',
                    '0.341' => '0.653',
                    '0.308' => '0.612',
                    '0.965' => '0.995',
                    '0.454' => '0.744',
                    '0.842' => '0.966',
                    '0.279' => '0.585',
                    '0.233' => '0.544',
                    '0.839' => '0.962',
                    '0.140' => '0.420',
                    '0.555' => '0.816',
                    '0.709' => '0.911',
                    '0.108' => '0.376',
                    '0.464' => '0.750',
                    '0.574' => '0.834',
                    '0.100' => '0.363',
                    '0.382' => '0.682',
                    '0.471' => '0.756',
                    '0.486' => '0.762',
                    '0.752' => '0.934',
                    '0.198' => '0.504',
                    '0.911' => '0.984',
                    '0.614' => '0.861',
                    '0.959' => '0.992',
                    '0.219' => '0.524',
                    '0.967' => '0.995',
                    '0.257' => '0.567',
                    '0.355' => '0.662',
                    '0.157' => '0.448',
                    '0.759' => '0.934',
                    '0.631' => '0.872',
                    '0.468' => '0.750',
                    '0.533' => '0.801',
                    '0.643' => '0.877',
                    '0.978' => '0.997',
                    '0.860' => '0.970',
                    '0.427' => '0.720',
                    '0.120' => '0.408',
                    '0.793' => '0.950',
                    '0.832' => '0.962',
                    '0.242' => '0.555',
                    '0.602' => '0.855',
                    '0.648' => '0.877',
                    '0.900' => '0.981',
                    '0.841' => '0.966',
                    '0.338' => '0.645',
                    '0.671' => '0.896',
                    '0.878' => '0.974',
                    '0.449' => '0.736',
                    '0.690' => '0.908',
                    '0.054' => '0.294',
                    '0.782' => '0.946',
                    '0.288' => '0.593',
                    '0.958' => '0.992',
                    '0.872' => '0.974',
                    '0.359' => '0.662',
                    '0.788' => '0.946',
                    '0.391' => '0.690',
                    '0.730' => '0.921',
                    '0.997' => '1.000',
                    '0.970' => '0.995',
                    '0.754' => '0.934',
                    '0.059' => '0.294',
                    '0.250' => '0.555',
                    '0.686' => '0.902',
                    '0.081' => '0.348',
                    '0.286' => '0.593',
                    '0.988' => '0.999',
                    '0.929' => '0.986',
                    '0.070' => '0.314',
                    '0.422' => '0.720',
                    '0.376' => '0.676',
                    '0.742' => '0.930',
                    '0.819' => '0.956',
                    '0.152' => '0.448',
                    '0.694' => '0.908',
                    '0.789' => '0.946',
                    '0.728' => '0.921',
                    '0.010' => '0.181',
                    '0.313' => '0.625',
                    '0.504' => '0.778',
                    '0.807' => '0.952',
                    '0.627' => '0.866',
                    '0.357' => '0.662',
                    '0.524' => '0.792',
                    '0.044' => '0.273',
                    '0.945' => '0.990',
                    '0.061' => '0.314',
                    '0.512' => '0.785',
                    '0.589' => '0.841',
                    '0.350' => '0.662',
                    '0.869' => '0.973',
                    '0.647' => '0.877',
                    '0.718' => '0.916',
                    '0.826' => '0.958',
                    '0.566' => '0.826',
                    '0.020' => '0.218',
                    '0.739' => '0.925',
                    '0.474' => '0.756',
                    '0.017' => '0.181',
                    '0.290' => '0.604',
                    '0.619' => '0.861',
                    '0.766' => '0.939',
                    '0.011' => '0.181',
                    '0.537' => '0.801',
                    '0.006' => '0.111',
                    '0.620' => '0.861',
                    '0.912' => '0.984',
                    '0.344' => '0.653',
                    '0.463' => '0.750',
                    '0.750' => '0.930',
                    '0.561' => '0.826',
                    '0.854' => '0.970',
                    '0.278' => '0.585',
                    '0.384' => '0.682',
                    '0.216' => '0.524',
                    '0.658' => '0.881',
                    '0.166' => '0.462',
                    '0.372' => '0.676',
                    '0.297' => '0.604',
                    '0.582' => '0.841',
                    '0.493' => '0.770',
                    '0.487' => '0.762',
                    '0.147' => '0.433',
                    '0.181' => '0.490',
                    '0.368' => '0.668',
                    '0.894' => '0.980',
                    '0.950' => '0.992',
                    '0.135' => '0.420',
                    '0.672' => '0.896',
                    '0.056' => '0.294',
                    '0.105' => '0.376',
                    '0.548' => '0.807',
                    '0.888' => '0.977',
                    '0.905' => '0.981',
                    '0.951' => '0.992',
                    '0.093' => '0.363',
                    '0.886' => '0.977',
                    '0.809' => '0.952',
                    '0.158' => '0.448',
                    '0.042' => '0.273',
                    '0.968' => '0.995',
                    '0.335' => '0.645',
                    '0.895' => '0.980',
                    '0.601' => '0.855',
                    '0.881' => '0.977',
                    '0.035' => '0.248',
                    '0.258' => '0.567',
                    '0.012' => '0.181',
                    '0.189' => '0.490',
                    '0.058' => '0.294',
                    '0.885' => '0.977',
                    '0.960' => '0.995',
                    '0.751' => '0.934',
                    '0.091' => '0.363',
                    '0.525' => '0.792',
                    '0.896' => '0.980',
                    '0.271' => '0.585',
                    '0.412' => '0.711',
                    '0.704' => '0.911',
                    '0.003' => '0.111',
                    '0.741' => '0.930',
                    '0.472' => '0.756',
                    '0.208' => '0.513',
                    '0.316' => '0.625',
                    '0.390' => '0.690',
                    '0.469' => '0.750',
                    '0.966' => '0.995',
                    '0.456' => '0.744',
                    '0.383' => '0.682',
                    '0.845' => '0.966',
                    '0.833' => '0.962',
                    '0.269' => '0.575',
                    '0.597' => '0.850',
                    '0.637' => '0.872',
                    '0.590' => '0.850',
                    '0.126' => '0.408',
                    '0.197' => '0.504',
                    '0.446' => '0.736',
                    '0.922' => '0.986',
                    '0.957' => '0.992',
                    '0.787' => '0.946',
                    '0.695' => '0.908',
                    '0.753' => '0.934',
                    '0.780' => '0.946',
                    '0.573' => '0.834',
                    '0.210' => '0.524',
                    '0.177' => '0.475',
                    '0.987' => '0.999',
                    '0.731' => '0.925',
                    '0.019' => '0.181',
                    '0.440' => '0.736',
                    '0.650' => '0.881',
                    '0.679' => '0.896',
                    '0.127' => '0.408',
                    '0.505' => '0.778',
                    '0.802' => '0.952',
                    '0.939' => '0.988',
                    '0.036' => '0.248',
                    '0.260' => '0.575',
                    '0.377' => '0.676',
                    '0.567' => '0.826',
                    '0.256' => '0.567',
                    '0.287' => '0.593',
                    '0.327' => '0.635',
                    '0.255' => '0.567',
                    '0.993' => '1.000',
                    '0.729' => '0.921',
                    '0.685' => '0.902',
                    '0.367' => '0.668',
                    '0.865' => '0.973',
                    '0.849' => '0.966',
                    '0.075' => '0.329',
                    '0.607' => '0.855',
                    '0.480' => '0.762',
                    '0.284' => '0.593',
                    '0.477' => '0.756',
                    '0.458' => '0.744',
                    '0.732' => '0.925',
                    '0.296' => '0.604',
                    '0.349' => '0.653',
                    '0.915' => '0.984',
                    '0.578' => '0.834',
                    '0.495' => '0.770',
                    '0.332' => '0.645',
                    '0.343' => '0.653',
                    '0.756' => '0.934',
                    '0.646' => '0.877',
                    '0.092' => '0.363',
                    '0.745' => '0.930',
                    '0.693' => '0.908',
                    '0.248' => '0.555',
                    '0.811' => '0.956',
                    '0.182' => '0.490',
                    '0.295' => '0.604',
                    '0.421' => '0.720',
                    '0.410' => '0.711',
                    '0.333' => '0.645',
                    '0.483' => '0.762',
                    '0.542' => '0.807',
                    '0.645' => '0.877',
                    '0.639' => '0.872',
                    '0.654' => '0.881',
                    '0.492' => '0.770',
                    '0.171' => '0.475',
                    '0.692' => '0.908',
                    '0.604' => '0.855',
                    '0.572' => '0.834',
                    '0.779' => '0.943',
                    '0.085' => '0.348',
                    '0.389' => '0.682',
                    '0.416' => '0.711',
                    '0.200' => '0.504',
                    '0.575' => '0.834',
                    '0.963' => '0.995',
                    '0.913' => '0.984',
                    '0.136' => '0.420',
                    '0.048' => '0.273',
                    '0.952' => '0.992',
                    '0.875' => '0.974',
                    '0.662' => '0.891',
                    '0.892' => '0.980',
                    '0.154' => '0.448',
                    '0.106' => '0.376',
                    '0.738' => '0.925',
                    '0.165' => '0.462',
                    '0.823' => '0.958',
                    '0.797' => '0.950',
                    '0.370' => '0.668',
                    '0.545' => '0.807',
                    '0.319' => '0.625',
                    '0.443' => '0.736',
                    '0.262' => '0.575',
                    '0.314' => '0.625',
                    '0.428' => '0.720',
                    '0.142' => '0.433',
                    '0.324' => '0.635',
                    '0.880' => '0.977',
                    '0.437' => '0.728',
                    '0.119' => '0.390',
                    '0.417' => '0.711',
                    '0.148' => '0.433',
                    '0.666' => '0.891',
                    '0.141' => '0.433',
                    '0.861' => '0.973',
                    '0.520' => '0.792',
                    '0.234' => '0.544',
                    '0.498' => '0.770',
                    '0.128' => '0.408',
                    '0.381' => '0.682',
                    '0.568' => '0.826',
                    '0.461' => '0.750',
                    '0.170' => '0.462',
                    '0.773' => '0.943',
                    '0.008' => '0.111',
                    '0.479' => '0.756',
                    '0.457' => '0.744',
                    '0.986' => '0.999',
                    '0.889' => '0.977',
                    '0.553' => '0.816',
                    '0.273' => '0.585',
                    '0.707' => '0.911',
                    '0.046' => '0.273',
                    '0.025' => '0.218',
                    '0.834' => '0.962',
                    '0.531' => '0.801',
                    '0.268' => '0.575',
                    '0.143' => '0.433',
                    '0.220' => '0.532',
                    '0.976' => '0.997',
                    '0.244' => '0.555',
                    '0.893' => '0.980',
                    '0.310' => '0.612',
                    '0.974' => '0.997',
                    '0.801' => '0.952',
                    '0.050' => '0.273',
                    '0.030' => '0.248',
                    '0.699' => '0.908',
                    '0.283' => '0.593',
                    '0.488' => '0.762',
                    '0.701' => '0.911',
                    '0.906' => '0.981',
                    '0.857' => '0.970',
                    '0.795' => '0.950',
                    '0.506' => '0.778',
                    '0.462' => '0.750',
                    '0.926' => '0.986',
                    '0.095' => '0.363',
                    '0.992' => '1.000',
                    '0.678' => '0.896',
                    '0.178' => '0.475',
                    '0.932' => '0.988',
                    '0.211' => '0.524',
                    '0.395' => '0.690',
                    '0.720' => '0.916',
                    '0.138' => '0.420',
                    '0.508' => '0.778',
                    '0.371' => '0.676',
                    '0.584' => '0.841',
                    '0.814' => '0.956',
                    '0.285' => '0.593',
                    '0.515' => '0.785',
                    '0.322' => '0.635',
                    '0.755' => '0.934',
                    '0.559' => '0.816',
                    '0.546' => '0.807',
                    '0.429' => '0.720',
                    '0.644' => '0.877',
                    '0.459' => '0.744',
                    '0.956' => '0.992',
                    '0.856' => '0.970',
                    '0.583' => '0.841',
                    '0.118' => '0.390',
                    '0.201' => '0.513',
                    '0.657' => '0.881',
                    '0.405' => '0.702',
                    '0.702' => '0.911',
                    '0.708' => '0.911',
                    '0.661' => '0.891',
                    '0.526' => '0.792',
                    '0.964' => '0.995',
                    '0.848' => '0.966',
                    '0.743' => '0.930',
                    '0.514' => '0.785',
                    '0.329' => '0.635',
                    '0.434' => '0.728',
                    '0.909' => '0.981',
                    '0.551' => '0.816',
                    '0.393' => '0.690',
                    '0.039' => '0.248',
                    '0.447' => '0.736',
                    '0.798' => '0.950',
                    '0.024' => '0.218',
                    '0.476' => '0.756',
                    '0.576' => '0.834',
                    '0.420' => '0.720',
                    '0.670' => '0.891',
                    '0.164' => '0.462',
                    '0.192' => '0.504',
                    '0.172' => '0.475',
                    '0.254' => '0.567',
                    '0.831' => '0.962',
                    '0.973' => '0.997',
                    '0.306' => '0.612',
                    '0.348' => '0.653',
                    '0.920' => '0.984',
                    '0.710' => '0.916',
                    '0.084' => '0.348',
                    '0.989' => '0.999',
                    '0.625' => '0.866',
                    '0.864' => '0.973',
                    '0.539' => '0.801',
                    '0.353' => '0.662',
                    '0.916' => '0.984',
                    '0.364' => '0.668',
                    '0.812' => '0.956',
                    '0.270' => '0.575',
                    '0.334' => '0.645',
                    '0.442' => '0.736',
                    '0.263' => '0.575',
                    '0.031' => '0.248',
                    '0.983' => '0.999',
                    '0.231' => '0.544',
                    '0.249' => '0.555',
                    '0.603' => '0.855',
                    '0.684' => '0.902',
                    '0.038' => '0.248',
                    '0.064' => '0.314',
                    '0.914' => '0.984',
                    '0.608' => '0.855',
                    '0.638' => '0.872',
                    '0.552' => '0.816',
                    '0.229' => '0.532',
                    '0.307' => '0.612',
                    '0.674' => '0.896',
                    '0.080' => '0.348',
                    '0.975' => '0.997',
                    '0.356' => '0.662',
                    '0.630' => '0.872',
                    '0.470' => '0.756',
                    '0.946' => '0.990',
                    '0.781' => '0.946',
                    '0.744' => '0.930',
                    '0.691' => '0.908',
                    '0.613' => '0.861',
                    '0.392' => '0.690',
                    '0.843' => '0.966',
                    '0.774' => '0.943',
                    '0.817' => '0.956',
                    '0.300' => '0.604',
                    '0.129' => '0.408',
                    '0.218' => '0.524',
                    '0.835' => '0.962',
                    '0.592' => '0.850',
                    '0.409' => '0.702',
                    '0.737' => '0.925',
                    '0.571' => '0.834',
                    '0.294' => '0.604',
                    '0.599' => '0.850',
                    '0.746' => '0.930',
                    '0.904' => '0.981',
                    '0.829' => '0.958',
                    '0.794' => '0.950',
                    '0.665' => '0.891',
                    '0.923' => '0.986',
                    '0.151' => '0.448',
                    '0.507' => '0.778',
                    '0.632' => '0.872',
                    '0.499' => '0.770',
                    '0.137' => '0.420',
                    '0.055' => '0.294',
                    '0.207' => '0.513',
                    '0.890' => '0.980',
                    '0.840' => '0.966',
                    '0.541' => '0.807',
                    '0.534' => '0.801',
                    '0.071' => '0.329',
                    '0.243' => '0.555',
                    '0.000' => '0.111',
                    '0.961' => '0.995',
                    '0.202' => '0.513',
                    '0.397' => '0.690',
                    '0.677' => '0.896',
                    '0.569' => '0.826',
                    '0.134' => '0.420',
                    '0.928' => '0.986',
                    '0.264' => '0.575',
                    '0.191' => '0.504',
                    '0.634' => '0.872',
                    '0.917' => '0.984',
                    '0.972' => '0.997',
                    '0.570' => '0.834',
                    '0.305' => '0.612',
                    '0.460' => '0.744',
                    '0.130' => '0.420',
                    '0.238' => '0.544',
                    '0.246' => '0.555',
                    '0.053' => '0.294',
                    '0.103' => '0.376',
                    '0.144' => '0.433',
                    '0.077' => '0.329',
                    '0.402' => '0.702',
                    '0.478' => '0.756',
                    '0.347' => '0.653',
                    '0.187' => '0.490',
                    '0.813' => '0.956',
                    '0.239' => '0.544',
                    '0.190' => '0.504',
                    '0.758' => '0.934',
                    '0.747' => '0.930',
                    '0.206' => '0.513',
                    '0.073' => '0.329',
                    '0.883' => '0.977',
                    '0.764' => '0.939',
                    '0.836' => '0.962',
                    '0.497' => '0.770',
                    '0.232' => '0.544',
                    '0.431' => '0.728',
                    '0.830' => '0.962',
                    '0.272' => '0.585',
                    '0.586' => '0.841',
                    '0.363' => '0.668',
                    '0.792' => '0.950',
                    '0.527' => '0.792',
                    '0.082' => '0.348',
                    '0.212' => '0.524',
                    '0.979' => '0.997',
                    '0.517' => '0.785',
                    '0.930' => '0.986',
                    '0.323' => '0.635',
                    '0.532' => '0.801',
                    '0.769' => '0.939',
                    '0.899' => '0.980',
                    '0.713' => '0.916',
                    '0.439' => '0.728',
                    '0.358' => '0.662',
                    '0.783' => '0.946',
                    '0.721' => '0.921',
                    '0.069' => '0.314',
                    '0.185' => '0.490',
                    '0.786' => '0.946',
                    '0.033' => '0.248',
                    '0.253' => '0.567',
                    '0.117' => '0.390',
                    '0.330' => '0.635',
                    '0.955' => '0.992',
                    '0.558' => '0.816',
                    '0.086' => '0.348',
                    '0.293' => '0.604',
                    '0.711' => '0.916',
                    '0.049' => '0.273',
                    '0.388' => '0.682',
                    '0.027' => '0.218',
                    '0.891' => '0.980',
                    '0.736' => '0.925',
                    '0.653' => '0.881',
                    '0.028' => '0.218',
                    '0.775' => '0.943',
                    '0.984' => '0.999',
                    '0.023' => '0.218',
                    '0.664' => '0.891',
                    '0.778' => '0.943',
                    '0.445' => '0.736',
                    '0.874' => '0.974',
                    '0.543' => '0.807',
                    '0.937' => '0.988',
                    '0.500' => '0.770',
                    '0.628' => '0.866',
                    '0.818' => '0.956',
                    '0.588' => '0.841',
                    '0.579' => '0.834',
                    '0.162' => '0.462',
                    '0.940' => '0.990',
                    '0.494' => '0.770',
                    '0.763' => '0.939',
                    '0.362' => '0.668',
                    '0.163' => '0.462',
                    '0.770' => '0.943',
                    '0.847' => '0.966',
                    '0.683' => '0.902',
                    '0.668' => '0.891',
                    '0.419' => '0.711',
                    '0.451' => '0.744',
                    '0.799' => '0.950',
                    '0.700' => '0.911',
                    '0.156' => '0.448',
                    '0.111' => '0.390',
                    '0.407' => '0.702',
                    '0.199' => '0.504',
                    '0.994' => '1.000',
                    '0.673' => '0.896',
                    '0.281' => '0.593',
                    '0.304' => '0.612',
                    '0.949' => '0.990',
                    '0.540' => '0.801',
                    '0.096' => '0.363',
                    '0.139' => '0.420',
                    '0.617' => '0.861',
                    '0.369' => '0.668',
                    '0.623' => '0.866',
                    '0.580' => '0.841',
                    '0.057' => '0.294',
                    '0.669' => '0.891',
                    '0.852' => '0.970',
                    '0.063' => '0.314',
                    '0.180' => '0.475',
                    '0.863' => '0.973',
                    '0.041' => '0.273',
                    '0.351' => '0.662',
                    '0.109' => '0.376',
                    '0.133' => '0.420',
                    '0.907' => '0.981',
                    '0.354' => '0.662',
                    '0.591' => '0.850',
                    '0.430' => '0.720',
                    '0.612' => '0.861',
                    '0.173' => '0.475',
                    '0.052' => '0.294',
                    '0.032' => '0.248',
                    '0.237' => '0.544',
                    '0.882' => '0.977',
                    '0.131' => '0.420',
                    '0.205' => '0.513',
                    '0.145' => '0.433',
                    '0.188' => '0.490',
                    '0.557' => '0.816',
                    '0.247' => '0.555',
                    '0.705' => '0.911',
                    '0.450' => '0.744',
                    '0.489' => '0.762',
                    '0.074' => '0.329',
                    '0.378' => '0.676',
                    '0.846' => '0.966',
                    '0.102' => '0.376',
                    '0.748' => '0.930',
                    '0.215' => '0.524',
                    '0.757' => '0.934',
                    '0.401' => '0.702',
                    '0.784' => '0.946',
                    '0.971' => '0.997',
                    '0.640' => '0.877',
                    '0.230' => '0.532',
                    '0.585' => '0.841',
                    '0.002' => '0.111',
                    '0.275' => '0.585',
                    '0.822' => '0.958',
                    '0.866' => '0.973',
                    '0.918' => '0.984',
                    '0.415' => '0.711',
                    '0.851' => '0.970',
                    '0.097' => '0.363',
                    '0.122' => '0.408',
                    '0.981' => '0.999',
                    '0.633' => '0.872',
                    '0.083' => '0.348',
                    '0.688' => '0.902',
                    '0.160' => '0.462',
                    '0.605' => '0.855',
                    '0.068' => '0.314',
                    '0.776' => '0.943',
                    '0.331' => '0.645',
                    '0.689' => '0.902',
                    '0.153' => '0.448',
                    '0.245' => '0.555',
                    '0.076' => '0.329',
                    '0.380' => '0.682',
                    '0.161' => '0.462',
                    '0.714' => '0.916',
                    '0.606' => '0.855',
                    '0.259' => '0.567',
                    '0.425' => '0.720',
                    '0.791' => '0.950',
                    '0.859' => '0.970',
                    '0.528' => '0.792',
                    '0.346' => '0.653',
                    '0.213' => '0.524',
                    '0.682' => '0.902',
                    '0.418' => '0.711',
                    '0.112' => '0.390',
                    '0.094' => '0.363',
                    '0.516' => '0.785',
                    '0.124' => '0.408',
                    '0.941' => '0.990',
                    '0.777' => '0.943',
                    '0.396' => '0.690',
                    '0.806' => '0.952',
                    '0.501' => '0.778',
                    '0.436' => '0.728',
                    '0.544' => '0.807',
                    '0.616' => '0.861',
                    '0.186' => '0.490',
                    '0.466' => '0.750',
                    '0.104' => '0.376',
                    '0.222' => '0.532',
                    '0.858' => '0.970',
                    '0.564' => '0.826',
                    '0.026' => '0.218',
                    '0.317' => '0.625',
                    '0.292' => '0.604',
                    '0.850' => '0.966',
                    '0.712' => '0.916',
                    '0.938' => '0.988',
                    '0.995' => '1.000',
                    '0.530' => '0.801',
                    '0.007' => '0.111',
                    '0.862' => '0.973',
                    '0.771' => '0.943',
                    '0.482' => '0.762',
                    '0.550' => '0.807',
                    '0.016' => '0.181',
                    '0.051' => '0.294',
                    '0.655' => '0.881',
                    '0.722' => '0.921',
                    '0.521' => '0.792',
                    '0.194' => '0.504',
                    '0.815' => '0.956',
                    '0.282' => '0.593',
                    '0.110' => '0.390',
                    '0.169' => '0.462',
                    '0.549' => '0.807',
                    '0.227' => '0.532',
                    '0.877' => '0.974',
                    '0.667' => '0.891',
                    '0.121' => '0.408',
                    '0.326' => '0.635',
                    '0.948' => '0.990',
                    '0.925' => '0.986',
                    '0.265' => '0.575',
                    '0.719' => '0.916',
                    '0.509' => '0.778',
                    '0.394' => '0.690',
                    '0.340' => '0.645',
                    '0.622' => '0.866',
                    '0.908' => '0.981',
                    '0.663' => '0.891',
                    '0.902' => '0.981'
                  },
          'tp' => {
                    '0.214' => '0.888',
                    '0.503' => '0.962',
                    '0.529' => '0.965',
                    '0.621' => '0.981',
                    '0.045' => '0.721',
                    '0.174' => '0.864',
                    '0.399' => '0.947',
                    '0.626' => '0.981',
                    '0.013' => '0.609',
                    '0.453' => '0.956',
                    '0.596' => '0.977',
                    '0.311' => '0.926',
                    '0.196' => '0.878',
                    '0.933' => '0.998',
                    '0.175' => '0.864',
                    '0.277' => '0.912',
                    '0.318' => '0.926',
                    '0.611' => '0.980',
                    '0.149' => '0.844',
                    '0.594' => '0.977',
                    '0.225' => '0.892',
                    '0.870' => '0.995',
                    '0.536' => '0.968',
                    '0.217' => '0.888',
                    '0.067' => '0.760',
                    '0.810' => '0.993',
                    '0.204' => '0.883',
                    '0.749' => '0.990',
                    '0.698' => '0.987',
                    '0.919' => '0.997',
                    '0.123' => '0.827',
                    '0.266' => '0.908',
                    '0.706' => '0.988',
                    '0.252' => '0.904',
                    '0.825' => '0.994',
                    '0.336' => '0.931',
                    '0.228' => '0.892',
                    '0.735' => '0.989',
                    '0.897' => '0.997',
                    '0.150' => '0.844',
                    '0.022' => '0.661',
                    '0.977' => '1.000',
                    '0.236' => '0.896',
                    '0.837' => '0.994',
                    '0.999' => '1.000',
                    '0.365' => '0.939',
                    '0.491' => '0.961',
                    '0.146' => '0.844',
                    '0.581' => '0.976',
                    '0.345' => '0.934',
                    '0.361' => '0.939',
                    '0.790' => '0.992',
                    '0.241' => '0.901',
                    '0.274' => '0.912',
                    '0.522' => '0.965',
                    '0.844' => '0.995',
                    '0.725' => '0.989',
                    '0.800' => '0.993',
                    '0.309' => '0.922',
                    '0.560' => '0.971',
                    '0.424' => '0.952',
                    '0.496' => '0.961',
                    '0.179' => '0.864',
                    '0.696' => '0.987',
                    '0.947' => '0.999',
                    '0.411' => '0.950',
                    '0.935' => '0.998',
                    '0.299' => '0.919',
                    '0.379' => '0.941',
                    '0.079' => '0.774',
                    '0.803' => '0.993',
                    '0.089' => '0.787',
                    '0.113' => '0.819',
                    '0.116' => '0.819',
                    '0.511' => '0.964',
                    '0.681' => '0.987',
                    '0.600' => '0.977',
                    '0.302' => '0.922',
                    '0.484' => '0.959',
                    '0.556' => '0.971',
                    '0.502' => '0.962',
                    '0.328' => '0.929',
                    '0.473' => '0.958',
                    '0.924' => '0.998',
                    '0.221' => '0.892',
                    '0.386' => '0.943',
                    '0.641' => '0.983',
                    '0.927' => '0.998',
                    '0.982' => '1.000',
                    '0.261' => '0.908',
                    '0.996' => '1.000',
                    '0.921' => '0.998',
                    '0.315' => '0.926',
                    '0.400' => '0.947',
                    '0.772' => '0.992',
                    '0.767' => '0.991',
                    '0.675' => '0.986',
                    '0.375' => '0.941',
                    '0.954' => '0.999',
                    '0.510' => '0.964',
                    '0.651' => '0.984',
                    '0.876' => '0.996',
                    '0.090' => '0.787',
                    '0.043' => '0.721',
                    '0.740' => '0.989',
                    '0.868' => '0.995',
                    '0.168' => '0.858',
                    '0.723' => '0.989',
                    '0.636' => '0.982',
                    '0.433' => '0.953',
                    '0.820' => '0.994',
                    '0.193' => '0.878',
                    '0.715' => '0.988',
                    '0.098' => '0.797',
                    '0.444' => '0.955',
                    '0.366' => '0.939',
                    '0.403' => '0.949',
                    '0.099' => '0.797',
                    '0.224' => '0.892',
                    '0.838' => '0.994',
                    '0.107' => '0.809',
                    '0.226' => '0.892',
                    '0.455' => '0.956',
                    '0.240' => '0.901',
                    '0.398' => '0.947',
                    '0.828' => '0.994',
                    '0.452' => '0.956',
                    '0.078' => '0.774',
                    '0.535' => '0.968',
                    '0.593' => '0.977',
                    '0.635' => '0.982',
                    '0.251' => '0.904',
                    '0.465' => '0.957',
                    '0.934' => '0.998',
                    '0.726' => '0.989',
                    '0.990' => '1.000',
                    '0.943' => '0.999',
                    '0.014' => '0.609',
                    '0.183' => '0.871',
                    '0.015' => '0.609',
                    '0.072' => '0.774',
                    '0.816' => '0.993',
                    '0.325' => '0.929',
                    '0.903' => '0.997',
                    '0.734' => '0.989',
                    '0.760' => '0.991',
                    '0.034' => '0.696',
                    '0.629' => '0.981',
                    '0.969' => '0.999',
                    '0.374' => '0.941',
                    '0.942' => '0.999',
                    '0.342' => '0.934',
                    '0.490' => '0.959',
                    '0.066' => '0.760',
                    '0.898' => '0.997',
                    '0.267' => '0.908',
                    '0.312' => '0.926',
                    '0.235' => '0.896',
                    '0.761' => '0.991',
                    '0.276' => '0.912',
                    '0.871' => '0.996',
                    '0.901' => '0.997',
                    '1.000' => '1.000',
                    '0.209' => '0.883',
                    '0.610' => '0.979',
                    '0.438' => '0.953',
                    '0.176' => '0.864',
                    '0.114' => '0.819',
                    '0.413' => '0.950',
                    '0.563' => '0.972',
                    '0.697' => '0.987',
                    '0.062' => '0.760',
                    '0.432' => '0.953',
                    '0.821' => '0.994',
                    '0.009' => '0.483',
                    '0.387' => '0.943',
                    '0.762' => '0.991',
                    '0.167' => '0.858',
                    '0.485' => '0.959',
                    '0.615' => '0.980',
                    '0.184' => '0.871',
                    '0.642' => '0.983',
                    '0.980' => '1.000',
                    '0.004' => '0.483',
                    '0.867' => '0.995',
                    '0.910' => '0.997',
                    '0.414' => '0.950',
                    '0.087' => '0.787',
                    '0.824' => '0.994',
                    '0.805' => '0.993',
                    '0.595' => '0.977',
                    '0.577' => '0.973',
                    '0.115' => '0.819',
                    '0.280' => '0.912',
                    '0.587' => '0.976',
                    '0.768' => '0.991',
                    '0.426' => '0.952',
                    '0.195' => '0.878',
                    '0.853' => '0.995',
                    '0.303' => '0.922',
                    '0.467' => '0.957',
                    '0.518' => '0.964',
                    '0.660' => '0.984',
                    '0.088' => '0.787',
                    '0.001' => '0.483',
                    '0.159' => '0.851',
                    '0.785' => '0.992',
                    '0.423' => '0.952',
                    '0.203' => '0.883',
                    '0.101' => '0.809',
                    '0.289' => '0.915',
                    '0.337' => '0.931',
                    '0.360' => '0.937',
                    '0.155' => '0.851',
                    '0.962' => '0.999',
                    '0.879' => '0.996',
                    '0.687' => '0.987',
                    '0.649' => '0.983',
                    '0.538' => '0.968',
                    '0.029' => '0.661',
                    '0.298' => '0.919',
                    '0.808' => '0.993',
                    '0.352' => '0.937',
                    '0.804' => '0.993',
                    '0.953' => '0.999',
                    '0.519' => '0.964',
                    '0.404' => '0.949',
                    '0.676' => '0.986',
                    '0.040' => '0.721',
                    '0.448' => '0.955',
                    '0.998' => '1.000',
                    '0.132' => '0.835',
                    '0.652' => '0.984',
                    '0.408' => '0.949',
                    '0.724' => '0.989',
                    '0.936' => '0.998',
                    '0.716' => '0.988',
                    '0.796' => '0.993',
                    '0.827' => '0.994',
                    '0.223' => '0.892',
                    '0.765' => '0.991',
                    '0.441' => '0.955',
                    '0.406' => '0.949',
                    '0.565' => '0.972',
                    '0.717' => '0.988',
                    '0.562' => '0.972',
                    '0.703' => '0.988',
                    '0.944' => '0.999',
                    '0.018' => '0.609',
                    '0.060' => '0.760',
                    '0.320' => '0.929',
                    '0.005' => '0.483',
                    '0.481' => '0.959',
                    '0.523' => '0.965',
                    '0.733' => '0.989',
                    '0.609' => '0.979',
                    '0.659' => '0.984',
                    '0.985' => '1.000',
                    '0.727' => '0.989',
                    '0.855' => '0.995',
                    '0.475' => '0.958',
                    '0.656' => '0.984',
                    '0.931' => '0.998',
                    '0.385' => '0.943',
                    '0.291' => '0.919',
                    '0.513' => '0.964',
                    '0.887' => '0.996',
                    '0.321' => '0.929',
                    '0.435' => '0.953',
                    '0.873' => '0.996',
                    '0.021' => '0.661',
                    '0.339' => '0.931',
                    '0.624' => '0.981',
                    '0.991' => '1.000',
                    '0.598' => '0.977',
                    '0.373' => '0.941',
                    '0.047' => '0.721',
                    '0.680' => '0.986',
                    '0.301' => '0.922',
                    '0.547' => '0.969',
                    '0.065' => '0.760',
                    '0.554' => '0.971',
                    '0.884' => '0.996',
                    '0.125' => '0.827',
                    '0.618' => '0.980',
                    '0.037' => '0.696',
                    '0.341' => '0.934',
                    '0.308' => '0.922',
                    '0.965' => '0.999',
                    '0.454' => '0.956',
                    '0.842' => '0.995',
                    '0.279' => '0.912',
                    '0.233' => '0.896',
                    '0.839' => '0.994',
                    '0.140' => '0.835',
                    '0.555' => '0.971',
                    '0.709' => '0.988',
                    '0.108' => '0.809',
                    '0.464' => '0.957',
                    '0.574' => '0.973',
                    '0.100' => '0.797',
                    '0.382' => '0.943',
                    '0.471' => '0.958',
                    '0.486' => '0.959',
                    '0.752' => '0.991',
                    '0.198' => '0.878',
                    '0.911' => '0.997',
                    '0.614' => '0.980',
                    '0.959' => '0.999',
                    '0.219' => '0.888',
                    '0.967' => '0.999',
                    '0.257' => '0.904',
                    '0.355' => '0.937',
                    '0.157' => '0.851',
                    '0.759' => '0.991',
                    '0.631' => '0.982',
                    '0.468' => '0.957',
                    '0.533' => '0.968',
                    '0.643' => '0.983',
                    '0.978' => '1.000',
                    '0.860' => '0.995',
                    '0.427' => '0.952',
                    '0.120' => '0.827',
                    '0.793' => '0.993',
                    '0.832' => '0.994',
                    '0.242' => '0.901',
                    '0.602' => '0.979',
                    '0.648' => '0.983',
                    '0.900' => '0.997',
                    '0.841' => '0.995',
                    '0.338' => '0.931',
                    '0.671' => '0.986',
                    '0.878' => '0.996',
                    '0.449' => '0.955',
                    '0.690' => '0.987',
                    '0.054' => '0.741',
                    '0.782' => '0.992',
                    '0.288' => '0.915',
                    '0.958' => '0.999',
                    '0.872' => '0.996',
                    '0.359' => '0.937',
                    '0.788' => '0.992',
                    '0.391' => '0.947',
                    '0.730' => '0.989',
                    '0.997' => '1.000',
                    '0.970' => '0.999',
                    '0.754' => '0.991',
                    '0.059' => '0.741',
                    '0.250' => '0.901',
                    '0.686' => '0.987',
                    '0.081' => '0.787',
                    '0.286' => '0.915',
                    '0.988' => '1.000',
                    '0.929' => '0.998',
                    '0.070' => '0.760',
                    '0.422' => '0.952',
                    '0.376' => '0.941',
                    '0.742' => '0.990',
                    '0.819' => '0.993',
                    '0.152' => '0.851',
                    '0.694' => '0.987',
                    '0.789' => '0.992',
                    '0.728' => '0.989',
                    '0.010' => '0.609',
                    '0.313' => '0.926',
                    '0.504' => '0.962',
                    '0.807' => '0.993',
                    '0.627' => '0.981',
                    '0.357' => '0.937',
                    '0.524' => '0.965',
                    '0.044' => '0.721',
                    '0.945' => '0.999',
                    '0.061' => '0.760',
                    '0.512' => '0.964',
                    '0.589' => '0.976',
                    '0.350' => '0.937',
                    '0.869' => '0.995',
                    '0.647' => '0.983',
                    '0.718' => '0.988',
                    '0.826' => '0.994',
                    '0.566' => '0.972',
                    '0.020' => '0.661',
                    '0.739' => '0.989',
                    '0.474' => '0.958',
                    '0.017' => '0.609',
                    '0.290' => '0.919',
                    '0.619' => '0.980',
                    '0.766' => '0.991',
                    '0.011' => '0.609',
                    '0.537' => '0.968',
                    '0.006' => '0.483',
                    '0.620' => '0.980',
                    '0.912' => '0.997',
                    '0.344' => '0.934',
                    '0.463' => '0.957',
                    '0.750' => '0.990',
                    '0.561' => '0.972',
                    '0.854' => '0.995',
                    '0.278' => '0.912',
                    '0.384' => '0.943',
                    '0.216' => '0.888',
                    '0.658' => '0.984',
                    '0.166' => '0.858',
                    '0.372' => '0.941',
                    '0.297' => '0.919',
                    '0.582' => '0.976',
                    '0.493' => '0.961',
                    '0.487' => '0.959',
                    '0.147' => '0.844',
                    '0.181' => '0.871',
                    '0.368' => '0.939',
                    '0.894' => '0.997',
                    '0.950' => '0.999',
                    '0.135' => '0.835',
                    '0.672' => '0.986',
                    '0.056' => '0.741',
                    '0.105' => '0.809',
                    '0.548' => '0.969',
                    '0.888' => '0.996',
                    '0.905' => '0.997',
                    '0.951' => '0.999',
                    '0.093' => '0.797',
                    '0.886' => '0.996',
                    '0.809' => '0.993',
                    '0.158' => '0.851',
                    '0.042' => '0.721',
                    '0.968' => '0.999',
                    '0.335' => '0.931',
                    '0.895' => '0.997',
                    '0.601' => '0.979',
                    '0.881' => '0.996',
                    '0.035' => '0.696',
                    '0.258' => '0.904',
                    '0.012' => '0.609',
                    '0.189' => '0.871',
                    '0.058' => '0.741',
                    '0.885' => '0.996',
                    '0.960' => '0.999',
                    '0.751' => '0.991',
                    '0.091' => '0.797',
                    '0.525' => '0.965',
                    '0.896' => '0.997',
                    '0.271' => '0.912',
                    '0.412' => '0.950',
                    '0.704' => '0.988',
                    '0.003' => '0.483',
                    '0.741' => '0.990',
                    '0.472' => '0.958',
                    '0.208' => '0.883',
                    '0.316' => '0.926',
                    '0.390' => '0.947',
                    '0.469' => '0.957',
                    '0.966' => '0.999',
                    '0.456' => '0.956',
                    '0.383' => '0.943',
                    '0.845' => '0.995',
                    '0.833' => '0.994',
                    '0.269' => '0.908',
                    '0.597' => '0.977',
                    '0.637' => '0.982',
                    '0.590' => '0.977',
                    '0.126' => '0.827',
                    '0.197' => '0.878',
                    '0.446' => '0.955',
                    '0.922' => '0.998',
                    '0.957' => '0.999',
                    '0.787' => '0.992',
                    '0.695' => '0.987',
                    '0.753' => '0.991',
                    '0.780' => '0.992',
                    '0.573' => '0.973',
                    '0.210' => '0.888',
                    '0.177' => '0.864',
                    '0.987' => '1.000',
                    '0.731' => '0.989',
                    '0.019' => '0.609',
                    '0.440' => '0.955',
                    '0.650' => '0.984',
                    '0.679' => '0.986',
                    '0.127' => '0.827',
                    '0.505' => '0.962',
                    '0.802' => '0.993',
                    '0.939' => '0.998',
                    '0.036' => '0.696',
                    '0.260' => '0.908',
                    '0.377' => '0.941',
                    '0.567' => '0.972',
                    '0.256' => '0.904',
                    '0.287' => '0.915',
                    '0.327' => '0.929',
                    '0.255' => '0.904',
                    '0.993' => '1.000',
                    '0.729' => '0.989',
                    '0.685' => '0.987',
                    '0.367' => '0.939',
                    '0.865' => '0.995',
                    '0.849' => '0.995',
                    '0.075' => '0.774',
                    '0.607' => '0.979',
                    '0.480' => '0.959',
                    '0.284' => '0.915',
                    '0.477' => '0.958',
                    '0.458' => '0.956',
                    '0.732' => '0.989',
                    '0.296' => '0.919',
                    '0.349' => '0.934',
                    '0.915' => '0.997',
                    '0.578' => '0.973',
                    '0.495' => '0.961',
                    '0.332' => '0.931',
                    '0.343' => '0.934',
                    '0.756' => '0.991',
                    '0.646' => '0.983',
                    '0.092' => '0.797',
                    '0.745' => '0.990',
                    '0.693' => '0.987',
                    '0.248' => '0.901',
                    '0.811' => '0.993',
                    '0.182' => '0.871',
                    '0.295' => '0.919',
                    '0.421' => '0.952',
                    '0.410' => '0.950',
                    '0.333' => '0.931',
                    '0.483' => '0.959',
                    '0.542' => '0.969',
                    '0.645' => '0.983',
                    '0.639' => '0.982',
                    '0.654' => '0.984',
                    '0.492' => '0.961',
                    '0.171' => '0.864',
                    '0.692' => '0.987',
                    '0.604' => '0.979',
                    '0.572' => '0.973',
                    '0.779' => '0.992',
                    '0.085' => '0.787',
                    '0.389' => '0.943',
                    '0.416' => '0.950',
                    '0.200' => '0.878',
                    '0.575' => '0.973',
                    '0.963' => '0.999',
                    '0.913' => '0.997',
                    '0.136' => '0.835',
                    '0.048' => '0.721',
                    '0.952' => '0.999',
                    '0.875' => '0.996',
                    '0.662' => '0.985',
                    '0.892' => '0.997',
                    '0.154' => '0.851',
                    '0.106' => '0.809',
                    '0.738' => '0.989',
                    '0.165' => '0.858',
                    '0.823' => '0.994',
                    '0.797' => '0.993',
                    '0.370' => '0.939',
                    '0.545' => '0.969',
                    '0.319' => '0.926',
                    '0.443' => '0.955',
                    '0.262' => '0.908',
                    '0.314' => '0.926',
                    '0.428' => '0.952',
                    '0.142' => '0.844',
                    '0.324' => '0.929',
                    '0.880' => '0.996',
                    '0.437' => '0.953',
                    '0.119' => '0.819',
                    '0.417' => '0.950',
                    '0.148' => '0.844',
                    '0.666' => '0.985',
                    '0.141' => '0.844',
                    '0.861' => '0.995',
                    '0.520' => '0.965',
                    '0.234' => '0.896',
                    '0.498' => '0.961',
                    '0.128' => '0.827',
                    '0.381' => '0.943',
                    '0.568' => '0.972',
                    '0.461' => '0.957',
                    '0.170' => '0.858',
                    '0.773' => '0.992',
                    '0.008' => '0.483',
                    '0.479' => '0.958',
                    '0.457' => '0.956',
                    '0.986' => '1.000',
                    '0.889' => '0.996',
                    '0.553' => '0.971',
                    '0.273' => '0.912',
                    '0.707' => '0.988',
                    '0.046' => '0.721',
                    '0.025' => '0.661',
                    '0.834' => '0.994',
                    '0.531' => '0.968',
                    '0.268' => '0.908',
                    '0.143' => '0.844',
                    '0.220' => '0.892',
                    '0.976' => '1.000',
                    '0.244' => '0.901',
                    '0.893' => '0.997',
                    '0.310' => '0.922',
                    '0.974' => '1.000',
                    '0.801' => '0.993',
                    '0.050' => '0.721',
                    '0.030' => '0.696',
                    '0.699' => '0.987',
                    '0.283' => '0.915',
                    '0.488' => '0.959',
                    '0.701' => '0.988',
                    '0.906' => '0.997',
                    '0.857' => '0.995',
                    '0.795' => '0.993',
                    '0.506' => '0.962',
                    '0.462' => '0.957',
                    '0.926' => '0.998',
                    '0.095' => '0.797',
                    '0.992' => '1.000',
                    '0.678' => '0.986',
                    '0.178' => '0.864',
                    '0.932' => '0.998',
                    '0.211' => '0.888',
                    '0.395' => '0.947',
                    '0.720' => '0.988',
                    '0.138' => '0.835',
                    '0.508' => '0.962',
                    '0.371' => '0.941',
                    '0.584' => '0.976',
                    '0.814' => '0.993',
                    '0.285' => '0.915',
                    '0.515' => '0.964',
                    '0.322' => '0.929',
                    '0.755' => '0.991',
                    '0.559' => '0.971',
                    '0.546' => '0.969',
                    '0.429' => '0.952',
                    '0.644' => '0.983',
                    '0.459' => '0.956',
                    '0.956' => '0.999',
                    '0.856' => '0.995',
                    '0.583' => '0.976',
                    '0.118' => '0.819',
                    '0.201' => '0.883',
                    '0.657' => '0.984',
                    '0.405' => '0.949',
                    '0.702' => '0.988',
                    '0.708' => '0.988',
                    '0.661' => '0.985',
                    '0.526' => '0.965',
                    '0.964' => '0.999',
                    '0.848' => '0.995',
                    '0.743' => '0.990',
                    '0.514' => '0.964',
                    '0.329' => '0.929',
                    '0.434' => '0.953',
                    '0.909' => '0.997',
                    '0.551' => '0.971',
                    '0.393' => '0.947',
                    '0.039' => '0.696',
                    '0.447' => '0.955',
                    '0.798' => '0.993',
                    '0.024' => '0.661',
                    '0.476' => '0.958',
                    '0.576' => '0.973',
                    '0.420' => '0.952',
                    '0.670' => '0.985',
                    '0.164' => '0.858',
                    '0.192' => '0.878',
                    '0.172' => '0.864',
                    '0.254' => '0.904',
                    '0.831' => '0.994',
                    '0.973' => '1.000',
                    '0.306' => '0.922',
                    '0.348' => '0.934',
                    '0.920' => '0.997',
                    '0.710' => '0.988',
                    '0.084' => '0.787',
                    '0.989' => '1.000',
                    '0.625' => '0.981',
                    '0.864' => '0.995',
                    '0.539' => '0.968',
                    '0.353' => '0.937',
                    '0.916' => '0.997',
                    '0.364' => '0.939',
                    '0.812' => '0.993',
                    '0.270' => '0.908',
                    '0.334' => '0.931',
                    '0.442' => '0.955',
                    '0.263' => '0.908',
                    '0.031' => '0.696',
                    '0.983' => '1.000',
                    '0.231' => '0.896',
                    '0.249' => '0.901',
                    '0.603' => '0.979',
                    '0.684' => '0.987',
                    '0.038' => '0.696',
                    '0.064' => '0.760',
                    '0.914' => '0.997',
                    '0.608' => '0.979',
                    '0.638' => '0.982',
                    '0.552' => '0.971',
                    '0.229' => '0.892',
                    '0.307' => '0.922',
                    '0.674' => '0.986',
                    '0.080' => '0.787',
                    '0.975' => '1.000',
                    '0.356' => '0.937',
                    '0.630' => '0.982',
                    '0.470' => '0.958',
                    '0.946' => '0.999',
                    '0.781' => '0.992',
                    '0.744' => '0.990',
                    '0.691' => '0.987',
                    '0.613' => '0.980',
                    '0.392' => '0.947',
                    '0.843' => '0.995',
                    '0.774' => '0.992',
                    '0.817' => '0.993',
                    '0.300' => '0.919',
                    '0.129' => '0.827',
                    '0.218' => '0.888',
                    '0.835' => '0.994',
                    '0.592' => '0.977',
                    '0.409' => '0.949',
                    '0.737' => '0.989',
                    '0.571' => '0.973',
                    '0.294' => '0.919',
                    '0.599' => '0.977',
                    '0.746' => '0.990',
                    '0.904' => '0.997',
                    '0.829' => '0.994',
                    '0.794' => '0.993',
                    '0.665' => '0.985',
                    '0.923' => '0.998',
                    '0.151' => '0.851',
                    '0.507' => '0.962',
                    '0.632' => '0.982',
                    '0.499' => '0.961',
                    '0.137' => '0.835',
                    '0.055' => '0.741',
                    '0.207' => '0.883',
                    '0.890' => '0.997',
                    '0.840' => '0.995',
                    '0.541' => '0.969',
                    '0.534' => '0.968',
                    '0.071' => '0.774',
                    '0.243' => '0.901',
                    '0.000' => '0.483',
                    '0.961' => '0.999',
                    '0.202' => '0.883',
                    '0.397' => '0.947',
                    '0.677' => '0.986',
                    '0.569' => '0.972',
                    '0.134' => '0.835',
                    '0.928' => '0.998',
                    '0.264' => '0.908',
                    '0.191' => '0.878',
                    '0.634' => '0.982',
                    '0.917' => '0.997',
                    '0.972' => '1.000',
                    '0.570' => '0.973',
                    '0.305' => '0.922',
                    '0.460' => '0.956',
                    '0.130' => '0.835',
                    '0.238' => '0.896',
                    '0.246' => '0.901',
                    '0.053' => '0.741',
                    '0.103' => '0.809',
                    '0.144' => '0.844',
                    '0.077' => '0.774',
                    '0.402' => '0.949',
                    '0.478' => '0.958',
                    '0.347' => '0.934',
                    '0.187' => '0.871',
                    '0.813' => '0.993',
                    '0.239' => '0.896',
                    '0.190' => '0.878',
                    '0.758' => '0.991',
                    '0.747' => '0.990',
                    '0.206' => '0.883',
                    '0.073' => '0.774',
                    '0.883' => '0.996',
                    '0.764' => '0.991',
                    '0.836' => '0.994',
                    '0.497' => '0.961',
                    '0.232' => '0.896',
                    '0.431' => '0.953',
                    '0.830' => '0.994',
                    '0.272' => '0.912',
                    '0.586' => '0.976',
                    '0.363' => '0.939',
                    '0.792' => '0.993',
                    '0.527' => '0.965',
                    '0.082' => '0.787',
                    '0.212' => '0.888',
                    '0.979' => '1.000',
                    '0.517' => '0.964',
                    '0.930' => '0.998',
                    '0.323' => '0.929',
                    '0.532' => '0.968',
                    '0.769' => '0.991',
                    '0.899' => '0.997',
                    '0.713' => '0.988',
                    '0.439' => '0.953',
                    '0.358' => '0.937',
                    '0.783' => '0.992',
                    '0.721' => '0.989',
                    '0.069' => '0.760',
                    '0.185' => '0.871',
                    '0.786' => '0.992',
                    '0.033' => '0.696',
                    '0.253' => '0.904',
                    '0.117' => '0.819',
                    '0.330' => '0.929',
                    '0.955' => '0.999',
                    '0.558' => '0.971',
                    '0.086' => '0.787',
                    '0.293' => '0.919',
                    '0.711' => '0.988',
                    '0.049' => '0.721',
                    '0.388' => '0.943',
                    '0.027' => '0.661',
                    '0.891' => '0.997',
                    '0.736' => '0.989',
                    '0.653' => '0.984',
                    '0.028' => '0.661',
                    '0.775' => '0.992',
                    '0.984' => '1.000',
                    '0.023' => '0.661',
                    '0.664' => '0.985',
                    '0.778' => '0.992',
                    '0.445' => '0.955',
                    '0.874' => '0.996',
                    '0.543' => '0.969',
                    '0.937' => '0.998',
                    '0.500' => '0.961',
                    '0.628' => '0.981',
                    '0.818' => '0.993',
                    '0.588' => '0.976',
                    '0.579' => '0.973',
                    '0.162' => '0.858',
                    '0.940' => '0.999',
                    '0.494' => '0.961',
                    '0.763' => '0.991',
                    '0.362' => '0.939',
                    '0.163' => '0.858',
                    '0.770' => '0.992',
                    '0.847' => '0.995',
                    '0.683' => '0.987',
                    '0.668' => '0.985',
                    '0.419' => '0.950',
                    '0.451' => '0.956',
                    '0.799' => '0.993',
                    '0.700' => '0.988',
                    '0.156' => '0.851',
                    '0.111' => '0.819',
                    '0.407' => '0.949',
                    '0.199' => '0.878',
                    '0.994' => '1.000',
                    '0.673' => '0.986',
                    '0.281' => '0.915',
                    '0.304' => '0.922',
                    '0.949' => '0.999',
                    '0.540' => '0.968',
                    '0.096' => '0.797',
                    '0.139' => '0.835',
                    '0.617' => '0.980',
                    '0.369' => '0.939',
                    '0.623' => '0.981',
                    '0.580' => '0.976',
                    '0.057' => '0.741',
                    '0.669' => '0.985',
                    '0.852' => '0.995',
                    '0.063' => '0.760',
                    '0.180' => '0.864',
                    '0.863' => '0.995',
                    '0.041' => '0.721',
                    '0.351' => '0.937',
                    '0.109' => '0.809',
                    '0.133' => '0.835',
                    '0.907' => '0.997',
                    '0.354' => '0.937',
                    '0.591' => '0.977',
                    '0.430' => '0.952',
                    '0.612' => '0.980',
                    '0.173' => '0.864',
                    '0.052' => '0.741',
                    '0.032' => '0.696',
                    '0.237' => '0.896',
                    '0.882' => '0.996',
                    '0.131' => '0.835',
                    '0.205' => '0.883',
                    '0.145' => '0.844',
                    '0.188' => '0.871',
                    '0.557' => '0.971',
                    '0.247' => '0.901',
                    '0.705' => '0.988',
                    '0.450' => '0.956',
                    '0.489' => '0.959',
                    '0.074' => '0.774',
                    '0.378' => '0.941',
                    '0.846' => '0.995',
                    '0.102' => '0.809',
                    '0.748' => '0.990',
                    '0.215' => '0.888',
                    '0.757' => '0.991',
                    '0.401' => '0.949',
                    '0.784' => '0.992',
                    '0.971' => '1.000',
                    '0.640' => '0.983',
                    '0.230' => '0.892',
                    '0.585' => '0.976',
                    '0.002' => '0.483',
                    '0.275' => '0.912',
                    '0.822' => '0.994',
                    '0.866' => '0.995',
                    '0.918' => '0.997',
                    '0.415' => '0.950',
                    '0.851' => '0.995',
                    '0.097' => '0.797',
                    '0.122' => '0.827',
                    '0.981' => '1.000',
                    '0.633' => '0.982',
                    '0.083' => '0.787',
                    '0.688' => '0.987',
                    '0.160' => '0.858',
                    '0.605' => '0.979',
                    '0.068' => '0.760',
                    '0.776' => '0.992',
                    '0.331' => '0.931',
                    '0.689' => '0.987',
                    '0.153' => '0.851',
                    '0.245' => '0.901',
                    '0.076' => '0.774',
                    '0.380' => '0.943',
                    '0.161' => '0.858',
                    '0.714' => '0.988',
                    '0.606' => '0.979',
                    '0.259' => '0.904',
                    '0.425' => '0.952',
                    '0.791' => '0.993',
                    '0.859' => '0.995',
                    '0.528' => '0.965',
                    '0.346' => '0.934',
                    '0.213' => '0.888',
                    '0.682' => '0.987',
                    '0.418' => '0.950',
                    '0.112' => '0.819',
                    '0.094' => '0.797',
                    '0.516' => '0.964',
                    '0.124' => '0.827',
                    '0.941' => '0.999',
                    '0.777' => '0.992',
                    '0.396' => '0.947',
                    '0.806' => '0.993',
                    '0.501' => '0.962',
                    '0.436' => '0.953',
                    '0.544' => '0.969',
                    '0.616' => '0.980',
                    '0.186' => '0.871',
                    '0.466' => '0.957',
                    '0.104' => '0.809',
                    '0.222' => '0.892',
                    '0.858' => '0.995',
                    '0.564' => '0.972',
                    '0.026' => '0.661',
                    '0.317' => '0.926',
                    '0.292' => '0.919',
                    '0.850' => '0.995',
                    '0.712' => '0.988',
                    '0.938' => '0.998',
                    '0.995' => '1.000',
                    '0.530' => '0.968',
                    '0.007' => '0.483',
                    '0.862' => '0.995',
                    '0.771' => '0.992',
                    '0.482' => '0.959',
                    '0.550' => '0.969',
                    '0.016' => '0.609',
                    '0.051' => '0.741',
                    '0.655' => '0.984',
                    '0.722' => '0.989',
                    '0.521' => '0.965',
                    '0.194' => '0.878',
                    '0.815' => '0.993',
                    '0.282' => '0.915',
                    '0.110' => '0.819',
                    '0.169' => '0.858',
                    '0.549' => '0.969',
                    '0.227' => '0.892',
                    '0.877' => '0.996',
                    '0.667' => '0.985',
                    '0.121' => '0.827',
                    '0.326' => '0.929',
                    '0.948' => '0.999',
                    '0.925' => '0.998',
                    '0.265' => '0.908',
                    '0.719' => '0.988',
                    '0.509' => '0.962',
                    '0.394' => '0.947',
                    '0.340' => '0.931',
                    '0.622' => '0.981',
                    '0.908' => '0.997',
                    '0.663' => '0.985',
                    '0.902' => '0.997'
                  }
        }
;

our $CONDEL_POLYPHEN_DATA = {
          'tn' => {
                    '0.214' => '0.411',
                    '0.621' => '0.303',
                    '0.529' => '0.325',
                    '0.503' => '0.332',
                    '0.174' => '0.429',
                    '0.045' => '0.549',
                    '0.399' => '0.357',
                    '0.626' => '0.303',
                    '0.596' => '0.308',
                    '0.453' => '0.345',
                    '0.013' => '0.667',
                    '0.311' => '0.376',
                    '0.933' => '0.188',
                    '0.196' => '0.419',
                    '0.175' => '0.429',
                    '0.611' => '0.307',
                    '0.318' => '0.375',
                    '0.277' => '0.387',
                    '0.149' => '0.445',
                    '0.594' => '0.309',
                    '0.225' => '0.407',
                    '0.870' => '0.223',
                    '0.536' => '0.324',
                    '0.217' => '0.410',
                    '0.810' => '0.247',
                    '0.067' => '0.514',
                    '0.204' => '0.415',
                    '0.749' => '0.266',
                    '0.919' => '0.196',
                    '0.698' => '0.281',
                    '0.123' => '0.464',
                    '0.706' => '0.280',
                    '0.266' => '0.393',
                    '0.825' => '0.243',
                    '0.252' => '0.398',
                    '0.336' => '0.370',
                    '0.735' => '0.269',
                    '0.228' => '0.406',
                    '0.897' => '0.211',
                    '0.150' => '0.444',
                    '0.977' => '0.142',
                    '0.022' => '0.619',
                    '0.236' => '0.404',
                    '0.837' => '0.236',
                    '0.999' => '0.018',
                    '0.365' => '0.364',
                    '0.491' => '0.334',
                    '0.581' => '0.312',
                    '0.146' => '0.447',
                    '0.361' => '0.365',
                    '0.345' => '0.368',
                    '0.790' => '0.250',
                    '0.241' => '0.401',
                    '0.274' => '0.388',
                    '0.844' => '0.234',
                    '0.522' => '0.326',
                    '0.800' => '0.247',
                    '0.725' => '0.273',
                    '0.560' => '0.318',
                    '0.309' => '0.377',
                    '0.496' => '0.333',
                    '0.424' => '0.351',
                    '0.696' => '0.283',
                    '0.179' => '0.428',
                    '0.947' => '0.175',
                    '0.411' => '0.354',
                    '0.935' => '0.186',
                    '0.379' => '0.361',
                    '0.299' => '0.381',
                    '0.803' => '0.247',
                    '0.079' => '0.500',
                    '0.113' => '0.474',
                    '0.089' => '0.491',
                    '0.511' => '0.329',
                    '0.116' => '0.469',
                    '0.681' => '0.287',
                    '0.600' => '0.308',
                    '0.484' => '0.337',
                    '0.302' => '0.381',
                    '0.556' => '0.318',
                    '0.502' => '0.332',
                    '0.473' => '0.340',
                    '0.328' => '0.373',
                    '0.924' => '0.193',
                    '0.221' => '0.408',
                    '0.386' => '0.360',
                    '0.927' => '0.191',
                    '0.641' => '0.299',
                    '0.982' => '0.131',
                    '0.996' => '0.073',
                    '0.261' => '0.394',
                    '0.921' => '0.194',
                    '0.772' => '0.256',
                    '0.400' => '0.357',
                    '0.315' => '0.375',
                    '0.767' => '0.258',
                    '0.675' => '0.290',
                    '0.375' => '0.363',
                    '0.954' => '0.167',
                    '0.510' => '0.329',
                    '0.651' => '0.297',
                    '0.876' => '0.222',
                    '0.090' => '0.489',
                    '0.740' => '0.268',
                    '0.043' => '0.552',
                    '0.868' => '0.224',
                    '0.723' => '0.274',
                    '0.168' => '0.433',
                    '0.636' => '0.300',
                    '0.820' => '0.243',
                    '0.433' => '0.348',
                    '0.715' => '0.276',
                    '0.193' => '0.421',
                    '0.444' => '0.346',
                    '0.098' => '0.486',
                    '0.403' => '0.357',
                    '0.366' => '0.364',
                    '0.099' => '0.485',
                    '0.838' => '0.236',
                    '0.224' => '0.407',
                    '0.107' => '0.477',
                    '0.226' => '0.407',
                    '0.455' => '0.345',
                    '0.828' => '0.241',
                    '0.398' => '0.357',
                    '0.240' => '0.401',
                    '0.452' => '0.345',
                    '0.535' => '0.324',
                    '0.078' => '0.501',
                    '0.635' => '0.300',
                    '0.593' => '0.309',
                    '0.251' => '0.398',
                    '0.934' => '0.186',
                    '0.465' => '0.343',
                    '0.726' => '0.273',
                    '0.990' => '0.108',
                    '0.943' => '0.178',
                    '0.183' => '0.425',
                    '0.014' => '0.660',
                    '0.816' => '0.244',
                    '0.015' => '0.655',
                    '0.072' => '0.508',
                    '0.903' => '0.207',
                    '0.325' => '0.374',
                    '0.760' => '0.260',
                    '0.734' => '0.269',
                    '0.034' => '0.573',
                    '0.969' => '0.151',
                    '0.629' => '0.302',
                    '0.942' => '0.178',
                    '0.374' => '0.363',
                    '0.490' => '0.334',
                    '0.342' => '0.369',
                    '0.898' => '0.210',
                    '0.066' => '0.515',
                    '0.267' => '0.392',
                    '0.312' => '0.376',
                    '0.235' => '0.404',
                    '0.761' => '0.259',
                    '0.276' => '0.387',
                    '0.871' => '0.223',
                    '0.901' => '0.208',
                    '1.000' => '0.000',
                    '0.610' => '0.307',
                    '0.209' => '0.412',
                    '0.438' => '0.348',
                    '0.176' => '0.429',
                    '0.114' => '0.471',
                    '0.697' => '0.282',
                    '0.563' => '0.317',
                    '0.413' => '0.353',
                    '0.821' => '0.243',
                    '0.432' => '0.349',
                    '0.062' => '0.521',
                    '0.009' => '0.705',
                    '0.387' => '0.360',
                    '0.762' => '0.258',
                    '0.615' => '0.305',
                    '0.485' => '0.336',
                    '0.167' => '0.434',
                    '0.980' => '0.135',
                    '0.642' => '0.299',
                    '0.184' => '0.424',
                    '0.004' => '0.775',
                    '0.910' => '0.203',
                    '0.867' => '0.224',
                    '0.414' => '0.353',
                    '0.824' => '0.243',
                    '0.087' => '0.492',
                    '0.805' => '0.247',
                    '0.595' => '0.308',
                    '0.577' => '0.314',
                    '0.115' => '0.470',
                    '0.768' => '0.258',
                    '0.587' => '0.310',
                    '0.280' => '0.387',
                    '0.426' => '0.351',
                    '0.195' => '0.420',
                    '0.853' => '0.230',
                    '0.660' => '0.294',
                    '0.518' => '0.327',
                    '0.467' => '0.343',
                    '0.303' => '0.379',
                    '0.088' => '0.491',
                    '0.159' => '0.439',
                    '0.001' => '0.891',
                    '0.785' => '0.251',
                    '0.423' => '0.351',
                    '0.203' => '0.415',
                    '0.360' => '0.365',
                    '0.337' => '0.370',
                    '0.289' => '0.384',
                    '0.101' => '0.484',
                    '0.962' => '0.158',
                    '0.155' => '0.441',
                    '0.879' => '0.221',
                    '0.687' => '0.286',
                    '0.649' => '0.298',
                    '0.538' => '0.323',
                    '0.029' => '0.589',
                    '0.808' => '0.247',
                    '0.298' => '0.381',
                    '0.352' => '0.367',
                    '0.953' => '0.168',
                    '0.804' => '0.247',
                    '0.519' => '0.327',
                    '0.404' => '0.357',
                    '0.676' => '0.288',
                    '0.998' => '0.049',
                    '0.448' => '0.346',
                    '0.040' => '0.560',
                    '0.132' => '0.456',
                    '0.652' => '0.297',
                    '0.724' => '0.274',
                    '0.408' => '0.354',
                    '0.936' => '0.185',
                    '0.716' => '0.276',
                    '0.796' => '0.249',
                    '0.827' => '0.241',
                    '0.223' => '0.407',
                    '0.765' => '0.258',
                    '0.441' => '0.347',
                    '0.406' => '0.356',
                    '0.717' => '0.276',
                    '0.565' => '0.316',
                    '0.703' => '0.281',
                    '0.562' => '0.317',
                    '0.944' => '0.177',
                    '0.018' => '0.639',
                    '0.060' => '0.524',
                    '0.320' => '0.375',
                    '0.481' => '0.337',
                    '0.005' => '0.762',
                    '0.733' => '0.269',
                    '0.523' => '0.326',
                    '0.609' => '0.307',
                    '0.985' => '0.123',
                    '0.659' => '0.294',
                    '0.855' => '0.228',
                    '0.727' => '0.272',
                    '0.931' => '0.189',
                    '0.656' => '0.295',
                    '0.475' => '0.340',
                    '0.385' => '0.360',
                    '0.513' => '0.328',
                    '0.291' => '0.384',
                    '0.887' => '0.218',
                    '0.435' => '0.348',
                    '0.321' => '0.375',
                    '0.873' => '0.223',
                    '0.021' => '0.622',
                    '0.339' => '0.369',
                    '0.991' => '0.104',
                    '0.624' => '0.303',
                    '0.598' => '0.308',
                    '0.373' => '0.363',
                    '0.680' => '0.287',
                    '0.047' => '0.546',
                    '0.547' => '0.321',
                    '0.301' => '0.381',
                    '0.884' => '0.219',
                    '0.554' => '0.318',
                    '0.065' => '0.516',
                    '0.618' => '0.304',
                    '0.125' => '0.462',
                    '0.037' => '0.567',
                    '0.341' => '0.369',
                    '0.965' => '0.156',
                    '0.308' => '0.377',
                    '0.454' => '0.345',
                    '0.842' => '0.235',
                    '0.279' => '0.387',
                    '0.233' => '0.405',
                    '0.839' => '0.236',
                    '0.555' => '0.318',
                    '0.140' => '0.449',
                    '0.709' => '0.278',
                    '0.108' => '0.477',
                    '0.574' => '0.314',
                    '0.464' => '0.343',
                    '0.471' => '0.340',
                    '0.382' => '0.361',
                    '0.100' => '0.484',
                    '0.486' => '0.336',
                    '0.752' => '0.266',
                    '0.911' => '0.203',
                    '0.198' => '0.418',
                    '0.614' => '0.306',
                    '0.959' => '0.161',
                    '0.967' => '0.154',
                    '0.219' => '0.409',
                    '0.257' => '0.395',
                    '0.355' => '0.366',
                    '0.759' => '0.260',
                    '0.157' => '0.440',
                    '0.631' => '0.301',
                    '0.468' => '0.343',
                    '0.533' => '0.324',
                    '0.978' => '0.140',
                    '0.860' => '0.227',
                    '0.643' => '0.299',
                    '0.427' => '0.350',
                    '0.793' => '0.249',
                    '0.120' => '0.468',
                    '0.832' => '0.238',
                    '0.900' => '0.208',
                    '0.648' => '0.298',
                    '0.602' => '0.308',
                    '0.242' => '0.401',
                    '0.841' => '0.235',
                    '0.878' => '0.221',
                    '0.671' => '0.291',
                    '0.338' => '0.370',
                    '0.449' => '0.346',
                    '0.690' => '0.284',
                    '0.054' => '0.533',
                    '0.782' => '0.251',
                    '0.288' => '0.385',
                    '0.958' => '0.162',
                    '0.872' => '0.223',
                    '0.788' => '0.251',
                    '0.359' => '0.366',
                    '0.391' => '0.359',
                    '0.730' => '0.270',
                    '0.997' => '0.065',
                    '0.970' => '0.150',
                    '0.754' => '0.263',
                    '0.059' => '0.525',
                    '0.250' => '0.399',
                    '0.686' => '0.286',
                    '0.286' => '0.385',
                    '0.081' => '0.497',
                    '0.988' => '0.114',
                    '0.929' => '0.190',
                    '0.422' => '0.351',
                    '0.070' => '0.509',
                    '0.742' => '0.267',
                    '0.376' => '0.362',
                    '0.819' => '0.244',
                    '0.152' => '0.444',
                    '0.694' => '0.283',
                    '0.789' => '0.250',
                    '0.728' => '0.272',
                    '0.010' => '0.694',
                    '0.807' => '0.247',
                    '0.504' => '0.332',
                    '0.313' => '0.376',
                    '0.627' => '0.303',
                    '0.524' => '0.326',
                    '0.357' => '0.366',
                    '0.945' => '0.176',
                    '0.044' => '0.550',
                    '0.589' => '0.310',
                    '0.512' => '0.328',
                    '0.061' => '0.522',
                    '0.869' => '0.224',
                    '0.350' => '0.367',
                    '0.647' => '0.299',
                    '0.826' => '0.241',
                    '0.718' => '0.276',
                    '0.566' => '0.316',
                    '0.020' => '0.628',
                    '0.739' => '0.268',
                    '0.474' => '0.340',
                    '0.017' => '0.643',
                    '0.766' => '0.258',
                    '0.619' => '0.304',
                    '0.290' => '0.384',
                    '0.537' => '0.324',
                    '0.011' => '0.680',
                    '0.620' => '0.303',
                    '0.006' => '0.739',
                    '0.912' => '0.203',
                    '0.463' => '0.344',
                    '0.344' => '0.369',
                    '0.750' => '0.266',
                    '0.854' => '0.229',
                    '0.561' => '0.317',
                    '0.278' => '0.387',
                    '0.384' => '0.361',
                    '0.658' => '0.294',
                    '0.216' => '0.411',
                    '0.372' => '0.363',
                    '0.166' => '0.435',
                    '0.297' => '0.381',
                    '0.582' => '0.312',
                    '0.493' => '0.333',
                    '0.487' => '0.335',
                    '0.894' => '0.212',
                    '0.368' => '0.364',
                    '0.147' => '0.446',
                    '0.181' => '0.426',
                    '0.950' => '0.172',
                    '0.672' => '0.291',
                    '0.135' => '0.453',
                    '0.905' => '0.206',
                    '0.888' => '0.217',
                    '0.548' => '0.321',
                    '0.105' => '0.480',
                    '0.056' => '0.529',
                    '0.951' => '0.171',
                    '0.093' => '0.487',
                    '0.886' => '0.218',
                    '0.809' => '0.247',
                    '0.158' => '0.439',
                    '0.968' => '0.153',
                    '0.042' => '0.556',
                    '0.335' => '0.370',
                    '0.895' => '0.211',
                    '0.881' => '0.220',
                    '0.601' => '0.308',
                    '0.035' => '0.571',
                    '0.258' => '0.395',
                    '0.012' => '0.673',
                    '0.189' => '0.423',
                    '0.960' => '0.160',
                    '0.885' => '0.218',
                    '0.058' => '0.526',
                    '0.751' => '0.266',
                    '0.896' => '0.211',
                    '0.525' => '0.326',
                    '0.091' => '0.489',
                    '0.704' => '0.280',
                    '0.412' => '0.354',
                    '0.271' => '0.389',
                    '0.741' => '0.267',
                    '0.003' => '0.803',
                    '0.472' => '0.340',
                    '0.390' => '0.359',
                    '0.316' => '0.375',
                    '0.208' => '0.413',
                    '0.469' => '0.343',
                    '0.966' => '0.155',
                    '0.456' => '0.345',
                    '0.845' => '0.233',
                    '0.833' => '0.238',
                    '0.383' => '0.361',
                    '0.637' => '0.300',
                    '0.597' => '0.308',
                    '0.269' => '0.391',
                    '0.590' => '0.309',
                    '0.126' => '0.462',
                    '0.957' => '0.163',
                    '0.922' => '0.194',
                    '0.446' => '0.346',
                    '0.197' => '0.418',
                    '0.787' => '0.251',
                    '0.695' => '0.283',
                    '0.753' => '0.265',
                    '0.780' => '0.252',
                    '0.573' => '0.314',
                    '0.210' => '0.412',
                    '0.987' => '0.115',
                    '0.177' => '0.428',
                    '0.731' => '0.269',
                    '0.019' => '0.634',
                    '0.440' => '0.348',
                    '0.650' => '0.298',
                    '0.679' => '0.287',
                    '0.802' => '0.247',
                    '0.505' => '0.331',
                    '0.127' => '0.460',
                    '0.939' => '0.181',
                    '0.377' => '0.362',
                    '0.260' => '0.395',
                    '0.036' => '0.568',
                    '0.567' => '0.316',
                    '0.327' => '0.374',
                    '0.287' => '0.385',
                    '0.256' => '0.396',
                    '0.993' => '0.094',
                    '0.729' => '0.271',
                    '0.685' => '0.286',
                    '0.255' => '0.397',
                    '0.865' => '0.225',
                    '0.367' => '0.364',
                    '0.849' => '0.232',
                    '0.607' => '0.307',
                    '0.075' => '0.502',
                    '0.480' => '0.338',
                    '0.284' => '0.386',
                    '0.477' => '0.340',
                    '0.732' => '0.269',
                    '0.458' => '0.344',
                    '0.915' => '0.199',
                    '0.578' => '0.313',
                    '0.296' => '0.381',
                    '0.349' => '0.367',
                    '0.495' => '0.333',
                    '0.756' => '0.262',
                    '0.646' => '0.299',
                    '0.332' => '0.372',
                    '0.343' => '0.369',
                    '0.745' => '0.266',
                    '0.693' => '0.284',
                    '0.092' => '0.488',
                    '0.248' => '0.400',
                    '0.811' => '0.247',
                    '0.182' => '0.426',
                    '0.295' => '0.381',
                    '0.421' => '0.351',
                    '0.410' => '0.354',
                    '0.333' => '0.372',
                    '0.542' => '0.322',
                    '0.483' => '0.337',
                    '0.645' => '0.299',
                    '0.639' => '0.300',
                    '0.654' => '0.296',
                    '0.492' => '0.333',
                    '0.692' => '0.284',
                    '0.171' => '0.431',
                    '0.604' => '0.307',
                    '0.572' => '0.315',
                    '0.779' => '0.252',
                    '0.085' => '0.494',
                    '0.389' => '0.359',
                    '0.416' => '0.353',
                    '0.200' => '0.415',
                    '0.575' => '0.314',
                    '0.963' => '0.158',
                    '0.913' => '0.201',
                    '0.136' => '0.453',
                    '0.048' => '0.544',
                    '0.952' => '0.169',
                    '0.875' => '0.223',
                    '0.662' => '0.293',
                    '0.892' => '0.212',
                    '0.154' => '0.442',
                    '0.106' => '0.479',
                    '0.738' => '0.268',
                    '0.165' => '0.435',
                    '0.823' => '0.243',
                    '0.797' => '0.248',
                    '0.370' => '0.364',
                    '0.545' => '0.321',
                    '0.319' => '0.375',
                    '0.443' => '0.347',
                    '0.262' => '0.394',
                    '0.314' => '0.375',
                    '0.428' => '0.350',
                    '0.142' => '0.449',
                    '0.880' => '0.221',
                    '0.324' => '0.374',
                    '0.437' => '0.348',
                    '0.119' => '0.468',
                    '0.417' => '0.352',
                    '0.666' => '0.292',
                    '0.148' => '0.446',
                    '0.141' => '0.449',
                    '0.861' => '0.227',
                    '0.520' => '0.327',
                    '0.234' => '0.405',
                    '0.498' => '0.332',
                    '0.128' => '0.460',
                    '0.568' => '0.316',
                    '0.381' => '0.361',
                    '0.461' => '0.344',
                    '0.170' => '0.432',
                    '0.773' => '0.256',
                    '0.008' => '0.716',
                    '0.479' => '0.338',
                    '0.986' => '0.119',
                    '0.889' => '0.214',
                    '0.457' => '0.344',
                    '0.553' => '0.318',
                    '0.707' => '0.279',
                    '0.273' => '0.388',
                    '0.046' => '0.548',
                    '0.834' => '0.237',
                    '0.531' => '0.325',
                    '0.025' => '0.603',
                    '0.268' => '0.391',
                    '0.976' => '0.143',
                    '0.143' => '0.448',
                    '0.220' => '0.408',
                    '0.244' => '0.400',
                    '0.893' => '0.212',
                    '0.974' => '0.146',
                    '0.801' => '0.247',
                    '0.310' => '0.376',
                    '0.050' => '0.542',
                    '0.030' => '0.584',
                    '0.699' => '0.281',
                    '0.701' => '0.281',
                    '0.283' => '0.386',
                    '0.488' => '0.335',
                    '0.906' => '0.206',
                    '0.857' => '0.228',
                    '0.795' => '0.249',
                    '0.506' => '0.331',
                    '0.462' => '0.344',
                    '0.926' => '0.191',
                    '0.992' => '0.098',
                    '0.678' => '0.288',
                    '0.095' => '0.487',
                    '0.932' => '0.188',
                    '0.178' => '0.428',
                    '0.211' => '0.411',
                    '0.720' => '0.275',
                    '0.395' => '0.358',
                    '0.138' => '0.450',
                    '0.508' => '0.330',
                    '0.371' => '0.364',
                    '0.584' => '0.311',
                    '0.814' => '0.245',
                    '0.285' => '0.385',
                    '0.515' => '0.327',
                    '0.322' => '0.374',
                    '0.755' => '0.262',
                    '0.559' => '0.318',
                    '0.546' => '0.321',
                    '0.429' => '0.350',
                    '0.644' => '0.299',
                    '0.956' => '0.164',
                    '0.856' => '0.228',
                    '0.583' => '0.312',
                    '0.459' => '0.344',
                    '0.118' => '0.468',
                    '0.201' => '0.415',
                    '0.657' => '0.295',
                    '0.405' => '0.356',
                    '0.702' => '0.281',
                    '0.708' => '0.278',
                    '0.661' => '0.294',
                    '0.526' => '0.325',
                    '0.964' => '0.157',
                    '0.848' => '0.232',
                    '0.743' => '0.267',
                    '0.514' => '0.327',
                    '0.329' => '0.373',
                    '0.434' => '0.348',
                    '0.909' => '0.203',
                    '0.551' => '0.319',
                    '0.393' => '0.358',
                    '0.798' => '0.247',
                    '0.039' => '0.562',
                    '0.447' => '0.346',
                    '0.576' => '0.314',
                    '0.024' => '0.608',
                    '0.476' => '0.340',
                    '0.670' => '0.291',
                    '0.420' => '0.352',
                    '0.164' => '0.437',
                    '0.192' => '0.422',
                    '0.831' => '0.239',
                    '0.172' => '0.430',
                    '0.254' => '0.397',
                    '0.973' => '0.147',
                    '0.306' => '0.377',
                    '0.348' => '0.367',
                    '0.920' => '0.194',
                    '0.710' => '0.277',
                    '0.989' => '0.111',
                    '0.084' => '0.495',
                    '0.625' => '0.303',
                    '0.864' => '0.226',
                    '0.539' => '0.323',
                    '0.353' => '0.367',
                    '0.916' => '0.199',
                    '0.812' => '0.247',
                    '0.364' => '0.364',
                    '0.270' => '0.390',
                    '0.334' => '0.370',
                    '0.442' => '0.347',
                    '0.263' => '0.394',
                    '0.983' => '0.127',
                    '0.031' => '0.582',
                    '0.231' => '0.405',
                    '0.249' => '0.400',
                    '0.603' => '0.307',
                    '0.684' => '0.286',
                    '0.914' => '0.199',
                    '0.608' => '0.307',
                    '0.064' => '0.516',
                    '0.038' => '0.564',
                    '0.638' => '0.300',
                    '0.552' => '0.319',
                    '0.229' => '0.406',
                    '0.674' => '0.290',
                    '0.307' => '0.377',
                    '0.080' => '0.497',
                    '0.975' => '0.145',
                    '0.356' => '0.366',
                    '0.630' => '0.302',
                    '0.946' => '0.175',
                    '0.781' => '0.251',
                    '0.744' => '0.266',
                    '0.691' => '0.284',
                    '0.470' => '0.340',
                    '0.613' => '0.306',
                    '0.843' => '0.234',
                    '0.774' => '0.255',
                    '0.392' => '0.359',
                    '0.817' => '0.244',
                    '0.300' => '0.381',
                    '0.129' => '0.458',
                    '0.835' => '0.237',
                    '0.218' => '0.409',
                    '0.592' => '0.309',
                    '0.409' => '0.354',
                    '0.737' => '0.268',
                    '0.571' => '0.315',
                    '0.294' => '0.382',
                    '0.599' => '0.308',
                    '0.746' => '0.266',
                    '0.904' => '0.207',
                    '0.829' => '0.241',
                    '0.794' => '0.249',
                    '0.665' => '0.292',
                    '0.923' => '0.193',
                    '0.151' => '0.444',
                    '0.507' => '0.331',
                    '0.632' => '0.301',
                    '0.499' => '0.332',
                    '0.137' => '0.451',
                    '0.055' => '0.531',
                    '0.207' => '0.413',
                    '0.890' => '0.214',
                    '0.840' => '0.235',
                    '0.541' => '0.322',
                    '0.534' => '0.324',
                    '0.071' => '0.509',
                    '0.243' => '0.400',
                    '0.000' => '1.000',
                    '0.961' => '0.159',
                    '0.677' => '0.288',
                    '0.569' => '0.315',
                    '0.202' => '0.415',
                    '0.397' => '0.358',
                    '0.134' => '0.455',
                    '0.928' => '0.191',
                    '0.264' => '0.394',
                    '0.634' => '0.300',
                    '0.191' => '0.422',
                    '0.972' => '0.148',
                    '0.917' => '0.197',
                    '0.570' => '0.315',
                    '0.305' => '0.378',
                    '0.460' => '0.344',
                    '0.246' => '0.400',
                    '0.238' => '0.403',
                    '0.130' => '0.457',
                    '0.053' => '0.536',
                    '0.144' => '0.448',
                    '0.103' => '0.482',
                    '0.077' => '0.501',
                    '0.402' => '0.357',
                    '0.478' => '0.339',
                    '0.347' => '0.367',
                    '0.813' => '0.245',
                    '0.187' => '0.423',
                    '0.239' => '0.403',
                    '0.758' => '0.260',
                    '0.190' => '0.422',
                    '0.747' => '0.266',
                    '0.206' => '0.414',
                    '0.883' => '0.219',
                    '0.073' => '0.507',
                    '0.764' => '0.258',
                    '0.836' => '0.236',
                    '0.497' => '0.332',
                    '0.232' => '0.405',
                    '0.431' => '0.349',
                    '0.830' => '0.240',
                    '0.272' => '0.388',
                    '0.586' => '0.311',
                    '0.792' => '0.250',
                    '0.363' => '0.365',
                    '0.527' => '0.325',
                    '0.212' => '0.411',
                    '0.082' => '0.496',
                    '0.979' => '0.138',
                    '0.517' => '0.327',
                    '0.930' => '0.190',
                    '0.323' => '0.374',
                    '0.532' => '0.324',
                    '0.769' => '0.258',
                    '0.899' => '0.208',
                    '0.713' => '0.277',
                    '0.439' => '0.348',
                    '0.783' => '0.251',
                    '0.358' => '0.366',
                    '0.721' => '0.275',
                    '0.786' => '0.251',
                    '0.185' => '0.424',
                    '0.069' => '0.511',
                    '0.033' => '0.574',
                    '0.253' => '0.398',
                    '0.117' => '0.468',
                    '0.330' => '0.372',
                    '0.955' => '0.165',
                    '0.558' => '0.318',
                    '0.711' => '0.277',
                    '0.086' => '0.493',
                    '0.293' => '0.382',
                    '0.049' => '0.543',
                    '0.388' => '0.360',
                    '0.027' => '0.598',
                    '0.891' => '0.213',
                    '0.736' => '0.269',
                    '0.653' => '0.297',
                    '0.028' => '0.593',
                    '0.984' => '0.125',
                    '0.775' => '0.255',
                    '0.664' => '0.292',
                    '0.023' => '0.616',
                    '0.778' => '0.252',
                    '0.874' => '0.223',
                    '0.445' => '0.346',
                    '0.543' => '0.322',
                    '0.937' => '0.182',
                    '0.500' => '0.332',
                    '0.628' => '0.302',
                    '0.818' => '0.244',
                    '0.588' => '0.310',
                    '0.579' => '0.313',
                    '0.940' => '0.181',
                    '0.162' => '0.438',
                    '0.494' => '0.333',
                    '0.763' => '0.258',
                    '0.362' => '0.365',
                    '0.847' => '0.232',
                    '0.770' => '0.257',
                    '0.163' => '0.437',
                    '0.683' => '0.287',
                    '0.668' => '0.292',
                    '0.419' => '0.352',
                    '0.799' => '0.247',
                    '0.700' => '0.281',
                    '0.451' => '0.345',
                    '0.156' => '0.441',
                    '0.111' => '0.474',
                    '0.407' => '0.354',
                    '0.199' => '0.418',
                    '0.994' => '0.091',
                    '0.673' => '0.290',
                    '0.949' => '0.172',
                    '0.540' => '0.323',
                    '0.281' => '0.387',
                    '0.304' => '0.379',
                    '0.139' => '0.450',
                    '0.096' => '0.487',
                    '0.617' => '0.304',
                    '0.623' => '0.303',
                    '0.580' => '0.313',
                    '0.369' => '0.364',
                    '0.669' => '0.291',
                    '0.057' => '0.528',
                    '0.852' => '0.231',
                    '0.863' => '0.227',
                    '0.180' => '0.428',
                    '0.063' => '0.517',
                    '0.041' => '0.558',
                    '0.351' => '0.367',
                    '0.133' => '0.456',
                    '0.109' => '0.476',
                    '0.907' => '0.205',
                    '0.354' => '0.367',
                    '0.591' => '0.309',
                    '0.612' => '0.307',
                    '0.430' => '0.349',
                    '0.173' => '0.430',
                    '0.052' => '0.539',
                    '0.237' => '0.403',
                    '0.032' => '0.580',
                    '0.882' => '0.219',
                    '0.205' => '0.414',
                    '0.131' => '0.457',
                    '0.557' => '0.318',
                    '0.188' => '0.423',
                    '0.145' => '0.447',
                    '0.705' => '0.280',
                    '0.247' => '0.400',
                    '0.450' => '0.345',
                    '0.489' => '0.335',
                    '0.074' => '0.504',
                    '0.378' => '0.362',
                    '0.846' => '0.232',
                    '0.748' => '0.266',
                    '0.102' => '0.483',
                    '0.757' => '0.261',
                    '0.215' => '0.411',
                    '0.971' => '0.149',
                    '0.784' => '0.251',
                    '0.640' => '0.300',
                    '0.401' => '0.357',
                    '0.230' => '0.406',
                    '0.585' => '0.311',
                    '0.002' => '0.838',
                    '0.275' => '0.388',
                    '0.866' => '0.225',
                    '0.822' => '0.243',
                    '0.918' => '0.197',
                    '0.851' => '0.231',
                    '0.415' => '0.353',
                    '0.097' => '0.486',
                    '0.981' => '0.133',
                    '0.122' => '0.467',
                    '0.633' => '0.301',
                    '0.083' => '0.495',
                    '0.688' => '0.285',
                    '0.605' => '0.307',
                    '0.160' => '0.438',
                    '0.776' => '0.254',
                    '0.068' => '0.512',
                    '0.689' => '0.284',
                    '0.331' => '0.372',
                    '0.245' => '0.400',
                    '0.153' => '0.443',
                    '0.076' => '0.502',
                    '0.380' => '0.361',
                    '0.714' => '0.277',
                    '0.606' => '0.307',
                    '0.161' => '0.438',
                    '0.791' => '0.250',
                    '0.259' => '0.395',
                    '0.425' => '0.351',
                    '0.859' => '0.227',
                    '0.528' => '0.325',
                    '0.346' => '0.368',
                    '0.213' => '0.411',
                    '0.682' => '0.287',
                    '0.418' => '0.352',
                    '0.112' => '0.474',
                    '0.094' => '0.487',
                    '0.516' => '0.327',
                    '0.941' => '0.179',
                    '0.124' => '0.464',
                    '0.777' => '0.254',
                    '0.396' => '0.358',
                    '0.806' => '0.247',
                    '0.501' => '0.332',
                    '0.544' => '0.321',
                    '0.436' => '0.348',
                    '0.616' => '0.304',
                    '0.186' => '0.423',
                    '0.466' => '0.343',
                    '0.104' => '0.481',
                    '0.858' => '0.228',
                    '0.564' => '0.317',
                    '0.222' => '0.408',
                    '0.026' => '0.600',
                    '0.317' => '0.375',
                    '0.850' => '0.231',
                    '0.712' => '0.277',
                    '0.292' => '0.384',
                    '0.995' => '0.080',
                    '0.938' => '0.182',
                    '0.530' => '0.325',
                    '0.862' => '0.227',
                    '0.007' => '0.729',
                    '0.771' => '0.257',
                    '0.482' => '0.337',
                    '0.550' => '0.320',
                    '0.016' => '0.649',
                    '0.655' => '0.296',
                    '0.051' => '0.540',
                    '0.722' => '0.274',
                    '0.521' => '0.326',
                    '0.815' => '0.245',
                    '0.194' => '0.421',
                    '0.282' => '0.387',
                    '0.549' => '0.320',
                    '0.169' => '0.433',
                    '0.110' => '0.475',
                    '0.877' => '0.221',
                    '0.227' => '0.407',
                    '0.667' => '0.292',
                    '0.121' => '0.467',
                    '0.326' => '0.374',
                    '0.948' => '0.175',
                    '0.925' => '0.192',
                    '0.719' => '0.275',
                    '0.265' => '0.394',
                    '0.509' => '0.330',
                    '0.394' => '0.358',
                    '0.340' => '0.369',
                    '0.622' => '0.303',
                    '0.908' => '0.204',
                    '0.663' => '0.292',
                    '0.902' => '0.208'
                  },
          'tp' => {
                    '0.214' => '0.804',
                    '0.621' => '0.708',
                    '0.529' => '0.731',
                    '0.503' => '0.737',
                    '0.174' => '0.817',
                    '0.045' => '0.881',
                    '0.399' => '0.762',
                    '0.626' => '0.708',
                    '0.596' => '0.714',
                    '0.453' => '0.747',
                    '0.013' => '0.926',
                    '0.311' => '0.779',
                    '0.933' => '0.583',
                    '0.196' => '0.811',
                    '0.175' => '0.817',
                    '0.611' => '0.710',
                    '0.318' => '0.777',
                    '0.277' => '0.786',
                    '0.149' => '0.824',
                    '0.594' => '0.715',
                    '0.225' => '0.802',
                    '0.870' => '0.630',
                    '0.536' => '0.729',
                    '0.217' => '0.804',
                    '0.810' => '0.656',
                    '0.067' => '0.866',
                    '0.204' => '0.808',
                    '0.749' => '0.677',
                    '0.919' => '0.595',
                    '0.698' => '0.690',
                    '0.123' => '0.836',
                    '0.706' => '0.688',
                    '0.266' => '0.790',
                    '0.825' => '0.650',
                    '0.252' => '0.795',
                    '0.336' => '0.774',
                    '0.735' => '0.681',
                    '0.228' => '0.801',
                    '0.897' => '0.612',
                    '0.150' => '0.824',
                    '0.977' => '0.502',
                    '0.022' => '0.911',
                    '0.236' => '0.800',
                    '0.837' => '0.644',
                    '0.999' => '0.127',
                    '0.365' => '0.769',
                    '0.491' => '0.739',
                    '0.581' => '0.718',
                    '0.146' => '0.826',
                    '0.361' => '0.770',
                    '0.345' => '0.773',
                    '0.790' => '0.663',
                    '0.241' => '0.798',
                    '0.274' => '0.786',
                    '0.844' => '0.642',
                    '0.522' => '0.733',
                    '0.800' => '0.660',
                    '0.725' => '0.684',
                    '0.560' => '0.724',
                    '0.309' => '0.779',
                    '0.496' => '0.738',
                    '0.424' => '0.756',
                    '0.696' => '0.690',
                    '0.179' => '0.816',
                    '0.947' => '0.567',
                    '0.411' => '0.759',
                    '0.935' => '0.581',
                    '0.379' => '0.765',
                    '0.299' => '0.781',
                    '0.803' => '0.658',
                    '0.079' => '0.859',
                    '0.113' => '0.843',
                    '0.089' => '0.853',
                    '0.511' => '0.735',
                    '0.116' => '0.841',
                    '0.681' => '0.694',
                    '0.600' => '0.714',
                    '0.484' => '0.741',
                    '0.302' => '0.781',
                    '0.556' => '0.725',
                    '0.502' => '0.737',
                    '0.473' => '0.743',
                    '0.328' => '0.776',
                    '0.924' => '0.591',
                    '0.221' => '0.802',
                    '0.386' => '0.763',
                    '0.927' => '0.589',
                    '0.641' => '0.705',
                    '0.982' => '0.482',
                    '0.996' => '0.338',
                    '0.261' => '0.792',
                    '0.921' => '0.593',
                    '0.772' => '0.668',
                    '0.400' => '0.762',
                    '0.315' => '0.778',
                    '0.767' => '0.671',
                    '0.675' => '0.696',
                    '0.375' => '0.767',
                    '0.954' => '0.556',
                    '0.510' => '0.735',
                    '0.651' => '0.703',
                    '0.876' => '0.627',
                    '0.090' => '0.853',
                    '0.740' => '0.680',
                    '0.043' => '0.883',
                    '0.868' => '0.631',
                    '0.723' => '0.684',
                    '0.168' => '0.819',
                    '0.636' => '0.706',
                    '0.820' => '0.652',
                    '0.433' => '0.753',
                    '0.715' => '0.685',
                    '0.193' => '0.812',
                    '0.444' => '0.749',
                    '0.098' => '0.849',
                    '0.403' => '0.761',
                    '0.366' => '0.768',
                    '0.099' => '0.848',
                    '0.838' => '0.644',
                    '0.224' => '0.802',
                    '0.107' => '0.845',
                    '0.226' => '0.802',
                    '0.455' => '0.747',
                    '0.828' => '0.648',
                    '0.398' => '0.762',
                    '0.240' => '0.799',
                    '0.452' => '0.747',
                    '0.535' => '0.730',
                    '0.078' => '0.859',
                    '0.635' => '0.706',
                    '0.593' => '0.716',
                    '0.251' => '0.795',
                    '0.934' => '0.581',
                    '0.465' => '0.744',
                    '0.726' => '0.683',
                    '0.990' => '0.435',
                    '0.943' => '0.572',
                    '0.183' => '0.815',
                    '0.014' => '0.924',
                    '0.816' => '0.652',
                    '0.015' => '0.922',
                    '0.072' => '0.863',
                    '0.903' => '0.608',
                    '0.325' => '0.776',
                    '0.760' => '0.673',
                    '0.734' => '0.682',
                    '0.034' => '0.893',
                    '0.969' => '0.526',
                    '0.629' => '0.706',
                    '0.942' => '0.573',
                    '0.374' => '0.767',
                    '0.490' => '0.740',
                    '0.342' => '0.773',
                    '0.898' => '0.612',
                    '0.066' => '0.866',
                    '0.267' => '0.790',
                    '0.312' => '0.779',
                    '0.235' => '0.800',
                    '0.761' => '0.673',
                    '0.276' => '0.786',
                    '0.871' => '0.630',
                    '0.901' => '0.610',
                    '1.000' => '0.000',
                    '0.610' => '0.711',
                    '0.209' => '0.806',
                    '0.438' => '0.751',
                    '0.176' => '0.816',
                    '0.114' => '0.842',
                    '0.697' => '0.690',
                    '0.563' => '0.723',
                    '0.413' => '0.759',
                    '0.821' => '0.651',
                    '0.432' => '0.753',
                    '0.062' => '0.869',
                    '0.009' => '0.937',
                    '0.387' => '0.763',
                    '0.762' => '0.672',
                    '0.615' => '0.709',
                    '0.485' => '0.740',
                    '0.167' => '0.819',
                    '0.980' => '0.489',
                    '0.642' => '0.705',
                    '0.184' => '0.814',
                    '0.004' => '0.957',
                    '0.910' => '0.603',
                    '0.867' => '0.632',
                    '0.414' => '0.758',
                    '0.824' => '0.650',
                    '0.087' => '0.854',
                    '0.805' => '0.657',
                    '0.595' => '0.715',
                    '0.577' => '0.719',
                    '0.115' => '0.841',
                    '0.768' => '0.670',
                    '0.587' => '0.717',
                    '0.280' => '0.785',
                    '0.426' => '0.755',
                    '0.195' => '0.811',
                    '0.853' => '0.638',
                    '0.660' => '0.701',
                    '0.518' => '0.734',
                    '0.467' => '0.744',
                    '0.303' => '0.780',
                    '0.088' => '0.853',
                    '0.159' => '0.821',
                    '0.001' => '0.981',
                    '0.785' => '0.665',
                    '0.423' => '0.756',
                    '0.203' => '0.808',
                    '0.360' => '0.770',
                    '0.337' => '0.774',
                    '0.289' => '0.783',
                    '0.101' => '0.848',
                    '0.962' => '0.541',
                    '0.155' => '0.823',
                    '0.879' => '0.625',
                    '0.687' => '0.693',
                    '0.649' => '0.703',
                    '0.538' => '0.729',
                    '0.029' => '0.901',
                    '0.808' => '0.656',
                    '0.298' => '0.781',
                    '0.352' => '0.771',
                    '0.953' => '0.558',
                    '0.804' => '0.657',
                    '0.519' => '0.734',
                    '0.404' => '0.761',
                    '0.676' => '0.695',
                    '0.998' => '0.251',
                    '0.448' => '0.748',
                    '0.040' => '0.887',
                    '0.132' => '0.832',
                    '0.652' => '0.703',
                    '0.724' => '0.684',
                    '0.408' => '0.759',
                    '0.936' => '0.579',
                    '0.716' => '0.685',
                    '0.796' => '0.661',
                    '0.827' => '0.649',
                    '0.223' => '0.802',
                    '0.765' => '0.672',
                    '0.441' => '0.750',
                    '0.406' => '0.760',
                    '0.717' => '0.685',
                    '0.565' => '0.722',
                    '0.703' => '0.689',
                    '0.562' => '0.723',
                    '0.944' => '0.571',
                    '0.018' => '0.918',
                    '0.060' => '0.871',
                    '0.320' => '0.777',
                    '0.481' => '0.741',
                    '0.005' => '0.952',
                    '0.733' => '0.682',
                    '0.523' => '0.732',
                    '0.609' => '0.712',
                    '0.985' => '0.468',
                    '0.659' => '0.701',
                    '0.855' => '0.637',
                    '0.727' => '0.683',
                    '0.931' => '0.584',
                    '0.656' => '0.701',
                    '0.475' => '0.743',
                    '0.385' => '0.764',
                    '0.513' => '0.734',
                    '0.291' => '0.783',
                    '0.887' => '0.620',
                    '0.435' => '0.753',
                    '0.321' => '0.777',
                    '0.873' => '0.629',
                    '0.021' => '0.912',
                    '0.339' => '0.773',
                    '0.991' => '0.425',
                    '0.624' => '0.708',
                    '0.598' => '0.714',
                    '0.373' => '0.767',
                    '0.680' => '0.695',
                    '0.047' => '0.880',
                    '0.547' => '0.727',
                    '0.301' => '0.781',
                    '0.884' => '0.621',
                    '0.554' => '0.726',
                    '0.065' => '0.867',
                    '0.618' => '0.709',
                    '0.125' => '0.835',
                    '0.037' => '0.890',
                    '0.341' => '0.773',
                    '0.965' => '0.534',
                    '0.308' => '0.779',
                    '0.454' => '0.747',
                    '0.842' => '0.643',
                    '0.279' => '0.785',
                    '0.233' => '0.800',
                    '0.839' => '0.644',
                    '0.555' => '0.726',
                    '0.140' => '0.829',
                    '0.709' => '0.687',
                    '0.108' => '0.845',
                    '0.574' => '0.720',
                    '0.464' => '0.744',
                    '0.471' => '0.743',
                    '0.382' => '0.764',
                    '0.100' => '0.848',
                    '0.486' => '0.740',
                    '0.752' => '0.676',
                    '0.911' => '0.602',
                    '0.198' => '0.810',
                    '0.614' => '0.709',
                    '0.959' => '0.546',
                    '0.967' => '0.532',
                    '0.219' => '0.803',
                    '0.257' => '0.793',
                    '0.355' => '0.771',
                    '0.759' => '0.673',
                    '0.157' => '0.822',
                    '0.631' => '0.706',
                    '0.468' => '0.744',
                    '0.533' => '0.730',
                    '0.978' => '0.498',
                    '0.860' => '0.635',
                    '0.643' => '0.704',
                    '0.427' => '0.754',
                    '0.793' => '0.663',
                    '0.120' => '0.839',
                    '0.832' => '0.647',
                    '0.900' => '0.610',
                    '0.648' => '0.703',
                    '0.602' => '0.713',
                    '0.242' => '0.798',
                    '0.841' => '0.643',
                    '0.878' => '0.625',
                    '0.671' => '0.698',
                    '0.338' => '0.774',
                    '0.449' => '0.747',
                    '0.690' => '0.693',
                    '0.054' => '0.875',
                    '0.782' => '0.666',
                    '0.288' => '0.783',
                    '0.958' => '0.548',
                    '0.872' => '0.629',
                    '0.788' => '0.664',
                    '0.359' => '0.770',
                    '0.391' => '0.763',
                    '0.730' => '0.682',
                    '0.997' => '0.305',
                    '0.970' => '0.524',
                    '0.754' => '0.675',
                    '0.059' => '0.872',
                    '0.250' => '0.796',
                    '0.686' => '0.694',
                    '0.286' => '0.783',
                    '0.081' => '0.857',
                    '0.988' => '0.448',
                    '0.929' => '0.587',
                    '0.422' => '0.756',
                    '0.070' => '0.864',
                    '0.742' => '0.680',
                    '0.376' => '0.766',
                    '0.819' => '0.652',
                    '0.152' => '0.824',
                    '0.694' => '0.692',
                    '0.789' => '0.664',
                    '0.728' => '0.683',
                    '0.010' => '0.934',
                    '0.807' => '0.657',
                    '0.504' => '0.736',
                    '0.313' => '0.778',
                    '0.627' => '0.708',
                    '0.524' => '0.732',
                    '0.357' => '0.771',
                    '0.945' => '0.570',
                    '0.044' => '0.883',
                    '0.589' => '0.717',
                    '0.512' => '0.734',
                    '0.061' => '0.870',
                    '0.869' => '0.631',
                    '0.350' => '0.772',
                    '0.647' => '0.704',
                    '0.826' => '0.649',
                    '0.718' => '0.685',
                    '0.566' => '0.722',
                    '0.020' => '0.915',
                    '0.739' => '0.681',
                    '0.474' => '0.743',
                    '0.017' => '0.919',
                    '0.766' => '0.671',
                    '0.619' => '0.709',
                    '0.290' => '0.783',
                    '0.537' => '0.729',
                    '0.011' => '0.931',
                    '0.620' => '0.708',
                    '0.006' => '0.949',
                    '0.912' => '0.601',
                    '0.463' => '0.745',
                    '0.344' => '0.773',
                    '0.750' => '0.677',
                    '0.854' => '0.638',
                    '0.561' => '0.724',
                    '0.278' => '0.785',
                    '0.384' => '0.764',
                    '0.658' => '0.701',
                    '0.216' => '0.804',
                    '0.372' => '0.768',
                    '0.166' => '0.820',
                    '0.297' => '0.782',
                    '0.582' => '0.718',
                    '0.493' => '0.739',
                    '0.487' => '0.740',
                    '0.894' => '0.614',
                    '0.368' => '0.768',
                    '0.147' => '0.825',
                    '0.181' => '0.816',
                    '0.950' => '0.562',
                    '0.672' => '0.698',
                    '0.135' => '0.831',
                    '0.905' => '0.607',
                    '0.888' => '0.620',
                    '0.548' => '0.727',
                    '0.105' => '0.846',
                    '0.056' => '0.874',
                    '0.951' => '0.561',
                    '0.093' => '0.852',
                    '0.886' => '0.620',
                    '0.809' => '0.656',
                    '0.158' => '0.822',
                    '0.968' => '0.529',
                    '0.042' => '0.885',
                    '0.335' => '0.774',
                    '0.895' => '0.613',
                    '0.881' => '0.623',
                    '0.601' => '0.714',
                    '0.035' => '0.891',
                    '0.258' => '0.792',
                    '0.012' => '0.928',
                    '0.189' => '0.813',
                    '0.960' => '0.545',
                    '0.885' => '0.621',
                    '0.058' => '0.872',
                    '0.751' => '0.677',
                    '0.896' => '0.613',
                    '0.525' => '0.732',
                    '0.091' => '0.852',
                    '0.704' => '0.689',
                    '0.412' => '0.759',
                    '0.271' => '0.787',
                    '0.741' => '0.680',
                    '0.003' => '0.962',
                    '0.472' => '0.743',
                    '0.390' => '0.763',
                    '0.316' => '0.778',
                    '0.208' => '0.807',
                    '0.469' => '0.744',
                    '0.966' => '0.532',
                    '0.456' => '0.747',
                    '0.845' => '0.642',
                    '0.833' => '0.646',
                    '0.383' => '0.764',
                    '0.637' => '0.705',
                    '0.597' => '0.714',
                    '0.269' => '0.788',
                    '0.590' => '0.716',
                    '0.126' => '0.835',
                    '0.957' => '0.550',
                    '0.922' => '0.593',
                    '0.446' => '0.748',
                    '0.197' => '0.811',
                    '0.787' => '0.664',
                    '0.695' => '0.691',
                    '0.753' => '0.676',
                    '0.780' => '0.667',
                    '0.573' => '0.720',
                    '0.210' => '0.806',
                    '0.987' => '0.453',
                    '0.177' => '0.816',
                    '0.731' => '0.682',
                    '0.019' => '0.916',
                    '0.440' => '0.750',
                    '0.650' => '0.703',
                    '0.679' => '0.695',
                    '0.802' => '0.658',
                    '0.505' => '0.736',
                    '0.127' => '0.834',
                    '0.939' => '0.576',
                    '0.377' => '0.766',
                    '0.260' => '0.792',
                    '0.036' => '0.890',
                    '0.567' => '0.721',
                    '0.327' => '0.776',
                    '0.287' => '0.783',
                    '0.256' => '0.793',
                    '0.993' => '0.399',
                    '0.729' => '0.683',
                    '0.685' => '0.694',
                    '0.255' => '0.794',
                    '0.865' => '0.633',
                    '0.367' => '0.768',
                    '0.849' => '0.640',
                    '0.607' => '0.712',
                    '0.075' => '0.861',
                    '0.480' => '0.742',
                    '0.284' => '0.784',
                    '0.477' => '0.742',
                    '0.732' => '0.682',
                    '0.458' => '0.746',
                    '0.915' => '0.598',
                    '0.578' => '0.719',
                    '0.296' => '0.782',
                    '0.349' => '0.772',
                    '0.495' => '0.738',
                    '0.756' => '0.675',
                    '0.646' => '0.704',
                    '0.332' => '0.775',
                    '0.343' => '0.773',
                    '0.745' => '0.678',
                    '0.693' => '0.692',
                    '0.092' => '0.852',
                    '0.248' => '0.797',
                    '0.811' => '0.656',
                    '0.182' => '0.816',
                    '0.295' => '0.782',
                    '0.421' => '0.756',
                    '0.410' => '0.759',
                    '0.333' => '0.774',
                    '0.542' => '0.728',
                    '0.483' => '0.741',
                    '0.645' => '0.704',
                    '0.639' => '0.705',
                    '0.654' => '0.702',
                    '0.492' => '0.739',
                    '0.692' => '0.692',
                    '0.171' => '0.818',
                    '0.604' => '0.713',
                    '0.572' => '0.720',
                    '0.779' => '0.667',
                    '0.085' => '0.855',
                    '0.389' => '0.763',
                    '0.416' => '0.758',
                    '0.200' => '0.809',
                    '0.575' => '0.719',
                    '0.963' => '0.539',
                    '0.913' => '0.600',
                    '0.136' => '0.831',
                    '0.048' => '0.879',
                    '0.952' => '0.560',
                    '0.875' => '0.628',
                    '0.662' => '0.700',
                    '0.892' => '0.615',
                    '0.154' => '0.823',
                    '0.106' => '0.846',
                    '0.738' => '0.681',
                    '0.165' => '0.820',
                    '0.823' => '0.651',
                    '0.797' => '0.661',
                    '0.370' => '0.768',
                    '0.545' => '0.728',
                    '0.319' => '0.777',
                    '0.443' => '0.749',
                    '0.262' => '0.791',
                    '0.314' => '0.778',
                    '0.428' => '0.754',
                    '0.142' => '0.828',
                    '0.880' => '0.624',
                    '0.324' => '0.777',
                    '0.437' => '0.752',
                    '0.119' => '0.840',
                    '0.417' => '0.757',
                    '0.666' => '0.699',
                    '0.148' => '0.825',
                    '0.141' => '0.828',
                    '0.861' => '0.635',
                    '0.520' => '0.733',
                    '0.234' => '0.800',
                    '0.498' => '0.738',
                    '0.128' => '0.834',
                    '0.568' => '0.721',
                    '0.381' => '0.765',
                    '0.461' => '0.745',
                    '0.170' => '0.818',
                    '0.773' => '0.668',
                    '0.008' => '0.941',
                    '0.479' => '0.742',
                    '0.986' => '0.461',
                    '0.889' => '0.617',
                    '0.457' => '0.747',
                    '0.553' => '0.726',
                    '0.707' => '0.688',
                    '0.273' => '0.786',
                    '0.046' => '0.881',
                    '0.834' => '0.646',
                    '0.531' => '0.730',
                    '0.025' => '0.907',
                    '0.268' => '0.788',
                    '0.976' => '0.505',
                    '0.143' => '0.827',
                    '0.220' => '0.803',
                    '0.244' => '0.798',
                    '0.893' => '0.615',
                    '0.974' => '0.510',
                    '0.801' => '0.659',
                    '0.310' => '0.779',
                    '0.050' => '0.878',
                    '0.030' => '0.899',
                    '0.699' => '0.690',
                    '0.701' => '0.689',
                    '0.283' => '0.784',
                    '0.488' => '0.740',
                    '0.906' => '0.606',
                    '0.857' => '0.636',
                    '0.795' => '0.662',
                    '0.506' => '0.736',
                    '0.462' => '0.745',
                    '0.926' => '0.589',
                    '0.992' => '0.412',
                    '0.678' => '0.695',
                    '0.095' => '0.850',
                    '0.932' => '0.584',
                    '0.178' => '0.816',
                    '0.211' => '0.805',
                    '0.720' => '0.684',
                    '0.395' => '0.762',
                    '0.138' => '0.830',
                    '0.508' => '0.735',
                    '0.371' => '0.768',
                    '0.584' => '0.718',
                    '0.814' => '0.654',
                    '0.285' => '0.784',
                    '0.515' => '0.734',
                    '0.322' => '0.777',
                    '0.755' => '0.675',
                    '0.559' => '0.725',
                    '0.546' => '0.727',
                    '0.429' => '0.754',
                    '0.644' => '0.704',
                    '0.956' => '0.552',
                    '0.856' => '0.637',
                    '0.583' => '0.718',
                    '0.459' => '0.746',
                    '0.118' => '0.840',
                    '0.201' => '0.809',
                    '0.657' => '0.701',
                    '0.405' => '0.760',
                    '0.702' => '0.689',
                    '0.708' => '0.687',
                    '0.661' => '0.700',
                    '0.526' => '0.731',
                    '0.964' => '0.536',
                    '0.848' => '0.641',
                    '0.743' => '0.679',
                    '0.514' => '0.734',
                    '0.329' => '0.776',
                    '0.434' => '0.753',
                    '0.909' => '0.604',
                    '0.551' => '0.726',
                    '0.393' => '0.763',
                    '0.798' => '0.661',
                    '0.039' => '0.888',
                    '0.447' => '0.748',
                    '0.576' => '0.719',
                    '0.024' => '0.908',
                    '0.476' => '0.742',
                    '0.670' => '0.698',
                    '0.420' => '0.757',
                    '0.164' => '0.820',
                    '0.192' => '0.812',
                    '0.831' => '0.647',
                    '0.172' => '0.817',
                    '0.254' => '0.794',
                    '0.973' => '0.514',
                    '0.306' => '0.780',
                    '0.348' => '0.772',
                    '0.920' => '0.594',
                    '0.710' => '0.687',
                    '0.989' => '0.441',
                    '0.084' => '0.856',
                    '0.625' => '0.708',
                    '0.864' => '0.633',
                    '0.539' => '0.728',
                    '0.353' => '0.771',
                    '0.916' => '0.598',
                    '0.812' => '0.655',
                    '0.364' => '0.769',
                    '0.270' => '0.787',
                    '0.334' => '0.774',
                    '0.442' => '0.750',
                    '0.263' => '0.791',
                    '0.983' => '0.477',
                    '0.031' => '0.898',
                    '0.231' => '0.801',
                    '0.249' => '0.796',
                    '0.603' => '0.713',
                    '0.684' => '0.694',
                    '0.914' => '0.599',
                    '0.608' => '0.712',
                    '0.064' => '0.868',
                    '0.038' => '0.889',
                    '0.638' => '0.705',
                    '0.552' => '0.726',
                    '0.229' => '0.801',
                    '0.674' => '0.697',
                    '0.307' => '0.780',
                    '0.080' => '0.858',
                    '0.975' => '0.507',
                    '0.356' => '0.771',
                    '0.630' => '0.706',
                    '0.946' => '0.568',
                    '0.781' => '0.667',
                    '0.744' => '0.679',
                    '0.691' => '0.693',
                    '0.470' => '0.743',
                    '0.613' => '0.710',
                    '0.843' => '0.642',
                    '0.774' => '0.668',
                    '0.392' => '0.763',
                    '0.817' => '0.652',
                    '0.300' => '0.781',
                    '0.129' => '0.833',
                    '0.835' => '0.645',
                    '0.218' => '0.804',
                    '0.592' => '0.716',
                    '0.409' => '0.759',
                    '0.737' => '0.681',
                    '0.571' => '0.721',
                    '0.294' => '0.782',
                    '0.599' => '0.714',
                    '0.746' => '0.678',
                    '0.904' => '0.608',
                    '0.829' => '0.648',
                    '0.794' => '0.662',
                    '0.665' => '0.699',
                    '0.923' => '0.592',
                    '0.151' => '0.824',
                    '0.507' => '0.736',
                    '0.632' => '0.706',
                    '0.499' => '0.738',
                    '0.137' => '0.830',
                    '0.055' => '0.875',
                    '0.207' => '0.807',
                    '0.890' => '0.617',
                    '0.840' => '0.643',
                    '0.541' => '0.728',
                    '0.534' => '0.730',
                    '0.071' => '0.864',
                    '0.243' => '0.798',
                    '0.000' => '1.000',
                    '0.961' => '0.542',
                    '0.677' => '0.695',
                    '0.569' => '0.721',
                    '0.202' => '0.809',
                    '0.397' => '0.762',
                    '0.134' => '0.832',
                    '0.928' => '0.587',
                    '0.264' => '0.791',
                    '0.634' => '0.706',
                    '0.191' => '0.813',
                    '0.972' => '0.516',
                    '0.917' => '0.597',
                    '0.570' => '0.721',
                    '0.305' => '0.780',
                    '0.460' => '0.746',
                    '0.246' => '0.797',
                    '0.238' => '0.800',
                    '0.130' => '0.833',
                    '0.053' => '0.876',
                    '0.144' => '0.826',
                    '0.103' => '0.847',
                    '0.077' => '0.860',
                    '0.402' => '0.761',
                    '0.478' => '0.742',
                    '0.347' => '0.772',
                    '0.813' => '0.654',
                    '0.187' => '0.814',
                    '0.239' => '0.799',
                    '0.758' => '0.674',
                    '0.190' => '0.813',
                    '0.747' => '0.677',
                    '0.206' => '0.807',
                    '0.883' => '0.622',
                    '0.073' => '0.863',
                    '0.764' => '0.672',
                    '0.836' => '0.644',
                    '0.497' => '0.738',
                    '0.232' => '0.801',
                    '0.431' => '0.754',
                    '0.830' => '0.648',
                    '0.272' => '0.786',
                    '0.586' => '0.718',
                    '0.792' => '0.663',
                    '0.363' => '0.769',
                    '0.527' => '0.731',
                    '0.212' => '0.805',
                    '0.082' => '0.857',
                    '0.979' => '0.496',
                    '0.517' => '0.734',
                    '0.930' => '0.586',
                    '0.323' => '0.777',
                    '0.532' => '0.730',
                    '0.769' => '0.670',
                    '0.899' => '0.611',
                    '0.713' => '0.686',
                    '0.439' => '0.750',
                    '0.783' => '0.666',
                    '0.358' => '0.770',
                    '0.721' => '0.684',
                    '0.786' => '0.664',
                    '0.185' => '0.814',
                    '0.069' => '0.864',
                    '0.033' => '0.894',
                    '0.253' => '0.794',
                    '0.117' => '0.840',
                    '0.330' => '0.775',
                    '0.955' => '0.554',
                    '0.558' => '0.725',
                    '0.711' => '0.687',
                    '0.086' => '0.854',
                    '0.293' => '0.783',
                    '0.049' => '0.879',
                    '0.388' => '0.763',
                    '0.027' => '0.903',
                    '0.891' => '0.616',
                    '0.736' => '0.681',
                    '0.653' => '0.703',
                    '0.028' => '0.902',
                    '0.984' => '0.472',
                    '0.775' => '0.668',
                    '0.664' => '0.700',
                    '0.023' => '0.910',
                    '0.778' => '0.667',
                    '0.874' => '0.628',
                    '0.445' => '0.748',
                    '0.543' => '0.728',
                    '0.937' => '0.578',
                    '0.500' => '0.737',
                    '0.628' => '0.707',
                    '0.818' => '0.652',
                    '0.588' => '0.717',
                    '0.579' => '0.719',
                    '0.940' => '0.575',
                    '0.162' => '0.821',
                    '0.494' => '0.738',
                    '0.763' => '0.672',
                    '0.362' => '0.770',
                    '0.847' => '0.641',
                    '0.770' => '0.669',
                    '0.163' => '0.821',
                    '0.683' => '0.694',
                    '0.668' => '0.699',
                    '0.419' => '0.757',
                    '0.799' => '0.660',
                    '0.700' => '0.690',
                    '0.451' => '0.747',
                    '0.156' => '0.823',
                    '0.111' => '0.843',
                    '0.407' => '0.759',
                    '0.199' => '0.810',
                    '0.994' => '0.386',
                    '0.673' => '0.697',
                    '0.949' => '0.563',
                    '0.540' => '0.728',
                    '0.281' => '0.785',
                    '0.304' => '0.780',
                    '0.139' => '0.829',
                    '0.096' => '0.850',
                    '0.617' => '0.709',
                    '0.623' => '0.708',
                    '0.580' => '0.719',
                    '0.369' => '0.768',
                    '0.669' => '0.699',
                    '0.057' => '0.873',
                    '0.852' => '0.639',
                    '0.863' => '0.633',
                    '0.180' => '0.816',
                    '0.063' => '0.868',
                    '0.041' => '0.886',
                    '0.351' => '0.772',
                    '0.133' => '0.832',
                    '0.109' => '0.844',
                    '0.907' => '0.604',
                    '0.354' => '0.771',
                    '0.591' => '0.716',
                    '0.612' => '0.710',
                    '0.430' => '0.754',
                    '0.173' => '0.817',
                    '0.052' => '0.877',
                    '0.237' => '0.800',
                    '0.032' => '0.896',
                    '0.882' => '0.623',
                    '0.205' => '0.807',
                    '0.131' => '0.833',
                    '0.557' => '0.725',
                    '0.188' => '0.813',
                    '0.145' => '0.826',
                    '0.705' => '0.688',
                    '0.247' => '0.797',
                    '0.450' => '0.747',
                    '0.489' => '0.740',
                    '0.074' => '0.862',
                    '0.378' => '0.765',
                    '0.846' => '0.642',
                    '0.748' => '0.677',
                    '0.102' => '0.847',
                    '0.757' => '0.674',
                    '0.215' => '0.804',
                    '0.971' => '0.519',
                    '0.784' => '0.666',
                    '0.640' => '0.705',
                    '0.401' => '0.761',
                    '0.230' => '0.801',
                    '0.585' => '0.718',
                    '0.002' => '0.969',
                    '0.275' => '0.786',
                    '0.866' => '0.632',
                    '0.822' => '0.651',
                    '0.918' => '0.596',
                    '0.851' => '0.639',
                    '0.415' => '0.758',
                    '0.097' => '0.850',
                    '0.981' => '0.486',
                    '0.122' => '0.839',
                    '0.633' => '0.706',
                    '0.083' => '0.856',
                    '0.688' => '0.693',
                    '0.605' => '0.713',
                    '0.160' => '0.821',
                    '0.776' => '0.667',
                    '0.068' => '0.865',
                    '0.689' => '0.693',
                    '0.331' => '0.775',
                    '0.245' => '0.797',
                    '0.153' => '0.823',
                    '0.076' => '0.860',
                    '0.380' => '0.765',
                    '0.714' => '0.685',
                    '0.606' => '0.712',
                    '0.161' => '0.821',
                    '0.791' => '0.663',
                    '0.259' => '0.792',
                    '0.425' => '0.755',
                    '0.859' => '0.635',
                    '0.528' => '0.731',
                    '0.346' => '0.773',
                    '0.213' => '0.805',
                    '0.682' => '0.694',
                    '0.418' => '0.757',
                    '0.112' => '0.843',
                    '0.094' => '0.851',
                    '0.516' => '0.734',
                    '0.941' => '0.574',
                    '0.124' => '0.836',
                    '0.777' => '0.667',
                    '0.396' => '0.762',
                    '0.806' => '0.657',
                    '0.501' => '0.737',
                    '0.544' => '0.728',
                    '0.436' => '0.752',
                    '0.616' => '0.709',
                    '0.186' => '0.814',
                    '0.466' => '0.744',
                    '0.104' => '0.847',
                    '0.858' => '0.636',
                    '0.564' => '0.723',
                    '0.222' => '0.802',
                    '0.026' => '0.905',
                    '0.317' => '0.778',
                    '0.850' => '0.640',
                    '0.712' => '0.686',
                    '0.292' => '0.783',
                    '0.995' => '0.357',
                    '0.938' => '0.577',
                    '0.530' => '0.730',
                    '0.862' => '0.634',
                    '0.007' => '0.945',
                    '0.771' => '0.668',
                    '0.482' => '0.741',
                    '0.550' => '0.727',
                    '0.016' => '0.921',
                    '0.655' => '0.702',
                    '0.051' => '0.878',
                    '0.722' => '0.684',
                    '0.521' => '0.733',
                    '0.815' => '0.653',
                    '0.194' => '0.812',
                    '0.282' => '0.785',
                    '0.549' => '0.727',
                    '0.169' => '0.819',
                    '0.110' => '0.844',
                    '0.877' => '0.626',
                    '0.227' => '0.801',
                    '0.667' => '0.699',
                    '0.121' => '0.839',
                    '0.326' => '0.776',
                    '0.948' => '0.566',
                    '0.925' => '0.590',
                    '0.719' => '0.684',
                    '0.265' => '0.791',
                    '0.509' => '0.735',
                    '0.394' => '0.762',
                    '0.340' => '0.773',
                    '0.622' => '0.708',
                    '0.908' => '0.604',
                    '0.663' => '0.700',
                    '0.902' => '0.609'
                  }
        }
;

1;
