package Bio::EnsEMBL::VDrawableContainer;

use strict;

use Bio::EnsEMBL::GlyphSet::Videogram;

sub new {
  my $class = shift;
  my $self  = $class->_init(@_);

  ########## loop over all the glyphsets the user wants:
  foreach (@{$self->{'contents'}}) {
    my ($container, $config) = @$_;
    
    $container->{'web_species'} ||= $ENV{'ENSEMBL_SPECIES'};
    
    my @chromosomes = ($container->{'chr'});
    my @configs     = $config->get_tracks;
    my $scalex      = $config->get_parameter('image_height') / $config->get_parameter('container_width');
    my $pos         = 100000;
    my $tmp         = {};
    my $flag        = 0;
    my (@glyphsets, %chr_glyphset_counts);
    
    if ($config->get_parameter('all_chromosomes') eq 'yes') { 
      @chromosomes = @{$config->species_defs->get_config($container->{'web_species'}, 'ENSEMBL_CHROMOSOMES') || []};
      @chromosomes = reverse @chromosomes if $container->{'format'} && $container->{'format'} eq 'pdf'; # reverse the order for drawing
      $flag        = 1;
    }
   
    $config->texthelper->{'_scalex'}           = $scalex;
    $config->{'transform'}->{'scalex'}         = $scalex;
    $config->{'transform'}->{'absolutescalex'} = 1;
    $config->{'transform'}->{'translatex'}    += $config->get_parameter('top_margin');
    
    foreach my $chr (@chromosomes) {
      $container->{'chr'} = $chr;
      
      foreach my $row_config (@configs) {
        my $display = $row_config->get('display') || ($row_config->get('on') eq 'on' ? 'normal' : 'off');
        
        next if $display eq 'off' || $display =~ /highlight/;
        
        my $classname = "$self->{'prefix'}::GlyphSet::" . $row_config->get('glyphset');
        
        next unless $self->dynamic_use($classname);
        
        my $glyphset;
        
        ## create a new glyphset for this row
        eval {
          $glyphset = $classname->new({
	          container  => $container,
	          config     => $config,
	          my_config  => $row_config,
	          strand     => 0,
	          highlights => $self->{'highlights'},
            display    => $display,
          });
        };
        
        if ($@ || !$glyphset) {
          my $reason = $@ || "No reason given just returns undef";
          warn "GLYPHSET: glyphset $classname failed (@{[$self->{container}{web_species}]}/$ENV{'ENSEMBL_SCRIPT'} at " . gmtime() . "\nGLYPHSET:  $reason";
	        next;
        }
        
        $glyphset->{'chr'} = $chr; 

        ## Parse and/or cache data for use on subsequent chromosomes
        if ($row_config->get('id')) {
          my $data = $self->{'storage'}{$row_config->get('id')};
          
          unless (defined($data->{$chr})) {
            $data = $glyphset->data(\@chromosomes); 
            $self->{'storage'}{$row_config->get('id')} = $data;
          }
          
          $glyphset->{'data'} = $data;
        }

        $glyphset->render_normal;

        if (@{$glyphset->{'glyphs'} || []}) {
	        push @glyphsets, $glyphset;
	        $chr_glyphset_counts{$chr}++;
        } elsif (!$row_config->get('hide_empty')) {
	        push @glyphsets, $glyphset;
	        $chr_glyphset_counts{$chr}++;
        }
      }
    }    

    ## Firstly lets work how many entries to draw per row!
    ## Then work out the minimum start for each of these rows
    ## We then shift up all these points up by that many base 
    ## pairs to close up any gaps

    my ($max_gs_chr)    = sort { $b <=> $a } values %chr_glyphset_counts;
    my $glyphsets       = @glyphsets;
    my $group_size      = $config->get_parameter('group_size') || $max_gs_chr;
    my $entries_per_row = $config->get_parameter('columns')    || (int(($glyphsets/$group_size - 1) / ($config->get_parameter('rows') || 1) + 1) * $group_size);
    $entries_per_row    = $max_gs_chr if $max_gs_chr > $entries_per_row;
    my $spacing         = $self->{'spacing'};
    my $entry_no        = 0;
    my $row_count       = 0;
    my $row_index       = 0;
    my $yoffset         = 0;
    my $current_chr     = undef;
    my (@min, @max);
    
    $config->set_parameter('max_height', 0);
    $config->set_parameter('max_width',  0);
    
    foreach my $glyphset (@glyphsets) {
      if ($current_chr ne $glyphset->{'chr'}) { ## Can we fit all the chr stuff in!
        $row_count += $chr_glyphset_counts{$glyphset->{'chr'}};
        
        if ($row_count > $entries_per_row) {
          $row_index++;
          $row_count = 0;
        }
        
        $current_chr = $glyphset->{'chr'};
      }
      
      $glyphset->{'row_index'} = $row_index;
      
      next unless defined $glyphset->minx;
      
      $min[$row_index] = $glyphset->minx if !defined $min[$row_index] || $min[$row_index] > $glyphset->minx;
    }
    
    ## Close up gap!

    my $translateX = shift @min;
    
    $config->{'transform'}->{'translatex'} -= $translateX * $scalex;
    
    my $xoffset   = -$translateX * $scalex;
    my $row_index = 0;

    foreach my $glyphset (@glyphsets) {
      if ($row_index != $glyphset->{'row_index'}) {  ## We are on a new row - so reset the yoffset [horizontal] to 0 
        my $translateX = shift @min;
        
        $row_index = $glyphset->{'row_index'};
        $yoffset   = 0;
        $xoffset  += $config->image_width - $translateX * $scalex;
        
        ## Shift down - and then close up gap!
        $config->{'transform'}->{'translatex'} += $config->image_width - $translateX * $scalex;
      }
      
      $config->set_parameter('max_width', $xoffset + $config->get_parameter('image_width'));
      
      ########## set up the label for this strip 
      ########## first we get the max width of label in characters
      my $feature_type_1 = $glyphset->my_config('feature_type')   || ($glyphset->my_config('keys') ? $glyphset->my_config('keys')->[0] : undef);
      my $feature_type_2 = $glyphset->my_config('feature_type_2') || ($glyphset->my_config('keys') ? $glyphset->my_config('keys')->[1] : undef);
      my $label_1        = $glyphset->my_config('label')          || ($feature_type_1 ? $glyphset->my_colour($feature_type_1, 'text')  : undef);
      my $label_2        = $glyphset->my_config('label_2')        || ($feature_type_2 ? $glyphset->my_colour($feature_type_2, 'text')  : undef);
      
      $label_1 = $glyphset->{'chr'} if $glyphset->{'my_config'}->id eq 'Videogram' && $flag;
      
      my $gw  = length(length $label_2 > length $label_1 ? $label_2 : $label_1);
      
      if ($gw > 0) {
        ########## and convert it to pels
        $gw = $config->texthelper->width('Small');
        
        ########## If the '_label' position is not 'above' move the labels below the image
        my $label_x  = $config->get_parameter('label') eq 'above' ? 0 : $config->get_parameter('image_height');
        $label_x    += 4 - $config->get_parameter('top_margin');
        my $label_y  = ($glyphset->maxy + $glyphset->miny - $gw) / 2;
        my $colour_1 = $glyphset->my_config('colour')   || ($feature_type_1 ? $glyphset->my_colour($feature_type_1, 'label') : undef);
        my $colour_2 = $glyphset->my_config('colour_2') || ($feature_type_2 ? $glyphset->my_colour($feature_type_2, 'label') : undef);
        
        if ($label_1) {
          $glyphset->push($glyphset->Text({
            x         => $label_x / $scalex,
            y         => ($glyphset->maxy + $glyphset->miny - length($label_1) * $gw) / 2,
            height    => $gw * length($label_1),
            font      => 'Small',
            text      => $label_1,
            absolutey => 1,
            colour    => $colour_1
          }));
        }
        
        if ($label_2) {
          $glyphset->push($glyphset->Text({
            x         => ($label_x + 2 + $config->texthelper->height('Tiny')) / $scalex,
            y         => ($glyphset->maxy + $glyphset->miny - length($label_2) * $gw) / 2,
            height    => $gw * length($label_2),
            font      => 'Small',
            text      => $label_2,
            absolutey => 1,
            colour    => $colour_2
          }));
        }
      }
      
      ########## remove any whitespace at the top of this row
      $config->{'transform'}->{'translatey'} = -$glyphset->miny + $spacing/2 + $yoffset;
      $glyphset->transform;
      
      ########## translate the top of the next row to the bottom of this one
      $yoffset += $glyphset->height + $spacing;
      $config->set_parameter('max_height',  $yoffset + $spacing) if $yoffset + $spacing > $config->get_parameter('max_height');
    }
    
    $self->{'glyphsets'} = \@glyphsets;
  }
  
  $self->timer_push("DrawableContainer->new: End GlyphSets");
  
  return $self;
}

sub _init {
  my $class    = shift;
  my $contents = shift;
  
  if (ref $contents eq 'ARRAY') {
    my $T = [];
    
    while (@$contents) {
      push @$T, [ splice @$contents, 0, 2 ];
    }
    
    $contents = $T;
  } else {
    $contents = [[ $contents, shift ]];
  }

  my ($highlights, $strandedness, $spacing, $storage) = @_;

  my $self = {
    glyphsets               => [],
    config                  => $contents->[0][1],
    storage                 => $storage,
    prefix                  => 'Bio::EnsEMBL',
    contents                => $contents,
    highlights              => $highlights   || [],
    spacing                 => $spacing      || $contents->[0][1]->get_parameter('spacing') || 0,
    strandedness            => $strandedness || 0,
    __extra_block_spacing__ => 0,
    timer                   => $contents->[0][1]->species_defs->timer
  };

  $self->{'strandedness'} = 1 if $self->{'config'}->get_parameter('text_export');

  bless $self, $class;
  return $self;
}

sub timer_push {
  my ($self, $tag, $dep) = @_;
  $self->{'timer'}->push($tag, $dep, 'draw');
}

########## render does clever drawing things
sub render {
  my ($self, $type) = @_;
  
  ########## build the name/type of render object we want
  my $renderer_type = qq(Bio::EnsEMBL::VRenderer::$type);
  ########## dynamic require of the right type of renderer

  return unless $self->dynamic_use($renderer_type);

  ########## big, shiny, rendering 'GO' button
  my $renderer = $renderer_type->new(
    $self->{'config'},
    $self->{'vc'},
    $self->{'glyphsets'}
  );
  
  return $renderer->canvas;
}

sub config {
  my ($self, $config) = @_;
  $self->{'config'} = $config if defined $config;
  return $self->{'config'};
}

sub glyphsets { return @{$_[0]->{'glyphsets'}}; }

sub dynamic_use {
  my ($self, $classname) = @_;
  my ($parent_namespace, $module) = $classname =~ /^(.*::)(.*?)$/;
  
  no strict 'refs';
  return 1 if $parent_namespace->{"$module::"}; # return if already used
  
  eval "require $classname";
  
  if ($@) {
    warn "VDrawableContainer: failed to use $classname\nVDrawableContainer: $@";
    return 0;
  }
  
  $classname->import;
  return 1;
}

1;
