// Copyright (c) 1999  Utrecht University (The Netherlands),
// ETH Zurich (Switzerland), Freie Universitaet Berlin (Germany),
// INRIA Sophia-Antipolis (France), Martin-Luther-University Halle-Wittenberg
// (Germany), Max-Planck-Institute Saarbruecken (Germany), RISC Linz (Austria),
// and Tel-Aviv University (Israel).  All rights reserved.
//
// This file is part of CGAL (www.cgal.org); you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; version 2.1 of the License.
// See the file LICENSE.LGPL distributed with CGAL.
//
// Licensees holding a valid commercial license may use this file in
// accordance with the commercial license agreement provided with the software.
//
// This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
//
// $URL: svn+ssh://scm.gforge.inria.fr/svn/cgal/branches/CGAL-3.3-branch/Kernel_23/include/CGAL/Vector_3.h $
// $Id: Vector_3.h 37197 2007-03-17 18:29:25Z afabri $
// 
//
// Author(s)     : Andreas Fabri, Stefan Schirra

#ifndef CGAL_VECTOR_3_H
#define CGAL_VECTOR_3_H

#include <CGAL/Origin.h>
#include <CGAL/Kernel/mpl.h>
#include <CGAL/representation_tags.h>
#include <boost/static_assert.hpp>
#include <boost/type_traits.hpp>
#include <CGAL/Kernel/Return_base_tag.h>

CGAL_BEGIN_NAMESPACE

template <class R_>
class Vector_3 : public R_::Kernel_base::Vector_3
{
  typedef typename R_::RT                    RT;
  typedef typename R_::FT                    FT;
  typedef typename R_::Segment_3             Segment_3;
  typedef typename R_::Ray_3                 Ray_3;
  typedef typename R_::Line_3                Line_3;
  typedef typename R_::Point_3               Point_3;
  typedef typename R_::Direction_3           Direction_3;
  typedef typename R_::Aff_transformation_3  Aff_transformation_3;

  typedef Vector_3                            Self;
  BOOST_STATIC_ASSERT((boost::is_same<Self, typename R_::Vector_3>::value));

public:

  typedef typename R_::Kernel_base::Vector_3         Rep;

  const Rep& rep() const
  {
    return *this;
  }

  Rep& rep()
  {
    return *this;
  }

  typedef          R_                       R;

  Vector_3() {}

  Vector_3(const Rep& v)
      : Rep(v) {}

  Vector_3(const Point_3& a, const Point_3& b)
    : Rep(typename R::Construct_vector_3()(Return_base_tag(), a, b)) {}

  Vector_3(const Segment_3& s)
    : Rep(typename R::Construct_vector_3()(Return_base_tag(), s)) {}

  Vector_3(const Ray_3& r)
    : Rep(typename R::Construct_vector_3()(Return_base_tag(), r)) {}

  Vector_3(const Line_3& l)
    : Rep(typename R::Construct_vector_3()(Return_base_tag(), l)) {}

  Vector_3(const Null_vector& v)
    : Rep(typename R::Construct_vector_3()(Return_base_tag(), v)) {}

  template < typename T1, typename T2, typename T3 >
#ifdef __INTEL_COMPILER
      Self
#else
  Vector_3
#endif
          (const T1 &x, const T2 &y, const T3 &z)
    : Rep(typename R::Construct_vector_3()(Return_base_tag(), x, y, z)) {}

  Vector_3(const RT& x, const RT& y, const RT& z, const RT& w)
    : Rep(typename R::Construct_vector_3()(Return_base_tag(), x, y, z, w)) {}

  Direction_3 direction() const
  {
    return R().construct_direction_3_object()(*this);
  }

  Vector_3 transform(const Aff_transformation_3 &t) const
  {
    return t.transform(*this);
  }

  Vector_3 operator-() const
  {
    return R().construct_opposite_vector_3_object()(*this);
  }

  Vector_3 operator-(const Vector_3& v) const
  {
    return R().construct_difference_of_vectors_3_object()(*this,v);
  }

  Vector_3 operator+(const Vector_3& v) const
  {
    return R().construct_sum_of_vectors_3_object()(*this,v);
  }

  Vector_3 operator/(const RT& c) const
  {
   return R().construct_divided_vector_3_object()(*this,c);
  }

  Vector_3 operator/(const typename First_if_different<FT,RT>::Type & c) const
  {
   return R().construct_divided_vector_3_object()(*this,c);
  }

  typename Qualified_result_of<typename R::Compute_x_3, Vector_3>::type
  x() const
  {
    return R().compute_x_3_object()(*this);
  }

  typename Qualified_result_of<typename R::Compute_y_3, Vector_3>::type
  y() const
  {
    return R().compute_y_3_object()(*this);
  }

  typename Qualified_result_of<typename R::Compute_z_3, Vector_3>::type
  z() const
  {
    return R().compute_z_3_object()(*this);
  }

  typename Qualified_result_of<typename R::Compute_hx_3, Vector_3>::type
  hx() const
  {
    return R().compute_hx_3_object()(*this);
  }

  typename Qualified_result_of<typename R::Compute_hy_3, Vector_3>::type
  hy() const
  {
    return R().compute_hy_3_object()(*this);
  }

  typename Qualified_result_of<typename R::Compute_hz_3, Vector_3>::type
  hz() const
  {
    return R().compute_hz_3_object()(*this);
  }

  typename Qualified_result_of<typename R::Compute_hw_3, Vector_3>::type
  hw() const
  {
    return R().compute_hw_3_object()(*this);
  }

  typename Qualified_result_of<typename R::Compute_x_3, Vector_3>::type
  cartesian(int i) const
  {
    CGAL_kernel_precondition( (i == 0) || (i == 1) || (i == 2) );
    if (i==0) return x();
    if (i==1) return y();
    return z();
  }

  typename Qualified_result_of<typename R::Compute_hw_3, Vector_3>::type
  homogeneous(int i) const
  {
    CGAL_kernel_precondition( (i >= 0) || (i <= 3) );
    if (i==0) return hx();
    if (i==1) return hy();
    if (i==2) return hz();
    return hw();
  }

  int dimension() const
  {
      return 3;
  }

  typename Qualified_result_of<typename R::Compute_x_3, Vector_3>::type
  operator[](int i) const
  {
      return cartesian(i);
  }

  typename Qualified_result_of<typename R::Compute_squared_length_3, Vector_3>::type
  squared_length() const
  {
    return R().compute_squared_length_3_object()(*this);
  }

};


template <class R >
std::ostream&
insert(std::ostream& os, const Vector_3<R>& v, const Cartesian_tag&) 
{
  switch(os.iword(IO::mode)) {
    case IO::ASCII :
      return os << v.x() << ' ' << v.y()  << ' ' << v.z();
    case IO::BINARY :
      write(os, v.x());
      write(os, v.y());
      write(os, v.z());
      return os;
    default:
      os << "VectorC3(" << v.x() << ", " << v.y() <<  ", " << v.z() << ")";
      return os;
  }
}

template <class R >
std::ostream&
insert(std::ostream& os, const Vector_3<R>& v, const Homogeneous_tag&)
{
  switch(os.iword(IO::mode))
  {
    case IO::ASCII :
        return os << v.hx() << ' ' << v.hy() << ' ' << v.hz() << ' ' << v.hw();
    case IO::BINARY :
        write(os, v.hx());
        write(os, v.hy());
        write(os, v.hz());
        write(os, v.hw());
        return os;
    default:
        return os << "VectorH3(" << v.hx() << ", "
                                 << v.hy() << ", "
                                 << v.hz() << ", "
                                 << v.hw() << ')';
  }
}

template < class R >
std::ostream&
operator<<(std::ostream& os, const Vector_3<R>& v)
{
  return insert(os, v, typename R::Kernel_tag() );
}


template <class R >
std::istream&
extract(std::istream& is, Vector_3<R>& v, const Cartesian_tag&) 
{
  typename R::FT x, y, z;
  switch(is.iword(IO::mode)) {
    case IO::ASCII :
      is >> x >> y >> z;
      break;
    case IO::BINARY :
      read(is, x);
      read(is, y);
      read(is, z);
      break;
    default:
      std::cerr << "" << std::endl;
      std::cerr << "Stream must be in ascii or binary mode" << std::endl;
      break;
  }
  if (is)
      v = Vector_3<R>(x, y, z);
  return is;
}

template <class R >
std::istream&
extract(std::istream& is, Vector_3<R>& v, const Homogeneous_tag&) 
{
  typename R::RT hx, hy, hz, hw;
  switch(is.iword(IO::mode))
  {
    case IO::ASCII :
        is >> hx >> hy >> hz >> hw;
        break;
    case IO::BINARY :
        read(is, hx);
        read(is, hy);
        read(is, hz);
        read(is, hw);
        break;
    default:
        std::cerr << "" << std::endl;
        std::cerr << "Stream must be in ascii or binary mode" << std::endl;
        break;
  }
  if (is)
    v = Vector_3<R>(hx, hy, hz, hw);
  return is;
}

template < class R >
std::istream&
operator>>(std::istream& is, Vector_3<R>& v)
{
  return extract(is, v, typename R::Kernel_tag() );
}

CGAL_END_NAMESPACE

#endif // CGAL_VECTOR_3_H
