/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: transfer.h,v 1.15 2004/05/01 14:34:17 bazsi Exp $
 *
 ***************************************************************************/
#ifndef ZORP_TRANSFER_H_INCLUDED
#define ZORP_TRANSFER_H_INCLUDED

#include <zorp/zorp.h>
#include <zorp/zobject.h>
#include <zorp/stream.h>
#include <zorp/proxy.h>
#include <zorp/poll.h>

/* transfer flags */
#define ZTF_COPY_TO_SERVER	0x0001 /* copy data to server direction */
#define ZTF_COPY_TO_CLIENT	0x0002 /* copy data to client direction */
#define ZTF_SEPARATE_SHUTDOWN	0x0004 /* shutdown directions separately */
#define ZTF_COMPLETE_COPY	0x0008 /* always copy the whole data until EOF, 
                                        * even if there was an error during
                                        * write */
                                        
#define ZTF_PROXY_STREAMS_POLLED 0x0010 /* proxy streams are already polled, do not add/remove from/to ZPoll */

/* endpoint indexes */

#define ZTE_STACKED     0x02

#define ZTE_CLIENT	0
#define ZTE_SERVER	1
#define ZTE_DOWN_CLIENT ZTE_CLIENT | ZTE_STACKED
#define ZTE_DOWN_SERVER ZTE_SERVER | ZTE_STACKED
#define ZTE_MAX		3

/* progress call reasons */
#define ZTP_STARTUP     0
#define ZTP_TIMED	1
#define ZTP_CHUNK_BASED 2

typedef struct _ZTransfer ZTransfer;

/* virtual function table for ZTransfer */
typedef struct _ZTransferFuncs
{
  ZObjectFuncs super;
  GIOStatus (*src_read)(ZTransfer *self, ZStream *s, gchar *buf, gsize count, gsize *bytes_read, GError **err);
  GIOStatus (*src_write)(ZTransfer *self, ZStream *s, const gchar *buf, gsize count, gsize *bytes_written, GError **err);
  GIOStatus (*src_shutdown)(ZTransfer *self, ZStream *s, gint shutdown_mode, GError **err);
  GIOStatus (*dst_read)(ZTransfer *self, ZStream *s, gchar *buf, gsize count, gsize *bytes_read, GError **err);
  GIOStatus (*dst_write)(ZTransfer *self, ZStream *s, const gchar *buf, gsize count, gsize *bytes_written, GError **err);
  GIOStatus (*dst_shutdown)(ZTransfer *self, ZStream *s, gint shutdown_mode, GError **err);
  gboolean (*progress)(ZTransfer *self, gint call_reason);
} ZTransferFuncs;


typedef struct _ZTransferIOBuffer
{
  gchar *buf;
  gsize ofs, end;
  guint packet_count, packet_bytes;
} ZTransferIOBuffer;

static inline gboolean
z_transfer_io_buffer_empty(ZTransferIOBuffer *b)
{
  return b->ofs == b->end;
}

/* status flags definitions */

#define ZTS_EOF_CLIENT_R         0x0001
#define ZTS_EOF_SERVER_R         0x0002
#define ZTS_EOF_CLIENT_W         0x0004
#define ZTS_EOF_SERVER_W         0x0008

#define ZTS_EOF_BITS		 0x00FF	      /* bits reserved for storing EOF state */
#define ZTS_FINISHED             0x0100       /* transfer is finished */
#define ZTS_FAILED		 0x0200       /* general Transfer-related failure */

/**
 * ZTransfer:
 * @super:
 * @endpoints: stores ZStream references indexed by ZTE_*
 * @owner: the owning ZProxy class
 * @poll: the poll used
 *
 * This is an abstract class to help implementing non-blocking data
 * transfers in various proxies. It takes care about buffering, timeouts,
 * the user needs to supply I/O functions to perform protocol conform I/O. 
 * I/O functions (src_read, src_write etc) may return G_IO_STATUS_AGAIN when
 * they feel that it is not actually possible to send traffic. 
 * G_IO_STATUS_EOF is used to indicate the end of data transfer (which may
 * not necessarily mean the physical end-of-file)
 **/
struct _ZTransfer
{
  ZObject super;
  /* no user code should touch these */
  ZStream *endpoints[ZTE_MAX + 1];
  ZStreamContext contexts[ZTE_MAX + 1];
  ZStackedProxy *stacked;
  ZTransferIOBuffer buffers[ZTE_MAX + 1];
  GSource *timeout_source;
  ZProxy *owner;
  ZPoll *poll;
  gsize buffer_size;
  glong timeout;
  gulong flags;
  gulong status;
  guint shutdown_mask;
  
  guint64 global_packet_count;
};

extern ZClass ZTransfer__class;

ZTransfer *z_transfer_new(ZClass *class, 
                          ZProxy *owner, 
                          ZPoll *poll, 
                          ZStream *source, ZStream *dest, 
                          ZStackedProxy *stacked,
                          gsize buf_size, 
                          glong timeout,
                          gulong flags);
void
z_transfer_set_stacked_proxy(ZTransfer *self, ZStackedProxy *proxy);

void z_transfer_free_method(ZObject *s);

gboolean z_transfer_run(ZTransfer *self);

void z_transfer_update_eof_mask(ZTransfer *self, guint add_mask);
void z_transfer_disable_copy(ZTransfer *self, guint direction);
void z_transfer_enable_copy(ZTransfer *self, guint direction);


/* helper functions for virtual methods */
static inline GIOStatus 
z_transfer_src_read(ZTransfer *self, ZStream *s, gchar *buf, gsize count, gsize *bytes_read, GError **err)
{
  return Z_FUNCS(self, ZTransfer)->src_read(self, s, buf, count, bytes_read, err);
}

static inline GIOStatus 
z_transfer_src_write(ZTransfer *self, ZStream *s, gchar *buf, gsize count, gsize *bytes_written, GError **err)
{
  return Z_FUNCS(self, ZTransfer)->src_write(self, s, buf, count, bytes_written, err);
}

static inline GIOStatus 
z_transfer_src_shutdown(ZTransfer *self, ZStream *s, gint shutdown_mode, GError **err)
{
  return Z_FUNCS(self, ZTransfer)->src_shutdown(self, s, shutdown_mode, err);
}

static inline GIOStatus 
z_transfer_dst_read(ZTransfer *self, ZStream *s, gchar *buf, gsize count, gsize *bytes_read, GError **err)
{
  return Z_FUNCS(self, ZTransfer)->dst_read(self, s, buf, count, bytes_read, err);
}

static inline GIOStatus 
z_transfer_dst_write(ZTransfer *self, ZStream *s, gchar *buf, gsize count, gsize *bytes_written, GError **err)
{
  return Z_FUNCS(self, ZTransfer)->dst_write(self, s, buf, count, bytes_written, err);
}

static inline GIOStatus 
z_transfer_dst_shutdown(ZTransfer *self, ZStream *s, gint shutdown_mode, GError **err)
{
  return Z_FUNCS(self, ZTransfer)->dst_shutdown(self, s, shutdown_mode, err);
}

static inline GIOStatus
z_transfer_progress(ZTransfer *self, gint call_reason)
{
  return Z_FUNCS(self, ZTransfer)->progress(self, call_reason);
}


#endif
