/* configparser.l

   Parser for configuration file

   (C) 1999 Stanislav Meduna <stano@eunet.sk>


   The format of the file is

   [section]
   entry = value...
   entry[qualif] = value...

   The parser is rather primitive and takes all values
   as simple strings - no quoting or something like that
   is possible.

   The caller gets the entries via callback.

*/
string    \"[^\n"]+\"
name [A-Za-z/_][A-Za-z0-9/_()]*

combegin  "#"|"//"

lbracket  "["
rbracket  "]"
equal     "="

white    [[:blank:]]

%s COMMENT ENTRY VALUES
%option prefix="configparser"
%option outfile="lex.yy.c"
%option noyywrap
%option yylineno
%{
#include <config.h>
#include <configuration.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <libintl.h>

#define _(s) dgettext(PACKAGE, s)

static char section[32];
static config_entry_t *entry = 0;


static void reset_entry(void)
{
	free_config_entry(entry);
	entry = 0;
}

static void got_section(const char *str)
{
	str++;
	strncpy(section, str, sizeof(section)-1);
	section[sizeof(section)-1] = 0;
	section[strlen(section)-1] = 0;
}

static void got_entry(const char *str)
{
	entry = (config_entry_t *) malloc(sizeof(config_entry_t));

	if (entry == NULL)
	{
		fprintf(stderr, _("Out of memory"));
		return;
	}

	strncpy(entry->section, section, sizeof(entry->section)-1);
	entry->section[sizeof(entry->section)-1] = 0;

	strncpy(entry->entry, str, sizeof(entry->entry)-1);
	entry->entry[sizeof(entry->entry)-1] = 0;

	*entry->qualif = 0;

	entry->nvalues = 0;
}

static void got_optional(const char *str)
{
	if (! entry)
		return;

	if (*entry->qualif != 0)
	{
		fprintf(stderr, _("Second qualifier ignored on line %d\n"), yylineno);
		return;
	}

	str++;

	strncpy(entry->qualif, str, sizeof(entry->qualif)-1);
	entry->qualif[sizeof(entry->qualif)-1] = 0;
	entry->qualif[strlen(entry->qualif)-1] = 0;
}

static void got_value(const char *str)
{
	int n;

	if (! entry)
		return;

	if (entry->nvalues >= sizeof(entry->values)/sizeof(entry->values[0]))
	{
		fprintf(stderr, _("Excessive values ignored on line %d\n"), yylineno);
		return;
	}


	if (*str == '\"')
		str++; /* skip quote */

	n = strlen(str);
	if (n)
		n--;

	entry->values[entry->nvalues] = strdup(str);

	if (entry->values[entry->nvalues] == NULL)
	{
		fprintf(stderr, _("Out of memory"));
		return;
	}

	if (str[n] == '\"')
		entry->values[entry->nvalues][n] = 0;

	entry->nvalues++;
}

static void got_all(void)
{
	if (! entry)
		return;

	config_entry_cb(entry);

	entry = 0;
}

static void syntax_error()
{
	fprintf(stderr, _("Syntax error in configuration at line %d\n"), yylineno);
	reset_entry();
}
%}
%%
	{ BEGIN(INITIAL); entry = 0; *section = 0; }

<INITIAL>{combegin}	{ BEGIN(COMMENT); }
<INITIAL>{white}	;
<INITIAL>\n		;
<INITIAL>{lbracket}{name}{rbracket} { got_section(yytext); BEGIN(COMMENT); }
<INITIAL>{name}		{ got_entry(yytext); BEGIN(ENTRY); }
<INITIAL>.		{ syntax_error(); BEGIN(COMMENT); }

<ENTRY>{lbracket}{name}{rbracket}	{ got_optional(yytext); }
<ENTRY>{equal}		{ BEGIN(VALUES); }
<ENTRY>{white}		;
<ENTRY>.		{ syntax_error(); BEGIN(COMMENT); }
<ENTRY>\n		{ syntax_error(); BEGIN(INITIAL); }

<VALUES>\n		{ got_all(); BEGIN(INITIAL); }
<VALUES>{white}		;
<VALUES>{name}		{ got_value(yytext); }
<VALUES>{string}	{ got_value(yytext); }
<VALUES>.		{ syntax_error(); BEGIN(COMMENT); }


<COMMENT>.	;
<COMMENT>\n	{ reset_entry(); BEGIN(INITIAL); }


%%

